/***************************************************************************************************************************
 
    FILE NAME :
      OURSTRNG.CPP
 
 
    DESCRIPTION :
      This module contains the methods used to implement the class OurString.   This class is intended to subsitute for the
      STL or ANSI String classes for compilers that do not support those classes.
 
 
    METHODS DEFINED IN THIS MODULE :
      OurString :: OurString   ( const char * cszSourceString = 0 )
      OurString :: OurString   ( size_t stLen , const char *  caSource = 0 )
      OurString :: OurString   ( const char OurString & rcourstring )

      virtual void  OurString :: clear   ()
      size_t        OurString :: length  () const

    OPERATORS DEFINED IN THIS MODULE :
      OurString   & OurString :: operator =   ( const char      * cszSource   )
      OurString   & OurString :: operator =   ( const OurString & rcourstring )
      OurString   & OurString :: operator +=  ( const char      * cszSource   )
      OurString   & OurString :: operator +=  ( const OurString & rcourstring )
      Boolean       OurString :: operator ==  ( const char      * cszSource   )
      Boolean       OurString :: operator ==  ( const OurString & rcourstrg   )
      Boolean       OurString :: operator !=  ( const char      * cszSource   )
      Boolean       OurString :: operator !=  ( const OurString & rcourstrg   )
      Boolean       OurString :: operator <=  ( const char      * cszSource   )
      Boolean       OurString :: operator <=  ( const OurString & rcourstrg   )
      Boolean       OurString :: operator >=  ( const char      * cszSource   )
      Boolean       OurString :: operator >=  ( const OurString & rcourstrg   )
      Boolean       OurString :: operator <   ( const char      * cszSource   )
      Boolean       OurString :: operator <   ( const OurString & rcourstrg   )
      Boolean       OurString :: operator >   ( const char      * cszSource   )
      Boolean       OurString :: operator >   ( const OurString & rcourstrg   )


      const OurString operator + ( const char       * csz         , const OurString & rcourstrg   )
      const OurString operator + ( const OurString  & rcourstrg   , const char      * csz         )
      const OurString operator + ( const OurString  & rcourstrg1  , const OurString & rcourstrg2  )




    Copyright 1997, Peter Garner
    This code may be used for any NON-MILITARY purpose.
 
***************************************************************************************************************************/
//  System Defined C Headers
# include <string.h>


//  Locally Defined C Headers
# include <strfcns.h>


//  Locally Defined C++ Headers
# include <ourstrng.hpp>




/***************************************************************************************************************************
  OurString :: OurString ( const char * cszSource = 0 ) 
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that will be the source for this DEEP copy constructor.


  DESCRIPTION :
    This constructor creates an OurString object that is a DEEP copy of 'cszSource'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
OurString :: OurString ( const char * cszSource ) 
  : CharArray ( strLen ( cszSource ) + 1 , cszSource )
{
  //  If we were passed a NULL String, set all of the receivers memory to 0.
  if ( ! strIsValid ( cszSource ) )
  {
    clear () ;

  } /* endif */

} /* OurString :: OurString ( const char * cszSource ) */



/***************************************************************************************************************************
  OurString :: OurString ( size_t stLen , const char * ccaSource )
****************************************************************************************************************************

  ARGUMENTS   :
    size_t    stLen
      The length of the character array 'ccaSource'.

    const T * ccaSource
      The address of a character array that will be the source of this DEEP copy constructor.  This need not be NULL
      terminated.


  DESCRIPTION :
    This constructor creates an OurString object that is a DEEP copy of the first 'stLen' characters in 'ccaSource'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
OurString :: OurString ( size_t stLen , const char * ccaSource )
  : CharArray ( stLen , ccaSource )
{
  //  Add an extra byte for the null terminator
  resize ( stLen + 1 ) ;

  //  If we were passed a NULL String, set all of the receivers memory to 0.
  if ( ! strIsValid ( ccaSource ) )
  {
    clear () ;

    return   ;

  } /* endif */

  //  Get the Base Address as a non const char.  This is needed since some compilers cannot resolve the difference between
  //  constant and non constant overloaded methods
  char  * caMem = (char *) getBase  () ;

  //  Null Terminate the Array
  caMem [stLen] = 0 ; 

} /* OurString :: OurString ( size_t stLen , const char * ccaSource ) */



/***************************************************************************************************************************
  OurString :: OurString ( const OurString & rcourstrng )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString < T > & crtemplarray
      A reference to a constant OurString object that will be the source for this DEEP copy constructor


  DESCRIPTION :
    This is the OurString copy constructor.  It creates a const OurString < T > & crtemplarray object that is a DEEP copy
    of the argument 'rcourstrng'.


***************************************************************************************************************************/
OurString :: OurString ( const OurString & rcourstrng )
  : CharArray ( rcourstrng )
{

} /* OurString :: OurString ( const OurString & rcourstrng ) */



/***************************************************************************************************************************
    const char * OurString :: className () const
****************************************************************************************************************************

  RETURNS     :
    A constant pointer to the ASCIIZ string "OurString".


  DESCRIPTION :
    This VIRTUAL constant method always returns the class name of the receiver.  This may be used to implement
    a primitive Run Time Type Identification, (RTTI), or anything else you like.  Note that this IS a virtual
    function, and will early bind in the constructors and destructor.  Thus you can also use this method to
    determine in which constructor level you are, although I don't know why you would want to know that!

***************************************************************************************************************************/
const char * OurString :: className () const
{
  return  "OurString" ;

} /* const char * OurString :: className () const */



/***************************************************************************************************************************
  virtual void OurString :: clear ()
****************************************************************************************************************************

  DESCRIPTION :
    This method sets the receivers' ENTIRE memory buffer to 0.  NO DESTRUCTORS are called!  This method actually just calls
    void TemplateArray < char > :: clear ().  That method is defined as protected in the parent class.  This method is
    defined just to expose it as public in this class, since this it is safe to call void TemplateArray < char > :: clear ()
    when the template is instantiated for char.  Please refer to the documentation for
    void TemplateArray < char > :: clear () for more information.

***************************************************************************************************************************/
void OurString :: clear ()
{
  CharArray :: clear () ;

} /* void OurString :: clear () */



/***************************************************************************************************************************
    size_t OurString :: length () const
****************************************************************************************************************************

  RETURNS     :
    This constant accessor method returns the length of the string encapsulated by the receiver NOT INCLUDING the NULL
    terminator.


  DESCRIPTION :
    This constant accessor method returns the length of the string encapsulated by the receiver NOT INCLUDING the NULL
    terminator.  This is the number of elements - 1.

***************************************************************************************************************************/
size_t OurString :: length () const
{
  return  numElems () - 1 ;

} /* size_t OurString :: length () const */



/***************************************************************************************************************************
  OurString & OurString :: operator = ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that will be the source for this DEEP copy initializer.


  RETURNS     :
    This operator returns a reference to the receiver.


  DESCRIPTION :
    This initializer initializes an OurString object to be a DEEP copy of 'cszSource'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
OurString & OurString :: operator = ( const char * cszSource )
{
  init ( strLen ( cszSource ) , cszSource ) ;

  return  * this ;

} /* OurString & OurString :: operator = ( const char * cszSource ) */



/***************************************************************************************************************************
  OurString & OurString :: operator = ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A pointer to the constant OurString object that will be the source for this DEEP copy initializer.


  RETURNS     :
    This operator returns a reference to the receiver.


  DESCRIPTION :
    This initializer initializs an OurString object to be a DEEP copy of 'rcourstrg'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
OurString & OurString :: operator = ( const OurString & rcourstrg )
{
  init ( rcourstrg ) ;

  return  * this ;

} /* OurString & OurString :: operator = ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  OurString & OurString :: operator += ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that will be the source for this concatenation operator


  RETURNS     :
    This operator returns a reference to the receiver.


  DESCRIPTION :
    This operator appends the string 'cszSource' to the receiver.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
OurString & OurString :: operator += ( const char * cszSource )
{
  size_t  stOldLen  = numElems  ()            ;
  size_t  stSrcLen  = strLen    ( cszSource ) ;

  resize ( numElems () + strLen ( cszSource ) ) ;

  if ( stSrcLen )
  {
    memcpy ( (char *) getBase () + stOldLen - 1, cszSource , stSrcLen + 1 ) ;

  } /* endif */

  return  * this ;

} /* OurString & OurString :: operator += ( const char * cszSource ) */



/***************************************************************************************************************************
  OurString & OurString :: operator += ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to the constant OurString object that will be the source for this concatenation operator


  RETURNS     :
    This operator returns a reference to the receiver.


  DESCRIPTION :
    This operator appends the string 'rcourstrg' to the receiver.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
OurString & OurString :: operator += ( const OurString & rcourstrg )
{
  return  OurString :: operator += ( (const char *) rcourstrg ) ;

} /* OurString & OurString :: operator += ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  Boolean OurString :: operator == ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the contents of the receiver and 'cszSource' are identical, false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator == ( const char * cszSource )
{
  Boolean   bRet  = false ;

  //  Check for a null argument
  if ( 0 == cszSource && 0 == getBase () )
  {
    //  If BOTH the receiver and cszSource are NULL, return true
    return  true ; 

  } /* endif */

  //  Since we just checked BOTH for NULL, if either is NULL, return false
  if ( 0 == cszSource || 0 == getBase ()  )
  {
    return  false ;

  } /* endif */

  //  If we get here, we are sure that NEITHER argument is NULL, so we may safely call strcmp.
  bRet  = strcmp ( getBase () , cszSource ) ? false : true ;    

  return  bRet  ;

} /* Boolean OurString :: operator == ( const char * cszSource ) */



/***************************************************************************************************************************
  Boolean OurString :: operator == ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to a constant OurString Object that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the contents of the receiver and 'cszSource' are identical, false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator == ( const OurString & rcourstrg )
{
  return  OurString :: operator == ( (const char *) rcourstrg ) ;

} /* Boolean OurString :: operator == ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  Boolean OurString :: operator != ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns false if the contents of the receiver and 'cszSource' are identical, true otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator != ( const char * cszSource )
{
  return  ! ( * this == cszSource ) ;

} /* Boolean OurString :: operator != ( const char * cszSource ) */



/***************************************************************************************************************************
  Boolean OurString :: operator != ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to a constant OurString Object that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns false if the contents of the receiver and 'cszSource' are identical, true otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator != ( const OurString & rcourstrg )
{
  return  ! ( * this == (const char *) rcourstrg ) ;

} /* Boolean OurString :: operator != ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  Boolean OurString :: operator <= ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the contents of the receiver and 'cszSource' are identical OR if the receiver is LESS
    than the argument 'cszSource', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator <= ( const char * cszSource )
{
  Boolean   bRet  = false ;

  //  Check for a null argument
  if ( 0 == cszSource && 0 == getBase () )
  {
    //  If BOTH the receiver and cszSource are NULL, return true
    return  true ; 

  } /* endif */

  //  Since we just checked BOTH for NULL, if either is NULL, return false
  if ( 0 == cszSource || 0 == getBase ()  )
  {
    return  false ;

  } /* endif */

  //  If we get here, we are sure that NEITHER argument is NULL, so we may safely call strcmp.
  bRet  = strcmp ( getBase () , cszSource ) <= 0 ? true : false ;

  return  bRet  ;

} /* Boolean OurString :: operator <= ( const char * cszSource ) */



/***************************************************************************************************************************
  Boolean OurString :: operator <= ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to a constant OurString Object that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the contents of the receiver and 'cszSource' are identical OR if the receiver is LESS
    than the argument 'cszSource', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator <= ( const OurString & rcourstrg )
{
  return  * this <= (const char *) rcourstrg ;

} /* Boolean OurString :: operator <= ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  Boolean OurString :: operator >= ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the contents of the receiver and 'cszSource' are identical OR if the receiver is GREATER
    than the argument 'cszSource', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator >= ( const char * cszSource )
{
  Boolean   bRet  = false ;

  //  Check for a null argument
  if ( 0 == cszSource && 0 == getBase () )
  {
    //  If BOTH the receiver and cszSource are NULL, return true
    return  true ; 

  } /* endif */

  //  Since we just checked BOTH for NULL, if either is NULL, return false
  if ( 0 == cszSource || 0 == getBase ()  )
  {
    return  false ;

  } /* endif */

  //  If we get here, we are sure that NEITHER argument is NULL, so we may safely call strcmp.
  bRet  = strcmp ( getBase () , cszSource ) >= 0 ? true : false ;

  return  bRet  ;

} /* Boolean OurString :: operator >= ( const char * cszSource ) */



/***************************************************************************************************************************
  Boolean OurString :: operator >= ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to a constant OurString Object that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the contents of the receiver and 'cszSource' are identical OR if the receiver is GREATER
    than the argument 'cszSource', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator >= ( const OurString & rcourstrg )
{
  return  * this >= (const char *) rcourstrg ;

} /* Boolean OurString :: operator >= ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  Boolean OurString :: operator < ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the receiver is LESS than the argument 'cszSource', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator < ( const char * cszSource )
{
  Boolean   bRet  = false ;

  //  Check for a null argument, remember that NULL is considered to be the lowest in the sort order
  if ( 0 == getBase () && 0 == cszSource || 0 != getBase () && 0 == cszSource )
  {
    //  If BOTH the receiver and cszSource are NULL, return false
    return  false ; 

  } /* endif */

  //  If the receiver is NULL, and the argument is not, return true
  if ( 0 == getBase () && 0 != cszSource )
  {
    return  true ;

  } /* endif */

  //  If we get here, we are sure that NEITHER argument is NULL, so we may safely call strcmp.
  bRet  = strcmp ( getBase () , cszSource ) < 0 ? true : false ;

  return  bRet  ;

} /* Boolean OurString :: operator < ( const char * cszSource ) */



/***************************************************************************************************************************
  Boolean OurString :: operator < ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to a constant OurString Object that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the receiver is LESS than the argument 'rcourstrg', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator < ( const OurString & rcourstrg )
{
  return  * this < (const char *) rcourstrg ;

} /* Boolean OurString :: operator < ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  Boolean OurString :: operator > ( const char * cszSource )
****************************************************************************************************************************

  ARGUMENTS   :
    const char  * cszSource
      A pointer to the constant ASCIIZ string that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the receiver is LESS than the argument 'cszSource', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator > ( const char * cszSource )
{
  Boolean   bRet  = false ;

  //  Check for a null argument, remember that NULL is considered to be the lowest in the sort order
  if ( 0 == getBase () && 0 == cszSource || 0 == getBase () && 0 != cszSource )
  {
    //  If BOTH the receiver and cszSource are NULL, return false
    return  false ; 

  } /* endif */

  //  If the receiver is NULL, and the argument is not, return true
  if ( 0 != getBase () && 0 == cszSource )
  {
    return  true ;

  } /* endif */

  //  If we get here, we are sure that NEITHER argument is NULL, so we may safely call strcmp.
  bRet  = strcmp ( getBase () , cszSource ) > 0 ? true : false ;

  return  bRet  ;

} /* Boolean OurString :: operator > ( const char * cszSource ) */



/***************************************************************************************************************************
  Boolean OurString :: operator > ( const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A reference to a constant OurString Object that we are comparing for equality with the receiver.


  RETURNS     :
    This operator returns true if the receiver is GREATER than the argument 'rcourstrg', false otherwise.


  DESCRIPTION :
    This compares the string 'cszSource' to the receiver.

***************************************************************************************************************************/
Boolean OurString :: operator > ( const OurString & rcourstrg )
{
  return  * this > (const char *) rcourstrg ;

} /* Boolean OurString :: operator > ( const OurString & rcourstrg ) */



/***************************************************************************************************************************
  const OurString operator + ( const char * csz , const OurString & rcourstrg )
****************************************************************************************************************************

  ARGUMENTS   :
    const char        * csz
      A pointer to a constant ASCIIZ string that will be the source for this DEEP copy add operator.

    const OurString   & rcourstrg
      A pointer to the constant OurString object that will be the SECOND source for this DEEP copy add operator.


  RETURNS     :
    This operator returns a constant OurString Object that is the concatenation of 'csz' and 'rcourstrg' .


  DESCRIPTION :
    This operator returns a constant OurString Object that is the concatenation of 'csz' and 'rcourstrg' .


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
const OurString operator + ( const char * csz , const OurString & rcourstrg )
{
  size_t      stLen   = strLen ( csz )  ;
  OurString   ourstrg ( strLen ( csz ) + rcourstrg.numElems () ) ;

  if ( stLen )
  {
    memcpy ( (char *) ourstrg , csz , stLen ) ;

  } /* endif */

  memcpy ( (char *) ourstrg + stLen , (const char *) rcourstrg , rcourstrg.numElems () ) ;

  return  ourstrg ;

} /* const OurString operator + ( const char * csz , const OurString & rcourstrg ) */



/***************************************************************************************************************************
  const OurString operator + ( const OurString & rcourstrg , const char * csz )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A pointer to the constant OurString object that will be the source for this DEEP copy add operator.

    const char        * csz
      A pointer to a constant ASCIIZ string that will be the SECOND source for this DEEP copy add operator.


  RETURNS     :
    This operator returns a constant OurString Object that is the concatenation of 'rcourstrg' and 'csz'.


  DESCRIPTION :
    This operator returns a constant OurString Object that is the concatenation of 'rcourstrg' and 'csz'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
const OurString operator + ( const OurString & rcourstrg , const char * csz )
{
  size_t      stLen   = strLen ( csz )  ;
  OurString   ourstrg ( strLen ( csz ) + rcourstrg.numElems () ) ;

  memcpy ( (char *) ourstrg , (const char *) rcourstrg , rcourstrg.length () ) ;

  if ( stLen )
  {
    memcpy ( (char *) ourstrg + rcourstrg.length () , csz , stLen + 1 ) ;

  } /* endif */

  return  ourstrg ;

} /* const OurString operator + ( const OurString & rcourstrg , const char * csz ) */



/***************************************************************************************************************************
  OurString operator + ( const OurString & rcourstrg1 , const OurString & rcourstrg2 )
****************************************************************************************************************************

  ARGUMENTS   :
    const OurString   & rcourstrg
      A pointer to the constant OurString object that will be the source for this DEEP copy add operator.

    const char        * csz
      A pointer to a constant ASCIIZ string that will be the SECOND source for this DEEP copy add operator.


  RETURNS     :
    This operator returns a constant OurString Object that is the concatenation of 'rcourstrg' and 'csz'.


  DESCRIPTION :
    This operator returns a constant OurString Object that is the concatenation of 'rcourstrg' and 'csz'.


  EXCEPTIONS  :
    If the attempt to allocate memory fails, an IOutOfMemory exception will be thrown.

***************************************************************************************************************************/
const OurString operator + ( const OurString & rcourstrg1 , const OurString & rcourstrg2 )
{
  //  We need room for only one NULL Terminator, hence the - 1 at the end of the following.
  OurString   ourstrg ( rcourstrg1.numElems () + rcourstrg2.numElems () - 1 ) ;

  memcpy ( (char *) ourstrg , (const char *) rcourstrg1 , rcourstrg1.length () ) ;

  memcpy ( (char *) ourstrg + rcourstrg1.length () , (const char *) rcourstrg2 , rcourstrg2.numElems () ) ;

  return  ourstrg ;

} /* const OurString operator + ( const OurString & rcourstrg1 , const OurString & rcourstrg2 ) */



/**************************************     END OF FILE OURSTRNG.CPP          *********************************************/

