/* ------------------------------------------------------------------------- */

/* sane - Scanner Access Now Easy.

   umax.c 

   (C) 1997 Oliver Rauch (for this file!)
   (other files have other copyrights, see each file!)

   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.

   This file implements a SANE backend for UMAX flatbed scanners.  */

/* ------------------------------------------------------------------------- */


/* SANE-FLOW-DIAGRAMM

	- sane_init() : initialize backend, attach scanners
	. - sane_get_devices() : query list of scanner-devices
	. - sane_open() : open a particular scanner-device
	. . - sane_set_io_mode : set blocking-mode
	. . - sane_get_select_fd : get scanner-fd
	. . - sane_get_option_descriptor() : get option informations
	. . - sane_control_option() : change option values
	. .
	. . - sane_start() : start image aquisition
	. .   - sane_get_parameters() : returns actual scan-parameters
	. .   - sane_read() : read image-data (from pipe)
	. .
	. . - sane_cancel() : cancel operation
	. - sane_close() : close opened scanner-device
	- sane_exit() : terminate use of backend
*/


/* ------------------------------------------------------------------------- */

#define UMAX_TO_SANE

#define BACKEND_NAME umax

/* ------------------------------------------------------------------------- */


#include "sane/config.h"

#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include "sane/sane.h"
#include "sane/sanei.h"
#include "sane/saneopts.h"
#include "sane/sanei_scsi.h"
#include "umax.h"

#include "sane/sanei_backend.h"

#ifndef PATH_MAX
# define PATH_MAX       1024
#endif


/* ----------------------------- SENSE_HANDLER ----------------------------- */


static SANE_Status sense_handler(int scsi_fd, u_char *result)
{
 u_char asc, ascq, sense;

  sense = result[0x02] & 0x0f;
  asc   = result[0x0c];
  ascq  = result[0x0d];

  if ( (result[0] & 0x80) == 0 ) 
  { 
    return SANE_STATUS_GOOD;                            /* sense key invalid */
  }

  switch(sense)
  {
    case 0x00: /* no sense */
      return SANE_STATUS_GOOD;
     break;

    case 0x03: /* medium error */
      if (ascq==0) { return SANE_STATUS_JAMMED; }
      if (ascq==1) { return SANE_STATUS_NO_DOCS; }
     break;

    case 0x04: /* hardware error */
    case 0x05: /* illegal request */
      return SANE_STATUS_IO_ERROR;
     break;
  }
 return SANE_STATUS_GOOD;
}


/* ---------------------- include lowlevel-driver  ------------------------- */

#include "umax-scsi.c"

/* ------------------------------------------------------------------------- */

#include <sane/sanei_config.h>
#define PATH_UMAX_CONFIG "umax.conf"

/* ------------------------------------------------------------------------- */

#define MM_PER_INCH	25.4

/* ------------------------------------------------------------------------- */

#define P_200_TO_255(per) SANE_UNFIX((per + 100) * 255/200 )
#define P_100_TO_255(per) SANE_UNFIX(per * 255/100 )

/* ------------------------------------------------------------------------- */


static SANE_String_Const scan_mode_list[7];


/* ------------------------------------------------------------------------- */


static SANE_String_Const source_list[4];


/* ------------------------------------------------------------------------- */


static const SANE_String_Const calibration_list[] =
{
    "Calibration ignored",
    "Calibration by image-type",
    "Calibration driver-selected",
    0
};



/* ------------------------------------------------------------------------- */


static const SANE_Int pattern_dim_list[] =
{
  8,                            /* # of elements */
  0, 2, 3, 4, 5, 6, 7, 8
};


/* ------------------------------------------------------------------------- */


static const SANE_Range u8_range =
{
    0,				/* minimum */
  255,				/* maximum */
    0				/* quantization */
};


/* ------------------------------------------------------------------------- */


static const SANE_Range u10_range =
{
    0,				/* minimum */
 1023,				/* maximum */
    0				/* quantization */
};


/* ------------------------------------------------------------------------- */


static const SANE_Range percentage_range =
{
  -100 << SANE_FIXED_SCALE_SHIFT,	/* minimum */
   100 << SANE_FIXED_SCALE_SHIFT,	/* maximum */
     1 << SANE_FIXED_SCALE_SHIFT	/* quantization */
};


/* ------------------------------------------------------------------------- */


static const SANE_Range percentage_range_100 =
{
     0 << SANE_FIXED_SCALE_SHIFT,	/* minimum */
   100 << SANE_FIXED_SCALE_SHIFT,	/* maximum */
     0 << SANE_FIXED_SCALE_SHIFT	/* quantization */
};


/* ------------------------------------------------------------------------- */


static int num_devices;
static Umax_Device *first_dev;
static Umax_Scanner *first_handle;


/* --------------------------- MAX STRING SIZE ----------------------------- */


static size_t max_string_size(const SANE_String_Const strings[])
{
 size_t size, max_size = 0;
 int i;

  for (i = 0; strings[i]; ++i)
  {
    size = strlen (strings[i]) + 1;
    if (size > max_size) { max_size = size; }
  }

return max_size;
}


/* -------------------------------- DO EOF --------------------------------- */


static SANE_Status do_eof(Umax_Scanner *scanner)
{
  DBG(15,"do_eof\n");

  if (scanner->pipe >= 0)
  {
    close(scanner->pipe);
    scanner->pipe = -1;
  }
 return SANE_STATUS_EOF;
}


/* -------------------------------- DO CANCEL ------------------------------ */


static SANE_Status do_cancel(Umax_Scanner *scanner)
{
  DBG(15,"do_cancel\n");

  scanner->scanning = SANE_FALSE;

  do_eof(scanner);                     /* close pipe and reposition scanner */

  if (scanner->reader_pid > 0)
  {
   int exit_status;

    DBG(20,"do_cancel: kill reader_process\n");

    /* ensure child knows it's time to stop: */
    kill (scanner->reader_pid, SIGTERM);
    while (wait (&exit_status) != scanner->reader_pid);
    scanner->reader_pid = 0;
  }

  if (scanner->device->us.sfd >= 0)
  {
    umax_reposition_scanner(&(scanner->device->us));
    umax_give_scanner(&(scanner->device->us));
    DBG(20,"do_cancel: close filedescriptor\n");
    sanei_scsi_close(scanner->device->us.sfd);
    scanner->device->us.sfd = -1;
  }

 return SANE_STATUS_CANCELLED;
}


/* ---------------------------- ATTACH SCANNER ----------------------------- */


static SANE_Status attach_scanner(const char *devicename, Umax_Device **devp)
{
Umax_Device *dev;
int sfd;

  DBG(15,"attach_scanner: %s\n", devicename);

  for (dev = first_dev; dev; dev = dev->next)
  {
    if (strcmp(dev->sane.name, devicename) == 0)
    {
      if (devp) { *devp = dev; }
      return SANE_STATUS_GOOD;
    }
  }

  DBG(3, "attach_scanner: opening %s\n", devicename);
  if (sanei_scsi_open(devicename, &sfd, sense_handler) != 0)
  {
    DBG(1, "attach_scanner: open failed\n");
    return SANE_STATUS_INVAL;
  }

  dev=malloc( sizeof(*dev) );
  if (!dev) { return SANE_STATUS_NO_MEM; }

  umax_init(&(dev->us));
  umax_initialize_values(&(dev->us));

  dev->us.devicename = strdup(devicename);
  dev->us.sfd        = sfd;

  if (umax_identify_scanner(&(dev->us)) != 0)
  {
    DBG(1, "attach_scanner: scanner-identification failed\n");
    sanei_scsi_close(dev->us.sfd);
    dev->us.sfd=-1;
    free(dev);
    return SANE_STATUS_INVAL;
  }

  umax_get_inquiry_values(&(dev->us));
  sanei_scsi_close(dev->us.sfd);
  dev->us.sfd=-1;

  dev->sane.name   = dev->us.devicename;
  dev->sane.vendor = dev->us.vendor;
  dev->sane.model  = dev->us.product;
  dev->sane.type   = "flatbed scanner"; 

  if (strcmp(dev->sane.model,"PSD ") == 0)
  { dev->sane.type = "page scanner"; }

  dev->x_range.min     = SANE_FIX(0);
  dev->x_range.quant   = SANE_FIX(0);
  dev->x_range.max     = SANE_FIX(dev->us.inquiry_fb_width  * MM_PER_INCH);

  dev->y_range.min     = SANE_FIX(0);
  dev->y_range.quant   = SANE_FIX(0);
  dev->y_range.max     = SANE_FIX(dev->us.inquiry_fb_length * MM_PER_INCH);

  dev->x_dpi_range.min   = SANE_FIX(1);
  dev->x_dpi_range.quant = SANE_FIX(1);
  dev->x_dpi_range.max   = SANE_FIX(dev->us.inquiry_x_res);
  dev->x_dpi_range.max   = SANE_FIX(dev->us.inquiry_optical_res);

  dev->y_dpi_range.min   = SANE_FIX(1);
  dev->y_dpi_range.quant = SANE_FIX(1);
  dev->y_dpi_range.max   = SANE_FIX(dev->us.inquiry_y_res);
  dev->y_dpi_range.max   = SANE_FIX(dev->us.inquiry_optical_res);

  dev->analog_gamma_range.min   = SANE_FIX(1.0);
  dev->analog_gamma_range.quant = SANE_FIX(0);
  dev->analog_gamma_range.max   = SANE_FIX(2.0);

  ++num_devices;
  dev->next = first_dev;
  first_dev = dev;

  if (devp) { *devp = dev; }

return SANE_STATUS_GOOD;
}


/* ---------------------------- READER PROCESS ----------------------------- */


/* This function is executed as a child process. */

static int reader_process(Umax_Scanner *scanner, int pipe_fd)
{
 int status;
 unsigned int data_left;
 unsigned int data_to_read;
 FILE *fp;

  DBG(15,"reader_process started\n");

  fp = fdopen (pipe_fd, "w");
  if (!fp) {return 1;}

  DBG(20,"reader_process: starting to READ data\n");

  data_left = scanner->params.lines * scanner->params.bytes_per_line;

  scanner->device->us.row_bufsize = scanner->device->us.bufsize;
  umax_trim_rowbufsize(&(scanner->device->us));              /* trim bufsize */

  DBG(25,"reader_process: reading %u bytes in blocks of %u bytes\n",
         data_left, scanner->device->us.row_bufsize);

  do
  {
    data_to_read = (data_left < scanner->device->us.row_bufsize) ? \
                    data_left : scanner->device->us.row_bufsize;

    umax_get_data_buffer_status(&(scanner->device->us));
    status = umax_read_data_block(&(scanner->device->us),data_to_read);
    if (status == 0) { continue; }
    if (status == -1)
    {
      DBG(1,"reader_process: unable to get image data from scanner!\n");
      fclose(fp);
      return(-1);
    }

    fwrite(scanner->device->us.buffer, 1, data_to_read, fp);
    fflush(fp);

    data_left -= data_to_read;
    DBG(25, "reader_process: buffer of %d bytes read; %d bytes to go\n",
            data_to_read, data_left);
  } while (data_left);

  fclose(fp);

  DBG(20,"reader_process: finished reading data\n");

return 0;
}


/* ------------------------------ INIT OPTIONS ----------------------------- */


static SANE_Status init_options(Umax_Scanner *scanner)
{
 int i;
 int scan_modes;

  DBG(15,"init_options\n");

  memset(scanner->opt, 0, sizeof (scanner->opt));
  memset(scanner->val, 0, sizeof (scanner->val));

  for (i = 0; i < NUM_OPTIONS; ++i)
  {
    scanner->opt[i].size = sizeof (SANE_Word);
    scanner->opt[i].cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
  }

  scanner->opt[OPT_NUM_OPTS].title = SANE_TITLE_NUM_OPTIONS;
  scanner->opt[OPT_NUM_OPTS].desc  = SANE_DESC_NUM_OPTIONS;
  scanner->opt[OPT_NUM_OPTS].cap   = SANE_CAP_SOFT_DETECT;
  scanner->val[OPT_NUM_OPTS].w     = NUM_OPTIONS;

  /* "Mode" group: */
  scanner->opt[OPT_MODE_GROUP].title = "Scan Mode";
  scanner->opt[OPT_MODE_GROUP].desc  = "";
  scanner->opt[OPT_MODE_GROUP].type  = SANE_TYPE_GROUP;
  scanner->opt[OPT_MODE_GROUP].cap   = 0;
  scanner->opt[OPT_MODE_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  scan_modes = -1;

  if (scanner->device->us.inquiry_lineart)
  { scan_mode_list[++scan_modes]="Lineart"; }

  if (scanner->device->us.inquiry_halftone)
  { scan_mode_list[++scan_modes]="Halftone"; }

  if (scanner->device->us.inquiry_grey)
  { scan_mode_list[++scan_modes]="Grey"; }

  if (scanner->device->us.inquiry_color)
  {
/*  
    if (scanner->device->us.inquiry_lineart)
    { scan_mode_list[++scan_modes]="Color Lineart"; }

    if (scanner->device->us.inquiry_halftone) 
    { scan_mode_list[++scan_modes]="Color Halftone"; }
*/

    scan_mode_list[++scan_modes]="Color"; 
  }

  {
   int i=0;
    { source_list[i++]="Flatbed"; }

    if (scanner->device->us.inquiry_adfmode)
    { source_list[i++]="Automatic Document Feeder"; }

    if (scanner->device->us.inquiry_transavail)
    { source_list[i++]="Transparency Adapter"; }
  }
  
  /* scan mode */
  scanner->opt[OPT_MODE].name  = SANE_NAME_SCAN_MODE;
  scanner->opt[OPT_MODE].title = SANE_TITLE_SCAN_MODE;
  scanner->opt[OPT_MODE].desc  = SANE_DESC_SCAN_MODE;
  scanner->opt[OPT_MODE].type  = SANE_TYPE_STRING;
  scanner->opt[OPT_MODE].size  = max_string_size(scan_mode_list);
  scanner->opt[OPT_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  scanner->opt[OPT_MODE].constraint.string_list = scan_mode_list;
  scanner->val[OPT_MODE].s     = strdup(scan_mode_list[0]);

  /* source */
  scanner->opt[OPT_SOURCE].name  = SANE_NAME_SCAN_SOURCE;
  scanner->opt[OPT_SOURCE].title = SANE_TITLE_SCAN_SOURCE;
  scanner->opt[OPT_SOURCE].desc  = SANE_DESC_SCAN_SOURCE;
  scanner->opt[OPT_SOURCE].type  = SANE_TYPE_STRING;
  scanner->opt[OPT_SOURCE].size  = max_string_size(source_list);
  scanner->opt[OPT_SOURCE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  scanner->opt[OPT_SOURCE].constraint.string_list = source_list;
  scanner->val[OPT_SOURCE].s     = strdup(source_list[0]);
  
  /* x-resolution */
  scanner->opt[OPT_X_RESOLUTION].name  = SANE_NAME_SCAN_RESOLUTION;
  scanner->opt[OPT_X_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
  scanner->opt[OPT_X_RESOLUTION].desc  = SANE_DESC_SCAN_RESOLUTION;
  scanner->opt[OPT_X_RESOLUTION].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_X_RESOLUTION].unit  = SANE_UNIT_DPI;
  scanner->opt[OPT_X_RESOLUTION].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_X_RESOLUTION].constraint.range =
                                         &scanner->device->x_dpi_range;
  scanner->val[OPT_X_RESOLUTION].w     = 100 << SANE_FIXED_SCALE_SHIFT;

  /* y-resolution */
  scanner->opt[OPT_Y_RESOLUTION].name  = "Y" SANE_NAME_SCAN_RESOLUTION;
  scanner->opt[OPT_Y_RESOLUTION].title = "Y-Resolution";
  scanner->opt[OPT_Y_RESOLUTION].desc  = SANE_DESC_SCAN_RESOLUTION;
  scanner->opt[OPT_Y_RESOLUTION].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_Y_RESOLUTION].unit  = SANE_UNIT_DPI;
  scanner->opt[OPT_Y_RESOLUTION].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_Y_RESOLUTION].constraint.range =
                                         &scanner->device->y_dpi_range;
  scanner->val[OPT_Y_RESOLUTION].w     = 100 << SANE_FIXED_SCALE_SHIFT;
  scanner->opt[OPT_Y_RESOLUTION].cap  |= SANE_CAP_INACTIVE;

  /* bind resolution */
  scanner->opt[OPT_RESOLUTION_BIND].name  = SANE_NAME_RESOLUTION_BIND;
  scanner->opt[OPT_RESOLUTION_BIND].title = SANE_TITLE_RESOLUTION_BIND;
  scanner->opt[OPT_RESOLUTION_BIND].desc  = SANE_DESC_RESOLUTION_BIND;
  scanner->opt[OPT_RESOLUTION_BIND].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_RESOLUTION_BIND].w     = SANE_TRUE;


  /* 10 bit mode */
  scanner->opt[OPT_TEN_BIT_MODE].name  = SANE_NAME_TEN_BIT_MODE;
  scanner->opt[OPT_TEN_BIT_MODE].title = SANE_TITLE_TEN_BIT_MODE;
  scanner->opt[OPT_TEN_BIT_MODE].desc  = SANE_DESC_TEN_BIT_MODE;
  scanner->opt[OPT_TEN_BIT_MODE].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_TEN_BIT_MODE].w     = SANE_FALSE;

  if ( (scanner->device->us.inquiry_GOB & 4) == 0)
  { scanner->opt[OPT_TEN_BIT_MODE].cap  |= SANE_CAP_INACTIVE; }


  /* negative */
  scanner->opt[OPT_NEGATIVE].name  = SANE_NAME_NEGATIVE;
  scanner->opt[OPT_NEGATIVE].title = SANE_TITLE_NEGATIVE;
  scanner->opt[OPT_NEGATIVE].desc  = SANE_DESC_NEGATIVE;
  scanner->opt[OPT_NEGATIVE].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_NEGATIVE].w     = SANE_FALSE;

  if (scanner->device->us.inquiry_reverse_multi == 0)
  { scanner->opt[OPT_NEGATIVE].cap  |= SANE_CAP_INACTIVE; }

  /* quality-calibration */
  scanner->opt[OPT_QUALITY].name  = SANE_NAME_QUALITY_CAL;
  scanner->opt[OPT_QUALITY].title = SANE_TITLE_QUALITY_CAL;
  scanner->opt[OPT_QUALITY].desc  = SANE_DESC_QUALITY_CAL;
  scanner->opt[OPT_QUALITY].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_QUALITY].w     = SANE_FALSE;

  if (scanner->device->us.inquiry_quality_ctrl == 0)
  { scanner->opt[OPT_QUALITY].cap  |= SANE_CAP_INACTIVE; }


  /* warmup */
  scanner->opt[OPT_WARMUP].name  = SANE_NAME_WARMUP;
  scanner->opt[OPT_WARMUP].title = SANE_TITLE_WARMUP;
  scanner->opt[OPT_WARMUP].desc  = SANE_DESC_WARMUP;
  scanner->opt[OPT_WARMUP].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_WARMUP].w     = SANE_FALSE;

  if (scanner->device->us.inquiry_max_warmup_time == 0)
  { scanner->opt[OPT_WARMUP].cap  |= SANE_CAP_INACTIVE; }


  /* ------------------------------ */

  /* "Geometry" group: */
  scanner->opt[OPT_GEOMETRY_GROUP].title = "Geometry";
  scanner->opt[OPT_GEOMETRY_GROUP].desc  = "";
  scanner->opt[OPT_GEOMETRY_GROUP].type  = SANE_TYPE_GROUP;
  scanner->opt[OPT_GEOMETRY_GROUP].cap   = SANE_CAP_ADVANCED;
  scanner->opt[OPT_GEOMETRY_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* top-left x */
  scanner->opt[OPT_TL_X].name  = SANE_NAME_SCAN_TL_X;
  scanner->opt[OPT_TL_X].title = SANE_TITLE_SCAN_TL_X;
  scanner->opt[OPT_TL_X].desc  = SANE_DESC_SCAN_TL_X;
  scanner->opt[OPT_TL_X].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_TL_X].unit  = SANE_UNIT_MM;
  scanner->opt[OPT_TL_X].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_TL_X].constraint.range = &(scanner->device->x_range);
  scanner->val[OPT_TL_X].w     = 0;

  /* top-left y */
  scanner->opt[OPT_TL_Y].name  = SANE_NAME_SCAN_TL_Y;
  scanner->opt[OPT_TL_Y].title = SANE_TITLE_SCAN_TL_Y;
  scanner->opt[OPT_TL_Y].desc  = SANE_DESC_SCAN_TL_Y;
  scanner->opt[OPT_TL_Y].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_TL_Y].unit  = SANE_UNIT_MM;
  scanner->opt[OPT_TL_Y].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_TL_Y].constraint.range = &(scanner->device->y_range);
  scanner->val[OPT_TL_Y].w     = 0;

  /* bottom-right x */
  scanner->opt[OPT_BR_X].name  = SANE_NAME_SCAN_BR_X;
  scanner->opt[OPT_BR_X].title = SANE_TITLE_SCAN_BR_X;
  scanner->opt[OPT_BR_X].desc  = SANE_DESC_SCAN_BR_X;
  scanner->opt[OPT_BR_X].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_BR_X].unit  = SANE_UNIT_MM;
  scanner->opt[OPT_BR_X].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_BR_X].constraint.range = &(scanner->device->x_range);
  scanner->val[OPT_BR_X].w     = scanner->device->x_range.max;

  /* bottom-right y */
  scanner->opt[OPT_BR_Y].name  = SANE_NAME_SCAN_BR_Y;
  scanner->opt[OPT_BR_Y].title = SANE_TITLE_SCAN_BR_Y;
  scanner->opt[OPT_BR_Y].desc  = SANE_DESC_SCAN_BR_Y;
  scanner->opt[OPT_BR_Y].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_BR_Y].unit  = SANE_UNIT_MM;
  scanner->opt[OPT_BR_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_BR_Y].constraint.range = &(scanner->device->y_range);
  scanner->val[OPT_BR_Y].w     = scanner->device->y_range.max;



  /* ------------------------------ */

  /* "Enhancement" group: */
  scanner->opt[OPT_ENHANCEMENT_GROUP].title = "Enhancement";
  scanner->opt[OPT_ENHANCEMENT_GROUP].desc  = "";
  scanner->opt[OPT_ENHANCEMENT_GROUP].type  = SANE_TYPE_GROUP;
  scanner->opt[OPT_ENHANCEMENT_GROUP].cap   = 0;
  scanner->opt[OPT_ENHANCEMENT_GROUP].constraint_type = SANE_CONSTRAINT_NONE;


  /* brightness */
  scanner->opt[OPT_BRIGHTNESS].name  = SANE_NAME_BRIGHTNESS;
  scanner->opt[OPT_BRIGHTNESS].title = SANE_TITLE_BRIGHTNESS;
  scanner->opt[OPT_BRIGHTNESS].desc  = SANE_DESC_BRIGHTNESS;
  scanner->opt[OPT_BRIGHTNESS].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_BRIGHTNESS].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_BRIGHTNESS].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_BRIGHTNESS].constraint.range = &percentage_range;
  scanner->val[OPT_BRIGHTNESS].w     = 0;

  /* contrast */
  scanner->opt[OPT_CONTRAST].name  = SANE_NAME_CONTRAST;
  scanner->opt[OPT_CONTRAST].title = SANE_TITLE_CONTRAST;
  scanner->opt[OPT_CONTRAST].desc  = SANE_DESC_CONTRAST;
  scanner->opt[OPT_CONTRAST].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_CONTRAST].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_CONTRAST].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_CONTRAST].constraint.range = &percentage_range;
  scanner->val[OPT_CONTRAST].w     = 0;


  /* threshold */
  scanner->opt[OPT_THRESHOLD].name  = SANE_NAME_THRESHOLD;
  scanner->opt[OPT_THRESHOLD].title = SANE_TITLE_THRESHOLD;
  scanner->opt[OPT_THRESHOLD].desc  = SANE_DESC_THRESHOLD;
  scanner->opt[OPT_THRESHOLD].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_THRESHOLD].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_THRESHOLD].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_THRESHOLD].constraint.range = &percentage_range_100;
  scanner->val[OPT_THRESHOLD].w     = SANE_FIX(50);



  /* highlight, white level */
  scanner->opt[OPT_HIGHLIGHT_BIND].name  = SANE_NAME_WHITE_BIND;
  scanner->opt[OPT_HIGHLIGHT_BIND].title = SANE_TITLE_WHITE_BIND;
  scanner->opt[OPT_HIGHLIGHT_BIND].desc  = SANE_DESC_WHITE_BIND;
  scanner->opt[OPT_HIGHLIGHT_BIND].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_HIGHLIGHT_BIND].w     = SANE_TRUE;

  scanner->opt[OPT_HIGHLIGHT].name  = SANE_NAME_WHITE_LEVEL;
  scanner->opt[OPT_HIGHLIGHT].title = SANE_TITLE_WHITE_LEVEL;
  scanner->opt[OPT_HIGHLIGHT].desc  = SANE_DESC_WHITE_LEVEL;
  scanner->opt[OPT_HIGHLIGHT].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_HIGHLIGHT].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_HIGHLIGHT].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_HIGHLIGHT].constraint.range = &percentage_range_100;
  scanner->val[OPT_HIGHLIGHT].w     = SANE_FIX(100);

  scanner->opt[OPT_HIGHLIGHT_R].name  = SANE_NAME_WHITE_LEVEL "r";
  scanner->opt[OPT_HIGHLIGHT_R].title = SANE_TITLE_WHITE_LEVEL " red";
  scanner->opt[OPT_HIGHLIGHT_R].desc  = SANE_DESC_WHITE_LEVEL " for red";
  scanner->opt[OPT_HIGHLIGHT_R].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_HIGHLIGHT_R].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_HIGHLIGHT_R].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_HIGHLIGHT_R].constraint.range = &percentage_range_100;
  scanner->val[OPT_HIGHLIGHT_R].w     = SANE_FIX(100);

  scanner->opt[OPT_HIGHLIGHT_G].name  = SANE_NAME_WHITE_LEVEL "g";
  scanner->opt[OPT_HIGHLIGHT_G].title = SANE_TITLE_WHITE_LEVEL " green";
  scanner->opt[OPT_HIGHLIGHT_G].desc  = SANE_DESC_WHITE_LEVEL " for green";
  scanner->opt[OPT_HIGHLIGHT_G].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_HIGHLIGHT_G].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_HIGHLIGHT_G].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_HIGHLIGHT_G].constraint.range = &percentage_range_100;
  scanner->val[OPT_HIGHLIGHT_G].w     = SANE_FIX(100);

  scanner->opt[OPT_HIGHLIGHT_B].name  = SANE_NAME_WHITE_LEVEL "b";
  scanner->opt[OPT_HIGHLIGHT_B].title = SANE_TITLE_WHITE_LEVEL " blue";
  scanner->opt[OPT_HIGHLIGHT_B].desc  = SANE_DESC_WHITE_LEVEL " for blue";
  scanner->opt[OPT_HIGHLIGHT_B].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_HIGHLIGHT_B].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_HIGHLIGHT_B].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_HIGHLIGHT_B].constraint.range = &percentage_range_100;
  scanner->val[OPT_HIGHLIGHT_B].w     = SANE_FIX(100);


  /* shadow, black level */
  scanner->opt[OPT_SHADOW_BIND].name  = SANE_NAME_BLACK_BIND;
  scanner->opt[OPT_SHADOW_BIND].title = SANE_TITLE_BLACK_BIND;
  scanner->opt[OPT_SHADOW_BIND].desc  = SANE_DESC_BLACK_BIND;
  scanner->opt[OPT_SHADOW_BIND].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_SHADOW_BIND].w     = SANE_TRUE;

  scanner->opt[OPT_SHADOW].name  = SANE_NAME_BLACK_LEVEL;
  scanner->opt[OPT_SHADOW].title = SANE_TITLE_BLACK_LEVEL;
  scanner->opt[OPT_SHADOW].desc  = SANE_DESC_BLACK_LEVEL;
  scanner->opt[OPT_SHADOW].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_SHADOW].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_SHADOW].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_SHADOW].constraint.range = &percentage_range_100;
  scanner->val[OPT_SHADOW].w     = 0;

  scanner->opt[OPT_SHADOW_R].name  = SANE_NAME_BLACK_LEVEL "r";
  scanner->opt[OPT_SHADOW_R].title = SANE_TITLE_BLACK_LEVEL " red";
  scanner->opt[OPT_SHADOW_R].desc  = SANE_DESC_BLACK_LEVEL " for red";
  scanner->opt[OPT_SHADOW_R].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_SHADOW_R].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_SHADOW_R].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_SHADOW_R].constraint.range = &percentage_range_100;
  scanner->val[OPT_SHADOW_R].w     = 0;

  scanner->opt[OPT_SHADOW_G].name  = SANE_NAME_BLACK_LEVEL "g";
  scanner->opt[OPT_SHADOW_G].title = SANE_TITLE_BLACK_LEVEL " green";
  scanner->opt[OPT_SHADOW_G].desc  = SANE_DESC_BLACK_LEVEL " for green";
  scanner->opt[OPT_SHADOW_G].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_SHADOW_G].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_SHADOW_G].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_SHADOW_G].constraint.range = &percentage_range_100;
  scanner->val[OPT_SHADOW_G].w     = 0;

  scanner->opt[OPT_SHADOW_B].name  = SANE_NAME_BLACK_LEVEL "b";
  scanner->opt[OPT_SHADOW_B].title = SANE_TITLE_BLACK_LEVEL " blue";
  scanner->opt[OPT_SHADOW_B].desc  = SANE_DESC_BLACK_LEVEL " for blue";
  scanner->opt[OPT_SHADOW_B].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_SHADOW_B].unit  = SANE_UNIT_PERCENT;
  scanner->opt[OPT_SHADOW_B].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_SHADOW_B].constraint.range = &percentage_range_100;
  scanner->val[OPT_SHADOW_B].w     = 0;



  /* analog-gamma */
  scanner->opt[OPT_ANALOG_GAMMA_BIND].name  = SANE_NAME_ANALOG_GAMMA_BIND;
  scanner->opt[OPT_ANALOG_GAMMA_BIND].title = SANE_TITLE_ANALOG_GAMMA_BIND;
  scanner->opt[OPT_ANALOG_GAMMA_BIND].desc  = SANE_DESC_ANALOG_GAMMA_BIND;
  scanner->opt[OPT_ANALOG_GAMMA_BIND].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_ANALOG_GAMMA_BIND].w     = SANE_TRUE;

  scanner->opt[OPT_ANALOG_GAMMA].name  = SANE_NAME_ANALOG_GAMMA;
  scanner->opt[OPT_ANALOG_GAMMA].title = SANE_TITLE_ANALOG_GAMMA;
  scanner->opt[OPT_ANALOG_GAMMA].desc  = SANE_DESC_ANALOG_GAMMA;
  scanner->opt[OPT_ANALOG_GAMMA].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_ANALOG_GAMMA].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_ANALOG_GAMMA].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_ANALOG_GAMMA].constraint.range =
                                 &(scanner->device->analog_gamma_range);
  scanner->val[OPT_ANALOG_GAMMA].w     = 1 << SANE_FIXED_SCALE_SHIFT;

  scanner->opt[OPT_ANALOG_GAMMA_R].name  = SANE_NAME_ANALOG_GAMMA_R;
  scanner->opt[OPT_ANALOG_GAMMA_R].title = SANE_TITLE_ANALOG_GAMMA_R;
  scanner->opt[OPT_ANALOG_GAMMA_R].desc  = SANE_DESC_ANALOG_GAMMA_R;
  scanner->opt[OPT_ANALOG_GAMMA_R].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_ANALOG_GAMMA_R].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_ANALOG_GAMMA_R].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_ANALOG_GAMMA_R].constraint.range =
                                   &(scanner->device->analog_gamma_range);
  scanner->val[OPT_ANALOG_GAMMA_R].w     = 1 << SANE_FIXED_SCALE_SHIFT;

  scanner->opt[OPT_ANALOG_GAMMA_G].name  = SANE_NAME_ANALOG_GAMMA_G;
  scanner->opt[OPT_ANALOG_GAMMA_G].title = SANE_TITLE_ANALOG_GAMMA_G;
  scanner->opt[OPT_ANALOG_GAMMA_G].desc  = SANE_DESC_ANALOG_GAMMA_G;
  scanner->opt[OPT_ANALOG_GAMMA_G].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_ANALOG_GAMMA_G].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_ANALOG_GAMMA_G].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_ANALOG_GAMMA_G].constraint.range =
                                   &(scanner->device->analog_gamma_range);
  scanner->val[OPT_ANALOG_GAMMA_G].w     = 1 << SANE_FIXED_SCALE_SHIFT;

  scanner->opt[OPT_ANALOG_GAMMA_B].name  = SANE_NAME_ANALOG_GAMMA_B;
  scanner->opt[OPT_ANALOG_GAMMA_B].title = SANE_TITLE_ANALOG_GAMMA_B;
  scanner->opt[OPT_ANALOG_GAMMA_B].desc  = SANE_DESC_ANALOG_GAMMA_B;
  scanner->opt[OPT_ANALOG_GAMMA_B].type  = SANE_TYPE_FIXED;
  scanner->opt[OPT_ANALOG_GAMMA_B].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_ANALOG_GAMMA_B].constraint_type  = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_ANALOG_GAMMA_B].constraint.range =
                                      &(scanner->device->analog_gamma_range);
  scanner->val[OPT_ANALOG_GAMMA_B].w     = 1 << SANE_FIXED_SCALE_SHIFT;


  /* custom-gamma table */
  scanner->opt[OPT_CUSTOM_GAMMA].name  = SANE_NAME_CUSTOM_GAMMA;
  scanner->opt[OPT_CUSTOM_GAMMA].title = SANE_TITLE_CUSTOM_GAMMA;
  scanner->opt[OPT_CUSTOM_GAMMA].desc  = SANE_DESC_CUSTOM_GAMMA;
  scanner->opt[OPT_CUSTOM_GAMMA].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_CUSTOM_GAMMA].w     = SANE_FALSE;

  /* grayscale gamma vector */
  scanner->opt[OPT_GAMMA_VECTOR].name  = SANE_NAME_GAMMA_VECTOR;
  scanner->opt[OPT_GAMMA_VECTOR].title = SANE_TITLE_GAMMA_VECTOR;
  scanner->opt[OPT_GAMMA_VECTOR].desc  = SANE_DESC_GAMMA_VECTOR;
  scanner->opt[OPT_GAMMA_VECTOR].type  = SANE_TYPE_INT;
  scanner->opt[OPT_GAMMA_VECTOR].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_GAMMA_VECTOR].size  =
                                 scanner->gamma_length * sizeof (SANE_Word);
  scanner->opt[OPT_GAMMA_VECTOR].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_GAMMA_VECTOR].constraint.range = &u8_range;
  scanner->val[OPT_GAMMA_VECTOR].wa    = &(scanner->gamma_table[0][0]);

  /* red gamma vector */
  scanner->opt[OPT_GAMMA_VECTOR_R].name  = SANE_NAME_GAMMA_VECTOR_R;
  scanner->opt[OPT_GAMMA_VECTOR_R].title = SANE_TITLE_GAMMA_VECTOR_R;
  scanner->opt[OPT_GAMMA_VECTOR_R].desc  = SANE_DESC_GAMMA_VECTOR_R;
  scanner->opt[OPT_GAMMA_VECTOR_R].type  = SANE_TYPE_INT;
  scanner->opt[OPT_GAMMA_VECTOR_R].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_GAMMA_VECTOR_R].size  =
                                   scanner->gamma_length * sizeof (SANE_Word);
  scanner->opt[OPT_GAMMA_VECTOR_R].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_GAMMA_VECTOR_R].constraint.range = &u8_range;
  scanner->val[OPT_GAMMA_VECTOR_R].wa    = &(scanner->gamma_table[1][0]);

  /* green gamma vector */
  scanner->opt[OPT_GAMMA_VECTOR_G].name  = SANE_NAME_GAMMA_VECTOR_G;
  scanner->opt[OPT_GAMMA_VECTOR_G].title = SANE_TITLE_GAMMA_VECTOR_G;
  scanner->opt[OPT_GAMMA_VECTOR_G].desc  = SANE_DESC_GAMMA_VECTOR_G;
  scanner->opt[OPT_GAMMA_VECTOR_G].type  = SANE_TYPE_INT;
  scanner->opt[OPT_GAMMA_VECTOR_G].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_GAMMA_VECTOR_G].size  =
                                   scanner->gamma_length * sizeof (SANE_Word);
  scanner->opt[OPT_GAMMA_VECTOR_G].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_GAMMA_VECTOR_G].constraint.range = &u8_range;
  scanner->val[OPT_GAMMA_VECTOR_G].wa    = &(scanner->gamma_table[2][0]);

  /* blue gamma vector */
  scanner->opt[OPT_GAMMA_VECTOR_B].name  = SANE_NAME_GAMMA_VECTOR_B;
  scanner->opt[OPT_GAMMA_VECTOR_B].title = SANE_TITLE_GAMMA_VECTOR_B;
  scanner->opt[OPT_GAMMA_VECTOR_B].desc  = SANE_DESC_GAMMA_VECTOR_B;
  scanner->opt[OPT_GAMMA_VECTOR_B].type  = SANE_TYPE_INT;
  scanner->opt[OPT_GAMMA_VECTOR_B].unit  = SANE_UNIT_NONE;
  scanner->opt[OPT_GAMMA_VECTOR_B].size  =
                                   scanner->gamma_length * sizeof (SANE_Word);
  scanner->opt[OPT_GAMMA_VECTOR_B].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_GAMMA_VECTOR_B].constraint.range = &u8_range;
  scanner->val[OPT_GAMMA_VECTOR_B].wa    = &(scanner->gamma_table[3][0]);

  /* halftone dimension */
  scanner->opt[OPT_HALFTONE_DIMENSION].name  = SANE_NAME_HALFTONE_DIMENSION;
  scanner->opt[OPT_HALFTONE_DIMENSION].title = SANE_TITLE_HALFTONE_DIMENSION;
  scanner->opt[OPT_HALFTONE_DIMENSION].desc  = SANE_DESC_HALFTONE_DIMENSION;
  scanner->opt[OPT_HALFTONE_DIMENSION].type  = SANE_TYPE_INT;
  scanner->opt[OPT_HALFTONE_DIMENSION].unit  = SANE_UNIT_PIXEL;
  scanner->opt[OPT_HALFTONE_DIMENSION].constraint_type =
                                               SANE_CONSTRAINT_WORD_LIST;
  scanner->opt[OPT_HALFTONE_DIMENSION].constraint.word_list = pattern_dim_list;
  scanner->val[OPT_HALFTONE_DIMENSION].w     = 0;

  /* halftone pattern */
  scanner->opt[OPT_HALFTONE_PATTERN].name  = SANE_NAME_HALFTONE_PATTERN;
  scanner->opt[OPT_HALFTONE_PATTERN].title = SANE_TITLE_HALFTONE_PATTERN;
  scanner->opt[OPT_HALFTONE_PATTERN].desc  = SANE_DESC_HALFTONE_PATTERN;
  scanner->opt[OPT_HALFTONE_PATTERN].type  = SANE_TYPE_INT;
  scanner->opt[OPT_HALFTONE_PATTERN].size  = 0;
  scanner->opt[OPT_HALFTONE_PATTERN].constraint_type = SANE_CONSTRAINT_RANGE;
  scanner->opt[OPT_HALFTONE_PATTERN].constraint.range = &u8_range;
  scanner->val[OPT_HALFTONE_PATTERN].wa    = scanner->halftone_pattern;


  /* ------------------------------ */

  /* "Advanced" group: */
  scanner->opt[OPT_ADVANCED_GROUP].title = "Advanced";
  scanner->opt[OPT_ADVANCED_GROUP].desc  = "";
  scanner->opt[OPT_ADVANCED_GROUP].type  = SANE_TYPE_GROUP;
  scanner->opt[OPT_ADVANCED_GROUP].cap   = SANE_CAP_ADVANCED;
  scanner->opt[OPT_ADVANCED_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* slow-scan */
  scanner->opt[OPT_SLOW].name  = "slow";
  scanner->opt[OPT_SLOW].title = "Slow speed";
  scanner->opt[OPT_SLOW].desc  = "Scan with slow speed";
  scanner->opt[OPT_SLOW].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_SLOW].w     = SANE_FALSE;

  /* smear */
  scanner->opt[OPT_SMEAR].name  = SANE_NAME_SMEAR;
  scanner->opt[OPT_SMEAR].title = SANE_TITLE_SMEAR;
  scanner->opt[OPT_SMEAR].desc  = SANE_DESC_SMEAR;
  scanner->opt[OPT_SMEAR].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_SMEAR].w     = SANE_FALSE;

  /* calibration mode */
  scanner->opt[OPT_CALIB_MODE].name  = "calibrationmode";
  scanner->opt[OPT_CALIB_MODE].title = "Calibration Mode";
  scanner->opt[OPT_CALIB_MODE].desc  = "Define calibration mode";
  scanner->opt[OPT_CALIB_MODE].type  = SANE_TYPE_STRING;
  scanner->opt[OPT_CALIB_MODE].size  = max_string_size(calibration_list);
  scanner->opt[OPT_CALIB_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  scanner->opt[OPT_CALIB_MODE].constraint.string_list = calibration_list;
  scanner->val[OPT_CALIB_MODE].s     = strdup(calibration_list[1]);

  if (scanner->device->us.inquiry_calibration == 0)
  { scanner->opt[OPT_CALIB_MODE].cap  |= SANE_CAP_INACTIVE; }


  /* RGB_PREVIEW_PATCH */
  scanner->opt[OPT_RGB_PREVIEW_PATCH].name  = SANE_NAME_RGB_PREVIEW_PATCH;
  scanner->opt[OPT_RGB_PREVIEW_PATCH].title = SANE_TITLE_RGB_PREVIEW_PATCH;
  scanner->opt[OPT_RGB_PREVIEW_PATCH].desc  = SANE_DESC_RGB_PREVIEW_PATCH;
  scanner->opt[OPT_RGB_PREVIEW_PATCH].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_RGB_PREVIEW_PATCH].w     = SANE_FALSE;

  /* START_SCAN_PATCH */
  scanner->opt[OPT_START_SCAN_PATCH].name  = SANE_NAME_START_SCAN_PATCH;
  scanner->opt[OPT_START_SCAN_PATCH].title = SANE_TITLE_START_SCAN_PATCH;
  scanner->opt[OPT_START_SCAN_PATCH].desc  = SANE_DESC_START_SCAN_PATCH;
  scanner->opt[OPT_START_SCAN_PATCH].type  = SANE_TYPE_BOOL;
  scanner->val[OPT_START_SCAN_PATCH].w     = SANE_FALSE;


  /* preview */
  scanner->opt[OPT_PREVIEW].name  = SANE_NAME_PREVIEW;
  scanner->opt[OPT_PREVIEW].title = SANE_TITLE_PREVIEW;
  scanner->opt[OPT_PREVIEW].desc  = SANE_DESC_PREVIEW;
  scanner->opt[OPT_PREVIEW].type  = SANE_TYPE_BOOL;
  scanner->opt[OPT_PREVIEW].cap   = SANE_CAP_SOFT_DETECT |
                                    SANE_CAP_SOFT_SELECT;
  scanner->val[OPT_PREVIEW].w     = SANE_FALSE;

  sane_control_option( scanner, OPT_MODE, SANE_ACTION_SET_VALUE,
                       (SANE_String *) scan_mode_list[scan_modes], NULL );
  
return SANE_STATUS_GOOD;
}


/* -------------------------------- SANE INIT ------------------------------ */


SANE_Status sane_init(SANE_Int *version_code, SANE_Auth_Callback authorize)
{
 char dev_name[PATH_MAX];
 size_t len;
 FILE *fp;

  DBG_INIT();

  DBG(10,"sane_init\n");

  if (version_code) { *version_code = SANE_VERSION_CODE(V_MAJOR, V_MINOR, 0); }

  fp = sanei_open_config(PATH_UMAX_CONFIG);
  if (!fp) 
  {
    attach_scanner("/dev/scanner", 0);      /* no config-file: /dev/scanner */
    return SANE_STATUS_GOOD;
  }

  while(fgets(dev_name, sizeof(dev_name), fp))
  {
    if (dev_name[0] == '#') { continue; }           /* ignore line comments */
    len = strlen (dev_name);
    if (dev_name[len - 1] == '\n') { dev_name[--len] = '\0'; }

    if (!len) { continue; }                           /* ignore empty lines */

    attach_scanner(dev_name, 0);
  }

  fclose(fp);

return SANE_STATUS_GOOD;
}


/* ------------------------------ SANE EXIT -------------------------------- */


void sane_exit(void)
{
 Umax_Device *dev, *next;

  DBG(10,"sane_exit\n");

  for (dev = first_dev; dev; dev = next)
  {
    next = dev->next;
    free(dev->us.devicename);
    free(dev);
  }
}


/* ----------------------------- SANE GET DEVICES -------------------------- */


SANE_Status sane_get_devices(const SANE_Device ***device_list, SANE_Bool local_only)
{
 static const SANE_Device **devlist = 0;
 Umax_Device *dev;
 int i;

  DBG(10,"sane_get_devices\n");

  if (devlist) { free (devlist); }

  devlist = malloc((num_devices + 1) * sizeof (devlist[0]));
  if (!devlist) { return SANE_STATUS_NO_MEM; }

  i = 0;

  for (dev = first_dev; i < num_devices; dev = dev->next)
  { devlist[i++] = &dev->sane; }

  devlist[i++] = 0;

  *device_list = devlist;

return SANE_STATUS_GOOD;
}


/* ------------------------------- SANE OPEN ------------------------------- */


SANE_Status sane_open(SANE_String_Const devicename, SANE_Handle *handle)
{
 Umax_Device *dev;
 SANE_Status status;
 Umax_Scanner *scanner;
 int i, j;

  DBG(10,"sane_open\n");

  if (devicename[0])                                /* search for devicename */
  {
    for (dev = first_dev; dev; dev = dev->next)
    { if (strcmp(dev->sane.name, devicename) == 0) { break; } }

    if (!dev)
    {
      status = attach_scanner(devicename, &dev);
      if (status != SANE_STATUS_GOOD) { return status; }
    }
  }
  else
  { dev = first_dev; }                /* empty devicname -> use first device */

  if (!dev) { return SANE_STATUS_INVAL; }

  scanner = malloc(sizeof (*scanner));
  if (!scanner) { return SANE_STATUS_NO_MEM; }

  memset(scanner, 0, sizeof (*scanner));

  scanner->device         = dev;
  scanner->device->us.sfd = -1;
  scanner->pipe           = -1;

  if (scanner->device->us.inquiry_GIB & 4)
  {
    scanner->gamma_length = 1024;                           /* 10 bits input */
    scanner->output_bytes  = 1;                             /* 8 bits output */
    scanner->device->us.gamma_input_bits = 4;
    DBG(20, "Using 10 bits for gamma input\n");
  }
  else
  {
    scanner->gamma_length = 256;                             /* 8 bits input */
    scanner->output_bytes  = 1;                             /* 8 bits output */
    DBG(20, "Using 8 bits for gamma input\n");
  }

  /* so gamma_table[0] converts GIB to GOB */
  for (j = 0; j < scanner->gamma_length; ++j)
  {
    scanner->gamma_table[0][j] = j * scanner->device->us.max_value
                                   / scanner->gamma_length;
  }

  /* gamma_table[1,2,3] does'nt convert (GIB->GIB) */
  for (i = 1; i < 4; ++i)
  {
    for (j = 0; j < scanner->gamma_length; ++j)
    { scanner->gamma_table[i][j] = j; }
  }

  init_options(scanner);

  /* insert newly opened handle into list of open handles: */
  scanner->next = first_handle;
  first_handle = scanner;

  *handle = scanner;
return SANE_STATUS_GOOD;
}


/* --------------------------------- SANE CLOSE ---------------------------- */


void sane_close(SANE_Handle handle)
{
 Umax_Scanner *prev, *scanner;

  DBG(10,"sane_close\n");

  /* remove handle from list of open handles: */
  prev = 0;

  for (scanner = first_handle; scanner; scanner = scanner->next)
  {
    if (scanner == handle) { break; }

    prev = scanner;
  }
  
  if (!scanner)
  {
    DBG(1, "close: invalid handle %p\n", handle);
    return;		/* oops, not a handle we know about */
  }

  if (scanner->scanning) { do_cancel(handle); } 

  if (prev) { prev->next = scanner->next; }
  else { first_handle = scanner; }

  free(handle);
}


/* --------------------- SANE GET OPTION DESCRIPTOR ------------------------ */


const SANE_Option_Descriptor * sane_get_option_descriptor(SANE_Handle handle, SANE_Int option)
{
 Umax_Scanner *scanner = handle;

  DBG(50,"sane_get_option_descriptor %d\n", option);

  if ((unsigned) option >= NUM_OPTIONS) { return 0; }

return scanner->opt + option;
}


/* ------------------------- SANE CONTROL OPTION --------------------------- */


SANE_Status sane_control_option(SANE_Handle handle, SANE_Int option, SANE_Action action, void *val, SANE_Int *info)
{
 Umax_Scanner *scanner = handle;
 SANE_Status status;
 SANE_Word w, cap;

  if (info) { *info = 0; }

  if (scanner->scanning) { return SANE_STATUS_DEVICE_BUSY; }

  if (option >= NUM_OPTIONS) { return SANE_STATUS_INVAL; }

  cap = scanner->opt[option].cap;

  if (!SANE_OPTION_IS_ACTIVE (cap)) { return SANE_STATUS_INVAL; }

  if (action == SANE_ACTION_GET_VALUE)
  {
    DBG(50,"sane_control_option %d, get value\n",option);
    switch (option)
    {
       /* word options: */
      case OPT_RESOLUTION_BIND:
      case OPT_X_RESOLUTION:
      case OPT_Y_RESOLUTION:
      case OPT_TL_X:
      case OPT_TL_Y:
      case OPT_BR_X:
      case OPT_BR_Y:
      case OPT_PREVIEW:
      case OPT_NUM_OPTS:
      case OPT_TEN_BIT_MODE:
      case OPT_NEGATIVE:
      case OPT_QUALITY:
      case OPT_WARMUP:
      case OPT_RGB_PREVIEW_PATCH:
      case OPT_START_SCAN_PATCH:
      case OPT_SLOW:
      case OPT_SMEAR:
      case OPT_ANALOG_GAMMA_BIND:
      case OPT_ANALOG_GAMMA:
      case OPT_ANALOG_GAMMA_R:
      case OPT_ANALOG_GAMMA_G:
      case OPT_ANALOG_GAMMA_B:
      case OPT_BRIGHTNESS:
      case OPT_CONTRAST:
      case OPT_THRESHOLD:
      case OPT_HIGHLIGHT_BIND:
      case OPT_HIGHLIGHT:
      case OPT_HIGHLIGHT_R:
      case OPT_HIGHLIGHT_G:
      case OPT_HIGHLIGHT_B:
      case OPT_SHADOW_BIND:
      case OPT_SHADOW:
      case OPT_SHADOW_R:
      case OPT_SHADOW_G:
      case OPT_SHADOW_B:
      case OPT_CUSTOM_GAMMA:
      case OPT_HALFTONE_DIMENSION:
        *(SANE_Word *) val = scanner->val[option].w;
       return SANE_STATUS_GOOD;

      /* word-array options: */
      case OPT_GAMMA_VECTOR:
      case OPT_GAMMA_VECTOR_R:
      case OPT_GAMMA_VECTOR_G:
      case OPT_GAMMA_VECTOR_B:
      case OPT_HALFTONE_PATTERN:
        memcpy (val, scanner->val[option].wa, scanner->opt[option].size);
       return SANE_STATUS_GOOD;

      /* string options: */
      case OPT_SOURCE:
        if ( (strcmp(val, "Flatbed") == 0) ||
             (strcmp(val, "Automatic Document Feeder") == 0) )
        {
          scanner->device->x_range.max = SANE_FIX(scanner->device->us.inquiry_fb_width  * MM_PER_INCH);
          scanner->device->y_range.max = SANE_FIX(scanner->device->us.inquiry_fb_length * MM_PER_INCH);
        }
        else if (strcmp(val, "Transparency Adapter") == 0)
        {
          scanner->device->x_range.max = SANE_FIX(scanner->device->us.inquiry_uta_width  * MM_PER_INCH);
          scanner->device->y_range.max = SANE_FIX(scanner->device->us.inquiry_uta_length * MM_PER_INCH);
        }
        if (info) { *info |= SANE_INFO_RELOAD_PARAMS; }
      /* fall through */
      case OPT_MODE:
      case OPT_CALIB_MODE:
        strcpy (val, scanner->val[option].s);
       return SANE_STATUS_GOOD;
    }
  }
  else if (action == SANE_ACTION_SET_VALUE)
  {
    DBG(50,"sane_control_option %d, set value\n",option);
    if (!SANE_OPTION_IS_SETTABLE (cap)) { return SANE_STATUS_INVAL; }

    status = sanei_constrain_value(scanner->opt+option, val, info);
    if (status != SANE_STATUS_GOOD) { return status; }

    switch (option)
    {
      /* (mostly) side-effect-free word options: */
      case OPT_X_RESOLUTION:
      case OPT_Y_RESOLUTION:
      case OPT_TL_X:
      case OPT_TL_Y:
      case OPT_BR_X:
      case OPT_BR_Y:
        if (info) { *info |= SANE_INFO_RELOAD_PARAMS; }
        /* fall through */
      case OPT_NUM_OPTS:
      case OPT_NEGATIVE:
      case OPT_QUALITY:
      case OPT_WARMUP:
      case OPT_RGB_PREVIEW_PATCH:
      case OPT_START_SCAN_PATCH:
      case OPT_SLOW:
      case OPT_SMEAR:
      case OPT_PREVIEW:
      case OPT_ANALOG_GAMMA:
      case OPT_ANALOG_GAMMA_R:
      case OPT_ANALOG_GAMMA_G:
      case OPT_ANALOG_GAMMA_B:
      case OPT_BRIGHTNESS:
      case OPT_CONTRAST:
      case OPT_THRESHOLD:
      case OPT_HIGHLIGHT:
      case OPT_HIGHLIGHT_R:
      case OPT_HIGHLIGHT_G:
      case OPT_HIGHLIGHT_B:
      case OPT_SHADOW:
      case OPT_SHADOW_R:
      case OPT_SHADOW_G:
      case OPT_SHADOW_B:
        scanner->val[option].w = *(SANE_Word *) val;
       return SANE_STATUS_GOOD;

      case OPT_TEN_BIT_MODE:
        scanner->val[option].w = *(SANE_Word *) val;
        if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }
	if (scanner->val[option].w == SANE_FALSE)              /* 8 bit mode */
	{
          scanner->opt[OPT_GAMMA_VECTOR].constraint.range = &u8_range;
          scanner->output_bytes  = 1;                        /* 8 bits output */
	}
	else                                                  /* 10 bit mode */
	{
          scanner->opt[OPT_GAMMA_VECTOR].constraint.range = &u10_range;
          scanner->output_bytes  = 2;                       /* 10 bits output */
	}
       return SANE_STATUS_GOOD;

      case OPT_ANALOG_GAMMA_BIND:
        scanner->val[option].w = *(SANE_Word *) val;
        if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }
	if (scanner->val[option].w == SANE_FALSE)
	{
          scanner->opt[OPT_ANALOG_GAMMA].cap   |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_ANALOG_GAMMA_R].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_ANALOG_GAMMA_G].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_ANALOG_GAMMA_B].cap &= ~SANE_CAP_INACTIVE;
	}
	else
	{
          scanner->opt[OPT_ANALOG_GAMMA].cap   &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_ANALOG_GAMMA_R].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_ANALOG_GAMMA_G].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_ANALOG_GAMMA_B].cap |= SANE_CAP_INACTIVE;
	}
       return SANE_STATUS_GOOD;

      case OPT_HIGHLIGHT_BIND:
        scanner->val[option].w = *(SANE_Word *) val;
        if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }
	if (scanner->val[option].w == SANE_FALSE)
	{
          scanner->opt[OPT_HIGHLIGHT].cap   |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_HIGHLIGHT_R].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_HIGHLIGHT_G].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_HIGHLIGHT_B].cap &= ~SANE_CAP_INACTIVE;
	}
	else
	{
          scanner->opt[OPT_HIGHLIGHT].cap   &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_HIGHLIGHT_R].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_HIGHLIGHT_G].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_HIGHLIGHT_B].cap |= SANE_CAP_INACTIVE;
	}
       return SANE_STATUS_GOOD;

      case OPT_SHADOW_BIND:
        scanner->val[option].w = *(SANE_Word *) val;
        if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }
	if (scanner->val[option].w == SANE_FALSE)
	{
          scanner->opt[OPT_SHADOW].cap   |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_SHADOW_R].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_SHADOW_G].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_SHADOW_B].cap &= ~SANE_CAP_INACTIVE;
	}
	else
	{
          scanner->opt[OPT_SHADOW].cap   &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_SHADOW_R].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_SHADOW_G].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_SHADOW_B].cap |= SANE_CAP_INACTIVE;
	}
       return SANE_STATUS_GOOD;

      case OPT_RESOLUTION_BIND:
        scanner->val[option].w = *(SANE_Word *) val;

        if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }
	if (scanner->val[option].w == SANE_FALSE)
	{ /* don't bind */
          scanner->opt[OPT_Y_RESOLUTION].cap &= ~SANE_CAP_INACTIVE;
          scanner->opt[OPT_X_RESOLUTION].title = "X-Resolution";
        }
	else
	{ /* bind */
          scanner->opt[OPT_Y_RESOLUTION].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_X_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
        }
       return SANE_STATUS_GOOD;

      /* side-effect-free word-array options: */
      case OPT_HALFTONE_PATTERN:
      case OPT_GAMMA_VECTOR:
      case OPT_GAMMA_VECTOR_R:
      case OPT_GAMMA_VECTOR_G:
      case OPT_GAMMA_VECTOR_B:
        memcpy (scanner->val[option].wa, val, scanner->opt[option].size);
       return SANE_STATUS_GOOD;

      /* side-effect-free single-string options: */
      case OPT_SOURCE:
      case OPT_CALIB_MODE:
        if (scanner->val[option].s) { free (scanner->val[option].s); }
        scanner->val[option].s = strdup (val);
      return SANE_STATUS_GOOD;

      /* options with side-effects: */

      case OPT_CUSTOM_GAMMA:
        w = *(SANE_Word *) val;
        if (w == scanner->val[OPT_CUSTOM_GAMMA].w) { return SANE_STATUS_GOOD; } 

        scanner->val[OPT_CUSTOM_GAMMA].w = w;
        if (w)
        {
           const char *mode = scanner->val[OPT_MODE].s;

           if (strcmp(mode, "Grey") == 0)
           { scanner->opt[OPT_GAMMA_VECTOR].cap &= ~SANE_CAP_INACTIVE; }
           else if (strcmp(mode, "Color") == 0)
           {
             scanner->opt[OPT_GAMMA_VECTOR].cap   &= ~SANE_CAP_INACTIVE;
             scanner->opt[OPT_GAMMA_VECTOR_R].cap &= ~SANE_CAP_INACTIVE;
             scanner->opt[OPT_GAMMA_VECTOR_G].cap &= ~SANE_CAP_INACTIVE;
             scanner->opt[OPT_GAMMA_VECTOR_B].cap &= ~SANE_CAP_INACTIVE;
           }
        }
        else
        {
          scanner->opt[OPT_GAMMA_VECTOR].cap   |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_GAMMA_VECTOR_R].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_GAMMA_VECTOR_G].cap |= SANE_CAP_INACTIVE;
          scanner->opt[OPT_GAMMA_VECTOR_B].cap |= SANE_CAP_INACTIVE;
        }
        if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }
       return SANE_STATUS_GOOD;

      case OPT_MODE:
      {
       int halftoning;

        if (scanner->val[option].s) { free (scanner->val[option].s); }
        scanner->val[option].s = strdup(val);

        if (info) {*info |=SANE_INFO_RELOAD_OPTIONS | SANE_INFO_RELOAD_PARAMS;}

        scanner->opt[OPT_NEGATIVE].cap           |= SANE_CAP_INACTIVE; 

        scanner->opt[OPT_TEN_BIT_MODE].cap       |= SANE_CAP_INACTIVE;

        scanner->opt[OPT_CUSTOM_GAMMA].cap       |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_GAMMA_VECTOR].cap       |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_GAMMA_VECTOR_R].cap     |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_GAMMA_VECTOR_G].cap     |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_GAMMA_VECTOR_B].cap     |= SANE_CAP_INACTIVE;

        scanner->opt[OPT_ANALOG_GAMMA_BIND].cap  |= SANE_CAP_INACTIVE;
        scanner->val[OPT_ANALOG_GAMMA_BIND].w     = SANE_TRUE;
        scanner->opt[OPT_ANALOG_GAMMA].cap       |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_ANALOG_GAMMA_R].cap     |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_ANALOG_GAMMA_G].cap     |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_ANALOG_GAMMA_B].cap     |= SANE_CAP_INACTIVE;

        scanner->opt[OPT_CONTRAST].cap           |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_BRIGHTNESS].cap         |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_THRESHOLD].cap          |= SANE_CAP_INACTIVE;

        scanner->opt[OPT_HIGHLIGHT_BIND].cap     |= SANE_CAP_INACTIVE;
        scanner->val[OPT_HIGHLIGHT_BIND].w        = SANE_TRUE;
        scanner->opt[OPT_HIGHLIGHT].cap          |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_HIGHLIGHT_R].cap        |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_HIGHLIGHT_G].cap        |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_HIGHLIGHT_B].cap        |= SANE_CAP_INACTIVE;

        scanner->opt[OPT_SHADOW_BIND].cap        |= SANE_CAP_INACTIVE;
        scanner->val[OPT_SHADOW_BIND].w           = SANE_TRUE;
        scanner->opt[OPT_SHADOW].cap             |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_SHADOW_R].cap           |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_SHADOW_G].cap           |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_SHADOW_B].cap           |= SANE_CAP_INACTIVE;

        scanner->opt[OPT_HALFTONE_DIMENSION].cap |= SANE_CAP_INACTIVE;
        scanner->opt[OPT_HALFTONE_PATTERN].cap   |= SANE_CAP_INACTIVE;

        halftoning = (strcmp(val, "Halftone") == 0 ||
                      strcmp(val, "Color Halftone") == 0);

        if (halftoning || strcmp(val, "Lineart") == 0 ||
            strcmp(val, "Color Lineart") == 0)
        { /* one bit modes */
          if (scanner->device->us.inquiry_reverse)
          { scanner->opt[OPT_NEGATIVE].cap  &= ~SANE_CAP_INACTIVE; }

          if (halftoning)
          { /* halftoning modes */
            scanner->opt[OPT_CONTRAST].cap           &= ~SANE_CAP_INACTIVE;
            scanner->opt[OPT_BRIGHTNESS].cap         &= ~SANE_CAP_INACTIVE;

	    if (scanner->device->us.inquiry_highlight)
            {scanner->opt[OPT_HIGHLIGHT].cap          &= ~SANE_CAP_INACTIVE;}

	    if (scanner->device->us.inquiry_shadow)
            {scanner->opt[OPT_SHADOW].cap             &= ~SANE_CAP_INACTIVE;}

            scanner->opt[OPT_HALFTONE_DIMENSION].cap &= ~SANE_CAP_INACTIVE;

            if (scanner->val[OPT_HALFTONE_DIMENSION].w)
            { scanner->opt[OPT_HALFTONE_PATTERN].cap &= ~SANE_CAP_INACTIVE; }
          }
	  else
	  { /* lineart modes */
            scanner->opt[OPT_THRESHOLD].cap  &= ~SANE_CAP_INACTIVE;
	  }
        }
        else
        { /* multi-bit modes(grey or color) */
          if ( (scanner->device->us.inquiry_GOB & 4) != 0)
          { scanner->opt[OPT_TEN_BIT_MODE].cap  &= ~SANE_CAP_INACTIVE; }

          if (scanner->device->us.inquiry_highlight)
          { scanner->opt[OPT_HIGHLIGHT].cap    &= ~SANE_CAP_INACTIVE; }

          if (scanner->device->us.inquiry_shadow)
          { scanner->opt[OPT_SHADOW].cap       &= ~SANE_CAP_INACTIVE; }

          if (scanner->device->us.inquiry_reverse_multi)
          { scanner->opt[OPT_NEGATIVE].cap     &= ~SANE_CAP_INACTIVE; }

          if (scanner->device->us.inquiry_gamma_dwload)
          { scanner->opt[OPT_CUSTOM_GAMMA].cap &= ~SANE_CAP_INACTIVE; }

          if (scanner->device->us.inquiry_analog_gamma)
          { scanner->opt[OPT_ANALOG_GAMMA].cap &= ~SANE_CAP_INACTIVE; }

          if (strcmp(val, "Color") == 0)
          {
            if (scanner->device->us.inquiry_analog_gamma)
            { scanner->opt[OPT_ANALOG_GAMMA_BIND].cap &= ~SANE_CAP_INACTIVE; }

            if (scanner->device->us.inquiry_highlight)
            { scanner->opt[OPT_HIGHLIGHT_BIND].cap &= ~SANE_CAP_INACTIVE; }

            if (scanner->device->us.inquiry_shadow)
            { scanner->opt[OPT_SHADOW_BIND].cap &= ~SANE_CAP_INACTIVE; }
          }
	}

        if (scanner->val[OPT_CUSTOM_GAMMA].w)
        {
          if (strcmp(val, "Grey") == 0)
          {
            if (scanner->device->us.inquiry_gamma_dwload)
            { scanner->opt[OPT_GAMMA_VECTOR].cap &= ~SANE_CAP_INACTIVE; }
          }
          else if (strcmp(val, "Color") == 0)
          {
            if (scanner->device->us.inquiry_gamma_dwload)
            {
	      scanner->opt[OPT_GAMMA_VECTOR].cap   &= ~SANE_CAP_INACTIVE;
              scanner->opt[OPT_GAMMA_VECTOR_R].cap &= ~SANE_CAP_INACTIVE;
              scanner->opt[OPT_GAMMA_VECTOR_G].cap &= ~SANE_CAP_INACTIVE;
              scanner->opt[OPT_GAMMA_VECTOR_B].cap &= ~SANE_CAP_INACTIVE;
	    }
          }
	}
       return SANE_STATUS_GOOD;
      }

      case OPT_HALFTONE_DIMENSION:
      /* halftone pattern dimension affects halftone pattern option: */
      {
        unsigned dim = *(SANE_Word *) val;

         scanner->val[option].w = dim;

         if (info) { *info |= SANE_INFO_RELOAD_OPTIONS; }

         scanner->opt[OPT_HALFTONE_PATTERN].cap |= SANE_CAP_INACTIVE;

         if (dim > 0)
         {
           scanner->opt[OPT_HALFTONE_PATTERN].cap &= ~SANE_CAP_INACTIVE;
           scanner->opt[OPT_HALFTONE_PATTERN].size = dim * sizeof (SANE_Word);
         }
        return SANE_STATUS_GOOD;
       }
    }
  } /* else */
return SANE_STATUS_INVAL;
}


/* ------------------------ SANE GET PARAMETERS --------------------------- */


SANE_Status sane_get_parameters(SANE_Handle handle, SANE_Parameters *params)
{
 Umax_Scanner *scanner = handle;
 const char *mode;

  DBG(10,"sane_get_parameters\n");

  if (!scanner->scanning)
  { /* not scanning, so lets use recent values */
    double width, height, x_dpi, y_dpi;

    memset(&scanner->params, 0, sizeof (scanner->params));

    width  = SANE_UNFIX(scanner->val[OPT_BR_X].w - scanner->val[OPT_TL_X].w);
    height = SANE_UNFIX(scanner->val[OPT_BR_Y].w - scanner->val[OPT_TL_Y].w);
    x_dpi  = SANE_UNFIX(scanner->val[OPT_X_RESOLUTION].w);
    y_dpi  = SANE_UNFIX(scanner->val[OPT_Y_RESOLUTION].w);
    if ( (scanner->val[OPT_RESOLUTION_BIND].w == SANE_TRUE) ||
         (scanner->val[OPT_PREVIEW].w == SANE_TRUE) )
    { y_dpi = x_dpi; }

    if (x_dpi > 0.0 && y_dpi > 0.0 && width > 0.0 && height > 0.0)
    {
      double x_dots_per_mm = x_dpi / MM_PER_INCH;
      double y_dots_per_mm = y_dpi / MM_PER_INCH;

      scanner->params.pixels_per_line = width *  x_dots_per_mm;
      scanner->params.lines           = height * y_dots_per_mm;
    }
  }

  mode = scanner->val[OPT_MODE].s;

  if (strcmp(mode, "Lineart") == 0 || strcmp(mode, "Halftone") == 0)
  {
    scanner->params.format = SANE_FRAME_GRAY;
    scanner->params.bytes_per_line = (scanner->params.pixels_per_line + 7) / 8;
    scanner->params.depth = 1;
  }
  else if (strcmp(mode, "Grey") == 0)
  {
    scanner->params.format = SANE_FRAME_GRAY;
    scanner->params.bytes_per_line =
      scanner->params.pixels_per_line * scanner->output_bytes;
    scanner->params.depth = 8 * scanner->output_bytes;
  }
  else if (strcmp(mode, "Color Lineart") == 0 ||
           strcmp(mode, "Color Halftone") == 0 )
  {
    if (scanner->device->us.inquiry_one_pass_color)
    {
      scanner->device->us.three_pass = 0;
      scanner->params.format = SANE_FRAME_RGB;
      scanner->params.bytes_per_line = 3 * scanner->params.pixels_per_line;
      scanner->params.depth = 8;
    }
    else /* three pass color */
    {
      scanner->device->us.three_pass = 1;
      scanner->params.format =
         SANE_FRAME_RED + scanner->device->us.three_pass_color - 1;
      scanner->params.bytes_per_line = scanner->params.pixels_per_line;
      scanner->params.depth = 8;
    }
  }
  else /* RGB */
  {
    if (scanner->device->us.inquiry_one_pass_color)
    {
      scanner->device->us.three_pass = 0;
      scanner->params.format = SANE_FRAME_RGB;
      scanner->params.bytes_per_line =
        3 * scanner->params.pixels_per_line * scanner->output_bytes;
      scanner->params.depth = 8 * scanner->output_bytes;
    }
    else /* three pass color */
    {
      scanner->device->us.three_pass = 1;
      scanner->params.format =
         SANE_FRAME_RED + scanner->device->us.three_pass_color - 1;
      scanner->params.bytes_per_line =
        scanner->params.pixels_per_line * scanner->output_bytes;
      scanner->params.depth = 8 * scanner->output_bytes;
    }
  }

  scanner->params.last_frame = (scanner->params.format != SANE_FRAME_RED
			  && scanner->params.format != SANE_FRAME_GREEN);

  if (params) { *params = scanner->params; }

return SANE_STATUS_GOOD;
}


/* ----------------------------- SANE START ------------------------------- */


SANE_Status sane_start(SANE_Handle handle)
{
 Umax_Scanner *scanner = handle;
 int fds[2];
 const char *mode;
 double x_dots_per_mm;
 double y_dots_per_mm;
 double basedots;
 const char *scan_source;

  DBG(10,"sane_start\n");

  mode = scanner->val[OPT_MODE].s;

  if (scanner->device->us.sfd < 0)               /* first call for this scan */
  {
    if ( sanei_scsi_open(scanner->device->sane.name,
         &(scanner->device->us.sfd), sense_handler) != SANE_STATUS_GOOD)
    {
       DBG(1, "sane_start: open of %s failed:\n",
               scanner->device->sane.name);
       return SANE_STATUS_INVAL;
    }

    scanner->device->us.three_pass_color = 1;

    /* test for adf and uta */
    scan_source = scanner->val[OPT_SOURCE].s;

    if (strcmp(scan_source, "Transparency Adapter") == 0)
    {
      if ( (scanner->device->us.inquiry_uta != 0) &&
           (scanner->device->us.inquiry_transavail != 0) )
      { scanner->device->us.uta=1; }
      else
      {
        DBG(1,"ERROR: Transparency Adapter not available\n");
        sanei_scsi_close(scanner->device->us.sfd);
        scanner->device->us.sfd=-1;
       return SANE_STATUS_INVAL;
      }
    }
    else
    {
      scanner->device->us.uta=0;

      if (strcmp(scan_source, "Automatic Document Feeder") == 0)
      {
        if ( (scanner->device->us.inquiry_adf) &&
             (scanner->device->us.inquiry_adfmode) )
        { scanner->device->us.adf=1; }
        else
        {
          DBG(1,"ERROR: Automatic Document Feeder not available\n");
         sanei_scsi_close(scanner->device->us.sfd);
         scanner->device->us.sfd=-1;
         return SANE_STATUS_INVAL;
        }
      }
      else
      { scanner->device->us.adf=0; }
    }

    /* grab scanner */
    if (umax_grab_scanner(&(scanner->device->us)))
    {
      sanei_scsi_close(scanner->device->us.sfd);
      scanner->device->us.sfd=-1;
      DBG(2,"WARNING: unable to reserve scanner: device busy\n");
     return SANE_STATUS_DEVICE_BUSY;
    }

    scanner->scanning = SANE_TRUE;

    umax_initialize_values(&(scanner->device->us));

    if (scanner->val[OPT_TEN_BIT_MODE].w == SANE_TRUE)        /* 10 bit mode */
    {
      scanner->device->us.bits_per_pixel = 4;
      scanner->device->us.max_value      = 1023;
      DBG(20,"Using 10 bit output mode\n");
    }

    scanner->device->us.reverse =
    scanner->device->us.reverse_multi = scanner->val[OPT_NEGATIVE].w;

    scanner->device->us.threshold =P_100_TO_255(scanner->val[OPT_THRESHOLD].w);
    scanner->device->us.brightness=P_200_TO_255(scanner->val[OPT_BRIGHTNESS].w);
    scanner->device->us.contrast  =P_200_TO_255(scanner->val[OPT_CONTRAST].w);

    scanner->device->us.quality     = scanner->val[OPT_QUALITY].w;
    scanner->device->us.preview     = scanner->val[OPT_PREVIEW].w;
    scanner->device->us.warmup      = scanner->val[OPT_WARMUP].w;
    scanner->device->us.slow        = scanner->val[OPT_SLOW].w;
    scanner->device->us.smear       = scanner->val[OPT_SMEAR].w;

    scanner->device->us.RGB_PREVIEW_PATCH = scanner->val[OPT_RGB_PREVIEW_PATCH].w;
    scanner->device->us.START_SCAN_PATCH  = scanner->val[OPT_START_SCAN_PATCH].w;

    scanner->device->us.analog_gamma_r = scanner->device->us.analog_gamma_g =
    scanner->device->us.analog_gamma_b =
      umax_calculate_analog_gamma(SANE_UNFIX(scanner->val[OPT_ANALOG_GAMMA].w));

    scanner->device->us.highlight_r = scanner->device->us.highlight_g = 
    scanner->device->us.highlight_b =
      P_100_TO_255(scanner->val[OPT_HIGHLIGHT].w);

    scanner->device->us.shadow_r = scanner->device->us.shadow_g = 
    scanner->device->us.shadow_b = 
      P_100_TO_255(scanner->val[OPT_SHADOW].w);

    if (strcmp(mode, "Lineart") == 0)
    { scanner->device->us.colormode = LINEART; }
    else if (strcmp(mode, "Halftone") == 0)
    { scanner->device->us.colormode = HALFTONE; }
    else if (strcmp(mode, "Grey") == 0)
    { scanner->device->us.colormode = GREYSCALE; }
    else if (strcmp(mode, "Color Lineart") == 0)
    { scanner->device->us.colormode = RGB_LINEART; }
    else if (strcmp(mode, "Color Halftone") == 0)
    { scanner->device->us.colormode = RGB_HALFTONE; }
    else if (strcmp(mode, "Color") == 0)
    {
      scanner->device->us.colormode = RGB;
      if (scanner->val[OPT_ANALOG_GAMMA_BIND].w == SANE_FALSE)
      {
        scanner->device->us.analog_gamma_r = umax_calculate_analog_gamma(
                         SANE_UNFIX(scanner->val[OPT_ANALOG_GAMMA_R].w) );
        scanner->device->us.analog_gamma_g = umax_calculate_analog_gamma(
                         SANE_UNFIX(scanner->val[OPT_ANALOG_GAMMA_G].w) );
        scanner->device->us.analog_gamma_b = umax_calculate_analog_gamma(
                         SANE_UNFIX(scanner->val[OPT_ANALOG_GAMMA_B].w) );
      }

      if (scanner->val[OPT_HIGHLIGHT_BIND].w == SANE_FALSE)
      {
        scanner->device->us.highlight_r =
                 P_100_TO_255(scanner->val[OPT_HIGHLIGHT_R].w);
        scanner->device->us.highlight_g = 
                 P_100_TO_255(scanner->val[OPT_HIGHLIGHT_G].w);
        scanner->device->us.highlight_b = 
                 P_100_TO_255(scanner->val[OPT_HIGHLIGHT_B].w);
      }

      if (scanner->val[OPT_SHADOW_BIND].w == SANE_FALSE)
      {
        scanner->device->us.shadow_r = 
                 P_100_TO_255(scanner->val[OPT_SHADOW_R].w);
        scanner->device->us.shadow_g = 
                 P_100_TO_255(scanner->val[OPT_SHADOW_G].w);
        scanner->device->us.shadow_b = 
                 P_100_TO_255(scanner->val[OPT_SHADOW_B].w);
      }
    }

    if (strcmp(scanner->val[OPT_CALIB_MODE].s,"Calibration ignored") == 0)
    { scanner->device->us.calibration = 0; }
    else if (strcmp(scanner->val[OPT_CALIB_MODE].s,"Calibration by image-type") == 0)
    { scanner->device->us.calibration = 1; }
    else if (strcmp(scanner->val[OPT_CALIB_MODE].s,"Calibration driver-selected") == 0)
    { scanner->device->us.calibration = 2; }

    /* get and set geometric values for scanning */
    scanner->device->us.x_resolution = 
             SANE_UNFIX(scanner->val[OPT_X_RESOLUTION].w);
    scanner->device->us.y_resolution =
             SANE_UNFIX(scanner->val[OPT_Y_RESOLUTION].w);

    if ( (scanner->val[OPT_RESOLUTION_BIND].w == SANE_TRUE) ||
         (scanner->val[OPT_PREVIEW].w == SANE_TRUE) )
    { scanner->device->us.y_resolution = scanner->device->us.x_resolution; }

    x_dots_per_mm = scanner->device->us.x_resolution / MM_PER_INCH;
    y_dots_per_mm = scanner->device->us.y_resolution / MM_PER_INCH;

    basedots = 1200 / MM_PER_INCH;

    scanner->device->us.upper_left_x =
      SANE_UNFIX(scanner->val[OPT_TL_X].w) * basedots;
    scanner->device->us.upper_left_y =
      SANE_UNFIX(scanner->val[OPT_TL_Y].w) * basedots;

    scanner->device->us.width_in_pixels  = ( x_dots_per_mm *
       SANE_UNFIX(scanner->val[OPT_BR_X].w - scanner->val[OPT_TL_X].w) );
    scanner->device->us.height_in_pixels = ( y_dots_per_mm *
       SANE_UNFIX(scanner->val[OPT_BR_Y].w - scanner->val[OPT_TL_Y].w) );

    umax_bound_scanarea(&(scanner->device->us));

    if (umax_check_values(&(scanner->device->us)) != 0)
    {
      DBG(1,"ERROR: invalid scan-values\n");
      scanner->scanning = SANE_FALSE;
      umax_give_scanner(&(scanner->device->us));
      sanei_scsi_close(scanner->device->us.sfd);
      scanner->device->us.sfd=-1;
     return SANE_STATUS_INVAL;
    }

    scanner->params.bytes_per_line  = scanner->device->us.row_len;
    scanner->params.pixels_per_line = scanner->device->us.width_in_pixels; 
    scanner->params.lines           = scanner->device->us.height_in_pixels;

    sane_get_parameters(scanner, 0);

    DBG(20,"x_resolution (dpi)      = %d\n",
           scanner->device->us.x_resolution);
    DBG(20,"y_resolution (dpi)      = %d\n",
           scanner->device->us.y_resolution);
    DBG(20,"upper_left_x (1200 dpi) = %d\n",scanner->device->us.upper_left_x);
    DBG(20,"upper_left_y (1200 dpi) = %d\n",scanner->device->us.upper_left_y);
    DBG(20,"width in pixels         = %d\n",
           scanner->device->us.width_in_pixels);
    DBG(20,"height in pixels        = %d\n",
           scanner->device->us.height_in_pixels);

    DBG(20,"bytes per line          = %d\n", scanner->params.bytes_per_line);
    DBG(20,"pixels_per_line         = %d\n", scanner->params.pixels_per_line);
    DBG(20,"lines                   = %d\n", scanner->params.lines);

    DBG(20,"analog_gamma            = %d %d %d\n",
            scanner->device->us.analog_gamma_r,
            scanner->device->us.analog_gamma_g,
            scanner->device->us.analog_gamma_b);

    DBG(20,"negative                = %d\n", scanner->device->us.reverse);

    DBG(20,"threshold (lineart)     = %d\n", scanner->device->us.threshold);
    DBG(20,"brightness (halftone)   = %d\n", scanner->device->us.brightness);
    DBG(20,"contrast   (halftone)   = %d\n", scanner->device->us.contrast);
    DBG(20,"highlight               = %d %d %d\n",
            scanner->device->us.highlight_r,
            scanner->device->us.highlight_g,
            scanner->device->us.highlight_b);
    DBG(20,"shadow                  = %d %d %d\n",
            scanner->device->us.shadow_r,
            scanner->device->us.shadow_g,
            scanner->device->us.shadow_b);

    DBG(20,"calibration             = %s\n", scanner->val[OPT_CALIB_MODE].s);
    DBG(20,"quality calibration     = %d\n", scanner->device->us.quality);
    DBG(20,"fast preview function   = %d\n", scanner->device->us.preview);
    DBG(20,"slow scan speed         = %d\n", scanner->device->us.slow);
    DBG(20,"smear                   = %d\n", scanner->device->us.smear);

    /* send halftonepattern */
    if ( (strcmp(mode, "Halftone") == 0) ||
         (strcmp(mode, "Color Halftone") == 0) )
    {
      umax_send_halftone_pattern(&(scanner->device->us), 
                                 (char *) &(scanner->halftone_pattern[0]),
                                 scanner->val[OPT_HALFTONE_DIMENSION].w );
      scanner->device->us.halftone = WD_halftone_download;
    } /* end of send halftonepattern */

 
  } /* ------------ end of first call -------------- */


  /* send gammacurves */
  if (scanner->val[OPT_CUSTOM_GAMMA].w == SANE_TRUE)
  {
    if (strcmp(mode, "Color") == 0)
    {
      if (scanner->device->us.three_pass == 0)        /* one pass color scan */
      {
       unsigned int i, dest, color, value;
       char gamma[3 * scanner->gamma_length * scanner->output_bytes];

        dest=0;
        for(color=1;color<=3;color++)
        {
          for(i=0; i < scanner->gamma_length; i++)
          {
            value = scanner->gamma_table[color][i];
            if (scanner->output_bytes == 2)
            { gamma[dest++] = scanner->gamma_table[0][value] / 256; }
            gamma[dest++] = (scanner->gamma_table[0][value] & 255);
          }
        }
        DBG(20,"sending 3 * %d bytes of gamma data\n",
                scanner->gamma_length * scanner->output_bytes);
        umax_send_gamma_data(&(scanner->device->us), &gamma[0], 3);
        scanner->device->us.digital_gamma_r =
        scanner->device->us.digital_gamma_g =
        scanner->device->us.digital_gamma_b = WD_gamma_download;
      }
      else                                          /* three pass color scan */
      {
       unsigned int i, dest, color, value;
       char gamma[scanner->gamma_length * scanner->output_bytes];

        dest  = 0;
        color = scanner->device->us.three_pass_color;
        for(i=0; i < scanner->gamma_length; i++)
        {
          value = scanner->gamma_table[color][i];
          if (scanner->output_bytes == 2)
          { gamma[dest++] = scanner->gamma_table[0][value] / 256; }
          gamma[dest++] = (scanner->gamma_table[0][value] & 255);
        }
        DBG(20,"sending %d bytes of gamma data\n",
                scanner->gamma_length * scanner->output_bytes);
        umax_send_gamma_data(&(scanner->device->us), &gamma[0], 1);
        scanner->device->us.digital_gamma_r =
        scanner->device->us.digital_gamma_g =
        scanner->device->us.digital_gamma_b = WD_gamma_download;
      }
    }
    else                                                   /* greyscale scan */
    {
     int i, dest, value;
     char gamma[scanner->gamma_length * scanner->output_bytes];

      dest=0;
      for(i=0; i < scanner->gamma_length; i++)
      {
          value = scanner->gamma_table[0][i];
          if (scanner->output_bytes == 2)
          { gamma[dest++] = scanner->gamma_table[0][value] / 256; }
          gamma[dest++] = (scanner->gamma_table[0][value] & 255);
      }
      DBG(20,"sending %d bytes of gamma data\n",
              scanner->gamma_length * scanner->output_bytes);
      umax_send_gamma_data(&(scanner->device->us), &gamma[0], 1);
      scanner->device->us.digital_gamma_r = WD_gamma_download;
    }
  } /* end of send gammacurves */


  umax_reposition_scanner(&(scanner->device->us));
  umax_set_window_param(&(scanner->device->us));
  umax_start_scan(&(scanner->device->us));
  umax_wait_scanner(&(scanner->device->us));

  /* create a pipe, fds[0]=read-fd, fds[1]=write-fd */
  if (pipe(fds) < 0)
  {
    DBG(1,"ERROR: could not create pipe\n");
    scanner->scanning = SANE_FALSE;
    umax_give_scanner(&(scanner->device->us));
    sanei_scsi_close(scanner->device->us.sfd);
    scanner->device->us.sfd=-1;
   return SANE_STATUS_IO_ERROR;
  }

  scanner->reader_pid = fork();
  if (scanner->reader_pid == 0)
  { /* reader_pid = 0 ===> child process */
    sigset_t ignore_set;
    struct SIGACTION act;

    close(fds[0]);

    sigfillset(&ignore_set);
    sigdelset(&ignore_set, SIGTERM);
    sigprocmask(SIG_SETMASK, &ignore_set, 0);

    memset(&act, 0, sizeof (act));
    sigaction (SIGTERM, &act, 0);

    /* don't use exit() since that would run the atexit() handlers... */
    _exit(reader_process(scanner, fds[1]));
  }
  close(fds[1]);
  scanner->pipe = fds[0];

 return SANE_STATUS_GOOD;
}


/* ---------------------------------- SANE READ ---------------------------- */


SANE_Status sane_read(SANE_Handle handle, SANE_Byte *buf, SANE_Int max_len, SANE_Int *len)
{
 Umax_Scanner *scanner = handle;
 ssize_t nread;

  *len = 0;

  nread = read(scanner->pipe, buf, max_len);
  DBG(20, "sane_read: read %ld bytes\n", (long) nread);

  if (!(scanner->scanning)) { return do_cancel(scanner); }

  if (nread < 0)
  {
    if (errno == EAGAIN) { return SANE_STATUS_GOOD; }
    else
    {
      do_cancel(scanner); 
     return SANE_STATUS_IO_ERROR;
    }
  }

  *len = nread;

  if (nread == 0)
  {
    if ( (scanner->device->us.three_pass == 0) ||
         (scanner->device->us.colormode<=RGB_LINEART) ||
         (++(scanner->device->us.three_pass_color) > 3) )
    { do_cancel(scanner); }

    return do_eof(scanner);                                   /* close pipe */
  }

return SANE_STATUS_GOOD;
}


/* ----------------------------- SANE CANCEL ------------------------------- */


void sane_cancel(SANE_Handle handle)
{
 Umax_Scanner *scanner = handle;

  DBG(10,"sane_cancel\n");

  if (scanner->reader_pid > 0) { kill(scanner->reader_pid, SIGTERM); }
  scanner->scanning = SANE_FALSE;
}


/* -------------------------- SANE SET IO MODE ----------------------------- */


SANE_Status sane_set_io_mode(SANE_Handle handle, SANE_Bool non_blocking)
{
 Umax_Scanner *scanner = handle;

  DBG(10,"sane_set_io_mode: non_blocking=%d\n",non_blocking);

  if (!scanner->scanning) { return SANE_STATUS_INVAL; }

  if (fcntl (scanner->pipe, F_SETFL, non_blocking ? O_NONBLOCK : 0) < 0)
  { return SANE_STATUS_IO_ERROR; }

return SANE_STATUS_GOOD;
}


/* ------------------------- SANE GET SELECT FD ---------------------------- */


SANE_Status sane_get_select_fd(SANE_Handle handle, SANE_Int *fd)
{
 Umax_Scanner *scanner = handle;

  DBG(10,"sane_get_select_fd\n");

  if (!scanner->scanning) { return SANE_STATUS_INVAL; }
  *fd = scanner->pipe;

return SANE_STATUS_GOOD;
}
