/****************************************************************************
*                   isoblob.c
*
*  This module implements functions that manipulate isoblobs.
*
*  By Lummox JR, July 1999
*
*  Almost all of the functions in this module were based on blob
*  functions by Dieter Bayer and Alexander Enzmann, and some on
*  isosurface functions created by R. Suzuki, D. Skarda, and T. Bily.
*  Most have been modified considerably from their original versions.
*
*****************************************************************************/

/****************************************************************************
*
*  Explanation:
*
*    -
*
*  Syntax:
*
*    isoblob
*    {
*      threshold THRESHOLD_VALUE
*      accuracy ACCURACY_VALUE
*      max_trace MAX_TRACE_VALUE
*
*      function { ... }    // density function 1
*      function { ... }    // density function 2
*      ...
*
*      sphere
*        {
*        <CENTER>, RADIUS, [strength] STRENGTH
*        [function] FUNCTION_NUMBER
*        [transformations, textures, etc.]
*        }
*
*      cylinder
*        {
*        <END1>, <END2>, RADIUS, [strength] STRENGTH
*        [function] FUNCTION_NUMBER
*        [transformations, textures, etc.]
*        }
*
*      [ hierarchy FLAG ]
*    }
*
*  Notes:
*
*  In spheres, density function is centered on <x,y,z>=<0,0,0>.
*  In cylinders, density function goes from <0,0,0> to <0,0,len>.
*  <r,s,t> always represents the coordinates in blob space.
*
*  Most density functions should use max(...,0) to avoid negative
*  densities, but the option is available. A normal density function
*  should range from 0 to 1.
*
*  ---
*
*  Jul 1999 : Creation.
*
*****************************************************************************/

#include "frame.h"
#include "povray.h"
#include "vector.h"
#include "povproto.h"
#include "isoblob.h"
#include "bbox.h"
#include "bsphere.h"
#include "lighting.h"
#include "matrices.h"
#include "objects.h"
#include "texture.h"

#ifdef IsoBlobPatch

#ifndef ISOSURFACE_OBJECT
#include "isosrf.h"
#endif
#include "f_expr.h"


/*****************************************************************************
* Local preprocessor defines
******************************************************************************/

/* Minimal intersection depth for a valid intersection. */

#define DEPTH_TOLERANCE 1.0e-2

/* Tolerance for inside test. */

#define INSIDE_TOLERANCE 1.0e-6

/* Ray enters/exits a component. */

#define ENTERING 0
#define EXITING  1



/*****************************************************************************
* Local typedefs
******************************************************************************/



/*****************************************************************************
* Static functions
******************************************************************************/

static void element_normal (VECTOR Result, VECTOR P, ISOBLOB_ELEMENT *Element, DBL accuracy, char normal_type);
static int intersect_element (VECTOR P, VECTOR D, ISOBLOB_ELEMENT *Element, DBL mindist, DBL *t0, DBL *t1, VECTOR v0, VECTOR v1);
static void insert_hit (ISOBLOB_ELEMENT *Element, DBL t0, DBL t1, VECTOR v0, VECTOR v1, ISOBLOB_INTERVAL *intervals, int *cnt);
static int determine_influences (VECTOR P, VECTOR D, ISOBLOB *Isoblob, DBL mindist, ISOBLOB_INTERVAL *intervals);
static DBL calculate_field_value (ISOBLOB *Isoblob, VECTOR P);
static DBL calculate_element_field (ISOBLOB_ELEMENT *Element, VECTOR P);

static int intersect_sphere (ISOBLOB_ELEMENT *Element, VECTOR P, VECTOR D, DBL mindist, DBL *tmin, DBL *tmax, VECTOR v0, VECTOR v1);
static int intersect_cylinder (ISOBLOB_ELEMENT *Element, VECTOR P, VECTOR D, DBL mindist, DBL *tmin, DBL *tmax, VECTOR v0, VECTOR v1);

static void get_element_bounding_sphere (ISOBLOB_ELEMENT *Element, VECTOR Center, DBL *Radius2);
static void build_bounding_hierarchy (ISOBLOB *Isoblob);

static void init_isoblob_element (ISOBLOB_ELEMENT *Element);
static void determine_element_texture (ISOBLOB *Isoblob,
  ISOBLOB_ELEMENT *Element, TEXTURE *Texture, VECTOR P, int *Count,
  TEXTURE **Textures, DBL *Weights);

static void insert_node (BSPHERE_TREE *Node, int *size);

static int  All_Isoblob_Intersections (OBJECT *Object, RAY *Ray, ISTACK *Depth_Stack);
static int  Inside_Isoblob (VECTOR point, OBJECT *Object);
static void Isoblob_Normal (VECTOR Result, OBJECT *Object, INTERSECTION *Inter);
static void Isoblob_UVCoord (UV_VECT Result, OBJECT *Object, INTERSECTION *Inter);
static ISOBLOB *Copy_Isoblob (OBJECT *Object);
static void Translate_Isoblob (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Rotate_Isoblob (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Scale_Isoblob (OBJECT *Object, VECTOR Vector, TRANSFORM *Trans);
static void Invert_Isoblob (OBJECT *Object);
static void Transform_Isoblob (OBJECT *Object, TRANSFORM *Trans);
static void Destroy_Isoblob (OBJECT *Object);
static void Compute_Isoblob_BBox (ISOBLOB *Isoblob);

/*****************************************************************************
* Local variables
******************************************************************************/

METHODS Isoblob_Methods =
{
  All_Isoblob_Intersections,
  Inside_Isoblob, Isoblob_Normal, Default_UVCoord,
  (COPY_METHOD)Copy_Isoblob,
  Translate_Isoblob, Rotate_Isoblob, Scale_Isoblob, Transform_Isoblob,
  Invert_Isoblob, Destroy_Isoblob
};

static BSPHERE_TREE **IsoblobQueue;

/* Maximum number of entries in queue during hierarchy traversal. */

static unsigned Max_IsoblobQueue_Size = 1024;



/*****************************************************************************
*
* FUNCTION
*
*   All_Isoblob_Intersections
*
* INPUT
*
*   Object      - Object
*   Ray         - Ray
*
* OUTPUT
*
*   Depth_Stack - Intersection stack
*
* RETURNS
*
*   int - TRUE, if a intersection was found
*   
* DESCRIPTION
*
*   Generate intervals of influence for each component. After these
*   are made, determine their aggregate effect on the ray. Add the
*   functions together for each component influencing the ray over
*   that interval, and look for any intersections.
*
*   If the total density function suddenly jumps across the
*   threshold at an interval boundary, an intersection occurs at
*   that point, with the bounding shape's normal as the intersection
*   normal.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static int All_Isoblob_Intersections(OBJECT *Object, RAY *Ray, ISTACK *Depth_Stack)
{
  register int j,k;
  int i, inf, cnt;
  int in_flag;
  int Intersection_Found = FALSE;
  DBL dist,start_dist;
  DBL Int_Begin[50], Int_End[50], Int_Length[50], lowvalue, hivalue, tmp=0;
  DBL lastint=0;
  register DBL t1,t2,l;
  VECTOR P, D, Ibegin, Iend, IbeginIsoblobspace, IendIsoblobspace;
  VECTOR IPoint;
  ISOBLOB_INTERVAL *intervals;
  ISOBLOB_ELEMENT *Element;
  ISOBLOB *Isoblob = (ISOBLOB *)Object;
  ISOBLOB_ELEMENT *firstelement,*thiselement;

  Increase_Counter(stats[Ray_Isoblob_Tests]);

  /* Transform the ray into isoblob space. */

  if (Isoblob->Trans != NULL)
  {
    MInvTransPoint(P, Ray->Initial, Isoblob->Trans);
    MInvTransDirection(D, Ray->Direction, Isoblob->Trans);
  }
  else
  {
    Assign_Vector(P, Ray->Initial);
    Assign_Vector(D, Ray->Direction);
  }

  /* Get the intervals along the ray where each component has an effect. */

  intervals = Isoblob->Data->Intervals;

  if ((cnt = determine_influences(P, D, Isoblob, DEPTH_TOLERANCE, intervals)) == 0)
  {
    /* Ray doesn't hit any of the component elements. */

    return (FALSE);
  }

  /* To avoid numerical problems we start at the first interval. */

  if ((start_dist = intervals[0].bound) < Small_Tolerance)
  {
    start_dist = 0.0;
  }

  else for (i = 0; i < cnt; i++)
  {
    intervals[i].bound -= start_dist;
  }

  /* Get the new starting point. */

  VAddScaledEq(P, start_dist, D);

  /* Clear out the linked list of elements. */

  firstelement=NULL;
  /* for(i=0;i<cnt;i++) intervals[i].Element->nextelement=NULL; */

  /*
   * Step through the list of intersection points, adding the
   * influence of each component as it appears. 
   */

  for (i = in_flag = 0; i < cnt; i++)
  {
    Element = intervals[i].Element;
    if ((intervals[i].type & 1) == ENTERING)
    {
      /*
       * Something is just starting to influence the ray.
       */

      /*
       * Calculate density of all shapes in the old interval.
       */
      VEvaluateRay(Iend,P,intervals[i].bound,D);
      func_R=Iend[X];
      func_S=Iend[Y];
      func_T=Iend[Z];
      for(lastint=-Isoblob->Data->Threshold,thiselement=firstelement;thiselement!=NULL;thiselement=thiselement->nextelement)
        {
        VEvaluateRay(func_XYZ,thiselement->PP,intervals[i].bound,thiselement->DD);
        evaluate_function(thiselement->Func);
        lastint+=(*calc_stack_top)*(thiselement->str);
        }
      in_flag++;

      /* Add element to queue */

      Element->nextelement=firstelement;
      firstelement=Element;

      /* use PP and DD to solve for the intersection */
      MInvTransPoint(Element->PP, P, Element->Trans);
      MInvTransDirection(Element->DD, D, Element->Trans);

      if(intervals[i].bound+start_dist>=Isoblob->Data->accuracy)
        {
        /*
         * Calculate density of new shape and add it to the total.
         */
        VEvaluateRay(func_XYZ,Element->PP,intervals[i].bound,Element->DD);
        evaluate_function(Element->Func);
        tmp=lastint+(*calc_stack_top)*(Element->str);
        /*
         * If density crosses the threshold at this boundary, then
         * there is an intersection here with a normal matching the
         * new bounding shape.
         */
        if(tmp*lastint<=0.0) {
          dist=intervals[i].bound+start_dist;
          VEvaluateRay (IPoint, P, dist, D);
          if (Point_In_Clip (IPoint, Object->Clip))
            {
            VEvaluateRay(IPoint, Ray->Initial, dist, Ray->Direction);
            push_normal_entry(dist,IPoint,intervals[i].normal,Object,Depth_Stack);
            Intersection_Found=TRUE;
            }
          }
        }
    }
    else
    {
      /* 
       * We are losing the influence of a component
       */

      if(intervals[i].bound+start_dist>=Isoblob->Data->accuracy)
        {
        /*
         * Calculate density of all shapes in the old interval.
         */
        VEvaluateRay(Iend,P,intervals[i].bound,D);
        func_R=Iend[X];
        func_S=Iend[Y];
        func_T=Iend[Z];
        for(lastint=-Isoblob->Data->Threshold,thiselement=firstelement;thiselement!=NULL;thiselement=thiselement->nextelement)
          {
          VEvaluateRay(func_XYZ,thiselement->PP,intervals[i].bound,thiselement->DD);
          evaluate_function(thiselement->Func);
          lastint+=(*calc_stack_top)*(thiselement->str);
          }
        /*
         * Calculate density of old shape and subtract it from the total.
         */
        VEvaluateRay(func_XYZ,Element->PP,intervals[i].bound,Element->DD);
        evaluate_function(Element->Func);
        tmp=lastint-(*calc_stack_top)*(Element->str);
        }

      /* Remove element from queue */

      if(firstelement==Element) {firstelement=Element->nextelement; Element->nextelement=NULL;}
      else if(firstelement!=NULL)
        {
        for(thiselement=firstelement;(thiselement->nextelement!=Element && thiselement->nextelement!=NULL);thiselement=thiselement->nextelement);
        if(thiselement->nextelement==Element) {thiselement->nextelement=Element->nextelement; Element->nextelement=NULL;}
        }

      /*
       * If density crosses the threshold at this boundary, then
       * there is an intersection here with a normal matching the
       * old bounding shape.
       */
      if(intervals[i].bound+start_dist>Isoblob->Data->accuracy && tmp*lastint<=0.0)
        {
        dist=intervals[i].bound+start_dist;
        VEvaluateRay (IPoint, P, dist, D);
        if (Point_In_Clip (IPoint, Object->Clip))
          {
          VEvaluateRay(IPoint, Ray->Initial, dist, Ray->Direction);
          push_normal_entry(dist,IPoint,intervals[i].normal,Object,Depth_Stack);
          Intersection_Found=TRUE;
          }
        }

      /* If no components are currently affecting the ray ---> skip ahead. */

      if (--in_flag == 0)
      {
        continue;
      }
    }

    /*
     *  Prepare to find an intersection within the interval.
     */

    Int_Begin[0]=intervals[i].bound;
    /* intervals[i+1] is always valid because in_flag>0 to reach this code */
    Int_End[0]=intervals[i+1].bound;
	Int_Length[0]=Int_End[0]-Int_Begin[0];
    j=0;
    inf=Isoblob->Data->max_trace;

    while (j>=0)
      {
      /*
       *  Isosurface method 1, modified
       *  Add up the low/hi intervals for each element and then subdivide
       *  as necessary.
       *
       *  <x,y,z> = component space coordinates
       *  <r,s,t> = isoblob space coordinates
       */
      t2=Int_End[j];
      t1=Int_Begin[j];
      l=Int_Length[j];
      VEvaluateRay(IbeginIsoblobspace,P,t1,D);
      VEvaluateRay(IendIsoblobspace,P,t2,D);
      if(D[X]<0)
        { func_R_low=IendIsoblobspace[X]; func_R_hi=IbeginIsoblobspace[X]; }
      else
        { func_R_low=IbeginIsoblobspace[X]; func_R_hi=IendIsoblobspace[X]; }
      if(D[Y]<0)
        { func_S_low=IendIsoblobspace[Y]; func_S_hi=IbeginIsoblobspace[Y]; }
      else
        { func_S_low=IbeginIsoblobspace[Y]; func_S_hi=IendIsoblobspace[Y]; }
      if(D[Z]<0)
        { func_T_low=IendIsoblobspace[Z]; func_T_hi=IbeginIsoblobspace[Z]; }
      else
        { func_T_low=IbeginIsoblobspace[Z]; func_T_hi=IendIsoblobspace[Z]; }
      for(lowvalue=hivalue=-Isoblob->Data->Threshold,thiselement=firstelement;thiselement!=NULL;thiselement=thiselement->nextelement)
        {
        VEvaluateRay(Ibegin,thiselement->PP,t1,thiselement->DD);
        VEvaluateRay(Iend,thiselement->PP,t2,thiselement->DD);
        if(thiselement->DD[X]<0.0)
          { func_XYZ_low[X]= Iend[X];   func_XYZ_hi[X]=Ibegin[X]; }
        else
          { func_XYZ_low[X]= Ibegin[X]; func_XYZ_hi[X]=Iend[X];   }
        if(thiselement->DD[Y]<0.0)
          { func_XYZ_low[Y]= Iend[Y];   func_XYZ_hi[Y]=Ibegin[Y]; }
        else
          { func_XYZ_low[Y]= Ibegin[Y]; func_XYZ_hi[Y]=Iend[Y];   }
        if(thiselement->DD[Z]<0.0)
          { func_XYZ_low[Z]= Iend[Z];   func_XYZ_hi[Z]=Ibegin[Z]; }
        else
          { func_XYZ_low[Z]= Ibegin[Z]; func_XYZ_hi[Z]=Iend[Z];   }
        evaluate_interval( thiselement->Func, l );
        if(thiselement->str>=0.0)
          {lowvalue+=(*cs_low_top)*(thiselement->str); hivalue+=(*cs_hi_top)*(thiselement->str);}
        else
          {hivalue+=(*cs_low_top)*(thiselement->str); lowvalue+=(*cs_hi_top)*(thiselement->str);}
        }
      if ( lowvalue * hivalue > 0.0)  /* does not contain 0 */
        { j--; }
      else
        {
        if (l > (Isoblob->Data->accuracy))
          {
          if(j>=49) /* Interval stack overflow; combine two intervals */
            {
            Int_Begin[0]=Int_Begin[1];
            Int_Length[0]+=Int_Length[1];
            for(k=1;k<j;++k)
              {
              Int_Begin[k]=Int_Begin[k+1];
              Int_End[k]=Int_End[k+1];
              Int_Length[k]=Int_Length[k+1];
              }
            --j;
            }
          Int_Length[j+1]=Int_Length[j]=l*0.5;
          Int_Begin[j+1]= Int_Begin[j];
          tmp= (Int_End[j]+Int_Begin[j])*0.5;
          Int_End[j+1]= Int_Begin[j]= tmp; 
          j++;
          }
        else
          {
          dist=t1+start_dist;
          if(dist>=Isoblob->Data->accuracy)
            {
            VEvaluateRay (IPoint, P, dist, D);
            if (Point_In_Clip (IPoint, Object->Clip))
              {
              VEvaluateRay(IPoint, Ray->Initial, dist, Ray->Direction);
              Make_Vector(Ibegin,0,0,0);
              push_normal_entry(dist,IPoint,Ibegin,Object,Depth_Stack);
              Intersection_Found=TRUE;
              }
            }
          inf--;
          j--;
          if(inf<=0) break;
          }
        }
      }

    /*
     * If the isoblob isn't used inside a CSG and we have found at least
     * one intersection then we are ready, because all possible intersections
     * will be further away (we have a sorted list!).
     */

    if (!(Isoblob->Type & IS_CHILD_OBJECT) && (Intersection_Found))
    {
      break;
    }
  }

  if (Intersection_Found)
  {
    Increase_Counter(stats[Ray_Isoblob_Tests_Succeeded]);
  }

  return (Intersection_Found);
}



/*****************************************************************************
*
* FUNCTION
*
*   insert_hit
*
* INPUT
*
*   Isoblob   - Pointer to isoblob structure
*   Element   - Element to insert
*   t0, t1    - Intersection depths
*   v0, v1    - Intersection normals
*
* OUTPUT
*
*   intervals - Pointer to sorted list of hits
*   cnt       - Number of hits in intervals
*
* RETURNS
*
* DESCRIPTION
*
*   Store the points of intersection. Keep track of: whether this is
*   the start or end point of the hit, which component was pierced
*   by the ray, and the point along the ray that the hit occured at.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void insert_hit(ISOBLOB_ELEMENT *Element, DBL t0, DBL  t1, VECTOR v0, VECTOR v1, ISOBLOB_INTERVAL *intervals, int *cnt)
{
  int k;

  /* We are entering the component. */

  intervals[*cnt].type    = Element->Type | ENTERING;
  intervals[*cnt].bound   = t0;
  intervals[*cnt].Element = Element;
  Assign_Vector(intervals[*cnt].normal,v0);

  for (k = 0; t0 > intervals[k].bound; k++);

  if (k < *cnt)
  {
    /*
     * This hit point is smaller than one that already exists -->
     * bump the rest and insert it here.
     */

    POV_MEMMOVE(&intervals[k+1], &intervals[k], (*cnt-k)*sizeof(ISOBLOB_INTERVAL));

    /* We are entering the component. */

    intervals[k].type    = Element->Type | ENTERING;
    intervals[k].bound   = t0;
    intervals[k].Element = Element;
    Assign_Vector(intervals[k].normal,v0);

    (*cnt)++;

    /* We are exiting the component. */

    intervals[*cnt].type    = Element->Type | EXITING;
    intervals[*cnt].bound   = t1;
    intervals[*cnt].Element = Element;
    Assign_Vector(intervals[*cnt].normal,v1);

    for (k = k + 1; t1 > intervals[k].bound; k++);

    if (k < *cnt)
    {
      POV_MEMMOVE(&intervals[k+1], &intervals[k], (*cnt-k)*sizeof(ISOBLOB_INTERVAL));

      /* We are exiting the component. */

      intervals[k].type    = Element->Type | EXITING;
      intervals[k].bound   = t1;
      intervals[k].Element = Element;
      Assign_Vector(intervals[k].normal,v1);
    }

    (*cnt)++;
  }
  else
  {
    /* Just plop the start and end points at the end of the list */

    (*cnt)++;

    /* We are exiting the component. */

    intervals[*cnt].type    = Element->Type | EXITING;
    intervals[*cnt].bound   = t1;
    intervals[*cnt].Element = Element;
    Assign_Vector(intervals[*cnt].normal,v1);

    (*cnt)++;
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   intersect_cylinder
*
* INPUT
*
*   Element    - Pointer to element structure
*   P, D       - Ray = P + t * D
*   mindist    - Min. valid distance
*
* OUTPUT
*
*   tmin, tmax - Intersection depths found
*   v0, v1     - Intersection normals
*
* RETURNS
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static int intersect_cylinder(ISOBLOB_ELEMENT *Element, VECTOR P, VECTOR  D, DBL mindist, DBL  *tmin, DBL  *tmax, VECTOR v0, VECTOR v1)
{
  DBL a, b, c, d, t, u, v, w, len;
  VECTOR PP, DD;

  /* Transform ray into cylinder space. */

  MInvTransPoint(PP, P, Element->Trans);
  MInvTransDirection(DD, D, Element->Trans);

  VLength(len, DD);
  VInverseScaleEq(DD, len);

  /* Intersect ray with cylinder. */

  a = DD[X] * DD[X] + DD[Y] * DD[Y];

  if (a > EPSILON)
  {
    b = PP[X] * DD[X] + PP[Y] * DD[Y];
    c = PP[X] * PP[X] + PP[Y] * PP[Y] - Element->rad2;

    d = b * b - a * c;

    if (d > EPSILON)
    {
      d = sqrt(d);

      t = ( - b + d) / a;

      w = PP[Z] + t * DD[Z];

      if ((w >= 0.0) && (w <= Element->len))
      {
        if (t < *tmin) { *tmin = t; Make_Vector(v0,PP[X]+t*DD[X],PP[Y]+t*DD[Y],0);}
        if (t > *tmax) { *tmax = t; Make_Vector(v1,PP[X]+t*DD[X],PP[Y]+t*DD[Y],0);}
      }

      t = ( - b - d) / a;

      w = PP[Z] + t * DD[Z];

      if ((w >= 0.0) && (w <= Element->len))
      {
        if (t < *tmin) { *tmin = t; Make_Vector(v0,PP[X]+t*DD[X],PP[Y]+t*DD[Y],0);}
        if (t > *tmax) { *tmax = t; Make_Vector(v1,PP[X]+t*DD[X],PP[Y]+t*DD[Y],0);}
      }
    }
  }

  /* Intersect base/cap plane. */

  if (fabs(DD[Z]) > EPSILON)
  {
    /* Intersect base plane. */

    t = - PP[Z] / DD[Z];

    u = PP[X] + t * DD[X];
    v = PP[Y] + t * DD[Y];

    if ((u * u + v * v) <= Element->rad2)
    {
      if (t < *tmin) { *tmin = t; Make_Vector(v0,0,0,-1);}
      if (t > *tmax) { *tmax = t; Make_Vector(v1,0,0,-1);}
    }

    /* Intersect cap plane. */

    t = (Element->len - PP[Z]) / DD[Z];

    u = PP[X] + t * DD[X];
    v = PP[Y] + t * DD[Y];

    if ((u * u + v * v) <= Element->rad2)
    {
      if (t < *tmin) { *tmin = t; Make_Vector(v0,0,0,1);}
      if (t > *tmax) { *tmax = t; Make_Vector(v1,0,0,1);}
    }
  }

  /* Check if the intersections are valid. */

  *tmin /= len;
  *tmax /= len;

  if (*tmin < mindist) { *tmin = 0.0; Make_Vector(v0,0,0,0);}
  if (*tmax < mindist) { *tmax = 0.0; Make_Vector(v1,0,0,0);}

  if (*tmin >= *tmax)
  {
    return (FALSE);
  }

  VDot(w,v0,v0);
  if(w>=EPSILON)
    {
    MTransNormal(v0,v0,Element->Trans);
    VNormalizeEq(v0);
    }
  VDot(w,v1,v1);
  if(w>=EPSILON)
    {
    MTransNormal(v1,v1,Element->Trans);
    VNormalizeEq(v1);
    }

  return (TRUE);
}


/*****************************************************************************
*
* FUNCTION
*
*   intersect_sphere
*
* INPUT
*
*   Element    - Pointer to element structure
*   P, D       - Ray = P + t * D
*   mindist    - Min. valid distance
*
* OUTPUT
*
*   tmin, tmax - Intersection depths found
*   v0, v1     - Intersection normals
*
* RETURNS
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static int intersect_sphere(ISOBLOB_ELEMENT *Element, VECTOR P, VECTOR  D, DBL mindist, DBL  *tmin, DBL  *tmax, VECTOR v0, VECTOR v1)
{
  DBL b, d, t, len;
  VECTOR PP, DD;

  MInvTransPoint(PP, P, Element->Trans);
  MInvTransDirection(DD, D, Element->Trans);

  VLength(len, DD);
  VInverseScaleEq(DD, len);

  VDot(b, PP, DD);
  VDot(t, PP, PP);

  d = b * b - t + Element->rad2;

  if (d < EPSILON)
  {
    return (FALSE);
  }

  d = sqrt(d);

  *tmax = ( - b + d) / len;
  if (*tmax < mindist) { *tmax = 0.0; Make_Vector(v0,0,0,0);}
  else { VEvaluateRay(v0,PP,*tmax,DD); }
  *tmin = ( - b - d) / len;  if (*tmin < mindist) { *tmin = 0.0; Make_Vector(v0,0,0,0);}
  else { VEvaluateRay(v0,PP,*tmin,DD); }

  if (*tmax == *tmin)
  {
    return (FALSE);
  }
  else
  {
    if (*tmax < *tmin)
    {
      d = *tmin;  *tmin = *tmax;  *tmax = d;
      Assign_Vector(PP,v0); Assign_Vector(v0,v1); Assign_Vector(v1,PP);
    }
  }

  VDot(d,v0,v0);
  if(d>=EPSILON)
    {
    MTransNormal(v0,v0,Element->Trans);
    VNormalizeEq(v0);
    }
  VDot(d,v1,v1);
  if(d>=EPSILON)
    {
    MTransNormal(v1,v1,Element->Trans);
    VNormalizeEq(v1);
    }

  return (TRUE);
}



/*****************************************************************************
*
* FUNCTION
*
*   intersect_element
*
* INPUT
*
*   P, D       - Ray = P + t * D
*   Element    - Pointer to element structure
*   mindist    - Min. valid distance
*
* OUTPUT
*
*   tmin, tmax - Intersection depths found
*   v0, v1     - Intersection normals
*
* RETURNS
*
* DESCRIPTION
*
*   -
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static int intersect_element(VECTOR P, VECTOR  D, ISOBLOB_ELEMENT *Element, DBL mindist, DBL  *tmin, DBL  *tmax, VECTOR v0, VECTOR v1)
{
#ifdef ISOBLOB_EXTRA_STATS
  Increase_Counter(stats[Isoblob_Element_Tests]);
#endif

  *tmin = BOUND_HUGE;
  *tmax = - BOUND_HUGE;

  switch (Element->Type)
  {
    case ISOBLOB_SPHERE:

      if (!intersect_sphere(Element, P, D, mindist, tmin, tmax, v0, v1))
      {
       return (FALSE);
      }

      break;

    case ISOBLOB_CYLINDER:

      if (!intersect_cylinder(Element, P, D, mindist, tmin, tmax, v0, v1))
      {
        return (FALSE);
      }

      break;
  }

#ifdef ISOBLOB_EXTRA_STATS
  Increase_Counter(stats[Isoblob_Element_Tests_Succeeded]);
#endif

  return (TRUE);
}



/*****************************************************************************
*
* FUNCTION
*
*   determine_influences
*
* INPUT
*
*   P, D       - Ray = P + t * D
*   Isoblob    - Pointer to isoblob structure
*   mindist    - Min. valid distance
*
* OUTPUT
*
*   intervals  - Sorted list of intersections found
*
* RETURNS
*
*   int - Number of intersection found
*
* DESCRIPTION
*
*   Make a sorted list of points along the ray at which the various isoblob
*   components start and stop adding their influence.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static int determine_influences(VECTOR P, VECTOR  D, ISOBLOB *Isoblob, DBL mindist, ISOBLOB_INTERVAL *intervals)
{
  int i;
  int cnt, size;
  DBL b, t, t0, t1;
  VECTOR V1, v0, v1, DNorm;
  BSPHERE_TREE *Tree;

  cnt = 0;

  if (Isoblob->Data->Tree == NULL)
  {
    /* There's no bounding hierarchy so just step through all elements. */

    for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
    {
      if (intersect_element(P, D, &Isoblob->Data->Entry[i], mindist, &t0, &t1, v0, v1))
      {
        if(Isoblob->Trans!=NULL) {
            MTransNormal(v0,v0,Isoblob->Trans);
            VNormalizeEq(v0);
            MTransNormal(v1,v1,Isoblob->Trans);
            VNormalizeEq(v1);
            }
        insert_hit(&Isoblob->Data->Entry[i], t0, t1, v0, v1, intervals, &cnt);
      }
    }
  }
  else
  {
    /* Use isoblob's bounding hierarchy. */

    size = 0;
	VNormalize(DNorm,D);

    IsoblobQueue[size++] = Isoblob->Data->Tree;

    while (size > 0)
    {
      Tree = IsoblobQueue[--size];

      /* Test if current node is a leaf. */

      if (Tree->Entries <= 0)
      {
        /* Test element. */

        if (intersect_element(P, D, (ISOBLOB_ELEMENT *)Tree->Node, mindist, &t0, &t1, v0, v1))
        {
          if(Isoblob->Trans!=NULL) {
              MTransNormal(v0,v0,Isoblob->Trans);
              VNormalizeEq(v0);
              MTransNormal(v1,v1,Isoblob->Trans);
              VNormalizeEq(v1);
              }
          insert_hit((ISOBLOB_ELEMENT *)Tree->Node, t0, t1, v0, v1, intervals, &cnt);
        }
      }
      else
      {
        /* Test all sub-nodes. */

        for (i = 0; i < (int)Tree->Entries; i++)
        {
#ifdef ISOBLOB_EXTRA_STATS
          Increase_Counter(stats[Isoblob_Bound_Tests]);
#endif

          VSub(V1, Tree->Node[i]->C, P);
		  /* This code needs direction to be normalized */
          VDot(b, V1, DNorm);
          VDot(t, V1, V1);

          if ((t - Sqr(b)) <= Tree->Node[i]->r2)
          {
#ifdef ISOBLOB_EXTRA_STATS
            Increase_Counter(stats[Isoblob_Bound_Tests_Succeeded]);
#endif

            insert_node(Tree->Node[i], &size);
          }
        }
      }
    }
  }

  return (cnt);
}



/*****************************************************************************
*
* FUNCTION
*
*   calculate_element_field
*
* INPUT
*
*   Element - Pointer to element structure
*   P       - Point whose field value is calculated
*
* OUTPUT
*
* RETURNS
*
*   DBL - Field value
*
* DESCRIPTION
*
*   Calculate the field value of a single element in a given point P
*   (which must already have been transformed into isoblob space).
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static DBL calculate_element_field(ISOBLOB_ELEMENT *Element, VECTOR P)
{
  DBL rad2;
  VECTOR PP;
  FUNCTION *Func = Element->Func;

  MInvTransPoint(PP, P, Element->Trans);

  /* Check bounding */
  if(Element->Type==ISOBLOB_SPHERE)
    {
    VDot(rad2,PP,PP);
    if(rad2>Element->rad2) return 0;
    }
  else if(Element->Type==ISOBLOB_CYLINDER)
    {
    if(PP[Z]<0 || PP[Z]>Element->len) return 0;
    if(Sqr(PP[X])+Sqr(PP[Y])>Element->rad2) return 0;
    }

  func_R = P[X];
  func_S = P[Y];
  func_T = P[Z];

  Assign_Vector(func_XYZ,PP);
  evaluate_function(Func);

  return (*calc_stack_top);
}



/*****************************************************************************
*
* FUNCTION
*
*   calculate_field_value
*
* INPUT
*
*   Isoblob - Pointer to isoblob structure
*   P       - Point whose field value is calculated
*
* OUTPUT
*
* RETURNS
*
*   DBL - Field value
*
* DESCRIPTION
*
*   Calculate the field value of a isoblob in a given point P
*   (which must already have been transformed into isoblob space).
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static DBL calculate_field_value(ISOBLOB *Isoblob, VECTOR P)
{
  int i;
  int size;
  DBL density, rad2;
  VECTOR V1;
  BSPHERE_TREE *Tree;

  density = 0.0;

  if (Isoblob->Data->Tree == NULL)
  {
    /* There's no tree --> step through all elements. */

    for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
    {
      density += calculate_element_field(&Isoblob->Data->Entry[i], P);
    }
  }
  else
  {
    /* A tree exists --> step through the tree. */

    size = 0;

    IsoblobQueue[size++] = Isoblob->Data->Tree;

    while (size > 0)
    {
      Tree = IsoblobQueue[--size];

      /* Test if current node is a leaf. */

      if (Tree->Entries <= 0)
      {
        density += calculate_element_field((ISOBLOB_ELEMENT *)Tree->Node, P);
      }
      else
      {
        /* Test all sub-nodes. */

        for (i = 0; i < (int)Tree->Entries; i++)
        {
          /* Insert sub-node if we are inside. */

          VSub(V1, P, Tree->Node[i]->C);

          VDot(rad2, V1, V1);

          if (rad2 <= Tree->Node[i]->r2)
          {
            insert_node(Tree->Node[i], &size);
          }
        }
      }
    }
  }

  return (density);
}



/*****************************************************************************
*
* FUNCTION
*
*   Inside_Isoblob
*
* INPUT
*
*   Test_Point - Point to test
*   Object     - Pointer to isoblob structure
*
* OUTPUT
*
* RETURNS
*
*   int - TRUE if Test_Point is inside
*
* DESCRIPTION
*
*   Calculate the density at the given point and then compare to
*   the threshold to see if we are in or out of the isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

static int Inside_Isoblob(VECTOR Test_Point, OBJECT *Object)
{
  VECTOR New_Point;
  ISOBLOB *Isoblob = (ISOBLOB *) Object;

  /* Transform the point into isoblob space. */

  if (Isoblob->Trans != NULL)
  {
    MInvTransPoint(New_Point, Test_Point, Isoblob->Trans);
  }
  else
  {
    Assign_Vector(New_Point, Test_Point);
  }

  if (calculate_field_value(Isoblob, New_Point) > Isoblob->Data->Threshold - INSIDE_TOLERANCE)
  {
    /* We are inside. */

    return (!Test_Flag(Isoblob, INVERTED_FLAG));
  }
  else
  {
    /* We are outside. */

    return (Test_Flag(Isoblob, INVERTED_FLAG));
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   element_normal
*
* INPUT
*
*   P       - Surface point
*   Element - Pointer to element structure
*
* OUTPUT
*
*   Result  - Element's normal
*
* RETURNS
*
* DESCRIPTION
*
*   Calculate the normal of a single element in the point P.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void element_normal(VECTOR Result, VECTOR  P, ISOBLOB_ELEMENT *Element, DBL accuracy, char normal_type)
{
  VECTOR PP,N;
  VECTOR TPoint;
  FUNCTION *Func=Element->Func;
  DBL func,rad2,val;

  MInvTransPoint(PP, P, Element->Trans);

  /* Check bounding */
  if(Element->Type==ISOBLOB_SPHERE)
    {
    VDot(rad2,PP,PP);
    if(rad2>Element->rad2) return;
    }
  else if(Element->Type==ISOBLOB_CYLINDER)
    {
    if(PP[Z]<0 || PP[Z]>Element->len) return;
    if(Sqr(PP[X])+Sqr(PP[Y])>Element->rad2) return;
    }

  if(normal_type>0)
    {
    Assign_Vector(func_XYZ,PP);
    func_R=P[X]; func_S=P[Y]; func_T=P[Z];
    evaluate_normal(Element->Func,accuracy);
    func=*calc_stack;
    if(fabs(func)<EPSILON) return; /* Nothing to contribute to normal */
    Make_Vector(N,-norm_stack->x,-norm_stack->y,-norm_stack->z);
    MTransNormal(N,N,Element->Trans);
    N[X]-=norm_stack->r;
    N[Y]-=norm_stack->s;
    N[Z]-=norm_stack->t;
    }
  else
    {
    Assign_Vector(func_XYZ,PP);
    func_R=P[X]; func_S=P[Y]; func_T=P[Z];
    evaluate_function(Element->Func); func=*calc_stack;
    if(fabs(func)<EPSILON) return; /* Nothing to contribute to normal */
    Assign_Vector(TPoint,P);
    func_R=TPoint[X]+=accuracy; MInvTransPoint(func_XYZ, TPoint, Element->Trans);
    evaluate_function(Element->Func); val=*calc_stack;
    func_R=TPoint[X]=P[X]-accuracy; MInvTransPoint(func_XYZ, TPoint, Element->Trans);
    evaluate_function(Element->Func);
    N[X]=((*calc_stack)-val)/(accuracy*2.0);
    TPoint[X]=P[X];
    func_S=TPoint[Y]+=accuracy; MInvTransPoint(func_XYZ, TPoint, Element->Trans);
    evaluate_function(Element->Func); val=*calc_stack;
    func_S=TPoint[Y]=P[Y]-accuracy; MInvTransPoint(func_XYZ, TPoint, Element->Trans);
    evaluate_function(Element->Func);
    N[Y]=((*calc_stack)-val)/(accuracy*2.0);
    TPoint[Y]=P[Y];
    func_T=TPoint[Z]+=accuracy; MInvTransPoint(func_XYZ, TPoint, Element->Trans);
    evaluate_function(Element->Func); val=*calc_stack;
    func_T=TPoint[Z]=P[Z]-accuracy; MInvTransPoint(func_XYZ, TPoint, Element->Trans);
    evaluate_function(Element->Func);
    N[Z]=((*calc_stack)-val)/(accuracy*2.0);
    }
  VNormalize(N,N);
  VScaleEq(N,Element->str*func);
  VAddEq(Result,N);

  /* Assign_Vector(TPoint,P); func=calculate_element_field(Element,TPoint);
  if(func<EPSILON) return;
  Assign_Vector(TPoint,P); TPoint[X]+=accuracy;
  Assign_Vector(TPoint2,P); TPoint2[X]-=accuracy; N[X]=calculate_element_field(Element,TPoint2)-calculate_element_field(Element,TPoint);
  Assign_Vector(TPoint,P); TPoint[Y]+=accuracy;
  Assign_Vector(TPoint2,P); TPoint2[Y]-=accuracy; N[Y]=calculate_element_field(Element,TPoint2)-calculate_element_field(Element,TPoint);
  Assign_Vector(TPoint,P); TPoint[Z]+=accuracy;
  Assign_Vector(TPoint2,P); TPoint2[Z]-=accuracy; N[Z]=calculate_element_field(Element,TPoint2)-calculate_element_field(Element,TPoint);
  if ((N[X]==0)&&(N[Y]==0)&&(N[Z]==0)) return;
  VNormalize(N, N);
  MTransNormal(N, N, Element->Trans);
  VNormalize(N, N);
  VScaleEq(N,Element->str*func);
  VAddEq(Result,N); */
}



/*****************************************************************************
*
* FUNCTION
*
*   Isoblob_Normal
*
* INPUT
*
*   Object  - Pointer to isoblob structure
*   Inter   - Pointer to intersection
*
* OUTPUT
*
*   Result  - Isoblob's normal
*
* RETURNS
*
* DESCRIPTION
*
*   Calculate the isoblob's surface normal in the intersection point.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void Isoblob_Normal(VECTOR Result, OBJECT *Object, INTERSECTION *Inter)
{
  int i;
  int size;
  DBL dist, val;
  VECTOR New_Point, V1;
  ISOBLOB *Isoblob = (ISOBLOB *) Object;
  BSPHERE_TREE *Tree;

  /* If intersection alrady has a normal, don't bother to supply one */
  VDot(dist,Inter->INormal,Inter->INormal);
  if(dist>EPSILON)
    {
    if(Isoblob->Trans!=NULL)
      {
      MTransNormal(Result,Inter->INormal,Isoblob->Trans);
      VNormalize(Result,Result);
      }
    else Assign_Vector(Result,Inter->INormal);
    return;
    }

  /* Transform the point into the isoblob space. */

  if (Isoblob->Trans != NULL)
  {
    MInvTransPoint(New_Point, Inter->IPoint, Isoblob->Trans);
  }
  else
  {
    Assign_Vector(New_Point, Inter->IPoint);
  }

  Make_Vector(Result, 0.0, 0.0, 0.0);

  /* For each component that contributes to this point, add its bit to the normal */

  if (Isoblob->Data->Tree == NULL)
  {
    /* There's no tree --> step through all elements. */

    for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
    {
      element_normal(Result, New_Point, &(Isoblob->Data->Entry[i]), Isoblob->Data->accuracy, Isoblob->Data->normal_type);
    }
  }
  else
  {
    /* A tree exists --> step through the tree. */

    size = 0;

    IsoblobQueue[size++] = Isoblob->Data->Tree;

    while (size > 0)
    {
      Tree = IsoblobQueue[--size];

      /* Test if current node is a leaf. */

      if (Tree->Entries <= 0)
      {
        element_normal(Result, New_Point, (ISOBLOB_ELEMENT *)Tree->Node, Isoblob->Data->accuracy, Isoblob->Data->normal_type);
      }
      else
      {
        /* Test all sub-nodes. */

        for (i = 0; i < (int)Tree->Entries; i++)
        {
          /* Insert sub-node if we are inside. */

          VSub(V1, New_Point, Tree->Node[i]->C);

          VDot(dist, V1, V1);

          if (dist <= Tree->Node[i]->r2)
          {
            insert_node(Tree->Node[i], &size);
          }
        }
      }
    }
  }

  VDot(val, Result, Result);

  if (val == 0.0)
  {
    Make_Vector(Result, 1.0, 0.0, 0.0);
  }
  else
  {
    /* Normalize normal vector. */

    val = 1.0 / sqrt(val);

    VScaleEq(Result, val);
  }

  /* Transform back to world space. */

  if (Isoblob->Trans != NULL)
  {
    MTransNormal(Result, Result, Isoblob->Trans);

    VNormalize(Result, Result);
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Translate_Isoblob
*
* INPUT
*
*   Vector - Translation vector
*
* OUTPUT
*
*   Object - Pointer to isoblob structure
*
* RETURNS
*
* DESCRIPTION
*
*   Translate a isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Translate_Isoblob(OBJECT *Object, VECTOR Vector, TRANSFORM *Trans)
{
  Transform_Isoblob(Object, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_Isoblob
*
* INPUT
*
*   Vector - Rotation vector
*
* OUTPUT
*
*   Object - Pointer to isoblob structure
*
* RETURNS
*
* DESCRIPTION
*
*   Rotate a isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Rotate_Isoblob(OBJECT *Object, VECTOR Vector, TRANSFORM *Trans)
{
  Transform_Isoblob(Object, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_Isoblob
*
* INPUT
*
*   Vector - Scaling vector
*
* OUTPUT
*
*   Object - Pointer to isoblob structure
*
* RETURNS
*
* DESCRIPTION
*
*   Scale a isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Scale_Isoblob(OBJECT *Object, VECTOR Vector, TRANSFORM *Trans)
{
  Transform_Isoblob(Object, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_Isoblob
*
* INPUT
*
*   Trans  - Pointer to transformation
*
* OUTPUT
*
*   Object - Pointer to isoblob structure
*
* RETURNS
*   
* DESCRIPTION
*
*   Transform a isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Transform_Isoblob(OBJECT *Object, TRANSFORM *Trans)
{
  int i;
  ISOBLOB *Isoblob = (ISOBLOB *)Object;

  if (Isoblob->Trans == NULL)
  {
    Isoblob->Trans = Create_Transform();
  }

  Recompute_BBox(&Object->BBox, Trans);

  Compose_Transforms(Isoblob->Trans, Trans);

  for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
  {
    Transform_Textures(Isoblob->Element_Texture[i], Trans);
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Invert_Isoblob
*
* INPUT
*
*   Object - Pointer to isoblob structure
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Invert a isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

static void Invert_Isoblob(OBJECT *Object)
{
  Invert_Flag(Object, INVERTED_FLAG);
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_Isoblob
*
* INPUT
*
*   Object - Pointer to isoblob structure
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Create a new isoblob.
*
* CHANGES
*
*   -
*
******************************************************************************/

ISOBLOB *Create_Isoblob()
{
  ISOBLOB *New;

  New = (ISOBLOB *)POV_MALLOC(sizeof (ISOBLOB), "isoblob");

  INIT_OBJECT_FIELDS(New, ISOBLOB_OBJECT, &Isoblob_Methods)

  Set_Flag(New, HIERARCHY_FLAG);

  New->Trans = NULL;

  New->Data = NULL;

  New->Element_Texture = NULL;

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Copy_Isoblob
*
* INPUT
*
*   Object - Pointer to isoblob structure
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Copy a isoblob.
*
*   NOTE: The components are not copied, only the number of references is
*         counted, so that Destroy_Isoblob() knows if they can be destroyed.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static ISOBLOB *Copy_Isoblob(OBJECT *Object)
{
  int i;
  ISOBLOB *New, *Old = (ISOBLOB *)Object;

  New = Create_Isoblob();

  /* Copy isoblob. */
  *New = *Old;
/*  COPY_OBJECT_FIELDS( New, Old ); */

  New->Trans = Copy_Transform(New->Trans);

  New->Data->References++;

  New->Element_Texture = (TEXTURE **)POV_MALLOC(New->Data->Number_Of_Components*sizeof(TEXTURE *), "isoblob texture list");

  for (i = 0; i < New->Data->Number_Of_Components; i++)
  {
    New->Element_Texture[i] = Copy_Textures(Old->Element_Texture[i]);
  }

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_Isoblob_List_Element
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
*   ISOBLOB_LIST * - Pointer to isoblob element
*
* DESCRIPTION
*
*   Create a new isoblob element in the component list used during parsing.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

ISOBLOB_LIST *Create_Isoblob_List_Element()
{
  ISOBLOB_LIST *New;

  New = (ISOBLOB_LIST *)POV_MALLOC(sizeof(ISOBLOB_LIST), "isoblob component");

  init_isoblob_element(&New->elem);

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_Isoblob_Func_Element
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
*   ISOBLOB_FUNC_LIST * - Pointer to isoblob function element
*
* AUTHOR
*
*   Lummox JR
*
* DESCRIPTION
*
*   Create a new function element in the function list used during parsing.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

ISOBLOB_FUNC_LIST *Create_Isoblob_Func_Element()
{
  ISOBLOB_FUNC_LIST *New;

  New = (ISOBLOB_FUNC_LIST *)POV_MALLOC(sizeof(ISOBLOB_FUNC_LIST), "isoblob function");

  New->func=NULL;
  New->next=NULL;

  return (New);
}



/*****************************************************************************
*
* FUNCTION
*
*   Destroy_Isoblob
*
* INPUT
*
*   Object - Pointer to isoblob structure
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Destroy a isoblob.
*
*   NOTE: The isoblob data is destroyed if they are no longer used by any copy.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void Destroy_Isoblob(OBJECT *Object)
{
  int i;
  ISOBLOB *Isoblob = (ISOBLOB *)Object;
  ISOBLOB_FUNC_LIST *thisfunc,*nextfunc;

  Destroy_Transform(Isoblob->Trans);

  for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
  {
    Destroy_Textures(Isoblob->Element_Texture[i]);
  }

  POV_FREE(Isoblob->Element_Texture);

  if (--(Isoblob->Data->References) == 0)
  {
    Destroy_Bounding_Sphere_Hierarchy(Isoblob->Data->Tree);

    for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
    {
      /*
       * Make sure to destroy multiple references of a texture
       * and/or transformation only once. Multiple references
       * are only used with cylindrical isoblobs. Thus it's
       * enough to ignore all cylinder caps.
       */

      if ((Isoblob->Data->Entry[i].Type == ISOBLOB_SPHERE) ||
          (Isoblob->Data->Entry[i].Type == ISOBLOB_CYLINDER))
      {
        Destroy_Transform(Isoblob->Data->Entry[i].Trans);

        Isoblob->Data->Entry[i].Trans   = NULL;
        Isoblob->Data->Entry[i].Texture = NULL;
      }
    }

    for(thisfunc=Isoblob->Data->firstfunc;thisfunc!=NULL;thisfunc=nextfunc)
      {
      nextfunc=thisfunc->next;
      /* do not free thisfunc->func because it is allocated elsewhere */
      POV_FREE(thisfunc);
      }

    POV_FREE(Isoblob->Data->Entry);

    POV_FREE(Isoblob->Data->Intervals);

    POV_FREE(Isoblob->Data);
  }

  POV_FREE(Object);
}



/*****************************************************************************
*
* FUNCTION
*
*   Compute_Isoblob_BBox
*
* INPUT
*
*   Isoblob - Isoblob
*
* OUTPUT
*
*   Isoblob
*
* RETURNS
*
* DESCRIPTION
*
*   Calculate the bounding box of a isoblob.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void Compute_Isoblob_BBox(ISOBLOB *Isoblob)
{
  int i;
  DBL radius, radius2;
  VECTOR Center, Min, Max;

  Make_Vector(Min, BOUND_HUGE, BOUND_HUGE, BOUND_HUGE);
  Make_Vector(Max, - BOUND_HUGE, - BOUND_HUGE, - BOUND_HUGE);

  for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
  {
    if (Isoblob->Data->Entry[i].str > 0.0)
    {
      get_element_bounding_sphere(&Isoblob->Data->Entry[i], Center, &radius2);

      radius = sqrt(radius2);

      Min[X] = min(Min[X], Center[X] - radius);
      Min[Y] = min(Min[Y], Center[Y] - radius);
      Min[Z] = min(Min[Z], Center[Z] - radius);
      Max[X] = max(Max[X], Center[X] + radius);
      Max[Y] = max(Max[Y], Center[Y] + radius);
      Max[Z] = max(Max[Z], Center[Z] + radius);
    }
  }

  Make_BBox_from_min_max(Isoblob->BBox, Min, Max);

  if (Isoblob->Trans != NULL)
  {
    Recompute_BBox(&Isoblob->BBox, Isoblob->Trans);
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   get_element_bounding_sphere
*
* INPUT
*
*   Element - Pointer to element
*   Center  - Bounding sphere's center
*   Radius2 - Bounding sphere's squared radius
*
* OUTPUT
*
*   Center, Radius2
*
* RETURNS
*
* DESCRIPTION
*
*   Calculate the bounding sphere of a isoblob element.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void get_element_bounding_sphere(ISOBLOB_ELEMENT *Element, VECTOR Center, DBL *Radius2)
{
  DBL r, r2 = 0.0;
  VECTOR C, H;

  switch (Element->Type)
  {
    case ISOBLOB_SPHERE:

      r2 = Element->rad2;

      Make_Vector(C, 0.0, 0.0, 0.0);

      break;

    case ISOBLOB_CYLINDER :

      Make_Vector(C, 0.0, 0.0, 0.5 * Element->len);

      r2 = Element->rad2 + Sqr(0.5 * Element->len);

      break;
  }

  /* Transform bounding sphere if necessary. */

  r = sqrt(r2);

  MTransPoint(C, C, Element->Trans);

  Make_Vector(H, r, r, r);

  MTransDirection(H, H, Element->Trans);

  r = max(max(fabs(H[X]), fabs(H[Y])), fabs(H[Z]));

  r2 = Sqr(r) + EPSILON;

  Assign_Vector(Center, C);

  *Radius2 = r2;
}



/*****************************************************************************
*
* FUNCTION
*
*   init_isoblob_element
*
* INPUT
*
*   Element - Pointer to isoblob element
*
* OUTPUT
*
*   Element
*
* RETURNS
*
* DESCRIPTION
*
*   Init isoblob element.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void init_isoblob_element(ISOBLOB_ELEMENT *Element)
{
  Element->Type = 0;

  Element->index = 0;

  Element->len =
  Element->rad2 = 0.0;

  Element->str = 0.0;

  Element->Texture = NULL;

  Element->Trans = NULL;
}



/*****************************************************************************
*
* FUNCTION
*
*   Make_Isoblob
*
* INPUT
*
*   Isoblob    - Pointer to isoblob structure
*   threshold  - Isoblob's threshold
*   accuracy   - Isoblob's accuracy
*   max_trace  - Isoblob's max_trace
*   IsoblobFuncList- Pointer to functions
*   IsoblobList- Pointer to elements
*   npoints    - Number of elements
*
* OUTPUT
*
*   Isoblob
*
* RETURNS
*
* DESCRIPTION
*
*   Create a isoblob after it was read from the scene file.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Make_Isoblob(ISOBLOB *Isoblob, DBL threshold, DBL accuracy, int max_trace, char normal_type, ISOBLOB_FUNC_LIST *IsoblobFuncList, ISOBLOB_LIST *IsoblobList, int npoints)
{
  int i, count;
  ISOBLOB_LIST *temp;
  ISOBLOB_ELEMENT *Entry;

  if (npoints < 1)
  {
    Error("Need at least one component in a isoblob.");
  }

  /* Figure out how many components there will be. */

  temp = IsoblobList;

  for (i = count = 0; i < npoints; i++)
  {
    count++;

    temp = temp->next;

    /* Test for too many components. [DB 12/94] */

    if (count >= MAX_ISOBLOB_COMPONENTS)
    {
      Error("There are more than %d components in a isoblob.\n", MAX_ISOBLOB_COMPONENTS);
    }
  }

  /* Initialize the isoblob data. */

  Isoblob->Data->Threshold = threshold;
  Isoblob->Data->accuracy = accuracy;
  Isoblob->Data->max_trace = max_trace;
  Isoblob->Data->normal_type = normal_type;

  Isoblob->Data->firstfunc = IsoblobFuncList;

  Entry = Isoblob->Data->Entry;

  for (i = 0; i < npoints; i++)
  {
    temp = IsoblobList;

    if ((fabs(temp->elem.str) < EPSILON) || (temp->elem.rad2 < EPSILON))
    {
      Warning(0, "Degenerate Isoblob element\n");
    }

    /* Initialize component. */

    *Entry = temp->elem;

    /* We have a multi-texture isoblob. */

    if (Entry->Texture != NULL)
    {
      Set_Flag(Isoblob, MULTITEXTURE_FLAG);
    }

    Entry++;

    /* Get rid of texture non longer needed. */

    IsoblobList = IsoblobList->next;

    Destroy_Textures(temp->elem.Texture);

    POV_FREE(temp);
  }

  for (i = 0; i < count; i++)
  {
    Isoblob->Data->Entry[i].index = i;
  }

  /* Compute bounding box. */

  Compute_Isoblob_BBox(Isoblob);

  /* Create bounding sphere hierarchy. */

  if (Test_Flag(Isoblob, HIERARCHY_FLAG))
  {
    build_bounding_hierarchy(Isoblob);
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Test_Isoblob_Opacity
*
* INPUT
*
*   Isoblob - Pointer to isoblob structure
*
* OUTPUT
*
*   Isoblob
*
* RETURNS
*
* DESCRIPTION
*
*   Set the opacity flag of the isoblob according to the opacity
*   of the isoblob's texture(s).
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Test_Isoblob_Opacity(ISOBLOB *Isoblob)
{
  int i;

  /* Initialize opacity flag to the opacity of the object's texture. */

  if ((Isoblob->Texture == NULL) || (Test_Opacity(Isoblob->Texture)))
  {
    Set_Flag(Isoblob, OPAQUE_FLAG);
  }

  if (Test_Flag(Isoblob, MULTITEXTURE_FLAG))
  {
    for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
    {
      if (Isoblob->Element_Texture[i] != NULL)
      {
        /* If component's texture isn't opaque the isoblob is neither. */

        if (!Test_Opacity(Isoblob->Element_Texture[i]))
        {
          Clear_Flag(Isoblob, OPAQUE_FLAG);
        }
      }
    }
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   build_bounding_hierarchy
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* DESCRIPTION
*
*   Create the bounding sphere hierarchy.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void build_bounding_hierarchy(ISOBLOB *Isoblob)
{
  int i, nElem, maxelements;
  BSPHERE_TREE **Elements;

  nElem = (int)Isoblob->Data->Number_Of_Components;

  maxelements = 2 * nElem;

  /*
   * Now allocate an array to hold references to these elements.
   */

  Elements = (BSPHERE_TREE **)POV_MALLOC(maxelements*sizeof(BSPHERE_TREE *), "isoblob bounding hierarchy");

  /* Init list with isoblob elements. */

  for (i = 0; i < nElem; i++)
  {
    Elements[i] = (BSPHERE_TREE *)POV_MALLOC(sizeof(BSPHERE_TREE), "isoblob bounding hierarchy");

    Elements[i]->Entries = 0;
    Elements[i]->Node    = (BSPHERE_TREE **)&Isoblob->Data->Entry[i];

    get_element_bounding_sphere(&Isoblob->Data->Entry[i], Elements[i]->C, &Elements[i]->r2);
  }

  Build_Bounding_Sphere_Hierarchy(&Isoblob->Data->Tree, nElem, Elements);

  /* Get rid of the Elements array. */

  POV_FREE(Elements);
}



/*****************************************************************************
*
* FUNCTION
*
*   Determine_Isoblob_Textures
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* DESCRIPTION
*
*   Determine the textures and weights of all components affecting
*   the given intersection point. The weights are calculated from
*   the field values and sum to 1.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Determine_Isoblob_Textures(ISOBLOB *Isoblob, VECTOR IPoint, int *Count, TEXTURE **Textures, DBL *Weights)
{
  int i;
  int size;
  DBL rad2, sum;
  VECTOR V1, P;
  ISOBLOB_ELEMENT *Element;
  BSPHERE_TREE *Tree;

  /* Make sure we have enough room in the textures/weights list. */

  Reinitialize_Lighting_Code(Isoblob->Data->Number_Of_Components, &Textures, &Weights);

  /* Transform the point into the isoblob space. */

  if (Isoblob->Trans != NULL)
  {
    MInvTransPoint(P, IPoint, Isoblob->Trans);
  }
  else
  {
    Assign_Vector(P, IPoint);
  }

  *Count = 0;

  if (Isoblob->Data->Tree == NULL)
  {
    /* There's no tree --> step through all elements. */

    for (i = 0; i < Isoblob->Data->Number_Of_Components; i++)
    {
      Element = &Isoblob->Data->Entry[i];

      determine_element_texture(Isoblob, Element, Isoblob->Element_Texture[i], P, Count, Textures, Weights);
    }
  }
  else
  {
    /* A tree exists --> step through the tree. */

    size = 0;

    IsoblobQueue[size++] = Isoblob->Data->Tree;

    while (size > 0)
    {
      Tree = IsoblobQueue[--size];

      /* Test if current node is a leaf. */

      if (Tree->Entries <= 0)
      {
        determine_element_texture(Isoblob, (ISOBLOB_ELEMENT *)Tree->Node, Isoblob->Element_Texture[((ISOBLOB_ELEMENT *)Tree->Node)->index], P, Count, Textures, Weights);
      }
      else
      {
        /* Test all sub-nodes. */

        for (i = 0; i < (int)Tree->Entries; i++)
        {
          /* Insert sub-node if we are inside. */

          VSub(V1, P, Tree->Node[i]->C);

          VDot(rad2, V1, V1);

          if (rad2 <= Tree->Node[i]->r2)
          {
            insert_node(Tree->Node[i], &size);
          }
        }
      }
    }
  }

  /* Normalize weights so that their sum is 1. */

  if (*Count > 0)
  {
    sum = 0.0;

    for (i = 0; i < *Count; i++)
    {
      sum += Weights[i];
    }

    if (sum > 0.0)
    {
      for (i = 0; i < *Count; i++)
      {
        Weights[i] /= sum;
      }
    }
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   determine_element_texture
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* DESCRIPTION
*
*   If the intersection point is inside the component calculate
*   the field density and store the element's texture and the field
*   value in the texture/weight list.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void determine_element_texture(ISOBLOB *Isoblob, ISOBLOB_ELEMENT *Element, TEXTURE *Texture, VECTOR P, int *Count, TEXTURE  **Textures, DBL *Weights)
{
  int i;
  DBL density;

  density = fabs(calculate_element_field(Element, P));

  if (density > 0.0)
  {
    if (Texture == NULL)
    {
      Textures[*Count] = Isoblob->Texture;
    }
    else
    {
      Textures[*Count] = Texture;
    }

    /* Test if this texture is already used. */

    for (i = 0; i < *Count; i++)
    {
      if (Textures[i] == Textures[*Count])
      {
        /* Add current weight to already existing texture weight. */

        Weights[i] += density;

        /* Any texture can only be in the list once --> exit. */

        return;
      }
    }

    Weights[(*Count)++] = density;
  }
}



/*****************************************************************************
*
* FUNCTION
*
*   Translate_Isoblob_Element
*
* INPUT
*
*   Element - Pointer to isoblob element
*   Vector  - Translation vector
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Translate a isoblob element.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Translate_Isoblob_Element(ISOBLOB_ELEMENT *Element, VECTOR Vector)
{
  TRANSFORM Trans;

  Compute_Translation_Transform(&Trans, Vector);

  Transform_Isoblob_Element(Element, &Trans);

  Transform_Textures(Element->Texture, &Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Rotate_Isoblob_Element
*
* INPUT
*
*   Element - Pointer to isoblob element
*   Vector  - Translation vector
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Rotate a isoblob element.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Rotate_Isoblob_Element(ISOBLOB_ELEMENT *Element, VECTOR Vector)
{
  TRANSFORM Trans;

  Compute_Rotation_Transform(&Trans, Vector);

  Transform_Isoblob_Element(Element, &Trans);

  Transform_Textures(Element->Texture, &Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Scale_Isoblob_Element
*
* INPUT
*
*   Element - Pointer to isoblob element
*   Vector  - Translation vector
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Scale a isoblob element.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Scale_Isoblob_Element(ISOBLOB_ELEMENT *Element, VECTOR Vector)
{
  TRANSFORM Trans;

  Compute_Scaling_Transform(&Trans, Vector);

  Transform_Isoblob_Element(Element, &Trans);

  Transform_Textures(Element->Texture, &Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Transform_Isoblob_Element
*
* INPUT
*
*   Element - Pointer to isoblob element
*   Trans   - Transformation
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Transform a isoblob element.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Transform_Isoblob_Element(ISOBLOB_ELEMENT *Element, TRANSFORM *Trans)
{
  /* this should never happen */
  if (Element->Trans == NULL)
  {
    Element->Trans = Create_Transform();
  }

  Compose_Transforms(Element->Trans, Trans);

  Transform_Textures(Element->Texture, Trans);
}



/*****************************************************************************
*
* FUNCTION
*
*   Invert_Isoblob_Element
*
* INPUT
*
*   Element - Pointer to isoblob element
*
* OUTPUT
*
*   Object
*
* RETURNS
*
* DESCRIPTION
*
*   Invert isoblob element by negating its strength.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Invert_Isoblob_Element(ISOBLOB_ELEMENT *Element)
{
  Element->str *= -1.0;
}



/*****************************************************************************
*
* FUNCTION
*
*   Init_Isoblob_Queue
*
* INPUT
*
* OUTPUT
*   
* RETURNS
*   
* DESCRIPTION
*
*   Init queues for isoblob intersections.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Init_Isoblob_Queue()
{
  IsoblobQueue = (BSPHERE_TREE **)POV_MALLOC(Max_IsoblobQueue_Size*sizeof(BSPHERE_TREE *), "isoblob queue");
}



/*****************************************************************************
*
* FUNCTION
*
*   Destroy_Isoblob_Queue
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* DESCRIPTION
*
*   Destroy queues for isoblob intersections.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Destroy_Isoblob_Queue()
{
  if (IsoblobQueue != NULL)
  {
    POV_FREE(IsoblobQueue);
  }

  IsoblobQueue = NULL;
}



/*****************************************************************************
*
* FUNCTION
*
*   insert_node
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* DESCRIPTION
*
*   Insert a node into the node queue.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

static void insert_node(BSPHERE_TREE *Node, int *size)
{
  /* Resize queue if necessary. */

  if (*size >= (int)Max_IsoblobQueue_Size)
  {
    if (Max_IsoblobQueue_Size >= INT_MAX/2)
    {
      Error("Isoblob queue overflow!\n");
    }

    Max_IsoblobQueue_Size *= 2;

    IsoblobQueue = (BSPHERE_TREE **)POV_REALLOC(IsoblobQueue, Max_IsoblobQueue_Size*sizeof(BSPHERE_TREE *), "isoblob queue");
  }

  IsoblobQueue[(*size)++] = Node;
}



/*****************************************************************************
*
* FUNCTION
*
*   Create_Isoblob_Element_Texture_List
*
* INPUT
*
* OUTPUT
*
* RETURNS
*
* DESCRIPTION
*
*   Create a list of all textures in the isoblob.
*
*   The list actually contains copies of the textures not
*   just references to them.
*
* CHANGES
*
*   Jul 1999 : Creation.
*
******************************************************************************/

void Create_Isoblob_Element_Texture_List(ISOBLOB *Isoblob, ISOBLOB_LIST *IsoblobList, int npoints)
{
  int i, element_count, count;
  ISOBLOB_LIST *temp;

  if (npoints < 1)
  {
    Error("Need at least one component in a isoblob.");
  }

  /* Figure out how many components there will be. */

  temp = IsoblobList;

  for (i = count = 0; i < npoints; i++)
  {
    count++;

    temp = temp->next;

    /* Test for too many components. [DB 12/94] */

    if (count >= MAX_ISOBLOB_COMPONENTS)
    {
      Error("There are more than %d components in an isoblob.\n", MAX_ISOBLOB_COMPONENTS);
    }
  }

  /* Allocate memory for components. */

  Isoblob->Data = (ISOBLOB_DATA *)POV_MALLOC(sizeof(ISOBLOB_DATA), "isoblob data");

  Isoblob->Data->References = 1;

  Isoblob->Data->Number_Of_Components = count;

  Isoblob->Data->Entry = (ISOBLOB_ELEMENT *)POV_MALLOC(count*sizeof(ISOBLOB_ELEMENT), "isoblob data");

  Isoblob->Data->Intervals = (ISOBLOB_INTERVAL *)POV_MALLOC(2*Isoblob->Data->Number_Of_Components*sizeof(ISOBLOB_INTERVAL), "isoblob intervals");

  Isoblob->Data->Tree = NULL;

  Isoblob->Data->firstfunc = NULL;

  /* Init components. */

  for (i = 0; i < count; i++)
  {
    init_isoblob_element(&Isoblob->Data->Entry[i]);
  }

  /* Allocate memory for list. */

  Isoblob->Element_Texture = (TEXTURE **)POV_MALLOC(count*sizeof(TEXTURE *), "isoblob texture list");

  for (i = 0; i < count; i++)
  {
    Isoblob->Element_Texture[i] = NULL;
  }

  for (i = element_count = 0; i < npoints; i++)
  {
    temp = IsoblobList;

    /* Copy textures. */

    /*
     * Copy texture into element texture list. This is neccessary
     * because individual textures have to be transformed too if
     * copies of the isoblob are transformed.
     */
    Isoblob->Element_Texture[element_count++] = Copy_Textures(temp->elem.Texture);

    IsoblobList = IsoblobList->next;
  }
}
#else
	static char dummy[2];
#endif