/****************************************************************************
*
*                   SciTech Multi-platform Graphics Library
*
*  ========================================================================
*
*    The contents of this file are subject to the SciTech MGL Public
*    License Version 1.0 (the "License"); you may not use this file
*    except in compliance with the License. You may obtain a copy of
*    the License at http://www.scitechsoft.com/mgl-license.txt
*
*    Software distributed under the License is distributed on an
*    "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
*    implied. See the License for the specific language governing
*    rights and limitations under the License.
*
*    The Original Code is Copyright (C) 1991-1998 SciTech Software, Inc.
*
*    The Initial Developer of the Original Code is SciTech Software, Inc.
*    All Rights Reserved.
*
*  ========================================================================
*
* Language:     ANSI C
* Environment:  Any
*
* Description:  Routines for clearing memory blocks.
*
****************************************************************************/

#include "mgl.h"

/*------------------------- Implementation --------------------------------*/

#if defined(NO_ASSEMBLER) || !defined(__INTEL__)

#ifndef __WATCOMC__

/****************************************************************************
DESCRIPTION:
Copies a block of memory as fast as possible.

HEADER:
mgraph.h

PARAMETERS:
dst - Pointer to destination block
src - Pointer to source block
n   - Number of bytes to copy

REMARKS:
This function copies a block of memory as fast as possible, and has been optimized
to copy the data 32 bits at a time for maximum performance. This function is
similar to the standard C library memcpy function, but can correctly handle copying
of memory blocks that are larger than 64Kb in size for 16 bit real mode
environments. Note also that this function is generally a lot faster than some
standard C library functions.

SEE ALSO:
MGL_memcpyVIRTSRC, MGL_memcpyVIRTDST
****************************************************************************/
void MGLAPI MGL_memcpy(
    void *dst,
    void *src,
    int n)
{ memcpy(dst,src,n); }

#endif

/****************************************************************************
DESCRIPTION:
Copies a block of memory as fast as possible.


HEADER:
mgraph.h

PARAMETERS:
dst - Pointer to destination block
src - Pointer to source block
n   - Number of bytes to copy

REMARKS:
This function copies a block of memory as fast as possible, and has been optimized
to copy the data 32 bits at a time for maximum performance. This function is
similar to the standard C library memcpy function, but can correctly handle copying
of memory blocks that are larger than 64Kb in size for 16 bit real mode
environments. Note also that this function is generally a lot faster than some
standard C library functions.

This function is identical to MGL_memcpy except that it is virtual linear
framebuffer safe, and should be used for copying data where the source pointer
resides in a virtualized linear surface.

SEE ALSO:
MGL_memcpy, MGL_memcpyVIRTDST
****************************************************************************/
void MGLAPI MGL_memcpyVIRTSRC(
    void *dst,
    void *src,
    int n)
{ memcpy(dst,src,n); }

/****************************************************************************
DESCRIPTION:
Copies a block of memory as fast as possible.

HEADER:
mgraph.h

PARAMETERS:
dst - Pointer to destination block
src - Pointer to source block
n   - Number of bytes to copy

REMARKS:
This function copies a block of memory as fast as possible, and has been optimized
to copy the data 32 bits at a time for maximum performance. This function is
similar to the standard C library memcpy function, but can correctly handle copying
of memory blocks that are larger than 64Kb in size for 16 bit real mode
environments. Note also that this function is generally a lot faster than some
standard C library functions.

This function is identical to MGL_memcpy except that it is virtual linear
framebuffer safe, and should be used for copying data where the destination pointer
resides in a virtualized linear surface.

SEE ALSO:
MGL_memcpyVIRTSRC, MGL_memcpy
****************************************************************************/
void MGLAPI MGL_memcpyVIRTDST(
    void *dst,
    void *src,
    int n)
{ memcpy(dst,src,n); }

/****************************************************************************
DESCRIPTION:
Clears a memory block with 8-bit values.

HEADER:
mgraph.h

PARAMETERS:
s   - Pointer to block to clear
c   - Value to clear with
n   - Number of bytes to clear

REMARKS:
This function clears a memory block to the specified 8 bit value. This function is
similar to the standard C library memset function, but can correctly handle clearing
of memory blocks that are larger than 64Kb in size for 16 bit real mode
environments.

SEE ALSO:
MGL_memsetw, MGL_memsetl
****************************************************************************/
void MGLAPI MGL_memset(
    void *p,
    int c,
    long n)
{
    uchar *p1 = p;
    while (n--)
        *p1++ = c;
}

/****************************************************************************
DESCRIPTION:
Clears a memory block with 16-bit values.

HEADER:
mgraph.h

PARAMETERS:
p   - Pointer to block to clear
c   - Value to clear with
n   - Number of words to clear

REMARKS:
This function clears a memory block to the specified 16 bit value. This function is
similar to the standard C library memset function, but can correctly handle clearing
of memory blocks that are larger than 64Kb in size for 16 bit real mode
environments, and allows you to specify a specific 16 bit value to be cleared.

SEE ALSO:
MGL_memset, MGL_memsetl
****************************************************************************/
void MGLAPI MGL_memsetw(
    void *p,
    int c,
    long n)
{
    ushort *p1 = p;
    while (n--)
        *p1++ = c;
}

/****************************************************************************
DESCRIPTION:
Clears a memory block with 32-bit values.

HEADER:
mgraph.h

PARAMETERS:
p   - Pointer to block to clear
c   - Value to clear with
n   - Number of dwords to clear

REMARKS:
This function clears a memory block to the specified 32 bit value. This function is
similar to the standard C library memset function, but can correctly handle clearing
of memory blocks that are larger than 64Kb in size for 16 bit real mode
environments, and allows you to specify a specific 32 bit value to be cleared.

SEE ALSO:
MGL_memset, MGL_memsetw
****************************************************************************/
void MGLAPI MGL_memsetl(
    void *p,
    long c,
    long n)
{
    long *p1 = p;
    while (n--)
        *p1++ = c;
}

#endif
