/*--------------------------------------------------------------*/
/*								*/
/*                Network Backup using GTAK                     */
/*								*/
/*--------------------------------------------------------------*/
/*
  INTRODUCTION
    This REXX procedure uses GTAK (version 2.40) to backup a Network.

  Remote Hosts on the network must support one of the following
   o LanManager or LanServer running as a Server (the prefered method)
   o REXEC for TCP/IP Hosts 
   o NFS for TCP/IP 

  Where possible, use Lan Manager or Lan Server. This is the fastest,
  most secure, and trouble-free method. Of course, this does require
  that each Host be a server and running OS/2! Well thats not quite
  true. If you can do a 'NET USE x: \\server\root' then thats enough.

  To backup Hosts where only TCP/IP is available, there are two options.
  If the Host runs any version of OS/2 then you should use REXEC because
  this processes Extended Attributes.
  If it is a UNIX, DOS, Windows Host or you dont care if the Extended Attributes
  are missed, use NFS because it more reliable and easier than REXEC.

  LIMITATIONS
    The GTAK programs do not produce consistent or meaningful return 
  codes. Example 'TAPE REWIND' will return 0 when no tape is loaded!
  This means that this REXX script cannot produce a secure backup. 
  It does produce a log which should be manually reviewed after each backup.

  CUSTOMISATION
    Look at the sample configuration values below and change as required

  OPERATION
    Just execute the script. If you are using REXEC, the remote scripts
  must be installed on each Remote Host that will use REXEC. Ensure
  REXECD is started with a long (or no) timeout!

  CONFIGURATION SETTINGS
  Hosts  		The number of Host.h entries following

  Host Entries. One set of entries for each Host to be processed
   Host.h		The LAN Server name, IP Name
   HostType.h		LOCAL	The Host running this script which and the Tape Unit.
			LAN	A LAN Server Name
			NFS	An IP Name
			REXEC	An IP Name		
   HostActive.h		TRUE    The Host will be processed
			FALSE   The Host will NOT be processed

   Drives.h		The number of Drive.x.y entries for Host h
    Drive.h.d		For LOCAL Hosts, drive letter or partition number (see below)
			For LAN   Hosts, share name on the server 
			For NFS   Hosts, drive letter on the Host
			For REXEC Hosts, drive letter on the Host

   Special Note for LOCAL processing. To backup partitions not accessable to your
   operating system (e.g, NTFS, Unix, hidden-primary) use 'TAPEPHD STATUS' to list
   all the partition numbers in your system. Then use the partition numbers in the
   Drive.h.d entries. Those partitions will then be backed up by reading every 
   sector in the partition and writing them directly to tape, without TAR.
*/
call Initialise

TapeName      = "TAPE$6"
TapeBlockSize = 65535
BackupDir     = "F:\BACKUP\"		/* Backup Log & QFA directories		*/
BackupLog     = "BACKUP.LOG"		
					/* For REXEC				*/
TCPSocket     = 2500			/*  - IP Socket number for TCPSERV.EXE	*/
TCPServer     = "LCLHOST"		/*  - IP Name of THIS Host (with Tape)	*/
					/* For LAN 				*/
LANDrive      = 'Y'			/*  - NET USE x:\\Host\drive		*/
					/* For NFS				*/
NFSDrive      = 'X'			/*  - MOUNT x:				*/ 
					/* DO NOT USE SAME DRIVE LETTER!	*/

/* ------------- Backup Host & Partition definitions ------------ */

Hosts = 5

Host.1 = "LCLHOST";	HostType.1 = LOCAL;	HostActive.1 = TRUE;
Drives.1 = 6
Drive.1.1 = "C"
Drive.1.2 = "D"
Drive.1.3 = "E"
Drive.1.4 = "F"
Drive.1.5 = "G"
Drive.1.6 = 2;		

Host.2 = "RMTHOST1";	HostType.2 = LAN;	HostActive.2 = TRUE;	
Drives.2 = 1
Drive.2.1 = "ROOTC"

Host.3 = "RMTHOST2";	HostType.3 = REXEC;	HostActive.3 = FALSE;	
Drives.3 = 2
Drive.3.1 = "C"
Drive.3.2 = "D"

Host.4 = "RMTHOST3";	HostType.4 = LAN;	HostActive.4 = FALSE;
Drives.4 = 2
Drive.4.1 = "ROOTC"
Drive.4.2 = "ROOTD"

Host.5 = "RMTHOST4";	HostType.5 = NFS;	HostActive.5 = FALSE;
Drives.5 = 1
Drive.5.1 = "C"

/* ------------- Backup Host & Partition definitions ------------ */

TCPServerStarted = FALSE

BackupLogPath = BackupDir||BackupLog
BackupTapeFiles = 0

"@ECHO File   Type   ----- Host ----- ---Label---  -Size- --Drive-- ---Date---- --Time--  -------- Tape Start Block -------- >"BackupDir||"TAPEINDX.TXT"

/*
   Remove old tape directories, make new ones
*/
rc = SysMkDir(BackupDir)
rc = SysFileDelete(BackupDir||BackupLog)

Msg = "I000 Backup Processing started "DATE()" "TIME()
say Msg
"@ECHO "Msg" >>"BackupLogPath
Msg = "I005 Preparing Logfiles and Tape"
say Msg
"@ECHO "Msg" >>"BackupLogPath

do HostNum = 1 to Hosts
  DirName = BackupDir||Host.HostNum
  rc = SysMkDir(DirName)
  if HostType.HostNum = REXEC then do
    if HostActive.HostNum = TRUE then
      if TCPServerStarted = FALSE then
        do 
          "START TCPSERV "TCPSocket" "TapeName
          TCPServerStarted = TRUE
        end
  end
  do DriveNum = 1 to Drives.HostNum
   if DATATYPE(Drive.HostNum.DriveNum, 'U') = 1 then 
     rc = SysFileDelete(DirName"\"Drive.HostNum.DriveNum)
  end
end

/*
   Issue dummy Tape command and ignore the error (GTAK has a SCSI bug!)
*/
"@TAPE STAT 1>NUL 2>NUL"

/*
   Setup tape blocksize
*/
"@ECHO I001 Setting the tape blocksize to 0 >>"BackupLogPath
"@TAPE BLOCKSIZE 0  >>"BackupLogPath" 2>>&1"
if rc > 0 then
  "ECHO W070 Setting the tape blocksize failed with rc="rc" >>"BackupLogPath

/*
   Rewind tape to ensure its at the load point
*/  
"@ECHO I002 Rewinding the tape to the load point >>"BackupLogPath
"@TAPE REWIND  >>"BackupLogPath" 2>>&1"
if rc > 0 then
  "ECHO W071 Tape rewind failed with rc="rc" >>"BackupLogPath

/*
   Put a confirmation message in the log
*/
"@ECHO I003 Tape Block number >>"BackupLogPath
"@TAPE TELL  >>"BackupLogPath" 2>>&1"

/*
   Write a Tape Mark at the beginning of the tape
*/
"@ECHO I004 Writing a tape mark at the begining of the volume >>"BackupLogPath
"@TAPE MARK 1  >>"BackupLogPath" 2>>&1"
if rc > 0 then
  "ECHO W072 Writing a Tape Mark failed with rc="rc" >>"BackupLogPath
"@ECHO I003 Tape Block number >>"BackupLogPath
"@TAPE TELL  >>"BackupLogPath" 2>>&1"

/* ------------------------------------------------------------ */
/*								*/
/*       Backup each Host defined in the above tables		*/
/*								*/
/* ------------------------------------------------------------ */

do HostNum = 1 to Hosts
  if HostActive.HostNum = TRUE then 
   do
    Partitions = 1
    select
      when HostType.HostNum = LOCAL then	/* Process this LOCAL Host 			*/
         call BackupLocal
  
      when HostType.HostNum = REXEC   then	/* Process a REMOTE Host using an IP Socket	*/
         call BackupUsingREXEC

      when HostType.HostNum = LAN   then	/* Process a REMOTE Host with a LAN connection	*/
         call BackupUsingLAN

      when HostType.HostNum = NFS   then	/* Process a REMOTE Host using TCPs NFS		*/
         call BackupUsingNFS
    
      otherwise
        do
          "ECHO E090 Incorrect entry for HostType."HostNum" >>"BackupLogPath
          ContinueAbort("E090 Incorrect entry for HostType."HostNum)      
        end
    end
   end
end

/*
   Create a summary log and append it to the tape
*/
if BackupTapeFiles > 0 then 
do
  '@TAPE WRITE 'BackupDir'TAPEINDX.TXT 1>>'BackupLogPath' 2>>&1'
  '@TAPE MARK 1  >>'BackupLogPath' 2>>&1'
end

Msg = "I016 Backup processing finished"
say Msg
"@ECHO "Msg" >>"BackupLogPath

"@TAPE -nowait UNLOAD 1>NUL 2>NUL"

exit

/*  ------------------------------------------  */
/*						*/
/*             Backup the LOCAL Host		*/
/*						*/
/*  ------------------------------------------  */
/*
    Backup the selected partitions on this Host which also 
    contains the Tape Unit.
*/
BackupLocal: 
 /* 
    A. Backup each partition 
 */
 Msg = "I010 Local Host "Host.HostNum" selected for backup"
 say Msg
 "@ECHO "Msg" >>"BackupLogPath
 do DriveNum = 1 to Drives.HostNum  
   Partitions = Partitions + 1
   if DATATYPE(Drive.HostNum.DriveNum, 'W') = 1 then do
     /*
       Process the selected Partition with Physical Storage
     */
     "@ECHO I003 Tape Block number >>"BackupLogPath
     "@TAPE TELL  >>"BackupLogPath" 2>>&1"
     BackupTapeFiles = BackupTapeFiles + 1
     "@TAPE TELL >temp.txt"
     TapePos    = DELWORD(LINEIN("temp.txt", 1, 1), 1, 1)
     ignore = STREAM('temp.txt', 'C', 'CLOSE')
     TapeFile   = SUBSTR(BackupTapeFiles,  1, 3, ' ')' '
     HostName   = SUBSTR(Host.HostNum, 1, 16, ' ')' '
     TapeFile   = SUBSTR(BackupTapeFiles,  1, 3, ' ')' '
     '@TAPEPHD STATUS='Drive.HostNum.DriveNum' >temp.txt'
     parse value LINEIN("temp.txt",1 ,1) with A B C D
     ignore = STREAM('temp.txt', 'C', 'CLOSE')
     PartSize   = RIGHT(B, 7, ' ')
     PartLabel  = SUBSTR(C, 1,12, ' ') 
     TapeHeader = '"'TapeFile'TAPEPHD  'HostName||PartLabel||PartSize'           'DATE()' 'TIME()'"'
     "@ECHO "TapeHeader" "TapePos" >>"BackupDir"TAPEINDX.TXT"
     Msg = "I011 Partition "PartLabel||Drive.HostNum.DriveNum"        selected for backup"
     say Msg
     "@ECHO "Msg" >>"BackupLogPath
     '@TAPEPHD READ='Drive.HostNum.DriveNum' 2>temp.txt | TAPE -block 'TapeBlockSize' WRITE  >>'BackupLogPath' 2>>&1'
     '@TYPE temp.txt >>'BackupLogPath
     call ReportBackupEnd
     '@TAPE MARK 1  >>'BackupLogPath' 2>>&1'
   end
   else do
     /*
       Process the selected Partition with Logical Storage
     */
     call BuildTapeDetails Drive.HostNum.DriveNum, Drive.HostNum.DriveNum
     '@TAR -c -b64 -E -p -pp -P -l -M -X EXCLUDE.LST --totals --reset-archive -D rel,'QFADir' -V 'TapeHeader' 'DiskDrive.DriveNum'/ 1>>'BackupLogPath' 2>>&1'
     call ReportBackupEnd
   end
 end  
 
 /* 
    B. Rewind Tape and Compare 
 */
 call RewindForCompare Partitions

 do DriveNum = 1 to Drives.HostNum  
   if DATATYPE(Drive.HostNum.DriveNum, 'W') = 1 then do
     Msg = "I012 Comparing Partition "PartLabel||Drive.HostNum.DriveNum
     say Msg
     "@ECHO "Msg" >>"BackupLogPath
     "@TAPE -block "TapeBlockSize" READ  2>temp.txt | TAPEPHD COMPARE="Drive.HostNum.DriveNum " >>"BackupLogPath" 2>>&1"
     '@TYPE temp.txt >>'BackupLogPath
   end
   else do
     Msg = "I012 Comparing Partition "DiskLabel.DriveNum" "DiskDrive.DriveNum
     say Msg
     "@ECHO "Msg" >>"BackupLogPath
     '@TAR --compare -b64 -E -p -pp -P 'DiskDrive.DriveNum'/* 1>>'BackupLogPath' 2>>&1'
   end
   call ReportCompareEnd 
 end  
 return

/*  ------------------------------------------  */
/*						*/
/*        Backup a Remote Host using LAN	*/
/*						*/
/*  ------------------------------------------  */
/*
   This Backup method requires that the Remote Host is running LanManager
   or LanServer and that share names have been setup for the root of
   each partition to be processed. This host must have access rights to
   those share names.

   Issue a 'NET USE x: \\server\share' to access the desired partition

   This is the prefered backup method where possible because.
    o Its using standard software designed to be integrated with OS/2 
    o Its the fastest to connect
    o The remote drive is processed just as a Local drive is 
    o Probably the fastest data transfer
*/
BackupUsingLAN: 

 /* 
   A. Backup each partition 
 */
 Msg = "I010 Remote Host "Host.HostNum" selected for backup using LAN"
 say Msg
 "@ECHO "Msg" >>"BackupLogPath
 do DriveNum = 1 to Drives.HostNum  
   Partitions = Partitions + 1
   NetRes = "\\"Host.HostNum"\"Drive.HostNum.DriveNum
   "@NET USE "LANDrive": /D 1>NUL 2>NUL"
   Msg = "I020 Connecting to network resource "NetRes
   say Msg
   "@ECHO "Msg" >>"BackupLogPath
   "@NET USE "LANDrive": "NetRes" 2>>"BackupLogPath" 1>>&2"
   if rc > 0 then
     "ECHO E072 Network Connection failed with rc="rc" >>"BackupLogPath
   else do
     call BuildTapeDetails LANDrive, Drive.HostNum.DriveNum
     '@TAR -c -b64 -E -p -pp -P -l -M -X EXCLUDE.LST --totals --reset-archive -D rel,'QFADir' -V 'TapeHeader' 'LANDrive':/ 1>>'BackupLogPath' 2>>&1'
     call ReportBackupEnd
   end
 end  

 /*
    B. Rewind Tape and Compare 
 */
 call RewindForCompare Partitions

 do DriveNum = 1 to Drives.HostNum  
   Msg = "I012 Comparing Partition "DiskLabel.DriveNum" "Drive.HostNum.DriveNum
   say Msg
   "@ECHO "Msg" >>"BackupLogPath
   '@TAR --compare -b64 -E -p -pp -P 'LANDrive':/* 1>>'BackupLogPath' 2>>&1'
   call ReportCompareEnd 
 end  
 "@NET USE "LANDrive": /D 1>NUL 2>NUL"
 return 

/*  ------------------------------------------  */
/*						*/
/*       Backup a Remote Host using REXEC 	*/
/*						*/
/*  ------------------------------------------  */
/*
    This backup method requires that REXECD is running on the remote host and
    that a userid & password of the Remote Host name has been setup. Also
    the remote BACKUP.CMD must be available in the PATH.

    This is the least prefered backup method because 
     o Its complicated (requires scripts on the remote host)
     o No backup log or QFA directory is saved on this host.
*/
BackupUsingREXEC: 

 /*
    A. Backup each partition 
 */
 Msg = "I010 Remote Host "Host.HostNum" selected for backup using IP Socket"
 say Msg
 "@ECHO "Msg" >>"BackupLogPath
 do DriveNum = 1 to Drives.HostNum  
   Partitions = Partitions + 1
   Msg = "I011 Partition "DiskLabel.DriveNum" "Drive.HostNum.DriveNum": selected for backup"
   say Msg
   "@ECHO "Msg" >>"BackupLogPath
   "@ECHO I003 Tape Block number >>"BackupLogPath
   "@TAPE TELL  >>"BackupLogPath" 2>>&1"
   "@REXEC "Host.HostNum" -l "Host.HostNum" -p "Host.HostNum" BACKUP.CMD "TCPSocket" "TCPServer" "Drive.HostNum.DriveNum' 1>>'BackupLogPath' 2>>&1'
   call ReportBackupEnd
 end

 /*
    B. Rewind Tape and Compare 
 */
 call RewindForCompare Partitions

 do DriveNum = 1 to Drives.HostNum  
   Msg = "I012 Comparing "Drive.HostNum.DriveNum": "
   say Msg
   "@ECHO "Msg" >>"BackupLogPath
   "@REXEC "IP.HostNum" -l "Host.HostNum" -p "Host.HostNum" COMPARE.CMD "TCPSocket" "TCPServer" "Drive.HostNum.DriveNum' 1>>'BackupLogPath' 2>>&1'
   call ReportCompareEnd
 end
 return 

/*  ------------------------------------------  */
/*						*/
/*       Backup a Remote Host using NFS		*/
/*						*/
/*  ------------------------------------------  */
/*
   This Backup method uses TCPs NFS to mount remote file systems (partitions).

   Although not a prefered method, it may be better that REXEC
    o Its slow 
    o Probably doesnt Backup the Extended Attributes! <--- NEEDS TESTING!

   It is assumed that specifying no Group ID and User ID is 
   authorised on the Remote Host.

   Issue a 'MOUNT x:HostName:Exp_File_system
*/
BackupUsingNFS: 

 Msg = "I010 Remote Host "Host.HostNum" selected for backup using NFS"
 say Msg
 "@ECHO "Msg" >>"BackupLogPath
 /*
    A. Backup each partition 
 */
 do DriveNum = 1 to Drives.HostNum  
   Partitions = Partitions + 1
   "@UMOUNT "NFSDrive" 1>NUL 2>NUL"
   Msg = "I020 Connecting to "Drive.HostNum.DriveNum": on NFS Server "Host.HostNum
   say Msg
   "@ECHO "Msg" >>"BackupLogPath
   "@MOUNT -g -u "NFSDrive": "Host.HostNum":"Drive.HostNum.DriveNum":/ 2>>"BackupLogPath" 1>>&2"
   if rc > 0 then
     "ECHO E072 NFS Connection failed with rc="rc" >>"BackupLogPath
   else do
     call BuildTapeDetails NFSDrive, Drive.HostNum.DriveNum
     '@TAR -c -b64 -E -p -pp -P -l -M -X EXCLUDE.LST --totals --reset-archive -D rel,'QFADir' -V 'TapeHeader' 'NFSDrive':/ 1>>'BackupLogPath' 2>>&1'
     call ReportBackupEnd
   end
 end

 /*
    B. Rewind Tape and Compare 
 */
 call RewindForCompare Partitions

 do DriveNum = 1 to Drives.HostNum  
   Msg = "I012 Comparing "Drive.HostNum.DriveNum": "
   say Msg
   "@ECHO "Msg" >>"BackupLogPath
   '@TAR --compare -b64 -E -p -pp -P 'NFSDrive':/* 1>>'BackupLogPath' 2>>&1'
   call ReportCompareEnd 
 end
 "@UMOUNT "NFSDrive" 1>NUL 2>NUL"
 return 

/*  ---------------------------------------  */
/*       Create an Archive Header	     */			
/*  ---------------------------------------  */
/*
    Each Disk Partition (Local or Remote) is written with a Header of
    the following format
    FileNum TAR Hostname PartitionLabel PartitionSize DriveLetter: date time
*/
BuildTapeDetails:
 parse arg  AccessDrive, ObjectName

 BackupTapeFiles = BackupTapeFiles + 1
 HostName  = SUBSTR(Host.HostNum, 1, 16, ' ')
 DiskDrive.DriveNum = AccessDrive||":"
 DiskLabel.DriveNum = SUBSTR(WORD(SysDriveInfo(DiskDrive.DriveNum), 4), 1, 11, ' ')  
 DiskBytes  = WORD(SysDriveInfo(DiskDrive.DriveNum), 3)
 select
   when DiskBytes > 1024*1024*1024 then
      DiskSize = SUBSTR(FORMAT(DiskBytes/(1024*1024*1024),4, 0)||"Gb", 1, 7, ' ')
   when DiskBytes > 1024*1024 then
      DiskSize = SUBSTR(FORMAT(DiskBytes/(1024 * 1024), 4, 0)||"Mb", 1, 7, ' ')
   otherwise
      DiskSize = SUBSTR(DiskBytes, 1, 7, ' ')
 end 
 DriveName  = SUBSTR(ObjectName, 1, 8, ' ')
 QFADir     = BackupDir||Host.HostNum"\"ObjectName".QFA"
 TapeFile   = SUBSTR(BackupTapeFiles,  1, 3, ' ')' '
 TapeHeader = '"'TapeFile'TAR      'HostName' 'DiskLabel.DriveNum'  'DiskSize' 'DriveName' 'DATE()' 'TIME()'"'
 "@TAPE TELL >temp.txt"
 TapePos = DELWORD(LINEIN("temp.txt", 1, 1), 1, 1)
 ignore = STREAM('temp.txt', 'C', 'CLOSE')
 "@ECHO "TapeHeader" "TapePos" >>"BackupDir"TAPEINDX.TXT"

 msg = "I011 Partition "DiskLabel.DriveNum" "DriveName" selected for backup"
 say Msg
 "@ECHO "Msg" >>"BackupLogPath
 "@ECHO I003 Tape Block number >>"BackupLogPath
 "@TAPE TELL  >>"BackupLogPath" 2>>&1"
 return

/*  ---------------------------------------  */
/*       Report the success of a Backup	     */			
/*  ---------------------------------------  */
/*
    One partition on a Host has been backed up so
    see if TAR produced a non-zero return code
    and put a message into the backup log.
*/
ReportBackupEnd:
 select
   when rc = 0 then
     "@ECHO I015 Partition Backup Finished, rc = 0 >>"BackupLogPath
   when rc = 1 then
     "@ECHO W073 Partition Backup complete with warnings, rc = 1 >>"BackupLogPath
   when rc = 2 then
     "@ECHO E080 Partition Backup terminated with errors, rc = 2 >>"BackupLogPath
   otherwise
     "@ECHO E081 Partition Backup failed with rc="rc" >>"BackupLogPath
 end
 "@ECHO I003 Tape Block number >>"BackupLogPath
 "@TAPE TELL  >>"BackupLogPath" 2>>&1"
 return

/*  ---------------------------------------  */
/*    Rewind the Tape after a Host Backup    */			
/*  ---------------------------------------  */
/*
    All the selected Partitions for a Host have
    been backed up. Rewind the tape to the first
    partition for this host. The archive will then
    be compared.
*/
RewindForCompare:
 parse arg Tapemarks
 "@TAPE FILE -"TapeMarks" >>"BackupLogPath" 2>>&1"
 if rc > 0 then
   "ECHO W071 Tape rewind failed with rc="rc" >>"BackupLogPath
 "@TAPE FILE 1  >>"BackupLogPath" 2>>&1"
 return

/*  ---------------------------------------  */
/*      Report the success of a Compare      */			
/*  ---------------------------------------  */
ReportCompareEnd:
 select
   when rc = 0 then
     "@ECHO I015 Partition Compare Finished, rc = 0 >>"BackupLogPath
   when rc = 1 then
     "@ECHO W073 Partition Compare complete with warnings, rc = 1 >>"BackupLogPath
   when rc = 2 then
     "@ECHO E080 Partition Compare terminated with errors, rc = 2 >>"BackupLogPath
   otherwise
     "@ECHO E081 Partition Compare failed with rc = "rc" >>"BackupLogPath
 end
 return

Initialise:
 call rxfuncadd sysloadfuncs, rexxutil, sysloadfuncs
 call sysloadfuncs

 TRUE  =  1
 FALSE =  0

 LOCAL  = 'LOCAL'
 REXEC  = 'REXEC'
 LAN    = 'LAN'
 NFS    = 'NFS'
 NTS    = 'NTS'

 return

ContinueAbort:
parse arg ErrorText

say "------------------- GTAK Backup Error ----------------------"
say 
say "GTAK Processing failed"
say
say "Error: "ErrorText
say
say "Enter 'C' to continue processing"
say "      'V' to view the Backup Log "BackupDir||BackupLog
say "          or anything else to abort"
say
say "-------------------------------------------------------------"

do forever
 pull answer
 select
   when answer = 'C' then 
     return 
   when answer = 'V' then
     e BackupLogPath
   otherwise
     exit
 end 
end

Abort:
exit

/* ------------------------------- INFORMATION --------------------------------

 Networked Backup using TAR. Remote Hosts use one of the following
 connections
   o LAN 	(using NET USE x: \\Server\Share)
   o REXEC 	(using IP Sockets)
   o NFS	(using MOUNT & UMOUNT)

 Each partition is saved in a seperate Tape Archive. A Tape Mark preceeds the
 first archive. A text summary file is written after the last tape file. This
 lists the volume headers of each Tape File written. Tape Files are one of two
 formats, normal TAR format or TAPEPHD format (these are physical sector backups).
  			
 Backup logs
   x:\BACKUP\BACKUP.LOG		The log for the complete backup
   x:\BACKUP\HostName\drive	QFA directories for each Drive on each Host
   x:\BACKUP\TAPEINDX.TXT	A summary list of each tape file. Tape files are 
				either 'TAR' or 'TAPEPHD' format.

 Log Information Messages, Warnings and Errors

   I000 Backup Processing started {date} {time}			

          First line of the Backup log

   I001 Setting the tape blocksize to 0 			

	  The tape blocksize is about to be set to variable
	
   I002 Rewinding the tape to the load point 				

          The tape is about to be rewound to the load point and TAPE TELL should
          report 0

   I003 Tape Block number {nnnnnnn}					

          Issues at various points in the backup to show where each Backup file
          starts.

   I004 Writing a tape mark at the begining of the volume

          Tape marks are written by TAR at the end of each execution. To enable the 
          compare process to function correctly, a tape mark is also written before
          the first tape archive.

   I005 Preparing Logfiles & Tape

   I010 {Local/Remote} Host {HostName} selected for backup {using IP Socket/LAN}

          The HostActive.n entry, setup by you, was set to TRUE. This will
          be processed.

   I011 Partition {label drive:} selected for backup

          The Drive.h.n entry is being processed.

   I012 Comparing Partition {label drive:}

          After all partitions have been backed up for a Host, the tape is
          rewound and the tape archive compared to the partition.

   I015 Partition Backup Finished, rc = 0

          TAR finished a Backup of a partition with return code 0. Normally TAR
          finished with a non-zero return code.

   I016 Backup processing finished 

          Last line in the backup log

   I020 Connecting to network resource \\Server\Share

          a NET USE x: \\Server\Share is about to be issued to connect the
          network drive.
 

   W070 Setting the tape blocksize failed with rc={n}

          The tape could not be set to variable size blocks. Review the following
          log messages.

   W071 Tape rewind failed with rc={n}

          The request to rewind the tape failed. Review the following log
          messages. 

   W072 Writing a Tape Mark failed with rc={n}

          A tape mark could not be written. Review the following log messages.

   W073 Partition {Backup/Compare} complete with warnings, rc = 1

          TAR finished a Backup of a partition but some warning messages were
          issued. Review the previous log messages.

   E080 Partition {Backup/Compare} terminated with errors, rc = 2 

          TAR finished a Backup of a partition but some error messages were
          issued. Review the previous log messages.

   E081 Partition Backup failed with rc={n}

          TAR finished a Backup of a partition but some failure messages were
          issued. Review the previous log messages.

   E090 Incorrect entry for HostType.{n}

          You have not specified either LOCAL, REXEC, NFS or LAN for the specified 
          HostType.n entry. Correct the entry.

  ---------------------------------------------------------------------- */
