------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS              --
--                                                                          --
--                 S Y S T E M . T I M E _ O P E R A T I O N S              --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                             --
--                                                                          --
--            Copyright (C) 1991-1997 Florida State University              --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is the OS/2 version of this package.

with System.Error_Reporting;
--  used for Shutdown

with System.OS_Interface;

with Interfaces.C;

package body System.Time_Operations is

   use System.Error_Reporting;
   use System.OS_Interface;

   --    In this package the use of DosSleep for waiting until a
   --    certain "real" time should be replaced by using DosAsyncTimer.
   --    DosSleep just gives up a certain number of timeslices and,
   --    because of that, may sleep much longer than intended.
   --    DosAsyncTimer will be more accurate especially when used by
   --    a high-priority thread as is often the case with ceiling locks.

   --    The disadvantage of this approach is the overhead: 1 system
   --    call for starting the timer, and one for waiting. For exact
   --    timing without the race condition of getting interrupted between
   --    calculating the relative delay and starting the timer, this
   --    code should be put in a critical section which costs an additional
   --    2 calls.

   -----------
   -- Clock --
   -----------

   function Clock return Duration renames System.OS_Interface.Clock;

   ---------------
   -- Delay_For --
   ---------------

   --  ??? We need some more thought here about the correct treatment of
   --  early wakeup due to a signal.  If the signal is for task abortion we
   --  need to check for it and proceed accordingly.
   --  This may require checking all the points where these are called.

   procedure Delay_For (Rel_Time : Duration) is
      Request : Duration;

   begin
      --  If the request is zero or negative, we need to add it to the
      --  tail of the ready queue for its priority.

      if Rel_Time <= 0.0 then
         Yield;
         return;
      end if;

      Request := Rel_Time + Clock_Delay_Correction;
      System.OS_Interface.Delay_For
        (Period => Duration_In_Millisec (Request * 1_000));
   end Delay_For;

   -----------------
   -- Delay_Until --
   -----------------

   procedure Delay_Until (Abs_Time : Duration) is
      Request      : Duration;
      Current_Time : Duration := Clock;

   begin
      --  If the requested time has passed. We need to add it to the
      --  tail of the ready queue for its priority.

      if Abs_Time <= Current_Time + Clock_Delay_Correction then
         Yield;
         return;
      end if;

      Request := Abs_Time - Current_Time + Clock_Delay_Correction;
      System.OS_Interface.Delay_For
        (Period => Duration_In_Millisec (Request * 1_000));
   end Delay_Until;

end System.Time_Operations;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.1
--  date: 1997/08/25 00:38:44;  author: dewar;  state: Exp;
--  Initial revision
--  ----------------------------
--  revision 1.2
--  date: 1997/08/27 03:42:31;  author: dewar;  state: Exp;  lines: +4 -7
--  Minor reformatting
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
