------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUN-TIME LIBRARY (GNARL) COMPONENTS              --
--                                                                          --
--    S Y S T E M . T A S K _ P R I M I T I V E S . O P E R A T I O N S     --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.7 $                             --
--                                                                          --
--            Copyright (C) 1991-1997 Florida State University              --
--                                                                          --
-- GNARL is free software; you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion. GNARL is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNARL; see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- As a special exception,  if other files  instantiate  generics from this --
-- unit, or you link  this unit with other files  to produce an executable, --
-- this  unit  does not  by itself cause  the resulting  executable  to  be --
-- covered  by the  GNU  General  Public  License.  This exception does not --
-- however invalidate  any other reasons why  the executable file  might be --
-- covered by the  GNU Public License.                                      --
--                                                                          --
-- GNARL was developed by the GNARL team at Florida State University. It is --
-- now maintained by Ada Core Technologies Inc. in cooperation with Florida --
-- State University (http://www.gnat.com).                                  --
--                                                                          --
------------------------------------------------------------------------------

--  This is an OS/2 version of this package.

with Interfaces.C;
--  used for size_t

with Interfaces.OS2Lib.Errors;
with Interfaces.OS2Lib.Threads;
with Interfaces.OS2Lib.Synchronization;

with System.Parameters;
--  used for Size_Type

with System.Tasking;
--  used for Task_ID

with System.Time_Operations;
--  used for Delay_For and Delay_Until

with System.Error_Reporting;
--  used for Shutdown

with System.Parameters;
--  used for Size_Type

with System.OS_Interface;
--  used for Clock_Delay_Correction and Yield

with System.Storage_Elements;

with Unchecked_Conversion;
with Unchecked_Deallocation;

package body System.Task_Primitives.Operations is

   package IC renames Interfaces.C;
   use Interfaces.OS2Lib;
   use Interfaces.OS2Lib.Errors;
   use Interfaces.OS2Lib.Threads;
   use Interfaces.OS2Lib.Synchronization;
   use System.Storage_Elements;
   use System.Tasking;
   use System.Error_Reporting;
   use System.Time_Operations;

   ----------------------
   --  Local Constants --
   ----------------------

   Clock_Delay_Correction : Duration
     renames System.OS_Interface.Clock_Delay_Correction;

   Max_Locks_Per_Task   : constant := 100;
   Suppress_Owner_Check : constant Boolean := False;

   ------------------
   --  Local Types --
   ------------------

   type Microseconds is new IC.long;
   subtype Lock_Level is Integer range 0 .. Max_Locks_Per_Task;

   ------------------
   --  Local Data  --
   ------------------

   --  The OS/2 DosAllocThreadLocalMemory API is used to allocate our TCB_Ptr.

   --  This API reserves a small range of virtual addresses that is backed
   --  by different physical memory for each running thread. In this case we
   --  create a pointer at a fixed address that points to the TCB_Ptr for the
   --  running thread. So all threads will be able to query and update their
   --  own TCB_Ptr without destroying the TCB_Ptr of other threads.

   type Thread_Local_Data is record
      Self_ID           : Task_ID; --  ID of the current thread
      Lock_Prio_Level   : Lock_Level; --  Nr of priority changes due to locks

      --  ... room for expansion here, if we decide to make access to
      --  jump-buffer and exception stack more efficient in future
   end record;

   type Access_Thread_Local_Data is access all Thread_Local_Data;

   --  Pointer to Thread Local Data
   Thread_Local_Data_Ptr : aliased Access_Thread_Local_Data;

   type PPTLD is access all Access_Thread_Local_Data;

   -----------------------
   -- Local Subprograms --
   -----------------------

   function To_PPVOID is new Unchecked_Conversion (PPTLD, PPVOID);
   function To_Address is new Unchecked_Conversion (Task_ID, System.Address);
   function To_PFNTHREAD is
     new Unchecked_Conversion (System.Address, PFNTHREAD);

   function To_MS (D : Duration) return ULONG;

   procedure Initialize_Lock (T : Task_ID);
   procedure Finalize_Lock   (T : Task_ID);

   procedure Set_Temporary_Priority
     (T            : in Task_ID;
      New_Priority : in System.Any_Priority);

   -----------
   -- To_MS --
   -----------

   function To_MS (D : Duration) return ULONG is
   begin
      return ULONG (D * 1_000);
   end To_MS;

   -----------
   -- Clock --
   -----------

   function Clock return Duration renames System.OS_Interface.Clock;

   -------------------
   -- Abort_Handler --
   -------------------

   --  OS/2 only has limited support for asynchronous signals.
   --  It seems not to be possible to jump out of an exception
   --  handler or to change the execution context of the thread.
   --  So asynchonous transfer of control is not supported.

   ----------
   -- Self --
   ----------

   function Self return Task_ID is
      Self_ID : Task_ID renames Thread_Local_Data_Ptr.Self_ID;

   begin
      --  Check that the thread local data has been initialized.

      pragma Assert
        ((Thread_Local_Data_Ptr /= null
          and then Thread_Local_Data_Ptr.Self_ID /= null)
        or else Shutdown ("Self: Thread_Local_Data_Ptr not initialized"));

      return Self_ID;
   end Self;

   ---------------------
   -- Initialize_Lock --
   ---------------------

   procedure Initialize_Lock
     (Prio : System.Any_Priority;
      L    : access Lock)
   is
   begin
      if DosCreateMutexSem
        (ICS.Null_Ptr, L.Mutex'Unchecked_Access, 0, False32) /= NO_ERROR
      then
         raise Storage_Error;
      end if;

      pragma Assert (L.Mutex /= 0, "Error creating Mutex");
      L.Priority := Prio;
      L.Owner_ID := Null_Address;
   end Initialize_Lock;

   procedure Initialize_Lock (L : access RTS_Lock) is
   begin
      if DosCreateMutexSem
        (ICS.Null_Ptr, L.Mutex'Unchecked_Access, 0, False32) /= NO_ERROR
      then
         raise Storage_Error;
      end if;

      pragma Assert (L.Mutex /= 0, "Error creating Mutex");

      L.Priority := System.Any_Priority'Last;
      L.Owner_ID := Null_Address;
   end Initialize_Lock;

   procedure Initialize_Lock (T : in Task_ID) is
   begin
      Initialize_Lock (T.LL.L'Unchecked_Access);
   end Initialize_Lock;

   -------------------
   -- Finalize_Lock --
   -------------------

   procedure Finalize_Lock (L : access Lock) is
   begin
      Must_Not_Fail (DosCloseMutexSem (L.Mutex));
   end Finalize_Lock;

   procedure Finalize_Lock (L : access RTS_Lock) is
   begin
      Must_Not_Fail (DosCloseMutexSem (L.Mutex));
   end Finalize_Lock;

   procedure Finalize_Lock (T : Task_ID) is
   begin
      Finalize_Lock (T.LL.L'Unchecked_Access);
   end Finalize_Lock;

   ----------------
   -- Write_Lock --
   ----------------

   procedure Write_Lock (L : access Lock; Ceiling_Violation : out Boolean) is
      Self_ID      : constant Task_ID := Thread_Local_Data_Ptr.Self_ID;
      Old_Priority : constant Any_Priority := Self_ID.LL.Current_Priority;

   begin
      if L.Priority < Old_Priority then
         Ceiling_Violation := True;
         return;
      end if;

      Ceiling_Violation := False;

      --  Increase priority before getting the lock
      --  to prevent priority inversion

      Thread_Local_Data_Ptr.Lock_Prio_Level :=
        Thread_Local_Data_Ptr.Lock_Prio_Level + 1;

      if L.Priority > Old_Priority then
         Set_Temporary_Priority (Self_ID, L.Priority);
      end if;

      --  Request the lock and then update the lock owner data

      Must_Not_Fail (DosRequestMutexSem (L.Mutex, SEM_INDEFINITE_WAIT));
      L.Owner_Priority := Old_Priority;
      L.Owner_ID := Self_ID.all'Address;
   end Write_Lock;

   procedure Write_Lock (L : access RTS_Lock) is
      Self_ID      : constant Task_ID := Thread_Local_Data_Ptr.Self_ID;
      Old_Priority : constant Any_Priority := Self_ID.LL.Current_Priority;

   begin
      --  Increase priority before getting the lock to prevent
      --  priority inversion

      Thread_Local_Data_Ptr.Lock_Prio_Level :=
        Thread_Local_Data_Ptr.Lock_Prio_Level + 1;

      if L.Priority > Old_Priority then
         Set_Temporary_Priority (Self_ID, L.Priority);
      end if;

      --  Request the lock and then update the lock owner data

      Must_Not_Fail (DosRequestMutexSem (L.Mutex, SEM_INDEFINITE_WAIT));
      L.Owner_Priority := Old_Priority;
      L.Owner_ID := Self_ID.all'Address;
   end Write_Lock;

   procedure Write_Lock (T : Task_ID) is
   begin
      --  Request the lock and then update the lock owner data

      Must_Not_Fail (DosRequestMutexSem (T.LL.L.Mutex, SEM_INDEFINITE_WAIT));
      T.LL.L.Owner_ID := Null_Address;
   end Write_Lock;

   ---------------
   -- Read_Lock --
   ---------------

   procedure Read_Lock (L : access Lock; Ceiling_Violation : out Boolean)
      renames Write_Lock;

   ------------
   -- Unlock --
   ------------

   procedure Unlock (L : access Lock) is
      Self_ID      : constant Task_ID := Thread_Local_Data_Ptr.Self_ID;
      Old_Priority : constant Any_Priority := L.Owner_Priority;

   begin
      --  Check that this task holds the lock

      pragma Assert (Suppress_Owner_Check
        or else L.Owner_ID = Self_ID.all'Address
        or else Shutdown ("GNULLI---Unlock: not lock owner"));

      --  Update the owner data

      L.Owner_ID := Null_Address;

      --  Do the actual unlocking. No more references
      --  to owner data of L after this point.

      Must_Not_Fail (DosReleaseMutexSem (L.Mutex));

      --  Reset priority after unlocking to avoid priority inversion

      Thread_Local_Data_Ptr.Lock_Prio_Level :=
        Thread_Local_Data_Ptr.Lock_Prio_Level - 1;

      if L.Priority /= Old_Priority then
         Set_Temporary_Priority (Self_ID, Old_Priority);
      end if;
   end Unlock;

   procedure Unlock (L : access RTS_Lock) is
      Self_ID      : constant Task_ID := Thread_Local_Data_Ptr.Self_ID;
      Old_Priority : constant Any_Priority := L.Owner_Priority;

   begin
      --  Check that this task holds the lock

      pragma Assert (Suppress_Owner_Check
        or else L.Owner_ID = Self_ID.all'Address
        or else Shutdown ("GNULLI---Unlock: not lock owner"));

      --  Update the owner data

      L.Owner_ID := Null_Address;

      --  Do the actual unlocking. No more references
      --  to owner data of L after this point.

      Must_Not_Fail (DosReleaseMutexSem (L.Mutex));

      --  Reset priority after unlocking to avoid priority inversion

      Thread_Local_Data_Ptr.Lock_Prio_Level :=
        Thread_Local_Data_Ptr.Lock_Prio_Level - 1;

      if L.Priority /= Old_Priority then
         Set_Temporary_Priority (Self_ID, Old_Priority);
      end if;
   end Unlock;

   procedure Unlock (T : Task_ID) is
   begin
      --  Check the owner data

      pragma Assert (Suppress_Owner_Check
        or else T.LL.L.Owner_ID = Null_Address
        or else Shutdown ("GNULLI---Unlock: not locked as task lock"));

      --  Do the actual unlocking. No more references
      --  to owner data of T.LL.L after this point.

      Must_Not_Fail (DosReleaseMutexSem (T.LL.L.Mutex));
   end Unlock;

   -----------
   -- Sleep --
   -----------

   procedure Sleep (Self_ID : Task_ID) is
      Count : aliased ULONG; -- Used to store dummy result

   begin
      --  Must reset Cond BEFORE L is unlocked.

      Must_Not_Fail (DosResetEventSem (Self_ID.LL.CV, Count'Unchecked_Access));
      Unlock (Self_ID);

      --  No problem if we are interrupted here.
      --  If the condition is signaled, DosWaitEventSem will simply not block.

      Must_Not_Fail (DosWaitEventSem (Self_ID.LL.CV, SEM_INDEFINITE_WAIT));

      --  Since L was previously accquired, lock operation should not fail.

      Write_Lock (Self_ID);
   end Sleep;

   ---------------
   -- Sleep_For --
   ---------------

   --  Pre-assertion: Cond is posted
   --                 Self is locked.

   --  Post-assertion: Cond is posted
   --                  Self is locked.

   procedure Sleep_For
     (Self_ID  : in Task_ID;
      Rel_Time : in Duration;
      Timedout : out Boolean)
   is
      Count : aliased ULONG; -- Used to store dummy result

   begin
      --  Must reset Cond BEFORE Self_ID is unlocked.

      Sem_Must_Not_Fail
        (DosResetEventSem (Self_ID.LL.CV,
         Count'Unchecked_Access));
      Unlock (Self_ID);

      --  No problem if we are interrupted here: if the condition is signaled,
      --  DosWaitEventSem will simply not block

      if Rel_Time < 0.0 then
         Timedout := True;
      else
         declare
            Time_Out  : ULONG;
            Result    : APIRET;

         begin
            Time_Out := To_MS (Rel_Time + Clock_Delay_Correction);
            Result := DosWaitEventSem (Self_ID.LL.CV, Time_Out);
            pragma Assert
             ((Result = NO_ERROR or Result = ERROR_TIMEOUT
                or Result = ERROR_INTERRUPT)
              or else Shutdown ("Sleep_For: DosWaitEventSem returned error"
                & Result'Img));

            --  ???
            --  What to do with error condition ERROR_NOT_ENOUGH_MEMORY? Can
            --  we raise an exception here?  And what about ERROR_INTERRUPT?
            --  Should that be treated as a simple timeout?
            --  For now, consider only ERROR_TIMEOUT to be a timeout.

            Timedout := Result = ERROR_TIMEOUT;
         end;
      end if;

      --  Ensure post-condition

      Write_Lock (Self_ID);

      if Timedout then
         Sem_Must_Not_Fail (DosPostEventSem (Self_ID.LL.CV));
      end if;

   end Sleep_For;

   -----------------
   -- Sleep_Until --
   -----------------

   procedure Sleep_Until
     (Self_ID  : in Task_ID;
      Abs_Time : in Duration;
      Timedout : out Boolean)
   is
      Rel_Time : Duration;

   begin
      --  Change Abs_time to a relative delay.
      --  Be careful not to reintroduce the race condition that gave birth
      --  to delay until.

      --  ???
      --  Why is this small critical section any safer than no
      --  critical section at all? If the Clock function is tasking-safe
      --  it is useless, since pre-emption just after the critical
      --  is as bad as in the critical section. Issuing extra system
      --  calls only raises the risk of getting pre-empted.

      --  A better implementation would be to set a timer in the critical
      --  section, and use a MuxWait semaphore to wait on both the timer and
      --  the condition semaphore.

      Must_Not_Fail (DosEnterCritSec);
      Rel_Time := Abs_Time - Clock + Clock_Delay_Correction;
      Must_Not_Fail (DosExitCritSec);

      --  There *is* a race condition when the task is preempted here IMO

      Sleep_For (Self_ID, Rel_Time, Timedout);
   end Sleep_Until;

   ------------
   -- Wakeup --
   ------------

   procedure Wakeup (T : Task_ID) is
   begin
      Sem_Must_Not_Fail (DosPostEventSem (T.LL.CV));
   end Wakeup;

   -----------
   -- Yield --
   -----------

   procedure Yield renames System.OS_Interface.Yield;

   ----------------------------
   -- Set_Temporary_Priority --
   ----------------------------

   procedure Set_Temporary_Priority
     (T            : Task_ID;
      New_Priority : System.Any_Priority)
   is
      use Interfaces.C;
      Delta_Priority : Integer;

   begin
      --  When Lock_Prio_Level = 0, we always need to set the
      --  Active_Priority. In this way we can make priority changes
      --  due to locking independent of those caused by calling
      --  Set_Priority.

      if Thread_Local_Data_Ptr.Lock_Prio_Level = 0
        or else New_Priority < T.LL.Active_Priority
      then
         Delta_Priority := T.LL.Active_Priority - T.LL.Current_Priority;
      else
         Delta_Priority := New_Priority - T.LL.Current_Priority;
      end if;

      if Delta_Priority /= 0 then

         --  ??? There is a race-condition here
         --  The TCB is updated before the system call to make
         --  pre-emption in the critical section less likely.

         T.LL.Current_Priority := T.LL.Current_Priority + Delta_Priority;
         Must_Not_Fail
           (DosSetPriority (Scope   => PRTYS_THREAD,
                            Class   => PRTYC_NOCHANGE,
                            Delta_P => long (Delta_Priority),
                            PorTid  => T.LL.Thread));
      end if;
   end Set_Temporary_Priority;

   ------------------
   -- Set_Priority --
   ------------------

   procedure Set_Priority (T : Task_ID; Prio : System.Any_Priority) is
   begin
      T.LL.Active_Priority := Prio;
      Set_Temporary_Priority (T, Prio);
   end Set_Priority;

   ------------------
   -- Get_Priority --
   ------------------

   function Get_Priority (T : Task_ID) return System.Any_Priority is
   begin
      return System.Any_Priority (T.LL.Active_Priority);
   end Get_Priority;

   ----------------
   -- Enter_Task --
   ----------------

   procedure Enter_Task (Self_ID : Task_ID) is
   begin
      --  Initialize thread local data

      Thread_Local_Data_Ptr.Self_ID := Self_ID;
      Thread_Local_Data_Ptr.Lock_Prio_Level := 0;

      --  For OS/2, we can set Self_ID.LL.Thread in
      --  Create_Task, since the thread is created suspended.
      --  That is, there is no danger of the thread racing ahead
      --  and trying to reference Self_ID.LL.Thread before it
      --  has been initialized.

      --  .... Do we need to do anything with signals for OS/2 ???
      null;
   end Enter_Task;

   ----------------------
   --  Initialize_TCB  --
   ----------------------

   procedure Initialize_TCB (Self_ID : Task_ID; Succeeded : out Boolean) is
   begin

      --  Set Succeeded to false, since Initialize_Lock may raise exception

      Succeeded := DosCreateEventSem (ICS.Null_Ptr,
        Self_ID.LL.CV'Unchecked_Access, 0, True32) = NO_ERROR;
      Initialize_Lock (Self_ID.LL.L'Unchecked_Access);

   exception
      --  Assumes any failure must be due to insufficient resources

      when Storage_Error =>
         Must_Not_Fail (DosCloseEventSem (Self_ID.LL.CV));
         Succeeded := False;
   end Initialize_TCB;

   -----------------
   -- Create_Task --
   -----------------

   procedure Create_Task
     (T          : Task_ID;
      Wrapper    : System.Address;
      Stack_Size : System.Parameters.Size_Type;
      Priority   : System.Any_Priority;
      Succeeded  : out Boolean)
   is
      Result              : aliased APIRET;
      Adjusted_Stack_Size : System.Parameters.Size_Type;
      use System.Parameters;

   begin
      --  In OS/2 the allocated stack size should be based on the
      --  amount of address space that should be reserved for the stack.
      --  Actual memory will only be used when the stack is touched anyway.

      --  The new minimum size is 12 kB, although the EMX docs
      --  recommend a minimum size of 32 kB.  (The original was 4 kB)
      --  Systems that use many tasks (say > 30) and require much
      --  memory may run out of virtual address space, since OS/2
      --  has a per-proces limit of 512 MB, of which max. 300 MB is
      --  usable in practise.

      if Stack_Size = Unspecified_Size then
         Adjusted_Stack_Size := Default_Stack_Size;
      elsif Stack_Size < Parameters.Minimum_Stack_Size then
         Adjusted_Stack_Size := Stack_Size + Minimum_Stack_Size;
      else
         Adjusted_Stack_Size := Stack_Size;
      end if;

      --  GB970222:
      --    Because DosCreateThread is called directly here, the
      --    C RTL doesn't get initialized for the new thead. EMX by
      --    default uses per-thread local heaps in addition to the
      --    global heap. There might be other effects of by-passing the
      --    C library here.

      --    When using _beginthread the newly created thread is not
      --    blocked initially. Does this matter or can I create the
      --    thread running anyway? The LL.Thread variable will be set
      --    anyway because the variable is passed by reference to OS/2.

      T.LL.Wrapper := To_PFNTHREAD (Wrapper);

      --  The OS implicitly gives the new task the priority of this task.

      T.LL.Current_Priority := Self.LL.Current_Priority;

      --  If task was locked before activator task was
      --  initialized, assume it has OS standard priority

      if T.LL.L.Owner_Priority not in Any_Priority'Range then
         T.LL.L.Owner_Priority := 1;
      end if;

      --  Create the thread, in blocked mode

      Result := DosCreateThread
        (F_ptid   => T.LL.Thread'Unchecked_Access,
         pfn      => T.LL.Wrapper,
         param    => To_Address (T),
         flag     => Block_Child + Commit_Stack,
         cbStack  => ULONG (Adjusted_Stack_Size));

      Succeeded := (Result = NO_ERROR);

      if not Succeeded then
         return;
      end if;

      --  Set the new thread's priority
      --  (child has inherited priority from parent)

      Set_Priority (T, Priority);

      --  Start the thread executing

      Must_Not_Fail (DosResumeThread (T.LL.Thread));

   end Create_Task;

   ------------------
   -- Finalize_TCB --
   ------------------

   procedure Finalize_TCB (T : Task_ID) is
   begin
      Must_Not_Fail (DosCloseEventSem (T.LL.CV));
      Finalize_Lock (T.LL.L'Unchecked_Access);

      --  Do not deallocate TCB here.
      --  GNARL layer is responsible for that.

   end Finalize_TCB;

   ---------------
   -- Exit_Task --
   ---------------

   procedure Exit_Task is
   begin
      DosExit (EXIT_THREAD, 0);

      --  Do not finalize TCB here.
      --  GNARL layer is responsible for that.

   end Exit_Task;

   ----------------
   -- Abort_Task --
   ----------------

   procedure Abort_Task (T : Task_ID) is
   begin
      null;

      --  Task abortion not implemented yet.
      --  Should perform other action ???

   end Abort_Task;

   ----------------
   -- Initialize --
   ----------------

   procedure Initialize (Environment_Task : Task_ID) is
      Succeeded : Boolean;

   begin
      --  Set ID of environment task.

      Thread_Local_Data_Ptr.Self_ID := Environment_Task;
      Environment_Task.LL.Thread := 1; --  By definition

      --  Initialize TCB for this task.
      --  This includes all the normal task-external initialization.
      --  This is also done by Initialize_ATCB, why ???

      Initialize_TCB (Environment_Task, Succeeded);

      --  Consider raising Storage_Error,
      --  if propagation can be tolerated ???

      pragma Assert (Succeeded
        or else Shutdown ("GNULLI failure---Initialize_TCB"));

      --  Do normal task-internal initialization,
      --  which depends on an initialized TCB.

      Enter_Task (Environment_Task);

      --  Insert here any other special
      --  initialization needed for the environment task.

   end Initialize;


begin
   --  Initialize pointer to task local data.
   --  This is done once, for all tasks.

   Must_Not_Fail (DosAllocThreadLocalMemory
      ((Thread_Local_Data'Size + 31) / 32,  --  nr of 32-bit words
       To_PPVOID (Thread_Local_Data_Ptr'Access)));

   --  Initialize thread local data for main thread

   Thread_Local_Data_Ptr.Self_ID := null;
   Thread_Local_Data_Ptr.Lock_Prio_Level := 0;

end System.Task_Primitives.Operations;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.5
--  date: 1997/08/27 03:49:02;  author: dewar;  state: Exp;  lines: +2 -2
--  Fix minor syntax error
--  ----------------------------
--  revision 1.6
--  date: 1997/08/27 03:51:48;  author: dewar;  state: Exp;  lines: +1 -2
--  Remove junk with of OS2Lib.Exceptions
--  ----------------------------
--  revision 1.7
--  date: 1997/08/27 11:42:21;  author: dewar;  state: Exp;  lines: +2 -5
--  Fix peculiar code referenced in 1.4 (it was temporary debug code)
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
