/*
    system.c -- run a command with the DOS shell.

    This system() function replaces the standard library version.
    It uses swap() to make more memory available to the spawned
    command.  Environment variables used:

      SHELL       Checked first for name of command interpreter.
      COMSPEC     Used if SHELL is not set.
      SHELLOPT    Flag used to introduce the system command,
                  defaults to "-c" if the command interpreter
                  came from SHELL, defaults to "/c" if the
                  interpreter came from COMSPEC.

    If the interpreter came from SHELL, the command is also
    enclosed in double quotes.  This isn't done if the interpreter
    came from COMSPEC.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "system.h"
#include "swap.h"
#include <stdio.h>

int system( const char *command )
{
    int  r;
    char *tmpfile = tempnam("","SPAWN");
    char exec_ret;
    char args[128];
    char *shell;
    char *shellopt;
    int  quote = 0;		/* Double quotes around command ? */

    if ( (shell = getenv("SHELL")) != NULL )
		quote = 2;
    else
		if ( (shell = getenv("COMSPEC")) == NULL )
	    	shell = "c:\\command.com";

    if ( (shellopt = getenv("SHELLOPT")) == NULL )
		shellopt = quote ? "-c" : "/c";

    if ( strlen(shellopt)+strlen(command)+quote+1 > sizeof(args) )
	{
		errno = E2BIG;
		return -1;
	}

    strcpy( args, shellopt );
    strcat( args, quote ? " \"" : " " );
    strcat( args, command  );
	if ( quote )
    	strcat( args, "\"" );

    r = swap( shell, args, &exec_ret, tmpfile );

	switch (r) {
		case SWAP_NO_SHRINK:
			printf ("Unable to shrink DOS memory block.");
			break;
		case SWAP_NO_SAVE:
			printf ("Unable to save program.");
			break;
		case SWAP_NO_EXEC:
			printf ("EXEC call failed.  DOS error is: ");
			switch (exec_ret) {
				case BAD_FUNC:
					printf ("Bad function.\n");
					break;
				case FILE_NOT_FOUND:
                        printf ("%s not found.\n", shell);
					break;
				case ACCESS_DENIED:
                        printf ("Access to %s denied.\n", shell);
					break;
				case NO_MEMORY:
					printf ("Insufficient memory.\n");
					break;
				case BAD_ENVIRON:
					printf ("Bad environment.\n");
					break;
				case BAD_FORMAT:
					printf ("Bad format.\n");
					break;
			}
			break;
	}
	r = r == 0 ? exec_ret : -1;
	free (tmpfile);
	return r;
}
