/* util.c -- functions for initializing new tree elements, and other things.
   Copyright (C) 1987, 1990 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <sys/types.h>
#if defined(USG) || defined(STDC_HEADERS)
#include <string.h>
#define index strchr
#define rindex strrchr
#else
#include <strings.h>
#endif
#include "defs.h"

char *find_pred_name ();
boolean pred_and ();

/* Return the last component of pathname FNAME, with leading slashes
   compressed into one slash. */

char *
basename (fname)
     char *fname;
{
  char *p;

  /* For "/", "//", etc., return "/". */
  for (p = fname; *p == '/'; ++p)
    /* Do nothing. */ ;
  if (*p == '\0')
    return p - 1;
  p = rindex (fname, '/');
  return (p == NULL ? fname : p + 1);
}

/* Return a pointer to a new predicate structure, which has been
   linked in as the last one in the predicates list.

   Set `predicates' to point to the start of the predicates list.
   Set `last_pred' to point to the new last predicate in the list.

   Set all cells in the new structure to the default values. */

struct predicate *
get_new_pred ()
{
  register struct predicate *new_pred;

  if (predicates == NULL)
    {
      predicates = (struct predicate *)
	xmalloc (sizeof (struct predicate));
      last_pred = predicates;
    }
  else
    {
      new_pred = (struct predicate *) xmalloc (sizeof (struct predicate));
      last_pred->pred_next = new_pred;
      last_pred = new_pred;
    }
  last_pred->pred_func = NULL;
#ifdef	DEBUG
  last_pred->p_name = NULL;
#endif	/* DEBUG */
  last_pred->p_type = NO_TYPE;
  last_pred->p_prec = NO_PREC;
  last_pred->side_effects = false;
  last_pred->need_stat = true;
  last_pred->args.str = NULL;
  last_pred->pred_next = NULL;
  last_pred->pred_left = NULL;
  last_pred->pred_right = NULL;
  return (last_pred);
}

/* Return a pointer to a new predicate, with operator check.
   Like get_new_pred, but it checks to make sure that the previous
   predicate is an operator.  If it isn't, the AND operator is inserted. */

struct predicate *
get_new_pred_chk_op ()
{
  struct predicate *new_pred;

  if (last_pred)
    switch (last_pred->p_type)
      {
      case NO_TYPE:
	fflush (stdout);
	error (1, 0, "oops -- invalid default insertion of and!");
	break;

      case VICTIM_TYPE:
      case CLOSE_PAREN:
	new_pred = get_new_pred ();
	new_pred->pred_func = pred_and;
#ifdef	DEBUG
	new_pred->p_name = find_pred_name (pred_and);
#endif	/* DEBUG */
	new_pred->p_type = BI_OP;
	new_pred->p_prec = AND_PREC;
	new_pred->need_stat = false;
	new_pred->args.str = NULL;

      default:
	break;
      }
  return (get_new_pred ());
}

/* Add a victim of predicate type PRED_FUNC to the predicate input list.

   Return a pointer to the predicate node just inserted.

   Fills in the following cells of the new predicate node:

   pred_func	    PRED_FUNC
   args(.str)	    NULL
   p_type	    VICTIM_TYPE
   p_prec	    NO_PREC

   Other cells that need to be filled in are defaulted by
   get_new_pred_chk_op, which is used to insure that the prior node is
   either not there at all (we are the very first node) or is an
   operator. */

struct predicate *
insert_victim (pred_func)
     boolean (*pred_func) ();
{
  struct predicate *new_pred;

  new_pred = get_new_pred_chk_op ();
  new_pred->pred_func = pred_func;
#ifdef	DEBUG
  new_pred->p_name = find_pred_name (pred_func);
#endif	/* DEBUG */
  new_pred->args.str = NULL;
  new_pred->p_type = VICTIM_TYPE;
  new_pred->p_prec = NO_PREC;
  return (new_pred);
}

void
usage (msg)
     char *msg;
{
  FILE *out = msg ? stderr : stdout;
  extern char *version_string;

  if (msg)
    fprintf (stderr, "\n%s: %s\n", program_name, msg);

  fprintf (out, "\n%s", version_string);
  fprintf (out, "\nUsage: %s [path...] [expression]\n", program_name);

  fprintf (out,
  "\nOptions:"
  "\n  -depth                       true; process dir contents before dir itself"
#ifndef OS2
  "\n  -follow                      Dereference symbolic links"
#endif
  "\n  -fulldays                    true; from day boundaries rather than from now"
  "\n  -maxdepth levels             Do not search more than levels directories deep"
  "\n  -version                     true; print find version number on stderr"
#ifndef OS2
  "\n  -xdev                        true; don't descend dirs with different st_dev"
#endif
  "\n");

  fprintf (out,
  "\nTests:"
  "\n  -atime n                     file last accessed n*24 hours ago"
  "\n  -ctime n                     file status last modified n*24 hours ago"
#ifndef OS2
  "\n  -fstype type                 file is on a filesystem of type type"
  "\n  -group gname                 file belongs to group gname (gid allowed)"
  "\n  -inum n                      file has inode number n"
  "\n  -links n                     file has n links"
#endif
  "\n  -mtime n                     file data last modified n*24 hours ago"
  "\n  -name pattern                base of path name matches glob pattern"
  "\n                               ('*' and '?' do not match '.' at start)"
  "\n  -newer file                  modtime is more recent than file's"
  );
  fprintf (out,
#ifndef OS2
  "\n  -nouser                      no user corresponds to file's uid"
  "\n  -nogroup                     no group corresponds to file's gid"
#endif
  "\n  -perm mode                   perm bits are exactly mode (octal or symbol)"
  "\n  -perm -mode                  all of perm bits mode are set (s,s,t checked)"
  "\n  -perm +mode                  any of perm bits mode are set (s,s,t checked)"
  "\n  -permmask mode               true; set significant bits mask for next -perm"
  "\n                               (allows testing for unset bits)"
  "\n  -regex pattern               path name matches regex pattern"
  "\n  -size n[c]                   file has n blocks (or chars)"
  "\n  -type c                      file type: b, c, d, p, f, l, s"
#ifndef OS2
  "\n  -user uname                  file is owned by uname (uid allowed)"
#endif
  "\n"
  );

  fprintf (out,
  "\nNumbers can be specified as"
  "\n  +n                           for greater than n, or"
  "\n  -n                           for less than n,"
  "\n  n                            for exactly n."
  "\n");

  fprintf (out,
  "\nActions:"
  "\n  -exec cmd                    exec cmd, true if 0 status returned"
  "\n  -ls                          true; list current file in 'ls -li' format"
  "\n  -ok cmd                      like exec but ask user first; false if not 'y'"
  "\n  -print                       true; print current full pathname"
  "\n  -print0                      true; print pathname followed by a null char"
  "\n  -prune                       (no -depth) true; do not descend current dir"
  "\n                               (-depth) false; no effect\n"
  "\nIf none of -print, -ls, -ok, -exec are given, -print is assumed."
  "\n");

  fprintf (out,
  "\nGrouping operators (in order of decreasing precendence):"
  "\n  ( expr )                     force precedence"
  "\n  ! expr                       true if expr is false"
  "\n  -not expr                    same as ! expr"
  "\n  expr1 expr2                  and (implied); expr2 not eval if expr1 false"
  "\n  expr1 -a expr2               same as expr1 expr2"
  "\n  expr1 -and expr2             same as expr1 expr2"
  "\n  expr1 -o expr2               or; expr2 not eval if expr1 true"
  "\n  expr1 -or expr2              same as expr1 -o expr2"
  "\n");

  fprintf (out,
  "\nFind processes files by applying each predicate in turn until the"
  "\noverall expression evaluates false.  The expression evaluation"
  "\ncontinues until the outcome is known (left hand side false for and,"
  "\ntrue for or).  Once this happens no more expressions are"
  "\nevaluated and find moves on to the next pathname.\n"
  "\nExits with status 0 if all files are processed successfully,"
  "\n>0 if error occurrs.\n"
  );

  exit (1);
}
