#pragma strings( readonly )

#include <math.h>
#include <string.h>
#include "clerical.h"
#include "error.h"
#include "global.h"

#define L_UPDATE_PERIOD       200L             // mseconds

#define L_BITMAP_CX            80L
#define L_BITMAP_CY            80L             // size of bitmap
#define L_LEFT_EYE_CENTER_X    20L
#define L_LEFT_EYE_CENTER_Y    40L             // centerposition of left eye
#define L_RIGHT_EYE_CENTER_X   60L
#define L_RIGHT_EYE_CENTER_Y   40L             // centerposition of right eye
#define L_EYE_CX               34L
#define L_EYE_CY               70L             // size of the whole eye
#define L_PUPIL_X               3L
#define L_PUPIL_Y               3L             // size of the pupil
#define D_PUPIL_MOVE_X        ( (double) 6L )
#define D_PUPIL_MOVE_Y        ( (double)10L )  // max pupildistance from center

MRESULT EXPENTRY DlgFktAbout( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 );

// ===========================================================================
VOID DlgAbout( HWND hwnd ) {
// ===========================================================================

    if ( ! WinDlgBox( HWND_DESKTOP     // parent
                    , hwnd             // owner
                    , DlgFktAbout      // function
                    , hModuleLanguage  // .dll as resource
                    , ID_DLG_ABOUT     // ID in resource
                    , NULL )) {        // inidata
        Error( "WinDlgBox RC(%X)"
             , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
    }
}

// ===========================================================================
MRESULT EXPENTRY DlgFktAbout( HWND   hwnd
                            , ULONG  msg
                            , MPARAM mp1
                            , MPARAM mp2 ) {
// ===========================================================================

           MRESULT  mReturn;

    static ULONG    ulTimerId;
    static HDC      hdcBitmap;
    static HPS      hpsBitmap;
    static HBITMAP  hBitmap;

    switch ( msg ) {

        case WM_INITDLG : {

            SIZEL        sizBitmap  = { L_BITMAP_CX, L_BITMAP_CY };

            LONG plInfoDC[ 2 ];     // 0 .. CAPS_COLOR_PLANES
                                    // 1 .. CAPS_COLOR_BITCOUNT

            BITMAPINFOHEADER2 bmp2;

            const DEVOPENSTRUC dop        = { NULL
                                            , "DISPLAY"
                                            , NULL
                                            , NULL
                                            , NULL
                                            , NULL
                                            , NULL
                                            , NULL
                                            , NULL };

            ARCPARAMS arcPara;

            POINTL ptlDraw, ptlBox = { L_BITMAP_CX, L_BITMAP_CY };

            // --- create bitmap in memory -----------------------------------

            if (( hdcBitmap = DevOpenDC( hAnchorBlock        // Anchor-block handle
                                       , OD_MEMORY           // Type of device context
                                       , "*"                 // Device-information token
                                       , 3L                  // Number of items
                                       , (PVOID)&dop         // Open-device-context data area
                                       , NULLHANDLE          // Compatible-device-context handle
                                       )) == DEV_ERROR ) {
                Error( "DevOpenDC RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            }
            if (( hpsBitmap = GpiCreatePS( hAnchorBlock      // Anchor-block handle
                                         , hdcBitmap         // Device-context handle
                                         , &sizBitmap        // Presentation-page size
                                         ,   PU_PELS         // Options
                                           | GPIA_ASSOC
                                           | GPIT_MICRO
                                         )) == GPI_ERROR ) {
                Error( "GpiCreatePS RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            }

            // --- color informations ----------------------------------------

            if ( DevQueryCaps( hdcBitmap           // Device-context handle
                             , CAPS_COLOR_PLANES   // First item of information
                             , 2L                  // Count of items of information
                             , plInfoDC            // Device capabilities
                             ) == FALSE ) {
                Error( "DevQueryCaps RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            }

            // --- initialize BITMAPINFOHEADER2 ------------------------------

            memset( &bmp2, 0, sizeof( BITMAPINFOHEADER2 ));

            bmp2.cbFix           = sizeof( BITMAPINFOHEADER2 );
            bmp2.cx              = L_BITMAP_CX;
            bmp2.cy              = L_BITMAP_CY;
            bmp2.cPlanes         = (USHORT)plInfoDC[ 0 ];
            bmp2.cBitCount       = (USHORT)plInfoDC[ 1 ];
            bmp2.ulCompression   = BCA_UNCOMP;
            bmp2.cxResolution    = 0;
            bmp2.cyResolution    = 0;
            bmp2.cclrUsed        = 0;
            bmp2.cclrImportant   = 0;
            bmp2.usUnits         = BRU_METRIC;
            bmp2.usRecording     = BRA_BOTTOMUP;
            bmp2.usRendering     = BRH_NOTHALFTONED;
            bmp2.cSize1          = 0;
            bmp2.cSize2          = 0;
            bmp2.ulColorEncoding = BCE_RGB;
            bmp2.ulIdentifier    = 0;

            // --- create bitmap ---------------------------------------------

            if (( hBitmap = GpiCreateBitmap( hpsBitmap // Presentation-space handle
                                           , &bmp2     // Bit-map information header
                                           , 0L        // Options
                                           , NULL      // Buffer address
                                           , NULL      // Bit-map information table
                                           )) == (HBITMAP)GPI_ERROR ) {
                Error( "GpiCreateBitmap RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            }

            if (( GpiSetBitmap( hpsBitmap  // Presentation-space handle
                              , hBitmap    // Handle of the bit map to be set
                              )) == (HBITMAP)BMB_ERROR ) {
                Error( "GpiSetBitmap RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            }

            // --- draw eyes into the bitmap ---------------------------------

            arcPara.lP = L_EYE_CX / 2L;
            arcPara.lQ = L_EYE_CY / 2L;
            arcPara.lR = 0L;
            arcPara.lS = 0L;
            GpiSetArcParams( hpsBitmap, &arcPara );

            ptlDraw.x = 0L;
            ptlDraw.y = 0L;
            GpiMove( hpsBitmap, &ptlDraw );
            GpiSetColor( hpsBitmap, SYSCLR_DIALOGBACKGROUND );
            GpiBox( hpsBitmap, DRO_OUTLINEFILL, &ptlBox, 0L, 0L );

            ptlDraw.x = L_LEFT_EYE_CENTER_X;
            ptlDraw.y = L_LEFT_EYE_CENTER_Y;
            GpiMove( hpsBitmap, &ptlDraw );
            GpiSetColor( hpsBitmap, CLR_WHITE );
            GpiFullArc( hpsBitmap, DRO_FILL, MAKEFIXED(1,0));
            GpiSetColor( hpsBitmap, CLR_DARKGRAY );
            GpiFullArc( hpsBitmap, DRO_OUTLINE, MAKEFIXED(1,0));

            ptlDraw.x = L_RIGHT_EYE_CENTER_X;
            ptlDraw.y = L_RIGHT_EYE_CENTER_Y;
            GpiMove( hpsBitmap, &ptlDraw );
            GpiSetColor( hpsBitmap, CLR_WHITE );
            GpiFullArc( hpsBitmap, DRO_FILL, MAKEFIXED(1,0));
            GpiSetColor( hpsBitmap, CLR_DARKGRAY );
            GpiFullArc( hpsBitmap, DRO_OUTLINE, MAKEFIXED(1,0));

            arcPara.lP = L_PUPIL_X;
            arcPara.lQ = L_PUPIL_Y;
            arcPara.lR = 0L;
            arcPara.lS = 0L;
            GpiSetArcParams( hpsBitmap, &arcPara );

            // --- start timer for periodic update ---------------------------

            ulTimerId = WinStartTimer( hAnchorBlock        // Anchor-block handle
                                     , hwnd                // Return a unique identity
                                     , ID_TIMER_EYE_UPDATE // Timer identifier ignored
                                     , L_UPDATE_PERIOD );  // Delay time in milliseconds

            mReturn = WinDefDlgProc( hwnd, msg, mp1, mp2 );

            break;
        }

        case WM_TIMER: {

            // --- init ------------------------------------------------------

            static POINTL ptlLeftEyePosition  = { L_LEFT_EYE_CENTER_X
                                                , L_LEFT_EYE_CENTER_Y }
                        , ptlRightEyePosition = { L_RIGHT_EYE_CENTER_X
                                                , L_RIGHT_EYE_CENTER_Y };

            POINTL ptlMousePosition
                 , ptlLeftEyeCenter  = { L_LEFT_EYE_CENTER_X
                                       , L_LEFT_EYE_CENTER_Y }     // position in Bitmap
                 , ptlRightEyeCenter = { L_RIGHT_EYE_CENTER_X
                                       , L_RIGHT_EYE_CENTER_Y };   // position in Bitmap

            double dLeftEyeAngle, dRightEyeAngle;

            // --- calculate EyeCenter in world-coordinates ------------------

            WinMapWindowPoints( WinWindowFromID( hwnd, ID_BMP_ABOUT )
                              , HWND_DESKTOP
                              , &ptlLeftEyeCenter
                              , 1L );
            WinMapWindowPoints( WinWindowFromID( hwnd, ID_BMP_ABOUT )
                              , HWND_DESKTOP
                              , &ptlRightEyeCenter
                              , 1L );

            // --- get mouse-coordinates -------------------------------------

            WinQueryPointerPos( HWND_DESKTOP, &ptlMousePosition );

            // --- calculate new eye-angles ----------------------------------

            dLeftEyeAngle  = atan2( (double)( ptlMousePosition.y - ptlLeftEyeCenter.y )
                                  , (double)( ptlMousePosition.x - ptlLeftEyeCenter.x ));
            dRightEyeAngle = atan2( (double)( ptlMousePosition.y - ptlRightEyeCenter.y )
                                  , (double)( ptlMousePosition.x - ptlRightEyeCenter.x ));

            // --- clear old positions ---------------------------------------

            GpiSetColor( hpsBitmap, CLR_WHITE );
            GpiMove( hpsBitmap, &ptlLeftEyePosition );
            GpiFullArc( hpsBitmap, DRO_OUTLINEFILL, MAKEFIXED(1,0));
            GpiMove( hpsBitmap, &ptlRightEyePosition );
            GpiFullArc( hpsBitmap, DRO_OUTLINEFILL, MAKEFIXED(1,0));

            // --- draw new position -----------------------------------------

            ptlLeftEyePosition.x  = L_LEFT_EYE_CENTER_X  + (LONG)( D_PUPIL_MOVE_X * cos( dLeftEyeAngle ));
            ptlLeftEyePosition.y  = L_LEFT_EYE_CENTER_Y  + (LONG)( D_PUPIL_MOVE_Y * sin( dLeftEyeAngle ));
            ptlRightEyePosition.x = L_RIGHT_EYE_CENTER_X + (LONG)( D_PUPIL_MOVE_X * cos( dRightEyeAngle ));
            ptlRightEyePosition.y = L_RIGHT_EYE_CENTER_Y + (LONG)( D_PUPIL_MOVE_Y * sin( dRightEyeAngle ));

            GpiSetColor( hpsBitmap, CLR_BLACK );
            GpiMove( hpsBitmap, &ptlLeftEyePosition );
            GpiFullArc( hpsBitmap, DRO_OUTLINEFILL, MAKEFIXED(1,0));
            GpiMove( hpsBitmap, &ptlRightEyePosition );
            GpiFullArc( hpsBitmap, DRO_OUTLINEFILL, MAKEFIXED(1,0));

            // --- display new eyes ------------------------------------------

            WinSendDlgItemMsg( hwnd                  // Parent-window handle
                             , ID_BMP_ABOUT          // control id
                             , SM_SETHANDLE          // Message identity
                             , MPFROMLONG( hBitmap ) // Message parameter 1 .. bit-map handle
                             , MPFROMLONG( 0L ));    // Message parameter 2 .. reserved
            mReturn = 0;

            break;

        }

        case WM_CLOSE   :
        case WM_COMMAND :

            // --- termination -----------------------------------------------

            WinStopTimer( hAnchorBlock // Anchor-block handle
                        , hwnd         // Window handle that is part of the timer identification
                        , ulTimerId ); // Timer identifier

            GpiSetBitmap( hpsBitmap     // Presentation-space handle
                        , NULLHANDLE ); // Free current bitmap
            GpiDeleteBitmap( hBitmap );
            GpiDestroyPS( hpsBitmap );
            DevCloseDC( hdcBitmap );

            WinDismissDlg( hwnd
                         , TRUE );
            mReturn = 0;

            break;

        default :

            mReturn = WinDefDlgProc( hwnd, msg, mp1, mp2 );

            break;
    }

    return mReturn;
}
