#pragma strings( readonly )

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "book.h"
#include "error.h"
#include "global.h"

// ===============================================================================================
BOOL BookConstructor( const ULONG ulMaxPages, const PPAGEDATA pPageData, const TID tidLoader ) {
// ===============================================================================================

    BOOL   bState = FALSE;   // internal state
    USHORT usCYTitleBar;
    APIRET rc;

    ULONG  ulFrameFlags =   FCF_TASKLIST
                          | FCF_TITLEBAR
                          | FCF_SYSMENU
                          | FCF_MINMAX
                          | FCF_SIZEBORDER
                          | FCF_ICON
                          | FCF_AUTOICON
                          | FCF_ACCELTABLE
                          | FCF_MENU;

    ULONG ulStyle = BKS_SPIRALBIND      // or BKS_SOLIDBIND
                  | BKS_BACKPAGESBR     // or BKS_BACKPAGESBL    or BKS_BACKPAGESTR or BKS_BACKPAGESTL
                  | BKS_MAJORTABRIGHT   // or BKS_MAJORTABLEFT   or BKS_MAJORTABTOP or BKS_MAJORTABBOTTOM
                  | BKS_SQUARETABS      // or BKS_ROUNDEDTABS    or BKS_POLYGONTABS
                  | BKS_TABTEXTLEFT     // or BKS_TABTEXTRIGHT   or BKS_TABTEXTCENTER
                  | BKS_STATUSTEXTRIGHT // or BKS_STATUSTEXTLEFT or BKS_STATUSTEXTCENTER
                  | WS_GROUP
                  | WS_TABSTOP
                  | WS_VISIBLE;

    CHAR   pszTitle[ MAX_RESOURCE_STRING_LENGTH ];

    // --- Create a standard window frame ----------------------------------------------------

    if ( WinLoadString( hAnchorBlock                      // Anchor-block handle
                      , hModuleLanguage                   // Resource identity containing the string
                      , ID_STRING_TITLE                   // String identifier
                      , MAX_RESOURCE_STRING_LENGTH        // Size of buffer
                      , pszTitle ) == 0 ) {               // Buffer that is to receive the string
        strcpy( pszTitle, DEFAULT_MAIN_TITLE );
    }

    hwndFrame = WinCreateStdWindow( HWND_DESKTOP       // Parent window handle
                                  , 0L                 // Frame window styles
                                  , &ulFrameFlags      // Creation flags
                                  , WC_NOTEBOOK        // window class name
                                  , pszTitle           // Title bar text
                                  , ulStyle            // Style of client window
                                  , hModuleLanguage    // Resource is in application module
                                  , ID_RESOURCE        // ID of resources for this frame window
                                  , &hwndNotebook );   // Returned client window handle

    if (( hwndFrame == NULLHANDLE ) || ( hwndNotebook == NULLHANDLE )) {

        Error( "WinCreateStdWindow for Notebook failed. RC(%X)"
             , ERRORIDERROR( WinGetLastError( hAnchorBlock )));

    } else if ( ! PreconditionsMeet( hwndFrame )) {

        Error( "Preconditions not meet. Scaled BitBlt not supported by your system." );

    } else {

        WinSetOwner( hwndNotebook, hwndFrame );

        // --- Set the pagebutton to the same size as the titlebar -------------------------------

        usCYTitleBar = (USHORT)( (ULONG)WinQuerySysValue( HWND_DESKTOP
                                                        , SV_CYTITLEBAR )
                                 & 0x0000FFFF );

        if ( usCYTitleBar <= 0 ) {

            Error( "WinQuerySysValue( SV_CYTITLEBAR ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));

        } else if ( ! WinSendMsg( hwndNotebook                        // Window handle
                                , BKM_SETDIMENSIONS                   // Message identity
                                , MPFROM2SHORT( usCYTitleBar
                                              , usCYTitleBar )        // Parameter 1
                                , MPFROMSHORT( BKA_PAGEBUTTON ))) {   // Parameter 2

            Error( "WinSendMsg( BKM_SETDIMENSIONS / BKA_PAGEBUTTON ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));

        } /* endif */

        // ---------------------------------------------------------------------------------------

        if ( ! BookAddPages( ulMaxPages, pPageData )) {
        } else if ( ! BookShow()){
        } else {

            // --- you need a windowprocedure to proceed some user actions and init the rest -----

            pfnwpNotebookOld = WinSubclassWindow( hwndNotebook, wpNotebookClient );

            // --- init will be continued ( let's PM process some messages ) ---------------------

            WinPostMsg( hwndNotebook
                      , WM_USER_INIT_PART_1
                      , MPFROMLONG( 0L )
                      , MPFROMLONG( 0L ));

            DosResumeThread( tidLoader );

            bState = TRUE;

        } /* endif */
    } /* endif */

    return bState;
}

// ===============================================================================================
VOID BookDestructor( VOID ) {
// ===============================================================================================

    // --- destroy the rest ----------------------------------------------------------------------

    if ( hwndBMP != NULLHANDLE ) {

        if ( ! WinDestroyWindow( hwndBMP )) {

            Error( "WinDestroyWindow( BMP ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
        } /* endif */

        hwndMLE = NULLHANDLE;
    } /* endif */

    if ( hwndMLE != NULLHANDLE ) {

        if ( ! WinDestroyWindow( hwndMLE )) {

            Error( "WinDestroyWindow( MLE ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
        } /* endif */

        hwndMLE = NULLHANDLE;
    } /* endif */

    if ( hwndNotebook != NULLHANDLE ) {

        if ( ! WinDestroyWindow( hwndNotebook )) {

            Error( "WinDestroyWindow( Notebook ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
        } /* endif */

        hwndNotebook = NULLHANDLE;
    } /* endif */

    if ( hwndFrame != NULLHANDLE ) {

        if ( ! WinDestroyWindow( hwndFrame )) {

            Error( "WinDestroyWindow( Frame ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
        } /* endif */

        hwndFrame = NULLHANDLE;
    } /* endif */
}

// ===============================================================================================
BOOL BookAddPages( const ULONG ulMaxPages, const PPAGEDATA pPageData ) {
// ===============================================================================================

    BOOL   bState = TRUE;   // internal state

    ULONG  ulPageId;
    ULONG  ulIndex;
    USHORT usPageStyle = BKA_STATUSTEXTON | BKA_AUTOPAGESIZE;
    CHAR   pszBuffer[ MAX_RESOURCE_STRING_LENGTH ];
    CHAR   pszStatus[ MAX_RESOURCE_STRING_LENGTH ];

    // --- Set the text into the status line for this page. --------------------------

    if ( WinLoadString( hAnchorBlock                      // Anchor-block handle
                      , hModuleLanguage                   // Resource identity containing the string
                      , ID_STRING_STATUS                  // String identifier
                      , MAX_RESOURCE_STRING_LENGTH        // Size of buffer
                      , pszStatus ) == 0 ) {              // Buffer that is to receive the string

        Error( "WinLoadString Id (%ld) for Notebook failed. RC(%X)"
             , ID_STRING_STATUS
             , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
        strcpy( pszStatus, "%ld / %ld" );

    } /* endif */

    for ( ulIndex = 0; ulIndex < ulMaxPages; ulIndex++ ) {

        sprintf( pszBuffer, pszStatus, ulIndex + 1, ulMaxPages );

        if ( pPageData[ ulIndex ].ulResIdTabText == ID_NO_RESOURCE ) {
            usPageStyle = (USHORT)( usPageStyle & (USHORT)~BKA_MAJOR ); // clear BKA_MAJOR bit
        } else {
            usPageStyle = (USHORT)( usPageStyle | BKA_MAJOR );  // set BKA_MAJOR bit
        } /* endif */

        ulPageId = LONGFROMMR( WinSendMsg( hwndNotebook
                                         , BKM_INSERTPAGE
                                         , MPFROMLONG( 0L )
                                         , MPFROM2SHORT( usPageStyle, BKA_LAST )));

        if ( ! ulPageId ) {

            Error( "WinSendMsg( BKM_INSERTPAGE ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            bState = FALSE;

            // --- Insert a pointer to this page's info into the space available -----------------
            //     in each page (its PAGE DATA that is available to the application).

        } else if ( ! WinSendMsg( hwndNotebook
                                , BKM_SETPAGEDATA
                                , MPFROMLONG( ulPageId )
                                , MPFROMP( &pPageData[ ulIndex ] ))) {

            Error( "WinSendMsg( BKM_SETPAGEDATA ) for Notebook failed. RC(%X)"
                 , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            bState = FALSE;

        } else if ( ! WinSendMsg( hwndNotebook
                                , BKM_SETSTATUSLINETEXT
                                , MPFROMLONG( ulPageId )
                                , MPFROMP( pszBuffer ))) {

            Error( "WinSendMsg( BKM_SETSTATUSLINETEXT ) for Notebook failed. RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
            bState = FALSE;

        } else  if ( pPageData[ ulIndex ].ulResIdTabText  != ID_NO_RESOURCE ) {

            // --- Set the text into the tab for this page. ------------------------------

            if ( WinLoadString( hAnchorBlock                         // Anchor-block handle
                              , hModuleLanguage                      // Resource identity containing the string
                              , pPageData[ ulIndex ].ulResIdTabText  // String identifier
                              , MAX_RESOURCE_STRING_LENGTH           // Size of buffer
                              , pszBuffer ) == 0 ) {                 // Buffer that is to receive the string

                Error( "WinLoadString Id (%ld) for Notebookpage (%ld) failed. RC(%X)"
                     , pPageData[ ulIndex ].ulResIdTabText
                     , ulIndex
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
                bState = FALSE;

            } else if ( ! WinSendMsg( hwndNotebook
                                    , BKM_SETTABTEXT
                                    , MPFROMLONG( ulPageId )
                                    , MPFROMP( pszBuffer ))) {

                Error( "WinSendMsg( BKM_SETTABTEXT ) for Notebook failed. RC(%X)"
                     , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
                bState = FALSE;

            } else {

            } /* endif */
        } /* endif */
    } /* endfor */

    return bState;
}

// ===============================================================================================
BOOL BookShow( VOID ) {
// ===============================================================================================

    BOOL   bState = TRUE;   // internal state

    SHORT  sPageCount;
    ULONG  ulFirstPageId;

    sPageCount = (SHORT)SHORT1FROMMR( WinSendMsg( hwndNotebook
                                                , BKM_QUERYPAGECOUNT
                                                , MPFROMLONG( 0L )
                                                , MPFROMSHORT( BKA_END )));
    if ( sPageCount > 0 ) {

        // --- display ( load ) first page ---------------------------------------------------

        ulFirstPageId = LONGFROMMR( WinSendMsg( hwndNotebook
                                              , BKM_QUERYPAGEID
                                              , MPFROMLONG( 0L )
                                              , MPFROMSHORT( BKA_TOP )));

        switch ( ulFirstPageId ) {

            case BOOKERR_INVALID_PARAMETERS : {

                // --- Returned if the page ID specified for the ulPageId parameter ----------
                //     for param1 is invalid when specifying either the BKA_PREV or
                //     BKA_NEXT attribute in the usQueryOrder parameter.

                Error( "WinSendMsg( BKM_QUERYPAGEID ) for Notebook returns BOOKERR_INVALID_PARAMETERS" );
                ulFirstPageId = 0;
                bState = FALSE;

                break;
            }

            case 0 : {

                // --- Requested page not found. This could be an indication that the end ----
                //     or front of the list has been reached, or that the notebook is empty.

                Error( (PSZ)"WinSendMsg ( BKM_QUERYPAGEID ) for Notebook returns NO_PAGE_FOUND" );
                bState = FALSE;

                break;
            }

            default : {

                // --- Retrieved page identifier. --------------------------------------------

                if ( ! WinSendMsg( hwndNotebook
                                 , BKM_TURNTOPAGE
                                 , MPFROMLONG( ulFirstPageId )
                                 , MPFROMP( NULL ))) {

                    Error( "WinSendMsg( BKM_TURNTOPAGE ) for Notebook failed. RC(%X)"
                         , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
                    bState = FALSE;
                } /* endif */

                break;
            }
        } /* switch */
    } /* endif */

    // --- set the new notebookposition ----------------------------------------------------------

    if ( ! WinSetWindowPos( hwndFrame             // Window handle
                          , NULLHANDLE            // Relative window-placement order
                          , 0L                    // Window position, x-coordinate
                          , 0L                    // Window position, y-coordinate
                          , 0L                    // Window size cx
                          , 0L                    // Window size cy
                          , SWP_SIZE )) {         // Window-positioning options

        Error( "WinSetWindowPos( SWP_SIZE ) for Notebook failed. RC(%X)"
             , ERRORIDERROR( WinGetLastError( hAnchorBlock )));
    } /* endif */

    return bState;
}

