 /* VESA package for emx/gcc --- Copyright (c) 1993 by Johannes Martin */
#include <sys/hw.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#define INCL_WIN
#define INCL_DOSPROCESS
#define INCL_DOSFILEMGR
#define INCL_DOSQUEUES
#define INCL_DOSSEMAPHORES
#define INCL_DOSMISC
#define INCL_DOSERRORS
#define INCL_DOSSESMGR

#include <os2emx.h>
#include <os2thunk.h>

#include "jmdive.h"
#include "vesa.h"
#include "vesadll.h"
#include "common.h"

#include "vesa_div.h"

static struct _ModeInfo _ModeInfos[] =
    { { 0x1, { 0x0f, 0x7, 0x0, 1024, 1024, 0x0, 0x0,
               OS2VesaSetWindow, OS2VesaGetWindow, 160,
               640, 400, 8, 16, 1, 4, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0 } },
      { 0xb, { 0x1f, 0x7, 0x0, 0x0, 0x0, 0x0, 0x0,
               OS2VesaSetWindow, OS2VesaGetWindow, 0x0,
               0x0, 0x0, 8, 16, 1, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0 } }
    };

static struct {
  VESAWORD modeNum;
  VESAWORD width, height;
  VESACHAR bits;
} vesaModes[] = { { 0x220,    0,    0,  0 },
                  { 0x013,  320,  200,  8 },
                  { 0x100,  640,  400,  8 },
                  { 0x101,  640,  480,  8 },
                  { 0x103,  800,  600,  8 },
                  { 0x105, 1024,  768,  8 },
                  { 0x107, 1280, 1024,  8 },
                  { 0x110,  640,  480, 15 },
                  { 0x111,  640,  480, 16 },
                  { 0x112,  640,  480, 24 },
                  { 0x113,  800,  600, 15 },
                  { 0x114,  800,  600, 16 },
                  { 0x115,  800,  600, 24 },
                  { 0x116, 1024,  768, 15 },
                  { 0x117, 1024,  768, 16 },
                  { 0x118, 1024,  768, 24 },
                  { 0x119, 1280, 1024, 15 },
                  { 0x11a, 1280, 1024, 16 },
                  { 0x11b, 1280, 1024, 24 },
                  { 0x221,  640,  480, 32 },
                  { 0x222,  800,  600, 32 },
                  { 0x223, 1024,  768, 32 },
                  { 0x224, 1280, 1024, 32 },
                  { 0xFFFF, 0, 0, 0 } };

static VESAWORD *Modes;
static struct _ModeInfo *ModeInfos = _ModeInfos;

static int  activeindex   = 0;
static int  screenLocked  = 0;

static HDIVE      hDive;
static HPIPE      hpipe;
static HMTX       hmtxLock;
static VESAWORD   currentBank;
static PVOID      graphBuffer;
static DATABUFFER *DataBuffer;
static VESASWITCH swfunc = NULL;

static VESACHAR palette[256][3]
   = { {  0,  0,  0 }, {  0,  0, 42 }, {  0, 42,  0 }, {  0, 42, 42 },
       { 42,  0,  0 }, { 42,  0, 42 }, { 42, 21,  0 }, { 42, 42, 42 },
       { 21, 21, 21 }, { 21, 21, 63 }, { 21, 63, 21 }, { 21, 63, 63 },
       { 63, 21, 21 }, { 63, 21, 63 }, { 63, 63, 21 }, { 63, 63, 63 },
       {  0,  0,  0 }, {  5,  5,  5 }, {  8,  8,  8 }, { 11, 11, 11 },
       { 14, 14, 14 }, { 17, 17, 17 }, { 20, 20, 20 }, { 24, 24, 24 },
       { 28, 28, 28 }, { 32, 32, 32 }, { 36, 36, 36 }, { 40, 40, 40 },
       { 45, 45, 45 }, { 50, 50, 50 }, { 56, 56, 56 }, { 63, 63, 63 },
       {  0,  0, 63 }, { 16,  0, 63 }, { 31,  0, 63 }, { 47,  0, 63 },
       { 63,  0, 63 }, { 63,  0, 47 }, { 63,  0, 31 }, { 63,  0, 16 },
       { 63,  0,  0 }, { 63, 16,  0 }, { 63, 31,  0 }, { 63, 47,  0 },
       { 63, 63,  0 }, { 47, 63,  0 }, { 31, 63,  0 }, { 16, 63,  0 },
       {  0, 63,  0 }, {  0, 63, 16 }, {  0, 63, 31 }, {  0, 63, 47 },
       {  0, 63, 63 }, {  0, 47, 63 }, {  0, 31, 63 }, {  0, 16, 63 },
       { 31, 31, 63 }, { 39, 31, 63 }, { 47, 31, 63 }, { 55, 31, 63 },
       { 63, 31, 63 }, { 63, 31, 55 }, { 63, 31, 47 }, { 63, 31, 39 },
       { 63, 31, 31 }, { 63, 39, 31 }, { 63, 47, 31 }, { 63, 55, 31 },
       { 63, 63, 31 }, { 55, 63, 31 }, { 47, 63, 31 }, { 39, 63, 31 },
       { 31, 63, 31 }, { 31, 63, 39 }, { 31, 63, 47 }, { 31, 63, 55 },
       { 31, 63, 63 }, { 31, 55, 63 }, { 31, 47, 63 }, { 31, 39, 63 },
       { 45, 45, 63 }, { 49, 45, 63 }, { 54, 45, 63 }, { 58, 45, 63 },
       { 63, 45, 63 }, { 63, 45, 58 }, { 63, 45, 54 }, { 63, 45, 49 },
       { 63, 45, 45 }, { 63, 49, 45 }, { 63, 54, 45 }, { 63, 58, 45 },
       { 63, 63, 45 }, { 58, 63, 45 }, { 54, 63, 45 }, { 49, 63, 45 },
       { 45, 63, 45 }, { 45, 63, 49 }, { 45, 63, 54 }, { 45, 63, 58 },
       { 45, 63, 63 }, { 45, 58, 63 }, { 45, 54, 63 }, { 45, 49, 63 },
       {  0,  0, 28 }, {  7,  0, 28 }, { 14,  0, 28 }, { 21,  0, 28 },
       { 28,  0, 28 }, { 28,  0, 21 }, { 28,  0, 14 }, { 28,  0,  7 },
       { 28,  0,  0 }, { 28,  7,  0 }, { 28, 14,  0 }, { 28, 21,  0 },
       { 28, 29,  0 }, { 21, 28,  0 }, { 14, 28,  0 }, {  7, 28,  0 },
       {  0, 28,  0 }, {  0, 28,  7 }, {  0, 28, 14 }, {  0, 28, 21 },
       {  0, 28, 28 }, {  0, 21, 28 }, {  0, 14, 28 }, {  0,  7, 28 },
       { 14, 14, 28 }, { 17, 14, 28 }, { 21, 14, 28 }, { 24, 14, 28 },
       { 28, 14, 28 }, { 28, 14, 24 }, { 28, 14, 21 }, { 28, 14, 17 },
       { 28, 14, 14 }, { 28, 17, 14 }, { 28, 21, 14 }, { 28, 24, 14 },
       { 28, 28, 14 }, { 24, 28, 14 }, { 21, 28, 14 }, { 17, 28, 14 },
       { 14, 28, 14 }, { 14, 28, 17 }, { 14, 28, 21 }, { 14, 28, 24 },
       { 14, 28, 28 }, { 14, 24, 28 }, { 14, 21, 28 }, { 14, 17, 28 },
       { 20, 20, 28 }, { 22, 20, 28 }, { 24, 20, 28 }, { 26, 20, 28 },
       { 28, 20, 28 }, { 28, 20, 26 }, { 28, 20, 24 }, { 28, 20, 22 },
       { 28, 20, 20 }, { 28, 22, 20 }, { 28, 24, 20 }, { 28, 26, 20 },
       { 28, 28, 20 }, { 26, 28, 20 }, { 24, 28, 20 }, { 22, 28, 20 },
       { 20, 28, 20 }, { 20, 28, 22 }, { 20, 28, 24 }, { 20, 28, 26 },
       { 20, 28, 28 }, { 20, 26, 28 }, { 20, 24, 28 }, { 20, 22, 28 },
       {  0,  0, 16 }, {  4,  0, 16 }, {  8,  0, 16 }, { 12,  0, 16 },
       { 16,  0, 16 }, { 16,  0, 12 }, { 16,  0,  8 }, { 16,  0,  4 },
       { 16,  0,  0 }, { 16,  4,  0 }, { 16,  8,  0 }, { 16, 12,  0 },
       { 16, 16,  0 }, { 12, 16,  0 }, {  8, 16,  0 }, {  4, 16,  0 },
       {  0, 16,  0 }, {  0, 16,  4 }, {  0, 16,  8 }, {  0, 16, 12 },
       {  0, 16, 16 }, {  0, 12, 16 }, {  0,  8, 16 }, {  0,  4, 16 },
       {  8,  8, 16 }, { 10,  8, 16 }, { 12,  8, 16 }, { 14,  8, 16 },
       { 16,  8, 16 }, { 16,  8, 14 }, { 16,  8, 12 }, { 16,  8, 10 },
       { 16,  8,  8 }, { 16, 10,  8 }, { 16, 12,  8 }, { 16, 14,  8 },
       { 16, 16,  8 }, { 14, 16,  8 }, { 12, 16,  8 }, { 10, 16,  8 },
       {  8, 16,  8 }, {  8, 16, 10 }, {  8, 16, 12 }, {  8, 16, 14 },
       {  8, 16, 16 }, {  8, 14, 16 }, {  8, 12, 16 }, {  8, 10, 16 },
       { 11, 11, 16 }, { 12, 11, 16 }, { 13, 11, 16 }, { 15, 11, 16 },
       { 16, 11, 16 }, { 16, 11, 15 }, { 16, 11, 13 }, { 16, 11, 12 },
       { 16, 11, 11 }, { 16, 12, 11 }, { 16, 13, 11 }, { 16, 15, 11 },
       { 16, 16, 11 }, { 15, 16, 11 }, { 13, 16, 11 }, { 12, 16, 11 },
       { 11, 16, 11 }, { 11, 16, 12 }, { 11, 16, 13 }, { 11, 16, 15 },
       { 11, 16, 16 }, { 11, 15, 16 }, { 11, 13, 16 }, { 11, 12, 16 },
       {  0,  0,  0 }, {  0,  0,  0 }, {  0,  0,  0 }, {  0,  0,  0 },
       {  0,  0,  0 }, {  0,  0,  0 }, {  0,  0,  0 }, {  0,  0,  0 } };

static int palSet = FALSE;

void ResetPalette() {
  if (palSet) {
    HPS hps;
    HDC hdc;

    hps = WinGetPS (HWND_DESKTOP);
    hdc = GpiQueryDevice (hps);
    GreUnrealizeColorTable(hdc);
    WinInvalidateRect (HWND_DESKTOP, (PRECTL)NULL, TRUE);
    WinReleasePS (hps);
    palSet = FALSE;
  }
}

void SetPalette() {
  HPS   hps;
  HDC   hdc;
  ULONG RGB2Entries[256];
  int   i;

  for (i = 0; i < 256; i++)
    RGB2Entries[i] = (palette[i][0] << 18) +
                     (palette[i][1] << 10) +
                     (palette[i][2] << 2);
  hps = WinGetPS(HWND_DESKTOP);
  hdc = GpiQueryDevice(hps);
  GpiCreateLogColorTable(hps, LCOL_PURECOLOR | LCOL_REALIZABLE,
                         LCOLF_CONSECRGB, 0, 256, (PLONG)RGB2Entries);
  GreRealizeColorTable(hdc);
  WinInvalidateRect(HWND_DESKTOP, (PRECTL)NULL, TRUE);
  WinReleasePS(hps);
  palSet = TRUE;
}

VESABOOL OS2VesaSetWindow(VESACHAR Window, VESAWORD Address)
{
  currentBank = Address;
  return(DiveSwitchBank(hDive, Address) == DIVE_SUCCESS);
}

VESABOOL OS2VesaGetWindow(VESACHAR Window, PVESAWORD Address)
{
  *Address = currentBank;
  return(TRUE);
}

VESABOOL OS2VesaGetInfo(PVESAINFO pVesaInfo)
{
  pVesaInfo->Version     = 0x102;
  pVesaInfo->OEMName     = "OS/2 VESA Driver by Johannes Martin";
  pVesaInfo->Modes       = Modes;
  pVesaInfo->MemoryBanks = 0;
  return(TRUE);
}

VESABOOL OS2VesaGetModeInfo(VESAWORD Mode, PVESAMODEINFO pVesaModeInfo)
{
  int i;

  Mode &= 0x7fff;
  for (i = 0; Modes[i] != Mode; i++)
    if (Modes[i] == 0xFFFF)
      return(FALSE);
  if (i == 0)
    *pVesaModeInfo = ModeInfos[0].Vesa;
  else {
    int j;

    for (j = 0; Modes[i] != vesaModes[j].modeNum; j++)
      if (vesaModes[j].modeNum == 0xFFFF)
        return(FALSE);
    *pVesaModeInfo = ModeInfos[1].Vesa;
    pVesaModeInfo->Width        = vesaModes[j].width;
    pVesaModeInfo->Height       = vesaModes[j].height;
  }
  pVesaModeInfo->WindowAStart = graphBuffer;
  pVesaModeInfo->WindowBStart = graphBuffer;
  return(TRUE);
}

VESABOOL OS2VesaSetMode(VESAWORD Mode)
{
  if (activeindex) {
    ULONG count;
    DosResetEventSem(DataBuffer->hev1, &count);
    WinPostMsg(DataBuffer->hwnd, WM_CLOSEWIN, 0, 0);
    DosWaitEventSem(DataBuffer->hev1, SEM_INDEFINITE_WAIT);
  }

  OS2VesaSetKeyboard(FALSE);
  DataBuffer->inGraphMode = FALSE;

  for (activeindex = 0; Modes[activeindex] != (Mode & 0x7fff); activeindex++)
    if (Modes[activeindex] == 0xFFFF) {
      activeindex = 0;
      return FALSE;
    }

  if (activeindex) {
    ULONG count;

    DataBuffer->inGraphMode = TRUE;
    DosResetEventSem(DataBuffer->hev1, &count);
    WinPostMsg(DataBuffer->hwnd, WM_OPENWIN, 0, 0);
    DosWaitEventSem(DataBuffer->hev1, SEM_INDEFINITE_WAIT);
    if ((Mode & 0x8000) == 0) {
      int LPWindow = ModeInfos[1].Vesa.WindowSize * 1024 / ModeInfos[1].Vesa.BytesPerScanline;
      int WinCount = vesaModes[0].height / LPWindow
                   + (vesaModes[0].height % LPWindow != 0);
      int WinFact  = ModeInfos[1].Vesa.WindowSize / ModeInfos[1].Vesa.WindowGranularity;
      int i;

//      printf("Grtafbuffer= %p, winsize= %d, winfak= %d \n",graphBuffer, ModeInfos[1].Vesa.WindowSize,WinFact);

      OS2VesaLockScreen(TRUE);
      for (i = 0; i < WinCount; i++) {
        OS2VesaSetWindow(0, i * WinFact);
        memset(graphBuffer, 0, ModeInfos[1].Vesa.WindowSize * 1024);
      }
      OS2VesaUnlockScreen();
    }
  }
  return TRUE;
}

VESABOOL OS2VesaGetMode(PVESAWORD Mode)
{
  *Mode = Modes[activeindex];
  return(TRUE);
}

VESABOOL OS2VesaSetScanlineLength(VESAWORD NumberOfPixels)
{
  return(NumberOfPixels == ModeInfos[activeindex != 0].Vesa.BytesPerScanline * 8 /
                           ModeInfos[activeindex != 0].Vesa.NumberOfBitsPerPixel);
}

VESABOOL OS2VesaGetScanlineLength(PVESAWORD BytesPerScanline,
                                  PVESAWORD NumberOfPixels,
                                  PVESAWORD NumberOfScanlines)
{
  *BytesPerScanline *= ModeInfos[activeindex != 0].Vesa.BytesPerScanline;
  *NumberOfPixels    = activeindex ? vesaModes[0].width : 640;
  *NumberOfScanlines = activeindex ? vesaModes[0].height : 400;
  return(TRUE);
}

static VOID APIENTRY RedrawWaitThread(ULONG Param)
{
  ULONG count;
  VESAWORD bank = 0;

  while (1) {
    DosWaitEventSem(DataBuffer->hev2, SEM_INDEFINITE_WAIT);
    if (DataBuffer->inGraphMode) {
      if (DataBuffer->visible) {
        if (ModeInfos[1].Vesa.NumberOfBitsPerPixel == 8)
          SetPalette();
        OS2VesaSetWindow(0, bank);
      } else
        DosRequestMutexSem(hmtxLock, SEM_INDEFINITE_WAIT);
      if (swfunc) {
        RECTL rectl;

        rectl.xLeft   = 0;
        rectl.yBottom = 0;
        rectl.xRight  = vesaModes[0].width;
        rectl.yTop    = vesaModes[0].height;
        DiveAcquireFrameBuffer(hDive, &rectl);
        screenLocked++;
        swfunc(DataBuffer->visible);
        screenLocked--;
        DiveDeacquireFrameBuffer(hDive);
      }
      if (!DataBuffer->visible) {
        if (ModeInfos[1].Vesa.NumberOfBitsPerPixel == 8)
          ResetPalette();
        OS2VesaGetWindow(0, &bank);
      } else
        DosReleaseMutexSem(hmtxLock);
    }
    DosResetEventSem(DataBuffer->hev2, &count);
    DosPostEventSem(DataBuffer->hev3);
  }
}

VESABOOL OS2VesaSetSwitchFunc(VESASWITCH SwitchFunc)
{
  swfunc = SwitchFunc;
  return(TRUE);
}

VESABOOL OS2VesaLockScreen(VESACHAR wait)
{
  if (screenLocked == 0) {
        RECTL rectl;

        rectl.xLeft   = 0;
        rectl.yBottom = 0;
        rectl.xRight  = vesaModes[activeindex].width;
        rectl.yTop    = vesaModes[activeindex].height;
     DiveAcquireFrameBuffer(hDive, &rectl);
  }
  screenLocked++;
  return(TRUE);
}

VESABOOL OS2VesaUnlockScreen(void)
{
  if (!screenLocked)
    return(FALSE);
  if (--screenLocked == 0) {
    DiveDeacquireFrameBuffer(hDive);
    DosReleaseMutexSem(hmtxLock);
  }
  return(TRUE);
}

VESABOOL OS2VesaSetModified(void)
{
  return(TRUE);
}

VESABOOL OS2VesaSetPalette(VESAWORD   StartIndex,
                           VESAWORD   NumberOfEntries,
                           PCVESACHAR Palette,
                           VESABOOL   Wait)
{
  if ((NumberOfEntries >= 1) && (StartIndex + NumberOfEntries <= 256)) {
    memcpy(palette + StartIndex * 3, Palette, NumberOfEntries * 3);
    if (DataBuffer->visible) {
      DosSleep(1);
      SetPalette();
    }
    return(TRUE);
  }
  return(FALSE);
}

VESABOOL OS2VesaGetPalette(VESAWORD  StartIndex,
                           VESAWORD  NumberOfEntries,
                           PVESACHAR Palette)
{
  if ((NumberOfEntries >= 1) && (StartIndex + NumberOfEntries <= 256)) {
    memcpy(Palette, palette + StartIndex * 3, NumberOfEntries * 3);
    return(TRUE);
  }
  return(FALSE);
}

static ULONG LastCode = -1;

VESABOOL OS2VesaGetCharacter(PVESACHAR pchar)
{
  if (DataBuffer->inGraphMode) {
    ULONG read;
    DosRead(hpipe, pchar, sizeof(VESACHAR), &read);
    return(read == sizeof(VESACHAR));
  }
  if (LastCode != -1) {
      *pchar   = LastCode;
      LastCode = -1;
  } else
    *pchar = _read_kbd(0, 1, 0);
  return(TRUE);
}

VESABOOL OS2VesaKeyboardHit(void)
{
  if (DataBuffer->inGraphMode) {
    VESACHAR  byte;
    ULONG     read, status;
    AVAILDATA avail;

    DosPeekNPipe(hpipe, &byte, sizeof(VESACHAR), &read, &avail, &status);
    return(read == sizeof(VESACHAR));
  }

  if (LastCode == -1)
    LastCode = _read_kbd(0, 0, 0);
  return(LastCode != -1);
}

VESABOOL OS2VesaSetKeyboard(VESABOOL raw)
{
  if (raw && !DataBuffer->inGraphMode)
    return(FALSE);
  if (DataBuffer->rawKeyboard != raw) {
    VESACHAR c;

    while (OS2VesaKeyboardHit()) {
      OS2VesaGetCharacter(&c);
    }
    DataBuffer->rawKeyboard = raw;
  }
  return(TRUE);
}

VESABOOL OS2MouseOpen(void)
{
  return(TRUE);
}

VESABOOL OS2MouseClose(void)
{
  return(TRUE);
}

VESABOOL OS2MouseShow(void)
{
  return(TRUE);
}

VESABOOL OS2MouseHide(void)
{
  return(TRUE);
}

VESABOOL OS2MouseSetPosition(VESAWORD x, VESAWORD y)
{
  WinPostMsg(DataBuffer->hwnd, WM_SETMOUSEPOS, MPFROM2SHORT(x, y), NULL);
  return(FALSE);
}

VESABOOL OS2MouseGetPosition(PVESAWORD x, PVESAWORD y)
{
  *x = DataBuffer->mousex;
  *y = DataBuffer->mousey;
  return(TRUE);
}

VESABOOL OS2MouseGetEvent(PVESAWORD x, PVESAWORD y, PVESAWORD status, VESABOOL wait)
{
  static VESAWORD oldx = 0;
  static VESAWORD oldy = 0;
  static VESAWORD oldstatus = 0;
  VESABOOL rc;

  do
    {
      *x      = DataBuffer->mousex;
      *y      = DataBuffer->mousey;
      *status = DataBuffer->mousestatus;
      rc = (*x != oldx) || (*y != oldy) || (*status != oldstatus);
      if (wait && !rc)
        DosSleep(100);
    }
  while (wait && !rc);
  oldx      = *x;
  oldy      = *y;
  oldstatus = *status;
  return(rc);
}

int _CRT_init(void);
void _CRT_term(void);

unsigned long _DLL_InitTerm(unsigned long modhandle, unsigned long flag)
{
  if (!flag)
    {
      int i, j;
      DIVE_CAPS diveCaps;
      STARTDATA StartData;
      char  params[20], prgname[256];
      ULONG sid, rc;
      PID   pid;
      PPIB  ppib;
      PTIB  ptib;
      TID   tid;

      if (_CRT_init())
        return(FALSE);

      memset(&diveCaps, 0, sizeof(diveCaps));
      diveCaps.ulStructLen = sizeof(diveCaps);
      DiveQueryCaps(&diveCaps, DIVE_BUFFER_SCREEN);
      if (!diveCaps.fScreenDirect)
        return(FALSE);
      if (DiveOpen(&hDive, FALSE, &graphBuffer) != DIVE_SUCCESS)
        return(FALSE);

      vesaModes[0].width  = diveCaps.ulHorizontalResolution;
      vesaModes[0].height = diveCaps.ulVerticalResolution;
      vesaModes[0].bits   = diveCaps.ulDepth;
      j = 2;
      for (i = 0; vesaModes[i].modeNum != 0xFFFF; i++)
        if ((vesaModes[i].width  <= diveCaps.ulHorizontalResolution) &&
            (vesaModes[i].height <= diveCaps.ulVerticalResolution) &&
            (vesaModes[i].bits   == diveCaps.ulDepth))
          j++;
      Modes = malloc(j * sizeof(VESAWORD));
      Modes[j = 0] = 0x03;
      for (i = 0; vesaModes[i].modeNum != 0xFFFF; i++)
        if ((vesaModes[i].width  <= diveCaps.ulHorizontalResolution) &&
            (vesaModes[i].height <= diveCaps.ulVerticalResolution) &&
            (vesaModes[i].bits   == diveCaps.ulDepth))
          Modes[++j] = vesaModes[i].modeNum;
      Modes[++j] = 0xFFFF;

      ModeInfos[1].Vesa.WindowGranularity    = diveCaps.ulApertureSize / 1024 + (diveCaps.ulApertureSize % 1024 != 0);
      ModeInfos[1].Vesa.WindowSize           = ModeInfos[1].Vesa.WindowGranularity;
      ModeInfos[1].Vesa.BytesPerScanline     = diveCaps.ulScanLineBytes;
      ModeInfos[1].Vesa.Width                = diveCaps.ulHorizontalResolution;
      ModeInfos[1].Vesa.Height               = diveCaps.ulVerticalResolution;
      ModeInfos[1].Vesa.NumberOfBitsPerPixel = diveCaps.ulDepth;
      switch (diveCaps.ulDepth) {
        case 8:
          ModeInfos[1].Vesa.MemoryModelType = 4;
          break;
        case 15:
          ModeInfos[1].Vesa.MemoryModelType      = 6;
          ModeInfos[1].Vesa.RedMaskSize          = 5;
          ModeInfos[1].Vesa.RedMaskPosition      = 10;
          ModeInfos[1].Vesa.GreenMaskSize        = 5;
          ModeInfos[1].Vesa.GreenMaskPosition    = 5;
          ModeInfos[1].Vesa.BlueMaskSize         = 5;
          ModeInfos[1].Vesa.BlueMaskPosition     = 0;
          ModeInfos[1].Vesa.ReservedMaskSize     = 1;
          ModeInfos[1].Vesa.ReservedMaskPosition = 15;
          break;
        case 16:
          ModeInfos[1].Vesa.MemoryModelType      = 6;
          ModeInfos[1].Vesa.RedMaskSize          = 5;
          ModeInfos[1].Vesa.RedMaskPosition      = 11;
          ModeInfos[1].Vesa.GreenMaskSize        = 6;
          ModeInfos[1].Vesa.GreenMaskPosition    = 5;
          ModeInfos[1].Vesa.BlueMaskSize         = 5;
          ModeInfos[1].Vesa.BlueMaskPosition     = 0;
          ModeInfos[1].Vesa.ReservedMaskSize     = 0;
          ModeInfos[1].Vesa.ReservedMaskPosition = 0;
          break;
        case 24:
          ModeInfos[1].Vesa.MemoryModelType      = 6;
          ModeInfos[1].Vesa.RedMaskSize          = 8;
          ModeInfos[1].Vesa.RedMaskPosition      = 16;
          ModeInfos[1].Vesa.GreenMaskSize        = 8;
          ModeInfos[1].Vesa.GreenMaskPosition    = 8;
          ModeInfos[1].Vesa.BlueMaskSize         = 8;
          ModeInfos[1].Vesa.BlueMaskPosition     = 0;
          ModeInfos[1].Vesa.ReservedMaskSize     = 0;
          ModeInfos[1].Vesa.ReservedMaskPosition = 0;
          break;
        case 32:
          ModeInfos[1].Vesa.MemoryModelType      = 6;
          ModeInfos[1].Vesa.RedMaskSize          = 8;
          ModeInfos[1].Vesa.RedMaskPosition      = 16;
          ModeInfos[1].Vesa.GreenMaskSize        = 8;
          ModeInfos[1].Vesa.GreenMaskPosition    = 8;
          ModeInfos[1].Vesa.BlueMaskSize         = 8;
          ModeInfos[1].Vesa.BlueMaskPosition     = 0;
          ModeInfos[1].Vesa.ReservedMaskSize     = 8;
          ModeInfos[1].Vesa.ReservedMaskPosition = 24;
          break;
      }

      DosAllocSharedMem((PPVOID) &DataBuffer, NULL, sizeof(DATABUFFER),
                        OBJ_GETTABLE | PAG_READ | PAG_WRITE | PAG_COMMIT);
      memset(DataBuffer, 0, sizeof(DATABUFFER));
      DataBuffer->hwnd        = 0;
      DataBuffer->inGraphMode = FALSE;
      DataBuffer->visible     = FALSE;
      DataBuffer->rawKeyboard = FALSE;
      DosCreateEventSem(NULL, &DataBuffer->hev1, DC_SEM_SHARED, 0);
      DosCreateEventSem(NULL, &DataBuffer->hev2, DC_SEM_SHARED, 0);
      DosCreateEventSem(NULL, &DataBuffer->hev3, DC_SEM_SHARED, 0);
      DosCreateMutexSem(NULL, &hmtxLock, 0, FALSE);
      DosCreateThread(&tid, RedrawWaitThread, 0, 0, 32 * 1024);
      DosGetInfoBlocks(&ptib, &ppib);
      strcpy(DataBuffer->pipeName, "\\PIPE\\VESA_DIV.");
      _itoa(ppib->pib_ulpid, DataBuffer->pipeName + 15, 16);
      DosCreateNPipe(DataBuffer->pipeName, &hpipe, NP_ACCESS_INBOUND, 1, 0, 256, 0);
      memset(&StartData, 0, sizeof(StartData));
      StartData.Length      = sizeof(StartData);
      StartData.Related     = 0;
      StartData.FgBg        = 0;
      StartData.TraceOpt    = 0;
      StartData.PgmTitle    = NULL;
      if (DosScanEnv("DIVESCR", &StartData.PgmName))
        {
          if (DosSearchPath(SEARCH_IGNORENETERRS | SEARCH_ENVIRONMENT | SEARCH_CUR_DIRECTORY,
                            "PATH", "DIVESCR.EXE", prgname, sizeof(prgname)))
            StartData.PgmName = "DIVESCR.EXE";
          else
            StartData.PgmName = prgname;
        }
      StartData.PgmInputs   = _itoa((ULONG) DataBuffer, params, 10);
      StartData.TermQ       = NULL;
      StartData.Environment = NULL;
      StartData.InheritOpt  = 1;
      StartData.SessionType = SSF_TYPE_PM;
      StartData.IconFile    = 0;
      StartData.PgmHandle   = 0;
      if (((rc = DosStartSession(&StartData, &sid, &pid)) != 0) &&
          (rc != ERROR_SMG_START_IN_BACKGROUND))
        {
          static char message[] = " not found.\r\n";
          ULONG written;

          DosWrite(2, StartData.PgmName, strlen(StartData.PgmName), &written);
          DosWrite(2, message, strlen(message), &written);
          return(FALSE);
        }
      DosConnectNPipe(hpipe);
      DosWaitEventSem(DataBuffer->hev1, SEM_INDEFINITE_WAIT);
      return(TRUE);
    }
  if (DataBuffer->hwnd != 0)
    {
      ULONG count;

      DosResetEventSem(DataBuffer->hev1, &count);
      if (DataBuffer->visible) {
        WinPostMsg(DataBuffer->hwnd, WM_FINISH, NULL, NULL);
        DosWaitEventSem(DataBuffer->hev2, 10000);
        if (DataBuffer->inGraphMode &&
            ModeInfos[1].Vesa.NumberOfBitsPerPixel == 8)
          ResetPalette();
        DosPostEventSem(DataBuffer->hev3);
      } else
        WinPostMsg(DataBuffer->hwnd, WM_FINISH, NULL, NULL);
      DosWaitEventSem(DataBuffer->hev1, 10000);
    }
  DosCloseMutexSem(hmtxLock);
  DosCloseEventSem(DataBuffer->hev1);
  DosCloseEventSem(DataBuffer->hev2);
  DosCloseEventSem(DataBuffer->hev3);
  DosDisConnectNPipe(hpipe);
  DosClose(hpipe);
  DosFreeMem(DataBuffer);
  DiveClose(hDive);
  _CRT_term();
  return(TRUE);
}
