/*****************************************************************************/
/*                                                                           */
/*                           OCO SOURCE MATERIALS                            */
/*                             IBM CONFIDENTIAL                              */
/*                                    OR                                     */
/*                        IBM CONFIDENTIAL RESTRICTED                        */
/*            WHEN COMBINED WITH THE AGGREGATED OCO SOURCE MODULES           */
/*                           FOR THIS PROGRAM PRODUCT                        */
/*                                                                           */
/*    5622-585 VisualAge Generator Workgroup Services for OS/2 V2.2          */
/*    5622-587 VisualAge Generator Workgroup Services for AIX V2.2           */
/*    5639-A26 VisualAge Generator Workgroup Services for Windows NT V2.2    */
/*                     (C) COPYRIGHT IBM CORP. 1994,1996                     */
/*                                                                           */
/*****************************************************************************/
#ifndef FCWEXP_H
#define FCWEXP_H

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <math.h>
#include <memory.h>
#include <iostream.h>
#include <iomanip.h>
#include <ctype.h>

#ifdef FCW_NT
#else
#include <wcstr.h>
#endif

enum Level { Leaf, Group };


// *********************************************************************
// Declare forward references
// *********************************************************************
class Item;
class Parent;
class RPCParm;
class quadnum;
class TraceFile;
class CHA;

// *********************************************************************
// Declare global functions
// *********************************************************************
void UserOverflow();
void MaxOverflow();

// ---------------------------------------------------------------------
//  Object for literal (CHA, MIX, DBCS) parameter on CALL statement
// ---------------------------------------------------------------------
class LitParm : public Parameter
{
  public:
   LitParm( char* p );
   virtual ~LitParm();

   long ParmSize();
   char* ParmData();

   // For RPC
   int& fNumberOfRPCEntries();
   LIST(RPCParm*)& RPCList();
   void SetupRpc();

  private:
   LIST(RPCParm*) LRPCParm;                            /* List of RPC Parms*/
   int            numberOfRPCEntries;
   FCWString      s;
   void*          pData;
};


// *********************************************************************
// Item
// *********************************************************************

class Item : public Parameter
{
  public:

    char*             Data(int i=1) const;              /* Empty Array item*/
    int               Length()      const;
    int               Byte()        const;
    Bool              IsLeaf()      const;
    int               Occurs()      const;
    void              SetOccurs(int);
    int               Offset()      const;
    ItemType          Type()        const;
    Bool              TypeIsPack()  const;
    Bool              TypeIsSQL()   const;
    Bool              TypeIsDBCS()  const;
    int               Scale()       const;
    void              ConvertData( char, FILE* );

    virtual void      SetEmpty() const { };
    virtual void      SetNull() const { };
    virtual void      SetClear() { };
    virtual FCWString StringForm();
    virtual int       LengthWOTBlanks();
    int&              fNumberOfRPCEntries();
    LIST(RPCParm*)&   RPCList();
    void              SetupRpc();

  protected:

    Item(Parent&, Level, int, int, int, int);
    Item(const Item&, int);
    virtual ~Item();

    Parent&           parent() const;
    void              SetByte(int);

    long              ParmSize();      /* Level 77 item only               */
    char*             ParmData();

    void              SetType(ItemType);
    void              SetScale(int);

  private:
    int            numberOfRPCEntries;
    LIST(RPCParm*) LRPCParm;                           /* List of RPC Parms*/
    Parent&        iparent;
    char *         parm;
    Level          level;
    int            offset;
    int            length;
    int            occurs;
    int            occurs_offset;
    ItemType       type;
    int            scale;
    int            byte;
    void*          pData;
};


// ----------------------------------------------------------
//  ChaItem :  for CHA, DBCS, HEX, and MIX
// ----------------------------------------------------------

class ChaItem : public Item
{
  protected:
    ChaItem(Parent&, Level, int, int, int, int);
    ChaItem(const ChaItem&, int);
    virtual ~ChaItem();

  public:
    operator FCWString() const;
    virtual int LengthWOTBlanks();

  private:
    void*    pData;
};

int  operator == ( const char *c1, const ChaItem& c2 );
int  operator <  ( const char *c1, const ChaItem& c2 );
int  operator >  ( const char *c1, const ChaItem& c2 );

inline
int  operator != ( const char *c1, const ChaItem& c2 )
{
   return (!( c1 == c2 ));
}

inline
int  operator <= ( const char *c1, const ChaItem& c2 )
{
   return (!( c1 > c2 ));
}

inline
int  operator >= ( const char *c1, const ChaItem& c2 )
{
   return (!( c1 < c2 ));
}


// *********************************************************************
// NumItem
// *********************************************************************

class NumItem : public Item
{
  public:
    int               Decimals() const;
    int               longValue() const;
    virtual           operator quadnum () const = 0;
    virtual void      operator = (const quadnum&) =0;
    virtual FCWString StringForm();

  protected:

    NumItem(Parent&, Level, int, int, int, int, int);
    NumItem(const NumItem&, int);
    virtual ~NumItem();

  private:

    int      decimals;
    void*    pData;
};

int     operator == ( const NumItem& n, const quadnum& q );
int     operator <  ( const NumItem& n, const quadnum& q );
int     operator >  ( const NumItem& n, const quadnum& q );

inline
int     operator != (const NumItem& n, const quadnum& q)
{
   return (!( n == q ));
}

inline
int     operator >= (const NumItem& n, const quadnum& q)
{
   return (!( n < q ));
}

inline
int     operator <= (const NumItem& n, const quadnum& q)
{
   return (!( n > q ));
}

int     operator == ( const NumItem& n, const NumItem& q );
int     operator <  ( const NumItem& n, const NumItem& q );
int     operator >  ( const NumItem& n, const NumItem& q );

inline
int     operator != (const NumItem& n, const NumItem& q)
{
   return (!( n == q ));
}

inline
int     operator >= (const NumItem& n, const NumItem& q)
{
   return (!( n < q ));
}

inline
int     operator <= (const NumItem& n, const NumItem& q)
{
   return (!( n > q ));
}

inline
int     operator == ( const quadnum& n, const NumItem& q )
{
   return ( q == n );
}

inline
int     operator <  ( const quadnum& n, const NumItem& q )
{
   return ( q > n );
}

inline
int     operator >  ( const quadnum& n, const NumItem& q )
{
   return ( q < n );
}

inline
int     operator != (const quadnum& n, const NumItem& q)
{
   return ( q != n );
}

inline
int     operator >= (const quadnum& n, const NumItem& q)
{
   return ( q <= n );
}

inline
int     operator <= (const quadnum& n, const NumItem& q)
{
   return ( q >= n );
}

// ************************************************************************
// BIN2
// ************************************************************************

class BIN2 : public NumItem
{
  public:

    BIN2 (Parent&, Level, int, int, int, int, int=0);
    BIN2 (const BIN2&, int);
    BIN2 (const BIN2&, quadnum );
    virtual ~BIN2();

    void SetEmpty () const;

    virtual operator quadnum () const;
    BIN2 operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const BIN2&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);      /* arith round operator     */

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );

  private:
    void*    pData;
};


// ************************************************************************
// BIN4
// ************************************************************************

class BIN4 : public NumItem
{
  public:

    BIN4 (Parent&, Level, int, int, int, int, int=0);
    BIN4 (const BIN4&, int);
    BIN4 (const BIN4&, quadnum);
    virtual ~BIN4();

    void SetEmpty () const ;

    virtual operator quadnum () const;
    BIN4 operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const BIN4&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);      /* arith round operator     */

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );

  private:
    void*    pData;
};

// ************************************************************************
// BIN8
// ************************************************************************

class BIN8 : public NumItem
{
  public:

    BIN8 (Parent&, Level, int, int, int, int, int=0);
    BIN8 (const BIN8&, int);
    BIN8 (const BIN8&, quadnum);
    virtual ~BIN8();

    void SetEmpty () const;

    virtual operator quadnum () const;
    BIN8 operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const BIN8&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );

  private:
    void*    pData;
};

// ************************************************************************
// NUM
// ************************************************************************

class NUM : public NumItem
{
  public:

    NUM (Parent&, Level, int, int, int, int, int=0);
    NUM (const NUM&, int);
    NUM (const NUM&, quadnum);
    virtual ~NUM();

    void SetEmpty () const;

    virtual operator quadnum () const;
    NUM operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const NUM&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );
    int Contains( const CHA& v,      int=1 );
    int Contains( const CHA& v,      NumItem& itm );

    // Interactions with CHA class
    friend class CHA;
    void operator = (const CHA& v);
    int operator == (const CHA& v) const;
    int operator != (const CHA& v) const;
    int operator >  (const CHA& v) const;
    int operator <  (const CHA& v) const;
    int operator >= (const CHA& v) const;
    int operator <= (const CHA& v) const;

  private:
    void*    pData;
};

inline
int NUM :: operator != (const CHA& v) const
{
   return (!( *this == v ));
}

inline
int NUM :: operator <= (const CHA& v) const
{
   return (!( *this > v ));
}

inline
int NUM :: operator >= (const CHA& v) const
{
   return (!( *this < v ));
}


// ************************************************************************
// NUMC
// ************************************************************************

class NUMC : public NumItem
{
  public:

    NUMC (Parent&, Level, int, int, int, int, int=0);
    NUMC (const NUMC&, int);
    NUMC (const NUMC&, quadnum);
    virtual ~NUMC();

    void SetEmpty () const;

    virtual operator quadnum () const;
    NUMC operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const NUMC&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );

  private:
    void*    pData;
};

// ************************************************************************
// PACK
// ************************************************************************

class PACK : public NumItem
{
  public:

    PACK (Parent&, Level, int, int, int, int, int=0);
    PACK (const PACK&, int);
    PACK (const PACK&, quadnum);
    virtual ~PACK();

    void SetEmpty () const;

    virtual operator quadnum () const;
    PACK operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const PACK&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );

  private:

    int      bytes;
    void*    pData;
};

// ************************************************************************
// PACF
// ************************************************************************

class PACF : public NumItem
{
  public:

    PACF (Parent&, Level, int, int, int, int, int=0);
    PACF (const PACF&, int);
    PACF (const PACF&, quadnum);
    virtual ~PACF();

    void SetEmpty () const;

    virtual operator quadnum () const;
    PACF operator [] (quadnum) const;

    quadnum operator - () const;
    quadnum operator + () const;

    void operator = (const PACF&);
    virtual void operator = (const quadnum& v);
    void operator <<= (const quadnum& v);

    int Contains( const quadnum& v,  int=1 );
    int Contains( const quadnum& v,  NumItem& itm );

  private:

    int      bytes;
    void*    pData;
};

// ************************************************************************
// HEX
// ************************************************************************

class HEX : public ChaItem
{
  public:

    HEX (Parent&, Level, int, int, int, int);
    HEX (const HEX&, int);
    HEX (const HEX&, quadnum);
    virtual ~HEX();

    HEX operator [] (quadnum) const;

    void SetEmpty () const;

    // Interactions with other HEX objects
    void operator =  (const HEX& v);
    int  operator == (const HEX& v) const;
    int  operator != (const HEX& v) const;
    int  operator >  (const HEX& v) const;
    int  operator <  (const HEX& v) const;
    int  operator >= (const HEX& v) const;
    int  operator <= (const HEX& v) const;

    // Interactions with CHA objects
    friend class CHA;
    void operator =  (const CHA& v);
    int  operator == (const CHA& v) const;
    int  operator != (const CHA& v) const;
    int  operator >  (const CHA& v) const;
    int  operator <  (const CHA& v) const;
    int  operator >= (const CHA& v) const;
    int  operator <= (const CHA& v) const;

    // Interactions with character literals
    void operator =  (const char *v);
    int  operator == (const char *v) const;
    int  operator != (const char *v) const;
    int  operator >  (const char *v) const;
    int  operator <  (const char *v) const;
    int  operator >= (const char *v) const;
    int  operator <= (const char *v) const;

    int Contains( const FCWString v, const quadnum i );

  private:
    void*    pData;
};

inline
int EXPORT HEX :: operator != (const HEX& v) const
{
   return (!( *this == v ));
}

inline
int EXPORT HEX :: operator <= (const HEX& v) const
{
   return (!( *this > v ));
}

inline
int EXPORT HEX :: operator >= (const HEX& v) const
{
   return (!( *this < v ));
}

inline
int HEX :: operator != (const char* v) const
{
   return (!( *this == v ));
}

inline
int HEX :: operator <= (const char* v) const
{
   return (!( *this > v ));
}

inline
int HEX :: operator >= (const char* v) const
{
   return (!( *this < v ));
}

inline
int operator == ( const char* v, const HEX& h )
{
    return ( h == v );
}

inline
int operator != ( const char* v, const HEX& h )
{
    return ( h != v );
}

inline
int operator >  ( const char* v, const HEX& h )
{
    return ( h <  v );
}

inline
int operator <  ( const char* v, const HEX& h )
{
    return ( h >  v );
}

inline
int operator >= ( const char* v, const HEX& h )
{
    return ( h <= v );
}

inline
int operator <= ( const char* v, const HEX& h )
{
    return ( h >= v );
}

// ************************************************************************
// CHA
// ************************************************************************

class CHA : public ChaItem
{
  public:

    CHA (Parent&, Level, int, int, int, int);
    CHA (const CHA&, int);
    CHA (const CHA&, quadnum);
    virtual ~CHA();

    CHA operator [] (quadnum) const;
    CHA operator [] (int) const;

    void SetEmpty () const;
    Bool IsNumeric () const;
    Bool IsBlank();

    void operator = (const CHA& v);
    int operator == (const CHA& v) const;
    int operator != (const CHA& v) const;
    int operator >  (const CHA& v) const;
    int operator <  (const CHA& v) const;
    int operator >= (const CHA& v) const;
    int operator <= (const CHA& v) const;

    friend class HEX;
    void operator = (const HEX& v);
    int operator == (const HEX& v) const;
    int operator != (const HEX& v) const;
    int operator >  (const HEX& v) const;
    int operator <  (const HEX& v) const;
    int operator >= (const HEX& v) const;
    int operator <= (const HEX& v) const;

    friend class MIX;
    void operator = (const MIX& v);
    int operator == (const MIX& v) const;
    int operator != (const MIX& v) const;
    int operator >  (const MIX& v) const;
    int operator <  (const MIX& v) const;
    int operator >= (const MIX& v) const;
    int operator <= (const MIX& v) const;

    friend class NUM;
    void operator = (const NUM& v);
    int operator == (const NUM& v) const;
    int operator != (const NUM& v) const;
    int operator >  (const NUM& v) const;
    int operator <  (const NUM& v) const;
    int operator >= (const NUM& v) const;
    int operator <= (const NUM& v) const;

    void operator = (const char *v);
    int operator == (const char *v) const;
    int operator != (const char *v) const;
    int operator >  (const char *v) const;
    int operator <  (const char *v) const;
    int operator >= (const char *v) const;
    int operator <= (const char *v) const;

    void operator =  (const int v );
    int  operator == (const int v ) const;
    int  operator != (const int v ) const;
    int  operator >  (const int v ) const;
    int  operator <  (const int v ) const;
    int  operator >= (const int v ) const;
    int  operator <= (const int v ) const;

    void operator =  (const quadnum& v );
    int  operator == (const quadnum& v ) const;
    int  operator != (const quadnum& v ) const;
    int  operator >  (const quadnum& v ) const;
    int  operator <  (const quadnum& v ) const;
    int  operator >= (const quadnum& v ) const;
    int  operator <= (const quadnum& v ) const;

    void operator = (const FCWString& v );

    int Contains( const FCWString v, const quadnum i );
    int Contains( const NUM&      v, const quadnum i );

  private:
    Bool IsValidNumeric () const;
    void*    pData;
};

inline
int CHA :: operator != (const CHA& v) const
{
   return (!( *this == v ));
}

inline
int CHA :: operator <= (const CHA& v) const
{
   return (!( *this > v ));
}

inline
int CHA :: operator >= (const CHA& v) const
{
   return (!( *this < v ));
}

inline
int HEX :: operator == (const CHA& v) const
{
   return ( *this == FCWString(v) );
}

inline
int HEX :: operator != (const CHA& v) const
{
   return ( *this != FCWString(v) );
}

inline
int HEX :: operator < (const CHA& v) const
{
   return ( *this < FCWString(v) );
}

inline
int HEX :: operator > (const CHA& v) const
{
   return ( *this > FCWString(v) );
}

inline
int HEX :: operator <= (const CHA& v) const
{
   return ( *this <= FCWString(v) );
}

inline
int HEX :: operator >= (const CHA& v) const
{
   return ( *this >= FCWString(v) );
}

inline
int CHA :: operator == (const HEX& v) const
{
  return ( v == *this );
}

inline
int CHA :: operator != (const HEX& v) const
{
  return ( v != *this );
}

inline
int CHA :: operator < (const HEX& v) const
{
  return ( v > *this );
}

inline
int CHA :: operator > (const HEX& v) const
{
  return ( v < *this );
}

inline
int CHA :: operator <= (const HEX& v) const
{
  return ( v >= *this );
}

inline
int CHA :: operator >= (const HEX& v) const
{
  return ( v <= *this );
}

inline
int CHA :: operator != (const MIX& v) const
{
   return (!( *this == v ));
}

inline
int CHA :: operator <= (const MIX& v) const
{
   return (!( *this > v ));
}

inline
int CHA :: operator >= (const MIX& v) const
{
   return (!( *this < v ));
}

inline
int CHA :: operator == (const NUM& v) const
{
   return ( v == *this );
}

inline
int CHA :: operator != (const NUM& v) const
{
   return ( v != *this );
}

inline
int CHA :: operator < (const NUM& v)  const
{
   return ( v > *this );
}

inline
int CHA :: operator > (const NUM& v)  const
{
   return ( v < *this );
}

inline
int CHA :: operator <= (const NUM& v) const
{
   return ( v >= *this );
}

inline
int CHA :: operator >= (const NUM& v) const
{
   return ( v <= *this );
}

inline
int CHA :: operator != (const char *v) const
{
   return (!( *this == v ));
}

inline
int CHA :: operator <= (const char *v) const
{
   return (!( *this > v ));
}

inline
int CHA :: operator >= (const char *v) const
{
   return (!( *this < v ));
}

inline
int  CHA :: operator != (const int v ) const
{
   return (!( *this == v ));
}

inline
int  CHA :: operator >= (const int v ) const
{
   return (!( *this < v ));
}

inline
int  CHA :: operator <= (const int v ) const
{
   return (!( *this > v ));
}

inline
int CHA :: operator != (const quadnum& v) const
{
   return (!( *this == v ));
}

inline
int CHA :: operator >= (const quadnum& v) const
{
   return (!( *this < v ));
}

inline
int CHA :: operator <= (const quadnum& v) const
{
   return (!( *this > v ));
}

inline
int operator == ( const quadnum& q, const CHA& c )
{
    return ( c == q );
}

inline
int operator != ( const quadnum& q, const CHA& c )
{
    return ( c != q );
}

inline
int operator >  ( const quadnum& q, const CHA& c )
{
    return ( c <  q );
}

inline
int operator <  ( const quadnum& q, const CHA& c )
{
    return ( c >  q );
}

inline
int operator >= ( const quadnum& q, const CHA& c )
{
    return ( c <= q );
}

inline
int operator <= ( const quadnum& q, const CHA& c )
{
    return ( c >= q );
}

inline
int operator == ( const char* q, const CHA& c )
{
    return ( c == q );
}

inline
int operator != ( const char* q, const CHA& c )
{
    return ( c != q );
}

inline
int operator >  ( const char* q, const CHA& c )
{
    return ( c <  q );
}

inline
int operator <  ( const char* q, const CHA& c )
{
    return ( c >  q );
}

inline
int operator >= ( const char* q, const CHA& c )
{
    return ( c <= q );
}

inline
int operator <= ( const char* q, const CHA& c )
{
    return ( c >= q );
}

// *********************************************************************
// DBCS
// *********************************************************************

class DBCS : public ChaItem
{
  public:

    DBCS (Parent&, Level, int, int, int, int);
    DBCS (const DBCS&, int);
    DBCS (const DBCS&, quadnum);
    virtual ~DBCS();

    DBCS operator [] (quadnum) const;

    void SetEmpty () const;
    Bool IsBlank();
    virtual FCWString StringForm();

    // Interactions with other DBCS objects
    void operator =  (const DBCS& v);
    int  operator == (const DBCS& v) const;
    int  operator != (const DBCS& v) const;
    int  operator >  (const DBCS& v) const;
    int  operator <  (const DBCS& v) const;
    int  operator >= (const DBCS& v) const;
    int  operator <= (const DBCS& v) const;

    // Interactions with character literals
    void operator =  (const char *v);
    int  operator == (const char *v) const;
    int  operator != (const char *v) const;
    int  operator >  (const char *v) const;
    int  operator <  (const char *v) const;
    int  operator >= (const char *v) const;
    int  operator <= (const char *v) const;

    int Contains( const FCWString v, const quadnum i );

    virtual int LengthWOTBlanks();

  private:
    void*    pData;
};

inline
int EXPORT DBCS :: operator != (const DBCS& v) const
{
   return (!( *this == v ));
}

inline
int EXPORT DBCS :: operator <= (const DBCS& v) const
{
   return (!( *this > v ));
}

inline
int EXPORT DBCS :: operator >= (const DBCS& v) const
{
   return (!( *this < v ));
}

inline
int EXPORT DBCS :: operator != (const char* v) const
{
   return (!( *this == v ));
}

inline
int EXPORT DBCS :: operator <= (const char* v) const
{
   return (!( *this > v ));
}

inline
int EXPORT DBCS :: operator >= (const char* v) const
{
   return (!( *this < v ));
}


// *********************************************************************
// MIX
// *********************************************************************

class MIX : public ChaItem
{
  public:

    MIX (Parent&, Level, int, int, int, int);
    MIX (const MIX&, int);
    MIX (const MIX&, quadnum);
    virtual ~MIX();

    MIX operator [] (quadnum) const;

    void SetEmpty () const;
    Bool IsNumeric () const;
    Bool IsBlank();

    // Interactions with other MIX objects
    void operator =  (const MIX& v);
    int  operator == (const MIX& v) const;
    int  operator != (const MIX& v) const;
    int  operator >  (const MIX& v) const;
    int  operator <  (const MIX& v) const;
    int  operator >= (const MIX& v) const;
    int  operator <= (const MIX& v) const;

    // Interactions with CHA objects
    friend class CHA;
    void operator =  (const CHA& v);
    int  operator == (const CHA& v) const;
    int  operator != (const CHA& v) const;
    int  operator >  (const CHA& v) const;
    int  operator <  (const CHA& v) const;
    int  operator >= (const CHA& v) const;
    int  operator <= (const CHA& v) const;

    // Interactions with character literals
    void operator =  (const char *v);
    int  operator == (const char *v) const;
    int  operator != (const char *v) const;
    int  operator >  (const char *v) const;
    int  operator <  (const char *v) const;
    int  operator >= (const char *v) const;
    int  operator <= (const char *v) const;

    int Contains( const FCWString v, const quadnum i );

  private:
    Bool IsValidNumeric () const;
    void*    pData;
};

inline
int EXPORT MIX :: operator != (const MIX& v) const
{
   return (!( *this == v ));
}

inline
int EXPORT MIX :: operator <= (const MIX& v) const
{
   return (!( *this > v ));
}

inline
int EXPORT MIX :: operator >= (const MIX& v) const
{
   return (!( *this < v ));
}

inline
int EXPORT MIX :: operator != (const CHA& v) const
{
   return (!( *this == v ));
}

inline
int EXPORT MIX :: operator <= (const CHA& v) const
{
   return (!( *this > v ));
}

inline
int EXPORT MIX :: operator >= (const CHA& v) const
{
   return (!( *this < v ));
}

inline
int EXPORT MIX :: operator != (const char* v) const
{
   return (!( *this == v ));
}

inline
int EXPORT MIX :: operator <= (const char* v) const
{
   return (!( *this > v ));
}

inline
int EXPORT MIX :: operator >= (const char* v) const
{
   return (!( *this < v ));
}


// *********************************************************************
// Stream overloads
// *********************************************************************

ostream& operator << (ostream&, quadnum&);
ostream& operator << (ostream&, const FCWString&);
ostream& operator << (ostream&, const CHA&);
ostream& operator << (ostream&, const DBCS&);
ostream& operator << (ostream&, const MIX&);
ostream& operator << (ostream&, const HEX&);
ostream& operator << (ostream&, const BIN2&);
ostream& operator << (ostream&, const BIN4&);
ostream& operator << (ostream&, const BIN8&);
ostream& operator << (ostream&, const NUM&);
ostream& operator << (ostream&, const NUMC&);
ostream& operator << (ostream&, const PACK&);
ostream& operator << (ostream&, const PACF&);

// *********************************************************************
// Trace overloads
// *********************************************************************

TraceFile& operator << (TraceFile&, quadnum&);
TraceFile& operator << (TraceFile&, const FCWString&);
TraceFile& operator << (TraceFile&, const CHA&);
TraceFile& operator << (TraceFile&, const DBCS&);
TraceFile& operator << (TraceFile&, const MIX&);
TraceFile& operator << (TraceFile&, const HEX&);
TraceFile& operator << (TraceFile&, const BIN2&);
TraceFile& operator << (TraceFile&, const BIN4&);
TraceFile& operator << (TraceFile&, const BIN8&);
TraceFile& operator << (TraceFile&, const NUM&);
TraceFile& operator << (TraceFile&, const NUMC&);
TraceFile& operator << (TraceFile&, const PACK&);
TraceFile& operator << (TraceFile&, const PACF&);

//  min(a,b) function :  takes quadnums as parameters
int  Min( const quadnum& a, const quadnum& b );

#endif
