/**************************************************************************
 *
 * SOURCE FILE NAME = QPUTIL.C
 *
 * DESCRIPTIVE NAME = PM Print Queue Processor
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1991, 1992
 *             Copyright Microsoft Corporation, 1990
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION : Utility functions for PM Print Queue Processor
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *  DATE      FLAG        APAR   CHANGE DESCRIPTION
 *  --------  ----------  -----  --------------------------------------
 *  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 ****************************************************************************/

#include "pmprint.h"

/*
 * Need global variable to ensure ChkMem is not optimized to        @97062
 *   not actually reference the pointers it is supposed to validate
 */
CHAR chMemChk[4];

/****************************************************************************
 *
 * FUNCTION NAME = EnterSem
 *
 * DESCRIPTION   = Get access to per-process MUTEX semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Caller has exclusive access to per-process data
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

VOID EnterSem( VOID )
{
    if (!fInitDone)
        Panic( "Spooler not initialized", "EnterSem", 0 );
    else
        DosRequestMutexSem(semPMPRINT, SEM_INDEFINITE_WAIT);
}

/****************************************************************************
 *
 * FUNCTION NAME = LeaveSem
 *
 * DESCRIPTION   = Release access to per-process MUTEX semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Caller gives up exclusive access to per-process data
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/
VOID LeaveSem( VOID )
{
      DosReleaseMutexSem(semPMPRINT);
}

/****************************************************************************
 *
 * FUNCTION NAME = ExitSem
 *
 * DESCRIPTION   = Terminate per-process MUTEX semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Caller closes exclusive access semaphore
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

VOID ExitSem( VOID )
{
      DosCloseMutexSem(semPMPRINT);
}

/****************************************************************************
 *
 * FUNCTION NAME = AsciiToInt
 *
 * DESCRIPTION   = Convert ascii string to integer, handling whitespace
 *
 * INPUT         = psz   - string to convert to numeric value
 *
 * OUTPUT        = numeric value of string
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

USHORT AsciiToInt( PSZ psz )
{
    USHORT n;
    UCHAR c;
    BOOL bNegative = FALSE;
    BOOL bDigitSeen= FALSE;

    while (*psz == ' ')
        psz++;

    c = *psz;
    if (c == '-')
        bNegative = TRUE;
    else
    if (c != '+')
        c = 0;

    if (c)
        psz++;

    n = 0;
    while (c = *psz++) {
        c -= '0';
        if (c > 9)
            break;

        else {
            bDigitSeen = TRUE;
            n = (n*10) + c;
            }
        }

    if (bNegative)
        return( -n );
    else
        return( n );
}

/****************************************************************************
 *
 * FUNCTION NAME = ParseKeyData
 *
 * DESCRIPTION   = Parse string for a separator character and allocate
 *                 and build array of strings from it.
 *                 Each string is null-terminated
 *
 * INPUT         = pszKeyData  - string to parse
 *                 chSep       - character separating keyname/keyvalue pairs
 *                 cbKeyData   - length of pszKeyData
 *                 pTempKeyBuf - temp buffer to return keydata in
 *                               Might not fit all keydata
 *                 cbTempBuf   - size(in bytes) of pTempKeyBuf
 *
 * OUTPUT        = pointer to key data structure
 *                 This will match pTempKeyBuf if keydata fits in the buffer
 *                 else the pointer is to memory allocated, which must
 *                 be freed by the caller.
 *
 * RETURN-NORMAL = pKeyData
 *
 *
 * RETURN-ERROR  = NULL
 *
 * NOTE: This routine expects the string being passed in to NOT contain
 *       double byte character set(DBCS).
 *       To accept DBCS strings use DosGetDBCSEnv().
 *
 ****************************************************************************/

PKEYDATA ParseKeyData( PSZ pszKeyData, UCHAR chSep, USHORT cbKeyData,
                       PKEYDATA pTempKeyBuf, USHORT cbTempBuf )
{
    USHORT   cTokens;
    USHORT   cb;
    USHORT   i;
    PKEYDATA pResult;
    PSZ      pDst;
    PSZ      psz;

    cTokens = 0;
    cb = cbKeyData;
    psz = pszKeyData;
    if (psz)
    {
        while (cb-- && *psz)
        {
            if (*psz == chSep)
                cTokens++;
            psz++ ;
        }

        if (psz[-1] != chSep)
            if (chSep != ';')
                cTokens++;
            else
                cTokens = 0;
    }

    if (cb || *psz || !cTokens)
        return( NULL );

    cb = sizeof( KEYDATA ) + (cTokens-1) * sizeof( PSZ ) + cbKeyData;
    if (cb > cbTempBuf)
    {
       /*
       ** Won't fit in given buffer so allocate a new one.
       ** Caller must free this buffer.
       */
       if (DosAllocMem( (PVOID)&pResult, cb, PAG_COMMIT|PAG_READ|PAG_WRITE))
       {
          return(NULL);
       }
    } else {
       pResult = pTempKeyBuf ;
    }

    pResult->cb = cb;
    pResult->cTokens = cTokens;
    pDst = (PSZ)pResult + (cb-cbKeyData);
    i = 0;

    while (cTokens--)
    {
       pResult->pTokens[ i ] = pDst;
       while (*pDst = *pszKeyData++) {
          if (*pDst != chSep)
          {
              pDst++ ;
          }
          else {
              *pDst = '\0';
              break;
              }
          }

       if (pResult->pTokens[ i ] == pDst)
           pResult->pTokens[ i ] = NULL;
       else
       {
           pDst++ ;
       }
       i++;
    }

    return( pResult );
}

/****************************************************************************
 *
 * FUNCTION NAME = EndStrcpy
 *
 * DESCRIPTION   = Copy string and return pointer to terminating '\0'
 *
 * INPUT         = pszDest  - gets new string
 *                 pszSrc   - source string to copy
 *
 * OUTPUT        = pszDest+strlen(pszSrc)
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

PSZ EndStrcpy( PSZ pszDest, PSZ pszSrc )
{
        strcpy( pszDest, pszSrc );
        return( (PSZ)(pszDest + SafeStrlen( pszDest )) );
}


/****************************************************************************
 *
 * FUNCTION NAME = SafeStrlen
 *
 * DESCRIPTION   = Return length of string, handle NULL pointer
 *
 * INPUT         = psz  - string to get length, may be NULL
 *
 * OUTPUT        = strlen(psz)
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/
int SafeStrlen( PSZ psz )
{
        if ( psz == (PSZ)NULL )
                return( 0 );
        else
                return( strlen( psz ) );
}

/****************************************************************************
 *
 * FUNCTION NAME = SafeStrcpy
 *
 * DESCRIPTION   = Copy source string to destination, handle NULL pointers.
 *
 * INPUT         = pszDest  - gets new string
 *                 pszSrc   - source string to copy
 *
 * OUTPUT        = pszDest
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/
PSZ SafeStrcpy( PSZ pszDest, PSZ pszSrc )
{
        if ( pszDest == (PSZ)NULL || pszSrc == (PSZ)NULL )
                return( pszDest );
        else
                return( strcpy( pszDest, pszSrc ) );
}

/****************************************************************************
 *
 * FUNCTION NAME = ChkMem
 *
 * DESCRIPTION   = Checks for access to a specific area of memory
 *                 Uses exception handling to determine if valid parms
 *                   were passed.
 *                 If an invalid pointer was given
 *                   an exception is raised when attempting to use the ptr,
 *                   our exception handler is called and we return FALSE
 *
 * INPUT         = pMem - Pointer to the memory to check
 *               = usMemSize - Size of the memory to check (0 = 64K)
 *               = usFlags - What access should be checked(read=0 or write=1)
 *
 * RETURN-NORMAL = 1(Memory can be accessed)
 *
 * RETURN-ERROR  = 0(Memory cannot be accessed)
 *
 ****************************************************************************/

BOOL ChkMem ( PVOID pMem, USHORT usMemSize, USHORT usFlags ) {

   BOOL             fSuccess;  /* TRUE if memory OK, FALSE if not accessable */
   ULONG            rcExcept;
   BIGXRR2          RegRec;
   PPIB             pPib ;   /* -> process ID block */
   PTIB             pTib ;   /* -> thread  ID block */
   PSZ              pszBuf;
   CHAR             ch;


   /*
    *  If NULL pointer then just return failure
    *    without setting exception handler.
    */
   if (!pMem) {
     return(FALSE);
   }

   /*
    *  When setjmp returns a nonzero number, it is returning from a longjmp
    *  [an exception has occurred]
    */
   if ( setjmp(RegRec.JBuf)) {
      /*
       * We are back here after an exception
       * Return to caller
       */
      return (FALSE);
   }

   /* Get thread information block pointer
    * Save this thread's exception chain pointer because we must
    *   restore this if we handle an exception
    */
   rcExcept = DosGetInfoBlocks(&pTib, &pPib);

   if (rcExcept) {

       RegRec.pTib         = (PTIB)NULL ;
       RegRec.pExceptChain = (PVOID)NULL ;

   } else {

       RegRec.pTib         = pTib ;
       RegRec.pExceptChain = pTib->tib_pexchain;
   }

   RegRec.pXRR = NULL;
   RegRec.Handler = &ExceptRoutine;

   rcExcept = DosSetExceptionHandler((PEXCEPTIONREGISTRATIONRECORD)&RegRec);
   /*
    * For now, nothing to do if rcExcept != 0
    */

    /*
     * A 0 size really equals 64K [Can't Check for segments anymore]
     *    Just check first byte for now.
     */

    if (usMemSize == 0) {
        usMemSize = 1;
    }

   pszBuf = (PSZ)pMem ;
   fSuccess = TRUE ;

   /*
    * Determine type of access to check
    *
    * Assigning global variable for READ/WRITE checks, because local
    *   variable would optimize to not perform check.
    */
   if (usFlags == CHK_READ) {
       chMemChk[0] = pszBuf[0];
       chMemChk[1] = pszBuf[usMemSize-1];
   } else if (usFlags == CHK_WRITE) {
       ch = pszBuf[0];
       pszBuf[0] = ch;
       if (ch) {
          chMemChk[2] = ch;
       }
       ch = pszBuf[usMemSize-1];
       pszBuf[usMemSize-1] = ch;
       if (ch) {
          chMemChk[3] = ch;
       }
   } else {
       fSuccess = FALSE ;
   }

   if (!rcExcept) {
       /* if able to set exception handler, disable it now */
       rcExcept = DosUnsetExceptionHandler( (PEXCEPTIONREGISTRATIONRECORD)
                                            &RegRec) ;
   }

   return(fSuccess);
}


/****************************************************************************
 *
 * FUNCTION NAME = ChkStr
 *
 * DESCRIPTION   = Checks for read access to a specific string.
 *                 Note: This function will fail if the memory in question has
 *                       more than one type[ie. committed vs. uncommitted]
 *
 * INPUT         = psz - String to check for read access
 *
 * RETURN-NORMAL = TRUE(String can be accessed)
 *
 * RETURN-ERROR  = FALSE(String cannot be accessed)
 *
 ****************************************************************************/

BOOL ChkStr ( PSZ psz )
{
   ULONG            rcExcept;
   BIGXRR2          RegRec;
   PPIB             pPib ;   /* -> process ID block */
   PTIB             pTib ;   /* -> thread  ID block */


   /*
    *  If NULL pointer then just return failure
    *    without setting exception handler.
    */
   if (!psz) {
     return(FALSE);
   }

   /*
    *  When setjmp returns a nonzero number, it is returning from a longjmp
    *  [an exception has occurred]
    */
   if ( setjmp(RegRec.JBuf)) {
      /*
       * We are back here after an exception
       * Return to caller
       */
      return (FALSE);
   }

   /* Get thread information block pointer
    * Save this thread's exception chain pointer because we must
    *   restore this if we handle an exception
    */
   rcExcept = DosGetInfoBlocks(&pTib, &pPib);

   if (rcExcept) {

       RegRec.pTib         = (PTIB)NULL ;
       RegRec.pExceptChain = (PVOID)NULL ;

   } else {

       RegRec.pTib         = pTib ;
       RegRec.pExceptChain = pTib->tib_pexchain;
   }

   RegRec.pXRR = NULL;
   RegRec.Handler = &ExceptRoutine;

   rcExcept = DosSetExceptionHandler((PEXCEPTIONREGISTRATIONRECORD)&RegRec);
   /*
    * For now, nothing to do if rcExcept != 0
    */

   /*
    * Assigning global variable for READ/WRITE checks, because local
    *   variable would optimize to not perform check.
    */
   chMemChk[0] = (CHAR)strlen(psz);

   if (!rcExcept) {
       /* if able to set exception handler, disable it now */
       rcExcept = DosUnsetExceptionHandler( (PEXCEPTIONREGISTRATIONRECORD)
                                            &RegRec) ;
   }

   return(TRUE);
}

/****************************************************************************
 *
 * FUNCTION NAME = ExceptRoutine
 *
 * DESCRIPTION   = PMPRINT's exception handling routine.
 *                 This takes care of thread exceptions while in PMPRINT.
 *                 Called when an exception occurrs and we have set an
 *                 exception handler.
 *                 This will restore all registers to the values
 *                 when setjmp() was called, and return to just past setjmp()
 *                 call with error code 5.
 *
 * INPUT         = pReportRec -> exception report record, contains type
 *                               of exception.
 *                 pLocalXRR  -> our big exception registration record
 *                               containing the setjmp() structure to
 *                               restore our registers
 *                 pContext   -> context record(not used by us)
 *                 pVoid      -> not used by us
 *
 * RETURN-NORMAL = returns to caller with XCPT_CONTINUE_SEARCH if we
 *                 don't handle the exception
 *
 *
 * RETURN-ERROR  = reset context to time of setjmp() call and return
 *                 exception number to caller of setjmp()
 *
 *
 ****************************************************************************/

INT   APIENTRY ExceptRoutine(PEXCEPTIONREPORTRECORD pReportRec ,
                           PEXCEPTIONREGISTRATIONRECORD pLocalXRR,
                           PCONTEXTRECORD pContext,
                           PVOID pVoid)
{
   PBIGXRR2      pLocalRR;


   if (pReportRec->ExceptionNum == XCPT_ACCESS_VIOLATION) {

      pLocalRR = (PBIGXRR2) pLocalXRR;
      /*
       * Restore the exception chain for this thread to its value
       * when the thread called our API
       */
      if (pLocalRR->pTib) {
          pLocalRR->pTib->tib_pexchain = pLocalRR->pExceptChain;
      }
      longjmp (pLocalRR->JBuf,XCPT_ACCESS_VIOLATION);

   } else {

      return XCPT_CONTINUE_SEARCH;
   }
   /*
    * Never gets here, but quiet compiler
    */
   return XCPT_CONTINUE_SEARCH;
}
