/**************************************************************************
 *
 * SOURCE FILE NAME = QPUTXT.C
 *
 * DESCRIPTIVE NAME = PM Print Queue Processor
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1991, 1992
 *             Copyright Microsoft Corporation, 1990
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION : Text mode print functions for PM Print Queue Processor
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *  DATE      FLAG        APAR   CHANGE DESCRIPTION
 *  --------  ----------  -----  --------------------------------------
 *  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 *  08/12/96  @IBMJ-RAW_TXT      feature 5j created by HS.
 ****************************************************************************/

#define INCL_DEV

#include "pmprint.h"


#ifdef    RAW_TXT                                           //@IBMJ-RAW_TXT

// definitions for tiny heap manager

#define PAGE_SIZE   4096                    // allocation unit

typedef struct _MCB {                       // memory control block
    ULONG        cbSize;                    // size of allocated memory
    struct _MCB *pNextMCB;
} MCB, *PMCB;

typedef struct _HCB {                       // heap control block
    ULONG        cbSize;                    // size of this heap unit
    struct _HCB *pNextHCB;                  // pointer to next heap unit
    PMCB         pNextMCB;                  // pointer to next memory control block
    ULONG        cbFreeSize;                // free space of this heap unit
} HCB, *PHCB;


// definitions for external file parser

typedef struct _RAWTOKEN {      /* rawtoken */
    ULONG       uFormat;
    ULONG       cbToken;
    PBYTE       pbToken;
} RAWTOKEN, *PRAWTOKEN;

typedef struct _RAWTOKENS {     /* rawtokens */
    struct _RAWTOKENS  *NextTok;
    RAWTOKEN            Tok;
} RAWTOKENS, *PRAWTOKENS;

#define TOK_ACTION_NONE             0
#define TOK_ACTION_REPLACE          1
#define TOK_ACTION_ADD_BEFORE       2
#define TOK_ACTION_ADD_AFTER        3


typedef struct _C_ESC_TBL {
    UCHAR   chr;
    UCHAR   value;
} C_ESC_TBL, *PC_ESC_TBL;

C_ESC_TBL c_esc_tbl [] = {
    {'a',  '\a'},       // alert
    {'b',  '\b'},       // back space
    {'f',  '\f'},       // form feed
    {'n',  '\n'},       // new line
    {'r',  '\r'},       // carriage return
    {'t',  '\t'},       // horizontal tab
    {'v',  '\v'},       // vertical tab
    {'\'', '\''},       // single quote
    {'\"', '\"'},       // double quote
    {'\?', '\?'},       // question
    {'\\', '\\'},       // back slash
//  {'x',  '\x'},       // hexadecimal
    {0,    0   }
};


ULONG QpCreateHeap  (ULONG ulInitSize);
ULONG QpDestroyHeap (ULONG hHeap);
PVOID QpAllocMem    (ULONG hHeap, ULONG ulSize);

PRAWTOKENS BuildIntToken (PRAWTOKENS *ppToks, ULONG hHeap);
PRAWTOKENS BuildExtToken (PRAWTOKENS *ppToks, ULONG hHeap, PULONG pulAction);
ULONG escstr2chr (PSZ pszIn, PUCHAR pchVal);

ULONG ParseRawData      (PQPROCINST pQProc, ULONG cbBuf, PUCHAR pchIn);
PVOID GetDefaultPrnOpts (PSZ pszQName, ULONG uCodePage);


// built in token

RAWTOKEN token [] =
{
    { FORMAT_TXT_OFF, 1, "\x1b" },                  // ESC
    { FORMAT_TXT_OFF, 1, "\x1c" },                  // FS
    { FORMAT_TXT_OFF, 5, "%!PS-" },                 // Adobe PostScript
    { FORMAT_TXT_OFF, 6, "\x04%!PS-" },             // Adobe PostScript with Ctrl+D
    { FORMAT_TXT_OFF, 4, "@PJL" },                  // HP PJL
    { FORMAT_TXT_OFF, 4, "@EJL" },                  // Epson EJL
    { 0,              0, 0 },                       // sentinel
};


/****************************************************************************
 *
 * FUNCTION NAME = VerifyRawFileMode
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

ULONG VerifyRawFileMode (PQPROCINST pQProc, PSZ pszFileName)
{
    BOOL   result;
    ULONG  rc;
    ULONG  cb;
    ULONG  cbRead;
    ULONG  cbTotal;


    result = FALSE;
    pQProc->pRawBuf = NULL;

    /*
    ** Open spool file.
    ** If this fails, OpenQPxxx call issues message to user
    */
    if (OpenQPInputFile (pQProc, pszFileName, TRUE))
    {
        /*
        ** Allocate max buf size of 4K to read spool file
        */
        if ((cbTotal = pQProc->cbFile) > PAGE_SIZE)
            cb = PAGE_SIZE;
        else
            cb = cbTotal;

        rc = DosAllocMem (&pQProc->pRawBuf, cb, PAG_COMMIT|PAG_READ|PAG_WRITE);
        if (rc == 0)
        {
            result = TRUE;

            cbRead = 0;
            rc = DosRead (pQProc->hFile, pQProc->pRawBuf, cb, &cbRead);
            if (rc || (cb != cbRead))
            {
                /*
                ** Read of spool file failed
                ** Give a message to user and fail print job
                ** Force read loop to terminate by zeroing cbTotal
                */
                cbTotal = 0 ;
                pQProc->fsStatus &= ~ QP_PAUSED;
                Panic ("QP: DosRead failed for %0s",
                       (PSZ)pQProc->szFileName, rc);
                SplQpMessage (pQProc->pszPortName,
                              SPL_ID_QP_FILE_NOT_FOUND,
                              PMERR_SPL_FILE_NOT_FOUND);
            }
            else
            {
                ParseRawData (pQProc, cbRead, pQProc->pRawBuf);
            }
        }
        else
        {
            /*
            ** Failed to allocate buffer to read file into
            */
            result = FALSE;
            SplQpMessage (pQProc->pszPortName,
                          SPL_ID_QP_MEM_ERROR,
                          PMERR_SPL_NO_MEMORY);
        }
    }

    if (pQProc->pRawBuf)
    {
        DosFreeMem (pQProc->pRawBuf);
        pQProc->pRawBuf = NULL;
    }

    CloseQPInputFile (pQProc);

    if (!result)
        Panic ("VerifyRawFileMode failed for %0s", pszFileName, 0);

    return (result);
}


ULONG ParseRawData (PQPROCINST pQProc, ULONG cbBuf, PUCHAR pchIn)
{
    ULONG       uFormat;
    ULONG       cbReadSkip;
    ULONG       hHeap;
    ULONG       ulAction;
    PRAWTOKENS  pToks = NULL;
    PRAWTOKENS  pToksInt, pToksIntLast;
    PRAWTOKENS  pToksExt, pToksExtLast;


    uFormat = pQProc->qparms.uFormat;

    if (uFormat != FORMAT_TXT_AUTO_DETECT)
        return TRUE;

    // skip null characters
    cbReadSkip = 0;
    for ( ; cbBuf && !*pchIn; cbBuf--, pchIn++)
        cbReadSkip++;

    hHeap = QpCreateHeap (4096);

    if (!hHeap)
        return FALSE;

    pToksIntLast = BuildIntToken (&pToksInt, hHeap);
    pToksExtLast = BuildExtToken (&pToksExt, hHeap, &ulAction);

    switch (ulAction)
    {
      case TOK_ACTION_REPLACE:
        pToks = pToksExt;
        break;
      case TOK_ACTION_ADD_BEFORE:
        pToks = pToksExt;
        if (pToksExtLast)
            pToksExtLast->NextTok = pToksInt;
        break;
      case TOK_ACTION_ADD_AFTER:
        pToks = pToksInt;
        if (pToksIntLast)
            pToksIntLast->NextTok = pToksExt;
        break;
      default:
        pToks = pToksInt;
        break;
    }


    while (pToks)
    {
        if (cbBuf >= pToks->Tok.cbToken && !memcmp (pchIn, pToks->Tok.pbToken, pToks->Tok.cbToken))
        {
            uFormat = pToks->Tok.uFormat;
            break;
        }
        pToks = pToks->NextTok;
    }

    QpDestroyHeap (hHeap);

    if (uFormat == FORMAT_TXT_AUTO_DETECT)
        uFormat = FORMAT_TXT_ON;

    pQProc->qparms.uFormat = uFormat;

    return TRUE;
}


PRAWTOKENS BuildIntToken (PRAWTOKENS *ppToks, ULONG hHeap)
{
    PRAWTOKEN   pTok;
    PRAWTOKENS  pToks, pToksPrev;


    pToks     = NULL;
    pToksPrev = NULL;

    for (pTok = &token[0]; pTok->cbToken; pTok++)
    {
        pToks = (PRAWTOKENS) QpAllocMem (hHeap, sizeof(RAWTOKENS));
        if (pToks)
        {
            if (!pToksPrev)
                *ppToks = pToks;
            else
                pToksPrev->NextTok = pToks;
            pToks->Tok.uFormat = pTok->uFormat;
            pToks->Tok.cbToken = pTok->cbToken;
            pToks->Tok.pbToken = (PBYTE) QpAllocMem (hHeap, pTok->cbToken);
            if (pToks->Tok.pbToken)
                memcpy (pToks->Tok.pbToken, pTok->pbToken, pTok->cbToken);
            else
                pToks->Tok.cbToken = 0;
            pToks->NextTok = NULL;
        }
        pToksPrev = pToks;
    }

    // return the pointer last allocated
    return pToks;
}


// External token file should be placed in the same directory as this queue processor
// and same file name but extension is ".DAT". (example x:\os2\dll\pmprntxt.dat).
//
// Format:
//  - strings after the semicolon ';' is ignored
//  - "action: replace" replace internal rules by the rules of external file.
//  - "action: add-before" the rules of external file are evaluated before the internal.
//  - "action: add-after" the rules of external file are evaluated after the internal.
//  - "binary: xxx" xxx is a binary mode rule.
//  - "text: xxx" xxx is a text mode rule.
//      where xxx is character string. Hex value can be used as \xhh.

PRAWTOKENS BuildExtToken (PRAWTOKENS *ppToks, ULONG hHeap, PULONG pulAction)
{
    PRAWTOKEN   pTok;
    PRAWTOKENS  pToks, pToksPrev;

    APIRET  rc;
    UCHAR   szBuf[CCHMAXPATH];
    LONG    i;
    HFILE   hFile;
    ULONG   actionTaken;
    FILESTATUS3 inFileStatus;
    ULONG   cbToken, cbLine;
    PBYTE   pbData, pbToken, pbNext, pb;
    BOOL    bCommentLine;
    UCHAR   uch;
    ULONG   uFormat;


    pToks     = NULL;
    pToksPrev = NULL;
    *pulAction = TOK_ACTION_NONE;


    // make full path name of external token file.
    // x:\os2\dll\pmxxx.qpr --> x:\os2\dll\pmxxx.dat

    rc = DosQueryModuleName (hModPMPrint, sizeof(szBuf), szBuf);

    if (rc)
        return pToks;

    for (i = strlen(szBuf); i >= 0; i--)
        if (szBuf[i] == '.')
            break;

    if (i <= 0)
        return pToks;

    szBuf[i+1] = '\0';
    strcat (szBuf, "DAT");

    rc = DosOpen (szBuf,
                  &hFile,
                  &actionTaken,
                  (ULONG)0,                         /* filesize NotApplicable */
                  (ULONG)0,                         /* no attributes */
                  OPEN_ACTION_OPEN_IF_EXISTS,       /* open only if exists */
                  OPEN_FLAGS_NOINHERIT|OPEN_ACCESS_READONLY|OPEN_SHARE_DENYNONE,
                  NULL);                            /* No EAs to set */

    if (rc)
        return pToks;

    rc = DosQueryFileInfo (hFile,
                           FIL_STANDARD,
                           &inFileStatus,
                           sizeof(inFileStatus));

    if (rc)
    {
        DosClose (hFile);
        return pToks;
    }

    // (cbFile + 1) for null terminator
    rc = DosAllocMem ((PVOID)&pbData, inFileStatus.cbFile+1, PAG_READ|PAG_WRITE|PAG_COMMIT);

    if (!rc)
        rc = DosRead (hFile, pbData, inFileStatus.cbFile, &cbToken);

    if (rc || (inFileStatus.cbFile != cbToken))
    {
        DosClose (hFile);
        return pToks;
    }

    *pulAction = TOK_ACTION_REPLACE;                // default action
    pbToken  = pbData;
    *(pbToken + cbToken) = '\0';
    cbToken++;                                      // buffer size = file size + 1

    while (cbToken)
    {
        // skip leading blank
        if (*pbToken == ' ')
        {
            cbToken++;
            pbToken++;
            continue;
        }

        // skip comment line
        if (*pbToken == ';')
        {
            pbNext = strpbrk (pbToken, "\x0d\x0a\x1a\0");
            pbNext++;
            cbToken -= pbNext - pbToken;
            pbToken = pbNext;
            continue;
        }

        pbNext = strpbrk (pbToken, ";\x0d\x0a\x1a\0");
        if (pbNext)
            pbNext++;
        else
            pbNext = pbToken + cbToken;

        cbLine = pbNext - pbToken;

        uFormat = FORMAT_TXT_AUTO_DETECT;

        if (!strncmp (pbNext, "action:", strlen ("action:")))
        {
            if (strstr (pbNext, "add-before"))
                *pulAction = TOK_ACTION_ADD_BEFORE;
            else if (strstr (pbNext, "add-after"))
                *pulAction = TOK_ACTION_ADD_AFTER;
        }
        else if (!strncmp (pbNext, "binary:", strlen("binary:")))
        {
            uFormat = FORMAT_TXT_OFF;
        }
        else if (!strncmp (pbNext, "text:", strlen("text:")))
        {
            uFormat = FORMAT_TXT_ON;
        }

        if (FORMAT_TXT_AUTO_DETECT != uFormat)
        {
            PUCHAR  p1, p2;

            p1 = strchr (pbNext, '"');
            if (p1)
            {
                p1++;
                p2 = strchr (p1, '"');
                if (p2)
                {
                    i = p2 - p1;
                    pb = NULL;
                    pToks = (PRAWTOKENS) QpAllocMem (hHeap, sizeof(RAWTOKENS));
                    if (pToks)
                    {
                        pToks->NextTok = NULL;
                        if (!pToksPrev)
                            *ppToks = pToks;
                        else
                            pToksPrev->NextTok = pToks;
                        pToks->Tok.uFormat = uFormat;
                        pToks->Tok.cbToken = 0;         // initial value
                        pToks->Tok.pbToken = pb = (PBYTE) QpAllocMem (hHeap, i);
                    }

                    *p2 = '\0';
                    while (*p1)
                    {
                        switch (*p1)
                        {
                          case '\\':
                            i = escstr2chr (p1, &uch);
                            if (pb)
                                *pb++ = uch;
                            p1 += i;
                            break;

                          default:
                            if (pb)
                                *pb++ = *p1;
                            p1++;
                            break;
                        }
                    }
                    if (pToks && pb)
                        pToks->Tok.cbToken = pb - pToks->Tok.pbToken;

                    pToksPrev = pToks;
                }
            }
        }

        cbToken -= cbLine;
        pbToken = pbNext;
    }

    DosFreeMem (pbData);

    DosClose (hFile);

    return pToks;
}


ULONG escstr2chr (PSZ pszIn, PUCHAR pchVal)
{
    ULONG   ulInc = 0;
    UCHAR   uch;
    INT     i;

    if ('\\' == *pszIn)
    {
        ulInc++;
        pszIn++;

        if (('x' == *pszIn) || ('X' == *pszIn))
        {
            ulInc++;
            pszIn++;
            *pchVal = 0;

            for (i = 2; i > 0; i--)
            {
                uch = *pszIn;
                ulInc++;
                pszIn++;

                if ((uch >= 'a') && (uch <= 'f'))
                    *pchVal += (uch - 'a') + 10;
                else if ((uch >= 'A') && (uch <= 'F'))
                    *pchVal += (uch - 'A') + 10;
                else if ((uch >= '0') && (uch <= '9'))
                    *pchVal += (uch - '0');

                if (2 == i)
                    *pchVal *= 16;
            }
        }
        else
        {
            ulInc++;
            pszIn++;

            for (i = 0; (uch = c_esc_tbl[i].chr) != '\0'; i++)
            {
                if (uch == *pszIn)
                {
                    *pchVal = c_esc_tbl[i].value;
                    break;
                }
            }
        }
    }

    return ulInc;
}


ULONG QpCreateHeap (ULONG ulInitSize)
{
    ULONG   rc;
    PHCB    phcb;


    ulInitSize += sizeof (HCB);

    if (ulInitSize < PAGE_SIZE)
        ulInitSize = PAGE_SIZE;

    ulInitSize = ((ulInitSize + PAGE_SIZE - 1) / PAGE_SIZE) * PAGE_SIZE;

    rc = DosAllocMem ((PVOID)&phcb, ulInitSize, PAG_READ|PAG_WRITE|PAG_COMMIT);

    if (rc)
    {
        return 0;
    }

    phcb->cbSize     = ulInitSize;
    phcb->pNextHCB   = NULL;
    phcb->pNextMCB   = (PMCB)(phcb + 1);
    phcb->cbFreeSize = phcb->cbSize - sizeof(HCB);

    return (ULONG)phcb;
}

ULONG QpDestroyHeap (ULONG hHeap)
{
    PHCB    phcb, phcb_next;

    phcb = (PHCB) hHeap;

    while (phcb)
    {
        phcb_next = phcb->pNextHCB;
        DosFreeMem (phcb);
        phcb = phcb_next;
    }
    return TRUE;
}


PVOID QpAllocMem (ULONG hHeap, ULONG ulSize)
{
    PHCB    phcb;
    PMCB    pmcb;

    phcb = (PHCB) hHeap;

    ulSize += sizeof(MCB);

    while (phcb)
    {
        if (ulSize <= phcb->cbFreeSize)
        {
            pmcb = phcb->pNextMCB;
            pmcb->cbSize = ulSize;
            pmcb->pNextMCB = pmcb + ulSize;

            phcb->pNextMCB    = pmcb->pNextMCB;
            phcb->cbFreeSize -= ulSize;

            return (PVOID)(pmcb + 1);
        }

        if (!phcb->pNextHCB)
            phcb->pNextHCB = (PHCB) QpCreateHeap (ulSize);

        phcb = phcb->pNextHCB;
    }

    return NULL;
}



/****************************************************************************
 *
 * FUNCTION NAME = SplQpRawPrintTextFile
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

BOOL SplQpRawPrintTextFile (PQPROCINST pQProc, PSZ pszFileName)
{
    BOOL            rc;
    HMODULE         hmodPMSPL;
    CHAR            szBadName[CCHMAXPATH];
    PRINTDEST       printdest;
    DEVOPENSTRUC    openData;
    PVOID           pPrnOpts;
    #define ORD_SPLPRINTFILE 623
    SPLERR (* APIENTRY pfnSplPrintFile) (ULONG      ulDestType,
                                         PVOID      pDestBuf,
                                         PSZ        pszFilenam,
                                         PVOID      pPrnOpts,
                                         PSZ        pszJobTitle);

    rc = FALSE;

    if (!DosLoadModule (szBadName, sizeof(szBadName), "PMSPL", &hmodPMSPL))
    {
        if (!DosQueryProcAddr (hmodPMSPL, ORD_SPLPRINTFILE, NULL, (PFN *)&pfnSplPrintFile))
        {
            memset (&openData, 0, sizeof(openData));
            openData.pszLogAddress      = (PSZ)pQProc->pszPortName;
//          openData.pszLogAddress      = (PSZ)pQProc->pszQName;
            openData.pszDriverName      = (PSZ)pQProc->pszDriverName;
            openData.pdriv              = pQProc->pDriverData;
            openData.pszDataType        = pQProc->pszDataType;
            openData.pszComment         = pQProc->pszComment;
            openData.pszQueueProcName   = (PSZ)NULL;
            openData.pszQueueProcParams = (PSZ)NULL;
            openData.pszSpoolerParams   = pQProc->pszSpoolerParams;

            memset (&printdest, 0, sizeof(printdest));
            printdest.cb         = sizeof(printdest);
            printdest.lType      = OD_DIRECT;
            printdest.pszToken   = "*";
            printdest.lCount     = SPL_PARAMS + 1;
            printdest.pdopData   = (PDEVOPENDATA)&openData;
            printdest.fl         = 0;
            printdest.pszPrinter = NULL;

            printdest.fl &= ~PD_JOB_PROPERTY;

            pPrnOpts = GetDefaultPrnOpts (pQProc->pszQName, pQProc->qparms.uCodePage);

            rc = pfnSplPrintFile (1,
                                  (PVOID) &printdest,
                                  pszFileName,
                                  pPrnOpts,
                                  pQProc->pszDocument);

            if (pPrnOpts)
                DosFreeMem (pPrnOpts);

            if (rc == NO_ERROR)
                rc = TRUE;
            else
                rc = FALSE;
        }
        DosFreeModule (hmodPMSPL);
    }

    return rc;
}


/****************************************************************************
 *
 * FUNCTION NAME = GetDefaultPrnOpts
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL = None
 *
 *
 * RETURN-ERROR  = None
 *
 *
 ****************************************************************************/

typedef struct _PRINTEROPTIONS
{
  LONG          lVersion;            /* version for this  structure            */
  FIXED         fxPointSize;         /* Pointsize for scalable font            */
  FATTRS        fAttrs;              /* Font-attribute structure for desired   */
  ULONG         flOptions;           /* Print options flags see PRINTOPT_      */
  USHORT        usPrintColumns;      /* Number of Columns per page             */
  USHORT        usNumberUpColumns;   /* Number of pages per form, columns-wise */
  USHORT        usNumberUpRows;      /* Number of pages per form, row-wise     */
  USHORT        usReserved;          /* reserved, must be 0                    */
  PSZ           pszInputFormat;      /* format of the data to be printed       */
  PSZ           pszFooter;           /* text page footer                       */
  PSZ           pszHeader;           /* text page header                       */
  PVOID         pReserved1;          /* reserved, must be NULL                 */
  PVOID         pReserved2;          /* reserved, must be NULL                 */
} PRINTEROPTIONS, *PPRINTEROPTIONS;

#define OFFSETTOPTR(p, np) np->p = (PVOID)(OFFSETOF(np->p) ? \
                                   ((PBYTE)np + OFFSETOF(np->p)) : (PBYTE)NULL)

#define PTEXT_KEYNAME     "PRINTER_OPTIONS"                  /* ini key name */


PVOID GetDefaultPrnOpts (PSZ pszQName, ULONG uCodePage)
{
  CHAR   szAppName[25];
  ULONG  ulDataLength=0;
  ULONG  rc = 0;
  PPRINTEROPTIONS   pDefaultPrnOpts = NULL;

  strcpy (szAppName, "PM_PrintObject:");
  strcat (szAppName, pszQName);

  rc = PrfQueryProfileSize( HINI_USER, szAppName, PTEXT_KEYNAME,&ulDataLength );

  if( !rc || (ulDataLength <  sizeof(PRINTEROPTIONS) ) )
     return( NULL );

  rc = DosAllocMem( (PPVOID)&(pDefaultPrnOpts), ulDataLength,
                     PAG_READ | PAG_WRITE | PAG_COMMIT);

  if( rc || (pDefaultPrnOpts == NULL) )
    return( NULL );

  memset( (PBYTE)pDefaultPrnOpts,0,sizeof(PRINTEROPTIONS) );

  rc = PrfQueryProfileData( HINI_USER, szAppName, PTEXT_KEYNAME,pDefaultPrnOpts, &ulDataLength );
                   /* if prf fails, continue with default zeroed PRINTEROPTIONS */
  OFFSETTOPTR(pszHeader,      pDefaultPrnOpts);
  OFFSETTOPTR(pszFooter,      pDefaultPrnOpts);
  OFFSETTOPTR(pszInputFormat, pDefaultPrnOpts);

  if (uCodePage)
      pDefaultPrnOpts->fAttrs.usCodePage = (USHORT)uCodePage;

  return pDefaultPrnOpts;
}

#endif // RAW_TXT                                           //@IBMJ-RAW_TXT

