/****************************************************************************/
/*  Copyright (C) 1993 IBM Corporation                                      */
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = util.c
 *
 * DESCRIPTIVE NAME = par1284 port driver utility routines for use with
 *                     structures, memory management, debug routines,...
 *
 * Copyright : COPYRIGHT IBM CORPORATION, 1994, 1995
 *             LICENSED MATERIAL - PROGRAM PROPERTY OF IBM
 *             REFER TO COPYRIGHT INSTRUCTION FORM#G120-2083
 *             RESTRICTED MATERIALS OF IBM
 *             IBM CONFIDENTIAL
 *
 * VERSION = V2.2
 *
 * DATE
 *
 * DESCRIPTION
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
 * CHANGE ACTIVITY =
 *  DATE      FLAG        APAR   CHANGE DESCRIPTION
 *  --------  ----------  -----  --------------------------------------
 *  mm/dd/yy  @Vr.mpppxx  xxxxx  xxxxxxx
 ****************************************************************************/

#include    "pdrconst.h"
#include    "pdrtypes.h"
#include    "pdrproto.h"

/*
** Global Variable for debugging
*/
#ifdef DEBUG_ALERT
 HFILE    hFileAlerts;      /* Handle to file to store logged data */
 BOOL     fLogAlerts=1;     /* Set to false to no longer log info  */
 BOOL     fDBterminal;      /* TRUE if sending output to PMDD.SYS  */
#endif

/*
** Need global variable to ensure ChkMem is not optimized to        @97062
**   not actually reference the pointers it is supposed to validate
*/
CHAR chMemChk[4];


/*
** Defines for converting DataTime into seconds past Jan 1,1970
*/
#define JULIANJAN 0
#define JULIANFEB 31
#define JULIANMAR (JULIANFEB + 28)
#define JULIANAPR (JULIANMAR + 31)
#define JULIANMAY (JULIANAPR + 30)
#define JULIANJUN (JULIANMAY + 31)
#define JULIANJUL (JULIANJUN + 30)
#define JULIANAUG (JULIANJUL + 31)
#define JULIANSEP (JULIANAUG + 31)
#define JULIANOCT (JULIANSEP + 30)
#define JULIANNOV (JULIANOCT + 31)
#define JULIANDEC (JULIANNOV + 30)
#define FEB     2
#define SECPERMIN 60
#define SECPERHOUR (SECPERMIN*60)
#define SECPERDAY (SECPERHOUR*24)
#define SECPERYEAR (SECPERDAY*365)
#define STARTYEAR 1970


const ULONG JulianMonth[] = { 0,
                              JULIANJAN,
                              JULIANFEB,
                              JULIANMAR,
                              JULIANAPR,
                              JULIANMAY,
                              JULIANJUN,
                              JULIANJUL,
                              JULIANAUG,
                              JULIANSEP,
                              JULIANOCT,
                              JULIANNOV,
                              JULIANDEC };


/*
** Local Functions
*/
ULONG       GetIniValue( PSZ pszApp, PSZ pszKey, ULONG ulDefValue );
VOID        FormatDateTime ( PSZ psz );
VOID        ShowHexByte ( PSZ pszBuf, BYTE bValue );
VOID        ShowHexAscii ( PSZ pszBuf, BYTE bValue );
ULONG       JulianDate (UCHAR day, UCHAR month, USHORT year);

/****************************************************************************
 *
 * FUNCTION NAME = AllocPdrMem
 *
 * DESCRIPTION   = Allocate memory from port driver heap
 *
 * INPUT         = cb  - Amount of memory to allocate in bytes
 *
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to allocated memory
 *
 * RETURN-ERROR  = NULL(Failure, memory could not be allocated)
 *
 ****************************************************************************/

PVOID AllocPdrMem ( ULONG cb ) {

    PVOID pMem = NULL;
    ULONG rc = 0;

    rc = DosSubAllocMem ( pHeapBase, &pMem, cb );
    if (rc) {
       PdrError ( 1 , 1 );
       return NULL;
    } else {
       return(pMem);
    }

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePdrMem
 *
 * DESCRIPTION   = Free memory from the port driver heap
 *
 * INPUT         = pMem - Pointer to the start of the memory block
 *               = cb - Size of the memory block to free
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0, Successful
 *
 * RETURN-ERROR  = rc, Failure
 *
 ****************************************************************************/

ULONG FreePdrMem ( PVOID pMem, ULONG cb ) {

    ULONG rc = 0;
    PFREEMEM    pFreeMem = (PFREEMEM) pMem;

    /*
     * Set signature and size [For debugging/validation]
     */
    if (cb >= sizeof(USHORT)) {
        pFreeMem->signature = FR_SIGNATURE;
    }
    if (cb >= (sizeof(USHORT)*2)) {
        pFreeMem->cb = (USHORT) cb;
    }
    /*
     * Free the memory
     */
    rc = DosSubFreeMem ( pHeapBase, pMem, cb );
    if (rc) {
        PdrPanic ( 2 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = EnterPdrSem
 *
 * DESCRIPTION   = Enter the port driver semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosRequestMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be requested)
 *
 ****************************************************************************/

ULONG EnterPdrSem ( VOID ) {

    ULONG rc = 0;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_PDRSEMS)
      {
         sprintf( logbuf,
                  "EnterPdrSem About to Request hSem=%lX\r\n",
                  (ULONG)semPortDrv );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */

    rc = DosRequestMutexSem ( semPortDrv , SEM_INDEFINITE_WAIT );
    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_PDRSEMS)
      {
         sprintf( logbuf,
                  "EnterPdrSem RequestSem rc=%d hSem=%lX\r\n",
                  rc, (ULONG)semPortDrv );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */
    if (rc) {
        PdrError ( 3 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = LeavePdrSem
 *
 * DESCRIPTION   = Leave the port driver semaphore
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosReleaseMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 ****************************************************************************/

ULONG LeavePdrSem ( VOID ) {

    ULONG rc = 0;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    rc = DosReleaseMutexSem ( semPortDrv );
    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_PDRSEMS)
      {
         sprintf( logbuf,
                  "LeavePdrSem ReleaseMutexSem rc=%d hSem=%lX\r\n",
                  rc, (ULONG)semPortDrv );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */
    if (rc) {
        PdrError ( 4 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = EnterDlgSem
 *
 * DESCRIPTION   = Enter the port settings dialog semaphore
 *
 * INPUT         = pPortDlgStruct -> dialog struct containing semaphore
 *
 * OUTPUT        = Calls DosRequestMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be requested)
 *
 ****************************************************************************/

ULONG EnterDlgSem ( PPORTDLGSTRUCT pPortDlgStruct )
{
    ULONG rc = 0;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif

    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_PDRSEMS)
      {
         sprintf( logbuf,
                  "EnterDlgSem About to Request hSem=%lX pPortDlgStruct=%lX\r\n",
                  (ULONG)pPortDlgStruct->hMtxDlg, (ULONG)pPortDlgStruct );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */

    rc = DosRequestMutexSem ( pPortDlgStruct->hMtxDlg , SEM_INDEFINITE_WAIT );
    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_PDRSEMS)
      {
         sprintf( logbuf,
                  "EnterDlgSem RequestSem rc=%d hSem=%lX\r\n",
                  rc, (ULONG)pPortDlgStruct->hMtxDlg );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */
    if (rc) {
        PdrError ( 5 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = LeaveDlgSem
 *
 * DESCRIPTION   = Leave the port settings dialog semaphore
 *
 * INPUT         = pPortDlgStruct -> dialog struct containing semaphore
 *
 * OUTPUT        = Calls DosReleaseMutexSem()
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 ****************************************************************************/

ULONG LeaveDlgSem ( PPORTDLGSTRUCT pPortDlgStruct )
{
    ULONG rc = 0;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif


    rc = DosReleaseMutexSem ( pPortDlgStruct->hMtxDlg );
    #ifdef DEBUG_ALERT
     {
      if (flChangeDebugLog & FL_PDRDB_ENABLE_PDRSEMS)
      {
         sprintf( logbuf,
                  "LeaveDlgSem ReleaseMutexSem rc=%d hSem=%lX pPortDlgStruct=%lX\r\n",
                  rc, (ULONG)pPortDlgStruct->hMtxDlg, (ULONG)pPortDlgStruct );
         LogCall( logbuf );
      }
     }
    #endif /* DEBUG */
    if (rc) {
        PdrError ( 6 , 1 );
    }
    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPdOpenInst
 *
 * DESCRIPTION   = Add new port instance to the list in the port driver
 *
 * INPUT         = pPdOpenInst       - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG AddPdOpenInst ( PPDOPENINST pPdOpenInst ) {

    if (pPdOpenInstList) {
        pPdOpenInst->pNext = pPdOpenInstList;
    }
    pPdOpenInstList = pPdOpenInst;

    return(0);

}

/****************************************************************************
 *
 * FUNCTION NAME = RemovePdOpenInst
 *
 * DESCRIPTION   = Remove port instance from the list in the port driver
 *
 * INPUT         = pPdOpenInst       - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG RemovePdOpenInst ( PPDOPENINST pPdOpenInst ) {

    PPDOPENINST pPdCurr = pPdOpenInstList;
    PPDOPENINST pPdPrev = NULL;

    while (pPdCurr) {
        if (pPdCurr == pPdOpenInst) {
            if (pPdPrev) {
                pPdPrev->pNext = pPdCurr->pNext;
            } else {
                pPdOpenInstList = pPdCurr->pNext;
            }
            return(0);
        }
        pPdPrev = pPdCurr;
        pPdCurr = pPdCurr->pNext;
    }

    return(ERROR_INVALID_HANDLE);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPortInst
 *
 * DESCRIPTION   = Add new port instance to the list in the port driver
 *
 * INPUT         = pszPortName  - Name of the port to add
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, port could not be added)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

PPORTINST AddPortInst ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;

    /*
     * See if port has been added previously
     */
    pPortInst = FindPortInst ( pszPortName );
    if ( pPortInst ) {
        return( pPortInst );
    }
    /*
     * Allocate a new port structure
     */
    if (!(pPortInst = NewPortInst ( pszPortName ))) {
        return(NULL);
    }
    /*
     * If port list exists, add to front of list
     */
    if (pPortInstList) {
        pPortInst->pNext = pPortInstList;
    }
    pPortInstList = pPortInst;

    return(pPortInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = NewPortInst
 *
 * DESCRIPTION   = Allocate new port instance
 *
 * INPUT         = pszPortName  - Name of the new port
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to port instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

PPORTINST NewPortInst ( PSZ pszPortName ) {

    PPORTINST pPortInst = NULL;
    ULONG     cb;
    ULONG     rc = 0;

    cb = sizeof( PORTINST ) + strlen ( pszPortName ) + 1;
    if (!(pPortInst = AllocPdrMem( cb ))) {
        return(NULL);
    }
    memset ( pPortInst, 0, cb );
    pPortInst->signature = PT_SIGNATURE;
    pPortInst->cb = cb;
    pPortInst->ulNoQueryTimeOut = DEF_TIMEOUT_QUERY_VALUE;
    pPortInst->ulNoJobTimeOut   = DEF_TIMEOUT_JOB_VALUE;
    pPortInst->pszPortName = (PSZ)((PBYTE)pPortInst + sizeof(PORTINST));
    strcpy ( pPortInst->pszPortName, pszPortName );
    rc = DosCreateMutexSem ( 0,
                             &(pPortInst->hPortSem),
                             0,
                             FALSE );       // Create Unowned
    if (rc) {
        FreePdrMem ( pPortInst, cb );
        return(NULL);
    }
    rc = DosCreateEventSem ( 0,
                             &(pPortInst->hevReadThread),
                             0,
                             FALSE );   // Create in Reset state
    /*
    ** Separately allocate a buffer to use for DosWrite.
    */
    rc = DosAllocMem((PVOID)&pPortInst->pbWriteBuf,
                      DEFAULT_BUFSIZE,
                      PAG_READ|PAG_WRITE|PAG_COMMIT);

    return(pPortInst);

}

/****************************************************************************
 *
 * FUNCTION NAME = FindPortInst
 *
 * DESCRIPTION   = Find port instance in the port driver
 *
 * INPUT         = pszPortName   -  Name of port to find
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = Pointer to port instance(Successful)
 *
 * RETURN-ERROR  = NULL(Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

PPORTINST FindPortInst ( PSZ pszPortName ) {

    PPORTINST pPortCurr = pPortInstList;

    while (pPortCurr) {
        if (!strcmp(pPortCurr->pszPortName, pszPortName)) {
            return(pPortCurr);
        }
        pPortCurr = pPortCurr->pNext;
    }

    return(NULL);

}

/****************************************************************************
 *
 * FUNCTION NAME = RemovePortInst
 *
 * DESCRIPTION   = Remove port instance from the list in the port driver
 *
 * INPUT         = pszPortName   -  Name of port to remove
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG RemovePortInst ( PSZ pszPortName ) {

    PPORTINST pPortCurr = pPortInstList;
    PPORTINST pPortPrev = NULL;

    while (pPortCurr) {
        if (!strcmp(pPortCurr->pszPortName, pszPortName)) {
            if (pPortPrev) {
                pPortPrev->pNext = pPortCurr->pNext;
            } else {
                pPortInstList = pPortCurr->pNext;
            }
            /*
            ** For now, we will not actually remove the port instance
            **  from memory, just remove it from our list.
            ** This is done because some of our threads might not
            **  be validating the port instance structure all the time.
            ** This is just a precaution to avoid traps;  if the
            **  code proves to be very good at revalidating pPortInst
            **  structures then we can free up this memory here.
            */
            pPortCurr->pNext = NULL;
            // FreePortInst ( pPortCurr );
            return(0);
        }
        pPortPrev = pPortCurr;
        pPortCurr = pPortCurr->pNext;
    }

    return(ERROR_INVALID_HANDLE);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreePortInst
 *
 * DESCRIPTION   = Free port instance memory
 *
 * INPUT         = pPortInst     -  Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0 (Successful)
 *
 * RETURN-ERROR  = rc (Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG FreePortInst ( PPORTINST pPortInst ) {

    ULONG rc = 0;

    /*
     * Close port semaphore
     */
    rc = DosCloseMutexSem ( pPortInst->hPortSem );
    rc = DosCloseEventSem ( pPortInst->hevReadThread );
    if (pPortInst->pbWriteBuf) {
       DosFreeMem(pPortInst->pbWriteBuf);
    }
    pPortInst->pbWriteBuf = NULL;
    FreePdrMem ( pPortInst, pPortInst->cb );

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = FreeAllPortInst
 *
 * DESCRIPTION   = Free port instance memory list
 *
 * INPUT         = None
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0 (Successful)
 *
 * RETURN-ERROR  = rc (Failure)
 *
 * NOTE: Caller must be in the port driver semaphore
 *
 ****************************************************************************/

ULONG FreeAllPortInst ( VOID ) {

    ULONG rc = 0;
    PPORTINST pPortInst = pPortInstList;
    PPORTINST pPortNext = NULL;

    while (pPortInst) {
        pPortNext = pPortInst->pNext;
        rc = FreePortInst ( pPortInst );
        pPortInst = pPortNext;
    }

    return(rc);

}

/****************************************************************************
 *
 * FUNCTION NAME = ValidatePdOpenInst
 *
 * DESCRIPTION   = Validate port instance from the list in the port driver
 *
 * INPUT         = pPdOpenInst       - Pointer to the port instance
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = 0(Successful)
 *
 * RETURN-ERROR  = rc(Failure, semaphore could not be released)
 *
 ****************************************************************************/

PPDOPENINST ValidatePdOpenInst ( PPDOPENINST pPdOpenInst ) {

    PPDOPENINST pPdCurr;

    if (!ChkMem( pPdOpenInst, sizeof(PDOPENINST), CHK_READ) )
    {
       return(NULL);
    }

  EnterPdrSem();

    pPdCurr = pPdOpenInstList;
    while (pPdCurr && (pPdCurr != pPdOpenInst)) {
        pPdCurr = pPdCurr->pNext;
    }

  LeavePdrSem();
    return(pPdCurr);

}

/****************************************************************************
 *
 * FUNCTION NAME = AddPortDlgStruct
 *
 * DESCRIPTION   = Add PORTDLGSTRUCT to list of open dialogs.  If found,
 *                 return the structure from the list.
 *
 * INPUT         = pPortDlgStruct - Pointer to structure contain the
 *                                  computer name and the port name of the
 *                                  PORTDLGSTRUCT to find in the list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = pOldPortDlgStruct - Pointer to PORTDLGSTRUCT from list
 *
 * RETURN-ERROR  = NULL - No PORTDLGSTRUCT found
 *
 ****************************************************************************/

BOOL AddPortDlgStruct ( PPORTDLGSTRUCT pPortDlgStruct ) {

    ULONG     rc = 0;

    /*
     * Make sure we have a structure to add
     */
    if (pPortDlgStruct) {

       rc = DosCreateMutexSem ( 0,
                                &(pPortDlgStruct->hMtxDlg),
                                0,
                                FALSE );       // Create Unowned
       EnterPdrSem();
        pPortDlgStruct->pNext = pOpenDlgList;
        pOpenDlgList = pPortDlgStruct;
       LeavePdrSem();
    }

    return(TRUE);

} /* end AddPortDlgStruct */

/****************************************************************************
 *
 * FUNCTION NAME = RemovePortDlgStruct
 *
 * DESCRIPTION   = Add PORTDLGSTRUCT to list of open dialogs.  If found,
 *                 return the structure from the list.
 *
 * INPUT         = pPortDlgStruct - Pointer to structure contain the
 *                                  computer name and the port name of the
 *                                  PORTDLGSTRUCT to find in the list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = pOldPortDlgStruct - Pointer to PORTDLGSTRUCT from list
 *
 * RETURN-ERROR  = NULL - No PORTDLGSTRUCT found
 *
 ****************************************************************************/

BOOL RemovePortDlgStruct ( PPORTDLGSTRUCT pPortDlgStruct ) {

    PPORTDLGSTRUCT pOldPortDlgStruct = NULL;
    PPORTDLGSTRUCT pPrevPortDlgStruct = NULL;
    BOOL fRemoved = FALSE;

    /*
     * Make sure we have a structure to remove
     */
    if (pPortDlgStruct) {
       EnterPdrSem();
        pOldPortDlgStruct = pOpenDlgList;
        /*
         * Go through the list and remove structure if found
         */
        while (pOldPortDlgStruct) {
            if (pOldPortDlgStruct == pPortDlgStruct) {
                /*
                 * If no previous structure, then it is the head of the list
                 */
                if (pPrevPortDlgStruct) {
                    pPrevPortDlgStruct->pNext = pPortDlgStruct->pNext;
                } else {
                    pOpenDlgList = pPortDlgStruct->pNext;
                }
                fRemoved = TRUE;
                break;
            }
        }
        DosCloseMutexSem(pPortDlgStruct->hMtxDlg);
        pPortDlgStruct->hMtxDlg = 0;
       LeavePdrSem();
    }

    return(fRemoved);

} /* end RemovePortDlgStruct */

/****************************************************************************
 *
 * FUNCTION NAME = FindPortDlgStruct
 *
 * DESCRIPTION   = Find PORTDLGSTRUCT in list of open dialogs.  If found,
 *                 return the structure from the list.
 *
 * INPUT         = pPortDlgStruct - Pointer to structure contain the
 *                                  computer name and the port name of the
 *                                  PORTDLGSTRUCT to find in the list
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = pOldPortDlgStruct - Pointer to PORTDLGSTRUCT from list
 *
 * RETURN-ERROR  = NULL - No PORTDLGSTRUCT found
 *
 ****************************************************************************/

PPORTDLGSTRUCT FindPortDlgStruct ( PPORTDLGSTRUCT pPortDlgStruct ) {

    PPORTDLGSTRUCT pOldPortDlgStruct = NULL;
    PSZ pszComputerName = NULL;
    PSZ pszPortName = NULL;
    BOOL fComputerMatch;

    if (pPortDlgStruct) {
        pszComputerName = pPortDlgStruct->pszComputerName;
        pszPortName = pPortDlgStruct->pszPortName;
       EnterPdrSem();
        pOldPortDlgStruct = pOpenDlgList;
        while (pOldPortDlgStruct) {
            fComputerMatch = FALSE;
            /*
             * Make sure computer names match
             */
            if ((pszComputerName && pOldPortDlgStruct->pszComputerName)) {
                if (!strcmp(pszComputerName,pOldPortDlgStruct->pszComputerName)) {
                    fComputerMatch = TRUE;
                }
            } else if (!pszComputerName && !(pOldPortDlgStruct->pszComputerName)) {
                fComputerMatch = TRUE;
            }
            /*
             * If computers match, make sure port names match
             */
            if (fComputerMatch &&
                   (pszPortName && pOldPortDlgStruct->pszPortName)) {
                if (!strcmp(pszPortName,pOldPortDlgStruct->pszPortName)) {
                    break;
                }
            }
            pOldPortDlgStruct = pOldPortDlgStruct->pNext;
        }
       LeavePdrSem();
    }

    return(pOldPortDlgStruct);

} /* end FindPortDlgStruct */

/****************************************************************************
 *
 * FUNCTION NAME = ValidatePortDlg
 *
 * DESCRIPTION   = Validate given Port dialog instance by checking its signature
 *
 * INPUT         = pPortDlgStruct instance pointer to validate
 *
 * OUTPUT        = pPortDlgStruct or NULL if invalid
 *
 * RETURN-NORMAL = pPortDlgStruct
 *
 * RETURN-ERROR  = NULL
 *
 * NOTE: This should be inside PdrSem() call to ensure DlgStruct is
 *       not freed while accessing it.
 *
 ****************************************************************************/

PPORTDLGSTRUCT ValidatePortDlg(PPORTDLGSTRUCT pPortDlgStruct)
{
    ULONG      rc;
    BIGXRR2    RegRec;      /* exception registration record              */
    PPIB       pPib ;       /* -> process ID block */
    PTIB       pTib ;       /* -> thread  ID block */


    // Set exception handler in case invalid pointer

    /*
     *  When setjmp returns a nonzero number, it is returning from a longjmp
     *  [an exception has occurred]
     */
    if ( setjmp(RegRec.JBuf)) {
       /*
        * We are back here after an exception
        * Return to caller
        */
       return (NULL);
    }

    /* Get thread information block pointer
     * Save this thread's exception chain pointer because we must
     *   restore this if we handle an exception
     */
    rc = DosGetInfoBlocks(&pTib, &pPib);

    if (rc) {

        RegRec.pTib         = (PTIB)NULL ;
        RegRec.pExceptChain = (PVOID)NULL ;

    } else {

        RegRec.pTib         = pTib ;
        RegRec.pExceptChain = pTib->tib_pexchain;
    }

    RegRec.pXRR = NULL;
    RegRec.Handler = &ExceptRoutine;

    rc = DosSetExceptionHandler((PEXCEPTIONREGISTRATIONRECORD)&RegRec);

    if (!pPortDlgStruct || (pPortDlgStruct->signature != PDLG_SIGNATURE) )
    {
       #ifdef DEBUG_ALERT
         char logbuf[260];

         DBPRINTF ((logbuf, "ValidatePortDlg invalid pPortDlgStruct=%lX", (ULONG)pPortDlgStruct));
       #endif
       pPortDlgStruct = NULL;
    }

    DosUnsetExceptionHandler( (PEXCEPTIONREGISTRATIONRECORD)&RegRec );
    return pPortDlgStruct;
}

/****************************************************************************
 *
 * FUNCTION NAME = InitWaitAlertBufs
 *
 * DESCRIPTION   = Allocate and initialize buffers to store alerts
 *
 * INPUT         = None
 *
 * OUTPUT        = Failure code if not zero
 *
 * NOTE          = Caller must be in the port driver semaphore on entry/exit
 *
 ****************************************************************************/
ULONG       InitWaitAlertBufs( VOID )
{
    ULONG  rc;
    #ifdef DEBUG_ALERT
      char logbuf[260];
    #endif

    /*
     * Must allocate WaitAlertBuffer for this port driver
     */
    rc = DosAllocMem((PVOID)&PdrWaitAlert.pCurrBuf,
                      DEF_WAITALERTBUF_SIZE,
                      PAG_READ|PAG_WRITE|PAG_COMMIT);
    if (rc) {
       #ifdef DEBUG_ALERT
        {
          sprintf( logbuf,
                   "InitWaitAlertBufs DosAllocMem failed rc=%d\r\n",
                   rc );
          LogCall( logbuf );
        }
       #endif /* DEBUG */
       PdrPanic ( 3 , 6 );
    } else {
       /*
       ** Initialize the waitalertbuf header
       */
       memset( PdrWaitAlert.pCurrBuf, 0, sizeof(WAITALERTBUF));
       PdrWaitAlert.pCurrBuf->cbBufSize = DEF_WAITALERTBUF_SIZE;
       PdrWaitAlert.pCurrBuf->cbBufUsed = sizeof(WAITALERTBUF) -
                                          sizeof(ALERTBUF);
       if (!PdrWaitAlert.pXlateBuf) {
          /*
           * Must allocate alternate WaitAlertBuffer for this port driver
           */
          rc = DosAllocMem((PVOID)&PdrWaitAlert.pXlateBuf,
                            DEF_WAITALERTBUF_SIZE,
                            PAG_READ|PAG_WRITE|PAG_COMMIT);
          if (rc) {
             #ifdef DEBUG_ALERT
              {
                sprintf( logbuf,
                         "InitWaitAlertBufs DosAllocMem failed rc=%d\r\n",
                         rc );
                LogCall( logbuf );
              }
             #endif /* DEBUG */
             PdrPanic ( 3 , 6 );
             DosFreeMem( PdrWaitAlert.pCurrBuf );
             PdrWaitAlert.pCurrBuf = NULL;
          } else {
             /*
             ** Initialize the alternate waitalertbuf header
             */
             memset( PdrWaitAlert.pXlateBuf, 0, sizeof(WAITALERTBUF));
             PdrWaitAlert.pXlateBuf->cbBufSize = DEF_WAITALERTBUF_SIZE;
             PdrWaitAlert.pXlateBuf->cbBufUsed = sizeof(WAITALERTBUF) -
                                                 sizeof(ALERTBUF);
             if (!PdrWaitAlert.hevWaitAlert) {
                rc = DosCreateEventSem ( 0,
                                         &(PdrWaitAlert.hevWaitAlert),
                                         0,
                                         FALSE );   // Create in Reset state
             }
          }
       }
    }
    return(rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = PdrError
 *
 * DESCRIPTION   = Routine called on a error condition in the port driver
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID PdrError ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = PdrPanic
 *
 * DESCRIPTION   = Routine called on a panic condition in the port driver
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID PdrPanic ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = PdrWarning
 *
 * DESCRIPTION   = Routine called on a warning condition in the protocol
 *                 converter
 *
 * INPUT         = ulCode      - Code of caller ( file dependent )
 *               = ulFile      - Number of file where caller is located
 *
 * OUTPUT        = None
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/

VOID PdrWarning ( ULONG ulCode, ULONG ulFile ) {

}

/****************************************************************************
 *
 * FUNCTION NAME = DisplayError
 *
 * DESCRIPTION   = Display error having string from the resource file.
 *
 * INPUT         = hwndOwner     - Owner of message box.
 *                                    if NULL, default is last active window.
 *                 usStringID    - ID of string in resource file.
 *                 usWinStyle    - Window style of message box.
 *                                    if NULL, default is MB_OK.
 *
 * OUTPUT        = User-response value returned by WimMessageBox API.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT DisplayError ( HWND hwndOwner,
                      USHORT usStringID,
                      USHORT usWinStyle )
{
   CHAR pszTitle[256];                       /*  Message-box window title   */
   CHAR pszText[256];                        /*  Message-box window message */
   USHORT usResponse;
   HAB    hAB;

   hAB = WinQueryAnchorBlock (HWND_DESKTOP);
   WinLoadString (hAB, hPdrMod, PORT_ERR_TITLE, 255, (PSZ)pszTitle);
   WinLoadString (hAB, hPdrMod, usStringID, 255, (PSZ)pszText);
   if (!hwndOwner)
   {
      hwndOwner = WinQueryActiveWindow (HWND_DESKTOP);
   }
   if (!usWinStyle)
   {
      usWinStyle = MB_OK;
   }
   usResponse = WinMessageBox (HWND_DESKTOP, hwndOwner, pszText, pszTitle, 1,
                                                           (ULONG)usWinStyle);
   return (usResponse);

}

/****************************************************************************
 *
 * FUNCTION NAME = CalcBufLength
 *
 * DESCRIPTION   = Determine how big buffer is needed to store all PORTNAMES
 *                 structures
 *
 * INPUT         = hab - anchor block handle
 *
 * OUTPUT        = length of buffer necessary to store all default port names
 *                 supported by this port driver.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG CalcBufLength ( HAB hab )
{
   ULONG cbRequired;
   USHORT usID;

   cbRequired = 0;

      /*
      ** calculate length required to fit all the port info.
      */
   for (usID = PORT_ID_FIRST; usID <= PORT_ID_LAST; usID += 2)
   {
      cbRequired += CalcStructLength (hab, usID);
   }

   return(cbRequired);
}

/****************************************************************************
 *
 * FUNCTION NAME = CalcStructLength
 *
 * DESCRIPTION   = Determine size of buffer needed to store PORTNAMES structure
 *                 for given string ID.
 *
 * INPUT         = hab     - anchor block handle
 *                 usID    - string ID for port name
 *
 * OUTPUT        = length of buffer necessary to store this port name
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG CalcStructLength ( HAB hab,
                         USHORT usID )
{
   ULONG cbRequired;
   CHAR chString[STR_LEN_PORTDESC];

   cbRequired = 0;

   WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, chString);
   cbRequired += strlen (chString) + 1;
   WinLoadString(hab, hPdrMod, (USHORT)(usID + 1), STR_LEN_PORTDESC, chString);
   cbRequired += strlen (chString) + 1;
   cbRequired += sizeof (PORTNAMES);
   return(cbRequired);
}

/****************************************************************************
 *
 * FUNCTION NAME = NumPortsCanFit
 *
 * DESCRIPTION   = Determine how many ports can fit in buffer
 *
 * INPUT         = hab     - anchor block handle
 *                 cbBuf   - size in bytes of buffer to hold PORTNAMES
 *
 * OUTPUT        = count of PORTNAMES structures that can fit in buffer
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

ULONG NumPortsCanFit ( HAB hab,
                       ULONG cbBuf )
{
   ULONG cbRequired;
   USHORT usID;
   ULONG ulNumPort;

   cbRequired = 0;
   ulNumPort = 0;

      /*
      ** calculate how many ports we can fit in buf.
      */
   for (usID = PORT_ID_FIRST; usID <= PORT_ID_LAST; usID += 2)
   {
      cbRequired += CalcStructLength (hab, usID);
      if (cbRequired > cbBuf)
      {
         return(ulNumPort);
      }
      ulNumPort++;
   }

   return(ulNumPort);
}

/****************************************************************************
 *
 * FUNCTION NAME = CopyNPorts
 *
 * DESCRIPTION   = Copy given number of ports into buffer
 *
 * INPUT         = hab     - anchor block handle
 *                 pBuf    - buffer to get PORTNAMES structures
 *                 ulReturned - number of ports to return
 *
 * OUTPUT        = pBuf is updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID  CopyNPorts ( HAB hab,
                   PCH pBuf,
                   ULONG ulReturned )
{
   USHORT usID;
   ULONG ulBeginText;
   ULONG ulBeginStruct;

   ulBeginText = ulReturned * sizeof (PORTNAMES);
   ulBeginStruct = 0;

   for (usID = PORT_ID_FIRST;
        usID <= PORT_ID_LAST && ulReturned;
        usID += 2, --ulReturned)
   {
      CopyStruct (hab, usID, pBuf, &ulBeginStruct, &ulBeginText);
   }
}

/****************************************************************************
 *
 * FUNCTION NAME = CopyStruct
 *
 * DESCRIPTION   = Copy single PORTNAMES structure to buffer
 *
 * INPUT         = hab     - anchor block handle
 *                 usID    - string ID for port to return
 *                 pBuf    - buffer to get PORTNAMES structures
 *                 pulBeginStruct - offset from begin of pBuf to store next
 *                                  PORTNAMES
 *                 pulBeginText   - offset from pBuf to store next string
 *
 * OUTPUT        = pBuf is updated
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID CopyStruct ( HAB hab,
                  USHORT usID,
                  PCH pBuf,
                  PULONG pulBeginStruct,
                  PULONG pulBeginText )
{
   PPORTNAMES pPortNames;

   pPortNames = (PPORTNAMES)(pBuf + *pulBeginStruct);
   *pulBeginStruct += sizeof (PORTNAMES);

      /*
      ** copy port name in the structure
      */
   pPortNames->pszPortName = pBuf + *pulBeginText;
   WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, pPortNames->pszPortName);
   *pulBeginText += strlen (pPortNames->pszPortName) + 1;

      /*
      ** copy port description to the structure
      */
   pPortNames->pszPortDesc = pBuf + *pulBeginText;
   WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, pPortNames->pszPortDesc);
   *pulBeginText += strlen (pPortNames->pszPortDesc) + 1;
}

/****************************************************************************
 *
 * FUNCTION NAME = GetPortDescription
 *
 * DESCRIPTION   = Get port description from our string resources.
 *
 * INPUT         = hab     - anchor block handle
 *                 pszPortName - name of port to get description for
 *                 pszPortDesc - gets port description
 *
 * OUTPUT        = TRUE  - if portname description is found
 *                 FALSE - if not
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL GetPortDescription ( HAB hab,
                          PSZ pszPortName,
                          PSZ pszPortDesc )
{
   USHORT usID;
   CHAR chBuf[STR_LEN_PORTDESC];

   for (usID = PORT_ID_FIRST; usID <= PORT_ID_LAST; usID += 2)
   {
      WinLoadString(hab, hPdrMod, usID, STR_LEN_PORTDESC, chBuf);
      if (!strcmp (pszPortName, chBuf))
      {
         strcpy (pszPortDesc, chBuf);
         return(TRUE);
      }
   }
   return(FALSE);
}

/****************************************************************************
 *
 * FUNCTION NAME = GetTimeOuts
 *
 * DESCRIPTION   = Retrieve timeouts for printer from system INI
 *
 * INPUT         = pszPort     -> name of infrared port
 *               = pPortInst   -> PORTINST structure to receive timeout values
 *
 *
 * RETURN-NORMAL = TRUE  - timeouts found
 *
 * RETURN-ERROR  = FALSE - timeouts not in INI, defaults used
 *
 * NOTE          = NOT in PDR Sem, but updates pPortInst timeouts
 *                 anyway.
 *
 *
 *
 ****************************************************************************/
BOOL GetTimeOuts( PSZ pszPort, PPORTINST pPortInst )
{
   CHAR     chAppName[CCHMAXPATH];


   strcpy (chAppName, APPNAME_LEAD_STR);
   strcat (chAppName, pszPort);

   pPortInst->ulPrintTimeOut               = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_PRINT,
                                                          DEF_TIMEOUT_PRINT_VALUE );
   pPortInst->ulNoQueryTimeOut             = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_QUERY,
                                                          DEF_TIMEOUT_QUERY_VALUE );
   pPortInst->ulNoJobTimeOut               = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_JOB,
                                                          DEF_TIMEOUT_JOB_VALUE );
   pPortInst->TimeOut.ulReadIntTimeOut     = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_READ_INTER,
                                                          DEF_TIMEOUT_READ_INTER );
   pPortInst->TimeOut.ulReadIdleTimeOut    = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_READ_ELAPSED,
                                                          DEF_TIMEOUT_READ_ELAPSED );
   pPortInst->TimeOut.ulWriteIntTimeOut    = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_WRITE_INTER,
                                                          DEF_TIMEOUT_WRITE_INTER );
   pPortInst->TimeOut.ulWriteIdleTimeOut   = GetIniValue( chAppName,
                                                          KEY_TIMEOUT_WRITE_ELAPSED,
                                                          DEF_TIMEOUT_WRITE_ELAPSED );
   pPortInst->ulModeSelected               = GetIniValue( chAppName,
                                                          KEY_MODE_SELECTED,
                                                          PARMODE_AUTODETECT );


   return(TRUE);
}


/****************************************************************************
 *
 * FUNCTION NAME = GetIniValue
 *
 * DESCRIPTION   = Retrieve numeric value from system INI
 *
 * INPUT         = pszApp      -> App Name in INI to query
 *                 pszKey      -> Key Name in INI to query
 *                 ulDefValue  -  Default value to return if no INI entry found
 *
 * OUTPUT        = numeric value from INI or ulDefValue if pszApp and pszKey
 *                 not found in System INI file.
 *
 * NOTE          = NOT in PDR Sem
 *                 If INI entry returns 0, this returns ulDefValue
 *
 ****************************************************************************/
ULONG GetIniValue( PSZ pszApp, PSZ pszKey, ULONG ulDefValue )
{
   BOOL     fSuccess;
   CHAR     chBuf[12];
   PSZ      pszSemi;
   ULONG    ulValue;


   ulValue  = 0;
   fSuccess = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                     pszApp,
                                     pszKey,
                                     NULL,
                                     chBuf,
                                     10 );
   if (fSuccess)
   {
     /*
     ** Remove trailing semi-colon
     */
     pszSemi = strchr( chBuf, ';' );
     if (pszSemi)
     {
        *pszSemi = '\0';
     }
     ulValue = atoi(chBuf);
   }
   if (!ulValue)
   {
     /*
     ** TIMEOUT_QUERY can have 0 set in INI file and it should not
     **  be overridden by the default timeout value.
     */
     if (!fSuccess || !strcmp(pszApp, KEY_TIMEOUT_QUERY) ) {
        ulValue = ulDefValue;
     }
   }

   return(ulValue);
}


/****************************************************************************
 *
 * FUNCTION NAME = RemoveLeadTrailBlanks
 *
 * DESCRIPTION   = Remove all the leading blanks and all the trailing blanks
 *                 from the source string. The target string contains no lead
 *                 or trail blanks. Source string is not altered.
 *
 * INPUT         = pTarget   - Target string.
 *                 pSource   - Source string.
 *
 * OUTPUT        = Void function.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 *      Note : Function accepts the same pointer for both source and target.
 *             This means that same buffer can be used as input and output.
 *
 ****************************************************************************/

VOID RemoveLeadTrailBlanks ( PCH pTarget,
                             PCH pSource )
{
   while (*pSource == ' ' || *pSource == '\t' || *pSource == '\n')
      pSource++;
   if (!(*pSource))
   {
      *pTarget = '\0';
      return;
   }
   while (*pSource)
      *pTarget++ = *pSource++ ;

   pTarget--;
   while (*pTarget == ' ' || *pTarget == '\t' || *pTarget == '\n')
      pTarget-- ;
   /*
   ** Set first blank to null
   */
   *++pTarget = '\0';

}

/****************************************************************************
 *
 * FUNCTION NAME = FormatDateTime
 *
 * DESCRIPTION   = Format the date and time into the given buffer in
 *                  the following format:
 *                    month-day hours:minutes:seconds
 *                  All fields will be 2 numbers(zero-preceded) and
 *                    the string will have 1 trailing blank
 *
 *                 Used by LogCall for debug purposes
 *
 * INPUT         = psz         - string buffer to get null terminated
 *                               formatted date and time
 * OUTPUT        = none        -
 *
 ****************************************************************************/
#ifdef DEBUG_ALERT

VOID FormatDateTime ( PSZ psz ) {

    DATETIME    DateTime;


    DosGetDateTime( &DateTime );
    psz[0]  = (DateTime.month / 10) + '0' ;
    psz[1]  = (DateTime.month % 10) + '0' ;
    psz[2]  = '-' ;
    psz[3]  = (DateTime.day   / 10) + '0' ;
    psz[4]  = (DateTime.day   % 10) + '0' ;
    psz[5]  = ' ' ;

    psz[6]  = (DateTime.hours / 10) + '0' ;
    psz[7]  = (DateTime.hours % 10) + '0' ;
    psz[8]  = ':' ;
    psz[9]  = (DateTime.minutes   / 10) + '0' ;
    psz[10] = (DateTime.minutes   % 10) + '0' ;
    psz[11] = ':' ;
    psz[12] = (DateTime.seconds   / 10) + '0' ;
    psz[13] = (DateTime.seconds   % 10) + '0' ;
    psz[14] = '.' ;
    psz[15] = (DateTime.hundredths   / 10) + '0' ;
    psz[16] = (DateTime.hundredths   % 10) + '0' ;
    psz[17] = ' ' ;
    psz[18] = '\0' ;

}

/****************************************************************************
 *
 * FUNCTION NAME = ShowHexByte
 *
 * DESCRIPTION   = Convert hex number to string
 *                 Used by DumpHex for debug purposes
 *
 * INPUT         = pszBuf    - buffer area to get converted number
 *                             This does NOT get null terminated!
 *                 bValue    - byte to format
 *
 * OUTPUT        = NONE
 *
 ****************************************************************************/
VOID ShowHexByte ( PSZ pszBuf, BYTE bValue )
{
    USHORT uValue;

    uValue = (USHORT)( bValue / 16 );

    if (uValue > 9)
       pszBuf[0] = (char) ((uValue - 10) + 'A');
    else
       pszBuf[0] = (char) (uValue + '0');

    uValue = (USHORT)( bValue % 16 );

    if (uValue > 9)
       pszBuf[1] = (char) ((uValue - 10) + 'A');
    else
       pszBuf[1] = (char) (uValue + '0');
}

/****************************************************************************
 *
 * FUNCTION NAME = ShowHexAscii
 *
 * DESCRIPTION   = Display hex byte as its Ascii value or as '.' if not valid
 *                 Used by DumpHex for debug purposes
 *
 * INPUT         = pszBuf    - buffer area to get converted number
 *                             This does NOT get null terminated!
 *                 bValue    - byte to format
 *
 * OUTPUT        = NONE
 *
 ****************************************************************************/
VOID ShowHexAscii ( PSZ pszBuf, BYTE bValue )
{

    if ( (bValue >= ' ') && (bValue <= '~') )
       pszBuf[0] = (char) (bValue);
    else
       pszBuf[0] = '.';
}

/****************************************************************************
 *
 * FUNCTION NAME = DumpHex
 *
 * DESCRIPTION   = Output hex/ascii representation of buffer into
 *                 log file
 *
 * INPUT         = pbHexChars  - buffer containing data to format
 *                 cb          - amount of data to format
 *
 * OUTPUT        = None
 *
 ****************************************************************************/
VOID DumpHex ( PBYTE pbHexChars, ULONG cb )
{
    CHAR   logbuf[70];
    ULONG  i;
    ULONG  numLines;
    ULONG  curLine;
    ULONG  charsLeftInLine;
    PBYTE  pbBytes;

    if (!cb) {
       return;
    }
    numLines = (cb / 16);

    if (cb % 16) {
      numLines++;
    }
    for (curLine = 1; curLine <= numLines; curLine++)
    {
       if (curLine < numLines) {
          charsLeftInLine = 16;
       } else {
          charsLeftInLine = cb % 16;
          if (charsLeftInLine == 0)
             charsLeftInLine = 16;
       }
       memset( logbuf, ' ', 70 );
       pbBytes = pbHexChars + ( (curLine-1) * 16 );
       for (i=0; i < charsLeftInLine; i++)
       {
          //
          // Create a line with following format:
          //
          // 00 11 22 33 44 55 66 77 88 99 AA BB CC DD EE FF  0123456789ABCDEF
          //
          // Where double numbers are hex values and
          //       single numbers are ascii representation of hex byte,
          //              or "." if not printable
          //
          ShowHexByte ( logbuf+(3*i), pbBytes[i] );
          ShowHexAscii( logbuf+48+i,  pbBytes[i] );

       }
       logbuf[48+16] = '\r';
       logbuf[49+16] = '\n';
       logbuf[50+16] = '\0';
       LogCall( logbuf );
    }
}

/****************************************************************************
 *
 * FUNCTION NAME = LogCall    - DEBUG API
 *
 * DESCRIPTION   = Log msg describing Bidi call in debug file C:\\ALERTIR
 *
 * INPUT         = pszMsg    - -> null terminated msg to log.
 *                                This can have newline characters.
 *
 * OUTPUT        = none
 *
 * NOTE          = Need not be in PdrSem on entry
 *
 ****************************************************************************/
VOID LogCall( PSZ pszMsg )
{

    ULONG       ActionTaken;
    ULONG       ulOpenAction;
    ULONG       cWritten;
    ULONG       rc ;
    CHAR        buf2[400];
    PPIB        ppib;
    PTIB        ptib;


    if (fLogAlerts)
    {
      /*
       * Get Local Process Info
       */
      DosGetInfoBlocks ( &ptib, &ppib );

      if ( hFileAlerts == 0)
      {
        ulOpenAction = OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_CREATE_IF_NEW ;
        //
        // Allow setting the debug alerts filename.
        // Default to debug console if not specified( "DB" )
        //
        buf2[0] = '\0';
        if (!PrfQueryProfileString(HINI_SYSTEMPROFILE,
                                   "PAR1284_DEBUG",
                                   "ALERTSFILE",
                                   (PSZ)NULL,
                                   buf2,
                                   sizeof(buf2) - 1) )
        {
           strcpy( buf2, "DB" );
        }
        //
        // Special filename "DB" makes us send to debug terminal
        //   using PMDD.SYS( need /C1 in config.sys )
        //
        if (strcmp(buf2, "DB") == 0 )
        {
           fDBterminal = TRUE;
           /**********************************/
           /* Open PMDD Device Driver Output */
           /**********************************/
           rc = DosOpen ("\\DEV\\SINGLEQ$",          /* File path name          */
                         &hFileAlerts,               /* File handle             */
                         &ActionTaken,               /* Action taken            */
                         0,                          /* File primary allocation */
                         FILE_NORMAL,                /* File attribute          */
                         OPEN_ACTION_OPEN_IF_EXISTS, /* Open function type      */
                         OPEN_FLAGS_NO_LOCALITY      /* Open mode of the file   */
                         | OPEN_FLAGS_NOINHERIT
                         | OPEN_SHARE_DENYNONE
                         | OPEN_ACCESS_READONLY,
                         NULL);                      /* Ptr to EA buffer        */
        } else {

           fDBterminal = FALSE;
           rc = DosOpen( buf2, &hFileAlerts, &ActionTaken,
                        0, 0,
                        ulOpenAction,
                        OPEN_FLAGS_NONSPOOLED | OPEN_FLAGS_FAIL_ON_ERROR |
                        OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYNONE,
                        NULL);
        }
        if (rc)
        {
          hFileAlerts = (ULONG)-1 ;
        }
      }
      if (hFileAlerts != (ULONG)-1)
      {
           FormatDateTime( buf2 );

           strcat( buf2, pszMsg );
           //
           // Re-use callers buffer - we know it is scratch buffer @BUGBUG
           //
           sprintf( buf2+strlen(buf2)-2, " PID=%d TID=%d\r\n",
                    ppib->pib_ulpid, ptib->tib_ptib2->tib2_ultid );

           if (fDBterminal)
           {
              //
              // Must use IOCtl to send to PMDD.SYS
              //
              ULONG  cbParmInOut = 0;
              ULONG  cbDataInOut = 0;

              DosDevIOCtl (hFileAlerts,       /* Device:   PMDD            */
                           0x03,              /* Catagory: SCRN/PTR DRAW   */
                           0x77,              /* Function: DEBUG OUTPUT    */
                           buf2,              /* Ptr to parms              */
                           strlen(buf2),      /* Parms Max Length          */
                           &cbParmInOut,      /* Ptr to Parm Length In/Out */
                           NULL,              /* Ptr to Data Area          */
                           0L,                /* Data Max Length           */
                           &cbDataInOut);     /* Ptr of Data Length In/Out */
           } else {

              /*
               * Append to file, don't overwrite other App's log entries
               */
              DosSetFilePtr( hFileAlerts, 0, FILE_END, &cWritten );
              DosWrite( hFileAlerts, buf2, strlen(buf2), &cWritten );
           }
      }
    } else {
      if ( (hFileAlerts != 0) && (hFileAlerts != (ULONG)-1) )
      {
         DosClose( hFileAlerts );
         hFileAlerts = 0;
      }

    }
}
#else
VOID LogCall( PSZ pszMsg )
{
}
VOID DumpHex ( PSZ pszHexChars, ULONG cb )
{
}
#endif /* DEBUG_ALERT */


/****************************************************************************
 *
 * FUNCTION NAME = ChkMem
 *
 * DESCRIPTION   = Checks for access to a specific area of memory
 *                 Uses exception handling to determine if valid parms
 *                   were passed.
 *                 If an invalid pointer was given
 *                   an exception is raised when attempting to use the ptr,
 *                   our exception handler is called and we return FALSE
 *
 * INPUT         = pMem - Pointer to the memory to check
 *               = usMemSize - Size of the memory to check (0 = 64K)
 *               = usFlags - What access should be checked(read=0 or write=1)
 *
 * RETURN-NORMAL = 1(Memory can be accessed)
 *
 * RETURN-ERROR  = 0(Memory cannot be accessed)
 *
 ****************************************************************************/

BOOL ChkMem ( PVOID pMem, USHORT usMemSize, USHORT usFlags ) {

   BOOL             fSuccess;  /* TRUE if memory OK, FALSE if not accessable */
   ULONG            rcExcept;
   BIGXRR2          RegRec;
   PPIB             pPib ;   /* -> process ID block */
   PTIB             pTib ;   /* -> thread  ID block */
   PSZ              pszBuf;
   CHAR             ch;


   /*
    *  If NULL pointer then just return failure
    *    without setting exception handler.
    */
   if (!pMem) {
     return(FALSE);
   }

   /*
    *  When setjmp returns a nonzero number, it is returning from a longjmp
    *  [an exception has occurred]
    */
   if ( setjmp(RegRec.JBuf)) {
      /*
       * We are back here after an exception
       * Return to caller
       */
      return (FALSE);
   }

   /* Get thread information block pointer
    * Save this thread's exception chain pointer because we must
    *   restore this if we handle an exception
    */
   rcExcept = DosGetInfoBlocks(&pTib, &pPib);

   if (rcExcept) {

       RegRec.pTib         = (PTIB)NULL ;
       RegRec.pExceptChain = (PVOID)NULL ;

   } else {

       RegRec.pTib         = pTib ;
       RegRec.pExceptChain = pTib->tib_pexchain;
   }

   RegRec.pXRR = NULL;
   RegRec.Handler = &ExceptRoutine;

   rcExcept = DosSetExceptionHandler((PEXCEPTIONREGISTRATIONRECORD)&RegRec);
   /*
    * For now, nothing to do if rcExcept != 0
    */

    /*
     * A 0 size really equals 64K [Can't Check for segments anymore]
     *    Just check first byte for now.
     */

    if (usMemSize == 0) {
        usMemSize = 1;
    }

   pszBuf = (PSZ)pMem ;
   fSuccess = TRUE ;

   /*
    * Determine type of access to check
    *
    * Assigning global variable for READ/WRITE checks, because local
    *   variable would optimize to not perform check.
    */
   if (usFlags == CHK_READ) {
       chMemChk[0] = pszBuf[0];
       chMemChk[1] = pszBuf[usMemSize-1];
   } else if (usFlags == CHK_WRITE) {
       ch = pszBuf[0];
       pszBuf[0] = ch;
       if (ch) {
          chMemChk[2] = ch;
       }
       ch = pszBuf[usMemSize-1];
       pszBuf[usMemSize-1] = ch;
       if (ch) {
          chMemChk[3] = ch;
       }
   } else {
       fSuccess = FALSE ;
   }

   if (!rcExcept) {
       /* if able to set exception handler, disable it now */
       rcExcept = DosUnsetExceptionHandler( (PEXCEPTIONREGISTRATIONRECORD)
                                            &RegRec) ;
   }

   return(fSuccess);
}


/****************************************************************************
 *
 * FUNCTION NAME = ChkStr
 *
 * DESCRIPTION   = Checks for read access to a specific string.
 *                 Note: This function will fail if the memory in question has
 *                       more than one type[ie. committed vs. uncommitted]
 *
 * INPUT         = psz - String to check for read access
 *
 * RETURN-NORMAL = TRUE(String can be accessed)
 *
 * RETURN-ERROR  = FALSE(String cannot be accessed)
 *
 ****************************************************************************/

BOOL ChkStr ( PSZ psz )
{
   ULONG            rcExcept;
   BIGXRR2          RegRec;
   PPIB             pPib ;   /* -> process ID block */
   PTIB             pTib ;   /* -> thread  ID block */


   /*
    *  If NULL pointer then just return failure
    *    without setting exception handler.
    */
   if (!psz) {
     return(FALSE);
   }

   /*
    *  When setjmp returns a nonzero number, it is returning from a longjmp
    *  [an exception has occurred]
    */
   if ( setjmp(RegRec.JBuf)) {
      /*
       * We are back here after an exception
       * Return to caller
       */
      return (FALSE);
   }

   /* Get thread information block pointer
    * Save this thread's exception chain pointer because we must
    *   restore this if we handle an exception
    */
   rcExcept = DosGetInfoBlocks(&pTib, &pPib);

   if (rcExcept) {

       RegRec.pTib         = (PTIB)NULL ;
       RegRec.pExceptChain = (PVOID)NULL ;

   } else {

       RegRec.pTib         = pTib ;
       RegRec.pExceptChain = pTib->tib_pexchain;
   }

   RegRec.pXRR = NULL;
   RegRec.Handler = &ExceptRoutine;

   rcExcept = DosSetExceptionHandler((PEXCEPTIONREGISTRATIONRECORD)&RegRec);
   /*
    * For now, nothing to do if rcExcept != 0
    */

   /*
    * Assigning global variable for READ/WRITE checks, because local
    *   variable would optimize to not perform check.
    */
   chMemChk[0] = (CHAR)strlen(psz);

   if (!rcExcept) {
       /* if able to set exception handler, disable it now */
       rcExcept = DosUnsetExceptionHandler( (PEXCEPTIONREGISTRATIONRECORD)
                                            &RegRec) ;
   }

   return(TRUE);
}

/****************************************************************************
 *
 * FUNCTION NAME = ExceptRoutine
 *
 * DESCRIPTION   = PAR1284's exception handling routine.
 *                 This takes care of thread exceptions while in PARALLEL.PDR
 *                 Called when an exception occurrs and we have set an
 *                 exception handler.
 *                 This will restore all registers to the values
 *                 when setjmp() was called, and return to just past setjmp()
 *                 call with error code 5.
 *
 * INPUT         = pReportRec -> exception report record, contains type
 *                               of exception.
 *                 pLocalXRR  -> our big exception registration record
 *                               containing the setjmp() structure to
 *                               restore our registers
 *                 pContext   -> context record(not used by us)
 *                 pVoid      -> not used by us
 *
 * RETURN-NORMAL = returns to caller with XCPT_CONTINUE_SEARCH if we
 *                 don't handle the exception
 *
 *
 * RETURN-ERROR  = reset context to time of setjmp() call and return
 *                 exception number to caller of setjmp()
 *
 *
 ****************************************************************************/

INT   APIENTRY ExceptRoutine(PEXCEPTIONREPORTRECORD pReportRec ,
                           PEXCEPTIONREGISTRATIONRECORD pLocalXRR,
                           PCONTEXTRECORD pContext,
                           PVOID pVoid)
{
   PBIGXRR2      pLocalRR;


   if (pReportRec->ExceptionNum == XCPT_ACCESS_VIOLATION) {

      pLocalRR = (PBIGXRR2) pLocalXRR;
      /*
       * Restore the exception chain for this thread to its value
       * when the thread called our API
       */
      if (pLocalRR->pTib) {
          pLocalRR->pTib->tib_pexchain = pLocalRR->pExceptChain;
      }
      longjmp (pLocalRR->JBuf,XCPT_ACCESS_VIOLATION);

   } else {

      return XCPT_CONTINUE_SEARCH;
   }
   /*
    * Never gets here, but quiet compiler
    */
   return XCPT_CONTINUE_SEARCH;
}

/****************************************************************************
 *
 * FUNCTION NAME = JulianDate
 *
 * DESCRIPTION   = Convert day, month to the Julian Date
 *
 * INPUT         = day - Current day of the month
 *               = month - Current month of the year
 *
 * OUTPUT        = Calculates the Julian Date using an array of monthly offsets
 *
 * RETURN-NORMAL = Results of the calculation
 *
 * RETURN-ERROR  = 0
 *
 ****************************************************************************/

ULONG JulianDate (UCHAR day, UCHAR month, USHORT year) {

    ULONG ulDay = 0;

    ulDay += (ULONG) day + JulianMonth [ month ];

    // Check for leap year and add one day if necessary

    if ((((year % 4) == 0) && ((year % 100) != 0)) && (month > FEB)) {
        ulDay++;
    } /* endif */
    return (ulDay);

}

/****************************************************************************
 *
 * FUNCTION NAME = time
 *
 * DESCRIPTION   = returns the number of seconds past Jan 1, 1970
 *
 * INPUT         = None
 *
 * OUTPUT        = Calls DosGetDateTime and converts the results to seconds
 *
 * RETURN-NORMAL = number of seconds past Jan 1, 1970
 *
 * RETURN-ERROR  = 0
 *
 ****************************************************************************/

ULONG time ( VOID ) {

    DATETIME DateTime;
    ULONG    ulTime;

    DosGetDateTime ( &DateTime );
    ulTime = DateTime.seconds;
    ulTime += (DateTime.minutes * SECPERMIN);
    ulTime += (DateTime.hours * SECPERHOUR);

    // Determine the Julian Date for the day, subtract 1 since you still are part
    // of that day

    ulTime += ((JulianDate (DateTime.day, DateTime.month, DateTime.year) - 1)
                 * SECPERDAY);

    // Determine interval in years from the start year

    ulTime += ((DateTime.year - STARTYEAR) * SECPERYEAR);

    // Check for the year 2000, it is not a leap year

    if (DateTime.year < 2000) {
        ulTime += ((((DateTime.year - STARTYEAR) + 2) / 4) * SECPERDAY);
    } else {
        ulTime += (((((DateTime.year - STARTYEAR) + 2) / 4) - 1) * SECPERDAY);
    }
    return ulTime;

}

/****************************************************************************
 *
 * FUNCTION NAME = GenerateUniquePort
 *
 * DESCRIPTION   = Generate a port name that is not yet in use
 *
 * INPUT         = pszPortName - buffer containing the default portname
 *                               This gets an ascii number appended to it
 *                               to make it a unique portname
 *
 * OUTPUT        = pszPortName gets updated
 *
 * RETURN-NORMAL = TRUE  - able to generate unique name
 *
 * RETURN-ERROR  = FALSE - unable to create unique port name
 *
 ****************************************************************************/

BOOL  GenerateUniquePortName( PSZ pszPortName )
{
    BOOL   fPortExists;
    PSZ    pszEndPortName;
    USHORT i;
    CHAR   chPortData[20];

    /*
    ** Generate a unique port name by adding numbers to the
    **   end of pszPortName
    ** Currently only 3 LPT ports are supported
    */
    pszEndPortName = pszPortName + strlen( pszPortName );
    i = 1;
    fPortExists = TRUE;
    while ( (i < 3) && fPortExists )
    {
       _itoa( i, pszEndPortName, 4);
       fPortExists = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                            APPNAME_PM_SPOOLER_PORT,
                                            pszPortName,
                                            NULL,
                                            chPortData,
                                            sizeof(chPortData) - 1);
       i++;
    }
    return(!fPortExists);
}

/*
** The following functions are for functions only exported by
**   the bidi spooler(PMSPL.DLL) available in a fixpak.
** These functions are not exported by the original Warp spooler,
**   so this port driver calls MyXXXX API instead of directly
**   calling an API in PMSPL.DLL that might not exist.
** Global(per-process) variable fLoadedPMSPL is set if we have
**   checked for the following APIs in PMSPL.DLL:
**   PrtQuery PrtSet SplProtSendCmd SplProtXlateCmd
*/
VOID LoadPMSPL( VOID )
{
   ULONG rc;
   HMODULE  hmod;
   CHAR  szFailName[80];

   rc = DosLoadModule( szFailName, 80, "PMSPL", &hmod ) ;
   if (rc == 0)
   {
     /*
      * Ordinal for PrtQuery is 603 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 603, NULL, (PFN *)&pfnPrtQuery ) ;
     /*
      * Ordinal for PrtSet is 604 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 604, NULL, (PFN *)&pfnPrtSet ) ;
     /*
      * Ordinal for SplProtSendCmd is 605 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 605, NULL, (PFN *)&pfnProtSendCmd ) ;
     /*
      * Ordinal for SplProtXlateCmd is 606 in PMSPL.DLL
      */
     rc = DosQueryProcAddr( hmod, 606, NULL, (PFN *)&pfnProtXlateCmd ) ;
   }
   fLoadedPMSPL = TRUE;
}

ULONG  MyPrtQuery ( PSZ    pszComputerName,
                    PSZ    pszDeviceName,
                    ULONG  ulType,
                    ULONG  ulCommand,
                    PVOID  pInData,
                    ULONG  cbInData,
                    PVOID  pOutData,
                    PULONG pcbOutData )
{
   ULONG rc;

   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnPrtQuery)
   {
      rc = (*pfnPrtQuery)( pszComputerName,
                           pszDeviceName,
                           ulType,
                           ulCommand,
                           pInData,
                           cbInData,
                           pOutData,
                           pcbOutData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}


ULONG MyPrtSet   ( PSZ    pszComputerName,
                   PSZ    pszDeviceName,
                   ULONG  ulType,
                   ULONG  ulCommand,
                   PVOID  pInData,
                   ULONG  cbInData )
{
   ULONG rc;


   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnPrtSet)
   {
      rc = (*pfnPrtSet)( pszComputerName,
                         pszDeviceName,
                         ulType,
                         ulCommand,
                         pInData,
                         cbInData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}

ULONG MySplProtSendCmd( PSZ    pszPortName,
                        ULONG  ulType,
                        ULONG  ulCommand,
                        PFN    pfnPdSendCmd,
                        PFN    pfnBaseProtSendCmd,
                        PVOID  pInData,
                        ULONG  cbInData,
                        PVOID  pOutData,
                        PULONG pcbOutData )
{
   ULONG rc;


   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnProtSendCmd)
   {
      rc = (*pfnProtSendCmd)( pszPortName,
                              ulType,
                              ulCommand,
                              pfnPdSendCmd,
                              pfnBaseProtSendCmd,
                              pInData,
                              cbInData,
                              pOutData,
                              pcbOutData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}

ULONG MySplProtXlateCmd( PSZ    pszPortName,
                         PFN    pfnBaseProtXlateCmd,
                         PVOID  pInData,
                         ULONG  cbInData,
                         PVOID  pAlertInfo,
                         PVOID  pOutData,
                         PULONG pcbOutData )
{
   ULONG rc;


   if (fLoadedPMSPL == FALSE)
   {
      LoadPMSPL();
   }
   if (pfnProtXlateCmd)
   {
      rc = (*pfnProtXlateCmd)( pszPortName,
                               pfnBaseProtXlateCmd,
                               pInData,
                               cbInData,
                               pAlertInfo,
                               pOutData,
                               pcbOutData );
   } else {
      rc = ERROR_NOT_SUPPORTED;
   }

   return(rc);
}

