/****************************************************************************/
/*  Copyright (C) 1995 IBM Corporation                                      */
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Sample BIDI Port Driver                                 */
/* FILENAME       : help.c                                                  */
/* DATE WRITTEN   : 02-08-94                                                */
/* DESCRIPTION    : Routines to implement help manager functions            */
/*                                                                          */
/****************************************************************************/

#include "pdrconst.h"
#include "pdrtypes.h"
#include "pdrproto.h"

/*
** We want to avoid automatically loading the help manager(HELPMGR.DLL),
**   since this takes up lots of memory.
** Do this by only linking to the HELPMGR if user selects help item.
** We replace the WinxxxxHelpInstance calls with our local versions.
** These versions use DosLoadModule() and DosQueryProcAddr() to
**   call the "real" help manager functions.
** The below function pointers, prototypes and variables are used
**   to this end.
*/

BOOL (* APIENTRY pfnWinAssociateHelpInstance)(HWND, HWND);
BOOL (* APIENTRY pfnWinCreateHelpInstance)(HAB, PHELPINIT);
BOOL (* APIENTRY pfnWinDestroyHelpInstance)(HWND);
BOOL (* APIENTRY pfnWinQueryHelpInstance)(HWND);

/*
** Global handle for helpmgr module, so it is only loaded once
*/
HMODULE  hvHelpmgrModule;

/*
** HELPINIT -- help manager initialization structure
*/
static HELPINIT hmiHelpData = {
    sizeof(HELPINIT),
    0L,
    (PSZ)     NULL,
    (PHELPTABLE) NULL,
    (HMODULE) NULL,
    (HMODULE) NULL,
    (USHORT)  NULL,
    (USHORT)  NULL,
    (PSZ)     NULL,
    CMIC_HIDE_PANEL_ID,
    (PSZ)     PDR_HELPFILE_NAME
};

/*
** Following routines replace Win help manager function calls.
** This is done to avoid automatically loading the help manager
** when the port driver is used.
** DosLoadModule is used to get the help manager function addresses
** and WinHook mechanism is used to get notified of F1 key.
**
** All CallxxHelpxx call equivalent WinxxHelpxx
*/
/****************************************************************************
 *
 * FUNCTION NAME = CALLAssociateHelpInstance
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL APIENTRY CALLAssociateHelpInstance ( HWND hwndHelpInstance,
                                          HWND hwndApp )
{
   ULONG    rc = 0;

   if (hvHelpmgrModule)
   {
      /*
      ** Check to see if we have the pointer from a previous call
      */
      if (!pfnWinAssociateHelpInstance)
      {
         /*
         ** Get pointer to the location of the function we want.
         ** We know ordinal 54 is Win32AssociateHelpInstance
         */
         rc = DosQueryProcAddr (hvHelpmgrModule,
                                     (ULONG)54,
                                     NULL,
                                     (PFN *)&pfnWinAssociateHelpInstance);
      }
      /*
      ** If no error continue.
      */
      if (!rc )
      {
         rc = (*pfnWinAssociateHelpInstance)(hwndHelpInstance, hwndApp);
         /*
         ** Function returns a bool
         */
         if (rc == TRUE)
            return(TRUE);
      }
   }
   return(FALSE);
}

/****************************************************************************
 *
 * FUNCTION NAME = CALLCreateHelpInstance
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

HWND APIENTRY  CALLCreateHelpInstance ( HAB hab,
                                        PHELPINIT phinitHMInitStructure )
{
   ULONG    rc = 0;
   HWND     hwnd = (HWND)NULLHANDLE;

      /*
      ** Check to see if we already have the handle
      */
   if (!hvHelpmgrModule)
      rc = DosLoadModule((PSZ)NULL, 0,
                         (PSZ)"HELPMGR", (PHMODULE)&hvHelpmgrModule);
   if (rc == 0)
   {
      if (!pfnWinCreateHelpInstance)
            /*
            ** Next get pointer to the location of the function we want.
            ** Ordinal 51 is Win32CreateHelpInstance
            */
         rc = DosQueryProcAddr (hvHelpmgrModule,
                                (ULONG)51,
                                NULL,
                                (PFN *)&pfnWinCreateHelpInstance);
         /*
         ** If no error continue.
         */
      if (!rc )
         hwnd = (*pfnWinCreateHelpInstance)(hab, phinitHMInitStructure );

   }
   return(hwnd);
}

/****************************************************************************
 *
 * FUNCTION NAME = CALLDestroyHelpInstance
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL APIENTRY   CALLDestroyHelpInstance ( HWND hwndHelpInstance )
{

   ULONG    rc = 0;

   if (hvHelpmgrModule)
   {
      if (!pfnWinDestroyHelpInstance)
            /*
            ** Next get pointer to the location of the function we want.
            ** Ordinal 52 is Win32DestroyHelpInstance
            */
         rc = DosQueryProcAddr (hvHelpmgrModule,
                                (ULONG)52,
                                NULL,
                                (PFN *)&pfnWinDestroyHelpInstance);
         /*
         ** If no error continue.
         */
      if (!rc )
      {
         rc = (*pfnWinDestroyHelpInstance)(hwndHelpInstance);
            /*
            ** Function returns a bool
            */
         if (rc == TRUE)
         {
           /* DosFreeModule(hvHelpmgrModule); */
            return(TRUE);
         }
      }
   }
   return(FALSE);
}

/****************************************************************************
 *
 * FUNCTION NAME = CALLQueryHelpInstance
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

HWND APIENTRY   CALLQueryHelpInstance ( HWND hwndApp )
{
   ULONG    rc = 0;
   HWND     hwnd = (HWND)NULLHANDLE;

   if (hvHelpmgrModule)
   {
      if (!pfnWinQueryHelpInstance)
            /*
            ** Get pointer to the location of the function we want.
            ** Ordinal 53 is Win32QueryHelpInstance
            */
         rc = DosQueryProcAddr (hvHelpmgrModule,
                                (ULONG)53,
                                NULL,
                                (PFN *)&pfnWinQueryHelpInstance);
         /*
         ** If no error continue.
         */
      if (!rc )
      {
            /*
            ** Make sure that the handle is associated with this instance
            **
            ** Make call
            */
         hwnd = (*pfnWinQueryHelpInstance)( hwndApp);

      }
   }
   return(hwnd);
}

/****************************************************************************
 *
 * FUNCTION NAME = SetHelpStubHook
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL EXPENTRY SetHelpStubHook()
{
    if(!HelpStubHookIsSet)
    {
        if(WinSetHook(0L, HMQ_CURRENT, HK_HELP, (PFN)HelpStubHook, 0L))
        {
            HelpStubHookIsSet = TRUE;
            return TRUE;
        }
    }
    return FALSE;
}

/****************************************************************************
 *
 * FUNCTION NAME = ReleaseHelpStubHook
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID EXPENTRY ReleaseHelpStubHook()
{
    if(HelpStubHookIsSet)
    {
        WinReleaseHook(0L, HMQ_CURRENT, HK_HELP, (PFN)HelpStubHook, 0L);
        HelpStubHookIsSet = FALSE;
    }
}

/****************************************************************************
 *
 * FUNCTION NAME = HelpStubHook
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

INT EXPENTRY HelpStubHook( HAB AppHAB,
                           USHORT Context,
                           USHORT IdTopic,
                           USHORT IdSubTopic,
                           PRECTL RectLPtr )
{

    InitializeHelp();

    return FALSE;
}

/****************************************************************************
 *
 * FUNCTION NAME = InitializeHelp
 *
 * DESCRIPTION   =
 *
 * INPUT         =
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID EXPENTRY InitializeHelp()
{
    HAB     hAB;
    HWND    hWnd;
    HWND    hWndActive;
    ULONG   rc = 0;
    PSZ     pszDot = NULL;
    CHAR    szBuf[256];
    CHAR    szHelpFile[CCHMAXPATH];

    if (HelpAlreadyInitialized)
       return;

       /*
       ** Initialize Help
       ** ---------------
       **
       ** Create an instance of the Help Manager, and associate it
       ** with the Frame.  If the Association fails, we handle it
       ** the same way as if the creation fails, ie hwndHelp
       ** (the Help Manager Object Window handle) is set to NULL.
       ** If we can't load the Module containing the Help Panel
       ** definitions, we forget Help altogether.
       */
    hWndActive = WinQueryActiveWindow(HWND_DESKTOP);
    hWnd = WinQueryWindow(hWndActive,QW_OWNER);
       /*
       ** if unable to get active window's owner
       **    use active window
       */
    if (hWnd == (HWND)NULL)
       hWnd = hWndActive ;

     hAB = (HAB)-1;
       /*
       ** Initialize a couple of the helpmgr structure elements
       ** First, get the title
       **
       ** Now load the title
       */
    WinLoadString (hAB, hPdrMod,
                   (USHORT)(PORT_HELP_TITLE), (SHORT)(256), szBuf);

    hmiHelpData.pszHelpWindowTitle = (PSZ)szBuf;
    /*
     * Get module name and change to help file by replacing .DRV with
     *     .HLP
     *  If an error occurs, just try the PDR_HELPFILE_NAME
     */
    rc = DosQueryModuleName ( hPdrMod, CCHMAXPATH, szHelpFile );
    if (rc) {
        hmiHelpData.pszHelpLibraryName = PDR_HELPFILE_NAME;
    } else {
        /*
         * Find last dot in szHelpFile, should be ".DRV"
         */
        pszDot = strrchr ( szHelpFile, '.' );
        if (pszDot) {
            strcpy ( pszDot, PDR_HELPFILE_EXT );
            hmiHelpData.pszHelpLibraryName = (PSZ)szHelpFile;
        } else {
            hmiHelpData.pszHelpLibraryName = PDR_HELPFILE_NAME;
        }
    }

    hAB = WinQueryAnchorBlock(hWnd);

       /*
       ** Only create a handle if we don't have one.
       */
    if (hwndHelp == 0L)
       hwndHelp = CALLCreateHelpInstance(hAB, &hmiHelpData);

       /*
       ** Always associate the helpmgr handle with the active window
       */
    if (hwndHelp != 0L)
    {
       if(!CALLAssociateHelpInstance(hwndHelp, hWnd) )
       {
           CALLDestroyHelpInstance(hwndHelp);
           hwndHelp = (HWND)NULL;
       }
    }

       /*
       ** If help was initialized, get rid of our hook. Otherwise, we have
       ** to ensure that our stub hook is the FIRST hook in the HK_HELP
       ** hook chain.
       */
    if(hwndHelp != 0L)
    {
        HelpAlreadyInitialized = TRUE;
        ReleaseHelpStubHook();
    }
    else
    {
        ReleaseHelpStubHook();
        SetHelpStubHook();
    }
}
