#define INCL_DEV      // include all DEV definitions
#define INCL_WIN
#define INCL_GPI      // include all Gpi definitions
#define INCL_ERRORS   // include all Gpi error definitions
#define INCL_GPIPRIMITIVES
#include <os2.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <math.h>
#include <pmord.h>
#include "PTTDLLDF.h"               // test case defines
#include "common1.h"
#include "common2.h"
#include "gpielmnt.h"

/*****************************************************************************
 *\\ddd
 * Subtask Name: GREAT
 *
 * Module Name: GPIELMNT
 *
 * Purpose: This module is called by PTT to test the Gpi Element API's,
 *       not including return codes.
 *
 * Revision Log: 9/3/91, GREAT, Stanley Logan
 *          - created initial module template
 *
 *
 * SRS References: Section ???
 *
 * Contents:
 *
 *   GpiElementTestApp1
 *   GpiElementTestApp2
 *
 *   GpiBeginAndEndElementExh
 *   GpiDeleteElementExh
 *   GpiDeleteElementRangeExh
 *   GpiDeleteElmntsBetweenLabelsExh
 *   GpiElementAndQryExh
 *   GpiGetAndPutDataExh
 *   GpiLabelAndSetElmtPtrAtLabelExh
 *   GpiOffsetElementPointerExh
 *   GpiQueryElementTypeExh
 *   GpiSetAndQryEditModeExh
 *   GpiSetAndQryElementPointerExh
 *
 *
 * Design Overview:
 *    This testcase is designed to test the Gpi element functions that:
 *
 *    1. Open an element bracket.
 *    2. Close an element bracket.
 *    3. Delete the element referenced by the current value of the element
 *       pointer.
 *    4. Delete a section of elements inclusively delimited by element pointer
 *       values.
 *    5. Delete a section of elements exclusively delimited by labels.
 *    6. Add a single element to the current segment.
 *    7. Query the graphics orders for a single element.
 *    8. Get the graphics orders for a specified segment.
 *    9. Add a buffer of graphics orders to the current segment.
 *   10. Insert a label into the current segment.
 *   11. Move the element pointer by a specified increment.
 *   12. Query information associated with the current element.
 *   13. Set the current edit mode to either insert or replace.
 *   14. Query the current edit mode setting.
 *   15. Set the value of the element pointer.
 *   16. Query the value of the element pointer.
 *   17. Set the value of the element pointer to point at a specified label.
 *
 *    This testcase includes application tests that are designed to test the
 *    behavior of these character functions with other Gpi's.
 *
 *    Also, testing includes both exhaustive and return error codes
 *    functions for each and every GPI function in this testcase. (Return
 *    error codes are tested in a supplementary module).
 *
 * Limitations:
 *
 *    All exhaustive element functions must be executed within a normal PS.
 *
 *\\end
 *****************************************************************************/
/*****************************************************************************
 *\ddd
 * Routine Name: InitTest
 *
 * Purpose: Initialize function name and address array for ptt.
 *
 * System Requirements: NONE
 *
 * Revision Log: 09/03/91, GREAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design
 *
 * Inputs:
 *   sel           //passed through to the InitTestSegVars function
 *   init_info_p   // name and address array used by ptt interface
 *
 * Outputs:
 *   init_info_p   // returned initialized with names and addresses
 *
 * Subroutines Required:
 *   InitTestSegVars()    // used by the ptt interface
 *
 * Limitations:
 *   init_info_p is of size ENTRYPOINTS defined in ptt.h and pttdlldf.h(both)
 *
 *\\end
 *****************************************************************************/
/*****************************************************************************
 *\\ algorithm
 * {
 *    for all test functions in this test case {
 *        init_info_p->function name = function name[i]
 *        init_info_p->function address = function address[i]
 *    }
 *    call inittestsegvars routine
 * }
 *\\end
 *****************************************************************************/

VOID APIENTRY InitTest(PVOID Selector, PTESTINITINFO init_info_p)
{
   register USHORT usTestCounter;

   init_info_p->ntests = NTESTS;
   strcpy(init_info_p->group_name, TestCaseName); //Testcase Name

   for (usTestCounter=0 ;usTestCounter < NTESTS; ++usTestCounter){

      init_info_p->test_addrs[usTestCounter] =
                EntryPointList[usTestCounter].TestAddress;
      strcpy(init_info_p->test_names[usTestCounter],
                EntryPointList[usTestCounter].TestName);
   }

   InitTestSegVars(Selector);  /* this call used by the ptt interface */

}



/***************************************************************************
 *\\ddd
 * Routine Name: GpiElementTestApp1
 *
 * Purpose:  This is an entry point for PTT that causes the first
 *           application test for the element GPI's to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   SETENTER
 *   SETEXIT
 *   SUBENTER
 *   SUBEXIT
 *   SHOWERROR
 *   DrawPyramid
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Log setup completion
 * Log subentry
 * Build pyramid segment consisting of one inch tall boxes
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Draw the pyramid segment
 * Log subexit
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiElementTestApp1(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiElementTestApp1";   // entry point name
   LONG           ErrorCount = 0;                   // error count
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Log setup completion

      SETEXIT();

// Log subentry

      SUBENTER(GpiElementTestApp1, "Pyramid with one inch boxes");

// Build pyramid segment consisting of translated copies of one element

      DrwPyramid(ONE_INCH);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_DISPLAY, DCTL_ON) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Log subexit

      SUBEXIT();

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}


/***************************************************************************
 *\\ddd
 * Routine Name: GpiElementTestApp2
 *
 * Purpose:  This is an entry point for PTT that causes the second
 *           application test for the element GPI's to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   SETENTER
 *   SETEXIT
 *   SUBENTER
 *   SUBEXIT
 *   SHOWERROR
 *   DrawPyramid
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Log setup completion
 * Log subentry
 * Build pyramid segment consisting of one-half inch tall boxes
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Draw the pyramid segment
 * Log subexit
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiElementTestApp2(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiElementTestApp2";   // entry point name
   LONG           ErrorCount = 0;                   // error count
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Log setup completion

      SETEXIT();

// Log subentry

      SUBENTER(GpiElementTestApp2, "Pyramid with one-half inch boxes");

// Build pyramid segment consisting of translated copies of one element

      DrwPyramid(HALF_INCH);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_DISPLAY, DCTL_ON) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Log subexit

      SUBEXIT();

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}


/***************************************************************************
 *\\ddd
 * Routine Name: GpiBeginAndEndElementExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           tests for GpiBeginElement and GpiEndElement to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   ERRSTATUS
 *   SETENTER
 *   SETEXIT
 *   SHOWERROR
 *   SUBENTER
 *   SUBEXIT
 *   randL
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set random number generator seed for reproducibility
 * Allocate space for the element description
 * Start loop from 0 to 10
 *    Set drawing mode to RETAIN
 *    Open graphics segment
 *    Log subentry
 *    Open an element bracket
 *    Start loop from 1 to 2 raised to the power of the outer loop index
 *       Set color to a random value between 1 and 15, inclusive
 *       Choose a random point within the viewing area
 *       Move current position to first point of line
 *       Increment y coordinate of point
 *       Draw a line from the current position up one pel
 *    Close the element bracket
 *    Log subexit
 *    Close the segment
 *    Enable output to the output medium
 *    Set drawing mode to DRAW-AND-RETAIN
 *    Draw the segment
 * Free space allocated for the element description
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiBeginAndEndElementExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiBeginAndEndElement";  // entry point name
   PSZ            ElementDesc;
   LONG           ElementID,
                  MaxPels,
                  PelNumber,
                  Color,
                  ErrorCount = 0;                     // error count
   SIZEL          sizePS;
   POINTL         ptlPel;
   LONG           tempval; //Temp val for power

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set random number generator seed to 1 for reproducibility

      srand(1);

// Allocate space for the element description

      if ((ElementDesc=(PSZ)malloc(50))==NULL)
         SHOWERROR("malloc", EntryName);

// Start loop to exhaustively go from 0 to 10

      for (ElementID = 0; ElementID <= 10; ElementID++)
      {

// Set drawing mode to RETAIN

         if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
            SHOWERROR("GpiSetDrawingMode", EntryName);

// Open segment

         if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
            SHOWERROR("GpiOpenSegment", EntryName);

// Log setup completion

         if (ElementID == 0)
            SETEXIT();

// Log subentry

         sprintf(LogText, "Element bracket #%d", ElementID);

         SUBENTER(GpiBeginAndEndElementExh, LogText);

// Open an element bracket

         if (GpiBeginElement(hPS, BOX_ELEMENT, ElementDesc) == FALSE)
            ErrorCount++;

         ERRSTATUS(TRUE, EntryName, "GpiBeginElement");

// Set upper limit on number of pels to turn on
// Compiler bug pow does not work properly, manual fix needed - BSC

         tempval = 0;
         MaxPels = 1;
         while (tempval != ElementID) {
            MaxPels *= 2;
            tempval++;
         } /* endwhile */

//         MaxPels = (LONG)pow(2.0, (double)ElementID);

// Start loop to iterate 2^n (n=0,..,10) times

         for (PelNumber = 1; PelNumber <= MaxPels;   PelNumber++)
         {

// Randomly choose a color to select and select it

            Color = randL(1L, 15L);

            if (GpiSetColor(hPS, Color) == FALSE)
               SHOWERROR("GpiSetColor", EntryName);


// Choose a random point within the viewing area

            ptlPel.x = randL(MapX(25), MapX(100 * (pg_size_x - 0.25)));
            ptlPel.y = randL(MapY(100), MapY(100 * (pg_size_y - 0.25)));

// Move current position to first point of line

            if (GpiMove(hPS, &ptlPel) == FALSE)
               SHOWERROR("GpiMove", EntryName);

// Increment y coordinate of point

            ptlPel.y += 1L;

// Draw a line from the current position to one pel above the current position

            if (GpiLine(hPS, &ptlPel) == GPI_ALTERROR)
               SHOWERROR("GpiLine", EntryName);

         }

// Close the element bracket

         if (GpiEndElement(hPS) == FALSE)
            ErrorCount++;

         ERRSTATUS(TRUE, EntryName, "GpiEndElement");

// Log subexit

         SUBEXIT();

// Close the segment

         if (GpiCloseSegment(hPS) == FALSE)
            SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

         if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
            SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

         if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
            SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the segment

         if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
            SHOWERROR("GpiDrawSegment", EntryName);

      }

// Free up allocated memory

      free(ElementDesc);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}


/***************************************************************************
 *\\ddd
 * Routine Name: GpiDeleteElementExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           test for GpiDeleteElement to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   ERRSTATUS
 *   MAKEFIXED
 *   SETENTER
 *   SETEXIT
 *   SUBENTER
 *   SUBEXIT
 *   SHOWERROR
 *   DrawPyramid
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Build the pyramid with half-inch boxes
 * Set drawing mode to RETAIN
 * Open pyramid segment
 * Log setup completion
 * Start loop to step through the levels
 *    Start loop to step through the boxes to be deleted in current level
 *       if (level = 1 + (n * 4)), skip two center boxes
 *       if (level = 3 + (n * 4)), delete two center boxes
 *       Build box label
 *       Set element pointer at label
 *       Increment element pointer by 2 to skip label and xform graphic ords.
 *       Log subentry
 *       Delete element bracket
 *       Log subexit
 *       Log subentry
 *       Delete model transform
 *       Log subexit
 *       Log subentry
 *       Delete label
 *       Log subexit
 * Close the segment
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Draw the segment
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiDeleteElementExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiDeleteElementExh";   // entry point name
   LONG           delete_index,
                  box_across = 1,
                  ErrorCount = 0,                    // error count
                  level;
   SIZEL          sizePS;
   FIXED          label;
   extern USHORT  Max_Box_Level;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Build the pyramid with half-inch boxes

      DrwPyramid(HALF_INCH);

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open pyramid segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Log setup completion

      SETEXIT();

// Start loop to step through the levels (first level= 2), deleting the
// desired boxes

      for (level = 2; level < (LONG)Max_Box_Level; level++)
      {

// Start loop to step through the boxes in current level

         for (delete_index = 1; delete_index <= level; delete_index += 2)
         {

// If (level = 1 + (n * 4)), skip two center boxes

            if (((level % 4) == 1) && (delete_index == ((level / 2) + 1)))
               delete_index += 1;

// If (level = 3 + (n * 4)), delete two center boxes

            if (((level % 4) == 3) && (delete_index == ((level / 2) + 2)))
               delete_index -= 1;

// Build box label

            label = MAKEFIXED(level, delete_index);

// Set element pointer at label

            if (GpiSetElementPointerAtLabel(hPS, (LONG)label) == FALSE)
               SHOWERROR("GpiSetElementPointerAtLabel", EntryName);

// Increment element pointer by 2 to skip Label and Transform graphic orders

            if (GpiOffsetElementPointer(hPS, 2L)==FALSE)
               SHOWERROR("GpiOffsetElementPointer", EntryName);

// Log subentry

            sprintf(LogText, "Deleting box (%ld, %ld)", level, delete_index);

            SUBENTER(GpiDeleteElementExh, LogText);

// Delete element bracket

            if (GpiDeleteElement(hPS) == FALSE)
               ErrorCount++;

            ERRSTATUS(TRUE, EntryName, "GpiDeleteElement");

// Log subexit

            SUBEXIT();

// Log subentry

            sprintf(LogText, "Deleting model transform");

            SUBENTER(GpiDeleteElementExh, LogText);

// Delete model transform

            if (GpiDeleteElement(hPS) == FALSE)
               ErrorCount++;

            ERRSTATUS(TRUE, EntryName, "GpiDeleteElement");

// Log subexit

            SUBEXIT();

// Log subentry

            sprintf(LogText, "Deleting label");

            SUBENTER(GpiDeleteElementExh, LogText);

// Delete label

            if (GpiDeleteElement(hPS) == FALSE)
               ErrorCount++;

            ERRSTATUS(TRUE, EntryName, "GpiDeleteElement");

// Log subexit

            SUBEXIT();

         }

      }

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: GpiDeleteElementRangeExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           test for GpiDeleteElementRange to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   ERRSTATUS
 *   MAKEFIXED
 *   SETENTER
 *   SETEXIT
 *   SUBENTER
 *   SUBEXIT
 *   SHOWERROR
 *   DrawPyramid
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Build the pyramid with half-inch boxes
 * Set drawing mode to RETAIN
 * Open pyramid segment
 * Log setup completion
 * Start loop to step through the all pyramid levels
 *    Build label for first box on this iteration's level
 *    Set element pointer to label representing first box on this level
 *    Query element pointer to determine current pointer position
 *    Log subentry
 *    Delete all inner boxes on this level
 *    Log subexit
 * Close the segment
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Draw the segment
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiDeleteElementRangeExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiDeleteElementRangeExh"; // entry point name
   extern USHORT  Max_Box_Level;
   LONG           label,
                  level,
                  ErrorCount = 0,                       // error count
                  ElementPtr;
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Build the pyramid with half-inch boxes

      DrwPyramid(HALF_INCH);

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open pyramid segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Log setup completion

      SETEXIT();

// Start loop to step through the levels (first level = 2), deleting every
// inner box on this iteration's level

      for (level = 2; level < (LONG)(Max_Box_Level - 1); level++)
      {

// Build label for first box on this iteration's level

         label = MAKEFIXED(level, 0L);

// Set element pointer to first box on this iteration's level

         if (GpiSetElementPointerAtLabel(hPS, (LONG)label)==FALSE)
            SHOWERROR("GpiSetElementPointer", EntryName);

// Query element pointer to determine current pointer position

         if ((ElementPtr = GpiQueryElementPointer(hPS))==GPI_ALTERROR)
            SHOWERROR("GpiQueryElementPointer", EntryName);

// Log subentry

         sprintf(LogText, "Deleting inner boxes from level %ld)", level);

         SUBENTER(GpiDeleteElementRangeExh, LogText);

// Delete all inner boxes on current level

         if (GpiDeleteElementRange(hPS, ElementPtr + 4,
                                            ElementPtr + (level * 3))== FALSE)
            ErrorCount++;

         ERRSTATUS(TRUE, EntryName, "GpiDeleteElementRange");

// Log subexit

         SUBEXIT();

      }

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: GpiDeleteElmntsBetweenLabelsExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           test for GpiDeleteElementsBetweenLabels to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   ERRSTATUS
 *   MAKEFIXED
 *   SETENTER
 *   SETEXIT
 *   SUBENTER
 *   SUBEXIT
 *   SHOWERROR
 *   DrawPyramid
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Build the pyramid with half-inch boxes
 * Set drawing mode to RETAIN
 * Open pyramid segment
 * Log setup completion
 * Start loop to step through the levels, deleting every other level
 *    Build label for first box on this iteration's level
 *    Build label for first box on next level
 *    Log subentry
 *    Delete all elements between the label on the current level and the label
 *       on the next level (exclusive)
 *    Log subexit
 *    Delete the level label for the current label
 * Close the segment
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Draw the segment
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiDeleteElmntsBetweenLabelsExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiDelElmntsBtwnLabelsExh"; // entry point name
   extern USHORT  Max_Box_Level;
   LONG           label1,
                  label2,
                  level,
                  ErrorCount = 0;                        // error count
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Build the pyramid with half-inch boxes

      DrwPyramid(HALF_INCH);

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open pyramid segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Log setup completion

      SETEXIT();

// Start loop to step through the levels (first level = 0), deleting every
// other line of boxes

      for (level = 0; level < (LONG)(Max_Box_Level - 1); level += 2)
      {

// Build label for first box on this iteration's level

         label1 = MAKEFIXED(level, 0L);

// Build label for first box on next level

         label2 = MAKEFIXED(level + 1, 0L);

// Log subentry

         sprintf(LogText, "Deleting level %ld", level);

         SUBENTER(GpiDeleteElmntsBetweenLabelsExh, LogText);

// Delete all boxes between the level label on the current level and the level
// label on the next level (exclusive)

         if (GpiDeleteElementsBetweenLabels(hPS, label1, label2)== FALSE)
            ErrorCount++;

         ERRSTATUS(TRUE, EntryName, "GpiDeleteElementsBetweenLabels");


// Log subexit

         SUBEXIT();

// Delete the level label for the current level

         if (GpiDeleteElement(hPS)==FALSE)
            SHOWERROR("GpiDeleteElement", EntryName);

      }

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}

 /***************************************************************************
 *\\ddd
 * Routine Name: GpiElementAndQryExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           tests for GpiElement and GpiQueryElement to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ABORTENTRY
 *   ENTER_MAIN
 *   EXIT_MAIN
 *   ERRSTATUS
 *   LOGINFO
 *   MAKEFIXED
 *   MAKEP
 *   SETENTER
 *   SETEXIT
 *   SUBENTER
 *   SUBEXIT
 *   SHOWERROR
 *   DrawPyramid
 *   MapX
 *   MapY
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set random seed for reproducibility
 * Allocate 64K segment for the element description string
 * Allocate 64K segment for the element data
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Open an element bracket
 * Insert 60K of comment data
 * Close the element bracket
 * Determine the length of the element data
 * Log setup completion
 * Log subentry
 * Query the element contents
 * Log subexit
 * Log subentry
 * Insert the element contents into the segment
 * Log subexit
 * Log subentry
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiElementAndQryExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiElementAndQryExh"; // entry point name
   LONG           ErrorCount,
                  ElementType,
                  index,
                  RetLength,
                  buffer_length = 0xffff;
   extern LONG    ElementBufferSize;
   SIZEL          sizePS;

   PVOID          pMemory,
                  pMemory2;

   PSZ            ElementDesc;
   PBYTE          pbData;
   MATRIXLF       MatStruct;
   POINTL         ptl[100];

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set random seed for reproducibility

      srand(5);

// Allocate 64K segment for the element description string


      //32 bit code memory allocation below
        if (DosAllocMem((PVOID)&pMemory,0xffff,
                        PAG_COMMIT | PAG_WRITE | PAG_READ) == NO_ERROR)
           ElementDesc = pMemory;
         else
            SHOWERROR("DosAllocMemory", EntryName);


// Allocate 64K segment for the element data


      //32 bit code memory allocation below
        if (DosAllocMem((PVOID)&pMemory2,0xffff,
                        PAG_COMMIT | PAG_WRITE | PAG_READ) == NO_ERROR)
          pbData = pMemory2;
        else
            SHOWERROR("DosAllocMemory", EntryName);

      sprintf(ElementDesc, "63 KB element");

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Open an element bracket

      if (GpiBeginElement(hPS, BOX_ELEMENT, ElementDesc) == FALSE)
         SHOWERROR("GpiBeginElement", EntryName);

// Insert 60000 bytes of comment data

      for (index = 0; index < 20000; index++)
         if (GpiComment(hPS, 1L, "a") == FALSE)
            SHOWERROR("GpiComment", EntryName);

// Close the element bracket

      if (GpiEndElement(hPS) == FALSE)
         SHOWERROR("GpiEndElement", EntryName);

// Determine the length of the element data

      ElementBufferSize = GpiQueryElementType(hPS, &ElementType, 0xffffL,
                                                                  ElementDesc);

      if (ElementBufferSize == GPI_ALTERROR)
         SHOWERROR("GpiQueryElementType", EntryName);

// Log setup completion

      SETEXIT();

// Log subentry

      sprintf(LogText, "Querying comment data...");

      SUBENTER(GpiQueryElement, LogText);

// Query the element contents

      if ((RetLength = GpiQueryElement(hPS, 0L, ElementBufferSize, pbData))
                                                               == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElement");

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Inserting comment data...");

      SUBENTER(GpiElement, LogText);

// Insert the element contents into the segment

      if (GpiElement(hPS, BOX_ELEMENT, ElementDesc, RetLength, pbData)
                                                                  == GPI_ERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiElement");

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Querying data with increasing offsets...");

      SUBENTER(GpiQueryElement, LogText);

      for (index = 1; index < 60000L; index *= 2L)
      {

// Query the element contents

         if ((RetLength = GpiQueryElement(hPS, index,
                           ElementBufferSize - index, pbData)) == GPI_ALTERROR)
            ErrorCount++;

         ERRSTATUS(TRUE, EntryName, "GpiQueryElement");

// If the number of bytes queried is not equal to the number of bytes returned,
// log an error

         if ((ElementBufferSize - index) != RetLength)
         {
            sprintf(LogText, "Error with GpiQueryElement.  Amount of data q"
                             "ueried does not correspond to expected amount"
                             " of data.");

            LOGINFO(4, "*", LogText);

            ErrorCount++;
         }
      }

// Log subexit

      SUBEXIT();

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Randomly create 76 points for 75 line polyline

      for (index = 0; index < 76; index++)
      {
         ptl[index].x = randL(0, MapX(pg_size_x * delta_x));
         ptl[index].y = randL(MapY(((pg_size_y / 2) + 1.5) * delta_y),
                                            MapY((pg_size_y - 0.5) * delta_y));
      }

// Move to the first random point

      if (GpiMove(hPS, &ptl[0]) == GPI_ALTERROR)
         SHOWERROR("GpiMove", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Open an element bracket

      if (GpiBeginElement(hPS, BOX_ELEMENT, ElementDesc) == FALSE)
         SHOWERROR("GpiBeginElement", EntryName);

// Insert the polyline into the segment

      if (GpiPolyLine(hPS, 75L, &ptl[1]) == GPI_ERROR)
         SHOWERROR("GpiPolyLine", EntryName);

// Close the element bracket

      if (GpiEndElement(hPS) == FALSE)
         SHOWERROR("GpiEndElement", EntryName);

// Query to determine the size of the element data

      ElementBufferSize = GpiQueryElementType(hPS, &ElementType, 0xffffL,
                                                                  ElementDesc);

      if (ElementBufferSize == GPI_ALTERROR)
         SHOWERROR("GpiQueryElementType", EntryName);

// Log subentry

      sprintf(LogText, "Querying poly line element data...");

      SUBENTER(GpiQueryElement, LogText);

// Query the element contents

      if ((RetLength = GpiQueryElement(hPS, 0L, ElementBufferSize, pbData))
                                                               == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElement");

// Log subexit

      SUBEXIT();

// Set Model Transform Matrix for this particular iteration's box

      MatStruct.fxM11 = MAKEFIXED(1, 0);
      MatStruct.fxM12 = MAKEFIXED(0, 0);
      MatStruct.lM13 = 0L;
      MatStruct.fxM21 = MAKEFIXED(0, 0);
      MatStruct.fxM22 = MAKEFIXED(1, 0);
      MatStruct.lM23 = 0L;
      MatStruct.lM31 = 1L;
      MatStruct.lM32 = -MapY((pg_size_y * delta_y) / 2);
      MatStruct.lM33 = 1L;

// Set model transform for this element

      if (GpiSetModelTransformMatrix(hPS, 9L, &MatStruct,TRANSFORM_REPLACE)
                                                                  == FALSE)
         SHOWERROR("GpiSetModelTransformMatrix", EntryName);

// Write element data into segment (draw box)

// Log subentry

      sprintf(LogText, "Inserting poly line element data...");

      SUBENTER(GpiElement, LogText);

// Insert the element contents into the segment

      if (GpiElement(hPS, BOX_ELEMENT, ElementDesc, RetLength, pbData)
                                                                  == GPI_ERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiElement");

// Log subexit

      SUBEXIT();

// Close graphics segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Log pyramid draw

      sprintf(LogText, "Drawing original and copied polyline elements...");

      LOGINFO(4, "*", LogText);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

      DosFreeMem(pbData);

      DosFreeMem(ElementDesc);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}

 /***************************************************************************
 *\\ddd
 * Routine Name: GpiGetAndPutDataExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           tests for GpiGetData and GpiPutData to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ENTER_MAIN
 *   SHOWERROR
 *   TEST_SIGNATURE
 *
 *
 *   EXIT_MAIN
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Build the pyramid with half-inch boxes
 * Open graphics segment
 * Fill segment with 1000 labels
 * Close segment
 * Log setup completion
 * Open graphics segment
 * Allocate 64K segment for copying label segment
 * Log subentry
 * Do
 *    Set offset to beginning of a complete graphics order
 *    Get label segment data
 *    If no error occurred and data was fetched, perform put function
 * While (GpiGetData retrieves data)
 * Log subexit
 * Free 64K segment for copying label segment
 * Close graphics segment
 * Open graphics segment
 * Allocate 64K segment for copying pyramid segment
 * Reset parameters for segment copying
 * Log subentry
 * Do
 *    Set offset to beginning of a complete graphics order
 *    Get pyramid segment data
 *    If no error occurred and data was fetched, perform put function
 * While (GpiGetData retrieves data)
 * Log subexit
 * Free 64K segment for copying pyramid segment
 * Close graphics segment
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Log pyramid draw
 * Draw the segment
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiGetAndPutDataExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiGetAndPutDataExh"; // entry point name
   LONG           ErrorCount = 0L,                 // error count
                  index,
                  OffSegment = 0L,
                  Offset = 0L,
                  Length = 0xfc00,
                  RetLength = 0L;
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;
   PVOID     pbData = NULL;


   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Build the pyramid with half-inch boxes

      DrwPyramid(TENTH_INCH);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 2) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Fill segment with labels

      for (index = 0; index < 1000; index ++)
         if (GpiLabel(hPS, index) == FALSE)
            SHOWERROR("GpiLabel", EntryName);

// Close segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Log setup completion

      SETEXIT();

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 3) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Allocate 64K segment for copying label segment

      //32 bit code memory allocation below
        if (DosAllocMem((PVOID)&pbData,65536L,
                        PAG_COMMIT | PAG_WRITE | PAG_READ) != NO_ERROR)
            SHOWERROR("DosAllocMemory", EntryName);


// Log subentry

      sprintf(LogText, "Copying label segment data...");

      SUBENTER(GpiGetData, LogText);

      do
      {

// Set offset to beginning of a complete graphics order

         Offset = (OffSegment += RetLength);

// Get label segment data

         if ((RetLength=GpiGetData(hPS, BOX_SEGMENT + 2, &Offset, DFORM_NOCONV,
                                              Length, pbData)) == GPI_ALTERROR)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiGetData");
         }

// If no error occurred and data was fetched, perform put function

         if(RetLength > 0)
         {

            ERRSTATUS(TRUE, EntryName, "GpiGetData");

            if (GpiPutData(hPS, DFORM_NOCONV, &RetLength, pbData) == GPI_ERROR)
               ErrorCount++;

            ERRSTATUS(TRUE, EntryName, "GpiPutData");

         }
      }
      while (RetLength>0);

// Log subexit

      SUBEXIT();

// Free 64K segment for copying label segment

      DosFreeMem(pbData);

// Close graphics segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Allocate 64K segment for copying pyramid segment


      //32 bit code memory allocation below
        if (DosAllocMem((PVOID)&pbData,65536L,
                        PAG_COMMIT | PAG_WRITE | PAG_READ) != NO_ERROR)
            SHOWERROR("DosAllocMemory", EntryName);

// Reset parameters for segment copying

      Offset = 0L;

      OffSegment = 0L;

      Length = 0xfc00;

      RetLength = 0L;

// Log subentry

      sprintf(LogText, "Copying pyramid segment data...");

      SUBENTER(GpiGetData, LogText);

      do
      {

// Set offset to beginning of a complete graphics order

         Offset = (OffSegment += RetLength);

// Get pyramid segment data

         if ((RetLength=GpiGetData(hPS, BOX_SEGMENT, &Offset, DFORM_NOCONV,
                                              Length, pbData)) == GPI_ALTERROR)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiGetData");
         }

// If no error occurred and data was fetched, perform put function

         if(RetLength > 0)
         {

            ERRSTATUS(TRUE, EntryName, "GpiGetData");

            if (GpiPutData(hPS, DFORM_NOCONV, &RetLength, pbData) == GPI_ERROR)
               ErrorCount++;

            ERRSTATUS(TRUE, EntryName, "GpiPutData");

         }
      }
      while (RetLength>0);

// Log subexit

      SUBEXIT();

// Free 64K segment for copying pyramid segment

      DosFreeMem(pbData);

// Close graphics segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Log pyramid draw

      sprintf(LogText, "Drawing copied pyramid segment...");

      LOGINFO(4, "*", LogText);

// Draw the segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}

 /***************************************************************************
 *\\ddd
 * Routine Name: GpiLblAndSetElmtPtrAtLblExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           tests for GpiLabel and GpiSetElementPointerAtLabel
 *           to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ENTER_MAIN
 *   SHOWERROR
 *   TEST_SIGNATURE
 *
 *
 *   EXIT_MAIN
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Set random number generator seed to 1 for reproducibility
 * Log setup completion
 * Log subentry
 * Log explanatory text concerning execution cycle
 * Insert 1000 elements into the segment
 * Log explanatory text concerning execution cycle
 * Begin loop to go from 0 to 99
 *    Set element pointer to position 0 so label will be found
 *    Set element pointer to label denoted by ElementPtr
 *    Query the position of the element pointer
 *    If pointer position returned is not equal to the pointer position which
 *       was set, log an error message
 * Log subexit
 * Close the segment
 * No device output, so suppress it
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiLabelAndSetElmtPtrAtLabelExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiLblAndSetElmtPtrAtLblExh"; // entry point name
   LONG           ErrorCount = 0,                          // error count
                  Element,
                  ElementPtr,
                  MaxElementPtr = 1000,
                  PostSetPtr;
   SIZEL          sizePS;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Set random number generator seed to 1 for reproducibility

      srand(1);

// Log setup completion

      SETEXIT();

// Log subentry

      sprintf(LogText, "Executing test cycle...");

      SUBENTER(GpiLabel, LogText);

// Log explanatory text concerning execution cycle

      sprintf(LogText, "Inserting 1000 labels into the segment.");

      LOGINFO(4, "*", LogText);

// Insert many elements into the segment

      for (Element = 1; Element <= MaxElementPtr; Element++)
      {

         if (GpiLabel(hPS, Element) == FALSE)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiLabel");
         }
      }

// Log explanatory text concerning execution cycle

      sprintf(LogText, "Randomly setting and querying element pointer 100"
                       " times over a range of 1000 labels.");

      LOGINFO(4, "*", LogText);

// Position element pointer to random element many times

      for (Element = 0; Element < 100; Element++)
      {

         ElementPtr = randL(1L, MaxElementPtr);

// Set element pointer to position 0 so label will be found

         if (GpiSetElementPointer(hPS, 0L) == FALSE)
            SHOWERROR("GpiSetElementPointerAtLabel", EntryName);


// Set element pointer to label denoted by ElementPtr

         if (GpiSetElementPointerAtLabel(hPS, ElementPtr) == FALSE)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiSetElementPointerAtLabel");
         }

// Query the position of the element pointer

         if ((PostSetPtr = GpiQueryElementPointer(hPS)) == GPI_ALTERROR)
            SHOWERROR("GpiQueryElementPointer", EntryName);

// If pointer position returned is not equal to the pointer position which
// was set, log an error message

         if (ElementPtr != PostSetPtr)
         {

            ErrorCount++;

            sprintf(LogText,  "Failure of GpiLabel or GpiSetElementPointerA"
                              "tLabel.  Label inserted was %ld and label re"
                              "presented by current position was %ld.",
                                                      ElementPtr, PostSetPtr);

            LOGINFO((INT)4, "@", LogText);

         }

      }

// Log subexit

      SUBEXIT();

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);

// No device output, so suppress it

      DeletePrinterOutput(EntryName);

   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: GpiOffsetElementPointerExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           test for GpiOffsetElementPointer to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ENTER_MAIN
 *   SHOWERROR
 *   TEST_SIGNATURE
 *
 *
 *   EXIT_MAIN
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Insert 1000 elements into the segment
 * Set random number generator seed to 1 for reproducibility
 * Log setup completion
 * Log subentry
 * Initialize element pointer for negative offset test
 * Log explanatory text concerning execution cycle
 * Test that negative offset sets element pointer to zero
 * If test fails, log failure
 * Reset element pointer to zero for exhaustive positive offset test
 * Log explanatory text concerning execution cycle
 * Begin loop to go from 0 to 999
 *    Determine position of element pointer before the pointer is moved
 *    Calculate some offset between 0 and MaxElementPtr
 *    Move element pointer.  If offset places pointer past last element,
 *       pointer is set to last element.
 *    Determine position of element pointer after the pointer is moved
 *    If position before move + offset is greater than the position of the last
 *       element, reset the element pointer to zero and make sure the position
 *       after move is set to the last element position.
 *    If position after move is not equal to the last element position,
 *       log error
 *    If position before move + offset is not greater than the position of the
 *       last element, verify that the position after move is properly placed
 * Log subexit
 * Close the segment
 * No device output, so suppress it
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiOffsetElementPointerExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiOffsetElementPointerExh"; // entry point name
   LONG           ErrorCount = 0,                         // error count
                  Element,
                  ElementPtr,
                  MaxElementPtr = 1000,
                  PreSetPtr,
                  PostSetPtr;
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Insert many elements into the segment

      for (Element = 1; Element <= MaxElementPtr; Element++)
      {

         if (GpiLabel(hPS, Element) == FALSE)
            SHOWERROR("GpiLabel", EntryName);

      }

// Set random number generator seed to 1 for reproducibility

      srand(1);

// Log setup completion

      SETEXIT();

// Log subentry

      sprintf(LogText, "Executing test cycle...");

      SUBENTER(GpiOffsetElementPointer, LogText);

// Initialize element pointer for negative offset test

      if (GpiSetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiSetElementPointer", EntryName);

// Log explanatory text concerning execution cycle

      sprintf(LogText, "Calling GpiOffsetElementPointer with negative offset.");

      LOGINFO(4, "*", LogText);

// Test that negative offset sets element pointer to zero

      if (GpiOffsetElementPointer(hPS, -1L) == FALSE)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiOffsetElementPointer");

      if ((PostSetPtr = GpiQueryElementPointer(hPS)) == GPI_ALTERROR)
         SHOWERROR("GpiQueryElementPointer", EntryName);

// If test fails, log failure

      if (PostSetPtr != 0L)
      {

         ErrorCount++;

         sprintf(LogText,  "Failure of GpiOffsetElementPointer.  Position"
                           "queried was %ld, not zero.", PostSetPtr);

         LOGINFO((INT)4, "@", LogText);

      }

// Reset element pointer to zero for exhaustive positive offset test

      if (GpiSetElementPointer(hPS, 0L) == FALSE)
         SHOWERROR("GpiSetElementPointer", EntryName);

// Log explanatory text concerning execution cycle

      sprintf(LogText, "Setting element pointer with random offsets %ld times",
                                                                MaxElementPtr);

      LOGINFO(4, "*", LogText);

      for (Element = 0; Element < 1000; Element++)
      {

// Determine position of element pointer before the pointer is moved

         if ((PreSetPtr = GpiQueryElementPointer(hPS)) == GPI_ALTERROR)
            SHOWERROR("GpiQueryElementPointer", EntryName);

// Calculate some offset between 0 and MaxElementPtr

         ElementPtr = randL(0L, MaxElementPtr);

// Move element pointer.  If offset places pointer past last element, pointer
// is set to last element.

         if (GpiOffsetElementPointer(hPS, ElementPtr) == FALSE)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiOffsetElementPointer");
         }

// Determine position of element pointer after the pointer is moved

         if ((PostSetPtr = GpiQueryElementPointer(hPS)) == GPI_ALTERROR)
            SHOWERROR("GpiQueryElementPointer", EntryName);

// If position before move + offset is greater than the position of the last
// element, reset the element pointer to zero and make sure the position after
// move is set to the last element position.

         if ((PreSetPtr + ElementPtr) > MaxElementPtr)
         {

            if (GpiSetElementPointer(hPS, 0L) == FALSE)
               SHOWERROR("GpiSetElementPointer", EntryName);

// If position after move is not equal to the last element position, log error

            if (PostSetPtr != MaxElementPtr)
            {

               ErrorCount++;

               sprintf(LogText,  "Failure of GpiOffsetElementPointer.  Posit"
                                 "ion queried was %ld, not %ld.",
                                                    PostSetPtr, MaxElementPtr);

               LOGINFO((INT)4, "@", LogText);

            }
         }

// If position before move + offset is not greater than the position of the
// last element, verify that the position after move is properly placed

         else
            if ((PreSetPtr + ElementPtr) != PostSetPtr)
            {

               ErrorCount++;

               sprintf(LogText,  "Failure of GpiOffsetElementPointer.  Posit"
                                 "ion queried was %ld, not %ld.",
                                           PostSetPtr, PreSetPtr + ElementPtr);

               LOGINFO((INT)4, "@", LogText);

            }
      }

// Log subexit

      SUBEXIT();

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);

// No device output, so suppress it

      DeletePrinterOutput(EntryName);

   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: GpiQueryElementTypeExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           test for GpiQueryElementType to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ENTER_MAIN
 *   SHOWERROR
 *   TEST_SIGNATURE
 *
 *
 *   EXIT_MAIN
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set initial box position
 * Build the pyramid with half-inch boxes
 * Make sure pyramid chain is not chained
 * Set drawing mode to DRAWANDRETAIN
 * Open graphics segment
 * Insert line bundle
 * Insert character bundle
 * Insert marker bundle
 * Insert area bundle
 * Insert image bundle
 * Close the segment
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Position element pointer to end of segment
 * Insert call segment
 * Insert polyline
 * Insert polyfillet
 * Insert polyfillet sharp
 * Insert polyspline
 * Insert polymarker
 * Insert full arc
 * Insert image
 * Insert character string at current position
 * Insert character string at given position
 * Begin user defined element insert
 *    Allocate space for the element description
 *    Query the element contents with GpiQueryElement for later copies with
 *       GpiElement
 *    Write element data into segment (draw box)
 *    Free space for the element description
 * End user defined element insert
 * Log setup completion
 * For each element type inserted except the last one
 *    Log subentry
 *    Query element type
 *    Set element pointer to next element whose type is to be queried for next
 *       query operation
 *    If type queried not equal to expected type, log error message
 *    Log subexit
 * For the last element type inserted
 *    Log subentry
 *    Query element type
 *    If type queried not equal to expected type, log error message
 *    Log subexit
 * Close the segment
 * No device output, so suppress it
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiQueryElementTypeExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiQueryElementTypeExh"; // entry point name
   LONG           ErrorCount = 0,                     // error count
                  RetLength,
                  Type;
   SIZEL          sizePS;
   FIXED          afxSharpness;
   extern LONG    ElementBufferSize;
   POINTL         ptl,
                  polyptl[5];
   PBUNDLE        Attrs;
   MATRIXLF       MatStruct;
   PSZ            ElementDesc;
   extern double  TopX,
                  TopY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Build the pyramid with half-inch boxes

      DrwPyramid(HALF_INCH);

// Make sure pyramid chain is not chained

      if (GpiSetSegmentAttrs(hPS, BOX_SEGMENT, ATTR_CHAINED, ATTR_OFF)
                                                                      == FALSE)
         SHOWERROR("GpiSetSegmentAttrs", EntryName);

// Set drawing mode to DRAWANDRETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

      ptl.x = ptl.y = 0;

      if (GpiMove(hPS, &ptl) == FALSE)
         SHOWERROR("GpiMove", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Begin line bundle insert

      if ((Attrs = (PBUNDLE)malloc(sizeof(LINEBUNDLE))) == NULL)
         SHOWERROR("malloc", EntryName);

      if (GpiQueryDefAttrs(hPS, PRIM_LINE, LnAttrMask, Attrs) == FALSE)
         SHOWERROR("GpiQueryDefAttrs", EntryName);

      if (GpiSetAttrs(hPS, PRIM_LINE, LnAttrMask, 0L, Attrs) == FALSE)
         SHOWERROR("GpiSetAttrs", EntryName);

      free (Attrs);

// End line bundle insert

// Begin character bundle insert

      if ((Attrs = (PBUNDLE)malloc(sizeof(CHARBUNDLE))) == NULL)
         SHOWERROR("malloc", EntryName);

      if (GpiQueryDefAttrs(hPS, PRIM_CHAR, ChAttrMask, Attrs) == FALSE)
         SHOWERROR("GpiQueryDefAttrs", EntryName);

      if (GpiSetAttrs(hPS, PRIM_CHAR, ChAttrMask, 0L, Attrs) == FALSE)
         SHOWERROR("GpiSetAttrs", EntryName);

      free (Attrs);

// End character bundle insert

// Begin marker bundle insert

      if ((Attrs = (PBUNDLE)malloc(sizeof(MARKERBUNDLE))) == NULL)
         SHOWERROR("malloc", EntryName);

      if (GpiQueryDefAttrs(hPS, PRIM_MARKER, MkAttrMask, Attrs) == FALSE)
         SHOWERROR("GpiQueryDefAttrs", EntryName);

      if (GpiSetAttrs(hPS, PRIM_MARKER, MkAttrMask, 0L, Attrs) == FALSE)
         SHOWERROR("GpiSetAttrs", EntryName);

      free (Attrs);

// End marker bundle insert

// Begin area bundle insert

      if ((Attrs = (PBUNDLE)malloc(sizeof(AREABUNDLE))) == NULL)
         SHOWERROR("malloc", EntryName);

      if (GpiQueryDefAttrs(hPS, PRIM_AREA, ArAttrMask, Attrs) == FALSE)
         SHOWERROR("GpiQueryDefAttrs", EntryName);

      if (GpiSetAttrs(hPS, PRIM_AREA, ArAttrMask, 0L, Attrs) == FALSE)
         SHOWERROR("GpiSetAttrs", EntryName);

      free (Attrs);

// End area bundle insert

// Begin image bundle insert

      if ((Attrs = (PBUNDLE)malloc(sizeof(IMAGEBUNDLE))) == NULL)
         SHOWERROR("malloc", EntryName);

      if (GpiQueryDefAttrs(hPS, PRIM_IMAGE, ImAttrMask, Attrs) == FALSE)
         SHOWERROR("GpiQueryDefAttrs", EntryName);

      if (GpiSetAttrs(hPS, PRIM_IMAGE, ImAttrMask, 0L, Attrs) == FALSE)
         SHOWERROR("GpiSetAttrs", EntryName);

      free (Attrs);

// End image bundle insert

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Position element pointer to end of segment

      if (GpiSetElementPointer(hPS, LONG_MAX) == FALSE)
         SHOWERROR("GpiSetElementPointer", EntryName);

// Begin call segment insert

// Call pyramid chain

      if (GpiCallSegmentMatrix(hPS, BOX_SEGMENT, 0L, &MatStruct,
                                               TRANSFORM_REPLACE) == GPI_ERROR)
         SHOWERROR("GpiCallSegmentMatrix", EntryName);

// End call segment insert

// Begin polyline insert

      ptl.y += 1;

      if (GpiLine(hPS, &ptl) == GPI_ERROR)
         SHOWERROR("GpiLine", EntryName);

// End polyline insert

// Begin polyfillet insert

      polyptl[0].x = polyptl[2].x = ptl.x;

      polyptl[0].y = polyptl[1].y = ptl.y;

      polyptl[1].x = polyptl[1].y + 1;

      polyptl[2].y = polyptl[2].x + 1;

      if (GpiMove(hPS, &polyptl[0]) == FALSE)
         SHOWERROR("GpiMove", EntryName);

      if (GpiPolyFillet(hPS, 2L, &polyptl[1]) == GPI_ERROR)
         SHOWERROR("GpiPolyFillet", EntryName);

// End polyfillet insert

// Begin polyfillet sharp insert

      polyptl[0].x = polyptl[2].x = ptl.x;

      polyptl[0].y = polyptl[1].y = ptl.y;

      polyptl[1].x = polyptl[1].y + 1;

      polyptl[2].y = polyptl[2].x + 1;

      if (GpiMove(hPS, &polyptl[0]) == FALSE)
         SHOWERROR("GpiMove", EntryName);

      afxSharpness = MAKEFIXED(1, 0);

      if (GpiPolyFilletSharp(hPS, 2L, &polyptl[1], &afxSharpness) == GPI_ERROR)
         SHOWERROR("GpiPolyFilletSharp", EntryName);

// End polyfillet sharp insert

// Begin polyspline insert

      polyptl[0].x = polyptl[2].x = ptl.x + 1;

      polyptl[0].y = polyptl[1].y = ptl.y + 1;

      polyptl[1].x = polyptl[1].y + 1;

      polyptl[2].y = polyptl[2].x + 1;

      if (GpiMove(hPS, &ptl) == FALSE)
         SHOWERROR("GpiMove", EntryName);

      if (GpiPolySpline(hPS, 3L, polyptl) == GPI_ERROR)
         SHOWERROR("GpiPolySpline", EntryName);

// End polyspline insert

// Begin polymarker insert

      if (GpiPolyMarker(hPS, 2L, polyptl) == GPI_ERROR)
         SHOWERROR("GpiPolyMarker", EntryName);

// End polymarker insert

// Begin full arc insert

      if (GpiFullArc(hPS, DRO_OUTLINE, MAKEFIXED(1, 0)) == GPI_ERROR)
         SHOWERROR("GpiFullArc", EntryName);

// End full arc insert

// Begin image insert

      ptl.x = 8;

      ptl.y = 8;

      if (GpiImage(hPS, 0L, (PSIZEL)&ptl, 8L, pbData) == GPI_ERROR)
         SHOWERROR("GpiImage", EntryName);

// End image insert

// Begin character string at current position insert

      if (GpiCharString(hPS, 5L, "Hello") == GPI_ERROR)
         SHOWERROR("GpiCharString", EntryName);

// End character string at current position insert

// Begin character string at given position insert

      if (GpiCharStringAt(hPS, &ptl, 5L, "Hello") == GPI_ERROR)
         SHOWERROR("GpiCharStringAt", EntryName);

// End character string at given position insert

// Begin user defined element insert

// Allocate space for the element description

      if ((ElementDesc=(PSZ)malloc(50))==NULL)
         SHOWERROR("malloc", EntryName);

// Query the element contents with GpiQueryElement for later copies with
// GpiElement

      if ((RetLength = GpiQueryElement(hPS, 0L, ElementBufferSize, pbData)) ==
                                                                  GPI_ALTERROR)
         SHOWERROR("GpiQueryElement", EntryName);

// Write element data into segment (draw box)

      if (GpiElement(hPS, BOX_ELEMENT, ElementDesc, (LONG)RetLength, pbData)
                                                                  == GPI_ERROR)
         SHOWERROR("GpiElement", EntryName);

// Free space for the element description

      free(ElementDesc);

// End user defined element insert

// Log setup completion

      SETEXIT();

// Begin test routine

      if (GpiSetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiSetElementPointer", EntryName);

// Log subentry

      sprintf(LogText, "Testing for line bundle...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_LINEBUNDLE)
      {
         sprintf(LogText, "Error reading line bundle element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                       ETYPE_LINEBUNDLE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for character bundle...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_CHARBUNDLE)
      {
         sprintf(LogText, "Error reading character bundle element type.  Exp"
                          "ected %lX, but value queried was %lX.",
                                                       ETYPE_CHARBUNDLE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for marker bundle...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_MARKERBUNDLE)
      {
         sprintf(LogText, "Error reading marker bundle element type.  Expec"
                          "ted %lX, but value queried was %lX.",
                                                     ETYPE_MARKERBUNDLE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for area bundle...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_AREABUNDLE)
      {
         sprintf(LogText, "Error reading area bundle element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                       ETYPE_AREABUNDLE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for image bundle...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_IMAGEBUNDLE)
      {
         sprintf(LogText, "Error reading image bundle element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                      ETYPE_IMAGEBUNDLE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for call segment...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_CALLSEGMENT)
      {
         sprintf(LogText, "Error reading call segment element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                      ETYPE_CALLSEGMENT, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for polyline...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 2L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_POLYLINE)
      {
         sprintf(LogText, "Error reading polyline element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                         ETYPE_POLYLINE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for polyfillet...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 2L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_POLYFILLET)
      {
         sprintf(LogText, "Error reading polyfillet element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                       ETYPE_POLYFILLET, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for polyfillet sharp...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 2L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_POLYFILLETSHARP)
      {
         sprintf(LogText, "Error reading polyfillet sharp element type.  Exp"
                          "ected %lX, but value queried was %lX.",
                                                  ETYPE_POLYFILLETSHARP, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for polyspline...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_POLYSPLINE)
      {
         sprintf(LogText, "Error reading polyspline element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                       ETYPE_POLYSPLINE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for polymarker...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_POLYMARKER)
      {
         sprintf(LogText, "Error reading polymarker element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                       ETYPE_POLYMARKER, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for full arc...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_FULLARC)
      {
         sprintf(LogText, "Error reading full arc element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                          ETYPE_FULLARC, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for image...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR(EntryName, "GpiOffsetElementPointer");

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_IMAGE)
      {
         sprintf(LogText, "Error reading image element type.  Expected"
                          " %lX, but value queried was %lX.",
                                                            ETYPE_IMAGE, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for string at current position...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_CURRCHARSTR)
      {
         sprintf(LogText, "Error reading current position character string "
                          "element type.  Expected %lX, but value queried wa"
                                            "s %lX.", ETYPE_CURRCHARSTR, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for string at given position...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// Set element pointer to next element whose type is to be queried for next
// query operation

      if (GpiOffsetElementPointer(hPS, 1L) == FALSE)
         SHOWERROR("GpiOffsetElementPointer", EntryName);

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_GIVNCHARSTR)
      {
         sprintf(LogText, "Error reading given position character string el"
                          "ement type.  Expected %lX, but value queried was "
                                              "%lX.", ETYPE_GIVNCHARSTR, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Testing for user defined element...");

      SUBENTER(GpiQueryElementType, LogText);

// Query element type

      if (GpiQueryElementType(hPS, &Type, 0L, ElementDesc) == GPI_ALTERROR)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryElementType");

// If type queried not equal to expected type, log error message

      if (Type != ETYPE_USERELEMENT)
      {
         sprintf(LogText, "Error reading user defined element type.  Expect"
                          "ed %lX, but value queried was %lX.",
                                                      ETYPE_USERELEMENT, Type);
         LOGINFO((INT)4, "@", LogText);
         ErrorCount++;
      }

// Log subexit

      SUBEXIT();

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);

// No device output, so suppress it

      DeletePrinterOutput(EntryName);

   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: GpiSetAndQryEditModeExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           tests for GpiSetEditMode and GpiQueryEditMode to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ENTER_MAIN
 *   SHOWERROR
 *   TEST_SIGNATURE
 *
 *
 *   EXIT_MAIN
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Log coordinates of top box of top pyramid
 * Set level number to allow maximum on visible output medium
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Draw all levels of pyramid, incrementing level with each iteration.
 *    The first level of the pyramid will consist of a box drawn with several
 *    polylines, while each subsequent level will consist of translated copies
 *    of this box.
 * Close the segment
 * Log coordinates of top box of bottom pyramid
 * Open graphics segment
 * Draw all levels of pyramid, incrementing level with each iteration.
 *    The first level of the pyramid will consist of a box drawn with several
 *    polylines, while each subsequent level will consist of translated copies
 *    of this box.
 * Close the segment
 * Set scale factors to be used for box placement
 * Log setup completion
 * Log subentry
 * Set edit mode to SEGEM_REPLACE
 * Log subexit
 * Log subentry
 * Query SEGEM_INSERT edit mode
 * Log subexit
 * Set initial box position for top pyramid
 * Open pyramid segment
 * Start loop to step through the levels (first level= 2), flipping the
 *    desired boxes
 *    Start loop to step through the boxes in current level
 *       If (level = 1 + (x * 4)), skip two center boxes
 *       If (level = 3 + (x * 4)), delete two center boxes
 *       Build box label
 *       Set element pointer at label
 *       Increment element pointer by 1 to skip Label graphics order
 *       Set Model Transform Matrix for this particular iteration's box
 *       Set angle of rotation
 *       Set point about which to rotate
 *       Use helper function to modify transform matrix
 *       Set new model transform for this element
 * Close the segment
 * Log subentry
 * Set edit mode to SEGEM_INSERT
 * Log subexit
 * Log subentry
 * Query SEGEM_INSERT edit mode
 * Log subexit
 * Set initial box position for bottom pyramid
 * Open pyramid segment
 * Start loop to step through the levels (first level= 2), flipping the
 *    desired boxes
 *    Start loop to step through the boxes in current level
 *       If (level = 1 + (x * 4)), skip two center boxes
 *       If (level = 3 + (x * 4)), delete two center boxes
 *       Build box label
 *       Set element pointer at label
 *       Increment element pointer by 1 to skip the Label graphics order
 *       Set Model Transform Matrix for this particular iteration's box
 *       Set angle of rotation
 *       Set point about which to rotate
 *       Use helper function to modify transform matrix
 *       Set new model transform for this element
 * Close the segment
 * Enable output to the output medium
 * Set drawing mode to DRAW-AND-RETAIN
 * Draw the top segment
 * Draw the bottom segment
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiSetAndQryEditModeExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiSetAndQryEditModeExh"; // entry point name
   LONG           ErrorCount = 0,                      // error count
                  flip_index = 0,
                  level;
   SIZEL          sizePS;
   extern double  TopX,
                  TopY;
   extern USHORT  Max_Box_Level;
   MATRIXLF       MatStruct;
   POINTL         ptl;
   FIXED          label,
                  Angle;
   double         ScaleX,
                  ScaleY;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

      TopX = pg_size_x / 2.0;
      TopY = pg_size_y - 0.75;

// Logging info

      sprintf(LogText, "Coordinates of top pyramid's top box: (%ld, %ld)",
                                                       (LONG)TopX, (LONG)TopY);
      LOGINFO((SHORT)4, "*", LogText);

// Set level number to allow maximum on visible output medium

      Max_Box_Level = (USHORT)((min(pg_size_x, pg_size_y - 1) - 2) /
                                                              (HALF_INCH * 2));

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

      level = 0;

// Draw all levels of pyramid, incrementing level with each iteration.
// The first level of the pyramid will consist of a box drawn with several
// polylines, while each subsequent level will consist of translated copies
// of this box.

      do
      {
         if (GpiLabel(hPS, level)==FALSE)
            SHOWERROR("GpiLabel", EntryName);

         DrawLevel(hPS, (double)level++, HALF_INCH, hWndClient, &ErrorCount,
                                                                    EntryName);
      }
      while (level < (LONG)Max_Box_Level);

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

      TopY = (pg_size_y / 2);

// Logging info

      sprintf(LogText, "Coordinates of bottom pyramid's top box: (%ld, %ld)",
                                                       (LONG)TopX, (LONG)TopY);
      LOGINFO((SHORT)4, "*", LogText);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

      level = 0;

// Draw all levels of pyramid, incrementing level with each iteration.
// The first level of the pyramid will consist of a box drawn with several
// polylines, while each subsequent level will consist of translated copies
// of this box.

      do
      {
         if (GpiLabel(hPS, level)==FALSE)
            SHOWERROR("GpiLabel", EntryName);

         DrawLevel(hPS, (double)level++, HALF_INCH, hWndClient, &ErrorCount,
                                                                    EntryName);
      }
      while (level < (LONG)Max_Box_Level);

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Set scale factors to be used for box placement

      ScaleX = HALF_INCH * delta_x;

      ScaleY = HALF_INCH * delta_y;

// Log setup completion

      SETEXIT();

// Log subentry

      sprintf(LogText, "Setting edit mode to SEGEM_REPLACE...");

      SUBENTER(GpiSetEditMode, LogText);

// Set edit mode to SEGEM_REPLACE

      if (GpiSetEditMode(hPS, SEGEM_REPLACE) == FALSE)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiSetEditMode");

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Querying edit mode...");

      SUBENTER(GpiQueryEditMode, LogText);

// Query SEGEM_INSERT edit mode

      if (GpiQueryEditMode(hPS) != SEGEM_REPLACE)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryEditMode");

// Log subexit

      SUBEXIT();

// Set initial box position

      TopX = pg_size_x / 2.0;
      TopY = (pg_size_y / 2);

// Open pyramid segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Start loop to step through the levels (first level= 2), flipping the
// desired boxes

      for (level = 2; level < (LONG)Max_Box_Level; level++)
      {

// Start loop to step through the boxes in current level

         for (flip_index = 1; flip_index <= level; flip_index += 2)
         {

// If (level = 1 + (x * 4)), skip two center boxes

            if (((level % 4) == 1) && (flip_index == ((level / 2) + 1)))
               flip_index += 1;

// If (level = 3 + (x * 4)), delete two center boxes

            if (((level % 4) == 3) && (flip_index == ((level / 2) + 2)))
               flip_index -= 1;

// Build box label

            label = MAKEFIXED(level, flip_index);

// Set element pointer at label

            if (GpiSetElementPointerAtLabel(hPS, (LONG)label) == FALSE)
               SHOWERROR("GpiSetElementPointerAtLabel", EntryName);

// Increment element pointer by 1 to skip Label graphics order

            if (GpiOffsetElementPointer(hPS, 1L)==FALSE)
               SHOWERROR("GpiOffsetElementPointer", EntryName);

// Set Model Transform Matrix for this particular iteration's box

            MatStruct.fxM11 = MAKEFIXED(1, 0);
            MatStruct.fxM12 = MAKEFIXED(0, 0);
            MatStruct.lM13  = 0L;
            MatStruct.fxM21 = MAKEFIXED(0, 0);
            MatStruct.fxM22 = MAKEFIXED(1, 0);
            MatStruct.lM23  = 0L;
            MatStruct.lM31  = MapX((level * -(ScaleX / 2.0)) +
                                                        (ScaleX * flip_index));
            MatStruct.lM32  = MapY(level * -(ScaleY));
            MatStruct.lM33  = 1L;

// Set angle of rotation

            Angle = MAKEFIXED(180, 0);

// Set point about which to rotate

            ptl.x = MapX(100 * TopX) + MatStruct.lM31;

            ptl.y = MapY(100 * TopY) + MatStruct.lM32 - MapY(0.75 * ScaleY);

// Use helper function to modify transform matrix

            if (GpiRotate(hPS, &MatStruct, TRANSFORM_ADD, Angle, &ptl)
                                                                      == FALSE)
               SHOWERROR("GpiRotate", EntryName);

// Set new model transform for this element

            if (GpiSetModelTransformMatrix(hPS, 9L, &MatStruct,
                                                   TRANSFORM_REPLACE) == FALSE)
               SHOWERROR("GpiSetModelTransformMatrix", EntryName);

         }

      }

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Log subentry

      sprintf(LogText, "Setting edit mode to SEGEM_INSERT...");

      SUBENTER(GpiSetEditMode, LogText);

// Set edit mode to SEGEM_INSERT

      if (GpiSetEditMode(hPS, SEGEM_INSERT) == FALSE)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiSetEditMode");

// Log subexit

      SUBEXIT();

// Log subentry

      sprintf(LogText, "Querying edit mode...");

      SUBENTER(GpiQueryEditMode, LogText);

// Query SEGEM_INSERT edit mode

      if (GpiQueryEditMode(hPS) != SEGEM_INSERT)
         ErrorCount++;

      ERRSTATUS(TRUE, EntryName, "GpiQueryEditMode");

// Log subexit

      SUBEXIT();

// Set initial box position

      TopY = pg_size_y - 0.75;

// Open pyramid segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Start loop to step through the levels (first level= 2), flipping the
// desired boxes

      for (level = 2; level < (LONG)Max_Box_Level; level++)
      {

// Start loop to step through the boxes in current level

         for (flip_index = 1; flip_index <= level; flip_index += 2)
         {


            if (((level % 4) == 1) && (flip_index == ((level / 2) + 1)))
               flip_index += 1;


            if (((level % 4) == 3) && (flip_index == ((level / 2) + 2)))
               flip_index -= 1;

// Build box label

            label = MAKEFIXED(level, flip_index);

// Set element pointer at label

            if (GpiSetElementPointerAtLabel(hPS, (LONG)label) == FALSE)
               SHOWERROR("GpiSetElementPointerAtLabel", EntryName);

// Increment element pointer by 1 to skip the Label graphics order

            if (GpiOffsetElementPointer(hPS, 1L)==FALSE)
               SHOWERROR("GpiOffsetElementPointer", EntryName);

// Set Model Transform Matrix for this particular iteration's box

            MatStruct.fxM11 = MAKEFIXED(1, 0);
            MatStruct.fxM12 = MAKEFIXED(0, 0);
            MatStruct.lM13  = 0L;
            MatStruct.fxM21 = MAKEFIXED(0, 0);
            MatStruct.fxM22 = MAKEFIXED(1, 0);
            MatStruct.lM23  = 0L;
            MatStruct.lM31  = MapX((level * -(ScaleX / 2.0)) +
                                                        (ScaleX * flip_index));
            MatStruct.lM32  = MapY(level * -(ScaleY));
            MatStruct.lM33  = 1L;

// Set angle of rotation

            Angle = MAKEFIXED(180, 0);

// Set point about which to rotate

            ptl.x = MapX(100 * TopX) + MatStruct.lM31;

            ptl.y = MapY(100 * TopY) + MatStruct.lM32 - MapY(0.75 * ScaleY);

// Use helper function to modify transform matrix

            if (GpiRotate(hPS, &MatStruct, TRANSFORM_ADD, Angle, &ptl)
                                                                      == FALSE)
               SHOWERROR("GpiRotate", EntryName);

// Set new model transform for this element

            if (GpiSetModelTransformMatrix(hPS, 9L, &MatStruct,
                                                   TRANSFORM_REPLACE) == FALSE)
               SHOWERROR("GpiSetModelTransformMatrix", EntryName);

         }

      }

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Enable output to the output medium

      if (GpiSetDrawControl(hPS, DCTL_ERASE, DCTL_OFF) == FALSE)
         SHOWERROR("GpiSetDrawControl", EntryName);

// Set drawing mode to DRAW-AND-RETAIN

      if (GpiSetDrawingMode(hPS, DM_DRAWANDRETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Draw the top segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Draw the bottom segment

      if (GpiDrawSegment(hPS, BOX_SEGMENT + 1) == FALSE)
         SHOWERROR("GpiDrawSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);
   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: GpiSetAndQryElementPointerExh
 *
 * Purpose:  This is an entry point for PTT that causes the exhaustive
 *           tests for GpiSetElementPointer and GpiQueryElementPointer
 *           to be run.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   ENTER_MAIN
 *   SHOWERROR
 *   TEST_SIGNATURE
 *
 *
 *   EXIT_MAIN
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log setup initialization
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Insert 1000 elements into the segment
 * Set random number generator seed to 1 for reproducibility
 * Log setup completion
 * Log subentry
 * Log explanatory text concerning execution cycle
 * Begin loop from 0 to 99
 *    Set element pointer to position denoted by ElementPtr
 *    Query element pointer position
 *    If position queried not equal to position which was set, log error
 *       message
 * Log subexit
 * Close the segment
 * No device output, so suppress it
 *
 *\\end
 ***************************************************************************/

VOID  APIENTRY GpiSetAndQryElementPointerExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   static PCHAR   EntryName="GpiSetAndQryElementPointerExh";
                                                           // entry point name
   LONG           ErrorCount = 0,                          // error count
                  Element,
                  ElementPtr,
                  MaxElementPtr = 1000,
                  PostSetPtr;
   SIZEL          sizePS;

   if (SelectionCall)
      WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
   else
   {

      if ((GpiQueryPS(hPS, &sizePS) & PS_TYPE) != GPIT_NORMAL)
      {
         ABORTENTRY(EntryName, "GPIT_MICRO");
         return;
      }

      ENTER_MAIN(EntryName);

// Log setup initialization

      SETENTER(EntryName, "SETUP");

// Set drawing mode to RETAIN

      if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
         SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

      if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
         SHOWERROR("GpiOpenSegment", EntryName);

// Insert many elements into the segment

      for (Element = 0; Element < MaxElementPtr; Element++)
      {

         if (GpiLabel(hPS, Element) == FALSE)
            SHOWERROR("GpiLabel", EntryName);

      }

// Set random number generator seed to 1 for reproducibility

      srand(1);

// Log setup completion

      SETEXIT();

// Log subentry

      sprintf(LogText, "Executing test cycle...");

      SUBENTER(GpiSetElementPointer, LogText);

// Log explanatory text concerning execution cycle

      sprintf(LogText, "Randomly setting and querying element pointer 100"
                       " times over a range of 1000 elements.");

      LOGINFO(4, "*", LogText);

// Position element pointer to random element many times

      for (Element = 0; Element < 100; Element++)
      {

// Generate a random pointer position

         ElementPtr = randL(0L, MaxElementPtr);

// Set element pointer to position denoted by ElementPtr

         if (GpiSetElementPointer(hPS, ElementPtr) == FALSE)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiSetElementPointer");
         }

// Query element pointer position

         if ((PostSetPtr = GpiQueryElementPointer(hPS)) == GPI_ALTERROR)
         {
            ErrorCount++;
            ERRSTATUS(TRUE, EntryName, "GpiQueryElementPointer");
         }

// If position queried not equal to position which was set, log error message

         if (ElementPtr != PostSetPtr)
         {

            ErrorCount++;

            sprintf(LogText,  "Failure of GpiSetElementPointer or GpiQuery"
                              "ElementPointer.  Pointer position set to %ld"
                              " and pointer position queried was %ld.",
                                                      ElementPtr, PostSetPtr);

            LOGINFO((INT)4, "@", LogText);

         }

      }

// Log subexit

      SUBEXIT();

// Close the segment

      if (GpiCloseSegment(hPS) == FALSE)
         SHOWERROR("GpiCloseSegment", EntryName);

// Exit entry point

      EXIT_MAIN(EntryName);

// No device output, so suppress it

      DeletePrinterOutput(EntryName);

   }
}

/***************************************************************************
 *\\ddd
 * Routine Name: DrawPyramid
 *
 * Purpose:  This is a function used to build a segment containing the
 *           necessary graphics orders to represent a pyramid of boxes
 *           of a specified scale.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   LOGINFO
 *   SHOWERROR
 *   DrawLevel
 *   min
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Log coordinates of top box of pyramid
 * Set level number to allow maximum on visible output medium
 * Set drawing mode to RETAIN
 * Open graphics segment
 * Do
 *    Draw all levels of pyramid, incrementing level with each iteration.
 *       The first level of the pyramid will consist of a box drawn with
 *       several polylines, while each subsequent level will consist of
 *       translated copies of this box.
 * While (Level < Max_Box_Level)
 * Close the segment
 *
 *\\end
 ***************************************************************************/

VOID DrawPyramid(HPS hPS, double Scale, HWND hWndClient, LONG *ErrorTotal,
                                                              PCHAR EntryName)
{
   extern USHORT   Max_Box_Level;
   USHORT          level = 0;          // box level
   LONG            ErrorCount = 0;

   ErrorCount = *ErrorTotal;

// Logging info

   sprintf(LogText, "Coordinates of top box: (%ld, %ld)", (LONG)TopX,
                                                                   (LONG)TopY);
   LOGINFO((SHORT)4, "*", LogText);

// Set level number to allow maximum on visible output medium

   Max_Box_Level = (USHORT)((min(pg_size_x, pg_size_y - 1) - 1) / Scale);

// Set drawing mode to RETAIN

   if (GpiSetDrawingMode(hPS, DM_RETAIN) == FALSE)
      SHOWERROR("GpiSetDrawingMode", EntryName);

// Open graphics segment

   if (GpiOpenSegment(hPS, BOX_SEGMENT) == FALSE)
      SHOWERROR("GpiOpenSegment", EntryName);

// Draw all levels of pyramid, incrementing level with each iteration.
// The first level of the pyramid will consist of a box drawn with several
// polylines, while each subsequent level will consist of translated copies
// of this box.

   do
   {
      if (GpiLabel(hPS, (LONG)level)==FALSE)
         SHOWERROR("GpiLabel", EntryName);

      DrwLevel((double)(level++));
   }
   while (level < Max_Box_Level);

// Close the segment

   if (GpiCloseSegment(hPS) == FALSE)
      SHOWERROR("GpiCloseSegment", EntryName);

   free(pbData);

   *ErrorTotal = ErrorCount;

   return;

}

/***************************************************************************
 *\\ddd
 * Routine Name: DrawLevel
 *
 * Purpose:  This is a function used to insert elements representing
 *           one level of a pyramid of boxes into a segment.
 *
 * System Requirements: Section ???
 *
 * Revision Log: 09/19/91, GreAT, Stanley Logan
 *               - created initial template
 *               - TestCase Design.
 *
 * Inputs:
 *    hPS            // presentation space
 *    hWndClient     // client window
 *    SelectionCall  // selection parameter
 *
 * Outputs:
 *   NONE
 *
 * Subroutines Required:
 *   MAKEFIXED
 *   SHOWERROR
 *   MapX
 *   MapY
 *
 * Limitations:
 *   NONE
 *
 *\\end
 ***************************************************************************/

/***************************************************************************
 *\\algorithm
 *
 * Allocate space for the element description
 * Log pyramid level information
 * If drawing the top level
 *    Set up coordinates for all corners of box
 *    Build element description string
 *    Build box label
 *    Insert label before element group
 *    Set Model Transform Matrix for the first box
 *    Start an element bracket
 *    Move the current position to the first point on the top face
 *    Set color for top face of box
 *    Set pattern for top face of box
 *    Begin area for filling
 *    Draw top face of box
 *    Close area for filling
 *    Move the current position to the first point on the left face
 *    Set color for top face of box
 *       Set pattern for top face of box
 *    Begin area for filling
 *    Draw left face of box
 *    Close area for filling
 *    Move the current position to the first point on the right face
 *    Set color for top face of box
 *    Set pattern for top face of box
 *    Begin area for filling
 *    Draw right face of box
 *    Close area for filling
 *    Close element bracket
 *    Set the element pointer to the box element
 *    Query the element type to determine space needed for element data
 *    Allocate space for the element orders
 *    Query the element contents with GpiQueryElement for later copies with
 *       GpiElement
 * Else
 *    Set scale factors for box positioning
 *    Set Model Transform Matrix for this particular iteration's box
 *    Build element description string
 *    Build box label
 *    Insert label before element group
 *    Set model transform for this element
 *    Write element data into segment with GpiElement
 * Free up allocated memory
 *
 *\\end
 ***************************************************************************/

VOID DrawLevel(HPS hPS, double pyramid_level, double Scale, HWND hWndClient,
                                            LONG *ErrorTotal, PCHAR EntryName)
{
   POINTL         PLine1[5],
                  PLine2[5],
                  PLine3[5];
   double         box_across = 0.0,
                  ScaleX,
                  ScaleY;
   extern double  TopX,
                  TopY;
   MATRIXLF       MatStruct;
   PSZ            ElementDesc;
   LONG           Type,
                  ErrorCount = 0;
   FIXED          label;
   static LONG    RetLength;            // number of bytes queried
   extern LONG    ElementBufferSize;

   ErrorCount = *ErrorTotal;

// Allocate space for the element description

   if ((ElementDesc=(PSZ)malloc(50))==NULL)
      SHOWERROR("malloc", EntryName);

// Log pyramid level information

   sprintf(ElementDesc, "Building level %ld", (LONG)pyramid_level);

   ERRSTATUS(FALSE, EntryName, ElementDesc);

   if (pyramid_level == 0.0)
   {

// Set up coordinates for all corners of box

      PLine1[0].x = PLine1[2].x = PLine1[4].x = PLine2[0].x = PLine2[3].x =
         PLine2[4].x = PLine3[0].x = PLine3[1].x =   PLine3[4].x =
                   MapX(100 * (TopX - ((Scale * pyramid_level) / 2.0)));
      PLine1[1].x = PLine2[1].x = PLine2[2].x =
                   MapX(100 * (TopX - ((Scale * (pyramid_level + 1)) / 2.0)));
      PLine1[3].x = PLine3[2].x = PLine3[3].x =
                   MapX(100 * (TopX + ((Scale * (pyramid_level + 1)) / 2.0)));
      PLine1[0].y = PLine1[4].y =
                   MapY(100 * (TopY - ((Scale * pyramid_level) / 1.0)));
      PLine1[1].y = PLine1[3].y = PLine2[1].y = PLine3[3].y =
                   MapY(100 * (TopY - ((Scale * (pyramid_level + 1)) / 2.0)));
      PLine1[2].y = PLine2[0].y = PLine2[2].y = PLine2[4].y = PLine3[0].y =
         PLine3[2].y = PLine3[4].y =
                   MapY(100 * (TopY - ((Scale * (pyramid_level + 1)) / 1.0)));
      PLine2[3].y = PLine3[1].y =
                   MapY(100 * (TopY - ((Scale * (pyramid_level + 1.5)) / 1.0)));

// Build element description string

      sprintf(ElementDesc, "Level: %f Box: %f", pyramid_level, box_across);

// Build box label

      label = MAKEFIXED((LONG)pyramid_level, (LONG)box_across);

// Insert label before element group

      if (GpiLabel(hPS, (LONG)label)==FALSE)
         SHOWERROR("GpiLabel", EntryName);

// Set Model Transform Matrix for the first box

      if (GpiSetModelTransformMatrix(hPS, 0L, &MatStruct, TRANSFORM_REPLACE)
                                                                     == FALSE)
         SHOWERROR("GpiSetModelTransformMatrix", EntryName);

// Start an element bracket

      if (GpiBeginElement(hPS, BOX_ELEMENT, ElementDesc) == FALSE)
         SHOWERROR("GpiBeginElement", EntryName);

// Move the current position to the first point on the top face

      if (GpiMove(hPS, PLine1) == FALSE)
         SHOWERROR("GpiMove", EntryName);

// Set color for top face of box

      if (GpiSetColor(hPS, CLR_BLUE) == FALSE)
         SHOWERROR("GpiSetColor", EntryName);

// Set pattern for top face of box

      if (GpiSetPattern(hPS, PATSYM_DENSE6) == FALSE)
         SHOWERROR("GpiSetPattern", EntryName);

// Begin area for filling

      if (GpiBeginArea(hPS, BA_BOUNDARY | BA_ALTERNATE) == FALSE)
         SHOWERROR("GpiBeginArea", EntryName);

// Draw top face of box

      if (GpiPolyLine(hPS, 5L, PLine1) == GPI_ERROR)
         SHOWERROR("GpiPolyLine", EntryName);

// Close area for filling

      if (GpiEndArea(hPS) == GPI_ERROR)
         SHOWERROR("GpiEndArea", EntryName);

// Move the current position to the first point on the left face

      if (GpiMove(hPS, PLine2) == FALSE)
         SHOWERROR("GpiMove", EntryName);

// Set color for top face of box

      if (GpiSetColor(hPS, CLR_RED) == FALSE)
         SHOWERROR("GpiSetColor", EntryName);

// Set pattern for top face of box

      if (GpiSetPattern(hPS, PATSYM_HALFTONE) == FALSE)
         SHOWERROR("GpiSetPattern", EntryName);

// Begin area for filling

      if (GpiBeginArea(hPS, BA_BOUNDARY | BA_ALTERNATE) == FALSE)
         SHOWERROR("GpiBeginArea", EntryName);

// Draw left face of box

      if (GpiPolyLine(hPS, 5L, PLine2) == GPI_ERROR)
         SHOWERROR("GpiPolyLine", EntryName);

// Close area for filling

      if (GpiEndArea(hPS) == GPI_ERROR)
         SHOWERROR("GpiEndArea", EntryName);

// Move the current position to the first point on the right face

      if (GpiMove(hPS, PLine3) == FALSE)
         SHOWERROR("GpiMove", EntryName);

// Set color for top face of box

      if (GpiSetColor(hPS, CLR_GREEN) == FALSE)
         SHOWERROR("GpiSetColor", EntryName);

// Set pattern for top face of box

      if (GpiSetPattern(hPS, PATSYM_DIAG4) == FALSE)
         SHOWERROR("GpiSetPattern", EntryName);

// Begin area for filling

      if (GpiBeginArea(hPS, BA_BOUNDARY | BA_ALTERNATE) == FALSE)
         SHOWERROR("GpiBeginArea", EntryName);

// Draw right face of box

      if (GpiPolyLine(hPS, 5L, PLine3) == GPI_ERROR)
         SHOWERROR("GpiPolyLine", EntryName);

// Close area for filling

      if (GpiEndArea(hPS) == GPI_ERROR)
         SHOWERROR("GpiEndArea", EntryName);

// Close element bracket

      if (GpiEndElement(hPS) == FALSE)
         SHOWERROR("GpiEndElement", EntryName);

// Set the element pointer to the box element

      if (GpiSetElementPointer(hPS, 4L) == FALSE)
         SHOWERROR("GpiSetElementPointer", EntryName);

// Query the element type to determine space needed for element data

      if ((ElementBufferSize = GpiQueryElementType(hPS, &Type, 0L, NULL))
                                                               ==GPI_ALTERROR)
         SHOWERROR("GpiQueryElementType", EntryName);

// Allocate space for the element orders

      if ((pbData = (PBYTE)malloc((size_t)ElementBufferSize)) == NULL)
         SHOWERROR("malloc", EntryName);

// Query the element contents with GpiQueryElement for later copies with
// GpiElement

      if ((RetLength = GpiQueryElement(hPS, 0L, ElementBufferSize, pbData)) ==
                                                                 GPI_ALTERROR)
         SHOWERROR("GpiQueryElement", EntryName);

   }
   else
   {

// Set scale factors for box positioning

      ScaleX = Scale * delta_x;

      ScaleY = Scale * delta_y;

      for (box_across = 0.0; box_across <= pyramid_level; box_across++)
      {

// Set Model Transform Matrix for this particular iteration's box

         MatStruct.fxM11 = MAKEFIXED(1, 0);
         MatStruct.fxM12 = MAKEFIXED(0, 0);
         MatStruct.lM13 = 0L;
         MatStruct.fxM21 = MAKEFIXED(0, 0);
         MatStruct.fxM22 = MAKEFIXED(1, 0);
         MatStruct.lM23 = 0L;
         MatStruct.lM31 =
         MapX((pyramid_level * -(ScaleX / 2.0)) + (ScaleX * box_across));
         MatStruct.lM32 = MapY(pyramid_level * -(ScaleY));
         MatStruct.lM33 = 1L;

// Build element description string

         sprintf(ElementDesc, "Level: %f Box: %f", pyramid_level, box_across);

// Build box label

         label = MAKEFIXED((LONG)pyramid_level, (LONG)box_across);

// Insert label before element group

         if (GpiLabel(hPS, (LONG)label)==FALSE)
            SHOWERROR("GpiLabel", EntryName);

// Set model transform for this element

         if (GpiSetModelTransformMatrix(hPS, 9L, &MatStruct,TRANSFORM_REPLACE)
                                                                     == FALSE)
            SHOWERROR("GpiSetModelTransformMatrix", EntryName);

// Write element data into segment (draw box)

         if (GpiElement(hPS, BOX_ELEMENT, ElementDesc, (LONG)RetLength,
                                                         pbData) == GPI_ERROR)
            SHOWERROR("GpiElement", EntryName);
      }
   }

// Free up allocated memory

   free(ElementDesc);

   *ErrorTotal = ErrorCount;

   return;

}
