#define INCL_DOS
#define INCL_WIN
#define INCL_GPI
#define INCL_ERRORS  // PM error definitions
#define INCL_DEV
#define MAINSOURCEFILE              //This main source code.

#include <string.h>
#include <stdio.h>
#include <os2.h>
#include "PTTDLLDF.h"  /* test case defines */
#include "routines.h"
#include "gpi_grid.h"  /* function to draw a precision d=grid on screen */
#include "std_util.h"
#include "gpiclip.h"  /* gpi clipping visible dll header file */

VOID set_clip_path(HPS hPS, POINTL ptl, INT size, HWND hWndClient, CHAR msg_circle[]);
VOID query_clip_box(HPS hPS, HWND hWndClient);
VOID exclude_clip_rect(HPS hps, HWND hWndClient);

INT error_count;

/*****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module Name: GPICLIP
 *
 * Purpose: This module is called by the PTT to test the Gpi Clipping
 *      Visible API's.
 *
 * Revision Log: 10/25/90, PDVT, Gisselle Labault
 *                    - created initial module template
 *               11/5/90, PDVT, Gisselle Labault
 *                    - testcase design
 *               11/8/90, PDVT, Gisselle Labault
 *                    - implementation of testcase design
 *
 * SRS References: Section 5.9
 *
 * Contents:  InitTest()
 *            GpiClipPre()
 *            GpiClipApp()
 *            GpiPtVisibleExh()
 *            GpiRectVisibleExh()
 *            GpiQueryClipBoxExh()
 *            GpiQueryClipRegionExh()
 *            GpiSetClipPathExh()
 *            GpiExcludeClipRectExh()
 *            GpiIntersectClipRectExh()
 *            GpiOffsetClipRegionExh()
 *
 * Design Overview:
 *  The GPICLIP testcase involves testing of the following GPIs:
 *  GpiExcludeClipRectangle(), GpiIntersectClipRectangle(),
 *  GpiOffsetClipRegion(), GpiSetClipRegion(), GpiSetClipPath(),
 *  GpiQueryClipBox(), GpiQueryClipRegion(), GpiPtVisible(),
 *  GpiRectVisible()
 *
 * Limitations:
 *  None.
 *
 *\\end
 *****************************************************************************/
/*****************************************************************************
 *\ddd
 * Routine Name: InitTest
 *
 * Purpose: Initialize function name and address for PTT
 *
 * Systems Requirements: None
 *
 * Revision Log: 10/25/90, PDVT, Gisselle Labault
 *               - created initial template
 *
 * Inputs:
 *   sel        // pass thru parameter for function calls
 *   pInitInfo  // pointer to the info array
 *
 * Outputs:
 *   pInitInfo  // updated with all entry point names ans addresses
 *
 * Subroutines Required:   InitTestSegVars()
 *
 * Limitations: none
 *
 *\\end
 *****************************************************************************/
/*****************************************************************************
 *\\ algorithm
 * {
 *  for each test routine
 *  {
 *     pInitInfo->function name = function name [i]
 *     pInitInfo->function address = function address[i]
 *  }
 *  call init set test seg vars function
 * }
 *\\end
 *****************************************************************************/
VOID APIENTRY InitTest(PVOID sel, PTESTINITINFO pInitInfo)
  {
    pInitInfo->ntests = NTESTS; /* set number of entry points */

    /* set each function name and address */
    strcpy(pInitInfo->group_name,"GPI Clip");

    strcpy(pInitInfo->test_names[0],"GpiClipPre");
    pInitInfo->test_addrs[0]=GpiClipPre;

    strcpy(pInitInfo->test_names[1],"GpiClipApp");
    pInitInfo->test_addrs[1]=GpiClipApp;

    strcpy(pInitInfo->test_names[2],"GpiPtVisibleExh");
    pInitInfo->test_addrs[2]=GpiPtVisibleExh;
    strcpy(pInitInfo->test_names[3],"GpiRectVisibleExh");
    pInitInfo->test_addrs[3]=GpiRectVisibleExh;
    strcpy(pInitInfo->test_names[4],"GpiQueryClipBoxExh");
    pInitInfo->test_addrs[4]=GpiQueryClipBoxExh;
    strcpy(pInitInfo->test_names[5],"GpiQueryClipRegionExh");
    pInitInfo->test_addrs[5]=GpiQueryClipRegionExh;
    strcpy(pInitInfo->test_names[6],"GpiSetClipPathExh");
    pInitInfo->test_addrs[6]=GpiSetClipPathExh;
    strcpy(pInitInfo->test_names[7],"GpiExcludeClipRectExh");
    pInitInfo->test_addrs[7]=GpiExcludeClipRectExh;
    strcpy(pInitInfo->test_names[8],"GpiIntersectClipRectExh");
    pInitInfo->test_addrs[8]=GpiIntersectClipRectExh;
    strcpy(pInitInfo->test_names[9],"GpiOffsetClipRegionExh");
    pInitInfo->test_addrs[9]=GpiOffsetClipRegionExh;


    InitTestSegVars(sel); /* this call used by thr PTT interface */
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiClipPre
 *
 * Purpose: This is an entry point for the PTT executes the precision tests
 *          by calling GpiClipPre.
 *
 * Systems Requirements: Section 5.9.2
 *
 * Revision Log: 10/25/90, PDVT, Gisselle Labault
 *               - created initial template
 *               11/7/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/7/90, PDVT, Gisselle Labault
 *                 - design implementation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   display_text(),  gpi_grid(),
 *                   check_rtn()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *  create a rectangular region
 *  set a clip to the region
 *  display text
 *  destroy clip and region
 *
 *  create a complex region
 *  set a clip to the region
 *  display text
 *  destroy clip and region
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiClipPre(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  RECTL     region[3];
  POINTL    ptl;
  HRGN      hrgn;
  LONG      complx;
  CHAR      str_msg[140];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;

  }
  else {
        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE, 0,"+ee+ GpiClipPre test - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        MapAndWriteString(hPS, delta_x, 5 * delta_y + delta_y/4, " 6 x 1 rectangular clipping area");
        /**********************/
        /* RECTANGULAR REGION */
        /**********************/
        region[0].xLeft = MapX(delta_x);
        region[0].yBottom = MapY(4 * delta_y);
        region[0].xRight = MapX(7 * delta_x);
        region[0].yTop = MapY(5 * delta_y);
        hrgn = GpiCreateRegion(hPS,  // Create region in presentation space
                               1L,   // of one rectangle using coordinates
                               region); // from region.

        /***********************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_RECT */
        /***********************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Rectangular Clipping Region - Entry\n");
        complx = GpiSetClipRegion(hPS, // set region created to a clipping region
                                 hrgn,
                                 NULL);
        if (complx == RGN_RECT) {  // check if indicator equals RGN_RECT
          ptl.x = delta_x;
          ptl.y = 6 * delta_y;
          display_text(hPS, ptl.x, ptl.y, "RECTANGULAR CLIPPING AREA...RECTANGULAR CLIPPING AREA...RECTANGULAR CLIPPING AREA...RECTANGULAR CLIIPING AREA");
          CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_RECT Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT)complx, hWndClient, "GpiSetClipRegion");
        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Rectangular Clipping Region - Exit\n");

        /******************/
        /* COMPLEX REGION */
        /******************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Complex Clipping Region - Entry\n");
        MapAndWriteString(hPS, delta_x, 3 * delta_y + delta_y/2, " Complex Clipping formed by two 2 x 1 rectangles");
        MapAndWriteString(hPS, delta_x, 3 * delta_y + delta_y/4, " and one 4 x 1 rectangle");
        region[0].xLeft = MapX(2 * delta_x);
        region[0].yBottom = MapY(delta_y);
        region[0].xRight = MapX(6 * delta_x);
        region[0].yTop = MapY(2 * delta_y);
        region[1].xLeft = MapX(delta_x);
        region[1].yBottom = MapY(2 * delta_y);
        region[1].xRight = MapX(3 * delta_x);
        region[1].yTop = MapY(3 * delta_y);
        region[2].xLeft = MapX(5 * delta_x);
        region[2].yBottom = MapY(2 * delta_y);
        region[2].xRight = MapX(7 * delta_x);
        region[2].yTop = MapY(3 * delta_y);
        hrgn = GpiCreateRegion(hPS,  // Create region in presentation space
                               3L,   // of three rectangles using coordinates
                               region); // from region.
        /**************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_COMPLEX */
        /**************************************************************/
        complx = GpiSetClipRegion(hPS, // set region created to a clipping region
                                 hrgn,
                                 NULL);
        if (complx == RGN_COMPLEX) {  // check if indicator equals RGN_COMPLEX
          ptl.x = delta_x;
          ptl.y = 6 * delta_y;
          display_text(hPS, ptl.x, ptl.y, "COMPLEX CLIPPING REGION...COMPLEX CLIPPING REGION...COMPLEX CLIPPING REGION...COMPLEX CLIPPING REGION");
          CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_COMPLEX Test                              -    PASS.\n");
        }
        else
          check_rtn((INT)complx, hWndClient, "GpiSetClipRegion");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Complex Clipping Region - Exit\n");

        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiClipPre                                                     -    PASS\n");
        else {
              sprintf(str_msg, "$ERRORCOUNT = %d.\n", error_count);
              CWRITELOG(L_HDR,0, str_msg);
              CWRITELOG(L_HDR,0,
"$GpiClipPre                                                     -    FAIL\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ GpiClipPre test - Exit\n");
  }
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiClipApp
 *
 * Purpose: This is an entry point for the PTT that executes the
 *          application test.
 *
 * Systems Requirements: Section 5.9.2
 *
 * Revision Log: 10/25/90, PDVT, Gisselle Labault
 *                 - created initial template
 *               11/5/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/6/90, PDVT, Gisselle Labault
 *                 - design implementation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   display_text(), gpi_grid(), check_rtn(),
 *                   DosSleep(), rvis_returned_rtn(), set_clip_path()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 *  create a rectangular region
 *  set a clip to the region
 *  display text
 *  call GpiPtVisible, and expect a PVIS_VISIBLE
 *  call GpiQueryClipBox, and expect a complexity indicator RGN_RECT
 *  call GpiRectVisible, and expect a RVIS_INVISIBLE
 *  call GpiQueryClipRegion, and expect no error
 *  call GpiOffsetClipRegion, and expect a complexity indicator RGN_RECT
 *  display text
 *  call GpiExcludeClipRectangle, and expect a complexity indicator RGN_COMPLEX
 *  display text
 *  call GpiIntersectClipRectangle, and expect a complexity indicator RGN_RECT
 *  display text
 *  destroy clip and region
 *  call GpiSetArcParams
 *  Begin Path
 *  Draw circle
 *  End Path
 *  call GpiSetClipPath, and expect no errors
 *  display text
 *  reset clip path
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiClipApp(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  RECTL    region;
  RECTL    rect_test;
  POINTL   ptl;
  POINTL   offset;
  HRGN     hrgn;
  HRGN     clip_region_handle;
  LONG     complx;
  CHAR     str_msg[140];
  INT      rvis_returned;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
       TEST_SIGNATURE(0, 0);
       LOG_SIGNATURE;
       CWRITELOG(L_LOTRACE,0,"+ee+ GpiClipApp test - Entry\n");
       gpi_grid(hPS, (LONG)INCH);
       error_count = 0;

       MapAndWriteString(hPS, delta_x, 5 * delta_y + delta_y/5, "Current clipping region");
       MapAndWriteString(hPS, delta_x/2, 2 * delta_y + delta_y/2, "Rectangle excluded from");
       MapAndWriteString(hPS, delta_x - 500, 2 * delta_y + delta_y/4, "clipping region ---------->");
       MapAndWriteString(hPS, 3 * delta_x + delta_x/2, 3 * delta_y + delta_y/6, "Clipping region has been moved");
       MapAndWriteString(hPS, 5 * delta_x, delta_y + delta_y/2, " <--- Intersect a");
       MapAndWriteString(hPS, 5 * delta_x, delta_y + delta_y/3, " new clip region");
       region.xLeft = MapX(delta_x);
       region.yBottom = MapY(3 * delta_y);
       region.xRight = MapX(3 * delta_x);
       region.yTop = MapY(5 * delta_y);
       hrgn = GpiCreateRegion(hPS,  // Create region in presentation space
                              1L,   // of one rectangle using coordinates
                              &region); // from region.

       /***********************************************************/
       /* Test GpiSetClipRegion for Complexity Indicator RGN_RECT */
       /***********************************************************/
       CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiSetClipRegion RGN_RECT - Entry\n");
       complx = GpiSetClipRegion(hPS, // set region created to a clipping region
                                hrgn,
                                NULL);
       if (complx == RGN_RECT) {  // check if indicator equals RGN_RECT
          memset(str_msg, '|', sizeof(str_msg));
          ptl.x = delta_x;
          ptl.y = 5 * delta_y;
          display_text(hPS, ptl.x, ptl.y, str_msg);
          CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_RECT Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT)complx, hWndClient, "GpiSetClipRegion");

        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiSetClipRegion RGN_RECT - Exit\n");

        /***********************************************************/
        /* Test GpiPtVisible for Visibility Indicator PVIS_VISIBLE */
        /***********************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiPtVisible PVIS_VISIBLE - Entry\n");
        ptl.x = delta_x;
        ptl.y = 4 * delta_y;
        MapPt(ptl);
        rvis_returned = (INT) GpiPtVisible(hPS, &ptl);
        if (rvis_returned == PVIS_VISIBLE)
          CWRITELOG(L_LOTRACE,0,
"$GpiPtVisible PVIS_VISIBLE Test                                 -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiPtVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiPtVisible PVIS_VISIBLE - Exit\n");

        /**********************************************************/
        /* Test GpiQueryClipBox for Complexity Indicator RGN_RECT */
        /**********************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiQueryClipBox RGN_RECT - Entry\n");
        complx = GpiQueryClipBox(hPS,          // query the region to get
                                 &rect_test);  // the tightest rectangle

        if (complx == RGN_RECT )   // check if indicator is RGN_RECT
            CWRITELOG(L_LOTRACE,0,
"$GpiQueryClipBox RGN_RECT Test                                  -    PASS.\n");
        else  // FAILURE
          check_rtn((INT) complx, hWndClient, "GpiQueryClipBox");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiQueryClipBox  RGN_RECT - Exit\n");

        /***************************************************************/
        /* Test GpiRectVisible for Visibility Indicator RVIS_INVISIBLE */
        /***************************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiRectVisible RVIS_INVISIBLE - Entry\n");
        region.xLeft = MapX(delta_x);
        region.yBottom = MapY(delta_y);
        region.xRight = MapX(2 * delta_x);
        region.yTop = MapY(2 * delta_y);
        rvis_returned = (INT) GpiRectVisible(hPS, &region);
        if (rvis_returned == RVIS_INVISIBLE)
          CWRITELOG(L_LOTRACE,0,
"$GpiRectVisible RVIS_INVISIBLE Test                             -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiRectVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiRectVisible RVIS_INVISIBLE - Exit\n");

        /**********************************************************/
        /* Test GpiQueryClipRegion for a valid clip region handle */
        /**********************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiQueryClipRegion - Entry\n");
        clip_region_handle = GpiQueryClipRegion(hPS);
        if ((clip_region_handle != HRGN_ERROR) &&
            (clip_region_handle != NULLHANDLE))
          CWRITELOG(L_LOTRACE,0,
"$GpiQueryClipRegion Test                                        -    PASS.\n");
        else  // FAILURE
          check_rtn((INT) complx, hWndClient, "GpiQueryClipRegion");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiQueryClipRegion - Exit\n");

        /**************************************************************/
        /* Test GpiOffsetClipRegion for Complexity Indicator RGN_RECT */
        /**************************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiOffsetClipRegion - Entry\n");
        offset.x = 2 * delta_x;
        offset.y = -2 * delta_y;
        MapPt(offset);
        complx = GpiOffsetClipRegion(hPS, &offset);
        if (complx == RGN_RECT) {   // check if indicator equals RGN_RECT
           memset(str_msg, '-', sizeof(str_msg));
           ptl.x = 3 * delta_x;
           ptl.y = 4 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiOffsetClipRegion RGN_RECT Test                              -    PASS.\n");
        }
        else  //  FAILURE
          check_rtn((INT) complx, hWndClient, "GpiOffsetClipRegion");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiOffsetClipRegion - Exit\n");

        /*********************************************************************/
        /* Test GpiExcludeClipRectangle for Complexity Indicator RGN_COMPLEX */
        /*********************************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiExcludeClipRectangle - Entry\n");
        region.xLeft = MapX(3 * delta_x);
        region.yBottom = MapY(2 * delta_y);
        region.xRight = MapX(4 * delta_x);
        region.yTop = MapY(2 * delta_y + delta_y/2);
        complx = GpiExcludeClipRectangle(hPS, &region);
        if (complx == RGN_COMPLEX)  {  // check if indicator equals RGN_COMPLEX
           memset(str_msg, '/', sizeof(str_msg));
           ptl.x = 2 * delta_x;
           ptl.y = 4 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiExcludeClipRectangle RGN_COMPLEX Test                       -    PASS.\n");
        }
        else  //  FAILURE
          check_rtn((INT) complx, hWndClient, "GpiExcludeClipRectangle");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiExcludeClipRectangle - Exit\n");

        /********************************************************************/
        /* Test GpiIntersectClipRectangle for Complexity Indicator RGN_RECT */
        /********************************************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiIntersectClipRectangle - Entry\n");
        region.xLeft = MapX(4 * delta_x);
        region.yBottom = MapY(delta_y);
        region.xRight = MapX(7 * delta_x);
        region.yTop = MapY(2 * delta_y);
        complx = GpiIntersectClipRectangle(hPS, &region);
        if (complx == RGN_COMPLEX) {   // check if indicator equals RGN_COMPLEX
           memset(str_msg, '\\', sizeof(str_msg));
           ptl.x = 2 * delta_x;
           ptl.y = 4 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiIntersectClipRectangle RGN_COMPLEX Test                     -    PASS.\n");
        }
        else  //  FAILURE
          check_rtn((INT) complx, hWndClient, "GpiIntersectClipRectangle");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiIntersectClipRectangle - Exit\n");

        /***********************/
        /* Test GpiSetClipPath */
        /***********************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test GpiSetClipPath - Entry\n");
        MapAndWriteString(hPS, 6 * delta_x, 6 * delta_y + delta_y/5, " Patterns:");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 6 * delta_y - delta_y/4, "|  Vertical Bar");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 6 * delta_y - delta_y/2, "\\  Back Slash");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 5 * delta_y + delta_y/4, "/  Slash");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 5 * delta_y, "-  Dash");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 5 * delta_y - delta_y/4, "x  Letter");
        MapAndWriteString(hPS, 3 * delta_x, 6 * delta_y - delta_y/4, "   Current clip path");
        memset(str_msg, 'x', sizeof(str_msg));
        ptl.x = 4 * delta_x + delta_x/2;
        ptl.y = 4 * delta_y + delta_y/2;
        set_clip_path(hPS, ptl, 2, hWndClient, str_msg);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test GpiSetClipPath - Exit\n");

        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiClipApp                                                     -    PASS.\n");
        else {
             sprintf(str_msg, "$ERRORCOUNT = %d.\n", error_count);
             CWRITELOG(L_HDR,0, str_msg);
             CWRITELOG(L_HDR,0,
"$GpiClipApp                                                     -    FAIL.\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ GpiClipApp test - Exit\n");
  }
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiPtVisibleExh
 *
 * Purpose: This function exhaustively tests for all visibility indicators
 *      returned by GpiPtVisible(). The syntax used to call the gpi is:
 *      (LONG) GpiPtVisible(
 *             HPS  hPS,
 *             POINTL pptl)
 *
 *      GpiPtVisible checks whether a point is visible within the clipping
 *      region of the device associated with the specified presentation
 *      space. The function returns the following visibility indicators:
 *
 *      PVIS_INVISIBLE
 *      PVIS_VISIBLE
 *      PVIS_ERROR
 *
 * Systems Requirements: Section 5.9.3
 *
 * Revision Log: 10/25/90, PDVT, Gisselle Labault
 *               - created initial template
 *               11/9/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/28/90, PDVT, Gisselle Labault
 *                    - redesign
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   display_text(), gpi_grid(), check_rtn(),
 *               rvis_returned_rtn()
 *
 * Limitations: none
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   Create a rectangular region
 *   Clip the region
 *   if region is not rectangular
 *     call check_rtn()
 *   endif
 *   display text in clipping region
 *
 *   // Test for PVIS_INVISIBLE
 *   pass to GpiPtVisible point that is not included in clipping region
 *   if GpiPtVisible doesn't return PVIS_INVISIBLE
 *     call rvis_returned_rtn()
 *   endif
 *
 *   // Test for PVIS_VISIBLE
 *   pass to GpiPtVisible point that is included in clipping region
 *   if GpiPtVisible doesn't return PVIS_VISIBLE
 *     call rvis_returned_rtn()
 *   endif
 *
 *   // Test for PVIS_ERROR
 *   pass an invalid presentation space to GpiPtVisible
 *   if GpiPtVisible returns PVIS_ERROR
 *     call CHECKRETURNERROR()
 *   else
 *     call rvis_returned_rtn()
 *   endif
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiPtVisibleExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  RECTL    region;
  POINTL   ptl;
  HRGN     hrgn;
  LONG     complx;
  CHAR     str_msg[100];
  INT      rvis_returned;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ Test GpiPtVisible - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        strcpy(str_msg, "VISIBLE POINTS...VISIBLE POINTS...VISIBLE POINTS...VISIBLE POINTS...VISIBLE POINTS");
        MapAndWriteString(hPS, delta_x, 6 * delta_y, "Checks whether a point is visible within the clipping region");
        MapAndWriteString(hPS, delta_x, 6 * delta_y - delta_y/5, "of the device associated with the specified presentation space");
        MapAndWriteString(hPS, 3 * delta_x - delta_x/5, 5 * delta_y -(delta_y/2), "<<< Rectangular Region >>>");
        MapAndWriteString(hPS, 2 * delta_x - delta_x/4, delta_y + delta_y/4, "2 x 3 inches");

        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator PVIS_INVISIBLE - Entry\n");
        region.xLeft = MapX(3 * delta_x);
        region.yBottom = MapY(1 * delta_y);
        region.xRight = MapX(5 * delta_x);
        region.yTop = MapY(4 * delta_y);
        hrgn = GpiCreateRegion(hPS,      // create region in presentation space
                               1L,       // of one rectangle using coordinates
                               &region);  // from region.
        /***********************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_RECT */
        /***********************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                 hrgn,
                                 NULL);
        if (complx == RGN_RECT) {   // check if indicator equals RGN_RECT
           ptl.x = 2 * delta_x;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_RECT Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        /************************************************/
        /* Test for Visibility Indicator PVIS_INVISIBLE */
        /************************************************/
        ptl.x = 5;
        ptl.y = 200;
        rvis_returned = (INT) GpiPtVisible(hPS, &ptl);
        if (rvis_returned == PVIS_INVISIBLE)
          CWRITELOG(L_LOTRACE,0,
"$GpiPtVisible PVIS_INVISIBLE Test                               -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiPtVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator PVIS_INVISIBLE - Exit\n");

        /**********************************************/
        /* Test for Visibility Indicator PVIS_VISIBLE */
        /**********************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator PVIS_VISIBLE - Entry\n");
        ptl.x = 4 * delta_x;
        ptl.y = 2 * delta_y;
        MapPt(ptl);
        rvis_returned = (INT) GpiPtVisible(hPS, &ptl);
        if (rvis_returned == PVIS_VISIBLE)
          CWRITELOG(L_LOTRACE,0,
"$GpiPtVisible PVIS_VISIBLE Test                                 -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiPtVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator PVIS_VISIBLE - Exit\n");

        /********************************************/
        /* Test for Visibility Indicator PVIS_ERROR */
        /********************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator PVIS_ERROR - Entry\n");
        rvis_returned = (INT) GpiPtVisible(0, &ptl);
        if (rvis_returned == PVIS_ERROR)
          error_count += CHECKRETURNERROR(rvis_returned, "GpiPtVisible", SEVERITY_ERROR, PMERR_INV_HPS);
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiPtVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator PVIS_ERROR - Exit\n");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);

        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiPtVisible                                                   -    PASS.\n");
        else {
              sprintf(str_msg, "$ERRORCOUNT = %d.\n", error_count);
              CWRITELOG(L_HDR,0, str_msg);
              CWRITELOG(L_HDR,0,
"$GpiPtVisible                                                   -    PASS.\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ Test GpiPtVisible - Exit\n");
  }
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiRectVisibleExh
 *
 * Purpose: This function exhaustively tests for all visibility indicators
 *      returned by GpiRectVisible(). The syntax used to call the gpi is:
 *      (LONG) GpiRectVisible(
 *             HPS  hPS,
 *             RECTL rectl)
 *
 *      GpiRectVisible checks whether any part of a rectangle lies within
 *      the clipping region of the device associated with the specified
 *      presentation space. The function returns the following visibility
 *      indicators:
 *
 *      RVIS_INVISIBLE
 *      RVIS_VISIBLE
 *      RVIS_PARTIAL
 *      RVIS_ERROR
 *
 * Systems Requirements: Section 5.9.3
 *
 * Revision Log: 10/25/90, PDVT, Gisselle Labault
 *               - created initial template
 * Revision Log: 11/7/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/28/90, PDVT, Gisselle Labault
 *                    - redesign
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   set_delt(), rvis_returned_rtn(), display_text()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   Create a complex region
 *   Clip the region
 *   if region is not complex
 *     call check_rtn()
 *   else
 *     display text in clipping region
 *   endif
 *
 *   // Test for RVIS_INVISIBLE
 *   pass to GpiRectVisible the coordinates that are not included in clipping region
 *   if GpiRectVisible doesn't return RVIS_INVISIBLE
 *     call rvis_returned_rtn()
 *   endif
 *
 *   // Test for RVIS_VISIBLE
 *   pass to GpiRectVisible the coordinates that are included in clipping region
 *   if GpiRectVisible doesn't return RVIS_VISIBLE
 *     call rvis_returned_rtn()
 *   endif
 *
 *   // Test for RVIS_PARTIAL
 *   pass to GpiRectVisible few coordinates that are included in clipping region
 *   if GpiRectVisible doesn't return RVIS_PARTIAL
 *     call rvis_returned_rtn()
 *   endif
 *
 *   // Test for RVIS_ERROR
 *   pass invalid coordinates to GpiRectVisible
 *   if GpiRectVisible returns PVIS_ERROR
 *     call CHECKRETURNERROR()
 *   else
 *     call rvis_returned_rtn()
 *   endif
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiRectVisibleExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  RECTL    region[5];
  POINTL   ptl;
  HRGN     hrgn;
  LONG     complx;
  int      rvis_returned;
  CHAR     str_msg[100];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ Test GpiRectVisible - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        strcpy(str_msg, "VISIBLE RECTANGLE...VISIBLE RECTANGLE...VISIBLE RECTANGLE...VISIBLE RECTANGLE...VISIBLE");
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RVIS_INVISIBLE - Entry\n");
        MapAndWriteString(hPS, delta_x, 5 * delta_y, "   Checks whether any part of a rectangle is visible on the");
        MapAndWriteString(hPS, delta_x, 5 * delta_y - (delta_y/4), "   device associated with the specified presentation space");
        MapAndWriteString(hPS, 3 * delta_x + delta_x/5, 4 * delta_y + delta_y/5, " <<< Complex Region >>>");

        region[0].xLeft = MapX(2 * delta_x);
        region[0].yBottom = MapY(delta_y);
        region[0].xRight = MapX(3 * delta_x);
        region[0].yTop = MapY(2 * delta_y);

        region[1].xLeft = MapX(3 * delta_x);
        region[1].yBottom = MapY(2 * delta_y);
        region[1].xRight = MapX(4 * delta_x);
        region[1].yTop = MapY(3 * delta_y);

        region[2].xLeft = MapX(4 * delta_x);
        region[2].yBottom = MapY(3 * delta_y);
        region[2].xRight = MapX(5 * delta_x);
        region[2].yTop = MapY(4 * delta_y);

        region[3].xLeft = MapX(5 * delta_x);
        region[3].yBottom = MapY(2 * delta_y);
        region[3].xRight = MapX(6 * delta_x);
        region[3].yTop = MapY(3 * delta_y);

        region[4].xLeft = MapX(6 * delta_x);
        region[4].yBottom = MapY(delta_y);
        region[4].xRight = MapX(7 * delta_x);
        region[4].yTop = MapY(2 * delta_y);
        hrgn = GpiCreateRegion(hPS,      // create region in presentation space
                               5L,       // of five rectangles using coordinates
                               region);  // from region.

        /**************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_COMPLEX */
        /**************************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                 hrgn,
                                 NULL);
        if (complx == RGN_COMPLEX) {  // check if indicator equals RGN_COMPLEX
           ptl.x = delta_x;
           ptl.y = 4 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_COMPLEX Test                              -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        /************************************************/
        /* Test for Visibility Indicator RVIS_INVISIBLE */
        /************************************************/
        region[0].xLeft = 5;
        region[0].yBottom = 20;
        region[0].xRight = 25;
        region[0].yTop = 50;
        rvis_returned = (INT) GpiRectVisible(hPS, &region[0]);
        if (rvis_returned == RVIS_INVISIBLE)
          CWRITELOG(L_LOTRACE,0,
"$GpiRectVisible RVIS_INVISIBLE Test                             -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiRectVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RVIS_INVISIBLE - Exit\n");

        /**********************************************/
        /* Test for Visibility Indicator RVIS_VISIBLE */
        /**********************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RVIS_VISIBLE - Entry\n");
        region[0].xLeft = MapX(3 * delta_x + delta_x/5);
        region[0].yBottom = MapY(2 * delta_y + delta_y/5);
        region[0].xRight = MapX(4 * delta_x - delta_x/5);
        region[0].yTop = MapY(3 * delta_y - delta_y/5);
        rvis_returned = (INT) GpiRectVisible(hPS, &region[0]);
        if (rvis_returned == RVIS_VISIBLE)
          CWRITELOG(L_LOTRACE,0,
"$GpiRectVisible RVIS_VISIBLE Test                               -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiRectVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RVIS_VISIBLE - Exit\n");

        /**********************************************/
        /* Test for Visibility Indicator RVIS_PARTIAL */
        /**********************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RVIS_PARTIAL - Entry\n");
        region[0].xLeft = MapX(3 * delta_x/2);
        region[0].yBottom = MapY(1 * delta_y);
        region[0].xRight = MapX(5 * delta_x);
        region[0].yTop = MapY(3 * delta_x);
        rvis_returned = (INT) GpiRectVisible(hPS, &region[0]);
        if (rvis_returned == RVIS_PARTIAL)
          CWRITELOG(L_LOTRACE,0,
"$GpiRectVisible RVIS_PARTIAL Test                               -    PASS.\n");
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiRectVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RVIS_PARTIAL - Exit\n");

        /********************************************/
        /* Test for Visibility Indicator RVIS_ERROR */
        /********************************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RVIS_ERROR - Entry\n");
        region[0].xLeft = MapX(100 * delta_x);
        region[0].yBottom = MapY((LONG)(999.99 * delta_y));
        region[0].xRight = MapX((1100 * delta_x));
        region[0].yTop = MapY((LONG)(98.229 * delta_x));
        rvis_returned = (INT) GpiRectVisible(hPS, &region[0]);
        if (rvis_returned == RVIS_ERROR)
          error_count += CHECKRETURNERROR(rvis_returned, "GpiRectVisible", SEVERITY_ERROR, PMERR_INV_RECT);
        else
          rvis_returned_rtn(rvis_returned, hWndClient, "GpiRectVisible");
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RVIS_ERROR - Exit\n");
        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);

        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiRectVisible                                                 -    PASS.\n");
        else {
              sprintf(str_msg, "$ERRORCOUNT = %d.\n", error_count);
              CWRITELOG(L_HDR,0, str_msg);
              CWRITELOG(L_HDR,0,
"$GpiRectVisible                                                 -    FAIL.\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ Test GpiRectVisible - Exit\n");
  }
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiExcludeClipRectExh
 *
 * Purpose: This function exhaustively tests for all visibility indicators
 *      returned by GpiExcludeClipRectangle(). The syntax used to call
 *      the gpi is:
 *      (LONG) GpiExcludeClipRectangle(
 *             HPS  hPS,
 *             RECTL rectl)
 *
 *      GpiExcludeClipRectangle excludes a rectangle from the clipping
 *      region. The function returns the following indicators:
 *
 *      RGN_NULL
 *      RGN_RECT
 *      RGN_COMPLEX
 *      RGN_ERROR
 *
 * Systems Requirements: Section
 *
 * Revision Log: 11/10/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/14/90, PDVT, Gisselle Labault
 *                 - design implementation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   gpi_grid(), CHECKRETURNERROR(),
 *                   check_rtn(), display_text()
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   // Test for RGN_NULL
 *   Create a null region
 *   Clip the region
 *   if region is not null
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   call GpiExcludeClipRectangle with valid coordinates
 *   if GpiExcludeClipRectangle doesn't return RGN_NULL
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   destroy region
 *
 *   // Test for RGN_COMPLEX
 *   Create a complex region
 *   Clip the region
 *   if region is not complex
 *     call check_rtn()
 *   endif
 *   call GpiExcludeClipRectangle with valid complex coordinates
 *   if GpiExcludeClipRectangle doesn't return RGN_COMPLEX
 *     call check_rtn()
 *   endif
 *   destroy region
 *
 *   // Test for RGN_RECT
 *   Create a rectangular region
 *   Clip the region
 *   if region is not rectangular
 *     call check_rtn()
 *   endif
 *   call GpiExcludeClipRectangle with valid coordinates for a rectangle
 *   if GpiExcludeClipRectangle doesn't return RGN_RECT
 *     call check_rtn()
 *   endif
 *   destroy region
 *
 *   // Test for RGN_ERROR
 *   Create a rectangular region
 *   Clip the region with an invalid presentation space handle
 *   if clipping region is in error
 *     call CHECKRETURNERROR()
 *   else
 *     call check_rtn()
 *   endif
 *   call GpiExcludeClipRectangle with valid coordinates for a rectangle
 *   if GpiExcludeClipRectangle returns RGN_ERROR
 *     call CHECKRETURNERROR()
 *   else
 *     call check_rtn()
 *   endif
 *   destroy region
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiExcludeClipRectExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else
    exclude_clip_rect(hPS, hWndClient);
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiQueryClipBoxExh
 *
 * Purpose: This function exhaustively tests for all complexity indicators
 *      returned by GpiQueryClipBox(). The syntax used to call
 *      the gpi is:
 *      (LONG) GpiQueryClipBox(
 *             HPS  hPS,
 *             RECTL rectl)
 *
 *      GpiQueryClipBox returns the dimensions of the tightest rectangle
 *      which completely encloses the intersection of all clipping
 *      definitions. The function returns the following complexity
 *      indicators:
 *
 *      RGN_NULL      Null region
 *      RGN_RECT      Rectangular region
 *      RGN_COMPLEX   Complex region
 *      RGN_ERROR     Error
 *
 * Revision Log: 11/09/90, PDVT, JANET TRENT
 *                 - initial
 *               11/19/90, PDVT, Gisselle Labault
 *                 - redesign
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   gpi_grid(), CHECKRETURNERROR(),
 *                   check_rtn(), display_text()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   // Test for RGN_NULL
 *   Create a null region
 *   Clip the region
 *   if region is not null
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   call GpiQueryClipBox()
 *   if GpiQueryClipBox doesn't return RGN_NULL
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   destroy region
 *
 *   // Test for RGN_COMPLEX
 *   Create a complex region
 *   Clip the region
 *   if region is not complex
 *     call check_rtn()
 *   endif
 *   call GpiQueryClipBox()
 *   if GpiQueryClipBox doesn't return RGN_COMPLEX
 *     call check_rtn()
 *   endif
 *   destroy region
 *
 *   // Test for RGN_RECT
 *   Create a rectangular region
 *   Clip the region
 *   if region is not rectangular
 *     call check_rtn()
 *   endif
 *   call GpiQueryClipBox()
 *   if GpiQueryClipBox doesn't return RGN_RECT
 *     call check_rtn()
 *   endif
 *   destroy region
 *
 *   // Test for RGN_ERROR
 *   Create a rectangular region
 *   Clip the region with an invalid presentation space handle
 *   if clipping region is in error
 *     call CHECKRETURNERROR()
 *   else
 *     call check_rtn()
 *   endif
 *   call GpiQueryClipBox()
 *   if GpiQueryClipBox returns RGN_ERROR
 *     call CHECKRETURNERROR()
 *   else
 *     call check_rtn()
 *   endif
 *   destroy region
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiQueryClipBoxExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else
    query_clip_box(hPS, hWndClient);
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiQueryClipRegionExh
 *
 * Purpose: This function tests for GpiQueryClipRegion functionality.
 *      The syntax used to call the gpi is:
 *      (HRGN) GpiQueryClipRegion(
 *             HPS  hPS)
 *
 *      GpiQueryClipRegion returns the handle of the currently selected
 *      clip region. The function returns the following return codes:
 *
 *      NULL          Null handle(no region selected)
 *      HRGN_ERROR    Error
 *      Otherwise     Clip region handle
 *
 * Revision Log: 11/09/90, PDVT, JANET TRENT
 *               - initial
 *               11/19/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/19/90, PDVT, Gisselle Labault
 *                 - design implementation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   gpi_grid(), CHECKRETURNERROR(),
 *                   check_rtn(), display_text()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   // Test for a valid clip region handle
 *   create 3 regions with valid coordinates, using GpiCreateRegion
 *   combines regions by calling GpiCombineRegion
 *   clip the region
 *   if region is not complex
 *     call check_rtn()
 *   endif
 *   call GpiQueryClipRegion()
 *   if GpiQueryClipRegion returns the clip region handle
 *     display text in clipping region
 *   else
 *     report difference to CWRITELOG
 *   endif
 *   destroy regions
 *
 *   // Test for NULL handle
 *   call GpiQueryClipRegion()
 *   if GpiQueryClipRegion doesn't return NULL
 *     report difference to CWRITELOG
 *   endif
 *
 *   // Test for HRGN_ERROR
 *   call GpiQueryClipRegion with an invalid presentation space handle
 *   if clipping region is in error
 *     call CHECKRETURNERROR()
 *   else
 *     report difference to CWRITELOG
 *   endif
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiQueryClipRegionExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
   POINTL ptl;
   RECTL region;
   HRGN hrgn1;
   HRGN hrgn2;
   HRGN hrgn3;
   HRGN clip_region_handle;
   LONG complx;
   CHAR str_msg[200];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE, 0, "+ee+ Test GpiQueryClipRegion - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        CWRITELOG(L_LOTRACE,0,"+ie+ Test HRGN is a clip region handle - Entry\n");
        region.xLeft = MapX(delta_x / 2);
        region.yBottom = MapY(delta_y);
        region.xRight = MapX(5 * delta_x + delta_x/2);
        region.yTop = MapY(4 * delta_y);

        hrgn1 = GpiCreateRegion(hPS,      // create region in presentation space
                                1L,       // of one rectangle using
                                &region); // region coordinates

        region.xLeft = MapX(2 * delta_x);
        region.yBottom = MapY(3 * delta_y);
        region.xRight = MapX(7 * delta_x);
        region.yTop = MapY(6 * delta_y);
        hrgn2 = GpiCreateRegion(hPS,    // create region in presentation space
                              1L,       // of one rectangle using
                              &region); // region coordinates

        region.xLeft = MapX(0);
        region.yBottom = MapY(0);
        region.xRight = MapX(0);
        region.yTop = MapY(0);
        hrgn3 = GpiCreateRegion(hPS,    // create region in presentation space
                              1L,       // of one rectangle using
                              &region); // region coordinates

        // combine regions in regions hrgn3
        GpiCombineRegion(hPS, hrgn3, hrgn1, hrgn2, CRGN_XOR);
        MapAndWriteString(hPS, 2 * delta_x + delta_x/2, 3 * delta_y + delta_y/2, "THIS AREA IS NOT PART OF");
        MapAndWriteString(hPS, 2 * delta_x + delta_x/2, 3 * delta_y + delta_y/4, "THE CLIPPING REGION");
        GpiSetPattern(hPS, PATSYM_DIAG4);
        GpiPaintRegion(hPS, hrgn3);
        /**************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_COMPLEX */
        /**************************************************************/
        complx = GpiSetClipRegion(hPS, hrgn3, NULL); // set region created to a
                                     // clipping region
        if (complx == RGN_COMPLEX)   // check if indicator is RGN_COMPLEX
          CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_COMPLEX Test                              -    PASS.\n");
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");
        clip_region_handle = GpiQueryClipRegion(hPS);  // get the handle of the
                                                       // current clip region

        if ((clip_region_handle != HRGN_ERROR) &&
            (clip_region_handle != NULLHANDLE)) {
           strcpy(str_msg, "GPIQUERYCLIPREGION-SUCCESSFUL-GPIQUERYCLIPREGION-SUCCESSFUL-GPIQUERYCLIPREGION-SUCCESSFUL-GPIQUERYCLIPREGION");
           ptl.x = 0;
           ptl.y = 6 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiQueryClipRegion Test                                        -    PASS.\n");
        }
        else {
          error_count += 1;
          CWRITELOG(L_FAIL,0,
"$Unexpected Error                                               -    FAIL.\n");
        }
        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn1);
        GpiDestroyRegion(hPS, hrgn2);
        GpiDestroyRegion(hPS, hrgn3);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test HRGN is a clip region handle - Exit\n");

       /********************/
       /* Test NULL handle */
       /********************/
       CWRITELOG(L_LOTRACE,0,"+ie+ Test NULL handle - Entry\n");
       clip_region_handle = GpiQueryClipRegion(hPS);  // get the handle of the
                                                      // current clip region
       if (clip_region_handle == NULLHANDLE)
         CWRITELOG(L_LOTRACE,0,
"$GpiQueryClipRegion Test NULL handle                            -    PASS.\n");
       else {
         error_count += 1;
         CWRITELOG(L_FAIL,0,
"$Unexpected Handle                                              -    FAIL.\n");
       }
       CWRITELOG(L_LOTRACE,0,"+ix+ Test NULL handle - Exit\n");

       /*******************/
       /* Test HRGN_ERROR */
       /*******************/
       CWRITELOG(L_LOTRACE,0,"+ie+ Test HRGN_ERROR - Entry\n");
       clip_region_handle = GpiQueryClipRegion(0);
       if (clip_region_handle != HRGN_ERROR)
         error_count += CHECKRETURNERROR(clip_region_handle, "GpiQueryClipRegion", SEVERITY_ERROR, PMERR_INV_HPS);
       else
         CWRITELOG(L_FAIL,0,
"$Unexpected Error                                               -    FAIL.\n");
       CWRITELOG(L_LOTRACE,0,"+ix+ Test HRGN_ERROR - Exit\n");

       if (!error_count)
         CWRITELOG(L_HDR,0,
"$GpiQueryClipRegion                                             -    PASS.\n");
       else {
             sprintf(str_msg, "$ERRORCOUNT = %d.\n", error_count);
             CWRITELOG(L_HDR,0, str_msg);
             CWRITELOG(L_HDR,0, "$GpiQueryClipRegion                                             -    FAIL.\n");
       }
       CWRITELOG(L_LOTRACE,0,"+ex+ Test GpiQueryClipRegion - Exit\n");
  }
} /* END - QUERY CLIP REGION EXHAUSTIVE TEST */
/***************************************************************************
 *\\ddd
 * Routine Name: GpiSetClipPathExh
 *
 * Purpose: This function tests for GpiSetClipPath functionality.
 *      The syntax used to call the gpi is:
 *      (BOOL) GpiSetClipPath(
 *             HPS   hPS,
 *             LONG  path,
 *             LONG  cmdoptions)
 *
 *      GpiSetClipPath selects a path as the current clip path
 *      The function returns:
 *      True    Successful completion
 *      False   Error occurred
 *
 * Revision Log: 11/09/90, PDVT, JANET TRENT
 *               - initial
 *               11/09/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/10/90, PDVT, Gisselle Labault
 *                 - design implementation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   gpi_grid(), CHECKRETURNERROR(),
 *                   set_clip_path()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   call set_clip_path() six times passing different patterns and sizes for
 *   the figure to be display on screen
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiSetClipPathExh(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
   POINTL  ptl;
   CHAR    str_msg[200];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ Test GpiSetClipPath - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        MapAndWriteString(hPS, delta_x, 5 * delta_y + delta_y/2, "Selects a path as the current clip path");
        MapAndWriteString(hPS, 6 * delta_x, 6 * delta_y + delta_y/5, " Patterns:");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 6 * delta_y - delta_y/4, "|  Vertical Bar");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 6 * delta_y - delta_y/2, "\\  Back Slash");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 5 * delta_y + delta_y/4, "/  Slash");
        MapAndWriteString(hPS, 6 * delta_x+delta_x/4, 5 * delta_y, "-  Dash");

        memset(str_msg, '-', sizeof(str_msg));
        ptl.x =  2 * delta_x;
        ptl.y =  3 * delta_y;
        set_clip_path(hPS, ptl, 3, hWndClient, str_msg);

        memset(str_msg, '|', sizeof(str_msg));
        ptl.x = 4 * delta_x + (delta_x / 2);
        ptl.y = 3 * delta_y;
        set_clip_path(hPS, ptl, 3, hWndClient, str_msg);

        memset(str_msg, '\\', sizeof(str_msg));
        ptl.x =  6 * delta_x + delta_x/4;
        ptl.y =  delta_y + delta_y/2;
        set_clip_path(hPS, ptl, 1, hWndClient, str_msg);

        memset(str_msg, '/', sizeof(str_msg));
        ptl.x =  3 * delta_x + delta_x/2;
        ptl.y =  4 * delta_y+delta_y/2;
        set_clip_path(hPS, ptl, 1, hWndClient, str_msg);

        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiSetClipPath                                                 -    PASS.\n");
        else {
             sprintf(str_msg, "$ERRORCOUNT = %d.\n", error_count);
             CWRITELOG(L_HDR,0, str_msg);
             CWRITELOG(L_HDR,0,
"$GpiSetClipPath                                                 -    FAIL.\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ Test GpiSetClipPath - Exit\n");
  }
} /* END - SET CLIP PATH EXHAUSTIVE TEST */

/***************************************************************************
 *\\ddd
 * Routine Name: GpiIntersectClipRectExh
 *
 * Purpose: This function exhaustively tests for all complexity indicators
 *      returned by GpiIntersectClipRectangle(). The syntax used to call
 *      the gpi is:
 *      (LONG) GpiIntersectClipRectangle(
 *             HPS  hPS,
 *             RECTL rectl)
 *
 *      GpiIntersectClipRect sets the new clip region to the intersection
 *      of the current clip region and the specified rectangle.
 *      The function returns the following complexity indicators:
 *
 *      RGN_NULL      Null region
 *      RGN_RECT      Rectangular region
 *      RGN_COMPLEX   Complex region
 *      RGN_ERROR     Error
 *
 * Revision Log: 11/19/90, PDVT, Gisselle Labault
 *                 - initial creation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   gpi_grid(), CHECKRETURNERROR(),
 *                   check_rtn(), display_text()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   // Test for RGN_NULL
 *   create a null region
 *   clip the region
 *   if region is not null
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   call GpiIntersectClipRectangle() with valid coordinates
 *   if GpiIntersectClipRectangle doesn't return RGN_NULL
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   destroy region
 *
 *   // Test for RGN_RECT
 *   create a rectangular region
 *   clip the region
 *   if region is not rectangular
 *     call check_rtn()
 *   endif
 *   call GpiIntersectClipRectangle() with valid coordinates for a rectangle
 *   if GpiIntersectClipRectangle doesn't return RGN_RECT
 *     call check_rtn()
 *   else
 *     display text in clipping area
 *   endif
 *   destroy region
 *
 *   // Test for RGN_COMPLEX
 *   create a complex region
 *   clip the region
 *   if region is not complex
 *     call check_rtn()
 *   endif
 *   call GpiIntersectClipRectangle() with valid complex coordinates
 *   if GpiIntersectClipRectangle doesn't return RGN_COMPLEX
 *     call check_rtn()
 *   else
 *     display text in clipping area
 *   endif
 *   destroy region
 *
 *   // Test for RGN_ERROR
 *   create a rectangular region
 *   clip the region with an invalid presentation space handle
 *   if clipping region is in error
 *     call CHECKRETURNERROR
 *   else
 *     call check_rtn()
 *   endif
 *   call GpiIntersectClipRectangle() with valid complex coordinates
 *   if GpiIntersectClipRectangle returns RGN_ERROR
 *     call CHECKRETURNERROR
 *   else
 *     call check_rtn()
 *   endif
 *   destroy region
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiIntersectClipRectExh(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  RECTL   intersect;
  RECTL   region[2];
  POINTL  ptl;
  HRGN    hrgn;
  LONG    complx;
  char    str1_msg[200],
          str2_msg[200];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else
  {
        /* * * * * * * * * * */
        /*  INITIALIZATIONS  */
        /* * * * * * * * * * */
        memset(str1_msg, '\\', sizeof(str1_msg));
        memset(str2_msg, '/', sizeof(str2_msg));
        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ Test GpiIntersectClipRectangle - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        /*******************************/
        /* Test for Indicator RGN_NULL */
        /*******************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_NULL - Entry\n");
        region[0].xLeft = MapX(delta_x);
        region[0].yBottom = MapY(delta_y);
        region[0].xRight = MapX(2 * delta_x);
        region[0].yTop = MapY(5 * delta_y);
        hrgn = GpiCreateRegion(hPS,     // Create region in presentation space
                               0L,      // with zero rectangle; using coordinates
                               region); // from region.

        /***********************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_NULL */
        /***********************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                hrgn,
                                NULL);
        if (complx == RGN_NULL) {   // check if indicator equals RGN_NULL
           /* This string should never appear on the screen */
           MapAndWriteString(hPS, delta_x, 3 * delta_y, "THIS SHOULD NEVER APPEAR ON THE SCREEN OR PAPER !!!");
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_NULL Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        intersect.xLeft = MapX(delta_x+delta_x/2);
        intersect.yBottom = MapY(3 * delta_y);
        intersect.xRight = MapX(2 * delta_x + delta_x/2);
        intersect.yTop =MapY(4 * delta_y);
        complx = GpiIntersectClipRectangle(hPS, &intersect);
        if (complx == RGN_NULL) {  // check if indicator equals RGN_NULL
           /* This string should never appear on the screen */
           MapAndWriteString(hPS, delta_x, 3 * delta_y, "THIS SHOULD NEVER APPEAR ON THE SCREEN OR PAPER !!!");
           CWRITELOG(L_LOTRACE,0,
"$GpiIntersectClipRectangle RGN_NULL Test                        -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiIntersectClipRectangle");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_NULL - Exit\n");

        /*******************************/
        /* Test for Indicator RGN_RECT */
        /*******************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_RECT - Entry\n");
        MapAndWriteString(hPS, delta_x, 6 * delta_y, "Sets a new clip region to the intersection of the current clip region");
        MapAndWriteString(hPS, delta_x/2, 5 * delta_y+delta_y/4, "<<< Rectangular Region >>>");

        hrgn = GpiCreateRegion(hPS,    // Create region in presentation space
                               1L,     // of one rectangle using coordinates
                               region); // from region.

        /***********************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_RECT */
        /***********************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                 hrgn,
                                 NULL);
        if (complx == RGN_RECT) {   // check if indicator equals RGN_RECT
           ptl.x = 0;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str1_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_RECT Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        complx = GpiIntersectClipRectangle(hPS, &intersect);
        if (complx == RGN_RECT) {  // check if indicator equals RGN_COMPLEX
           ptl.x = 0;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str2_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiIntersectClipRectangle RGN_RECT Test                        -    PASS.\n");
        }
        else  //  FAILURE
          check_rtn((INT) complx, hWndClient, "GpiIntersectClipRectangle");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_RECT - Exit\n");

        /**********************************/
        /* Test for Indicator RGN_COMPLEX */
        /**********************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_COMPLEX - Entry\n");
        MapAndWriteString(hPS, 4 * delta_x + delta_x/2, 5 * delta_y+delta_y/4, "<<< Complex Region >>>");
        region[0].xLeft = MapX(4 * delta_x);
        region[0].yBottom = MapY(delta_y);
        region[0].xRight = MapX(5 * delta_x);
        region[0].yTop = MapY(5 * delta_y);

        region[1].xLeft = MapX(6 * delta_x);
        region[1].yBottom = MapY(delta_y);
        region[1].xRight = MapX(7 * delta_x);
        region[1].yTop = MapY(5 * delta_y);
        hrgn = GpiCreateRegion(hPS,      // create region in presentation space
                                2L,      // of two rectangles using coordinates
                                region); // from region.

        /**************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_COMPLEX */
        /**************************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                 hrgn,
                                NULL);
        if (complx == RGN_COMPLEX) {   // check if indicator equals RGN_COMPLEX
           ptl.x = 3 * delta_x;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str1_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_COMPLEX Test                              -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        intersect.xLeft = MapX(4 * delta_x + delta_x/2);
        intersect.yBottom = MapY(2 * delta_y);
        intersect.xRight = MapX(6 * delta_x + delta_x/2);
        intersect.yTop =MapY(3 * delta_y);
        complx = GpiIntersectClipRectangle(hPS, &intersect);
        if ( complx == RGN_COMPLEX ) {   // check if indicator is RGN_COMPLEX
           ptl.x = 3 * delta_x;
           ptl.y = 7 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str2_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiIntersectClipRectangle RGN_COMPLEX Test                     -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiIntersectClipRectangle");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_COMPLEX - Exit\n");

        /********************************/
        /* Test for Indicator RGN_ERROR */
        /********************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_ERROR - Entry\n");
        hrgn = GpiCreateRegion( hPS,     // create region in presentation space
                                1L,      // of one rectangle using coordinates
                                region); // from region.
        /************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_ERROR */
        /************************************************************/
        complx = GpiSetClipRegion(0,  // set invalid presentation handle
                                  hrgn,
                                  NULL);
        if (complx == RGN_ERROR)   // check if indicator equals RGN_ERROR
           error_count += CHECKRETURNERROR(complx, "GpiSetClipRegion", SEVERITY_ERROR, PMERR_INV_HPS);
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        complx = GpiIntersectClipRectangle(0, &intersect);
        if (complx == RGN_ERROR)   // check if indicator is RGN_ERROR
          error_count += CHECKRETURNERROR(complx, "GpiIntersectClipRectangle", SEVERITY_ERROR, PMERR_INV_HPS);
        else
          check_rtn((INT) complx, hWndClient, "GpiIntersectClipRectangle");
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_ERROR - Exit\n");
        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiIntersectClipRectangle                                      -    PASS.\n");
        else {
             sprintf(str1_msg, "$ERRORCOUNT = %d.\n", error_count);
             CWRITELOG(L_HDR,0, str1_msg);
             CWRITELOG(L_HDR,0,
"$GpiIntersectClipRectangle                                      -    FAIL.\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ Test GpiIntersectClipRectangle - Exit\n");
  }
}
/***************************************************************************
 *\\ddd
 * Routine Name: GpiOffsetClipRegionExh
 *
 * Purpose: This function exhaustively tests for all complexity indicators
 *      returned by GpiOffsetClipRegion(). The syntax used to call
 *      the gpi is:
 *      (LONG) GpiOffsetClipRegion(
 *             HPS  hPS,
 *             POINTL pptl)
 *
 *      GpiOffsetClipRegion moves the clipping region by the specified
 *      displacement.
 *      The function returns the following complexity indicators:
 *
 *      RGN_NULL      Null region
 *      RGN_RECT      Rectangular region
 *      RGN_COMPLEX   Complex region
 *      RGN_ERROR     Error
 *
 * Revision Log: 11/20/90, PDVT, Gisselle Labault
 *                 - initial creation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), TEST_SIGNATURE(), LOG_SIGNATURE,
 *                   gpi_grid(), CHECKRETURNERROR(),
 *                   check_rtn(), display_text()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   // Test for RGN_NULL
 *   create a null region
 *   clip the region
 *   if region is not null
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   call GpiOffsetClipRegion() with displacement
 *   if GpiOffsetClipRegion doesn't return RGN_NULL
 *     call check_rtn()
 *   else
 *     // this string shouldn't never appear on screen or paper
 *     display string
 *   endif
 *   destroy region
 *
 *   // Test for RGN_RECT
 *   create a rectangular region
 *   clip the region
 *   if region is not rectangular
 *     call check_rtn()
 *   endif
 *   call GpiOffsetClipRegion() with displacement
 *   if GpiOffsetClipRegion doesn't return RGN_RECT
 *     call check_rtn()
 *   else
 *     display text in clipping area
 *   endif
 *   call GpiOffsetClipRegion() with displacement
 *   if GpiOffsetClipRegion doesn't return RGN_RECT
 *     call check_rtn()
 *   else
 *     display text in clipping area
 *   endif
 *   destroy region
 *
 *   // Test for RGN_COMPLEX
 *   create a complex region
 *   clip the region
 *   if region is not complex
 *     call check_rtn()
 *   endif
 *   call GpiOffsetClipRegion() with negative displacement
 *   if GpiOffsetClipRegion doesn't return RGN_COMPLEX
 *     call check_rtn()
 *   else
 *     display text in clipping area
 *   endif
 *   destroy region
 *
 *   // Test for RGN_ERROR
 *   create a rectangular region
 *   clip the region with an invalid presentation space handle
 *   if clipping region is in error
 *     call CHECKRETURNERROR
 *   else
 *     call check_rtn()
 *   endif
 *   call GpiOffsetClipRegion() with negative displacement
 *   if GpiOffsetClipRegion returns RGN_ERROR
 *     call CHECKRETURNERROR
 *   else
 *     call check_rtn()
 *   endif
 *   destroy region
 * }
 *\\end
 ***************************************************************************/
VOID APIENTRY GpiOffsetClipRegionExh(HPS hPS, HWND hWndClient, BOOL SelectionCall)
{
  RECTL    region[3];
  POINTL   offset;
  POINTL   ptl;
  HRGN     hrgn;
  LONG     complx;
  CHAR     str1_msg[200],
           str2_msg[200];

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else
  {
        /* * * * * * * * * * */
        /*  INITIALIZATIONS  */
        /* * * * * * * * * * */
        memset(str1_msg, '\\', sizeof(str1_msg));
        memset(str2_msg, '|', sizeof(str2_msg));

        TEST_SIGNATURE(0, 0);
        LOG_SIGNATURE;
        CWRITELOG(L_LOTRACE,0,"+ee+ GpiOffsetClipRegion - Entry\n");
        gpi_grid(hPS, (LONG)INCH);
        error_count = 0;
        /*******************************/
        /* Test for Indicator RGN_NULL */
        /*******************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_NULL - Entry\n");
        region[0].xLeft = MapX(delta_x);
        region[0].yBottom = MapY(2 * delta_y);
        region[0].xRight = MapX(delta_x);
        region[0].yTop = MapY(4 * delta_y);
        hrgn = GpiCreateRegion(hPS,    // Create region in presentation space
                               0L,      // with zero rectangle; using coordinates
                               region); // from region.

        /***********************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_NULL */
        /***********************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                hrgn,
                                NULL);
        if (complx == RGN_NULL) { //  check if indicator equals RGN_NULL
           /* This string should never appear on the screen */
           MapAndWriteString(hPS, delta_x, 3 * delta_y, "THIS SHOULD NEVER APPEAR ON THE SCREEN OR PAPER !!!");
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_NULL Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        offset.x = delta_x;
        offset.y = -1 *  delta_y;
        MapPt(offset);
        complx = GpiOffsetClipRegion(hPS, &offset);
        if (complx == RGN_NULL) {  // check if indicator equals RGN_NULL
           /* This string should never appear on the screen */
           MapAndWriteString(hPS, delta_x, 2 * delta_y, "THIS SHOULD NEVER APPEAR ON THE SCREEN OR PAPER !!!");
           CWRITELOG(L_LOTRACE,0,
"$GpiOffsetClipRegion RGN_NULL Test                              -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiOffsetClipRegion");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_NULL - Exit\n");

        /*******************************/
        /* Test for Indicator RGN_RECT */
        /*******************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_RECT - Entry\n");
        MapAndWriteString(hPS, 0L, 6 * delta_y - delta_y/5, "  <<< Rectangular Region >>>");
        MapAndWriteString(hPS, 0L, 5 * delta_y + delta_y/2, "Clipping Area will be moved three");
        MapAndWriteString(hPS, 0L, 5 * delta_y + delta_y/4, "times using three different patterns");
        region[0].xLeft = MapX(0);
        region[0].yBottom = MapY(2 * delta_y);
        region[0].xRight = MapX(delta_x);
        region[0].yTop = MapY(4 * delta_y);
        hrgn = GpiCreateRegion(hPS,     // Create region in presentation space
                               1L,      // of one rectangle using coordinates
                               region); // from region.

        /***********************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_RECT */
        /***********************************************************/
        complx = GpiSetClipRegion(hPS, // set region created to a clipping region
                                 hrgn,
                                 NULL);
        if (complx == RGN_RECT){  // check if indicator equals RGN_RECT
           ptl.x = 0;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str1_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_RECT Test                                 -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        offset.x = delta_x;
        offset.y = -1 * delta_y;
        MapPt(offset);
        complx = GpiOffsetClipRegion(hPS, &offset);
        if (complx == RGN_RECT) {   // check if indicator equals RGN_RECT
           ptl.x = 0;
           ptl.y = 4 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str2_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiOffsetClipRegion RGN_RECT Test 1                            -    PASS.\n");
        }
        else  //  FAILURE
          check_rtn((INT) complx, hWndClient, "GpiOffsetClipRegion");

        memset(str1_msg, '+', sizeof(str1_msg));
        offset.x = delta_x;
        offset.y = 2 * delta_y;
        MapPt(offset);
        complx = GpiOffsetClipRegion(hPS, &offset);
        if (complx == RGN_RECT) {   // check if indicator equals RGN_RECT
           ptl.x = delta_x;
           ptl.y = 6 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str1_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiOffsetClipRegion RGN_RECT Test 2                            -    PASS.\n");
        }
        else  //  FAILURE
          check_rtn((INT) complx, hWndClient, "GpiOffsetClipRegion");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_RECT - Exit\n");

        /**********************************/
        /* Test for Indicator RGN_COMPLEX */
        /**********************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_COMPLEX - Entry\n");
        MapAndWriteString(hPS, 5 * delta_x, 6 * delta_y - delta_y/5, "  <<< Complex Region >>>");
        MapAndWriteString(hPS, 5 * delta_x - delta_x/5, 5 * delta_y + delta_y/2, " Clipping Area will be move one");
        MapAndWriteString(hPS, 5 * delta_x - delta_x/5, 5 * delta_y + delta_y/4, " time using two different patterns");

        region[0].xLeft = MapX(5 * delta_x);
        region[0].yBottom = MapY(4 * delta_y);
        region[0].xRight = MapX(6 * delta_x);
        region[0].yTop = MapY(5 * delta_y);
        region[1].xLeft = MapX(6 * delta_x);
        region[1].yBottom = MapY(3 * delta_y);
        region[1].xRight = MapX(7 * delta_x);
        region[1].yTop = MapY(4 * delta_y);
        region[2].xLeft = MapX(6 * delta_x);
        region[2].yBottom = MapY(2 * delta_y);
        region[2].xRight = MapX(7 * delta_x);
        region[2].yTop = MapY(3 * delta_y);

        hrgn = GpiCreateRegion(hPS,      // create region in presentation space
                                3L,      // of three rectangles using coordinates
                                region); // from region.
        /**************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_COMPLEX */
        /**************************************************************/
        complx = GpiSetClipRegion(hPS,  // set region created to a clipping region
                                 hrgn,
                                 NULL);

        if ( complx == RGN_COMPLEX)  {  // check if indicator is RGN_COMPLEX
           ptl.x = 3 * delta_x;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str2_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiSetClipRegion RGN_COMPLEX Test                              -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");

        offset.x = -1 * delta_x;
        offset.y = -1 * delta_y;
        MapPt(offset);
        complx = GpiOffsetClipRegion(hPS, &offset);
        if ( complx == RGN_COMPLEX ) {   // check if indicator is RGN_COMPLEX
           memset(str2_msg, '/', sizeof(str2_msg));
           ptl.x = 3 * delta_x;
           ptl.y = 5 * delta_y;
           display_text(hPS, ptl.x, ptl.y, str2_msg);
           CWRITELOG(L_LOTRACE,0,
"$GpiOffsetClipRegion RGN_COMPLEX Test                           -    PASS.\n");
        }
        else
          check_rtn((INT) complx, hWndClient, "GpiOffsetClipRegion");

        GpiSetClipRegion(hPS, NULLHANDLE, NULL);
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_COMPLEX - Exit\n");

        /********************************/
        /* Test for Indicator RGN_ERROR */
        /********************************/
        CWRITELOG(L_LOTRACE,0,"+ie+ Test Indicator RGN_ERROR - Entry\n");
        hrgn = GpiCreateRegion(hPS,      // create region in presentation space
                               1L,       // of one rectangle using coordinates
                               region); // from region.
        /************************************************************/
        /* Test GpiSetClipRegion for Complexity Indicator RGN_ERROR */
        /************************************************************/
        complx = GpiSetClipRegion(0, // set region created to a clipping region
                                  hrgn,
                                  NULL);
        if (complx == RGN_ERROR)  // check if indicator equals RGN_ERROR
           error_count += CHECKRETURNERROR(complx, "GpiSetClipRegion", SEVERITY_ERROR, PMERR_INV_HPS);
        else
          check_rtn((INT) complx, hWndClient, "GpiSetClipRegion");
        complx = GpiOffsetClipRegion(0, &offset);
        if (complx == RGN_ERROR)   // check if indicator is RGN_ERROR
           error_count += CHECKRETURNERROR(complx, "GpiOffsetClipRegion", SEVERITY_ERROR, PMERR_INV_HPS);
        else
          check_rtn((INT) complx, hWndClient, "GpiOffsetClipRegion");
        GpiDestroyRegion(hPS, hrgn);
        CWRITELOG(L_LOTRACE,0,"+ix+ Test Indicator RGN_ERROR - Exit\n");

        if (!error_count)
           CWRITELOG(L_HDR,0,
"$GpiOffsetClipRegion                                            -    PASS.\n");
        else {
              sprintf(str1_msg, "$ERRORCOUNT = %d.\n", error_count);
              CWRITELOG(L_HDR,0, str1_msg);
              CWRITELOG(L_HDR,0,
"$GpiOffsetClipRegion                                            -    FAIL.\n");
        }
        CWRITELOG(L_LOTRACE,0,"+ex+ Test GpiOffsetClipRegion - Exit\n");
  }
}


/***************************************************************************
 *\\ddd
 * Routine Name: set_clip_path
 *
 * Purpose: This function tests for GpiSetClipPath functionality.
 *
 * Revision Log: 11/09/90, PDVT, Gisselle Labault
 *                 - testcase design
 *               11/10/90, PDVT, Gisselle Labault
 *                 - design implementation
 *
 * Inputs:
 *   hPS            // presentation space
 *   hWndClient     // client window
 *   SelectionCall  // selection parameter
 *
 * Outputs: none
 *
 * Subroutines Required: CWRITELOG(), CHECKRETURNERROR, GpiSetArcParams(),
 *                   GpiFullArc(), display_text(), MapArc(), MapPt()
 *
 * Limitations: none
 *
 *\\end
 ***************************************************************************/
/***************************************************************************
 *\\algorithm
 * {
 *   begin path with GpiBeginPath
 *   set location with GpiMove
 *   call GpiFullArc
 *   end path with GpiEndPath
 *   call GpiSetClipPath(hps,1, SCP_ALTERNATE | SCP_AND)
 *   call CHECKRETURNERROR
 *   if no error
 *     display pattern
 *   else
 *     report difference to CWRITELOG
 *   endif
 *   call GpiSetClipPath(hps,0, SCP_RESET) to stop current clip path
 *   call CHECKRETURNERROR
 * }
 *\\end
 ***************************************************************************/
void set_clip_path(HPS hPS, POINTL ptl, INT size, HWND hWndClient, CHAR msg_circle[])
{
   INT return_code;
   ARCPARAMS arc_parameters;

   /***************************************************/
   /* Define the shape and orientation of the ellipse */
   /***************************************************/
   arc_parameters.lR = 0L;
   arc_parameters.lQ = delta_y/2; //800L;
   arc_parameters.lP = delta_x/2; //600L;
   arc_parameters.lS = 0L;
   MapArc(arc_parameters);
   /**********************************/
   /* set the current arc parameters */
   /**********************************/
   GpiSetArcParams(hPS, &arc_parameters);
   MapPt(ptl);
   GpiBeginPath(hPS, 1L);
   GpiMove(hPS, &ptl);
   GpiFullArc(hPS, DRO_OUTLINE, MAKEFIXED(size, 0));
   GpiEndPath(hPS);

   return_code = GpiSetClipPath(hPS, 1L, SCP_ALTERNATE | SCP_AND);
   // error_count += CHECKRETURNERROR(return_code, "GpiSetClipPath(SCP_ALTERNATE | SCP_AND) PMERR_OK", SEVERITY_ERROR, PMERR_OK);

   if (return_code) {
     ptl.x = 0;
     ptl.y = 7 * delta_y;
     display_text(hPS, ptl.x, ptl.y, msg_circle);
     CWRITELOG(L_LOTRACE,0,
"$GpiSetClipPath(SCP_ALTERNATE | SCP_AND) PMERR_OK Test          -    PASS.\n");
   }
   else {
     error_count++;
     CWRITELOG(L_FAIL,0,
"$Unexpected Error                                               -    FAIL.\n");
   }
   if (return_code = GpiSetClipPath(hPS, 0L, SCP_RESET))
     CWRITELOG(L_LOTRACE,0,
"$GpiSetClipPath(SCP_RESET) PMERR_OK Test                        -    PASS.\n");
   else
     error_count += BADERROR("GpiSetClipPath(SCP_RESET)", FALSE, NULL);

} /* END - SET CLIP PATH TEST */

