#define INCL_GPI
#define INCL_ERRORS    // PM error definitions
#include <string.h>
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include "PTTDLLDF.h"
#include "arc_util.h"
#include "gpiarc.h"
#include "arc_grid.h"

/****************************************************************************
 *\\ddd
 * Subtask Name: PDVT
 *
 * Module name:  GPIARC
 *
 * Purpose:  This module contains all the GPI Arc precision tests.
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - created initial module template
 *
 * SRS References: Section 5.3
 *
 * Contents: gpi_arc_pre1(),
 *           gpi_arc_pre2()
 *
 * Limitations:  NONE.
 *
 *\\end
****************************************************************************/

/****************************************************************************
 *\\ddd
 * Routine Name: gpi_arc_pre1()
 *
 * Purpose: This entry point for PTT causes the precision test to begin.
 *
 * System Requirements: Section 5.3.2
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS                  - presentation space handle
 *         hWndClient           - client window handle
 *         SelectionCall        - selection indicator
 *
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiSetArcParams
 *      GpiFullArc
 *      GpiPartialArc
 *      GpiPointArc
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *  GpiFullArc draws a 1" then 2" circle to appear as two concentric circles
 *  GpiPointArc creates to semi-circles one with 1" chord and one with 2"chord
 *  GpiPartialArc creates to semi-circles 1" then 2" radius
 *\\end
 ***************************************************************************/
VOID APIENTRY gpi_arc_pre1(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL ptl;
  POINTL aptl[6];
  ARCPARAMS arcparam1;
  LONG xpos, ypos;              // current x and y position
  int i;
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"GPIARC Precision Test 1 - Entry.\n");
    lErrorCount = 0L;

    arc_grid (hPS, (LONG)INCH);
    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);


    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetArcParams Test - Entry.\n");

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", TRUE, "GpiArcPre1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetArcParams Test - Exit.\n");

    // GpiFullArc precision
    ptl.x = 2*delta_x;
    ptl.y = 8*delta_y;
    MapAndWriteString1 (hPS, delta_x, ptl.y - 2*delta_y - 2*line_space,
                       "GpiFullArc - 1 & 2");
    MapAndWriteString1 (hPS, delta_x, ptl.y - 2*delta_y - 3*line_space,
                       "inch circles");
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test1 - Entry.\n");

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(1,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiArcPre1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test1 - Exit.\n");


    CWRITELOG(L_LOTRACE,0,"+ie+ GpiFullArc Test2 - Entry.\n");

    if (GpiFullArc (hPS, DRO_OUTLINE, MAKEFIXED(2,0)) == FALSE) {
        lErrorCount += BADERROR("GpiFullArc", TRUE, "GpiArcPre1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiFullArc Test2 - Exit.\n");

    // GpiPartialArc precision

    ptl.x = 3*delta_x;
    ptl.y = 1*delta_y;

    MapAndWriteString1 (hPS, ptl.x + (3*delta_x)/2, ptl.y + (3*delta_y)/2,
                       "GpiPartialArc - 1 & 2");

    MapAndWriteString1 (hPS, 4*delta_x + delta_x/2, (2*delta_y + delta_y/2) - line_space,
                       "inch semi-circles");

    MapPt(ptl);
    GpiMove (hPS, &ptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPartialArc Test1 - Entry.\n");

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(1,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiArcPre1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPartialArc Test1 - Exit.\n");

    ptl.x = 3*delta_x;
    ptl.y = 1*delta_y;
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPartialArc Test2 - Entry.\n");

    if (GpiPartialArc (hPS, &ptl, MAKEFIXED(2,0),
                                  MAKEFIXED(0,0),
                                  MAKEFIXED(90,0)) == FALSE) {
        lErrorCount += BADERROR("GpiPartialArc", TRUE, "GpiArcPre1");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPartialArc Test2 - Exit.\n");

    // GpiPointArc precision
    xpos = 4*delta_x;
    ypos = 4*delta_y;
    MapAndWriteString1 (hPS, xpos, ypos - delta_y/2 -line_space,
                       "GpiPointArc - 1 & 2");

    MapAndWriteString1 (hPS, xpos, ypos - delta_y/2 - 2*line_space,
                       "inch chords");

    for (i = 1; i <= 2; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = ptl.x + i*delta_x;
        aptl[0].y = ptl.y;
        aptl[1].x = ptl.x + i*delta_x;
        aptl[1].y = ptl.y + i*delta_y;
        MapArrayOfPoints (2, aptl);
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPointArc Test1 - Entry.\n");

        if (GpiPointArc (hPS, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPointArc", TRUE, "GpiArcPre1");
            return;
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPointArc Test1 - Exit.\n");
    }
   if(!lErrorCount){
     CWRITELOG(L_HDR,0,
"$GPIARC Precision Test 1                                        -    PASS.\n");
   }
   else{
     sprintf(out_str,"$ERRORCOUNT = %d.\n",lErrorCount);
     CWRITELOG(L_HDR,0,out_str);
     CWRITELOG(L_HDR,0,
"$GPIARC Precision Test 1                                        -    FAIL.\n");
   }

   CWRITELOG(L_LOTRACE,0,"GPIARC Precision Test 1 - Exit.\n");

  }
}

/****************************************************************************
 *\\ddd
 * Routine Name: gpi_arc_pre2()
 *
 * Purpose: This entry point for PTT causes the second precision test to begin.
 *
 * System Requirements: Section 5.3.2
 *
 * Revision Log: 11/08/90, PDVT, Frederik Johannessen
 *                     - created initial detailed design and code
 *               10/24/90, PDVT, Donna Williams
 *                     - initial creation
 *
 * Inputs: hPS                  - presentation space handle
 *         hWndClient           - client window handle
 *         SelectionCall        - selection indicator
 *
 *
 * Outputs:
 *
 * Subroutines Required:
 *
 *      GpiPolyFillet
 *      GpiPolyFilletSharp
 *      GpiPolySpline
 *      BADERROR
 *      CWRITELOG
 *      TEST_SIGNATURE
 *      LOG_SIGNATURE
 *      MapAndWriteString1
 *
 * Limitations:
 *
 *\\end
 ***************************************************************************/

/****************************************************************************
 *\\algorithm
 *  GpiPolyFillet creates concentric 1" and 2" diameter circles
 *  GpiPolyFilletSharp creates consecutive 1" and 2" parabolas
 *  GpiPolySpline joins two small parabolas with two large parabolas
 *\\end
 ***************************************************************************/
VOID APIENTRY gpi_arc_pre2(HPS hPS,HWND hWndClient,BOOL SelectionCall)
{
  POINTL ptl;
  POINTL aptl[12];
  ARCPARAMS arcparam1;
  LONG xpos, ypos;              // current x and y position
  LONG i;
  FIXED afx[1];
  CHAR out_str[80];
  LONG lErrorCount;

  if (SelectionCall) {
    WinSendMsg (hWndClient, WM_PTHREAD_DOES_DEVCALLS, NULL, NULL);
    *pBasisX = 10000;
    *pBasisY = 10000;
  }
  else {
    TEST_SIGNATURE(0,0);
    LOG_SIGNATURE;
    CWRITELOG(L_LOTRACE,0,"GPIARC Precision Test 2 - Entry.\n");
    lErrorCount = 0L;

    arc_grid (hPS, (LONG)INCH);
    arcparam1.lR = 0L;
    arcparam1.lQ = delta_y;
    arcparam1.lP = delta_x;
    arcparam1.lS = 0L;
    MapArc(arcparam1);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiSetArcParams Test - Entry.\n");

    if (GpiSetArcParams (hPS, &arcparam1) == FALSE) {
        lErrorCount += BADERROR("GpiSetArcParams", TRUE, "GpiArcPre2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiSetArcParams Test - Exit.\n");

    // GpiPolyFillet precision
    xpos = 2*delta_x;
    ypos = 6*delta_y;

    MapAndWriteString1 (hPS, delta_x, 6*delta_y - 2*line_space,
                       "GpiPolyFillet - 1 & 2");

    MapAndWriteString1 (hPS, delta_x, 6*delta_y - 3*line_space,
                       "inch circles");

    for (i = 1; i <= 2; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = xpos + (i*delta_x)/2;
        aptl[0].y = ypos;
        aptl[1].x = xpos + (i*delta_x)/2;
        aptl[1].y = ypos + i*delta_y;
        aptl[2].x = xpos - (i*delta_x)/2;
        aptl[2].y = ypos + i*delta_y;
        aptl[3].x = xpos - (i*delta_x)/2;
        aptl[3].y = ypos;
        aptl[4].x = xpos;
        aptl[4].y = ypos;
        MapArrayOfPoints (5, aptl);
        MapPt(ptl);
        GpiMove (hPS, &ptl);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFillet Test - Entry.\n");

        if (GpiPolyFillet (hPS, 5L, aptl) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFillet", TRUE, "GpiArcPre2");
            return;
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFillet Test - Exit.\n");
    }

    // GpiPolyFilletSharp precision
    xpos = 5*delta_x;
    ypos = 6*delta_y;
    MapAndWriteString1 (hPS, xpos, ypos - 2*line_space,
                       "GpiPolyFilletSharp -");
    MapAndWriteString1 (hPS, xpos, ypos - 3*line_space,
                       "1 & 2 inch parabolas");

    for (i = 1; i <= 2; i++) {
        ptl.x = xpos;
        ptl.y = ypos;
        aptl[0].x = xpos + (i*delta_x)/2;
        aptl[0].y = ypos + i*delta_y*2;
        aptl[1].x = xpos + i*delta_x;
        aptl[1].y = ypos;
        MapArrayOfPoints (2, aptl);
        MapPt(ptl);
        GpiMove (hPS, &ptl);
        afx[0] = MAKEFIXED(1,0);

        CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFilletSharp Test1 - Entry.\n");

        if (GpiPolyFilletSharp (hPS, 2L, aptl, afx) == FALSE) {
            lErrorCount += BADERROR("GpiPolyFilletSharp", TRUE, "GpiArcPre2");
            return;
        }

        CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFilletSharp Test1 - Exit.\n");

        if (i == 1) {
            CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolyFilletSharp Test2 - Entry.\n");

            if (GpiPolyFilletSharp (hPS, 2L, aptl, afx) == FALSE) {
                lErrorCount += BADERROR("GpiPolyFilletSharp", TRUE, "GpiArcPre2");
                return;
            }

            CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolyFilletSharp Test2 - Exit.\n");
        }
    }

    // GpiPolySpline precision
    xpos = 2*delta_x;
    ypos = 4*delta_y;

    MapAndWriteString1 (hPS, xpos+delta_x, delta_y + delta_y/2,
                       "GpiPolySpline - 1, 2");

    MapAndWriteString1 (hPS, xpos+delta_x, (delta_y + delta_y/2) - line_space,
                       "& 4 inch parabolas");

    ptl.x = xpos;
    ptl.y = ypos;
    MapPt(ptl);
    GpiMove (hPS, &ptl);

    aptl[0].x = xpos;
    aptl[0].y = ypos + delta_y;
    aptl[1].x = xpos + delta_x;
    aptl[1].y = ypos + delta_y;
    aptl[2].x = xpos + delta_x;
    aptl[2].y = ypos;

    aptl[3].x = xpos + delta_x;
    aptl[3].y = ypos - 2*delta_y;
    aptl[4].x = xpos + 3*delta_x;
    aptl[4].y = ypos - 2*delta_y;
    aptl[5].x = xpos + 3*delta_x;
    aptl[5].y = ypos;

    aptl[6].x = xpos + 3*delta_x;
    aptl[6].y = ypos + delta_y;
    aptl[7].x = xpos + 4*delta_x;
    aptl[7].y = ypos + delta_y;
    aptl[8].x = xpos + 4*delta_x;
    aptl[8].y = ypos;



    MapArrayOfPoints (9L, aptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolySpline Test1 - Entry.\n");

    if (GpiPolySpline (hPS, 9L, aptl) == GPI_ERROR) {
        lErrorCount += BADERROR("GpiPolySpline", TRUE, "GpiArcPre2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolySpline Test1 - Exit.\n");

    for (i = 0L; i < 9L; i++) {
      GpiMarker (hPS, &aptl[i]);
    }


    aptl[0].x = xpos + 4*delta_x;
    aptl[0].y = ypos - 3*delta_y;
    aptl[1].x = xpos;
    aptl[1].y = ypos - 3*delta_y;
    aptl[2].x = xpos;
    aptl[2].y = ypos;
    MapArrayOfPoints (3L, aptl);

    CWRITELOG(L_LOTRACE,0,"+ie+ GpiPolySpline Test2 - Entry.\n");

    if (GpiPolySpline (hPS, 3L, aptl) == GPI_ERROR) {
        lErrorCount += BADERROR("GpiPolySpline", TRUE, "GpiArcPre2");
        return;
    }

    CWRITELOG(L_LOTRACE,0,"+ix+ GpiPolySpline Test2 - Exit.\n");

    for (i = 0L; i < 3L; i++) {
        if (GpiMarker (hPS, &aptl[i]) == GPI_ERROR) {
            lErrorCount += BADERROR("GpiMarker", TRUE, "GpiArcPre2");
            return;
        }
    }

   if(!lErrorCount){
     CWRITELOG(L_HDR,0,
"$GPIARC Precision Test 2                                        -    PASS.\n");
   }
   else{
     sprintf(out_str,"$ERRORCOUNT = %d.\n",lErrorCount);
     CWRITELOG(L_HDR,0,out_str);
     CWRITELOG(L_HDR,0,
"$GPIARC Precision Test 2                                        -    FAIL.\n");
   }

   CWRITELOG(L_LOTRACE,0,"GPIARC Precision Test 2 - Exit.\n");

 }
}

