/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = PPD2BIN.C
 *
 * DESCRIPTIVE NAME = PPD Formatter
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION  Parses a PPD files and puts the information in an
 *              output file
 *
 *
 * FUNCTIONS
 *                 AmbFilename
 *                 atoRound
 *                 RepWarning
 *                 RepError
 *                 SkipNumber
 *                 SkipBlank
 *                 MovePast
 *                 CopyString
 *                 CopyInQuote
 *                 Openppd
 *                 ScanParam
 *                 ScanProcedure
 *                 AddExtension
 *                 FreeAll
 *                 RemoveExtension
 *                 GetArgs
 *                 SaveCommand
 *                 SaveProcedure
 *                 SaveString
 *                 FormPpd
 *                 bld_compare
 *                 CntInpFiles
 *                 WriteRCFile
 *                 main
 *
 *
 * NOTES        Uses the following format for output file:
 *              The output file consists of following segments:
 *
 *              1. Signature
 *              2. Printer directory segment
 *              3. One or more printer information segments
 *
 *              Signature has got the following structure:
 *              Identifying name               40 bytes,
 *              Total no of entries in dir      2 bytes,
 *              Present no of entries in dir    2 bytes,
 *              free                            4 bytes.
 *
 *              Each entry in Printer directory segment has got the
 *              following structure:
 *              Printer name                   40 bytes,
 *              Offset of printer segment       4 bytes,
 *              free                            4 bytes.
 *
 *              Each printer information segment has got the structure
 *              stored in format DESPPD. Towards the end of printer info
 *              segment lies the list buffer of length specified by the
 *              first two bytes of DESPPD structure. The buffer contains
 *              various command string lists and dimension lists.
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#define INCL_GENPLIB_LAYOUT
#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <builtin.h>
#include <genplib.h>
#include "ppd2bin.h"
#include "..\inc\ppdialog.h"
#include <fontres.h>
// @V3.0129238 - Increase the buffer size
// #define  IBUFFSIZE     2048
#define  IBUFFSIZE     (1024 * 35)     // upped from 11 [wgs]
#define  MAX_ESC_STR_LEN 256L          
//@V3.0CMPS01
#define  COMPRESS TRUE
#define  NOCOMPRESS FALSE
//end @V3.0CMPS01

#define  CSTR_NORMAL             0     
#define  CSTR_INCLUDE_SLASH      1     
#define  CSTR_HEX2CHAR           2     
#define  CSTR_EXCLUDE_DQUOTE     4     

/**************************************************************************
* The headers #def INT as int. Just following this, there is a #define int SHORT
* This will cause trouble with using PINTs since it expects a 32 bit value
* to be pointed to. And that definintion will not be affected by the #define int SHORT
* To avoid the problem, I undef INT and the use typedef to reassign it.
* [wgs] 1/17/98
***********************************************************************/
#ifdef INT
#undef INT
typedef int INT;
#endif

#define int SHORT
#define unsigned USHORT
#define MAX_PATH_SIZE 256   //@V3.0MAXPATH

// @V4.CACHE Begin
typedef struct
{
   FILE *fhFile;
   long lFileLen;
   long lCurPos;
   char *pbBuffer;
} FILEBUFFER, *PFILEBUFFER;

FILEBUFFER fbIn;
// @V4.CACHE End

int SkipBlank(char *);

//@V3.0CMPS01
FILE *ppdOut;
BOOL fWriteWords = FALSE;
//end @V3.0CMPS01
long test_prefix = 0x12345678;
//FILE *fhIn = NULL;                   // @V4.CACHE
FILE *fhOut = NULL;
long test_post = 0x12345678;
short iInfiles;                        /* count of input files               */
long ofsOutfile = 0;                   /* to indicate current offset in
                                          output file                        */
char *apszInfiles[MAX_INPUT];          /* input file names pointer list      */
char szOutDir[MAX_PATH_SIZE];          //@V3.0MAXPATH
char szGeneric[MAX_FNAMESIZE];
// char abBuffin[1024 * 4];              /* buffer required for reading input  */
// @V3.0129238 - Increment buffer size
char abBuffin[IBUFFSIZE];             /* buffer required for reading input  */
/* the following appears only to be used in processing the current file and not stored */
/* [wgs] and since the input file is might be over 64k(esp. the hp8000_6.ppd) */
struct
{
  long usFileStart;
  long usFileEnd;
} UIFileGroup;

char acDefFont[] = "Courier";
int cbBuffout = 0;                     /* number of bytes filled in output
                                          buffer                             */
DESPPD desPpd;                         /* printer descriptor segments        */
DRENTRY *pdrTblEntry;
SIGNATURE *psigFile;
char *pbDirBuffer;                     /* pointer to directory buffer tobe
                                          allocated                          */
PBYTE pbItemsBuffer;                   /* pointer to items                   */
// @V3.0129238 - Increment buffer size
// static char pbPrBuffer[1024 * 4];
static char pbPrBuffer[ IBUFFSIZE ];
USHORT USBlockHeader[ 50 ];

// @V3.0129238 - Global variable to tell if current key is a UI
BOOL BUI = FALSE;

#define MAX_FORM_SIZE 127
SHORT sFormCount;
PCH   pFormTable[64];                   /* allow 64 forms */

UINT   nErrorLogLevel=0;

/*
** D74609
*/
#define TRUNCATE_FLOAT   (float) 0.0
#define ROUND_FLOAT      (float) 0.9999

int MovePast( char *, char );
VOID roundImgAreaVals(int *, float, CHAR );  /* changed from an INT [wgs] */

BOOL VerifyPPDLineSize( PCHAR );
BOOL BErrorWasDisplayed = FALSE;

/* @V3.0129238 New functions */
VOID ProcessUIList( VOID );
UINT ProcessUIBlock( PUI_LIST, PUI_BLOCK, UINT, PUSHORT );     //@V4.Media
int CopyString( char *, char *, int, UINT );      //@V4.HexChar

/* @V3.0CMPS01 start
** This is a new section for compression
*/
#include "..\inc\ppdtable.h"
#define HASHSLOTS 251

typedef struct _WORDELEMENT
{
  PSZ pszWord;      //Pointer to keyword
  SHORT sIndex;     //0 based index plus adjustment
  SHORT sList;      //0 based list
  struct _WORDELEMENT *pNext;
} WORDELEMENT, *PWORDELEMENT;

PWORDELEMENT pPSKeyWords;
PWORDELEMENT aHashTable[ HASHSLOTS ];
INT iShrinkage;     //When the ppb is expanded this is the diff between
                    //compressed and uncompressed size

INT MatchKeywords( PBYTE, PUI_LIST, PBYTE, BOOL );
PUI_BLOCK QueryBlockFromKeyword( PUI_LIST, PBYTE, PBYTE, PINT );
PUI_ENTRY QueryEntryFromOption( PBYTE, PUI_BLOCK, PBYTE, PINT );
VOID ProcessCmdsAsUIs( VOID );
INT ProcessUIConstraints( VOID );
INT CopyWord( PCHAR, PCHAR );
VOID VerifyUICList( VOID );
LONG CountFilesFound( VOID );
PCHAR SearchKeySubst( PCHAR );                        // @V4.COLL2FP

BOOL ParsePPDLang( PSZ, PSZ );
FILE *MRILangFile;
CHAR  MRIPathString[ CCHMAXPATH ];

#if 0

long old_errno = 0;
long o_old_errno = 0;

long my_ftell(FILE *the_file)
{
   long  ret_val;
   int   xx;

   if (errno == 29)
      clearerr(the_file);

   if (the_file == NULL)
      {
      xx = 2;
      ret_val = -1;
      }
   else
      {
      xx  = errno;
      if (xx == 8)
         {
         xx = feof(the_file);
         xx = xx  & 0x0000ffff;
         }
      o_old_errno = old_errno;
      if (the_file == fhIn)
         printf("ftell error %d\n", errno);
      if (old_errno && the_file == fhIn)
         ret_val = -1;
      ret_val = ftell(the_file);
      if (ret_val == -1)
         {
         old_errno = errno;
         if (the_file == fhIn && old_errno)
            xx = 1;
         }
      }

   return(ret_val);
} /* my_ftell */
long my_fclose(FILE *the_file)
{
   fclose(the_file);
   if (the_file == fhIn )
      printf("\nfile closed\n");

}
long my_fseek(FILE *the_file, long offset, INT origin)
{
   long  ret;
   INT   err_num;

   if (test_prefix != 0x12345678 || test_post != 0x12345678 )
      {
      printf("corruption error\n\a");
      }
   ret = fseek(the_file, offset, origin);

   if (the_file == fhIn)
      {
      err_num = errno;
      printf("seek to %ld, origin %d bytes from infile, errno = %ld, returns %ld\n", offset, origin, errno, ret);
      if (errno == 29)
         clearerr(fhIn);
      }
   return(ret);
}

long my_fread(void *buff, size_t size, size_t count, FILE *the_file)
{
   long  ret;

   ret = fread(buff, size, count, the_file);
   if (the_file == fhIn)
      {
      old_errno = errno;
      printf("read (%ld, %ld) %ld bytes from infile, errno = %ld\n", (long)size, (long)count, ret, old_errno);

      }
   return(ret);
}

#define ftell(x) my_ftell((x))
#define fclose(x) my_fclose((x))
#define fread(x, a, b, c) my_fread((x), (a), (b), (c))
#define fseek(a, b, c) my_fseek((a), (b), (c))

#endif

// @V4.CACHE Begin

/*****************************************************************************\
**
** FUNCTION = fbseek
**
** DESCRIPTION = Simulates fseek for file buffer
**
\*****************************************************************************/

long fbseek(PFILEBUFFER pfb, long offset, INT origin)
{
   long lCurPos;

   if( !pfb->pbBuffer || !pfb->fhFile )
      return TRUE;
   
   lCurPos = pfb->lCurPos;
   
   switch(origin)
   {
   case SEEK_SET:
      lCurPos = offset;
      break;
   case SEEK_CUR:
      lCurPos += offset;
      break;
   case SEEK_END:
      lCurPos = pfb->lFileLen + offset;
      break;
   default:
      return TRUE;
   }
   if( lCurPos>=0 && lCurPos<=pfb->lFileLen )
   {
      pfb->lCurPos = lCurPos;
      return FALSE;
   }
   else
      return TRUE;
}

/*****************************************************************************\
**
** FUNCTION = fbtell
**
** DESCRIPTION = Simulates ftell for file buffer
**
\*****************************************************************************/

long fbtell(PFILEBUFFER pfb)
{
   if( !pfb->pbBuffer || !pfb->fhFile )
      return -1;
   
   if( pfb->lCurPos>=0 && pfb->lCurPos < pfb->lFileLen )
      return pfb->lCurPos;
   else
      return -1;
}

/*****************************************************************************\
**
** FUNCTION = fbread
**
** DESCRIPTION = Simulates fread for file buffer
**
\*****************************************************************************/

long fbread(void *buff, long size, long count, PFILEBUFFER pfb)
{
   long lRealCount;

   if( !pfb->pbBuffer || !pfb->fhFile )
      return 0;
   if( size<=0 || count<=0)
      return 0;
   
   if( pfb->lCurPos>=0 && pfb->lCurPos < pfb->lFileLen )
   {
      if( pfb->lCurPos+size*count>pfb->lFileLen )
         lRealCount = (pfb->lFileLen - pfb->lCurPos)/size;
      else
         lRealCount = count;
      
      memcpy(buff, pfb->pbBuffer+pfb->lCurPos, lRealCount*size );
      pfb->lCurPos += lRealCount*size;

      return lRealCount;
   }
   else
      return 0;
}

/*****************************************************************************\
**
** FUNCTION = fbclose
**
** DESCRIPTION = Simulates fbclose for file buffer
**
\*****************************************************************************/

long fbclose(PFILEBUFFER pfb)
{
   if( pfb->pbBuffer && pfb->fhFile )
   {
      fclose(pfb->fhFile);
      free(pfb->pbBuffer);
      memset( pfb, 0, sizeof( FILEBUFFER ) );
      
      return 0;
   }
   else
      return -1;
}

// @V4.CACHE End


/*****************************************************************************\
**
** FUNCTION = MakeHashKey
**
** DESCRIPTION = Given an inpust psz string will create a ulong "key"
**               The key is generated viewing the string as seires of ulongs
**               and add them up.
**
\*****************************************************************************/

ULONG MakeHashKey( PSZ pszValue )
{
  int i;
  int iLen;
  int iLongs;
  int iRem;
  ULONG ulKey = 0;
  PULONG pulL;

  iLen = strlen( pszValue );        /* length of input strings   */
  iLongs = iLen / sizeof( ULONG );  /* number of longs in string */
  iRem =   iLen % sizeof( ULONG );  /* bytes left over           */
  pulL = (PULONG)pszValue;

  if ( iLongs )   /* if any longs */
  {
    for ( i = 0; i < iLongs; i ++ )   /* for each long in string */
    {
      ulKey += *pulL;                 /* add together */
      pulL++;
    }
  }

  switch ( iRem )   /* Switch on remainder bytes(s) */
  {                 /* Need to  ignore extra bytes of last word */
    case 0:
      break;
    case 1:
      ulKey += *pulL & 0x000000FF;
      break;
    case 2:
      ulKey += *pulL & 0x0000FFFF;
      break;
    case 3:
      ulKey += *pulL & 0x00FFFFFF;
  }

  return ulKey;

}


/*****************************************************************************\
**
** FUNCTION = AddHashEntry
**
** DESCRIPTION = Adds entry to hash table
**
\*****************************************************************************/

VOID  AddHashEntry( LONG lSlot, PWORDELEMENT pW )
{
  PWORDELEMENT pWOld;

  /*
  ** If no entrys for slot just put in
  */
  if ( aHashTable[ lSlot ] == NULL )
  {
    aHashTable[ lSlot ] = pW;
    return;
  }

  /*
  ** Someone in slot, add to end chain
  */

  pWOld = aHashTable[ lSlot ];

  while ( pWOld->pNext )  /* Loop through to end of chain */
  {
    pWOld = pWOld->pNext;
  }

  pWOld->pNext = pW;

  return;

}


/*****************************************************************************\
**
** FUNCTION = BuildHashTable
**
** DESCRIPTION = Builds an array of pointers one for each word
**
\*****************************************************************************/

VOID BuildHashTable( VOID )
{
  int i, j, k, iCount, iAdjust;
  PWORDELEMENT pW;
  LONG lSlot;

  /* Zero out hash table */
  memset( aHashTable, 0, sizeof( aHashTable ) );

  iCount = 0;
  for ( i = 0; i < PSLISTCOUNT; i++ )
  {
    iCount += sListSize[ i ];
  }

  pPSKeyWords = (PWORDELEMENT)malloc( iCount * sizeof( WORDELEMENT ) );

  pW = pPSKeyWords;

  k = 0;
  iAdjust = 128;
  for ( i = 0; i < PSLISTCOUNT; i++ )   /* For each list */
  {
    for ( j = 0; j < sListSize[ i ]; j++ )  /* For each element in the list */
    {
      pW->pszWord = achPSKeyWords + sPSKeyWordOffset[ k ];
      pW->sList = i;
      pW->pNext = NULL;

      /*
      ** For the first 127 (0-126) add 128 to element ( sets high bit )
      ** for 127 to 254 subtract 126 ( 128 - 254 = 126 )
      ** Cant use 0 or 255 as index since they have special meaning
      */
      pW->sIndex = k + iAdjust;

      lSlot = MakeHashKey( pW->pszWord ) % HASHSLOTS ;

      AddHashEntry( lSlot, pW );

      k++;
      pW++;
    }

    iAdjust -= 254;
  }

/* Debug area ****************/
#if 0
///*
//** Dump out list
//*/
//printf( "KeyWord structures\n" );
//pW = pPSKeyWords;
//for ( i = 0; i < iCount; i++ )
//{
//  printf( "%3.3d, %2.2d, %s\n", pW->sIndex, pW->sList, pW->pszWord );
//  pW++;
//}
//
///*
//**  Dump out hash table
//*/
//printf( "Hash table\n" );
//for ( i = 0; i < HASHSLOTS; i++ )
//{
//  pW =  aHashTable[ i ];
//
//  printf("Slot %4.4d ",i);
//  while( pW )
//  {
//    printf( "%s ",pW->pszWord );
//    pW = pW->pNext;
//  }
//  printf( "\n" );
//}
#endif
}


/*****************************************************************************\
**
** FUNCTION = SearchHashTable
**
** DESCRIPTION = Searches hash table for match
**
\*****************************************************************************/

PWORDELEMENT SearchHashTable( PSZ pszString )
{
  PWORDELEMENT pW;
  ULONG        ulKey;

  ulKey = MakeHashKey( pszString ) % HASHSLOTS; /* Make key from input string */

  pW = aHashTable[ ulKey ];

  while ( pW )  /* Loop thru list */
  {
    if ( strcmp( pszString, pW->pszWord ) == 0 )
    {
      break;    /* If match break */
    }

    pW = pW->pNext;
  }

  return pW;
}

/* @V3.0CMPS01 end */

/*
** D74609
*/
/***************************************************************************
 *
 * FUNCTION NAME = roundImgAreaVals
 *
 * DESCRIPTION   = Used for the *ImageableArea key.  This function either
 *                 rounds the floating point value to the nearest integer, or
 *                 truncates the floating point value to the base integer.
 *                 The new integer value is stored in the output buffer.
 *
 * INPUT         = i - Offset to the input buffer where the floating point
 *                   string is read.
 *                 fIncFrac - Contains one of the following values:
 *                   TRUNCATE_FLOAT - Truncate the floating point value to the
 *                   base integer (i.e.  25.2 -> 25, 14.8 -> 14).
 *                   ROUND_FLOAT - Rounds the floating point value to the next
 *                   higher integer value, providing that there is a fractional
 *                   value (i.e 25.1 -> 26, 79.8 -> 80, 15.0 -> 15, 4.0 -> 4).
 *                   Any other value will yield unpredictable results.
 *                 cMovePastChar - Character value that is passed on to
 *                   MovePast to indicate what character to move past on the
 *                   buffer.  For this function's first call, the argument
 *                   is usually '"' (quote), and most subsequent calls are ' '
 *                   (space).
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = Returns the latest offset (passed in from i above).
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */
VOID roundImgAreaVals(int *i, float fIncFrac, CHAR cMovePastChar )
{
  USHORT j;

  *i += MovePast( pbPrBuffer + *i, cMovePastChar );

  /*
  ** Since spaces are used as terminators in this function, a leading space
  ** may cause the string offset to stop before skipping to the next value.
  ** By removing all leading spaces, this ensures that the string offset moves
  ** to the next value.
  */
  /*
  ** @V3.0100963
  ** Check for all valid spaces, not just for the whitespace.  This is to
  ** prevent a repeat performance in what was happening with MovePast().
  */
  while (isspace( *(pbPrBuffer + *i) ) )
  {
    (*i)++;
  }

  /*
  ** For the imageable area, only the integer value is to be stored.  By
  ** adding a fraction (fIncFrac) to the whole number taken from the PPD, it
  ** either increase the integer value by one, or keep the integer unchanged.
  ** Then the fraction can be discarded (convert it to a USHORT).
  */
  j = (USHORT) (atof( pbPrBuffer + *i ) + fIncFrac);

  memcpy((pbItemsBuffer + cbBuffout), (PCHAR) &j, sizeof( USHORT ) );
  cbBuffout += sizeof( USHORT );
}



/***************************************************************************
 *
 * FUNCTION NAME = atoRound
 *
 * DESCRIPTION   = rounds off a fractional number to nearest integer
 *
 * INPUT         = pchBuffer - pointer to buffer containing number
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int atoRound( char *pchBuffer )
{
  int i;

  i = atoi( pchBuffer );

  while (*pchBuffer >= '0' && *pchBuffer <= '9')
  {
    pchBuffer++;
  }

  if (*pchBuffer == '.')
  {
    pchBuffer++;

    if (*pchBuffer >= '5')
    {
      i++;
    }
  }
  return( i );
}

/***************************************************************************
 *
 * FUNCTION NAME = RepWarning
 *
 * DESCRIPTION   = prints the given message (pszMsg) with the given
 *                 line fragment (pszLine), including the current
 *                 input line number.  If line fragment is NULL, then
 *                 only the message is displayed.
 *
 * INPUT         = err     - error number
 *                 pszLine - optional text
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID RepWarning( ErrType err, char *pszLine )
{
  fprintf( stderr, "%s", szErrmsgs[err] );

  if (pszLine)
  {
    fprintf( stderr, " %s", pszLine );
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = RepError
 *
 * DESCRIPTION   = Same as RepWarning, but terminates as well.
 *
 * INPUT         = err     - error number
 *                 pszLine - optional text
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID RepError( ErrType err, char *pszLine )
{
  RepWarning( err, pszLine );
  exit( 1 );
}

/***************************************************************************
 *
 * FUNCTION NAME = SkipNumber
 *
 * DESCRIPTION   = This routine moves the input buffer pointer forward
 *                 to skip the next number. Returns the number of bytes
 *                 skipped.
 *
 * INPUT         = pszLine - line to skip
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int SkipNumber( char *pszLine )
{
  int i;

  i = SkipBlank( pszLine );

  if (*(pszLine+i) == '+' || *(pszLine+i) == '-' || *(pszLine+i) == '.')
  {
    i++;
  }

  while (*(pszLine+i) >= '0' && *(pszLine+i) <= '9')
  {
    i++;
  }

  if (*(pszLine + i) == '.')
  {
    i++;

    while (*(pszLine+i) >= '0' && *(pszLine+i) <= '9')
    {
      i++;
    }
  }
  return( i );
}

/***************************************************************************
 *
 * FUNCTION NAME = SkipBlank
 *
 * DESCRIPTION   = This routine moves the input buffer pointer forward
 *                 to the next non-white character.  Returns the
 *                 number of bytes skipped.
 *
 * INPUT         = pszLine - line to skip
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int SkipBlank( char *pszLine )
{
  int i;

  i = 0;

  while (*(pszLine+i) && (*(pszLine+i) == ' ' || *(pszLine+i) == '\t'))
  {
    i++;
  }
  return( i );
}

/***************************************************************************
 *
 * FUNCTION NAME = MovePast
 *
 * DESCRIPTION   = This routine returns the offset to skip past the
 *                 first occurrence of character chSkip.
 *
 * INPUT         = pbBuffer - pointer to buffer containing text
 *                 chSkip   - character to skip
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int MovePast( char *pbBuffer, char chSkip )
{
  int i;

  i = 0;

  /*
  ** Check for tab characters (0x09) as well as whitespaces.  Theoretically, it
  ** would be easier to replace the ' ' and 0x09 check with isspace(), but
  ** to make things safer, just add the 0x09 check (with the excpetion of the
  ** HP 4V, it has been working up to now.  We don't want to possibly break
  ** another PPD by replacing with isspace() ).
  */
  /*
  ** We have 10 ppd's where are used TAB (0x09) instead of space
  ** So if MovePast is searching for space, search also for TAB char.
  */
  while ( ( *(pbBuffer+i) != chSkip ) &&
          ( *(pbBuffer+i) != 0x09 || chSkip != ' ' ) &&      
          ( *(pbBuffer+i) >= ' ' || *(pbBuffer+i) == 0x09 )
        )
  {
    i++;
  }
  i++;

  // @V3.OEM There may be more than one blank in the PPD file, so get
  //         past all of them
  while ( ( *(pbBuffer+i) == chSkip) ||
          ( *(pbBuffer+i) == 0x09 && chSkip == ' ' )         
        )
  {
    i++;
  }
  return( i );
}


/*
** This function previously copied the quoted contents to a destination buffer
** and stored the string length in the first byte.  Now, this function does a
** standard NULL-terminated string copy for the contents within the quotes.
*/
/*
** @V3.0129238
** Include the forward slash ('/') as a delimeter if bIncludeSlash is TRUE.
** If bIncludeSlash is FALSE, do not include the forward slash as a
** delimiter.
*/
/***************************************************************************
 *
 * FUNCTION NAME = CopyString
 *
 * DESCRIPTION   = This routine copies a string from source to
 *                 destination with delimiters as a double Quote,
 *                 blank, colon, or forward slash (optional).  The string
 *                 is NULL terminated.
 *
 * INPUT         = szDestn  - destination string
 *                 szSource - source string
 *                 iMaxLen  - length of destination string
 *                 uiOpt    - Options for copying           //@V4.HexChar
 *                             CSTR_NORMAL
 *                             CSTR_INCLUDE_SLASH - Include the forward slash ('/')
 *                                                  as a delimiter.
 *                             CSTR_HEX2CHAR      - Convert hex <e1> strings to chars.
 *                             CSTR_EXCLUDE_DQUOTE- Exclude the double quote (")
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = Number of characters in the copied string.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int CopyString(char *szDestn, char *szSource, int iMaxlen, UINT uiOpt)
{
  int   i = 0;
  char  cHex;
  int   iInHex = 0;


  iMaxlen--;  /* maxlen is actually 1 based, i is a 0 based index, if you hit the max
               * then add a NULL you may have a memory violation, so adjust by 1 [wgs] */

  if (!(*szSource))       /* if the string is empty, do nothing */
     return(0);

  /* @V4.HexChar Begin
  ** OldCode
  ** while (*szSource && *szSource != '"' && (*szSource >= ' ' || *szSource < 0) &&
  **        *szSource != ':' && ((bIncludeSlash == TRUE && *szSource != '/') ||
  **        bIncludeSlash != TRUE) && i < iMaxlen)
  ** {
  **   *(szDestn + i++) = *(szSource++);
  ** }
  **
  ** New Code translates also hex strings <e1> to chars 
  ** if CSTR_HEX2CHAR is set.
  */
  
  while ( *szSource &&
          (*szSource != '"' || (uiOpt & CSTR_EXCLUDE_DQUOTE) ) &&
          (*szSource >= ' ' || *szSource < 0) &&
          *szSource != ':' &&
          ( ( (uiOpt & CSTR_INCLUDE_SLASH) && *szSource != '/') ||
            !(uiOpt & CSTR_INCLUDE_SLASH) )
          && i < iMaxlen)
  {
    if ( uiOpt & CSTR_HEX2CHAR )
    {
      if( !iInHex )                              //Not in Hex string
      {
        //search for Hex strings
        if( *szSource == '<' )                   //Begining of Hex char
        {
          cHex = 0;
          iInHex = 1;
        }
        else
          *(szDestn + i++) = *szSource;
      }
      else                                       //In Hex string
      {
        if( iInHex <= 3 && *szSource == '>' )    //End of Hex string
        {
          *(szDestn + i++) = cHex;
          iInHex = 0;
        }
        else                                     //between < >
        {
          if (*szSource >= '0' && *szSource <= '9' && iInHex < 3 )
            cHex = (cHex << 4 ) | (*szSource - '0');
          else if (*szSource >= 'a' && *szSource <= 'f' && iInHex < 3 )
            cHex = (cHex << 4 ) | (*szSource - 'a' + 0xA );
          else if (*szSource >= 'A' && *szSource <= 'F' && iInHex < 3 )
            cHex = (cHex << 4 ) | (*szSource - 'A' + 0xA );
          else                                  
          { 
            // Bad Hex String
            // return to next of '<'
            szSource -= iInHex;
            *(szDestn + i++) = *szSource ;
            iInHex = -1;                        //Next line set iInHex=0
          }
          iInHex++;
        }
      }
    }
    else
      *(szDestn + i++) = *szSource;
      
    szSource++;
  }
  // @V4.HexChar End

  *(szDestn + i++) = 0;

  return( i );
} /* CopyString */


/*
** This function previously copied the quoted contents to a destination buffer
** and stored the string length in the first byte.  Now, this function does a
** standard NULL-terminated string copy for the contents within the quotes.
*/
/***************************************************************************
 *
 * FUNCTION NAME = CopyInQuote
 *
 * DESCRIPTION   = This routine copies a NULL-terminated string from source to
 *                 destination with delimiters as a double Quote.  This
 *                 function returns the string length, in bytes, including
 *                 the terminator.
 *
 *                 If fRemoveDots is TRUE the dots in the target
 *                 string will be replaced with underscores.  This is
 *                 to get around a in the IBM spooler which chokes on
 *                 more than one dot in a printer model name.
 *
 * INPUT         = szDestn     - destination string
 *                 szSource    - source string
 *                 fRemoveDots - flag
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int CopyInQuote(char *szDestn,char *szSource,BOOL fRemoveDots, BOOL CompressFlag)
{
  /*@V3.0CMPS01 start */
  int   j, k;
//ULONG ulKey; NOTUSED
  PWORDELEMENT pW;
  PSZ pszP;
  /*@V3.0CMPS01 end   */
  register int i = 0;
  BOOL fInHexString = FALSE;

  /*
  ** Remove MAX_ESC_STR_LEN.  Some PPD commands may exceed 256 characters.
  */

  if (!(*szSource))       /* if the string is empty, do nothing */
     return(i);

  /* Remove excess leading spaces */
  while (*szSource && isspace( *szSource ))
  {
    szSource++;
  }

  /*
  ** Change any dots to underscores if needed
  */
  if ( fRemoveDots )
  {
    pszP = szSource;

    while ( *pszP != '"' )
    {
      if ( *pszP == '.' )
      {
        *pszP = '_';
      }

      pszP++;
    }
  }


  while (*szSource && *szSource != '"')   /* make sure we check the null too. [wgs] */
  {
    if ( isalnum( *szSource ) )
    {
      /* copy ANs to buffer */
      j = 0;
      /*
      ** A word is alphanumeric.  Dots are allowed as long as in
      ** the middle
      */
      while ( isalnum( *szSource )                             ||
              ((*szSource == '.') && isalnum( *(szSource+1) ) ) )
      {
        *(szDestn + i + j++ ) = *(szSource++);
      }
      if ( j > 1 )  /* a multi char word */
      {
        if (CompressFlag == TRUE)
        {
          *(szDestn + i + j ) = '\0';
          if ( fWriteWords )    /* Write out words if needed */
          {
            fprintf (ppdOut, "%s\n",szDestn+i);
          }

          if ( ( pW = SearchHashTable( (PSZ) szDestn+i ) ) != 0 )  //Look in hash table
          {
            /*
            ** For each list we pass write a 0xff out - first list no FFs
            ** secon write one FF, third write two ...
            */
            for ( k = 0; k < pW->sList; k++ )
            {
              *(szDestn + i++) = '\xFF';
            }
            *(szDestn + i++) = pW->sIndex;
          }
          else
          {
            i += j; /* move i by word */
          }
        }
        else
        {
          i += j; /* move i by word */
        }
      }
      else
      {
        i += j; /* move i by word */
      }
    }

    else
    if (isspace( *szSource ))
    {
      /*
      ** If space use one space char and skip rest
      --
      ** Keep it a space Not tab CR LF etc...
      */
      BOOL fFoundLF = FALSE;

      if ( *szSource != '\n' )
      {
        *(szDestn + i++) = ' ';
      }
      else
      {
        *(szDestn + i++) = '\n';
        fFoundLF = TRUE;
      }

      szSource++;

      /* Remove excess any spaces */
      while ( isspace( *szSource )                     &&
              ( *szSource != '\n' || fFoundLF == TRUE ) )
      {
        szSource++;
      }
    }

    else
    {
      // @V4.1200417
      // This is either a hex string or dict entry
      if ( *szSource == '<' )
      {
        // It's a Dict entry
        if ( *(szSource+1) == '<' )
        {
          *(szDestn + i++) = *(szSource++);
        }
        else
        {
          // Do not compress hex strings
          // If compression is already off then nothing really needs to change
          fInHexString = CompressFlag;
          CompressFlag = FALSE;
        }
      }
      else
      if ( *szSource == '>'    &&
           fInHexString == TRUE )
      {
        fInHexString = FALSE;
        CompressFlag = TRUE;
      }

      *(szDestn + i++) = *(szSource++);
    }
  }

  /*
  ** NULL-terminate the destination string.
  */

  *(szDestn + i++) = 0;

  return( i );
} /* CopyInQuote */

/***************************************************************************
 *
 * FUNCTION NAME = Openppd
 *
 * DESCRIPTION   = Opens the file specified in read only text mode and
 *                 returns True if open successful ,else FALSE.  The
 *                 file handle is stored in fhIn
 *
 * INPUT         = szFileName - filename
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

BOOL Openppd( char *szFilename )
{
//long i; NOTUSED
  //fhIn = fopen( szFilename, "rb" );               // @V4.CACHE

  fbIn.fhFile = fopen( szFilename, "rb" );          // @V4.CACHE

  //if (!fhIn)                                      // @V4.CACHE
  if (!fbIn.fhFile)                                 // @V4.CACHE
  {
    /*
    ** @V3.1141419
    ** Previously, if a file couldn't be found, then the compiler would stop.
    ** However, confidential OEM PPD files will not be out on the DDK.
    ** Therefore, change to simply ignore files that aren't found.  However,
    ** do display it for build purposes.
    */
//    RepWarning( err_cantopen, szFilename );
    printf( "%s INFO - File not found.  Ignore and continue.\n",
            szFilename );
    *szFilename = 0;
    return( FALSE );
  }

  // @V4.CACHE Begin
  fseek( fbIn.fhFile, 0L, SEEK_END);
  fbIn.lFileLen = ftell( fbIn.fhFile );
  fseek( fbIn.fhFile, 0L, SEEK_SET);

  fbIn.pbBuffer = (char *)malloc( fbIn.lFileLen * sizeof( char ) );
  fread( fbIn.pbBuffer, sizeof( char ), fbIn.lFileLen, fbIn.fhFile );
  // @V4.CACHE End

  fbseek(&fbIn, 0L, SEEK_SET);                      // @V4.CACHE
  printf( "Converting  %s\n", szFilename );
  return( TRUE );
}

/***************************************************************************
 *
 * FUNCTION NAME = ScanParam
 *
 * DESCRIPTION   = This routine scans the szSrchstring in the input
 *                 file and provides the remaining parameter in the
 *                 return buffer provided.  If szSrchstring is found
 *                 this routine returns TRUE else FALSE.
 *
 * INPUT         = szSrchstring - search string address
 *
 * OUTPUT        = pbBufaddress - return buffer address
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

BOOL ScanParam( char *szSrchstring, char *pbBufreturn )
{
  BOOL fIseof, fIsFound;
  /* @V3.0129238 */
  INT  j = 0, k;
  INT  i;
  long li, lc;
  UINT uiStrLen = strlen( szSrchstring );
  // @V3.0129238
  long   uiCurrFileLoc;
  INT x;

  fIseof = FALSE;

  /*
  ** search for parameter till token found or file eof encountered
  */
  while (!fIseof)
  {
    if ((lc = fbtell(&fbIn)) == -1L)
    {
      return( FALSE );
    }

    /*
    ** @V3.0129238 - For UI groups, query the current file location
    **
    */
    uiCurrFileLoc = fbtell( &fbIn );

    if ((i = fbread(abBuffin,1,IBUFFSIZE,&fbIn)) != IBUFFSIZE)
    {
      fIseof = TRUE;
    }

    /*
    ** Verify that the line size does not exceed IBUFFSIZE bytes.
    ** Report an error and skip the command if this exists.
    */
    if (VerifyPPDLineSize( abBuffin ) == TRUE)
    {
      if (i <= 1)
      {
        break;
      }

      /*
      ** Ignore the last partially read line
      */
      if (!fIseof)         /* didn't hit end of file, then don't need do adjust */
         {                 /* ignore the partially read line */
         if (i > 3)
            i--;
         for (li = 1;i;li++, i--)
            {
            if ((abBuffin[i] == '*') && (abBuffin[i - 1] == '\n'))
               {
               break;
               }
            } /* for */

         /*
         ** shift back the current access pointer of the file
         ** make it point to the beginning of the partially read line
         */
         if (li > 1 && li < IBUFFSIZE)
            {
            if (fbseek( &fbIn, -(li-1), SEEK_CUR) == -1L)      /* oops, something       */
               {
               return( FALSE );
               }
            } /* check for reasonable backup values */
         } /* if (!fIseof) */

      j = 0;
      fIsFound = FALSE;
      while (j < i)
      {
        if (!strncmp( abBuffin+j, szSrchstring, uiStrLen ))
        {
          /* @COOK */
          if (isspace( *(abBuffin + j + uiStrLen) ) ||
              *(abBuffin + j + uiStrLen) == ':')
          {
            fIsFound = TRUE;
          }
        }
        k = j;

        for (;;j++)
        {
          /*
          ** Check to see if a comment is on the next line.  This will
          ** prevent a trap if the compiler finds a comment immediately
          ** following a command line.  The compiler won't read the
          ** comment and possibly exceed buffer space.
          */
          if ((abBuffin[j] == 0xA) && (abBuffin[j+1] == 0x2A ||
               abBuffin[j+1] == '%'))
          {
            j++;
            break;
          }

          if  (j == i)
          {
            break;
          }
        }

        if (fIsFound)
        {
          /*
          ** @V3.0129238 - If current key is a UI...
          */
          if (BUI == TRUE)
          {
            fbseek( &fbIn, uiCurrFileLoc + k, SEEK_SET );
            lc = fbtell( &fbIn );

            /*
            ** @V3.1150229
            ** Change '1024 * 4' to IBUFFSIZE.
            */
//            fread( abBuffin, 1, 1024 * 4, fhIn );
            fbread( abBuffin, 1, IBUFFSIZE, &fbIn );
            j = MovePast( abBuffin, ' ' );
            memset( pbBufreturn, 0, IBUFFSIZE );
            memcpy( pbBufreturn, abBuffin + j, IBUFFSIZE - j );
            fbseek( &fbIn, lc + 10, SEEK_SET );
          }
          else
          {
            k += uiStrLen;
            k += SkipBlank( abBuffin + k );

            if (j >= k)
            {
              x = j - k;
            }
            else
            {
              x = 0;
              while (abBuffin[ k + x ] != 0x0A)
              {
                x++;
              }
            }

            memcpy( pbBufreturn, abBuffin + k, x );
          }
          break;
        }
      }

      /*
      ** set the access pointer so that in next access the current
      ** token is not reencountered
      */
      if (fIsFound)
      {
        fbseek( &fbIn, lc+(long)j, SEEK_SET );
        break;
      }
    }
  }
  return( fIsFound );
} /* ScanParam */

/***************************************************************************
 *
 * FUNCTION NAME = ScanProcedure
 *
 * DESCRIPTION   = This routine scans the szSrchstring in the input
 *                 file and provides the remaining parameter in the
 *                 return buffer provided.  If szSrchstring is found
 *                 this routine returns TRUE else FALSE.
 *
 * INPUT         = szSrchstring - search string address
 *
 * OUTPUT        = pbBufaddress - return buffer address
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

BOOL ScanProcedure( char *szSrchstring,char *pbBufreturn )
{
  BOOL fIseof, fIsFound;
  INT  i, j, k, iStrlen;
  long li;

  fIsFound = FALSE;               
  fIseof = FALSE;

  /*
  ** search for parameter till token found or file eof encountered
  */
  while (!fIseof)
  {
    if (fbseek( &fbIn, 0L, SEEK_CUR ) == -1L)
    {
      return (FALSE);
    }

    /*
    ** @V3.1150229
    ** Change '1024 * 4' to IBUFFSIZE.
    */
    if ((i = fbread( abBuffin, 1, IBUFFSIZE, &fbIn )) != IBUFFSIZE)
    {
      fIseof = TRUE;
    }

    if (i <= 0)
    {
      break;
    }

    /* Ignore the last partially read line */
    for (li = 1;i;li++, i--)
    {
      if ((abBuffin[IBUFFSIZE - li] == 0x2A) &&
          (abBuffin[IBUFFSIZE - li - 1] == 0xA))
      {
        break;
      }
    }

    i--;

    /* shift back the current access pointer of the file */
    if (li > 1 && li < IBUFFSIZE)
    {
      if (fbseek( &fbIn, -li, SEEK_CUR ) == -1L)
      {
        return (FALSE);
      }
    }

    j = 0;
    fIsFound = FALSE;
    iStrlen = 0;

    while (*(szSrchstring + iStrlen) != '\0')
    {
      iStrlen++;
    }

    while (j < i)
    {
      if (!strncmp(abBuffin+j, szSrchstring, iStrlen))
      {
        if (*(abBuffin+j+iStrlen) == ' ' || *(abBuffin+j+iStrlen) == ':')
        {
          fIsFound = TRUE;
        }
      }
      k = j;

      for (;;j++)
      {
        if ((abBuffin[j] == 0xA) && (abBuffin[j+1] == 0x2A))
        {
          j++;
          break;
        }
        if  (j == i)
        {
          break;
        }
      }

      if (fIsFound)
      {
        /*
        ** now reset the file to start of where the token has been
        ** encountered and read a large block so that entire procedure
        ** might be covered.
        */
        if (fbseek( &fbIn, (long) - (i - k), SEEK_CUR) == -1L)
        {
          return( FALSE );
        }

        if ((i = fbread( abBuffin, 1, IBUFFSIZE, &fbIn)) <= 0)
        {
          return( FALSE );
        }
        k = iStrlen;
        k += MovePast( abBuffin+k, '"' );
        j = 0;

        while (abBuffin[k] != '"' && k < i)
        {
          if (abBuffin[k] != '\r')
          {
            *(pbBufreturn + 2 + j++) = abBuffin[k++];
          }
          else
          {
            k++;
          }
        }
        *(pbBufreturn+2+j) = '\0';
        *(int *)pbBufreturn = (1+j);
        return (TRUE);
      }
    }
  }
  return (fIsFound);
}

/***************************************************************************
 *
 * FUNCTION NAME = AddExtension
 *
 * DESCRIPTION   = If given filename has no extension, add the given default
 *
 * INPUT         = pszName   - filename
 *                 pszDefext - default extension
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID AddExtension( char *pszName, char *pszDefext )
{
  char           c;
  int            l;
  register char *p;

  l = strlen( pszName );
  p = pszName + l;
  c = '.';

  while (l-- > 0)
  {
    c = *--p;

    if (c == '.' || c == '\\' || c == ':')
    {
      break;
    }
  }

  if (c != '.')
  {
    strcat( p, pszDefext );
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = FreeAll
 *
 * DESCRIPTION   = To free all the memory allocated from heap.
 *
 * INPUT         = NONE.
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

void FreeAll(void)
{
  int i;

  for (i = 0; i < iInfiles; i++)
  {
    if (apszInfiles[i])
       free( apszInfiles[i] );
  }
  if (pbDirBuffer)
     free( pbDirBuffer );
} /* FreeAll */

/***************************************************************************
 *
 * FUNCTION NAME = RemoveExtension
 *
 * DESCRIPTION   = Removes any extension from the given filename.
 *
 * INPUT         = pszName - file name
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID RemoveExtension( char *pszName )
{
  char           c;
  int            l;
  register char *p;

  l = strlen( pszName );
  p = pszName+l;
  c = '.';

  while (l-- > 0)
  {
    c = *--p;

    if (c == '.' || c == '\\' || c == ':')
    {
      break;
    }
  }

  if (c == '.')
  {
    *p = 0;
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = GetArgs
 *
 * DESCRIPTION   = Parses all args passed to the program.
 *
 * INPUT         = argc - number of commandline arguments
 *                 argv - pointer to array of commandline arguments
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID GetArgs( int argc, char **argv )
{
  register char *p;
  int i;

  RepWarning( err_ver, NULL );

  /*
  ** To check the range of arguments
  */
  if (argc <= 4)
  {
    RepError( err_usage, NULL );
  }
  iInfiles = 0;
  p = *++argv;

  /*
  ** pick up the names of input files
  */

  if (*p == '-' && tolower(*(p + 1)) == 'i')
  {
    p = *++argv;

    while (*p != '-' && iInfiles < MAX_INPUT && --argc > 0)
    {
      apszInfiles[iInfiles] = malloc( MAX_PATH_SIZE );    ///@V3.0MAXPATH

      if (apszInfiles[iInfiles] == NULL)
      {
        RepError( err_arg, NULL );
      }
      strcpy( apszInfiles[iInfiles], p );
      p = *++argv;
      iInfiles++;
    }
  }
  else
  {
    RepError( err_usage, NULL );
  }

  if (*p == '-' && tolower(*(p+1)) == 'o' && argc > 0)
  {
    p = *++argv;
    strcpy( szOutDir, p );

    p = *++argv;
  }
  else
  {
    RepError( err_usage, NULL );
  }

  if (*p == '-' && tolower(*(p+1)) == 'g' && argc > 0)
  {
    p = *++argv;
    strcpy( szGeneric, p );
    AddExtension( szGeneric, DEF_IEXT );
    p = *++argv;  /*@V3.0CMPS01 */
  }
  else
  {
    RepError( err_usage, NULL );
  }

  MRIPathString[ 0 ] = 0;
  if (p != NULL)
  {
    if (argc > 0 && *p == '-' && tolower( *(p + 1) ) == 'p')
    {
      strcpy( MRIPathString, p + 2 );
      i = strlen( MRIPathString );
      if (MRIPathString[ i - 1 ] == '\\')
      {
        MRIPathString[ i - 1 ] = 0;
      }
      strcat( MRIPathString, "\\mrilang.lst" );
    }
    p = *++argv;  /*@V3.0CMPS01 */
  }

  /* -e# Error loging */
  if ( p && *p == '-' && tolower(*(p+1)) == 'e' && argc > 0)
  {
    if( *(p+2)>='0' && *(p+2)<='9' )
      nErrorLogLevel = *(p+2) - '0';
    else
      nErrorLogLevel = 0;

    p = *++argv;
  }

  /*@V3.0CMPS01 */
  /* Write out Words   This is last parm */
  if ( p && *p == '-' && tolower(*(p+1)) == 'w' && argc > 0)
  {
    argv += 2;
    p = *argv;
    fWriteWords = TRUE;
  }

  for (i = 0; i < iInfiles; i++)
  {
    strlwr( apszInfiles[i] );
  }
  strlwr( szOutDir );
  strlwr( szGeneric );
}

/***************************************************************************
 *
 * FUNCTION NAME = SaveCommand
 *
 * DESCRIPTION   = This routine scans the parameter buffer and saves
 *                 the command within quotes in pbItemsBuffer .This
 *                 routine is written to save code.
 *
 * INPUT         = szString - pointer to string to be searched
 *
 * OUTPUT        = piOffset - pointer to variable where offset is to be
 *                            stored
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID SaveCommand(char *szString,short *piOffset,BOOL CompressFlag)
{
  int  i, j;
  BOOL fPrnName;

  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam(szString, pbPrBuffer))
  {
    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, '"' );
    fPrnName =  (strcmp( szString, "*NickName" ) == 0) |
                (strcmp( szString, "*ShortNickName" ) == 0);
    *piOffset = cbBuffout;

    /* The data within quotes is NULL */
    if (pbPrBuffer[i-1] == '"' && pbPrBuffer[i-2] == '"')
    {
      *piOffset = 0;
      return;
    }

    /*
    ** copies the string delimited by quote with ist byte as length.
    */
    j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, fPrnName,
                     CompressFlag);
    cbBuffout += j;
  }
  else
  {
    /*
    ** Null pointer
    */
    *piOffset = -1;
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = SaveProcedure
 *
 * DESCRIPTION   = This routine scans the parameter buffer and saves
 *                 the procedure within quotes in pbItemsBuffer .This
 *                 routine is written to save code.
 *
 * INPUT         = szString - pointer to string to be searched.
 *
 * OUTPUT        = piOffset - pointer to variable where offset is to be
 *                            stored.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID SaveProcedure( char *szString, short *piOffset )
{
  int j;

  fbseek( &fbIn, (long) 0, SEEK_SET );

  if (ScanProcedure( szString, pbItemsBuffer + cbBuffout ))
  {
    *piOffset = cbBuffout;
    j = *(int *)(pbItemsBuffer + cbBuffout );

    /*
    ** 2 bytes extra to take care of length
    */
    cbBuffout += ( j + 2 );
  }
  else
  {
    /*
    ** Null pointer
    */
    *piOffset = -1;
  }
}

/***************************************************************************
 *
 * FUNCTION NAME = SaveString
 *
 * DESCRIPTION   = This routine scans the parameter buffer and saves
 *                 the keyword following colon in pbItemsBuffer . This
 *                 routine is written to save code/
 *
 * INPUT         = szString - pointer to string to be searched.
 *
 * OUTPUT        = piOffset - pointer to variable where offset is to be
 *                            stored.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

VOID SaveString( char *szString, short *piOffset )
{
  int i, j;

  fbseek( &fbIn, (long) 0, SEEK_SET );

  if (ScanParam( szString, pbPrBuffer ))
  {
    /*
    ** skip the " character
    */

    i = MovePast( pbPrBuffer, ':' );
    *piOffset = cbBuffout;
    i += SkipBlank(pbPrBuffer+i );

    /*
    ** copies the string delimited by a blank with ist byte as length.
    ** Strings are not null terminated
    */
    if (strncmp( pbPrBuffer + i, "Unknown", 7 ))
    {
      // @V3.0129238
      j = CopyString( pbItemsBuffer + cbBuffout, 
                      pbPrBuffer + i, 
                      80, 
                      CSTR_NORMAL );    // @V4.HexChar
      cbBuffout += j;
    }
    else
    {
      *piOffset = -1;
    }
  }
  else
  {
    /*
    ** Null pointer
    */
    *piOffset = -1;
  }
}

/*****************************************************************************\
**
** FUNCTION NameToIndex
**
** Will look up form in table and put index in PPB.  Adds form if not there
**
\*****************************************************************************/

SHORT NameToIndex( VOID )
{
  SHORT i;
  SHORT sNameLen;
  CHAR  acFormName[MAX_FORM_SIZE+1];       /* Buffer for found form name */
  PCH   pFormName;

  /*
  */
  // @V3.0129238
  sNameLen = CopyString( acFormName, 
                         pbPrBuffer, 
                         MAX_FORM_SIZE, 
                         CSTR_NORMAL );       // @V4.HexChar
  acFormName[MAX_PSIZE-1] = '\0';   /* For now limit to 64 char */
  pFormName = acFormName;


  /* find form in table */

  for ( i = 0; i < sFormCount; i++ )
  {
    if ( !strcmp( pFormName, pFormTable[i] ) )
      break ;  /* Found - stop loop */
  }

  /* Warn if about to overflow table */
  if ( sFormCount == 63 )
  {
    printf( "ERROR ERROR Too many form names\n" );
    i = 0;  /* set to first form */
  }

  if ( i >= sFormCount )
  { /* Couldn't find it so add to table */
    pFormTable[ sFormCount ] = (PCH)malloc( strlen( pFormName ) + 1 );
    strcpy( pFormTable[ sFormCount ], pFormName );
    i = sFormCount++;
  }

  *(PSHORT)(pbItemsBuffer+cbBuffout) = i;

  cbBuffout += sizeof(SHORT);

  return sNameLen;
}

/*****************************************************************************\
**
** FUNCTION ProcessFormTable
**
** Writes out the form table and index
**
\*****************************************************************************/

VOID ProcessFormTable( VOID )
{
  SHORT i;
  LONG  lIndexTable[64];
//PUI_BLOCK pBlock;NOTUSED
//INT iNumOfPages; NOTUSED
//INT iNumOfPageRegions; NOTUSED

#if 0
///*
//** Do a sanity check
//*/
//if ( desPpd.desPage.iCmpgpairs  != sFormCount  ||
//     desPpd.desPage.iDmpgpairs  != sFormCount  ||
//     desPpd.desPage.iImgpgpairs != sFormCount   )
//{
//  printf( "WARNING - Mismatched forms counts: PageRegion %d, PaperDim %d, "
//          "ImageableArea %d, Total Count %d\n", desPpd.desPage.iCmpgpairs,
//           desPpd.desPage.iDmpgpairs, desPpd.desPage.iImgpgpairs, sFormCount );
//}
#endif

  desPpd.desForms.usFormCount  = sFormCount;  /* store form count */
  desPpd.desForms.ofsFormTable = cbBuffout;   /* store start of form table */

  /* Write out the table - it's regular null term strings */
  for ( i = 0; i < sFormCount; i++ )
  {
    PCHAR p;
    INT j;
    INT iLen;

    lIndexTable[i] = cbBuffout;     /* Keep track of where each form goes */
//  strcpy( pbItemsBuffer + cbBuffout, pFormTable[ i ] );
//@V3.0CMPS01
    p = pFormTable[i];
    iLen = strlen( p );
    *(p + iLen) = '"';    //Replace null term with quote
    j = CopyInQuote( pbItemsBuffer + cbBuffout, p, FALSE, COMPRESS);
    iShrinkage += ( iLen + 1 ) - j;   //J includes null term


    free( pFormTable[ i ] );
//  cbBuffout += strlen( pFormTable[ i ] ) + 1;
//@V3.0CMPS01
    cbBuffout += j;
  }

  desPpd.desForms.ofsFormIndex = cbBuffout;   /* store start of index table */

  for ( i = 0; i < sFormCount; i++ )
  {
    *(PLONG)(pbItemsBuffer+cbBuffout) = lIndexTable[ i ];
    cbBuffout += sizeof(LONG);
  }

}

/*****************************************************************************\
**
** FUNCTION DoDefaultForm
**
** Converts the default string found by the old method to the index of a form.
** This means that desPpd.desPage.ofsDfpgsz is NOT an offset but an index value
**
\*****************************************************************************/

VOID DoDefaultForm( VOID )
{
  PBYTE ptr;
  PBYTE pImageableArea;
  SHORT sLen;
  SHORT i;
  SHORT j;
  PCH   pchSlash;
  SHORT sFirstForm;


  /* if value -1 no default form so use the first form */
  if ( desPpd.desPage.ofsDfpgsz == -1 )
  {
    desPpd.desPage.ofsDfpgsz = 0;
    printf( "INFO: No default form\n" );
    return;
  }

  ptr = pbItemsBuffer + desPpd.desPage.ofsDfpgsz;
  sLen = strlen( ptr );

  /* Point to imageable area */
  pImageableArea = pbItemsBuffer + desPpd.desPage.ofsImgblPgsz;
  sFirstForm = *((PSHORT)pImageableArea);

  /*
  */
#if 0
//  sLen = (SHORT)*ptr ;
//  ptr++;              /* point to name */
//  *(ptr+sLen) = 0;    /* NULL terminate */
#endif

  for ( j = 0; j < desPpd.desPage.iImgpgpairs; j++ )
  {

    i = *((PSHORT)pImageableArea);
    pImageableArea +=  sizeof( SHORT ) * 5;

    /*
    ** If Xlate string, block it by temp replacing slash with zero
    */
    if ( ( pchSlash = strchr( pFormTable[i], '/' ) ) != 0 )
    {
      *pchSlash = 0;
    }

    if ( !strcmp( (PCH)ptr, pFormTable[i] ) )
    {
      if ( pchSlash )
      {
        *pchSlash = '/';
      }
      break;
    }
    if ( pchSlash )
    {
      *pchSlash = '/';
    }
  }

  if ( j >= desPpd.desPage.iImgpgpairs )  /* Not found, make it zero */
  {
    i = sFirstForm;
    printf( "INFO: Default form is %s\n", ptr );
  }

  desPpd.desPage.ofsDfpgsz = i;

  cbBuffout -= sLen + 1;  /* Erase the name */

  return;
}


/***************************************************************************
 *
 * FUNCTION NAME = FormPpd
 *
 * DESCRIPTION   = To form a binary output segment out of ppd file.
 *                 Returns false if any error encountered in input
 *                 file format else returns True.
 *
 * INPUT         = NONE.
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

BOOL FormPpd(void)
{
  /*
  ** D74609
  */
/* register int  i; */
  int           i;
  int           j, k;
  char         *p, *q;
  char          scratch[64 + 1];
//int           iResType;    /* QMS fix */

  PUI_BLOCK pBlock;
  PUI_BLOCK pTempBlock;
  INT y;

  /*
  ** Zero out structure
  */
  // @V3.0129238 - Assign a pointer to the block list.
  p = (char *) desPpd.stUIList.pBlockList;
  q = (char *) desPpd.stUICList.puicBlockList;
  memset( &desPpd, 0, sizeof( desPpd ) );
  desPpd.stUIList.pBlockList = (PUI_BLOCK) p;
  desPpd.stUICList.puicBlockList = (PUIC_BLOCK) q;
  memset( desPpd.stUIList.pBlockList, 0, 4000 );

  desPpd.desItems.ofsPswrd = -1;

  /*
  */
  /*
  ** @V3.0101290
  ** For v4.2 PPDs, JCLBegin and JCLToPSInterpreter replaces
  ** InitPostScriptMode.
  */
  SaveCommand( szSearch[initstring], (short *) &desPpd.desItems.ofsInitString,
               COMPRESS);
  if (desPpd.desItems.ofsInitString == -1)
  {
    SaveCommand (szSearch[ JCLBegin ], (short *) &desPpd.desItems.ofsInitString,
                COMPRESS);
    /*
    ** @V3.0115171
    ** Add a separate offset for the JCL to PS interpreter command.  Do not
    ** append it to the *JCLBegin command.  This is needed so if JCLResolution
    ** is used, this command converts the device back to Postscript.
    ** JCLResolution must be inserted between the JCLBegin and the
    ** JCL-to-PS commands.
    */
    SaveCommand( szSearch[ JCLToPSInterpreter ],
                 (short *) &desPpd.desItems.ofsJCLToPS, COMPRESS);
#if 0
//  /*
//  ** Combine the following two strings: JCLBegin & JCLToPSInterpreter.
//  ** Decrement cbBuffout to remove the terminator between the two strings.
//  */
//  if (desPpd.desItems.ofsInitString != -1)
//  {
//    cbBuffout--;
//    SaveCommand (szSearch[ JCLToPSInterpreter ],
//                (short *)&desPpd.desItems.ofsInitString, COMPRESS);
//  }
#endif
  }

  /*
  */
  /*
  ** @V3.0101290
  ** For v4.2PPDs, JCLEnd is the command that replaces TermPostScriptMode.
  */
  SaveCommand( szSearch[termstring], (short *) &desPpd.desItems.ofsTermString,
               COMPRESS);
  if (desPpd.desItems.ofsTermString == -1)
  {
    SaveCommand( szSearch[ JCLEnd ], (short *) &desPpd.desItems.ofsTermString,
                 COMPRESS);
  }

  /*
  ** scan and read the throughput parameter
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[throughput], pbPrBuffer ))
  {
    /*
    ** skip the " character
    */

    i = MovePast( pbPrBuffer, '"' );
    desPpd.desItems.iPpm = (short) atoi( pbPrBuffer + i );
  }
  else
  {
    /*
    ** Null value
    */
    desPpd.desItems.iPpm = -1;
  }

  /*
  ** scan and read the printer memory parameter
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[freevm], pbPrBuffer ))
  {
    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, '"' );
    desPpd.desItems.lFreeVM = atol( pbPrBuffer + i );
  }
  else
  {
    /*
    ** Null value
    */
    desPpd.desItems.lFreeVM = (long) -1;
  }

// @V3.OEM
  /*
  ** scan and read the PCFileName
  */
  desPpd.desItems.ofsPCFileName = cbBuffout;
  SaveCommand( szSearch[PCFileName], (short *) &desPpd.desItems.ofsPCFileName,
               NOCOMPRESS);

  /*
  ** scan and read printer type classified under product name
  */

  desPpd.desItems.ofsPrType = cbBuffout;
  desPpd.desItems.ofsPrType = -1;

  /*
  ** scan and read printer name
  ** Try ShortNickName first
  */
  desPpd.desItems.ofsPrName = cbBuffout;
  SaveCommand( szSearch[shortnickname], (short *) &desPpd.desItems.ofsPrName,
               NOCOMPRESS);

  if ( desPpd.desItems.ofsPrName == -1 )
  {
    /*
    ** scan and read printer name
    ** Use NickName
    */
    desPpd.desItems.ofsPrName = cbBuffout;
    SaveCommand( szSearch[printername], (short *) &desPpd.desItems.ofsPrName,
                 NOCOMPRESS);
  }

  /*
  ** Make sure name is not too long
  */
  /*
  */
  if (strlen( pbItemsBuffer + desPpd.desItems.ofsPrName ) > NAME_LEN - 1)
  {
    printf( "...ERROR...ERROR...ERROR... Nickname too long\n" );
  }

#if 0
//** Scan and read available resolutions.
//*/
//desPpd.desItems.ResList.uNumOfRes  = 0;
//desPpd.desItems.ResList.uResOffset = 0;
//
//// @V3.0115171
//desPpd.desItems.ResList.bIsJCLResolution = FALSE;
//
//fseek( fhIn, (long) 0, SEEK_SET );
//
///* QMS Fix
//** Some printers (QMS) use SetJobResolution instead of SetResolution
//** Look for SetJobResolution first
//*/
///* @COOK */
//if ( ScanParam( szSearch [SetJobResolution], pbPrBuffer ) )
//{
//  iResType = SetJobResolution;
//}
//else
//{
//  /*
//  ** @V3.0115171
//  ** Query for *JCLResolution.
//  */
//  fseek( fhIn, (long) 0, SEEK_SET );
//  if (ScanParam( szSearch[ JCLResolution ], pbPrBuffer ))
//  {
//    iResType = JCLResolution;
//    desPpd.desItems.ResList.bIsJCLResolution = TRUE;
//  }
//  else
//  {
//    fseek( fhIn, (long) 0, SEEK_SET );
//    if ( ScanParam( szSearch [Resolution], pbPrBuffer ) )
//    {
//      iResType = Resolution;
//    }
//    else
//    {
//      iResType = setresolution;
//    }
//  }
//}
//
//fseek( fhIn, (long) 0, SEEK_SET );
//
//while (ScanParam( szSearch [iResType], pbPrBuffer ))
//{
//  /*
//  ** Get the resolution value from the string and put it into
//  ** the variable buffer.
//  */
//  uDPIVal = (SHORT) atoi( pbPrBuffer );
//  *(unsigned *)(pbItemsBuffer + cbBuffout) =  uDPIVal;
//
//  /*
//  ** If the resolution buffer offset indicator is 0, assign it to
//  ** the beginning of the resolution buffer.
//  */
//  if (desPpd.desItems.ResList.uResOffset == 0)
//  {
//    desPpd.desItems.ResList.uResOffset = cbBuffout;
//  }
//
//  /*
//  ** Increment cbBuffout because two bytes are being used by
//  ** the resolution value.
//  ** Also, increment the varaiable that keeps track of the number
//  ** of available resolutions.
//  */
//  cbBuffout += 2;
//  desPpd.desItems.ResList.uNumOfRes++;
//
//  /*
//  ** skip the " character
//  */
//  i = MovePast( pbPrBuffer, '"' );
//
//  /*
//  ** @V3.0129238 START
//  ** It is possible that there are no Postscript commands (NULL).
//  ** Currently, we don't check for that which can lead to garbage output.
//  ** If command does not exist, write out NULL.
//  */
//  if (*(pbPrBuffer + i) != 0x0D || *(pbPrBuffer + i - 2) != '"')
//  {
//    /*
//    ** copies the string delimited by a blank or quote.
//    */
//    j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, FALSE,
//                     COMPRESS);
//  }
//  else
//  {
//    *(pbItemsBuffer + cbBuffout) = 0;
//    j = 1;
//  }
//  cbBuffout += j;
//}
#endif

  /*
  ** @V3.0115171
  ** Move the Default resolution processing after the standard resolution
  ** processing.  If no default resolution is provided, assign the default
  ** to the first resolution found.
  */
  /*
  ** scan and read default resolution
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[defaultres], pbPrBuffer ))
  {
    /*
    ** skip the : character
    */
    i = MovePast( pbPrBuffer, ':' );
    desPpd.desItems.iResDpi = (short) atoi( pbPrBuffer + i );
  }

  /* If no default found try default JCL */
  if ( desPpd.desItems.iResDpi == 0 )
  {
    fbseek( &fbIn, (long)0, SEEK_SET );
    if (ScanParam( szSearch[defaultJCLRes], pbPrBuffer ))
    {
      i = MovePast( pbPrBuffer, ':' );
      desPpd.desItems.iResDpi = (short) atoi( pbPrBuffer + i );
    }
  }
#if 0
//else
//{
//  /*
//  ** @V3.0115171
//  ** No default found.  Assign the default to the first resolution.
//  */
//  /*
//  ** Null value
//  */
//  desPpd.desItems.iResDpi =
//    *((PSHORT) (pbItemsBuffer + desPpd.desItems.ResList.uResOffset));
//  printf( "WARNING... No default resolution, set to %d\n",
//          desPpd.desItems.iResDpi );
//}
#endif

  if (desPpd.desItems.iResDpi == 0)
  {
    desPpd.desItems.iResDpi = 300;
  }

  /*
  ** scan and read screen frequency
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam(szSearch[screenfreq], pbPrBuffer))
  {
    /*
    ** skip the : character
    */

    i = MovePast( pbPrBuffer, '"' );
    desPpd.desItems.lScrFreq = (long)(atof( pbPrBuffer + i) * 100.00 );
  }
  else
  {
    /*
    ** Null value
    */
    desPpd.desItems.lScrFreq = -1L;
  }

  /*
  ** To read the flag that indicates whether the device supports
  ** colour or not.
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[colordevice], pbPrBuffer ))
  {
    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, ':' );
    i += SkipBlank( pbPrBuffer + i );

    if (!strnicmp("TRUE", pbPrBuffer+i, 4))
    {
      desPpd.desItems.fIsColorDevice = TRUE;
    }
    else
    {
      desPpd.desItems.fIsColorDevice = NONE;
    }
  }
  else
  {
    desPpd.desItems.fIsColorDevice = NONE;
  }

  /*
  ** To read the True or False value indicating whether the Postscript
  ** device has a built in file system.
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[filesystem], pbPrBuffer ))
  {
    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, ':' );
    i += SkipBlank(pbPrBuffer + i );

    if (!strnicmp( "TRUE", pbPrBuffer + i, 4 ))
    {
      desPpd.desItems.fIsFileSystem = TRUE;
    }
    else
    {
      desPpd.desItems.fIsFileSystem = FALSE;
    }
  }
  else
  {
    desPpd.desItems.fIsFileSystem = FALSE;
  }

  /*
  ** To read the Postscript sequence that will perform a soft
  ** restart of the printer.
  */
  desPpd.desItems.ofsReset = -1;

  /*
  ** Read the appropriate postscript sequence to exit the job server loop
  */
  desPpd.desItems.ofsExitserver = -1;

  /*
  ** Read the halftone screen angle
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[screenangle], pbPrBuffer ))
  {
    /*
    ** skip the : character
    */
    i = MovePast( pbPrBuffer, '"' );
    desPpd.desItems.iScreenAngle = (long)(atof( pbPrBuffer + i) * 100.0 );
  }
  else
  {
    /*
    ** Null value
    */
    desPpd.desItems.iScreenAngle = -1L;
  }

  /*
  ** Read the value indicating whether the device supports
  ** infinitely variable paper sizes
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[variablepaper], pbPrBuffer ))
  {
    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, ':' );
    i += SkipBlank( pbPrBuffer + i );

    if (!strnicmp( "TRUE", pbPrBuffer + i, 4 ))
    {
      desPpd.desPage.fIsVariablePaper = TRUE;
    }
    else
    {
      desPpd.desPage.fIsVariablePaper = FALSE;
    }
  }
  else
  {
    desPpd.desPage.fIsVariablePaper = FALSE;
  }

  /*
  ** Read the default imageable area paper type .
  */
  desPpd.desPage.ofsDefimagearea = -1;

  /*
  ** Read the keyword for the Default paper dimension.
  ** This value should always be letter
  */
  desPpd.desPage.ofsDefpaperdim = -1;

  /*
  ** This gives the font name which is the default font provided by
  ** findfont if the requested font is not available
  */
  SaveString( szSearch[defaultfont], (short *)&desPpd.desFonts.ofsDeffont );

  /*
  ** If no default font provided make it Courier
  */
  if (desPpd.desFonts.ofsDeffont == -1)
  {
    /*
    ** Copy the default font (no ending null)
    */
    desPpd.desFonts.ofsDeffont = cbBuffout;
    *(pbItemsBuffer + cbBuffout) = sizeof( acDefFont ) - 1;
    cbBuffout++;

    for (i = 0; i < sizeof( acDefFont )-1; i++, cbBuffout++)
    {
      *(pbItemsBuffer + cbBuffout) = acDefFont[i];
    }
  }

#if 0
  /*
  ** Read the complete list of page names and page select commands
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

#if 0
///*
//** count of pairs formed
//*/
//k = 0;
//desPpd.desPage.ofsLspgCmnds = cbBuffout;
//
//if (ScanParam( szSearch[variablepaper], pbPrBuffer ))
//{
//  /*
//  ** Copies the string name
//  */
//
//  i = NameToIndex( );   /* Change name to index */
//
//  /*
//  ** string length is one more than actual scanned
//  */
//  i--;
//
//  /*
//  ** skip the " character
//  */
//  i += MovePast( pbPrBuffer+i, '"' );
//
//  /*
//  ** copies the string delimited by a blank or quote with ist
//  ** byte as length. Strings are not null terminated
//  */
//  j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, FALSE,
//                   COMPRESS);
//  cbBuffout += j;
//  k++;
//}
  sPpd.desPage.iCmpgpairs = k;
#endif

/*
EE** Read the list of paper type supported and the associated paper sizes
  */
  fbseek( &fbIn, (long)0, SEEK_SET );
  #endif


  /*
  ** Read the complete list of paper types and associated image coordinates
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  /*
  ** This has been moved to after paper names have been read so we can
  ** fill in a default size if not provided.  Done in the next block below
  */
  /*
  ** Read default page size at power on :-whether letter ,A4 etc
  */
#if 0
//SaveString( szSearch[defaultpsize], (short *)&desPpd.desPage.ofsDfpgsz );
//
//DoDefaultForm( );  /* Change default form to index value */
//
///*
//** @V3.0129238 - Read in page sizes
//*/
//desPpd.desInpbins.ofsPageSizes = cbBuffout;
//while (ScanParam( szSearch[ pagenamelist ], pbPrBuffer))
//{
////i = CopyString( pbItemsBuffer + cbBuffout, pbPrBuffer, 80, TRUE );  //@V3.1137565
////cbBuffout += i;                                                     //@V3.1137565
//  i = NameToIndex( );   /* Change name to index */                    //@V3.1137565
//
//  /*
//  ** string length is one more than actual scanned
//  */
//  i--;
//
//  /*
//  ** skip the " character
//  */
//  i += MovePast( pbPrBuffer + i, '"' );
//
//  /*
//  ** copies the string delimited by a blank or quote.
//  */
//  j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, FALSE,
//                   COMPRESS );
//  cbBuffout += j;
//  k++;
//}
//desPpd.desInpbins.iNumOfPageSizes = k;
//
///*
//** Read the default condition of the manual feed operation.
//** This can be used to determine the initial setting of the
//** manualfeed operation
//*/
//
//fseek( fhIn, (long)0, SEEK_SET );
//
//if (ScanParam( szSearch[defmanualfeed], pbPrBuffer))
//{
//  /*
//  ** skip the " character
//  */
//  i = MovePast( pbPrBuffer, ':' );
//  i += SkipBlank( pbPrBuffer + i );
//
//  if (!strnicmp("TRUE", pbPrBuffer+i, 4))
//  {
//    desPpd.desInpbins.iManualfeed = TRUE;
//  }
//  else if (!strnicmp("NONE", pbPrBuffer+i, 4))
//  {
//    desPpd.desInpbins.iManualfeed = NONE;
//  }
//  else
//  {
//    desPpd.desInpbins.iManualfeed = FALSE;
//  }
//}
//else
//{
//  desPpd.desInpbins.iManualfeed = NONE;
//}
//
///*
//** Read the invocation string for disabling or enabling manual feed
//*/
//
//fseek( fhIn, (long)0, SEEK_SET );
//
//while (ScanParam( szSearch[manualfeed], pbPrBuffer))
//{
//  if (!strnicmp("TRUE", pbPrBuffer, 4))
//  {
//    desPpd.desInpbins.ofsManualtrue = cbBuffout;
//  }
//  else
//  {
//    desPpd.desInpbins.ofsManualfalse = cbBuffout;
//  }
//
//  /*
//  ** skip the " character
//  */
//  i = MovePast( pbPrBuffer, '"' );
//
//  /*
//  ** copies the string delimited by a blank or quote.
//  */
//  j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, FALSE,
//                   COMPRESS );
//  cbBuffout += j;
//}
#endif

// @V3.1142031
#if 0
///*
//** READ THE DEFAULT CONDITION OF THE DUPLEX OPTION.
//** DUPLEX PROCESSING PUT IN BY DCR 1462
//*/
//
//fseek( fhIn, (long)0, SEEK_SET );
//
//if (ScanParam( szSearch[defaultduplex], pbPrBuffer))
//{
//  /*
//  ** skip the " character
//  */
//
//  i = MovePast( pbPrBuffer, ':' );
//  i += SkipBlank( pbPrBuffer + i );
//
//  /*
//  ** no duplex available
//  */
//
//  if (!strnicmp("NONE", pbPrBuffer+i, 4))
//  {
//    desPpd.desItems.sDefaultDuplex = DUPLEX_NONE;
//  }
//  else
//  {
//    /*
//    ** duplex avail, turned off
//    */
//    if (!strnicmp("FALSE", pbPrBuffer+i, 5))
//    {
//      desPpd.desItems.sDefaultDuplex = DUPLEX_FALSE;
//    }
//    else
//    {
//      /*
//      ** duplex of some type, turn off
//      */
//      desPpd.desItems.sDefaultDuplex = DUPLEX_FALSE;
//    }
//  }
//}
//else
//{
//  desPpd.desItems.sDefaultDuplex = DUPLEX_NONE;
//}
//
///*
//** Read the invocation string for disabling or enabling duplex option
//*/
//fseek( fhIn, (long)0, SEEK_SET );
//
//while (ScanParam( szSearch[duplex], pbPrBuffer))
//{
//  if (!strnicmp( "FALSE", pbPrBuffer, 5) ||
//      !strnicmp( "NONE",  pbPrBuffer, 4) )
//  {
//    desPpd.desItems.ofsDuplexFalse = cbBuffout;
//  }
//  else if (!strnicmp( "DUPLEXNOTUMBLE", pbPrBuffer, 14))
//  {
//    desPpd.desItems.ofsDuplexNoTumble = cbBuffout;
//  }
//  else if (!strnicmp( "DUPLEXTUMBLE", pbPrBuffer, 12))
//  {
//    desPpd.desItems.ofsDuplexTumble = cbBuffout;
//  }
//  else
//  {
//    desPpd.desItems.ofsDuplexNoTumble = cbBuffout;
//  }
//
//  /*
//  ** skip the " character
//  */
//  i = MovePast( pbPrBuffer, '"' );
//
//  /*
//  ** copies the string delimited by a blank or quote with ist byte
//  ** as length. Strings are not null terminated
//  */
//  j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, FALSE,
//                   COMPRESS );
//  cbBuffout += j;
//}
#endif

  /*
  ** Read the default value for the input slot category
  */
#if 0
//SaveString( szSearch[definputslot], (short *)&desPpd.desInpbins.ofsDefinputslot );
//fseek( fhIn, (long)0, SEEK_SET );
//
///*
//** count of pairs formed
//*/
//
//k = 0;
//desPpd.desInpbins.ofsCmInpbins = cbBuffout;
//
//while (ScanParam( szSearch[inputbinlist], pbPrBuffer))
//{
//  /*
//  ** copies the Input bin name
//  */
//  // @V3.0129238
//  i = CopyString( pbItemsBuffer + cbBuffout, pbPrBuffer, 80, FALSE );
//  cbBuffout += i;
//
//  /*
//  ** string length is one more than actual scanned
//  */
//  i--;
//
//  /*
//  ** skip the " character
//  */
//  i += MovePast( pbPrBuffer + i, '"' );
//
//  /*
//  ** @V3.0129238 START
//  ** It is possible that there are no Postscript commands (NULL).
//  ** Currently, we don't check for that which can lead to garbage output.
//  ** If command does not exist, write out NULL.
//  */
//  if (*(pbPrBuffer + i) != 0x0D || *(pbPrBuffer + i - 2) != '"')
//  {
//    /*
//    ** copies the string delimited by a blank or quote.
//    */
//    j = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer + i, FALSE,
//                     COMPRESS );
//  }
//  else
//  {
//    *(pbItemsBuffer + cbBuffout) = 0;
//    j = 1;
//  }
//  // @V3.0129238 END
//
//  cbBuffout += j;
//  k++;
//}
//
//desPpd.desInpbins.iInpbinpairs = k;
//
//if (desPpd.desInpbins.ofsDefinputslot == -1)
//{
//  desPpd.desInpbins.ofsDefinputslot = desPpd.desInpbins.ofsCmInpbins;
//}
#endif

  /*
  ** Read the keyword for the default output order which can be normal
  ** or reverse
  */

  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[defoutputorder], pbPrBuffer))
  {
    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, ':' );
    i += SkipBlank( pbPrBuffer + i );

    if (!strnicmp( "NORMAL", pbPrBuffer + i, 6))
    {
      desPpd.desOutbins.fIsDefoutorder = NORMAL;
    }
    else
    {
      desPpd.desOutbins.fIsDefoutorder = REVERSE;
    }
  }
  else
  {
    desPpd.desOutbins.fIsDefoutorder = REVERSE;
  }

  /*
  ** Read the invocation strings for selecting normal or
  ** reverse output order
  */
  desPpd.desOutbins.ofsOrdernormal = -1;
  desPpd.desOutbins.ofsOrderreverse = -1;
  fbseek( &fbIn, (long)0, SEEK_SET );

  while (ScanParam( szSearch[outputorder], pbPrBuffer))
  {
    if (!strnicmp( "NORMAL", pbPrBuffer, 4))
    {
      desPpd.desOutbins.ofsOrdernormal = cbBuffout;
    }
    else
    {
      desPpd.desOutbins.ofsOrderreverse = cbBuffout;
    }

    /*
    ** skip the " character
    */
    i = MovePast( pbPrBuffer, '"' );

    /*
    ** copies the string delimited by a blank or quote with ist byte
    ** as length. Strings are not null terminated
    */
    j = CopyInQuote( pbItemsBuffer+cbBuffout, pbPrBuffer+i, FALSE,
                     COMPRESS );
  cbBuffout += j;
  }

  /*
  ** Read the complete procedure of Transfer Normalized & inverse
  */
  SaveProcedure( szSearch[transfernor], (short *)&desPpd.desItems.ofsTransferNor );
  SaveProcedure( szSearch[transferinv], (short *)&desPpd.desItems.ofsTransferInv );

  /*
  ** Read the invocation strings for various output bins
  */
  fbseek( &fbIn, (long)0, SEEK_SET );
  k = 0;

  desPpd.desOutbins.iOutbinpairs = k;
  desPpd.desOutbins.ofsCmOutbins = -1;

  /*
  ** Insert the section of code that searches for the "*LanguageLevel"
  ** PostScript command.
  */
  fbseek( &fbIn, (long)0, SEEK_SET );

  if (ScanParam( szSearch[LanguageLevel], pbPrBuffer ))
  {
    /*
    ** Skip past the colon, spaces, and the first quote, until a numeric
    ** character is found.  The reason for the loop below is, even though
    ** the PPD specs require the PS level to be in quotes, this is not the
    ** case for all PPDs.  Since this compiler cannot assume that the
    ** level number is in quotes, it will have to continue to increment
    ** the offset until the first numeric character is found.
    */
    i = 0;
    while (*(pbPrBuffer + i) < '0' || *(pbPrBuffer + i) > '9')
    {
      i++;
    }

    /*
    ** Save the language level offset.
    */
    desPpd.desItems.usLanguageLevel = atoi( pbPrBuffer + i );
  }
  else
  {
    desPpd.desItems.usLanguageLevel = 1;
  }
  /*
  */

  /* @V3.0129238 - Add new file group */
  fbseek( &fbIn, (long)0, SEEK_SET );
  memset( &UIFileGroup, 0, sizeof( UIFileGroup ) );
  while (ScanParam( szSearch[ OpenGroup ], pbPrBuffer ))
  {
    i = MovePast( pbPrBuffer, ' ' );
    if (!strnicmp( "InstallableOptions", pbPrBuffer + i, 18))
    {
      UIFileGroup.usFileStart = fbtell( &fbIn );        /* took out the cast, since it's the same size */

      if (ScanParam( szSearch[ CloseGroup ], pbPrBuffer ))
      {
        UIFileGroup.usFileEnd = fbtell( &fbIn );        /* took out the cast, since it's the same size */
      }
    }
  }
  ProcessUIList( );
  ProcessCmdsAsUIs( );

  /*
  ** count of pairs formed
  */
  k = 0;
  desPpd.desPage.ofsDimxyPgsz = cbBuffout;

  fbseek( &fbIn, (long)0, SEEK_SET );
  if ((pBlock = QueryBlockFromKeyword( &desPpd.stUIList, pbItemsBuffer,
                                       UINAME_PAGESIZE, NULL )) != NULL)
  {
    while (ScanParam( szSearch[pagesizelist], pbPrBuffer ))
    {
      y = 0;
      i = 0;
      while (*(pbPrBuffer + i) != ' ' && *(pbPrBuffer + i) != ':' &&
             *(pbPrBuffer + i) != '/')
      {
        scratch[ y++ ] = *(pbPrBuffer + i++);
      }
      scratch[ y ] = 0;

      pTempBlock = pBlock;
      QueryEntryFromOption( (PBYTE) pbItemsBuffer, pTempBlock,
                            (PBYTE) scratch, (PINT) &y );
      *((PSHORT) (pbItemsBuffer + cbBuffout)) = (SHORT) y;
      cbBuffout += 2;

      /*
      ** warn if formname longer than 31
      */
      if (i > MAX_PSIZE - 1)
      {
        strncpy( scratch, pbItemsBuffer + cbBuffout - i, i );
        scratch[i] = (char) 0;
        printf( "...ERROR... %s\nFormname > MaxPaperSize [%d]\n", scratch, i );
      }

      if ( *(pbPrBuffer + i) == '/' )
      {
        i += MovePast( pbPrBuffer+i, ':' );
      }
      /*
      ** skip the " character
      */
      i += MovePast( pbPrBuffer+i, '"' );
      j = atoRound( pbPrBuffer + i );

      /*
      ** copies the x-size into items buffer
      */
      memcpy( (pbItemsBuffer + cbBuffout), (char *) &j, sizeof( SHORT ) );
      cbBuffout += sizeof(SHORT );
      i += SkipNumber( pbPrBuffer + i );
      j = atoRound( pbPrBuffer + i );

      /*
      ** copies the y-size into items buffer
      */
      memcpy( (pbItemsBuffer + cbBuffout), (char *) &j, sizeof( SHORT ) );
      cbBuffout += sizeof( SHORT );
      k++;
    }
    desPpd.desPage.iDmpgpairs = k;

    /*
    ** count of pairs formed
    */
    fbseek( &fbIn, (long)0, SEEK_SET );
    k = 0;
    desPpd.desPage.ofsImgblPgsz = cbBuffout;

    /*
    ** D74609
    ** *ImageableArea: value1, value2, value3, value4
    ** value1 and value2 must be rounded up.
    ** value3 and value4 must be truncated.
    */
    while (ScanParam( szSearch[imageablearea], pbPrBuffer))
    {
      CHAR  achOldSuffix[ 64 + 1 ];
      PCHAR pcSlash;

      j = 0;
      y = 0;
      i = 0;
      while (*(pbPrBuffer + i) != ':' && *(pbPrBuffer + i) != '/')
      {
        scratch[ y++ ] = *(pbPrBuffer + i++);
      }
      scratch[ y ] = 0;

      /* See if there is a translation string */

      if ( *( pbPrBuffer + i ) == '/' )
      {
        pcSlash =  pbPrBuffer + (i++) + 1;  /* don't need slash */
        while( *pcSlash != ':' )    /* There is so copy it over */
        {
          achOldSuffix[ j++ ] = *(pcSlash++);
          i++;
        }
      }
      achOldSuffix[ j ]  = '\0';    /* If no slash OldSuffix will be null */

      pTempBlock = pBlock;
      QueryEntryFromOption( (PBYTE) pbItemsBuffer, pTempBlock,
                            (PBYTE) scratch, (PINT) &y );
      *((PSHORT) (pbItemsBuffer + cbBuffout)) = (SHORT) y;
      cbBuffout += 2;
      /*
      ** Bottom left X corner.
      ** Round up to nearest integer.
      */
      roundImgAreaVals( &i, ROUND_FLOAT, '"' );

      /*
      ** Bottom left y corner.
      ** Round up to nearest integer.
      */
      roundImgAreaVals( &i, ROUND_FLOAT, ' ' );

      /*
      ** Top right x corner.
      ** Truncate floating point value.
      */
      roundImgAreaVals( &i, TRUNCATE_FLOAT, ' ' );

      /*
      ** Top right y corner.
      ** Truncate floating point value.
      */
      roundImgAreaVals( &i, TRUNCATE_FLOAT, ' ' );

      /* Copy the old name out into resource */
      strcpy( pbItemsBuffer + cbBuffout, achOldSuffix );
      cbBuffout += strlen( achOldSuffix ) + 1;

      k++;
    }
    desPpd.desPage.iImgpgpairs = k;
  }
  else
  {
    desPpd.desPage.iDmpgpairs = 0;
    desPpd.desPage.ofsDimxyPgsz = 0;

    desPpd.desPage.iImgpgpairs = 0;
    desPpd.desPage.ofsImgblPgsz = 0;
  }

  ProcessUIConstraints( );
  VerifyUICList( );

  /*
  ** Read the fonts name list supported by printer
  */

  fbseek( &fbIn, (long)0, SEEK_SET );

  /*
  ** count of pairs formed
  */

  k = 0;
  desPpd.desFonts.ofsFontnames = cbBuffout;

  /*
  ** @V3.0129238 - Modify code for font name for UI
  */
  while (ScanParam( szSearch[fontnamelist], pbPrBuffer))
  {
    for (j = 0 ; j < 32 ; j++)
    {
      // @V3.0129238
      if (isspace( pbPrBuffer[ j ] ) || pbPrBuffer[ j ] == ':' ||
          pbPrBuffer[ j ] == 0)
      {
        pbPrBuffer[ j ] = 0;
        break;
      }
    }

    /*
    ** check fontres.h to see if font is known
    */
    i = 1;

    while (FontResourceName[i] != 0)
    {
      // @V3.0129238
      if (!strcmp( FontResourceName[ i ], pbPrBuffer ))
      {
        break;
      }

#if 0
//    /*
//    ** since FontResourceName is far ptrs must do compare here instead
//    ** of str calls
//    */
//    j = 0;
//
//    while (FontResourceName[i][j] == pbPrBuffer[j])
//    {
//      j++;
//    }
//
//    /*
//    ** names matched
//    */
//    if ((FontResourceName[i][j] == 0) && (pbPrBuffer[j] == ':'))
//    {
//      break;
//    }
#endif
      i++;
    }

    /*
    ** couldn't match font
    */
    if (FontResourceName[i] == 0)
    {
      // @V3.0129238
#if 0
//    p = strchr(pbPrBuffer, ':' );
//    *p = 0;
#endif
      if ( nErrorLogLevel<1 )
        printf( "...WARNING..WARNING..WARNING... font %s not found\n", pbPrBuffer );
      continue;
    }
    *((BYTE *)(pbItemsBuffer + cbBuffout)) = i;
    cbBuffout += 1;
    k++;
  }
  desPpd.desFonts.iFonts = k;
  ProcessFormTable();

  fbclose( &fbIn );
//@V3.0CMPS01
//desPpd.desItems.iSizeBuffer = cbBuffout;
  desPpd.desItems.iSizeBuffer = cbBuffout + iShrinkage;
  return( TRUE );
}


/***************************************************************************
 *
 * FUNCTION NAME = CntInpFiles
 *
 * DESCRIPTION   = This routine opens the input list and reads the
 *                 list of ppd file names.  It stores all the file
 *                 names by allocating memory for them and saves their
 *                 pointers.  The semicolon is used as comment
 *                 delimiter, semicolons in col 1 means skip the line.
 *
 *                 Note - the list files contain the ppd name with
 *                 extension but no path.
 *
 * INPUT         = szInputList - name of file containing list of ppds.
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = count of ppd files
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

int CntInpFiles( char *szInputList )
{
  char szOutFile[] = "pswords.dat";   //@V3.0CMPS01
  char  szPath[256];
  char  pTemp[ 255 ];
  int   i;
  FILE *fh;
  char *p;
  char *p2;
  BOOL fLineParsed;

  /*@V3.0CMPS01 vvv */
  /* Open words file */
  if ( fWriteWords )
  {
    ppdOut = fopen (szOutFile, "w");
    if (!ppdOut)
    {
      printf ("can't open %s\n", szOutFile);
      exit (1);
    }
  }
  /*@V3.0CMPS01 ^^^ */

  if ((MRILangFile = fopen( MRIPathString, "r" )) != NULL)
  {
    fgets( szPath, sizeof( szPath ) - 1, MRILangFile );
    ParsePPDLang( szPath, MRIPathString );
  }

  i = 0;

  /*
  ** open the list file
  */
  if ((fh = fopen( szInputList, "r")) == NULL)
  {
    /*
    ** report failure
    */
    RepError( err_cantopen, szInputList );
  }

  /*
  ** read each line in the file saveing the file names
  */
  while (fgets( szPath, sizeof( szPath ) - 1, fh) != NULL)
  {
    /*
    ** check for comment
    */
    if ((p = strchr( szPath, ';' )) != NULL)
    {
      /*
      ** chop it off
      */
      *p = (char)NULL;

      /*
      ** comment marker was in col 1
      ** skip to next line
      */
      if ((p-szPath) == 0)
      {
        continue;
      }

    }

    if (MRILangFile != NULL)
    {
      fLineParsed = ParsePPDLang( szPath, (PSZ) pTemp );
    }
    else if ((p2 = strstr( szPath, "LANGUAGE" )) != NULL)
    {
      *p2 = 0;
      fLineParsed = TRUE;
    }
    else
    {
      fLineParsed = FALSE;
    }

    if ((MRILangFile == NULL && p2 == NULL) ||
        (MRILangFile != NULL &&
          (strcmpi( pTemp, MRIPathString ) == 0 || fLineParsed == FALSE)))
    {
      apszInfiles[i] = malloc( strlen( szPath ) + 1 );
      strcpy( apszInfiles[i++], szPath );
    }
  }
  return( i );
}

/***************************************************************************
 *
 * FUNCTION NAME = WriteRCFile
 *
 * DESCRIPTION   = Will write out the RC file to get the PPBs and
 *                 hreader into the drv.  First the type_id is #defined
 *                 to the include value The resource_id for header is
 *                 0 The rest of the resource_ids are set to 1 based
 *                 file index
 *
 * INPUT         = iFileCount - count of PPB files
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

void WriteRCFile( int iFileCount, UINT uiOutDirLen )
{
  CHAR  FileName[128];
  FILE *phOutFile;
  int   i;
  CHAR szOutDir2[128];            //PPC Quotes & Slashes
  PCHAR pCh;                      //PPC Quotes & Slashes
  INT z;

  /*
  ** open the rc file
  */
  /* @V3.0104222 */
  strcpy( &szOutDir[ uiOutDirLen ], RC_FILE_NAME );
  if ((phOutFile = fopen( szOutDir, "w" )) == 0)
  {
    /*
    ** open failed
    */
    RepError( err_cantcreate, RC_FILE_NAME );
  }
  szOutDir[ uiOutDirLen ] = 0;

  /*
  ** major define statement
  */
  fprintf( phOutFile, "\n#define  ALLPPDS %d\n\n", ALLPPDS );

  //PPC add leading quote and make double slashes
  szOutDir2[0] = '"';
  i = 1;
  pCh = szOutDir;
  while ( *pCh )
  {
    if ( ( szOutDir2[i++] = *pCh ) == '\\' )
    {
      szOutDir2[i++] = '\\';
    }
    pCh++;
  }
  szOutDir2[i] = 0;
  //PPC Quotes & Slashes   end

  /*
  ** what resource signature and directory are in
  */
  strcpy( FileName, szOutDir2 );  //PPC Quotes & Slashes
  strcat( FileName, DIRECTORY_NAME );
  strcat( FileName, "\"" );       //PPC Quotes & Slashes
  fprintf( phOutFile, "RESOURCE ALLPPDS %d LOADONCALL %s\n", DIRECTORY_PPB,
           FileName );

  z = 2;
  for (i = 0; i < iFileCount; i++)
  {
    /*
    ** @V3.1141419
    ** For files that were not parsed, do not put into the RC file.
    */
    if (apszInfiles[ i ][ 0 ] != 0)
    {
      strcpy( FileName, szOutDir2 );  //PPC Quotes & Slashes
      strcat( FileName, apszInfiles[i] );
      strcat( FileName, "\"" );       //PPC Quotes & Slashes
      fprintf( phOutFile, "RESOURCE ALLPPDS %d LOADONCALL %s\n", z,
               FileName );
      z++;
    }
  }

  /*
  ** Tack on an blank
  */
  fprintf( phOutFile, "\n" );
  fclose( phOutFile );
}




/*
** V2.191412
** New function.
*/
/***************************************************************************
 *
 * FUNCTION NAME = VerifyPPDLineSize
 *
 * DESCRIPTION   = Since a PPD line with a quoted value may exceed one
 *                 physical file line, it is important to verify that the
 *                 current PPD line that is being read does not exceed the
 *                 buffer.  If the PPD line does exceed the buffer, the
 *                 compiler is not equipped to handle oversized lines, so the
 *                 specific command must be ignored.
 *                 This function verifies that the line does not exceed the
 *                 buffer by searching for both starting and ending quotes of
 *                 the quoted value.  If the starting quote exists, the value
 *                 is a quoted value and the function searches for the ending
 *                 quote.  If an ending quote is found, the line does not
 *                 exceed the buffer.  If an ending quote is not found, the
 *                 line exceeds the buffer and the function returns an error.
 *
 * INPUT         = pBuffer - Pointer to buffer containing current PPD line.
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = TRUE - Buffer does not exceed IBUFFSIZE bytes.
 * RETURN-ERROR  = FALSE - Buffer exceeds IBUFFSIZE bytes.
 *
 *************************************************************************   */
BOOL VerifyPPDLineSize( PCHAR pBuffer )
{
  UINT uiLoop1;
  UINT uiLoop2;
  BOOL bRC = TRUE;

  /*
  ** Search for the first quote.
  */
  for (uiLoop1 = 0 ; uiLoop1 < IBUFFSIZE ; uiLoop1++)
  {
    if (*(pBuffer + uiLoop1) == '"')
    {
      break;
    }
  }

  /*
  ** In the above loop, if uiLoop1 is IBUFFSIZE, it could not find a starting
  ** quote.  Otherwise, a starting quote was found so find the ending quote.
  */
  if (uiLoop1 < IBUFFSIZE)
  {
    for (uiLoop2 = uiLoop1 + 1 ; uiLoop2 < IBUFFSIZE ; uiLoop2++)
    {
      if (*(pBuffer + uiLoop2) == '"')
      {
        break;
      }
    }

    /*
    ** If uiLoop2 equals IBUFFSIZE, no ending quote was found.  The buffer
    ** was exceeded.
    */
    if (uiLoop2 == IBUFFSIZE)
    {
      /*
      ** Since this compiler passes the PPD file for each command, if a
      ** specific command exceeded the buffer, an error will be generated for
      ** each pass.  Rather than display the same error for each pass, display
      ** the error for the first pass only.
      */
      if (BErrorWasDisplayed == FALSE)
      {
        strtok( pBuffer, " " );
        /*
        ** Change "ERROR" to "INFO".  This was confusing the build group.
        ** They were thinking that an error occurred and that the build was
        ** not successful.
        */
        printf( "INFO - Command %s, line exceeds maximum buffer size of %u K.\nThis line is ignored.\n",
                pBuffer, IBUFFSIZE );
      }
      BErrorWasDisplayed = TRUE;
      bRC = FALSE;
    }
  }

  return( bRC );
}





/***************************************************************************
 *
 * FUNCTION NAME = main
 *
 * DESCRIPTION   = Function Mainline
 *
 *
 * INPUT         = argc - count of commandline arguments
 *                 argv - array of pointers to commandline arguments
 *
 * OUTPUT        = NONE.
 *
 * RETURN-NORMAL = NONE.
 * RETURN-ERROR  = NONE.
 *
 *************************************************************************   */

main( int argc, char **argv )
{
  int   i;
//int   j, z, y; NOTUSED
  int   iPPBCount;
  long  lResNum;
  char *pc;
  UINT  uiOutDirLen;

  /*
  ** @V3.0129238
  */
//UINT      uiLoop;      NOTUSED
//PUI_BLOCK pUIBlock;    NOTUSED
  PUI_LIST  pUIList = &desPpd.stUIList;

  GetArgs( argc, argv );

  /*
  ** @V3.0129238 - Clear UI list
  */
  memset( pUIList, 0, sizeof( UI_LIST ) );
  DosAllocMem( (PPVOID) &pUIList->pBlockList, 4096,
               PAG_READ | PAG_WRITE | PAG_COMMIT );
  memset( pUIList->pBlockList, 0, 4096 );

  memset( &desPpd.stUICList, 0, sizeof( UIC_LIST ) );
  DosAllocMem( (PPVOID) &desPpd.stUICList.puicBlockList, 4096,
               PAG_READ | PAG_WRITE | PAG_COMMIT );

  /*
  ** if there is one input file specified
  */
  if (iInfiles == 1)
  {
    iInfiles = CntInpFiles( apszInfiles[0] );
  }

  if (iInfiles == 0)
  {
    RepError( err_cantopen, NULL );
  }

  /*
  ** Add error checking to make sure that the number of PPD files in the
  ** list does not exceed the maximum amount of PPDs allowed by the
  ** compiler.
  */
  if (iInfiles > MAX_INPUT)
  {
    RepError( err_maxPPDs, NULL );
  }

  /* @V3.0CMPS01 Build hash table */
  BuildHashTable();

  /*
  ** create the output file so that ppd segments can be written onto it
  */
  i = 0;

  /*
  ** allocate memory for directory segment
  */
  if ((pbDirBuffer = malloc(ofsOutfile = (long)(iInfiles+1)*DRENTRYSIZE)) ==
     NULL)
  {
    RepError( err_cantopen, NULL );
  }

  /*
  ** fill with zeros
  */
  memset( pbDirBuffer, 0, ofsOutfile );

  /*
  ** allocate memory for items buffer
  */
  // @V3.0129238 - Increase buffer size
  if ((pbItemsBuffer = malloc(PRBUFSZ * 24)) == NULL)
  {
    RepError( err_cantopen, NULL );
  }
  psigFile = (SIGNATURE *)pbDirBuffer;

  /*
  ** get the printer name of generic printer
  */
  if (!Openppd( szGeneric ))
  {
    RepWarning( err_cantopen, szGeneric );
    return 0;
  }

  if (!FormPpd())
  {
    RepWarning( err_cantopen, szGeneric );
    return 0;
  }

  /*
  */
  strcpy( (char *) psigFile, pbItemsBuffer + desPpd.desItems.ofsPrName );

  memcpy( (char *) psigFile+40, (char *) &iInfiles, sizeof(USHORT) );
  pdrTblEntry = (DRENTRY *) (pbDirBuffer + DRENTRYSIZE );
  cbBuffout = 0;

  /* V3.0104222 */
  uiOutDirLen = strlen( szOutDir );
  lResNum = 2;
  while (i < iInfiles)
  {
    if (!Openppd(apszInfiles[i]))
    {
      /*
      ** @V3.1141419
      ** Ignore PPD files that aren't found.
      */
      i++;
      continue;
    }

    sFormCount = 0; /* init for each ppd */
    /*
    ** Parse and create the PPD segment
    */
    BErrorWasDisplayed = FALSE;
    iShrinkage = 0; //@V3.0CMPS01
    if (FormPpd())
    {
      /*
      ** Create a ppb for each ppd file
      */
      /* @V3.0104222 */
      RemoveExtension( apszInfiles[i] );
      AddExtension( apszInfiles[i], DEF_OEXT );
      strcpy( &szOutDir[ uiOutDirLen ], apszInfiles[ i ] );
      fhOut = fopen( szOutDir, "w+b" );

      if (!fhOut)
      {
        RepError( err_cantcreate, apszInfiles[i] );
      }

      /*
      ** write the printer descriptor segment onto output file
      */
      if (fwrite( (char *) &desPpd,1,sizeof( desPpd ),fhOut) != sizeof(desPpd))
      {
        RepError( err_output, NULL );
      }

      /*
      ** store the resource number (i+1)
      */
      memcpy( (char *)pdrTblEntry+MAX_FNAMESIZE, (void *)&lResNum, sizeof(long) );
      lResNum++;

      /*
      ** @V3.0129238 - Write out UI buffer list
      */
      if (pUIList->pBlockList != NULL)
      {
        fwrite( pUIList->pBlockList, 1, desPpd.stUIList.usBlockListSize,
                fhOut );
        memset( pUIList->pBlockList, 0, 4096 );
      }
      pUIList->usNumOfBlocks = 0;
      pUIList->usBlockListSize = 0;

      if (desPpd.stUICList.usNumOfUICs > 0)
      {
        fwrite( desPpd.stUICList.puicBlockList, 1,
                desPpd.stUICList.usNumOfUICs * sizeof( UIC_BLOCK ), fhOut );
        memset( desPpd.stUICList.puicBlockList, 0, 4096 );
      }

      /*
      ** write the items buffer which contain various names ,commands and
      ** dimensions lists
      */
      /* ftell( fhOut ); debug */
      if (fwrite( pbItemsBuffer, 1, cbBuffout, fhOut) != cbBuffout)
      {
        RepError( err_output1, NULL );
      }
      cbBuffout = 0;

      /*
      */
      strcpy( (char *) pdrTblEntry, pbItemsBuffer + desPpd.desItems.ofsPrName );

      pc = (char *)pdrTblEntry;
      pc += DRENTRYSIZE;
      pdrTblEntry = (DRENTRY *)pc;

      /*
      ** Close the ppb
      */
      fclose( fhOut );
    }
    i++;
  }
  /* @V3.0104222 */
  szOutDir[ uiOutDirLen ] = 0;

  iPPBCount = i;
  i = CountFilesFound( );

  memcpy( (char *)psigFile+40+sizeof(USHORT), (char *)&i, sizeof(USHORT) );

  /*
  ** Write out directory
  */
  /* @V3.0104222 */
  strcpy( &szOutDir[ uiOutDirLen ], DIRECTORY_NAME );
  fhOut = fopen( szOutDir, "w+b" );
  ftell(fhOut);


  szOutDir[ uiOutDirLen ] = 0;

  if (!fhOut)
  {
    RepError( err_cantcreate, DIRECTORY_NAME );
  }
  i = (iInfiles + 1) * DRENTRYSIZE;

  if (fwrite( (char *)psigFile,1,(unsigned)i,fhOut) != i)
  {
    RepError( err_output2, NULL );
  }
  WriteRCFile( iPPBCount, uiOutDirLen );

  /*
  ** release all the allocated memory
  */
  FreeAll( );

  /*
  ** @V3.0129238
  ** Free the block list
  */
  DosFreeMem( desPpd.stUIList.pBlockList );
  DosFreeMem( desPpd.stUICList.puicBlockList );
  fclose( fhOut );
  /* @V3.0CMPS01 vvv */
  if ( fWriteWords )
  {
    fclose (ppdOut);
  }
  /* @V3.0CMPS01 ^^^ */

  return 0;
}


//@V4.Media
#define PUIB_IGNORE_MEDIA  0x00
#define PUIB_SEARCH_MEDIA  0x01
#define PUIB_MEDIA_FOUND   0x02
#define PUIB_CREATE_MEDIA  0x04
#define PUIB_MEDIA_CREATED 0x08

/*
** @V3.0129238 - From here to end - new function.
*/
VOID ProcessUIList( )
{
  PCHAR     pUICurrBlock;
  PCHAR     pEndToken;
  UINT      uiStrLen;
  SHORT     sDisplayOrder;
  UINT      uiLoop, uiLoop2;
  long      usFileLoc;
  PUI_BLOCK pUIBlock1, pUIBlock2;
  PUI_LIST  pUIList = &desPpd.stUIList;
  // @V3.0134840
  PCHAR pTemp;
  INT   ofsTemp;
  PSZ   pUICmd;
  USHORT usCMDLoop;
  USHORT    usFlags = PUIB_SEARCH_MEDIA;                       //@V4.Media

  BUI = TRUE;
  memset( USBlockHeader, 0, sizeof( USBlockHeader ) );


  if (pUIList->pBlockList != NULL)
  {
    pUICmd = (PSZ) szSearch[ OpenUI ];
    for (usCMDLoop = 0 ; usCMDLoop <= 1 ; usCMDLoop++)
    {
      pUICurrBlock = (PCHAR) pUIList->pBlockList;
      fbseek( &fbIn, (long) 0, SEEK_SET );

      while (ScanParam( pUICmd, pbPrBuffer ))
      {
        pEndToken = pbPrBuffer;
        while (*pEndToken != '/' && *pEndToken != ':')
        {
          pEndToken++;
        }
        uiStrLen = (UINT) (pEndToken - pbPrBuffer);

        /*
        ** Query the UI string list to see if the name already exists.
        ** Process the UI if the string does not exist.
        */
        for (uiLoop = 0 ; uiLoop < maximum ; uiLoop++)
        {
          if (!strncmp( pbPrBuffer, szSearch[ uiLoop ], uiStrLen ))
          {
            break;
          }
        }

        /*
        ** If uiLoop = maximum, then no matching strings were found in the
        ** above IF statement.
        */
        /*
        ** *IMPORTANT
        ** 'ColorAdjust' is used by Brother HS-1PS2.  The problem with this
        ** command is that the command string is more than 2K in size.
        ** Incrementing the buffer more than 2K broke other parts of the
        ** compiler.  Ignore this command, just for the time being.
        */
        if (uiLoop == maximum && strncmp( pbPrBuffer, "*ColorAdjust", 12 ))
        {
          USBlockHeader[ pUIList->usNumOfBlocks++ ] =
            pUIList->usBlockListSize;
          pUIBlock1 = (PUI_BLOCK) (pUICurrBlock + pUIList->usBlockListSize);

          usFileLoc = fbtell( &fbIn );
          if (usFileLoc > UIFileGroup.usFileStart &&
              usFileLoc < UIFileGroup.usFileEnd)
          {
            pUIBlock1->ucGroupType = UIGT_INSTALLABLEOPTION;
          }
          else
          {
            pUIBlock1->ucGroupType = UIGT_DEFAULTOPTION;
          }

          pUIList->usBlockListSize += (USHORT) ProcessUIBlock( pUIList,
                                                               pUIBlock1,
                                                               uiStrLen,
                                                               &usFlags );  //@V4.Media


          /* @V4.Media
          ** If last UI Block was InputSlot and there are MediaType
          ** simulate new UI Block MediaType
          */
          if ( usFlags & PUIB_MEDIA_FOUND )
          {
            if ( nErrorLogLevel<2 )
               printf(">>>WARNING...MediaType Simulation...\n");
            usFlags = PUIB_CREATE_MEDIA;

            USBlockHeader[ pUIList->usNumOfBlocks++ ] =
              pUIList->usBlockListSize;
            
            pUIBlock1 = (PUI_BLOCK) (pUICurrBlock + pUIList->usBlockListSize);
            pUIBlock1->ucGroupType = UIGT_DEFAULTOPTION;
            
            pUIList->usBlockListSize += (USHORT) ProcessUIBlock( pUIList,
                                                                 pUIBlock1,
                                                                 uiStrLen,
                                                                 &usFlags );
            usFlags = PUIB_MEDIA_CREATED;
          }

        }
      }

      pUICmd = (PSZ) szSearch[ JCLOpenUI ];
    }

    /*
    ** Now that the UI's have been processed, it is time to sort them.
    ** They will be sorted in order, from low to high value, taken from
    ** the "*OrderDependency" command in the UI block.  Each command
    ** is followed by a numeric value.
    */
    // @V3.0134840
    if (pUIList->usNumOfBlocks > 0)
    {
      sDisplayOrder = 0;
      for (uiLoop = 0 ; uiLoop < pUIList->usNumOfBlocks ; uiLoop++)
      {
        pUIBlock1 = (PUI_BLOCK) ((PCHAR) pUIList->pBlockList +
                                 USBlockHeader[ uiLoop ]);
        pUIBlock1->usDisplayOrder = (USHORT) sDisplayOrder++;

        /*
        ** @V3.1149556
        ** If "JCLResolution" is found, then treat it as a standard
        ** resolution.  Do this by converting the offset to point to
        ** "Resolution", rather than "JCLResoluton".
        */
        pTemp = (PCHAR) (pbItemsBuffer + pUIBlock1->ofsUIName);
        if (!strcmp( pTemp, "JCLResolution" ))
        {
          // Increment by 3 to skip past "JCL".
          pUIBlock1->ofsUIName += 3;
        }

        // @V3.1157420
        if (pUIBlock1->usUILocation == 0)
        {
          pUIBlock1->usUILocation = UI_ORDER_PROLOGSETUP;
        }
      }

      DosAllocMem( (PPVOID) &pTemp, 4096,
                   PAG_READ | PAG_WRITE | PAG_COMMIT );
      ofsTemp = 0;

      for (uiLoop = 0 ; uiLoop < pUIList->usNumOfBlocks ; uiLoop++)
      {
        for (uiLoop2 = 0 ; uiLoop2 < pUIList->usNumOfBlocks - 1 ; uiLoop2++)
        {
          pUIBlock1 = (PUI_BLOCK) ((PCHAR) pUIList->pBlockList +
                                   USBlockHeader[ uiLoop2 ]);
          pEndToken = (PCHAR) (pUIBlock1->ofsUIName + pbItemsBuffer);

          pUIBlock2 = (PUI_BLOCK) ((PCHAR) pUIList->pBlockList +
                                   USBlockHeader[ uiLoop2 + 1 ]);
          pEndToken = (PCHAR) (pUIBlock2->ofsUIName + pbItemsBuffer);

          if (pUIBlock1->usOrderDep > pUIBlock2->usOrderDep)
          {
            ofsTemp = USBlockHeader[ uiLoop2 ];
            USBlockHeader[ uiLoop2 ] = USBlockHeader[ uiLoop2 + 1 ];
            USBlockHeader[ uiLoop2 + 1 ] = ofsTemp;
          }
        }
      }

      ofsTemp = 0;
      for (uiLoop = 0 ; uiLoop < pUIList->usNumOfBlocks ; uiLoop++)
      {
        pUIBlock1 = (PUI_BLOCK) ((PCHAR) pUIList->pBlockList +
                                 USBlockHeader[ uiLoop ]);
        pUIBlock1->usOrderDep = (USHORT) uiLoop;
        pEndToken = (PCHAR) (pUIBlock1->ofsUIName + pbItemsBuffer);
        memcpy( pTemp + ofsTemp, pUIBlock1, QUERY_BLOCK_SIZE( pUIBlock1 ) );
        ofsTemp += QUERY_BLOCK_SIZE( pUIBlock1 );
      }

      memcpy( pUIList->pBlockList, pTemp, pUIList->usBlockListSize );

      DosFreeMem( pTemp );
    }

    pbPrBuffer[ 0 ] = 0;
  }

//  DosFreeMem( pUIList->pBlockList );
  BUI = FALSE;
}



UINT ProcessUIBlock( PUI_LIST pUIList, PUI_BLOCK pUIBlock, UINT uiStrLen, PUSHORT pusFlags )
{
  CHAR aTempDefName[ 31 ];
  INT  iLoop;
  UINT uiEndOfString;
  PCHAR pTransString;
#ifdef DEBUG
  PCHAR pStorage = pbItemsBuffer + cbBuffout;
  PCHAR pLine = pbPrBuffer + uiStrLen;
#endif
  PCHAR pCompare;
  CHAR  cGroupType;
  PCHAR pKeyword;                                    // @V4.COLL2FP
  
//PSZ   pTemp; NOTUSED
  BOOL  bInputSlot = FALSE;                      //@V4.Media
  BOOL  bSkipEntry;                              //@V4.Media

  cGroupType = pUIBlock->ucGroupType;
  memset( pUIBlock, 0, sizeof( UI_BLOCK ) - sizeof( UI_ENTRY ) );
  pUIBlock->ucGroupType = cGroupType;

  /* @V4.Media
  ** If PUIB_CREATE_MEDIA then simulating Media type
  */
  if ( *pusFlags & PUIB_CREATE_MEDIA )
  {
    bInputSlot = TRUE;
    if ((pUIBlock->ofsUIName = (USHORT) MatchKeywords( UINAME_MEDIATYPE, pUIList,
                                        pbItemsBuffer, TRUE )) == (USHORT) -1)
    {
      pUIBlock->ofsUIName = cbBuffout;
      cbBuffout += CopyString( pbItemsBuffer + cbBuffout,
                               UINAME_MEDIATYPE,
                               255,
                               CSTR_INCLUDE_SLASH );       // @V4.HexChar
    }

    pUIBlock->ucPanelID = UIP_PREDEF_FEATURE;
  }
  else
  {
    // @V4.COLL2FP 
    // Search for keyword to replace current UI block keyword
    pKeyword = SearchKeySubst( pbPrBuffer + 1 );
  
    if ((pUIBlock->ofsUIName = (USHORT) MatchKeywords( pKeyword, pUIList,
                                      pbItemsBuffer, TRUE )) == (USHORT) -1)
    {
      pUIBlock->ofsUIName = cbBuffout;
      pCompare = pbItemsBuffer + cbBuffout;
      cbBuffout += CopyString( pbItemsBuffer + cbBuffout, 
                               pKeyword,                   // @V4.COLL2FP 
                               255,
                               CSTR_INCLUDE_SLASH );       // @V4.HexChar
    }
    /*
    ** @V3.1151556
    ** If a name already exists, we still need to extract it to verify if it
    ** is a predefined UI or not.
    */
    else
    {
      pCompare = aTempDefName;
      pTransString = pbPrBuffer + 1;
      iLoop = 0;
  
      while (*pTransString != ' ' && *pTransString != ':')
      {
        aTempDefName[ iLoop ] = *pTransString;
        pTransString++;
        iLoop++;
      }
      aTempDefName[ iLoop ] = 0;
      pCompare = SearchKeySubst( aTempDefName );             // @V4.COLL2FP 
      
    }
    
    /* @V4.Media
    ** Search for UI Block=InputSlot if PUIB_SEARCH_MEDIA
    */
    if( ( *pusFlags & PUIB_SEARCH_MEDIA ) &&
        !strcmpi( UINAME_INPUTSLOT, pCompare ))
    {
       bInputSlot = TRUE;
    }

    //@V3.MEDIATYPE
    // see if this is a predefined feature
    pUIBlock->ucPanelID = UIP_OS2_FEATURE;
    for( iLoop = 0; iLoop < MAX_PREDEFINED ; iLoop++ )
    {
    //@V3.0FIX
    //  if( !strcmp( szPredefined[iLoop], pCompare ))
      if( !strcmpi( szPredefined[iLoop], pCompare ))
        pUIBlock->ucPanelID = UIP_PREDEF_FEATURE;
    }
    //@V3.MEDIATYPE end
  }

  /*
  ** Copy the translation string.
  */
  if (*(pbPrBuffer + uiStrLen) == '/' && pUIBlock->ucPanelID !=
      UIP_PREDEF_FEATURE)
  {
    if ((pUIBlock->ofsUITransString =
         (USHORT) MatchKeywords( pbPrBuffer + ++uiStrLen, pUIList,
                                 pbItemsBuffer, TRUE )) == (USHORT) -1)
    {
      pUIBlock->ofsUITransString = cbBuffout;
      cbBuffout += CopyString( pbItemsBuffer + cbBuffout,
                               pbPrBuffer + uiStrLen, 
                               255, 
                               CSTR_INCLUDE_SLASH | CSTR_HEX2CHAR); // @V4.HexChar
    }
  }
  else
  {
    /*
    ** Assign the UI name as the Translation string since a Translation
    ** string wasn't provided.  Ignore the asterisk ('*') for the
    ** Translation string.
    */
    // @V3.OEM This seems to work fine without ignoring the asterisk;
    //         adding 1 caused the first character of the UI name
    //         to be chopped of in cases where the PPD had a variable
    //         number of spaces before the '*'.  MovePast has been
    //         changed to handle this.
//  pUIBlock->ofsUITransString = pUIBlock->ofsUIName + 1;
    pUIBlock->ofsUITransString = pUIBlock->ofsUIName;
  }

  uiStrLen      += MovePast( pbPrBuffer + uiStrLen, ':' );
  uiStrLen      += SkipBlank( pbPrBuffer + uiStrLen );
  uiEndOfString  = MovePast( pbPrBuffer + uiStrLen, '\n' );

#ifdef DEBUG
  pStorage = pbItemsBuffer + cbBuffout;
  pLine = pbPrBuffer + uiStrLen;
#endif

  /*
  ** The UI has three types of selections: BOOLEAN, PICK ONE, and
  ** PICK MANY.
  */
  if (!strncmp( pbPrBuffer + uiStrLen, "Boolean", 7 ))
  {
    pUIBlock->usSelectType = (UCHAR) UI_SELECT_BOOLEAN;
  }
  else if (!strncmp( pbPrBuffer + uiStrLen, "PickMany", 8 ))
  {
    pUIBlock->usSelectType = (UCHAR) UI_SELECT_PICKMANY;
  }
  else
  {
    pUIBlock->usSelectType = (UCHAR) UI_SELECT_PICKONE;
  }

  /*
  ** Go to the next line.
  */
  // @V3.0FIX - Remove "+ 1;"
  uiStrLen += uiEndOfString;

  /*
  ** Run through the loop until a CLOSEUI is encountered.
  */
  do
  {
#ifdef DEBUG
    pStorage = pbItemsBuffer + cbBuffout;
    pLine = pbPrBuffer + uiStrLen;
#endif

    if (*(pbPrBuffer + uiStrLen) == 0 &&
        *(pbPrBuffer + uiStrLen + 1) == 0)
    {
      fbread( pbPrBuffer, 1, IBUFFSIZE, &fbIn );
      uiStrLen = 0;
    }

    /*
    ** Parse the appropriate key.
    */
    if (!strncmp( pbPrBuffer + uiStrLen, "*Default", 8 ))
    {
      uiStrLen += MovePast( pbPrBuffer + uiStrLen, ':' );
      uiStrLen += SkipBlank( pbPrBuffer + uiStrLen );

      uiStrLen += CopyString( aTempDefName, 
                              pbPrBuffer + uiStrLen, 
                              31, 
                              CSTR_NORMAL );           // @V4.HexChar
      /* @V4.Media      ----
      ** Tektronic printers have Transparent medias in InputSlot block.
      ** Parser should ignore them. Search for DefaultInputSlot: Papar or AutoSelect
      ** FYI - Kyocera printers DefaultInputSlot: Internal or PF30A
      */
      if( ( bInputSlot ) &&
          ( *pusFlags & PUIB_SEARCH_MEDIA ) &&
          ( ( !strcmp( aTempDefName, "Paper") ) ||
            ( !strcmp( aTempDefName, "AutoSelect") ) ) )
      {
        *pusFlags = PUIB_IGNORE_MEDIA;
        bInputSlot = FALSE;
      }

    }
    else if (!strncmp( pbPrBuffer + uiStrLen, "*OrderDependency", 16 ) ||
             !strncmp( pbPrBuffer + uiStrLen, "OrderDependency", 15 ))
    {
      uiStrLen += MovePast( pbPrBuffer + uiStrLen, ':' );
      uiStrLen += SkipBlank( pbPrBuffer + uiStrLen );

      pUIBlock->usOrderDep = atoi( pbPrBuffer + uiStrLen );

      uiStrLen      += MovePast( pbPrBuffer + uiStrLen, ' ' );
      uiStrLen      += SkipBlank( pbPrBuffer + uiStrLen );
      uiEndOfString  = MovePast( pbPrBuffer + uiStrLen, ' ' );

      if (!strncmp( pbPrBuffer + uiStrLen, "ExitServer", 10 ))
      {
        pUIBlock->usUILocation = UI_ORDER_EXITSERVER;
      }
      else if (!strncmp( pbPrBuffer + uiStrLen, "DocumentSetup", 13 ))
      {
        pUIBlock->usUILocation = UI_ORDER_DOCSETUP;
      }
      else if (!strncmp( pbPrBuffer + uiStrLen, "PageSetup", 9 ))
      {
        pUIBlock->usUILocation = UI_ORDER_PAGESETUP;
      }
      else if (!strncmp( pbPrBuffer + uiStrLen, "JCLSetup", 8 ))
      {
        pUIBlock->usUILocation = UI_ORDER_JCLSETUP;
      }
      else
      {
        pUIBlock->usUILocation = UI_ORDER_PROLOGSETUP;
      }
    }
    else if (!strncmp( pbPrBuffer + uiStrLen, "*CloseUI", 8 )     ||
             !strncmp( pbPrBuffer + uiStrLen, "*JCLCloseUI", 11 ) ||
             pUIBlock->usNumOfEntries >= 30                        )  //@V4.1200417
    {
      break;
    }
    else if (*(pbPrBuffer + uiStrLen) == '*' &&
             *(pbPrBuffer + uiStrLen + 1) != '?' &&
             *(pbPrBuffer + uiStrLen + 1) != '%' &&
             strncmp( pbPrBuffer + uiStrLen, "*End", 4 ))
    {
      uiStrLen += MovePast( pbPrBuffer + uiStrLen, ' ' );

      /* @V4.Media
      ** If in current block is InputSlot
      ** search keywords for predefined MediaType's
      ** Skip predefined entry if we are in real (first) InputSlot processing
      ** Skip other entry if we are simulating MediaType
      */
      bSkipEntry = FALSE;
      if ( bInputSlot )
      {
        if ( *pusFlags & PUIB_CREATE_MEDIA )
          bSkipEntry = TRUE;

        for( iLoop = 0; iLoop < MAX_PREDEF_MEDIA; iLoop++ )
        {
          if( !strncmp( pbPrBuffer + uiStrLen,
                        szPredefMedia[iLoop],
                        strlen(szPredefMedia[iLoop]) ) )
          {
            *pusFlags |= PUIB_MEDIA_FOUND;
            bSkipEntry = !bSkipEntry;
            break;
          }
        }
      }

      if ( bSkipEntry )
      {
         goto SkipEntry;
      }

      if ((pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsOption =
           (USHORT) MatchKeywords( pbPrBuffer + uiStrLen, pUIList,
                                   pbItemsBuffer, TRUE )) == (USHORT) -1)
      {
        pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsOption = cbBuffout;
        cbBuffout += CopyString( pbItemsBuffer + cbBuffout,
                                 pbPrBuffer + uiStrLen, 
                                 31, 
                                 CSTR_INCLUDE_SLASH );       // @V4.HexChar
      }

      pTransString = pbPrBuffer + uiStrLen;
      while (*pTransString != '/' && *pTransString != ':')
      {
        pTransString++;
      }

      if (*pTransString == '/')
      {
        if ((pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsTransString =
             (USHORT) MatchKeywords( ++pTransString, pUIList, pbItemsBuffer,
                                     TRUE )) == (USHORT) -1)
        {
          pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsTransString =
            cbBuffout;
          cbBuffout += CopyString( pbItemsBuffer + cbBuffout, 
                                   pTransString,
                                   40, 
                                   CSTR_NORMAL  | CSTR_HEX2CHAR | CSTR_EXCLUDE_DQUOTE ); // @V4.HexChar
        }
      }
      else
      {
        pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsTransString =
          pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsOption;
      }
      uiStrLen += MovePast( pbPrBuffer + uiStrLen, ':' );
      uiStrLen += SkipBlank( pbPrBuffer + uiStrLen );
// @V4.1200417
//// No needed that hex strings are done
////  if (pUIBlock->usUILocation == UI_ORDER_JCLSETUP)
////  {
////    pTemp = (PSZ) (pbPrBuffer + uiStrLen + 2);
////
////    while (*pTemp != '"')
////    {
////      pTemp++;
////    }
////
////    if (*(pTemp - 1) == '>')
////    {
////      *pTemp = ' ';
////      while (*pTemp != '<')
////      {
////        pTemp--;
////      }
////      *pTemp = '"';
////    }
////  }

      pUIBlock->uiEntry[ pUIBlock->usNumOfEntries ].ofsValue = cbBuffout;
      iLoop = CopyInQuote( pbItemsBuffer + cbBuffout, pbPrBuffer +
                           ++uiStrLen, FALSE, COMPRESS );
      cbBuffout += iLoop;

      if (pUIBlock->usUILocation == UI_ORDER_JCLSETUP)
      {
        *(pbItemsBuffer + cbBuffout - 1) = '\n';
        *(pbItemsBuffer + cbBuffout++) = 0;
      }

      if (*(pbPrBuffer + uiStrLen - 1) == '"')
      {
        while (*(pbPrBuffer + uiStrLen) != '"')
        {
          uiStrLen++;
        }
        uiStrLen++;
      }
      else
      {
        uiStrLen += iLoop;
      }

      pUIBlock->usNumOfEntries++;
    }

SkipEntry:                                                      //@V4.Media
    if (*(pbPrBuffer + uiStrLen) != '\n')
    {
      uiStrLen += MovePast( pbPrBuffer + uiStrLen, '\n' );
    }
    else
    {
      uiStrLen++;
    }
  } while (TRUE);

  /*
  ** Run through the list of entries in the current block and find
  ** the default entry name.  The reason     the list starts at the
  ** bottom is so if no matching names are found, the default is assigned
  ** to the first entry, which is the last one to be encountered in
  ** this loop.
  */
  for (iLoop = pUIBlock->usNumOfEntries - 1 ; iLoop >= 0 ; iLoop--)
  {
    /*
    ** If a match is found, get out of the loop.
    */
    if (iLoop == 0 ||
        !strcmp( pbItemsBuffer + pUIBlock->uiEntry[ iLoop ].ofsOption,
                 aTempDefName ) )
    {
//@V3.0FIX
//    pUIBlock->usDefaultEntry = (USHORT) (1 << iLoop);
      pUIBlock->usDefaultEntry = (USHORT) iLoop;
      break;
    }
  }

  if (pUIBlock->usNumOfEntries > 0)
  {
    iLoop = (INT) pUIBlock->usNumOfEntries - 1;
  }
  else
  {
    iLoop = 0;
  }

  return( sizeof( UI_BLOCK ) + (sizeof( UI_ENTRY ) * iLoop) );
}

/*
** @V4.COLL2FP 
** Function SearchKeySubst
** This function search pKeyToSearch in SRKeywords massive
** If founds returns Keyword need to replace with
** If not    returns pKeyToSearch
*/
PCHAR SearchKeySubst( PCHAR pKeyToSearch )
{
   INT      iLoop;
   CHAR     aKey[ MAX_PSIZE ];                 // Key string to compare
   
   CopyString( aKey, pKeyToSearch, MAX_PSIZE, CSTR_INCLUDE_SLASH );
   
   for ( iLoop = 0; SRKeywords[ iLoop ].szSearch ; iLoop++ )
   {
      if ( !strcmp( aKey, SRKeywords[ iLoop ].szSearch ) )
      {
         return SRKeywords[ iLoop ].szReplace;
      }
   }
   
   return pKeyToSearch;
}




INT MatchKeywords( PBYTE pKeyToMatch, PUI_LIST pUIList, PBYTE pPPBBuff,
                   BOOL bCompareEntries )
{
  CHAR      aKey[ MAX_PSIZE ];                 // Key string to compare
  INT       iListIndex;                        // List Index counter
  INT       iBlockIndex;                       // Block index counter
  PBYTE     pKeyword;                          // Current UI keyword/T.S.
  PUI_ENTRY pUIEntry;                          // UI entry pointer
  PUI_BLOCK pUIBlock = pUIList->pBlockList;    // UI Block pointer
  INT       iRC = -1;                          // Return code

  /*
  ** At this point, pKeyToMatch points to the current keyword, but it is not
  ** NULL-terminated.  Copy the keyword to a temporary buffer so that it is
  ** NULL-terminated.  It is not recommended to insert a 0 in pKeyToMatch since
  ** that pointer points to the actual PPD string data.
  */
  CopyString( aKey, pKeyToMatch, MAX_PSIZE, CSTR_INCLUDE_SLASH ); // @V4.HexChar

  /*
  ** Run through the UI list.
  */
  for (iListIndex = 0 ; iListIndex < (INT) pUIList->usNumOfBlocks - 1 ;
       iListIndex++)
  {
    pKeyword = (PBYTE) (pUIBlock->ofsUIName + pPPBBuff);

    /*
    ** Compare the current keyword with the string to match.
    ** Set the return code to the offset if a match is found.
    */
    if (!strcmp( pKeyword, aKey ))
    {
      iRC = (INT) pUIBlock->ofsUIName;
      break;
    }

    /*
    ** Compare the Translation String only if the T.S. is not the same as
    ** as the keyword for this has already been done above.
    */
    if (pUIBlock->ofsUIName != pUIBlock->ofsUITransString)
    {
      pKeyword = (PBYTE) (pUIBlock->ofsUITransString + pPPBBuff);
      if (!strcmp( pKeyword, aKey ))
      {
        /*
        ** @V3.1151556
        ** Copy T.S. offset, not UI name offset.
        */
//        iRC = (INT) pUIBlock->ofsUIName;
        iRC = (INT) pUIBlock->ofsUITransString;
        break;
      }
    }

    /*
    ** Compare entries, only if requested.
    */
    if (bCompareEntries == TRUE)
    {
      /*
      ** At this point, no match has been found.  At this time, compare the
      ** keywords and the Translation Strings for each entry in the current
      ** block.
      */
      for (iBlockIndex = 0 ; iBlockIndex < (INT) pUIBlock->usNumOfEntries ;
           iBlockIndex++)
      {
        pUIEntry = &pUIBlock->uiEntry[ iBlockIndex ];

        /*
        ** Compare the entry's keyword.
        */
        pKeyword = (PBYTE) (pUIEntry->ofsOption + pPPBBuff);
        if (!strcmp( pKeyword, aKey ))
        {
          iRC = (INT) pUIEntry->ofsOption;
          break;
        }

        /*
        ** Compare the entry's Translation String, provided that the offset to
        ** the T.S. is not the same as the keyword.
        */
        if (pUIEntry->ofsTransString != pUIEntry->ofsOption)
        {
          pKeyword = (PBYTE) (pUIEntry->ofsTransString + pPPBBuff);
          if (!strcmp( pKeyword, aKey ))
          {
            iRC = (INT) pUIEntry->ofsTransString;
            break;
          }
        }
      }
    }

    INCREMENT_BLOCK_PTR( pUIBlock );
  }

  return( iRC );
}





INT ProcessUIConstraints( )
{
  LONG       ofsBlock1 = 0;
  LONG       ofsBlock2 = 0;
  CHAR       aKeyword1[ MAX_PSIZE ];
  CHAR       aOption1[ MAX_PSIZE ];
  CHAR       aKeyword2[ MAX_PSIZE ];
  CHAR       aOption2[ MAX_PSIZE ];
  ULONG      uOptionBit1, uOptionBit2;
  PUI_BLOCK  pBlock1;
  PUI_BLOCK  pBlock2;
//PUIC_ENTRY pUICEntry1;  NOTUSED
//PUIC_ENTRY pUICEntry2;  NOTUESD
  INT        ofsOption = 0;
  PUIC_LIST  pUICList;
  PUIC_BLOCK pUICCompare;
  UINT       uiStrLen;
  INT        iUICIndex;
  PUI_LIST   pUIList = &desPpd.stUIList;
  PUIC_BLOCK pUICBlock = desPpd.stUICList.puicBlockList;
  INT        iRC = cbBuffout;

//PUIC_BLOCK pUICTemp;  NOTUSED

  pUICList = &desPpd.stUICList;
  pUICList->puicBlockList = pUICList->puicBlockList;

  fbseek( &fbIn, (LONG) 0, SEEK_SET );

  while (ScanParam( szSearch[ UIConstraints ], pbPrBuffer ))
  {
    /*
    ** Set all arrays to NULL.
    */
    aKeyword1[ 0 ] = aKeyword2[ 0 ] = aOption1[ 0 ] = aOption2[ 0 ] = 0;

    /*
    ** Retrieve the first keyword string.  The first keyword is required.
    */
    uiStrLen = MovePast( pbPrBuffer, ' ' );
    CopyWord( aKeyword1, pbPrBuffer + uiStrLen + 1 );

    /*
    ** Retrieve the first option string.  The option string is optioal.  An
    ** option string can be determined if the string does not begin with a '*'.
    */
    uiStrLen += MovePast( pbPrBuffer + uiStrLen, ' ' );
    if (*(pbPrBuffer + uiStrLen) != '*')
    {
      CopyWord( aOption1, pbPrBuffer + uiStrLen );
    }

    /*
    ** Retrieve the second keyword string.  This string is also required.
    */
    uiStrLen += MovePast( pbPrBuffer + uiStrLen, ' ' );
    if (*(pbPrBuffer + uiStrLen) != 0x0D && *(pbPrBuffer + uiStrLen) == '*')
    {
      CopyWord( aKeyword2, pbPrBuffer + uiStrLen + 1 );
    }

    /*
    ** Retrieve the second option string.  The option string is optioal.  An
    ** option string can be determined if the string does not begin with a '*'.
    */
    uiStrLen += MovePast( pbPrBuffer + uiStrLen, ' ' );
    if (*(pbPrBuffer + uiStrLen) != 0x0D)
    {
      CopyWord( aOption2, pbPrBuffer + uiStrLen );
    }

    /*
    ** From the keywords extracted above, query the respective UI blocks.
    */
    pBlock1 = QueryBlockFromKeyword( pUIList, pbItemsBuffer,
                                     SearchKeySubst ( aKeyword1 ), //@V4.COLL2FP 
                                     (PINT) &ofsBlock1 );
    pBlock2 = QueryBlockFromKeyword( pUIList, pbItemsBuffer,
                                     SearchKeySubst ( aKeyword2 ), //@V4.COLL2FP 
                                     (PINT) &ofsBlock2 );

    /*
    ** If at least one block is NULL, then the keyword string is invalid.  No
    ** processing needs to be done for invalid strings.
    */
    /*
    ** Do not include PageRegion in with the constraints.  Constraints are
    ** reserved for UI blocks that have matching controls in either Printer
    ** or Job Properties, while PageRegion has no matching control (the form
    ** listbox is managed by PageSize).
    */
    if (pBlock1 != NULL && pBlock2 != NULL &&
        strcmp( aKeyword1, "PageRegion" ) && strcmp( aKeyword2, "PageRegion" ) )
    {
      /*
      ** If a valid option string is provided for the first and second
      ** keywords, then query the zero-based offset (ofsOption) of that entry.
      ** From that offset, set the appropriate zero-based bit (uOptionBit).  If
      ** no option string exists for the appropriate keyword, then set all of
      ** the bits in uOptionBit to 1.  This is because the lack of an option
      ** string for a keyword means that *ALL* options apply.  Therefore, set
      ** all bits to 1.
      ** Again, this applies to both the first and second option (aOption1 and
      ** aOption2).
      */
      if (aOption1[ 0 ] != 0)
      {
        /*
        ** If the option string is invalid (no matching entry), then set
        ** the option bit to 0.
        */
        if (QueryEntryFromOption( pbItemsBuffer, pBlock1, aOption1,
                                  (PINT) &ofsOption ) != NULL)
        {
          uOptionBit1 = 1 << ofsOption;
        }
        else
        {
          uOptionBit1 = 0;
        }
      }
      else
      {
        /*
        ** @V3.1151556
        ** The default value should never be a constraint.
        */
//        uOptionBit1 = (ULONG) -1;
        uOptionBit1 = (ULONG) ~(1 << pBlock1->usDefaultEntry);
      }

      if (aOption2[ 0 ] != 0)
      {
        /*
        ** If the option string is invalid (no matching entry), then set
        ** the option bit to 0.
        */
        if (QueryEntryFromOption( pbItemsBuffer, pBlock2, aOption2,
                                  (PINT) &ofsOption ) != NULL)
        {
          uOptionBit2 = 1 << ofsOption;
        }
        else
        {
          uOptionBit2 = 0;
        }
      }
      else
      {
        /*
        ** @V3.1151556
        ** The default value should never be a constraint.
        */
//        uOptionBit2 = (ULONG) -1;
        uOptionBit2 = (ULONG) ~(1 << pBlock2->usDefaultEntry);
      }

      /*
      ** Just like the IF statement above for blocks, verify that the option
      ** selections are valid (not 0).  If any option is invalid, then the
      ** constraint does not apply.
      */
      if (uOptionBit1 != 0 && uOptionBit2 != 0)
      {
        pUICCompare = pUICList->puicBlockList;
        for (iUICIndex = 0 ; iUICIndex < (INT) pUICList->usNumOfUICs ;
             iUICIndex++)
        {
          if (pUICCompare->uicEntry1.ofsUIBlock == ofsBlock1 &&
              pUICCompare->uicEntry2.ofsUIBlock == ofsBlock2)
          {
            if (pUICCompare->uicEntry1.bOption == uOptionBit1)
            {
              pUICCompare->uicEntry2.bOption |= uOptionBit2;
              break;
            }
            else if (pUICCompare->uicEntry2.bOption == uOptionBit2)
            {
              pUICCompare->uicEntry1.bOption |= uOptionBit1;
              break;
            }
          }
          else if (pUICCompare->uicEntry1.ofsUIBlock == ofsBlock2 &&
                   pUICCompare->uicEntry2.ofsUIBlock == ofsBlock1)
          {
            if (pUICCompare->uicEntry1.bOption == uOptionBit2)
            {
              pUICCompare->uicEntry2.bOption |= uOptionBit1;
              break;
            }
            else if (pUICCompare->uicEntry2.bOption == uOptionBit1)
            {
              pUICCompare->uicEntry1.bOption |= uOptionBit2;
              break;
            }
          }
          pUICCompare++;
        }

        if (iUICIndex >= (INT) pUICList->usNumOfUICs ||
            pUICList->usNumOfUICs == 0)
        {
          pUICBlock = pUICList->puicBlockList + pUICList->usNumOfUICs++;

          pUICBlock->uicEntry1.ofsUIBlock = (USHORT) ofsBlock1;
          pUICBlock->uicEntry1.bOption    = uOptionBit1;

          pUICBlock->uicEntry2.ofsUIBlock = (USHORT) ofsBlock2;
          pUICBlock->uicEntry2.bOption    = uOptionBit2;
        }
      }
    }
  }

  return( iRC );
}





PUI_BLOCK QueryBlockFromKeyword( PUI_LIST pUIList, PBYTE pPSStringBuff,
                                 PBYTE pKeyword, PINT pofsBlock )
{
  UINT      uiLoop;                                   // Local loop counter
  PBYTE     pBlockName;                               // Name of current block
  PUI_BLOCK pInBlock = pUIList->pBlockList;           // Input block pointer
  PUI_BLOCK pUIBlockRC = NULL;                        // Block return code

  if (pofsBlock != NULL)
  {
    *pofsBlock = -1;
  }

  for (uiLoop = 0 ; uiLoop < pUIList->usNumOfBlocks ; uiLoop++)
  {
    pBlockName = pInBlock->ofsUIName + pPSStringBuff;

    if (!stricmp( pBlockName, pKeyword ))
    {
      pUIBlockRC = pInBlock;

      if (pofsBlock != NULL)
      {
        *pofsBlock = (INT) uiLoop;
      }

      break;
    }

    INCREMENT_BLOCK_PTR( pInBlock );
  }

  return( pUIBlockRC );
}





PUI_ENTRY QueryEntryFromOption( PBYTE pPSStringBuff, PUI_BLOCK pUIBlock,
                                PBYTE pOption, PINT pofsEntry )
{
  UINT      uiLoop;             // Local loop counter
  PBYTE     pEntryName;         // Current UI Entry name
  PUI_ENTRY pEntryRC = NULL;    // Current entry return code

  if (pofsEntry != NULL)
  {
    *pofsEntry = -1;
  }

  for (uiLoop = 0 ; uiLoop < pUIBlock->usNumOfEntries ; uiLoop++)
  {
    pEntryName = pUIBlock->uiEntry[ uiLoop ].ofsOption + pPSStringBuff;

    if (!strcmp( pOption, pEntryName ))
    {
      pEntryRC = &pUIBlock->uiEntry[ uiLoop ];

      if (pofsEntry != NULL)
      {
        *pofsEntry = (INT) uiLoop;
      }

      break;
    }
  }

  return( pEntryRC );
}





VOID ProcessCmdsAsUIs( )
{
  CHAR      aUIString[ MAX_PSIZE ];
  CHAR      aTempString[ MAX_PSIZE ];
  CHAR      aUIDefString[ MAX_PSIZE ];
  BOOL      bBlockProcessed;
  INT       iBlockIndex;
  INT       iStrLen;
  INT       iCurrEntry;
  INT       iIndex;
  PSZ       pTransString;
  INT       iVar;
  BOOL      fMatchFound;
  BOOL      fKeywordFound;
  PUI_BLOCK pUIBlock = desPpd.stUIList.pBlockList;
//USHORT    usOrderDep = 0; NOTUSED

  for (iIndex = 0 ; iIndex < (INT) desPpd.stUIList.usNumOfBlocks ; iIndex++)
  {
    INCREMENT_BLOCK_PTR( pUIBlock );
  }

  strcpy( aUIDefString, "*Default" );
  for (iIndex = 0 ; iIndex < MAX_PREDEFINED ; iIndex++)
  {
    fMatchFound = FALSE;
    if (!strcmp( szPredefined[ iIndex ], "JCLResolution" ))
    {
      if (QueryBlockFromKeyword( &desPpd.stUIList, pbItemsBuffer,
                                 UINAME_RESOLUTION, NULL ) != NULL)
      {
        fMatchFound = TRUE;
      }
    }

    if (QueryBlockFromKeyword( &desPpd.stUIList, pbItemsBuffer,
                               szPredefined[ iIndex ], NULL ) == NULL &&
        fMatchFound == FALSE)
    {
      iCurrEntry = 0;
      aUIString[ 0 ] = '*';
      strcpy( &aUIString[ 1 ], szPredefined[ iIndex ] );

      bBlockProcessed = FALSE;
      fKeywordFound = FALSE;
      fbseek( &fbIn, (long) 0, SEEK_SET );
      while (ScanParam( aUIString, pbPrBuffer ))
      {
        fKeywordFound = TRUE;
        iStrLen = 0;
        if (pUIBlock->ofsUIName == 0)
        {
          iBlockIndex = (INT) MatchKeywords( pbPrBuffer + 1, &desPpd.stUIList,
                                             pbItemsBuffer, TRUE );

          if (iBlockIndex == -1)
          {
            pUIBlock->ofsUIName = pUIBlock->ofsUITransString = cbBuffout;

            /*
            ** @V3.1155950
            ** The default location is Document Setup.  If not inserted,
            ** then the string is never sent out with the job.
            */
            pUIBlock->usUILocation = UI_ORDER_DOCSETUP;

            /*
            ** If found, then this is a pre-defined feature (do not display
            ** in "Features" page).
            */
            pUIBlock->ucPanelID = UIP_PREDEF_FEATURE;

            if (strcmp( szPredefined[ iIndex ], "SetResolution" ))
            {
              cbBuffout += CopyString( pbItemsBuffer + cbBuffout,
                                       szPredefined[ iIndex ], 
                                       MAX_PSIZE,
                                       CSTR_INCLUDE_SLASH );      // @V4.HexChar
            }
            else
            {
              cbBuffout += CopyString( pbItemsBuffer + cbBuffout,
                                       &szPredefined[ iIndex ][ 3 ], 
                                       MAX_PSIZE,
                                       CSTR_INCLUDE_SLASH );      // @V4.HexChar
            }
          }
          else
          {
            pUIBlock->ofsUIName = pUIBlock->ofsUITransString = iBlockIndex;
          }
        }

        pTransString = (PSZ) pbPrBuffer;
        while (*pTransString != ' ' && *pTransString != '/')
        {
          pTransString++;
        }

        if (*pTransString == '/')
        {
          *pTransString = 0;
        }
        else
        {
          pTransString = NULL;
        }

        /*
        ** Copy the Option.
        */
        iStrLen = CopyString( pbItemsBuffer + cbBuffout, 
                              pbPrBuffer,
                              80, 
                              CSTR_NORMAL );                    // @V4.HexChar
        pUIBlock->uiEntry[ iCurrEntry ].ofsOption = cbBuffout;

        cbBuffout += iStrLen;

        if (pTransString != NULL)
        {
          iVar = CopyString( pbItemsBuffer + cbBuffout,
                             pbPrBuffer + iStrLen, 
                             80, 
                             CSTR_NORMAL );                     // @V4.HexChar
          pUIBlock->uiEntry[ iCurrEntry ].ofsTransString = cbBuffout;
          cbBuffout += iVar;
          iStrLen += iVar;
        }
        else
        {
          pUIBlock->uiEntry[ iCurrEntry ].ofsTransString =
          pUIBlock->uiEntry[ iCurrEntry ].ofsOption;
        }

        /*
        ** skip the " character
        */
        do
        {
          iStrLen++;
        } while (*(pbPrBuffer + iStrLen - 1) != '"');

        /*
        ** copies the string delimited by a blank or quote.
        */
        iStrLen = CopyInQuote( pbItemsBuffer + cbBuffout,
                               pbPrBuffer + iStrLen, FALSE, NOCOMPRESS );
        pUIBlock->uiEntry[ iCurrEntry++ ].ofsValue = cbBuffout;
        cbBuffout += iStrLen;

        pUIBlock->usNumOfEntries++;
        bBlockProcessed = TRUE;
      }

      if (fKeywordFound == TRUE)
      {
        strcpy( &aUIDefString[ 8 ], szPredefined[ iIndex ] );
        fbseek( &fbIn, (long) 0, SEEK_SET );

        if (ScanParam( aUIDefString, pbPrBuffer ))
        {
          iStrLen = 0;
          while (*(pbPrBuffer + iStrLen) == ':' ||
                 *(pbPrBuffer + iStrLen) == ' ')
          {
            iStrLen++;
          }

          CopyWord( aTempString, pbPrBuffer + iStrLen );

          for (iBlockIndex = 0 ; iBlockIndex < pUIBlock->usNumOfEntries ;
               iBlockIndex++)
          {
            if (!strcmp( pUIBlock->uiEntry[ iBlockIndex ].ofsOption +
                         pbItemsBuffer, aTempString ))
            {
              pUIBlock->usDefaultEntry = (USHORT) iBlockIndex;
              break;
            }
          }
        }

        if (bBlockProcessed == TRUE)
        {
          desPpd.stUIList.usBlockListSize += QUERY_BLOCK_SIZE( pUIBlock );
          desPpd.stUIList.usNumOfBlocks++;
          INCREMENT_BLOCK_PTR( pUIBlock );
        }
      }
    }
  }

  pUIBlock = desPpd.stUIList.pBlockList;
  for (iVar = 0 ; iVar < desPpd.stUIList.usNumOfBlocks ; iVar++)
  {
    pUIBlock->usOrderDep = (USHORT) iVar;
    INCREMENT_BLOCK_PTR( pUIBlock );
  }
}




INT CopyWord( PCHAR pDestn, PCHAR pSource )
{
  INT iStrLen = 0;

  while (!isspace( *pSource ) && *pSource != ':')
  {
    *(pDestn++) = *(pSource++);
    iStrLen = 0;
  }
  *pDestn = 0;

  return( iStrLen );
}





BOOL ParsePPDLang( PSZ pFileLine,
                   PSZ pOutString
                 )
{
  PSZ  pColonChar;
  PSZ  pParseChar;
  BOOL fRC = FALSE;

  *pOutString = 0;

  if ((pColonChar = strchr( pFileLine, ':' )) != NULL)
  {
    pParseChar = pColonChar + 1;
    while (*pParseChar == ' ')
    {
      pParseChar++;
    }

    while (*pParseChar != '\n' && *pParseChar != ' ' && *pParseChar != 0)
    {
      *(pOutString++) = *(pParseChar++);
    }
    *pOutString = 0;

    while (*pColonChar != 'L' && pColonChar >= pFileLine)
    {
      pColonChar--;
    }
    *pColonChar = 0;

    fRC = TRUE;
  }

  return( fRC );
}





VOID VerifyUICList( )
{
  LONG       lLoop;
  PUIC_BLOCK pUICBlock = desPpd.stUICList.puicBlockList;

  for (lLoop = 0 ; lLoop < (LONG) desPpd.stUICList.usNumOfUICs ; lLoop++)
  {
    if (pUICBlock->uicEntry1.bOption == (UI_SEL) -1)
    {
      pUICBlock->uicEntry1.bOption = 0;
    }

    if (pUICBlock->uicEntry2.bOption == (UI_SEL) -1)
    {
      pUICBlock->uicEntry2.bOption = 0;
    }

    pUICBlock++;
  }
}





LONG CountFilesFound(
)
{
  LONG lLoop = 0;
  LONG lRC = 0;

  while (apszInfiles[ lLoop ] != NULL)
  {
    if (apszInfiles[ lLoop++ ][ 0 ] != 0)
    {
      lRC++;
    }
  }

  return( lRC );
}
