/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*****************************************************************************
 *
 * SOURCE FILE NAME = JPDLG.C
 *
 * DESCRIPTIVE NAME = Job Properties Dialog Box
 *
 *
 * VERSION = V1.0
 *
 * DATE
 *
 * AUTHORS      Debbie Cook, Larry Moore
 *
 * DESCRIPTION  Contains functions used to maintain the Job Properties
 *              dialog box for PostScript level 2 support.
 *
 *
 * FUNCTIONS
 *            FeaturePageDlgProc
 *            FeaturePageInitControls
 *            FeaturePageSaveSettings
 *            DisplayFormSize
 *            EffectsPageDlgProc
 *            EffectsPageInitControls
 *            EffectsPageSaveSettings
 *            FormPageDlgProc
 *            FormPageInitControls
 *            FormPageSaveSettings
 *            JobPropertiesDlgProc
 *            OutputPageDlgProc
 *            OutputPageInitControls
 *            OutputPageSaveSettings
 *
 *
 * ORDER OF FUNCTIONS
 *            Entry Point:
 *            - JobPropertiesDlgProc - PM procedure for Job Properties dialog.
 *
 *            Functions/Procedures for Notebook pages (in page order):
 *              - FormPageDlgProc - PM procedure for the Form Page in the
 *                Job Properties notebook.
 *                - FormPageInitControls - Initialize the controls for the
 *                  Forms page.
 *                - FormPageSaveSettings - Saves current settings in the page.
 *                - DisplayFormSize - Displays the currently selected form
 *                  size.
 *              - OutputPageDlgProc - PM procedure for the Output page.
 *                - OutputPageInitControls - Initializes the controls for the
 *                  Output page.
 *                - OutputPageSaveSettings - Saves the current page settings.
 *              - EffectPageDlgProc - PM procedure for the Effects page.
 *                - EffectsPageInitControls - Initializes the controls for the
 *                  page.
 *                - EffectsPageSaveSettings - Saves the current page settings.
 *              - FeaturePageDlgProc - PM procedure for the Feature page.
 *                - FeaturePageInitControls - Initializes page controls.
 *                - FeaturePageSaveSettings - Saves the current page settings.
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/
#pragma pack(1)
#define  INCL_PM
#define  INCL_TKTP
#define  INCL_GENPLIB_ERROR
#define  INCL_GENPLIB_LAYOUT
#define  INCL_GENPLIB_GPLUTIL
#define  INCL_GENPLIB_DIALOGPAGES
#include <os2.h>
#include <string.h>
#include <stdlib.h>
#include <genplib.h>
#include "inc\ppdialog.h"
#include "inc\dlgproc.h"
#include "inc\uinames.h"
#include "inc\profile.h"
#include "inc\config.h"
#include "inc\ppdialog.h"
#include "inc\pspagtun.h"
#include "inc\dlg.h"
#include "inc\enumpage.h"
// @V3.1141419
#ifndef DDKBLD
#include "inc\oem.h"
#include "inc\os2_oem.h"
#endif

BOOL VerifyUISelectionList( PCNFDATA );
MRESULT APIENTRY LayoutCallbackProc( PGLCB, PVOID );


/****************************************************************************\
** DEFINES START                                                            **
\****************************************************************************/
#define NUM_OF_PAGES           4       // Number of notebook pages

#define  DPDM_USERPROPERTIES   3

/*
** These defines make up the tick information for the Gamma Correction slide
** bar.  For more information on ticks, refer to the Programming Guide, vol 2.
** GC_SLIDE_HOME_VAL is the value of the home position.
** GC_SLIDE_INCREMENT_VALUE is the increment value from one tick to another.
**   For example, if this is 2 and GC_SLIDE_MIN_VAL is -10, then the next tick
**   mark to the right of GC_SLIDE_MIN_VALUE is -8.  Fractional values are
**   allowed, but limited from 0.1 to 0.9.
** GC_SLIDE_NUM_OF_TICKS is ((GC_SLIDE_MAX_VAL - GC_SLIDE_MIN_VAL) *
**   GC_SLIDE_INCREMENT_VALUE) + 1 (the '+ 1' is to include 0).  This must
**   be hard coded because the resource compiler can't handle math.
** GC_SLIDE_MAJOR_TICK_PIXEL_LEN and GC_SLIDE_MINOR_TICK_PIXEL_LEN
**   deterine the pixel length of the ticks.  The pixel length is from the bar
**   to the numeric value above the tick.  The major tick is always twice as
**   long as the minor tick.
** GC_SLIDE_MAJOR_TICK_PLACE determines the pattern for placing major ticks.
**   If the value is 1, every tick is major.  If 2, every second tick is major.
**   If 3, every third tick is major, etc.  The pattern algorithm is to use
**   modular division.  If the result is 0, a major tick is inserted.
**   Otherwise, a minor tick is inserted.
**   ** This must be an integer value. **
**   ** No error checking is performed on these values. **
*/
#define GC_SLIDE_HOME_VAL              1
#define GC_SLIDE_INCREMENT_VALUE       1   // Increment value of ticks
#define GC_SLIDE_NUM_OF_TICKS          50
#define GC_SLIDE_MAJOR_TICK_PIXEL_LEN  6   // Length of tick mark from bar to #
#define GC_SLIDE_MINOR_TICK_PIXEL_LEN  (INT) (GC_SLIDE_MAJOR_TICK_PIXEL_LEN / 2)
#define GC_SLIDE_MAJOR_TICK_PLACE      10  // Pattern where maj. ticks are used

#define SETFLAG( flag, bit )   ((flag) |= (ULONG)(bit))
#define CLEARFLAG( flag, bit ) ((flag) &= ~((ULONG)(bit)))
#define CHECKFLAG( flag, bit ) ((flag) & (ULONG)(bit))

/****************************************************************************\
** DEFINES END                                                              **
\****************************************************************************/


/****************************************************************************\
** USER DEFINED MESSAGES FOR THE JOB PROPERTIES DIALOG BOX                  **
\****************************************************************************/
/*
** WPJP_CHANGEICON - Sent to the Job Properties dialog box to change an
** icon for a given control.  mp1 should contain the control.  mp2 should
** be the return code for a check-box-query-check message.
**
** WMPS_INITSETTINGS - This is sent to all dialog pages in the Job
** Properties notebook when the window is initialized.  This is so that
** the dialog page initializes all its controls from the given Job
** Properties Information Structure.  The difference between this and
** WM_INITDLG is that WM_INITDLG should not be accessing the info structure,
** since it has not been yet initialized at that time.  WM_INITDLG can be
** used to set general parameters for controls (i.e. set min/max numeric
** values in spin buttons.  WMPS_INITSETTINGS should initialize controls
** that it reads from the info structure.
**
** WMPS_SAVESETTINGS - This is sent to all dialog pages from the Job
** Properties dialog box to indicate to save all settings pertaining to that
** dialog page.  This is sent only if "OK" is selected.
**
** WMPS_VERIFYSETTINGS - This is sent to all pages before the data is saved.
** When received, the pages must verify that their data is correct.  If all
** is OK (or if no verification is needed), the page must return TRUE.  If
** the data is not correct, the page must return FALSE.  Each page is
** responsible for handling its own error recovery.
*/
#define COLOR_INDEX              0
#define MONOCHROME_INDEX         1
#define REFRESH_DISPLAY          0
#define SAVE_SETTINGS            1
/****************************************************************************\
** USER DEFINED MESSAGE END                                                 **
\****************************************************************************/





/****************************************************************************\
** TYPEDEFS START                                                           **
\****************************************************************************/
typedef PDLGHDR PJPINFO;
/****************************************************************************\
** TYPEDEFS END                                                             **
\****************************************************************************/


/****************************************************************************\
** FUNCTION PROTOTYPES START                                                **
\****************************************************************************/
MRESULT EXPENTRY JobPropertiesDlgProc( HWND, ULONG, MPARAM, MPARAM );
MRESULT EXPENTRY FormPageDlgProc( HWND, ULONG, MPARAM, MPARAM );
MRESULT EXPENTRY OutputPageDlgProc( HWND, ULONG, MPARAM, MPARAM );
MRESULT EXPENTRY EffectsPageDlgProc( HWND, ULONG, MPARAM, MPARAM );
VOID FormPageInitControls( HWND, PJPINFO, INT );
VOID OutputPageInitControls( HWND, PJPINFO, INT );
VOID EffectsPageInitControls( HWND, PJPINFO, INT );
UINT FormPageSaveSettings( HWND, PJPINFO );
UINT OutputPageSaveSettings( HWND, PJPINFO );
UINT EffectsPageSaveSettings( HWND, PJPINFO );
VOID DisplayGammaValue( HWND, ULONG, LONG );
VOID InsertSliderVals( HWND, ULONG);
VOID GetOffset ( HWND, ULONG, BOOL);
VOID SaveGammaVals (HWND, ULONG, PLONG);
BOOL FixUpColors (INT ,PINT, PINT, PINT, PINT, PINT);
VOID FeaturePageDefaultControls( HWND, PJPINFO, BOOL );
VOID FormPageDefaultControls( HWND, PJPINFO );
VOID EffectsPageDefaultControls( HWND, PJPINFO );
VOID OutputPageDefaultControls( HWND, PJPINFO );
LONG QueryTrayArray( PDLGHDR, PSZ );
VOID SetJobpropPageData( HWND, PDLGHDR );
PGLCB InitNUPDlg( PDLGHDR );
VOID SaveNUPDlg( PDLGHDR );
/****************************************************************************\
** FUNCTION PROTOTYPES END                                                  **
\****************************************************************************/


/****************************************************************************\
** EXTERNAL DEFINITIONS START                                               **
\****************************************************************************/

extern FIXED frdiv( FIXED, FIXED );
extern FIXED frmul( FIXED, FIXED );
extern VOID QueryUserForm( PCNFDATA );
//extern VOID DisplayGammaValue( HWND, ULONG, LONG );
extern BOOL ValidFilename( PSZ );

extern PBYTE QueryUIOptionString( PDESPPD, PUI_SEL, PBYTE, PINT, PUI_BLOCK * );
extern INT SetUIOption( PDESPPD, PUI_SEL, PBYTE, PBYTE );
extern BOOL _System szIsEqual( PSZ, PSZ );
extern PSZ GetDefaultPageSize( PDESPPD, PSZ );
extern VOID SaveCurrentUISelections( PDESPPD, PCNFDATA );
extern PUI_BLOCK QueryBlockFromKeyword( PUI_LIST, PBYTE, PBYTE, PINT );
// @V3.1142412
VOID PrintErrorHandler( PCNFDATA );
extern PGJNCB QueryLayoutCBFromCNFDATA( PCNFDATA );
extern SHORT *PaperDimensions( CHAR *, PDESPPD );
/****************************************************************************\
** EXTERNAL DEFINITIONS END                                                 **
\****************************************************************************/


/****************************************************************************\
** GLOBAL VARIABLES START                                                   **
\****************************************************************************/
extern PSZ FontResourceName[];

BOOL   bNewChange = TRUE;
INT    iOldRed;
INT    iOldGreen;
INT    iOldBlue;
INT    iNumOfPages;
//HWND   hNotebook;

/*
** This table is used to initialize the notebook for the Job Properties.
** The fields are defined as follows:
** - uiDialogID - ID of dialog box (in PSCRIPT.DLG) that is to be displayed
**   in a notebook page.
** - pfnDlgProc - Procedure to the dialog that is called when the dialog
**   page is active.
** - usTabID - Notebook tab ID for the dialog page.  It is used as an ID to
**   the tab string in the string table that is to be inserted in the dialog
**   page's tab.
** - usTabType - The type of tab that is to be inserted.  This should either
**   be BKA_MAJOR or BKA_MINOR.
**
** No error checking is performed on this table.
**
** The list define's below the structure are the index numbers for each of
** the given page.  If the pages are moved, these defines must also be updated
** as well.
*/
GDPTMPL DlgJPTmplTable[ NUM_OF_PAGES ] =
{
  JPNB_DLG_FORM,   (PFNWP) FormPageDlgProc,    JPF_TAB,  GTAB_ID, 0,
  JPNB_DLG_OUTPUT, (PFNWP) OutputPageDlgProc,  JPO_TAB,  GTAB_ID, 0,
  JPNB_DLG_EFFECTS,(PFNWP) EffectsPageDlgProc, JPE_TAB,  GTAB_ID, 0,
  JPNB_DLG_UI,     (PFNWP) FeaturePageDlgProc, JPUI_TAB, GTAB_ID, 0
};
#define JP_FORM_PAGE_INDEX       0
#define JP_OUTPUT_PAGE_INDEX     1
#define JP_EFFECTS_PAGE_INDEX    2
#define JP_FEATURE_PAGE_INDEX    3
/****************************************************************************\
** GLOBAL VARIABLES END                                                     **
\****************************************************************************/





/******************************************************************************
 *
 * FUNCTION NAME = JobPropertiesDlgProc
 *
 * DESCRIPTION
 * This is the procedure for the Job Properties dialog box.  All messages
 * concerning the dialog are received here.
 * This dialog box processes the WMJP_CHANGEICON message to modify the current
 * icon display for the status bar.
 *
 * *NOTE - CHANGING THE ICONS IN THE STATUS BAR (WMJP_CHANGEICON message)*
 * Currently, the dialog contains an icon control for each icon supported
 * see PSCRIPT.DLG).  The current icon is hidden and the new one is
 * displayed.  Another way of handling this is query the window rectangle,
 * load the icon pointer, fill the rectangle with the dialog background
 * color (WinFillRect), and set the new icon handle (SM_SETHANDLE).  The
 * reason     the former is done that it takes up less code than the latter.
 * A WM_PAINT message is needed here to re-draw the icons.
 * The above text was written as a record for future use.  Below is an
 * algorithm:
 *   WinQueryWindowRect( icon handle, &icon rectangle );
 *   Icon PS = WinGetPS( icon handle );
 *   Icon memory handle = WinLoadPointer( HWND_DESKTOP, icon ID );
 *   WinFillRect( Icon PS, &icon rectangle, SYSCLR_DIALOGBACKGROUND );
 *   WinSendMsg( icon handle, SM_SETHANDLE, icon memory handle );
 *   WinDestroyPointer( icon handle );
 *
 * INPUT
 * hDlg - Dialog handle
 * ulMsg - Message ID
 * mp1, mp2 - General message parameters.  mp2 contains the DC structure
 *  passed in from the driver.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 *                 FALSE
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
MRESULT EXPENTRY JobPropertiesDlgProc( HWND hDlg, ULONG ulMsg, MPARAM mp1,
                                       MPARAM mp2 )
{
  UINT     uiLoop;                         // Loop counter
  BOOL     bSegRC;                         // Return code from LoadInfoSegment
  PJPINFO  pJPInfo;                        // Job prop. info structure pointer
  PB      *apResources;                    // Poiner to printer resources
  HWND     hIcon;                          // Status Bar Icon handle
  USHORT   usInSBIcon;                     // Incomming ID for Status Bar
  USHORT   usInSBGroup;                    // Group ID for Status Bar
  MRESULT  mRC = (MRESULT) FALSE;          // Return code
  INT      iOEMPages = 0;  
  GDPTMPL  stOEMPrinters;
  PVOID    pPtr;

  switch( ulMsg )
  {
  case WM_INITDLG:
       /*
       ** Initialize the Job Properties Information structure.
       */
       apResources = (PB *) mp2;

#ifndef DDKBLD
       // @V3.1141419
       memset( (PCHAR) &stOEMPrinters, 0, sizeof( stOEMPrinters ) );

       iOEMPages = FindOEMPrntr( hDlg, apResources, &stOEMPrinters);
#endif

       if ((pJPInfo = (PJPINFO) InitializeDialogHeader( hDlg, mp2,
                                NUM_OF_PAGES + iOEMPages)) != NULL)
       {
         InitializeDialog( hDlg, pJPInfo, DPDM_POSTJOBPROP, DlgJPTmplTable,
                           &stOEMPrinters, NUM_OF_PAGES, iOEMPages,
                           JP_FEATURE_PAGE_INDEX );

//         pPtr = (PVOID) QueryLayoutCBFromCNFDATA( pJPInfo->pCNFData );

//         pJPInfo->hLayoutInstance = GplLayoutCreatePage( pJPInfo->hNBInstance,
//                                    (PGLCB) pPtr, NULL, NULL, 0 );
           InitNUPDlg( pJPInfo );
       }
       else
       {
         GplErrSetError( PMERR_HEAP_OUT_OF_MEMORY );
         WinDismissDlg( hDlg, FALSE );
       }
       break;

  case WM_PAINT:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
       UpdateIcon( 0, hDlg, JPSB_PORTRAIT, 0 );
       UpdateIcon( 0, hDlg, JPSB_MONOCHROME, 0 );
       UpdateIcon( 0, hDlg, JPSB_DUPLEX_NONE, 0 );
       UpdateIcon( 0, hDlg, JPSB_OUTPUT_PRINTER, 0 );
       break;

  case WM_COMMAND:
       pJPInfo = (PJPINFO) WinQueryWindowULong( hDlg, QWL_USER );

       switch( SHORT1FROMMP( mp1 ) )
       {
       // OK - Exit dialog box, save settings.
       case PSB_OK:
            if ( SaveDialogSettings(WinWindowFromID(hDlg, PS_NOTEBOOK_GROUP)) )
            if (VerifyAndSaveSettings( pJPInfo ) == PMERR_OK)
            {
              GplLayoutSaveSettings(pJPInfo->hLayoutInstance);
              SaveNUPDlg(pJPInfo);
              GplLayoutDestroyPage(0, pJPInfo->hLayoutInstance);

              pJPInfo->pCNFData->uSaveOptions = DEV_OK;
              ClosePSDlg( hDlg, pJPInfo, TRUE );
            }
            break;

       // CANCEL - exit dialog box, do not save settings.
       case DID_CANCEL:
       case PSB_CANCEL:
            pJPInfo->pCNFData->uSaveOptions = DPDM_NONE;
            GplLayoutDestroyPage(0, pJPInfo->hLayoutInstance);

            ClosePSDlg( hDlg, pJPInfo, FALSE );
            break;

       case PSB_HELP:
            // PUT SOME HELP CODE IN HERE!!!!!
            break;

       }        /* end switch */
       break;

  case WM_CLOSE:
       /*
       ** The OK button ID already contains code to save the current data.
       ** Rather than duplicate the code, simply send an OK message since the
       ** processing for both messages is the same.
       */
       WinSendMsg( hDlg, WM_COMMAND, (MPARAM) PSB_OK, (MPARAM) 0 );
       break;

  case WM_CONTROLPOINTER:
       mRC = SetContextHelp( hDlg, ulMsg, mp1, mp2, NULL, 0, 0 );
       break;
       
  case WM_HITTEST:
       // Simulate WM_CONTROLPOINTER for controls such as spinbutton.
       GplHitTestHandler(hDlg, mp1, mp2);
       break;

  default:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
  }            /* end switch */

  return( mRC );
}
/*---------------------------------------------------------------------------*\
* JobPropertiesDlgProc End                                                    *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = FormPageDlgProc
 *
 * DESCRIPTION
 * This is the procedure for the Forms dialog box that is in the Job
 * Properties notebook.  Among the standard procedure responsibilites, this
 * procedure initializes and saves all data pertaining to the dialog, with
 * regard to the Job Properties Information structure.
 *
 * INPUT
 * hDlg - Dialog handle
 * ulMsg - Message ID
 * mp1, mp2 - General message parameters
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 *                 FALSE
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
MRESULT EXPENTRY FormPageDlgProc( HWND hDlg, ULONG ulMsg, MPARAM mp1,
                                  MPARAM mp2 )
{
  PDLGHDR    pJPInfo;                 // Job Properties info structure pointer
  HWND       hCtrl;                   // Local control handle
  SHORT      sIndex;                  // Listbox item index
  SHORT      sHandle;                 // Listbox item handle
  FORMSTRUCT stForms;
  MRESULT    mRC = (MRESULT) FALSE;   // Function return code
//@V3.130814
  USHORT     usOldIcon;
//@V3.130814 end
  PSZ        pDuplexSelected;
  BOOL       fUpdateIcon;
  INT        iStatusBar;
  INT        iIconID;
  PUI_BLOCK  pUIBlock;
  BOOL       bMixedFeed;

  if (ulMsg != WM_INITDLG)
  {
    pJPInfo = (PDLGHDR) WinQueryWindowULong( hDlg, QWL_USER );
  }

  switch( ulMsg )
  {
  // @V3.1142412
  case WM_INITDLG:
       /*
       ** Initialize the numerical limits of the spin button.
       */
       WinSendDlgItemMsg( hDlg, JPF_COPIESNUM, SPBM_SETLIMITS,
                          MPFROMLONG( 999 ), MPFROMLONG( 1 ) );
       WinSendDlgItemMsg( hDlg, JPF_COPIESNUM, SPBM_SETTEXTLIMIT,
                          MPFROMLONG( 3 ), (MPARAM) 1 );

       pJPInfo = InitDlgULong( hDlg, mp2 );
       FormPageInitControls( hDlg, pJPInfo, UCD_INITIALIZEPAGE );
       break;

  case WM_PAINT:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
       UpdateIcon( hDlg, 0, JPSB_PORTRAIT, 0 );
       // @V3.1142412
       UpdateIcon( hDlg, 0, JPSB_DUPLEX_NONE, 0 );
       break;

  /*
  ** This must be here so that FALSE is returned.  If not, the dialog box
  ** in the notebook will disappear if the user presses ESC.
  */
  case WM_COMMAND:
       switch ( SHORT1FROMMP( mp1 ) )
       {
       case PSB_UNDO:
         FormPageInitControls( hDlg, pJPInfo, UCD_UNDOPAGE );
         break;

       case PSB_DEFAULT:
         FormPageDefaultControls( hDlg, pJPInfo );
         break;

       case DID_CANCEL:
         WinSendMsg( pJPInfo->hDlg, WM_COMMAND, (MPARAM) PSB_CANCEL, (MPARAM) 0 );
       }
       break;

  case WM_CONTROL:
       // @V3.1142412
       fUpdateIcon = FALSE;
       pDuplexSelected = NULL;

       switch( SHORT1FROMMP( mp1 ) )
       {
       /*
       ** For the following controls, send a message to the status bar
       ** to change to the respective icons.
       */
       case JPF_FORMNAMELIST:
            if (SHORT2FROMMP( mp1 ) == CBN_EFCHANGE)
            {
              SetJobpropPageData( hDlg, pJPInfo );
            }
            break;
            
       
       case JPF_MIXED_FEED:                                        // @MIXED_FEED Begin
            if (SHORT2FROMMP( mp1 ) == BN_CLICKED)
            {
              bMixedFeed = WinQueryButtonCheckstate( hDlg, JPF_MIXED_FEED );

              // if mp2=0 simulated message, don't change state
              if ( mp2 )
              {
                bMixedFeed = bMixedFeed ? FALSE : TRUE;
                WinSendDlgItemMsg( hDlg, JPF_MIXED_FEED,
                                   BM_SETCHECK,
                                   MPFROMSHORT( bMixedFeed ),
                                   (MPARAM) 0 );
              }

              // enable/disable Mixed Feed controls
              for( sIndex = JPF_FORMNAMETEXT_1; sIndex <= JPF_MEDIATYPELIST_1; sIndex++)
              {
                //WinShowWindow( WinWindowFromID( hDlg, sIndex ), bMixedFeed );
                // If printer don't have mediatypes, also disable 1st media controls
                if ( ( sIndex == JPF_MEDIATYPELIST_1 || sIndex == JPF_MEDIATYPETEXT_1 ) &&
                     WinIsWindowEnabled( WinWindowFromID( hDlg, JPF_MEDIATYPELIST ) ) == FALSE )
                  WinEnableControl( hDlg, sIndex, FALSE );
                else
                  WinEnableControl( hDlg, sIndex, bMixedFeed );
              }
              
            }
            break;                                                // @MIXED_FEED End

       case JPF_PORTRAIT:
            iStatusBar = JPSB_PORTRAIT;
            iIconID = ICON_PORTRAIT;
            fUpdateIcon = TRUE;
            mp1 = (MPARAM) GJT_PORTRAIT;
            GplJFUpdateFormSettings( pJPInfo->hNBInstance, &mp1,
                                     GJMF_ORIENTATION, 1 );
            break;

       case JPF_LANDSCAPE:
            iStatusBar = JPSB_PORTRAIT;
            iIconID = ICON_LANDSCAPE;
            fUpdateIcon = TRUE;
            mp1 = (MPARAM) GJT_LANDSCAPE;
            GplJFUpdateFormSettings( pJPInfo->hNBInstance, &mp1,
                                     GJMF_ORIENTATION, 1 );
            break;

       case JPF_DUPLEXNONE:
            fUpdateIcon = TRUE;
            iStatusBar = JPSB_DUPLEX_NONE;
            iIconID = ICON_DUPLEX_NONE;
            pDuplexSelected = UINAME_DUPLEXNONE;
            break;

       case JPF_DUPLEXNOTUMBLE:
            fUpdateIcon = TRUE;
            iStatusBar = JPSB_DUPLEX_NONE;
            iIconID = ICON_DUPLEX_BOOK;
            pDuplexSelected = UINAME_DUPLEXNOTUMBLE;
            break;

       case JPF_DUPLEXTUMBLE:
            fUpdateIcon = TRUE;
            iStatusBar = JPSB_DUPLEX_NONE;
            iIconID = ICON_DUPLEX_FLIP;
            pDuplexSelected = UINAME_DUPLEXTUMBLE;
            break;
       }

       if (pDuplexSelected != NULL)
       {
         sIndex = (SHORT) SetUIOption( pJPInfo->pdesPPD,
                                       pJPInfo->pCurrUISelList, UINAME_DUPLEX,
                                       (PBYTE) pDuplexSelected );

         if (!(pJPInfo->usFlags & UIC_NO_UPDATE))
         {
           pUIBlock = QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList,
                                             pJPInfo->pItemsBuff,
                                             UINAME_DUPLEX, NULL );
           SendConstraintsMessages( pJPInfo, (INT) SHORT1FROMMP( mp1 ),
                                    (INT) sIndex, pUIBlock );
         }
       }

       if (fUpdateIcon == TRUE)
       {
         UpdateIcon( hDlg, pJPInfo->hDlg, iStatusBar, iIconID );
       }
       break;

  case WMPS_VERIFYSETTINGS:
       mRC = (MRESULT) PMERR_OK;
       break;

  case WMPS_SAVESETTINGS:
       /*
       ** 'OK' was pressed.  Save the settings.
       */
       mRC = (MRESULT) FormPageSaveSettings( hDlg, pJPInfo );
       break;

  case WM_CONTROLPOINTER:
       mRC = SetContextHelp( hDlg, ulMsg, mp1, mp2, NULL, 0, 0 );
       break;

  case WMPS_CONSTRAINTS:
       // If CONSTRAINTS sent by FORMNAMELIST or TRAYNAMELIST, update constraints only for
       // specific MEDIATYPELIST or MEDIATYPELIST_1, not both, because
       // 1st form or tray shouldn't constraint rest page Medias and rest form, tray - 1st Media.
       switch ( SHORT1FROMMP( mp1 ) )
       {
         case JPF_FORMNAMELIST:
         case JPF_TRAYNAMELIST:
           UpdateConstraints( hDlg, pJPInfo, JPF_MEDIATYPELIST, mp1, mp2 );
           break;
         case JPF_TRAYNAMELIST_1:                                   // @MIXED_FEED
         case JPF_FORMNAMELIST_1:                                 // @MIXED_FEED
           UpdateConstraints( hDlg, pJPInfo, JPF_MEDIATYPELIST_1, mp1, mp2 );
           break;
         default:
           UpdateConstraints( hDlg, pJPInfo, JPF_MEDIATYPELIST, mp1, mp2 );
           UpdateConstraints( hDlg, pJPInfo, JPF_MEDIATYPELIST_1, mp1, mp2 );
       }
       // @V4.COLL2FP
       UpdateConstraints( hDlg, pJPInfo, JPF_COLLATELIST, mp1, mp2 );

       sHandle = (SHORT) SHORT1FROMMP( mp1 );
       if (sHandle != JPF_DUPLEXNONE && sHandle != JPF_DUPLEXNOTUMBLE 
           && sHandle != JPF_DUPLEXTUMBLE 
           && sHandle != JPF_TRAYNAMELIST_1 // MIXED_FEED first page tray constraint messages don't apply //@V4.0VV01
           && sHandle != JPF_FORMNAMELIST_1 // MIXED_FEED first page form constraint messages don't apply //@V4.0VV01
          )
       {
         SetDuplexControls( hDlg, pJPInfo );
       }
       break;

  default:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
  }    /* end switch */

  return( mRC );
}
/*---------------------------------------------------------------------------*\
* FormPageDlgProc End                                                         *
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\
* Initialize tray list for JP Form page
\*---------------------------------------------------------------------------*/
VOID FormPageInitTrays( HWND hDlg, PJPINFO pJPInfo, ULONG ulCtrlID, PSZ aTraySelected )
{
  UINT      uiOffset;
  ULONG     aulStringID[ 2 ];
  LONG      alStringHandle[ 2 ];
  PCNFDATA  pCNFData = pJPInfo->pCNFData;  // Configuration data
  
  if (InitializeUIControl( hDlg, pJPInfo, UINAME_INPUTSLOT, ulCtrlID,
                           ulCtrlID-1 ) ==  NULL)
  {
    aulStringID[ 0 ]    = JPF_AUTOTRAYSELECT;
    alStringHandle[ 0 ] = AUTOTRAY_HANDLE;
    aulStringID[ 1 ]    = JPF_MANUALFEED;
    alStringHandle[ 1 ] = MANUALFEED_HANDLE;
    
    GplPageInsertListItems( hDlg, ulCtrlID, aulStringID,
                            GDPF_STRTYPEID, alStringHandle, 2, 0,
                            pscript_module );
    
    if (!strcmp( aTraySelected, MANUALFEED_STRING ))
    {
      uiOffset = MANUALFEED_HANDLE;
    }
    else
    {
      uiOffset = AUTOTRAY_HANDLE;
    }
    
    GplCtrlLBSelectItem( hDlg, ulCtrlID, (MPARAM) uiOffset,
                         GLF_SELECTHANDLE );
  }

}


/******************************************************************************
 *
 * FUNCTION NAME = FormPageInitControls
 *
 * DESCRIPTION
 * Initializes the controls for the "Forms" page to the current settings for
 * the DC.  This includes listing all supported forms in the "Forms"
 * listbox for the feature.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID FormPageInitControls( HWND hDlg, PJPINFO pJPInfo, INT iModifyPgFlag )
{
  PCHAR     pBuffer;
  UINT      uiOffset;
  UINT      uiIndex;
  PUI_BLOCK pUIBlock;
  ULONG     aulStringID[ 2 ];
  LONG      alStringHandle[ 2 ];
  PCNFDATA  pCNFData = pJPInfo->pCNFData;  // Configuration data
  PDESPPD   pdesPPD  = pJPInfo->pdesPPD;   // PPD data structure
  PUI_ENTRY puiEntry;
  INT       ofsEntry;
  HWND      hCtrl;                         // handle of control to send message to //@V4.0VV01
  LONG      lIndex;                        // index of tray                        //@V4.0VV01

  if (iModifyPgFlag == UCD_INITIALIZEPAGE)
  {
    pUIBlock = QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList,
                                      pJPInfo->pItemsBuff, UINAME_PAGESIZE,
                                      NULL);

    if (pCNFData->jobProperties.szFormName[ 0 ] != 0)
    {
      if (SetUIOption( pJPInfo->pdesPPD, pJPInfo->pCurrUISelList,
                       UINAME_PAGESIZE, pCNFData->jobProperties.szFormName )
          == -1)
      {
        *(pJPInfo->pCurrUISelList + pUIBlock->usOrderDep) = 0;
      }
    }
    else
    {
      *(pJPInfo->pCurrUISelList + pUIBlock->usOrderDep) =
        CONVERT_OFS_TO_UISEL( pUIBlock->usDefaultEntry );
    }

    // @MIXED_FEED Begin
    FormPageInitTrays( hDlg, pJPInfo, JPF_TRAYNAMELIST_1, pCNFData->aTraySelected_1 );

    InitializeUIControl( hDlg, pJPInfo, UINAME_PAGESIZE, JPF_FORMNAMELIST_1,
                         JPF_FORMNAMETEXT_1 );
    InitializeUIControl( hDlg, pJPInfo, UINAME_MEDIATYPE, JPF_MEDIATYPELIST_1,
                         JPF_MEDIATYPETEXT_1 );
    WinSendDlgItemMsg( hDlg, JPF_FORMNAMELIST_1, UCM_LISTITEMS, (MPARAM) 0, (MPARAM) 0 );
    // Disable MIXED_FEED if booklet selected
    WinEnableControl( hDlg, JPF_MIXED_FEED, (pCNFData->gjfncb.ulFlags & GNDF_BOOKLETENABLED)? FALSE : TRUE );
    // @MIXED_FEED End

    FormPageInitTrays( hDlg, pJPInfo, JPF_TRAYNAMELIST, pCNFData->u.iv.aTraySelected );

    InitializeUIControl( hDlg, pJPInfo, UINAME_PAGESIZE, JPF_FORMNAMELIST,
                         JPF_FORMNAMETEXT );
    InitializeUIControl( hDlg, pJPInfo, UINAME_MEDIATYPE, JPF_MEDIATYPELIST,
                         JPF_MEDIATYPETEXT );

                         
    
    if (InitializeUIControl( hDlg, pJPInfo, UINAME_COLLATE, JPF_COLLATELIST,
                             JPF_COLLATETEXT ) == NULL)                       //@V4.COLL2FP
    {
      /*
      ** If printer don't support hardware collate at all
      ** add None and Simulated options to Collate combo box.
      */
      aulStringID[ 0 ]    = IDS_None;
      alStringHandle[ 0 ] = NONE_COLLATE_HANDLE;
      aulStringID[ 1 ]    = IDS_COLLATE_SIMULATION;
      alStringHandle[ 1 ] = SIM_COLLATE_HANDLE;
      GplPageInsertListItems( hDlg, JPF_COLLATELIST, aulStringID,
                              GDPF_STRTYPEID, alStringHandle, 2, LIT_END,
                              pscript_module );

      WinEnableWindow( WinWindowFromID( hDlg, JPF_COLLATELIST ), TRUE );
      WinEnableWindow( WinWindowFromID( hDlg, JPF_COLLATETEXT ), TRUE );

      GplCtrlLBSelectItem( hDlg,
                           JPF_COLLATELIST,
                           (MPARAM)
                           (CHECKFLAG(pCNFData->ulFlags, SIM_COLLATE)?
                             SIM_COLLATE_HANDLE : NONE_COLLATE_HANDLE ),
                           GLF_SELECTHANDLE );
    }
    else
    {
      /*
      ** If printer support hardware collate in some cases
      ** add only Simulated options to Collate combo box.
      */
      aulStringID[ 0 ]    = IDS_COLLATE_SIMULATION;
      alStringHandle[ 0 ] = SIM_COLLATE_HANDLE;
      GplPageInsertListItems( hDlg, JPF_COLLATELIST, aulStringID,
                              GDPF_STRTYPEID, alStringHandle, 1, LIT_END,
                              pscript_module );

      if ( CHECKFLAG(pCNFData->ulFlags, SIM_COLLATE) )
      {
        GplCtrlLBSelectItem( hDlg,
                             JPF_COLLATELIST,
                             (MPARAM) SIM_COLLATE_HANDLE,
                             GLF_SELECTHANDLE );
      }
    }
    
    
    // We need to send constraint message for JPF_TRAYNAMELIST to disable duplex printing  //@V4.0VV01
    // if tray is not compliant. This is already being done when user selects trays from   //@V4.0VV01
    // dropdown box, but call on dialog initialization is needed too!                      //@V4.0VV01
    if ((hCtrl = WinWindowFromID( hDlg, JPF_TRAYNAMELIST )) != (HWND) 0)                   //@V4.0VV01
    {                                                                                      //@V4.0VV01
       // query which tray is currently selected                                           //@V4.0VV01
       lIndex = UpdateSelection( hCtrl, pJPInfo, JPF_TRAYNAMELIST,                         //@V4.0VV01
                                 FALSE ); //IGNORE_CONSTRAINT_MSG                          //@V4.0VV01
       
       if(!(pJPInfo->usFlags & UIC_NO_UPDATE))                                             //@V4.0VV01
       {                                                                                   //@V4.0VV01
         SendConstraintsMessages( pJPInfo, JPF_TRAYNAMELIST, lIndex, pUIBlock );           //@V4.0VV01
       }                                                                                   //@V4.0VV01
    }                                                                                      //@V4.0VV01

  }
  else //UCD_UNDO
  {
    WinSendDlgItemMsg( hDlg, JPF_FORMNAMELIST, UCM_LISTITEMS, (MPARAM) 0,
                       (MPARAM) 0 );
    WinSendDlgItemMsg( hDlg, JPF_MEDIATYPELIST, UCM_UNDO, (MPARAM) 0,
                       (MPARAM) 0 );

    // @MIXED_FEED Begin
    WinSendDlgItemMsg( hDlg, JPF_FORMNAMELIST_1, UCM_LISTITEMS, (MPARAM) 0,
                       (MPARAM) 0 );
    // @MIXED_FEED End

    pUIBlock = (PUI_BLOCK) WinQueryWindowULong( WinWindowFromID( hDlg, JPF_COLLATELIST ),
                                                QWL_USER );
    
    if (pUIBlock)
    {
      if ( CHECKFLAG(pCNFData->ulFlags, SIM_COLLATE) )
      {
        GplCtrlLBSelectItem( hDlg,
                             JPF_COLLATELIST,
                             (MPARAM) SIM_COLLATE_HANDLE,
                             GLF_SELECTHANDLE );
      }
      else
      {
        WinSendDlgItemMsg( hDlg, JPF_COLLATELIST, UCM_UNDO, (MPARAM) 0,
                           (MPARAM) 0 );                              //@V4.COLL2FP
      }
    }
    else
    {
      
      GplCtrlLBSelectItem( hDlg,
                           JPF_COLLATELIST,
                           (MPARAM)
                           (CHECKFLAG(pCNFData->ulFlags, SIM_COLLATE)?
                             SIM_COLLATE_HANDLE : NONE_COLLATE_HANDLE ),
                           GLF_SELECTHANDLE );
    }
    
    
    pUIBlock = QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList,
                                      pJPInfo->pItemsBuff,
                                      UINAME_DUPLEX, NULL );
    if( pUIBlock )
    {                           
      *(pJPInfo->pCurrUISelList + pUIBlock->usOrderDep) =
        *(ASSIGN_UISELLIST_PTR( pCNFData ) + pUIBlock->usOrderDep);
    }
  }

  // @MIXED_FEED Begin
  pUIBlock = QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList,
                                    pJPInfo->pItemsBuff, UINAME_MEDIATYPE, NULL);
  if ( pUIBlock )
  {
    puiEntry = QueryEntryFromOption( pJPInfo->pItemsBuff, pUIBlock, pCNFData->szMediaType_1, &ofsEntry );
    
    GplCtrlLBSelectItem( hDlg, JPF_MEDIATYPELIST_1, (MPARAM) ofsEntry,
                         GLF_SELECTHANDLE );
  }
  if ( WinIsWindowEnabled( WinWindowFromID( hDlg, JPF_MIXED_FEED ) ) )
  {
    WinSendDlgItemMsg( hDlg, JPF_MIXED_FEED,
                       BM_SETCHECK,
                       MPFROMSHORT( CHECKFLAG( pCNFData->ulFlags, MIXED_FEED)?TRUE:FALSE ),
                       (MPARAM) 0 );
  }
  else
  {
    WinSendDlgItemMsg( hDlg, JPF_MIXED_FEED,
                       BM_SETCHECK,
                       MPFROMSHORT( FALSE ),
                       (MPARAM) 0 );
  }

  // Send BN_CHECK notify message. Set MP2=0, to indicate, that message is simulated
  WinSendMsg(hDlg, WM_CONTROL, MPFROM2SHORT(JPF_MIXED_FEED,BN_CLICKED), MPFROMSHORT(0));
  // @MIXED_FEED End

  
  /*
  ** Set the radio buttons for the Orientation group.
  */
  /*
  ** If output is encapsulated, disable the landscape button.
  */
  if (pCNFData->iDestnType == ENCAPS)
  {
    WinEnableWindow( WinWindowFromID( hDlg, JPF_LANDSCAPE ), FALSE );
    pCNFData->jobProperties.iOrient = PORTRAIT;
  }

  if (pCNFData->jobProperties.iOrient == PORTRAIT)
  {
    uiIndex  = JPF_PORTRAIT;
    uiOffset = ICON_PORTRAIT;
  }
  else   // Landscape
  {
    uiIndex  = JPF_LANDSCAPE;
    uiOffset = ICON_LANDSCAPE;
  }
  /*
  ** Set the desired button.
  */
  WinSendDlgItemMsg( hDlg, (ULONG) uiIndex, BM_CLICK, MPFROMLONG( TRUE ),
                     (MPARAM) 0 );

  /*
  ** Send this message to set the current orientation icon.
  */
  UpdateIcon( hDlg, pJPInfo->hDlg, JPSB_PORTRAIT, uiOffset );

  /* DUPLEX START ************************************************************/
  SetDuplexControls( hDlg, pJPInfo );
  /* DUPLEX END **************************************************************/

  WinSendDlgItemMsg( hDlg, JPF_COPIESNUM, SPBM_SETCURRENTVALUE,
                     MPFROMLONG( pCNFData->iCntCopies ),
                     (MPARAM) 0 );

}
/*---------------------------------------------------------------------------*\
* FormPageInitControls End                                                    *
\*---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*\
* Query the tray name that was selected.
\*---------------------------------------------------------------------------*/
VOID QuerySelectedTray( HWND hDlg, PDLGHDR pJPHdr, ULONG ulCtrlID, PSZ aTraySelected )
{
  PUI_BLOCK pUIBlock;
  HWND      hCtrl;
  LONG      lSelected;
  PSZ       pString;
  PSZ       pManFeedSelected;
  
  hCtrl = WinWindowFromID( hDlg, ulCtrlID );
  pUIBlock = (PUI_BLOCK) WinQueryWindowULong( hCtrl, QWL_USER );
  
  if ( pUIBlock == NULL )
  {
    *aTraySelected = 0;
    return;
  }
  
  GplPageQueryLBSelection( hCtrl, 0, &lSelected, NULL );

  /*
  ** The tray will either contain a tray name, "MANUALFEED", or
  ** "AUTOTRAYSELECT".  Query the handle to the selected item and set the
  ** appropriate string.  If manual feed is selected, set the MF UI to TRUE,
  ** otherwise, set to FALSE.
  */
  pManFeedSelected = (PSZ) "False";
  switch (lSelected)
  {
  case AUTOTRAY_HANDLE:
       pString = AUTOTRAY_STRING;
       break;

  case MANUALFEED_HANDLE:
       pString = MANUALFEED_STRING;
       pManFeedSelected = (PSZ) "True";
       break;

  default:
       pString = (PSZ) (pJPHdr->pItemsBuff +
                        pUIBlock->uiEntry[ lSelected ].ofsOption);
       break;

  }

  strcpy( aTraySelected, pString );

  if ( ulCtrlID == JPF_TRAYNAMELIST )
    SetUIOption( pJPHdr->pdesPPD, pJPHdr->pCurrUISelList, UINAME_MANUALFEED,
                 (PBYTE) pManFeedSelected );
  return;
}

/*---------------------------------------------------------------------------*\
* Query form name that was selected.
\*---------------------------------------------------------------------------*/
VOID QuerySelectedForm( HWND hDlg, PDLGHDR pJPHdr, ULONG ulCtrlID, PSZ szFormName )
{
  PUI_BLOCK pUIBlock;
  HWND      hCtrl;
  LONG      lSelected;
  PSZ       pString;

  /*
  ** Query the form name that was selected.
  */
  hCtrl = WinWindowFromID( hDlg, ulCtrlID );
  pUIBlock = (PUI_BLOCK) WinQueryWindowULong( hCtrl, QWL_USER );
  
  if ( pUIBlock == NULL )
  {
    *szFormName = 0;
    return;
  }

  GplPageQueryLBSelection( hCtrl, 0, &lSelected, NULL );

  if (!IS_UNQ_FORMID( lSelected ))
  {
    pString = (PSZ) (pJPHdr->pItemsBuff +
                     pUIBlock->uiEntry[ lSelected ].ofsOption);
  }
  else
  {
    pString = (PSZ) (pJPHdr->stUnqFrmLst[ CONV_LBID_TO_UNQID( lSelected ) ].
                     aFormName);
  }

  strcpy( szFormName, pString );
  
  return;
}


/******************************************************************************
 *
 * FUNCTION NAME = FormPageSaveSettings
 *
 * DESCRIPTION
 * Saves the current settings for the Forms page.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 * RETURN-ERROR  - FALSE
 *
 *****************************************************************************/
UINT FormPageSaveSettings( HWND hDlg, PDLGHDR pJPHdr )
{
  PUI_BLOCK pUIBlock;
  HWND      hCtrl;
  LONG      lSelected;
  PSZ       pString;
  PCNFDATA  pCNFData = pJPHdr->pCNFData;

  /*
  ** Save the current form setting.
  */
  QuerySelectedForm(hDlg, pJPHdr, JPF_FORMNAMELIST, pCNFData->jobProperties.szFormName );
  PrfWriteProfileString( HINI_SYSTEMPROFILE,
                         (PSZ) pCNFData->szKeyApp, PAPERSOURCE_INI,
                         (PSZ) pCNFData->jobProperties.szFormName );

  /*
  ** Query the tray name that was selected.
  */
  QuerySelectedTray(hDlg, pJPHdr, JPF_TRAYNAMELIST, pCNFData->u.iv.aTraySelected );

  // @MIXED_FEED Begin
  lSelected = (LONG) WinSendDlgItemMsg( hDlg, JPF_MIXED_FEED, BM_QUERYCHECK,
                                        (MPARAM) 0, (MPARAM) 0 );
  if (lSelected == 1)
  {
    SETFLAG( pCNFData->ulFlags, MIXED_FEED);
  }
  else
  {
    CLEARFLAG( pCNFData->ulFlags, MIXED_FEED);
    //*(pCNFData->szFormName_1)=0x0;
    //*(pCNFData->aTraySelected_1)=0x0;
    //*(pCNFData->szMediaType_1)=0x0;
  }

  //Get selected form for subsequent pages
  QuerySelectedForm(hDlg, pJPHdr, JPF_FORMNAMELIST_1, pCNFData->szFormName_1 );
  
  //Get selected tray for subsequent pages
  QuerySelectedTray(hDlg, pJPHdr, JPF_TRAYNAMELIST_1, pCNFData->aTraySelected_1 );
  
  //Get selected media for subsequent pages
  hCtrl = WinWindowFromID( hDlg, JPF_MEDIATYPELIST_1 );
  pUIBlock = (PUI_BLOCK) WinQueryWindowULong( hCtrl, QWL_USER );
  if (pUIBlock != NULL)
  {
    GplPageQueryLBSelection( hCtrl, 0, &lSelected, NULL );
    pString = (PSZ) (pJPHdr->pItemsBuff +
                     pUIBlock->uiEntry[ lSelected ].ofsOption);
    strcpy( pCNFData->szMediaType_1, pString );
  }
  // @MIXED_FEED End

  /*
  ** Store the orientation.
  */
  lSelected = (LONG) WinSendDlgItemMsg( hDlg, JPF_PORTRAIT, BM_QUERYCHECK,
                                        (MPARAM) 0, (MPARAM) 0 );
  if (lSelected == 1)
  {
    pCNFData->jobProperties.iOrient = PORTRAIT;
  }
  else
  {
    pCNFData->jobProperties.iOrient = LANDSCAPE;
  }

  /*
  ** Save the driver version.
  */
  pCNFData->lVersion = DRIVERSION;

  /*
  ** Media Type.
  */
  if (WinIsWindowEnabled( WinWindowFromID( hDlg, JPUI_PICKONE ) ) == TRUE)
  {
    lSelected = (LONG) WinSendDlgItemMsg( hDlg, JPUI_PICKONE,
                       LM_QUERYSELECTION, MPFROMSHORT( LIT_FIRST ),
                       (MPARAM) 0 );

    pUIBlock = QueryBlockFromKeyword( &pJPHdr->pdesPPD->stUIList,
                                      pJPHdr->pItemsBuff, UINAME_MEDIATYPE,
                                      NULL );

    SetUIOption( pJPHdr->pdesPPD, pJPHdr->pCurrUISelList,
                 pJPHdr->pItemsBuff + pUIBlock->ofsUIName,
                 pJPHdr->pItemsBuff +
                         pUIBlock->uiEntry[ lSelected ].ofsOption );
  }

  /*
  ** Number Of Copies.
  */
  WinSendDlgItemMsg( hDlg, JPF_COPIESNUM, SPBM_QUERYVALUE,
                     (MPARAM) &lSelected, MPFROMSHORT( SPBQ_UPDATEIFVALID ) );
  pCNFData->iCntCopies = lSelected;

  /*
  ** Set or clear flags to simulate collated output
  */
  hCtrl = WinWindowFromID( hDlg, JPF_COLLATELIST );
  pUIBlock = (PUI_BLOCK) WinQueryWindowULong( hCtrl, QWL_USER );
  GplPageQueryLBSelection( hCtrl, 0, &lSelected, NULL );

  switch (lSelected)
  {
  case NONE_COLLATE_HANDLE:
     CLEARFLAG( pCNFData->ulFlags, SIM_COLLATE );
     // Need to set pUIBlock to default
     break;
  case SIM_COLLATE_HANDLE:
     SETFLAG( pCNFData->ulFlags, SIM_COLLATE );
     // Need to set pUIBlock to default
     break;
  default:
     CLEARFLAG( pCNFData->ulFlags, SIM_COLLATE );
     break;
  }
  
  
  
  return( PMERR_OK );
}
/*---------------------------------------------------------------------------*\
* FormPageSaveSettings End                                                    *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = OutputPageDlgProc
 *
 * DESCRIPTION
 * This is the procedure for the Output Options dialog box that is in the Job
 * Properties notebook.  Among the standard procedure responsibilites, this
 * procedure initializes and saves all data pertaining to the dialog, with
 * regard to the Job Properties Information structure.
 *
 * INPUT
 * hDlg - Dialog handle
 * ulMsg - Message ID
 * mp1, mp2 - General message parameters
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 *                 FALSE
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
MRESULT EXPENTRY OutputPageDlgProc( HWND hDlg, ULONG ulMsg, MPARAM mp1,
                                    MPARAM mp2 )
{
  PDLGHDR pJPInfo;                 // Job Properties info structure
  HWND    hwndEntry;               // 'File' entry field handle
  UINT    uiVal1, uiVal2;          // Local variables
  CHAR    aTempFileName[ CCHMAXPATH ];  // Temporary file name
  INT     iFileControl;            // 'TRUE' if to file, 'FALSE' if to printer
  USHORT  usStatusBar;             // Status Bar Icon ID
  MRESULT mRC = (MRESULT) FALSE;   // Return code
  // @V3.1142412
//  INT     iDuplexControl = FALSE;  // 'TRUE' to change icon, 'FALSE' not to
//@V3.130814
  INT     iPrinterControl = FALSE; // 'TRUE' to change icon, 'FALSE' not to
  BOOL    fUpdateIcon;
  INT     iStatusBar;
  INT     iIconID;
//  PSZ     pDuplexSelected;

  if (ulMsg != WM_INITDLG)
  {
    pJPInfo = (PDLGHDR) WinQueryWindowULong( hDlg, QWL_USER );
  }

  switch( ulMsg )
  {
  /*
  ** Set the spin button controls to recognize the acceptable min/max
  ** values that can be given to them.  The actual contents of the
  ** spin buttons will be updated when this procedure receives a
  ** WPJM_INITSETTINGS message.
  ** Do not access the JPINFO structure because it has not been set
  ** at this time.  It will be for WPJM_INITSETTINGS.
  */
  case WM_INITDLG:
       pJPInfo = InitDlgULong( hDlg, mp2 );

       OutputPageInitControls( hDlg, pJPInfo, UCD_INITIALIZEPAGE );
       break;

  case WM_PAINT:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
       UpdateIcon( hDlg, 0, JPSB_OUTPUT_PRINTER, 0 );
       break;

  /*
  ** This must be here so that FALSE is returned.  If not, the dialog box
  ** in the notebook will disappear if the user presses ESC.
  */
  case WM_COMMAND:
       switch ( SHORT1FROMMP( mp1 ) )
       {
       // @V3.1142412
       case JPO_ERRHNDLR:
            PrintErrorHandler( pJPInfo->pCNFData );
            break;

       case PSB_UNDO:
            OutputPageInitControls( hDlg, pJPInfo, UCD_UNDOPAGE );
            break;

       case PSB_DEFAULT:
            OutputPageDefaultControls( hDlg, pJPInfo );
            break;

       case DID_CANCEL:
            WinSendMsg( pJPInfo->hDlg, WM_COMMAND, (MPARAM) PSB_CANCEL,
                        (MPARAM) 0 );
       }
       break;

  case WM_CONTROL:
       iFileControl = -1;
       fUpdateIcon = FALSE;
//       pDuplexSelected = NULL;

       switch( SHORT1FROMMP( mp1 ) )
       {
       case JPO_PRINTER:
            fUpdateIcon = TRUE;
            iStatusBar = JPSB_OUTPUT_PRINTER;
            iIconID = ICON_OUTPUT_PRINTER;
            iFileControl = FALSE;
            uiVal1 = FALSE;
            break;

       case JPO_RAWFILE:
            fUpdateIcon = TRUE;
            iStatusBar = JPSB_OUTPUT_PRINTER;
            iIconID = ICON_OUTPUT_RAW;
            iFileControl = TRUE;
            uiVal1 = FALSE;
            break;

       case JPO_EPS:
            fUpdateIcon = TRUE;
            iStatusBar = JPSB_OUTPUT_PRINTER;
            iIconID = ICON_OUTPUT_EPS;
            iFileControl = TRUE;
            uiVal1 = TRUE;
            break;

       case JPO_FILEENTRY:
            break;

       case JPO_REPDOWNFONT:
            /*            ** Call PM to get state of check box
            ** 0 = Unchecked, 1 = checked;
            */
            if ((!(SHORT)WinSendDlgItemMsg( hDlg,
                  JPO_REPDOWNFONT, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 )) &&
                (!(SHORT)WinSendDlgItemMsg( hDlg,
                  JPO_REPDEVFONTS, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 )))
            {
              WinSendDlgItemMsg( hDlg, JPO_REPDEVFONTS, BM_SETCHECK,
                                 (MPARAM) TRUE, (MPARAM) 0 );
            }
           break;

       case JPO_REPDEVFONTS:
            /*
            ** Call PM to get state of check box
            ** 0 = Unchecked, 1 = checked;
            */
            if ((!(SHORT)WinSendDlgItemMsg( hDlg,
                  JPO_REPDOWNFONT, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 )) &&
                (!(SHORT)WinSendDlgItemMsg( hDlg,
                  JPO_REPDEVFONTS, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 )))
            {  //Turn on DLfonts
              WinSendDlgItemMsg( hDlg, JPO_REPDOWNFONT, BM_SETCHECK,
                                 (MPARAM) TRUE, (MPARAM) 0 );
            }

            break;
       }

       if (fUpdateIcon == TRUE)
       {
         UpdateIcon( hDlg, pJPInfo->hDlg, iStatusBar, iIconID );
       }

       /*
       ** Enable the 'File' entry field, only if 'Raw File' or
       ** 'Encapsulated PostScript File' is selected.  Disable it
       ** if output is going to the printer.
       ** If iFileControl is -1, that means none of the output radio
       ** buttons were selected.
       */
       if (pJPInfo != NULL && iFileControl != -1)
       {
         /*
         ** Display the appropriate icon in the Status Bar.
         */
         if (uiVal1 == TRUE)
         {
           /*
           ** Force the 'Portrait' radio button to be checked.
           */
           WinSendDlgItemMsg( pJPInfo->stPage[ JP_FORM_PAGE_INDEX ].hPage,
                              JPF_PORTRAIT, BM_SETCHECK, MPFROMSHORT( 1 ),
                              (MPARAM) 0 );

           /*
           ** Send a message to the 'Forms' page, indicating that Portrait
           ** was selected.  This is preferred rather than to simply change
           ** the status icon just in case additional processing is done
           ** when 'Portrait' is selected.
           */
           WinSendMsg( pJPInfo->stPage[ JP_FORM_PAGE_INDEX ].hPage,
                       WM_CONTROL, MPFROM2SHORT( JPF_PORTRAIT, BN_CLICKED ),
                       (MPARAM) 0 );
           uiVal2 = FALSE;
         }
         else
         {
           uiVal2 = TRUE;
         }
         WinEnableWindow( WinWindowFromID( pJPInfo->stPage[ JP_FORM_PAGE_INDEX ].
                                           hPage, JPF_LANDSCAPE ),
                          uiVal2 );

         WinEnableWindow( WinWindowFromID( hDlg, JPO_FILETEXT ),
                          (BOOL) iFileControl );
         hwndEntry = WinWindowFromID( hDlg, JPO_FILEENTRY );
         if (iFileControl == (INT) FALSE)
         {
           WinSendMsg( hwndEntry, EM_CLEAR, (MPARAM) 0, (MPARAM) 0 );
           WinEnableWindow( hwndEntry, FALSE );
         }
         else
         {
           WinEnableWindow( hwndEntry, TRUE );
           WinSetFocus( HWND_DESKTOP, hwndEntry );
           WinSetDlgItemText( hDlg, JPO_FILEENTRY,
                              (PSZ) pJPInfo->pCNFData->szDestnFile );
         }
       }
       break;

  case WMPS_VERIFYSETTINGS:
       mRC = (MRESULT) PMERR_OK;
       if (WinSendDlgItemMsg( hDlg, JPO_PRINTER, BM_QUERYCHECK, (MPARAM) 0,
                              (MPARAM) 0 ) != (MRESULT) 1)
       {
         WinQueryDlgItemText( hDlg, JPO_FILEENTRY, CCHMAXPATH,
                              (PSZ) &aTempFileName );

         if (aTempFileName[ 0 ] == 0 ||
             !ValidFilename( (PSZ) aTempFileName ))
         {
           /*
           ** let the user know the filename is invalid.
           */
           DisplayMessageBox( hDlg, IDM_BadFileName, 0,
                              MB_OK | MB_ICONEXCLAMATION );

           /*
           ** tell the dialog box to select the entire edit field.
           */
           WinSendDlgItemMsg( hDlg, JPO_FILEENTRY, EM_SETSEL,
                              MPFROM2SHORT((SHORT) 0, (SHORT) 3),
                              (MPARAM) 0L );

           /*
           ** put the cursor on the file edit field.
           */
           WinEnableWindow( hDlg, TRUE );
           mRC = (MRESULT) WinWindowFromID( hDlg, JPO_FILEENTRY );
         }
       }
       break;

  case WMPS_SAVESETTINGS:
       /*
       ** 'OK' was pressed.  Save the settings.
       */
       mRC = (MRESULT) OutputPageSaveSettings( hDlg, pJPInfo );
       break;

  case WM_CONTROLPOINTER:
       mRC = SetContextHelp( hDlg, ulMsg, mp1, mp2, NULL, 0, 0 );
       break;

  case WMPS_CONSTRAINTS:
       UpdateConstraints( hDlg, pJPInfo, JPO_RESOLUTIONLIST, mp1, mp2 );
       break;

  default:
    mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
  }       /* end switch */

  return( mRC );
}
/*---------------------------------------------------------------------------*\
* OutputPageDlgProc End                                                       *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = OutputPageInitControls
 *
 * DESCRIPTION
 * Initializes the controls for the "Output" page to the current settings.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID OutputPageInitControls( HWND hDlg, PJPINFO pJPInfo, INT iModifyPgFlag )
{
  ULONG      ulSelectButton;
  BOOL       bEnableControl;
  CHAR       szString[ MAX_PSIZE ];         // Local string buffer
  UINT       uiIndex;
  HWND       hCtrl;
  PCNFDATA   pCNFData = pJPInfo->pCNFData;
  PDESPPD    pdesPPD  = pJPInfo->pdesPPD;   // PPD data structure

  /* PRINT TO GROUP START ****************************************************/
  switch( pCNFData->iDestnType )
  {
  case RAW:
       ulSelectButton = JPO_RAWFILE;
       bEnableControl = TRUE;
       break;

  case ENCAPS:
       ulSelectButton = JPO_EPS;
       bEnableControl = TRUE;
       break;

  default:
       ulSelectButton = JPO_PRINTER;
       bEnableControl = FALSE;
  }
  WinSetDlgItemText( hDlg, JPO_FILEENTRY, "" );

  WinEnableWindow( WinWindowFromID( hDlg, JPO_FILETEXT ),
                   (BOOL) bEnableControl );
  WinEnableWindow( WinWindowFromID( hDlg, JPO_FILEENTRY ),
                   (BOOL) bEnableControl );
  WinSendDlgItemMsg( hDlg, ulSelectButton, BM_SETCHECK, MPFROMSHORT( 1 ),
                     (MPARAM) 0 );

  if (bEnableControl == TRUE)
  {
    WinSetDlgItemText( hDlg, JPO_FILEENTRY, (PSZ) pCNFData->szDestnFile );
  }

  /*
  ** This message forces a repaint of the Output group of the Status Bar.
  ** Without this message, the Status Bar output group may be blank at
  ** initialization.
  */
  WinSendMsg( hDlg, WM_CONTROL, MPFROMSHORT( ulSelectButton ), (MPARAM) 0 );
  /* PRINT TO GROUP END ******************************************************/

  /* MISC START **************************************************************/
  WinSendDlgItemMsg( hDlg, JPO_REPDOWNFONT, BM_SETCHECK,
                               MPFROMSHORT( pCNFData->sUseDLFonts ),
                               (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPO_REPDEVFONTS, BM_SETCHECK, (MPARAM) (LONG)
                     pCNFData->sUsePDFonts, (MPARAM) 0L );

  WinSendDlgItemMsg( hDlg, JPO_LVL1COMPAT, BM_SETCHECK,
                               MPFROMSHORT( pCNFData->usPSLevel1 ),
                               (MPARAM) 0 );

  if (pJPInfo->pdesPPD->desItems.usLanguageLevel == 1)
  {
    WinSendDlgItemMsg( hDlg, JPO_LVL1COMPAT, BM_SETCHECK, (MPARAM) 0,
                       (MPARAM) 0 );
    bEnableControl = FALSE;
  }
  else
  {
    bEnableControl = TRUE;
  }
  WinEnableWindow( WinWindowFromID( hDlg, JPO_LVL1COMPAT ), bEnableControl );

  /*
  ** Check/uncheck Unix/AIX Compatibility.
  */
  if ((pCNFData->ulFlags & UNIX_COMPAT) != 0)
  {
    ulSelectButton = 1;
  }
  else
  {
    ulSelectButton = 0;
  }

  WinSendDlgItemMsg( hDlg, JPO_UNIXCOMPAT, BM_SETCHECK,
                     (MPARAM) ulSelectButton, (MPARAM) 0 );

  // Set up Font substitution
  WinSendDlgItemMsg( hDlg, JPO_FONTSUB, BM_SETCHECK,
                     (MPARAM) ((pCNFData->ulFlags >> 7) & 1), NULL );

  // TURN OFF for now.....  Remove when full support is active
  WinEnableWindow( WinWindowFromID( hDlg, JPO_FONTSUB ), FALSE );


  /* MISC END ****************************************************************/

  /* RESOLUTION START ********************************************************/
  if (QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList, pJPInfo->pItemsBuff,
                             UINAME_RESOLUTION, NULL ) != NULL)
  {
    if (iModifyPgFlag == UCD_INITIALIZEPAGE)
    {
      InitializeUIControl( hDlg, pJPInfo, UINAME_RESOLUTION, JPO_RESOLUTIONLIST,
                           JPO_RESOLUTIONTEXT );
    }
    else
    {
      WinSendDlgItemMsg( hDlg, JPO_RESOLUTIONLIST, UCM_UNDO, (MPARAM) 0,
                         (MPARAM) 0 );
    }
  }
  else
  {
    if (iModifyPgFlag == UCD_INITIALIZEPAGE)
    {
      hCtrl = WinWindowFromID( hDlg, JPO_RESOLUTIONLIST );
      uiIndex = pJPInfo->pdesPPD->desItems.iResDpi;
      _itoa( uiIndex, szString, 10 );
      
      WinSendMsg( hCtrl, LM_DELETEALL, 0, 0 );
      uiIndex = (UINT) WinSendMsg( hCtrl, LM_INSERTITEM, MPFROMSHORT( LIT_END ),
                                   MPFROMP( szString ) );
      WinSendMsg( hCtrl, LM_SETITEMHANDLE, MPFROMLONG( uiIndex ),
                  (MPARAM) 0 );
      WinSendMsg( hCtrl, LM_SELECTITEM, (MPARAM) 0, (MPARAM) TRUE );
    }
  }
  /* RESOLUTION END **********************************************************/
}
/*****************************************************************************\
* OutputPageInitControls End                                                  *
\*****************************************************************************/





/******************************************************************************
 *
 * FUNCTION NAME = OutputPageSaveSettings
 *
 * DESCRIPTION
 * Saves the current settings for the Output page.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 * RETURN-ERROR  - TRUE
 *
 *****************************************************************************/
UINT OutputPageSaveSettings( HWND hDlg, PJPINFO pJPInfo )
{
  UINT uiVar;
  UINT uiValue;
  UINT uiRC = PMERR_OK;
  PCNFDATA   pCNFData = pJPInfo->pCNFData;  // Configuration data

  /* PRINT TO... START *******************************************************/
  if (WinSendDlgItemMsg( hDlg, JPO_PRINTER, BM_QUERYCHECK, (MPARAM) 0,
                         (MPARAM) 0 ))
  {
    pJPInfo->pCNFData->iDestnType = SYSTEM;
  }
  else if (WinSendDlgItemMsg( hDlg, JPO_RAWFILE, BM_QUERYCHECK, (MPARAM) 0,
                              (MPARAM) 0 ))
  {
    pJPInfo->pCNFData->iDestnType = RAW;
  }
  else
  {
    pJPInfo->pCNFData->iDestnType = ENCAPS;
  }

  if (pJPInfo->pCNFData->iDestnType != SYSTEM)
  {
    WinQueryDlgItemText( hDlg, JPO_FILEENTRY, MAX_PSIZE,
                         (PSZ) pJPInfo->pCNFData->szDestnFile );
  }
  /* PRINT TO... END *********************************************************/

  /* MISC START **************************************************************/
  pJPInfo->pCNFData->sUseDLFonts = (SHORT)WinSendDlgItemMsg( hDlg,
                    JPO_REPDOWNFONT, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 );

  pJPInfo->pCNFData->sUsePDFonts = (SHORT)WinSendDlgItemMsg( hDlg,
            JPO_REPDEVFONTS, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 );

  pJPInfo->pCNFData->usPSLevel1 = (SHORT) WinSendDlgItemMsg( hDlg,
                       JPO_LVL1COMPAT, BM_QUERYCHECK, (MPARAM) 0, (MPARAM) 0 );

  /*
  ** Query if "Unix/AIX Compatibility is set.  If so, set the appropriate
  ** flag.
  */
  if (WinSendDlgItemMsg( hDlg, JPO_UNIXCOMPAT, BM_QUERYCHECK, (MPARAM) 0,
                         (MPARAM) 0 ) == (MPARAM) 1)
  {
    pCNFData->ulFlags |= UNIX_COMPAT;
  }
  else
  {
    pCNFData->ulFlags &= ~UNIX_COMPAT;
  }

  /*
  ** Query if "Font Substitution is set.  If so, set the appropriate
  ** flag.
  */
  if (WinSendDlgItemMsg( hDlg, JPO_FONTSUB, BM_QUERYCHECK, (MPARAM) 0,
                         (MPARAM) 0 ) == (MPARAM) 1)
  {
    pCNFData->ulFlags |= USEFONTSUB;
  }
  else
  {
    pCNFData->ulFlags &= ~USEFONTSUB;
  }
  /* MISC END ****************************************************************/

  return( uiRC );
}
/*---------------------------------------------------------------------------*\
* OutputPageSaveSettings End                                                  *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = EffectsPageDlgProc
 *
 * DESCRIPTION
 * This is the procedure for the Output Options dialog box that is in the Job
 * Properties notebook.  Among the standard procedure responsibilites, this
 * procedure initializes and saves all data pertaining to the dialog, with
 * regard to the Job Properties Information structure.
 *
 * INPUT
 * hDlg - Dialog handle
 * ulMsg - Message ID
 * mp1, mp2 - General message parameters
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 *                 FALSE
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
MRESULT EXPENTRY EffectsPageDlgProc( HWND hDlg, ULONG ulMsg, MPARAM mp1,
                                     MPARAM mp2 )
{
  PJPINFO pJPInfo;                 // Job Properties info structure
  INT     iLoop;                  // Loop counter
  INT     iTick;                   // Tick offset for slide bar
  FIXED   fxValue;                 // FIXED value of current slider selection
  MRESULT mRC = (MRESULT) FALSE;   // Return code
//@V3.130814
  HAB        hAB = WinQueryAnchorBlock( hDlg );
  HWND       hwndListBox;
  BOOL       fColorFlag;
  CHAR       aColorString[ MAX_PSIZE ];
  PUI_BLOCK  pUIBlock;
  POWNERITEM poi;
  INT        iIndexFound = -1;
  INT        ofsEntry;                // Offset of entry array
  USHORT     usOldIcon;
  INT        iNewIcon;
//@V3.130814 end

  if (ulMsg != WM_INITDLG)
  {
    pJPInfo = (PDLGHDR) WinQueryWindowULong( hDlg, QWL_USER );
  }

  switch( ulMsg )
  {
  case WM_INITDLG:
       /*
       ** The loop inserts tick marks on the slider.  A major tick mark
       ** is GC_SLIDE_MAJOR_TICK_PIXEL_LEN long and a minor tick is
       ** GC_SLIDE_MINOR_TICK_PIXEL_LEN long.  The value that determines
       ** which tick to use is GC_SLIDE_MAJOR_TICK_PLACE.
       */
       for (iLoop = GC_SLIDE_HOME_VAL ; iLoop <= GC_SLIDE_NUM_OF_TICKS ;
            iLoop++)
       {
         /*
         ** Insert a major tick if one of the following occurs: the current
         ** value is the start of the base GC_SLIDE_MAJOR_TICK_PLACE.
         ** Also, always insert a tick at the start and end of the bar.
         ** Otherwise, insert a minor tick.
         */
         if (iLoop % GC_SLIDE_MAJOR_TICK_PLACE == 0 ||
             iLoop == GC_SLIDE_HOME_VAL || iLoop == GC_SLIDE_NUM_OF_TICKS)
         {
           iTick = GC_SLIDE_MAJOR_TICK_PIXEL_LEN;
         }
         else
         {
           iTick = GC_SLIDE_MINOR_TICK_PIXEL_LEN;
         }

         WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_R, SLM_SETTICKSIZE,
                            MPFROM2SHORT( iLoop - GC_SLIDE_HOME_VAL, iTick ),
                            (MPARAM) 0 );
         WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_G, SLM_SETTICKSIZE,
                            MPFROM2SHORT( iLoop - GC_SLIDE_HOME_VAL, iTick ),
                            (MPARAM) 0 );
         WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_B, SLM_SETTICKSIZE,
                            MPFROM2SHORT( iLoop - GC_SLIDE_HOME_VAL, iTick ),
                            (MPARAM) 0 );
       }

       /*
       ** Insert the minimum, maximum, and 'center' values for the slider.
       ** These values are hard-coded, so if the above format changes, these
       ** values may also need to be changed.
       */
       InsertSliderVals(hDlg, JPE_GAMMASLIDE_R);
       InsertSliderVals(hDlg, JPE_GAMMASLIDE_G);
       InsertSliderVals(hDlg, JPE_GAMMASLIDE_B);

       /*
       ** Initialize the spin buttons for this dialog.  Set from 0 to 999,
       ** and allow no more than three characters for each control.
       */
       WinSendDlgItemMsg( hDlg, JPE_SCALENUM, SPBM_SETLIMITS,
                          MPFROMLONG( 999 ), MPFROMLONG( 0 ) );
       WinSendDlgItemMsg( hDlg, JPE_SCALENUM, SPBM_SETTEXTLIMIT,
                          MPFROMLONG( 3 ), (MPARAM) 0 );

       pJPInfo = InitDlgULong( hDlg, mp2 );
       EffectsPageInitControls( hDlg, pJPInfo, UCD_INITIALIZEPAGE );
       break;

  case WM_PAINT:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
       UpdateIcon( hDlg, 0, JPSB_MONOCHROME, 0 );
       break;

  /*
  ** This must be here so that FALSE is returned.  If not, the dialog box
  ** in the notebook will disappear if the user presses ESC.
  */
  case WM_COMMAND:
       switch ( SHORT1FROMMP( mp1 ) )
       {
       case PSB_UNDO:
         EffectsPageInitControls( hDlg, pJPInfo, UCD_UNDOPAGE );
         break;

       case PSB_DEFAULT:
         EffectsPageDefaultControls( hDlg, pJPInfo );
         break;

       case DID_CANCEL:
         WinSendMsg( pJPInfo->hDlg, WM_COMMAND, (MPARAM) PSB_CANCEL, (MPARAM) 0 );
       }
       break;

  case WM_CONTROL:
       switch( SHORT1FROMMP( mp1 ) )
       {
       case JPE_GAMMASLIDE_R:
       case JPE_GAMMASLIDE_G:
       case JPE_GAMMASLIDE_B:
            iTick = (INT) SHORT2FROMMP( mp1 );

            /*
            ** Display the gamma value if the slider bar is moved (changed).
            */
            if (iTick == SLN_CHANGE || iTick == SLN_SLIDERTRACK)
            {
              // find out if the LOCKRGB is checked -- this will move
              // all the colors at once
              if (WinSendDlgItemMsg( hDlg, JPE_LOCKRGB, BM_QUERYCHECK,
                                     (MPARAM) 0,
                                     (MPARAM) 0 ) == (MRESULT) 1)
                GetOffset( hDlg, SHORT1FROMMP( mp1 ), TRUE);
              else
              {
                bNewChange = TRUE;
                GetOffset( hDlg, SHORT1FROMMP( mp1 ), FALSE);
              }
            }

            break;

       /*
       ** If this message is received in this procedure, then the color model
       ** is not a UI.
       */
       case JPE_COLOR_LIST:
            if (SHORT2FROMMP( mp1 ) == LN_SELECT)
            {
              if (!(pJPInfo->usFlags & IS_COLOR_UI))
              {
                iLoop = (INT) WinSendDlgItemMsg( hDlg, JPE_COLOR_LIST,
                                                 LM_QUERYSELECTION,
                                                 MPFROMSHORT( LIT_FIRST ),
                                                 (MPARAM) 0 );

                if (iLoop == 0)
                {
                  iNewIcon = ICON_COLOR;
                }
                else
                {
                  iNewIcon = ICON_MONOCHROME;
                }

                // @V3.1141419
#ifndef DDKBLD
                OEMSetStatus( hDlg, (MPARAM) fColorFlag );
#endif
                UpdateIcon( hDlg, pJPInfo->hDlg, JPSB_MONOCHROME, iNewIcon );
              }
            }
            break;
       }
       break;

//@V3.130814 end

  case WM_DRAWITEM:
       if ((poi = (POWNERITEM) PVOIDFROMMP(mp2)) != NULL)
       {
         /*
         ** This code will draw the ribbon a specific color for the gamma
         ** sliders.
         */
         if (poi->idItem == SDA_RIBBONSTRIP)
         {
           iIndexFound = (INT) SHORT1FROMMP( mp1 );

           switch (iIndexFound)
           {
           case JPE_GAMMASLIDE_R:
                ofsEntry = CLR_RED;
                break;

           case JPE_GAMMASLIDE_G:
                ofsEntry = CLR_DARKGREEN;
                break;

           case JPE_GAMMASLIDE_B:
                ofsEntry = CLR_BLUE;
                break;
           }
         }
         else
         {
           mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
         }

         if (iIndexFound != -1)
         {
           WinFillRect( poi->hps, &(poi->rclItem), ofsEntry );
           mRC = (MRESULT) TRUE;
         }
       }
       else
       {
         mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
       }
       break;

  case WMPS_VERIFYSETTINGS:
       mRC = (MRESULT) PMERR_OK;
       break;

  case WMPS_SAVESETTINGS:
       /*
       ** 'OK' was pressed.  Save the settings.
       */
       mRC = (MRESULT) EffectsPageSaveSettings( hDlg, pJPInfo );
       break;

  case WM_CONTROLPOINTER:
       mRC = SetContextHelp( hDlg, ulMsg, mp1, mp2, NULL, 0, 0 );
       break;

  case WMPS_CONSTRAINTS:
       UpdateConstraints( hDlg, pJPInfo, JPE_COLOR_LIST, mp1, mp2 );
       break;

  default:
    mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
  }       /* end switch */

  return( mRC );
}
/*****************************************************************************\
* EffectsPageDlgProc End                                                      *
\*****************************************************************************/





/******************************************************************************
 *
 * FUNCTION NAME = EffectsPageInitControls
 *
 * DESCRIPTION
 * Initializes the controls for the "Effects" page to the current settings.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 * RETURN-ERROR  - FALSE
 *
 *****************************************************************************/
VOID EffectsPageInitControls( HWND hDlg, PJPINFO pJPInfo, INT iModifyPgFlag )
{
  HWND       hCtrl;
  PUI_BLOCK  pUIBlock;
  INT        iIndexFound;
  CHAR       aColorString[ MAX_PSIZE ];
  HAB        hAB = WinQueryAnchorBlock( hDlg );
  PCNFDATA   pCNFData = pJPInfo->pCNFData;
  PDESPPD    pdesPPD  = pJPInfo->pdesPPD;   // PPD data structure

  /* GAMMA CORRECTION GROUP START ********************************************/
  iOldRed = pCNFData->sGammaValues.lRed - 1;
  WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_R, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iOldRed ));
  iOldGreen = pCNFData->sGammaValues.lGreen - 1;
  WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_G, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iOldGreen ) );
  iOldBlue = pCNFData->sGammaValues.lBlue - 1;
  WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_B, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iOldBlue ) );
  /* GAMMA CORRECTION GROUP END **********************************************/

  /* GAMMA LOCK **************************************************************/
  if ( pCNFData->bGammaLock )
  {
    // check LOCK
    WinSendDlgItemMsg( hDlg, JPE_LOCKRGB, BM_SETCHECK,
                       MPFROMSHORT( (BOOL) TRUE ),
                       (MPARAM) 0 );
  }
  else
  {
    // uncheck LOCK
    WinSendDlgItemMsg( hDlg, JPE_LOCKRGB, BM_SETCHECK,
                       MPFROMSHORT( (BOOL) FALSE ),
                       (MPARAM) 0 );
  }
  /* GAMMA LOCK end **********************************************************/

  /* MISC START **************************************************************/
  WinSendDlgItemMsg( hDlg, JPE_FLIPTOPBTM, BM_SETCHECK,
                               MPFROMSHORT( pCNFData->effOutput.fIsFliptb ),
                               (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPE_FLIPLEFTRIGHT, BM_SETCHECK,
                     MPFROMSHORT( pCNFData->effOutput.fIsFliplr ),
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPE_DRAWINVERTED, BM_SETCHECK,
                     MPFROMSHORT( pCNFData->effOutput.fIsDrawInverted ),
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPE_SCALENUM, SPBM_SETCURRENTVALUE,
                     MPFROMLONG( pCNFData->jobProperties.uScale ),
                     (MPARAM) 0 );
  /* MISC END ****************************************************************/

  /* ENABLE COLOR START ******************************************************/
  /*
  ** Is color supported?
  */
  hCtrl = WinWindowFromID( hDlg, JPE_COLOR_LIST );

  if (pdesPPD->desItems.fIsColorDevice == TRUE)
  {
    pUIBlock = QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList,
                                      pJPInfo->pItemsBuff, UINAME_COLORMODEL,
                                      &iIndexFound );

    if (pUIBlock != NULL)
    {
      if (pUIBlock->ucPanelID != UIP_OEM_FEATURE)
      {
        /*
        ** This flag is set if "*ColorModel" is not in the PPD.  This tells
        ** the driver that the color combo box is not a UI control.
        */
        pJPInfo->usFlags |= IS_COLOR_UI;

        if (iModifyPgFlag == UCD_INITIALIZEPAGE)
        {
          InitializeUIControl( hDlg, pJPInfo, UINAME_COLORMODEL,
                               JPE_COLOR_LIST, JPE_COLOR_GROUP );
        }
        else
        {
          WinSendDlgItemMsg( hDlg, JPE_COLOR_LIST, UCM_UNDO, (MPARAM) 0,
                             (MPARAM) 0 );
        }
      }
    }
    else
    {
      if (iModifyPgFlag == UCD_INITIALIZEPAGE)
      {
        WinEnableWindow( hCtrl, TRUE );
        WinEnableControl( hDlg, JPE_COLOR_GROUP, TRUE );
//        WinSendMsg( hCtrl, LM_DELETEALL, (MPARAM) 0, (MPARAM) 0 );

        WinLoadString( hAB, pscript_module, DLG_CMYK_STRING, MAX_PSIZE,
                       aColorString );
        WinSendMsg( hCtrl, LM_INSERTITEM, MPFROMLONG( 0 ),
                    MPFROMP( aColorString ) );

        WinLoadString( hAB, pscript_module, DLG_GRAY_STRING, MAX_PSIZE,
                       aColorString );
        WinSendMsg( hCtrl, LM_INSERTITEM, MPFROMLONG( 1 ),
                    MPFROMP( aColorString ) );
      }

      if (pJPInfo->pCNFData->jobProperties.fIsColorDevice == TRUE)
      {
        iIndexFound = 0;
      }
      else
      {
        iIndexFound = 1;
      }

      WinSendMsg( hCtrl, LM_SELECTITEM, MPFROMLONG( iIndexFound ),
                  MPFROMSHORT( TRUE ) );
    }
  }
  else
  {
    UpdateIcon( hDlg, pJPInfo->hDlg, JPSB_MONOCHROME, ICON_MONOCHROME );
  }
}
/*---------------------------------------------------------------------------*\
* EffectsPageInitControls End                                                 *
\*---------------------------------------------------------------------------*/


/******************************************************************************
 *
 * FUNCTION NAME = EffectsPageSaveSettings
 *
 * DESCRIPTION
 * Saves the current settings for the Effects page.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 * RETURN-ERROR  - FALSE
 *
 *****************************************************************************/
UINT EffectsPageSaveSettings( HWND hDlg, PJPINFO pJPInfo )
{
  INT  iValue;
  UINT uiRC = PMERR_OK;
  PCNFDATA   pCNFData = pJPInfo->pCNFData;

  /*
  ** Query the current zero-based offset position of the slider from the
  ** home position.  Since the GENPLIB gamma function receives a gamma value
  ** in a scale of 10 (gamma value * 10), multiply the value by 10 and store
  ** it.  This value can be directly passed into the gamma function.  The
  ** gamma function will convert it to its real value.
  ** Since the slider is a zero-based offset, add the value of the home
  ** position to get the value.
  */
  SaveGammaVals (hDlg, JPE_GAMMASLIDE_R,
                 &(pJPInfo->pCNFData->sGammaValues.lRed));
  SaveGammaVals (hDlg, JPE_GAMMASLIDE_G,
                 &(pJPInfo->pCNFData->sGammaValues.lGreen));
  SaveGammaVals (hDlg, JPE_GAMMASLIDE_B,
                 &(pJPInfo->pCNFData->sGammaValues.lBlue));

  /*
  ** Store the LOCKRGB option.
  */
  if (WinSendDlgItemMsg( hDlg, JPE_LOCKRGB, BM_QUERYCHECK, (MPARAM) 0,
                         (MPARAM) 0 ) == (MRESULT) 1)
  {
    pCNFData->bGammaLock = TRUE;
  }
  else
  {
    pCNFData->bGammaLock = FALSE;
  }

  WinSendDlgItemMsg( hDlg, JPE_SCALENUM, SPBM_QUERYVALUE, (MPARAM) &iValue,
                     MPFROMSHORT( SPBQ_UPDATEIFVALID ) );
  pJPInfo->pCNFData->jobProperties.uScale = iValue;

  pJPInfo->pCNFData->effOutput.fIsFliptb =
    (SHORT) WinSendDlgItemMsg( hDlg, JPE_FLIPTOPBTM, BM_QUERYCHECK,
                               (MPARAM) 0, (MPARAM) 0 );
  pJPInfo->pCNFData->effOutput.fIsFliplr =
    (SHORT) WinSendDlgItemMsg( hDlg, JPE_FLIPLEFTRIGHT, BM_QUERYCHECK,
                               (MPARAM) 0, (MPARAM) 0 );
  pJPInfo->pCNFData->effOutput.fIsDrawInverted =
    (SHORT) WinSendDlgItemMsg( hDlg, JPE_DRAWINVERTED, BM_QUERYCHECK,
                               (MPARAM) 0, (MPARAM) 0 );

  /*
  ** Store the color option.
  */
  iValue = (INT) WinSendDlgItemMsg( hDlg, JPE_COLOR_LIST,
                                    LM_QUERYSELECTION,
                                    MPFROMSHORT( LIT_FIRST ), (MPARAM) 0 );

  if (iValue == 0)
  {
    pJPInfo->pCNFData->jobProperties.fIsColorDevice = TRUE;
  }
  else
  {
    pJPInfo->pCNFData->jobProperties.fIsColorDevice = FALSE;
  }

  return( uiRC );
}
/*---------------------------------------------------------------------------*\
* EffectsPageSaveSettings End                                                 *
\*---------------------------------------------------------------------------*/



/******************************************************************************
 *
 * FUNCTION NAME = FeaturePageDlgProc
 *
 * DESCRIPTION
 * This is the procedure for the Feature Options dialog box that is in the Job
 * Properties notebook.  Among the standard procedure responsibilites, this
 * procedure initializes and saves all data pertaining to the dialog, with
 * regard to the Job Properties Information structure.
 * This procedure is also responsible for displaying the correct selection
 * control.  The top part of the dialog contains a list of all supported UI
 * options for the Feature.  The bottom portion contains a group box that will
 * contain one of the following controls for the selected UI option.  The
 * ID that determines which control to use is in the PPD file with "*OpenUI":
 * - True/False radio buttons if an option is set to "Boolean".
 * - A drop-down list box if an option is set to "PickOne".  This list box
 *   allows for only one selection.
 * - A standard listbox if an option is set to "PickMany".  This list box
 *   allows for more than one selection.
 * Each time a new UI option is selected, the matching selection control is
 * displayed with the given selections.  A text line at the bottom instructs
 * the user what can be selected.  This text line is also updated in this
 * procedure.
 * The Feature page is used to allow user interaction for the Postscript
 * User Interface (UI) support.
 *
 * INPUT
 * hDlg - Dialog handle
 * ulMsg - Message ID
 * mp1, mp2 - General message parameters
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 *                 FALSE
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
MRESULT EXPENTRY FeaturePageDlgProc( HWND hDlg, ULONG ulMsg, MPARAM mp1,
                                     MPARAM mp2 )
{
  PJPINFO     pJPInfo;                 // Job Properties Info structure
  PSZ         pBlockString;            // UI block string
  LONG        lSelectType;             // Selection type: one or many
  LONG        lHideWindow;             // Show/hide control
  PUI_BLOCK   pUIBlock;                // Block pointer
  MRESULT     mRC = (MRESULT) FALSE;   // Functionr return code

  /*
  ** At WM_INITDLG, the Dialog Header (pJPInfo) has not yet been set.
  */
  if (ulMsg != WM_INITDLG)
  {
    pJPInfo = (PDLGHDR) WinQueryWindowULong( hDlg, QWL_USER );
  }

  switch( ulMsg )
  {
  case WM_INITDLG:
       pJPInfo = InitDlgULong( hDlg, mp2 );
       FeaturePageInitControls( hDlg, pJPInfo );
       break;

  case WM_COMMAND:
       switch ( SHORT1FROMMP( mp1 ) )
       {
       case PSB_UNDO:
            FeaturePageDefaultControls( hDlg, pJPInfo, FALSE );
            break;

       case PSB_DEFAULT:
            FeaturePageDefaultControls( hDlg, pJPInfo, TRUE );
            break;

       case DID_CANCEL:
         WinSendMsg( pJPInfo->hDlg, WM_COMMAND, (MPARAM) PSB_CANCEL, (MPARAM) 0 );
       }
       break;

  case WM_CONTROL:
       switch (SHORT1FROMMP( mp1 ))
       {
       case JPUI_UILIST:
            if (SHORT2FROMMP( mp1 ) == LN_SELECT)
            {
              /*
              ** The handle for the current item is a pointer to its matching
              ** UI block.
              */
              GplPageQueryLBSelection( hDlg, JPUI_UILIST, (PLONG) &pUIBlock,
                                        NULL );


              /*
              ** With block pointer, get the string.
              */
              pBlockString = (PSZ) (pJPInfo->pItemsBuff + pUIBlock->ofsUIName);

              /*
              ** For single-selection blocks (UI_SELECT_BOOLEAN and
              ** UI_SELECT_PICKONE), display the combo box that only allows
              ** single selections.  Otherwise, display the listbox that
              ** allows multiple selections (JPUI_PICKMANY).
              */
              if (pUIBlock->usSelectType == UI_SELECT_PICKMANY)
              {
                lSelectType = JPUI_PICKMANY;
                lHideWindow = JPUI_PICKONE;
              }
              else
              {
                lSelectType = JPUI_PICKONE;
                lHideWindow = JPUI_PICKMANY;
              }

              WinShowWindow( WinWindowFromID( hDlg, lHideWindow ), FALSE );
              WinShowWindow( WinWindowFromID( hDlg, lSelectType ), TRUE );

              InitializeUIControl( hDlg, pJPInfo, pBlockString, lSelectType,
                                   0 );
            }
            break;
       }
       break;

  case WMPS_VERIFYSETTINGS:
       mRC = (MRESULT) PMERR_OK;
       break;

  case WMPS_SAVESETTINGS:
       mRC = (MRESULT) PMERR_OK;
       break;

  case WM_CONTROLPOINTER:
       mRC = SetContextHelp( hDlg, ulMsg, mp1, mp2, NULL, 0, 0 );
       break;

  case WMPS_CONSTRAINTS:
       /*
       ** Constraints are not applied to installable options.  Since
       ** Installable Options appear only in Printer Properties, do
       ** not apply constraints when Printer Properties is displayed.
       */
       if (pJPInfo->uiPropertyType == DPDM_POSTJOBPROP)
       {
         UpdateConstraints( hDlg, pJPInfo, JPUI_PICKONE, mp1, mp2 );
       }
       break;

  default:
       mRC = WinDefDlgProc( hDlg, ulMsg, mp1, mp2 );
  }

  return( mRC );
}
/*---------------------------------------------------------------------------*\
* FeaturePageDlgProc End                                                       *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = FeaturePageInitControls
 *
 * DESCRIPTION
 * Initializes the controls for the "Feature" page to the current settings.
 * This is for the PostScript User Interface (UI).
 * In the UI list, each block is listed in the order defined by the
 * "*OrderDependency" keyword (the lowest value is the first in the list).
 * However, the display order in the listbox is that the first "*OpenUI"
 * keyword in the PPD is the first keyword to be displayed in the list.
 * The field 'usDisplayOrder' contains an incremental value, starting at 0,
 * that indicates which keyword is to be displayed first.  Search the UI
 * list and display in order found in each 'usDisplayOrder'.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID FeaturePageInitControls( HWND hDlg, PJPINFO pJPInfo )
{
  INT       iBlockLoop;
  INT       iDisplayLoop;
  UINT      uiGroupType;
  PUI_BLOCK pUIBlock;
  PSZ       pString;
  ULONG     ulString;
  LONG      lHandle;
  HWND      hCtrl = WinWindowFromID( hDlg, JPUI_UILIST );
  USHORT    usDisplayOrder = 0;
  INT       iNumOfBlocks = (INT) pJPInfo->pdesPPD->stUIList.usNumOfBlocks;

  WinSendMsg( hCtrl, LM_DELETEALL, (MPARAM) 0, (MPARAM) 0 );

  /*
  ** For Printer Properties, display all available options.  For Job
  ** Properties, display all default options.
  */
  if (pJPInfo->uiPropertyType == DPDM_CHANGEPROP)
  {
    uiGroupType = UIGT_INSTALLABLEOPTION;
  }
  else
  {
    uiGroupType = UIGT_DEFAULTOPTION;
  }

  for (iBlockLoop = 0 ; iBlockLoop < iNumOfBlocks ; iBlockLoop++)
  {
    pUIBlock = pJPInfo->pdesPPD->stUIList.pBlockList;
    pString = NULL;

    /*
    ** Since the block list is in "*OrderDependency" order and that the
    ** display order is not always the same as the OrderDependency order,
    ** then the UI list needs to be searched each time for the next
    ** incremental 'usDisplayOrder' keyword.  Do that here.
    */
    for (iDisplayLoop = 0 ; iDisplayLoop < iNumOfBlocks ; iDisplayLoop++)
    {
      if (pUIBlock->usDisplayOrder == usDisplayOrder)
      {
        pString = (PSZ) (pJPInfo->pItemsBuff + pUIBlock->ofsUITransString);
        break;
      }
      else
      {
        /*
        ** Go to the next block.
        */
        INCREMENT_BLOCK_PTR( pUIBlock );
      }
    }
    usDisplayOrder++;

    /*
    ** Display the keyword if the block matches the criteria.
    */
    if (pUIBlock->ucPanelID == UIP_OS2_FEATURE &&
        pUIBlock->ucGroupType == (UCHAR) uiGroupType &&
        pString != NULL)
    {
      ulString = (ULONG) pString;
      lHandle  = (LONG) pUIBlock;
      GplPageInsertListItems( hCtrl, 0, &ulString, GDPF_STRTYPEPTR,
                              &lHandle, 1, LIT_END, pscript_module );
    }
  }

  /*
  ** Select the first item in the list.
  */
  WinSendMsg( hCtrl, LM_SELECTITEM, (MPARAM) 0, MPFROMSHORT( TRUE ) );
}
/*---------------------------------------------------------------------------*\
* FeaturePageInitControls End                                                  *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = DisplayGammaValue
 *
 * DESCRIPTION
 * Displays a gamma value in a control field.  The gamma field contains the
 * gamma value times 10, since the GENPLIB function converts the value to
 * fraction.  The value is displayed as in decimal format.  If the variable
 * contains 1, the value is displayed as .1.  If 90 is provided, the value
 * is displayed as 9.0.  The only restriction here is that the value must
 * always be positive.
 *
 * INPUT
 * hParent - Handle of the parent to the slider.
 * ulControlID - Control ID where fixed value is to be displayed.
 * lValue - Value to be displayed in text control.
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID DisplayGammaValue( HWND hParent, ULONG ulControlID, LONG lValue )
{
  UINT uiStrLen;          // String length
  CHAR aSVStr[ 4 ];       // Slider value string buffer

  /*
  ** For the integer value, divide by 10 and truncate the fraction.  For
  ** the fraction, use modular divsion by 10.
  */
  aSVStr[ 0 ] = (CHAR) (lValue / 10) + '0';
  aSVStr[ 1 ] = '.';
  aSVStr[ 2 ] = (CHAR) (lValue % 10) + '0';
  aSVStr[ 3 ] = 0;

  /*
  ** Display the value string in the control.
  */
  WinSetDlgItemText( hParent, ulControlID, aSVStr );
}
/*---------------------------------------------------------------------------*\
* DisplayGammaValue End                                                       *
\*---------------------------------------------------------------------------*/


/******************************************************************************
 *
 * FUNCTION NAME = InsertSliderVals
 *
 * DESCRIPTION
 * Insert the slider values for the RGB gamma slider bars.
 *
 * INPUT
 * HWND hDlg - Job Properties dialog box handle.
 * ULONG ulControlID - Slider Control
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID InsertSliderVals( HWND hDlg, ULONG ulControlID)
{

  WinSendDlgItemMsg( hDlg, ulControlID, SLM_SETSCALETEXT,
                     MPFROMLONG( 0 ), MPFROMP( "0.1" ) );
  WinSendDlgItemMsg( hDlg, ulControlID, SLM_SETSCALETEXT,
                     MPFROMLONG( 9 ), MPFROMP( "1.0" ) );
  WinSendDlgItemMsg( hDlg, ulControlID, SLM_SETSCALETEXT,
                     MPFROMLONG( GC_SLIDE_NUM_OF_TICKS - 1 ),
                     MPFROMP( "5.0" ) );
  return;
}
/*---------------------------------------------------------------------------*\
* InsertSliderVals End                                                       *
\*---------------------------------------------------------------------------*/


/******************************************************************************
 *
 * FUNCTION NAME = GetOffset
 *
 * DESCRIPTION
 * Get offset from the slider and display the gamma value.
 *
 * INPUT
 * HWND hDlg - Job Properties dialog box handle.
 * ULONG ulControlID - Slider Control
 * BOOL bBoxChecked - is Box Checked?
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID GetOffset ( HWND hDlg, ULONG ulControlID, BOOL bBoxChecked)
{
  INT     iLoop;                  // Loop counter
  INT     iRed;
  INT     iGreen;
  INT     iBlue;
  INT     iNewColor;
  static  ULONG ulFirstColor;
  static  SHORT sCount = 0;

  iLoop = (INT) WinSendDlgItemMsg( hDlg, ulControlID,
                                  SLM_QUERYSLIDERINFO,
                  MPFROM2SHORT( SMA_SLIDERARMPOSITION,
                                SMA_INCREMENTVALUE ),
                  (MPARAM) 0 );

  /*
  ** Display the gamma value, based on the offset from the
  ** slider.
  */
  if ((bBoxChecked == TRUE) && (bNewChange))
  {
    bNewChange = FALSE;
    ulFirstColor = ulControlID;

    switch (ulFirstColor)
    {
      case JPE_GAMMASLIDE_R:
        if (!FixUpColors( iLoop, &iGreen, &iBlue,
                          &iOldRed, &iOldGreen, &iOldBlue ))
        {
          //Red can't go as far as we wanted to because Green or Blue
          //hit the end.  Red is reset to as far as it can go, so now
          //send a message to get that one changed.
          iLoop = iOldRed;
          WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_R, SLM_SETSLIDERINFO,
                       MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                       MPFROMSHORT( iLoop ) );
        }
        WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_G, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iGreen ) );
        WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_B, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iBlue ) );
        break;
      case JPE_GAMMASLIDE_G:
        if (!FixUpColors( iLoop, &iRed, &iBlue,
                          &iOldGreen, &iOldRed, &iOldBlue ))
        {
          //Green can't go as far as we wanted to because Red or Blue
          //hit the end.  Green is reset to as far as it can go, so now
          //send a message to get that one changed.
          iLoop = iOldGreen;
          WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_G, SLM_SETSLIDERINFO,
                       MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                       MPFROMSHORT( iLoop ) );
        }
        WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_R, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iRed ) );
        WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_B, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iBlue ) );
        break;
      case JPE_GAMMASLIDE_B:
        if (!FixUpColors( iLoop, &iRed, &iGreen,
                          &iOldBlue, &iOldRed, &iOldGreen ))
        {
          //Blue can't go as far as we wanted to because Red or Green
          //hit the end.  Blue is reset to as far as it can go, so now
          //send a message to get that one changed.
          iLoop = iOldBlue;
          WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_B, SLM_SETSLIDERINFO,
                       MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                       MPFROMSHORT( iLoop ) );
        }
        WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_R, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iRed ) );
        WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_G, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iGreen ) );
        break;
    }
  }

  if (bBoxChecked == TRUE)
    sCount++;
  if ((sCount >= 3) || (bBoxChecked == FALSE))
  {
    sCount = 0;
    bNewChange = TRUE;
  }

  switch (ulControlID)
  {
    case JPE_GAMMASLIDE_R:
      iOldRed = iLoop;
      DisplayGammaValue( hDlg, JPE_GAMMAVALNUM_R, (LONG) iLoop + 1 );
      break;
    case JPE_GAMMASLIDE_G:
      iOldGreen = iLoop;
      DisplayGammaValue( hDlg, JPE_GAMMAVALNUM_G, (LONG) iLoop + 1 );
      break;
    case JPE_GAMMASLIDE_B:
      iOldBlue = iLoop;
      DisplayGammaValue( hDlg, JPE_GAMMAVALNUM_B, (LONG) iLoop + 1 );
      break;
  }



  return;
}
/*---------------------------------------------------------------------------*\
* GetOffset        End                                                       *
\*---------------------------------------------------------------------------*/


/******************************************************************************
 *
 * FUNCTION NAME = SaveGammaVals
 *
 * DESCRIPTION
 * Save the gamma values from the sliders.
 *
 * INPUT
 * HWND hDlg - Job Properties dialog box handle.
 * ULONG ulControlID - Slider Control
 * PLONG pulColor - Color
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID SaveGammaVals ( HWND hDlg, ULONG ulControlID, PLONG plColor)
{
  INT  iValue;

  iValue = (INT) WinSendDlgItemMsg( hDlg, ulControlID,
                                    SLM_QUERYSLIDERINFO,
                                    MPFROM2SHORT( SMA_SLIDERARMPOSITION,
                                                  SMA_INCREMENTVALUE ),
                                    (MPARAM) 0 );

  *plColor = (LONG) (iValue + 1);

  return;
}
/*---------------------------------------------------------------------------*\
* SaveGammaVals    End                                                       *
\*---------------------------------------------------------------------------*/



/******************************************************************************
 *
 * FUNCTION NAME = FixUpColors
 *
 * DESCRIPTION
 * Save the gamma values from the sliders.
 *
 * INPUT
 * HWND hDlg - Job Properties dialog box handle.
 * ULONG ulControlID - Slider Control
 * PLONG pulColor - Color
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
BOOL FixUpColors ( INT iColor1, PINT pColor2, PINT pColor3,
                  PINT pOld1, PINT pOld2, PINT pOld3)
{

  INT     iColorDiff;
  INT     iMinSlide;
  INT     iMaxSlide;
  INT     iNewColor1;

  iMinSlide = MIN_GAMMA -1;
  iMaxSlide = MAX_GAMMA -1;

  if (*pOld1 > iColor1)
  {
    iColorDiff = *pOld1 - iColor1;
    *pColor2 = *pOld2 - iColorDiff;
    *pColor3 = *pOld3 - iColorDiff;
  }
  else
  {
    iColorDiff = iColor1 - *pOld1;
    *pColor2 = *pOld2 + iColorDiff;
    *pColor3 = *pOld3 + iColorDiff;
  }

  iNewColor1 = iColor1;
  if ((*pColor2 > iMaxSlide) ||
      (*pColor3 > iMaxSlide) ||
      (*pColor2 < iMinSlide) ||
      (*pColor3 < iMinSlide))
  {
    iNewColor1 = *pOld1;
    *pColor2 = *pOld2;
    *pColor3 = *pOld3;
    return(FALSE);
  }

  *pOld1 = iNewColor1;
  *pOld2 = *pColor2;
  *pOld3 = *pColor3;
  return(TRUE);

}
/*---------------------------------------------------------------------------*\
* SaveGammaVals    End                                                       *
\*---------------------------------------------------------------------------*/



/******************************************************************************
 *
 * FUNCTION NAME = FormPageDefaultControls
 *
 * DESCRIPTION
 * Defaults the controls for the "Forms" page.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID FormPageDefaultControls( HWND hDlg, PJPINFO pJPInfo )
{
  PCHAR      pBuffer;                       // Local buffer pointer
  PDESPPD    pdesPPD  = pJPInfo->pdesPPD;   // PPD data structure
  PUI_BLOCK  pUIBlock;
  LONG       lSelection;
  PSZ        pDefFormName;
  INT        iIndexFound = -1;

  WinSendDlgItemMsg( hDlg, JPF_FORMNAMELIST, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );
  //Default the tray list
  WinSendDlgItemMsg( hDlg, JPF_TRAYNAMELIST, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );
  WinSendDlgItemMsg( hDlg, JPF_MEDIATYPELIST, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );

  // @MIXED_FEED Begin
  WinSendDlgItemMsg( hDlg, JPF_MIXED_FEED, BM_SETCHECK, MPFROMSHORT( FALSE ), (MPARAM) 0 );
  // Send BN_CHECK notify message. Set MP2=0, to indicate, that message is simulated
  WinSendMsg(hDlg, WM_CONTROL, MPFROM2SHORT(JPF_MIXED_FEED,BN_CLICKED), MPFROMSHORT(0));
  
  WinSendDlgItemMsg( hDlg, JPF_FORMNAMELIST_1, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );
  //Default the tray list
  WinSendDlgItemMsg( hDlg, JPF_TRAYNAMELIST_1, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );
  WinSendDlgItemMsg( hDlg, JPF_MEDIATYPELIST_1, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );
  // @MIXED_FEED End
  
  pUIBlock = (PUI_BLOCK) WinQueryWindowULong( WinWindowFromID( hDlg, JPF_COLLATELIST ), QWL_USER );
  if (pUIBlock)
  {
    WinSendDlgItemMsg( hDlg, JPF_COLLATELIST, UCM_SETDEFAULT, (MPARAM) 0,
                       (MPARAM) 0 );                          //@V4.COLL2FP
  }
  else
  {
    GplCtrlLBSelectItem( hDlg,
                         JPF_COLLATELIST,
                         (MPARAM) NONE_COLLATE_HANDLE,
                         GLF_SELECTHANDLE );
  }


  /*
  ** NOTE:  The defaults are taken from PRDGEXTF.H, which proved to be a
  **        pain to include in this file.
  **                DEFAULT_ORIENT = PORTRAIT;
  */
  WinSendDlgItemMsg( hDlg, (ULONG) JPF_PORTRAIT, BM_CLICK,
                     MPFROMLONG( TRUE ), (MPARAM) 0 );

  /*
  ** Send this message to set the current orientation icon.
  */
  WinSendMsg( hDlg, WM_CONTROL, MPFROMSHORT( (SHORT) JPF_PORTRAIT ),
              (MPARAM) 0 );

  /* DUPLEX START ************************************************************/
  if ((pUIBlock = QueryBlockFromKeyword( &pJPInfo->pdesPPD->stUIList,
                                         pJPInfo->pItemsBuff,
                                         UINAME_DUPLEX, &iIndexFound ))
       != NULL)
  {
    *(pJPInfo->pCurrUISelList + iIndexFound) =
       ((UI_SEL) 1 << pUIBlock->usDefaultEntry );

    pBuffer = (PCHAR) (pJPInfo->pItemsBuff +
              pUIBlock->uiEntry[ pUIBlock->usDefaultEntry ].ofsOption);

    if (!strcmp( pBuffer, UINAME_DUPLEXTUMBLE ))
    {
      lSelection = JPF_DUPLEXTUMBLE;
    }
    else if (!strcmp( pBuffer, UINAME_DUPLEXNOTUMBLE ))
    {
      lSelection = JPF_DUPLEXNOTUMBLE;
    }
    else
    {
      lSelection = JPF_DUPLEXNONE;
    }
  }
  else
  {
    lSelection = JPF_DUPLEXNONE;
  }

  WinSendDlgItemMsg( hDlg, lSelection, BM_SETCHECK, MPFROMSHORT( 1 ),
                     (MPARAM) 0 );

  /*
  ** Send this message to set the current duplex icon.
  */
  WinSendMsg( hDlg, WM_CONTROL, MPFROMSHORT( lSelection ), (MPARAM) 0 );
  /* DUPLEX END **************************************************************/

  WinSendDlgItemMsg( hDlg, JPF_COPIESNUM, SPBM_SETCURRENTVALUE,
                     MPFROMLONG( 1 ), (MPARAM) 0 );

}
/*---------------------------------------------------------------------------*\
* FormPageDefaultControls End                                                 *
\*---------------------------------------------------------------------------*/



/******************************************************************************
 *
 * FUNCTION NAME = OutputPageDefaultControls
 *
 * DESCRIPTION
 * Defaults the controls for the "Output" page.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - None
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
VOID OutputPageDefaultControls( HWND hDlg, PJPINFO pJPInfo )
{
  PDESPPD    pdesPPD  = pJPInfo->pdesPPD;   // PPD data structure
  HWND       hCtrl = WinWindowFromID( hDlg, JPO_FILEENTRY );
  ULONG      ulControl;
  USHORT     usChoice;
  SHORT      sResCount;
  SHORT      sLoop;
  CHAR       szResString[ MAX_PSIZE ];         // Local string buffer
  CHAR       szString[ MAX_PSIZE ];            // Local string buffer

  
  pJPInfo->usFlags |= UIC_NO_ASK; // keep unconstraining quiet //@243371

  /* PRINT TO GROUP START ****************************************************/
  // Check the printer button and disable and clear the file entry field
  WinSendDlgItemMsg( hDlg, JPO_PRINTER, BM_SETCHECK,
                     MPFROMSHORT( TRUE ), (MPARAM) 0 );
  WinEnableWindow( WinWindowFromID( hDlg, JPO_FILETEXT ),
                   (BOOL) FALSE );
  WinSetDlgItemText( hDlg, JPO_FILEENTRY, "" );
  WinEnableWindow( hCtrl, (BOOL) FALSE );

  /*
  ** This message forces a repaint of the Output group of the Status Bar.
  ** Without this message, the Status Bar output group may be blank at
  ** initialization.
  */
  WinSendMsg( hDlg, WM_CONTROL, MPFROMSHORT( JPO_PRINTER ), (MPARAM) 0 );

  /* PRINT TO GROUP END ******************************************************/
  WinSendDlgItemMsg( hDlg, JPO_REPDOWNFONT, BM_SETCHECK,
                     MPFROMSHORT( TRUE ),
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPO_REPDEVFONTS, BM_SETCHECK,
                     MPFROMSHORT( TRUE ),
                     (MPARAM) 0 );

  // Should we be checking pdesPPD->desItems.usLanguageLevel?  Right
  // now, we're just defaulting the checkbox to off
  usChoice = FALSE;
  WinSendDlgItemMsg( hDlg, JPO_LVL1COMPAT, BM_SETCHECK,
                     MPFROMSHORT( usChoice ),
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPO_LVL1COMPAT, BM_SETCHECK, (MPARAM) 0,
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPO_FONTSUB, BM_SETCHECK, NULL, NULL );

  // @V3.1142412
  WinSendDlgItemMsg( hDlg, JPO_ERRHNDLR, BM_SETCHECK, (MPARAM) 0, (MPARAM) 0 );

  // @V4.0190777
  WinSendDlgItemMsg( hDlg, JPO_UNIXCOMPAT, BM_SETCHECK, NULL, NULL );
  /* MISC END ****************************************************************/

  /* RESOLUTION START ********************************************************/
  WinSendDlgItemMsg( hDlg, JPO_RESOLUTIONLIST, UCM_SETDEFAULT, (MPARAM) 0,
                     (MPARAM) 0 );
  /* RESOLUTION END **********************************************************/

  pJPInfo->usFlags &= ~UIC_NO_ASK; // be interactive again //@243371

}
/*****************************************************************************\
* OutputPageDefaultControls End                                               *
\*****************************************************************************/




/******************************************************************************
 *
 * FUNCTION NAME = EffectsPageDefaultControls
 *
 * DESCRIPTION
 * Defaults the controls for the "Effects" page to the current settings.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 * RETURN-ERROR  - FALSE
 *
 *****************************************************************************/
VOID EffectsPageDefaultControls( HWND hDlg, PJPINFO pJPInfo )
{
  HWND       hwndLocal;
  PDESPPD    pdesPPD  = pJPInfo->pdesPPD;   // PPD data structure
  UINT       uiOffset;
//@V3.130814
  HAB        hAB = WinQueryAnchorBlock( hDlg );
  HWND       hCtrl = WinWindowFromID( hDlg, JPE_COLOR_LIST );
  CHAR       aColorString[ MAX_PSIZE ];
  PUI_BLOCK  pUIBlock;
  INT        iIndexFound = -1;
  BOOL       fColorFlag;
//@V3.130814 end

  /* GAMMA CORRECTION GROUP START ********************************************/
  iOldRed = NO_GAMMA - 1;
  WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_R, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iOldRed ));
  iOldGreen = NO_GAMMA - 1;
  WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_G, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iOldGreen ) );
  iOldBlue = NO_GAMMA - 1;
  WinSendDlgItemMsg( hDlg, JPE_GAMMASLIDE_B, SLM_SETSLIDERINFO,
                     MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE ),
                     MPFROMSHORT( iOldBlue ) );

  // uncheck LOCK
  WinSendDlgItemMsg( hDlg, JPE_LOCKRGB, BM_SETCHECK,
                     MPFROMSHORT( (BOOL) FALSE ),
                     (MPARAM) 0 );

  /* GAMMA CORRECTION GROUP END **********************************************/

  /* MISC START **************************************************************/
  // NOTE:  The defaults are taken from PRDGEXTF.H, which proved to be a
  //        pain to include in this file.
  //            DEFAULT_TOPBOTTOM = FALSE;
  //            DEFAULT_LEFTRIGHT = FALSE;
  //            DEFAULT_INVERT    = FALSE;

  WinSendDlgItemMsg( hDlg, JPE_FLIPTOPBTM, BM_SETCHECK,
                               MPFROMSHORT( (BOOL) FALSE ),
                               (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPE_FLIPLEFTRIGHT, BM_SETCHECK,
                     MPFROMSHORT( (BOOL) FALSE ),
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPE_DRAWINVERTED, BM_SETCHECK,
                     MPFROMSHORT( (BOOL) FALSE ),
                     (MPARAM) 0 );

  WinSendDlgItemMsg( hDlg, JPE_SCALENUM, SPBM_SETCURRENTVALUE,
                     MPFROMLONG( (SHORT) 100 ),
                     (MPARAM) 0 );
  /* MISC END   **************************************************************/

  /* COLOR START *************************************************************/
  if (WinIsWindowEnabled( WinWindowFromID( hDlg, JPE_COLOR_LIST ) ) == TRUE)
  {
    if (pJPInfo->usFlags & IS_COLOR_UI)
    {
      WinSendDlgItemMsg( hDlg, JPE_COLOR_LIST, UCM_SETDEFAULT, (MPARAM) 0,
                         (MPARAM) 0 );
    }
    else
    {
      WinSendMsg( hCtrl, LM_SELECTITEM, (MPARAM) 0, MPFROMSHORT( TRUE ) );
    }
  }

//@V3.130814 end
}
/*---------------------------------------------------------------------------*\
* EffectsPageDefaultControls End                                              *  *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = FeaturePageDefaultControls
 *
 * DESCRIPTION
 * Default/Undo the controls for the "Feature" page.
 *
 * INPUT
 * hDlg - Form page dialog handle
 * pJPInfo - Job Information Structure pointer
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL - TRUE
 * RETURN-ERROR  - FALSE
 *
 *****************************************************************************/
VOID FeaturePageDefaultControls( HWND hDlg, PJPINFO pJPInfo, BOOL bIsDefault )
{
  INT       iIndex;
  PUI_BLOCK pUIBlock = pJPInfo->pdesPPD->stUIList.pBlockList;
  PUI_SEL   pTempUISel = pJPInfo->pCurrUISelList;
  PUI_SEL   pUISelBase = ASSIGN_UISELLIST_PTR( pJPInfo->pCNFData );
  UCHAR     ucTypeCheck;

  /*
  ** pUISelBase is the pointer to the UI_SELLIST in pCNFData.  This
  ** is the list that existed at the dialog initialization.
  ** pTempUISel is a pointer to a temporary UI_SELLIST.  This list
  ** exists as long as the dialog exists.  At save and exist, this
  ** list is copied to pCNFData.
  ** Therefore, for any copy, copy data from the pCNFData UI_SELLIST
  ** to the temporary buffer.
  */
  if (pJPInfo->uiPropertyType == DPDM_CHANGEPROP)
  {
    ucTypeCheck = UIGT_INSTALLABLEOPTION;
  }
  else
  {
    ucTypeCheck = UIGT_DEFAULTOPTION;
  }

  /*
  ** Run through the list of supported UIs
  */
  for (iIndex = 0 ; iIndex < (INT)
       pJPInfo->pdesPPD->stUIList.usNumOfBlocks ; iIndex++)
  {
    /*
    ** Only assign values to the types of blocks that correspond to the
    ** input type:
    **        UIGT_DEFAULTOPTION for Job Properties
    **        UIGT_INSTALLABLEOPTION for Printer Properties
    */
    if (pUIBlock->ucGroupType == ucTypeCheck )
    {
      if (bIsDefault)
      {
        *pTempUISel = (UI_SEL) (1 << pUIBlock->usDefaultEntry);
      }
      else
      {
        *pTempUISel = *pUISelBase;
      }
    }

    pUISelBase++;
    pTempUISel++;
    INCREMENT_BLOCK_PTR( pUIBlock );
  }

  // select the first one in the list . . .
  WinSendDlgItemMsg( hDlg, JPUI_UILIST, LM_SELECTITEM, (MPARAM) 0,
                     MPFROMSHORT( TRUE ) );
  WinSendMsg( hDlg, WM_CONTROL, MPFROM2SHORT( JPUI_UILIST, LN_SELECT ),
              MPFROMSHORT( TRUE ) );
}
/*---------------------------------------------------------------------------*\
* FeaturePageDefaultControls End                                              *  *
\*---------------------------------------------------------------------------*/





LONG ListJobPropFormEntries( HWND      hCtrl,
                             PDLGHDR   pDlgHdr,
                             PUI_BLOCK pFormBlock
                           )
{
  #define NO_FORM_SELECTED   ((INPBINS + 10) * -1)

  PSZ       pCurrRealForm;
  PSZ       pCurrDispForm;
  LONG      lIndex;
  PUI_BLOCK pTrayBlock;
  PUI_ENTRY pUIEntry;
  PULONG    pStrPtrList;
  PLONG     pHndlList;
  PBYTE     pBasePtr;
  UI_SEL    uiSel;
  LONG      lFormSelected = NO_FORM_SELECTED;
  LONG      lNumOfForms = 0;
  PSZ       pSelectedForm;
  PUNQFRM   pUnqFormList = pDlgHdr->stUnqFrmLst;

  // @MIXED_FEED Begin
  switch(WinQueryWindowUShort( hCtrl, QWS_ID ))
  {
    case JPF_FORMNAMELIST:
      pSelectedForm = (PSZ) pDlgHdr->pCNFData->jobProperties.szFormName;
      break;
    case JPF_FORMNAMELIST_1:
      pSelectedForm = (PSZ) pDlgHdr->pCNFData->szFormName_1;
      break;
  }
  // @MIXED_FEED End
  
  /*
  ** To calculate the amount of memory needed, count the number of predefined
  ** forms, plus the total possible number of unique forms.  Since the values
  ** are to be stored in ULONG's, multiply that value by the size of a ULONG.
  */
  lIndex = (LONG) ((pFormBlock->usNumOfEntries + NUM_OF_UNQ_FORMS)
                   * sizeof( ULONG ));

  /*
  ** Allocate memory needed where the string pointers and handles will be
  ** stored.  Double the amount of memory calculated in lIndex (above).  One
  ** buffer will be for the string pointer list, the other will be for the
  ** matching handle list.
  */
  if ((pBasePtr = (PBYTE) GplMemoryAlloc( (HMCB) 0, lIndex * 2 ))
      != NULL)
  {
    pStrPtrList = (PULONG) pBasePtr;
    pHndlList   = (PLONG) (pBasePtr + lIndex);

    pTrayBlock = QueryBlockFromKeyword( &pDlgHdr->pdesPPD->stUIList,
                                        pDlgHdr->pItemsBuff, UINAME_INPUTSLOT,
                                        NULL );

    // List the predefined forms.
    pUIEntry = pFormBlock->uiEntry;
    for (lIndex = 0 ; lIndex < (LONG) pFormBlock->usNumOfEntries ; lIndex++)
    {
      // From the UI entry offset, query the current form string.
      pCurrDispForm = (PSZ) (pDlgHdr->pItemsBuff + pUIEntry->ofsTransString);
      pCurrRealForm = (PSZ) (pDlgHdr->pItemsBuff + pUIEntry->ofsOption);

      /*
      ** Display the form IF:
      ** - The "Display Selected Forms Only" checkbox was not checked in
      **   Printer Properties (DISP_SEL_FORMS == 0)
      ** OR
      ** - DISP_SEL_FORMS is checked and the form name cannot be found in
      **   the array of trays.  This means that the form was not associated
      **   with a tray, so do not display.
      */
      if ((pDlgHdr->usFlags & DISP_SEL_FORMS) == 0 ||
          QueryTrayArray( pDlgHdr, pCurrRealForm ) >= 0)
      {
        pStrPtrList[ lNumOfForms ] = (ULONG) pCurrDispForm;
        pHndlList[ lNumOfForms++ ] = lIndex;

        /*
        ** If the current form matches the selected form, then set the
        ** appropriate flag so that the form can be highlighted later.
        */
        if (!strcmp( pSelectedForm, pCurrRealForm ))
        {
          lFormSelected = lIndex;
        }
      }

      pUIEntry++;
    }


    // List any unique forms.
    pUnqFormList = pDlgHdr->stUnqFrmLst;
    for (lIndex = 0 ; lIndex < NUM_OF_UNQ_FORMS ; lIndex++)
    {
      if (pUnqFormList->aFormName[ 0 ] != 0)
      {
        pStrPtrList[ lNumOfForms ] = (ULONG) pUnqFormList->aFormName;
        pHndlList[ lNumOfForms++ ] = CONV_UNQID_TO_LBID( lIndex );

        if (!strcmp( pSelectedForm, pUnqFormList->aFormName ))
        {
          lFormSelected = CONV_UNQID_TO_LBID( lIndex );
        }
      }

      pUnqFormList++;
    }

    /*
    ** If any fors were listed in the arrays, insert them now.
    */
    if (lNumOfForms > 0)
    {
      GplPageInsertListItems( hCtrl, 0, pStrPtrList, GDPF_STRTYPEPTR,
                              pHndlList, lNumOfForms, LIT_END,
                              pscript_module );
    }

    /*
    ** If a form was not selected, then select the default form.
    */
    if (lFormSelected == NO_FORM_SELECTED)
    {
      uiSel = *(pDlgHdr->pCurrUISelList + pFormBlock->usOrderDep);
      lFormSelected = -1;

      do
      {
        uiSel >>= 1;

        lFormSelected++;
      } while (uiSel != 0);
    }

    GplPageSelectListItem( hCtrl, 0, lFormSelected, GDPI_HANDLE );

    GplMemoryFree( pBasePtr );
  }                                           // GplMemoryAlloc

  return( lNumOfForms );
}





/******************************************************************************
 *
 * FUNCTION NAME = QueryTrayArray
 *
 * DESCRIPTION
 * This function runs through the array of predefined tray/form mapped strings
 * in the PSOURCE structure to see if a match exists between any string in
 * the array and a given string.
 *
 * Each element in this array corresponds to a tray.  The order is set by the
 * PPD.  For example, the first tray in the PPD corresponds to the first array
 * element.  The string in each array contains the form name that the tray is
 * mapped.
 *
 * INPUT
 * pDlgHdr - Dialog header structure.
 * pFormString - Form string to find the match.
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - 0 or greater - Contains the array element where the match
 *                 was found.
 * RETURN-ERROR  - -1 - No match found.
 *
 *****************************************************************************/
LONG QueryTrayArray( PDLGHDR pDlgHdr,
                     PSZ     pFormString
                   )
{
  LONG    lIndex;
  PSOURCE pSource = pDlgHdr->pCNFData->u.iv.pSourcePaper;
  LONG    lRC = -1;

  for (lIndex = 0 ; lIndex < INPBINS ; lIndex++)
  {
    if (!strcmp( pSource->szPaperName[ lIndex ], pFormString ))
    {
      lRC = lIndex;
      break;
    }
  }

  return( lRC );
}





/******************************************************************************
 *
 * FUNCTION NAME = ListJobPropTrayEntries
 *
 * DESCRIPTION
 * Lists the trays that have been assigned to the currently displayed form
 * (this form is displayed in the "Paper" listbox).  This assignment was set
 * in Printer Properties.
 * This function also displays two additional forms:
 * - "Auto Tray Select" - This is displayed only if 0 or more than 1 tray
 *   was set in Printer Properties.  Auto Tray Select tells the printer to
 *   automatically select the first available tray to the current form.  The
 *   reason     this is not displayed for only one tray is, although it is OK
 *   to display both trays (Auto Tray and the assigned tray), it is redundant
 *   and confusing to the user.  Auto Tray was provided for multiple trays or
 *   for zero trays to allow the user to temporarily change the tray without
 *   having to make any modifications in Printer Properties.
 * - "Manual Feed" - This is displayed only if the printer supports Manual
 *   Feed.
 *
 * INPUT
 * hTrayCtrl - Tray listbox handle.
 * pDlgHdr - Dialog box header.
 * pTrayBlock - UI block for the trays.
 * hFormCtrl - Job Properties form control handle.
 * lFormSelection - Zero-based offset of the current item selected in the
 *   Forms listbox.
 *
 * OUTPUT
 * None.
 *
 * RETURN-NORMAL - Number of trays listed.
 * RETURN-ERROR  - None
 *
 *****************************************************************************/
LONG ListJobPropTrayEntries( HWND      hTrayCtrl,
                             PDLGHDR   pDlgHdr,
                             PUI_BLOCK pTrayBlock,
                             HWND      hFormCtrl,
                             LONG      lFormSelection
                           )
{
  PSOURCE   pSource;
  CHAR      aATString[ MAX_PSIZE ];
  CHAR      aMFString[ MAX_PSIZE ];
  ULONG     aulStrList[ INPBINS ];
  LONG      alHndlList[ INPBINS ];
  LONG      lIndex;
  PSZ       pFormName;
  PSZ       aTraySelected;
  LONG      lTraySelected = 0;
  LONG      lExcludeATString;
  PUI_ENTRY pTrayEntry = pTrayBlock->uiEntry;
  PUI_BLOCK pFormBlock = (PUI_BLOCK) WinQueryWindowULong( hFormCtrl, QWL_USER );
  LONG      lNumOfTrays = 1;               // Reserve for Auto Tray Select

  // @MIXED_FEED Begin
  switch(WinQueryWindowUShort( hTrayCtrl, QWS_ID ))
  {
    case JPF_TRAYNAMELIST:
      aTraySelected = pDlgHdr->pCNFData->u.iv.aTraySelected;
      break;
    case JPF_TRAYNAMELIST_1:
      aTraySelected = pDlgHdr->pCNFData->aTraySelected_1;
      break;
  }
  // @MIXED_FEED End

  
  // Clear any existing strings.
  WinSendMsg( hTrayCtrl, LM_DELETEALL, (MPARAM) 0, (MPARAM) 0 );

  /*
  ** Retrieve the handle and convert to either a pre-defined form or a user-
  ** defined form.  We need the form name to match what is set with the trays.
  */
  if (!IS_UNQ_FORMID( lFormSelection ))
  {
    pFormName = (PSZ) (pFormBlock->uiEntry[ lFormSelection ].ofsOption +
                       pDlgHdr->pItemsBuff);
  }
  else              // lIndex is negative -> user-defined form
  {
    lIndex = CONV_UNQID_TO_LBID( lFormSelection );
    pFormName = (PSZ) pDlgHdr->stUnqFrmLst[ lIndex ].aFormName;
  }

  /*
  ** With the given form name, list all trays that are mapped to the form.
  ** Remember that lNumOfTrays is set to 1 (above).  This is because
  ** GplPageInsertListItems() inserts the strings in the order that they are
  ** found in the array.  Since we want Auto Tray Select to be first in the
  ** list, then it must be first in the array.  Insert the predefined trays
  ** starting on the second array element.
  */
  pSource = pDlgHdr->pCNFData->u.iv.pSourcePaper;
  for (lIndex = 0 ; lIndex < (LONG) pTrayBlock->usNumOfEntries ; lIndex++)
  {
    // Match?  Then display.
    if (!strcmp( pFormName, pSource->szPaperName[ lIndex ]))
    {
      // Set the string address in the array.
      aulStrList[ lNumOfTrays ] =
        (ULONG) (pDlgHdr->pItemsBuff + pTrayEntry->ofsTransString);

      /*
      ** If the current tray matches the selected one, then set the current
      ** handle to be highlighted later.
      */
      if (!strcmp( aTraySelected,
                   (PCHAR) (pDlgHdr->pItemsBuff + pTrayEntry->ofsOption) ))  
      
      {
        lTraySelected = lIndex;
      }

      alHndlList[ lNumOfTrays++ ] = lIndex;
    }

    pTrayEntry++;
  }

  /*
  ** "Auto Tray Select" should be displayed if there are no predefined trays,
  ** or if there are two or more predefined trays listed.  For one tray,
  ** A.T.S. is not needed because it is redundant.  If desired, set the first
  ** array element for A.T.S. and load the string.
  */
  if (lNumOfTrays == 1 || lNumOfTrays > 2)
  {
    aulStrList[ 0 ] = (ULONG) aATString;
    alHndlList[ 0 ] = AUTOTRAY_HANDLE;

    WinLoadString( (HAB) 0, pscript_module, JPF_AUTOTRAYSELECT,
                   sizeof( aATString ), (PSZ) aATString );

    // If this is the current selection, set the current selection flag.
    if (!strcmp( aTraySelected, AUTOTRAY_STRING ))
    {
      lTraySelected = AUTOTRAY_HANDLE;
    }

    lExcludeATString = 0;          // Include A.T. array element (0).
  }
  else
  {
    lExcludeATString = 1;         // Ignore A.T. array, go to array 1.
  }

  /*
  ** If Manual Feed is supported, then add "Manual Feed" in at the end of the
  ** array list.
  */
  if (QueryBlockFromKeyword( &pDlgHdr->pdesPPD->stUIList,
                             pDlgHdr->pItemsBuff, UINAME_MANUALFEED,
                             NULL) != NULL)
  {
    aulStrList[ lNumOfTrays ]   = (ULONG) aMFString;
    alHndlList[ lNumOfTrays++ ] = MANUALFEED_HANDLE;

    WinLoadString( (HAB) 0, pscript_module, JPF_MANUALFEED,
                   sizeof( aMFString ), (PSZ) aMFString );

    // If this is the current selection, set the flag.
    if (!strcmp( aTraySelected, MANUALFEED_STRING ))
    {
      lTraySelected = MANUALFEED_HANDLE;
    }
  }

  /*
  ** If any trays were inserted in the array, display them here.  If Auto Tray
  ** Select is not to be inserted, then lExcludeATString must be set to 1.
  */
  if ((lNumOfTrays - lExcludeATString) > 0)
  {
    GplPageInsertListItems( hTrayCtrl, 0, &aulStrList[ lExcludeATString ],
                            GDPF_STRTYPEPTR, &alHndlList[ lExcludeATString ],
                            lNumOfTrays - lExcludeATString, LIT_END,
                            pscript_module );
  }

  /*
  ** Highlight the current selection.  If the function returns FALSE, then
  ** highlight the first item.
  */
  if (GplPageSelectListItem( hTrayCtrl, 0, lTraySelected, GDPI_HANDLE )
      == FALSE)
  {
    WinSendMsg( hTrayCtrl, LM_SELECTITEM, (MPARAM) 0, (MPARAM) TRUE );
  }

  return( lNumOfTrays - lExcludeATString );
}





VOID UpdateJobPropFormSelection( HWND      hFormCtrl,
                                 PDLGHDR   pDlgHdr,
                                 PUI_BLOCK pFormBlock,
                                 ULONG     ulCtrlID
                               )
{
  LONG lHandle;
  LONG lTrueHandle;
  ULONG ulTrayNameCtrlID;


  // @MIXED_FEED Begin
  switch(ulCtrlID)
  {
    case JPF_FORMNAMELIST:
      ulTrayNameCtrlID = JPF_TRAYNAMELIST;
      break;
    case JPF_FORMNAMELIST_1:
      ulTrayNameCtrlID = JPF_TRAYNAMELIST_1;
      break;
  }
  // @MIXED_FEED Begin

  /*
  ** Query the current selection from the form control.
  */
  GplPageQueryLBSelection( hFormCtrl, 0, &lHandle, NULL );

  /*
  ** If the handle is less than zero, then it is a user-defined form.
  ** What is needed is to map the user-defined form to the matching real
  ** form.  This is stored in sPrtHndlMap.
  */
  if (lHandle < 0)
  {
    lTrueHandle = CONV_LBID_TO_UNQID( lHandle );
    lTrueHandle = (LONG) pDlgHdr->stUnqFrmLst[ lHandle ].sPrtHndlMap;
  }
  else
  {
    lTrueHandle = lHandle;
  }

  /*
  ** Set the UI Selector bit to the real form handle.
  */
  *(pDlgHdr->pCurrUISelList + pFormBlock->usOrderDep) =
    CONVERT_OFS_TO_UISEL( lTrueHandle );

  /*
  ** Send a message to the tray list to tell it that a form has been selected.
  */
  WinSendDlgItemMsg( WinQueryWindow( hFormCtrl, QW_PARENT ), ulTrayNameCtrlID,
                     UCM_LISTITEMS, (MPARAM) hFormCtrl, (MPARAM) lHandle );

  SendConstraintsMessages( pDlgHdr, ulCtrlID, lTrueHandle, pFormBlock );
}





ULONG DisplayCSHText( HWND hDlg, PSZ pString )
{
  return WinSetDlgItemText( hDlg, PS_CONTEXTHELP, pString );
}






VOID SetJobpropPageData(
  HWND    hPage,
  PDLGHDR pDlgHdr
)
{
  #define NUM_OF_VALUES   2

  PSZ        pPageSelected;
  PSHORT     pPaperDim;
  PUI_BLOCK  pFormBlock;
  LONG       lSelected;
  PUNQFRM    pUniqueForm;
  FORMSTRUCT formStruct;
  SIZEL      pageSize;
  RECTL      rclImgArea;
  MPARAM     mpValues[ NUM_OF_VALUES ];

  GplPageQueryLBSelection( hPage, JPF_FORMNAMELIST, &lSelected, NULL );

  if (IS_UNQ_FORMID( lSelected ))
  {
    pUniqueForm = &pDlgHdr->stUnqFrmLst[ CONV_LBID_TO_UNQID( lSelected ) ];
    if ( pUniqueForm->usCustWidth && pUniqueForm->usCustHeight )
    {
      pageSize.cx = pUniqueForm->usCustWidth;
      pageSize.cy = pUniqueForm->usCustHeight;

      rclImgArea.xLeft  = rclImgArea.yBottom = 0;
      rclImgArea.xRight = pageSize.cx;
      rclImgArea.yTop   = pageSize.cy;
    }
    else
    {
      lSelected = pUniqueForm->sPrtHndlMap;
    }
  }
  
  if (!IS_UNQ_FORMID( lSelected ))
  {
    pFormBlock = QueryBlockFromKeyword( &pDlgHdr->pdesPPD->stUIList,
                                        pDlgHdr->pItemsBuff, UINAME_PAGESIZE,
                                        NULL );
    pPageSelected = (PSZ) (pDlgHdr->pItemsBuff +
                           pFormBlock->uiEntry[ lSelected ].ofsOption);

    pPaperDim = (PSHORT) PaperDimensions( (CHAR *) pPageSelected,
                                          pDlgHdr->pdesPPD );

    pageSize.cx = (LONG) *(pPaperDim++);
    pageSize.cy = (LONG) *pPaperDim;

    GetImageableArea( (SHORT) lSelected, &formStruct, pDlgHdr->pdesPPD,
                      pDlgHdr->pItemsBuff );
    pPaperDim = (PSHORT) formStruct.Data;
    rclImgArea.xLeft   = (LONG) *(pPaperDim++);
    rclImgArea.yBottom = (LONG) *(pPaperDim++);
    rclImgArea.xRight  = (LONG) *(pPaperDim++);
    rclImgArea.yTop    = (LONG) *pPaperDim;
  }

  mpValues[ 0 ] = (MPARAM) &pageSize;
  mpValues[ 1 ] = (MPARAM) &rclImgArea;
  GplJFUpdateFormSettings( pDlgHdr->hNBInstance, mpValues, GJMF_PAGESIZE,
                           NUM_OF_VALUES );
}





PGLCB InitNUPDlg(
  PDLGHDR pDlgHdr
)
{
  PGLCB pRC;

  if (pDlgHdr != NULL)
  {
    if ((pRC = GplMemoryAlloc( pProcessHeap, sizeof( *pRC ))) != NULL)
    {
      PGJNCB  pNUP = &pRC->layout.nUP;
      PGJNCB2 pOldNUP = &pDlgHdr->pCNFData->gjfncb;

      pDlgHdr->pLayoutCB = pRC;

      GplLayoutInitCB( pRC, GL_SET_DEFAULT_LAYOUT, 0 );

      pNUP->ulNumPgSheet = pOldNUP->ulNumPgSheet;
      pNUP->ulFlags      = pOldNUP->ulFlags;

     strcpy( pNUP->szBookletRangeOfPages, 
             pDlgHdr->pCNFData->szBookletRangeOfPages );  

     pDlgHdr->hLayoutInstance = GplLayoutCreatePage( pDlgHdr->hNBInstance,
                                                     pRC,
                                                     LayoutCallbackProc,
                                                     (PVOID) pDlgHdr,
                                                     NULL,
                                                     0,
                                                     GNDF_NOFTP | GNDF_NOSCALING );
    }
  }
  else
  {
    pRC = NULL;
  }

  return( pRC );
}




VOID SaveNUPDlg(
  PDLGHDR pDlgHdr
)
{
  PGJNCB  pNUP = &pDlgHdr->pLayoutCB->layout.nUP;
  PGJNCB2 pOldNUP = &pDlgHdr->pCNFData->gjfncb;

  pOldNUP->ulNumPgSheet = pNUP->ulNumPgSheet;
  pOldNUP->ulFlags      = pNUP->ulFlags;

  strcpy( pDlgHdr->pCNFData->szBookletRangeOfPages, 
          pNUP->szBookletRangeOfPages );  

  GplMemoryFree( pDlgHdr->pLayoutCB );
}

MRESULT APIENTRY LayoutCallbackProc(
  PGLCB pLayoutCB,
  PVOID pInst
)
{
  PDLGHDR       pDlgHdr = (PDLGHDR) pInst;
  
  // Current type of layout selected
  switch (pLayoutCB->ulLayoutType)
  {
    // n-up
    case GL_LAYOUT_NUP:
    {
      HWND       hDlg;
      ULONG      PageID;
      ULONG      hNotebook[3];
      PGJNCB     pNupCB = &pLayoutCB->layout.nUP;
      
      GplJFQueryNBInstParamList( pDlgHdr->hNBInstance, hNotebook, GJNI_QUERY_NOTEBOOK_HANDLE, 1 );
      
      // query notebook Paper page ID (first page in notebook)
      PageID = (ULONG) WinSendMsg( hNotebook[0],
                                   BKM_QUERYPAGEID,
                                   (MPARAM)0,
                                   MPFROMLONG(BKA_FIRST) );
      
      hDlg = (HWND) WinSendMsg( hNotebook[0],
                                BKM_QUERYPAGEWINDOWHWND,
                                MPFROMLONG(PageID),
                                (MPARAM)0 );

      // Disable MIXED_FEED if booklet selected
      if( pNupCB->ulFlags & GNDF_BOOKLETENABLED )
      {
        WinEnableControl( hDlg, JPF_MIXED_FEED, FALSE );
        WinSendDlgItemMsg( hDlg, JPF_MIXED_FEED, BM_SETCHECK, MPFROMSHORT( FALSE ), (MPARAM) 0 );
      }
      else
      {
        WinEnableControl( hDlg, JPF_MIXED_FEED, TRUE );
      }
      // Send BN_CHECK notify message. Set MP2=0, to indicate, that message is simulated
      WinSendMsg(hDlg, WM_CONTROL, MPFROM2SHORT(JPF_MIXED_FEED,BN_CLICKED), MPFROMSHORT(0));
      
      break;
    }
  }
  
  return( NULL );
}

