/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/*************************************************************************
 *
 * SOURCE FILE NAME = INIT.C
 *
 * DESCRIPTIVE NAME = Initialization Code
 *
 *
 * VERSION = V1.0
 *
 * DATE
 *
 * DESCRIPTION : This file contains initialization code for jobs and for
 *               Dynamic Job Properties.
 *
 *
 * FUNCTIONS :
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/
#pragma pack(1)
#define  INCL_ERRORS
#define  INCL_PM
#define  INCL_DOS
#define  INCL_DEV
#include "inc\prdinclt.h"
#include "inc\prdgextf.h"
#include "inc\prdeextf.h"
#include "inc\prdmath.h"
#include "inc\prderre.h"
#include "inc\utl.h"

#define  INCL_GENPLIB_ERROR
#define  INCL_GENPLIB_MEMORY
#define  INCL_GENPLIB_HANDLER
#define  INCL_GENPLIB_GAMMA
#define  INCL_GENPLIB_WATERMARKS
#define  INCL_GENPLIB_LAYOUT
#include <genplib.h>

#define  INCL_WINSHELLDATA
#define  INCL_INCL_WINSHELLDATA
#define  OD_MEMORY     8L
#define  NGREMAX  222
#include <string.h>
//#include <pmshl.h>
#define  INCL_SPL
#define  INCL_SPLDOSPRINT
#define  INCL_SPLERRORS
#include <pmspl.h>
#include <pmerr.h>
#include "inc\config.h"
#include "inc\pspagtun.h"
#include "inc\init.h"
#include "inc\uinames.h"
#include "inc\profile.h"





/****************************************************************************\
** DEFINES                                                                  **
\****************************************************************************/
#define INI_KEY_ORIENTATION  "ORIENTATION"
#define INI_KEY_OUTPUTORDER  "OUTPUTORDER"
#define INI_KEY_FONTNAME     "FONTNAME"
#define INI_KEY_EFFECTS      "EFFECTS"
//#define INI_KEY_FONTCOUNT    "FONTCOUNT"
/****************************************************************************\
** DEFINES END                                                              **
\****************************************************************************/





/****************************************************************************\
** LOCAL FUNCTION PROTOTYPES                                                **
\****************************************************************************/
PDRIVDATA AllocDRIVDATAStruc( PDESPPD, PVOID, PSZ );            //@V3.1151350
VOID VerifyCNFDATAContents( PCNFDATA, PDESPPD, PSZ, PSZ );
VOID LoadCurrentUISelections( PDESPPD, PCNFDATA );
BOOL VerifyUISelectionList( PDESPPD, PCNFDATA );
LONG MapUserDefForm( PDESPPD, PCNFDATA, PSZ );
PUI_BLOCK ClearUISelection( PDESPPD, PCNFDATA, PSZ );
LONG CopyDataElements( PDDCOPY, LONG, PDDCOPY, LONG );
VOID UpdateCopyFields( PDDCOPY, LONG );
/****************************************************************************\
** LOCAL FUNCTION PROTOTYPES END                                            **
\****************************************************************************/





/****************************************************************************\
** EXTERNAL FUNCTION PROTOTYPES                                             **
\****************************************************************************/
extern PBYTE QueryUIOptionString( PDESPPD, PUI_SEL, PBYTE, PINT, PUI_BLOCK * );
extern VOID ConvertColorModel( PDESPPD, PCNFDATA, PDV );
extern VOID _System utl_memcopy( PB, PB, short int );
extern ULONG GetCountryCode( VOID );
extern INT _Optlink CompareRealNames( PSZ, PSZ );
extern VOID szNewCopy( register PSZ, register PSZ, register SHORT );
extern LONG DefaultFontCount( PDESPPD );
extern LONG UsePrinterDeviceFonts( VOID );
extern INT SetUIOption( PDESPPD, PUI_SEL, PBYTE, PBYTE );
extern PSZ GetDefaultPageSize( PDESPPD, PBYTE );
extern PUI_BLOCK QueryBlockFromKeyword( PUI_LIST, PBYTE, PBYTE, PINT );
extern VOID  ConvertOldJpToNew( PDRIVDATA, PDESPPD );
extern VOID  ConvertNewJpToOld( PDRIVDATA, PDESPPD );
extern INT NewUserForms( PCNFDATA, INT, PSZ, PSZ );
// @V4.0169885
extern BOOL InsertQueueString( PSZ, PSZ, PSZ );
/****************************************************************************\
** EXTERNAL FUNCTION PROTOTYPES END                                         **
\****************************************************************************/





/****************************************************************************\
** EXTERNAL GLOBAL VARIABLES                                                **
\****************************************************************************/
extern PVOID pProcessHeap;
extern PBYTE StringTable[MAX_STRINGS];
extern HMODULE pscript_module;                 /* memory.c */
/****************************************************************************\
** EXTERNAL GLOBAL VARIABLES END                                            **
\****************************************************************************/

typedef PBYTE *PPBYTE;





/***************************************************************************
 *
 * FUNCTION NAME = InitializeLocalDRIVDATA
 *
 * DESCRIPTION
 * The local DRIVATA for this driver includes a CNFDATA structure and a UI
 * Selection Buffer.  The CNFDATA structure contains general user selections
 * from Job Properties.  The UI Selection Buffer contains specific User
 * Interface selections.
 *
 * This function initializes a local DRIVDATA, including CNFDATA and UI
 * Selection Buffer.
 *
 * Diagram of the CNFDATA structure:
 *
 *             PCNFDATA buffer
 *      +--- +-----------------+  <-- start of DRIVDATA.abGeneralData
 *      |    |                 |      +------------------+
 *      |    |  pSourcePaper   | -->  | allocated memory |
 * size of   |                 |      +------------------+
 * CNFDATA   |                 |
 * structure |  ofsSelectList  | -+
 *      |    |                 |  |
 *      +--- |-----------------|<-+  --+
 *           |  UI Selection   |       | Size is number of UI blocks times the
 *           |  List buffer    |       | size of the UI_SEL macro (in bytes).
 *           +-----------------+ ------+
 *           +-----------------+
 *           |Layout (NUP) data|
 *           +-----------------+
 *
 * The CNFDATA buffer is found in PPDIALOG.H.
 * In order to support backward compatibility, the set of functions must take
 * into considerations that there might be older CNFDATAs, ones that may be
 * smaller than the current structure.  It must also be taken into
 * consideration that the UI Selection List Buffer may not exist on older
 * versions.
 * This function initializes the CNFDATA structure and ensures that valid
 * data are copied and set.
 *
 * INPUT
 * plDriveData - DRIVDATA buffer provided by the user.  This contains a
 *   CNFDATA structure.  This may be NULL.  If NULL, a new DRIVDATA is created
 *   and initialized.  If not NULL, then the data is copied to the new
 *   DRIVDATA.
 * pdesPPD - PPD data buffer.  This is required.
 * pQueueName - Pointer to the queue name, name as it appears on the Print
 *   Object.  This may be NULL.  If NULL, then the queue name is not copied
 *   to the new DRIVDATA.
 * pDeviceName - Pointer to the device name.  This may be NULL.  If NULL, then
 *   the device name is not copied to the new DRIVDATA.
 *
 * OUTPUT
 * pPCNFDataRtn - If a pointer is provided, returns the CNFDATA portion of
 *   the DRIVDATA.
 *
 * RETURN-NORMAL = Pointer to local CNFDATA structure.  The contents of this
 *   structure contain data passed in from plDriveData.
 *
 * RETURN-ERROR  = NULL - allocation failure or PDESPPD not provided.
 *
 **************************************************************************/
PDRIVDATA InitializeLocalDRIVDATA( PDRIVDATA pInDrivData, PDESPPD pdesPPD,
                                   PSZ pQueueName, PSZ pDeviceName,
                                   PDV pdv,PCNFDATA *ppCNFDataRtn )
{
  INT        iDDSize;
  PCNFDATA   pCNFData;
  PVOID      pHeap;
  PVOID      pPtr;
  PDRIVDATA  pLocalDrivData = NULL;
  PLDRIVDATA plInDrivData = (PLDRIVDATA) pInDrivData;

  // LMMOORE
  /*
  ** This is just for a reference.  When debugging, it is a good idea to get
  ** the size of the CNFDATA structure.
  */
#if IPMD
  LONG l = sizeof( CNFDATA );
#endif

  if (ppCNFDataRtn != NULL)
  {
    *ppCNFDataRtn = NULL;
  }

  /*
  ** pdesPPD is required.
  */
  if (pdesPPD != NULL)
  {
    /*
    ** If pdv, then use the DC heap, else use per-process heap.
    */
    if ( pdv )
    {
      pHeap = pdv->pDCHeap;
    }
    else
    {
      pHeap = pProcessHeap;
    }

    /*
    ** Allocate memory for the return DRIVDATA.
    */
    if ((pLocalDrivData = AllocDRIVDATAStruc( pdesPPD, pHeap,
                                          pDeviceName )) != NULL) //@V3.1151350
    {
      pCNFData = (PCNFDATA) pLocalDrivData->abGeneralData;

      if (ppCNFDataRtn != NULL)
      {
        *ppCNFDataRtn = pCNFData;
      }

      /*
      ** Copy the contents of the CNFDATA in LDRIVDATA to the local CNFDATA
      ** buffer.
      */
      CopyDRIVDATA( pLocalDrivData, pInDrivData );

      ConvertOldJpToNew( pLocalDrivData, pdesPPD );

      /*
      ** Read the INI data into the CNFDATA.  Also verify that valid data
      ** exists.
      */
      VerifyCNFDATAContents( pCNFData, pdesPPD, pQueueName, pDeviceName );

      /*
      ** Ensure that the module handle is the correct value.
      */
      pCNFData->hmod = pscript_module;
    }
    else
    {
      GplErrSetError( ERROR_NOT_ENOUGH_MEMORY );
    }
  }
  else
  {
    /*
    ** pdesPPD is not provided, and it was required.  Return an error.
    */
    GplErrSetError( -1 );
  }

  return( pLocalDrivData );
}
/*---------------------------------------------------------------------------*\
* InitializeLocalDRIVDATA End                                                 *
\*---------------------------------------------------------------------------*/





/***************************************************************************
 *
 * FUNCTION NAME = AllocDRIVDATAStruc
 *
 * DESCRIPTION
 * Allocates memory for the DRIVDATA structure.  This buffer includes the
 * CNFDATA structure, starting at DRIVDATA.abGeneralData; and the UI Selection
 * List Buffer, which immediately follows the CNFDATA.
 * Memory for the PSOURCE structure in CNFDATA is also allocated.
 *
 * NOTE
 * This function is the compliment of FreeLocalDRIVDATA() which frees all
 * memory allocated in this function.  Any modifications made in this function
 * may also need to be modified in FreeLocalDRIVDATA().
 *
 * INPUT
 * pdesPPD - PPD descriptor structure.
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = Allocated memory for DRIVDATA structure.
 *
 * RETURN-ERROR  = NULL - allocation failure.
 *
 * @V3.1151350 - added new parm
 **************************************************************************/
PDRIVDATA AllocDRIVDATAStruc( PDESPPD pdesPPD, PVOID pHeap, PSZ pDeviceName )
{
  PDRIVDATA pDrivData;                          // DRIVDATA pointer
  PCNFDATA  pCNFData;                           // Subset of DRIVDATA
  UINT      uiUISelLength;                      // Length of UI selection list
  UINT      uiTotalDDSize;                      // Total size of DRIVDATA
  BOOL      bAllocSuccess = TRUE;               // TRUE if all alloc is success

  /*
  ** Query the total length of the CNFDATA buffer.  This includes
  ** the size of the UI selection buffer.
  */
  uiUISelLength = pdesPPD->stUIList.usNumOfBlocks * sizeof( UI_SEL );

  /*
  ** First, calculate the total size of DRIVDATA.
  */
  uiTotalDDSize = (UINT) QueryDrivdataSize( pdesPPD );

  /*
  ** Allocate memory for the DRIVDATA structure.
  */
  if ((pDrivData = GplMemoryAlloc( pHeap, uiTotalDDSize )) != NULL)
  {
    pDrivData->cb = (LONG) uiTotalDDSize;
    /* @V3.1151350 fill in the version and device name */
    pDrivData->lVersion = DRIVERSION;
    strcpy( pDrivData->szDeviceName, pDeviceName );
    pCNFData = (PCNFDATA) pDrivData->abGeneralData;

    if (pCNFData->lVersion == 0)
    {
      pCNFData->lVersion = pDrivData->lVersion;
    }

    /*
    ** Allocate memory for the source paper buffer.
    */
    if ((pCNFData->u.iv.pSourcePaper =
         (PSOURCE) GplMemoryAlloc( pHeap, sizeof( SOURCE ) )) == NULL)
    {
      bAllocSuccess = FALSE;
    }

    /*
    ** Set the size of the UI selection list and assign the offset to the
    ** first byte in the UI selection buffer.  This buffer immediately
    ** follows the CNFDATA structure.
    */
    pCNFData->stUISelList.usCurrUIListSize = (USHORT) uiUISelLength;
    pCNFData->stUISelList.ofsSelectList    = sizeof( CNFDATA );
  }
  else
  {
    bAllocSuccess = FALSE;
  }

  /*
  ** Allocation failed.
  */
  if (bAllocSuccess == FALSE)
  {
    FreeLocalDRIVDATA( pDrivData );
    pCNFData = NULL;

    GplErrSetError( ERROR_NOT_ENOUGH_MEMORY );
  }

  return( pDrivData );
}
/*---------------------------------------------------------------------------*\
* AllocDRIVDATAStruc End                                                      *
\*---------------------------------------------------------------------------*/





/*****************************************************************************
 *
 * FUNCTION NAME = FreeLocalDRIVDATA
 *
 * DESCRIPTION
 * Frees allocated memory for the DRIVDATA structure. First, this function
 * frees all allocated pointers within the structure, then the DRIVDATA
 * buffer itself is freed.
 *
 * NOTE
 * This function frees all data allocated in AllocCNFStructure().  Any changes
 * made to AllocCNFStructure() may need to be modified here as well.
 *
 * INPUT
 * pDrivData - DRIVDATA pointer.
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = None
 *
 * RETURN-ERROR  = None
 *
 ****************************************************************************/
VOID FreeLocalDRIVDATA( PDRIVDATA pDrivData )
{
  PCNFDATA pCNFData;

  if (pDrivData != NULL)
  {
    pCNFData = (PCNFDATA) pDrivData->abGeneralData;

    /*
    ** Free the source paper buffer.
    */
    if (pCNFData->u.iv.pSourcePaper != NULL)
    {
      GplMemoryFree( pCNFData->u.iv.pSourcePaper );
      pCNFData->u.iv.pSourcePaper = NULL;
    }

    /*
    ** Free the CNFData structure.
    */
    GplMemoryFree( pDrivData );
  }
}
/*---------------------------------------------------------------------------*\
* FreeLocalDRIVDATA End                                                       *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = CopyDRIVDATA
 *
 * DESCRIPTION
 * Copies all data from a source DRIVDATA to a destination DRIVDATA.
 * The copy is done in groups.  This is required since each group can grow
 * independently over time.  When copying a group, the smallest size of each
 * group is the size of the copy.  For example, if the size of the source
 * group is larger than the destination, the size of the destination is queried
 * and that size is copied from the source to the destination.
 * This allows the driver to add more data without having to be too restrictive
 * in size limitations.
 *
 * NOTE:  In copying the CNFDATA, the pointer to the list of paper names
 * (pSourcePaper) must be saved from the destination buffer, then restored
 * after the copy.  This pointer is not passed to the queue, but is allocated
 * in AllocDRIVDATA(), and the address from the source is most likely
 * invalid.
 *
 * INPUT
 * pddSrc - Source data to copy.
 *
 * OUTPUT
 * pddDest - Buffer where data was copied.
 *
 * RETURN CODES
 * TRUE - All copying successful
 * FALSE - Some data does not yet exist, so entire copy was not made.
 *
 *****************************************************************************/
BOOL CopyDRIVDATA( PDRIVDATA pddDest, PDRIVDATA pddSrc )
{
  DDCOPY     ddDestBuff;
  DDCOPY     ddSrcBuff;
  LONG       lDestBuffSize;
  LONG       lSrcBuffSize;
  PCNFDATA   pCNFDest;
  PSOURCE    pPaper;
  PCNFDATA   pCNFSrc;
  UI_SELLIST uiTemp;
  BOOL       fRC = FALSE;

  if (pddSrc != NULL)
  {
    pCNFSrc = (PCNFDATA) pddSrc->abGeneralData;
  }

  /*
  ** For some apps, like PageMaker, the byte count may be invalid.  This can
  ** be determined by querying the size.  If it is negative, or if 'oldhmod'
  ** is not 0, then the DRIVDATA is invalid, so do not copy.
  */
  if (pddDest != NULL && pddSrc != NULL &&
      pddSrc->cb >= sizeof( DRIVDATA )  &&
      (pddSrc->cb & 0xFFF0000) == 0     &&
      pCNFSrc->oldhmod == 0)
  {
    ddDestBuff.pBuffer   = (PBYTE) pddDest;
    ddDestBuff.lBuffSize = pddDest->cb;

    ddSrcBuff.pBuffer   = (PBYTE) pddSrc;
    ddSrcBuff.lBuffSize = pddSrc->cb;

    // Copy DRIVDATA
    lDestBuffSize = sizeof( DRIVDATA ) - sizeof( CHAR );
    CopyDataElements( &ddDestBuff, lDestBuffSize, &ddSrcBuff, lDestBuffSize );

    /*
    ** Copy CNFDATA
    ** Don't forget to save the pointer to the paper (pSourcePaper).  This
    ** value is not passed to the queue, so the pointer from the source is
    ** invalid.
    */
    pCNFDest = (PCNFDATA) &pddDest->abGeneralData;
    pPaper   = pCNFDest->u.iv.pSourcePaper;
    pCNFSrc  = (PCNFDATA) &pddSrc->abGeneralData;

    if (ddSrcBuff.lBuffSize > PRE_UI_CNFDATA_SIZE)
    {
      /*
      ** It is possible that the source DRIVDATA may contain invalid UI
      ** data.  If the size of the source and dest. DRIVDATA's are the
      ** same, then verify that the source UI contains valid data.  If
      ** not, then update.  After these values are updated, they will
      ** be preserved.
      */
      if (pddDest->cb == pddSrc->cb)
      {
        if (pCNFSrc->stUISelList.usCurrUIListSize >= pddSrc->cb         ||
            pCNFSrc->stUISelList.ofsSelectList    >= pddSrc->cb         ||
            pCNFSrc->stUISelList.ofsSelectList    < PRE_UI_CNFDATA_SIZE)
        {
          pCNFSrc->stUISelList.ofsSelectList    = sizeof( CNFDATA );
          pCNFSrc->stUISelList.usCurrUIListSize = pddSrc->cb - lDestBuffSize -
                                                  sizeof( CNFDATA );
        }
      }

      lSrcBuffSize = pCNFSrc->stUISelList.ofsSelectList;
    }
    else
    {
      lSrcBuffSize = ddSrcBuff.lBuffSize;
    }

    /*
    ** PRE_UI_CNFDATA_SIZE is the size of the CNFDATA prior to the UI
    ** variable length additions.  If the size is greater than PRE_UI_...
    ** then the UI variable length data exists, so prepare for the
    ** copy.
    */
    if (ddDestBuff.lBuffSize > PRE_UI_CNFDATA_SIZE)
    {
      /*
      ** ofsSelectList contains the offset to the beginning of the UI
      ** data.  This is also the current CNFDATA size.
      */
      lDestBuffSize = pCNFDest->stUISelList.ofsSelectList;

      if (lDestBuffSize == 0)
      {
        lDestBuffSize = lSrcBuffSize;
      }
      else
      {
        /*
        ** We need to szve the contents of stUISelList since they contain
        ** data needed to access the UI.
        */
        memcpy( &uiTemp, &pCNFDest->stUISelList, sizeof( uiTemp ) );
      }
    }
    else
    {
      lDestBuffSize = ddDestBuff.lBuffSize;

    }

    if (lSrcBuffSize > 0)
    {
      CopyDataElements( &ddDestBuff, lDestBuffSize, &ddSrcBuff,
                        lSrcBuffSize );
    }
    else
    {
      goto copyerrexit;
    }
    pCNFDest->u.iv.pSourcePaper = pPaper;

    // UI Selection List.
    if (ddDestBuff.lBuffSize > 0 &&
        ddSrcBuff.lBuffSize  > 0)
    {
      memcpy( &pCNFDest->stUISelList, &uiTemp, sizeof( uiTemp ) );

      lDestBuffSize = pCNFDest->stUISelList.usCurrUIListSize;
      lSrcBuffSize  = pCNFSrc->stUISelList.usCurrUIListSize;

      CopyDataElements( &ddDestBuff, lDestBuffSize, &ddSrcBuff,
                        lSrcBuffSize );
    }
    else
    {
      goto copyerrexit;
    }

    /*
    ** In this section, we need to verify that the destination DRIVDATA
    ** contains valid data.
    */
    pddDest->lVersion = DRIVERSION;
    if (*pddDest->szDeviceName == 0)
    {
      strcpy( pddDest->szDeviceName, pCNFDest->szPrtName );
    }

    // NUP
#if 0
//  if (ddDestBuff.lBuffSize > 0 &&
//      ddSrcBuff.lBuffSize  > 0 &&
//      GplLayoutVerifyCB( (PGLCB) ddSrcBuff.pBuffer ) == PMERR_OK)
//  {
//    GplLayoutInitCB( (PGLCB) ddDestBuff.pBuffer, GL_LAYOUT_NUP, 0 );
//    GplLayoutCopyCB( (PGLCB) ddDestBuff.pBuffer,
//                     (PGLCB) ddSrcBuff.pBuffer );
//
//    UpdateCopyFields( &ddDestBuff,
//                      GplJFQueryCBSize( (PVOID) ddDestBuff.pBuffer ) );
//    UpdateCopyFields( &ddSrcBuff,
//                      GplJFQueryCBSize( (PVOID) ddSrcBuff.pBuffer ) );
//  }
//  else
//  {
//    goto copyerrexit;
//  }
#endif

    // Watermarks
#if 0
//  if (ddDestBuff.lBuffSize > 0 &&
//      ddSrcBuff.lBuffSize  > 0 &&
//      GplWMVerifyCB( (PGJWCB) ddSrcBuff.pBuffer ) == PMERR_OK)
//  {
//    GplWMInitCB( (PGJWCB) ddDestBuff.pBuffer );
//    GplWMCopyCB( (PGJWCB) ddDestBuff.pBuffer,
//                 (PGJWCB) ddSrcBuff.pBuffer );
//
//    UpdateCopyFields( &ddDestBuff,
//                      GplJFQueryCBSize( (PVOID) ddDestBuff.pBuffer ) );
//    UpdateCopyFields( &ddSrcBuff,
//                      GplJFQueryCBSize( (PVOID) ddSrcBuff.pBuffer ) );
//  }
//  else
//  {
//    goto copyerrexit;
//  }
#endif

    fRC = TRUE;
  }

copyerrexit:

  return( fRC );
}
/*---------------------------------------------------------------------------*\
* CopyDRIVDATA End                                                            *
\*---------------------------------------------------------------------------*/





/******************************************************************************
 *
 * FUNCTION NAME = CopyDataElements
 *
 * DESCRIPTION
 * Copies the minimum amount of data from a source to a destination buffer.
 * The minimum is determined by querying the minimum from lDestBuffSize and
 * lSrcBuffSize.
 * The buffer pointer and size are adjusted accordingly to their respective
 * sizes provided in lDestBuffSize and lSrcBuffSize.
 *
 * INPUT
 * pddDest - Destination buffer structure containing a buffer pointer and
 *   buffer size.
 * lDestBuffSize - Size of portion of destination buffer to copy.
 * pddSrc - Source buffer structure containing a buffer pointer and buffer
 *   size.
 * lSrcBuffSize - Size of portion of source buffer to copy.
 *
 * OUTPUT
 * None
 *
 * RETURN CODES
 * Length of actual data copied.
 *
 *****************************************************************************/
LONG CopyDataElements(
  PDDCOPY pddDest,
  LONG    lDestBuffSize,
  PDDCOPY pddSrc,
  LONG    lSrcBuffSize
)
{
  LONG lRC = 0;

  if (pddDest != NULL && pddSrc != NULL)
  {
    if (lDestBuffSize > 0 && lSrcBuffSize > 0)
    {
      lRC = min( lDestBuffSize, lSrcBuffSize );
      memcpy( pddDest->pBuffer, pddSrc->pBuffer, lRC );
    }

    UpdateCopyFields( pddDest, lDestBuffSize );
    UpdateCopyFields( pddSrc, lSrcBuffSize );
  }

  return( lRC );
}





/******************************************************************************
 *
 * FUNCTION NAME = UpdateCopyFields
 *
 * DESCRIPTION
 * Updates the following elements of a DDCOPY structure:
 * - Increments the buffer pointer (pBuffer) the amount of bytes in lBuffIncr.
 * - Decrements the buffer size (lBuffSize) the amount of bytes in lBuffIncr.
 *
 * INPUT
 * pddBuffer - Copy structure.
 * lBuffIncr - Byte size to update the fields in the structure.
 *
 * OUTPUT
 * None
 *
 * RETURN CODES
 * None
 *
 *****************************************************************************/
VOID UpdateCopyFields(
  PDDCOPY pddBuffer,
  LONG    lBuffIncr
)
{
  if (pddBuffer != NULL)
  {
    pddBuffer->pBuffer   += lBuffIncr;
    pddBuffer->lBuffSize -= lBuffIncr;
  }
}





#if 0
/******************************************************************************
 *
 * FUNCTION NAME = InitializeDRIVDATAContents
 *
 * DESCRIPTION
 * Initializes the following pointers/values to a given DRIVDATA buffer:
 * - Sets a PCNFDATA pointer to the CNFDATA portion of the local DRIVDATA
 *   buffer.  Sets the size of the CNFDATA portion.  To verify if a CNFDATA
 *   buffer exists, the size of DRIVDATA.cb must exceed the size of a
 *   DRIVDATA structure and that cb must not be greater than 65535.
 * - Sets a pointer to the UI Selection List and sets the size of the list.
 *   To determine if a UI Selection Buffer exists:
 *   1) The size of the CNFDATA must exceed PRE_UI_CNFDATA_SIZE.
 *   2) The contents of the UI Selection structure within the CNFDATA must
 *      be valid.
 *
 * If any of the buffers listed above cannot be initialized, the pointers and
 * values are set to 0.
 *
 * INPUT
 * plDrivData - Pointer to LDRIVDATA.
 *
 * OUTPUT
 * pplCNFData - If a CNFDATA exists with plDrivData, this points to it.
 *   Otherwise, it is NULL.
 * piCNFSize - If pplCNFData is not NULL, this contains the size of the
 *   CNFDATA.  If pplCNFData is NULL, this is 0.
 * pplUISelList - If a UI Selection List exists, this points to it.  If no
 *   list exists, this is NULL.
 * piUISelList - This returns the size of the UI Selection list.  If
 *   pplUISelList is NULL, this is 0.
 *
 * RETURN-COLOR  - None
 * RETURN-GRAY   - None
 *
 *****************************************************************************/
VOID InitializeDRIVDATAContents( PDRIVDATA pDrivData, PCNFDATA *pplCNFData,
                                 PINT piCNFSize, PUI_SEL *pplUISelList,
                                 PINT piUISelList )
{
  /*
  ** Initialize all values to NULL.
  */
  *pplCNFData   = NULL;
  *pplUISelList = NULL;
  *piCNFSize  = *piUISelList = 0;

  /*
  ** For the copy to be valid, at least the size must be larger than a
  ** DRIVDATA structure.
  ** It is also known that even though 'cb' is a doubleword, the value
  ** (at least as Postscript is concerned) the value will never exceed a WORD.
  ** Since some apps may pass in invalid values verify that the high word
  ** is 0.  If true, then the value is valid.  If false, than an invalid
  ** value is passed in.
  */
  if (pDrivData->cb > sizeof( DRIVDATA ) &&
      (pDrivData->cb & 0xFFFFF000) == 0)
  {
    *pplCNFData = (PCNFDATA) &pDrivData->abGeneralData;
    *piCNFSize  = (INT) (pDrivData->cb - (sizeof( DRIVDATA ) + 1));

    /*
    ** Verify that the CNFDATA from the LDRIVDATA may contain the UI Selection
    ** list.  This is achieved by verifying that the size of the CNFDATA
    ** portion is at least as big as the current CNFDATA up to the point
    ** prior to the UI_SELLIST pointer.  If this is true, then the UI_SELLIST
    ** structure is valid and (maybe) the UI Selection List Buffer exists.
    ** If the size does include the UI Selection List, then also verify that
    ** the offset value is value (that it is greater than 0 and that the
    ** offset does not extend past the total LDRIVDATA size).
    */
    if (*piCNFSize > PRE_UI_CNFDATA_SIZE &&
        ((*pplCNFData)->stUISelList.ofsSelectList > 0 &&
         (*pplCNFData)->stUISelList.ofsSelectList <= pDrivData->cb))
    {
      *pplUISelList = (PUI_SEL) ((PBYTE) *pplCNFData +
                                 (*pplCNFData)->stUISelList.ofsSelectList);
      *piUISelList   = (INT) (*pplCNFData)->stUISelList.usCurrUIListSize;
    }
  }
}
/*---------------------------------------------------------------------------*\
* InitializeDRIVDATAContents End                                              *
\*---------------------------------------------------------------------------*/
#endif





/***************************************************************************
 *
 * FUNCTION NAME = VerifyCNFDATAContents
 *
 * DESCRIPTION
 * Verifies that the data within the CNFDATA are valid.  If any field is not
 * valid, it is reset to a default value.  This verification includes
 * retrieving data from the INI file for the CNFDATA.
 *
 * INPUT
 * pCNFData - Pointer to the CNFDATA structure.  This contains current user
 *   selections.
 * pdesPPD - Pointer to the PPD data structure.
 * pQueueName - Pointer to the queue name, name as it appears on the Print
 *   Object.  If NULL, the queue name is not copied.
 * pDeviceName - Pointer to the device name.  If NULL, the device name is
 *   not copied.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = None
 * RETURN-ERROR  = None
 *
 ****************************************************************************/
VOID VerifyCNFDATAContents( PCNFDATA pCNFData, PDESPPD pdesPPD,
                            PSZ pQueueName, PSZ pDeviceName )
{
  CHAR  aINIString[ MAX_PSIZE ];
  CHAR  aINIEntry[ MAX_PSIZE ];
  PCHAR pINIString;
  INT   iCountry;
  PSZ   pszDefString;
  PSZ   pszDefPageSize;
  PVOID pBuffer;
  INT   iLoop;

  /*
  ** Validate the contents in the UI_SELLIST structure.
  */
  pCNFData->stUISelList.ofsSelectList = sizeof( CNFDATA );
  pCNFData->stUISelList.usCurrUIListSize = (USHORT) (sizeof( UI_SEL ) *
                                            pdesPPD->stUIList.usNumOfBlocks);

  /*
  ** @V3.2155950
  ** The next series of code ensures that the correct device and app name
  ** are set and that the correct data is read from the INI.
  */
  if ( pQueueName != NULL )
  {
    // @V4.0169885
    InsertQueueString( pCNFData->szKeyApp, pQueueName, pDeviceName );

#if 0
//  strcpy( pCNFData->szKeyApp, "PM_DD_" );
//  strcat( (PSZ) pCNFData->szKeyApp, pQueueName );
//  strcat( pCNFData->szKeyApp, ",PSCRIPT." );
#endif
  }

  if ( pDeviceName != NULL )
  {
    strcpy( pCNFData->szPrtName, pDeviceName );
  }

  /* @V4.0158676
  ** If no form name is provided make a note of this.
  ** Later if HardCopyCaps is called we wont set the CURRENT bit since
  ** The DC which opened us didn't have any form associated with it
  ** BTW the spooler opens us with a minimal set of JPs which means
  ** it will not have current form.
  */
  if ( pCNFData->jobProperties.szFormName[ 0 ] == '\0' )
  {
    pCNFData->ulFlags |= NO_FORM_IN_JP;  //Set Flag
  }
  else
  {
    pCNFData->ulFlags &= ~NO_FORM_IN_JP; //Clear flag
  }

  ReadTrayPageMapINI( pdesPPD, pCNFData );

  /*
  ** Orientation
  */
  if (pCNFData->jobProperties.iOrient != LANDSCAPE &&
      pCNFData->jobProperties.iOrient != PORTRAIT)
  {
    pCNFData->jobProperties.iOrient = PORTRAIT;
  }

  /*
  ** Initialize the Ouput Order data.
  */
  if (pCNFData->jobProperties.iOutPlace != REVERSE &&
      pCNFData->jobProperties.iOutPlace != NORMAL)
  {
    pCNFData->jobProperties.iOutPlace = NORMAL;
  }

  if (pCNFData->iDestnType != SYSTEM &&
      pCNFData->iDestnType != ENCAPS &&
      pCNFData->iDestnType != RAW)
  {
    pCNFData->iDestnType = SYSTEM;
  }

  if (pCNFData->iCntCopies > 999 ||
      pCNFData->iCntCopies < 1)
  {
    pCNFData->iCntCopies = 1;
  }

  if (pCNFData->jobProperties.szFormName[0] == 0)
  {
    iCountry = (INT) GetCountryCode( );

    if (iCountry == 1 || iCountry == 2 )      //@V4.1191730 Don't include Brazil
    {
      pszDefString = (PSZ) StringTable[ IDS_Letter - STRING_BASE ];
    }
    else
    {
      pszDefString = (PSZ) StringTable[ IDS_A4 - STRING_BASE ];
    }

    pszDefPageSize = (PSZ) GetDefaultPageSize( pdesPPD,
                                               pdesPPD->pPSStringBuff );

    /*
    ** If default real name is same as the string table name ( letter or A4
    ** Then use the offical name in case of xlate names
    */
    if ( ! CompareRealNames( pszDefPageSize, pszDefString ) )
    {
      pszDefString = pszDefPageSize;
    }

    szNewCopy( (PSZ) pCNFData->jobProperties.szFormName, (PSZ) pszDefString,
               MAX_PSIZE );
  }

  PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pCNFData->szKeyApp, (PSZ)
                         INI_KEY_EFFECTS, NULL, aINIString,
                         sizeof( aINIString ) );

  /*
  ** Get the number of downloaded font limit
  */
  iCountry = sizeof( ULONG );
  if ( !PrfQueryProfileData( HINI_SYSTEMPROFILE, (PSZ) pCNFData->szKeyApp,
      (PSZ) FONTCOUNT_INI, (PLONG) &pCNFData->lFontCount,
      (PULONG) &iCountry ) )
  {
    pCNFData->lFontCount = -1L;
  }

// Now done in ConvertOldJpToNew  @V3.1152199
#if 0
///*
//** @V3.1147743
//** Query for old duplex support.  If set, and if duplex is supported,
//** set the new duplex format.
//*/
//if (pdesPPD->desItems.sDefaultDuplex != DUPLEX_FALSE &&
//    QueryBlockFromKeyword( &pdesPPD->stUIList, pdesPPD->pPSStringBuff,
//                           UINAME_DUPLEX, NULL ) != NULL)
//{
//  if (pCNFData->sDuplexMode == DUPLEX_DUPLEXNOTUMBLE)
//  {
//    SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ), UINAME_DUPLEX,
//                 UINAME_DUPLEXNOTUMBLE );
//  }
//  else if (pCNFData->sDuplexMode == DUPLEX_DUPLEXTUMBLE)
//  {
//    SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ), UINAME_DUPLEX,
//                 UINAME_DUPLEXTUMBLE );
//  }
//}
#endif

#if 0
//if ((pdesPPD->desItems.sDefaultDuplex == DUPLEX_FALSE) &&
//   (pCNFData->sDuplexMode == DUPLEX_NONE))
//{
//  pCNFData->sDuplexMode = DUPLEX_FALSE;
//}
//else if (pdesPPD->desItems.sDefaultDuplex == DUPLEX_NONE)
//{
//  pCNFData->sDuplexMode = DUPLEX_NONE;
//}
#endif

  /*  ** Retrieve the default resolution.
  */
  if (pCNFData->uResolution == 0)
  {
    pCNFData->uResolution = pdesPPD->desItems.iResDpi;
  }

  /*  ** Set initial amount
  */
  if (pCNFData->lFontCount == -1L )
  {
    pCNFData->lFontCount =  DefaultFontCount( pdesPPD  );
  }

  if (pCNFData->jobProperties.uScale == 0)
  {
    pCNFData->jobProperties.uScale = 100;
  }

// Now done in ConvertOldJpToNew  @V3.1152199
#if 0
///*
//** Older versions stored the Manual Feed in 'iManualFeed'.  The current
//** version stores "MANUALFEED" in 'aTraySelected'.  Ensure that both
//** values are up-to-date in case in receiving/printing from/to other
//** versions.
//*/
//if (pCNFData->jobProperties.iManualfeed == TRUE &&
//    szIsEqual( pCNFData->u.iv.aTraySelected, MANUALFEED_STRING ) == FALSE)
//{
//    strcpy( pCNFData->u.iv.aTraySelected, MANUALFEED_STRING );
//}
//
///*
//** This field is no longer needed.  To ensure that this does not conflict
//** with other settings, set to FALSE.
//*/
//pCNFData->jobProperties.iManualfeed = FALSE;
#endif

  /*
  ** Boolean flag flip top to bottom
  */
  if (pCNFData->effOutput.fIsFliptb != FALSE &&
      pCNFData->effOutput.fIsFliptb != TRUE)
  {
    pCNFData->effOutput.fIsFliptb = FALSE;
  }

  /*
  ** Boolean flag draw inverted
  */
  if (pCNFData->effOutput.fIsDrawInverted != FALSE &&
      pCNFData->effOutput.fIsDrawInverted != TRUE)
  {
    pCNFData->effOutput.fIsDrawInverted = FALSE;
  }

  /*
  ** Boolean flag flip left to right
  */
  if (pCNFData->effOutput.fIsFliplr != FALSE &&
      pCNFData->effOutput.fIsFliplr != TRUE)
  {
    pCNFData->effOutput.fIsFliplr = FALSE;
  }

  /*
  ** default value of scale factor is 100% i.e. unity
  */
  if (pCNFData->jobProperties.uScale > 999)
  {
    pCNFData->jobProperties.uScale = 100;
  }

  /*  ** Use all fonts by default
  */
  if (pCNFData->u.iv.fJobPropSet == FALSE)
  {
    pCNFData->sUseDLFonts = 1;
  }
  else
  if (pCNFData->sUseDLFonts != 1 &&
      pCNFData->sUseDLFonts != 0)
  {
    pCNFData->sUseDLFonts = 1;
  }

  /*
  ** For some countries, UserPrinterDeviceFonts() will always return 1.
  ** However, if the user disabled 'Use Printer Device Fonts' in Job
  ** Properties, this function will still return 1 and thus, will always
  ** case sUsePDFonts to be set.
  ** Therefore, call UsePrinterDeviceFonts() only when initializing and
  ** ignore on subsequent passes.
  */
  if ((pCNFData->ulFlags & PD_FONTS_SET) == 0)
  {
    pCNFData->sUsePDFonts = UsePrinterDeviceFonts();
    pCNFData->ulFlags |= PD_FONTS_SET;
  }

  /* D74609
  ** Set usPSLevel1 to false
  */
  if (pCNFData->usPSLevel1 != 0 &&
      pCNFData->usPSLevel1 != 1)
  {
    pCNFData->usPSLevel1 = 0;
  }

  /*
  ** @V3.0GAMMA1
  ** Set initial gamma value to 10.
  */
  if (pCNFData->u.iv.fJobPropSet == FALSE)
  {
    pCNFData->sGammaValues.lRed = NO_GAMMA;
    pCNFData->sGammaValues.lGreen = NO_GAMMA;
    pCNFData->sGammaValues.lBlue = NO_GAMMA;
  }
  else
  {
    if (pCNFData->sGammaValues.lRed > 50 ||
        pCNFData->sGammaValues.lRed < 1)
    {
      pCNFData->sGammaValues.lRed = NO_GAMMA;
    }
    if (pCNFData->sGammaValues.lGreen > 50 ||
        pCNFData->sGammaValues.lGreen < 1)
    {
      pCNFData->sGammaValues.lGreen = NO_GAMMA;
    }
    if (pCNFData->sGammaValues.lBlue > 50 ||
        pCNFData->sGammaValues.lBlue < 1)
    {
      pCNFData->sGammaValues.lBlue = NO_GAMMA;
    }
  }

  /*
  ** Read the Default Font name setting from os2.ini
  */
  if (pCNFData->szFontName[ 0 ] == 0)
  {
    strcpy( pCNFData->szFontName, pdesPPD->pPSStringBuff +
                                  pdesPPD->desFonts.ofsDeffont );
  }

#if 0
//if (pCNFData->szKeyApp[ 0 ] == 0 &&
//    pQueueName != NULL)
//{
//  strcpy( pCNFData->szKeyApp, "PM_DD_" );
//  strcat( (PSZ) pCNFData->szKeyApp, pQueueName );
//  strcat( pCNFData->szKeyApp, ",PSCRIPT." );
//  strcat( pCNFData->szKeyApp, pDeviceName );
//}
//
//if (pCNFData->szPrtName[ 0 ] == 0 &&
//    pDeviceName != NULL)
//{
//  strcpy( pCNFData->szPrtName, pDeviceName );
//}
//
//if (pCNFData->szSegName[ 0 ] == 0)
//{
//  strcpy( pCNFData->szSegName, pDeviceName );
//}
#endif

  if (ReadPrtPropOptionsINI( pCNFData->szKeyApp, INISUBKEY_MODESWITCH,
                             (PBOOL) &pCNFData->uPrinterPropFlags ) == FALSE)
  {
    /*
    ** If no init string
    */
    if (pdesPPD->desItems.ofsInitString == -1)
    {
      pCNFData->uPrinterPropFlags &= ~MODESWITCH;   /* Turn off           */
    }
    else  
    {
      /*
      ** @V3.1146873
      ** Read INI for current Mode Switch setting.
      */
      PrfQueryProfileString( HINI_SYSTEMPROFILE, (PSZ) pCNFData->szKeyApp,
                             (PSZ) INI_KEY_EFFECTS, (PSZ) "",
                             (PSZ) aINIString, sizeof( aINIString ) );

      if (aINIString[ 0 ] != 0)
      {
        /*
        ** @V3.1141419
        ** The modeswitch setting is part of a string with semicolons as
        ** delimiters.  The modeswitch settings is at offset 10.
        ** Query the INI string and skip past the previous 9 settings.
        ** QueryNextSubkey() relies on semicolons as delimiters, so this
        ** works out perfectly.
        ** If for some reason, the INI string does not have enough settings,
        ** then pBuffer will be NULL before the modeswitch settings is
        ** encountered.
        */
        pINIString = (PSZ) aINIString;
        for (iLoop = 0 ; iLoop < 10 ; iLoop++)
        {
          QueryNextSubkey( &pINIString, aINIEntry, sizeof( aINIEntry ) );
          if (pINIString == NULL)
          {
            break;
          }
        }

        if (pINIString != NULL && *pINIString != 0)
        {
          pCNFData->uPrinterPropFlags = atoi( aINIEntry );
        }
      }
      else
      {
        pCNFData->uPrinterPropFlags |= MODESWITCH;  // Default setting
      }

      if ( strncmp( pCNFData->szSegName, "IBM 4019", 8 ) == 0 )
      {
        pCNFData->uPrinterPropFlags &= ~MODESWITCH;   /* Turn off           */
      }
    }
  }

  /*
  ** For color devices, if "ColorModel" is not supported, then
  ** set the default to color.
  */
  if (pdesPPD->desItems.fIsColorDevice == TRUE)
  {
    if (pCNFData->u.iv.fJobPropSet == FALSE)
    {
      pCNFData->jobProperties.fIsColorDevice = TRUE;
    }
    else
    {
      if (pCNFData->jobProperties.fIsColorDevice != TRUE &&
          (pCNFData->jobProperties.fIsColorDevice != FALSE) &&
           pCNFData->jobProperties.fIsColorDevice != NONE)
      {
        pCNFData->jobProperties.fIsColorDevice = TRUE;
      }
    }
  }
  else
  {
    pCNFData->jobProperties.fIsColorDevice = FALSE;
  }

  if (pCNFData->iDestnType < SYSTEM || pCNFData->iDestnType > ENCAPS)
  {
    pCNFData->iDestnType = SYSTEM;
  }

  LoadCurrentUISelections( pdesPPD, pCNFData );

  /* Job props have been init-ed */
  if (pCNFData->u.iv.fJobPropSet == FALSE)
  {
    pCNFData->u.iv.fJobPropSet = TRUE;
  }

  /*
  ** If the orientation has been set to landscape, we must reverse duplex
  ** (if it has been set).  Here, we'll set the flag to reverse duplex.
  ** The work is actually done in ConvertUIBitsToStrings().
  */
  pszDefString = QueryUIOptionString( pdesPPD,
                                      ASSIGN_UISELLIST_PTR( pCNFData ),
                                      UINAME_DUPLEX, NULL, NULL );
  // If duplex exists...
  if (pszDefString != NULL)
  {
    if (pCNFData->jobProperties.iOrient == LANDSCAPE &&
        strcmp( pszDefString, UINAME_DUPLEXNONE ))

    {
      SETFLAG( pCNFData->ulFlags, DUPLEX_CHANGED );
    }
  }

  // Nup
  if (pCNFData->gjfncb.ulNumPgSheet > 16    ||
      pCNFData->gjfncb.ulNumPgSheet < 1)
  {
    memset( &pCNFData->gjfncb, 0, sizeof( pCNFData->gjfncb ) );
    pCNFData->gjfncb.ulNumPgSheet = 1;
  }

//  pBuffer = (PVOID) QueryLayoutCBFromCNFData( pCNFData );
//  GplLayoutInitCB( (PGLCB) pBuffer, GL_SET_DEFAULT_LAYOUT, 0 );

//  pBuffer = (PVOID) QueryWCBFromCNFData( pCNFData );
//  GplWMInitCB( (PGJWCB) pBuffer );
}
/*---------------------------------------------------------------------------*\
* VerifyCNFDataContents End                                                   *
\*---------------------------------------------------------------------------*/





/***************************************************************************
 *
 * FUNCTION NAME = LoadCurrentUISelections
 *
 * DESCRIPTION
 * Loads the current UI (printer feature) data into the CNFDATA structure.
 * This function first looks into the INI file for the selections.  If found,
 * they are loaded.  If not, the default UI selections are read from the PPB
 * resource (taken from the "*Default..." key in the *OpenUI / *CloseUI).
 * The INI UI data is stored in string format.  See the header for
 * SaveCurrentUISelections() for the INI format.
 *
 * INPUT
 * pdesPpd - PPD description structure.
 * pCNFData - Printer configuration data structure.  This includes the UI
 *   selection buffer pUISelectList.
 *
 * OUTPUT
 * pCNFData->pUISelectList - Updated with saved selections.
 *
 * RETURN-NORMAL = None
 * RETURN-ERROR  = None
 *
 ****************************************************************************/
 // @V3.1147743 - Modify function
VOID LoadCurrentUISelections( PDESPPD pdesPPD, PCNFDATA pCNFData )
{
  PUI_BLOCK pUIBlock;          // Current Block pointer
  CHAR      aINIString[ 255 ]; // Buffer where INI strings are stored
  BOOL      fNotEnd;           // Boolean value of substring indicator
  CHAR      aKeyword[ MAX_PSIZE ];  // Keyword buffer
  CHAR      aOption[ MAX_PSIZE ];   // Option buffer
  UINT      uiLoop;            // Local loop counter
  BOOL      bINIExists;        // TRUE if INI string exists, FALSE otherwise
  ULONG     ulLength = 255;    // Size of INI string buffer
  PBYTE     pINIString = aINIString;     // Current INI string pointer
  PUI_SEL   pUICurrentSel = ASSIGN_UISELLIST_PTR( pCNFData );
                               // UI Selection Buffer
#if IPMD_DISPLAY_UI
PSZ pKeyword;
#endif

  /*
  ** Read the string, if it exists.
  */
  if ((bINIExists = PrfQueryProfileData( HINI_SYSTEMPROFILE,
                    (PSZ) pCNFData->szKeyApp, (PSZ) INI_KEY_UI,
                    (PVOID) aINIString, (PULONG) &ulLength )) == TRUE)
  {
    while (*pINIString != 0)
    {
      /*
      ** Retrieve the keyword.
      */
      fNotEnd = QueryNextSubkey( &pINIString, aKeyword, MAX_PSIZE );

      /*
      ** Retrieve all option strings that exist.
      */
      while (fNotEnd == TRUE)
      {
        fNotEnd = QueryNextSubkey( &pINIString, aOption, MAX_PSIZE );
        SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ),
                     aKeyword, aOption );
      }
    }
  }

  /*
  ** First, verify if the UI Selection List is invalid.  There is no
  ** reason to load the selection list if it already exists.
  */
  pUIBlock = pdesPPD->stUIList.pBlockList;
  for (uiLoop = 0 ; uiLoop < (UINT) pdesPPD->stUIList.usNumOfBlocks ; uiLoop++)
  {

#if IPMD_DISPLAY_UI
  pKeyword = (PSZ) (pdesPPD->pPSStringBuff + pUIBlock->ofsUIName);
#endif

    if (*pUICurrentSel == 0 ||
        *pUICurrentSel >= (1 << (pUIBlock->usNumOfEntries + 1)))
    {
      *pUICurrentSel = (UI_SEL) (1 << pUIBlock->usDefaultEntry);
    }

    INCREMENT_BLOCK_PTR( pUIBlock );
    pUICurrentSel++;
  }

// @V3.1147743
#if 0
//if (VerifyUISelectionList( pdesPPD, pCNFData ) == FALSE)
//{
//  pUIBlock = pdesPPD->stUIList.pBlockList;
//  for (uiLoop = 0 ; uiLoop < pdesPPD->stUIList.usNumOfBlocks ; uiLoop++)
//  {
//    *(pUICurrentSel++) = (UI_SEL) (1 << pUIBlock->usDefaultEntry);
//
//    INCREMENT_BLOCK_PTR( pUIBlock );
//  }
//}
#endif
}
/*---------------------------------------------------------------------------*\
* LoadCurrentUISelections End                                                 *
\*---------------------------------------------------------------------------*/




// @V3.1147743 - No longer needed
#if 0
/***************************************************************************
 *
 * FUNCTION NAME = VerifyUISelectionList
 *
 * DESCRIPTION
 * Verifies that the data in the UI selection list is valid.  Each bit in
 * each entry of the UI selection list corresponds to a selection that the
 * user has made in Job Properties.  There are at most 32 selections for a
 * given entry.  If a bit is 1, then the selection was made.  This allows a
 * user to make multiple selections (if available), for a given UI entry.
 *
 * For example, suppose there are two entries for a UI block.  If bit 0 is
 * set to 1, then the first entry is the selection.  If bit 1 is set, then
 * the second entry is the selection.
 *
 * For multiple selection entries, multiple bits may be set, but the value
 * of the total bits may not exceed the value if all bits were set.  In the
 * above example, if the UI block allowed multiple selections, then the
 * highest possible value would be 3 (bit 0 and bit 1 both set to 1).
 *
 * This function verifies the following conditions:
 * 1) That no selections are set to zero.  If bits are used, then at least one
 *    bit must be set at all times.
 * 2) That the value of the UI selection must not exceed the total value
 *    if all bits were set, as mentioned above.  Using the above example, 3
 *    would be the maximum possible value.  However, if the value was 4, then
 *    that would mean that third selection was made.  However, there are only
 *    two selections, so the value is invalid.
 * 3) This applies to single selection entries only.  For single selections,
 *    one and only one bit may be set.  If more than one bits are set, then
 *    the value is invalid.
 *
 * If any values are invalid, the function exists the loop and returns an
 * error.
 *
 * INPUT
 * pdesPPD - PPD descriptor pointer.
 * pCNFData - Printer configuration data structure.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = TRUE - The UI selections are valid.
 * RETURN-ERROR  = FALSE - UI selections are invalid.
 *
 ****************************************************************************/
BOOL VerifyUISelectionList( PDESPPD pdesPPD, PCNFDATA pCNFData )
{
  INT       iNumOfFeatures;                     // Number of features
  INT       iIndex;                             // Loop counter
  INT       iNumOfKeywords;                     // Number of keywords per block
  UI_SEL    uiTempSel;                          // Temporary UI selection value
  PUI_BLOCK pUICurrBlock = pdesPPD->stUIList.pBlockList;   // Curr UI block
  PUI_SEL   pUICurrSelection = ASSIGN_UISELLIST_PTR( pCNFData );  // Selections
  BOOL      bRC = TRUE;                         // Return code

  /*
  ** Query the number of blocks/features for the list.
  */
  iNumOfFeatures = (INT) pdesPPD->stUIList.usNumOfBlocks;

  for (iIndex = 0 ; iIndex < iNumOfFeatures && bRC == TRUE ; iIndex++)
  {
    /*
    ** Query the number of entries/keywords for a given block.
    */
    iNumOfKeywords = (INT) pUICurrBlock->usNumOfEntries;

    /*
    ** Since each bit in the UI Selection entry represents a selection, that
    ** means that each UI Selection entry can never be zero.  If zero, then
    ** this is invalid.
    */
    if (*pUICurrSelection != 0)
    {
      /*
      ** The value of the UI Selection entry must never exceed the total
      ** bits for each entry it represents.  Increment the number of
      ** keywords to ensure that the value does not exceed the total value
      ** if all bits were set to 1.
      */
      if (*pUICurrSelection < (1 << (iNumOfKeywords + 1)))
      {
        /*
        ** This next test is for single selection UI entries.  For single
        ** selection UI entries, one and only one bit may be set.  Although
        ** the value may be within the range checked above, more than one
        ** bit set means that there is an error.  Remove the lowest bit.  If
        ** a non-zero value still exists, then the selection is invalid.
        */
        if (pUICurrBlock->usSelectType != UI_SELECT_PICKMANY)
        {
          uiTempSel = *pUICurrSelection;

          while (uiTempSel != 0 && (uiTempSel & 1) == 0)
          {
            uiTempSel >>= 1;
          }

          if ((uiTempSel >> 1) != 0)
          {
            bRC = FALSE;
          }
        }
      }
      else
      {
        bRC = FALSE;
      }
    }
    else
    {
      bRC = FALSE;
    }

    /*
    ** Increment pointers.
    */
    pUICurrSelection++;
    INCREMENT_BLOCK_PTR( pUICurrBlock );
  }

  return( bRC );
}
/*---------------------------------------------------------------------------*\
* VerifyUISelectionList End                                                   *
\*---------------------------------------------------------------------------*/
#endif





/***************************************************************************
 *
 * FUNCTION NAME = SaveCurrentUISelections
 *
 * DESCRIPTION
 * Saves the current UI selection data, taken from pcnfData->pCurrUIEntry,
 * into the OS2SYS.INI file.
 *
 * The UI string data is stored in the following format:
 *
 * Keyword1;option1[,option2,...,optionn]0[Keyword2;option1[,option2...
 * ..,optionn]0...Keywordn;option1[,option2,...,optionn]00
 *
 * Each keyword references a specific block.  Each option references a
 * specific entry within the block (keyword).  The separator between the
 * keyword and matching options is the semicolon.  The separator between each
 * option of a given keyword is the comma.  The separater between each
 * keyword/option group is the NULL terminator.  The terminator for the entire
 * string is another NULL terminator.  This function saves the selected UI
 * in the above string format.
 *
 * The application name for the UI string is the printer name.
 * The UI application name in the INI is "UI".
 *
 * INPUT
 * pdesPpd - PPD description structure.
 * pcnfData - Printer configuration data structure.  This includes the UI
 *   selection buffer pUISelectList.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = None
 * RETURN-ERROR  = None
 *
 ****************************************************************************/
VOID SaveCurrentUISelections( PDESPPD pdesPPD, PCNFDATA pcnfData )
{
  CHAR      aINIString[ 255 ];           // Buffer of profile string
  INT       iLoop;                       // Local loop counter
  PBYTE     pKeywordStr;                 // Current pointer string
  PBYTE     pOptionStr;                  // Current option string
  INT       ofsSelBit;                   // Offset of current UI Selection bit
  PUI_BLOCK pUIBlock = pdesPPD->stUIList.pBlockList;  // UI Block pointer
  PUI_BLOCK pDummy = NULL;

  aINIString[ 0 ] = 0;

  /*
  ** Go through the UI Selection list.
  */
  for (iLoop = 0 ; iLoop < (INT) pdesPPD->stUIList.usNumOfBlocks ; iLoop++)
  {
    if (pUIBlock->ucGroupType == UIGT_INSTALLABLEOPTION)
    {
      /*
      ** Retrieve the keyword string and append it to the buffer.
      */
      pKeywordStr = pUIBlock->ofsUIName + pdesPPD->pPSStringBuff;
      SaveINIGroupData( aINIString, pKeywordStr );

      ofsSelBit = 0;
      while (ofsSelBit != -1)
      {
        /*
        ** With the given UI Selection list, retrieve the first option
        ** string.
        */
        pOptionStr = QueryUIOptionString( pdesPPD,
                                          ASSIGN_UISELLIST_PTR( pcnfData ),
                                          pKeywordStr, &ofsSelBit, &pDummy );

        /*
        ** This is TRUE if the end of the UI Option Entry is not NULL.
        */
        if (pOptionStr != NULL)
        {
          SaveINIGroupData( aINIString, pOptionStr );
        }
      }

      SaveINIGroupData( aINIString, NULL );
    }

    INCREMENT_BLOCK_PTR( pUIBlock );
  }

  /*
  ** Write the profile buffer to the INI.  Include a NULL terminator.
  */
  if (aINIString[ 0 ] != 0)
  {
    PrfWriteProfileData( HINI_SYSTEMPROFILE, (PSZ) pcnfData->szKeyApp,
                         (PSZ) INI_KEY_UI, (PVOID) aINIString,
                         (UINT) strlen( aINIString ) + 1 );
  }

}
/*---------------------------------------------------------------------------*\
* SaveCurrentUISelections End                                                 *
\*---------------------------------------------------------------------------*/





/***************************************************************************
 *
 * FUNCTION NAME = VerifyAndSetJobContents
 *
 * DESCRIPTION
 * Verifies that certain internal selections are valid for a job.  This also
 * sets parameters for a job.  This function should be called when a job is
 * initialized (like ps_startdoc()).
 * This is not like VerifyCNFDATAContents().  VerifyCNFDATAContents() verifies
 * that the contents of the CNFDATA structure are valid.  This function
 * verifies settings that affect the job output only, and is only needed when
 * a job is sent out.
 * It is not needed for DevPostDeviceModes.
 *
 * INPUT
 * pdv-> Structure part of PDDC.
 * fIsDestnLAN - This is TRUE if the destination is to a LAN device or FALSE
 *   if to a local device.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = None
 * RETURN-ERROR  = None
 *
 ****************************************************************************/
VOID VerifyAndSetJobContents( PDV pdv )
{
  PDESPPD  pdesPPD  = pdv->pdesPPD;
  PCNFDATA pCNFData = pdv->pCNFData;

  /*
  ** Convert the internal UI format to a string format if it has not yet been
  ** done (if pUISelectList == NULL).
  */
  if (pdv->pUISelectList == NULL)
  {
    pdv->pUISelectList = ConvertUIBitsToStrings( pdv, pCNFData );
  }

  /*
  ** If the user set the flag for Unix compatibility, set the NO_CTL_D flag.
  */
  if ((pCNFData->ulFlags & UNIX_COMPAT) != 0)
  {
    SETFLAG( pdv->ulGenFlags, NO_CTL_D );
  }

  ConvertColorModel( pdesPPD, pCNFData, pdv );
}
/*---------------------------------------------------------------------------*\
* VerifyAndSetJobContents End                                                 *
\*---------------------------------------------------------------------------*/





/***************************************************************************
 *
 * FUNCTION NAME = ConvertUIBitsToStrings
 *
 * DESCRIPTION
 * With a given list of selected bits from the UI_SEL array in the CNFDATA
 * structure, converts the bits to matching strings in the following format:
 *
 *  "KEYWORD1,OPTION1A,OPTION1B,...OPTION1N;KEYWORD2,OPTION2A,...OPTION2N"
 *
 * with the KEYWORD being the block name and OPTION being the UI entry
 * corresponding to each bit that was set in the UI_SEL list.
 *
 * The reason for this is to be compatible with other versions of the OS/2
 * keyword selection.  If a printer adds a new keyword name and this is passed
 * across the LAN, no matter what keywords are added, the keyword will always
 * remain the same.  If new keywords are added, the comparasion will be against
 * the string, rather than the UI_SEL bits, which can change if a new keyword
 * is added.
 * This string buffer is stored in PDV, which is passed onto the spooler.
 *
 * INPUT
 * pdesPPD - Structure containing output data.
 * pCNFData - Configuration data structure.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = Pointer to the new string list.
 * RETURN-ERROR  = NULL, no data allocated.
 *
 ****************************************************************************/
PBYTE ConvertUIBitsToStrings( PDV pdv, PCNFDATA pCNFData )
{
  UINT      uiListIndex;                  // Index loop counter
  INT       ofsSelBit;                    // Offset of bit selected
  PBYTE     pBlockName;                   // Current block name
  PBYTE     pEntryName;                   // Current entry name
  PSZ       pOldDuplexSelected = NULL;    // Selected duplex
  PSZ       pDuplexSelected;              // Selected duplex
  PUI_BLOCK pUIBlock;                     // Current UI block
  PDESPPD   pdesPPD = pdv->pdesPPD;
  UINT      uiNumOfBlocks = (UINT) pdesPPD->stUIList.usNumOfBlocks;
  PUI_SEL   pExistingSel = ASSIGN_UISELLIST_PTR( pCNFData );
  PBYTE     pRC = NULL;                   // Function return code

  /*
  ** Allocate a page of memory for the new selection string list.
  */
  if ((pRC = (PBYTE) GplMemoryAlloc( (HMCB) 0, 4000 )) != NULL)
  {
#if 0
    /*
    ** @V4.0168335
    ** For the duplex, there is a problem.  When printing landscape, if
    ** NoTumble is selected, the output comes out tumbled and visa versa.
    ** The reason is that the duplex format is assumed to be *FOR PORTRAIT*.
    ** When landscape is selected, the printer gets confused.
    ** Therefore, if TUMBLE is selected, set to NOTUMBLE and visa versa.
    ** Also, do this in RAW mode only.  If the data type is not checked, then
    ** standard will convert duplex to the correct format, but raw will convert
    ** it back to the       format.
    ** *This is for landscape only*
    */
    if (pCNFData->jobProperties.iOrient == LANDSCAPE           &&
        pdv->usDataType == PM_Q_RAW                            &&
        CHECKFLAG( pCNFData->ulFlags, DUPLEX_CHANGED ) != 0)
    {
      ofsSelBit = 0;
      if ((pOldDuplexSelected = (PSZ) QueryUIOptionString( pdesPPD,
                                   ASSIGN_UISELLIST_PTR( pCNFData ),
                                   UINAME_DUPLEX, &ofsSelBit, NULL )) != NULL)
      {
        if (!strcmp( pOldDuplexSelected, UINAME_DUPLEXTUMBLE ))
        {
          pDuplexSelected = UINAME_DUPLEXNOTUMBLE;
        }
        else if (!strcmp( pOldDuplexSelected, UINAME_DUPLEXNOTUMBLE ))
        {
          pDuplexSelected = UINAME_DUPLEXTUMBLE;
        }
        else
        {
          pDuplexSelected = NULL;
        }

        if (pDuplexSelected != NULL)
        {
          SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ),
                       UINAME_DUPLEX, pDuplexSelected );
        }
      }

      // No longer needed - turn it off
      CLEARFLAG( pCNFData->ulFlags, DUPLEX_CHANGED );
    }
#endif

    /*
    ** If a form name is not supplied, set the default form name.
    */
    if (pCNFData->jobProperties.szFormName[ 0 ] == 0)
    {
      strcpy( pCNFData->jobProperties.szFormName,
              GetDefaultPageSize( pdesPPD, pdesPPD->pPSStringBuff ) );
    }

    /*
    ** The PageSize is stored in 'szFormName' and not in the UI Selection
    ** List.  Set the UI_SEL bit for PageSize.
    */
    // @V3.1157420
    if (SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ),
                     UINAME_PAGESIZE, pCNFData->jobProperties.szFormName )
        == -1)
    {
      MapUserDefForm( pdesPPD, pCNFData, pCNFData->jobProperties.szFormName );
    }

    /*
    ** If "AUTOTRAYSELECT" is selected, then we do not want PageRegion and
    ** InputSlot to be included in the job.
    ** Otherwise, PageRegion is the only UI that is not set in Job or Printer
    ** Properties and InputSlot is stored in 'aTraySelected'.  The PageRegion
    ** option will always match PageSize.  Set PageRegion and InputSlot.
    */
    uiListIndex = (UINT) strcmp( pCNFData->u.iv.aTraySelected,
                                 AUTOTRAY_STRING );

    /*
    ** If AUTOTRAY is not selection (uiListIndex != 0), then set the
    ** tray name, and clear all selections to PageSize, since PageSize is not
    ** to be set if a specific tray is selected.
    */
    if (uiListIndex != 0)
    {
      SetUIOption( pdesPPD, pExistingSel, UINAME_INPUTSLOT,
                   pCNFData->u.iv.aTraySelected );

      /*
      ** For non-AUTOTRAY's, do not send the *PageSize string.
      */
      ClearUISelection( pdesPPD, pCNFData, (PSZ) UINAME_PAGESIZE );

      /*
      ** Ensure that the correct bit is set for PageRegion.
      */
      SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ),
                   UINAME_PAGEREGION, pCNFData->jobProperties.szFormName );
    }
    else                        // AUTOTRAY is selected
    {
      /*
      ** AUTOTRAY is selected.  Therefore, PageSize is to be used, but
      ** PageRegion is not.  Clear out the PageRegion selection.
      */
      ClearUISelection( pdesPPD, pCNFData, (PSZ) UINAME_PAGEREGION );
    }

    /*
    ** AUTOTRAY or MANUALFEED is selected.  Clear all selections for
    ** InputSlot since the specific tray is not needed.
    */
    if (uiListIndex == 0 ||
        !strcmp( pCNFData->u.iv.aTraySelected, MANUALFEED_STRING ))
    {
      ClearUISelection( pdesPPD, pCNFData, (PSZ) UINAME_INPUTSLOT );
    }

    pUIBlock = pdesPPD->stUIList.pBlockList;
    for (uiListIndex = 0 ; uiListIndex < uiNumOfBlocks ; uiListIndex++)
    {
      /*
      ** Save the current keyword (block name).
      */
      ofsSelBit = 0;
      pBlockName = pdesPPD->pPSStringBuff + pUIBlock->ofsUIName;
      SaveINIGroupData( (PBYTE) pRC, pBlockName );

      /*
      ** Run through the list and save all selected options (entry names).
      */
      while ((pEntryName = QueryUIOptionString( pdesPPD, pExistingSel,
                                                pBlockName, &ofsSelBit, NULL ))
              != NULL)
      {
        SaveINIGroupData( (PBYTE) pRC, pEntryName );
      }

      /*
      ** Terminate the current group.
      */
      SaveINIGroupData( (PBYTE) pRC, NULL );

      INCREMENT_BLOCK_PTR( pUIBlock );
    }
  }

  return( pRC );
}
/*---------------------------------------------------------------------------*\
* ConvertUIBitsToStrings End                                                  *
\*---------------------------------------------------------------------------*/





/***************************************************************************
 *
 * FUNCTION NAME = MapUserDefForm
 *
 * DESCRIPTION
 * With a given user-defined form name, this function sets the appropriate
 * bits for the UI keys PageSize and PageRegion for the matching real form
 * name.
 *
 * INPUT
 * pdesPPD - PPD description structure.  This includes the option string that
 *   will be returned.
 * pCNFData - Configuration data structure.
 * pUserDefFormName - User-defined form name.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = Number of user-defined forms.
 * RETURN-ERROR  = 0 - No user-defined forms exist.
 *
 ****************************************************************************/
LONG MapUserDefForm( PDESPPD pdesPPD, PCNFDATA pCNFData, PSZ pUserDefFormName )
{
  LONG lNumOfUserForms;                 // Number of user-defined forms
  LONG lIndex;                          // Loop counter
  CHAR aUserString[ MAX_PSIZE ];        // User-defined string buffer
  CHAR aRealString[ MAX_PSIZE ];        // Matching real form buffer

  /*
  ** Query the number of user-defined forms.  If no forms exist, return 0.
  */
  if ((lNumOfUserForms = NewUserForms( pCNFData, QUERY_NUM_OF_USER_FORMS,
                                       NULL, NULL )) > 0)
  {
    /*
    ** This loop runs through the list of user-defined forms to find a
    ** match (provided).
    */
    for (lIndex = 0 ; lIndex < lNumOfUserForms ; lIndex++)
    {
      /*
      ** Query the current user-defined form.
      */
      NewUserForms( pCNFData, (INT) lIndex, (PSZ) aUserString,
                    (PSZ) aRealString );

      /*
      ** If a match exists, with the given real form name, set the PageSize
      ** and PageRegion values.
      */
      if (!strcmp( aUserString, pUserDefFormName ))
      {
        SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ),
                     UINAME_PAGEREGION, aRealString );
        SetUIOption( pdesPPD, ASSIGN_UISELLIST_PTR( pCNFData ),
                     UINAME_PAGESIZE, aRealString );
      }
    }
  }

  return( lNumOfUserForms );
}
/*---------------------------------------------------------------------------*\
* MapUserDefForms End                                                         *
\*---------------------------------------------------------------------------*/





/***************************************************************************
 *
 * FUNCTION NAME = ClearUISelection
 *
 * DESCRIPTION
 * With a given UI keyword string, this function clears all selections for
 * the given UI.
 *
 * INPUT
 * pdesPPD - PPD description structure.  This includes the option string that
 *   will be returned.
 * pCNFData - Configuration data structure.
 * pUIString - UI keyword string.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = Pointer to matching UI block.
 * RETURN-ERROR  = NULL - no block exists for the string.
 *
 ****************************************************************************/
PUI_BLOCK ClearUISelection( PDESPPD pdesPPD, PCNFDATA pCNFData, PSZ pUIString )
{
  PUI_BLOCK pUIBlock;            // UI block
  PUI_SEL   pUISel;              // Pointer to UI selection list

  /*
  ** Query the UI block.
  */
  if ((pUIBlock = QueryBlockFromKeyword( &pdesPPD->stUIList,
                                         pdesPPD->pPSStringBuff,
                                         (PBYTE) pUIString, NULL )) != NULL)
  {
    /*
    ** The UI exists.  Clear all current settings.
    */
    pUISel = ASSIGN_UISELLIST_PTR( pCNFData );
    *(pUISel + pUIBlock->usOrderDep) = 0;
  }

  return( pUIBlock );
}
/*---------------------------------------------------------------------------*\
* ClearUISelection End                                                        *
\*---------------------------------------------------------------------------*/





PGLCB QueryLayoutCBFromCNFDATA(
  PCNFDATA pCNFData
)
{
  PGLCB pRC;

  if (pCNFData != NULL)
  {
    pRC = (PGLCB) ((PBYTE) pCNFData + sizeof( *pCNFData ) +
                    pCNFData->stUISelList.usCurrUIListSize);

  }
  else
  {
    pRC = NULL;
  }

  return( pRC );
}





/***************************************************************************
 *
 * FUNCTION NAME = QueryWCBFromCNFDATA
 *
 * DESCRIPTION
 * With a given CNFDATA pointer, this function returns a pointer to the
 * Watermark Control Block (WCB).
 * The watermark control block is a section of the buffer that follows the
 * UI selection list, which follows the CNFDATA structure:
 *
 *       +-----------------------+
 *       |   CNFDATA structure   |
 *       +-----------------------+
 *       |   UI Selection list   |
 *       +-----------------------+
 *       |Watermark control block|
 *       +-----------------------+
 *
 * INPUT
 * pCNFData - Configuration data structure.
 *
 * OUTPUT
 * None
 *
 * RETURN-NORMAL = Pointer to the watermark control block.
 * RETURN-ERROR  = NULL
 *
 ****************************************************************************/
#if 0
//PGJWCB QueryWCBFromCNFDATA(
//  PCNFDATA pCNFData
//)
//{
//  PGJWCB pRC;
//  LONG lSize = sizeof( CNFDATA ) + pCNFData->stUISelList.usCurrUIListSize +
//               sizeof( GLCB );
//
//  GplWMQueryCB( (PVOID) pCNFData, lSize + sizeof( GJWCB ), &pRC );
//
//  if (pRC == NULL)
//  {
//    pRC = (PGJWCB) ((PBYTE) pCNFData + lSize);
//  }
//
//  return( pRC );
//}
#endif





LONG QueryDrivdataSize(
  PDESPPD pdesPPD
)
{
  LONG lRC;

  if (pdesPPD != NULL)
  {
#if 0
//  lRC = ((sizeof( DRIVDATA ) - sizeof( CHAR ))                +
//         sizeof( CNFDATA )                                    +
//         (pdesPPD->stUIList.usNumOfBlocks * sizeof( UI_SEL )) +
//         sizeof( GLCB )                                       +
//         sizeof( GJWCB ));
#endif
    lRC = ((sizeof( DRIVDATA ) - sizeof( CHAR ))                +
           sizeof( CNFDATA )                                    +
           (pdesPPD->stUIList.usNumOfBlocks * sizeof( UI_SEL )));
  }
  else
  {
    lRC = 0;
  }

  return( lRC );
}
