/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = FONT.C
 *
 * DESCRIPTIVE NAME = POSTSCRIPT DRIVER
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION         Font support code & data
 *
 *
 * FUNCTIONS           FullToFontName
 *                     MatchFaceName
 *                     prda_DefaultFontIndex
 *                     prda_DeviceQueryFontAttributes
 *                     GetMappedFontName
 *                     szIsEqual
 *                     PfntFromIndex
 *                     GetPFMetrics
 *                     FreeFontResource
 *                     SFDownload
 *                     RemapBoldItalic
 *                     szLength
 *                     prda_CountKernPairs
 *                     RemapCodePage
 *                     CPDownLoaded
 *
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#pragma pack(1)
#ifdef IPMD
  #define DEBUG
#endif
#define  INCL_DEV
#define INCL_PM
#include <string.h>
#include <stdlib.h>
#include "inc\prdinclt.h"
#include <pmdev.h>
#define  INCL_SPLDOSPRINT
#include <pmspl.h>
#include "inc\utl.h"
#include "inc\prdgextf.h"
#include "inc\prdtcone.h"
#include "inc\prdtextf.h"
#include "inc\prdmath.h"
#include "inc\allfonts.h"
#include "inc\fremap.h"
#include "inc\ofm.h"
#include "inc\pspagtun.h"             /* V2.174057  Page Tuning */
#include "inc\mapping.h"
#include "inc\usermacr.h"                                               //@DBCS
#define  INCL_GENPLIB_ERROR
#define  INCL_GENPLIB_MEMORY
#define  INCL_GENPLIB_GPLEDC
#define  INCL_GENPLIB_FONTMGR
#include <genplib.h>

#define   OD_MEMORY   8L
/*
** #define TESTING
*/

extern SHORT   prda_DefaultFontIndex(PDDC);

extern VOID   GetMappedFontName(PDDC,PB,int,SHORT);
extern PFNT   PfntFromIndex(PDDC,SHORT);/* charstr2.c             */
extern PFONTMETRICS   GetPFMetrics(PFONTDEF);/* charstr2.c         */
extern VOID   FreeFontResource( PDV, PFONTDEF,PFNT);/* charstr2.c        */
extern VOID   SFDownload(PDDC,PSZ);/* download.c           */
extern HMODULE pscript_module;
// @V3.1140397
extern BOOL IsDBCSChar( PDDC, UCHAR );

extern PDDC EnterDriver(PDDC);
extern VOID ExitDriver(PDDC);
VOID   RemapBoldItalic(PDDC,PFATTRS);
SHORT  szLength(PSZ);
LONG   prda_CountKernPairs(PDDC);
BOOL   RemapCodePage(PDDC,SHORT,SHORT,PSZ);
BOOL   CPDownLoaded(PDDC,SHORT,PSHORT,USHORT);
/* DBCS enabling start */                                               //@DBCS
extern SHORT IsDBCSFont(PDDC);
VOID   IsBoldItalic(PDDC,PFATTRS,SHORT);
SHORT  CheckDispatchGothic(PDDC);
SHORT  SetDispatchEuropean(PDDC,PSZ,SHORT,SHORT);
BOOL   IsEuroInCP( USHORT );
VOID   ResProcessEndPage( PDDC );                                   //@V4.1198538
/* DBCS enabling end   */                                           //@DBCS
VOID   MoveTextMetrics(PFOCAMETRICS, PFONTMETRICS );                
BOOL   IsDynFnt( PDDC pddc, SHORT usIndex );                        
PFATTRS CheckBitMapFont( PFATTRS, PFATTRS );                        //@BITSUB

#define  RF_DELETE_ENGINE_FONT 4
#define  ERROR_NEGONE  (-1L)
#define  NO_CODEPAGE   65400
#define  GRAVE         96    /* a few accent glyph ids                      */
#define  ACUTE         239
#define  CIRCUMFLEX    94
#define  TILDE         126
#define  DIERESIS      249
#define  RING          248
#define  CEDILLA       247
#define  CARON         307
#define  EURO          213
#define  PS_EURO_GLYPH 315
#define  DOTLESSI      315
#define  PS_DOTLESSI_GLYPH 213

/* DBCS enabling start */                                               //@DBCS
static ULONG Dispatch[DBCSFONT_ALL] = {HELVETICA,      TIMES_ROMAN,    HELVETICA,
                                       TIMES_ROMAN,    TIMES_BOLD,
                                       HELVETICA_BOLD, COURIER} ;
/* DBCS enabling end   */                                               //@DBCS

#define eUNKNOWN 0
#define eTRUE    1
#define eFALSE   2
INT  EuroEnable = eUNKNOWN;

/*
**           !!CR FIND OUT WHAT TO DO IN CASE AN INVALID CODE PAGE
**           IS GIVEN AS INPUT
*/
/***************************************************************************
 *
 * FUNCTION NAME =  FullToFontName
 *
 *
 * DESCRIPTION   =  Maps the given Face Name to the appropriate Font Name.
 *                   Copy fontmetrics from source to destination,
 *
 *                  Effects: Scans the list of all fonts offered by
 *                           the current printer until one is found
 *                           whose Face name is the same as that requested.
 *
 * Warnings: Fonts have two different names.  The FullName is the most
 *           complete; an example is "ITC Avant Garde Gothic Book Oblique".
 *           This is the name used for matching.  However, the FontNames
 *           used in the printer font list have an abbreviated form like
 *           "AvantGarde-BookOblique".
 *
 * INPUT         = PSZ      Face Name
 *                 PDDC     pddc
 *
 *
 *
 * OUTPUT        = NONE
 *
 *
 *
 *
 * RETURN-NORMAL = Font Name PSZ.
 *                  WARNING:
 *                  This function overwrites the data at
 *                  pMetricsSrc.  If this data is
 *                  valuable, it should be copied before
 *                  entry to here.
 *
 *
 *
 * RETURN-ERROR  = FALSE
 *
 *
 *
 ****************************************************************************/

PSZ FullToFontName( PSZ pszFaceName, PDDC pddc )
{
  SHORT i;
  
  PDYNFNT_LIST  prDynFnt;

  // Device Name of dynamic font is stored in DynFnt record
  prDynFnt = DynFntListSearchItem ( pddc->pdv->pDynFntList, DYNFNT_SEARCH_FONTNAME, (ULONG) pszFaceName );
  if ( prDynFnt )
  {
    return (PSZ) prDynFnt->rFontProfile.szDeviceName;
  }
  

  for (i = 0; i < pddc->pdv->cFonts; i++)
  {
    if ( (*pddc->pdv->paFonts)[i].pszFullName  &&  
         szIsEqual(pszFaceName, (PSZ)(*pddc->pdv->paFonts)[i].pszFullName))
    {
      return (PSZ) (*pddc->pdv->paFonts)[i].pszFontName;
    }
  }
  return (PSZ) 0L;            /* not found                                   */
}

/***************************************************************************
 *
 * FUNCTION NAME =  MatchFaceName
 *
 * DESCRIPTION   =  Returns the index number (0-n), in the current printer's
 *                  list of supported fonts, of the font that has the
 *                  specified face name.
 *
 *                  Effects: Scans the list of all fonts offered by
 *                           the current printer until one is found
 *                           whose Face name is the same as that requested.
 *                           The index of this matching font is then returned.
 *                           If no match is found, -1 is returned.
 *
 * Warnings: Fonts have two different names.  The FullName is the most
 *           complete; an example is "ITC Avant Garde Gothic Book Oblique".
 *           This is the name used for matching.  However, the FontNames
 *           used in the printer font list have an abbreviated form like
 *           "AvantGarde-BookOblique".
 *
 * INPUT         = PSZ      pointer to Face Name
 *                 PDDC     pointer to display context
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = Index number of desired font, or -1 if not found.
 *
 * RETURN-ERROR  = FALSE
 *
 ****************************************************************************/

SHORT MatchFaceName( PSZ pszFaceName, PDDC pddc, BOOL fOnlySoftFont )
{
  SHORT i;
  PFNT  pFnt;

  /*
  ** Find the index number corresponding to this fontfullname
  */
  for (i = 0; i < pddc->pdv->cFonts; i++)
  {
    pFnt = (PFNT)&(*pddc->pdv->paFonts)[i];

    /*
    ** The tests - must have a fullname, if want softfont then usresource must
    ** be zero, and the names must match
    */
    if ( pFnt->pszFullName    &&
       ( ( fOnlySoftFont && ((SHORT)pFnt->usResource <= 0 )) || !fOnlySoftFont ) &&
         szIsEqual( pszFaceName, pFnt->pszFullName ) )
    {
      return  i;
    }
  }

  return -1;                 /* not found                                   */
}

/***************************************************************************
 *
 * FUNCTION NAME =  FontDownLoad
 *
 * DESCRIPTION   =  Download a soft font if necessary. NOP HW fonts.
 *
 * INPUT         = PDDC     pointer to display context
 *                 PSZ      pointer to Face Name
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

VOID FontDownload( PDDC pddc, PSZ pszName, BOOL fInStartDoc )
{
  SHORT i;
  PFNT  pFnt;
  PDV   pdv = pddc->pdv;

  
  // If dynamic font, it is already downloaded!!!!!!!
  if ( !pddc->pddcb->text.fDynamicFont &&
       0 <= (i = MatchFaceName( pszName, pddc, pddc->pddcb->text.fUseSoftFont )))
  {
    pFnt = (PFNT) &(*pdv->paFonts)[i];

    /*
    ** If soft font
    ** and not yet loaded.
    */
    if ((SHORT) pFnt->usResource <= 0
       && pFnt->bLoaded == FALSE)
    {
      /*
      **  Make sure channel is open before
      **  downloading fonts
      */
      if (!pddc->fHeaderSent           &&
           pdv->usDataType == PM_Q_RAW && 
           fInStartDoc     == FALSE     )                                //@RES
      {
        ps_startdoc(pddc);
      }

      // Since the startdoc can call FontResourceDownload recheck flag
      // If loaded return
      if ( pFnt->bLoaded != FALSE )
      {
        return;
      }

      
      SFDownload( pddc, (PSZ)pFnt->pszPFB);/* define soft font               */

      // If there is knowledge of resource use then fonts are only
      // downloaded once so mark it as locked
      if ( CHECKFLAG( pdv->MRCB.ulFlags, RESOURCE_DATA_SET ) != 0 )
      {
        pFnt->bLoaded = TRUE | LOCKED;  // Mark it as locked
      }
      else
      {
        pFnt->bLoaded = TRUE;           // Mark it as loaded
      }
    }
  }
  return ;
}


/***************************************************************************
 *
 * FUNCTION NAME =  NormalizeMetrics
 *
 * DESCRIPTION   =  Adjusts font metrics received from graphics engine
 *                  for use in driver. Actually scales them for resolution
 *                  of 1000. These values are usually passed in as for
 *                  2028 resolution, but that does not suit us.
 *
 *                  The code is actually taken from fntlib\ttfps.c 
 *                  function PSprocessTTF(). Consider it magic.
 *
 *
 * INPUT          
 *            fm =  Pointer to FONTMETRICS structure that needs to
 *                  be adjusted
 *
 * OUTPUT        =  fm structure memory is modified
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
*/
VOID NormalizeMetrics(PFONTMETRICS fm)
{
   double   mult;

/* now fix all numbers so that the device resolutions are 1000 */
   mult = (double)1000 / (double)fm->sXDeviceRes;
   
   fm->sXDeviceRes = 1000;
   fm->sYDeviceRes = 1000;
   
   fm->lEmHeight = (long)(fm->lEmHeight * mult);
   fm->lMaxAscender = (long)(fm->lMaxAscender * mult);
   fm->lMaxDescender = (long)(fm->lMaxDescender * mult);
   fm->lLowerCaseAscent = (long)(fm->lLowerCaseAscent * mult);
   fm->lLowerCaseDescent = (long)(fm->lLowerCaseDescent * mult);
   fm->lInternalLeading = (long)(fm->lInternalLeading * mult);
   fm->lExternalLeading = (long)(fm->lExternalLeading * mult);
   fm->lAveCharWidth = (long)(fm->lAveCharWidth * mult);
   fm->lMaxCharInc = (long)(fm->lMaxCharInc * mult);
   fm->lEmInc = (long)(fm->lEmInc * mult);
   fm->lMaxBaselineExt = (long)(fm->lMaxBaselineExt * mult);
   fm->lSubscriptXSize = (long)(fm->lSubscriptXSize * mult);
   fm->lSubscriptYSize = (long)(fm->lSubscriptYSize * mult);
   fm->lSubscriptXOffset = (long)(fm->lSubscriptXOffset * mult);
   fm->lSubscriptYOffset = (long)(fm->lSubscriptYOffset * mult);
   fm->lSuperscriptXSize = (long)(fm->lSuperscriptXSize * mult);
   fm->lSuperscriptYSize = (long)(fm->lSuperscriptYSize * mult);
   fm->lSuperscriptXOffset = (long)(fm->lSuperscriptXOffset * mult);
   fm->lSuperscriptYOffset = (long)(fm->lSuperscriptYOffset * mult);
   fm->lUnderscoreSize = (long)(fm->lUnderscoreSize * mult);
   fm->lUnderscorePosition = (long)(fm->lUnderscorePosition * mult);
   fm->lStrikeoutSize = (long)(fm->lStrikeoutSize * mult);
   fm->lStrikeoutPosition = (long)(fm->lStrikeoutPosition * mult);

   return ;
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_RealizeFont
 *
 * DESCRIPTION   =  Attempts to match a device font to a requested
 *                  logical font and returns a LONG handle to the
 *                  font if successful.
 *
 *            pfatLogFont    =  Pointer to a FATTRS structure con-
 *                              taining logical font information
 *                              if this is a RF_DEVICE_FONT command.
 *            lFontHandle    =  The handle of the logical font to be
 *                              deleted if this is a RF_DELETE_FONT
 *                              command.
 *
 *
 *          RF_DELETE_FONT:
 *             1                Success
 *             0                error
 *
 * Actions:
 *          RF_DEVICE_FONT      Attempt to match font, return handle.
 *          RF_LOAD_ENGINE_FONT Return 0L, font can't be realized.
 *          RF_DELETE_FONT      Attempt to delete font with handle
 *                              specified.
 *
 * RF_DEVICE_FONT   In this case, the PostScript driver will always
 *              return a matching device font unless a code page is
 *              requested that we cannot support.  If a legitimate
 *              code page is requested but no match is provided and
 *              we can't match the facename, the  default font will
 *              be used.  The algorithm used is:
 *
 *              Check the lMatch field of the FATTRS structure.  If
 *              negative, it is presumably one of the "magic cookies"
 *              we gave the engine from DeviceQueryFonts, and we choose
 *              that font.  If positive, then the engine is telling
 *              us it will do font simulations, and we return a handle
 *              of zero.  If lMatch is zero, we look for a facename
 *              in the FATTRS structure.  If the facename is not null,
 *              we attempt to match it to one of the facenames in our
 *              list of fonts supported by the current printer.  If we
 *              match, we select the matched font.  If we don't match
 *              or the facename is null, we choose the default font.
 *
 *              Once we have chosen a font, we deal with code pages.
 *              If the usCodePage field of the FATTRS structure is
 *              zero, we simply choose the code page provided by the
 *              font.  Otherwise, we try to get a mapping vector from
 *              the requested code page to the code page supported by
 *              the font.  If we cannot get such a vector, THE FONT
 *              IS NOT REALIZABLE; we return zero to the engine.
 *
 *              Once we have chosen a font and found a suitable code
 *              page mapping vector, we look for space in our logical
 *              font info table in the pddc.  If an entry is free,
 *              we allocate it, initialize its records, and return
 *              the negative of the index as the 32 bit handle. If
 *              the table is full, we must return an error.
 *
 * RF_LOAD_ENGINE_FONT   We always return 0L.
 *
 * RF_DELETE_FONT        If the handle provided is legitimate, we
 *                       erase the corresponding font from our logical
 *                       font info table and return 1L.  Otherwise
 *                       we must return an error.  If the handle
 *                       provided corresponds to the logical font
 *                       currently in use, we first select the default
 *                       font before deleting the logical font.
 *
 *
 * INPUT         =  RF_DEVICE_FONT or
 *                  RF_LOAD_ENGINE_FONT or
 *                  RF_DELETE_FONT
 *
 * OUTPUT        = RF_DEVICE_FONT or RF_LOAD_ENGINE_FONT:
 *                 >0 32-bit handle for the new logical font
 *                  0 Font cannot be realized
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/
/*
**              !!!CR FIND OUT WHAT CODE PAGE TO USE FOR SYMBOL FONT.
*/

ULONG prda_RealizeFont( HDC hdc, ULONG ulCommand, PFATTRS pfa,
                        LONG lFontHandle, PDDC pddc, ULONG FunN)
{
  SHORT        prda_SetTextAttrs(SHORT, SHORT, PDCHARBUNDLE, PDDC);
  LONG         lFont;                /* font cookie we will return        */
  LONG         lOffset;              /* font offset from MatchFaceName    */
  USHORT       usAvailableCodePage;/* code page the font supports         */
  PUSHORT      pusRequestedVector = 0L;/* vector for requested code page  */
  USHORT       usRequestedCodePage;                              //@V4.0179015
  LONG         i;
  PFNT         pFnt;
  PFONTDEF     pfdFont;
  PFONTMETRICS pfmMetrics;
  USHORT       usIndex, usCodePage;
  CHAR         szFont[FACESIZE];
  BOOL         fMustUseSoftFont;    
  PCNFDATA     pCNFData = pddc->pdv->pCNFData;                   // @V3.1142031
  BOOL         fNeedBackupFont = FALSE;                               //@GPLEDC
  
  PFOCAFONT    pffFont = (PFOCAFONT) lFontHandle;
  PDYNFNT_LIST prDynFnt = NULL;
  
/* DBCS enabling start */                                               //@DBCS
  BOOL         fDBCS;
  FATTRS       NewFATTRS;                                             //@BITSUB

  fDBCS = (pddc->pddcb->text.ubDBCSVector[0]) ? TRUE : FALSE;
/* DBCS enabling end   */                                               //@DBCS

  EnterDriver( pddc );

  if (pddc->iType == OD_MEMORY)
  {
    ExitDriver( pddc );
    return (ulCommand == RF_DELETE_FONT ? GPI_ERROR : 0);
  }

  // In order change name from bitmap to outline or remap to bold &| italic
  // must change pfa
  // Since it is not good write in user supplied data we make a local
  // copy and update that
  if ( pfa != NULL )
  {
    NewFATTRS = *pfa;   // Make copy
    pfa = &NewFATTRS;   // Change the pfa ptr to point to local copy
  }

  /*
  **  !!!CR GET THESE CONSTANTS DEFINED
  */
  switch ((SHORT)ulCommand)
  {
  case RF_DEVICE_FONT :   /* Realize a logical font                      */
       /*
       ** check lMatch
       */
       /*
       **            !!!CR COMMENT ON WHAT THE DIFFERENT LMATCH VALUES MEAN
       */
       /*
       ** if we have a positive lMatch just return 0, ie no match.
       */
       /*
       ** Since we are using ATM fonts lets see if we have the
       ** softfont on a positive lmatch
       */
       if (pfa->lMatch > 0L)
       {
         if ( (PSZ) pfa->szFacename == NULL )
         {
           ExitDriver(pddc);
           return (0L);
         }
         else
         {
           fMustUseSoftFont = TRUE;
         }
       }
       else
       {
         /*
         ** for 0 or neg lmatch
         */
         fMustUseSoftFont = FALSE;
       }

       /*
       ** Use the magic cookie given by the caller
       */
       lFont = pfa->lMatch;

       /*
       ** save the selection options for use in charstringpos
       */
       pddc->pddcb->cgs.usSelection = pfa->fsSelection;

       /*
       ** check Facename
       */
       if ((PSZ) pfa->szFacename != NULL)
       {
         PrintLog( (PSZ) "FaceName = %ls", (PSZ) pfa->szFacename);

         //@BITSUB
         // If system bitmap font Tms Rmn or Helv the sub it out
         pfa = CheckBitMapFont( pfa, &NewFATTRS );

         /*
         ** if the BOLD and/or ITALIC attributes are set,
         ** remap the font to take the attributes into
         **  account.
         */
         if (pfa->fsSelection & (FATTR_SEL_BOLD + FATTR_SEL_ITALIC))
         {
           RemapBoldItalic(pddc, pfa);
         }

         /*
         ** MatchFaceName attempts to find the facename in
         ** our current printer font list.  If successful, it
         ** returns the font's offset in the directory.  If
         ** unsuccessful, it returns -1
         **
         **
         **  !!!CR Cast MatchFacename to a long
         **  !!!CR lOffset should be lFontIndex
         */
         if ((lOffset = MatchFaceName((PSZ) pfa->szFacename, pddc,
                                      fMustUseSoftFont) ) >= 0L )
         {

           /*
           ** convert the font offset to a magic cookie
           */
           lFont = OffsetToCookie( lOffset );
           PrintLog( (PSZ) "...FaceName matched \n");
         }

         /*
         ** matches failed: choose default printer font
         */
         else
         {
           PrintLog( (PSZ) "...no match: choosing default font\n");
           lFont = OffsetToCookie((LONG)prda_DefaultFontIndex(pddc));

           /*
           **  The facename specified is not supported by driver .
           ** We should give a chance to engine to download the
           ** specified font.Hence we return 0 i.e. font is not
           ** realizable .
           */
           ExitDriver( pddc );
           return (0L);
         }
       }
       else
       {
         /*
         ** They didn't give an lMatch or a facename
         */
         PrintLog( (PSZ) "No lMatch or facename; using default font\n" );
         lFont = OffsetToCookie( (LONG) prda_DefaultFontIndex( pddc ) );
       }

       PrintLog( (PSZ) "RealizeFont--Cookie for chosen font = %ld\n", lFont );

       /*
       ** Settle the issue of code page
       */
       /*
       **  !!!CR EXPLAIN WHAT "SETTLE THE CODE PAGE ISSUE" MEANS
       */
       PrintLog( (PSZ) "Requested code page: %d\n", pfa->usCodePage );

       /*
       ** See what code page the chosen font supports
       */
       usIndex = (SHORT) CookieToOffset( lFont );
       pFnt = PfntFromIndex( pddc, usIndex );

       
       // If DynamycFont let the engine realize it in RF_LOAD_ENGINEFONT
       if ( IsDynFnt( pddc, usIndex ) )
       {
           ExitDriver( pddc );
           return (0L);
       }
       

       /* @V4.0187422
       ** If the Font LMatch is zero and the driver finds the font is
       ** is downloaded the mark it so
       */
       if ( pfa->lMatch == 0     &&
            pFnt->usResource <= 0 )
       {
         fMustUseSoftFont = TRUE;
       }

       /*       ** Check to see if enough memory on printer for softfont
       */
       if ( (SHORT) pFnt->usResource <= 0                     &&
             CHECKFLAG( pFnt->ulFlags, FNT_FONTCOUNTED ) == 0 &&
             pCNFData->lFontCount < 999 )  /* 999 turns off check */
       {
         // @V3.1142031
         if ( pCNFData->lFontCount <= 0 )
         {
           ExitDriver(pddc);
           return ( 0L );
         }
         else
         {
           pCNFData->lFontCount--;
           SETFLAG( pFnt->ulFlags, FNT_FONTCOUNTED );
         }
       }

       pfdFont = prda_LoadFontResource(pddc, usIndex );

       if (!pfdFont)          /* Couldn't get resource                       */
       {
         /*
         ** !!CR Magic number, and ulong
         */
         ExitDriver(pddc );
         return ((LONG)-1 );   /* Minor function saves error code             */
       }

       pfmMetrics = GetPFMetrics( pfdFont );
       usAvailableCodePage = pfmMetrics->usCodePage;
       PrintLog( (PSZ) "This font supports code page %d\n", usAvailableCodePage) ;
       FreeFontResource( pddc->pdv, pfdFont, pFnt);

/*
** -----------------------------------------------------------------------------
** Jump to here from LOADENGINE, if we have dynamic font
** In this case we should return 0 anyway
*/
FillLFI:

       // Set the requested code page to pfa supplied one         //@V4.0179015
       usRequestedCodePage = pfa->usCodePage;                     //@V4.0179015
                                                                  //@V4.0179015
       // If supplied CP is 0 then use default                    //@V4.0179015
       if ( usRequestedCodePage == 0 )                            //@V4.0179015
       {                                                          //@V4.0179015
         usRequestedCodePage = pddc->pddcb->text.usDefCodePg;     //@V4.0179015
       }                                                          //@V4.0179015

       /*
       ** Determine if we can map the code page that the caller wishes to
       ** use into the code page that this font supports.  If not, then
       ** we cannot realize this font.
       **
       ** !!!CR make a define for 850 instead of using a magic number
       */
       /*       ** Don't worry about application/caller codepage compatibility if font
       ** has 65400 (no code page)
       */
       if ( usAvailableCodePage != NO_CODEPAGE )
       {
         if ((usAvailableCodePage == 850) ||                            //@DBCS
             (fDBCS))           // The code page is for DBCS.           //@DBCS
         {
           /*
           ** This font offers Winthorn multi-codepage support, but has the caller
           ** requested a codepage that the engine is not familiar with?
           */
           if ( usRequestedCodePage != 0 )                        //@V4.0179015
           {
             pusRequestedVector = (PUSHORT) GreQueryCodePageVector((ULONG)
                                  usRequestedCodePage );          //@V4.0179015

             if (!pusRequestedVector)
             {
               PrintLog( (PSZ)
                  "QueryCodePageVector doesn't know requested code page\n" );
               PrintLog( (PSZ) "Unknown codepage requested. Cannot realize font\n" );
               ExitDriver( pddc );
               return (0L );
             }
           }
         }
         else
         {
           /*
           ** This font only works with one specific code page.  The caller
           ** must have either requested that code page number explicitly,
           ** or requested code page zero.
           */
           if ( usRequestedCodePage != usAvailableCodePage  &&    //@V4.0179015
                usRequestedCodePage != 0                     )    //@V4.0179015
           {
             PrintLog( (PSZ)
                "Font requires specific code page. Cannot realize font\n" );
             ExitDriver( pddc );
             return (0L );
           }
         }
       }

       /*
       ** Now we must find a logical font info entry that is free
       */
       for (i = MIN_FONT_LCID; (i <= MAX_FONT_LCID) &&
       (pddc->pddcb->text.lfiEntry[i].usFlags&LFONT_IN_USE ); i++);

       if (i <= MAX_FONT_LCID)
       {
         /*
         */
         pddc->pddcb->text.fUseSoftFont = fMustUseSoftFont;

         /*
         ** Set up entry in logical font info table
         **
         ** !!!CR create constants for these flags instead of BIT1, BIT0, ETC
         */
         pddc->pddcb->text.lfiEntry[i].sOffset = (SHORT) CookieToOffset( lFont );
         pddc->pddcb->text.lfiEntry[i].usFlags = LFONT_IN_USE;

         if (pfa->fsType&BIT2)
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_HAS_KERNING;
           PrintLog( (PSZ) "This font has kerning\n" );
         }

         if (pfa->fsSelection&FATTR_SEL_UNDERSCORE)
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_HAS_UNDERSCORE;
           PrintLog( (PSZ) "This font has underscore\n" );
         }

         if (pfa->fsSelection&FATTR_SEL_STRIKEOUT)
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_HAS_STRIKEOUT;
           PrintLog( (PSZ) "This font has strikeout\n" );
         }

         if (pfa->fsSelection&FATTR_SEL_OUTLINE)
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= (LFONT_IS_OUTLINE+
              LFONT_OUTLINE_NOTDONE );
           PrintLog( (PSZ) "This font is outline font\n" );
         }

/* DBCS enabling start */                                               //@DBCS
         /*
         ** Set the attribute of Bold or italic.
         */
         if (pfa->fsSelection & FATTR_SEL_ITALIC)
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_HAS_ITALIC;
           PrintLog( (PSZ) "This font is italic font\n" );
         }

         if (pfa->fsSelection & FATTR_SEL_BOLD)
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_HAS_BOLD;
           PrintLog( (PSZ) "This font is bold font\n" );
         }

         /*
         ** Check process for Bold/Itablic font.
         */
         IsBoldItalic( pddc, pfa, i );
/* DBCS enabling end   */                                               //@DBCS

         /*
         ** if no codepages have been remapped, this means
         ** that we are still using codepage 850, and don't
         ** have any remapping to worry about.
         */

         /*
         ** mark this newly defined font as having not been
         ** remapped to a codepage.
         */
         ClearFontRemap( i );

         /*
         ** make sure this newly defined font works with
         ** the current codepage.  use the codepage passed
         ** in the FATTRS structure.  if, however, it is zero,
         ** use the default codepage.
         */
         if ((PSZ) pfa->szFacename != NULL)
         {
           szCopy((PSZ) szFont, (PSZ) pfa->szFacename, FACESIZE );
         }
         else
         {
           // @V3.1142031
           szCopy((PSZ) szFont, (PSZ) pCNFData->szFontName, FACESIZE );
         }

         /*
         ** If the font is NO_CODEPAGE keep it instead of using the
         ** process codepage
         */
         if ( usAvailableCodePage == NO_CODEPAGE )  /* APL problem */
         {
           usCodePage = NO_CODEPAGE;
         }
         else
         {
           // The code page set above                             //@V4.0179015
           {
             usCodePage = usRequestedCodePage;
           }
         }

         /*
         ** save the codepage for the the logical font we
         ** are realizing.
         */
         pddc->usCodePages[i] = usCodePage;

         
         // If LOAD_ENGINE_FONT we should return 0 anyway
         // It is dynamic font!!!!
         if ( (SHORT)ulCommand == RF_LOAD_ENGINE_FONT )
         {
           //i is logical font entry index for this font
           prDynFnt->ulDeviceFont = i;
           ExitDriver( pddc );
           return( 0 );
         }

#if      0
//////////*
/////////** there is no need to remap the font if no
/////////** codepages have been remapped, and the codepage
/////////** is 850. This is an optimization which should
/////////** prevent the remapping code from being downloaded
/////////** for most print jobs. There is code in QuoteChar
/////////** which checks the bytes as they are output, so
/////////** even if we are using codepage 850, if we use
/////////** non-ascii characters, the remapping will get done
/////////** when necessary.
/////////**
/////////** NO_CODEPAGE is the codepage which means leave this
/////////** font alone; don't do any remapping. also, i have
/////////** put in a check to see if we have codepage 1004 or
/////////** 850. both of these codepages have the identical
/////////** first 128 characters. if an extended character is
/////////** chosen for either of these fonts, the remapping will
/////////** be done at print time by prda_QuoteChar. - kentse
/////////*/
/////////if (((pddc->pddcb->cgs.cCodePagesRemapped != 0) ||           //@DBCS
/////////    (usCodePage != 850 && !fDBCS)                            //@DBCS
/////////  ) && (usCodePage != NO_CODEPAGE))
/////////{
/////////  if (!RemapCodePage(pddc, usCodePage, (SHORT)i, (PSZ)szFont))
/////////  {
/////////    ExitDriver( pddc );
/////////    return( 0L );
/////////  }
/////////  SetFontRemap(i );
/////////}
#endif
         lFont = OffsetToCookie( i );

         // For gpledc
         if ( lFontHandle != NULL ) // Engine supports backup fonts
         {
           // NOTE
           // It has been brought to my attention that the user can pick ANY
           // codepage at runtime so all codepages need backup fonts
////////// // If current CodePage is 850 or 850 is in system prepared codepages
////////// // Set up a backup font
////////// if ( IsEuroInCP( pddc->usCodePages[i] ) != TRUE )
////////// {
//////////   ULONG ulCPs[3];
//////////   ULONG ulRetCount;
//////////   INT j;
//////////   // Check prepared CPs
//////////   DosQueryCP( sizeof( ulCPs ), ulCPs, &ulRetCount );
//////////   ulRetCount /= sizeof( ULONG );   // RetCount is total bytes - convert to longs
//////////   for ( j = 0; j < ulRetCount; j++ ) // Loop thru
//////////   {
//////////     if ( IsEuroInCP( ulCPs[j] ) == TRUE )
//////////     {
//////////       fNeedBackupFont = TRUE;
//////////       break;
//////////     }
//////////   }
////////// }
////////// else
////////// {
             fNeedBackupFont = TRUE;
////////// }
         }

         if ( fNeedBackupFont == TRUE )
         {
           *((PSZ *)lFontHandle) = (PSZ)(pfa->szFacename);
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_HAS_BACKUP;
         }

         //@V4.1198538
         //Need to note that font is soft font
         if ( fMustUseSoftFont != 0 )
         {
           pddc->pddcb->text.lfiEntry[i].usFlags |= LFONT_IS_SOFTFONT;
         }

         ExitDriver( pddc );
         return( lFont );
       }
       else
       {
         /*
         ** Logical font info table is full
         */
         PrintLog( (PSZ) "Logical Font info table is full! " );
         PrintLog( (PSZ) "-- Not realized\n" );

         /*
         **  TALK TO PAULK ABOUT THE CORRECT ERROR CODE TO RETURN
         */
         GplErrSetError(  PMERR_INSUFFICIENT_MEMORY );

         /*
         ** !!CR magic number and ulong
         */
         ExitDriver( pddc );
         return( (ULONG)-1 );
       }

  case RF_LOAD_ENGINE_FONT :
       /*
       ** PostScript driver must return 0L for this command
       */
/*
**!!! THIS CODE IS A FIX  TO GET AROUND AN ENGINE BUG.  THE ENGINE
**!!! IS CALLING US WITH THE DELETE_ENGINE_FONT COMMAND INSTEAD OF
**!!! THE DELETE_FONT COMMAND.  ONCE THE ENGINE IS FIXED, TAKE THIS
**!!! CODE OUT OF HERE!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
/*
**!!! THIS CODE IS A FIX  TO GET AROUND AN ENGINE FUNCTION.  THE ENGINE
**!!! IS CALLING US WITH THE DELETE_ENGINE_FONT COMMAND INSTEAD OF
**!!! THE DELETE_FONT COMMAND.
*/

       
       /*
       ** See if the font handle is valid
       */
       if (lFontHandle > 0 && pfa->szFacename != NULL)
       {

           PrintLog( (PSZ) "FaceName = %ls", (PSZ) pfa->szFacename);

           //@BITSUB
           // If system bitmap font Tms Rmn or Helv the sub it out
           pfa = CheckBitMapFont( pfa, &NewFATTRS );

           /*
           ** if the BOLD and/or ITALIC attributes are set,
           ** remap the font to take the attributes into
           **  account.
           */
           if (pfa->fsSelection & (FATTR_SEL_BOLD + FATTR_SEL_ITALIC))
           {
             RemapBoldItalic(pddc, pfa);
           }

           fMustUseSoftFont = FALSE;

           if ((lOffset = MatchFaceName((PSZ) pfa->szFacename, pddc,
                                        fMustUseSoftFont) ) >= 0L )
           {

             /*
             ** convert the font offset to a magic cookie
             */
             usIndex = (USHORT) lOffset;
             pFnt = PfntFromIndex( pddc, usIndex );

             // If DynamycFont let the engine realize it
             // but do all stuff like in device font case
             if ( IsDynFnt( pddc, usIndex ) )
             {
               prDynFnt = pFnt->prDynFnt;
               if ( prDynFnt->ulRefCount == 0 )
               {
                 prDynFnt->ulRefCount = 1;
                 prDynFnt->pffGreFont = pffFont;

                 // acquire metrics from engine
                 MoveTextMetrics ( &(pffFont->fmMetrics), (PFONTMETRICS) &(prDynFnt->FontMetrics) );
                 // normalize those metrics for use with driver              //@V1.0VV03
                 // (adjust some values)                                     //@V1.0VV03
                 NormalizeMetrics( &(prDynFnt->FontMetrics) );               //@V1.0VV03

                 usAvailableCodePage = prDynFnt->FontMetrics.usCodePage;
                 lFont = usIndex;
                 goto FillLFI;
               }
               else
               {
                 prDynFnt->ulRefCount++;
               }
             }
           }
       }
       
       ExitDriver( pddc );
       return( 0L );           /* !!CR magic number                           */

/*
**!!! END OF FIX !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
       /*
       ** !!!CR                    FIND OUT WHEN THIS IS CALLED
       ** !!!CR AND COMMENT  THE REASON HERE.
       */

// @V3.1140397
#if 0
//    PrintLog( (PSZ) "RF_DELETE_ENGINE_FONT - returning error\n" );
//    ExitDriver( pddc );
//    return( 0L );           /* !!CR magic number                           */
#endif
  case RF_DELETE_ENGINE_FONT :
       /*
       ** PostScript driver must return 0L for this command
       ** See if the font handle is valid
       */
       if (lFontHandle > 0)
       {
         
         //If Font have Dynamic Font, delete assignment
         prDynFnt = DynFntListSearchItem ( pddc->pdv->pDynFntList, DYNFNT_SEARCH_GREFONT, lFontHandle );
         if ( prDynFnt )
         {
           if ( prDynFnt->ulRefCount > 0 )
           {
             prDynFnt->ulRefCount--;

             if ( prDynFnt->ulRefCount == 0 )
             {
               pddc->pddcb->text.lfiEntry[ prDynFnt->ulDeviceFont ].usFlags = 0;
             }
           }
           else
           {
             PrintLog( (PSZ) "RF_DELETE_FONT - Font already deleted\n", lFontHandle );
             GplErrSetError(  PMERR_INV_FONT_ATTRS );
           }

         }
         ExitDriver( pddc );
         return( 0 );
       }
       lOffset = CookieToOffset( lFontHandle );

       if ((lOffset < MIN_FONT_LCID) || (lOffset > MAX_FONT_LCID))
       {
         PrintLog( (PSZ) "RF_DELETE_FONT - Illegal handle %ld\n", lFontHandle );
         GplErrSetError(  PMERR_INV_FONT_ATTRS );
         ExitDriver( pddc );
         return( 0L );         /* !!CR magic number                           */
       }

       /*
       ** Handle is valid -- free up the logical font table entry
       */
       pddc->pddcb->text.lfiEntry[lOffset].usFlags = 0;
       PrintLog( (PSZ) "RF_DELETE_FONT - Deleted font entry %ld\n", lOffset );
       ExitDriver( pddc );
       return( 0L );           /* !!CR magic number                           */

/*
**!!! END OF FIX !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
*/
       /*
       ** !!!CR                    FIND OUT WHEN THIS IS CALLED
       ** !!!CR AND COMMENT  THE REASON HERE.
       */

// @V3.1140397
#if 0
//    PrintLog( (PSZ) "RF_DELETE_ENGINE_FONT - returning error\n" );
//    ExitDriver( pddc );
//    return( 0L );           /* !!CR magic number                           */
#endif


  case RF_DELETE_FONT :
       /*
       ** See if the font handle is valid
       */

      lOffset = CookieToOffset( lFontHandle );

      if ((lOffset < MIN_FONT_LCID) || (lOffset > MAX_FONT_LCID))
      {
        PrintLog( (PSZ) "RF_DELETE_FONT - Illegal handle %ld\n", lFontHandle );
        GplErrSetError(  PMERR_INV_FONT_ATTRS );
        ExitDriver( pddc );
        return( 0L );         /* !!CR magic number                           */
      }

      /*
      ** Handle is valid -- free up the logical font table entry
      */
      pddc->pddcb->text.lfiEntry[lOffset].usFlags = 0;
      PrintLog( (PSZ) "RF_DELETE_FONT - Deleted font entry %ld\n", lOffset );
      ExitDriver( pddc );
      return( 1L );           /* !!CR magic number                           */

  default:
      PrintLog( (PSZ) "--? Unknown command --?\n" );
      GplErrSetError(  PMERR_INV_FONT_ATTRS );
      ExitDriver( pddc );
      return( 0L );           /* magic number !!CR                           */
  }
}

/***************************************************************************
 *
 * FUNCTION NAME =  RemapBoldItalic
 *
 * DESCRIPTION   =
 * This routine is called when we are about to realize a font, and the
 * BOLD and/or ITALIC attributes have been set.  This routine maps the
 * given font into the appropriate font, depending on which attributes
 * have been set.  For example, if the user chooses Courier as the font
 * and has the BOLD attribute set, this routine will select
 * Courier-Bold as the font.
 *
 * INPUT         =  PDDC
 *                  PFATTRS
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = NONE
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

VOID RemapBoldItalic( PDDC pddc, PFATTRS pfa )
{
  FONT_ID_NUM  FIDNum;
  FONT_INFO    FInfo;
  USHORT       usSelection;

  // Search the list of Device/System fonts
  // If the font is found then we can switch it for one in the same
  // family with the requested attribute

  FIDNum = FindFontIDNum( pddc, pfa->szFacename, (pfa->lMatch > 0L) );

  /*
  ** do nothing if we could not find the font
  */
  if (FIDNum == FONT_NOT_FOUND)
  {
    return;
  }

  // Expand the ID num
  FInfo.iFamily = HIUSHORT( FIDNum );
  FInfo.iType = LOUSHORT( FIDNum );

  // Set us a switch - note the attribute(s) are or'ed in so we do not
  // do not downgrade a font - for example Courier Bold with italic bit set
  // should become Courier Bold Italic not Courier Italic

  usSelection = pfa->fsSelection & ( FATTR_SEL_BOLD | FATTR_SEL_ITALIC );
  switch( usSelection )
  {
  case FATTR_SEL_ITALIC:
    FInfo.iType |= ITALIC;
    break;

  case FATTR_SEL_BOLD:
    FInfo.iType |=  BOLD;
    break;

  case FATTR_SEL_BOLD | FATTR_SEL_ITALIC:
    FInfo.iType |= BOLDITALIC;
    break;
  }

  // If the family has the desired type copy it over
  if ( AllFonts[FInfo.iFamily][FInfo.iType] != NULL )
  {
    strcpy( (PSZ)pfa->szFacename, AllFonts[FInfo.iFamily][FInfo.iType] );
  }

  return;
}

/***************************************************************************
 *
 * FUNCTION NAME =  szLength
 *
 * DESCRIPTION   =  This routine calculated the length of a given
 *                  string, including the terminating NULL.
 *
 * INPUT         =  PSZ     pointer to the string.
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = Length of the given string.
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

SHORT szLength( PSZ pszScan )
{
  SHORT i;

  i = 1;

  while (*pszScan++ != '\0')
  {
    i++;
  }
  return( i );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_countKernPairs
 *
 * DESCRIPTION   = Count the number of kerning pairs that are available for
 *                 the current font and codepage.  This may be different from
 *                 the maximum number of kerning pairs available if the
 *                 some of the kerned characters are not in the current
 *                 codepage.
 *
 * INPUT         =  PSZ     pointer to the string.
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = Number of Kerning Pairs.
 *
 * RETURN-ERROR  = NONE
 *
 ****************************************************************************/

LONG prda_CountKernPairs( PDDC pddc )
{
  PKERNPAIRS pkp;            /* Pointer to our kern table                   */
  SHORT      sCodePoint1;    /* Kerning pair: first character               */
  SHORT      sCodePoint2;    /* Kerning pair: second character              */
  LONG       i = 0;          /* The kerning table index                     */
  LONG       nPairs = 0;     /* The number of kern pairs                    */

  /*
  ** Make sure the font resource is loaded.
  */
  if (!pddc->pddcb->cgs.fValidFont)
  {
    if (!prda_CheckFontResource(pddc))/* couldn't get resources             */
    {
      return( ERROR_NEGONE ); /* Minor function saves error code             */
    }

    /*
    ** ps_selectfont(pddc );   **utlps.c
    */
  }

  /*
  ** Point to start of kerning table
  */
  pkp = (PKERNPAIRS) (((PB) pddc->pddcb->text.pfdfFontDef) +
                       pddc->pddcb->text.pfdfFontDef->Header.usKerningDataOffset );

  for (i = 0; i < pddc->pddcb->text.pfmFontMetrics->sKerningPairs; ++i)
  {
    /*
    ** The code points listed in a font's kerning table are stored in
    ** the code page of the font (usually 850). Before a kerning pair
    ** may be returned to the caller, the code points must be
    ** translated into the caller's code page.  This is what the
    ** prda_BackMapChar minor function does.
    */
    sCodePoint1 = prda_BackMapChar( pkp[i].g1, pddc );
    sCodePoint2 = prda_BackMapChar( pkp[i].g2, pddc );

    if (!((sCodePoint1 == -1) || (sCodePoint2 == -1)))
    {
      ++nPairs;
    }
  }
  return( (LONG) nPairs );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_DeviceQueryFontAttributes
 *
 * DESCRIPTION   = Copies the FONTMETRICS data for the current font to the
 *                 caller's buffer.
 *
 * Effects: Copies the entire FONTMETRICS data structure to a temporary
 *          buffer, rescales it, and then copies the requested amount to
 *          the caller's buffer.
 *
 * Warnings: A request for more bytes of information than are contained in
 *           a FONTMETRICS structure will be silently reduced to a request
 *           for exactly as many bytes as are in a FONTMETRICS structure.
 *
 * INPUT  =  hdc;             Display context handle
 *           ulNumBytes;      How many bytes of FONTMETRICS data to return
 *           pfmFontMetrics;  Pointer to buffer where data shall be placed
 *           pddc;            Pointer to display context instance data
 *           FunN;            Engine function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL = SUCCESS
 *
 * RETURN-ERROR  = FAILURE
 *
 ****************************************************************************/

ULONG prda_DeviceQueryFontAttributes( HDC hdc, ULONG ulNumBytes,
                                      PFONTMETRICS pfmFontMetrics,
                                      PDDC pddc, ULONG FunN)
  /*
  ** HDC          hdc;               Display context handle
  ** ULONG        ulNumBytes;        How many bytes of FONTMETRICS data to return
  ** PFONTMETRICS pfmFontMetrics;    Pointer to buffer where data shall be placed
  ** PDDC         pddc;              Pointer to display context instance data
  ** ULONG        FunN;              Engine function number
  */

{
  LONG lMatch;               /* The unique negativematch value for this
                                font */
  EnterDriver( pddc );

  #if      XDEBUG
    LogCall( "prda_DeviceQueryFontAttributes (%08lx, %lp, %lp, %08lx)\n", ((PB)
             &hdc)+sizeof(hdc) );
  #endif /* XDEBUG                                     */

  PrintLog( (PSZ) "ulNumBytes = {%ld}\n", ulNumBytes );

  /*
  ** Raise an error in the case of memory dc.
  */
  if (pddc->iType == OD_MEMORY)
  {
    GplErrSetError( (LONG)PMERR_DEV_FUNC_NOT_INSTALLED );
    ExitDriver( pddc );
    return( GPI_ERROR );
  }

  /*
  ** Make sure the current font's resources are loaded
  */
  if (!pddc->pddcb->cgs.fValidFont)
  {
    if (!prda_CheckFontResource( pddc )) /* couldn't get resources*/
    {
      ExitDriver( pddc );
      return( FAILURE );      /* Minor function saves error code*/
    }

    /*
    **        ps_selectfont(pddc );   **utlps.c
    */
  }

  /*
  ** Copy a maximum of sizeof(FONTMETRICS) bytes
  */
  ulNumBytes = min(ulNumBytes, (ULONG) sizeof(FONTMETRICS) );

  /*
  ** Convert the index of the current font into a "magic cookie" which
  ** we will return in the lMatch field of the font metrics, so that the
  ** caller will be able to identify this font in the future.
  */

  lMatch = OffsetToCookie( (LONG) pddc->pddcb->text.usFont );

  /*
  ** Copy metrics to the caller's buffer, converting to world coordinates
  */
  prda_CopyMetrics( pfmFontMetrics, pddc->pddcb->text.pfmFontMetrics,
                    ulNumBytes, lMatch, pddc );
/* DBCS enabling start */                                               //@DBCS
  /*
  ** If the font is a DBCS font, we have to correct the metrics for DBCS.
  */
  if ((pfmFontMetrics->fsType & FM_TYPE_MBCS) &&
      (pfmFontMetrics->fsType & FM_TYPE_FIXED))
  {
/// pfmFontMetrics->sXDeviceRes       =                            //@V3.1147608
    pfmFontMetrics->sXDeviceRes       = pddc->pdv->canvas.Res.x;   //@V3.1147608
/// pfmFontMetrics->sYDeviceRes       = pddc->pdv->canvas.iRes;    //@V3.1147608
    pfmFontMetrics->sYDeviceRes       = pddc->pdv->canvas.Res.y;   //@V3.1147608
    pfmFontMetrics->usCodePage        = pddc->pddcb->text.usDefCodePg;
    pfmFontMetrics->lXHeight          = pfmFontMetrics->lEmHeight/2L;
    pfmFontMetrics->lMaxAscender      = pfmFontMetrics->lEmHeight;
    pfmFontMetrics->lMaxDescender     = 0L;
    pfmFontMetrics->lLowerCaseDescent = 0L;
    pfmFontMetrics->lInternalLeading  = 0L;
    pfmFontMetrics->lExternalLeading  = 0L;
    pfmFontMetrics->lAveCharWidth     = pfmFontMetrics->lMaxCharInc;
  }
/* DBCS enabling end   */                                               //@DBCS
  pfmFontMetrics->sKerningPairs = (SHORT) prda_CountKernPairs(pddc );
  PrintLog( (PSZ) "QueryFontAttributes: sKerningPairs = %d\n",
            pddc->pddcb->text.pfmFontMetrics->sKerningPairs );
  PrintLog( (PSZ) "Setting lMatch to %ld\n", lMatch );
  ExitDriver( pddc );
  return( SUCCESS );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_DeviceQueryFonts
 *
 * DESCRIPTION   =
 * Copies the FONTMETRICS data for the requested font to the caller's buffer.
 * If the given facename is null, then data for all fonts will be returned,
 * buffer size permitting.
 *
 * Effects: On return, pulNumFonts will have been changed from the maximum
 *          number of fonts the caller would like returned to the actual
 *          number of fonts returned in the caller's buffer.  This will
 *          either be zero (an invalid font facename was given), one (a
 *          legitimate facename was given), or a larger number.  The latter
 *          case would occur if the caller gave a null font facename.  In
 *          that situation as many fonts would be returned as could be fit
 *          into the caller's buffer.
 *
 * Warnings: A request for more bytes of information than are contained in
 *           a FONTMETRICS structure will return an error.
 *
 *           rest with zeros.
 *
 *
 *           The driver presently does not distinguish between public and
 *           private fonts, so the options parameter is ignored.
 *
 * INPUT  =  hdc;               Display context handle
 *           ulOptions;         Options flags -- presently ignored
 *           pszFaceName;       Font to return FONTMETRICS data for
 *           pfmFontMetrics;    Pointer to buffer where data shall be placed
 *           ulNumBytes;        How many bytes of FONTMETRICS data to return
 *                              for each matching font
 *           pulNumFonts;       The maximum number of fonts to be returned
 *           pddc;              Pointer to display context instance data
 *           FunN;              Engine function number                          = NONE
 *
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =  The total number of fonts matching the requested
 *                  face name, but not returned because the buffer was
 *                  not big enough.
 *
 * RETURN-ERROR  =  -1
 *
 ****************************************************************************/

LONG prda_DeviceQueryFonts( HDC hdc, ULONG ulOptions, PSZ pszFaceName,
                            PFONTMETRICS pfmFontMetrics, ULONG ulNumBytes,
                            PULONG pulNumFonts, PDDC pddc, ULONG FunN)
{
  PFONTMETRICS pfmSrcFont;
  LONG         lMatch;
  PFNT         pFnt;
  PFONTDEF     pfdFont;
  SHORT        usStartFont, usHowManyFonts;
  SHORT        sIndex;
  ULONG        ulNumCopied;
  ULONG        ulOurFMsize;  
  register     SHORT i;      

  EnterDriver( pddc );

  #if      XDEBUG
    LogCall( "prda_DeviceQueryFonts(%08lx, %lp, %lp, %08lx)\n", ((PB)&hdc)+sizeof
            (hdc) );
    (pszFaceName) ? PrintLog( (PSZ) "FaceName = %ls\n", pszFaceName) :
                    PrintLog( (PSZ) "NULL Facename\n" );
  #endif  /* XDEBUG                                            */

  if (pddc->iType == OD_MEMORY)
  {
    /*
    ** Device fonts can't be realized for memory DC.
    */
    *pulNumFonts = 0 ;
    ExitDriver( pddc );
    return( 0 );
  }

  PrintLog( (PSZ) "ulNumBytes = %ld\n", ulNumBytes );
  PrintLog( (PSZ) "NumFonts = %ld\n", *pulNumFonts );

  /*
  ** The driver returns failure if its being asked for only
  ** private fonts because the driver doesn't have any private
  ** fonts.  All driver fonts are public fonts.
  */
  if ((ulOptions&QF_PRIVATE) && (!(ulOptions&QF_PUBLIC)))
  {
    ExitDriver( pddc );
    return( 0L );
  }

/*
** PTR OSDD SM10841 - the FONTMETRICS structure has increased by
** 8 bytes in 2.0 we just fill in what we know.
**
**
**     If number of bytes requested from each font's metrics is greater
**     than the size of a font metrics structure, we return an error.
**
**
**  if ( ulNumBytes > sizeof(FONTMETRICS) )
**  {
**      GplErrSetError (   PMERR_INV_LENGTH_OR_COUNT ) ;
**      return( ERROR_NEGONE) ;
**  }
*/

  ulOurFMsize = min(ulNumBytes, (ULONG)sizeof(FONTMETRICS) );

  /*
  ** If the facename is null, then every font is said to have matched.
  */
  if (pszFaceName == NULL)
  {
    usStartFont = 0;                                /* First font      */
    usHowManyFonts = prda_NumPrinterFonts(pddc );   /* Number of fonts */
    PrintLog( (PSZ) "NULL FaceName; Startfont = %d, HowMany = %d\n",
              usStartFont, usHowManyFonts );
  }

  /*
  ** Otherwise, attempt to match the facename to one of our fonts. This
  ** will result in either zero or one matches.
  */
  else
  {
    /*
    */
    if (0 > (sIndex = MatchFaceName(pszFaceName, pddc, FALSE ))
        || IsDynFnt( pddc, (SHORT) sIndex ) )                    
    {
      PrintLog( (PSZ) "Couldn't match facename %ls\n", pszFaceName );
      usStartFont = 0;
      usHowManyFonts = 0;
    }
    else                     /* Found a matching font                       */
    {
      usStartFont = (SHORT) sIndex;
      usHowManyFonts = 1;
      PrintLog( (PSZ) "%ls: Startfont = %d, HowMany = 1\n", pszFaceName,
         usStartFont );
    }
  }

  /*
  ** Now we know how many fonts have matched, and where the matched
  ** fonts begin in the font list. The number of fonts we will return is
  ** either the number that matched or the number the caller asked for,
  ** whichever is less. Once we know how many to return, loop through
  ** that many times, and then return the "extra" number.
  */
  ulNumCopied = min( (ULONG) usHowManyFonts, *pulNumFonts );
  PrintLog( (PSZ) "Caller asked for %ld fonts, and %d fonts actually ",
            *pulNumFonts, usHowManyFonts );
  PrintLog( (PSZ) "matched.\nSo we will return %ld fonts.\n", ulNumCopied );

  while (usHowManyFonts && *pulNumFonts)
  {
    /*
    ** Load a font's resource so we can get at its metrics
    */
    pFnt = PfntFromIndex( pddc, usStartFont );

    
    if ( IsDynFnt( pddc, usStartFont ) )
    {
      /*
      ** Temporary (may be Not)
      ** Skip all Dynamic fonts
      ** Anyway, prda_NumPrinterFonts(pddc ) don't count Dynamic fonts,
      ** because of pFnt->Resource = 0
      */
    }
    else
    {
    

      /*      ** Want to avoid duplicate font names.  Dup OFM fonts will be marked with
      ** with -1s so we ignore them
      */
      if ( (SHORT) pFnt->usResource >= 0 )
      {
        pfdFont = prda_LoadFontResource(pddc, usStartFont );

        if (!pfdFont)            /* Couldn't get resource                       */
        {
          ExitDriver( pddc );
          return( ERROR_NEGONE ); /* Minor function saves error code             */
        }
        pfmSrcFont = GetPFMetrics(pfdFont );

        /*
        ** Convert the index of the font into a "magic cookie" which
        ** we will return in the lMatch field of the font metrics, so that
        ** the caller can identify this font in the future.
        */
        lMatch = OffsetToCookie((LONG)usStartFont );

        /*
        ** Copy metrics to the caller's buffer, converting to world
        ** coordinates. Then bump pointers for the next iteration.
        **
        **    PTR OSDD SM10841 zero out the unused part
        */
        /*
        ** Copy metrics to the caller's buffer, converting to world
        ** coordinates. Then bump pointers for the next iteration.
        */
        for (i = sizeof(FONTMETRICS ); i < (SHORT) ulNumBytes; i++)
        {
          *((PB)pfmFontMetrics+i) = 0;
        }
        prda_CopyMetrics( pfmFontMetrics, pfmSrcFont, ulOurFMsize, lMatch, pddc );
        FreeFontResource(pddc->pdv, pfdFont, pFnt);

  /* DBCS enabling start */                                               //@DBCS
        /*
        ** If the font is a DBCS font, we have to correct the metrics for DBCS.
        */
        if ((pfmFontMetrics->fsType & FM_TYPE_MBCS) &&
            (pfmFontMetrics->fsType & FM_TYPE_FIXED))
        {
  /////// pfmFontMetrics->sXDeviceRes       =                          //@V3.1147608
          pfmFontMetrics->sXDeviceRes       = pddc->pdv->canvas.Res.x; //@V3.1147608
  /////// pfmFontMetrics->sYDeviceRes       = pddc->pdv->canvas.iRes;  //@V3.1147608
          pfmFontMetrics->sYDeviceRes       = pddc->pdv->canvas.Res.y; //@V3.1147608
          pfmFontMetrics->usCodePage        = pddc->pddcb->text.usDefCodePg;
          pfmFontMetrics->lXHeight          = pfmFontMetrics->lEmHeight/2L;
          pfmFontMetrics->lMaxAscender      = pfmFontMetrics->lEmHeight;
          pfmFontMetrics->lMaxDescender     = 0L;
          pfmFontMetrics->lLowerCaseDescent = 0L;
          pfmFontMetrics->lInternalLeading  = 0L;
          pfmFontMetrics->lExternalLeading  = 0L;
          pfmFontMetrics->lAveCharWidth     = pfmFontMetrics->lMaxCharInc;
        }
  /* DBCS enabling end   */                                               //@DBCS

        PrintLog( (PSZ) "QueryFonts: pfm->szFaceName = %ls\n", (PSZ)
                  pfmFontMetrics->szFacename );
        PrintLog( (PSZ) "QueryFonts: lMatch = %ld\n", lMatch );
        usHowManyFonts--;
        (*pulNumFonts)--;
        pfmFontMetrics = (PFONTMETRICS)(((PB)pfmFontMetrics)+ulNumBytes );
      }
    }                                                    
    usStartFont++;
  }
  *pulNumFonts = ulNumCopied;
  PrintLog( (PSZ) "Extra fonts matched but not returned = %d\n", usHowManyFonts );
  ExitDriver( pddc );
  return( (LONG)usHowManyFonts );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_GetKerningTable
 *
 * DESCRIPTION   =
 * Copies kerning pairs from internal table to caller's buffer.
 *
 * Effects: Scans internal kerning table and copies entries one at a time
 *          to caller's buffer until we reach the end of the table or have
 *          copied the desired number of pairs.  The internal kerning table
 *          lists code points in the font's code page (usually extended
 *          code page 850).  The kerning amount is converted into world
 *          coordinates before being returned to the caller.
 *
 * Warnings: If the font supports extended code page 850 and the caller is
 *           using a code page that is a subset of this, it is possible
 *           for a kerning pair to reference a character that does not
 *           exist in the caller's code page.  Such kerning pairs will
 *           not be returned to the caller.  Therefore it is possible for
 *           the caller to get back fewer kerning pairs than were asked
 *           for, even if the caller asked for the exact number of kerning
 *           pairs offered by the font.
 *
 *           It would be logical for the kerning pairs to be returned to
 *           the caller in sorted order.  However, the specification does
 *           not mention this, so the caller cannot rely on kerning pairs
 *           being returned in any specific order.  For this reason, no
 *           attempt is made to return the kerning pairs in sorted order.
 *
 * INPUT  =  hdc;           Display context handle
 *           ulNumPairs;    Number of kerning pairs desired
 *           pkrnprDest;    Pointer to where pairs shall be stored   ;
 *           pddc;          Pointer to display context instance data ics;
 *           FunN;          Engine function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =  The number of Kerning Pairs.
 *
 * RETURN-ERROR  =  -1
 *
 ****************************************************************************/

LONG  prda_GetPairKerningTable( HDC hdc, ULONG ulNumPairs,
                                PKERNINGPAIRS pkrnprDest, PDDC pddc,
                                ULONG FunN )
{

  PKERNPAIRS pkrnprSrc;      /* Pointer to our kern table                   */
  PKERNINGPAIRS pkrnprTemp;
  PKERNINGPAIRS pkrnprTemp1;
  PB pbTemp;                 /* Temp used to find our kern table            */
  LONG lNumReturned = 0L;    /* How many pairs we actually retn             */
  SHORT sKernTableIndex = 0; /* Index in our kern table                     */
  SHORT sCodePoint1;         /* Kerning pair: first character               */
  SHORT sCodePoint2;         /* Kerning pair: second character              */
  SHORT sKernAmount;         /* Amount to kern between chars                */
  SHORT sTopElement;
  SHORT sCurElement;
  SHORT us1,us2;
  ULONG tl;

  EnterDriver(pddc);

  if (pddc->iType == OD_MEMORY)
  {
    ExitDriver( pddc );
    return( 0 );
  }

  /*
  ** save it for subsequent sorting
  */
  pkrnprTemp = pkrnprDest;

  /*
  ** if the engine wants to do vector font simulaitons, we must call
  ** back to the engine's default function here.
  */
  if (pddc->pddcb->text.fFontSimulation)
  {
    tl =  (LONG)(*pfnlGetPairKerningTable)( hdc, ulNumPairs, pkrnprDest, pddc, FunN );
    ExitDriver( pddc );
    return ( tl );
  }

  #if      XDEBUG
    LogCall( "prda_GetPairKerningTable (%08lx, %lp, %lp, %08lx\n)", ((PB)&hdc) +
             sizeof(hdc));
  #endif

  PrintLog( (PSZ)"Pairs requested = %ld\n", ulNumPairs );

  /*
  ** Make sure the font resource is loaded.
  */
  if (!pddc->pddcb->cgs.fValidFont)
  {
    /*
    ** Couldn't get resources
    */
    if (!prda_CheckFontResource( pddc ))
    {
      ExitDriver( pddc );
      return ( ERROR_NEGONE );    /* Minor function saves error code */
    }

    /*
    **        ps_selectfont(pddc);   **utlps.c
    */
  }

  PrintLog( (PSZ)"Number of kerning pairs available = %d\n",
            pddc->pddcb->text.pfmFontMetrics->sKerningPairs );

  /*
  ** Point to start of kerning table
  */
  pbTemp = (PB) pddc->pddcb->text.pfdfFontDef;
  pbTemp += pddc->pddcb->text.pfdfFontDef->Header.usKerningDataOffset;
  pkrnprSrc = (PKERNPAIRS) pbTemp;

  /*
  ** Convert kerning pairs one at a time into world coordinates and
  ** the caller's code page.  Return only the ones that will be valid
  ** in the caller's code page.  Loop until we have as many kerning
  ** pairs as the caller asked for, or we run out of pairs to examine.
  */
  while ((lNumReturned < ulNumPairs) && (sKernTableIndex <
          pddc->pddcb->text.pfmFontMetrics->sKerningPairs))
  {
    /*
    ** The code points listed in a font's kerning table are stored in
    ** the code page of the font (usually 850). Before a kerning pair
    ** may be returned to the caller, the code points must be
    ** translated into the caller's code page.  This is what the
    ** prda_BackMapChar minor function does.
    */

    sCodePoint1 = prda_BackMapChar( pkrnprSrc[sKernTableIndex].g1, pddc );
    sCodePoint2 = prda_BackMapChar( pkrnprSrc[sKernTableIndex].g2, pddc );
    sKernAmount = (SHORT) FxToLong( frmul( pddc->pddcb->text.fxXScale, LongToFx
                          ((LONG)pkrnprSrc[sKernTableIndex].KernUnits)));

    if (!((sCodePoint1 == -1) || (sCodePoint2 == -1)))
    {
      pkrnprDest->sFirstChar = sCodePoint1;
      pkrnprDest->sSecondChar = sCodePoint2;
      pkrnprDest->lKerningAmount = sKernAmount;
      PrintLog((PSZ) "Kerning data[%d] = \042%c\042, \042%c\042, %d\n",
               sKernTableIndex, pkrnprDest->sFirstChar, pkrnprDest->sSecondChar,
               pkrnprDest->lKerningAmount);
      pkrnprDest++;
      lNumReturned++;
    }
    sKernTableIndex++;
  }

  /*
  **  Sort the returned kern pair table in ascending order of sFirstChar as the
  **  major key and sSecondChar as the minor key. Since sorting time is not
  **  important we are using the most inefficient algorithm on earth. Pick up
  **  the lightest and place it at its right place.
  */
  for (sTopElement = 0; sTopElement < (SHORT)lNumReturned; sTopElement++)
  {
    us1 = pkrnprTemp->sFirstChar;
    us2 = pkrnprTemp->sSecondChar;
    pkrnprDest = pkrnprTemp;
    pkrnprTemp1 = pkrnprTemp;

    for (sCurElement = sTopElement + 1; sCurElement < (SHORT) lNumReturned;
       sCurElement++)
    {
      pkrnprDest++;

      if (us1 > pkrnprDest->sFirstChar)
      {
        pkrnprTemp1 = pkrnprDest;
        us1 = pkrnprDest->sFirstChar;
        us2 = pkrnprDest->sSecondChar;
      }
      else if (us1 == pkrnprDest->sFirstChar)
      {
        if (us2 > pkrnprDest->sSecondChar)
        {
          pkrnprTemp1 = pkrnprDest;
          us1 = pkrnprDest->sFirstChar;
          us2 = pkrnprDest->sSecondChar;
        }
      }
    }

    if (pkrnprTemp1 != pkrnprTemp)/* the element at begin is not lightest
                                so replace it with the lightest             */
    {
      sCodePoint1 = pkrnprTemp1->sFirstChar;
      sCodePoint2 = pkrnprTemp1->sSecondChar;
      sKernAmount = pkrnprTemp1->lKerningAmount;
      pkrnprTemp1->sFirstChar = pkrnprTemp->sFirstChar;
      pkrnprTemp1->sSecondChar = pkrnprTemp->sSecondChar;
      pkrnprTemp1->lKerningAmount = pkrnprTemp->lKerningAmount;
      pkrnprTemp->sFirstChar = sCodePoint1;
      pkrnprTemp->sSecondChar = sCodePoint2;
      pkrnprTemp->lKerningAmount = sKernAmount;
    }
    pkrnprTemp++;
  }
  PrintLog( (PSZ)"Pairs returned = %ld\n", lNumReturned );
  ExitDriver( pddc );
  return ( lNumReturned );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_QueryWidthTable
 *
 * DESCRIPTION   =
 * Returns the widths of characters in the currently selected font.
 *
 * Effects: Copies a portion of the currently selected font's character
 *          width table into the caller's buffer, converting into world
 *          coordinates and translating points into the caller's code page.
 *
 * Warnings: The width of the default character will be returned in place
 *           of any invalid character.  An invalid character is one with a
 *           code point less than zero, greater than 255, or outside the
 *           range supported by the font.
 *
 *
 *
 * INPUT  =  hdc                 Display context handle
 *           lFirstChar          First character whose width is desired
 *           lSize               Number of character widths desired
 *           pulWidthTable       Buffer to hold the character widths
 *           pddc                Pointer to display context instance data
 *           FunN                Engine function number
 *
 * OUTPUT        = NONE
 *
 *
 *
 *
 * RETURN-NORMAL =  SUCCESS
 *
 * RETURN-ERROR  =  FAILURE
 *
 *
 *
 ****************************************************************************/

ULONG prda_QueryWidthTable( HDC hdc, LONG lFirstChar, LONG lSize,
                            PULONG pulWidthTable, PDDC pddc,
                            ULONG FunN)
  /*
  ** HDC     hdc
  ** LONG    lFirstChar            First char whose width is desired
  ** LONG    lSize                 Number of character widths desired
  ** PULONG  pulWidthTable         Buffer to hold character widths
  ** PDDC    pddc
  ** ULONG   FunN
  */
{
  register SHORT i;
  PB             pbTemp;
  PCHARWIDTH     pcdWidths;       /* Pointer to top of width table        */
  SHORT          sFirst;          /* First char offered by this font      */
  SHORT          sLast;           /* Last char offered by this font       */
  LONG           lScaledWidth;    /* Width of char in world coordinates   */
//SHORT          sCurrentChar;    /* Current char to return width for     *///@DBCS
  SHORT          sRemappedChar;   /* Same, after all remapping is done    */
  PCHARWIDTH     pcdCurWidth;     /* Width of current char of interest    */
  ULONG          tl;
/* DBCS enabling start */                                               //@DBCS
  USHORT         sCurrentChar;
  BOOL           fDBCS;
  CHARWIDTH     CharWidth;

  fDBCS = (pddc->pddcb->text.ubDBCSVector[0]) ? TRUE : FALSE;
/* DBCS enabling end   */                                               //@DBCS

  EnterDriver( pddc );

  if (pddc->iType == OD_MEMORY)
  {
     ExitDriver( pddc );
     return( FAILURE );
  }

  /*
  ** if the engine wants to do vector font simulaitons, we must call
  ** back to the engine's default function here.
  */
  if (pddc->pddcb->text.fFontSimulation)
  {
    tl = (ULONG) (*pfnlQueryWidthTable)( hdc, lFirstChar, lSize,
                                         pulWidthTable, pddc, FunN );
    ExitDriver( pddc );
    return( tl );
  }

  #if      XDEBUG
    LogCall( "prda_QueryWidthTable(%08lx, %lp, %lp, %lp, %08lx\n", ((PB)&hdc) +
             sizeof(hdc));
  #endif
  PrintLog( (PSZ)"Requested first codepoint (in current code page) = %ld\n",
            lFirstChar );
  PrintLog( (PSZ)"Widths requested = %ld\n", lSize );

  /*
  ** Confirm that the number of widths requested isn't less than zero
  */
  if ((lSize < 0L) || (lSize > 256L))
  {
    PrintLog( (PSZ) "--? Number of char width requested < 0\n" );
    GplErrSetError(  PMERR_INV_LENGTH_OR_COUNT );
    ExitDriver( pddc );
    return( FAILURE );
  }

/* DBCS enabling start */                                               //@DBCS
  /*
  ** The code page is for SBCS.
  */
  if (!fDBCS)
  {
/* DBCS enabling end   */                                               //@DBCS
    if (lFirstChar < 0L || lFirstChar > 255L)
    {
      PrintLog( (PSZ)"First char requested < 0 || > 255\n" );
      GplErrSetError(  PMERR_INV_FIRST_CHAR );
      ExitDriver( pddc );
      return( FAILURE );
    }
/* DBCS enabling start */                                               //@DBCS
  }
  /*
  ** When the code page is for DBCS, we have to check the range of
  ** the first char.
  */
  else
  {
    if (lFirstChar < 0L || lFirstChar > 65535L)
    {
      PrintLog( (PSZ)"First char is out of range\n" );
      GplErrSetError(  PMERR_INV_FIRST_CHAR );
      ExitDriver( pddc );
      return ( FAILURE );
    }
  }
/* DBCS enabling end   */                                               //@DBCS

  /*
  ** Prepare for mapping characters between the code pages
  */
  sFirst = pddc->pddcb->text.pfmFontMetrics->sFirstChar;
  sLast = pddc->pddcb->text.pfmFontMetrics->sLastChar;
//sCurrentChar = (SHORT) lFirstChar;                                    //@DBCS
  sCurrentChar = (USHORT)lFirstChar;                                    //@DBCS

  /*
  ** Make sure the font resource is loaded.
  */
  if (!pddc->pddcb->cgs.fValidFont)
  {
    /*
    ** Couldn't get resources.
    */
    if (!prda_CheckFontResource(pddc))
    {
      ExitDriver( pddc );
      return( FAILURE );      /* Minor function saves error code */
    }

    /*
    **        ps_selectfont(pddc);   **utlps.c
    */
  }

  /*
  ** Now we want to copy the width data into the caller's buffer.
  ** The character metrics of the resource file are arranged as follows:
  **
  ** For each character:
  **
  **    0000  byte    Character code for PostScript show command
  **    0001  word    Character width vector ("WX" arguments)
  **
  **
  ** Point to start of char metrics data
  */
  pbTemp = (PB) pddc->pddcb->text.pfdfFontDef;
  pbTemp += pddc->pddcb->text.pfdfFontDef->Header.usCharMetricsOffset;
  pcdWidths = (PCHARWIDTH) pbTemp;

/* DBCS enabling start */                                               //@DBCS
  /*
  ** When the current code page is for DBCS.
  */
  if (fDBCS)
  {
    /*
    ** A DBCS char must be greater than 256.
    */
    if (sCurrentChar < 256)
    {
      /*
      ** The first byte is in the range of DBCS char.
      */
      if (IsDBCSChar( pddc, sCurrentChar ))
      {
        PrintLog( (PSZ)"First char is invalid\n" );
        ExitDriver( pddc );
        return ( FAILURE );
      }
    }
    else
    {
      sRemappedChar = sFirst + pddc->pddcb->text.pfmFontMetrics->sDefaultChar;
      pcdCurWidth = pcdWidths + (sRemappedChar - sFirst);
      lScaledWidth = FxToLong( frmul( LongToFx( (LONG)(pcdCurWidth->wx * 2) ),
                                      pddc->pddcb->text.fxXScale ) );
      PrintLog( (PSZ)"(scaled width = %ld)\n", lScaledWidth );

      /*
      ** The DBCS char is fixed pitch, then the char width doesn't change.
      */
      for (i = 0; i < (SHORT)lSize; i++)
      {
        *pulWidthTable++ = lScaledWidth;
      }
      ExitDriver( pddc );
      return ( SUCCESS );
    }
  }
/* DBCS enabling end   */                                               //@DBCS

  for (i = 0; i < (SHORT)lSize; i++)
  {
    /*
    ** The character width table is stored in the font resource in
    ** extended code page 850 order if the font offers winthorn multi-
    ** codepage support. In this case we must reorder the width table
    ** so that we return character widths to the caller in their code
    ** page. This involves cycling through the caller's code points,
    ** remapping them one at a time. All of the caller's code points
    ** must be in the range of 0-255; any outside this range will be
    ** mapped to the default character.
    */
    // @V3.1140397 - Use SHORT typecast.
    if (((SHORT) sCurrentChar < 0) || ((SHORT) sCurrentChar > 255))
    {
      sRemappedChar = sFirst+pddc->pddcb->text.pfmFontMetrics->sDefaultChar;
    }
    else
    {
/* DBCS enabling start */                                               //@DBCS
      /*
      ** The character enter the range of DBCS char.
      */
      if (IsDBCSChar( pddc, sCurrentChar ))
      {
        PrintLog( (PSZ)"The character don't have to enter of the DBCS range\n" );
        ExitDriver( pddc );
        return ( FAILURE );
      }
/* DBCS enabling end   */                                               //@DBCS

      if (pddc->pddcb->text.pusCurCodePgVector)
      {
        sRemappedChar = (SHORT)*(pddc->pddcb->text.pusCurCodePgVector+
                                 sCurrentChar);
      }
      else
      {
        sRemappedChar = sCurrentChar;
      }

      if ((sRemappedChar < sFirst) || (sRemappedChar > (sFirst + sLast)))
      {
        sRemappedChar = sFirst+pddc->pddcb->text.pfmFontMetrics->sDefaultChar;
      }
    }

    pcdCurWidth = pcdWidths + (sRemappedChar - sFirst);

    // gpledc
    if ( sRemappedChar == EURO )
    {
      BOOL fRC;
      ABCSPACE abc;
      fRC = GplEDCGetABCSpace( pddc->hdc, pddc,
                              EDC_SRC_UGL | EDC_NOTRANSFORM, EURO, &abc );
      if ( fRC == TRUE )
      {
        CharWidth.glyph = EURO;
        CharWidth.wx    = abc.lA + abc.ulB + abc.lC;
        CharWidth.llx   = abc.lA;
        CharWidth.urx   = abc.lA + abc.ulB;
        pcdCurWidth = &CharWidth;
      }
    }

    /*
    ** Scale width and copy to caller's buffer
    */
    lScaledWidth = FxToLong( frmul( LongToFx( (LONG) pcdCurWidth->wx ),
                             pddc->pddcb->text.fxXScale ) );
    PrintLog( (PSZ) "Width of code point #%d is %d ", sCurrentChar,
                    pcdCurWidth->wx);
    PrintLog( (PSZ) "(scaled width = %ld)\n", lScaledWidth);
    *pulWidthTable++ = lScaledWidth;
    sCurrentChar++;
  }
  ExitDriver( pddc );
  return ( SUCCESS );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_SetCodePage
 *
 * DESCRIPTION   =
 * Sets the code page for characters written with the default font.
 *
 * Effects: Stores the requested code page number in the DDC.  If the
 *          default font is presently selected, then the current code
 *          page number is changed as well.
 *
 * Warnings: Note that this routine is only used to change the code page
 *           of the default font; a logical font's code page is selected
 *           at CreateLogicalFont() time, and cannot be changed.
 *
 *           SERIOUS WARNING: IF AN APP REQUESTS AN INVALID CODE PAGE,
 *           GREQUERYCODEPAGEVECTOR() WILL GENERATE A RIP INSTEAD OF
 *           RETURNING A NULL POINTER.  THIS IS EXTREMELY BAD !!
 *           (This is true as of the 1.1 maintenance release of PM,
 *           in 1.2 this has been fixed)
 *
 * INPUT  = hdc              Display context handle
 *          ulCodePage       Desired code page number
 *          pddc             Pointer to display context instance data
 *          FunN             Engine function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =  SUCCESS
 *
 * RETURN-ERROR  =  FAILURE
 *
 ****************************************************************************/

LONG prda_SetCodePage( HDC hdc, ULONG ulCodePage, PDDC pddc, ULONG FunN )
{
  PSHORT pusNewVector;      /* Pointer to remap vector for new code pg     */
  ULONG  ulOldCP;
  ULONG  ulRet;
  PCNFDATA pCNFData = pddc->pdv->pCNFData;                       // @V3.1142031

  EnterDriver( pddc );

  #if      XDEBUG
    LogCall( "prda_SetCodePage (%08lx, %lp, %08lx)\n", ((PB)&hdc)+sizeof(hdc) );
  #endif /* XDEBUG                                     */

  PrintLog( (PSZ) "Requested new code page = {%ld}\n", ulCodePage );

  if(pddc->iType == OD_MEMORY)
  {
     ulRet = InnerGreSetCodePage (pddc->hdcMemory, ulCodePage, FunN );
     ExitDriver( pddc );
     return( ulRet );
  }

  /*
  ** Get mapping vector for the requested code page from engine
  */
  pusNewVector = (PSHORT) GreQueryCodePageVector(ulCodePage );

  if (pusNewVector == NULL)
  {
    RIP("SetCodePage: QueryCodePageVector returned NULL pointer" );
    GplErrSetError(  PMERR_INV_CODEPAGE );
    ExitDriver( pddc );
    return( FAILURE );
  }

  /*
  ** If we are currently using the default font, we must also change the
  ** current code page right here and now.
  */
  if (!pddc->pddcb->text.bLogFontEntry)/* 0 = default font is in use now    */
  {
    pddc->pddcb->text.pusCurCodePgVector = pusNewVector;
    PrintLog( (PSZ) "Changed default as well as current code page\n" );
  }

  /*
  ** if the default font is being given a new code page here, then
  ** download the new code page mapping.  make sure we redefine
  ** the font after the new mapping has been saved in the pddc.
  ** ps_DownloadFont relies on the information store in the pddc.
  */
  ulOldCP = (ULONG)pddc->pddcb->text.usDefCodePg;

  if (ulOldCP != ulCodePage)
  {
    /*
    ** Save the new code page in the DC instance data
    */
    pddc->pddcb->text.usDefCodePg = (SHORT) ulCodePage;
    pddc->pddcb->text.pusDefCodePgVector = pusNewVector;

    /*
    ** SetCodePage is supposed to affect only the default font.
    ** call remapcodepage to remap the default font to the new
    ** codepage.
    */
/* DBCS enabling start */                                               //@DBCS
    /* If the code page is not for DBCS and the font is not DBCS */
    if (!pddc->pddcb->text.ubDBCSVector[0] && !IsDBCSFont( pddc ))
    {
/* DBCS enabling end   */                                               //@DBCS
    // @V3.1142031
//      if (!RemapCodePage(pddc, (SHORT) ulCodePage, 0, (PSZ) pddc->pdv->szDefFont) )
      if (!RemapCodePage( pddc, (SHORT) ulCodePage, 0,
                          (PSZ) pCNFData->szFontName ) )
      {
        ExitDriver( pddc );
        return( FAILURE );
      }
      SetFontRemap( 0 );
    }                                                                   //@DBCS
    ps_SetDownloadedFont(pddc );
  }

  /*
  ** set the default font's CodePage
  */
  pddc->usCodePages[0] = (SHORT) ulCodePage;
  ExitDriver( pddc );
  return( SUCCESS );
}

/***************************************************************************
 *
 * FUNCTION NAME =  prda_GetCodePage
 *
 * DESCRIPTION   =
 * Returns the code page that will be used for characters written with
 * the default font.
 *
 * Warnings: Note that this routine returns the code page for the default
 *           font only; if the current font is a logical font, then the
 *           code page number returned here may not be the same as the
 *           code page number currently in use.
 *
 * INPUT  = hdc        Display context handle
 *          pddc       Pointer to display context instance data
 *          FunN       Engine function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =  Code page number of the default font.
 *
 * RETURN-ERROR  =  NONE
 *
 ****************************************************************************/

LONG prda_GetCodePage( HDC hdc,PDDC pddc,ULONG FunN )
{

  ULONG ulRet;

  EnterDriver( pddc );

  #if      XDEBUG
    LogCall( "prda_GetCodePage (%08lx, %lp, %08lx)\n", ((PB)&hdc)+sizeof(hdc) );
  #endif  /* XDEBUG                                            */

  if (pddc->iType == OD_MEMORY)
  {
    ulRet = InnerGreGetCodePage( pddc->hdcMemory, FunN );
  }
  else
  {
    ulRet = (ULONG) pddc->pddcb->text.usDefCodePg ;
  }

  ExitDriver( pddc );

  return( ulRet );
}


/***************************************************************************
 *
 * FUNCTION NAME =  RemapCodePage
 *
 * DESCRIPTION   =
 * Action: The driver calls this routine to download a new font
 *         definition to the Postscript printer.
 *
 * INPUT  = hdc        Display context handle
 *          pddc       Pointer to display context instance data
 *          FunN       Engine function number
 *
 * OUTPUT        = NONE
 *
 * RETURN-NORMAL =  TRUE - Success
 *
 * RETURN-ERROR  =  False - Failure
 *                  Error in getting necessary resource.
 *
 ****************************************************************************/

BOOL RemapCodePage( PDDC pddc,SHORT usCP,SHORT usFont,PSZ szFont )
{
  CHAR           szFontName[10];
  PVOID          pResource;
  SHORT          uErr,cCP;
  PFONTREMAPCODE pfrcData;
  PSZ            psz;
  PSHORT         pusCodePageVector;
  SHORT          i, j;
  PSHORT         pusOffset;
  SHORT          sLineLen;
  SHORT          sFirst;
  PBYTE          pbTemp;
  PCHARWIDTH     pcData;
  SHORT          sDefaultChar;
  SHORT          sLast;
  PFNT           pFnt;
  SHORT          usCPType;                                            
  CHAR           ch;
  PTXT_STATE     pText = &pddc->pddcb->text;                          //@GPLEDC
  SHORT          sEntry= pText->bLogFontEntry;                        //@GPLEDC
  SHORT          usCPNew[256];
  BOOL           bReDownLoadCP;

  /*
  ** if we have not yet have a startdoc,
  ** we don't want to do anything.  this is necessary for the following
  ** reasons.  if this is called before a startdoc has been done, nothing
  ** will be output to the output channel, but flags will have been set
  ** which will cause unruly behavior when trying to print characters.
  */
  if (pddc->pdv->fDocStarted == FALSE)
  {
    return( TRUE );
  }

  #ifdef   TESTING
    PrintChannel(pddc, (PSZ) "Entering RemapCodePage\n" );
  #endif                       /* TESTING                                     */

  /*  ** Make sure not neg ( from new frame code )
  */
  if ( (USHORT)usCP != NO_CODEPAGE )
    usCP = abs( usCP );

  /*
  ** Get current code page vector.  It had better not be NULL.
  */
  if (!(pusCodePageVector = (PSHORT) GreQueryCodePageVector((ULONG)usCP)))
 {
   /*
   **    if ( !(pusCodePageVector = pddc->pddcb->text.pusCurCodePgVector) )
   */
   RIP("RemapCodePage: NULL codepage vector passed in" );
   return( FALSE );
 }

  #ifdef   TESTING
    PrintChannel(pddc, (PSZ) "RemapCodePage: pusCodePageVector = %lx\n",
                  pusCodePageVector );
  #endif                       /* TESTING                                     */

  PrintLog( (PSZ) "Downloading font %ls\n", (PSZ) szFontName );
  PrintLog( (PSZ) "Loading remap data resource file fntremap.txt\n" );

  /*
  ** get our module handle so we can get the font remapping code,
  ** which is stored as a resource.
  */
  if ((uErr = DosGetResource(pscript_module, MYFREMAP, REMAP_COD, &pResource)) != 0)
  {
    RIP("RemapCodePage: DosGetResource failed" );
    return( FALSE );
  }

  /*
  ** we now have a pointer to our font remapping code.
  */
  pfrcData = (PFONTREMAPCODE)(pResource );

  /*
  ** if the font remapping header has not been downloaded to the
  ** printer, do it now.  this header is sent at most once per job.
  */
  if (!(pddc->pddcb->cgs.fFRedefDownld))
  {
    psz = (PSZ) pfrcData;
    psz += pfrcData->sOffsetRoutine;  /* Point to remap header text           */

    while (*psz)             /* Print text a line at a time                 */
    {
      PrintChannel(pddc, (PSZ) "%ls\n", (PSZ) psz );

      while (*psz)
      {
        psz++;
      }
      psz++;
    }
    pddc->pddcb->cgs.fFRedefDownld = TRUE;
  }

  /*
  ** Make sure the font is there.  This is trivial for a HW font,
  ** but a Soft font must be downloaded before we can redefine it.
  */
  FontDownload( pddc, (PSZ) szFont, FALSE );                             //@RES

  /*
  ** Create unique name for the new font we will download
  */
  GetMappedFontName( pddc, (PSZ) szFontName, sizeof(szFontName), usFont );

  /*  ** Moved the following section up to here so we can see if the font is
  ** a device or soft (OFM) font
  **
  ** If font is device font procede in the old syle remaping
  ** If font is OFM then use the new more complete remapping
  */

  /*
  **
  **   Make sure we have a font matrix loaded for the current font.
  **   In the following FOR loop, this statement cases a trap D:
  **      j = pddc->pddcb->text.pfmFontMetrics->sFirstChar   +
  **          pddc->pddcb->text.pfmFontMetrics->sDefaultChar ;
  **   The item "pfmFontMetrics" points to a block of memory, but this
  **   process does not own that memory - thus Trap D.
  */
  prda_CheckFontResource(pddc );

  i = MatchFaceName( szFont, pddc, pddc->pddcb->text.fUseSoftFont );
  if ( i < 0 )
  {
    RIP("RemapCodePage: MatchFaceName failed" );
    return( FALSE );
  }
  pFnt = (PFNT) &(*pddc->pdv->paFonts)[i];

  if( IsDynFnt( pddc, i ) )
  {
     // see if current Dynamic Font in question is TYPE1 font                  //@V1.0VV02
     if(!strcmp(pFnt->prDynFnt->rFontProfile.szFontType, FONT_TYPE_TYPE1))     //@V1.0VV02
     {                                                                         //@V1.0VV02
        // for TYPE1 font we must not create unicode mapping! we need plain    //@V1.0VV02
        // old CP_NOOFM                                                        //@V1.0VV02
        usCPType = CP_NOOFM;                                                   //@V1.0VV02
     }                                                                         //@V1.0VV02
     else                                                                      //@V1.0VV02
     {                                                                         //@V1.0VV02
        DynFntCharSet( pFnt->prDynFnt, DYNFNT_CS_SET_CP, usCP);
      
        usCPType = CP_UNICODE;
     }                                                                         //@V1.0VV02   
  }
  else
  {
    if ( pFnt->usResource <= 0 )
    {
      usCPType = CP_OFM;
    }
    else
    {
      usCPType = CP_NOOFM;
    }
  }

  sFirst = pddc->pddcb->text.pfmFontMetrics->sFirstChar;
  sLast  = pddc->pddcb->text.pfmFontMetrics->sLastChar;
  sDefaultChar =  pddc->pddcb->text.pfmFontMetrics->sDefaultChar;


  if ( usCPType == CP_OFM )   
  {
    pbTemp = (PBYTE)pddc->pddcb->text.pfdfFontDef;
    pbTemp += pddc->pddcb->text.pfdfFontDef->Header.usCharMetricsOffset;
  }

  /*
  ** we now need to look through our table to see if this codepage
  ** has been downloaded to the printer.  if it has, we just
  ** redefine the current font, using the proper codepage.
  */

  bReDownLoadCP = FALSE;

  if (CPDownLoaded(pddc, usCP, &cCP, usCPType ))
  {
    if( usCPType != CP_OFM )                    //OFM type can be other
    {
      goto redefine_cur_font;
    }
  }
  else  //Not downloaded
  {
    bReDownLoadCP = TRUE;

    /*
    ** the codepage mapping has not been sent down to the printer.
    ** add the current codepage to our table, then download the
    ** mapping.
    */
    cCP = pddc->pddcb->cgs.cCodePagesRemapped;
  }

    //CP Downloaded
    //Check if this CP OK for current font
    if( usCPType == CP_OFM )                    //Old style mapings are OK
    {
      for (i = 0; i < 256; i++)
      {
        j = *(pusCodePageVector + i );

        if ((j <= 0) || (j > sLast))
        {
          j = sFirst + sDefaultChar;
        }
        else    /* Make sure CP is defined */
        {
          pcData = (PCHARWIDTH)pbTemp;
          pcData += ( j - sFirst );  /* point to char's metrics  */

          if ( pcData->glyph == 32 )
          {
            j = sFirst + sDefaultChar;
          }
        }

        /*
        ** Check for the moved glyph euro or dotlessi
        ** In the old versions UniCode 213 = dotless1, 315 was acsiitilde
        ** In the new version          213 = Euro      315 = dotlessi
        */
        if ( j == PS_DOTLESSI_GLYPH ||
             j == PS_EURO_GLYPH      )
        {
          // This check needs to be only once per process
          if ( EuroEnable == eUNKNOWN )
          {
            ch = WinCpTranslateChar((HAB)0, 1252, 0x80, 850);
            if ( ch == EURO )
            {
              EuroEnable = eTRUE;
            }
            else
            {
              EuroEnable = eFALSE;
            }
          }

          // If EuroEnable is true then then use new mapping
          if ( EuroEnable == eTRUE )
          {
            if ( j == PS_DOTLESSI_GLYPH )  // Looking for Euro
            {
              j = PS_EURO_GLYPH;
            }
          }
        }

        usCPNew[i] = j;

        if ( ! bReDownLoadCP )
        {
          if ( pddc->pddcb->cgs.CPTable[cCP].usCPVector[i] !=j ) //CP should be redownloaded
          {
            bReDownLoadCP = TRUE;
          }
        }
      }
    }


  if( ! bReDownLoadCP )                    //skip redownloading
  {
    goto redefine_cur_font;
  }

  if( cCP == pddc->pddcb->cgs.cCodePagesRemapped ) //Wasn't downloaded
  {
    /*
    ** There is a limit of 10 codepages being defined at a time.  I am
    ** imposing this to help keep postscript printers from running
    ** out of memory.  once we are asked to download the 11th codepage
    ** it simply replaces the 10th.
    */
    if (cCP >= CP_LIMIT)
    {
      cCP = CP_LIMIT_MINUS_ONE;
    }
    pddc->pddcb->cgs.CPTable[cCP].usCP = usCP;
    pddc->pddcb->cgs.CPTable[cCP].usCPType = usCPType ;
    pddc->pddcb->cgs.cCodePagesRemapped = cCP + 1;
    /*
    ** count new remapped code page in the current document page.
    */
    ++pddc->pddcb->cgs.usNewCPs;
  }
  else          //Was downloaded but should be redownloaded
  {
    if( cCP < pddc->pddcb->cgs.cCodePagesRemapped - pddc->pddcb->cgs.usNewCPs )
    {
      //cCP should be redownloaded at next NewFrame
      pddc->pddcb->cgs.usNewCPs = pddc->pddcb->cgs.cCodePagesRemapped - cCP;
    }
  }

  /*
  ** End B724427
  */
  /*
  ** download the mapping for the given codepage.
  */
  PrintChannel( pddc, (PSZ) "/CP%d_%d [ 0 \n  ", usCP, usCPType);
  sLineLen = 2;

  for (i = 0; i < 256; i++)
  {
    /*
    ** If the line is too long, start a new one
    */
    if (sLineLen > 66)
    {
      PrintChannel( pddc, (PSZ) "\n  " );
      sLineLen = 2;
    }

    switch ( usCPType )
    {
      case CP_NOOFM:

        if (i < 0x11 || i > 0x1B || i == 0x14 || i == 0x15)
        {
          j = *(pusCodePageVector + i );

          if (i == 0xFF && j == 0xFF)                                     //@DBCS
            j = *(pusCodePageVector + 0x20) ;                             //@DBCS

          if ((j <= 0) || (j >= pfrcData->sNumGlyphNames))
          {
            j = sFirst + sDefaultChar;
          }
        }
        else
        {
          switch (i)
          {
          case 0x11:
               j = GRAVE;
               break;

          case 0x12:
               j = ACUTE;
               break;

          case 0x13:
               j = CIRCUMFLEX;
               break;

          case 0x16:
               j = TILDE;
               break;

          case 0x17:
               j = DIERESIS;
               break;

          case 0x18:
               j = RING;
               break;

          case 0x19:
               j = CEDILLA;
               break;

          case 0x1A:
               j = CARON;
               break;

          case 0x1B:
               j = DOTLESSI;
               break;
          }
        }

        /*
        ** Point to corresponding glyph name as known by PostScript and print
        */
        pusOffset = (PSHORT) &pfrcData->sOffsetpGlyph;
        psz = (PSZ) pfrcData;
        psz += *(pusOffset+j );

        PrintChannel(pddc, (PSZ) "/%ls", (PSZ) psz );

        /*
        ** Update the line length
        */
        sLineLen +=  strlen(psz) + 1;

        break;

      case CP_OFM:

        psz = (PSZ)PMGlyphName[usCPNew[i]];
        pddc->pddcb->cgs.CPTable[cCP].usCPVector[i] = usCPNew[i];

        PrintChannel(pddc, (PSZ) "/%ls", (PSZ) psz );

        /*
        ** Update the line length
        */
        sLineLen +=  strlen(psz) + 1;

        break;

      case CP_UNICODE:
        pddc->pddcb->cgs.CPTable[cCP].usCPVector[i]=pFnt->prDynFnt->CharSet.usCharSet[i];
        PrintChannel(pddc, (PSZ) "/U%04x ", pddc->pddcb->cgs.CPTable[cCP].usCPVector[i]);
        sLineLen += 9;

        break;
     }
  }
  PrintChannel( pddc, (PSZ) "\n] def\n" );


  /*
  ** redefine the current font using the codepage mapping we
  ** just downloaded to the printer.
  */

redefine_cur_font:

  /*
  ** output the postscript commands to redefine the current font
  ** using the current codepage mapping.
  */
  PrintChannel( pddc, (PSZ) "CP%d_%d /%ls /%ls redefine\n", usCP, usCPType, (PSZ) szFontName,
                (PSZ) FullToFontName((PSZ) szFont, pddc) );
  szCopy( pddc->pddcb->cgs.szFont, (PSZ) (szFontName), sizeof
          (pddc->pddcb->cgs.szFont) );

  // GPLEDC will need to remap after code page change
  if ( (pText->lfiEntry[sEntry].usFlags & LFONT_HAS_BACKUP) != 0 )
  {
    if ( GplEDCSetFlags( pddc, 0, EDC_ACTIVE, QUERY ) == TRUE )
    {
      GplEDCSetFlags( pddc, 0, EDC_NEED_REMAP, SET );
    }
  }

  #ifdef   TESTING
    PrintChannel(pddc, (PSZ) "Leaving ps_RemapCodePage\n" );
  #endif                       /* TESTING                                     */

  return( TRUE );
}

/***************************************************************************
 *
 * FUNCTION NAME =  CPDownLoaded
 *
 * DESCRIPTION   =
 * This routine returns TRUE if the passed in codepage has been.
 * downloaded to the printer.  Otherwise, it returns FALSE.
 *
 * INPUT  = pddc       Pointer to display context instance data
 *          usCP       Code page
 *
 * OUTPUT =
 *          pusCPTable Returns CP number in CPTable
 *
 * RETURN-NORMAL =  TRUE - Success
 *
 * RETURN-ERROR  =  False - Failure
 *
 ****************************************************************************/

BOOL CPDownLoaded( PDDC pddc, SHORT usCP, PSHORT pusCPTable, USHORT usCPType)
{
  SHORT i, cCP;

  /*
  ** bag out if no codepages have been downloaded.
  */
  cCP = pddc->pddcb->cgs.cCodePagesRemapped;

  if ( !cCP || !usCPType)
  {
    return( FALSE );
  }

  /*
  ** some codepages have been remapped.  we need to look through
  ** our table to see if the current one has been downloaded.
  */
  for (i = 0; i < cCP; i++)
  {
    if (pddc->pddcb->cgs.CPTable[i].usCP == usCP &&
        pddc->pddcb->cgs.CPTable[i].usCPType == usCPType)
    {
      if ( pusCPTable )
      {
        *pusCPTable = i;          //Returns CP number in CPTable
      }
      return( TRUE );
    }
  }

  /*
  ** the current code page was not found in the table.
  */

  return( FALSE );
}




/* DBCS enabling start */                                               //@DBCS
/***************************************************************************
 *
 * FUNCTION NAME =  IsBoldItalic
 *
 * DESCRIPTION   =  Check whether the font is bold or italic.
 *
 * INPUT         =  PDDC    pddc     Pointer to display context instance data
 *                  PFATTRS pfa      Pointer to font attributes
 *                  SHORT   sIndex   Font index
 *
 * OUTPUT        =  NONE
 *
 * RETURN-NORMAL =  NONE
 *
 * RETURN-ERROR  =  NONE
 *
 ****************************************************************************/

VOID IsBoldItalic (PDDC pddc, PFATTRS pfa, SHORT sIndex)
{
  SHORT   i;
  FONT_INFO    FInfo;
  FONT_ID_NUM  FIDNum;

  // Search the font name in the font list

  FIDNum = FindFontIDNum( pddc, pfa->szFacename, (pfa->lMatch > 0L) );

  // Not found - return
  if ( FIDNum == FONT_NOT_FOUND )
  {
    return;
  }

  // Expand
  FInfo.iFamily = HIUSHORT( FIDNum );
  FInfo.iType = LOUSHORT( FIDNum );

  // Switch on type - then or in the right attribute bit(s)

  switch( FInfo.iType )
  {
  case BOLDITALIC:
    pddc->pddcb->text.lfiEntry[sIndex].usFlags |=
                             (LFONT_HAS_ITALIC + LFONT_HAS_BOLD);
    break;

  case BOLD:
    pddc->pddcb->text.lfiEntry[sIndex].usFlags |= LFONT_HAS_BOLD;
    break;

  case ITALIC:
    pddc->pddcb->text.lfiEntry[sIndex].usFlags |= LFONT_HAS_ITALIC;
    break;
  }

#if 0
///* Symbol and ZapfDingbats fonts are not italic or bold */
///* ZapfChanceryMediumItalic is not bold                 */
///* Sergejs Roze 1998                                    */
//if ((i == SYMBOL) || (i == ZAPFDINGBATS) || (i == ZAPFCHANCERY_MEDIUMITALIC))
//  return;
#endif
  return;
}


/***************************************************************************
 *
 * FUNCTION NAME =  CheckDispatchGothic
 *
 * DESCRIPTION   =  Check whether the font is what has to display DBCS
 *                  by gothic font or not.
 *
 * INPUT         =  PDDC    pddc     Pointer to display context instance data
 *
 * OUTPUT        =  SHORT            TURE  : Display Japanese by gothic font.
 *                                   FALSE : Not.
 *
 * RETURN-NORMAL =  NONE
 *
 * RETURN-ERROR  =  NONE
 *
 ****************************************************************************/

SHORT CheckDispatchGothic (PDDC pddc)
{
//SHORT    i;
  FONT_INFO    FInfo;
  FONT_ID_NUM  FIDNum;

  /* Search the font name in the font name list */
  FIDNum = FindFontIDNum( pddc, (PSZ)pddc->pddcb->text.szFont, FALSE );

  // Not found - return
  if ( FIDNum == FONT_NOT_FOUND )
  {
    return FALSE;
  }

  // Expand
  FInfo.iFamily = HIUSHORT( FIDNum );
  FInfo.iType = LOUSHORT( FIDNum );

  /* Check the font is avant */
  if ( IS_AVANT_GARDE( FInfo.iFamily ) )
  {
    return TRUE;
  }

  /* Check the font is helvetica */
  if ( IS_HELVETICA( FInfo.iFamily ) )
  {
    return TRUE;
  }

  return FALSE;
}


/***************************************************************************
 *
 * FUNCTION NAME =  SetDispatchEuropean
 *
 * DESCRIPTION   =  Return the substitute for the european font.
 *
 * INPUT         =  PDDC    pddc        Pointer to display context instance data
 *                  PSZ     szSetFont   Save are for font name to change
 *                  SHORT   DBCSFont    Index of the DBCS font
 *                  SHORT   sFontType   0: Return with full name
 *                                      1: Return with face name
 *
 * OUTPUT        =  SHORT               Index of font
 *
 * RETURN-NORMAL =  NONE
 *
 * RETURN-ERROR  =  NONE
 *
 ****************************************************************************/

SHORT SetDispatchEuropean (PDDC pddc, PSZ szSetFont,
                           SHORT DBCSFont, SHORT sFontType)
{
  BYTE    bEntry;
  CHAR    szFont[256];
  USHORT  usFlags;
  FONT_INFO    FInfo;
  FONT_ID_NUM  FIDNum;

  #if XDEBUG
  PrintLog( (PSZ) "\nSetDispatchEuropean\n" );
  #endif
  /*
  ** If the current font is DBCS font, substitute European font.
  */
  if (DBCSFont != -1)
  {
    // Look in table to which font this maps to
    FIDNum = Dispatch[DBCSFont];

    // Expand
    FInfo.iFamily = HIUSHORT( FIDNum );
    FInfo.iType = LOUSHORT( FIDNum );

    bEntry = pddc->pddcb->text.bLogFontEntry;
    usFlags = pddc->pddcb->text.lfiEntry[bEntry].usFlags &
                                         ( LFONT_HAS_ITALIC | LFONT_HAS_BOLD );
    // Select the right attributes
    switch( usFlags )
    {
    case LFONT_HAS_ITALIC | LFONT_HAS_BOLD:
      FInfo.iType |= BOLDITALIC;
      break;

    case LFONT_HAS_ITALIC:
      FInfo.iType |= ITALIC;
      break;

    case LFONT_HAS_BOLD:
      FInfo.iType |= BOLD;
      break;

    default:
      FInfo.iType |= REGULAR;
      break;
    }
  }
  else
  {
    /* In the case of non DBCS font, check the font name */
    /* Search the font name in the font name list */
    FIDNum = FindFontIDNum( pddc, (PSZ)pddc->pddcb->text.szFont, FALSE );

    // Not found - return
    if ( FIDNum == FONT_NOT_FOUND )
    {
      return -1;
    }

    // Expand
    FInfo.iFamily = HIUSHORT( FIDNum );
    FInfo.iType = LOUSHORT( FIDNum );
  }

  // Copy in the right font
  strcpy( szFont, AllFonts[FInfo.iFamily][FInfo.iType] );

  if (sFontType)
    szCopy( (PSZ)szSetFont, FullToFontName ((PSZ)szFont, pddc), 256 );
  else
    szCopy( (PSZ)szSetFont, (PSZ)szFont, 256 );

  #if XDEBUG
  PrintLog( (PSZ) "SetDispatchEuropean : szSetFont = %ls\n", szSetFont );
  PrintLog( (PSZ) "SetDispatchEuropean : ret = %d\n", FInfo.iFamily );
  #endif

  return FInfo.iFamily;
}
/* DBCS enabling end   */                                               //@DBCS


/*****************************************************************************/
/******************** New Font/Resource Handling Functions *******************/
/*****************************************************************************/

/*****************************************************************************\
**
** FUNCTION: GetCurrentPageNum
**
** DESCRIPTION:
**  When spooling STD Devesc_newframe is not called so must query spooler to
**  get current page
**
** RETURNS:
**  page number
**
\*****************************************************************************/

LONG GetCurrentPageNum( PDDC pddc )
{
  PDV         pdv          = pddc->pdv;
  LONG        lCurrentPage = pdv->shPageno;
  PCN         pcn;
  PQMJOBINFO  pQMJobInfo;
  PRJINFO4    prjInfo4;
  ULONG       ulSizeNeeded;
  APIRET      ulRC;

  // If not standard then shPageNo is correct
  if ( pdv->usDataType != PM_Q_STD )
  {
    return lCurrentPage;
  }

  pcn = &(pdv->cn);

  // If JobInfo is null there is nothing to do
  if ( pcn->pQMJobInfo == NULL )
  {
    return lCurrentPage;    // Might as well try this
  }

  pQMJobInfo = pcn->pQMJobInfo;
  ulRC = SplQueryJob( pQMJobInfo->pszComputerName,
                      pQMJobInfo->pszQueueName,
                      pQMJobInfo->ulJobID,
                      4,
                      (PVOID)&prjInfo4,
                      sizeof (prjInfo4),
                      &ulSizeNeeded);

  if ( ulRC == NO_ERROR       ||
       ulRC == ERROR_MORE_DATA )
  {
    return prjInfo4.ulPagesSpooled + 1;
  }
  else
  {
    return lCurrentPage;
  }
}


/*****************************************************************************\
**
** FUNCTION: GetFontRB
**
** DESCRIPTION:
**  Will get the specified RB for font name and type.  If doesn't exist
**  will create one
**
\*****************************************************************************/

PFONTRESOURCEBLOCK GetFontRB( PDDC  pddc        ,  // Ptr to ddc
                              PSZ   pszFontName ,  // Ptr to text state
                              LONG  lType       )  // Type of font - Dev, DL ..
{
  PDV                          pdv   = pddc->pdv;
  PMASTERRESOURCECONTROLBLOCK  pMRCB = &pdv->MRCB;
  PRESOURCECONTROLBLOCK        pRCB;
  PFONTRESOURCEBLOCK           pFRB;
  INT                          i;
  ULONG                        ulpSave;

  // Look for the FontCB;
  pRCB =  (PRESOURCECONTROLBLOCK)pMRCB->ulpRCB;
  for ( i = 0; i < pMRCB->lCount; i++ )
  { // If this RCB is font quit loop
    ulpSave = (ULONG)pRCB; // Save last looked at
    if ( pRCB->header.ulType == FONT_CONTROL_BLOCK )
    {
      break;
    }
    // Point to next RCB
    pRCB = (PRESOURCECONTROLBLOCK)pRCB->header.ulpNext;
  }

  // No Font CB found, alloc a RCB and update pointers
  if ( pRCB == NULL      ||
       i >= pMRCB->lCount )
  {
    // First block so set up MRCB
    if ( i == 0 )
    {
      pMRCB->header.ulType = MASTER_RESOURCE_CONTROL_BLOCK;
      pMRCB->header.lSize  = sizeof( MASTERRESOURCECONTROLBLOCK );
      pMRCB->ulVersion     = V1_0;
      pMRCB->lTotalSize    = 0;
      pMRCB->lLastPageProcessed = 0;                              //@V4.1198538
      SETFLAG( pMRCB->ulFlags, RESOURCE_FREE_BLOCKS );
    }

    if ( ( pRCB = (PRESOURCECONTROLBLOCK)GplMemoryAlloc( pdv->pDCHeap,
                                   sizeof( RESOURCECONTROLBLOCK ) ) ) == NULL )
    {
      return NULL;
    }
    pMRCB->lTotalSize += sizeof( RESOURCECONTROLBLOCK );

    pRCB->header.ulType = FONT_CONTROL_BLOCK;
    pRCB->header.lSize = sizeof( RESOURCECONTROLBLOCK );

    if ( pMRCB->ulpRCB == NULL )
    { // List empty
      pMRCB->ulpRCB = (ULONG)pRCB;
    }
    else
    { // Insert into chain
      ((PRESOURCECONTROLBLOCK)ulpSave)->header.ulpNext = (ULONG)pRCB;
    }

    pMRCB->lCount++;  // Bump count
  }

  /* Search thru the font blocks to find matching name and type
  ** The same name can have several types - device courier and DL version
  */
  pFRB = (PFONTRESOURCEBLOCK)pRCB->ulpResource;
  for ( i = 0; i < pRCB->lCount; i++ )
  {
    ulpSave = (ULONG)pFRB;
    if ( pFRB->lFontType == lType                     &&
         strcmp( pszFontName, pFRB->achFontName ) == 0 )
    {
      break;
    }

    // To Next
    pFRB = (PFONTRESOURCEBLOCK)pFRB->header.ulpNext;
  }

  if ( pFRB == NULL     ||  // Nothing or past end
       i >= pRCB->lCount )
  {
    if ( ( pFRB = (PFONTRESOURCEBLOCK)GplMemoryAlloc( pdv->pDCHeap,
                                   sizeof( FONTRESOURCEBLOCK ) ) ) == NULL )
    {
      return NULL;
    }

    pMRCB->lTotalSize += sizeof( FONTRESOURCEBLOCK );

    pFRB->header.ulType = FONT_RESOURCE_BLOCK;
    pFRB->header.lSize = sizeof( FONTRESOURCEBLOCK );

    if ( pRCB->ulpResource == NULL )
    { // List empty
      pRCB->ulpResource = (ULONG)pFRB;
    }
    else
    { // Insert into chain
      ((PFONTRESOURCEBLOCK)ulpSave)->header.ulpNext = (ULONG)pFRB;
    }

    pRCB->lCount++;  // Bump count

    // Init the structure

    pFRB->ulUsage = 0; // First time
    pFRB->lStartPage = GetCurrentPageNum( pddc );
    pFRB->lFontType = lType;
    strcpy( pFRB->achFontName, pszFontName );
  }

  return pFRB;
}


/*****************************************************************************\
**
** FUNCTION: LogFontUsage
**
** DESCRIPTION:
*/
VOID LogFontUsage( PDDC pddc       ,  // PDDC pointer
                   LONG lCharCount )  // Number of characters in line
{
  PDV                 pdv      = pddc->pdv;
  TXT_STATE          *pText    = &pddc->pddcb->text;
  PFONTRESOURCEBLOCK  pFontRB;
  LONG                lType;
  LONG                lCurrentPage;                               //@V4.1198538

  // If the font data has been collected from another pass just return
  if ( CHECKFLAG( pdv->MRCB.ulFlags, RESOURCE_DATA_SET ) != 0 )
  {
    return;
  }

  //@V4.1198538
  // Get the current page - if it's greater than last page processed plus one
  // then new frame did not call ResProcessEndPage so do it now.
  // Reason     it's GE lLastPageProcessed plus one is current page is already
  // one ahead of last processed.

  lCurrentPage = GetCurrentPageNum( pddc );
  if ( lCurrentPage > pdv->MRCB.lLastPageProcessed + 1 )
  {
    ResProcessEndPage( pddc );
  }

  // If font simulation bit is on then this is an engine font
  if ( pText->fFontSimulation == TRUE )
  {
    lType = ENGINEFONT;
  }
  else  // If use soft font switch is set this is a download font
  {
    //@V4.1198538 use the flag value - not pText.fUseSoftFont entry
    BYTE bEntry = pText->bLogFontEntry;

    if ( pText->lfiEntry[ bEntry ].usFlags & LFONT_IS_SOFTFONT )
    {
      lType = DOWNLOAD;
    }
    else  // It is a device font
    {
      lType = DEVICEFONT;
    }
  }

  pFontRB = GetFontRB( pddc, pText->szFont, lType );

  if ( pFontRB == NULL )
  {
    return;
  }

  // Record if in a saved state.  If not char/line will need to send down
  // at top of page
  if ( pddc->usSavedDCs > 0 )
  {
    SETFLAG( pFontRB->ulFlags, IN_SAVE_STATE );
  }

  switch ( pFontRB->ulUsage )
  {
    case MULTI_PAGE: // Been here before, maxed out usage
      break;

    case 0: // New page
      // Only one char - could be singe till next call
      if ( lCharCount == 1 )
      {
        pFontRB->ulUsage = SINGLE_CHAR;
      }
      else
      {
        pFontRB->ulUsage = SINGLE_LINE;
      }
      break;

    case SINGLE_CHAR:
    case SINGLE_LINE:
    case SINGLE_PAGE:
    case TOP_OF_PAGE:
      // Single char and lines are single page if on same start page
      // multi if on more
      if ( pFontRB->lStartPage == lCurrentPage ) //use above value  @V4.1198538
      {
        if ( CHECKFLAG( pFontRB->ulFlags, IN_SAVE_STATE ) != 0 )
        {
          pFontRB->ulUsage = TOP_OF_PAGE;
        }
        else
        {
          pFontRB->ulUsage = SINGLE_PAGE;
        }
      }
      else
      {
        pFontRB->ulUsage = MULTI_PAGE;
      }
      break;

    default:
      pFontRB->ulUsage = MULTI_PAGE;
      break;
  }

  return;
}


/*****************************************************************************\
**
** FUNCTION: ProcessRawResources
**
** DESCRIPTION:
**  Will printout list of fonts at end of RAW document and free memory in list
**
\*****************************************************************************/

VOID ProcessRawResources( PDDC pddc )
{
  PDV                          pdv   = pddc->pdv;
  PMASTERRESOURCECONTROLBLOCK  pMRCB = &pdv->MRCB;
  PRESOURCECONTROLBLOCK        pRCB;
  PFONTRESOURCEBLOCK           pFRB;
  INT                          i;
  PBYTE                        pbRCBsave;
  PBYTE                        pbFRBsave;    // Hold old ptr
  BOOL                         fFreeBlock;
  BOOL                         fFreeBuf;

  //  @V4.1218347
  // If nothing, return
  if ( pMRCB->lCount == 0 )
  {
    return;
  }

  // Write out section header
  if ( pMRCB->lCount )
  {
    PrintChannel( pddc,"%%%%DocumentSuppliedResources:\n" );
  }

  // If the no_free flag is off then free blocks as processed
  fFreeBlock = ( CHECKFLAG( pMRCB->ulFlags, RESOURCE_FREE_BLOCKS ) != 0 ); //@V4.1218347
  fFreeBuf   = ( CHECKFLAG( pMRCB->ulFlags, RESOURCE_FREE_BUF    ) != 0 ); //@V4.1218347

  // For each resource control block process its resource blocks
  pRCB =  (PRESOURCECONTROLBLOCK)pMRCB->ulpRCB;
  for ( i = 0; i < pMRCB->lCount; i++ )
  {
    pbRCBsave = (PBYTE)pRCB;
    switch ( pRCB->header.ulType )
    {
      case FONT_CONTROL_BLOCK:
        // If there is some fonts write out font header
        pFRB = (PFONTRESOURCEBLOCK)pRCB->ulpResource;
        // Look at each RB
        for ( i = 0; i < pRCB->lCount; i++ )
        {
          pbFRBsave = (PBYTE)pFRB;
          // Only want download fonts to show
          if ( pFRB->lFontType == DOWNLOAD )
          {
            PrintChannel( pddc, "%%%%+ font %ls\n", pFRB->achFontName );
///DEBUG/// PrintChannel( pddc, "%%%%+ font %ls, %ld, %ld, %ld\n", pFRB->achFontName,
///DEBUG///               pFRB->ulUsage, pFRB->lStartPage, pFRB->lFontType );
          }

          // To Next
          pFRB = (PFONTRESOURCEBLOCK)pFRB->header.ulpNext;

          if ( fFreeBlock == TRUE )
          {
            GplMemoryFree( pbFRBsave );
          }
        }
        break;

      default:
        break;
    }

    // Point to next RCB
    pRCB = (PRESOURCECONTROLBLOCK)pRCB->header.ulpNext;

    if ( fFreeBuf == TRUE )                                       //@V4.1218347
    {
      GplMemoryFree( pbRCBsave );
    }
  }

  // Reset MCB to clear state
  pMRCB->header.lSize = sizeof(MASTERRESOURCECONTROLBLOCK);
  pMRCB->header.ulpNext = 0;
  pMRCB->lCount = 0;
  pMRCB->lTotalSize = 0;
  pMRCB->ulpRCB = 0;

  return;
}


/*****************************************************************************\
**
** FUNCTION: ProcessSTDResources
**
** DESCRIPTION:
**  The Font log is a group of non continous resource blocks.  In order to put
**  them in Job props they have to be continous. The old blocks are freed as
**  they are copied over
**
\*****************************************************************************/

VOID ProcessSTDResources( PDDC pddc )
{
  PDV                          pdv      = pddc->pdv;
  PCNFDATA                     NewpCNFData;
  PMASTERRESOURCECONTROLBLOCK  pMRCB = &pdv->MRCB;
  PRESOURCECONTROLBLOCK        pRCB;
  PRESHEADER                   pResHeader;
  PBYTE                        pbResHeaderSave;    // Hold old ptr
  PBYTE                        pb;
  INT                          i, j;
  PBYTE                        pbRCBsave;
  // For setting job
  PCN                          pcn = &pddc->pdv->cn;
  PQMJOBINFO                   pQMJobInfo;
  APIRET                       ulRC;
  PPRJINFO3                    pPrjInfo3;
  PLDRIVDATA                   pLDrivData;
  ULONG                        ulSize;
  BOOL                         fFreeBlock;
  ULONG                        ulOffset;
  #define  NERR_BufTooSmall 2123

  // Return if no resource logged
  if ( pMRCB->lCount == 0 )
  {
    return;
  }

  // If JobInfo is null there is nothing to do
  if ( pcn->pQMJobInfo == NULL )
  {
    return;
  }

  pQMJobInfo = pcn->pQMJobInfo;

  // Call spooler to get job props size first
  ulRC = SplQueryJob( pQMJobInfo->pszComputerName,
                      pQMJobInfo->pszQueueName,
                      pQMJobInfo->ulJobID,
                      3,
                      NULL,
                      0,
                      &ulSize );

  if ( ulRC == ERROR_MORE_DATA ||
       ulRC == NERR_BufTooSmall )
  {
    if ( ( pPrjInfo3 = (PPRJINFO3)GplMemoryAlloc( pdv->pDCHeap, ulSize ) ) == NULL )
    {
      return;
    }

    ulRC = SplQueryJob( pQMJobInfo->pszComputerName,
                        pQMJobInfo->pszQueueName,
                        pQMJobInfo->ulJobID,
                        3,
                        pPrjInfo3,
                        ulSize,
                        &ulSize);
  }

  // Check spooler call
  if ( ulRC != NO_ERROR )
  {
    return;
  }

  // Need to allocate a buffer big enough to hold resources
  ulSize = pPrjInfo3->pDriverData->cb;      // Size of drivdata
  i = ulSize + pMRCB->lTotalSize;

  // Get new buffer
  if ( ( pLDrivData = (PLDRIVDATA)GplMemoryAlloc( pdv->pDCHeap, i ) ) == NULL )
  {
    return;
  }

  // Copy spooler data over
  memcpy( pLDrivData, pPrjInfo3->pDriverData, ulSize );

  // Free spooler data
  GplMemoryFree( pPrjInfo3 );

  // Point to end of data
  pb = ((PBYTE)pLDrivData) + pLDrivData->cb;

  // Record offest for later - note offset is measured from begining of cnfdata
  ulOffset = pb - (PBYTE)&(pLDrivData->cnfData);

  // If the no_free flag is off then free blocks as processed
  fFreeBlock = ( CHECKFLAG( pMRCB->ulFlags, RESOURCE_FREE_BLOCKS ) != 0 ); //@V4.1218347

  // For each resource control block process its resource blocks
  pRCB =  (PRESOURCECONTROLBLOCK)pMRCB->ulpRCB;
  for ( i = 0; i < pMRCB->lCount; i++ )
  {
    pbRCBsave = (PBYTE)pRCB;  // Save for freeing

    memcpy( pb, (PBYTE)pRCB, pRCB->header.lSize ); // Copy RCB
    pb += pRCB->header.lSize;

    // Get the first resource
    pResHeader = (PRESHEADER)(pRCB->ulpResource);
    // Loop thru the resource chain
    for ( j = 0; j < pRCB->lCount; j++ )
    {
      pbResHeaderSave = (PBYTE)pResHeader;  // Save for later
      // Copy the resource over
      memcpy( pb, pResHeader, pResHeader->lSize );
      pb += pResHeader->lSize;

      // On to next, free old
      pResHeader = (PRESHEADER)pResHeader->ulpNext;
      if ( fFreeBlock == TRUE )
      {
        GplMemoryFree( pbResHeaderSave );
      }
    }

    // Point to next RCB
    pRCB = (PRESOURCECONTROLBLOCK)(pRCB->header.ulpNext);

    if ( fFreeBlock == TRUE )
    {
      GplMemoryFree( pbRCBsave );
    }
  }

  // Copy the MRCB over
  NewpCNFData = (PCNFDATA)&(pLDrivData->cnfData);
  NewpCNFData->u.iv.MRCB = *pMRCB;

  // Change pointer to offset
  NewpCNFData->u.iv.MRCB.ulpRCB = ulOffset;

  // Indicate MRCB has resource data in it
  SETFLAG( NewpCNFData->u.iv.MRCB.ulFlags, RESOURCE_DATA_SET );

  // These are now contiguous so do free individual blocks
  SETFLAG( NewpCNFData->u.iv.MRCB.ulFlags, RESOURCE_DO_NOT_FREE );

  // Bump up the size
  pLDrivData->cb += NewpCNFData->u.iv.MRCB.lTotalSize;

  // If resources are a single buffer delete it
  if ( CHECKFLAG( pMRCB->ulFlags, RESOURCE_FREE_BUF ) != 0 )      //@V4.1218347
  {
    GplMemoryFree( (PBYTE)(pMRCB->ulpRCB) );
  }

  // Now set the job prop
  SplSetJob( pQMJobInfo->pszComputerName,
             pQMJobInfo->pszQueueName,
             pQMJobInfo->ulJobID,
             3,
             pLDrivData,
             pLDrivData->cb,
             PRJ_DRIVERDATA_PARMNUM );

  // @V4.1218347
  // Free temp JP data
  GplMemoryFree( pLDrivData );

  return;
}


/*****************************************************************************\
**
** FUNCTION: SetUpJPResource
**
** DESCRIPTION:
**  Converts the offsets in Resource blocks to pointers
**
\*****************************************************************************/
VOID SetUpJPResource( PDV         pdv        ,
                      PLDRIVDATA  pLDrivData )
{
  PMASTERRESOURCECONTROLBLOCK  pMRCB  = &pdv->MRCB;
  PRESOURCECONTROLBLOCK        pRCB;
  PRESOURCECONTROLBLOCK        pRCBSave;
  PRESHEADER                   pResHeader;
  PCNFDATA                     pCNFData;
  INT                          i, j;
  PB                           pb;

  /* If there are resources get size
  ** Make sure there is a full cnfdata block
  */
  if ( pLDrivData == NULL              ||
       pLDrivData->cb < sizeof(CNFDATA) )
  {
    return;
  }

  // Set up cnf pointer
  pCNFData = &(pLDrivData->cnfData);

  // Copy the MRCB over to pdv
  *pMRCB = pCNFData->u.iv.MRCB;


  // Verify data
  if ( pMRCB->header.ulType != MASTER_RESOURCE_CONTROL_BLOCK ||
       pMRCB->ulVersion     != V1_0                           ) //Change test when adding next version
  {
    // Zero out the MRCB data to prevent other errors
    memset( pMRCB, 0, sizeof( MASTERRESOURCECONTROLBLOCK ) );
/// memset( &(pCNFData->u.iv.MRCB), 0, sizeof( MASTERRESOURCECONTROLBLOCK ) );

    return;
  }

  // Make sure something to do
  if ( pMRCB->lCount == 0 )
  {
    return;
  }

  // Get some memory for resources and copy over.  JP can change between
  // pages and resource list lost
  if ( ( pb = GplMemoryAlloc( pdv->pDCHeap, pMRCB->lTotalSize ) ) == NULL )
  {
    return;
  }
  memcpy( pb, ((PBYTE)pCNFData) + pMRCB->ulpRCB, pMRCB->lTotalSize );


  // Convert the offset tp RCBs to address
  // The RCBs start at offset from begining of resource data
  pMRCB->ulpRCB = (ULONG)( pb );

  // For each resource control block process its resource blocks
  pRCB =  (PRESOURCECONTROLBLOCK)(pMRCB->ulpRCB);
  for ( i = 0; i < pMRCB->lCount; i++ )
  {
    // Save the RCB ptr so they can be linked
    pRCBSave = pRCB;

    // Point to first resource block
    pRCB->ulpResource = (ULONG)(((PBYTE)pRCB) + pRCB->header.lSize);
    pResHeader = (PRESHEADER)(pRCB->ulpResource);

    for ( j = 0; j < pRCB->lCount; j++ )
    {
      pResHeader->ulpNext = (ULONG)(((PBYTE)pResHeader) + pResHeader->lSize );
      pResHeader = (PRESHEADER)(pResHeader->ulpNext);
    }

    // Point RCB next field to next RCB
    pRCBSave->header.ulpNext = (ULONG)pResHeader;
    pRCB = (PRESOURCECONTROLBLOCK)pResHeader;
  }

  // These are now contiguous so do free individual blocks
  SETFLAG( pMRCB->ulFlags, RESOURCE_FREE_BUF );
  CLEARFLAG( pMRCB->ulFlags, RESOURCE_FREE_BLOCKS );              //@V4.1218347

  return;
}


/*****************************************************************************\
**
** FUNCTION: GetRCB
**
** DESCRIPTION: Finds the requested Resource Control Block
**
\*****************************************************************************/

PRESOURCECONTROLBLOCK GetRCB( PMASTERRESOURCECONTROLBLOCK  pMRCB,
                              ULONG                        ulType )
{
  PRESOURCECONTROLBLOCK        pRCB  = NULL;
  INT                          i;

  // Look for the FontCB;
  pRCB =  (PRESOURCECONTROLBLOCK)pMRCB->ulpRCB;
  for ( i = 0; i < pMRCB->lCount; i++ )
  { // If this RCB is font quit loop
    if ( pRCB->header.ulType == ulType )
    {
      break;
    }
    // Point to next RCB
    pRCB = (PRESOURCECONTROLBLOCK)pRCB->header.ulpNext;
  }

  // If off list set pRCB to null
  if ( i >= pMRCB->lCount )
  {
    pRCB = NULL;
  }

  return pRCB;
}

/*****************************************************************************\
**
** FUNCTION: FontResourceDownload
**
** DESCRIPTION
**   Called at the start of job and start of each page will download any
**   Multipage fonts.  Single page fonts are taken care of by FontDownLoad
**
\*****************************************************************************/

VOID FontResourceDownload( PDDC  pddc    ,
                           ULONG ulUsage )  // What fonts to download
{
  PDV                          pdv   = pddc->pdv;
  PMASTERRESOURCECONTROLBLOCK  pMRCB = &pdv->MRCB;
  PRESOURCECONTROLBLOCK        pRCB;
  PFONTRESOURCEBLOCK           pFRB;
  INT                          i;
  INT                          iCurrentPage;

  // If font data has not already been recorded from STD pass the return
  // since there is no knowlwdge of font usage
  if ( CHECKFLAG( pMRCB->ulFlags, RESOURCE_DATA_SET ) == 0 )
  {
    return;
  }

  // Look for the FontCB;
  pRCB =  GetRCB( pMRCB, FONT_CONTROL_BLOCK );

  // IF no FRCB found then return
  if ( pRCB == NULL )
  {
    return;
  }

  // Move current page to local var
  iCurrentPage = GetCurrentPageNum( pddc );

  // Look at font list - if the fonts start page is the current page and it's
  // a multipage and a downloaded font send it down;
  pFRB = (PFONTRESOURCEBLOCK)pRCB->ulpResource;
  for ( i = 0; i < pRCB->lCount; i++ )
  {
    if ( pFRB->lStartPage == iCurrentPage &&
         pFRB->lFontType  == DOWNLOAD     &&
         (pFRB->ulUsage & ulUsage ) != 0  &&
         CHECKFLAG( pFRB->ulFlags, FONT_LOADED ) == 0 )
    {
      FontDownload( pddc, (PSZ)pFRB->achFontName, TRUE );
      SETFLAG( pFRB->ulFlags, FONT_LOADED );
    }

    // To Next
    pFRB = (PFONTRESOURCEBLOCK)pFRB->header.ulpNext;
  }

  return;
}


/*****************************************************************************\
**
** FUNCTION: ResProcessEndPage
**
** DESCRIPTION
**
\*****************************************************************************/

VOID ResProcessEndPage( PDDC  pddc )
{
  PDV                          pdv   = pddc->pdv;
  PMASTERRESOURCECONTROLBLOCK  pMRCB = &pdv->MRCB;
  PRESOURCECONTROLBLOCK        pRCB;
  PFONTRESOURCEBLOCK           pFRB;
  INT                          i;
  INT                          iCurrentPage;


  pMRCB->lLastPageProcessed++;  //Bump processed pages              @V4.1198538

  // If a reset DC was done and generating resource info must change any
  // Single page fonts to multipage fonts
  if ( CHECKFLAG( pdv->ulGenFlags, DID_RESETDC )      != 0 &&
       CHECKFLAG( pMRCB->ulFlags, RESOURCE_DATA_SET ) == 0  )
  {
    // Turn off flag
    CLEARFLAG( pdv->ulGenFlags, DID_RESETDC ) ;

    // Look for the FontCB;
    pRCB =  GetRCB( pMRCB, FONT_CONTROL_BLOCK );

    // IF no FRCB found then return
    if ( pRCB == NULL )
    {
      return;
    }

    // Move current page to local var
    // When spooling and get newframe/enddoc the the spooler has already bumped
    // the pagenum so decrement it
    iCurrentPage = GetCurrentPageNum( pddc ) - 1;

    // Look at font list - if the fonts start page is the current page and it's
    // a single page d a downloaded font send it down;
    pFRB = (PFONTRESOURCEBLOCK)pRCB->ulpResource;
    for ( i = 0; i < pRCB->lCount; i++ )
    {
      if ( pFRB->lStartPage == iCurrentPage                               &&
           (pFRB->ulUsage & (SINGLE_LINE | SINGLE_PAGE | TOP_OF_PAGE)) != 0 )
      {
        pFRB->ulUsage = MULTI_PAGE;
      }

      // To Next
      pFRB = (PFONTRESOURCEBLOCK)pFRB->header.ulpNext;
    }
  }

  return;
}


//*****************************************************************************
//
// FUNCTION: IsEuroInCP
//
// DESCRIPTION: Given a codepage will see if in list of codepages which have
//              a Euro
//
//   Note: A list of what codepages have Euro can be found by scanning the
//         *.ucmap files in \culs\src\uls\codepage for Server 99 src
//
//*****************************************************************************

BOOL IsEuroInCP( USHORT usCodePage )
{
  switch ( usCodePage )
  {
  case    0:
  case  819:
  case  850:
  case  857:
  case 1004:
  case 1250:
  case 1251:
  case 1252:
  case 1253:
  case 1254:
  case 1255:
  case 1256:
  case 1257:
    return TRUE;

  default:
    return FALSE;
  }

  return FALSE;
}

//*****************************************************************************
//
// FUNCTION: MoveTextMetrics
//
// DESCRIPTION: Copy Font Metrics from FOCAMETICS to FONTMETRICS
//
//*****************************************************************************

VOID MoveTextMetrics(PFOCAMETRICS pfoca, PFONTMETRICS pfm)
{
   PULONG  tul;
   PUSHORT tus,sus;
   USHORT  ii;

   utl_memcopy(pfm->szFamilyname, pfoca->szFamilyname, FACESIZE);
   utl_memcopy(pfm->szFacename, pfoca->szFacename, FACESIZE);

   tus = (PUSHORT)&(pfm->idRegistry);
   sus = (PUSHORT)&(pfoca->usRegistryId);

//   pfm->idRegistry   = pfoca->usRegistryId;
//   pfm->usCodePage   = pfoca->usCodePage;

   for(ii=0;ii<2;ii++,tus++,sus++)
       *tus=*sus;

   // CYRILLIC | LATIN2
   // change 850 fonts to 0
   //if(pfm->usCodePage == UGL_FONT)
   //   pfm->usCodePage = 0;

   tul = (PULONG)tus;

//   pfm->lEmHeight         = (LONG)pfoca->yEmHeight;
//   pfm->lXHeight          = (LONG)pfoca->yXHeight;
//   pfm->lMaxAscender      = (LONG)pfoca->yMaxAscender;
//   pfm->lMaxDescender     = (LONG)pfoca->yMaxDescender;
//   pfm->lLowerCaseAscent  = (LONG)pfoca->yLowerCaseAscent;
//   pfm->lLowerCaseDescent = (LONG)pfoca->yLowerCaseDescent;
//   pfm->lInternalLeading  = (LONG)pfoca->yInternalLeading;
//   pfm->lExternalLeading  = (LONG)pfoca->yExternalLeading;
//   pfm->lAveCharWidth     = (LONG)pfoca->xAveCharWidth;
//   pfm->lMaxCharInc       = (LONG)pfoca->xMaxCharInc;
//   pfm->lEmInc            = (LONG)pfoca->xEmInc;
//   pfm->lMaxBaselineExt   = (LONG)pfoca->yMaxBaselineExt;

   for(ii=0;ii<12;ii++,tul++,sus++)
       *tul=(ULONG)*sus;

   tus = (PUSHORT)tul;

//   pfm->sCharSlope        = pfoca->sCharSlope;
//   pfm->sInlineDir        = pfoca->sInlineDir;
//   pfm->sCharRot          = pfoca->sCharRot;

//   pfm->usWeightClass     = pfoca->usWeightClass;
//   pfm->usWidthClass      = pfoca->usWidthClass;

//   pfm->sXDeviceRes       = pfoca->xDeviceRes;
//   pfm->sYDeviceRes       = pfoca->yDeviceRes;
//   pfm->sFirstChar        = pfoca->usFirstChar;
//   pfm->sLastChar         = pfoca->usLastChar;
//   pfm->sDefaultChar      = pfoca->usDefaultChar;
//   pfm->sBreakChar        = pfoca->usBreakChar;
//   pfm->sNominalPointSize = pfoca->usNominalPointSize;
//   pfm->sMinimumPointSize = pfoca->usMinimumPointSize;
//   pfm->sMaximumPointSize = pfoca->usMaximumPointSize;

//   pfm->fsType            = (USHORT)pfoca->fsTypeFlags;
//   pfm->fsDefn            = (USHORT)pfoca->fsDefn;
//   pfm->fsSelection       = (USHORT)pfoca->fsSelectionFlags;
//   pfm->fsCapabilities    = (USHORT)pfoca->fsCapabilities;

   for(ii=0;ii<18;ii++,tus++,sus++)
       *tus=*sus;

   tul = (PULONG)tus;

//   pfm->lSubscriptXSize     = (LONG)pfoca->ySubscriptXSize;
//   pfm->lSubscriptYSize     = (LONG)pfoca->ySubscriptYSize;
//   pfm->lSubscriptXOffset   = (LONG)pfoca->ySubscriptXOffset;
//   pfm->lSubscriptYOffset   = (LONG)pfoca->ySubscriptYOffset;
//   pfm->lSuperscriptXSize   = (LONG)pfoca->ySuperscriptXSize;
//   pfm->lSuperscriptYSize   = (LONG)pfoca->ySuperscriptYSize;
//   pfm->lSuperscriptXOffset = (LONG)pfoca->ySuperscriptXOffset;
//   pfm->lSuperscriptYOffset = (LONG)pfoca->ySuperscriptYOffset;
//   pfm->lUnderscoreSize     = (LONG)pfoca->yUnderscoreSize;
//   pfm->lUnderscorePosition = (LONG)pfoca->yUnderscorePosition;
//   pfm->lStrikeoutSize      = (LONG)pfoca->yStrikeoutSize;
//   pfm->lStrikeoutPosition  = (LONG)pfoca->yStrikeoutPosition;

   for(ii=0;ii<12;ii++,tul++,sus++)
       *tul=(ULONG)*sus;

   tus = (PUSHORT)tul;

//   pfm->sKerningPairs = pfoca->usKerningPairs;
//   pfm->sFamilyClass  = pfoca->sFamilyClass;

   for(ii=0;ii<2;ii++,tus++,sus++)
       *tus=*sus;

   return;
}

//*****************************************************************************
//
// FUNCTION: IsDynFnt
//
// DESCRIPTION: Returns TRUE if given font index is Dynamic Font
//
//*****************************************************************************

BOOL IsDynFnt( PDDC pddc, SHORT usIndex )
{
  PFNT         pFnt;

  pFnt = PfntFromIndex( pddc, usIndex );

  if( pFnt->bLoaded & DYNFNT )
     return TRUE;
  else
     return FALSE;
}


//@BITSUB
//*****************************************************************************
//
// FUNCTION: CheckBitMapFont
//
// DESCRIPTION: If the requested font name is a system bitmap font tis routine
// will change the name to the equivalent system outline(downloadable) font
//
// RETURNS: Either the orig pfa if no sub or NewFATTRS if sub-ed
//
//*****************************************************************************

PFATTRS CheckBitMapFont( PFATTRS pfa,         // Orig pfa
                         PFATTRS NewFATTRS )  // Our copy
{
  // Names of System Bitmap Fonts - note the order
  static PSZ pBitmapNames[] = {
    "Helv",
    "Helv Italic",
    "Helv Bold",
    "Helv Bold Italic",
    "Tms Rmn",
    "Tms Rmn Italic",
    "Tms Rmn Bold",
    "Tms Rmn Bold Italic" };

  // Names of System Outline fonts - order specific
  static PSZ pOutlineNames[] = {
    "Helvetica",
    "Helvetica Italic",
    "Helvetica Bold",
    "Helvetica Bold Italic",
    "Times New Roman",
    "Times New Roman Italic",
    "Times New Roman Bold",
    "Times New Roman Bold Italic" };
  #define HELV             0
  #define TMSRMN           4

  INT iIndex = -1;  // Set to none
  INT i;

  // Sniff if font name starts H
  if ( pfa->szFacename[0] == 'H' )
  {
    iIndex = HELV;
  }
  else // Sniff if font name starts T
  if ( pfa->szFacename[0] == 'T' )
  {
    iIndex = TMSRMN;
  }

  // Neither so return
  if ( iIndex == -1 )
  {
    return pfa;
  }

  // Look thru the bitmap fonts for name match
  for ( i = 0; i < 4; i++ )
  {
    if ( strcmp( (PSZ)pfa->szFacename, pBitmapNames[iIndex] ) == 0 )
    { // Good - a match
      strcpy( (PSZ)pfa->szFacename, pOutlineNames[iIndex] );
      break;
    }
    iIndex++;
  }

  return pfa;
}


//*****************************************************************************
//
// FUNCTION: FindFontIDNum
//
// DESCRIPION: Given any fontname will look in AllFonts to see if in there
//
//*****************************************************************************

FONT_ID_NUM  FindFontIDNum( PDDC pddc,
                            PSZ  pszFacename,
                            BOOL fSystemFont )
{
  FONT_INFO fi; // Formal font type
  FONT_ID_NUM FIDNum = FONT_NOT_FOUND;  // Indicates is no find

  fi.iType = REGULAR;
  if ( fSystemFont == TRUE )
  {
    fi.iFamily = SYSTEM_FONTS;  // Set to system fonts
  }
  else
  {
    fi.iFamily = 0;             // Set to start of list
  }

  // Loop thru all the listed fonts
  // Outer loop is for each family, Inner loop is for types
  while( AllFonts[fi.iFamily][REGULAR] != NULL )
  {
    for ( fi.iType = REGULAR; fi.iType <= BOLDITALIC; fi.iType++ )
    {
      if ( AllFonts[fi.iFamily][fi.iType] != NULL &&
           strcmp( pszFacename, AllFonts[fi.iFamily][fi.iType] ) == 0 )
      {
        FIDNum = MAKEULONG( fi.iType, fi.iFamily );   // Create Font ID Num
        return FIDNum;                                // Return It
      }
    }
    fi.iFamily++;
  }

  return FIDNum;
}

