/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
#define  INCL_DEV                      /* Include Device stuff              */
#define  INCL_DEVDJP                   /* Include DJP stuff                 */
#define  INCL_SPL                      /* Spooler stuff                     */
#define  INCL_ERRORS

#include "plotters.h"
#include "utils.h"
#include "profile.h"

extern BOOL         get_device_id       (PSZ,PUSHORT,PUSHORT);
extern LOCAL VOID   initialize_pdevice  (PVOID,PSZ,PPDEVICE,BOOL);
extern VOID         build_app_name      (PSZ,PSZ,PSZ,PSZ);
extern LOCAL VOID   GetPageData         (PPDEVICE,SHORT,PHCINFO,PPROFILEINFO);
extern SHORT        NumBits             (ULONG ulNum);

/* Function prototypes...
*/
APIRET       InnerDeviceQuery          (ULONG          ulFuncNum,
                                        PPDEVICE       pPDevice,
                                        PDRIVDATA      pGoodDrivData,
                                        ULONG          cbParm1,
                                        PBYTE          pbParm1,
                                        ULONG          cbParm2,
                                        PBYTE          pbParm2);
SHORT        FormIndex                 (USHORT         usMask);
LONG         ReturnFormID              (USHORT         usFormBit);
PSZ          pszProperty               (ULONG          ulProperty);
PSZ          pszType                   (LONG           lType);
BOOL         FormsMatch                (PDJPT_FORM     pS,
                                        PHCINFO        pD);
BOOL         FillInFormInfo            (PDJPT_FORM     pDJPForm,
                                        PPDEVICE       pPDevice,
                                        SHORT          Size,
                                        PPROFILEINFO   pJobProp);

/****************************************************************************/
/* PROCEDURE NAME : OS2_PM_DRV_QUERY                                        */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 8/30/95                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG APIENTRY
OS2_PM_DRV_POSTESCAPE (PSZ       pszDriverName,
                       PSZ       pszDeviceName,
                       PSZ       pszQueueName,
                       PSZ       pszSplPrinterName,
                       ULONG     ulFuncNum,
                       ULONG     cbParm1,
                       PBYTE     pbParm1,
                       ULONG     cbParm2,
                       PBYTE     pbParm2)
{
   HMCB               hmcbHeap      = (HMCB)0;
   PPDEVICE           pPDevice      = (PPDEVICE)NULL;
   ULONG              pcbNeeded;
   PDRIVDATA          pGoodDrivData = (PDRIVDATA)NULL;
   PPROFILEINFO       pJobProp;
   PDRIVDATA          pDrivData;
   APIRET             rc;
   ULONG              ulrc;

   ulrc = DEV_ERROR;

   // This api is not supported on pre-DAX engines
#ifndef DISABLE_DJP_SUPPORT
   if (RASTER_ENGINE_234 > GreQueryEngineVersion ())
#endif
      goto depart;

   // Assume an bad parameter condition
   ulrc = DEV_BAD_PARAMETERS;

   // Validate driver name
   if (0 != strcmp (pszDriverName, APP_NAME))
      goto depart;

   // Validate queue name
   rc = SplQueryQueue ((PSZ)NULL,
                       pszQueueName,
                       3,
                       (PVOID)NULL,
                       0,
                       &pcbNeeded);
   assertT (NERR_BufTooSmall != rc);
   if (NERR_BufTooSmall != rc)
      goto depart;

   // Validate spooler printer name
   rc = SplQueryDevice ((PSZ)NULL,
                        pszSplPrinterName,
                        3,
                        (PVOID)NULL,
                        0,
                        &pcbNeeded);
   assertT (NERR_BufTooSmall != rc);
   if (NERR_BufTooSmall != rc)
      goto depart;

   ulrc = DEV_OK;

   // Need good job properties?
   switch (ulFuncNum)
   {
   case DEVPE_QUERYSIZE:
   case DEVPE_QUERYJOBPROPERTIES:
   case DEVPE_SETJOBPROPERTIES:
   case DEVPE_DEFAULTJOBPROPERTIES:
   {
      /*
      ** Create a heap and grow it by the same amount.
      ** Do not allocate anything greater than 16k from the heap-use dosallocmem.
      ** Make the heap non shareable
      */
      hmcbHeap = GplMemoryCreateInstance (HEAPSIZE, HEAPSIZE, 0x4000, PROCESS_MEMORY);

      /* Take the input job properties and make sure the size is set
      ** correctly.
      */
      pDrivData  = (PDRIVDATA)pbParm2;
      pDrivData->cb = min (cbParm2, pDrivData->cb);

      pGoodDrivData = return_driver_data (hmcbHeap,
                                          pDrivData,
                                          pszSplPrinterName,
                                          TRUE);
      assertF (pGoodDrivData);
      pJobProp = (PPROFILEINFO)(&pGoodDrivData->abGeneralData);

      /* Is the size of the input driver data is not the same as
      ** the size of our good data?
      */
      if (pDrivData->cb != pGoodDrivData->cb)
         ulrc = DEV_PROP_BUF_TOO_SMALL;

      /* Or, if the two don't compare, then fail the call.
      */
      if (0 != memcmp (pDrivData, pGoodDrivData, pGoodDrivData->cb))
         ulrc = DEV_INV_INP_JOBPROPERTIES;
   }
   }

   if (DEV_OK != ulrc)
      goto depart;

   // Need a device block?
   switch (ulFuncNum)
   {
   case DEVPE_QUERYSIZE:
   case DEVPE_QUERYJOBPROPERTIES:
   case DEVPE_SETJOBPROPERTIES:
   case DEVPE_DEFAULTJOBPROPERTIES:
   {
      CHAR  achAppName[255];

      // Allocate a device instance
      pPDevice = (PPDEVICE)GplMemoryAlloc (hmcbHeap, sizeof(PDEVICE));
      assertF (pPDevice);
      pPDevice->paPaletteEntry = (PPALETTEENTRY)NULL;

      // Set up needed elements!
      pPDevice->pDriverData    = pGoodDrivData;
      pPDevice->pSetup         = (PPROFILEINFO)pPDevice->pDriverData->abGeneralData;
      pPDevice->pszPrinterName = pszSplPrinterName;

      build_app_name (APP_NAME,                               // Driver name
                      pPDevice->pDriverData->szDeviceName,    // Device name
                      pPDevice->pszPrinterName,
                      achAppName);

      get_device_id (pPDevice->pDriverData->szDeviceName,
                     &pPDevice->Plotter,
                     &pPDevice->StringID);

      initialize_pdevice (hmcbHeap, achAppName, pPDevice, FALSE);
      break;
   }
   }

   ulrc = InnerDeviceQuery (ulFuncNum,
                            pPDevice,
                            pGoodDrivData,
                            cbParm1,
                            pbParm1,
                            cbParm2,
                            pbParm2);

depart:
  /*
  ** free mem allocated in initialize_pdevice
  */
  if (pPDevice &&
      pPDevice->paPaletteEntry)
     GplMemoryFree ((PVOID)pPDevice->paPaletteEntry);

  /*
  ** free the drivdata
  */
  if (pGoodDrivData)
     GplMemoryFree (pGoodDrivData);

  /*
  ** free pPDevice allocated in this module
  */
  if (pPDevice)
    GplMemoryFree ((PVOID)pPDevice);

   /*
   ** free the heap
   */
   if (hmcbHeap)
      GplMemoryDeleteInstance (hmcbHeap);

   return ulrc;
}

/****************************************************************************/
/* PROCEDURE NAME : InnerDeviceQuery                                        */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 8/30/95                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
APIRET
InnerDeviceQuery (ULONG          ulFuncNum,
                  PPDEVICE       pPDevice,
                  PDRIVDATA      pGoodDrivData,
                  ULONG          cbParm1,
                  PBYTE          pbParm1,
                  ULONG          cbParm2,
                  PBYTE          pbParm2)
{
   LONG               lResult    = DEV_OK;
   PPROFILEINFO       pJobProp   = (PPROFILEINFO)pGoodDrivData->abGeneralData;

   switch (ulFuncNum)
   {
   case DEVPE_QUERYSUPPORT:
   {
      ULONG    ulCheck;

      if (sizeof (ULONG) > cbParm1)
         // Not big enough!
         break;

      ulCheck = *(PULONG)pbParm1;

      switch (ulCheck)
      {
      case DEVPE_QUERYSUPPORT:
      case DEVPE_QUERYSIZE:
      case DEVPE_QUERYJOBPROPERTIES:
      case DEVPE_SETJOBPROPERTIES:
         // good result
         lResult = DEV_OK;
         break;

      default:
         // We don't support these
         lResult = DEVPE_NOTSUPPORTED;
         break;
      }
      break;
   }

   case DEVPE_QUERYSIZE:
   {
      INT         cbQueries  = cbParm1;
      PQUERYSIZE  pQuerySize = (PQUERYSIZE)pbParm1;
      PQUERYTUPLE pTuple     = pQuerySize->aTuples;
      INT         cbElmSize  = sizeof (QUERYTUPLE);
      INT         iCount     = 1;
      INT         iMaxCount;
      INT         iSize;

      // Take off the header
      cbQueries -= QUERYSIZE_HEADER_SIZE;

      // Must be an even multiple
      assertT (cbQueries % cbElmSize);
      if (cbQueries % cbElmSize)
      {
         GplErrSetError (PMERR_INVALID_PARAMETER);
         break;
      }

      // Initialize the returned size
      pQuerySize->ulSizeNeeded = 0;

      while (cbQueries)
      {
         switch (pTuple->ulProperty)
         {
         case DJP_NONE:
            iSize     = sizeof (((PDJP_ITEM)NULL)->ulValue);
            iMaxCount = 1;
            break;

         case DJP_SJ_ORIENTATION:
            iSize     = sizeof (DJPT_ORIENTATION);
            iMaxCount = 2;
            break;

         case DJP_CJ_FORM:
            iSize     = sizeof (DJPT_FORM);
            iMaxCount = NumBits (PlotterClass[pPDevice->Plotter].DeviceInfo.fsMediaSizeSupport);
            break;

         case DJP_SJ_PAPERSIZE:
            iSize     = sizeof (DJPT_PAPERSIZE);
            iMaxCount = NumBits (PlotterClass[pPDevice->Plotter].DeviceInfo.fsMediaSizeSupport);
            break;

         case DJP_CJ_MIXEDFORMS:
            iSize     = sizeof (DJPT_MIXEDFORMS);
            iMaxCount = 1;
            break;

         case DJP_SJ_COPIES:
            iSize     = sizeof (DJPT_COPIES);
            iMaxCount = 1;
            break;

         case DJP_SJ_FEED:
            iSize     = sizeof (DJPT_FEED);
            iMaxCount = 2;
            break;

         case DJP_CJ_RESOLUTION:
         case DJP_SJ_BITSPERPEL:
         case DJP_SJ_COLOR:
         case DJP_SJ_PRINTQUALITY:
         case DJP_SJ_FONTDOWNLOADING:
         case DJP_SJ_DUPLEX:
         case DJP_SJ_COLLATE:
         case DJP_SJ_SCALING:
         case DJP_SJ_TRAYTYPE:
         case DJP_SJ_MEDIA:
         case DJP_SJ_FORMFEEDCONTROL:
         default:
            iSize         = sizeof (((PDJP_ITEM)NULL)->ulValue);
            iMaxCount     = 1;
            DBPRINTF (("Unknow query '%s' = %d!\n",
                        pszProperty (pTuple->ulProperty),
                        pTuple->ulProperty));
            break;
         }

         DBPRINTF (("Query size '%s'/%d = %d.  Max count = %d\n",
                    pszProperty (pTuple->ulProperty),
                    pTuple->ulProperty,
                    iSize, iMaxCount));

         if (DJP_NONE    == pTuple->ulProperty ||
             DJP_CURRENT == pTuple->lType       )
         {
            iCount = 1;
         }
         else if (DJP_ALL == pTuple->lType)
         {
            iCount = iMaxCount;
         }

         // Bump up the returned size
         pQuerySize->ulSizeNeeded += DJP_HEADER_SIZE + iCount * iSize;

         cbQueries  -= cbElmSize;
         pTuple++;
      }

      // good result
      lResult = DEV_OK;
      break;
   }

   case DEVPE_QUERYJOBPROPERTIES:
   {
      INT            cbQueries   = cbParm1;
      PBYTE          pbQueries   = pbParm1;
      PDJP_ITEM      pIQuery     = (PDJP_ITEM)pbQueries;
      BOOL           fError      = FALSE;
      PBYTE          pbNext      = (PBYTE)NULL;
      INT            iNumQueries;
      PQUERYTUPLE    pTuples     = NULL,
                     pTupleCur;

      // Output buffer must be at least one item in size
      if (sizeof (DJP_ITEM) > cbQueries)
      {
         lResult = DEV_BAD_PARAMETERS;
         break;
      }

      // Step 1.  Count the number of queries
      iNumQueries = 0;
      pIQuery     = (PDJP_ITEM)pbQueries;
      while (pIQuery->ulProperty != DJP_NONE)
      {
         iNumQueries++;
         pIQuery = DJP_NEXT_STRUCTP (pIQuery);
      }
      iNumQueries++;   // Add an EOL marker

      // Step 2.  Allocate and copy the query types
      pTuples = (PQUERYTUPLE)GplMemoryAlloc (pPDevice->hmcbHeap,
                                             iNumQueries * sizeof (QUERYTUPLE));
      assertF (pTuples);
      if (!pTuples)
      {
         lResult = DEV_ERROR;
         GplErrSetError (PMERR_INSUFFICIENT_MEMORY);
         break;
      }

      /* Since we will be overwriting the input list to the output list, we
      ** need to copy the original input list into an array of tuples
      ** (the only thing that matters)
      */
      pIQuery   = (PDJP_ITEM)pbQueries;
      pTupleCur = pTuples;
      while (pIQuery->ulProperty != DJP_NONE)
      {
         pTupleCur->ulProperty = pIQuery->ulProperty;
         pTupleCur->lType      = pIQuery->lType;

         pIQuery = DJP_NEXT_STRUCTP (pIQuery);
         pTupleCur++;
      }
      // And copy the EOL
      pTupleCur->ulProperty = DJP_NONE;
      pTupleCur->lType      = DJP_NONE;

      // Step 3.  Fill in the output list
      pIQuery = (PDJP_ITEM)pbQueries;
      pTupleCur = pTuples;
      while (0 < iNumQueries)
      {
         BOOL  fCurError = FALSE;

         pIQuery->ulProperty    = pTupleCur->ulProperty;
         pIQuery->lType         = pTupleCur->lType;
         pIQuery->ulNumReturned = 0;

         if (sizeof (DJP_ITEM) > cbQueries)
         {
            fError = DEV_PROP_BUF_TOO_SMALL;
            assertstring ("Not enough space left!\n");
            break;
         }

         DBPRINTF (("Query '%s'/%d.  Query type '%s'/%d\n",
                    pszProperty (pTupleCur->ulProperty),
                    pTupleCur->ulProperty,
                    pszType (pTupleCur->lType),
                    pTupleCur->lType));

         if (pTupleCur->lType == DJP_CURRENT)
         {
            pIQuery->ulNumReturned = 1;

            // Assume a simple type... complex types will override it
            pbNext = (PBYTE)(pIQuery + 1);

            switch (pTupleCur->ulProperty)
            {
            case DJP_SJ_ORIENTATION:
               if (OR_PORTRAIT == pJobProp->Orientation)
                  pIQuery->ulValue = DJP_ORI_PORTRAIT;
               else
                  pIQuery->ulValue = DJP_ORI_LANDSCAPE;
               break;

            case DJP_SJ_PAPERSIZE:
            case DJP_CJ_FORM:
            {
               PDJPT_FORM      pDJPForm = DJP_ELEMENTP (*pIQuery, DJPT_FORM);
               PDJPT_PAPERSIZE pDJPSize = DJP_ELEMENTP (*pIQuery, DJPT_PAPERSIZE);
               PDEVICEINFO     pDevice;
               USHORT          usMask;
               LONG            lRet;

               usMask = 1 << pJobProp->Size;
               pDevice = &PlotterClass[pPDevice->Plotter].DeviceInfo;

               if (DJP_SJ_PAPERSIZE == pTupleCur->ulProperty)
               {
                  lRet = ReturnFormID (usMask);
                  if (-1 != lRet)
                     pIQuery->ulValue = lRet;
                  else
                     fCurError = TRUE;
               }
               else /* if (DJP_CJ_FORM == pTupleCur->ulProperty) */
               {
                  FillInFormInfo (pDJPForm,
                                  pPDevice,
                                  pJobProp->Size,
                                  pJobProp);

                  /* NOTE: we do not have any tray or media info to return!
                  */
                  pDJPForm++;
                  pbNext = (PBYTE)pDJPForm;
               }
               break;
            }

            case DJP_CJ_MIXEDFORMS:
            {
               PDJPT_MIXEDFORMS pDJPMXForm = DJP_ELEMENTP (*pIQuery, DJPT_MIXEDFORMS);

               /* We only have one form for the document.
               */
               pDJPMXForm->lStartRange = DJP_MXF_INFINITY;
               pDJPMXForm->lEndRange   = DJP_MXF_INFINITY;
               FillInFormInfo (&pDJPMXForm->djpfmForm,
                               pPDevice,
                               pJobProp->Size,
                               pJobProp);

               pDJPMXForm++;
               pbNext = (PBYTE)pDJPMXForm;
               break;
            }

            case DJP_SJ_COPIES:
               pIQuery->ulValue = (pJobProp->usFlags & FL_UNCOLLATEDMASK) + 1;
               break;

            case DJP_SJ_FEED:
               if (pJobProp->PaperFeed)
                  pIQuery->ulValue = DJP_FED_AUTOMATIC;
               else
                  pIQuery->ulValue = DJP_FED_MANUAL;
               break;

            case DJP_NONE:
               // Nothing to do!
               pIQuery->ulValue = DJP_NONE;
               break;

            case DJP_CJ_RESOLUTION:
            case DJP_SJ_BITSPERPEL:
            case DJP_SJ_COLOR:
            case DJP_SJ_PRINTQUALITY:
            case DJP_SJ_FONTDOWNLOADING:
            case DJP_SJ_DUPLEX:
            case DJP_SJ_COLLATE:
            case DJP_SJ_SCALING:
            case DJP_SJ_TRAYTYPE:
            case DJP_SJ_MEDIA:
            case DJP_SJ_FORMFEEDCONTROL:
            default:
               fCurError = TRUE;
               DBPRINTF (("Unknow query '%s' = %d!\n",
                           pszProperty (pTupleCur->ulProperty),
                           pTupleCur->ulProperty));
               break;
            }
         }
         else if (pTupleCur->lType == DJP_ALL)
         {
            switch (pTupleCur->ulProperty)
            {
            case DJP_SJ_ORIENTATION:
            {
               PDJPT_ORIENTATION pTmp = DJP_ELEMENTP (*pIQuery, DJPT_ORIENTATION);

               *pTmp++ = DJP_ORI_PORTRAIT;
               *pTmp++ = DJP_ORI_LANDSCAPE;

               pIQuery->ulNumReturned = 2;
               pbNext = (PBYTE)pTmp;
               break;
            }

            case DJP_CJ_FORM:
            case DJP_SJ_PAPERSIZE:
            {
               ULONG           ulNumDefined;
               PDJPT_FORM      pDJPForm     = DJP_ELEMENTP (*pIQuery, DJPT_FORM);
               PDJPT_PAPERSIZE pDJPSize     = DJP_ELEMENTP (*pIQuery, DJPT_PAPERSIZE);
               PDEVICEINFO     pDevice;
               INT             iNumReturned = 0;
               USHORT          usMask;
               SHORT           Size;
               LONG            lRet;

               pDevice = &PlotterClass[pPDevice->Plotter].DeviceInfo;

               for (usMask = 0x8000, Size = 15; usMask; usMask >>= 1, Size--)
               {
                  if (pDevice->fsMediaSizeSupport & usMask)
                  {
                     if (DJP_SJ_PAPERSIZE == pTupleCur->ulProperty)
                     {
                        lRet = ReturnFormID (usMask);
                        if (-1 != lRet)
                        {
                           *pDJPSize++ = lRet;
                           iNumReturned++;
                           pbNext = (PBYTE)pDJPSize;
                        }
                        else
                           fCurError = TRUE;
                     }
                     else /* if (DJP_CJ_FORM == pTupleCur->ulProperty) */
                     {
                        FillInFormInfo (pDJPForm,
                                        pPDevice,
                                        Size,
                                        pJobProp);

                        /* NOTE: we do not have any tray or media info to return!
                        */
                        iNumReturned++;
                        pDJPForm++;
                        pbNext = (PBYTE)pDJPForm;
                     }
                  }
               }

               pIQuery->ulNumReturned = iNumReturned;
               break;
            }

            case DJP_CJ_MIXEDFORMS:
            {
               PDJPT_MIXEDFORMS pDJPMXForm = DJP_ELEMENTP (*pIQuery, DJPT_MIXEDFORMS);

               /* We only have one form for the document.
               */
               pDJPMXForm->lStartRange = DJP_MXF_INFINITY;
               pDJPMXForm->lEndRange   = DJP_MXF_INFINITY;
               FillInFormInfo (&pDJPMXForm->djpfmForm,
                               pPDevice,
                               pJobProp->Size,
                               pJobProp);

               pDJPMXForm++;
               pbNext = (PBYTE)pDJPMXForm;
               pIQuery->ulNumReturned = 1;
               break;
            }

            case DJP_SJ_COPIES:
            {
               pIQuery->lType         = DJP_ERROR_NOT_ENUM;
               pIQuery->ulNumReturned = 0;
               pIQuery->ulValue       = DJP_NONE;

               pbNext = (PBYTE)(pIQuery + 1);

               fCurError = TRUE;
               break;
            }

            case DJP_SJ_FEED:
            {
               PDJPT_FEED pTmp = DJP_ELEMENTP (*pIQuery, DJPT_FEED);

               *pTmp++ = DJP_FED_MANUAL;
               *pTmp++ = DJP_FED_AUTOMATIC;

               pIQuery->ulNumReturned = 2;
               pbNext = (PBYTE)pTmp;
               break;
            }

            case DJP_NONE:
            {
               // Nothing to do!
               pIQuery->ulNumReturned = 0;
               pIQuery->ulValue       = DJP_NONE;

               pbNext = (PBYTE)(pIQuery + 1);
               break;
            }

            case DJP_CJ_RESOLUTION:
            case DJP_SJ_BITSPERPEL:
            case DJP_SJ_COLOR:
            case DJP_SJ_PRINTQUALITY:
            case DJP_SJ_FONTDOWNLOADING:
            case DJP_SJ_DUPLEX:
            case DJP_SJ_COLLATE:
            case DJP_SJ_SCALING:
            case DJP_SJ_TRAYTYPE:
            case DJP_SJ_MEDIA:
            case DJP_SJ_FORMFEEDCONTROL:
            default:
               fCurError = TRUE;

               pbNext = (PBYTE)(pIQuery + 1);

               DBPRINTF (("Unknow query '%s' = %d!\n",
                           pszProperty (pTupleCur->ulProperty),
                           pTupleCur->ulProperty));
               break;
            }
         }
         else if (DJP_NONE != pTupleCur->lType)
         {
            fCurError = TRUE;

            assertstring ("Unknown query type!\n");
         }

         // We now know the final size
         pIQuery->cb = pbNext - (PBYTE)pIQuery;

         /* @TBD We need some way to stop overwriting memory
         */
         assertT (cbQueries < pIQuery->cb);

         cbQueries -= pIQuery->cb;

         if (fCurError)
         {
            pIQuery->ulNumReturned = 0;
            pIQuery->ulValue       = DJP_NONE;
            pIQuery->lType         = DJP_ERROR_NOT_SUPPORTED;
            fError                 = TRUE;
         }

         // Move to the next query item
         pIQuery = DJP_NEXT_STRUCTP (pIQuery);

         iNumQueries--;
         pTupleCur++;
      }

      // Clean up
      if (pTuples)
         GplMemoryFree (pTuples);

      if (fError)
      {
         GplErrSetWarning (PMERR_DATATYPE_ENTRY_INVALID);
         lResult = DEV_WARNING;
      }
      else
      {
         lResult = DEV_OK;
      }
      break;
   }

   case DEVPE_SETJOBPROPERTIES:
   {
      INT            cbQueries   = cbParm1;
      PBYTE          pbQueries   = pbParm1;
      PDJP_ITEM      pISet       = (PDJP_ITEM)pbQueries;
      ULONG          cbDrivData;
      PDRIVDATA      pDrivData   = (PDRIVDATA)NULL;
      BOOL           fError      = FALSE;
      ULONG          ulMaxSize;

      // Update the job properties with the input set list
      do
      {
         switch (pISet->ulProperty)
         {
         case DJP_SJ_ORIENTATION:
         {
            if (pISet->ulValue == DJP_ORI_PORTRAIT)
            {
               pJobProp->Orientation = OR_PORTRAIT;
            }
            else if (pISet->ulValue == DJP_ORI_LANDSCAPE)
            {
               pJobProp->Orientation = OR_LANDSCAPE;
            }
            else
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
            }
            break;
         }

         case DJP_SJ_PAPERSIZE:
         {
            USHORT  usMask = 0;

            switch (pISet->ulValue)
            {
              case DJP_PSI_LETTER:           usMask = MEDSZ_A;   break;
              case DJP_PSI_TABLOID:          usMask = MEDSZ_B;   break;
              case DJP_PSI_CSHEET:           usMask = MEDSZ_C;   break;
              case DJP_PSI_DSHEET:           usMask = MEDSZ_D;   break;
              case DJP_PSI_ESHEET:           usMask = MEDSZ_E;   break;
              case DJP_PSI_A4:               usMask = MEDSZ_A4;  break;
              case DJP_PSI_A3:               usMask = MEDSZ_A3;  break;
              case DJP_PSI_A2:               usMask = MEDSZ_A2;  break;
              case DJP_PSI_A1:               usMask = MEDSZ_A1;  break;
              case DJP_PSI_A0:               usMask = MEDSZ_A0;  break;
              case DJP_PSI_ARCHITECT_BSHEET: usMask = MEDSZ_AB;  break;
              case DJP_PSI_ARCHITECT_CSHEET: usMask = MEDSZ_AC;  break;
              case DJP_PSI_ARCHITECT_DSHEET: usMask = MEDSZ_AD;  break;
              case DJP_PSI_ARCHITECT_ESHEET: usMask = MEDSZ_AE;  break;
            }

            if (!usMask)
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
               DBPRINTF (("Invalid form size %d\n", pISet->ulValue));
            }
            else
            {
               pJobProp->Size = FormIndex (usMask);
            }
            break;
         }

         case DJP_CJ_FORM:
         {
            PDJPT_FORM      pDJPForm  = DJP_ELEMENTP (*pISet, DJPT_FORM);
            BOOL            fFound    = FALSE;
            HCINFO          Form;
            PDEVICEINFO     pDevice;
            USHORT          usMask;

            pDevice = &PlotterClass[pPDevice->Plotter].DeviceInfo;

            for (usMask = 0x8000; usMask; usMask >>= 1)
            {
               if (pDevice->fsMediaSizeSupport & usMask)
               {
                  GetPageData (pPDevice, (FormIndex (usMask)), &Form, pJobProp);

                  fFound = FormsMatch (pDJPForm, &Form);
                  if (fFound)
                     break;
               }
            }

            if (!fFound)
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
               DBPRINTF (("Invalid form size!\n"));
            }
            else
            {
               pJobProp->Size = FormIndex (usMask);
            }
            break;
         }

         case DJP_CJ_MIXEDFORMS:
         {
            PDJPT_MIXEDFORMS pDJPMXForm = DJP_ELEMENTP (*pISet, DJPT_MIXEDFORMS);
            BOOL             fFound     = FALSE;
            HCINFO           Form;
            PDEVICEINFO      pDevice;
            USHORT           usMask;

            if (-1 != pDJPMXForm->lStartRange ||
                -1 != pDJPMXForm->lEndRange    )
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
               DBPRINTF (("Invalid range!\n"));
            }

            pDevice = &PlotterClass[pPDevice->Plotter].DeviceInfo;

            for (usMask = 0x8000; usMask; usMask >>= 1)
            {
               if (pDevice->fsMediaSizeSupport & usMask)
               {
                  GetPageData (pPDevice, (FormIndex (usMask)), &Form, pJobProp);

                  fFound = FormsMatch (&pDJPMXForm->djpfmForm, &Form);
                  if (fFound)
                     break;
               }
            }

            if (!fFound)
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
               DBPRINTF (("Invalid form size!\n"));
            }
            else
            {
               pJobProp->Size = FormIndex (usMask);
            }
            break;
         }

         case DJP_SJ_COPIES:
         {
            INT iValue = (INT)pISet->ulValue;

            if (0 >= iValue || iValue > FL_UNCOLLATEDMASK + 1)
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
               DBPRINTF (("Invalid # copies %d!\n", iValue));
            }
            else
            {
               pJobProp->usFlags = ((pISet->ulValue - 1) &  FL_UNCOLLATEDMASK) |
                                   (pJobProp->usFlags    & ~FL_UNCOLLATEDMASK);
            }
            break;
         }

         case DJP_SJ_FEED:
         {
            if (DJP_FED_AUTOMATIC == pISet->ulValue)
            {
               pJobProp->PaperFeed = TRUE;
            }
            else if (DJP_FED_MANUAL == pISet->ulValue)
            {
               pJobProp->PaperFeed = FALSE;
            }
            else
            {
               pISet->lType = DJP_ERROR_OUT_OF_RANGE;
            }
            break;
         }

         case DJP_NONE:
            // Nothing to do!
            break;

         case DJP_CJ_RESOLUTION:
         case DJP_SJ_BITSPERPEL:
         case DJP_SJ_COLOR:
         case DJP_SJ_PRINTQUALITY:
         case DJP_SJ_FONTDOWNLOADING:
         case DJP_SJ_DUPLEX:
         case DJP_SJ_COLLATE:
         case DJP_SJ_SCALING:
         case DJP_SJ_TRAYTYPE:
         case DJP_SJ_MEDIA:
         case DJP_SJ_FORMFEEDCONTROL:
         default:
            pISet->lType = DJP_ERROR_NOT_SUPPORTED;
            DBPRINTF (("Unknow set '%s' = %d!\n",
                        pszProperty (pISet->ulProperty),
                        pISet->ulProperty));
            break;
         }

         // Move to the set item
         pISet = DJP_NEXT_STRUCTP (pISet);

      } while (pISet->ulProperty != DJP_NONE);

      /* Copy the updated job properties onto the user's job properties
      */
      cbDrivData = cbParm2;
      pDrivData  = (PDRIVDATA)pbParm2;
      ulMaxSize  = min (pGoodDrivData->cb, cbDrivData);
      memcpy (pDrivData, pGoodDrivData, ulMaxSize);
      pDrivData->cb = ulMaxSize;

      if (fError)
      {
         GplErrSetWarning (PMERR_DATATYPE_ENTRY_INVALID);
         lResult = DEV_ERROR;
      }
      else
      {
         lResult = DEV_OK;
      }
      break;
   }

   case DEVPE_DEFAULTJOBPROPERTIES:
   {
      INT            cbQueries   = cbParm1;
      PBYTE          pbQueries   = pbParm1;
      PDJP_ITEM      pISet       = (PDJP_ITEM)pbQueries;
      ULONG          cbDrivData;
      PDRIVDATA      pDrivData   = (PDRIVDATA)NULL;
      PDEFPROFILE    pDefProfile = (PDEFPROFILE)NULL;
      PDEVICEINFO    pDeviceInfo = (PDEVICEINFO)NULL;
      BOOL           fError      = FALSE;
      ULONG          ulMaxSize;

      pDefProfile = &PlotterClass[pPDevice->Plotter].DefProfile;
      pDeviceInfo = &PlotterClass[pPDevice->Plotter].DeviceInfo;

      // Update the job properties with the input set list
      do
      {
         /* NOTE:
         **    This should match get_defaults () !!!
         */
         switch (pISet->ulProperty)
         {
         case DJP_SJ_ORIENTATION:
            pJobProp->Orientation = OR_LANDSCAPE;
            break;

         case DJP_SJ_PAPERSIZE:
         case DJP_CJ_FORM:
            pJobProp->Size = pDefProfile->sSize;
            break;

         case DJP_SJ_COPIES:
            pJobProp->usFlags = (0                    &  FL_UNCOLLATEDMASK) |
                                (pJobProp->usFlags    & ~FL_UNCOLLATEDMASK);
            break;

         case DJP_SJ_FEED:
            pJobProp->PaperFeed = pDeviceInfo->PaperFeedSupport;
            break;

         case DJP_NONE:
            // Nothing to do!
            break;

         case DJP_CJ_MIXEDFORMS:
         case DJP_CJ_RESOLUTION:
         case DJP_SJ_BITSPERPEL:
         case DJP_SJ_COLOR:
         case DJP_SJ_PRINTQUALITY:
         case DJP_SJ_FONTDOWNLOADING:
         case DJP_SJ_DUPLEX:
         case DJP_SJ_COLLATE:
         case DJP_SJ_SCALING:
         case DJP_SJ_TRAYTYPE:
         case DJP_SJ_MEDIA:
         case DJP_SJ_FORMFEEDCONTROL:
         default:
            fError = TRUE;
            pISet->lType = DJP_ERROR_NOT_SUPPORTED;
            DBPRINTF (("Unknow set '%s' = %d!\n",
                        pszProperty (pISet->ulProperty),
                        pISet->ulProperty));
            break;
         }

         // Move to the set item
         pISet = DJP_NEXT_STRUCTP (pISet);

      } while (pISet->ulProperty != DJP_NONE);

      /* Copy the updated job properties onto the user's job properties
      */
      cbDrivData = cbParm2;
      pDrivData  = (PDRIVDATA)pbParm2;
      ulMaxSize  = min (pGoodDrivData->cb, cbDrivData);
      memcpy (pDrivData, pGoodDrivData, ulMaxSize);
      pDrivData->cb = ulMaxSize;

      lResult = DEV_OK;
      break;
   }

   default:
      lResult = DEVPE_NOTIMPLEMENTED;
      break;
   }

depart:

   return lResult;
}

SHORT
FormIndex (USHORT usMask)
{
   INT iValue = 0;

   /* To convert from a MEDSZ_ to the job property Size
   ** convert the bit that is set into a 0 based bit position.
   */
   while (usMask)
   {
      iValue++;
      usMask >>= 1;
   }

   return iValue - 1;
}

LONG
ReturnFormID (USHORT usFormBit)
{
   switch (usFormBit)
   {
   case MEDSZ_A:   return DJP_PSI_LETTER;
   case MEDSZ_B:   return DJP_PSI_TABLOID;
   case MEDSZ_C:   return DJP_PSI_CSHEET;
   case MEDSZ_D:   return DJP_PSI_DSHEET;
   case MEDSZ_E:   return DJP_PSI_ESHEET;
   case MEDSZ_A4:  return DJP_PSI_A4;
   case MEDSZ_A3:  return DJP_PSI_A3;
   case MEDSZ_A2:  return DJP_PSI_A2;
   case MEDSZ_A1:  return DJP_PSI_A1;
   case MEDSZ_A0:  return DJP_PSI_A0;
   case MEDSZ_AB:  return DJP_PSI_ARCHITECT_BSHEET;
   case MEDSZ_AC:  return DJP_PSI_ARCHITECT_CSHEET;
   case MEDSZ_AD:  return DJP_PSI_ARCHITECT_DSHEET;
   case MEDSZ_AE:  return DJP_PSI_ARCHITECT_ESHEET;
   case MEDSZ_R24: return -1; // not supported
   case MEDSZ_R36: return -1; // not supported
   default:        return -1; // unknown
   }
}

PSZ
pszProperty (ULONG ulProperty)
{
   switch (ulProperty)
   {
   case DJP_NONE:               return "DJP_NONE";
   case DJP_SJ_ORIENTATION:     return "DJP_SJ_ORIENTATION";
   case DJP_CJ_RESOLUTION:      return "DJP_CJ_RESOLUTION";
   case DJP_SJ_BITSPERPEL:      return "DJP_SJ_BITSPERPEL";
   case DJP_SJ_COLOR:           return "DJP_SJ_COLOR";
   case DJP_SJ_PRINTQUALITY:    return "DJP_SJ_PRINTQUALITY";
   case DJP_CJ_FORM:            return "DJP_CJ_FORM";
   case DJP_SJ_PAPERSIZE:       return "DJP_SJ_PAPERSIZE";
   case DJP_SJ_TRAYTYPE:        return "DJP_SJ_TRAYTYPE";
   case DJP_SJ_MEDIA:           return "DJP_SJ_MEDIA";
   case DJP_SJ_MEDIA_COLOR:     return "DJP_SJ_MEDIA_COLOR";
   case DJP_CJ_MIXEDFORMS:      return "DJP_CJ_MIXEDFORMS";
   case DJP_SJ_FONTDOWNLOADING: return "DJP_SJ_FONTDOWNLOADING";
   case DJP_SJ_DUPLEX:          return "DJP_SJ_DUPLEX";
   case DJP_SJ_COLLATE:         return "DJP_SJ_COLLATE";
   case DJP_SJ_FEED:            return "DJP_SJ_FEED";
   case DJP_SJ_COPIES:          return "DJP_SJ_COPIES";
   case DJP_SJ_SCALING:         return "DJP_SJ_SCALING";
   case DJP_SJ_FORMFEEDCONTROL: return "DJP_SJ_FORMFEEDCONTROL";
   case DJP_SJ_N_UP:            return "DJP_SJ_N_UP";
   default:                     return "*** unknown ***";
   }
}

PSZ
pszType (LONG lType)
{
   switch (lType)
   {
   case DJP_NONE:                return "DJP_NONE";
   case DJP_ALL:                 return "DJP_ALL";
   case DJP_CURRENT:             return "DJP_CURRENT";
   case DJP_ERROR_NOT_SUPPORTED: return "DJP_ERROR_NOT_SUPPORTED";
   case DJP_ERROR_OUT_OF_RANGE:  return "DJP_ERROR_OUT_OF_RANGE";
   case DJP_ERROR_NOT_ENUM:      return "DJP_ERROR_NOT_ENUM";
   case DJP_ERROR_INV_PARMS:     return "DJP_ERROR_INV_PARMS";
   default:                      return "*** unknown ***";
   }
}

BOOL
FormsMatch (PDJPT_FORM pS, PHCINFO pD)
{
   if (0 != strcmp (pS->szFormname, pD->szFormname))
      return FALSE;

   return TRUE;
}

BOOL
FillInFormInfo (PDJPT_FORM   pDJPForm,
                PPDEVICE     pPDevice,
                SHORT        Size,
                PPROFILEINFO pJobProp)
{
   PDEVICEINFO     pDevice;
   USHORT          usMask;
   LONG            lRet;

   // Set up the device *nfo pointer
   usMask = 1 << Size;
   pDevice = &PlotterClass[pPDevice->Plotter].DeviceInfo;

   // Clear out the returned structure...
   memset (pDJPForm, 0, sizeof (*pDJPForm));

   // Fill in form stuff
   GetPageData (pPDevice, Size, &pDJPForm->hcInfo, pJobProp);

   strncpy (pDJPForm->szFormname,
            pDJPForm->hcInfo.szFormname,
            sizeof (pDJPForm->szFormname));
   strncpy (pDJPForm->szDisplayFormname,
            pDJPForm->hcInfo.szFormname,
            sizeof (pDJPForm->szDisplayFormname));

   lRet = ReturnFormID (usMask);
   if (-1 != lRet)
      pDJPForm->djppsFormID = lRet;
   else
      pDJPForm->djppsFormID = 0;

   return TRUE;
}
