/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Omni Driver Query Functions                             */
/* SOURCE NAME    : query.c                                                 */
/* AUTHOR         : Matt Rutkowski & Mark Hamzy                             */
/* DESCRIPTION    : provides entry points for:                              */
/*                                                                          */
/*                  OS/2 API called             Omni Handling Function      */
/*                  ===============             ======================      */
/*                                                                          */
/*                  DevQueryCaps();             QueryDeviceCaps();          */
/*                  DevQueryHardcopyCaps();     QueryHardcopyCaps();        */
/*                                              QueryDeviceBitmaps();       */
/*                                              QueryDevResource2();        */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @WORDPRO- 04/10/96 - MFR [IBM] - Changed behavior of CAPS_PHYS_COLORS    */
/*                      value returned for mono printers to please Lotus.   */
/*                                                                          */
/*                                                                          */
/****************************************************************************/

#define INCL_WINMENUS
#define INCL_DOS
#define INCL_PM
#include <os2.h>

#define INCL_GRE_DEVICE
#define INCL_DDIMISC
#define INCL_WINSEI
#define INCL_WINSHELLDATA
#define INCL_GPIERRORS
// @DBCS
#define INCL_VMANDDI
#include <ddi.h>
#include <pmddi.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"

#define DEFAULT_PITCH        10
#define DEFAULT_POINTSIZE    12

/****************************************************************************/
/* PROCEDURE NAME : QueryDeviceBitmaps                                      */
/*                                                                          */
/* AUTHOR         : Monte Copeland  & Matt Rutkowski                        */
/* DATE WRITTEN   : 06/10/93                                                */
/* DESCRIPTION    : Returns all bitmap formats supported by the device      */
/*                                                                          */
/* PARAMETERS:      HDC     hdc                                             */
/*                  PLONG   paOutData                                       */
/*                  LONG    cOutData                                        */
/*                  PDDC    pddc                                            */
/*                  ULONG   ulFunction                                      */
/*                                                                          */
/* RETURN VALUES:   LONG    lrc                                             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/
LONG ENGENTRY
QueryDeviceBitmaps (HDC    hdc,
                    PLONG  paOutData,
                    LONG   cOutData,
                    PDDC   pddc,
                    ULONG  ulFunction)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   REGREC      regrec;
   ULONG       ulException;
   LONG        lrc;

#ifdef DEBUG
   PDEBUGINFO  pDbg           = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* EXCEPTION MANAGEMENT                                                   */
   /*------------------------------------------------------------------------*/
   REGISTERHANDLER (regrec, globals.hModule);
   ulException = setjmp (regrec.jmp);
   if (ulException)
   {
      // clean up here
      assertstring ("Exception caught!\n");
      CheckForTermination (&regrec, ulException, pddc);

      // error result
      lrc = FALSE;
      goto depart;
   }

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bQUERYDEVICEBITMAPS, "QueryDeviceBitmaps(): Enter\n"));

   ENTERDDC (pddc);

   lrc = GreQueryDeviceBitmaps (pddc->hdcShadow, paOutData, cOutData);
   assertF (lrc);

depart:
   EXITDDC (pddc);
   UNREGISTERHANDLER (regrec);

   DBPRINTIF ((pDbg->bQUERYDEVICEBITMAPS, "QueryDeviceBitmaps(): Exit\n"));

   return lrc;

} /* end QueryDeviceBitmaps */

/****************************************************************************/
/* PROCEDURE NAME : QueryDeviceCaps                                         */
/*                                                                          */
/* AUTHOR         : Monte Copeland & Matt Rutkowski [IBM]                   */
/* DATE WRITTEN   : 11/13/93                                                */
/* DESCRIPTION    : Returns various capabilities supported by the device    */
/*                  associated with the HDC passed in.                      */
/*                                                                          */
/*                  This function is invoked by the DevQueryCaps() API      */
/*                  from OS/2 PM applications.                              */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      HDC     hdc                                             */
/*                  LONG    lIndex                                          */
/*                  PLONG   paOutData                                       */
/*                  LONG    cOutData                                        */
/*                  PDDC    pddc                                            */
/*                  ULONG   ulFunction                                      */
/*                                                                          */
/* RETURN VALUES:   LONG    lrc                                             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/
LONG ENGENTRY
QueryDeviceCaps (HDC    hdc,
                 LONG   lIndex,
                 PLONG  paOutData,
                 LONG   cOutData,
                 PDDC   pddc,
                 ULONG  ulFunction)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   REGREC      regrec;
   ULONG       ulException;
   ULONG       lCap;
   LONG        lrc;
   LONG        i;
   BOOL        fOK;
   PLONG       plCap;
   PLONG       alShadowCaps, plShadowCaps;

#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* EXCEPTION MANAGEMENT                                                   */
   /*------------------------------------------------------------------------*/
   REGISTERHANDLER (regrec, globals.hModule);
   ulException = setjmp (regrec.jmp);
   if (ulException)
   {
      assertstring ("Exception caught!\n");

      // clean up here
      CheckForTermination (&regrec, ulException, pddc);

      // error result
      GplErrSetSevereError (PMERR_INV_LENGTH_OR_COUNT);

      lrc = FALSE;
      goto depart;
   }

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bQUERYDEVICECAPS, "QueryDeviceCaps(): Enter; lIndex = %d\n", lIndex));

   assertF (pddc);
   assertF (pddc->pdb);

   // check args;
   if (  lIndex < 0
      || cOutData  < 1
      || !paOutData
      )
      RAISEEXCEPTION (XCPT_USER);

   if (GRE_22 > globals.ulGreVersion)
   {
      // Not the 2.2 Engine

      // get the capabilities of the shadow DC
      alShadowCaps = (PLONG)GplMemoryAlloc (pddc->pdb->hmcbHeap,
                                            sizeof (LONG) * cOutData);
      assertF (alShadowCaps);

      // Get capabilities of current display driver which we are
      // using for our shadow DC to do rendering for us
      fOK = DevQueryCaps (pddc->hdcShadow, lIndex, cOutData, alShadowCaps);
      assertF (fOK);
   }

   // for each capability requested
   for (i = 0, lCap = lIndex; i < cOutData; i++, lCap++)
   {
      plCap = paOutData + i;

      if (GRE_22 > globals.ulGreVersion)
      {
         // Not the 2.2 Engine
         plShadowCaps = alShadowCaps + i;
         DBPRINTIF ((pDbg->bQUERYDEVICECAPS, "QueryDeviceCaps(): lCap = %d, plShadowCaps = %d\n",
                     lCap, *plShadowCaps));
      }

      InnerQueryCaps (lCap, plCap, plShadowCaps, pddc->pdb);
   }

   if (GRE_22 > globals.ulGreVersion)
   {
      // Not the 2.2 Engine
      GplMemoryFree (alShadowCaps);
   }

   lrc = TRUE;

   DBPRINTIF ((pDbg->bQUERYDEVICECAPS, "QueryDeviceCaps(): Exit\n"));

depart:
   UNREGISTERHANDLER (regrec);

#ifndef DEBUG
   fOK++;                       // Avoid compiler warnings
#endif

   return lrc;

} /* end QueryDeviceCaps */

VOID ENGENTRY
InnerQueryCaps (LONG lCap, PLONG plCap, PLONG plShadowCaps, PDEVICEBLOCK pdb)
{
   DBPRINTIF ((FALSE, "InnerQueryCaps(): Enter; lCap == %lu; plCap = %x\n", lCap, plCap));
   DBPRINTIF ((FALSE, "InnerQueryCaps(): plShadowCap = %x\n", lCap, plShadowCaps));

   /*--------------------------------*/
   /* Return what SHADOW DC supports */
   /*--------------------------------*/
   switch (lCap)
   {
   // @TBD-move marker height ???
   case CAPS_BACKGROUND_MIX_SUPPORT:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         *plCap = CAPS_BM_OVERPAINT | CAPS_BM_LEAVEALONE;
      break;
   }

   case CAPS_BITMAP_FORMATS:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         *plCap = 2;
      break;
   }

   case CAPS_COLORS:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         // simulated "logical" color count
         *plCap = Power (2, pdb->pPrintMode->usLogBitCount);
      break;
   }

   case CAPS_COLOR_BITCOUNT:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         // bpp for simulated "logical" colors
         *plCap = pdb->pPrintMode->usLogBitCount;
      break;
   }

   case CAPS_COLOR_INDEX:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         // simulated "logical" colors
         *plCap = Power (2, pdb->pPrintMode->usLogBitCount) - 1;
      break;
   }

   case CAPS_COLOR_PLANES:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         *plCap = 1;
      break;
   }

   case CAPS_COLOR_TABLE_SUPPORT:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         // Unsupported
         *plCap = 0;
      break;
   }

   case CAPS_FOREGROUND_MIX_SUPPORT:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         *plCap = CAPS_FM_OR             |
                  CAPS_FM_OVERPAINT      |
                  CAPS_FM_XOR            |
                  CAPS_FM_LEAVEALONE     |
                  CAPS_FM_AND            |
                  CAPS_FM_GENERAL_BOOLEAN;
      break;
   }

   case CAPS_RASTER_CAPS:
   {
      if (GRE_22 > globals.ulGreVersion)
         // these capabilities of the printer are the same as the shadow DC
         *plCap = *plShadowCaps;
      else
         *plCap = CAPS_RASTER_BITBLT  |
                  CAPS_RASTER_SET_PEL |
                  CAPS_RASTER_FONTS   ;
      break;
   }

   case CAPS_PHYS_COLORS:
   {
      if (  pdb->pPrintMode->ulColorTech == COLOR_TECH_MONO
         || pdb->pPrintMode->ulColorTech == COLOR_TECH_K
         )
      {
         /* @WORDPRO
         **      HACK      - as requested by Lotus, we for mono printers only
         ** report we have 2 physical colors so that Lotus can show the user
         ** monochrome options, especially their own grayscaling (default on
         ** of course) because ours is not good enough; ya right >:\    - MFR
         */
         *plCap = 2;
      }
      else
      {
         /* Use simulated "logical" colors since we grayscale for mono and
         ** dither for color (original method used for all color techs.)
         */
         *plCap = Power (2, pdb->pPrintMode->usBitsPerPel );
      }
      break;
   }

   case CAPS_MARKER_HEIGHT:
   {
      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
         *plCap = pdb->pResInfo->ulYRes/DEFAULT_PITCH;
      else
         *plCap = pdb->pResInfo->ulXRes/DEFAULT_PITCH;

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      break;
   }

   case CAPS_MARKER_WIDTH:
   {
      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
         *plCap = pdb->pResInfo->ulXRes/DEFAULT_PITCH;
      else
         *plCap = pdb->pResInfo->ulYRes/DEFAULT_PITCH;

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      break;
   }

   case CAPS_SMALL_CHAR_HEIGHT:
   case CAPS_SMALL_CHAR_WIDTH:
   {
      *plCap = 0;
      break;
   }

   case CAPS_CHAR_HEIGHT:
   case CAPS_GRAPHICS_CHAR_HEIGHT:
   {

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
         *plCap = (DEFAULT_POINTSIZE * pdb->pResInfo->ulYRes) / 72;
      else
         *plCap = (DEFAULT_POINTSIZE * pdb->pResInfo->ulXRes) / 72;

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      break;
   }

   case CAPS_CHAR_WIDTH:
   case CAPS_GRAPHICS_CHAR_WIDTH:
   {
      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
         *plCap = (DEFAULT_POINTSIZE * pdb->pResInfo->ulXRes) / 72;
      else
         *plCap = (DEFAULT_POINTSIZE * pdb->pResInfo->ulYRes) / 72;

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      break;
   }

   case CAPS_VIO_LOADABLE_FONTS:
   {
      *plCap = 0;
      break;
   }

   // @TBD - move these CAPS to pdevice structure
   case CAPS_ADDITIONAL_GRAPHICS:
   {
      if (  (pdb->ulType == OD_MEMORY)
         && (GRE_22 > globals.ulGreVersion)
         )
         *plCap = *plShadowCaps
                   &~CAPS_FONT_OUTLINE_DEFAULT
                   &~CAPS_FONT_IMAGE_DEFAULT
                   &~CAPS_FONT_OUTLINE_MANAGE
                   &~CAPS_FONT_IMAGE_MANAGE
                   &~CAPS_ENHANCED_FONTMETRICS
                   | CAPS_CLIP_FILLS ;
      else
         *plCap = 0;
      break;
   }

   // @TBD - vary according to device/system generic font
   // based on lmatch and GENERIC flag
   case CAPS_HORIZONTAL_FONT_RES:
   case CAPS_VERTICAL_FONT_RES:
   {
      ULONG ulRes;

      // return cap in pels per meter; there are 39.37 inches per meter
      // Get current dpi from Resolution selected in Job Props
      // and use to calculate pels/meter
      assertF (pdb->pResInfo);

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
      {
         if (CAPS_HORIZONTAL_FONT_RES == lCap)
            ulRes = pdb->pResInfo->ulXRes;
         else
            ulRes = pdb->pResInfo->ulYRes;
      }
      else
      {
         if (CAPS_HORIZONTAL_FONT_RES == lCap)
            ulRes = pdb->pResInfo->ulYRes;
         else
            ulRes = pdb->pResInfo->ulXRes;
      }

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      *plCap = ulRes;
      break;
   }

   case CAPS_FAMILY:
   {
      // customary to return OD_DIRECT (5) for this
      *plCap = OD_DIRECT;
      break;
   }

   case CAPS_IO_CAPS:
   {
      // device supports output (2)
      *plCap = CAPS_IO_SUPPORTS_OP;
      break;
   }

   case CAPS_TECHNOLOGY:
   {
      *plCap = pdb->pDevice->ulOS2DeviceTechnology;
      break;
   }

   case CAPS_DRIVER_VERSION:
   {
      *plCap = pdb->pDriver->ulDriverVersion;
      break;
   }

   case CAPS_HEIGHT:
   {
      // this is the height for the default form or for
      // form specified in DevOpen Parms (FORM=)
      // as setup in enable( FILL_PHYSICAL)
      *plCap = pdb->pFormInfo->hcInfo.yPels;
      DBPRINTF (("InnerQueryCaps(): CAPS_HEIGHT; lCap == %lu; *plCap = %x\n", lCap, *plCap));
      break;
   }

   case CAPS_WIDTH:
   {
      // this is the width for the default form or for
      // form specified in DevOpen Parms (FORM=)
      // as setup in enable( FILL_PHYSICAL)
      *plCap = pdb->pFormInfo->hcInfo.xPels;
      DBPRINTF (("InnerQueryCaps(): CAPS_WIDTH; lCap == %lu; *plCap = %x\n", lCap, *plCap));
      break;
   }

   case CAPS_HEIGHT_IN_CHARS:
   case CAPS_WIDTH_IN_CHARS:
   {
      // assume 6 lines per inch of 12 point text on Letter
      // assume 80 chars per line of 12 point text on Letter
      // TBD - @TTY

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
      {
         if (CAPS_HEIGHT_IN_CHARS == lCap)
            *plCap = 66;
         else
            *plCap = 80;
      }
      else
      {
         if (CAPS_HEIGHT_IN_CHARS == lCap)
            *plCap = 80;
         else
            *plCap = 66;
      }

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      break;
   }

   case CAPS_VERTICAL_RESOLUTION:
   case CAPS_HORIZONTAL_RESOLUTION:
   {
      ULONG ulRes;

      // return cap in pels per meter; there are 39.37 inches per meter
      // Get current dpi from Resolution selected in Job Props
      // and use to calculate pels/meter
      assertF (pdb->pResInfo);

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      if (ORIENTATION_PORTRAIT == pdb->pJobProperties->ulOrientation)
      {
         if (CAPS_HORIZONTAL_RESOLUTION == lCap)
            ulRes = pdb->pResInfo->ulXRes;
         else
            ulRes = pdb->pResInfo->ulYRes;
      }
      else
      {
         if (CAPS_HORIZONTAL_RESOLUTION == lCap)
            ulRes = pdb->pResInfo->ulYRes;
         else
            ulRes = pdb->pResInfo->ulXRes;
      }

      if (pdb->ulNupPages == 2)
         SWAP (pdb->pResInfo->ulXRes, pdb->pResInfo->ulYRes); 

      //@239651:There is a round mistake for a pels per meter value computation.
      //      *plCap = ulRes * INCHES_PER_METER;
      
      *plCap = (ulRes * 3937)/100;
      break;
   }

   /*--------------------------------*/
   /* Return UNSUPPORTED caps        */
   /*--------------------------------*/
   // @TBD - move down device font stuff
   case CAPS_DEVICE_FONTS:
   case CAPS_DEVICE_FONT_SIM:
   case CAPS_DEVICE_POLYSET_POINTS:
   case CAPS_DEVICE_WINDOWING:
   case CAPS_GRAPHICS_SUBSET:
   case CAPS_GRAPHICS_VECTOR_SUBSET:
   case CAPS_GRAPHICS_VERSION:
   case CAPS_LINEWIDTH_THICK:
   case CAPS_MOUSE_BUTTONS:
   case CAPS_WINDOW_BYTE_ALIGNMENT:
   default:
   {
      // unsupported capabilities
      *plCap = 0;
      DBPRINTF (("InnerQueryCaps(): Unsupported CAP: lCap == %lu\n", lCap));
      break;
   }
   }

} /* end InnerQueryCaps */

//----------------------------------------------------------------------------------------

LONG ENGENTRY
QueryDevResource2 (HDC            hdc,
                   ULONG          lType,
                   ULONG          id,
                   PDDC           pddc,
                   ULONG          ulFunction)
{
   REGREC      regrec;
   ULONG       ulException;
   LONG        lrc;
#ifdef DEBUG
   PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

   REGISTERHANDLER (regrec, globals.hModule);
   ulException = setjmp (regrec.jmp);
   if (ulException)
   {
      CheckForTermination (&regrec, ulException, pddc);

      // error result
      GplErrSetError (PMERR_INV_LENGTH_OR_COUNT);

      lrc = GPI_ALTERROR;
      goto depart;
   }

   DBPRINTIF ((pDbg->bQUERYDEVRESOURCE2, "QueryDevResource2(): Enter\n"));

   // none available
   lrc = 0;

depart:
   UNREGISTERHANDLER (regrec);

   DBPRINTIF ((pDbg->bQUERYDEVRESOURCE2, "QueryDevResource2(): Exit; lrc = %d\n", lrc));

   return lrc;

} /* end QueryDeviceResource2 */


/****************************************************************************/
/* PROCEDURE NAME : QueryHardcopyCaps                                       */
/*                                                                          */
/* AUTHOR         : Matt Rutkowski [IBM]                                    */
/* DATE WRITTEN   : 11/03/93                                                */
/* DESCRIPTION    : Returns # forms supported by a device and form          */
/*                  form information (HCINFO structures)                    */
/*                                                                          */
/*                  This function is invoked by the DevQueryHardcopyCaps()  */
/*                  API from OS/2 PM applications.                          */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      HDC     hdc                                             */
/*                  LONG    lStart                                          */
/*                  LONG    cCount                                          */
/*                  PHCINFO pInfo                                           */
/*                  PDDC    pddc                                            */
/*                  ULONG   ulFunction                                      */
/*                                                                          */
/* RETURN VALUES:   LONG    lrc                                             */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/
LONG ENGENTRY
QueryHardcopyCaps (HDC     hdc,
                   LONG    lStart,
                   LONG    cCount,
                   PHCINFO pInfo,
                   PDDC    pddc,
                   ULONG   ulFunction)
{
   /*------------------------------------------------------------------------*/
   /* LOCAL VARIABLES                                                        */
   /*------------------------------------------------------------------------*/
   REGREC          regrec;
   ULONG           ulException;
   LONG            lrc;
   LONG            i;
   FORMINFO2       fi2;
   PFORMINFO2      pForms;
   USHORT          usUserForms;
   PDEVICEINFO     pDeviceLcl;
   PDRIVERINFO     pDriverLcl;
   USHORT          usTotalForms;
   ULONG           ulDeviceForms;
   LONG            lUserStart;
   LONG            cDriver,
                   cUser;
   PUSERFORM       pUserForm;
   USERFORM        UserForm;
   ULONG           ulCurrentFormID;
#ifdef DEBUG
   PDEBUGINFO      pDbg              = &globals.DebugInfo;
#endif

   /*------------------------------------------------------------------------*/
   /* EXCEPTION MANAGEMENT                                                   */
   /*------------------------------------------------------------------------*/
   REGISTERHANDLER (regrec, globals.hModule);
   ulException = setjmp (regrec.jmp);
   if (ulException)
   {
      assertstring ("Exception caught!\n");
      CheckForTermination (&regrec, ulException, pddc);

      // error result
      GplErrSetError (PMERR_INV_LENGTH_OR_COUNT);

      lrc = DQHC_ERROR;
      goto depart;
   }

   /*------------------------------------------------------------------------*/
   /* BEGIN CODE                                                             */
   /*------------------------------------------------------------------------*/
   DBPRINTIF ((pDbg->bQUERYHARDCOPYCAPS,
               "QueryHardcopyCaps(): Enter; lStart = %d, cCount = %d\n",
               lStart, cCount));

   assertF (pddc);
   assertF (pddc->pdb);
   assertF (pddc->pdb->pDevice);

   // get pDevice & pDriver from pdb for easier dereference
   pDeviceLcl = pddc->pdb->pDevice;
   pDriverLcl = pddc->pdb->pDriver;

   // add device defined forms to user defined to get total # we have
   ulDeviceForms = pDeviceLcl->ulNumForms;
   usUserForms   = pDeviceLcl->pUserDefData->usNumForms;

   usTotalForms  = ulDeviceForms + usUserForms;

   // User wants to know how many forms we support
   if (cCount == 0)
   {
      // return count of hard copy forms supported for this device
      lrc = usTotalForms;
   }
   else
   {
      // get current default form from DRIVDATA (Job Props)
      // and use for matching forms to mark HCAPS_CURRENT
      GetIDsFromConnID (pDriverLcl,
                        pDeviceLcl,
                        pddc->pdb->pJobProperties->ulDefConnID,
                        NULL,
                        &ulCurrentFormID,
                        NULL);

      // Validate user request for forms is within range
      if (lStart >= usTotalForms)
      {
         // start index is out of range
         lrc = DQHC_ERROR;
      }
      else
      {
         cDriver = min (ulDeviceForms - lStart, cCount);
         cDriver = max (cDriver, 0);

         //DBPRINTF (("QueryHardcopyCaps(): cDriver = %d\n", cDriver));

         if (cDriver > 0)
         {
            pForms = pDeviceLcl->pFORMS + lStart;

            // for every driver defined form we can return
            for (i = 0; i < cDriver; i++)
            {
               FormInfoFromID (pddc->pdb->pDriver,
                               pDeviceLcl,
                               pForms->ulFormID,
                               &fi2);
               FormNameFromID (pddc->pdb->pDriver,
                               pDeviceLcl,
                               fi2.hcInfo.szFormname,
                               pForms->ulFormID);

               // Mark this form if it is part of a connection as selectable
               MarkIfSelectableCurrentForm (pddc->pdb->pDriver,
                                            pDeviceLcl,
                                            pForms->ulFormID,
                                            &fi2);

               DBPRINTIF ((pDbg->bQUERYHARDCOPYCAPS, "QueryHardcopyCaps(): Adding Driver defined Form = %s\n", fi2.hcInfo.szFormname));

               CalculateFormSize (pddc->pdb, pddc->pdb->pJobProperties, &fi2);

               // set HCAPS_CURRENT based on current job props if none set first in array
               if (pForms->ulFormID == ulCurrentFormID)
                  fi2.hcInfo.flAttributes |= HCAPS_CURRENT;

               *pInfo = fi2.hcInfo;

               pInfo++;
               pForms++;
            }
         }

         cUser = cCount - cDriver;  // # requested - # fulfilled by driver defs
         cUser = min (cUser, usUserForms ); // limit request to what we have

         // DBPRINTF (("QueryHardcopyCaps(): cUser = %d\n", cUser));

         // if we still have user forms to return
         if (cUser > 0)
         {
            // calculate start index if we have forms to process
            lUserStart = max (lStart - ulDeviceForms , 0);

 ///////////DBPRINTF (("QueryHardcopyCaps(): lUserStart = %d\n", lUserStart));

            // get pointer to user forms
            pUserForm = pDeviceLcl->pUserDefData->pUserFORMS;
            assertF (pUserForm);

            // @TBD - advance user def list pointer to new start position
            for (i = 0; i < lUserStart; i++)
            {
               pUserForm = pUserForm->pNextForm;
               assertF (pUserForm);
            }

            // for every driver defined form we can return
            for (i = 0; i < cUser; i++)
            {
               // Copy in the data to private space..
               UserForm = *pUserForm;

               // Mark this form if it is part of a connection as selectable
               MarkIfSelectableCurrentForm (pddc->pdb->pDriver,
                                            pDeviceLcl,
                                            UserForm.fiUser.ulFormID,
                                            &UserForm.fiUser);

               CalculateFormSize (pddc->pdb,
                                  pddc->pdb->pJobProperties,
                                  &UserForm.fiUser);

               // set HCAPS_CURRENT based on current job props if none set first in array
               if (UserForm.fiUser.ulFormID == ulCurrentFormID)
                  UserForm.fiUser.hcInfo.flAttributes |= HCAPS_CURRENT;

               // @TBD - access userdef structs
               *pInfo = UserForm.fiUser.hcInfo;

               DBPRINTIF ((pDbg->bQUERYHARDCOPYCAPS, "QueryHardcopyCaps(): Adding User defined form: = %s\n", pInfo->szFormname));

               if (pUserForm->pNextForm != NULL)
               {
                   pUserForm = pUserForm->pNextForm;
                   assertF (pUserForm);

                   pInfo++;
               }
            }
         }

         // return to user how many we can return
         lrc = min (usTotalForms - lStart, cCount);
      }
   }

depart:
   UNREGISTERHANDLER (regrec);

   DBPRINTIF ((pDbg->bQUERYHARDCOPYCAPS, "QueryHardcopyCaps(): Exit; lrc = %d\n", lrc));

   return lrc;

} /* end QueryHardcopyCaps */
