/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Font Dialog Control Functions                           */
/* SOURCE NAME    : FONTS.C                                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/26/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @3632   - 04/08/98 - UCC [IBMJ]- Add 2 flag for DevFnt sup use.          */
/*                                  fRecoding is for HiRes devfnt print prob*/
/*                                  fOldDevfnt is for backward compatibility*/
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/* @JPN2H99- 08/01/99 - UCC [IBMJ]- Printer properties option page support  */
/*                                                                          */
/****************************************************************************/

// os2 includes
#define INCL_DOS
#define INCL_DOSERRORS
#define INCL_PM
#include <os2.h>

#define INCL_VMANDDI
#include <ddi.h>
#include <pmddi.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"
#include "fonts.h"

/* defines...
*/
#define FIXEDFMFSIZE (sizeof (FOCAFONT) + 8)

/* Global variables...
*/
LONG valPointSizes[]  = { 8, 10, 12, 24, 32 };

/* Function prototypes...
*/
LONG          FillDevFontMetricsStrucs       (PDEVICEBLOCK    pdb,
                                              PDEVFONTMETRICS pdfm,
                                              PFOCAMETRICSEXT pfmeStart);
VOID          ScaleType3Font                 (PFOCAFONT       pff,
                                              PRESINFO        pRes,
                                              LONG            lPointSize);
VOID          FixupFontInformation           (PFOCAFONT       pff,
                                              PFONTINFO2      pFontInfo,
                                              PRESINFO        pRes,
                                              LONG            lFMFSize,
                                              INT             lPointSize);

/****************************************************************************/
/* PROCEDURE NAME : FillFocaMetricsListBox                                  */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Uses font info already returned from GpiQueryFonts      */
/*                  to display fontmetric for a selected font in a listbox. */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Clear out font metric information listbox           */
/*                  (b) Interpret each field of the (FONTMETRIC) structure  */
/*                      at index 'i' of the list returned by                */
/*                      GpiQueryFonts(); an insert interpreted data into    */
/*                      a listbox.                                          */
/*                                                                          */
/* PARAMETERS:      (HWND)    hwnd         - caller's window handle         */
/*                  (USHORT)  i            - Index into (FONTMETRICS) list  */
/*                                           for font whose metrics this    */
/*                                           function will display          */
/*                  (SHORT *) MinPointSize - Minimum point size of indexed  */
/*                                           font.                          */
/*                  (SHORT *) MaxPointSize - Maximum point size of indexed  */
/*                                           font.                          */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @DBCS       09-16-95 - Matt Rutkowski -  Function adapted from FontTest  */
/*                        (i.e FT32.EXE) from displaying FONTMETRICS to     */
/*                        instead display FOCAMETRICS which are similar.    */
/*                                                                          */
/****************************************************************************/
VOID
FillFocaMetricsListBox (HWND         hwnd,
                        ULONG        ulLBoxID,
                        PDLGINSTANCE pdi,
                        PFOCAFONT    pFocaFont)
{
   CHAR         szBuffer[256];
   HWND         hwndLB         = WinWindowFromID (hwnd, ulLBoxID);
   PFOCAMETRICS pfm;

   WinSendDlgItemMsg (hwnd, ulLBoxID, LM_DELETEALL, NULL, NULL);

   if (!pFocaFont)
      return;

   pfm = &pFocaFont->fmMetrics;

   /*------------------------------------------------------*/
   /* Facename                                             */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "Facename = %s", pfm->szFacename);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* Familyname                                           */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "Familyname = %s", pfm->szFamilyname);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* Identity                                             */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "Identity = %d", pfm->ulIdentity);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* Registry ID                                          */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "Registry ID = %d", pfm->usRegistryId);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* CodePage                                             */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usCodePage = %d", pfm->usCodePage);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* Defn                                                 */
   /*------------------------------------------------------*/
   /* FM_DEFN_OUTLINE         0x0001                       */
   /* FM_DEFN_IFI             0x0002                       */
   /* FM_DEFN_WIN             0x0004                       */
   /* FM_DEFN_GENERIC         0x8000                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "fsDefn = 0x%x", pfm->fsDefn);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   if (pfm->fsDefn & FM_DEFN_OUTLINE)
   {
      sprintf (szBuffer, "      FM_DEFN_OUTLINE (0x%x)", FM_DEFN_OUTLINE);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }
   else
   {
      sprintf (szBuffer, "      BITMAP FONT");
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsDefn & FM_DEFN_IFI)
   {
      sprintf (szBuffer, "      FM_DEFN_IFI (0x%x)", FM_DEFN_IFI);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsDefn & FM_DEFN_WIN)
   {
      sprintf (szBuffer, "      FM_DEFN_WIN (0x%x)", FM_DEFN_WIN);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsDefn & FM_DEFN_GENERIC)
   {
      sprintf (szBuffer, "      FM_DEFN_GENERIC (0x%x)", FM_DEFN_GENERIC);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   /*------------------------------------------------------*/
   /* Type                                                 */
   /*------------------------------------------------------*/
   /* FM_TYPE_FIXED           0x0001                       */
   /* FM_TYPE_LICENSED        0x0002                       */
   /* FM_TYPE_KERNING         0x0004                       */
   /* FM_TYPE_DBCS            0x0010                       */
   /* FM_TYPE_MBCS            0x0018                       */
   /* FM_TYPE_64K             0x8000                       */
   /* FM_TYPE_ATOMS           0x4000                       */
   /* FM_TYPE_FAMTRUNC        0x2000                       */
   /* FM_TYPE_FACETRUNC       0x1000                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "fsTypeFlags = 0x%x", pfm->fsTypeFlags);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   if (pfm->fsTypeFlags & FM_TYPE_FIXED)
   {
      sprintf (szBuffer, "      FM_TYPE_FIXED (0x%x)", FM_TYPE_FIXED);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_LICENSED)
   {
      sprintf (szBuffer, "      FM_TYPE_LICENSED (0x%x)", FM_TYPE_LICENSED);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_KERNING)
   {
      sprintf (szBuffer, "      FM_TYPE_KERNING (0x%x)", FM_TYPE_KERNING);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_DBCS)
   {
      sprintf (szBuffer, "      FM_TYPE_DBCS (0x%x)", FM_TYPE_DBCS);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_MBCS)
   {
      sprintf (szBuffer, "      FM_TYPE_MBCS (0x%x)", FM_TYPE_MBCS);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_64K)
   {
      sprintf (szBuffer, "      FM_TYPE_64K (0x%x)", FM_TYPE_64K);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_ATOMS)
   {
      sprintf (szBuffer, "      FM_TYPE_ATOMS (0x%x)", FM_TYPE_ATOMS);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_FAMTRUNC)
   {
      sprintf (szBuffer, "      FM_TYPE_FAMTRUNC (0x%x)", FM_TYPE_FAMTRUNC);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsTypeFlags & FM_TYPE_FACETRUNC)
   {
      sprintf (szBuffer, "      FM_TYPE_FACETRUNC (0x%x)", FM_TYPE_FACETRUNC);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   /*------------------------------------------------------*/
   /* Selection                                            */
   /*------------------------------------------------------*/
   /* FM_SEL_ITALIC           0x0001                       */
   /* FM_SEL_UNDERSCORE       0x0002                       */
   /* FM_SEL_NEGATIVE         0x0004                       */
   /* FM_SEL_OUTLINE          0x0008                       */
   /* FM_SEL_STRIKEOUT        0x0010                       */
   /* FM_SEL_BOLD             0x0020                       */
   /* FM_SEL_ISO9241_TESTED   0x0040                       */
   /*------------------------------------------------------*/

   sprintf (szBuffer, "fsSelectionFlags = 0x%x", pfm->fsSelectionFlags);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   if (pfm->fsSelectionFlags & FM_SEL_OUTLINE)
   {
      sprintf (szBuffer, "      FM_SEL_OUTLINE (0x%x)", FM_SEL_OUTLINE);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsSelectionFlags & FM_SEL_ITALIC)
   {
      sprintf (szBuffer, "      FM_SEL_ITALIC (0x%x)", FM_SEL_ITALIC);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsSelectionFlags & FM_SEL_UNDERSCORE)
   {
      sprintf (szBuffer, "      FM_SEL_UNDERSCORE (0x%x)", FM_SEL_UNDERSCORE);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsSelectionFlags & FM_SEL_NEGATIVE)
   {
      sprintf (szBuffer, "      FM_SEL_NEGATIVE (0x%x)", FM_SEL_NEGATIVE);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsSelectionFlags & FM_SEL_STRIKEOUT)
   {
      sprintf (szBuffer, "      FM_SEL_STRIKEOUT (0x%x)", FM_SEL_STRIKEOUT);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsSelectionFlags & FM_SEL_BOLD)
   {
      sprintf (szBuffer, "      FM_SEL_BOLD (0x%x)", FM_SEL_BOLD);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   if (pfm->fsSelectionFlags & FM_SEL_ISO9241_TESTED)
   {
      sprintf (szBuffer, "      FM_SEL_ISO9241_TESTED (0x%x)", FM_SEL_ISO9241_TESTED);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   /*------------------------------------------------------*/
   /* Capabilities                                         */
   /*------------------------------------------------------*/
   /* FM_CAP_NOMIX            0x0001                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "fsCapabilities = 0x%x", pfm->fsCapabilities);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   if (pfm->fsCapabilities & FM_CAP_NOMIX)
   {
      sprintf (szBuffer, "      FM_CAP_NOMIX (0x%x)", FM_CAP_NOMIX);
      WinInsertLboxItem (hwndLB, LIT_END, szBuffer);
   }

   /*------------------------------------------------------*/
   /* MaxBaselineExt                                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yMaxBaselineExt = %d", pfm->yMaxBaselineExt);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* AveCharWidth                                         */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "xAveCharWidth = %d", pfm->xAveCharWidth);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* NominalPointSize                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usNominalPointSize = %d", pfm->usNominalPointSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* MinimumPointSize                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usMinimumPointSize = %d", pfm->usMinimumPointSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* MaximumPointSize                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usMaximumPointSize = %d", pfm->usMaximumPointSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* EmHeight                                             */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yEmHeight = %d", pfm->yEmHeight);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* XHeight                                              */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yXHeight = %d", pfm->yXHeight);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* MaxAscender                                          */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yMaxAscender = %d", pfm->yMaxAscender);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* MaxDescender                                         */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yMaxDescender = %d", pfm->yMaxDescender);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* LowerCaseAscent                                      */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yLowerCaseAscent = %d", pfm->yLowerCaseAscent);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* LowerCaseDescent                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yLowerCaseDescent = %d", pfm->yLowerCaseDescent);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* InternalLeading                                      */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yInternalLeading = %d", pfm->yInternalLeading);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* ExternalLeading                                      */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yExternalLeading = %d", pfm->yExternalLeading);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* MaxCharInc                                           */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "xMaxCharInc = %d", pfm->xMaxCharInc);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* EmInc                                                */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "xEmInc = %d", pfm->xEmInc);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* sCharSlope                                           */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "sCharSlope = %d", pfm->sCharSlope);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* InLineDir                                            */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "sInlineDir = %d", pfm->sInlineDir);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* CharRot                                              */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "sCharRot = %d", pfm->sCharRot);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* FirstChar                                            */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usFirstChar = %d", pfm->usFirstChar);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* LastChar                                             */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usLastChar = %d", pfm->usLastChar);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* DefaultChar                                          */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usDefaultChar = %d", pfm->usDefaultChar);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* BreakChar                                            */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usBreakChar = %d", pfm->usBreakChar);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* KerningPairs                                         */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usKerningPairs = %d", pfm->usKerningPairs);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* FamilyClass                                          */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "sFamilyClass = %d", pfm->sFamilyClass);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* WeightClass                                          */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usWeightClass = %d", pfm->usWeightClass);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* WidthClass                                           */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "usWidthClass = %d", pfm->usWidthClass);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SubscriptXSize                                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySubscriptXSize = %d", pfm->ySubscriptXSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SubscriptYSize                                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySubscriptYSize = %d", pfm->ySubscriptYSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SubscriptXOffset                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySubscriptXOffset = %d", pfm->ySubscriptXOffset);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SubscriptYOffset                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySubscriptYOffset = %d", pfm->ySubscriptYOffset);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SuperscriptXSize                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySuperscriptXSize = %d", pfm->ySuperscriptXSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SuperscriptYSize                                     */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySuperscriptYSize = %d", pfm->ySuperscriptYSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SuperscriptXOffset                                   */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySuperscriptXOffset = %d", pfm->ySuperscriptXOffset);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* SuperscriptYOffset                                   */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "ySuperscriptYOffset = %d", pfm->ySuperscriptYOffset);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* UnderscoreSize                                       */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yUnderscoreSize = %d", pfm->yUnderscoreSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* UnderscorePosition                                   */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yUnderscorePosition = %d", pfm->yUnderscorePosition);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* StrikeoutSize                                        */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yStrikeoutSize = %d", pfm->yStrikeoutSize);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

   /*------------------------------------------------------*/
   /* StrikeoutPosition                                    */
   /*------------------------------------------------------*/
   sprintf (szBuffer, "yStrikeoutPosition = %d", pfm->yStrikeoutPosition);
   WinInsertLboxItem (hwndLB, LIT_END, szBuffer);

} /* end FillFontMetricsListBox */

/****************************************************************************/
/* PROCEDURE NAME : ScaleType3Font                                          */
/*                                                                          */
/* AUTHOR         : Mark H.                                                 */
/* DATE WRITTEN   : 5/1/98                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/****************************************************************************/
VOID
ScaleType3Font (PFOCAFONT pff, PRESINFO pRes, LONG lPointSize)
{
#define SCALEDOWN(x,ptSize,newRes,oldRes)  if (0 != x) x = ( ( (float)(x)           \
                                                             * (float)ptSize        \
                                                             * (float)newRes        \
                                                             )                      \
                                                           / ((float)oldRes * 72.0) \
                                                           )
#define SCALEUP(x,ptSize,newRes,oldRes)    if (0 != x) x = ( ( (float)(x)           \
                                                             * (float)ptSize        \
                                                             * (float)newRes        \
                                                             + 0.5                  \
                                                             )                      \
                                                           / ((float)oldRes * 72.0) \
                                                           )
#define SCALEABC(a,b,c) {                                                      \
                        long int total;                                        \
                        total = a + b + c;                                     \
                        SCALEDOWN (a,     lPointSize, sxDeviceRes, sxFontRes); \
                        SCALEDOWN (c,     lPointSize, sxDeviceRes, sxFontRes); \
                        SCALEUP   (total, lPointSize, sxDeviceRes, sxFontRes); \
                        total -= a;                                            \
                        total -= c;                                            \
                        b      = total;                                        \
                        }

   PFOCAMETRICS          pfm;
   PFONTDEFINITIONHEADER pfdh;
   SHORT                 sxDeviceRes,
                         sxFontRes;
   SHORT                 syDeviceRes,
                         syFontRes;
   PTYPE3CHARDEF         pT3Def;
   BOOL                  fCompressed = FALSE;
#ifdef DEBUG
   BOOL                  fDump       = FALSE;
#endif
   register INT          j;

   pfm         = &pff->fmMetrics;
   pfdh        = &pff->fdDefinitions;

   sxDeviceRes = (SHORT)pRes->ulXRes;
   syDeviceRes = (SHORT)pRes->ulYRes;
   sxFontRes   = pfm->xDeviceRes;
   syFontRes   = pfm->yDeviceRes;

   pfm->fsDefn &= ~FM_DEFN_OUTLINE;

   SCALEDOWN (pfm->yEmHeight,           lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yXHeight,            lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yMaxAscender,        lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yMaxDescender,       lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yLowerCaseAscent,    lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yLowerCaseDescent,   lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yInternalLeading,    lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yExternalLeading,    lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->xAveCharWidth,       lPointSize, sxDeviceRes, sxFontRes);
   SCALEDOWN (pfm->xMaxCharInc,         lPointSize, sxDeviceRes, sxFontRes);
   SCALEDOWN (pfm->xEmInc,              lPointSize, sxDeviceRes, sxFontRes);
   SCALEDOWN (pfm->yMaxBaselineExt,     lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySubscriptXSize,     lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySubscriptYSize,     lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySubscriptXOffset,   lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySubscriptYOffset,   lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySuperscriptXSize,   lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySuperscriptYSize,   lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySuperscriptXOffset, lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->ySuperscriptYOffset, lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yUnderscoreSize,     lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yUnderscorePosition, lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yStrikeoutSize,      lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfm->yStrikeoutPosition,  lPointSize, syDeviceRes, syFontRes);

   pfm->usNominalPointSize = lPointSize * 10;
   pfm->usMinimumPointSize = lPointSize * 10;
   pfm->usMaximumPointSize = lPointSize * 10;

   SCALEDOWN (pfdh->xCellWidth,      lPointSize, sxDeviceRes, sxFontRes);
   SCALEDOWN (pfdh->yCellHeight,     lPointSize, syDeviceRes, syFontRes);
   SCALEDOWN (pfdh->xCellIncrement,  lPointSize, sxDeviceRes, sxFontRes);
   SCALEDOWN (pfdh->pCellBaseOffset, lPointSize, sxDeviceRes, sxFontRes);

   // Special case
   SCALEABC (pfdh->xCellA, pfdh->xCellB, pfdh->xCellC);

   if (0 < pfm->loffExtension)
      fCompressed = TRUE;
   else
      fCompressed = FALSE;

   pT3Def = (PTYPE3CHARDEF)(pff + 1);

#ifdef DEBUG
   if (fDump)
   {
      DBPRINTF (("Foca metrics for %s@%d\n",
                 pfm->szFacename,
                 pfm->usNominalPointSize / 10));
   }
#endif

   if (fCompressed)
   {
      PFOCAMETRICSEXT pfmeRangeInfo;

      pfmeRangeInfo = (PFOCAMETRICSEXT)((LONG)pff + pfm->loffExtension);

      for (j = 0; j < pfmeRangeInfo->ulGlyphCount; j++, pT3Def++)
      {
         if (pT3Def->sASpace || pT3Def->usBSpace || pT3Def->sCSpace)
         {
            // Special case
            SCALEABC (pT3Def->sASpace, pT3Def->usBSpace, pT3Def->sCSpace);

            DBPRINTIF ((fDump,
                        "For glyph %3d, (a, b, c) (%3d, %3d, %3d) = %3d\n",
                        j,
                        pT3Def->sASpace,
                        pT3Def->usBSpace,
                        pT3Def->sCSpace,
                        pT3Def->sASpace + pT3Def->usBSpace + pT3Def->sCSpace));
         }
      }
   }
   else
   {
      for (j = 0; j < pff->fmMetrics.usLastChar; j++, pT3Def++)
      {
         if (pT3Def->sASpace || pT3Def->usBSpace || pT3Def->sCSpace)
         {
            // Special case
            SCALEABC (pT3Def->sASpace, pT3Def->usBSpace, pT3Def->sCSpace);

            DBPRINTIF ((fDump,
                        "For glyph %3d, (a, b, c) (%3d, %3d, %3d) = %3d\n",
                        j,
                        pT3Def->sASpace,
                        pT3Def->usBSpace,
                        pT3Def->sCSpace,
                        pT3Def->sASpace + pT3Def->usBSpace + pT3Def->sCSpace));
         }
      }
   }

   pfm->xDeviceRes = sxDeviceRes;
   pfm->yDeviceRes = syDeviceRes;
}

/****************************************************************************/
/* PROCEDURE NAME : ReturnNumberOfDeviceFonts                               */
/*                                                                          */
/* AUTHOR         : Mark H.                                                 */
/* DATE WRITTEN   : 5/1/98                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/****************************************************************************/
LONG
ReturnNumberOfDeviceFonts (PDRIVERINFO           pDriver,
                           PDEVICEINFO           pDevice)
{
   LONG                  lNumFonts       = (LONG)pDevice->usNumFonts;
   LONG                  lCurrentMatchID = -1;
   PFONTINFO2            pFontInfo;
   register INT          i;

   // for each supported font (indicated by Font ID)
   for (i = 0; i < pDevice->usNumFonts; i++)
   {
      // if Font ID found and string returned
      if (FontInfoFromID (pDriver, pDevice, pDevice->pulFONTS[i], &pFontInfo))
      {
         if (pFontInfo->ulType == FONT_TYPE_SCALABLE_SPECIAL)
         {
            /* @TODO
            ** Fix up the lmatch number so that the numbers are
            ** reasonable and valid.
            ** However, multiple devices cannot share different ranges
            ** at the same time because the lmatch numbers will be off!
            */
            pFontInfo->lDevMatch = lCurrentMatchID;

            lNumFonts       += dimof (valPointSizes) - 1;
            lCurrentMatchID -= dimof (valPointSizes) - 1;
         }

         lCurrentMatchID--;
      }
   }

   return lNumFonts;
}

/****************************************************************************/
/* PROCEDURE NAME : QueryDevFntMetrics                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-18-95                                                 */
/* DESCRIPTION    : New entry point for getting device font metrics         */
/*                                                                          */
/*                  This is utilized when the PM Graphics Engine supports   */
/*                  managing our device fonts (OS/2 for PowerPC).  We       */
/*                  can return an array of DEVFONTMETRICS, one for each     */
/*                  device font we want PM GRE to manage for us.            */
/*                                                                          */
/*                  The DEVFONTMETRICS structure can return font info to    */
/*                  the GRE in 2 formats.  The first method is to pass      */
/*                  the fully-qualified filename of a valid OS/2 font file. */
/*                  The second method is to supply the GRE with a           */
/*                  FOCAMETRICS structure with the needed font information. */
/*                  The DEVFONTMETRICS contains fields for either method.   */
/*                                                                          */
/*                  A valid font file means that OS/2 has a font driver     */
/*                  that knows how to read in the needed font data and      */
/*                  supply it to the GRE.  These can include Adobe Type 1,  */
/*                  .FON (OS/2 bitmap), and TrueType font.                  */
/*                                                                          */
/* PARAMETERS:      xInnerGre32Entry5 - The GRE entry point                 */
/*                  (PDDC)              pddc                                */
/*                  (PPDEVFONTMETRICS)  pdfm,                               */
/*                  (PULONG)            pulOptions                          */
/*                  (ULONG)             ulReserved1                         */
/*                  (ULONG)             ulReserved2                         */
/*                                                                          */
/* RETURN VALUES: (ULONG)ulNumFonts                                         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG APIENTRY
QueryDevFntMetrics (HDC    hdc,
                    PVOID *pv,
                    ULONG  ulOptions,
                    PDDC   pddc,
                    ULONG  ulR2)
{
   PDEVFONTMETRICS  *ppdfm       = (PDEVFONTMETRICS *)pv;
   PFNSUBCLASS       pfnSubClass = (PFNSUBCLASS)NULL;
   LONG              lNumFonts   = 0;
   PDEVICEBLOCK      pdb         = (PDEVICEBLOCK)NULL;
   PDEVICEINFO       pDevice     = (PDEVICEINFO)NULL;

   DBPRINTIF ((TRUE,
               "%s(): Enter hdc=%08X, pv=%08X, ulOptions=%08X, pddc=%08X, ulR2=%08X\n",
               __FUNCTION__, hdc, pv, ulOptions, pddc, ulR2));

   if (  !pddc
      || !pddc->pdb
      || !pddc->pdb->pDevice
      )
      // Error
      return 0;

   if (GRE_23C == globals.ulGreVersion)
   {
      // Broken engine!  Disable device fonts!
      return 0;
   }

   pdb         = pddc->pdb;
   pDevice     = pdb->pDevice;
   pfnSubClass = pDevice->pSubclassedFunctions;

   
   // Disable device fonts if Booklet or Nup is enabled
   // Can not coexist with Nup at this time 99.08.01
   // System or App will use default font.
   if ( pdb->ulNupPages >= 2 )
   {
      return 0;
   }

   // @JPN2H99 Check device font disabled flag
   if ( pdb->pPrnProperties->fOmniOptions & OMNI_OPTION_DEVICEFONT_OFF)
   {
      // return no fonts.
      return 0;
   }

   // get # Fonts defined
   lNumFonts = ReturnNumberOfDeviceFonts (pdb->pDriver, pdb->pDevice);

   DBPRINTIF ((TRUE, "%s(): lNumFonts = %d\n", __FUNCTION__, lNumFonts));

//@SCALABLEFONT - begin
// Check PMGRE version here. Before Fixpak 5 PMGRE sets ulOptions
// with its address pointer. So we can check current PMGRE version
// by checking ulOptions value.
#define DFO_ALLOC_FONTS 0
#define DFO_FREE_FONTS  1
#define DFO_QUERY_CAPS  2
#define DFO_LAST_VALUE  2

   
   if (ulOptions > DFO_LAST_VALUE)
   {
      // Old PMGRE just want us return metrics.
      ulOptions       = DFO_ALLOC_FONTS;
      pdb->fOldDevfnt = TRUE;
   }
   

   switch (ulOptions)
   {
   case DFO_ALLOC_FONTS:
   {
      if (lNumFonts > 0)
      {
         LONG    cbSize;

         // Notify Device file that the font information needs to be current
         // (or updated) for any queries from the GRE or PM applications
         // This includes CODEPAGE glyph mappings (from current process
         // codepages can be updated and font file names changed).
         if (GRE_23D > globals.ulGreVersion)
            cbSize = sizeof (OLDDEVFONTMETRICS) * lNumFonts;
         else
            cbSize = sizeof (DEVFONTMETRICS) * lNumFonts;

         if (!pddc->pdb->pdfm)
         {
            *ppdfm = (PDEVFONTMETRICS)GplMemoryAlloc (globals.pvSharedHeap,
                                                      cbSize);
            pddc->pdb->pdfm = *ppdfm;
         }
         else
         {
            *ppdfm = pddc->pdb->pdfm;
         }

         lNumFonts = FillDevFontMetricsStrucs (pdb,
                                               *ppdfm,
                                               (PFOCAMETRICSEXT)(*ppdfm + lNumFonts));

         // If device has subclassed functions and one of them is "Query Fonts"
         if (  pfnSubClass
            && pfnSubClass->pfnQueryFonts
            )
         {
             // Call subclassed function in device file (if supplied) to allow
             // device file to modify/update values within the PDEVFONTMETRICS
             // pSubclassedFunctions->pfnQueryFonts( );
             return lNumFonts;
         }
      }

      DBPRINTIF ((TRUE, "%s(): Exit-success with %d fonts\n", __FUNCTION__, lNumFonts));

      return lNumFonts;

//@SCALABLEFONT - begin
   }

   case DFO_FREE_FONTS:   // Free allocated memories
   {                      // Free pFocaFont allocated
#if 0
      // NOTE: Do not add this code without rewriting it and sanity checking it!
      //       The font logic code has changed a lot.
      //       And anyways, all of the data is freed already.  I don't know what
      //       real requirement called for the DFO_FREE_FONTS option.

//////LONG       i;
//////PFONTINFO2 pFontInfo;
//////
//////for( i=0; i < lNumFonts; i++ )
//////{
//////   // if Font ID found and string returned
//////   if( FontInfoFromID( pdb->pDriver, pDevice, pDevice->pulFONTS[i], &pFontInfo ) )
//////   {
//////      if( pFontInfo->pFocaFont)
//////      {
//////         GplMemoryFree( pFontInfo->pFocaFont);
//////         pFontInfo->pFocaFont = NULL;
//////      }
//////   }
//////}
#endif
   }
   break;

   case DFO_QUERY_CAPS:   // Return caps
   {
      // Default caps: all simulate by engine.
      ULONG ulFlag = 0L;

#if 0
      /* These are the allowable flags:
      **
      **    PDEVFNT_BOLD_SUPPORT
      **    PDEVFNT_ITALIC_SUPPORT
      **    PDEVFNT_SHEAL_SUPPORT
      **    PDEVFNT_ANGLE_SUPPORT
      **    PDEVFNT_OUTLINE_SUPPORT
      **    PDEVFNT_UNDERSCORE_SUPPORT
      **    PDEVFNT_STRIKEOUT_SUPPORT
      */
#endif

      if (  pfnSubClass
         && pfnSubClass->pfnQueryFonts
         )
      {
         // Ask plug-in its device font capabilities.
         // PMGRE will check the flag to switch its
         // simulation code.
         ulFlag = pfnSubClass->pfnQueryFonts (pddc, NULL, DFQF_QUERYCAPS);
      }

      return ulFlag;
   }
   }
//@SCALABLEFONT - end

   DBPRINTIF ((TRUE, "%s(): Exit-failure!\n", __FUNCTION__));

   return 0L;

} /* end QueryDevFntMetrics */

/****************************************************************************/
/* PROCEDURE NAME : FillDevFontMetricsStrucs                                */
/* AUTHOR         : Mark H.                                                 */
/* DATE WRITTEN   : 5/1/98                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
LONG
FillDevFontMetricsStrucs (PDEVICEBLOCK    pdb,
                          PDEVFONTMETRICS pdfmArray,
                          PFOCAMETRICSEXT pfmeStart)
{
   LONG                  lrc                = -1;  // -1 == error
   LONG                  lFontsFound        = 0;
   LONG                  lNumExpandedFonts;
   LONG                  lNumFontsInList;
   LONG                  lPointSize;
   PRESINFO              pRes               = pdb->pResInfo;
   PDRIVERINFO           pDriver            = pdb->pDriver;
   PDEVICEINFO           pDevice            = pdb->pDevice;
   PFONTINFO2            pFontInfo;
   PDEVFONTMETRICS       pdfm;
   LONG                  lFMFSize;
   register INT          j, i               = 0;

   pdfm = pdfmArray;

   // get # Fonts defined
   lNumFontsInList   = pDevice->usNumFonts;
   lNumExpandedFonts = ReturnNumberOfDeviceFonts (pdb->pDriver, pdb->pDevice);

   // If device has no device fonts
   if (0 == lNumExpandedFonts)
      // return error
      return lrc;

   // for each supported font (indicated by Font ID)
   for (i = 0, j = 0; j < lNumFontsInList; i++, j++)
   {
      // if Font ID found and string returned
      if (FontInfoFromID (pDriver, pDevice, pDevice->pulFONTS[j], &pFontInfo))
      {
         INT  iWhichPointSize = 0;
         BOOL fDone           = FALSE;

         
         if (pdb->fOldDevfnt == TRUE)
         {
            // Skip Scalable font when old PMGRE.
            if (pFontInfo->ulType == FONT_TYPE_SCALABLE)
            {
               continue;
            }
         }
         

         do
         {
            if (FONT_TYPE_SCALABLE_SPECIAL == pFontInfo->ulType)
               lPointSize = valPointSizes[iWhichPointSize];
            else if (FONT_TYPE_SCALABLE_SPECIAL2 == pFontInfo->ulType)
               lPointSize = pFontInfo->ulTypeExtra;
            else
               lPointSize = 0;

            // Check if exist pFoca, if not, then delete the entry.
            // Passing empty pFoca will trap PMGRE!!
            lFMFSize = ReadFocaMetricsFile (pdb->hmcbHeap,
                                            pFontInfo,
                                            &pdfm->pFoca,
                                            pRes,
                                            lPointSize);
            if (0 < lFMFSize)
            {
               lFontsFound++;

               pdfm->flOptions            = 0L;
               pdfm->pUniPanhose          = NULL;
               pdfm->achFileName[0]       = '\0';
               if (*pFontInfo->szFamilyName)
                  pdfm->pszFullFamilyName = pFontInfo->szFamilyName;
               else
                  pdfm->pszFullFamilyName = pdfm->pFoca->fmMetrics.szFamilyname;
               if (*pFontInfo->szFaceName)
                  pdfm->pszFullFaceName   = pFontInfo->szFaceName;
               else
                  pdfm->pszFullFaceName   = pdfm->pFoca->fmMetrics.szFacename;
               strcpy (pdfm->achGlyphListName, pFontInfo->szGlyphListName);
               pdfm->pfnDevQueryCodePage  = NULL;
               pdfm->lDevMatch            = pFontInfo->lDevMatch;

               if (GRE_23D <= globals.ulGreVersion)
               {
                  /* New engine.  First, tell the engine that we are using
                  ** the new fields.
                  */
                  pdfm->flOptions |= DEVFONT_EXTENSION;

                  /* Then, fill in additional fields
                  */
                  // Current size that we know about
                  pdfm->ulSize        = sizeof (DEVFONTMETRICS);
                  // Name of backup font
                  if (*pFontInfo->szBackupFontName)
                     pdfm->pszBackupFont = pFontInfo->szBackupFontName;
                  else
                     pdfm->pszBackupFont = pdfm->pszFullFaceName;

                  // Clear out device specific part of the dfm
                  memset (pdfm->aulDeviceSpecific,
                          0,
                          sizeof (pdfm->aulDeviceSpecific));
               }

               if (pdb->pJobProperties->ulDefFontID == pdfm->lDevMatch)
                  pdfm->flOptions |= DEVFONT_DEFAULT;

               if (FONT_TYPE_SCALABLE_SPECIAL == pFontInfo->ulType)
               {
                  pdfm->lDevMatch = pFontInfo->lDevMatch - iWhichPointSize;

                  if (0 != iWhichPointSize)
                     pdfm->flOptions &= ~DEVFONT_DEFAULT;
               }

///////////////DBPRINTF (("Foca metrics for %s@%d are at %08X\n",
///////////////           pFontInfo->szFaceName,
///////////////           pdfm->pFoca->fmMetrics.usNominalPointSize / 10,
///////////////           pdfm->pFoca));
            }

            iWhichPointSize++;

            fDone = (  (FONT_TYPE_SCALABLE_SPECIAL != pFontInfo->ulType)
                    || (  (FONT_TYPE_SCALABLE_SPECIAL == pFontInfo->ulType)
                       && (iWhichPointSize == dimof (valPointSizes))
                       )
                    );

            if (!fDone)
            {
               i++;
            }

            if (GRE_23D > globals.ulGreVersion)
               // Old engine
               pdfm = (PDEVFONTMETRICS)(((POLDDEVFONTMETRICS)pdfm) + 1);
            else
               // New engine
               pdfm++;

         } while (!fDone);
      }
#ifdef DEBUG
      else
      {
         assertstring ("Font match failed!\n");
      }
#endif
   }

   lrc = lFontsFound;

   pdb->ulNumDevFonts = lFontsFound;

   DBPRINTF (("%s(): Exit with %d device fonts\n", __FUNCTION__, lrc));

   return lrc;

} /* end FillDevFontMetricsStrucs */

/****************************************************************************/
/* PROCEDURE NAME : ReadFocaMetricsFile                                     */
/*                                                                          */
/* AUTHOR         : Mark H.                                                 */
/* DATE WRITTEN   : 5/1/98                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
LONG
ReadFocaMetricsFile (HMCB       hmcb,
                     PFONTINFO2 pFontInfo,
                     PFOCAFONT *ppFoca,
                     PRESINFO   pRes,
                     LONG       lPointSize)
{
   LONG       lFileSize       = 0;  // 0 == error (no bytes read)
   HFILE      hFile;
   ULONG      ulAction;
   ULONG      cbActual        = 0;
   SHORT      sEnd;
   CHAR       szFontPath[CCHMAXPATH];
   APIRET     rc;

   rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
   assertT (rc);

   // Is there a copy already loaded?
   if (pFontInfo->vpGlobalFocaFont)
   {
      assertT (*ppFoca);
      *ppFoca = (PFOCAFONT)GplMemoryAlloc (hmcb, pFontInfo->cbGlobalFocaFont);
      assertF (*ppFoca);

      // Copy the global font into the local font
      if (*ppFoca)
      {
         memcpy (*ppFoca, pFontInfo->vpGlobalFocaFont, pFontInfo->cbGlobalFocaFont);

         FixupFontInformation (*ppFoca,
                               pFontInfo,
                               pRes,
                               pFontInfo->cbGlobalFocaFont,
                               lPointSize);

         rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
         assertT (rc);

         return pFontInfo->cbGlobalFocaFont;
      }

      rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
      assertT (rc);

      // Error
      return 0;
   }

   rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
   assertT (rc);

   strcpy (szFontPath, globals.szModule);

   // Strip name 'OMNI.DRV' from end of fully-qualified path
   // in order to get directory where font files are located
   for (sEnd = strlen (globals.szModule) - 1; sEnd >= 0; sEnd--)
   {
      if ('\\' == globals.szModule[sEnd])
      {
         szFontPath[sEnd + 1] = 0;
         break;
      }
   }

   // get file name from pFonts
   strcat (szFontPath, pFontInfo->szFileName);

   // open input file to read in Foca Font data
   rc = DosOpen2 (szFontPath,
                  &hFile,
                  &ulAction,
                  0L,
                  0L,
                  1L,
                  (ULONG)0x20,
                  (PEAOP2)0L);
   assertT (rc);

   // If DosOpen successful
   if (NO_ERROR == rc)
   {
      ULONG    ulFileSize = 0;

      DBPRINTIF ((TRUE, "%s(): DosOpen2('%s') Success\n", __FUNCTION__, szFontPath));

      // Find the file size
      rc = DosSetFilePtr (hFile,
                          0,
                          FILE_END,
                          &ulFileSize);
      assertT (rc);

      if (  NO_ERROR == rc
         && 0 < ulFileSize
         )
      {
         assertT (*ppFoca);
         *ppFoca = (PFOCAFONT)GplMemoryAlloc (hmcb, ulFileSize);
         assertF (*ppFoca);

         if (*ppFoca)
         {
            ULONG   ulTmp = 0;

            rc = DosSetFilePtr (hFile,
                                0,
                                FILE_BEGIN,
                                &ulTmp);
            assertT (rc);
            rc = DosRead (hFile,
                          (PBYTE)*ppFoca,
                          ulFileSize,
                          &cbActual);
            assertT (rc);

            if (NO_ERROR == rc)
            {
               /* @TODO UCCTmp
               */
               SafeStrNCpy (pFontInfo->szFaceName,
                            (*ppFoca)->fmMetrics.szFacename,
                            sizeof ((*ppFoca)->fmMetrics.szFacename));
               SafeStrNCpy (pFontInfo->szFamilyName,
                            (*ppFoca)->fmMetrics.szFamilyname,
                            sizeof ((*ppFoca)->fmMetrics.szFamilyname));

               lFileSize += cbActual;

               rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
               assertT (rc);

               // Store the copy in global data
               pFontInfo->vpGlobalFocaFont = (PFOCAFONT)GplMemoryAlloc (globals.pvSharedHeap,
                                                                        ulFileSize);
               assertF (pFontInfo->vpGlobalFocaFont);

               if (pFontInfo->vpGlobalFocaFont)
               {
                  pFontInfo->cbGlobalFocaFont = ulFileSize;

                  memcpy (pFontInfo->vpGlobalFocaFont, *ppFoca, ulFileSize);

//////////////////DBPRINTF (("Global Foca metrics for %s@%d are at %08X\n",
//////////////////           pFontInfo->vpGlobalFocaFont->fmMetrics.szFacename,
//////////////////           pFontInfo->vpGlobalFocaFont->fmMetrics.usNominalPointSize / 10,
//////////////////           pFontInfo->vpGlobalFocaFont));
               }

               rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
               assertT (rc);

               FixupFontInformation (*ppFoca,
                                     pFontInfo,
                                     pRes,
                                     ulFileSize,
                                     lPointSize);
            }
         }
      }

      // Close the Foca Font data file
      rc = DosClose (hFile);
      assertT (rc);
   }
   else
   {
      // Failure
      DBPRINTIF ((TRUE, "%s(): DosOpen2('%s') FAILED!\n", __FUNCTION__, szFontPath));
   }

   // Reset Font Path to remove previous Font file name
   szFontPath[sEnd + 1] = '\0';

   return lFileSize;

} /* end ReadFocaMetricsFile */

/****************************************************************************/
/* PROCEDURE NAME : FixupFontInformation                                    */
/*                                                                          */
/* AUTHOR         : Mark H.                                                 */
/* DATE WRITTEN   : 5/1/98                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
FixupFontInformation (PFOCAFONT  pff,
                      PFONTINFO2 pFontInfo,
                      PRESINFO   pRes,
                      LONG       lFMFSize,
                      INT        lPointSize)
{
   PFOCAMETRICS          pfm;
   PFONTDEFINITIONHEADER pfdh;

   pfm  = &pff->fmMetrics;
   pfdh = &pff->fdDefinitions;

   // Resolution is changable but *.fmf file is coding with static
   // resoution value. So here we patch the metrics read from files,
   // so that PMGRE can handle the fonts correctly.
   // Maybe we have to redesign this later!!            -Y.Shih
   if (  (pFontInfo->ulType == FONT_TYPE_FIXED)
      && (  pfm->xDeviceRes != (SHORT)pRes->ulXRes
         || pfm->yDeviceRes != (SHORT)pRes->ulYRes
         )
      )
   {
      SHORT sxDeviceRes,
            sxFontRes;
      SHORT syDeviceRes,
            syFontRes;

      sxDeviceRes = (SHORT)pRes->ulXRes;
      syDeviceRes = (SHORT)pRes->ulYRes;
      sxFontRes   = pfm->xDeviceRes;
      syFontRes   = pfm->yDeviceRes;

      if (sxFontRes && syFontRes)
      {
         pfm->yEmHeight          =
         pfm->yXHeight           =
         pfm->yMaxAscender       =
         pfm->yLowerCaseAscent   =
         pfm->yMaxBaselineExt    =
         pfdh->yCellHeight       = pfdh->yCellHeight * syDeviceRes / syFontRes;

         pfm->xAveCharWidth      =
         pfm->xMaxCharInc        =
         pfm->xEmInc             =
         pfdh->xCellWidth        =
         pfdh->xCellIncrement    = pfdh->xCellWidth * sxDeviceRes / sxFontRes;

         pfm->yStrikeoutPosition = pfm->yStrikeoutPosition * syDeviceRes / syFontRes;
         pfm->yExternalLeading   = pfm->yExternalLeading * syDeviceRes / syFontRes;

         pfm->xDeviceRes         = sxDeviceRes;
         pfm->yDeviceRes         = syDeviceRes;
      }
   }
   /* HACK: Scale the font into a fixed size
   */
   else if (  (  FONT_TYPE_SCALABLE_SPECIAL  == pFontInfo->ulType
              || FONT_TYPE_SCALABLE_SPECIAL2 == pFontInfo->ulType
              )
           && (lFMFSize > FIXEDFMFSIZE)         //UCCTry
           )
   {
      ScaleType3Font (pff, pRes, lPointSize);
   }
}

/****************************************************************************/
/* PROCEDURE NAME : CreateUniqueFontID                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10-18-93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG
CreateUniqueFontID (PDEVICEINFO pDevice)
{
   ULONG     ulFontID = 0;
   PUSERFONT pUserFont;
   BOOL      bIDClash = FALSE;

   assertF (pDevice);
   assertF (pDevice->pUserDefData);

   // if we have user defined Fonts to check against
   if (pDevice->pUserDefData->usNumFonts > 0)
   {
      // sanity check pointers
      assertF (pDevice->pUserDefData->pUserFONTS);

      // as long as we don't exceed some large max ID number
      // we'll keep trying to assign a unique ID
      while (ulFontID < MAX_USER_DEF_ID)
      {
         // assign Font list pointer to local variable
         pUserFont = pDevice->pUserDefData->pUserFONTS;

         // for each entry in linked list of user defined Fonts
         while ((pUserFont != NULL) && !bIDClash)
         {
            if (pUserFont->fiUser.ulFontID == (ulFontID | DEF_TYPE_USER))
            {
               bIDClash = TRUE;
               break;
            }

            pUserFont = pUserFont->pNextFont;
         }

         // we had a clash try next ID
         if (bIDClash)
         {
            ulFontID++;
            bIDClash = FALSE;
         }
         else
         {
            // we have found a match let's get out of while loop
            break;
         }
      }
   }

   return ulFontID | DEF_TYPE_USER;

} /* end CreateUniqueFontID */

/****************************************************************************/
/* PROCEDURE NAME : AddUserDefinedFontEntry                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
AddUserDefinedFontEntry (PDEVICEINFO pDevice, PFONTINFO2 pNewFontDef)
{
   PUSERFONT pUserFont;
   BOOL      bSuccess = TRUE;
   BOOL      rc;

   DBPRINTF (("AddUserDefinedFontEntry(): Enter\n"));

   assertF (pDevice);
   assertF (pNewFontDef);

   // allocate user Font struct and add to current list
   pUserFont = (PUSERFONT)GplMemoryAlloc (globals.pvSharedHeap,
                                          sizeof (USERFONT));

   assertF (pUserFont);

   // if memory is allocated all right
   if (pUserFont)
   {
      /* Since we are altering global data, request a global semaphore
      ** for the changes we need to make...
      */
      rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
      assertT (rc);

      // Copy in new Font entry from one passed in
      pUserFont->pNextFont       = NULL;
      pUserFont->fiUser.ulFontID = pNewFontDef->ulFontID;

/// @DBCS
///   pUserFont->fiUser.fm       = pNewFontDef->fm;
      strcpy (pUserFont->fiUser.szCmdSelectFont, pNewFontDef->szCmdSelectFont);

/// @DBCS
///   DBPRINTF (("AddUserDefinedFontEntry(): Adding Font = %s\n",
///              pUserFont->fiUser.fm.szFacename));

      // Add new Font structure to linked list
      if (pDevice->pUserDefData->pUserFONTS == NULL)
      {
         // first Font being defined add at top of list
         pDevice->pUserDefData->pUserFONTS = pUserFont;
      }
      else
      {
         // add at end of list
         pDevice->pUserDefData->pLastUserFONTS->pNextFont = pUserFont;
      }

      pDevice->pUserDefData->pLastUserFONTS = pUserFont;
      pDevice->pUserDefData->usNumFonts++;

      rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
      assertT (rc);
   }

   DBPRINTF (("AddUserDefinedFontEntry(): Exit\n"));

#ifndef DEBUG
   rc++;       // Get rid of warning!
#endif

   return bSuccess;

} /* end AddUserDefinedFontEntry */

/****************************************************************************/
/* PROCEDURE NAME : DeleteUserDefinedFontEntry                              */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES  :                                                         */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
DeleteUserDefinedFontEntry (PDEVICEINFO pDevice, ULONG ulFontID)
{
   PUSERFONT pCurrent;
   PUSERFONT pLast    = NULL;
   BOOL      bSuccess = FALSE;
   BOOL      rc;

   /* Since we are altering global data, request a global semaphore
   ** for the changes we need to make...
   */
   rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
   assertT (rc);

   // search for Font ID in linked list and once found delete entry
   // from list and fix chain
   pCurrent = pDevice->pUserDefData->pUserFONTS;

   while (pCurrent != NULL)
   {
      // matching FontID
      if (pCurrent->fiUser.ulFontID == ulFontID)
      {
         // we are in middle of list chain
         if (pLast != NULL)
         {
            pLast->pNextFont = pCurrent->pNextFont;
         }
         else
         {
            // deleting first entry
            pDevice->pUserDefData->pUserFONTS = pCurrent->pNextFont;
         }

         GplMemoryFree (pCurrent);

         pDevice->pUserDefData->usNumFonts--;

         bSuccess = TRUE;

         break;
      }

      // advnce list pointers
      pLast    = pCurrent;
      pCurrent = pCurrent->pNextFont;
   }

   rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
   assertT (rc);

#ifndef DEBUG
   rc++;       // Get rid of warning!
#endif

   return bSuccess;

} /* end DeleteUserDefinedFontEntry */

/****************************************************************************/
/* PROCEDURE NAME : FreeUserDefinedFontList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID
FreeUserDefinedFontList (PDEVICEINFO pDevice)
{
   PUSERFONT pDelete, pNext;
   BOOL      rc;

   DBPRINTF (("FreeUserDefinedFontList(): Enter\n"));

   assertF (pDevice);

   if (!pDevice || !pDevice->pUserDefData)
      return;

   /* Since we are altering global data, request a global semaphore
   ** for the changes we need to make...
   */
   rc = DosRequestMutexSem (procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT);
   assertT (rc);

   // search for Font ID in linked list and once found delete entry
   // from list and fix chain
   pDelete = pDevice->pUserDefData->pUserFONTS;

   while (pDelete != NULL)
   {
      pNext = pDelete->pNextFont;

      GplMemoryFree (pDelete);

      pDelete = pNext;
   }

   // reset linked list values
   pDevice->pUserDefData->bFontsInit     = FALSE;
   pDevice->pUserDefData->pUserFONTS     = NULL;
   pDevice->pUserDefData->pLastUserFONTS = NULL;
   pDevice->pUserDefData->usNumFonts     = 0;

   rc = DosReleaseMutexSem (procdata.hmtxGlobalSem);
   assertT (rc);

#ifndef DEBUG
   rc++;       // Get rid of warning!
#endif

   DBPRINTF (("FreeUserDefinedFontList(): Exit\n"));

} /* FreeUserDefinedFontList */

/****************************************************************************/
/* PROCEDURE NAME : SaveUserDefinedFontList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
SaveUserDefinedFontList (PSZ pszAppName, PDEVICEINFO pDevice)
{
   USHORT     usFontsProcessed    = 0;
   USHORT     usTotalFonts;
   PUSERFONT  pCurrent;
   CHAR       szTemp[LEN_STRING];
   PFONTINFO2 pFont, pBase;
   ULONG      ulDataSize;
   ULONG      ulFontInfoSize      = sizeof (FONTINFO2);
   BOOL       bSuccess            = TRUE;

   // write out Num Fonts to INI
   usTotalFonts = pDevice->pUserDefData->usNumFonts;

   _itoa (usTotalFonts, szTemp, 10);

   PrfWriteProfileString (HINI_SYSTEMPROFILE,
                          pszAppName,
                          (PSZ)KEYNAME_NUM_USER_FONTS,
                          (PVOID)szTemp);

   // allocate memory to hold all Fontinfo and cast to FONTINFO2 struct
   ulDataSize = usTotalFonts * sizeof (FONTINFO2);

   // if we have Font strucs to allocate for
   if (ulDataSize > 0)
   {
      pBase = GplMemoryAlloc (0, ulDataSize);

      assertF (pBase);

      if (pBase)
      {
         pFont = pBase;

         // set up start of list pointer
         pCurrent = pDevice->pUserDefData->pUserFONTS;

         // while we have Fonts in linked list to process
         while (pCurrent != NULL && usFontsProcessed <= usTotalFonts)
         {
            // copy out to memory area
            memcpy (pFont, &(pCurrent->fiUser), ulFontInfoSize);

            // increment counter
            usFontsProcessed++;

            // advance buffer pointer
            pFont++;

            // advance list pointer
            pCurrent = pCurrent->pNextFont;
         }

         PrfWriteProfileData (HINI_SYSTEMPROFILE,
                              pszAppName,
                              (PSZ)KEYNAME_USER_FONTS,
                              (PVOID)pBase,
                              ulDataSize);

         GplMemoryFree (pBase);
      }
      else
      {
         bSuccess = FALSE;
      }
   }
   else
   {
      // overwrite any old entries with NULL
      PrfWriteProfileData (HINI_SYSTEMPROFILE,
                           pszAppName,
                           (PSZ)KEYNAME_USER_FONTS,
                           (PVOID)NULL,
                           ulDataSize);
   }

   return bSuccess;

} /* SaveUserDefinedFontList */

/****************************************************************************/
/* PROCEDURE NAME : ReadUserDefinedFontList                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ReadUserDefinedFontList (PSZ pszAppName, PDEVICEINFO pDevice)
{
   PFONTINFO2 pFont, pBase;
   ULONG      ulDataSize;
   BOOL       bSuccess            = TRUE;
   CHAR       szTemp[LEN_STRING];
   USHORT     usFontsProcessed    = 0;
   USHORT     usNumUserFonts      = 0;
   BOOL       bRC;

   DBPRINTF (("ReadUserDefinedFontList(): Enter; Appname = %s\n", pszAppName));

   assertF (pDevice);

   if (!pDevice->pUserDefData->bFontsInit)
   {
      ulDataSize = PrfQueryProfileString (HINI_SYSTEMPROFILE,
                                          pszAppName,
                                          (PSZ)KEYNAME_NUM_USER_FONTS,
                                          (PSZ)NULL,
                                          (PVOID)szTemp,
                                          LEN_STRING);
      // If we have an INI entry
      if (ulDataSize != 0)
      {
         // write out Num Fonts to INI
         usNumUserFonts = atoi (szTemp);

         if (usNumUserFonts > 0)
         {
            // allocate memory to hold all Fontinfo and cast to FONTINFO2 struct
            ulDataSize = usNumUserFonts * sizeof (FONTINFO2);

            pBase = GplMemoryAlloc (0, ulDataSize);

            assertF (pBase);

            if (pBase)
            {
               // read in profile data
               bRC = PrfQueryProfileData (HINI_SYSTEMPROFILE,
                                          pszAppName,
                                          (PSZ)KEYNAME_USER_FONTS,
                                          (PVOID)pBase,
                                          &ulDataSize);

               assertF (bRC);

               // if data read in correctly
               if (bRC)
               {
                  // init current entry pointer to each Font structure
                  pFont = pBase;

                  // while we have Fonts in linked buffer to process
                  while (usFontsProcessed < usNumUserFonts)
                  {
                     AddUserDefinedFontEntry (pDevice, pFont);

                     // increment counter
                     usFontsProcessed++;

                     // advance buffer pointer
                     pFont++;
                  }

                  // free memory
                  GplMemoryFree (pBase);
               }
               else
               {
                  bSuccess = FALSE;
               }
            }
            else
            {
               bSuccess = FALSE;
            }
         }
      }
      else
      {
         DBPRINTF (("ReadUserDefinedFontList(): No INI data\n"));
      }

      // indicate we have read in font data
      pDevice->pUserDefData->bFontsInit = TRUE;
   }

   DBPRINTF (("ReadUserDefinedFontList(): Exit\n"));

   return bSuccess;

} /* ReadUserDefinedFontList */

/****************************************************************************/
/* PROCEDURE NAME : FontNameFromID                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/23/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
FontNameFromID (PDRIVERINFO pDriver,
                PDEVICEINFO pDevice,
                PSZ         pszFontName,
                ULONG       ulFontID)
{
   ULONG      ulDefType;
   BOOL       bSuccess       = FALSE;
   ULONG      i;
   ULONG      ulNumDefined;
   PFONTINFO2 pFonts;
   PFOCAFONT  pFocaFont;

   // Find out how Font defined (in driver, in device, by user )
   ulDefType = MAKEULONG (0, HIUSHORT(ulFontID));

   switch (ulDefType)
   {
   case DEF_TYPE_DRIVER:  // No hi word value
   {
      ulNumDefined = pDriver->ulNumFonts;
      pFonts       = pDriver->pFONTS;

      // For now only driver table definitions allowed
      // loop through table of Fonts until matching Font ID
      // found
      for (i = 0; i < ulNumDefined; i++)
      {
         if (ulFontID == pFonts[i].ulFontID)
         {
            if (pFonts[i].vpGlobalFocaFont)
            {
               pFocaFont = pFonts[i].vpGlobalFocaFont;
               strcpy (pszFontName, pFocaFont->fmMetrics.szFacename);
               bSuccess = TRUE;
            }
            break;
         }
      }

      break;
   }

   case DEF_TYPE_DEVICE:
   {
      // @TBD - fill in???
      break;
   }

   case DEF_TYPE_USER:
   {
      PUSERFONT pUserFont;

      // assign Font list pointer to local variable
      pUserFont = pDevice->pUserDefData->pUserFONTS;

      // for each entry in linked list of user defined Fonts
      while (pUserFont != NULL)
      {
         if (pUserFont->fiUser.ulFontID == ulFontID)
         {
            if (pUserFont->fiUser.vpGlobalFocaFont)
            {
               pFocaFont = pUserFont->fiUser.vpGlobalFocaFont;
               strcpy (pszFontName, pFocaFont->fmMetrics.szFacename);
               bSuccess = TRUE;
            }
            break;
         }

         pUserFont = pUserFont->pNextFont;
      }

      break;
   }

   default:
   {
      assertstring ("Unhandled case!\n");
      break;
   }
   }

   return bSuccess;

} /* end FontNameFromID */

/****************************************************************************/
/* PROCEDURE NAME : FontInfoFromID                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/23/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
FontInfoFromID (PDRIVERINFO pDriver,
                PDEVICEINFO pDevice,
                ULONG       ulFontID,
                PPFONTINFO2 pFont)
{
   ULONG       ulDefType;
   BOOL        bSuccess     = FALSE;
   ULONG       i;
   ULONG       ulNumDefined;
   PFONTINFO2  pFonts;

   // Find out how Font defined (in driver, in device, by user )
   ulDefType = MAKEULONG (0, HIUSHORT (ulFontID));

   switch (ulDefType)
   {
   case DEF_TYPE_DRIVER:  // No hi word value
   {
      ulNumDefined = pDriver->ulNumFonts;
      pFonts       = pDriver->pFONTS;

      // For now only driver table definitions allowed
      // loop through table of Fonts until matching Font ID
      // found
      for (i = 0; i < ulNumDefined; i++)
      {
         if (ulFontID == pFonts[i].ulFontID)
         {
            *pFont   = &(pFonts[i]);
            bSuccess = TRUE;
            break;
         }
      }

      break;
   }

   case DEF_TYPE_DEVICE:
   {
      // @TBD - fill in???
      break;
   }

   case DEF_TYPE_USER:
   {
      PUSERFONT pUserFont;

      // assign Font list pointer to local variable
      pUserFont = pDevice->pUserDefData->pUserFONTS;

      // for each entry in linked list of user defined Fonts
      while (pUserFont != NULL)
      {
         if (pUserFont->fiUser.ulFontID == ulFontID)
         {
            *pFont = &pUserFont->fiUser;

            bSuccess = TRUE;
            break;
         }

         pUserFont = pUserFont->pNextFont;
      }

      break;
   }

   default:
   {
      assertstring ("Unhandled case!\n");
      break;
   }
   }

   return bSuccess;

} /* end FontInfoFromID */

/****************************************************************************/
/* PROCEDURE NAME : FillFontList                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/22/93                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @EXJOB  - 07/13/98 - UCC [IBMJ]- Expansible job data support             */
/*                                                                          */
/****************************************************************************/
VOID
FillFontList (HWND         hwndDlg,
              PDLGINSTANCE pdi,
              PDEVICEINFO  pDevice,
              ULONG        ulListboxID)
{
   PRESINFO   pRes            = pdi->pRes;
   USHORT     usNumFonts;
   USHORT     i;
   ULONG      ulFontID;
   CHAR       szFontName[LEN_FONTNAME+1];
   PUSERFONT  pUserFont;
   USHORT     usMatchIdx      = 0;
   BOOL       bMatchFound     = FALSE;
   PFONTINFO2 pFontInfo;
   INT        iWhichPointSize = 0;
   BOOL       fDone           = FALSE;
#ifdef DEBUG
   PDEBUGINFO pDbg            = &globals.DebugInfo;
#endif

   DBPRINTIF ((pDbg->bFILLFONTLIST, "%s(): Enter\n", __FUNCTION__));

   // initial pointer validation
   assertF (pdi);
   assertF (pDevice);

   // free existing list
   WinSendDlgItemMsg (hwndDlg,
                      ulListboxID,
                      LM_DELETEALL,
                      MPVOID,
                      MPVOID);

   /************************************/
   /* Driver Defined Fonts             */
   /************************************/

   // get # Fonts defined
   usNumFonts = pDevice->usNumFonts;

   if (usNumFonts > 0)
   {
      assertF (pDevice->pulFONTS);

      // add each Font defined into listbox
      for (i = 0; i < usNumFonts; i++)
      {
         ulFontID = pDevice->pulFONTS[i];

         // if Font ID found and string returned
         if (FontInfoFromID (pdi->pDriver, pDevice, ulFontID, &pFontInfo))
         {
            SHORT     sIdx;
            PFOCAFONT pFocaFont       = NULL;

            ulFontID = pFontInfo->lDevMatch;

            iWhichPointSize = 0;
            do
            {
               pFocaFont = NULL;
               ReadFocaMetricsFile (pdi->hmcbHeap,
                                    pFontInfo,
                                    &pFocaFont,
                                    pRes,
                                    valPointSizes[iWhichPointSize]);

               if (pFocaFont)
               {
                  strcpy (szFontName, pFontInfo->szFaceName);

                  if (FONT_TYPE_SCALABLE_SPECIAL == pFontInfo->ulType)
                  {
                     sprintf (szFontName, "%s @ %d",
                              pFocaFont->fmMetrics.szFacename,
                              pFocaFont->fmMetrics.usNominalPointSize / 10);
                  }
                  else if (FONT_TYPE_SCALABLE_SPECIAL2 == pFontInfo->ulType)
                  {
                     sprintf (szFontName, "%s @ %d",
                              pFocaFont->fmMetrics.szFacename,
                              pFontInfo->ulTypeExtra);
                  }

                  sIdx = (SHORT)WinSendDlgItemMsg (hwndDlg,
                                                   ulListboxID,
                                                   LM_INSERTITEM,
                                                   MPFROM2SHORT (LIT_END, 0),
                                                   MPFROMP (szFontName));

                  // Set Item Handle to be the Font ID
                  WinSendDlgItemMsg (hwndDlg,
                                     ulListboxID,
                                     LM_SETITEMHANDLE,
                                     MPFROMSHORT (sIdx),
                                     MPFROMLONG (ulFontID));

                  // If we are to highlite correct index
                  if (!bMatchFound)
                  {
                     // If we match default form ID
                     // @EXJOB
                     if (ulFontID == pdi->pJobProperties->ulDefFontID)
                     {
                        usMatchIdx = sIdx;
                        bMatchFound = TRUE;
                     }
                  }

                  GplMemoryFree (pFocaFont);
               }

               iWhichPointSize++;

               fDone = (  (FONT_TYPE_SCALABLE_SPECIAL != pFontInfo->ulType)
                       || (  (FONT_TYPE_SCALABLE_SPECIAL == pFontInfo->ulType)
                          && (iWhichPointSize == dimof (valPointSizes))
                          )
                       );

               if (!fDone)
               {
                  ulFontID = ((LONG)ulFontID - 1);
               }

            } while (!fDone);
         }
      }
   }

   /************************************/
   /* User Defined Fonts               */
   /************************************/

   // get # Fonts defined in linked list by user
   usNumFonts = pDevice->pUserDefData->usNumFonts;

   // assign base pointer to linked list
   pUserFont = pDevice->pUserDefData->pUserFONTS;

   // add each Font defined into listbox
   for (i = 0; i < usNumFonts; i++)
   {
      // Get local copy of Font ID
      ulFontID = pUserFont->fiUser.ulFontID;

      // if Font ID found and string returned
      if (FontNameFromID (pdi->pDriver, pDevice, szFontName, ulFontID))
      {
         SHORT     sIdx;
         PFOCAFONT pFocaFont = NULL;

         ulFontID = pUserFont->fiUser.lDevMatch;

         iWhichPointSize = 0;
         do
         {
            pFocaFont = NULL;
            ReadFocaMetricsFile (pdi->hmcbHeap,
                                 pFontInfo,
                                 &pFocaFont,
                                 pRes,
                                 valPointSizes[iWhichPointSize]);

            if (pFocaFont)
            {
               strcpy (szFontName, pFontInfo->szFaceName);

               if (  FONT_TYPE_SCALABLE_SPECIAL  == pFontInfo->ulType
                  || FONT_TYPE_SCALABLE_SPECIAL2 == pFontInfo->ulType
                  )
               {
                  sprintf (szFontName, "%s @ %d",
                           pFocaFont->fmMetrics.szFacename,
                           pFocaFont->fmMetrics.usNominalPointSize / 10);
               }

               sIdx = (SHORT)WinSendDlgItemMsg (hwndDlg,
                                                IDL_DEVICE_FONTS,
                                                LM_INSERTITEM,
                                                MPFROM2SHORT (LIT_END, 0),
                                                MPFROMP (szFontName));

               // Set Item Handle to be the Font ID
               WinSendDlgItemMsg (hwndDlg,
                                  IDL_DEVICE_FONTS,
                                  LM_SETITEMHANDLE,
                                  MPFROMSHORT (sIdx),
                                  MPFROMLONG (ulFontID));

               DBPRINTIF ((pDbg->bFILLFONTLIST, "%s(): Adding Font: %s (ID=%d)\n", __FUNCTION__, szFontName, ulFontID));

               // If we are to highlite correct index
               if (!bMatchFound)
               {
                  // If we match default form ID
                  // @EXJOB
                  if (ulFontID == pdi->pJobProperties->ulDefFontID)
                  {
                     usMatchIdx = sIdx;
                     bMatchFound = TRUE;
                  }
               }

               GplMemoryFree (pFocaFont);
            }

            iWhichPointSize++;

            fDone = (  (FONT_TYPE_SCALABLE_SPECIAL != pFontInfo->ulType)
                    || (  (FONT_TYPE_SCALABLE_SPECIAL == pFontInfo->ulType)
                       && (iWhichPointSize == dimof (valPointSizes))
                       )
                    );

            if (!fDone)
            {
               ulFontID = ((LONG)ulFontID - 1);
            }

         } while (!fDone);
      }

      // increment list pointer
      pUserFont = pUserFont->pNextFont;
   }

   // @TBD - highlite index may need to check if no match found
   // what to do as default then
   WinSendDlgItemMsg (hwndDlg,
                      ulListboxID,
                      LM_SELECTITEM,
                      MPFROMLONG (usMatchIdx),
                      MPFROMLONG (TRUE));

   DBPRINTIF ((pDbg->bFILLFONTLIST, "%s(): Highliting Font index: (%d)\n", __FUNCTION__, usMatchIdx));

   DBPRINTIF ((pDbg->bFILLFONTLIST, "%s(): Exit\n", __FUNCTION__));

} /* end FillFontList */

/****************************************************************************/
/* PROCEDURE NAME : GetDefaultFontInfo                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 11/19/93                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
GetDefaultFontInfo (PDEVICEBLOCK pdb,
                    ULONG        ulDefFontID,
                    PPFONTINFO2  pFontMatch)
{
   PDRIVERINFO pDriver         = pdb->pDriver;
   PDEVICEINFO pDevice         = pdb->pDevice;
   BOOL        bSuccess        = FALSE;
   ULONG       ul;
   ULONG       ulNumDefined;
   PFONTINFO2  pFonts;
#ifdef DEBUG
   PDEBUGINFO  pDbg            = &globals.DebugInfo;
#endif

   DBPRINTIF ((pDbg->bGETDEFAULTFONTINFO, "%s(): Enter\n", __FUNCTION__));

   assertF (pDevice);
   assertF (pFontMatch);

   /************************************/
   /* Driver Defined Fonts             */
   /************************************/
   ulNumDefined = pDriver->ulNumFonts;
   pFonts       = pDriver->pFONTS;

   // For now only driver table definitions allowed
   // loop through table of Fonts until matching Font ID
   // found
   for (ul = 0; ul < ulNumDefined; ul++)
   {
      // Now we must determine if the string is in our stringtable
      // or a it is a string the user defined in hcInfo structure
      if (pFonts[ul].ulFontID == ulDefFontID)
      {
         *pFontMatch = &(pFonts[ul]);

         bSuccess = TRUE;
         break;
      }
   }

   /************************************/
   /* User Defined Fonts               */
   /************************************/

   // search user defined Fonts if no match found yet
   if (!bSuccess)
   {
      PUSERFONT pUserFont;

      // assign Font list pointer to local variable
      pUserFont = pDevice->pUserDefData->pUserFONTS;

      // for each entry in linked list of user defined Fonts
      while (pUserFont != NULL)
      {
         if (pUserFont->fiUser.ulFontID == ulDefFontID)
         {
            *pFontMatch = &(pUserFont->fiUser);

            bSuccess = TRUE;
            break;
         }

         pUserFont = pUserFont->pNextFont;
      }
   }

   DBPRINTIF ((pDbg->bGETDEFAULTFONTINFO, "%s(): Exit\n", __FUNCTION__));

   return bSuccess;

} /* end GetDefaultFontInfo */

/****************************************************************************/
/* PROCEDURE NAME : FontIdFromLMatch                                        */
/* AUTHOR         : Mark H                                                  */
/* DATE WRITTEN   : 06/19/98                                                */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG
FontIdFromLMatch (PDRIVERINFO  pDriver,
                  PDEVICEINFO  pDevice,
                  LONG         lMatch,
                  PLONG        plPointSize)
{
   ULONG       ulNumDefined;
   PFONTINFO2  pFonts;
   ULONG       ul;

   ulNumDefined = pDriver->ulNumFonts;
   pFonts       = pDriver->pFONTS;

   // For now only driver table definitions allowed
   // loop through table of Fonts until matching Font ID
   // found
   for (ul = 0; ul < ulNumDefined; ul++)
   {
      if (FONT_TYPE_SCALABLE_SPECIAL == pFonts[ul].ulType)
      {
         if (  lMatch <= pFonts[ul].lDevMatch
            && (pFonts[ul].lDevMatch - (dimof (valPointSizes) - 1)) <= lMatch
            )
         {
            *plPointSize = valPointSizes[-lMatch + pFonts[ul].lDevMatch];

            return pFonts[ul].ulFontID;
         }
      }
      else if (FONT_TYPE_SCALABLE_SPECIAL2 == pFonts[ul].ulType)
      {
         if (lMatch == pFonts[ul].lDevMatch)
         {
            *plPointSize = pFonts[ul].ulTypeExtra;

            return pFonts[ul].ulFontID;
         }
      }
      else if (lMatch == pFonts[ul].lDevMatch)
      {
         if (pFonts[ul].vpGlobalFocaFont)
            *plPointSize = pFonts[ul].vpGlobalFocaFont->fmMetrics.usNominalPointSize / 10;

         return pFonts[ul].ulFontID;
      }
   }

   return 0;
}
