/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Omni Driver Enable File                                 */
/* SOURCE NAME    : SAMP_PCL.C                                              */
/* AUTHOR         : Matt Rutkowski                                          */
/* DESCRIPTION    : Contains definitions and functions for adding           */
/*                  sample monochrome and sample color PCL language         */
/*                  printer devices to the core "Omni" driver.              */
/*                                                                          */
/*                  This simply requires defining the TRAYS, PAPER SIZES,   */
/*                  MEDIA TYPES, BITMAP FORMATS, RESOLUTIONS and FONTS      */
/*                  supported by the printers we will be supporting,        */
/*                                                                          */
/*                  and...  writing a few simple functions.                 */
/*                                                                          */
/*                  After filling these out, you should get basic output    */
/*                  to your printer models.  This basic support can then    */
/*                  be enhanced to add escape codes, compression and        */
/*                  dither algorithms unique to the printers you support.   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                          support                         */
/*                                                                          */
/****************************************************************************/

//****************************************************************************
//*  Include GENPLIB headers files needed by this module
//****************************************************************************
#define INCL_GENPLIB_THREAD_WRITE
#define INCL_GENPLIB_COMPRESS
//#define INCL_GENPLIB_DBCS
//#define INCL_GENPLIB_STRINGSORT
//#define INCL_GENPLIB_FAX
#define INCL_GENPLIB_LAYOUT  
#include <genplib.h>

//****************************************************************************
//*  Device Support Flags (unique to SampPCL printer family)
//****************************************************************************
#define PCL_ENABLE_GRAY_BALANCE 0

#define PCL_SUPPORT_DEPLETION          0x00000004  // Ink depletion supported
#define PCL_SUPPORT_SHINGLE            0x00000008  // Dot spacing/Shingling supported
#define PCL_SUPPORT_PRINT_QUALITY      0x00000010  // Supports Print Quality cmd
#define PCL_SUPPORT_GRAY_BALANCE       0x00000008  // Supports firmware gray balance for composite black

#define PCL_SUPPORT_SHINGLE_DEPLETION  ( PCL_SUPPORT_DEPLETION | PCL_SUPPORT_SHINGLE )

#define PCL_SUPPORT_CMY_ONLY           0x00010000  // model supports only CMY ink (not K)
#define PCL_SUPPORT_CMYK               0x00020000  // model supports CMY and K inks

// @CHANGE - Here we simply define bit flag constants that represent
// the levels of compression supported by the printers we will be
// adding.  These are based off of #defines in the GENPLIB compression
// library package.  This allows us to assign a "set" of compression
// modes with a single define.
// See below for use within printer model definitions
#define SAMPPCL_RLL_TIFF     GPLCOMPRESS_RLL  | GPLCOMPRESS_TIFF
#define SAMPPCL_DELTAROW     SAMPPCL_RLL_TIFF | GPLCOMPRESS_DELTAROW
#define SAMPPCL_RLLDELTAROW  SAMPPCL_DELTAROW | GPLCOMPRESS_RLLDELTAROW

//****************************************************************************
//*  Device MACROs (unique to SampPCL family)
//****************************************************************************

//****************************************************************************
//*
//*               F E A T U R E    I D   #'s
//*
//****************************************************************************
//*  This section contains all the unique IDs that will be used to
//*  reference unique printer models and specific features of
//*  individual printer models.
//*
//*  Including:
//*         PRINT MODEL   (e.g. each printer model gets a unique ID)
//*         PRINT QUALITY (e.g. each resolution referenced gets a unique ID)
//*         PRINT MODE    ("")
//*         DEVICE FONT   ("")
//*         PAPER TRAY
//*         FORM SIZE
//*         MEDIA TYPE
//*         FORM CONNECTION
//*
//*  Please note that different printer models can reference the
//*  same ID if they have the same capability.  This is often true
//*  when a "printer family" exists and all printers in the family
//*  support the same paper sizes, trays etc.
//*
//* !!!!!!!!!!!!!!!!!!!!!!! WARNINGS ABOUT IDs !!!!!!!!!!!!!!!!!!!!!!!!
//*
//* 1)     These IDs should NEVER be changed once added and shipped with
//*        a printer or cross-network printing could fail (i.e. a newer
//*        and older driver need to communicate data across network).
//*
//* 2)     Each group of IDs should have the default option for the
//*        group assigned the ID == 0 (zero).  This helps the core
//*        driver fail invalid IDs and pick 0 (zero) as a default
//*        fallback in an emergency.
//*
//* !!!!!!!!!!!!!!!!!!!!!!! WARNINGS ABOUT IDs !!!!!!!!!!!!!!!!!!!!!!!!
//*
//****************************************************************************

//****************************************************************************
//* PRINT MODEL IDs
//****************************************************************************
//* Assign a new unique internal model ID for every new printer added
//* @CHANGE - Change these model IDs to ones for your PCL compatable printer
//****************************************************************************
#define PRN_ID_SAMPPCL_K             9000
#define PRN_ID_SAMPPCL_CMY           9001
#define PRN_ID_SAMPPCL_CMYK          9002

//****************************************************************************
//* PRINT QUALITY IDs
//****************************************************************************
//* Assign a new unique internal Print Quality ID for each quality
//* setting available on your printer models.
//*
//* A unique print quality typically corresponds to each setting your
//* printer allows to control the # print head passes per scanline
//* (e.g. draft, letter quality, fine) and for each resolution supported).
//*
//* Example: A Draft quality selection would cause one pass per scanline,
//* whereas a Presentation selection would make 4 passes per scanline.
//*
//* Result: Draft printout would have visible gaps b/w scanlines whereas
//* Presentation printout would have no visible scanline artifacts
//*
//* @CHANGE - Change these IDs for the # of print qualities you wish
//* supported for your printer models.
//****************************************************************************
#define PCL_RES_ID_DRAFT             0
#define PCL_RES_ID_NORMAL            1
#define PCL_RES_ID_PRESENTATION      2

//****************************************************************************
//*  PRINT MODES IDs
//****************************************************************************
//* Assign a new unique internal Print Mode ID for each
//* bitmap format you want to support on your printer using
//* your rendering software or the ones provided in GenPLib.
//*
//* Note: Available formats that work on OS/2 Warp include:
//*
//*  8 bit/pel logical on a  1 bit/pel physical bitmap (256 grayscale)
//*  8 bit/pel logical on a  8 bit/pel physical bitmap (256 color)
//* 24 bit/pel logical on a 24 bit/pel physical bitmap (16.7 million color)
//*
//****************************************************************************
#define PCL_PMODE_ID_BLACK           0
#define PCL_PMODE_ID_COLOR           1
#define PCL_PMODE_ID_BLK_CLR         2
#define PCL_PMODE_ID_24_CLR          3
#define PCL_PMODE_ID_24_BLK_CLR      4

//****************************************************************************
//* FONTS IDs
//****************************************************************************
// Not implemented for these PCL models
//****************************************************************************

//****************************************************************************
//* TRAYS IDs
//****************************************************************************
//* Assign a new unique internal Tray ID for each physical tray
//* your printer models support.
//****************************************************************************
#define PCL_TRAY_ID_TRAY             0
#define PCL_TRAY_ID_UPPER            1
#define PCL_TRAY_ID_ENV              2
#define PCL_TRAY_ID_NOSELECT         3
#define PCL_TRAY_ID_50SHEET          4
#define PCL_TRAY_ID_AUTO             5
#define PCL_TRAY_ID_MANUAL           6
#define PCL_TRAY_ID_ENV2             7
#define PCL_TRAY_ID_PORTABLE         8

//****************************************************************************
//* FORMS IDs
//****************************************************************************
//  Assign a unique ID for each unique paper size supported by your printer
//  models.
//
//  NOTE: You can have multiple form definitions for the same paper size
//
//  Example: Three IDs can exist for one paper size like "Letter"
//           each with different margins.
//
//  This is because different printer models within a printer family
//  may have different margins for the same paper size.
//
//  There may even be different margins used on one form size
//  "Letter" within the same printer model depending on which
//  tray or printhead is being used.
//****************************************************************************

#define PCL_FORM_ID_LETTER           0
#define PCL_FORM_ID_LEGAL            1
#define PCL_FORM_ID_A4               2
#define PCL_FORM_ID_EXECUTIVE        3
#define PCL_FORM_ID_C10_ENV          4
#define PCL_FORM_ID_DL_ENV           5

//****************************************************************************
//* MEDIA TYPE IDs
//****************************************************************************
//* Assign a new unique internal Media Type ID for each different
//* media type your printer models support.
//****************************************************************************
#define PCL_MEDIA_ID_PLAIN           0      // Normal copier paper
#define PCL_MEDIA_ID_TRANS           1      // Transparency paper
#define PCL_MEDIA_ID_GLOSSY          2      // Chrome Coated paper (Shiny)
#define PCL_MEDIA_ID_SPECIAL         3      // Coated Paper

//****************************************************************************
//* FORM CONNECTION IDs
//****************************************************************************
//* Assign a new unique internal Form Connection ID for each different
//* combination of "Tray", "Form", and "Media Type" a user can select
//* Not every permutation need be displayed since the user can define
//* uncommon combinations from the Omni's Printer Property panels.
//****************************************************************************
#define CONN_ID_1                    0
#define CONN_ID_2                    1
#define CONN_ID_3                    2
#define CONN_ID_4                    3
#define CONN_ID_5                    4
#define CONN_ID_6                    5
#define CONN_ID_7                    6
#define CONN_ID_8                    7
#define CONN_ID_9                    8
#define CONN_ID_10                   9
#define CONN_ID_11                   10
#define CONN_ID_12                   11

//****************************************************************************
//  DEVICE HANDLE (unique data area for "plug-in" modules)
//****************************************************************************
//   The device handle is simply a data structure that is unique to
//   a specific "plug-in" module.  A pointer to this data area can be
//   given to the "core" Omni driver and this sample pointer will be
//   passed as an input parameter to all functions the "plug-in" module
//   will subclass.
//
//   This is a useful place to put state data and variables that are
//   re-used by the plug-in module during rendereing a print job
//****************************************************************************

#define SAMPPCL_SIG     0x5350434C      // 'SPCL'

typedef struct _SampPCLHandle {
   ULONG         cb;
   ULONG         ulSig;
   COMPRESSINFO  Comp;
   DITHEREQUEST  Req;
   DITHERESULT   Reply;
   PVOID         pdh;         // handle pointer to dither structure
   BOOL          fHaveSetupPrinter;

   ULONG         ulMoveDown;               // @176300 Y offset before printing
   ULONG         ulMoveRight;              // @176300 X offset before printing
} SAMPPCLHANDLE, *PSAMPPCLHANDLE;

//****************************************************************************
//* PRIVATE/SUBCLASSED FUNCTION PROTOTYPES
//****************************************************************************
FNSTD              SampPCLBeginJob;
FNSTD              SampPCLEndJob;
FNSTD              SampPCLNewFrame;
ABORT_FUNC         SampPCLAbortJob;
DEVICE_QUERY_FUNC  SampPCLDeviceQuery;
RASTER_FUNC        SampPCLChooseRasterize;

BOOL  _System SampPCLSetupPrinter  (PVOID, PVOID);

//****************************************************************************
//* Shingling/Depletion Tables  (SampPCL Specific)
//****************************************************************************
//
// Shingling is a process of applying only a percentage of the total ink
// for a given scanline per print head pass.  This allows ink to dry more
// evenly producing better quality printouts. Options usually are based upon
// the % of ink layed down per head pass (i.e. 25% = best quality,
// 50% = good quality, 100% or None = draft quality.
//
// Array of Shingle and Depletion for Monochrome CMY Printer
// Note the table is organized as follow:
//
//   The columns represent the Media Types of
//   Pain, Special, Glossy  and  Transparency.
//
//   The rows are presentation types of
//   Draft, Normal and Presentation.
//
//   The entries are the Shingle and Depletion for the
//   Combinations.
//
// NOTE: The concept of "shingling" may be known by different names
//       depending on printer manufacturer.  Not all printers support
//       this feature.
//
//****************************************************************************

typedef struct _Shingle_Depletion {
   INT iShingle;
   INT iDepletion;
} Shingle_Depletion;

#define DEPL_0      1
#define DEPL_25     2
#define DEPL_50     3

#define SHING_0     0
#define SHING_25    2
#define SHING_50    1


// Array of Shingle and Depletion for CMY Printers (MONOCHROME)
Shingle_Depletion aMono_CMY[3][4] =
{
               /*        Plain               Trans                 Glossy            Special */
/* Draft  */     { {SHING_0, DEPL_0},  {SHING_0, DEPL_0},   {SHING_0, DEPL_25},   {SHING_0,DEPL_0} },
/* Normal */     { {SHING_0, DEPL_0},  {SHING_50,DEPL_0},   {SHING_50,DEPL_25},   {SHING_0,DEPL_0} },
/* Pres   */     { {SHING_50,DEPL_0},  {SHING_25,DEPL_0},   {SHING_25,DEPL_25},   {SHING_0,DEPL_0} },
};

// Array of Shingle and Depletion for CMY Printers (COLOR)
Shingle_Depletion aColor_CMY[3][4] =
{
               /*        Plain               Trans                 Glossy            Special */
/* Draft  */     { {SHING_0, DEPL_25}, {SHING_0,  DEPL_0},  {SHING_0, DEPL_25},   {SHING_0, DEPL_25} },
/* Normal */     { {SHING_0, DEPL_25}, {SHING_50, DEPL_0},  {SHING_50,DEPL_25},   {SHING_50,DEPL_25} },
/* Pres   */     { {SHING_50,DEPL_25}, {SHING_25, DEPL_0},  {SHING_25,DEPL_25},   {SHING_25,DEPL_25} },
};

// Array of Shingle and Depletion for CMY Printers (MONOCHROME)
Shingle_Depletion aMono_CMYK[3][4] =
{
               /*        Plain               Trans                 Glossy            Special */
/* Draft  */     { {SHING_0 ,DEPL_25}, {SHING_0, DEPL_0},   {SHING_0, DEPL_25},   {SHING_0, DEPL_0} },
/* Normal */     { {SHING_0 ,DEPL_25}, {SHING_50,DEPL_0},   {SHING_50,DEPL_0 },   {SHING_0, DEPL_0} },
/* Pres   */     { {SHING_50,DEPL_25}, {SHING_25,DEPL_0},   {SHING_25,DEPL_25},   {SHING_0, DEPL_0} },
};


// Array of Shingle and Depletion for CMY Printers (COLOR)
Shingle_Depletion aColor_CMYK[3][4] =
{
               /*        Plain               Trans                 Glossy            Special */
/* Draft  */     { {SHING_0 ,DEPL_25}, {SHING_0, DEPL_0},    {SHING_0, DEPL_25},  {SHING_0, DEPL_25}},
/* Normal */     { {SHING_0 ,DEPL_25}, {SHING_50,DEPL_0},    {SHING_50,DEPL_25},  {SHING_50,DEPL_25}},
/* Pres   */     { {SHING_50,DEPL_25}, {SHING_25,DEPL_0},    {SHING_25,DEPL_25},  {SHING_25,DEPL_25}},
};

// Array of Shingle and Depletion for CMY Printers (COLOR)
Shingle_Depletion aShingDepl850c[4][4] =
{
               /*        Plain               Trans                 Glossy            Special */
/* Low    */     { {SHING_0 ,DEPL_0}, {SHING_0, DEPL_0}, {SHING_0, DEPL_0}, {SHING_0, DEPL_0}}, // 0,0,0,0
/* Draft  */     { {SHING_0 ,DEPL_0}, {SHING_0, DEPL_0}, {SHING_0, DEPL_0}, {SHING_0, DEPL_0}}, // 0,0,0,0
/* Normal */     { {SHING_0 ,DEPL_0}, {SHING_50,DEPL_0}, {SHING_50,DEPL_0}, {SHING_50,DEPL_0}}, // 0,0,0,0
/* Pres   */     { {SHING_50,DEPL_0}, {SHING_25,DEPL_0}, {SHING_25,DEPL_0}, {SHING_25,DEPL_0}}, // 1,1,1,1
};

/****************************************************************************/
/* PROCEDURE NAME : ConfigureShingleDepletion                               */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/95                                                */
/* DESCRIPTION    : Some PCL printers have 2 commands that help control     */
/*                  the amount and layout of ink printed to aid in          */
/*                  preventing ink "bleeding" or "pooling".                 */
/*                                                                          */
/*                  SHINGLING: ( Esc*o#Q )                                  */
/*                  This command controls which rows of inkjet nozzles      */
/*                  spray per printhead pass.  Without shingling all        */
/*                  nozzles spray at once and can cause "bleeding" with     */
/*                  shingling multiple printhead passes are made            */
/*                  and only a percentage of the inkjet nozzles spray       */
/*                  on a given pass.                                        */
/*                                                                          */
/*                                                                          */
/*                  DEPLETION: ( Esc*o#D )                                  */
/*                  This command reduces the number of nozzles that         */
/*                  spray on a given pass by a certain percentage.          */
/*                  This is typically done on certain media types           */
/*                  that are not very absorbtive to prevent "bleeding".     */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL
ConfigureShingleDepletion (PDDC pddc, PSAMPPCLHANDLE pHandle)
{
   INT       iShingle     = 0;
   INT       iDepletion   = 0;
   ULONG     ulDeviceBits = pddc->pdb->pDevice->ulDeviceBits;
   PRESINFO  pResInfo     = pddc->pdb->pResInfo;
   ULONG     ulRes        = pResInfo->ulResID;
   ULONG     ulMedia      = pddc->pdb->pMediaInfo->ulMediaID;

   // The Shingling and Depletion commands we'll be sending
   CHAR      achDEP[]     = _ESC_ "*o%dq%dD";
   CHAR      achSHINGLE[] = _ESC_ "*o%dQ";
   CHAR      achDEPONLY[] = _ESC_ "*o%dD";

   DBPRINTF (("Enter: ConfigShingleDepletion: ulRes = %d, ulMedia = %d\n",
              ulRes,
              ulMedia));

   // If this printer supports both a black and color pen
   // use one set of tables that makes allowances for mixing 2 ink types
   if (ulDeviceBits & PCL_SUPPORT_CMYK)
   {
      // "Black and Color" selected
      switch (pddc->pdb->pPrintMode->ulColorTech)
      {
      case COLOR_TECH_CMYK:
      {
         // "Black and Color" Selected
         iShingle   = aColor_CMYK[ulRes][ulMedia].iShingle;
         iDepletion = aColor_CMYK[ulRes][ulMedia].iDepletion;
         break;
      }

      case COLOR_TECH_CMY:
      {
         // "Color Only" Selected
         iShingle   = aColor_CMY[ulRes][ulMedia].iShingle;
         iDepletion = aColor_CMY[ulRes][ulMedia].iDepletion;
         break;
      }

      case COLOR_TECH_K:
      default:
      {
         // "Black Only" Selected
         iShingle   = aMono_CMYK[ulRes][ulMedia].iShingle;
         iDepletion = aMono_CMYK[ulRes][ulMedia].iDepletion;
         break;
      }
      }
   }
   else
   {
      // If printer only supports 1 pen at a time (i.e. black or color)
      if (ulDeviceBits & PCL_SUPPORT_CMY_ONLY)
      {
         switch (pddc->pdb->pPrintMode->ulColorTech)
         {
         case COLOR_TECH_CMY:
         {
            // "Color Only" Selected
            iShingle   = aColor_CMY[ulRes][ulMedia].iShingle;
            iDepletion = aColor_CMY[ulRes][ulMedia].iDepletion;
            break;
         }

         case COLOR_TECH_K:
         default:
         {
            // "Black Only" Selected
            iShingle   = aMono_CMY[ulRes][ulMedia].iShingle;
            iDepletion = aMono_CMY[ulRes][ulMedia].iDepletion;
            break;
         }
         }
      }
      else
      {
         // Otherwise this a monochrom device and we'll use same
         // tables as we would for a CMYK device in "Black Only"
         // color technology selected.
         iShingle   = aMono_CMYK[ulRes][ulMedia].iShingle;
         iDepletion = aMono_CMYK[ulRes][ulMedia].iDepletion;
      }
   }

   DBPRINTF(( "iShingle = %d, iDepletion = %d\n", iShingle, iDepletion ));

   // Both Shingle and Depletion are supported on this model
   // This is the most common occurrence for most SampPCL printers
   if (  (ulDeviceBits & PCL_SUPPORT_SHINGLE)
      && (ulDeviceBits & PCL_SUPPORT_DEPLETION)
      )
   {
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       strlen(achDEP),
                                       achDEP,
                                       iShingle,
                                       iDepletion);
   }
   else
   {
      // Shingling ONLY supported on this printer
      // NOTE: We test this bit first since having this option is more
      // common than just having depletion
      if (ulDeviceBits & PCL_SUPPORT_SHINGLE)
      {
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          strlen (achSHINGLE),
                                          achSHINGLE,
                                          iShingle);
      }
      else
      {
         // Depletion ONLY supported on this printer
         // NOTE: We've already tested that at least 1 is supported
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          strlen(achDEPONLY),
                                          achDEPONLY,
                                          iDepletion);
      }
   }

   DBPRINTF (("Exit: ConfigShingleDepletion: iShingle = %d, iDepletion = %d\n", iShingle, iDepletion));

   return TRUE;

} /* end ConfigureShingleDepletion */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLBeginJob                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/93                                                */
/* DESCRIPTION    : This functions will send to the output thread all the   */
/*                  PCL commands needed to Initialize a PCL print job.      */
/*                  In addition, we initialize all variables we need to     */
/*                  indicate this is a new print job, as well as a new      */
/*                  page of output.                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/*                  [A] Send Printer Initialization Command                 */
/*                  [B] Set printheads initial position on page             */
/*                  [C] Initialize flags to indicate:                       */
/*                      1) Start of new page                                */
/*                      2) Graphics not yet initialized for print job       */
/*                                                                          */
/* NOTES:           We defer graphics PCL commands until we are sure        */
/*                  something has been successfully rendered.               */
/*                                                                          */
/* PARAMETERS:      (PVOID)pv == (PDDC) pddc : Omni Driver magic cookie     */
/*                  (PVOID)pvHandle : Our own PCL handle                    */
/*                                                                          */
/* RETURN VALUES:   TRUE:  Success - We always succeed                      */
/*                  FALSE: Failure - you may add failure conditions         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLBeginJob (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc       = (PDDC)pv;
   PDEVICEINFO    pDevice    = pddc->pdb->pDevice;
   PSAMPPCLHANDLE pHandle  = (PSAMPPCLHANDLE)pvHandle;

   if (pDevice->pCmds->ulCmdInit)
   {
      // Init printer
      GplThreadWriteBlock (pddc->pdb->hThread,
                           pDevice->pCmds->pszCmdInit,
                           pDevice->pCmds->ulCmdInit);
   }

   // Set printer's physical head position in our ddc
   pddc->ptlPrintHead.x = 0;
   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.yPels-1;
   else
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.xPels-1;

   // Reset flag for beginning of a new (next) page
   pHandle->Req.SrcOptions.fStartOfPage = TRUE;
   pHandle->fHaveSetupPrinter = FALSE;

   return TRUE;

} /* end SampPCLBeginJob */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLSetupPrinter                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/95                                                */
/* DESCRIPTION    : Function we call whenever we wish to send all the       */
/*                  PCL commands that will setup graphics rendering         */
/*                  in our printer.                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/*                  [A] First verify we are actually need to send PCL.      */
/*                      This function can be called for PM_Q_STD printing   */
/*                      where a metafile will be generated.  The core       */
/*                      Omni driver will handle this case by default.       */
/*                                                                          */
/*                  [B] Send PCL Units of Motion Command                    */
/*                  [C] Send PCL Form Size Command                          */
/*                  [D] Send Print Quality command                          */
/*                  [E] Send Gray Balance/Shingling/Depletion PCL           */
/*                      Commands if supported by this model.                */
/*                                                                          */
/*                  [F] Send PCL Tray selection Command                     */
/*                  [G] Send PCL Margin Command                             */
/*                  [H] Send PCL Printhead initial (X,Y) position           */
/*                                                                          */
/* NOTES:                                                                   */
/*                  Basic PCL Language structure                            */
/*                  ============================                            */
/*                                                                          */
/*                  UEL  - PJL                                              */
/*                                                                          */
/*                  EscE - reset                                            */
/*                                                                          */
/*                  Job  Control Commands                                   */
/*                  =====================                                   */
/*                                                                          */
/*                  Esc&l#X - # copies                                      */
/*                  Esc&l#S - duplex/simplex                                */
/*                  Esc&l#U - left binding margin (shifts log page left)    */
/*                  Esc&l#Z - top binding margin (shifts log page down)     */
/*                  Esc&a#G - Duplex select page side                       */
/*                  Esc&l#G - Output bin select                             */
/*                  Esc&u#D - units of measure (units/inch == dpi)          */
/*                                                                          */
/*                  Page Control Commands                                   */
/*                  =====================                                   */
/*                                                                          */
/*                  Esc&l#A - Page Size                                     */
/*                  Esc&l#H - Paper Source                                  */
/*                  Esc&l#O - Orientation                                   */
/*                  Esc&l#E - Top Margin Cmd                                */
/*                  Esc&l#L - Perf. Skip Cmd                                */
/*                  Esc*p#X - Horz Motion                                   */
/*                  Esc*p#Y - Vert Motion                                   */
/*                  Esc&k0G - Line Sep                                      */
/*                                                                          */
/*                  Font Commands                                           */
/*                  EscE - reset                                            */
/*                  UEL  - PJL                                              */
/*                                                                          */
/* PARAMETERS:      (PVOID)pv == (PDDC) pddc : Omni Driver magic cookie     */
/*                  (PVOID)pvHandle : Our own PCL handle                    */
/*                                                                          */
/* RETURN VALUES:   TRUE:  Success - We always succeed                      */
/*                  FALSE: Failure - you may add failure conditions         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLSetupPrinter (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc         = (PDDC)pv;
   PRESINFO       pResInfo     = pddc->pdb->pResInfo;
   PFORMINFO2     pFormInfo    = pddc->pdb->pFormInfo;
   CHAR           achUOM[]     = _ESC_ "*t%dR";
   PSAMPPCLHANDLE pHandle      = (PSAMPPCLHANDLE)pvHandle;

   assertF (pddc);

   // Validate our PDDC is valid before dereferencing
   if (!pddc)
      return FALSE;

   // We have an actual OS/2 job not a DOS/Windows job to send
   if (pddc->pdb->fRawOnly == FALSE)
   {
      pHandle->fHaveSetupPrinter = TRUE;

      /*------------------------*/
      /* Units of Motion (res)  */
      /*------------------------*/
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       strlen (achUOM),
                                       achUOM,
                                       pResInfo->ulXRes);

      /*------------------------*/
      /* Form Size              */
      /*------------------------*/
      GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                  pddc->pdb->pFormInfo->szCmdSelectForm);


      /*------------------------*/
      /* Print Quality (res)    */
      /*------------------------*/
      GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                  pResInfo->szCmdSelectRes);

#if PCL_ENABLE_GRAY_BALANCE
      /*------------------------*/
      /* Gray Balance           */
      /*------------------------*/
      // Select Gray Balance if device supports this and device is to
      // print in CMY mode  and printmode is not draft - enable Gray balance
      if (  (pddc->pdb->pDevice->ulDeviceBits & PCL_GRAY_BALANCE)
         && pddc->pdb->pPrintMode->ulColorTech == COLOR_TECH_CMY
         && pddc->pdb->pResInfo->ulResID != PCL_RES_ID_DRAFT
         )
         GplThreadWriteASCIICommand (pddc->pdb->hThread, _ESC_"*b1B");
#endif

      /*-------------------*/
      /* Shingle/Depletion */
      /*-------------------*/

      // if either shingling or depletion are supported on the printer
      // reference tables based upon resolution, color technology and
      // media type to derive the correct command to send to printer
      if (pddc->pdb->pDevice->ulDeviceBits & PCL_SUPPORT_SHINGLE ||
          pddc->pdb->pDevice->ulDeviceBits & PCL_SUPPORT_DEPLETION )
      {
         ConfigureShingleDepletion (pddc, pHandle);
      }

      /*-------------------*/
      /* Tray Selection    */
      /*-------------------*/
      // NOTE: Some PCL models will perform an extra FF if tray cmd is sent AND
      // paper size cmd is sent, so only send form size
      // Select the tray (paper source)
      // GplThreadWriteASCIICommand (pddc->pdb->hThread,
      //                             pddc->pdb->pTrayInfo->szCmdSelectTray);

      /*-------------------*/
      /* Top Margin        */
      /*-------------------*/

      // Set Top Margin to maximum logical page (no 1/2 inch default)
      GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                  pddc->pdb->pDevice->pCmds->pszCmdSetTopMargin);

      /*-------------------*/
      /* Initial Position  */
      /*-------------------*/

      // go to upper left corner to start document
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pddc->pdb->pDevice->pCmds->ulCmdSetXYPos,
                                       pddc->pdb->pDevice->pCmds->pszCmdSetXYPos,
                                       0,
                                       0);

      // @176300 User defined form?
      pHandle->ulMoveRight = 0;
      pHandle->ulMoveDown  = 0;
      if (IsUserDefinedForm (pFormInfo))
      {
         double  dyClipPels;
         double  dxClipPels;

         dxClipPels = pFormInfo->hcInfo.xLeftClip;
         dyClipPels = pFormInfo->hcInfo.cy - pFormInfo->hcInfo.yTopClip;

         if (0.01 <= dyClipPels)
         {
            double  dyPels;

            dyPels  = dyClipPels;
            dyPels /= 25.40L;
            dyPels *= pResInfo->ulYRes;

            DBPRINTF (("```dyPels  = %f\n", dyPels));

            pHandle->ulMoveDown = dyPels;

            if (pddc->pdb->pDevice->pCmds->ulCmdSetYPos)
            {
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                pddc->pdb->pDevice->pCmds->ulCmdSetYPos,
                                                pddc->pdb->pDevice->pCmds->pszCmdSetYPos,
                                                pHandle->ulMoveDown);
            }
         }

         if (0.01 <= dxClipPels)
         {
            double  dxPels;

            dxPels  = dxClipPels;
            dxPels /= 25.40L;
            dxPels *= pResInfo->ulXRes;

            DBPRINTF (("```dxPels  = %f\n", dxPels));

            pHandle->ulMoveRight = dxPels;
         }
      }
   }

   return TRUE;

} /* end SampPCLSetupPrinter */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLEndJob                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/93                                                */
/* DESCRIPTION    : These function sends all the commands needed to         */
/*                  tell the printer the print job has ended.               */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/*                  [A] Send PCL reset command to prepare printer for       */
/*                      next print job.                                     */
/*                                                                          */
/* NOTES:                                                                   */
/*                                                                          */
/* PARAMETERS:      (PVOID)pv == (PDDC) pddc : Omni Driver magic cookie     */
/*                  (PVOID)pvHandle : Our own PCL handle                    */
/*                                                                          */
/* RETURN VALUES:   TRUE:  Success - We always succeed                      */
/*                  FALSE: Failure - you may add failure conditions         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLEndJob (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc     = (PDDC)pv;

   // Simply send the EscE command to reset printer for next job
   GplThreadWriteASCIICommand (pddc->pdb->hThread,
                               pddc->pdb->pDevice->pCmds->pszCmdInit);

   return TRUE;

} /* end SampPCLEndJob */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLAbortJob                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/93                                                */
/* DESCRIPTION    : This function is called to fill in the "Abort Data"     */
/*                  structure.  This structure will contain all the         */
/*                  commands and/or data needed to reset the printer        */
/*                  to a known state after a print job was aborted by       */
/*                  an application or user.                                 */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* NOTES:           We elect to send 0x0256 of the NULL character to        */
/*                  the printer in case the printer was in graphics         */
/*                  mode and expecting graphics data.  After this the       */
/*                  printer should once again accept PCL commands           */
/*                  and then we send a PCL Abort command.                   */
/*                                                                          */
/* PARAMETERS:      (PVOID)pv == (PDDC) pddc : Omni Driver magic cookie     */
/*                  (PABORT_DATA)   : Abort data to send to printer         */
/*                  (PVOID)pvHandle : Our own PCL handle                    */
/*                                                                          */
/* RETURN VALUES:   TRUE:  Success - We always succeed                      */
/*                  FALSE: Failure - you may add failure conditions         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLAbortJob (PVOID pv, PABORT_DATA pAbortData, PVOID pvHandle)
{
   PDDC           pddc     = (PDDC)pv;

   // In raster transfer mode, if we were to stop sending data, then
   // printer would lock up.  Therefore, send a whole bunch of zeroes
   // that will fill up the rest of the raster transfer line.
   pAbortData->bSendRepeatBlock   = TRUE;
   pAbortData->uchAbortChar       = 0;
   pAbortData->ulAbortRepeatCount = 0x0256;

   pAbortData->bSendString    = TRUE;
   pAbortData->pszAbortString = pddc->pdb->pDevice->pCmds->pszCmdAbort;

   return TRUE;

} /* end SampPCLAbortJob */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLNewFrame                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/93                                                */
/* DESCRIPTION    : This function handle all PCL calls needed to eject      */
/*                  a page and setup printer/internal state for             */
/*                  rendering the next page of output.                      */
/* NOTES:                                                                   */
/*                                                                          */
/* PARAMETERS:      (PVOID)pv == (PDDC) pddc : Omni Driver magic cookie     */
/*                  (PVOID)pvHandle : Our own PCL handle                    */
/*                                                                          */
/* RETURN VALUES:   TRUE:  Success - We always succeed                      */
/*                  FALSE: Failure - you may add failure conditions         */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLNewFrame (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc     = (PDDC)pv;
   PSAMPPCLHANDLE pHandle  = (PSAMPPCLHANDLE)pvHandle;

   // send out blank pages except for last page being blank
   GplThreadWriteASCIICommand (pddc->pdb->hThread,
                               pddc->pdb->pDevice->pCmds->pszCmdPageEject);

   // go to upper left corner to start document
   GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                    pddc->pdb->pDevice->pCmds->ulCmdSetXYPos,
                                    pddc->pdb->pDevice->pCmds->pszCmdSetXYPos,
                                    0,
                                    0);

   // Set printer's physical head position in our ddc
   pddc->ptlPrintHead.x = 0;
   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.yPels-1;
   else
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.xPels-1;

   // Reset error buffer for next page (for error diffusion alorithms)
   // so that errors from previous page's dither are not carried over
   GPL_DITHER_RESET_ERR_BUFFERS (pHandle->Req)

   // Reset flag for beginning of a new (next) page
   pHandle->Req.SrcOptions.fStartOfPage = TRUE;

   // Reset blank lines count for the page
   pHandle->Comp.ulBlankLineCount = 0;

   return TRUE;

} /* end SampPCLNewFrame */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLDeviceQuery                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/12/93                                                */
/* DESCRIPTION    : This function is the main entry point for the "Plug-in" */
/*                  module.  It is called to enable the module, disable     */
/*                  the module, and query the module for subclassed         */
/*                  functions.                                              */
/*                                                                          */
/*  NOTE: "ulType"s:  DEVICE_ALLOC_HANDLE:                                  */
/*                       Signals the "plug-in" module to allocate and       */
/*                       fill-in the "Device Handle" it will use for        */
/*                       storing information during the print job.          */
/*                                                                          */
/*                       Set function return code to be the pointer to      */
/*                       "Device Handle" you allocate                       */
/*                       rc = (ULONG)pHandle;                               */
/*                                                                          */
/*                    DEVICE_FREE_HANDLE:                                   */
/*                       Signals the "plug-in" module to free the           */
/*                       "Device Handle" it used during the print job       */
/*                                                                          */
/*                       Set function return code to TRUE if "Device        */
/*                       Handle" was successfully freed or FALSE if         */
/*                       a failure occurred.                                */
/*                                                                          */
/*                    DEVICE_QUERY_ORIGIN:                                  */
/*                       Returns where on the printed page the printer      */
/*                       considers the page origin to be located.           */
/*                                                                          */
/*                       Parameters are retrieved as follows:               */
/*                       pulFlags   = (PULONG)pvArg1;                       */
/*                       pptlOrigin = (PPOINTL)va_arg (list, ULONG);        */
/*                                                                          */
/*                       Set one of the following origins:                  */
/*                       *pulFlags = DEVICE_BOTTOM_ORIGIN                   */
/*                       *pulFlags = DEVICE_TOP_ORIGIN                      */
/*                                                                          */
/*                       Fill-in initial position from the origin:          */
/*                       pptlOrigin->x = 0;                                 */
/*                       pptlOrigin->y = 120;                               */
/*                                                                          */
/*                       Return: TRUE for success, FALSE for failure        */
/*                                                                          */
/*                    DEVICE_QUERY_ROTATION_CAPS:                           */
/*                       Returns what orientations are possible on          */
/*                       the printer.  The Omni driver currently            */
/*                       assumes portrait printing and automatically        */
/*                       rotates landscape images for the "plug-in"         */
/*                       module if this call is not hooked.                 */
/*                                                                          */
/*                       Parameters are retrieved as follows:               */
/*                       pulFlags   = (PULONG)pvArg1;                       */
/*                                                                          */
/*                       Set one of the following origins:                  */
/*                       *pulFlags = DEVICE_PORTRAIT_OR_LANDSCAPE:          */
/*                       *pulFlags = DEVICE_ALWAYS_PORTRAIT:                */
/*                       *pulFlags = DEVICE_ALWAYS_LANDSCAPE:               */
/*                                                                          */
/*                       Return: TRUE for success, FALSE for failure        */
/*                                                                          */
/*                    DEVICE_QUERY_FUNCTIONS:                               */
/*                       This function is the dynamic way the Omni driver   */
/*                       queries the functions the driver wishes to         */
/*                       subclass.  If this "ulType" is not handled then    */
/*                       the Omni driver will use the functions the plug-in */
/*                       module filled out in the static function table     */
/*                                                                          */
/*                       Retrieve the function table as follows:            */
/*                       PFNSUBCLASS pfnSub = (PFNSUBCLASS)pvArg1;          */
/*                       Then hook a function as follows:                   */
/*                       pfnSub->pfnSpoolerOpen     = MySpoolerOpen;        */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (PVOID)pv1   : Typically the pddc for the Omni driver   */
/*                  (PVOID)ulType: Type of Device Query                     */
/*                  others       : Vary depending on "ulType" passed in     */
/*                                                                          */
/* RETURN VALUES:   Vary depending on "ulType" passed in                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLDeviceQuery (PVOID pv1, ULONG ulType, PVOID pvArg1, ...)
{
   BOOL           rc       = TRUE;
   PDDC           pddc     = (PDDC)pv1;
   PSAMPPCLHANDLE pHandle;
   INT            iBitsInArray,
                  iDWordBitsInArray,
                  i;
   va_list        list;

   va_start (list, pvArg1);

   DBPRINTF (("ulType = %d\r\n", ulType));

   // This function will return various information to the core
   // Omni driver depending on the "ulType" parameter passed in on the call.
   switch (ulType)
   {
   case DEVICE_ALLOC_HANDLE:
   {
      PSAMPPCLHANDLE pOldHandle = (PSAMPPCLHANDLE)va_arg (list, PSAMPPCLHANDLE);
      HMCB           hmcbHeap;

      hmcbHeap = (HMCB)pvArg1;

      if (!pOldHandle)
      {
         // Allocate our own data area for PCL-type printers
         pHandle = (PSAMPPCLHANDLE)GplMemoryAlloc (hmcbHeap,
                                                   sizeof (SAMPPCLHANDLE));
         assertF (pHandle);
      }
      else
         pHandle = pOldHandle;

      if (!pHandle)
         break;

      if (!pOldHandle)
         // Initilialize SampPCL's data
         memset (pHandle, 0, sizeof (SAMPPCLHANDLE));

      pHandle->cb                    = sizeof (SAMPPCLHANDLE);
      pHandle->ulSig                 = SAMPPCL_SIG;
      pHandle->Comp.ulBlankLineCount = 0;
      pHandle->fHaveSetupPrinter     = FALSE;

      if (!pddc)
      {
         // Cannot go further without a pddc...
         rc = (ULONG)pHandle;
         break;
      }

      /*------------------------*/
      /* Row Size & Orientation */
      /*------------------------*/
      // Determine width of a band based on orientation
      // Number of pels in a source row
      // number of bytes in Source, DwordAligned
      // Number of bytes in Destination, Byte Aligned
      // Set printer's physical head column position in our ddc

      if (pddc->pdb->bRasterPortrait)
      {
         iBitsInArray = BUMP_TO_NEXT_MODULUS (pddc->bmp.cx, 8);
         iDWordBitsInArray  = BUMP_TO_NEXT_MODULUS (pddc->bmp.cx * pddc->bmp.cBitCount,32);
      }
      else
      {
         iBitsInArray = BUMP_TO_NEXT_MODULUS (pddc->bmp.cy, 8);
         iDWordBitsInArray  = BUMP_TO_NEXT_MODULUS (pddc->bmp.cy * pddc->bmp.cBitCount, 32);
      }

      /*----------------------*/
      /* Printhead Init       */
      /*----------------------*/

      // Set printer's physical head position in our ddc
      pddc->ptlPrintHead.x = 0;

      if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
         pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.yPels-1;
      else
         pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.xPels-1;

      /*-------------------*/
      /* Compression Mode  */
      /*-------------------*/

      // get local variable of all comp modes supported by current device
      pHandle->Comp.ulSupportedCompModes = pddc->pdb->pJobProperties->ulCompressModes;

      pHandle->Comp.ulColorTech = pddc->pdb->pPrintMode->ulColorTech;
      pHandle->Comp.ulBytesPerRow = iBitsInArray>>3;

      GplCompressCreateInstance (&(pHandle->Comp), pddc->pdb->hmcbHeap);

      /*----------------------*/
      /* CMY or CMYK printing */
      /*----------------------*/
      if (pddc->pdb->pPrintMode->ulColorTech != COLOR_TECH_K)
      {
         LONG lCGamma, lMGamma, lYGamma;

         /*-------------------*/
         /* Dither Request    */
         /*-------------------*/

         // Init modify flag telling Gamma/HSV that we may replace/change
         // values in the source bitmap.  1,2,4,8 bit pel bitmaps, then
         // if C,M,Y are to be printed, the RGB index will be set to white.
         // If 16, 24 bit bitmaps, then the R,G,B value is replaced with white.
         pHandle->Req.lDitherType = pddc->pdb->pJobProperties->ulAlgorithm;

         // Sets up default flags for dithering code based only on algorithm
         // se gpldith.h header file from GENPLIB code shipped with DDK
         GPL_DITHER_SETUP_DEFAULTS (pHandle->Req,  pddc->pdb->pJobProperties->ulLevel)

         // Allow dithering code to modify our device surface buffer for perf.
         pHandle->Req.SrcOptions.fModify  = TRUE;

         pHandle->Req.iSrcRowPels = iBitsInArray;

         // number of bytes in Source, DwordAligned
         pHandle->Req.iNumSrcRowBytes32 = iDWordBitsInArray>>3;

         // Number of bytes in Destination, Byte Aligned
         pHandle->Req.iNumDestRowBytes = pHandle->Comp.ulBytesPerRow;

         // Fill in Number of colors (derived from bitcount)
         pHandle->Req.iNumColors = Power (2, pddc->bmp.cBitCount);

         // Initialize # of dither rows (or height of band buffer)
         pHandle->Req.iNumDitherRows = 1;

         /*-------------------*/
         /* Color Algorithm   */
         /*-------------------*/
         switch (pddc->pdb->pPrintMode->ulColorTech)
         {
         case COLOR_TECH_CMYK:
            pHandle->Req.bReq   = CMYK_ALL;
            break;

         case COLOR_TECH_CMY:
            pHandle->Req.bReq   = CMY_ALL;
            break;
         }

         /*-------------------*/
         /* Gamma             */
         /*-------------------*/
         i = FindGamma (pddc->pdb->pDriver->pGammas->pSelect,
                        pddc->pdb->pDriver->pGammas->pFamily,
                        (ULONG)pddc->pdb->pDevice->usDeviceID,
                        pddc->pdb->pDriver->pGammas->pRes,
                        pddc->pdb->pResInfo->ulResID,
                        pddc->pdb->pDriver->pGammas->pMedia,
                        pddc->pdb->pMediaInfo->ulMediaID,
                        pddc->pdb->pDriver->pGammas->pMode,
                        pddc->pdb->pPrintMode->ulPrintModeID,
                        pddc->pdb->pJobProperties->ulAlgorithm);

         // Use table lookup values to set default gammas for this device
         // NOTE: Red   gamma is really Cyan gamma
         //       Green gamma is really Magenta gamma
         //       Blue  gamma is really Yellow gamma
         pHandle->Req.lRGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bCGamma;
         pHandle->Req.lGGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bMGamma;
         pHandle->Req.lBGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bYGamma;
         pHandle->Req.lKGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bKGamma;

         pHandle->Req.ulRBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;
         pHandle->Req.ulGBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;
         pHandle->Req.ulBBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;
         pHandle->Req.ulKBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;

         // Since users want to truly adjust the colors Red, Green and Blue
         // and we as printers deal with Cyan Magenta and Yellow we need to
         // translate RGB to CMY ( R == M + Y, G == C + Y, and B = C + M )
         // So take and translate RGB adjustment values from Jon Properties
         GplGammaConvertRGBtoCMY (pddc->pdb->pJobProperties->lRedAdjust,
                                  pddc->pdb->pJobProperties->lGreenAdjust,
                                  pddc->pdb->pJobProperties->lBlueAdjust,
                                  &lCGamma,
                                  &lMGamma,
                                  &lYGamma);

         // Check for Gamma and Bias adjustment from Job Properties.
         // This dialog adjustment can be used to calibrate new device's default gammas
         pHandle->Req.lRGamma = pHandle->Req.lRGamma + lCGamma;
         pHandle->Req.lGGamma = pHandle->Req.lGGamma + lMGamma;
         pHandle->Req.lBGamma = pHandle->Req.lBGamma + lYGamma;
         pHandle->Req.lKGamma = pHandle->Req.lKGamma + pddc->pdb->pJobProperties->sBlackAdjust;

         pHandle->Req.ulRBias = (ULONG)((LONG)pHandle->Req.ulRBias + pddc->pdb->pJobProperties->lRedBias);
         pHandle->Req.ulGBias = (ULONG)((LONG)pHandle->Req.ulGBias + pddc->pdb->pJobProperties->lGreenBias);
         pHandle->Req.ulBBias = (ULONG)((LONG)pHandle->Req.ulBBias + pddc->pdb->pJobProperties->lBlueBias);
         pHandle->Req.ulKBias = (ULONG)((LONG)pHandle->Req.ulKBias + (LONG)pddc->pdb->pJobProperties->bBlackBias);

         /*-------------------*/
         /* HSV               */
         /*-------------------*/
         pHandle->Req.delta.lHue        = pddc->pdb->pJobProperties->lHue;
         pHandle->Req.delta.lSaturation = pddc->pdb->pJobProperties->lSaturation;
         pHandle->Req.delta.lValue      = pddc->pdb->pJobProperties->lValue;

         GplDitherCreateInstance ((PPVOID)&pHandle->pdh,
                                  hmcbHeap,
                                  (PDITHEREQUEST)&pHandle->Req,
                                  (PDITHERESULT)&pHandle->Reply);
      }
      else
      {
         // number of bytes in Source, DwordAligned
         pHandle->Req.iNumSrcRowBytes32 = iDWordBitsInArray>>3;

         // Number of bytes in Destination, Byte Aligned
         pHandle->Req.iNumDestRowBytes = pHandle->Comp.ulBytesPerRow;
      }

      rc = (ULONG)pHandle;
      break;
   }

   case DEVICE_FREE_HANDLE:
   {
      BOOL fReAllocate = va_arg (list, BOOL);

      // Grab our handle
      pHandle = (PSAMPPCLHANDLE)pvArg1;
      assertF (pHandle);

      // Free work buffers we allocated with GplMemoryAlloc()
      if (pHandle)
      {
         assertT (SAMPPCL_SIG != pHandle->ulSig);

         // tell the dither function to deallocate all it's buffers
         if (pHandle->pdh)
            GplDitherDeleteInstance ((PVOID)pHandle->pdh);

         // tell compression to deallocate all its's buffers
         GplCompressDeleteInstance ((PVOID)&(pHandle->Comp));

         if (!fReAllocate)
            // Free our handle
            GplMemoryFree (pHandle);
      }

      rc = TRUE;
      break;
   }

   default:
   {
      rc = FALSE;
      break;
   }
   }

   va_end (list);

   return rc;

} /* end SampPCLDeviceQuery */

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
VOID
SampPCLFillInPDL (PPDL   pPDL,
                  ULONG  ulDeviceID,
                  ULONG  ulSubPDL)
{
   /* OK. Return PDL # ulSubPDL.
   */
   switch (ulSubPDL)
   {
   case 1:
   {
      pPDL->ulPDL                         = PDL_Deskjet;
      pPDL->ulPDLLevel                    = PDL_DONTCARE;
      pPDL->ulMajorPDLVersion             = PDL_DONTCARE;
      pPDL->ulMinorPDLVersion             = PDL_DONTCARE;
      pPDL->szMinorCharID[0]              = '\0';
      pPDL->ulReserved                    = 0;
      pPDL->ulFlags                       = PDLCAPS_CURRENT | PDLCAPS_MACROSUPPORT;
      pPDL->szPDLDescription[0]           = '\0';
      pPDL->szTranslatedPDLDescription[0] = '\0';

#if 0
      /* Can specialize here
      */
      switch (ulDeviceID)
      {
      case PRN_ID_SAMPPCL_K:
      case PRN_ID_SAMPPCL_CMY:
      case PRN_ID_SAMPPCL_CMYK:
         pPDL->ulPDLLevel = PDL_DONTCARE;
         break;
      }
#endif
      break;
   }

   default:
      assertstring ("Unknown sub pdl level number.\n");
      break;
   }
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
BOOL
SampPCLQueryPDL (ULONG  ulType,
                 PPDL   pPDL,
                 ULONG  ulDeviceID,
                 ...)
{
   va_list          list;

   va_start (list, ulDeviceID);

   switch (ulType)
   {
   case QUERYPDL_QUERY_NUMBER:
   {
      ULONG *pulNumPDLs = va_arg (list, ULONG *);

      /* Return the number of Printer Description Languages that you support
      ** based on the device id (ulDeviceID).  For this driver, we only support
      ** 1 language for every device.
      ** Ex:
      ** switch (ulDeviceID)
      ** {
      ** case DEVICE_A:
      ** case DEVICE_B: *pulNumPDLs = 1; break;
      ** case DEVICE_C: *pulNumPDLs = 2; break;
      ** }
      */
      *pulNumPDLs = 1;

      return TRUE;
   }

   case QUERYPDL_QUERY_PDL:
   {
      ULONG ulSubPDL = va_arg (list, ULONG);

      /* Return the Printer Description Languages.  This is called for each element
      ** in the array that you support (total number returned in QUERYPDL_QUERY_NUMBER).
      ** The numbering scheme is 1 .. *pulNumPDLs.
      */
      SampPCLFillInPDL (pPDL, ulDeviceID, ulSubPDL);

      return TRUE;
   }

   case QUERYPDL_ISOK_PDL:
   {
      ULONG ulSubPDL = va_arg (list, ULONG);
      PDL   MyPDL;

      /* Return whether or not this is a PDL that you support (essentially
      ** one that you filled in previously).
      */
      SampPCLFillInPDL (&MyPDL, ulDeviceID, ulSubPDL);

      if (MyPDL.ulPDL             == pPDL->ulPDL                 &&
          MyPDL.ulMajorPDLVersion == pPDL->ulMajorPDLVersion     &&
          MyPDL.ulMinorPDLVersion == pPDL->ulMinorPDLVersion     &&
          0 == strcmp (MyPDL.szMinorCharID, pPDL->szMinorCharID) &&
          MyPDL.ulReserved        == pPDL->ulReserved            &&
          MyPDL.ulFlags           == pPDL->ulFlags                )
      {
         return TRUE;
      }
      else
      {
         return FALSE;
      }
   }
   }

   va_end (list);

   return FALSE;
}

/****************************************************************************/
/* PROCEDURE NAME : SampPCLMonoRasterize                                    */
/* AUTHOR         : Matt Rutkowski & Tom Spurrier                           */
/* DATE WRITTEN   : 8/1/94                                                  */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLMonoRasterize ( PBYTE             pbBits,
                       PBITMAPINFO2      pbmi,
                       PSIZEL            psizelBuffer,
                       PSIZEL            psizelPage,
                       PRECTL            prectlPageLocation,
                       PVOID             pvHandle,
                       PVOID             pvDDC)
{
   static BYTE    Mask[8]  = {
      0xFF, 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE
   };
   PDDC           pddc     = (PDDC)pvDDC;
   PSAMPPCLHANDLE pHandle  = (PSAMPPCLHANDLE)pvHandle;
   INT            iAmount;
   ULONG          cy, cx;
   INT            iWorldY;
   register INT   iCurrentRow;
   INT            iRemainder;
   INT            i;

   // setup height and width of buffer we are to process
   // based on orientation
   // check to see if we need to move for the start of the band
   // iAmount is the amount to move down the page
   // NOTE: ptlPrinthead has (0,0) at top-left of page
   // incrementing downwards
   // check to see if we need to move for the start of the band
   // iAmount is the amount to move down the page

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
   {
      cy = min (psizelBuffer->cy,
                (prectlPageLocation->yTop - prectlPageLocation->yBottom + 1));
      cx = min (psizelBuffer->cx,
                (prectlPageLocation->xRight - prectlPageLocation->xLeft + 1));
      iWorldY = prectlPageLocation->yTop;
   }
   else
   {
      cy = min (psizelBuffer->cy,
                (prectlPageLocation->xRight - prectlPageLocation->xLeft + 1));
      cx = min (psizelBuffer->cx,
                (prectlPageLocation->yTop - prectlPageLocation->yBottom + 1));
      iWorldY = prectlPageLocation->xRight;
   }

   // We want to keep iRemainder of the left-most bits of the last byte.
   // NOTE: 0 <= iRemainder <= 7

   PRINT_VAR (pHandle->Req.iNumDestRowBytes);
   PRINT_VAR (cx);
   iRemainder = cx - (pHandle->Req.iNumDestRowBytes - 1) * 8;

   if (8 == iRemainder)
      iRemainder = 0;
   assertF (0 <= iRemainder && iRemainder <= 7);

   // @176300
   if (pHandle->ulMoveRight                   &&
       pddc->pdb->pDevice->pCmds->ulCmdSetXPos )
   {
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pddc->pdb->pDevice->pCmds->ulCmdSetXPos,
                                       pddc->pdb->pDevice->pCmds->pszCmdSetXPos,
                                       pHandle->ulMoveRight);
   }

   // Enter into raster graphics transfer mode
   GplThreadWriteASCIICommand (pddc->pdb->hThread,
                      pddc->pdb->pDevice->pCmds->pszCmdBeginRasterGraphics);

   // check to see if we need to move for the start of the band
   // iAmount is the amount to move down the page
   iAmount = pddc->ptlPrintHead.y - iWorldY;

   if (iAmount > 0)
   {
      // undate physical print head because we will move here
      pddc->ptlPrintHead.y -= iAmount ;

      GplThreadWriteCommandWithPrintf( pddc->pdb->hThread,
                              pddc->pdb->pDevice->pCmds->ulCmdSetYPos,
                              pddc->pdb->pDevice->pCmds->pszCmdSetYPos,
                              iAmount);
   }

   // Process each scanline (1 pel line). Compress it then send it to printer
   for (iCurrentRow = psizelBuffer->cy - 1;
        iCurrentRow >= (INT)(psizelBuffer->cy - cy);
        iCurrentRow--)
   {
      // Copy raster bits into our buffer
      pHandle->Reply.pbKBuffer = pbBits + iCurrentRow * pHandle->Req.iNumSrcRowBytes32;

      // Make sure that the extraneous bits are set to 0.
      *(pHandle->Reply.pbKBuffer + pHandle->Req.iNumDestRowBytes - 1) &= Mask[iRemainder];

      // Check for blank line
      for (i = 0; i < pHandle->Req.iNumDestRowBytes; i++)
         if (*(pHandle->Reply.pbKBuffer + i) != 0) break;

      if (i == pHandle->Req.iNumDestRowBytes)   // Is line blank
         ++pHandle->Comp.ulBlankLineCount;
      else
      {
         if (pHandle->Comp.ulBlankLineCount) // output leading blank lines
         {
            if (pHandle->Comp.iCurrentCompMode != 0)   // Set no compression mode
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                pddc->pdb->pDevice->pCmds->ulCmdSetCompression,
                                                pddc->pdb->pDevice->pCmds->pszCmdSetCompression,
                                                0);

            // Skip leading blank lines
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             pddc->pdb->pDevice->pCmds->ulCmdSetYPos,
                                             pddc->pdb->pDevice->pCmds->pszCmdSetYPos,
                                             pHandle->Comp.ulBlankLineCount);

            pHandle->Comp.ulBlankLineCount = 0;

            if (pHandle->Comp.iCurrentCompMode != 0)
            {
               if (pHandle->Comp.iMaxCompMode > MAXCOMPRESSTIFF)
                  if (pHandle->Comp.pbLastLineK)
                     memset (pHandle->Comp.pbLastLineK, 0, pHandle->Req.iNumDestRowBytes);

               pHandle->Comp.iCurrentCompMode = 0;
            }
         }

         CompressRasterPlane (pHandle->Reply.pbKBuffer,
                              pHandle->Req.iNumDestRowBytes,
                              TRUE,
                              pHandle->Comp.pbLastLineK,
                              pHandle->Comp.pbCompress,
                              &pHandle->Comp.iCurrentCompMode,
                              pHandle->Comp.ulSupportedCompModes,
                              pvDDC,
                              pHandle->Comp.pusDelta);
      }
   }

   // If we exit raster graphics the most PCL printers will reset to default
   // compression.  Therefore we must invalidate the current compression
   // mode to force us to reselect it next time through.
   pHandle->Comp.iCurrentCompMode = GPLCOMPRESS_INVALID;

   
   if (pHandle->Comp.ulBlankLineCount) // output leading blank lines
   {
      if (pHandle->Comp.iCurrentCompMode != 0)   // Set no compression mode
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          pddc->pdb->pDevice->pCmds->ulCmdSetCompression,
                                          pddc->pdb->pDevice->pCmds->pszCmdSetCompression,
                                          0);

      // Skip leading blank lines
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pddc->pdb->pDevice->pCmds->ulCmdSetYPos,
                                       pddc->pdb->pDevice->pCmds->pszCmdSetYPos,
                                       pHandle->Comp.ulBlankLineCount);

      pHandle->Comp.ulBlankLineCount = 0;

      if (pHandle->Comp.iCurrentCompMode != 0)
      {
         if (pHandle->Comp.iMaxCompMode > MAXCOMPRESSTIFF)
            if (pHandle->Comp.pbLastLineK)
               memset (pHandle->Comp.pbLastLineK, 0, pHandle->Req.iNumDestRowBytes);

         pHandle->Comp.iCurrentCompMode = 0;
      }
   }
   

   // Exit raster graphics transfer mode
   GplThreadWriteASCIICommand (pddc->pdb->hThread,
                               pddc->pdb->pDevice->pCmds->pszCmdEndRasterGraphics);

   // Update printhead location
   pddc->ptlPrintHead.y -= cy;

   return FALSE;

} /* end SampPCLMonoRasterize */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLColorRasterize                                   */
/* AUTHOR         : Matt Rutkowski & Tom Spurrier                           */
/* DATE WRITTEN   : 8/1/94                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* 11/11/94 - Matt Rutkowski - rewrote                                      */
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLColorRasterize (PBYTE             pbBits,
                       PBITMAPINFO2      pbmi,
                       PSIZEL            psizelBuffer,
                       PSIZEL            psizelPage,
                       PRECTL            prectlPageLocation,
                       PVOID             pvHandle,
                       PVOID             pvDDC)
{
   PDDC           pddc         = (PDDC)pvDDC;
   PSAMPPCLHANDLE pHandle      = (PSAMPPCLHANDLE)pvHandle;
   PDEVICEINFO    pDevice      = pddc->pdb->pDevice;
   HTHREAD        hThread      = pddc->pdb->hThread;
   PCMDINFO       pCmds        = pDevice->pCmds;
   INT            iWorldY,
                  iAmount;
   ULONG          cy;
   register INT   iCurrentRow;
   PDITHEREQUEST  pCMYReq      = (PDITHEREQUEST)&pHandle->Req;
   USHORT         saveCy;

   /*-------------------*/
   /* Band Calculations */
   /*-------------------*/
   // setup height and width of buffer we are to process
   // based on orientation
   // check to see if we need to move for the start of the band
   // iAmount is the amount to move down the page
   // NOTE: ptlPrinthead has (0,0) at top-left of page
   // incrementing downwards
   // check to see if we need to move for the start of the band
   // iAmount is the amount to move down the page

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
   {
      cy = min (psizelBuffer->cy,
                (prectlPageLocation->yTop - prectlPageLocation->yBottom + 1));
      iWorldY = prectlPageLocation->yTop;
   }
   else
   {
      cy = min (psizelBuffer->cy,
                (prectlPageLocation->xRight - prectlPageLocation->xLeft + 1));
      iWorldY = prectlPageLocation->xRight;
   }

   // @176300
   if (  pHandle->ulMoveRight
      && pCmds->ulCmdSetXPos
      )
   {
      GplThreadWriteCommandWithPrintf (hThread,
                                       pCmds->ulCmdSetXPos,
                                       pCmds->pszCmdSetXPos,
                                       pHandle->ulMoveRight);
   }

   /*-------------------------*/
   /* Send Planes Per Row Cmd */
   /*-------------------------*/

   if (pddc->pdb->pPrintMode->ulColorTech == COLOR_TECH_CMY)
      GplThreadWriteASCIICommand (hThread, _ESC_"*r-3U");
   else
      GplThreadWriteASCIICommand (hThread, _ESC_"*r-4U");

   /*--------------------------------*/
   /* Send Begin Raster Graphics Cmd */
   /*--------------------------------*/

   // Enter into raster graphics transfer mode
   GplThreadWriteASCIICommand (hThread,
                               pCmds->pszCmdBeginRasterGraphics);

   /*--------------------------------*/
   /* Send Initial Print Head Pos.   */
   /*--------------------------------*/

   // check to see if we need to move for the start of the band
   // iAmount is the amount to move down the page
   // NOTE: ptlPrinthead has (0,0) at top-left of page
   // incrementing downwards
   iAmount = pddc->ptlPrintHead.y - iWorldY;

   // If we need to reposition for this new graphics band
   if (iAmount > 0)
   {
      // undate physical print head because we will move here
      pddc->ptlPrintHead.y -= iAmount;

      GplThreadWriteCommandWithPrintf (hThread,
                                       pCmds->ulCmdSetYPos,
                                       pCmds->pszCmdSetYPos,
                                       iAmount);
   }

   /*--------------------------------*/
   /* Init Dither Req struct         */
   /*--------------------------------*/
   pCMYReq->pSrcInfo = pbmi;

   /*--------------------------------*/
   /* 1st band of each page          */
   /*--------------------------------*/
   if (!pHandle->Req.SrcOptions.fStartOfPage)
   {
      GplCompressZeroLastLineBuffers ((PVOID)&(pHandle->Comp));
   }

   saveCy = pbmi->cy;
   pbmi->cy = 1;

   // Process each scanline (1 pel line). Compress it then send it to printer
   for (iCurrentRow = psizelBuffer->cy - 1;
        iCurrentRow >= (INT)(psizelBuffer->cy - cy);
        iCurrentRow--)
   {
      // Copy raster bits into our buffer
      pCMYReq->pbSrc = pbBits + iCurrentRow * pHandle->Req.iNumSrcRowBytes32;

      // First Dither scanline from an RGB format to CMYK format
      GplDitherRGBtoCMYK ((PPVOID)&pHandle->pdh,
                          pCMYReq,
                          (PDITHERESULT)&pHandle->Reply);

      // Is this a blank line (i.e. CMY and K planes all blank)
      if (GPL_DITHER_PLANES_BLANK (pHandle->Reply))
      {
         ++pHandle->Comp.ulBlankLineCount;
      }
      else
      {
         // We have determined we have data to send out, first output any
         // blank lines between current position and last real scanline
         if (pHandle->Comp.ulBlankLineCount)
         {
            if (pHandle->Comp.iCurrentCompMode != GPLCOMPRESS_NONE)   // Set no compression mode
               GplThreadWriteCommandWithPrintf (hThread,
                                                pCmds->ulCmdSetCompression,
                                                pCmds->pszCmdSetCompression,
                                                0);

            // Skip leading blank lines
            GplThreadWriteCommandWithPrintf (hThread,
                                             pCmds->ulCmdSetYPos,
                                             pCmds->pszCmdSetYPos,
                                             pHandle->Comp.ulBlankLineCount);

            pHandle->Comp.ulBlankLineCount = 0;

            if (pHandle->Comp.iCurrentCompMode != GPLCOMPRESS_NONE)
            {
               if (pHandle->Comp.iMaxCompMode > MAXCOMPRESSTIFF)
               {
                  GplCompressZeroLastLineBuffers ((PVOID)&(pHandle->Comp));
               }

               // Reset current compression mode to uncompressed
               pHandle->Comp.iCurrentCompMode = GPLCOMPRESS_NONE;
            }
         }

         /*--------------------------------*/
         /* Send out compressed KCMY data  */
         /*--------------------------------*/

         // If CMYK then we have to compress the Black plane 1st
         if (pddc->pdb->pPrintMode->ulColorTech == COLOR_TECH_CMYK)
         {
            CompressRasterPlane (pHandle->Reply.pbKBuffer,
                                 pHandle->Req.iNumDestRowBytes,
                                 FALSE,
                                 pHandle->Comp.pbLastLineK,
                                 pHandle->Comp.pbCompress,
                                 &pHandle->Comp.iCurrentCompMode,
                                 pHandle->Comp.ulSupportedCompModes,
                                 pvDDC,
                                 pHandle->Comp.pusDelta);
         }

         CompressRasterPlane (pHandle->Reply.pbXBuffer,
                              pHandle->Req.iNumDestRowBytes,
                              FALSE,
                              pHandle->Comp.pbLastLineX,
                              pHandle->Comp.pbCompress,
                              &pHandle->Comp.iCurrentCompMode,
                              pHandle->Comp.ulSupportedCompModes,
                              pvDDC,
                              pHandle->Comp.pusDelta);

         CompressRasterPlane (pHandle->Reply.pbYBuffer,
                              pHandle->Req.iNumDestRowBytes,
                              FALSE,
                              pHandle->Comp.pbLastLineY,
                              pHandle->Comp.pbCompress,
                              &pHandle->Comp.iCurrentCompMode,
                              pHandle->Comp.ulSupportedCompModes,
                              pvDDC,
                              pHandle->Comp.pusDelta);

         CompressRasterPlane (pHandle->Reply.pbZBuffer,
                              pHandle->Req.iNumDestRowBytes,
                              TRUE,
                              pHandle->Comp.pbLastLineZ,
                              pHandle->Comp.pbCompress,
                              &pHandle->Comp.iCurrentCompMode,
                              pHandle->Comp.ulSupportedCompModes,
                              pvDDC,
                              pHandle->Comp.pusDelta);
      }
   }

   pbmi->cy = saveCy;

   // If we exit raster graphics most PCL printers will reset to default
   // compression.  Therefore we must invalidate the current compression
   // mode to force us to reselect it next time through.
   pHandle->Comp.iCurrentCompMode = GPLCOMPRESS_INVALID;

   
   // We have determined we have data to send out, first output any
   // blank lines between current position and last real scanline
   if (pHandle->Comp.ulBlankLineCount)
   {
      if (pHandle->Comp.iCurrentCompMode != GPLCOMPRESS_NONE)   // Set no compression mode
         GplThreadWriteCommandWithPrintf (hThread,
                                          pCmds->ulCmdSetCompression,
                                          pCmds->pszCmdSetCompression,
                                          0);

      // Skip leading blank lines
      GplThreadWriteCommandWithPrintf (hThread,
                                       pCmds->ulCmdSetYPos,
                                       pCmds->pszCmdSetYPos,
                                       pHandle->Comp.ulBlankLineCount);

      pHandle->Comp.ulBlankLineCount = 0;

      if (pHandle->Comp.iCurrentCompMode != GPLCOMPRESS_NONE)
      {
         if (pHandle->Comp.iMaxCompMode > MAXCOMPRESSTIFF)
         {
            GplCompressZeroLastLineBuffers ((PVOID)&(pHandle->Comp));
         }

         // Reset current compression mode to uncompressed
         pHandle->Comp.iCurrentCompMode = GPLCOMPRESS_NONE;
      }
   }
   

   // Exit raster graphics transfer mode
   GplThreadWriteASCIICommand (pddc->pdb->hThread,
                     pddc->pdb->pDevice->pCmds->pszCmdEndRasterGraphics);

   // Update current position
   pddc->ptlPrintHead.y -= cy;  // Update physical print head at end

   //DBPRINTIF(( pDbg->bSUBRASTER,">>>>> %s: Exit <<<<<",__FUNCTION__ ));
   return FALSE;

} /* end SampPCLColorRasterize */

/****************************************************************************/
/* PROCEDURE NAME : SampPCLChooseRasterize                                  */
/* AUTHOR         : Matt Rutkowski & Tom Spurrier                           */
/* DATE WRITTEN   : 8/1/94                                                  */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* 11/11/94 - Matt Rutkowski - rewrote                                      */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
BOOL _System
SampPCLChooseRasterize (PBYTE            pbBits,
                        PBITMAPINFO2     pbmi,
                        PSIZEL           psizelBuffer,
                        PSIZEL           psizelPage,
                        PRECTL           prectlPageLocation,
                        PVOID            pvHandle,
                        PVOID            pvDDC)
{
   PDDC           pddc     = (PDDC)pvDDC;
   PSAMPPCLHANDLE pHandle  = (PSAMPPCLHANDLE)pvHandle;

   if (!pHandle->fHaveSetupPrinter)
   {
      SampPCLSetupPrinter (pvDDC, pvHandle);
      GplBookletResetOutput (pddc->pdb->hThread); 
   }

   switch (pddc->pdb->pPrintMode->ulColorTech)
   {
   case COLOR_TECH_K:   // ~~ test for now to be changed back
   {                    // to SampPCLMonoRasterize - see Matt

      SampPCLMonoRasterize (pbBits,
                            pbmi,
                            psizelBuffer,
                            psizelPage,
                            prectlPageLocation,
                            pvHandle,
                            pvDDC);
      break;
   }

   case COLOR_TECH_CMYK:
   case COLOR_TECH_CMY:
   {
      SampPCLColorRasterize (pbBits,
                             pbmi,
                             psizelBuffer,
                             psizelPage,
                             prectlPageLocation,
                             pvHandle,
                             pvDDC);

      break;
   }

   default:
   {
      assertstring ("Bad SampPCL Print Mode ID");
   }
   }

   return FALSE;

} /* end SampPCLChooseRasterize */

// ****************************************************************************
// * TABLES SECTION
// ****************************************************************************

//--------------------------------------------------------------
// Memory table: not used, but cannot be NULL in DEVICE table
//--------------------------------------------------------------
// Define all Memory types used by all SampPCL models
MEMORYINFO SampPCLMemory = {
  MEMORY_FIXED_SIZE,   // ulMemFlags;
  0,                   // ulBase;
  0,                   // ulMaximum;
  0                    // ulIncrement;
};

//----------------------------------
// GAMMA correction tables (NEW)
//----------------------------------
// Optional tables that derive a
// gamma value depending on current
// printer model, resolution, media
// and print mode.
//----------------------------------
ULONG abSampPCLModeTable [][2] = {
        MODE_ID_BLK_COLOR,  PCL_PMODE_ID_BLK_CLR   ,
        MODE_ID_BLK_COLOR,  PCL_PMODE_ID_24_BLK_CLR,
        MODE_ID_COLOR    ,  PCL_PMODE_ID_COLOR     ,
        MODE_ID_COLOR    ,  PCL_PMODE_ID_24_CLR    ,
        0,                  0
};

ULONG abSampPCLMediaTable [][2] = {
        MEDIA_ID_PLAIN     ,  PCL_MEDIA_ID_PLAIN   ,
        MEDIA_ID_TRANS     ,  PCL_MEDIA_ID_TRANS   ,
        MEDIA_ID_GLOSSY    ,  PCL_MEDIA_ID_GLOSSY  ,
        MEDIA_ID_1_SPECIAL ,  PCL_MEDIA_ID_SPECIAL ,
        0,0
};

ULONG abSampPCLResTable [][2] = {
        RES_ID_150,  PCL_RES_ID_DRAFT            ,
        RES_ID_300,  PCL_RES_ID_NORMAL           ,
        RES_ID_300,  PCL_RES_ID_PRESENTATION     ,
        0,           0
};

ULONG abSampPCLFamily[][2] = {
        1,  PRN_ID_SAMPPCL_K,
        1,  PRN_ID_SAMPPCL_CMY,
        1,  PRN_ID_SAMPPCL_CMYK,
        0,  0
};

GAMMA_STRUCT  abSampPCLGammaTable [] = {
 0, 0,          0,                  0,                 0,                   30, 30, 30, 30, 0,      // Element zero is the default
 1, RES_ID_150, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 20, 20, 20, 10, 0,
 1, RES_ID_150, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 20, 20, 20, 10, 0,
 1, RES_ID_150, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 20, 20, 20, 10, 0,
 1, RES_ID_150, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 20, 20, 20, 10, 0,

 1, RES_ID_300, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_MATRIX   , 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 32, 32, 32, 10, 0,
 1, RES_ID_300, MEDIA_ID_TRANS    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 26, 26, 26, 10, 0,
 1, RES_ID_300, MEDIA_ID_TRANS    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 26, 26, 26, 10, 0,
 1, RES_ID_300, MEDIA_ID_TRANS    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 26, 26, 26, 10, 0,  // tested
 1, RES_ID_300, MEDIA_ID_TRANS    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 26, 26, 26, 10, 0,  // tested
 1, RES_ID_300, MEDIA_ID_GLOSSY   , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 26, 26, 26, 10, 0,
 1, RES_ID_300, MEDIA_ID_GLOSSY   , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 26, 26, 26, 10, 0,
 1, RES_ID_300, MEDIA_ID_GLOSSY   , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 26, 26, 26, 10, 0,
 1, RES_ID_300, MEDIA_ID_GLOSSY   , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 26, 26, 26, 10, 0,

 // jk1115 start new gamma correction
 1, RES_ID_150, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 10,10,10,10,0,
 1, RES_ID_150, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 10,10,10,30,0,

 1, RES_ID_300, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 13,13,13,20,0,
 1, RES_ID_300, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 13,13,13,20,0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_TRANS    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 13,13,13,20,0,
 1, RES_ID_300, MEDIA_ID_TRANS    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_GLOSSY   , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 13,13,13,20,0,
 1, RES_ID_300, MEDIA_ID_GLOSSY   , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 13,13,13,30,0,

 1, RES_ID_150, MEDIA_ID_PLAIN,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION,  12,12,12,13,0,
 1, RES_ID_150, MEDIA_ID_PLAIN,     MODE_ID_COLOR,     DITHER_ID_NEW_DIFFUSION,  12,12,12,13,0,

 1, RES_ID_300, MEDIA_ID_PLAIN,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION,  13,13,13,32,0,
 1, RES_ID_300, MEDIA_ID_PLAIN,     MODE_ID_COLOR,     DITHER_ID_NEW_DIFFUSION,  13,13,13,32,0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION,  13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR,     DITHER_ID_NEW_DIFFUSION,  13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_TRANS,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION,  13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_TRANS,     MODE_ID_COLOR,     DITHER_ID_NEW_DIFFUSION,  13,13,13,30,0,
 1, RES_ID_300, MEDIA_ID_GLOSSY,    MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION,  13,13,13,29,0,
 1, RES_ID_300, MEDIA_ID_GLOSSY,    MODE_ID_COLOR,     DITHER_ID_NEW_DIFFUSION,  13,13,13,29,0,
 // jk1115 end

 // sx start new gamma correction
 1, RES_ID_150, MEDIA_ID_PLAIN,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX ,  36,36,36,13,0,
 1, RES_ID_150, MEDIA_ID_PLAIN,     MODE_ID_COLOR,     DITHER_ID_NEW_MATRIX ,  36,36,36,13,0,

 1, RES_ID_300, MEDIA_ID_PLAIN,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX ,  38,38,38,32,0,
 1, RES_ID_300, MEDIA_ID_PLAIN,     MODE_ID_COLOR,     DITHER_ID_NEW_MATRIX ,  38,38,38,32,0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX ,  38,38,38,30,0,
 1, RES_ID_300, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR,     DITHER_ID_NEW_MATRIX ,  38,38,38,30,0,
 1, RES_ID_300, MEDIA_ID_TRANS,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX ,  38,38,38,30,0,
 1, RES_ID_300, MEDIA_ID_TRANS,     MODE_ID_COLOR,     DITHER_ID_NEW_MATRIX ,  38,38,38,30,0,
 1, RES_ID_300, MEDIA_ID_GLOSSY,    MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX ,  38,38,38,29,0,
 1, RES_ID_300, MEDIA_ID_GLOSSY,    MODE_ID_COLOR,     DITHER_ID_NEW_MATRIX ,  38,38,38,29,0,
 // sx end

 0,0,0,0,0,0,0,0,0,0      // Terminate with zero elements
};

GAMMAINFO SAMPPCL_pGammas[] = {
  {
    (PULONG)abSampPCLFamily,                 // pFamily
    (PULONG)abSampPCLResTable,               // pRes;
    (PULONG)abSampPCLMediaTable,             // pMedia;
    (PULONG)abSampPCLModeTable,              // pMode;
    (PGAMMA_STRUCT)abSampPCLGammaTable,      // pSelect;
  }
};

// ****************************************************************************
// * PRINT QUALITY TABLES
// ****************************************************************************
// @TBD -  use name IDs NOT _STRING_ names

// Define all Print Qualities used by all SampPCL models
RESINFO SAMPPCL_pRes[] = {
 {  PCL_RES_ID_DRAFT,              // ulResID;
    RES_STRING_DRAFT,              // ulNameID;
    "",                            // szResName[32];
    150,                           // ulXRes;
    150,                           // ulYRes;
    5,                             // usCmdLength;
    _ESC_"*r1Q",                   // szCmdSelectRes[32];
    0,                             // ulCapabilities
    DJP_PQL_DRAFT,                 // ulDJPInfo
    0,                             // usPrintMethod;
    0,                             // usNumPhysicalPins;
    0,                             // bAdjacentDotPrinting;
    1,                             // usDotsPerColumn;
    0,                             // usBytesPerColumn;
    0,                             // usLenLineSpacing;
    _NUL_,                         // szCmdLineSpacing[8];
    0,                             // ulUnitsLineSpacing;
    0,                             // usLenVerticalDensity;
    _NUL_,                         // szCmdVerticalDensity[8];
    0                              // ulUnitsVerticalDensity;
  }
,
  { PCL_RES_ID_NORMAL,             // ulResID;
    RES_STRING_NORMAL,             // ulNameID;
    "",                            // szResName[32];
    300,                           // ulXRes;
    300,                           // ulYRes;
    5,                             // usCmdLength;
    _ESC_"*r2Q",                   // szCmdSelectRes[32];
    0,                             // ulCapabilities
    DJP_PQL_HIGH,                  // ulDJPInfo
    0,                             // usPrintMethod;
    0,                             // usNumPhysicalPins;
    0,                             // bAdjacentDotPrinting;
    1,                             // usDotsPerColumn;
    0,                             // usBytesPerColumn;
    0,                             // usLenLineSpacing;
    _NUL_,                         // szCmdLineSpacing[8];
    0,                             // ulUnitsLineSpacing;
    0,                             // usLenVerticalDensity;
    _NUL_,                         // szCmdVerticalDensity[8];
    0                              // ulUnitsVerticalDensity;
  }
,
  { PCL_RES_ID_PRESENTATION,       // ulResID;
    RES_STRING_PRESENTATION,       // ulNameID;
    "",                            // szResName[32];
    300,                           // ulXRes;
    300,                           // ulYRes;
    5,                             // usCmdLength;
    _ESC_"*r2Q",                   // szCmdSelectRes[32];
    0,                             // ulCapabilities
    DJP_PQL_HIGH,                  // ulDJPInfo
    0,                             // usPrintMethod;
    0,                             // usNumPhysicalPins;
    0,                             // bAdjacentDotPrinting;
    1,                             // usDotsPerColumn;
    0,                             // usBytesPerColumn;
    0,                             // usLenLineSpacing;
    _NUL_,                         // szCmdLineSpacing[8];
    0,                             // ulUnitsLineSpacing;
    0,                             // usLenVerticalDensity;
    _NUL_,                         // szCmdVerticalDensity[8];
    0                              // ulUnitsVerticalDensity;
  }
};
#define SAMPPCL_DEFINED_RESOLUTIONS       (sizeof (SAMPPCL_pRes)/sizeof (SAMPPCL_pRes[0]))

//****************************************************************************
//* PRINT MODE TABLES
//****************************************************************************
PRINTMODE SAMPPCL_pPrintModes[] =
{
  {
    PCL_PMODE_ID_BLACK,             // ulPrintModeID;
    COLOR_TECH_K,                   // ulColorTech;
    PRINT_MODE_STRING_BLACK_ONLY,   // ulNameID;
    "",                             // szPrintMode[32];
    1,                              // usBitsPerPel;   // Physical bit count
    8,                              // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    PCL_PMODE_ID_COLOR,             // ulPrintModeID;
    COLOR_TECH_CMY,                 // ulColorTech;
    PRINT_MODE_STRING_COLOR_ONLY,   // ulNameID;
    "",                             // szPrintMode[32];
    8,                              // usBitsPerPel;   // Physical bit count
    8,                              // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    PCL_PMODE_ID_BLK_CLR,           // ulPrintModeID;
    COLOR_TECH_CMYK,                // ulColorTech;
    PRINT_MODE_STRING_BLACK_COLOR,  // ulNameID;
    "",                             // szPrintMode[32];
    8,                              // usBitsPerPel;   // Physical bit count
    8,                              // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    PCL_PMODE_ID_24_CLR,            // ulPrintModeID;
    COLOR_TECH_CMY,                 // ulColorTech;
    PRINT_MODE_STRING_24_CLR,       // ulNameID;
    "",                             // szPrintMode[32];
    24,                             // usBitsPerPel;   // Physical bit count
    24,                             // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    PCL_PMODE_ID_24_BLK_CLR,        // ulPrintModeID;
    COLOR_TECH_CMYK,                // ulColorTech;
    PRINT_MODE_STRING_24_BLK_CLR,   // ulNameID;
    "",                             // szPrintMode[32];
    24,                             // usBitsPerPel;   // Physical bit count
    24,                             // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
}; /* end Print Modes */
#define SAMPPCL_DEFINED_PRINT_MODES       (sizeof (SAMPPCL_pPrintModes)/sizeof (SAMPPCL_pPrintModes[0]))

//****************************************************************************
//* FONT DEFINITION TABLES
//****************************************************************************
// Not implemented for this module


//****************************************************************************
//* TRAY DEFINITION TABLES
//****************************************************************************
// @TBD -  use name IDs NOT _STRING_ names

// Define all Trays used by all SampPCL models
TRAYINFO SAMPPCL_pTrays[] =
{
  {
     PCL_TRAY_ID_UPPER,                       // ulTrayID;
     TRAY_UPPER_TRAY,                         // ulTrayCap;
     DJP_TRY_UPPER,                           // ulDJPid;
     TRAY_STRING_TRAY,                        // ulNameID; (NOTE: string reads "Paper Tray")
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType;
     5,                                       // ulCmdSelectTray;
     _ESC_ "&l1H",                            // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_ENV,                         // ulTrayID;
     TRAY_ENV_FEED,                           // ulTrayCap;
     DJP_TRY_ENVELOPE,                        // ulDJPid;
     TRAY_STRING_MANUAL_ENV,                  // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_MANUAL,                        // ulTrayType;
     5,                                       // ulCmdSelectTray;
     _ESC_ "&l3H",                            // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_NOSELECT,                    // ulTrayID;
     TRAY_NONE,                               // ulTrayCap;
     DJP_TRY_MANUAL,                          // ulDJPid;
     TRAY_STRING_MANUAL_FEED,                 // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_MANUAL,                        // ulTrayType
     0,                                       // ulCmdSelectTray;
     "",                                      // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_50SHEET,                     // ulTrayID;
     TRAY_SHEET_FEEDER,                       // ulTrayCap;
     DJP_TRY_AUTO,                            // ulDJPid;
     TRAY_STRING_SHEET_FEEDER,                // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     0,                                       // ulCmdSelectTray;
     "",                                      // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_AUTO,                        // ulTrayID;
     TRAY_AUTO_FEEDER,                        // ulTrayCap;
     DJP_TRY_AUTO,                            // ulDJPid;
     TRAY_STRING_AUTO_FEEDER,                 // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     0,                                       // ulCmdSelectTray;
     "",                                      // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_MANUAL,                      // ulTrayID;
     TRAY_MANUAL_FEED,                        // ulTrayCap;
     DJP_TRY_MANUAL,                          // ulDJPid;
     TRAY_STRING_MANUAL_FEEDER,               // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_MANUAL,                        // ulTrayType
     0,                                       // ulCmdSelectTray;
     "",                                      // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_ENV2,                        // ulTrayID;
     TRAY_MANUAL_ENV,                         // ulTrayCap;
     DJP_TRY_ENVMANUAL,                       // ulDJPid;
     TRAY_STRING_ENV_FEED,                    // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType;
     5,                                       // ulCmdSelectTray;
     _ESC_ "&l3H",                            // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     PCL_TRAY_ID_PORTABLE,                    // ulTrayID;
     TRAY_PORTABLE_SHEET,                     // ulTrayCap;
     DJP_TRY_MANUAL,                          // ulDJPid;
     TRAY_STRING_PORTABLE_SHEET,              // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     0,                                       // ulCmdSelectTray;
     "",                                      // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
};
#define SAMPPCL_DEFINED_TRAYS             (sizeof (SAMPPCL_pTrays)/sizeof (SAMPPCL_pTrays[0]))

//****************************************************************************
//* FORM DEFINITION TABLES
//****************************************************************************
//* NOTE : We do not fill in xPels or yPels in this structure
//*        but determine this value dynamically based on
//*        current resolution
//*        also HCAPS_SELECTABLE and CURRENT are determined from
//*        printer and job props dynamically as well
//****************************************************************************

/*----------------------------------*/
/* Standard PCL margins             */
/*----------------------------------*/
FORMINFO2 SAMPPCL_pForms[] = {
  {
    PCL_FORM_ID_LETTER,         // ulFormID;
    FORM_LETTER,                // ulFormCap;
    DJP_PSI_LETTER,             // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LETTER,         // ulNameID;
    5,                          // usLenCmdSelectForm;
    { _ESC_ "&l2A" },           // szCmdSelectForm[LEN_CMD];
    {                           // hcInfo
      "",                       // szFormname[32] optional
      US_LETTER_WIDTH,          // cx; (mm)
      US_LETTER_HEIGHT,         // cy; (mm)
      630,                      // xLeftClip; (mm)
      1490,                     // yBottomClip; (mm)
      US_LETTER_WIDTH-630,      // xRightClip; (mm)
      US_LETTER_HEIGHT,         // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    },
    FALSE                       // bHasBeenSetup;
  }
,
  {
    PCL_FORM_ID_LEGAL,          // ulFormID;
    FORM_LEGAL,                 // ulFormCap;
    DJP_PSI_LEGAL,              // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LEGAL,          // ulNameID;
    5,                          // usLenCmdSelectForm;
    { _ESC_ "&l3A" },           // szCmdSelectForm[LEN_CMD];
    {                           // hcInfo
      "",                       // szFormname[32]
      LEGAL_WIDTH,              // cx; (mm)
      LEGAL_HEIGHT,             // cy; (mm)
      630,                      // xLeftClip; (mm)
      1490,                     // yBottomClip; (mm)
      LEGAL_WIDTH-630,          // xRightClip; (mm)
      LEGAL_HEIGHT-102,         // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    },
    FALSE                       // bHasBeenSetup;
  }
,
  {
    PCL_FORM_ID_A4,             // ulFormID;
    FORM_A4,                    // ulFormCap;
    DJP_PSI_A4,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A4,             // ulNameID;
    6,                          // usLenCmdSelectForm;
    { _ESC_ "&l26A" },          // szCmdSelectForm[LEN_CMD];
    {                           // hcInfo
      "",                       // szFormname[32] optional
      A4_WIDTH,                 // cx; (mm)
      A4_HEIGHT,                // cy; (mm)
      340,                      // xLeftClip; (mm)
      1490,                     // yBottomClip; (mm)
      A4_WIDTH-340,             // xRightClip; (mm)
      A4_HEIGHT-102,            // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    },
    FALSE                       // bHasBeenSetup;
  }
,
  {
    PCL_FORM_ID_EXECUTIVE,      // ulFormID;
    FORM_EXECUTIVE,             // ulFormCap;
    DJP_PSI_EXECUTIVE,          // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_EXECUTIVE,      // ulNameID;
    5,                          // usLenCmdSelectForm;
    { _ESC_ "&l1A" },           // szCmdSelectForm[LEN_CMD];
    {                           // hcInfo
      "",                       // szFormname[32] optional
      EXECUTIVE_WIDTH,          // cx; (mm)
      EXECUTIVE_HEIGHT,         // cy; (mm)
      640,                      // xLeftClip; (mm)
      1490,                     // yBottomClip; (mm)
      EXECUTIVE_WIDTH-640,      // xRightClip; (mm)
      EXECUTIVE_HEIGHT-102,     // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    },
    FALSE                       // bHasBeenSetup;
  }
,
  {
    PCL_FORM_ID_C10_ENV,        // ulFormID;
    FORM_C10_ENV,               // ulFormCap;
    DJP_PSI_ENV_C10,            // ulDJPid;
    FORM_CLASS_ENVELOPE,        // ulFormClass;
    FORM_STRING_C10_ENV,        // ulNameID;
    7,                          // usLenCmdSelectForm;
    { _ESC_ "&l-81A" },         // szCmdSelectForm[LEN_CMD];
    {                           // hcInfo
      "",                       // szFormname[32] optional
      C10_WIDTH,                // cx; (mm)
      C10_HEIGHT,               // cy; (mm)
      320,                      // xLeftClip; (mm)
      2130,                     // yBottomClip; (mm)
      C10_WIDTH-320,            // xRightClip; (mm)
      C10_HEIGHT-102,           // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    },
    FALSE                       // bHasBeenSetup;
  }
,
  {
    PCL_FORM_ID_DL_ENV,         // ulFormID;
    FORM_DL_ENV,                // ulFormCap;
    DJP_PSI_ENV_DL,             // ulDJPid;
    FORM_CLASS_ENVELOPE,        // ulFormClass;
    FORM_STRING_DL_ENV,         // ulNameID;
    7,                          // usLenCmdSelectForm;
    { _ESC_ "&l-90A" },         // szCmdSelectForm[LEN_CMD];
    {                           // hcInfo
      "",                       // szFormname[32] optional
      DL_WIDTH,                 // cx; (mm)
      DL_HEIGHT,                // cy; (mm)
      320,                      // xLeftClip; (mm)
      2800,                     // yBottomClip; (mm)
      DL_WIDTH-320,             // xRightClip; (mm)
      DL_HEIGHT-102,            // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    },
    FALSE                       // bHasBeenSetup;
  }
}; /* end Forms */
#define SAMPPCL_DEFINED_FORMS    (sizeof (SAMPPCL_pForms)/sizeof (SAMPPCL_pForms[0]))

//****************************************************************************
//* MEDIA DEFINITION TABLES
//****************************************************************************
MEDIAINFO SAMPPCL_pMedias[] = {
  {
    PCL_MEDIA_ID_PLAIN,           // ulMediaID;
    MEDIA_TYPE_PLAIN,             // ulMediaCap;
    DJP_MED_PLAIN,                // ulDJPid;
    MEDIA_STRING_PLAIN,           // ulNameID;
    "",                           // szMediaName[LEN_MEDIANAME];
    5,                            // ulCmdSelectMedia
    _ESC_ "&l0M",                 // szMediaTypeCmd;
    FALSE,                        // ulColorAdjustRequired
    HEAVY_ABSORPTION              // ulAbsorptionAdjust
  } /* end Plain */
,
  {
    PCL_MEDIA_ID_TRANS,           // ulMediaID;
    MEDIA_TYPE_TRANSPARENCY,      // ulMediaCap;
    DJP_MED_TRANSPARENCY,         // ulDJPid;
    MEDIA_STRING_TRANSPARENCY,    // ulNameID;
    "",                           // szMediaName[LEN_MEDIANAME];
    5,                            // ulCmdSelectMedia
    _ESC_ "&l4M",                 // szMediaTypeCmd;
    TRUE,                         // ulColorAdjustRequired
    NO_ABSORPTION                 // ulAbsorptionAdjust
  } /* end Transparency */
,
  {
    PCL_MEDIA_ID_GLOSSY,          // ulMediaID;
    MEDIA_TYPE_GLOSSY,            // ulMediaCap;
    DJP_MED_GLOSSY,               // ulDJPid;
    MEDIA_STRING_GLOSSY,          // ulNameID;
    "",                           // szMediaName[LEN_MEDIANAME];
    5,                            // ulCmdSelectMedia
    _ESC_ "&l3M",                 // szMediaTypeCmd;
    TRUE,                         // ulColorAdjustRequired
    NO_ABSORPTION                 // ulAbsorptionAdjust
  } /* end Glossy */
,
  {
    PCL_MEDIA_ID_SPECIAL,         // ulMediaID;
    MEDIA_TYPE_SPECIAL,           // ulMediaCap;
    DJP_MED_SPECIAL,              // ulDJPid;
    MEDIA_STRING_SPECIAL,         // ulNameID;
    "",                           // szMediaName[LEN_MEDIANAME];
    5,                            // ulCmdSelectMedia
    _ESC_ "&l2M",                 // szMediaTypeCmd;
    FALSE,                        // ulColorAdjustRequired
    LIGHT_ABSORPTION              // ulAbsorptionAdjust
  }  /* end Special */
}; /* end Medias */
#define SAMPPCL_DEFINED_MEDIAS  (sizeof (SAMPPCL_pMedias)/sizeof (SAMPPCL_pMedias[0]))

//****************************************************************************
//* FORM CONNECTION TABLES
//****************************************************************************
FORMCONNECTION SAMPPCL_pConnects[] = {
  //------------------------------------------------------------------
  // Standard mono and color PCL form sizes
  //------------------------------------------------------------------
  { CONN_ID_1,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_A4,        PCL_MEDIA_ID_PLAIN   },
  { CONN_ID_2,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_A4,        PCL_MEDIA_ID_TRANS   },
  { CONN_ID_3,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_A4,        PCL_MEDIA_ID_GLOSSY  },
  { CONN_ID_4,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_A4,        PCL_MEDIA_ID_SPECIAL },
  { CONN_ID_5,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_LETTER,    PCL_MEDIA_ID_PLAIN   },
  { CONN_ID_6,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_LETTER,    PCL_MEDIA_ID_TRANS   },
  { CONN_ID_7,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_LETTER,    PCL_MEDIA_ID_GLOSSY  },
  { CONN_ID_8,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_LETTER,    PCL_MEDIA_ID_SPECIAL },
  { CONN_ID_9,  PCL_TRAY_ID_UPPER,    PCL_FORM_ID_LEGAL,     PCL_MEDIA_ID_PLAIN   },
  { CONN_ID_10, PCL_TRAY_ID_UPPER,    PCL_FORM_ID_EXECUTIVE, PCL_MEDIA_ID_PLAIN   },
  { CONN_ID_11, PCL_TRAY_ID_UPPER,    PCL_FORM_ID_C10_ENV,   PCL_MEDIA_ID_PLAIN   },
  { CONN_ID_12, PCL_TRAY_ID_UPPER,    PCL_FORM_ID_DL_ENV,    PCL_MEDIA_ID_PLAIN   }
}; /* end Connects */

#define SAMPPCL_DEFINED_CONNECTIONS     (sizeof (SAMPPCL_pConnects)/sizeof (SAMPPCL_pConnects[0]))

//****************************************************************************
//* FEATURE SUPPORT
//****************************************************************************
FEATUREINFO SAMPPCL_Features = {
  FALSE,               // bSupUserDefTrays;
  TRUE,                // bSupUserDefForms;
  FALSE,               // bSupUserDefFonts;
  TRUE                 // bSupUserDefConnects;
}; /* end SAMPPCL_Features */

//****************************************************************************
//* COMMAND SUPPORT
//****************************************************************************
CMDINFO SAMPPCL_Commands = {
  7                                                  ,  // length
  _ESC_ ASCII(E) _ESC_ "&k0G"                        ,  // pszCmdInit;
  2                                                  ,  // length
  _ESC_ ASCII(E)                                     ,  // pszCmdTerm;
  2                                                  ,  // length
  _ESC_ ASCII(E)                                     ,  // pszCmdReset;
  7                                                  ,  // length
  _ESC_ ASCII(E) _ESC_ "&k0G"                        ,  // pszCmdAbort
  1                                                  ,  // length
  _FF_                                               ,  // pszCmdPageEject;
  5                                                  ,  // length
  _ESC_ "&l0O"                                       ,  // pszCmdPortrait
  5                                                  ,  // length
  _ESC_ "&l1O"                                       ,  // pszCmdLandscape
  9                                                  ,  // length
  _ESC_ "*p%dx%dY"                                   ,  // pszCmdSetXYPos
  6                                                  ,  // length
  _ESC_ "*p%dX"                                      ,  // pszCmdSetXPos
  6                                                  ,  // length
  _ESC_ "*b%dY"                                      ,  // pszCmdSetYPos
  2                                                  ,  // length
  _ESC_ ASCII(Z)                                     ,  // pszCmdSelfTest;
  5                                                  ,  // length
  _ESC_ "&s0C"                                       ,  // pszCmdEOLWrapOn;
  5                                                  ,  // length
  _ESC_ "&s1C"                                       ,  // pszCmdEOLWrapOff;
  6                                                  ,  // length
  _ESC_ "%k%dG"                                      ,  // pszCmdLineTerm;
  5                                                  ,  // length
  _ESC_ "&k0W"                                       ,  // pszCmdTextLtoR;
  5                                                  ,  // length
  _ESC_ "&K1W"                                       ,  // pszCmdBidi;
  5                                                  ,  // length
  _ESC_ "&k5W"                                       ,  // pszCmdTextScaleOn;
  5                                                  ,  // length
  _ESC_ "&k6W"                                       ,  // pszCmdTextScaleOff;
  5                                                  ,  // length
  _ESC_ "*r1A"                                       ,  // pszCmdBeginRasterGraphics;
  4                                                  ,  // length
  _ESC_ "*rC"                                        ,  // pszCmdEndRasterGraphicsLine;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdMoveToNextRasterGraphicsLine;
  5                                                  ,  // length
  _ESC_ "*rbC"                                       ,  // pszCmdEndRasterGraphics;
  6                                                  ,  // length
  _ESC_ "*t%dR"                                      ,  // pszCmdSetResolution;
  6                                                  ,  // length
  _ESC_ "*b%dM"                                      ,  // pszCmdSetCompression
  6                                                  ,  // length
  _ESC_ "*b%dW"                                      ,  // pszCmdTransferRasterBlock
  5                                                  ,  // length
  _ESC_ "&l0E"                                       ,  // pszCmdSetTopMargin;
  6                                                  ,  // length
  _ESC_ "*b%dV"                                      ,  // pszCmdTransferRasterPlane
}; /* end SAMPPCL_Commands */

//------------------------------------------
// Subclassed functions for SampPCL Driver
//------------------------------------------
FNSUBCLASS SAMPPCL_Sub_Functions = {
   SampPCLBeginJob,        // pfnInitJob;
   SampPCLChooseRasterize, // pfnRasterizeBand;
   SampPCLNewFrame,        // pfnNextPage;
   SampPCLEndJob,          // pfnTermJob;
   SampPCLAbortJob,        // pfnAbortJob;
   SampPCLDeviceQuery,     // pfnDeviceQuery;
   NULL,                   // pfnQueryFonts;
   NULL,                   // pfnDeviceOpen;
   NULL,                   // pfnDeviceClose;
   NULL,                   // pfnDeviceWrite;
   NULL,                   // pfnSpoolerOpen;
   NULL,                   // pfnSpoolerStartJob;
   NULL,                   // pfnSpoolerEndJob;
   NULL,                   // pfnSpoolerClose;
   (PFNBANDINGSUPPORT)NULL,// pfnBandingSupport;
   SampPCLQueryPDL         // pfnQueryPDL;
}; /* end SAMPPCL_Sub_Functions */

//****************************************************************************
//*
//*               S U P P O R T E D      F E A T U R E S
//*
//****************************************************************************
//*
//*  The following arrays contain IDs that reference tables defined above
//*  for the following features:
//*
//*         PRINT QUALITIES (e.g. resolutions)
//*         PRINT MODES     (e.g. color/mono)
//*         DEVICE FONTS
//*         PAPER TRAYS
//*         FORM SIZES
//*         MEDIA TYPES
//*         FORM CONNECTIONS (tray/form size/media type)
//*
//* NOTE: first value in array is default value for device
//****************************************************************************

//****************************************************************************
//* PRINT QUALITIES SUPPORTED (by device)
//****************************************************************************

//------------------------------------------------------------------
// Used by: DJ 500c, 540, 550c, 560c, Portable, 310, 320, 600, 600c
//------------------------------------------------------------------
ULONG SAMPPCL_pulResolutions[] = {
   PCL_RES_ID_NORMAL,
   PCL_RES_ID_DRAFT,
   PCL_RES_ID_PRESENTATION
};
#define SAMPPCL_SUPPORTED_RESOLUTIONS     (sizeof (SAMPPCL_pulResolutions)/sizeof (SAMPPCL_pulResolutions[0]))

//****************************************************************************
//* PRINT MODES SUPPORTED (by device)
//****************************************************************************

ULONG SAMPPCL_pulPrintModes_K[]   = {
   PCL_PMODE_ID_BLACK
};
#define SAMPPCL_SUPPORTED_PRINT_MODES_K (sizeof (SAMPPCL_pulPrintModes_K)/sizeof (SAMPPCL_pulPrintModes_K[0]))

ULONG SAMPPCL_pulPrintModes_CMY[] = {
   PCL_PMODE_ID_COLOR,
   PCL_PMODE_ID_BLACK,
   PCL_PMODE_ID_24_CLR
};
#define SAMPPCL_SUPPORTED_PRINT_MODES_CMY (sizeof (SAMPPCL_pulPrintModes_CMY)/sizeof (SAMPPCL_pulPrintModes_CMY[0]))

ULONG SAMPPCL_pulPrintModes_CMYK[] = {
   PCL_PMODE_ID_BLK_CLR,
   PCL_PMODE_ID_BLACK,
   PCL_PMODE_ID_COLOR,
   PCL_PMODE_ID_24_CLR,
   PCL_PMODE_ID_24_BLK_CLR
};
#define SAMPPCL_SUPPORTED_PRINT_MODES_CMYK (sizeof (SAMPPCL_pulPrintModes_CMYK)/sizeof (SAMPPCL_pulPrintModes_CMYK[0]))

//****************************************************************************
// TRAY TYPES SUPPORTED (by device)
//****************************************************************************
ULONG SAMPPCL_pulTrays[] = {
   PCL_TRAY_ID_UPPER
};
#define SAMPPCL_SUPPORTED_TRAYS  (sizeof (SAMPPCL_pulTrays)/sizeof (SAMPPCL_pulTrays[0]))

//****************************************************************************
//* MEDIA TYPES SUPPORTED (by device)
//****************************************************************************

ULONG SAMPPCL_pulMedias[] = {
   PCL_MEDIA_ID_PLAIN,
   PCL_MEDIA_ID_SPECIAL,
   PCL_MEDIA_ID_GLOSSY,
   PCL_MEDIA_ID_TRANS
};
#define SAMPPCL_SUPPORTED_MEDIAS (sizeof (SAMPPCL_pulMedias)/sizeof (SAMPPCL_pulMedias[0]))

/**************************************************************************
*                                                                         *
*                        C O N N E C T I O N S                            *
*                                                                         *
**************************************************************************/

//------------------------------------------------------------------
// Standard PCL Form Connections
//------------------------------------------------------------------
ULONG SAMPPCL_pulConnects[] = {
   CONN_ID_1,
   CONN_ID_2,
   CONN_ID_3,
   CONN_ID_4,
   CONN_ID_5,
   CONN_ID_6,
   CONN_ID_7,
   CONN_ID_8,
   CONN_ID_9,
   CONN_ID_10,
   CONN_ID_11,
   CONN_ID_12
};
#define SAMPPCL_SUPPORTED_CONNECTIONS   (sizeof (SAMPPCL_pulConnects)/sizeof (SAMPPCL_pulConnects[0]))


USERDEFDATA SAMPPCL_UserData = {
  FALSE, // BOOL             bTraysInit;
  0,     // USHORT           usNumTrays;
  NULL,  // PUSERTRAY        pUserTRAYS;
  NULL,  // PUSERTRAY        pLastUserTRAYS;
  FALSE, // BOOL             bFormsInit;
  0,     // USHORT           usNumForms;
  NULL,  // PUSERFORM        pUserFORMS;
  NULL,  // PUSERFORM        pLastUserFORMS;
  FALSE, // BOOL             bConnsInit;
  0,     // USHORT           usNumConnects;
  NULL,  // PUSERCONNECT     pUserCONNECTS;
  NULL,  // PUSERCONNECT     pLastUserCONNECTS;
  FALSE, // BOOL             bFontsInit;
  0,     // USHORT           usNumFonts;
  NULL,  // PUSERFONT        pUserFONTS;
  NULL   // PUSERFONT        pLastUserFONTS;
};

DEVICEINFO SAMPPCL_pDevices[] = {
  {
    PRN_ID_SAMPPCL_K,
    "Sample PCL Monochrome",                      // pszDeviceName;
    "Sample PCL Monochrome",                      // pszDeviceDesc;
    "IBM",                                        // pszDeviceVendor;
    1,                                            // usDeviceMajorVersion;
    0,                                            // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                     // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,              // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                              // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                         // Raster info
    0,                                            // ulDeviceBits;
    SAMPPCL_RLLDELTAROW,                          // compression supported
    &SampPCLMemory,                               // pMemory;
    &SAMPPCL_Commands,                            // pCommands;
    &SAMPPCL_Sub_Functions,                       // pSubclassedFunctions;
    &SAMPPCL_Features,                            // pFeatures;
    SAMPPCL_SUPPORTED_RESOLUTIONS,                // usNumRes;
    SAMPPCL_pulResolutions,                       // pulRES;
    0,                                            // usNumFonts;
    NULL,                                         // pFONTS;
    SAMPPCL_SUPPORTED_TRAYS,                      // usNumTrays;
    SAMPPCL_pulTrays,                             // pTRAYS;
    SAMPPCL_DEFINED_FORMS,                        // ulNumForms;
    SAMPPCL_pForms,                               // pFORMS;
    SAMPPCL_SUPPORTED_MEDIAS,                     // ulNumMedias;
    SAMPPCL_pulMedias,                            // pulMedias;
    NULL,                                         // pInkInfo;
    SAMPPCL_SUPPORTED_CONNECTIONS,                // usNumConnects;
    SAMPPCL_pulConnects,                          // pConnects;
    SAMPPCL_SUPPORTED_PRINT_MODES_K,
    SAMPPCL_pulPrintModes_K,
    0,                                            // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                      // pTripletModifiers;
    &SAMPPCL_UserData,
    {
       0,                         // cb        ***IGNORED***
       {0},                       // achDriver ***IGNORED***
       0,                         // ulSig     ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       CONN_ID_1,                 // ulDefConnID
       CONN_ID_5,                 // ulDefNonMetricConnID;
       0,                         // ulDefFontID
       PCL_RES_ID_NORMAL,         // ulDefResID
       PCL_PMODE_ID_BLACK,        // ulDefPrintModeID
       SAMPPCL_RLLDELTAROW,       // compression supported
       COLOR_BLACK,               // usForeground = COLOR_BACK
       COLOR_WHITE,               // usBackground = COLOR_WHITE
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_NONE,               // bFFControlType
       0,                         // ulJobPhaseControl
       0L,                        // Gamma Adjustment Red
       0L,                        // Gamma Adjustment Green
       0L,                        // Gamma Adjustment Blue
       0L,                        // Bias of Red Gamma
       0L,                        // Bias of Green Gamma
       0L,                        // Bias of Blue Gamma
       0L,                        // Duplex Options
       0,                         // Black Gamma Adjustment
       0,                         // Bias of Black Gamma
       0                          // Black Reduction %
    }
  }
,
  {
    PRN_ID_SAMPPCL_CMY,
    "Sample PCL Color-Only",                      // pszDeviceName;
    "Sample PCL Color-Only",                      // pszDeviceDesc;
    "IBM",                                        // pszDeviceVendor;
    1,                                            // usDeviceMajorVersion;
    0,                                            // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                     // ulOS2DeviceTechnology;
    OMNI_CAP_COLOR,                               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                              // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                         // Raster info
    PCL_SUPPORT_CMY_ONLY |
    PCL_SUPPORT_SHINGLE_DEPLETION,                // ulDeviceBits;
    SAMPPCL_RLLDELTAROW,                          // compression supported
    &SampPCLMemory,                               // pMemory;
    &SAMPPCL_Commands,                            // pCommands;
    &SAMPPCL_Sub_Functions,                       // pSubclassedFunctions;
    &SAMPPCL_Features,                            // pFeatures;
    SAMPPCL_SUPPORTED_RESOLUTIONS,                // usNumRes;
    SAMPPCL_pulResolutions,                       // pulRES;
    0,                                            // usNumFonts;
    NULL,                                         // pFONTS;
    SAMPPCL_SUPPORTED_TRAYS,                      // usNumTrays;
    SAMPPCL_pulTrays,                             // pTRAYS;
    SAMPPCL_DEFINED_FORMS,                        // ulNumForms;
    SAMPPCL_pForms,                               // pFORMS;
    SAMPPCL_SUPPORTED_MEDIAS,                     // ulNumMedias;
    SAMPPCL_pulMedias,                            // pulMedias;
    NULL,                                         // pInkInfo;
    SAMPPCL_SUPPORTED_CONNECTIONS,                // usNumConnects;
    SAMPPCL_pulConnects,                          // pConnects;
    SAMPPCL_SUPPORTED_PRINT_MODES_CMY,
    SAMPPCL_pulPrintModes_CMY,
    0,                                            // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                      // pTripletModifiers;
    &SAMPPCL_UserData,
    {
       0,                         // cb        ***IGNORED***
       {0},                       // achDriver ***IGNORED***
       0,                         // ulSig     ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       CONN_ID_1,                 // ulDefConnID
       CONN_ID_5,                 // ulDefNonMetricConnID;
       0,                         // ulDefFontID
       PCL_RES_ID_NORMAL,         // ulDefResID
       PCL_PMODE_ID_COLOR,        // ulDefPrintModeID
       SAMPPCL_RLLDELTAROW,       // compression supported
       COLOR_BLACK,               // usForeground = COLOR_BACK
       COLOR_WHITE,               // usBackground = COLOR_WHITE
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_NONE,               // bFFControlType
       0,                         // ulJobPhaseControl
       0L,                        // Gamma Adjustment Red
       0L,                        // Gamma Adjustment Green
       0L,                        // Gamma Adjustment Blue
       0L,                        // Bias of Red Gamma
       0L,                        // Bias of Green Gamma
       0L,                        // Bias of Blue Gamma
       0L,                        // Duplex Options
       0,                         // Black Gamma Adjustment
       0,                         // Bias of Black Gamma
       0                          // Black Reduction %
    }
  }
,
  {
    PRN_ID_SAMPPCL_CMYK,
    "Sample PCL Black & Color",                   // pszDeviceName;
    "Sample PCL Black & Color",                   // pszDeviceDesc;
    "IBM",                                        // pszDeviceVendor;
    1,                                            // usDeviceMajorVersion;
    0,                                            // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                     // ulOS2DeviceTechnology;
    OMNI_CAP_COLOR,                               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                              // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                         // Raster info
    PCL_SUPPORT_CMY_ONLY |
    PCL_SUPPORT_SHINGLE_DEPLETION |
    PCL_SUPPORT_GRAY_BALANCE,                     // ulDeviceBits;
    SAMPPCL_RLLDELTAROW,                          // compression supported
    &SampPCLMemory,                               // pMemory;
    &SAMPPCL_Commands,                            // pCommands;
    &SAMPPCL_Sub_Functions,                       // pSubclassedFunctions;
    &SAMPPCL_Features,                            // pFeatures;
    SAMPPCL_SUPPORTED_RESOLUTIONS,                // usNumRes;
    SAMPPCL_pulResolutions,                       // pulRES;
    0,                                            // usNumFonts;
    NULL,                                         // pFONTS;
    SAMPPCL_SUPPORTED_TRAYS,                      // usNumTrays;
    SAMPPCL_pulTrays,                             // pTRAYS;
    SAMPPCL_DEFINED_FORMS,                        // ulNumForms;
    SAMPPCL_pForms,                               // pFORMS;
    SAMPPCL_SUPPORTED_MEDIAS,                     // ulNumMedias;
    SAMPPCL_pulMedias,                            // pulMedias;
    NULL,                                         // pInkInfo;
    SAMPPCL_SUPPORTED_CONNECTIONS,                // usNumConnects;
    SAMPPCL_pulConnects,                          // pConnects;
    SAMPPCL_SUPPORTED_PRINT_MODES_CMYK,
    SAMPPCL_pulPrintModes_CMYK,
    0,                                            // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                      // pTripletModifiers;
    &SAMPPCL_UserData,
    {
       0,                         // cb        ***IGNORED***
       {0},                       // achDriver ***IGNORED***
       0,                         // ulSig     ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       CONN_ID_1,                 // ulDefConnID
       CONN_ID_5,                 // ulDefNonMetricConnID
       0,                         // ulDefFontID
       PCL_RES_ID_NORMAL,         // ulDefResID
       PCL_PMODE_ID_BLK_CLR,      // ulDefPrintModeID
       SAMPPCL_RLLDELTAROW,       // compression supported
       COLOR_BLACK,               // usForeground = COLOR_BACK
       COLOR_WHITE,               // usBackground = COLOR_WHITE
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_NONE,               // bFFControlType
       0,                         // ulJobPhaseControl
       0L,                        // Gamma Adjustment Red
       0L,                        // Gamma Adjustment Green
       0L,                        // Gamma Adjustment Blue
       0L,                        // Bias of Red Gamma
       0L,                        // Bias of Green Gamma
       0L,                        // Bias of Blue Gamma
       0L,                        // Duplex Options
       0,                         // Black Gamma Adjustment
       0,                         // Bias of Black Gamma
       0                          // Black Reduction %
    }
  }
};

#define SAMPPCL_SUPPORTED_DEVICES         (sizeof (SAMPPCL_pDevices)/sizeof (SAMPPCL_pDevices[0]))

DRIVERINFO pSAMPPCLDriver[] = {
  {
    "Sample PCL Driver",             // pszDriverName;
    "IBM",                           // pszDriverVendor;
    0x0210,                          // usDriverVersion;
    SAMPPCL_DEFINED_RESOLUTIONS,     // usNumRes;
    SAMPPCL_pRes,                    // pRES;
    0,                               // usNumFonts;
    NULL,                            // pFONTS;
    SAMPPCL_DEFINED_TRAYS,           // usNumTrays;
    SAMPPCL_pTrays,                  // pTRAYS;
    SAMPPCL_DEFINED_FORMS,           // usNumForms;
    SAMPPCL_pForms,                  // pFORMS;
    SAMPPCL_DEFINED_MEDIAS,          // ulNumMedias;
    SAMPPCL_pMedias,                 // pMEDIAS;
    SAMPPCL_DEFINED_CONNECTIONS,     // usNumConnects;
    SAMPPCL_pConnects,               // pCONNECTS;
    SAMPPCL_DEFINED_PRINT_MODES,     // usNumPrintModes;
    SAMPPCL_pPrintModes,             // pPrintModes;
    SAMPPCL_SUPPORTED_DEVICES,       // usNumDevices;
    SAMPPCL_pDevices,                // pDEVICES;
    SAMPPCL_pGammas                  // pGammaTables;
  }
};
