/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Omni Driver Enable File                                 */
/* SOURCE NAME    : SAMP_EPS.C                                              */
/* AUTHOR         : Mark Hamzy                                              */
/* DESCRIPTION    : Contains definitions and functions for adding sample    */
/*                  monochrome and color Escape(2)P language printers       */
/*                  that have 9 or 24 printheads to the "core" Omni         */
/*                  driver.                                                 */
/*                                                                          */
/*                  This simply requires defining the TRAYS, PAPER SIZES,   */
/*                  MEDIA TYPES, BITMAP FORMATS, RESOLUTIONS and FONTS      */
/*                  supported by the printers we will be supporting,        */
/*                                                                          */
/*                  and...  writing a few simple functions.                 */
/*                                                                          */
/*                  After filling these out, you should get basic output    */
/*                  to your printer models.  This basic support can then    */
/*                  be enhanced to add escape codes, compression and        */
/*                  dither algorithms unique to the printers you support.   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                          support                         */
/*                                                                          */
/****************************************************************************/

#define INCL_GENPLIB_THREAD_WRITE
#define INCL_GENPLIB_COMPRESS
#define INCL_GENPLIB_DBCS
#define INCL_GENPLIB_STRINGSORT
//#define INCL_GENPLIB_FAX
#define INCL_GENPLIB_LAYOUT  
#include <genplib.h>

//****************************************************************************
//*
//*               F E A T U R E    I D   #'s
//*
//****************************************************************************
//*
//*  The following IDs are used to reference specific features
//*  such as:
//*
//*         PRINT MODEL   (e.g. each printer model gets a unique ID)
//*         PRINT QUALITY (e.g. each resolution referenced gets a unique ID)
//*         PRINT MODE    ("")
//*         DEVICE FONT   ("")
//*         PAPER TRAY
//*         FORM SIZE
//*         MEDIA TYPE
//*         FORM CONNECTION
//*
//* !!!!!!!!!!!!!!!!!!!!!!! WARNING !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//*
//* NOTE: These IDs can NEVER be changed once added and shipped in an
//*       OS/2 product or cross-network printing could fail.
//*
//* !!!!!!!!!!!!!!!!!!!!!!! WARNING !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
//*
//****************************************************************************

//****************************************************************************
//* PRINT MODEL IDs
//****************************************************************************
//* Assign a new unique internal model ID for every new printer added
//****************************************************************************
#define PRN_ID_EPSON_9PIN_GENERIC_NARROW      2000
#define PRN_ID_EPSON_9PIN_GENERIC_WIDE        2001
#define PRN_ID_EPSON_24PIN_GENERIC_NARROW     2002
#define PRN_ID_EPSON_24PIN_GENERIC_WIDE       2003
#define PRN_ID_EPSON_48PIN_GENERIC            2004
#define PRN_ID_EPSON_ESC2P_GENERIC            2005

/*************************************************/
/* We made all the code and data based on VP-960 */
/* user reference. It has Japanese char set.     */
/*************************************************/
#define PRN_ID_GEN_ESCP24J84                  8000
#define LOCALBUFSIZE                          513

//****************************************************************************
//* PRINT QUALITY IDs
//****************************************************************************
//* Assign a new unique internal Print Quality ID for each printer model
//* that comes out with a new resolution/print quality
//****************************************************************************
#define EPSON_RES_ID_60_60                  0
#define EPSON_RES_ID_80_60                  1
#define EPSON_RES_ID_90_60                  2
#define EPSON_RES_ID_120_60                 3
#define EPSON_RES_ID_240_60                 4
#define EPSON_RES_ID_60_72                  5
#define EPSON_RES_ID_72_72                  6
#define EPSON_RES_ID_80_72                  7
#define EPSON_RES_ID_90_72                  8
#define EPSON_RES_ID_120_72                 9
#define EPSON_RES_ID_144_72                10
#define EPSON_RES_ID_240_72                11
#define EPSON_RES_ID_60_180                12
#define EPSON_RES_ID_120_180               13
#define EPSON_RES_ID_90_180                14
#define EPSON_RES_ID_180_180               15
#define EPSON_RES_ID_360_180               16
#define EPSON_RES_ID_60_360                17
#define EPSON_RES_ID_90_360                18
#define EPSON_RES_ID_120_360               19
#define EPSON_RES_ID_180_360               20
#define EPSON_RES_ID_360_360               21
#define EPSON_RES_ID_180_180_2P            22
#define EPSON_RES_ID_360_180_2P            23
#define EPSON_RES_ID_360_360_2P            24
#define EPSON_RES_ID_720_720_2P            25
#define EPSON_RES_ID_60_72_MX80            26
#define EPSON_RES_ID_120_72_MX80           27
#define EPSON_RES_ID_360_360_24            28
#define EPSON_RES_ID_360_720_2P            29

//****************************************************************************
//* FONTS IDs
//****************************************************************************
#define  FONT_ID_MINCHO_WIDE           0 // epsonmwd.fmf
#define  FONT_ID_MINCHO_10CPI          1 // epsonm10.fmf
#define  FONT_ID_MINCHO_12CPI          2 // epsonm12.fmf
#define  FONT_ID_MINCHO_15CPI          3 // epsonm15.fmf
#define  FONT_ID_GOTHIC_WIDE           4 // epsongwd.fmf
#define  FONT_ID_GOTHIC_10CPI          5 // epsong10.fmf
#define  FONT_ID_GOTHIC_12CPI          6 // epsong12.fmf
#define  FONT_ID_GOTHIC_15CPI          7 // epsong15.fmf

//****************************************************************************
//*  PRINT MODES IDs
//****************************************************************************
//* Assign a new unique internal Print Mode ID for each
//* bitmap format supported by output device and rendering s/w
//****************************************************************************
#define EPSON_PRINT_MODE_ID_MONO           1
#define EPSON_PRINT_MODE_ID_BLK_COLOR      2
#define EPSON_PRINT_MODE_ID_BLK_24BIT      3
#define EPSON_PRINT_MODE_ID_COLOR          4     // Add CMY only
#define EPSON_PRINT_MODE_ID_24BIT          5     //  "   "    "

#define RES_METHOD_EPSON_COLUMN            0x10
#define RES_METHOD_EPSON_ESC2P             0x20

#define EPSON_TRAY_ID_TRAY                 1
#define EPSON_TRAY_ID_CSF                  2
#define EPSON_TRAY_ID_CSF_ESCP2            3
#define EPSON_TRAY_ID_REAR_CONTINUOUS      4
#define EPSON_TRAY_ID_FRONT_CONTINUOUS     5
#define EPSON_TRAY_ID_SINGLE_SHEET         6
#define EPSON_TRAY_ID_MANUAL_FEED          7
#define EPSON_TRAY_ID_BIN1                 8
#define EPSON_TRAY_ID_BIN2                 9

#define EPSON_FORM_ID_LETTER               1
#define EPSON_FORM_ID_LEGAL                2
#define EPSON_FORM_ID_A3                   3
#define EPSON_FORM_ID_A4                   4
#define EPSON_FORM_ID_A5                   5
#define EPSON_FORM_ID_B5                   6
#define EPSON_FORM_ID_LEDGER               7
#define EPSON_FORM_ID_WIDE                 8

#define EPSON_MEDIA_ID_PLAIN               0
#define EPSON_MEDIA_ID_360_SPECIAL         1
#define EPSON_MEDIA_ID_720_SPECIAL         2

#define EPSON_CONN_ID_1                    1
#define EPSON_CONN_ID_2                    2
#define EPSON_CONN_ID_3                    3
#define EPSON_CONN_ID_4                    4
#define EPSON_CONN_ID_5                    5
#define EPSON_CONN_ID_6                    6
#define EPSON_CONN_ID_7                    7
#define EPSON_CONN_ID_8                    8
#define EPSON_CONN_ID_9                    9
#define EPSON_CONN_ID_10                   10
#define EPSON_CONN_ID_11                   11
#define EPSON_CONN_ID_12                   12
#define EPSON_CONN_ID_13                   13
#define EPSON_CONN_ID_14                   14
#define EPSON_CONN_ID_15                   15
#define EPSON_CONN_ID_16                   16
#define EPSON_CONN_ID_17                   17
#define EPSON_CONN_ID_18                   18
#define EPSON_CONN_ID_19                   19
#define EPSON_CONN_ID_20                   20
#define EPSON_CONN_ID_21                   21
#define EPSON_CONN_ID_22                   22
#define EPSON_CONN_ID_23                   23
#define EPSON_CONN_ID_24                   24
#define EPSON_CONN_ID_25                   25
#define EPSON_CONN_ID_26                   26
#define EPSON_CONN_ID_27                   27
#define EPSON_CONN_ID_28                   28
#define EPSON_CONN_ID_29                   29
#define EPSON_CONN_ID_30                   30
#define EPSON_CONN_ID_31                   31
#define EPSON_CONN_ID_32                   32
#define EPSON_CONN_ID_33                   33
#define EPSON_CONN_ID_34                   34
#define EPSON_CONN_ID_35                   35
#define EPSON_CONN_ID_36                   36
#define EPSON_CONN_ID_37                   37
#define EPSON_CONN_ID_38                   38
#define EPSON_CONN_ID_39                   39
#define EPSON_CONN_ID_40                   40
#define EPSON_CONN_ID_41                   41
#define EPSON_CONN_ID_42                   42
#define EPSON_CONN_ID_43                   43
#define EPSON_CONN_ID_44                   44
#define EPSON_CONN_ID_45                   45
#define EPSON_CONN_ID_46                   46
#define EPSON_CONN_ID_47                   47
#define EPSON_CONN_ID_48                   48
#define EPSON_CONN_ID_49                   49

#define SAMPEPSON_SIG     0x504F5350              // 'SEPS'
typedef struct _EpsonHandle {
   ULONG            cb;
   ULONG            ulSig;

   USHORT           usSpacingDividend;        // Line spacing

   ULONG            ulMoveDown;               // @176300 Y offset before printing
   ULONG            ulMoveRight;              // @176300 X offset before printing

   BOOL             fGraphicsHaveBeenSent;
   BOOL             fHaveEnteredGraphicsMode;
   BOOL             fHaveSetupPrinter;

   PBYTE            pbCompress;
   INT              cBytesInCompress;

   PBYTE            pbBuffer;                 // Intermediate Source buffer for bands @EPSON

   DITHEREQUEST     Req;
   DITHERESULT      Reply;
   PVOID            pdh;

   ULONG            ulSystemCharSet;                              //@DEVFNT @MATTR - moved to local from global
   HSORTER          hSort;
   /********************************************************/
   /* There is a case only text data will be printed to be */
   /* concerned.                                           */
   /********************************************************/
   BOOL             fTextHaveBeenSent;
   /********************************************************/
   /* For performance reason. There is a case that may     */
   /* textblt once a glyph. We allocate local buffer for it*/
   /********************************************************/
   PBYTE            pbStringBuf;
   /********************************************************/
   /* In some case there need to define system environmet  */
   /* and devices enviroment each DC.                      */
   /********************************************************/
   HSTRINOUT        hStrInOut;
} EPSONHANDLE, *PEPSONHANDLE;

BOOL _System  EpsonBeginJob          (PVOID        pv,
                                      PVOID        pvHandle);
VOID          EpsonSetupPrinter      (PDDC         pddc,
                                      PEPSONHANDLE pHandle);
BOOL _System  EpsonEndJob            (PVOID        pv,
                                      PVOID        pvHandle);
BOOL _System  EpsonAbortJob          (PVOID        pv,
                                      PABORT_DATA  pAbortData,
                                      PVOID        pvHandle);
BOOL _System  EpsonNewFrame          (PVOID        pv,
                                      PVOID        pvHandle);
ULONG _System EpsonDeviceQuery       (PVOID        pv1,
                                      ULONG        ulType,
                                      PVOID        pvArg1,
                                                   ...);
BOOL _System  EpsonChooseRasterize   (PBYTE        pbBits,
                                      PBITMAPINFO2 pbmi,
                                      PSIZEL       psizelBuffer,
                                      PSIZEL       psizelPage,
                                      PRECTL       prectlPageLocation,
                                      PVOID        pvHandle,
                                      PVOID        pvDDC);
VOID          SetYPosition           (PDDC         pddc,
                                      INT          iWorldY);
BOOL          MoveToYPosition        (PDDC         pddc,
                                      PVOID        pvHandle,
                                      INT          iWorldY,
                                      BOOL         fAbsolute);
BOOL          MoveToXPosition        (PDDC         pddc,
                                      PVOID        pvHandle,
                                      INT          iWorldX);


VOID
SetYPosition (PDDC pddc, INT iWorldY)       // DLLwork
{
   pddc->ptlPrintHead.y = iWorldY;
}

#define BUMP_TO_NEXT_MODULUS(n,m) ((((n)+(m)-1)/(m))*(m))

INT
GrabPrintHeadBand (PBYTE pbBits,                  // DLLwork
                   PBYTE pbBuffer,
                   INT   iMaxX,
                   INT   iCurrentY,
                   INT   iBytesPerColumn,
                   INT   iBytesInScanLine,
                   BOOL  fInterleaved,
                   PINT  piMaxRight)
{
   static BYTE  bMaxBit[256] = {
                              0, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              3, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              2, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              3, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              1, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              3, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              2, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              3, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8,
                              4, 8, 7, 8, 6, 8, 7, 8, 5, 8, 7, 8, 6, 8, 7, 8
   };
   static BYTE  bBitMask[8] = {0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01};
   static BYTE  bRemMask[8] = {0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE, 0xFF};
   BOOL         fAllZero    = TRUE;
   BYTE         currentbyte,
                columnbyte0, columnbyte1, columnbyte2, columnbyte3,
                columnbyte4, columnbyte5, columnbyte6, columnbyte7;
   INT          iPosInArray,
                iBytesInCX,
                iRemX,
                iMaxRight;
   INT          offset2, offset3, offset4, offset5, offset6, offset7;
   register INT i, j, x;

   // Sanity checks
   assertF (pbBits);
   assertF (pbBuffer);
   assertF (iMaxX);
   assertF (iBytesPerColumn);
   assertF (iBytesInScanLine);

   iMaxRight = 0;

   // How many bytes are there for the iMaxX bits?
   iBytesInCX = BUMP_TO_NEXT_MODULUS (iMaxX, 8) / 8;

   /* The for loop and the other tests expect the number to be one less than
   ** the real number.
   */
   iBytesInCX--;

   // How many bits are there in the last byte?
   iRemX = iMaxX & 7;
   if (0 == iRemX)
      /* Since cx is divisible by 8, the "end of the byte" is in effect and
      ** you want to address every bit in that byte.
      */
      iRemX = 8;
   assertF (1 <= iRemX && iRemX <= 8);

   // Calculate the offsets in advance
   offset2  = iBytesPerColumn << 1;
   offset3  = offset2 + iBytesPerColumn;
   offset4  = offset3 + iBytesPerColumn;
   offset5  = offset4 + iBytesPerColumn;
   offset6  = offset5 + iBytesPerColumn;
   offset7  = offset6 + iBytesPerColumn;

   /* Loop through every pel in the X direction one byte at a time.
   ** 8-bit move along the x axis.
   */
   for (x = 0; x <= iBytesInCX; x++)
   {
      /* Calculate the offset into the bitmap array (OS/2 PM defined).
      ** This is a 1-bpp bitmap format.  Note that x (in pels) runs in
      ** multiples of 8, so to access the bitmap data, you need to divide
      ** by 8.
      */
      iPosInArray = x + iCurrentY * iBytesInScanLine;

      /* Loop through the number of bytes in the printer head.
      ** 8-bit move along the y axis.
      */
      for (j = 0; j < iBytesPerColumn; j++)
      {
         // Clear out the column bits
         columnbyte0 = columnbyte1 = columnbyte2 = columnbyte3 =
         columnbyte4 = columnbyte5 = columnbyte6 = columnbyte7 = 0;

         if (iPosInArray >= 0)
         {
            /* Construct the 90 deg rotated bytes from column of bits.
            ** Loop through each scanline in the 8-bit byte of the printer head.
            ** 1-bit move along the y axis.
            */
            for (i = 0; i <= 7; i++)
            {
               // Grab the byte of bitmap data
               currentbyte = pbBits[iPosInArray];

               if (x == iBytesInCX)
                  /* If this is the last byte then, only look at the remaining
                  ** bits (mask off extraneous bits).
                  */
                  currentbyte &= bRemMask[iRemX - 1];

               // Is there something there?
               if (currentbyte)
               {
                  fAllZero = FALSE;

                  /* Keep track of the maximum bit that was set for this
                  ** byte of bitmap data.
                  */
                  iMaxRight = max (((x << 3) + bMaxBit[currentbyte]),
                                   iMaxRight);

                  /* Set the corresponding rotated bits in the column bytes.
                  */
                  if (currentbyte & 0x80) columnbyte0 |= bBitMask[i];
                  if (currentbyte & 0x40) columnbyte1 |= bBitMask[i];
                  if (currentbyte & 0x20) columnbyte2 |= bBitMask[i];
                  if (currentbyte & 0x10) columnbyte3 |= bBitMask[i];
                  if (currentbyte & 0x08) columnbyte4 |= bBitMask[i];
                  if (currentbyte & 0x04) columnbyte5 |= bBitMask[i];
                  if (currentbyte & 0x02) columnbyte6 |= bBitMask[i];
                  if (currentbyte & 0x01) columnbyte7 |= bBitMask[i];
               }

               /* Decrement the scan line pointer.  This has the effect of
               ** keeping x the same and moving one less for y.
               */
               iPosInArray -= iBytesInScanLine;
               if (fInterleaved)
                  iPosInArray -= iBytesInScanLine;

               if (iPosInArray < 0)
                  // We are outside the bitmap array.  Leave!
                  break;
            }
         }

         // Are we on the last byte?
         if (x == iBytesInCX)
         {
            /* Check the number of bits remaining before writing out the data.
            ** We are guarenteed at least one.
            */
            *(pbBuffer + j) = columnbyte0;
            if (1 < iRemX)
               *(pbBuffer + iBytesPerColumn + j) = columnbyte1;
            if (2 < iRemX)
               *(pbBuffer + offset2         + j) = columnbyte2;
            if (3 < iRemX)
               *(pbBuffer + offset3         + j) = columnbyte3;
            if (4 < iRemX)
               *(pbBuffer + offset4         + j) = columnbyte4;
            if (5 < iRemX)
               *(pbBuffer + offset5         + j) = columnbyte5;
            if (6 < iRemX)
               *(pbBuffer + offset6         + j) = columnbyte6;
            if (7 < iRemX)
               *(pbBuffer + offset7         + j) = columnbyte7;
         }
         else
         {
            /* Write the 8 column block of bitmap data.
            */
            *(pbBuffer +                   j) = columnbyte0;
            *(pbBuffer + iBytesPerColumn + j) = columnbyte1;
            *(pbBuffer + offset2         + j) = columnbyte2;
            *(pbBuffer + offset3         + j) = columnbyte3;
            *(pbBuffer + offset4         + j) = columnbyte4;
            *(pbBuffer + offset5         + j) = columnbyte5;
            *(pbBuffer + offset6         + j) = columnbyte6;
            *(pbBuffer + offset7         + j) = columnbyte7;
         }
      }

      /* We have done an 8 byte block for however many byte columns there
      ** are.  Increment the output buffer pointer...
      */
      pbBuffer += (iBytesPerColumn << 3);
   }

   // Return the maximum bit that was set in the x axis.
   if (piMaxRight)
      *piMaxRight = iMaxRight;

   // Return if this band is empty or not.
   return fAllZero;
}

/****************************************************************************/
/* PROCEDURE NAME : EPSONTextBlt                                            */
/* AUTHOR         : Matt Rutkowski                                          */
/* DATE WRITTEN   : 9/21/95                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*    typedef struct _TEXTBLTINFO {              // textbltinfo             */
/*            ULONG               ulLength;                                 */
/*            ULONG               flOptions;                                */
/*            ULONG               lGlyphCnt;                                */
/*            PLONG               pGlyphIndicies;                           */
/*            ULONG               ulFGMix;                                  */
/*            ULONG               ulBGMix;                                  */
/*            ULONG               ulFGColor;                                */
/*            ULONG               ulBGColor;                                */
/*            PBMAPINFO           pDstBmapInfo;                             */
/*            PDEVFONTINFO        pDevFntInfo;                              */
/*            ULONG               ulClpCnt;                                 */
/*            PRECTL              abrClipRects;                             */
/*            PPOINTL             aptlSrcOrg;                               */
/*            PBLTRECT            abrDst;                                   */
/*    } TEXTBLTINFO;                                                        */
/*    typedef TEXTBLTINFO *PTEXTBLTINFO;                                    */
/*                                                                          */
/*                                                                          */
/*    typedef struct _BLTRECT {         // bltrect                          */
/*       ULONG  ulXOrg;                                                     */
/*       ULONG  ulYOrg;                                                     */
/*       ULONG  ulXExt;                                                     */
/*       ULONG  ulYExt;                                                     */
/*    } BLTRECT;                                                            */
/*    typedef BLTRECT  *PBLTRECT;                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:   None.                                                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
LONG APIENTRY EPSONTextBlt( PTEXTBLTINFO ptbi, PDDC pddc )
{
   PULONG pulChar;
   ULONG  ul, ulIndex;
   ULONG  ul2 = 0;
   RECTL  rclTextBox;
   PEPSONHANDLE   pHandle;
   PSZ    pszGlyphs;
   APIRET Rc;

   /*Restrictions*******************************************/
   /* 0. GRE ClipText() in 32Devfnt.c has a fatal bug, that*/
   /*    cause a exception when it want to TextBlt.        */
   /* 1. We do not support Clip function as of date 08 Mar */
   /*    Because GRE should do clip and pass clipped str to*/
   /*    TextBlt.                                          */
   /* 2. DBCS glyph code never receive from GRE.           */
   /* 3. Default Jobproperties Panels's default font list  */
   /*    box never lists fonts list.                       */
   /* 4. GRE should handle when pfnTextBlt() reports an    */
   /*    error.                                            */
   /********************************************************/
   assertF( pddc );
   assertF( pddc->pdb );
   assertF( pddc->pdb->pvDeviceCookie );

   // Get a copy of our device's memory area
   pHandle     = (PEPSONHANDLE)pddc->pdb->pvDeviceCookie;

   assertF( pHandle->pbStringBuf );
   pszGlyphs   = pHandle->pbStringBuf;

   /**************************************************/
   /* Maybe we can move SorterCreateInstance and     */
   /* SorterDeleteInstance to EpsonDeviceQuery()     */
   /**************************************************/
   // Create string sorter if not already found
   if( !pHandle->hSort )
   {
       Rc = GplStringSorterCreateInstance( (HDC)pddc,
                                           &(pHandle->hSort),
                                           0L,
                                           0L );

       if (NO_ERROR != Rc)
          return RC_ERROR;
   }

   /***************************************************/
   /* For performance reason, we check number of bytes*/
   /* we need for making up a string here and select  */
   /* static buffer or dynamic buffer for handling    */
   /* string.                                         */
   /***************************************************/
   if( (ptbi->lGlyphCnt * 2 + 1) > LOCALBUFSIZE)
   {
      pszGlyphs   = (PSZ) GplMemoryAlloc( pddc->pdb->hmcbHeap,
                                          (ptbi->lGlyphCnt * 2 + 1));
      assertF( pszGlyphs);
      if( !pszGlyphs)
         return RC_ERROR;
   }
   /***************************************************/
   /* For safty reason, we make our string pointer to */
   /* the pointer already allocated. But remember its */
   /* size is LOCALBUFSIZE only. We have to trick some*/
   /* here.                                           */
   /***************************************************/
   if( !pszGlyphs)
   {
      pszGlyphs       = pHandle->pbStringBuf;
      if( ptbi->lGlyphCnt > LOCALBUFSIZE-1)
          ptbi->lGlyphCnt = LOCALBUFSIZE-1;
   }

#ifdef DEBUG
   if( ptbi->pDevFntInfo->fFontInfo & DFI_FIXED_FONT )
      DBPRINTF(( "EPSONTextBlt(): Fixed Width Font\n" ));

   if( ptbi->pDevFntInfo->fFontInfo & DFI_DBCS_FONT )
      DBPRINTF(( "EPSONTextBlt(): DBCS Font\n" ));
#endif

   pulChar = (PULONG)ptbi->pGlyphIndices;

   for ( ul = 0; ul < ptbi->lGlyphCnt; ul++ )
   {
      ulIndex = pulChar[ul];

      /* Specification ***************************/
      /* GlyphIndex indicates it's SBCS when it's*/
      /* below 0x400.                            */
      /*******************************************/
      if ( ulIndex < 0x400 )
      {
         *(pszGlyphs+ul+ul2)   = (CHAR)ulIndex;
      } /* end if */
      else
      {
         /*Gre Unimplemented*************************/
         /* We don't know but GRE did not pass glyph*/
         /* through this path.                      */
         /* This path should be tested when DBCS    */
         /* glyph passes through this path!!!!!     */
         /*******************************************/
         *(pszGlyphs+ul+ul2++) = (CHAR) (ulIndex >> 8);
         *(pszGlyphs+ul+ul2  ) = (CHAR)  ulIndex;
      } /* end else */
   } /* end for */
   *(pszGlyphs+ul+ul2) = 0;

   rclTextBox.xLeft   = ptbi->abrDst->ulXOrg;
   rclTextBox.yBottom = ptbi->abrDst->ulYOrg;

   rclTextBox.xRight  = rclTextBox.xLeft   + ptbi->abrDst->ulXExt;
   rclTextBox.yTop    = rclTextBox.yBottom + ptbi->abrDst->ulYExt;

#ifdef DEBUG
   DBPRINTF(( "EPSONTextBlt(): ( %4d, %4d ),", rclTextBox.xLeft, rclTextBox.yBottom));
   DBPRINTF(( "( %4d, %4d )\n", rclTextBox.xRight, rclTextBox.yTop));
   DBPRINTF(( "EPSONTextBlt(): String = %s\n", pszGlyphs));
#endif
   /*******************************************************/
   /* We insert textblted string here with text box and   */
   /* font information.                                   */
   /*******************************************************/
   Rc = GplStringSorterInsertString( pHandle->hSort,
                                     pszGlyphs,
                                     strlen (pszGlyphs) + 1,
                                     rclTextBox,
                                     (PVOID)ptbi->pDevFntInfo->ulUniqueFntID,
                                     NULL );
   if( Rc == NO_ERROR)
      Rc =  RC_SUCCESS;
   else
      Rc =  RC_ERROR;

   if( pszGlyphs && (pszGlyphs != pHandle->pbStringBuf ))
      GplMemoryFree( pszGlyphs);

   return Rc;

} /* end EPSONTextBlt */

BOOL _System
EpsonBeginJob (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc       = (PDDC)pv;
   PDEVICEINFO    pDevice    = pddc->pdb->pDevice;
   PDEVICESURFACE pds;


   if (pDevice->pCmds->ulCmdInit)
   {
      // Init printer
      GplThreadWriteBlock (pddc->pdb->hThread,
                           pDevice->pCmds->pszCmdInit,
                           pDevice->pCmds->ulCmdInit);
   }

   // Set printer's physical head position in our ddc
   pddc->ptlPrintHead.x = 0;
   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.yPels - 1;
   else
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.xPels - 1;

   /**************************************************************/
   /* We assume if there is a pointer point to a Fonts table, it */
   /* can deal device fonts printing.                            */
   /* Also, for landscape, can not do the text print, don't hook */
   /* the SoftDraw in the case.                                  */
   /**************************************************************/
   if( (pDevice->pulFONTS ) &&
       (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation))
   {
      // Subclass the text blt routine from SoftDraw.
      //We will handle device fonts
      pds  = (PDEVICESURFACE)pddc->pdb->pDeviceSurface;

      if( pds->pfnTextBlt != (PFN)EPSONTextBlt )
      {
         // Save GRE's original Text Blt function
         pddc->pdb->pfnTextBlt = (PFN)pds->pfnTextBlt;

         // @DEVFONT - Initialize pfnTextBlt to our function
         pds->pfnTextBlt = (PFN)EPSONTextBlt;

      } /* end if */

   } /* end if */

   return TRUE;

} /* end EpsonBeginJob */

VOID
EpsonSetupPrinter (PDDC pddc, PEPSONHANDLE pHandle)
{
   PDEVICEINFO    pDevice    = pddc->pdb->pDevice;
   PRESINFO       pResInfo   = pddc->pdb->pResInfo;
   PTRAYINFO      pTrayInfo  = pddc->pdb->pTrayInfo;
   PMEDIAINFO     pMediaInfo = pddc->pdb->pMediaInfo;
   PFORMINFO2     pFormInfo  = pddc->pdb->pFormInfo;

   assertF (pddc);
   assertF (pHandle);
   assertF (pDevice);
   assertF (pResInfo);
   assertF (pTrayInfo);
   assertF (pMediaInfo);

   pHandle->fHaveSetupPrinter = TRUE;

   // @176300 User defined form?
   pHandle->ulMoveRight = 0;
   pHandle->ulMoveDown  = 0;
   if (IsUserDefinedForm (pFormInfo))
   {
      double  dyClipPels;
      double  dxClipPels;

      dxClipPels = pFormInfo->hcInfo.xLeftClip;
      dyClipPels = pFormInfo->hcInfo.cy - pFormInfo->hcInfo.yTopClip;

      if (0.01 <= dyClipPels)
      {
         double  dyPels;

         dyPels  = dyClipPels;
         dyPels /= 25.40L;
         dyPels *= pResInfo->ulYRes;

         DBPRINTF (("```dyPels  = %f\n", dyPels));

         pHandle->ulMoveDown = dyPels;
      }

      if (0.01 <= dxClipPels)
      {
         double  dxPels;

         dxPels  = dxClipPels;
         dxPels /= 25.40L;
         dxPels *= pResInfo->ulXRes;

         DBPRINTF (("```dxPels  = %f\n", dxPels));

         pHandle->ulMoveRight = dxPels;
      }
   }

   /* ESC/P2 ???
   */
   if (RES_METHOD_EPSON_ESC2P == pResInfo->usPrintMethod)
   {
      PSZ   pszHack = _ESC_ "(C" HEX(02) HEX(00) "%W";
      float flSize;

      // @166185 - Conditionally enter into raster graphics mode...
      if (!pHandle->fHaveEnteredGraphicsMode)
      {
         pHandle->fHaveEnteredGraphicsMode = TRUE;

         // Enter into raster graphics mode...
         GplThreadWriteBlock (pddc->pdb->hThread,
                              pDevice->pCmds->pszCmdBeginRasterGraphics,
                              pDevice->pCmds->ulCmdBeginRasterGraphics);
      }

      // Set the units in x * 1/3600 units
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pDevice->pCmds->ulCmdSetResolution,
                                       pDevice->pCmds->pszCmdSetResolution,
                                       3600 / pResInfo->ulYRes);

      // Grab the form length
      if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
         flSize = pFormInfo->hcInfo.cy;
      else
         flSize = pFormInfo->hcInfo.cx;

      // @176300 User defined form?
      if (0 < pHandle->ulMoveDown)
      {
         flSize += pFormInfo->hcInfo.cy - pFormInfo->hcInfo.yTopClip;
      }

      // units are now in milimeters (not in hundred millimeters)
      flSize /= 25.40;

      flSize *= pResInfo->ulYRes;

      // Tell the printer the size of the form
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread, 6, pszHack, (INT)flSize);
   }
   else
   {
      // Set line spacing
      pHandle->usSpacingDividend = pResInfo->ulUnitsLineSpacing  *
                                     pResInfo->usDotsPerColumn   /
                                     pResInfo->ulYRes;

      if (pResInfo->usLenVerticalDensity)
         // Set vertical density
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          pResInfo->usLenVerticalDensity,
                                          pResInfo->szCmdVerticalDensity,
                                          pResInfo->ulUnitsVerticalDensity
                                          / pResInfo->ulYRes);

      // Set line spacing
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pResInfo->usLenLineSpacing,
                                       pResInfo->szCmdLineSpacing,
                                       pHandle->usSpacingDividend);

#if 0 /* @TEST */
      // Set form size for user defined forms
      if (!(pFormInfo->ulFormCap & FORM_CAP_FF_TO_END))
      {
         // Set the form size
         PSZ     pszHack = _ESC_ "C" "%c";
         float   flSize;
         BYTE    bResult;

         // Grab the form length
         if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
            flSize = pFormInfo->hcInfo.cy;
         else
            flSize = pFormInfo->hcInfo.cx;

         // units are now in milimeters (not in hundred millimeters)
         flSize /= 25.40;

DBPRINTF (("!!!!!!!!!!Form length (inches) = %f\n", flSize));
DBPRINTF (("!!!!!!!!!!Line spacing         = 1/%d\"\n", pResInfo->ulUnitsLineSpacing));
         flSize = flSize
                * (float)pResInfo->ulUnitsLineSpacing
                / (float)pHandle->usSpacingDividend;

         bResult = (BYTE)flSize;
DBPRINTF (("!!!!!!!!!!usSpacingDividend    = %d\n", pHandle->usSpacingDividend));
DBPRINTF (("!!!!!!!!!!result (float)       = %f\n", flSize));
DBPRINTF (("!!!!!!!!!!result (byte)        = %d\n", bResult));

         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          4,
                                          pszHack,
                                          bResult);
      }
#endif
   }

   // Set unidirectional
   if (pDevice->pCmds->ulCmdBidi)
   {
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pDevice->pCmds->ulCmdBidi,
                                       pDevice->pCmds->pszCmdBidi,
                                       '1');
   }

   // ESC/P2?
   if (RES_METHOD_EPSON_ESC2P == pResInfo->usPrintMethod)
   {
      if (((EPSON_RES_ID_720_720_2P == pResInfo->ulResID)        ||
           (EPSON_RES_ID_360_360_2P == pResInfo->ulResID)         ) &&
          (RES_METHOD_EPSON_ESC2P  == pResInfo->usPrintMethod)      &&
          ((EPSON_MEDIA_ID_360_SPECIAL == pMediaInfo->ulMediaID) ||
           (EPSON_MEDIA_ID_720_SPECIAL == pMediaInfo->ulMediaID)  )  )
      {
         PSZ  pszHack = _ESC_ "(i" HEX(01) HEX(00) "1";

         /* Turn on micro weave for special paper types...
         */
         GplThreadWriteBlock (pddc->pdb->hThread, pszHack, 6);
      }

   }

   // @176300 User defined form?
   if (0 < pHandle->ulMoveDown)
   {
      MoveToYPosition (pddc, pHandle, pHandle->ulMoveDown, TRUE);
   }
}

BOOL _System
EpsonEndJob (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc      = (PDDC)pv;
   PDEVICEINFO    pDevice   = pddc->pdb->pDevice;
   PEPSONHANDLE   pHandle   = (PEPSONHANDLE)pvHandle;
   ////PRESINFO       pResInfo  = pddc->pdb->pResInfo;
   PFORMINFO2     pFormInfo = pddc->pdb->pFormInfo;
   LONG           lMax;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      lMax = pddc->ptlPrintHead.y;
   else
      lMax = pddc->ptlPrintHead.x;

   // This is a zero based offset, so bump it up
   lMax++;
///assertT (lMax % pResInfo->usDotsPerColumn);  // Should be evenly divisible

DBPRINTF (("##########lMax                 = %d\n", lMax));
DBPRINTF (("##########current pos          = (%d, %d)\n",
           pddc->ptlPrintHead.x, pddc->ptlPrintHead.y));

   // If last page is blank don't print it
   // if (pHandle->fGraphicsHaveBeenSent &&
   //     0 < lMax                        )
   /****************************************************/
   /* If there is hSort allocated, text string is out, */
   /* So here we should send FF to printer.            */
   /****************************************************/
   if ((pHandle->fGraphicsHaveBeenSent || pHandle->hSort) &&
       0 < lMax                        )
   {
      if (pFormInfo->ulFormCap & FORM_CAP_FF_TO_END)
         // Standard form
         GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                     pddc->pdb->pDevice->pCmds->pszCmdPageEject);
      else
         // User defined form /* @TEST */
         MoveToYPosition (pddc, pvHandle, -1, FALSE);
   }

   if (pDevice->pCmds->ulCmdTerm)
   {
      // Terminate printer
      GplThreadWriteBlock (pddc->pdb->hThread,
                           pDevice->pCmds->pszCmdTerm,
                           pDevice->pCmds->ulCmdTerm);
   }

   if( pHandle->hSort)
       GplStringSorterDeleteInstance( pHandle->hSort);

   return TRUE;

} /* end EpsonEndJob */

BOOL _System
EpsonAbortJob (PVOID pv, PABORT_DATA pAbortData, PVOID pvHandle)
{
   PDDC        pddc     = (PDDC)pv;
   PRESINFO    pResInfo = pddc->pdb->pResInfo;
   ULONG       ulSize;
   INT         cx;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      cx = pddc->pdb->pFormInfo->hcInfo.xPels;
   else
      cx = pddc->pdb->pFormInfo->hcInfo.yPels;

   /* In raster transfer mode, if we were to stop sending data, then
   ** printer would lock up.  Therefore, send a whole bunch of zeroes
   ** that will fill up the rest of the raster transfer line.
   */
   if (RES_METHOD_EPSON_ESC2P != pResInfo->usPrintMethod)
      /* Use varying print band heights...
      */
      ulSize = pResInfo->usBytesPerColumn * (cx + 7) / 8;
   else
      /* The largest column size for ESC-P2 printers is 24 lines
      ** high.
      */
      ulSize = 24 * (cx + 7) / 8;

   pAbortData->bSendRepeatBlock   = TRUE;
   pAbortData->uchAbortChar       = 0;
   pAbortData->ulAbortRepeatCount = ulSize;

   pAbortData->bSendString    = TRUE;
   pAbortData->pszAbortString = pddc->pdb->pDevice->pCmds->pszCmdAbort;

   return TRUE;
}

BOOL _System
EpsonNewFrame (PVOID pv, PVOID pvHandle)
{
   PDDC           pddc      = (PDDC)pv;
   PEPSONHANDLE   pHandle   = (PEPSONHANDLE)pvHandle;
   PDEVICEINFO    pDevice   = pddc->pdb->pDevice;
///PRESINFO       pResInfo  = pddc->pdb->pResInfo;
   LONG           lMax;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      lMax = pddc->ptlPrintHead.y;
   else
      lMax = pddc->ptlPrintHead.x;

   // This is a zero based offset, so bump it up
   lMax++;
///assertT (lMax % pResInfo->usDotsPerColumn);  // Should be evenly divisible

DBPRINTF (("##########lMax                 = %d\n", lMax));
DBPRINTF (("##########current pos          = (%d, %d)\n",
           pddc->ptlPrintHead.x, pddc->ptlPrintHead.y));

   // if (pHandle->fGraphicsHaveBeenSent &&
   /****************************************************/
   /* Check text has been send or not, if text string  */
   /* is out, here we should send FF to printer.       */
   /****************************************************/
   if( (pHandle->fGraphicsHaveBeenSent || pHandle->fTextHaveBeenSent) &&
       0 < lMax                        )
   {

//
//      if (pFormInfo->ulFormCap & FORM_CAP_FF_TO_END)
//         // Standard form
//         GplThreadWriteASCIICommand (pddc->pdb->hThread,
//                                     pddc->pdb->pDevice->pCmds->pszCmdPageEject);
//      else
//         // User defined form /* @TEST */
//         MoveToYPosition (pddc, pvHandle, -1, FALSE);

      GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                  pddc->pdb->pDevice->pCmds->pszCmdPageEject);

   }

   if (pDevice->pCmds->ulCmdBidi)
   {
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pDevice->pCmds->ulCmdBidi,
                                       pDevice->pCmds->pszCmdBidi,
                                       '1');
   }

   // Set printer's physical head position in our ddc
   pddc->ptlPrintHead.x = 0;
   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.yPels - 1;
   else
      pddc->ptlPrintHead.y = pddc->pdb->pFormInfo->hcInfo.xPels - 1;

   // Reset error buffer for next page (for error diffusion alorithms)
   // so that errors from previous page's dither are not carried over
   GPL_DITHER_RESET_ERR_BUFFERS (pHandle->Req);

   pHandle->fGraphicsHaveBeenSent = FALSE;

   pHandle->Req.SrcOptions.fStartOfPage = TRUE;

   pHandle->fTextHaveBeenSent     = FALSE;
   if( pHandle->hSort)
       GplStringSorterReset( pHandle->hSort);

   return TRUE;

} /* end EsponNewFrame */

ULONG _System
EpsonDeviceQuery (PVOID pv1, ULONG ulType, PVOID pvArg1, ...)
{
   PDDC         pddc       = (PDDC)pv1;
   PRESINFO     pResInfo;
   PPRINTMODE   pPrintMode;
   ULONG        rc         = TRUE;
   PEPSONHANDLE pHandle    = NULL;
   va_list      list;
   register INT i;

   va_start (list, pvArg1);

   switch (ulType)
   {
   case DEVICE_ALLOC_HANDLE:
   {
      PEPSONHANDLE   pOldHandle = (PEPSONHANDLE)va_arg (list, PEPSONHANDLE);
      HMCB           hmcbHeap;
      INT            cx;
      INT            iBytesPerColumn,
                     iColumnSize,
                     iOutputBytesInArray;
      ULONG          ulRc;

      hmcbHeap = (HMCB)pvArg1;

      if (!pOldHandle)
      {
         pHandle = (PEPSONHANDLE)GplMemoryAlloc (hmcbHeap,
                                                 sizeof (EPSONHANDLE));
         assertF (pHandle);
      }
      else
         pHandle = pOldHandle;

      if (!pHandle)
         break;

      if (!pOldHandle)
         // Initilialize Epson's data
         memset (pHandle, 0, sizeof (EPSONHANDLE));

      pHandle->cb                    = sizeof (EPSONHANDLE);
      pHandle->ulSig                 = SAMPEPSON_SIG;
      pHandle->fGraphicsHaveBeenSent = FALSE;
      pHandle->fTextHaveBeenSent     = FALSE;
      pHandle->fHaveSetupPrinter     = FALSE;

      if (!pddc)
      {
         // Cannot go further without a pddc...
         rc = (ULONG)pHandle;
         break;
      }

      pResInfo   = pddc->pdb->pResInfo;
      pPrintMode = pddc->pdb->pPrintMode;

      if (pddc->pdb->ulNupPages == 2) 
         if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
            cx = pddc->pdb->pFormInfo->hcInfo.xPels;
         else
            cx = pddc->pdb->pFormInfo->hcInfo.yPels;

      iBytesPerColumn = pResInfo->usNumPhysicalPins / 8;

      iOutputBytesInArray = BUMP_TO_NEXT_MODULUS (cx, 8) / 8;

      /* Buffer allocations
      ** Intermediate Source partial band, work buffer
      */
      pHandle->pbBuffer = (PBYTE)GplMemoryAlloc (hmcbHeap,
                                                 cx * iBytesPerColumn);
      assertF (pHandle->pbBuffer);

      /*****************************************************/
      /* Allocate buffer for TextBlt                       */
      /*****************************************************/
      pHandle->pbStringBuf = (PBYTE)GplMemoryAlloc (hmcbHeap,
                                                    LOCALBUFSIZE);
      assertF (pHandle->pbStringBuf);

      /*****************************************************/
      /* Allocate text string attribute work buffer for ddc*/
      /*****************************************************/
      ulRc = GplDBCSConvertCreateInstance (pddc->pdb->hmcbHeap,
                                           &(pHandle->hStrInOut));
      assertF (pHandle->hStrInOut);

      if (ulRc)
         pHandle->hStrInOut = NULL;
      else if (pHandle->hStrInOut)
      {
         PSTRINOUT pStrInOut           = pHandle->hStrInOut;

         // Query Character Set used for this print job
         GplNLSQuerySystemCharSetID (pddc->pdb->ctryCode.country,
                                     &(pHandle->ulSystemCharSet));

         pStrInOut->ulInCharSet          = pHandle->ulSystemCharSet;
         pStrInOut->pszStringOut         = pHandle->pbStringBuf;
         pStrInOut->ulStrOutLen          = LOCALBUFSIZE;
         pStrInOut->ulInCharSetExtension = 0;

         if (pddc->pdb->ctryInfo.country == COUNTRY_JAPAN)
            pStrInOut->ulInCharEncoding  = CHAR_ENCODING_SHIFT_JIS;

         /*********************************************/
         /* Do we need to set default code page here ?*/
         /*********************************************/
      }

      // Compression buffer
      if (RES_METHOD_EPSON_ESC2P == pResInfo->usPrintMethod)
      {
         INT   iSize;

         iSize = iOutputBytesInArray * 24;
         iSize = iSize + iSize / 20;

         // Don't give a heap handle and you will get process memory (DosAlloc)
         // for large allocations such as this...
         pHandle->pbCompress = (PBYTE)GplMemoryAlloc (0, iSize);
         assertF (pHandle->pbCompress);

         pHandle->cBytesInCompress = iSize;
      }

      /* Are we color?
      */
      if ((COLOR_TECH_CMY  == pPrintMode->ulColorTech) ||
          (COLOR_TECH_CMYK == pPrintMode->ulColorTech)  )
      {
         /*------------------------------*/
         /*     Setup Dither Request     */
         /*------------------------------*/
         if (pResInfo->ulXRes >= pResInfo->ulYRes)
            pHandle->Req.ulResolution = pResInfo->ulXRes;
         else
            pHandle->Req.ulResolution = pResInfo->ulYRes;

         // number of bytes in Source, DwordAligned
         pHandle->Req.iNumSrcRowBytes32  = (BUMP_TO_NEXT_MODULUS (cx * pddc->bmp.cBitCount, 32)) / 8;

         if (RES_METHOD_EPSON_ESC2P == pResInfo->usPrintMethod)
         {
            iColumnSize = 24;
         }
         else
         {
            if (EPSON_RES_ID_360_360_24 == pResInfo->ulResID)
               iColumnSize = pResInfo->usNumPhysicalPins * 2;
            else
               iColumnSize = pResInfo->usNumPhysicalPins;
         }

         // Number of bytes in Destination, Byte Aligned
         pHandle->Req.iNumDestRowBytes = iOutputBytesInArray;
         pHandle->Req.iSrcRowPels      = cx;
         pHandle->Req.iNumDitherRows   = iColumnSize;
         pHandle->Req.iNumColors       = Power (2, pddc->bmp.cBitCount);

         if (COLOR_TECH_CMY == pPrintMode->ulColorTech)
            pHandle->Req.bReq = CMY_ALL;
         else if (COLOR_TECH_CMYK == pPrintMode->ulColorTech)
            pHandle->Req.bReq = CMYK_ALL;

         /* Init modify flag telling Gamma/HSV that we may replace/change
         ** values in the source bitmap.  1,2,4,8 bit pel bitmaps, then
         ** if C,M,Y are to be printed, the RGB index will be set to white.
         ** If 16, 24 bit bitmaps, then the R,G,B value is replaced with white.
         */
         pHandle->Req.SrcOptions.fModify = TRUE;

         /* check for override of Gamma and Bias from Job Properties.  This
         ** is used for calibrating new devices from Job Properties notebook.
         */
         i = FindGamma (pddc->pdb->pDriver->pGammas->pSelect,
                        pddc->pdb->pDriver->pGammas->pFamily,
                        (ULONG)pddc->pdb->pDevice->usDeviceID,
                        pddc->pdb->pDriver->pGammas->pRes,
                        pddc->pdb->pResInfo->ulResID,
                        pddc->pdb->pDriver->pGammas->pMedia,
                        pddc->pdb->pMediaInfo->ulMediaID,
                        pddc->pdb->pDriver->pGammas->pMode,
                        pPrintMode->ulPrintModeID,
                        pddc->pdb->pJobProperties->ulAlgorithm);

         {
            LONG lCGamma, lMGamma, lYGamma;

           // Use table lookup values to set default gammas for this device
           // NOTE: Red   gamma is really Cyan gamma
           //       Green gamma is really Magenta gamma
           //       Blue  gamma is really Yellow gamma
           pHandle->Req.lRGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bCGamma;
           pHandle->Req.lGGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bMGamma;
           pHandle->Req.lBGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bYGamma;
           pHandle->Req.lKGamma = (pddc->pdb->pDriver->pGammas->pSelect+i)->bKGamma;

           pHandle->Req.ulRBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;
           pHandle->Req.ulGBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;
           pHandle->Req.ulBBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;
           pHandle->Req.ulKBias = (ULONG)(pddc->pdb->pDriver->pGammas->pSelect+i)->bBias;

           // Since users want to truly adjust the colors Red, Green and Blue
           // and we as printers deal with Cyan Magenta and Yellow we need to
           // translate RGB to CMY ( R == M + Y, G == C + Y, and B = C + M )
           // So take and translate RGB adjustment values from Jon Properties
           GplGammaConvertRGBtoCMY (pddc->pdb->pJobProperties->lRedAdjust,
                                    pddc->pdb->pJobProperties->lGreenAdjust,
                                    pddc->pdb->pJobProperties->lBlueAdjust,
                                    &lCGamma,
                                    &lMGamma,
                                    &lYGamma);

           // Check for Gamma and Bias adjustment from Job Properties.
           // This dialog adjustment can be used to calibrate new device's default gammas
           pHandle->Req.lRGamma = pHandle->Req.lRGamma + lCGamma;
           pHandle->Req.lGGamma = pHandle->Req.lGGamma + lMGamma;
           pHandle->Req.lBGamma = pHandle->Req.lBGamma + lYGamma;
           pHandle->Req.lKGamma = pHandle->Req.lKGamma + pddc->pdb->pJobProperties->sBlackAdjust;

           pHandle->Req.ulRBias = (ULONG)((LONG)pHandle->Req.ulRBias + pddc->pdb->pJobProperties->lRedBias);
           pHandle->Req.ulGBias = (ULONG)((LONG)pHandle->Req.ulGBias + pddc->pdb->pJobProperties->lGreenBias);
           pHandle->Req.ulBBias = (ULONG)((LONG)pHandle->Req.ulBBias + pddc->pdb->pJobProperties->lBlueBias);
           pHandle->Req.ulKBias = (ULONG)((LONG)pHandle->Req.ulKBias + (LONG)pddc->pdb->pJobProperties->bBlackBias);
         }

         // *** HSV ***
         pHandle->Req.delta.lHue        = pddc->pdb->pJobProperties->lHue;
         pHandle->Req.delta.lSaturation = pddc->pdb->pJobProperties->lSaturation;
         pHandle->Req.delta.lValue      = pddc->pdb->pJobProperties->lValue;

         // Dither type
         pHandle->Req.lDitherType = pddc->pdb->pJobProperties->ulAlgorithm;
         GPL_DITHER_SETUP_DEFAULTS (pHandle->Req, pddc->pdb->pJobProperties->ulLevel);

         // Just Do It (tm)!
         GplDitherCreateInstance ((PPVOID)&pHandle->pdh,
                                  hmcbHeap,
                                  (PDITHEREQUEST)&pHandle->Req,
                                  (PDITHERESULT)&pHandle->Reply);
      }

      rc = (ULONG)pHandle;
      break;
   }

   case DEVICE_FREE_HANDLE:
   {
      BOOL fReAllocate = va_arg (list, BOOL);

      // Grab our handle
      pHandle = (PEPSONHANDLE)pvArg1;
      assertF (pHandle);

      if (pHandle)
      {
         assertT (SAMPEPSON_SIG != pHandle->ulSig);

         if (pHandle->pdh)
            // tell the dither function to deallocate all it's buffers
            GplDitherDeleteInstance ((PVOID)pHandle->pdh);

         if (pHandle->hStrInOut)
            GplDBCSConvertDeleteInstance (pHandle->hStrInOut);

         if (pHandle->pbStringBuf)
            GplMemoryFree(pHandle->pbStringBuf);

         // Free work buffers
         if (pHandle->pbBuffer)
            GplMemoryFree (pHandle->pbBuffer);

         if (pHandle->pbCompress)
            GplMemoryFree (pHandle->pbCompress);

         if (!fReAllocate)
            // Finally, free our handle!
            GplMemoryFree (pHandle);
      }

      rc = TRUE;
      break;
   }

   default:
      rc = FALSE;
      break;
   }

   va_end (list);

   return rc;
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
VOID
SampEPSFillInPDL (PPDL   pPDL,
                  ULONG  ulDeviceID,
                  ULONG  ulSubPDL)
{
   /* OK. Return PDL # ulSubPDL.
   */
   switch (ulSubPDL)
   {
   case 1:
   {
      pPDL->ulPDL                         = PDL_Epson;
      pPDL->ulMajorPDLVersion             = 1;
      pPDL->ulMinorPDLVersion             = 0;
      pPDL->szMinorCharID[0]              = '\0';
      pPDL->ulReserved                    = 0;
      pPDL->ulFlags                       = PDLCAPS_CURRENT | PDLCAPS_MACROSUPPORT;
      pPDL->szPDLDescription[0]           = '\0';
      pPDL->szTranslatedPDLDescription[0] = '\0';

      switch (ulDeviceID)
      {
      case PRN_ID_EPSON_9PIN_GENERIC_NARROW:
      case PRN_ID_EPSON_9PIN_GENERIC_WIDE:
         pPDL->ulPDLLevel = LEVEL_ESC;
         break;

      case PRN_ID_EPSON_24PIN_GENERIC_NARROW:
      case PRN_ID_EPSON_24PIN_GENERIC_WIDE:
      case PRN_ID_EPSON_48PIN_GENERIC:
         pPDL->ulPDLLevel = LEVEL_ESCP;
         break;

      case PRN_ID_EPSON_ESC2P_GENERIC:
         pPDL->ulPDLLevel = LEVEL_ESCP_2;
         break;
      }
      break;
   }

   default:
      assertstring ("Unknown sub pdl level number.\n");
      break;
   }
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
BOOL
SampEPSQueryPDL (ULONG  ulType,
                 PPDL   pPDL,
                 ULONG  ulDeviceID,
                 ...)
{
   va_list          list;

   va_start (list, ulDeviceID);

   switch (ulType)
   {
   case QUERYPDL_QUERY_NUMBER:
   {
      ULONG *pulNumPDLs = va_arg (list, ULONG *);

      /* Return the number of Printer Description Languages that you support
      ** based on the device id (ulDeviceID).  For this driver, we only support
      ** 1 language for every device.
      ** Ex:
      ** switch (ulDeviceID)
      ** {
      ** case DEVICE_A:
      ** case DEVICE_B: *pulNumPDLs = 1; break;
      ** case DEVICE_C: *pulNumPDLs = 2; break;
      ** }
      */
      *pulNumPDLs = 1;

      return TRUE;
   }

   case QUERYPDL_QUERY_PDL:
   {
      ULONG ulSubPDL = va_arg (list, ULONG);

      /* Return the Printer Description Languages.  This is called for each element
      ** in the array that you support (total number returned in QUERYPDL_QUERY_NUMBER).
      ** The numbering scheme is 1 .. *pulNumPDLs.
      */
      SampEPSFillInPDL (pPDL, ulDeviceID, ulSubPDL);

      return TRUE;
   }

   case QUERYPDL_ISOK_PDL:
   {
      ULONG ulSubPDL = va_arg (list, ULONG);
      PDL   MyPDL;

      /* Return whether or not this is a PDL that you support (essentially
      ** one that you filled in previously).
      */
      SampEPSFillInPDL (&MyPDL, ulDeviceID, ulSubPDL);

      if (MyPDL.ulPDL             == pPDL->ulPDL                 &&
          MyPDL.ulMajorPDLVersion == pPDL->ulMajorPDLVersion     &&
          MyPDL.ulMinorPDLVersion == pPDL->ulMinorPDLVersion     &&
          0 == strcmp (MyPDL.szMinorCharID, pPDL->szMinorCharID) &&
          MyPDL.ulReserved        == pPDL->ulReserved            &&
          MyPDL.ulFlags           == pPDL->ulFlags                )
      {
         return TRUE;
      }
      else
      {
         return FALSE;
      }
   }
   }

   va_end (list);

   return FALSE;
}

/****************************************************************************/
/* PROCEDURE NAME : MoveToYPosition                                         */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
BOOL
MoveToYPosition (PDDC   pddc,
                 PVOID  pvHandle,
                 INT    iWorldY,
                 BOOL   fAbsolute)
{
   PDEVICEINFO  pDevice  = pddc->pdb->pDevice;
   PRESINFO     pResInfo = pddc->pdb->pResInfo;
   PEPSONHANDLE pHandle  = (PEPSONHANDLE)pvHandle;
   INT          iAmount,
                iResult,
                iRemainder;

   if (fAbsolute)
   {
      iAmount = iWorldY;
   }
   else
   {
      if (pddc->ptlPrintHead.y == iWorldY)
         // No movement required
         return TRUE;

      if (pddc->ptlPrintHead.y < iWorldY)
      {
         // Wants to move backwards!
         DBPRINTF (("%s: Move Backwards!?! to %d when at %d", __FUNCTION__, iWorldY, pddc->ptlPrintHead.y ));
         return FALSE;
      }

      iAmount = pddc->ptlPrintHead.y - iWorldY;
   }

   if (pDevice->pCmds->ulCmdSetYPos)
   {
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pDevice->pCmds->ulCmdSetYPos,
                                       pDevice->pCmds->pszCmdSetYPos,
                                       iAmount);
   }
   else
   {
      /* We have to do it the hard way...
      */
      iResult    = iAmount / pResInfo->usDotsPerColumn;
      iRemainder = iAmount - pResInfo->usDotsPerColumn * iResult;

      PRINT_VAR (iAmount);
      PRINT_VAR (iResult);
      PRINT_VAR (iRemainder);

      if (iRemainder)
      {
         INT  iNewSpacing;

         iNewSpacing = pResInfo->ulUnitsLineSpacing
                     * iRemainder
                     / pResInfo->ulYRes;

         // Set new line spacing
         DBPRINTF (("New spacing = %d/%d\n", iNewSpacing, pResInfo->ulUnitsLineSpacing));
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          pResInfo->usLenLineSpacing,
                                          pResInfo->szCmdLineSpacing,
                                          iNewSpacing);

         // Move down
         GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                     pDevice->pCmds->pszCmdMoveToNextRasterGraphicsLine);
         GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                     pDevice->pCmds->pszCmdEndRasterGraphicsLine);

         // Restore old line spacing
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          pResInfo->usLenLineSpacing,
                                          pResInfo->szCmdLineSpacing,
                                          pHandle->usSpacingDividend);
      }

      if (iResult)
      {
         register INT i;

         for (i = iResult; i; i--)
         {
            // Move down
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdMoveToNextRasterGraphicsLine);
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdEndRasterGraphicsLine);
         }
      }
   }

   return TRUE;
}

/****************************************************************************/
/* PROCEDURE NAME : MoveToXPosition                                         */
/* AUTHOR         : Yu-Chih,Shih                                            */
/* DATE WRITTEN   : 2/09/96                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/****************************************************************************/
BOOL
MoveToXPosition (PDDC pddc, PVOID pvHandle, INT iWorldX)
{
   PDEVICEINFO  pDevice  = pddc->pdb->pDevice;
   PRESINFO     pResInfo = pddc->pdb->pResInfo;
   INT          iAmount;

   if (pddc->ptlPrintHead.x == iWorldX)
      // No movement required
      return TRUE;

   iAmount = iWorldX * (180 / pResInfo->ulXRes);

   if (pDevice->pCmds->ulCmdSetXPos)
   {
      GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                  pDevice->pCmds->pszCmdEndRasterGraphicsLine);
      GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                       pDevice->pCmds->ulCmdSetXPos,
                                       pDevice->pCmds->pszCmdSetXPos,
                                       iAmount);
   }
   else
   {
      assertstring ("I don't know how to move right!");
      return FALSE;
   }

   return TRUE;

} /* end MoveToXPosition */

/****************************************************************************/
/* PROCEDURE NAME : EpsonColumnRasterize                                    */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* 097499 - 11-17-94  Output postion commands for blank lines - mjones      */
/****************************************************************************/

BOOL _System
EpsonColumnRasterize (PBYTE         pbBits,
                      PBITMAPINFO2  pbmi,
                      PSIZEL        psizelBuffer,
                      PSIZEL        psizelPage,
                      PRECTL        prectlPageLocation,
                      PVOID         pvHandle,
                      PVOID         pvDDC)
{
   BOOL         bRC;
   BOOL         bBlankLines  = FALSE;
   BOOL         bStartofBand = TRUE;
   PDDC         pddc         = (PDDC)pvDDC;
   PDEVICEINFO  pDevice      = pddc->pdb->pDevice;
   PRESINFO     pResInfo     = pddc->pdb->pResInfo;
   PEPSONHANDLE pHandle      = (PEPSONHANDLE)pvHandle;
   ULONG        cy           = psizelBuffer->cy,
                cx           = psizelBuffer->cx;
   INT          iLinesPerBuffer,
                iNumBlocks,
                iColumnSize,
                iBytesPerColumn,
                iBytesInArray,
                iScanLineY,
                iWorldY,
                iMaxDataX;
   PBYTE        pbBuffer;
   BOOL         fAllZero;
   PSTRINOUT    pStrInOut    = pHandle->hStrInOut;
   BOOL         fHaveString  = FALSE;
   APIRET       StrSorterRc  = -1;
   PSZ          pszString;
   RECTL        rectlString;
   LONG         lUniqueFntID = 0, lCurrFontID;
   INT          iDeltaLinesY;

   DBPRINTF (("prectlPageLocation (%d, %d) - (%d, %d)\n",
              prectlPageLocation->xLeft,  prectlPageLocation->yBottom,
              prectlPageLocation->xRight, prectlPageLocation->yTop));
   iScanLineY      = cy-1;
   iLinesPerBuffer = BUMP_TO_NEXT_MODULUS (cy, pResInfo->usNumPhysicalPins);
   iColumnSize     = pResInfo->usNumPhysicalPins;
   iNumBlocks      = iLinesPerBuffer/iColumnSize;
   iBytesPerColumn = iColumnSize/8;
   iBytesInArray   = (BUMP_TO_NEXT_MODULUS (cx, 32)) >> 3;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      iWorldY      = prectlPageLocation->yTop;
   else
      iWorldY      = prectlPageLocation->xRight;

   pbBuffer = pHandle->pbBuffer;
   assertF (pbBuffer);

   // If we have created a string sorter, then we have been called
   // at pfnTextBlt and have at least 1 device font to output
   if (pHandle->hSort && pDevice->pulFONTS)
   {
      // Get first string in page band
      StrSorterRc = GplStringSorterGetFirstString (pHandle->hSort,
                                                   STR_INCL_PART_IN_BAND,
                                                   *prectlPageLocation,
                                                   &pszString,
                                                   &rectlString,
                                                   (PPVOID)&lUniqueFntID);
      if (StrSorterRc == NO_ERROR)
      {
         pHandle->fTextHaveBeenSent = TRUE;
         DBPRINTF (("EpsonColumnRasterize(): GetFirst(): string '%s' text box (%d,%d) (%d,%d)\n",
                    pszString,
                    rectlString.xLeft,
                    rectlString.yBottom,
                    rectlString.xRight,
                    rectlString.yTop
                    ));
      }

      lCurrFontID = lUniqueFntID;

      // Validate the font ID, represents a valid index into our device
      // font array.  If so, get a pointer to the new font structure
      // If no strings are found we default ot first font in array
      if (-(lCurrFontID+1) < pddc->pdb->pDevice->usNumFonts)
      {
         SetDevFntInfo (pddc, pStrInOut, lCurrFontID);
      }

      // If we have not yet allocated a buffer to hold output string
      // after codepoint is remapped for printers charset, do so now
      if (!pStrInOut->pszStringOut)
      {
         pStrInOut->pszStringOut = GplMemoryAlloc (pddc->pdb->hmcbHeap,
                                                   LOCALBUFSIZE);
         assertF (pStrInOut->pszStringOut);

         if (!pStrInOut->pszStringOut)
            return FALSE;

         pStrInOut->ulStrOutLen = LOCALBUFSIZE;
      }
   }

   for (; iNumBlocks && iWorldY >= 0; iNumBlocks--)
   {
      //Check if there are string to be send out in the column!!
      if (StrSorterRc == NO_ERROR)
      {
         // Check if we have string in this column
         if((rectlString.yTop) >= (iWorldY-iColumnSize))
         {
            fHaveString = TRUE;
         }
      }

      // Send image to printer first
      /* Move bits for the print so they are in the correct format for the
      ** printer.
      */
      fAllZero = GrabPrintHeadBand (pbBits,
                                    pbBuffer,
                                    cx,
                                    iScanLineY,
                                    iBytesPerColumn,
                                    iBytesInArray,
                                    FALSE,
                                    &iMaxDataX);

      if (fAllZero && !fHaveString)
         // all zero = blank line
         bBlankLines = TRUE;
      else
      {
         // not all zero
         // Process image first.
         if(!fAllZero)
         {
            if (bBlankLines)   // output leading blank lines
            {
               bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
               assertF (bRC);
               // Update current position
               SetYPosition (pddc, iWorldY);
               bBlankLines = FALSE;
               bStartofBand = FALSE;
            }
            else if (bStartofBand)  // check for motion for start of band
            {
               bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
               assertF (bRC);
               // Update current position
               SetYPosition (pddc, iWorldY);
               bStartofBand = FALSE;
            }

            if (!pHandle->fGraphicsHaveBeenSent)
            {
               PSZ  pszHack = _ESC_ "r0";

               // Start with black ink (default)
               GplThreadWriteBlock (pddc->pdb->hThread, pszHack, 3);
            }
            pHandle->fGraphicsHaveBeenSent = TRUE;

            // @176300
            if (pHandle->ulMoveRight        &&
                pDevice->pCmds->ulCmdSetXPos )
            {
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                pDevice->pCmds->ulCmdSetXPos,
                                                pDevice->pCmds->pszCmdSetXPos,
                                                pHandle->ulMoveRight);
            }

            // Write raster block header
            GplThreadWriteBlock (pddc->pdb->hThread,
                                 pResInfo->szCmdSelectRes,
                                 pResInfo->usCmdLength);

            // Count of raster data
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             2,
                                             "%w",
                                             iMaxDataX);

            // Write raster block
            GplThreadWriteBlock (pddc->pdb->hThread, pbBuffer, iMaxDataX * iBytesPerColumn);

            // Bring head to left most position
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdEndRasterGraphicsLine);

         }

         // If there are string in this column, send them out!
         // Otherwise send LF to printer.
         if (fHaveString)
         {
            // Send string out while string exist in the column.
            do
            {
               // We check if there is other string exist in the column.
               // So here, we turn off the flag for default.
               fHaveString = FALSE;

               // Calculate length to move to LT(left top) from head position.
               iDeltaLinesY = iWorldY - rectlString.yTop;

               if (bBlankLines || iDeltaLinesY)
               {
                  DBPRINTF (("EpsonColumnRasterize(): MoveYPos(): iWorldY = %d, iDeltaLinesY = %d\n",
                             iWorldY,
                             iDeltaLinesY));

                  bRC = MoveToYPosition (pddc, pHandle, rectlString.yTop, FALSE);
                  assertF (bRC);

                  SetYPosition (pddc, rectlString.yTop);
               }

               // Bring head to real X position.
               MoveToXPosition (pddc, pHandle, rectlString.xLeft);

               // Change font if need.
               if (lUniqueFntID != lCurrFontID)
               {
                  lCurrFontID = lUniqueFntID;

                  if (-(lCurrFontID+1) < pddc->pdb->pDevice->usNumFonts)
                  {
                     SetDevFntInfo (pddc, pStrInOut, lCurrFontID);
                  }
               }

               // Convert code point if need.
               /*************************************************/
               /* Japanese Version Specific.....                */
               /* In Japan We may have to do some code convert, */
               /* because it is different between OS and device */
               /* OS/2 Japanese uses Shift JIS Char Code, and   */
               /* Epson Device  uses JIS Char Code.             */
               /* Here we assume if text string attribute handle*/
               /* is allocated, it may have to do some convrsion*/
               /*************************************************/
               if (pStrInOut)
               {
                  BOOL      bError = FALSE;
                  ULONG     ulRcGplDbcs         = GPL_DBCS_NOERROR;

                  pStrInOut->pszStringIn        = pszString;
                  pStrInOut->ulStrInLen         = strlen( pszString);

                  /**********************************************/
                  /* After call GplDBCSCodeConvert, if out buf  */
                  /* size isn't enough, it will receive an err  */
                  /* GPL_DBCS_CONV_BUFOVERRUN, and reamined     */
                  /* unprocessed bytes will be returned in      */
                  /* STRINOUT.ulStrInLen. And pszStringIn is    */
                  /* pointed to next char to be processed.      */
                  /**********************************************/
                  do
                  {
                     do
                     {
                        ulRcGplDbcs = GplDBCSCodeConvert( pStrInOut);

                        // Check ALL!!! possible return codes to avoid infinate loop
                        switch( ulRcGplDbcs )
                        {
                        case GPL_DBCS_NOERROR:
                        case GPL_DBCS_CONV_BUFOVERRUN:
                        {
                           // Output string as converted to correct DBCS charset
                           GplThreadWriteASCIICommand (pddc->pdb->hThread, pStrInOut->pszStringOut);

                           DBPRINTF(("GplThreadWriteASCIICommand(): Output Text: '%s' to thread\n",
                                     pStrInOut->pszStringOut));
                           break;
                        }

                        case GPL_DBCS_NOTCONVERTED:
                        case GPL_DBCS_DBCSENV_INVALID:
                        case GPL_DBCS_CONV_UNKNOWNTYPE:
                        case GPL_DBCS_NOTIMPLEMENTED:
                        case GPL_DBCS_INVALID_CODEPAGE:
                        case GPL_DBCS_DOSAPI_ERROR:
                        case GPL_DBCS_USERENV_INVALID:
                        case GPL_DBCS_ERR_MEMORY_ALLOC:
                        case GPL_DBCS_ERR_MEMORY_FREE:
                        default:
                        {
                           DBPRINTF (("GplDBCSCodeConvert(): No Conversion/Error!!! rc=%x\n",
                                      ulRcGplDbcs));
                           assertT (ulRcGplDbcs);

                           // Output string passed in, no conversion
                           GplThreadWriteASCIICommand (pddc->pdb->hThread, pszString);

                           DBPRINTF (("GplThreadWriteASCIICommand(): Output Text: '%s' to thread\n",
                                      pszString));

                           bError = TRUE;
                           break;
                        }
                        }

                     } while (ulRcGplDbcs == GPL_DBCS_CONV_BUFOVERRUN && !bError);

                  } while ((pStrInOut->ulStrInLen != 0) && !bError);
               }
               else // If string need not conversion, send it out.
                  GplThreadWriteASCIICommand (pddc->pdb->hThread, pszString);

               // Get next string in string sorter.
               StrSorterRc = GplStringSorterGetNextString (pHandle->hSort,
                                                           &pszString,
                                                           &rectlString,
                                                           (PPVOID)&lUniqueFntID);

               DBPRINTF (("EpsonColumnRasterize(): GetNext(): String '%s' text box (%d,%d) (%d,%d)\n",
                          pszString,
                          rectlString.xLeft,
                          rectlString.yBottom,
                          rectlString.xRight,
                          rectlString.yTop));

               // If there still has string to be process...
               if (StrSorterRc == NO_ERROR)
               {
                  // Check the string exist in current column or not.
                  if ((rectlString.yTop) >= (iWorldY - iColumnSize))
                  {
                     DBPRINTF (("EpsonColumnRasterize(): String '%s' top coord (%d) >= %d \n",
                                pszString,
                                rectlString.yTop,
                                (iWorldY - iColumnSize)));

                     fHaveString = TRUE;
                  }
                  else
                  {
                     DBPRINTF (("EpsonColumnRasterize(): Current text '%s' above coordinate %d\n",
                                pszString,
                                (iWorldY - iColumnSize)));
                  }
               }
               else if (StrSorterRc == ERROR_NO_MORE_ITEMS)
               {
                  DBPRINTF (("EpsonColumnRasterize(): GetNext(): return 'ERROR_NO_MORE_ITEMS' (rc=%d)\n",
                             StrSorterRc));
               }
               else
               {
                  DBPRINTF (("EpsonColumnRasterize(): GetNext(): return 'Unknown' (rc=%d)\n",
                              StrSorterRc));
               }

               // Bring head to left most position.
               GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                           pDevice->pCmds->pszCmdEndRasterGraphicsLine);

            } while (fHaveString);

            // Move head down to next raster column if need.
            if (iDeltaLinesY != iColumnSize)
            {
               bRC = MoveToYPosition (pddc, pHandle, iWorldY - iColumnSize, FALSE);
               assertF (bRC);

               SetYPosition (pddc, iWorldY - iColumnSize);

               bBlankLines  = FALSE;
            }
         }
         else
         {
            if (  !OmniBookletEnabled (pddc) 
               || !GplJournalIsLastBand (pddc->hJournal)  
               || (  iNumBlocks - 1                       
                  && (iWorldY - iColumnSize) >= 0         
                  )                                       
               )                                          
            {
               // No String, send LF to printer to end raster operation.
               GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                           pDevice->pCmds->pszCmdMoveToNextRasterGraphicsLine);
            }
         }
      }

      iScanLineY -= iColumnSize;
      iWorldY    -= iColumnSize;

      if (!fAllZero)
         SetYPosition (pddc, iWorldY);
   }

   
   if (OmniBookletEnabled (pddc) && bBlankLines)   
   {
      bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
      assertF (bRC);
      // Update current position
      SetYPosition (pddc, iWorldY);
   }
   

   return TRUE;

} /* end EpsonColumnRasterize */

/****************************************************************************/
/* PROCEDURE NAME : EpsonColumn_360_24                                      */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
EpsonColumn_360_24 (PBYTE         pbBits,
                    PBITMAPINFO2  pbmi,
                    PSIZEL        psizelBuffer,
                    PSIZEL        psizelPage,
                    PRECTL        prectlPageLocation,
                    PVOID         pvHandle,
                    PVOID         pvDDC)
{
   BOOL         bRC;
   BOOL         bBlankLines  = FALSE;
   BOOL         bStartofBand = TRUE;
   PDDC         pddc         = (PDDC)pvDDC;
   PDEVICEINFO  pDevice      = pddc->pdb->pDevice;
   PRESINFO     pResInfo     = pddc->pdb->pResInfo;
   PEPSONHANDLE pHandle      = (PEPSONHANDLE)pvHandle;
   ULONG        cy           = psizelBuffer->cy,
                cx           = psizelBuffer->cx;
   INT          iLinesPerBuffer,
                iNumBlocks,
                iColumnSize,
                iBytesPerColumn,
                iBytesInArray,
                iScanLineY,
                iWorldY,
                iMaxDataX;
   USHORT       usNumVirtPins;
   PBYTE        pbBuffer;
   BOOL         fAllZero,
                fInterLeaved;

   /* The following code is for printers that support 1/360" line spacing, but
   ** do not support 1/360" pel size.  These are the printers which understand
   ** ESC/P but do not understand ESC/P2.  Let's use the LQ-510 as an example
   ** (a 24-pin printer).  The pel size is 1/180" of an inch but it can move in
   ** 1/360" increments.  So, in order to print out in 360 dpi, we must
   ** interleave 2 1/180" raster bands.  The second band is moved down 1/360".
   ** The effect looks sort of like this:
   ** band #1 =========> pin 1
   **                           pin 1  <============ band #2
   **                        2
   **                               2
   **                        3
   **                               3
   **                        4
   **                               4
   **                        5
   **                               5
   **                   etc...
   **                           etc...
   */
   iScanLineY      = cy - 1;
   usNumVirtPins   = pResInfo->usNumPhysicalPins * 2;
   iLinesPerBuffer = BUMP_TO_NEXT_MODULUS (cy, usNumVirtPins);
   iColumnSize     = usNumVirtPins;
   iNumBlocks      = iLinesPerBuffer / iColumnSize * 2;
   iBytesPerColumn = pResInfo->usNumPhysicalPins / 8;
   iBytesInArray   = (BUMP_TO_NEXT_MODULUS (cx, 32)) >> 3;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      iWorldY      = prectlPageLocation->yTop;
   else
      iWorldY      = prectlPageLocation->xRight;

   pbBuffer = pHandle->pbBuffer;
   assertF (pbBuffer);

   fInterLeaved = TRUE;
   for (; iNumBlocks && iWorldY >= 0; iNumBlocks--)
   {
      /* Move bits for the print so they are in the correct format for the
      ** printer.
      */
      fAllZero = GrabPrintHeadBand (pbBits,
                                    pbBuffer,
                                    cx,
                                    iScanLineY,
                                    iBytesPerColumn,
                                    iBytesInArray,
                                    TRUE,
                                    &iMaxDataX);

      if (fAllZero)
         // all zero = blank line
         bBlankLines = TRUE;
      else
      {
         // not all zero
         if (bBlankLines)   // output leading blank lines
         {
            bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
            assertF (bRC);
            // Update current position
            SetYPosition (pddc, iWorldY);
            bBlankLines = FALSE;
            bStartofBand = FALSE;
         }
         else if (bStartofBand)  // check for motion for start of band
         {
            bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
            assertF (bRC);
            // Update current position
            SetYPosition (pddc, iWorldY);
            bStartofBand = FALSE;
         }

         if (!pHandle->fGraphicsHaveBeenSent)
         {
            PSZ  pszHack = _ESC_ "r0";

            // Start with black ink (default)
            GplThreadWriteBlock (pddc->pdb->hThread, pszHack, 3);
         }

         pHandle->fGraphicsHaveBeenSent = TRUE;

         // @176300
         if (pHandle->ulMoveRight        &&
             pDevice->pCmds->ulCmdSetXPos )
         {
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             pDevice->pCmds->ulCmdSetXPos,
                                             pDevice->pCmds->pszCmdSetXPos,
                                             pHandle->ulMoveRight);
         }

         // Write raster block header
         GplThreadWriteBlock (pddc->pdb->hThread,
                              pResInfo->szCmdSelectRes,
                              pResInfo->usCmdLength);

         // Count of raster data
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          2,
                                          "%w",
                                          iMaxDataX);

         // Write raster block
         GplThreadWriteBlock (pddc->pdb->hThread,
                              pbBuffer,
                              iMaxDataX * iBytesPerColumn);

         // Dynamically change the vertical spacing
         GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                          pResInfo->usLenLineSpacing,
                                          pResInfo->szCmdLineSpacing,
                                          (fInterLeaved
                                              ? 1
                                              : (iColumnSize - 1)));

         // End the raster line
         GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                     pDevice->pCmds->pszCmdMoveToNextRasterGraphicsLine);

         if (  !OmniBookletEnabled (pddc) 
            || !GplJournalIsLastBand (pddc->hJournal)  
            || (  iNumBlocks - 1                       
               && (iWorldY - iColumnSize + 1) >= 0     
               )                                       
            )                                          
         {
            // Move to the next line
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdEndRasterGraphicsLine);
         }
      }

      if (fInterLeaved)
      {
         /* We need to interleave the following print head block with the
         ** current one.  Only move down one scan line.
         */
         fInterLeaved = FALSE;
         iScanLineY--;
         iWorldY--;
      }
      else
      {
         /* This was the matching print head block.  Move down the remaining
         ** "virtual" print head size.
         */
         fInterLeaved = TRUE;
         iScanLineY  -= iColumnSize - 1;
         iWorldY     -= iColumnSize - 1;
      }

      if (!fAllZero)
         SetYPosition (pddc, iWorldY);
   }

   
   if (OmniBookletEnabled (pddc) && bBlankLines)   
   {
      bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
      assertF (bRC);
      // Update current position
      SetYPosition (pddc, iWorldY);
   }
   

   return TRUE;
}

/****************************************************************************/
/* PROCEDURE NAME : EpsonColorColumnRasterize                               */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* 097499 - 11-17-94  Output postion commands for blank lines - mjones      */
/****************************************************************************/
BOOL _System
EpsonColorColumnRasterize (PBYTE        pbBits,
                           PBITMAPINFO2 pbmi,
                           PSIZEL       psizelBuffer,
                           PSIZEL       psizelPage,
                           PRECTL       prectlPageLocation,
                           PVOID        pvHandle,
                           PVOID        pvDDC)
{
   BOOL          bRC;
   BOOL          bBlankLines  = FALSE;
   BOOL          bStartofBand = TRUE;
   PDDC          pddc         = (PDDC)pvDDC;
   PDEVICEINFO   pDevice      = pddc->pdb->pDevice;
   PRESINFO      pResInfo     = pddc->pdb->pResInfo;
   PPRINTMODE    pPrintMode   = pddc->pdb->pPrintMode;
   PEPSONHANDLE  pHandle      = (PEPSONHANDLE)pvHandle;
   ULONG         cy           = psizelBuffer->cy,
                 cx           = psizelBuffer->cx;
   PDITHEREQUEST pCMYReq      = (PDITHEREQUEST)&pHandle->Req;
   INT           iLinesPerBuffer,
                 iNumBlocks,
                 iColumnSize,
                 iBytesPerColumn,
                 iScanLineY,
                 iWorldY,
                 iBytesInArray,
                 iOutputBytesInArray,
                 iMaxDataX;
   BOOL          fAllZero;
   PBYTE         pbBuffer;
   PBYTE         pbKBuffer;
   PBYTE         pbCBuffer;
   PBYTE         pbMBuffer;
   PBYTE         pbYBuffer;
   PBYTE         pbColorBits;
   INT           iPass,
                 iMaxPass,
                 iLastColor = -1;   /* Not a valid color to start with */
   PSZ           pszHack;

   iScanLineY           = cy - 1;
   iLinesPerBuffer      = BUMP_TO_NEXT_MODULUS (cy, pResInfo->usNumPhysicalPins);
   iColumnSize          = pResInfo->usNumPhysicalPins;
   iNumBlocks           = iLinesPerBuffer/iColumnSize;
   iBytesPerColumn      = iColumnSize/8;
   iOutputBytesInArray  = BUMP_TO_NEXT_MODULUS (cx*1, 8) / 8;
   iBytesInArray        = BUMP_TO_NEXT_MODULUS (cx*pbmi->cBitCount, 32) / 8;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      iWorldY = prectlPageLocation->yTop;
   else
      iWorldY = prectlPageLocation->xRight;

   if (COLOR_TECH_CMYK == pPrintMode->ulColorTech)
      iMaxPass = 4;
   else
      iMaxPass = 3;

   /* Grab the buffers from our handle
   */
   pbBuffer = pHandle->pbBuffer;
   assertF (pbBuffer);

   /* Reindex the CMYK buffer pointers dynamically since the output
   ** size can change.
   */
   pbKBuffer = pHandle->Reply.pbKBuffer;
   pbCBuffer = pHandle->Reply.pbXBuffer;
   pbMBuffer = pHandle->Reply.pbYBuffer;
   pbYBuffer = pHandle->Reply.pbZBuffer;

   /* Set up the dynamic parts of the separate request structure.
   */
   pCMYReq->pSrcInfo = pbmi;

   for (; iNumBlocks && iWorldY >= 0; iNumBlocks--)
   {
      if (iScanLineY + 1 < iColumnSize)   // if remainder of a printhead band
      {
         // Point to raster bits
         pCMYReq->pbSrc = pbBits;
         pbmi->cy = iScanLineY + 1;        // just do remainder lines
      }
      else
      {
         // Point to raster bits
         pCMYReq->pbSrc = pbBits + (iScanLineY + 1 - iColumnSize) * iBytesInArray;
         pbmi->cy = iColumnSize;
      }

      // Separate the colors!
      GplDitherRGBtoCMYK ((PPVOID)&pHandle->pdh,
                          (PDITHEREQUEST)&pHandle->Req,
                          (PDITHERESULT)&pHandle->Reply);

      // Is the entire block empty?
      if (GPL_DITHER_PLANES_BLANK (pHandle->Reply))
         fAllZero = TRUE;
      else
         fAllZero = FALSE;

      if (fAllZero)
         bBlankLines = TRUE;
      else
      {
         if (bBlankLines)    // output leading blank lines
         {
            bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
            assertF (bRC);
            SetYPosition (pddc, iWorldY);
            bBlankLines = FALSE;
            bStartofBand = FALSE;
         }
         else if (bStartofBand)  // check for motion at start of band
         {
            bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
            assertF (bRC);
            // Update current position
            SetYPosition (pddc, iWorldY);
            bStartofBand = FALSE;
         }

         pHandle->fGraphicsHaveBeenSent = TRUE;

         pbKBuffer = pHandle->Reply.pbKBuffer;
         pbCBuffer = pHandle->Reply.pbXBuffer;
         pbMBuffer = pHandle->Reply.pbYBuffer;
         pbYBuffer = pHandle->Reply.pbZBuffer;

         for (iPass = 0; iPass < iMaxPass; iPass++)
         {
            switch (iPass)
            {
            case 0:  /* YELLOW */
               if (pHandle->Reply.fEmpty.bYellow)
                  continue;

               pbColorBits = pbYBuffer;
               pszHack = _ESC_"r4";
               break;

            case 1:  /* MAGENTA */
               if (pHandle->Reply.fEmpty.bMagenta)
                  continue;

               pbColorBits = pbMBuffer;
               pszHack = _ESC_"r1";
               break;

            case 2:  /* CYAN */
               if (pHandle->Reply.fEmpty.bCyan)
                  continue;

               pbColorBits = pbCBuffer;
               pszHack = _ESC_"r2";
               break;

            case 3:  /* BLACK */
               if (pHandle->Reply.fEmpty.bBlack)
                  continue;

               pbColorBits = pbKBuffer;
               pszHack = _ESC_"r0";
               break;
            }

            if (iLastColor != iPass)
            {
               // Write color select command
               GplThreadWriteASCIICommand (pddc->pdb->hThread, pszHack);
               iLastColor = iPass;
            }

            // Build output buffer
            GrabPrintHeadBand (pbColorBits,
                               pbBuffer,
                               cx,
                               iColumnSize - 1,
                               iBytesPerColumn,
                               iOutputBytesInArray,
                               FALSE,
                               &iMaxDataX);

            // @176300
            if (pHandle->ulMoveRight        &&
                pDevice->pCmds->ulCmdSetXPos )
            {
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                pDevice->pCmds->ulCmdSetXPos,
                                                pDevice->pCmds->pszCmdSetXPos,
                                                pHandle->ulMoveRight);
            }

            // Write raster block header
            GplThreadWriteBlock (pddc->pdb->hThread,
                                 pResInfo->szCmdSelectRes,
                                 pResInfo->usCmdLength);

            // Count of raster data
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             2,
                                             "%w",
                                             iMaxDataX);

            // Write raster block
            GplThreadWriteBlock (pddc->pdb->hThread,
                                 pbBuffer,
                                 iMaxDataX * iBytesPerColumn);

            // End the raster line
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdEndRasterGraphicsLine);

         }  /* for (iPass = 0; iPass < 4; iPass++) */

         if (  !OmniBookletEnabled (pddc) 
            || !GplJournalIsLastBand (pddc->hJournal)  
            || (  iNumBlocks - 1                       
               && (iWorldY - iColumnSize) >= 0         
               )                                       
            )                                          
         {
            // Move to the next line
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdMoveToNextRasterGraphicsLine);
         }
      }  /* if (!fAllZero) */

      iScanLineY -= iColumnSize;
      iWorldY    -= iColumnSize;

      if (!fAllZero)
         SetYPosition (pddc, iWorldY);   // Update current position

   }  /* for (; iNumBlocks && iWorldY >= 0; iNumBlocks--) */

   
   if (OmniBookletEnabled (pddc) && bBlankLines)    
   {
      bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
      assertF (bRC);
      SetYPosition (pddc, iWorldY);
   }
   

   return TRUE;

} /* end EpsonColorColumnRasterize */

/****************************************************************************/
/* PROCEDURE NAME : EpsonColorColumn_360_24                                 */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
EpsonColorColumn_360_24 (PBYTE        pbBits,
                         PBITMAPINFO2 pbmi,
                         PSIZEL       psizelBuffer,
                         PSIZEL       psizelPage,
                         PRECTL       prectlPageLocation,
                         PVOID        pvHandle,
                         PVOID        pvDDC)
{
   BOOL          bRC;
   BOOL          bBlankLines  = FALSE;
   BOOL          bStartofBand = TRUE;
   PDDC          pddc         = (PDDC)pvDDC;
   PDEVICEINFO   pDevice      = pddc->pdb->pDevice;
   PRESINFO      pResInfo     = pddc->pdb->pResInfo;
   PPRINTMODE    pPrintMode   = pddc->pdb->pPrintMode;
   PEPSONHANDLE  pHandle      = (PEPSONHANDLE)pvHandle;
   ULONG         cy           = psizelBuffer->cy,
                 cx           = psizelBuffer->cx;
   PDITHEREQUEST pCMYReq      = (PDITHEREQUEST)&pHandle->Req;
   INT           iLinesPerBuffer,
                 iNumBlocks,
                 iColumnSize,
                 iBytesPerColumn,
                 iScanLineY,
                 iWorldY,
                 iBytesInArray,
                 iOutputBytesInArray,
                 iMaxDataX;
   BOOL          fAllZero,
                 fInterLeaved;
   USHORT        usNumVirtPins;
   PBYTE         pbBuffer;
   PBYTE         pbKBuffer;
   PBYTE         pbCBuffer;
   PBYTE         pbMBuffer;
   PBYTE         pbYBuffer;
   PBYTE         pbColorBits;
   INT           iPass,
                 iMaxPass,
                 iLastColor = -1;   /* Not a valid color to start with */
   PSZ           pszHack;

   /* The following code is for printers that support 1/360" line spacing, but
   ** do not support 1/360" pel size.  These are the printers which understand
   ** ESC/P but do not understand ESC/P2.  Let's use the LQ-510 as an example
   ** (a 24-pin printer).  The pel size is 1/180" of an inch but it can move in
   ** 1/360" increments.  So, in order to print out in 360 dpi, we must
   ** interleave 2 1/180" raster bands.  The second band is moved down 1/360".
   ** The effect looks sort of like this:
   ** band #1 =========> pin 1
   **                           pin 1  <============ band #2
   **                        2
   **                               2
   **                        3
   **                               3
   **                        4
   **                               4
   **                        5
   **                               5
   **                   etc...
   **                           etc...
   */
   iScanLineY           = cy - 1;
   usNumVirtPins        = pResInfo->usNumPhysicalPins * 2;
   iLinesPerBuffer      = BUMP_TO_NEXT_MODULUS (cy, usNumVirtPins);
   iColumnSize          = usNumVirtPins;
   iNumBlocks           = iLinesPerBuffer / iColumnSize * 2;
   iBytesPerColumn      = pResInfo->usNumPhysicalPins / 8;
   iOutputBytesInArray  = BUMP_TO_NEXT_MODULUS (cx*1, 8) / 8;
   iBytesInArray        = BUMP_TO_NEXT_MODULUS (cx*pbmi->cBitCount, 32) / 8;

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
      iWorldY = prectlPageLocation->yTop;
   else
      iWorldY = prectlPageLocation->xRight;

   if (COLOR_TECH_CMYK == pPrintMode->ulColorTech)
      iMaxPass = 4;
   else
      iMaxPass = 3;

   /* Grab the buffers from our handle
   */
   pbBuffer = pHandle->pbBuffer;
   assertF (pbBuffer);

   /* Reindex the CMYK buffer pointers dynamically since the output
   ** size can change.
   */
   pbKBuffer = pHandle->Reply.pbKBuffer;
   pbCBuffer = pHandle->Reply.pbXBuffer;
   pbMBuffer = pHandle->Reply.pbYBuffer;
   pbYBuffer = pHandle->Reply.pbZBuffer;

   /* Set up the dynamic parts of the separate request structure.
   */
   pCMYReq->pSrcInfo = pbmi;

   fInterLeaved = TRUE;
   for (; iNumBlocks && iWorldY >= 0; iNumBlocks--)
   {
      if (fInterLeaved)
      {
         if (iScanLineY + 1 < iColumnSize)   // if remainder of a printhead band
         {
            // Point to raster bits
            pCMYReq->pbSrc = pbBits;
            pbmi->cy = iColumnSize = iScanLineY + 1;
         }
         else
         {
            pbmi->cy = iColumnSize = usNumVirtPins;

            // Point to raster bits
            pCMYReq->pbSrc = pbBits + (iScanLineY + 1 - iColumnSize) * iBytesInArray;
         }

         // Separate the colors!
         GplDitherRGBtoCMYK ((PPVOID)&pHandle->pdh,
                             (PDITHEREQUEST)&pHandle->Req,
                             (PDITHERESULT)&pHandle->Reply);
      }

      // Is the entire block empty?
      if (GPL_DITHER_PLANES_BLANK (pHandle->Reply))
         fAllZero = TRUE;
      else
         fAllZero = FALSE;

      if (fAllZero)
         bBlankLines = TRUE;
      else
      {
         if (bBlankLines)    // output leading blank lines
         {
            bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
            assertF (bRC);
            SetYPosition (pddc, iWorldY);
            bBlankLines = FALSE;
            bStartofBand = FALSE;
         }
         else if (bStartofBand)  // check for motion at start of band
         {
            bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
            assertF (bRC);
            // Update current position
            SetYPosition (pddc, iWorldY);
            bStartofBand = FALSE;
         }

         pHandle->fGraphicsHaveBeenSent = TRUE;

         /* Reindex the CMYK buffer pointers dynamically since the output
         ** size can change.
         */
         pbKBuffer = pHandle->Reply.pbKBuffer;
         pbCBuffer = pHandle->Reply.pbXBuffer;
         pbMBuffer = pHandle->Reply.pbYBuffer;
         pbYBuffer = pHandle->Reply.pbZBuffer;

         for (iPass = 0; iPass < iMaxPass; iPass++)
         {
            switch (iPass)
            {
            case 0:  /* YELLOW */
               if (pHandle->Reply.fEmpty.bYellow)
                  continue;

               pbColorBits = pbYBuffer;
               pszHack = _ESC_"r4";
               break;

            case 1:  /* MAGENTA */
               if (pHandle->Reply.fEmpty.bMagenta)
                  continue;

               pbColorBits = pbMBuffer;
               pszHack = _ESC_"r1";
               break;

            case 2:  /* CYAN */
               if (pHandle->Reply.fEmpty.bCyan)
                  continue;

               pbColorBits = pbCBuffer;
               pszHack = _ESC_"r2";
               break;

            case 3:  /* BLACK */
               if (pHandle->Reply.fEmpty.bBlack)
                  continue;

               pbColorBits = pbKBuffer;
               pszHack = _ESC_"r0";
               break;
            }

            if (iLastColor != iPass)
            {
               // Write color select command
               GplThreadWriteASCIICommand (pddc->pdb->hThread, pszHack);
               iLastColor = iPass;
            }

            // Build output buffer
            GrabPrintHeadBand (pbColorBits,
                               pbBuffer,
                               cx,
                               iColumnSize - 1,
                               iBytesPerColumn,
                               iOutputBytesInArray,
                               TRUE,
                               &iMaxDataX);

            if (iMaxDataX)
            {
               // @176300
               if (pHandle->ulMoveRight        &&
                   pDevice->pCmds->ulCmdSetXPos )
               {
                  GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                   pDevice->pCmds->ulCmdSetXPos,
                                                   pDevice->pCmds->pszCmdSetXPos,
                                                   pHandle->ulMoveRight);
               }

               // Write raster block header
               GplThreadWriteBlock (pddc->pdb->hThread,
                                    pResInfo->szCmdSelectRes,
                                    pResInfo->usCmdLength);

               // Count of raster data
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                2,
                                                "%w",
                                                iMaxDataX);

               // Write raster block
               GplThreadWriteBlock (pddc->pdb->hThread,
                                    pbBuffer,
                                    iMaxDataX * iBytesPerColumn);
            }

            // Dynamically change the vertical spacing
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             pResInfo->usLenLineSpacing,
                                             pResInfo->szCmdLineSpacing,
                                             (fInterLeaved
                                                 ? 1
                                                 : (iColumnSize - 1)));

            // End the raster line
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdEndRasterGraphicsLine);

         }  /* for (iPass = 0; iPass < 4; iPass++) */

         if (  !OmniBookletEnabled (pddc) 
            || !GplJournalIsLastBand (pddc->hJournal)  
            || (  iNumBlocks - 1                       
               && (iWorldY - iColumnSize + 1) >= 0     
               )                                       
            )                                          
         {
            // Move to the next line
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdMoveToNextRasterGraphicsLine);
         }
      }  /* if (!fAllZero) */

      if (fInterLeaved)
      {
         /* We need to interleave the following print head block with the
         ** current one.  Only move down one scan line.
         */
         fInterLeaved = FALSE;
         iScanLineY--;
         iWorldY--;
      }
      else
      {
         /* This was the matching print head block.  Move down the remaining
         ** "virtual" print head size.
         */
         fInterLeaved = TRUE;
         iScanLineY  -= iColumnSize - 1;
         iWorldY     -= iColumnSize - 1;
      }

      if (!fAllZero)
         SetYPosition (pddc, iWorldY);   // Update current position

   }  /* for (; iNumBlocks && iWorldY >= 0; iNumBlocks--) */

   
   if (OmniBookletEnabled (pddc) && bBlankLines)    
   {
      bRC = MoveToYPosition (pddc, pHandle, iWorldY, FALSE);
      assertF (bRC);
      SetYPosition (pddc, iWorldY);
   }
   

   return TRUE;

} /* end EpsonColorColumnRasterize */

/****************************************************************************/
/* PROCEDURE NAME : CompressEpsonRLE                                        */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                  Data is organized as counter bytes followed by          */
/*                  data bytes.  Two types of counter bytes can be          */
/*                  used: repeat counters and data-length counters.         */
/*                                                                          */
/*                  repeat counters - specify the number of times (minus 1) */
/*                                    to repeat the following single byte   */
/*                                    of data.                              */
/*                                    -1 <= repeat counter <= -127          */
/*                                                                          */
/*                  data-length counters - specify the number of            */
/*                                    bytes (minus 1) of print data         */
/*                                    data following the counter.           */
/*                                    This data is only printed once.       */
/*                                                                          */
/*                           0 <= data-length counter <= 127                */
/*                                                                          */
/*     The first byte of compressed data must be a counter.                 */
/*     EX: -3 0 1 60 61 -4 15 expands into 0 0 0 0 60 61 15 15 15 15 15     */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                pbData,        // raster data to compress                 */
/*                pbReturn,      // compressed data will be written         */
/*                iTotalBytes )  // count of bytes in                       */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
INT APIENTRY
CompressEpsonRLE (PBYTE  pbData,
                  INT    cBytesInData,
                  PBYTE  pbReturn,
                  INT    cBytesInReturn)
{
   INT           iFrom = 0,
                 iTo   = 0,
                 dups;
   unsigned char data;
   unsigned char *puch;

   /* Test for a special cases
   */
   if (0 > cBytesInData)
   {
      // Negative count
      assertT (0 > cBytesInData);
      return 0;
   }

   // loop through the buffer
   while (iFrom < cBytesInData)
   {
      /* There have to be two or more chars to examine
      ** cBytesInData is 1 based and iFrom is 0 based
      */
      if (1 == cBytesInData - iFrom)
      {
         // just one character to compress
         *(pbReturn + iTo) = (unsigned char)'\0';
         iTo++;
         *(pbReturn + iTo) = (unsigned char)pbData[iFrom];
         iTo++;

         return iTo;
      }

      data = pbData[iFrom];            // seed start point
      iFrom++;                         // look at the next char

      // is the next byte equal?
      if (data == pbData[iFrom])
      {
         iFrom++;                      // point to the next data byte
         dups = 2;                     // initial count of dups

         // count dups and not go off the end
         while (iFrom < cBytesInData  &&
                data == pbData[iFrom] &&
                dups < 128             )
         {
            dups++;
            iFrom++;
         }

         *(pbReturn + iTo) = (unsigned char)(256 - dups + 1); // return count
         iTo++;
         *(pbReturn + iTo) = (unsigned char)data;             // return data
         iTo++;

         assertF (dups <= 128);
      }
      else
      {
         dups = 1;                     // initial count of non dups
         puch = pbReturn + iTo;        // Remember initial location of count
         iTo++;                        // Reserve its space

         *(pbReturn + iTo) = (unsigned char)data; // copy the data
         iTo++;                                   // move to next output

          // count non dups and not go off the end
         while (iFrom < cBytesInData &&
                dups < 128            )
         {
            if (iFrom == cBytesInData - 1         ||   // We are at the end
                pbData[iFrom] != pbData[iFrom + 1] )   // Still different
            {
               // copy the data
               *(pbReturn + iTo) = (unsigned char)*(pbData + iFrom);
               iTo++;
               dups++;
            }
            else
               // Not not the same... ok not different
               break;

            iFrom++;
         }

         *puch = (unsigned char)(dups - 1);       // update count

         assertF (dups <= 128);
      }
   }

   return iTo;

} /* end CompressEpsonRLE */

/****************************************************************************/
/* PROCEDURE NAME : EpsonEsc2PRasterize                                     */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
EpsonEsc2PRasterize (PBYTE        pbBits,
                     PBITMAPINFO2 pbmi,
                     PSIZEL       psizelBuffer,
                     PSIZEL       psizelPage,
                     PRECTL       prectlPageLocation,
                     PVOID        pvHandle,
                     PVOID        pvDDC)
{
   static INT   iBandSizes[]= {24, 8, 1, 0};
   static BYTE  Mask[8]     = {0xFF, 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE};
   PDDC         pddc        = (PDDC)pvDDC;
   PDEVICEINFO  pDevice     = pddc->pdb->pDevice;
   PRESINFO     pResInfo    = pddc->pdb->pResInfo;
   PMEDIAINFO   pMediaInfo  = pddc->pdb->pMediaInfo;
   PEPSONHANDLE pHandle     = (PEPSONHANDLE)pvHandle;
   ULONG        cy          = psizelBuffer->cy,
                cx          = psizelBuffer->cx,
                ulPageSize;
   INT          iBandSizeIdx  = 0,
                iNumScanLines,
                iScanLineY,
                iWorldY,
                iBytesInArray,
                iBytesInCX;
   BOOL         bDirty;
   PBYTE        pbBuffer;
   PBYTE        pbCompress;
   INT          iCompressed;
   INT          iRemainder;
   register INT sl;

   if (((EPSON_RES_ID_720_720_2P == pResInfo->ulResID)        ||
        (EPSON_RES_ID_360_360_2P == pResInfo->ulResID)         ) &&
       (RES_METHOD_EPSON_ESC2P  == pResInfo->usPrintMethod)      &&
       ((EPSON_MEDIA_ID_360_SPECIAL == pMediaInfo->ulMediaID) ||
        (EPSON_MEDIA_ID_720_SPECIAL == pMediaInfo->ulMediaID)  )  )
      /* Micro weaving has been turned on!  We must set a band size of 1! */
      iBandSizeIdx = 2;

   if (EPSON_RES_ID_360_720_2P == pResInfo->ulResID)
      iBandSizeIdx = 2;

   
   if (OmniBookletEnabled (pddc))   
      if (pddc->iShift == 0)
      {
         prectlPageLocation->yTop    += pddc->pdb->ulTopMove;
         prectlPageLocation->yBottom += pddc->pdb->ulTopMove;
         pddc->iShift                 = pddc->pdb->ulTopMove;
      }
      else
      {
         pddc->iShift                += pddc->pdb->ulTopMove;
      }
   

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
   {
      ulPageSize = psizelPage->cy;
      iWorldY    = prectlPageLocation->yTop;

      // @176300
      ulPageSize += pHandle->ulMoveDown;
   }
   else
   {
      ulPageSize = psizelPage->cx;
      iWorldY    = prectlPageLocation->xRight;
   }

   iBytesInArray = BUMP_TO_NEXT_MODULUS (cx*pbmi->cBitCount, 32) / 8;
   iBytesInCX    = BUMP_TO_NEXT_MODULUS (cx*pbmi->cBitCount, 8) / 8;
   iScanLineY    = cy - 1;
   iNumScanLines = min (cy, iWorldY+1);

   /* We want to keep iRemainder of the left-most bits of the last
   ** byte.
   ** NOTE: 0 <= iRemainder <= 7
   */
   iRemainder = cx - (iBytesInCX - 1)*8;
   if (8 == iRemainder)
      iRemainder = 0;
   assertF (0 <= iRemainder && iRemainder <= 7);

   // Grab our compression buffer
   pbCompress = pHandle->pbCompress;
   assertF (pbCompress);

   while (iNumScanLines)
   {
      while (iNumScanLines >= iBandSizes[iBandSizeIdx])
      {
         /* See if this block of raster has some bits set in it!
         */
         bDirty = FALSE;
         for (sl = 0; sl < iBandSizes[iBandSizeIdx] && !bDirty; sl++)
         {
            register INT x;

            // Check all but the last byte
            for (x = 0; x <= iBytesInCX - 2 && !bDirty; x++)
               bDirty |= pbBits[x + (iScanLineY - sl)*iBytesInArray];
            // Check the remaining bits
            bDirty |= pbBits[x + (iScanLineY - sl)*iBytesInArray] & Mask[iRemainder];
         }

         if (bDirty)
         {
            if (!pHandle->fGraphicsHaveBeenSent)
            {
               PSZ  pszHack = _ESC_ "r0";

               // Start with black ink (default)
               GplThreadWriteBlock (pddc->pdb->hThread, pszHack, 3);
            }
            pHandle->fGraphicsHaveBeenSent = TRUE;

            // Set Absolute Vertical Print Position
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             pDevice->pCmds->ulCmdSetYPos,
                                             pDevice->pCmds->pszCmdSetYPos,
                                             (ulPageSize - iWorldY
                                             + pddc->iShift)); 

            // @176300
            if (pHandle->ulMoveRight        &&
                pDevice->pCmds->ulCmdSetXPos )
            {
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                pDevice->pCmds->ulCmdSetXPos,
                                                pDevice->pCmds->pszCmdSetXPos,
                                                pHandle->ulMoveRight);
            }

            // Send raster transfer header
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             pResInfo->usCmdLength,
                                             pResInfo->szCmdSelectRes,
                                             1,                          // Compression
                                             3600/pResInfo->ulYRes,      // V density
                                             3600/pResInfo->ulXRes,      // H density
                                             iBandSizes[iBandSizeIdx]);  // V dot count

            // Count of raster data
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             2,
                                             "%w",
                                             cx);

            for (sl = 0; sl < iBandSizes[iBandSizeIdx]; sl++)
            {
               pbBuffer = pbBits + iScanLineY*iBytesInArray;

               // Make sure that the extraneous bits are set to 0.
               pbBuffer[iBytesInCX-1] &= Mask[iRemainder];

               iCompressed = CompressEpsonRLE (pbBuffer,
                                               iBytesInCX,
                                               pbCompress,
                                               pHandle->cBytesInCompress);
               GplThreadWriteBlock (pddc->pdb->hThread, pbCompress, iCompressed);

               // Move down a scan line
               iScanLineY--;
               iWorldY--;
            }

            // End the raster line
            GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                        pDevice->pCmds->pszCmdEndRasterGraphicsLine);
         }
         else
         {
            iWorldY    -= iBandSizes[iBandSizeIdx];
            iScanLineY -= iBandSizes[iBandSizeIdx];
         }

         // Done with a block of scan lines
         iNumScanLines -= iBandSizes[iBandSizeIdx];
      }

      // Move to the next scan line block size
      iBandSizeIdx++;
   }

   return TRUE;

} /* end EpsonEsc2PRasterize */

/****************************************************************************/
/* PROCEDURE NAME : EpsonColorEsc2PRasterize                                */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
EpsonColorEsc2PRasterize (PBYTE        pbBits,
                          PBITMAPINFO2 pbmi,
                          PSIZEL       psizelBuffer,
                          PSIZEL       psizelPage,
                          PRECTL       prectlPageLocation,
                          PVOID        pvHandle,
                          PVOID        pvDDC)
{
   static INT    iBandSizes[] = {24, 8, 1, 0};
///static BYTE   Mask[8]      = {0xFF, 0x80, 0xC0, 0xE0, 0xF0, 0xF8, 0xFC, 0xFE};
   PDDC          pddc         = (PDDC)pvDDC;
   PDEVICEINFO   pDevice      = pddc->pdb->pDevice;
   PRESINFO      pResInfo     = pddc->pdb->pResInfo;
   PMEDIAINFO    pMediaInfo   = pddc->pdb->pMediaInfo;
   PPRINTMODE    pPrintMode   = pddc->pdb->pPrintMode;
   PEPSONHANDLE  pHandle      = (PEPSONHANDLE)pvHandle;
   ULONG         cy           = psizelBuffer->cy,
                 cx           = psizelBuffer->cx,
                 ulPageSize;
   PDITHEREQUEST pCMYReq      = (PDITHEREQUEST)&pHandle->Req;
   INT           iBandSizeIdx = 2,
                 iNumScanLines,
                 iScanLineY,
                 iWorldY,
                 iBytesInArray,
                 iOutputBytesInArray;
   PBYTE         pbBuffer;
   PBYTE         pbKBuffer;
   PBYTE         pbCBuffer;
   PBYTE         pbMBuffer;
   PBYTE         pbYBuffer;
   PBYTE         pbColorBits;
   PBYTE         pbCompress;
   INT           iCompressed;
   INT           iPass,
                 iMaxPass,
                 iLastColor   = -1;   /* Not a valid color to start with */
   PSZ           pszHack;
   register INT  sl;

   // Change the band size based on resolution and media type.
   if (((EPSON_RES_ID_720_720_2P == pResInfo->ulResID)        ||
        (EPSON_RES_ID_360_360_2P == pResInfo->ulResID)         ) &&
       (RES_METHOD_EPSON_ESC2P  == pResInfo->usPrintMethod)      &&
       ((EPSON_MEDIA_ID_360_SPECIAL == pMediaInfo->ulMediaID) ||
        (EPSON_MEDIA_ID_720_SPECIAL == pMediaInfo->ulMediaID)  )  )
      /* Micro weaving has been turned on!  We must set a band size of 1! */
      iBandSizeIdx = 2;

   
   if (OmniBookletEnabled(pddc))    
      if (pddc->iShift == 0)
      {
         prectlPageLocation->yTop    += pddc->pdb->ulTopMove;
         prectlPageLocation->yBottom += pddc->pdb->ulTopMove;
         pddc->iShift                 = pddc->pdb->ulTopMove;
      }
      else
      {
         pddc->iShift                += pddc->pdb->ulTopMove;
      }
   

   if (ORIENTATION_PORTRAIT == pddc->pdb->pJobProperties->ulOrientation)
   {
      ulPageSize = psizelPage->cy;
      iWorldY    = prectlPageLocation->yTop;
   }
   else
   {
      ulPageSize = psizelPage->cx;
      iWorldY    = prectlPageLocation->xRight;
   }

   iScanLineY           = psizelBuffer->cy-1;
   iNumScanLines        = min (cy, iWorldY+1);
   iOutputBytesInArray  = BUMP_TO_NEXT_MODULUS (cx, 8) / 8;
   iBytesInArray        = BUMP_TO_NEXT_MODULUS (cx*pbmi->cBitCount, 32) / 8;

   // Grab our compression buffer
   pbCompress = pHandle->pbCompress;
   assertF (pbCompress);

   /* Set up the dynamic parts of the separate request structure.
   */
   pCMYReq->pSrcInfo = pbmi;

   if (COLOR_TECH_CMYK == pPrintMode->ulColorTech)
     iMaxPass = 4;
   else if (COLOR_TECH_CMY == pPrintMode->ulColorTech)
     iMaxPass = 3;

   /* The input bitmap block will be sub-divided into a block of 24, 8, or 1
   */
   while (iNumScanLines)
   {
      while (iNumScanLines >= iBandSizes[iBandSizeIdx])
      {
         // Point to raster bits
         pCMYReq->pbSrc = pbBits + ( iScanLineY + 1 - iBandSizes[iBandSizeIdx] )
                                   * iBytesInArray;

         // Set the current request block size
         pbmi->cy = iBandSizes[iBandSizeIdx];

         // Separate the colors!
         GplDitherRGBtoCMYK ((PPVOID)&pHandle->pdh,
                             (PDITHEREQUEST)&pHandle->Req,
                             (PDITHERESULT)&pHandle->Reply);

         pbKBuffer = pHandle->Reply.pbKBuffer;
         pbCBuffer = pHandle->Reply.pbXBuffer;
         pbMBuffer = pHandle->Reply.pbYBuffer;
         pbYBuffer = pHandle->Reply.pbZBuffer;

         // Is there something in this band?
         if (!(GPL_DITHER_PLANES_BLANK (pHandle->Reply)))
         {
            pHandle->fGraphicsHaveBeenSent = TRUE;

            // Set Absolute Vertical Print Position
            GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                             pDevice->pCmds->ulCmdSetYPos,
                                             pDevice->pCmds->pszCmdSetYPos,
                                             (ulPageSize - iWorldY
                                             + pddc->iShift)); 

            /* Loop through the four color passes/planes:
            **     YELLOW .. MAGENTA .. CYAN .. BLACK
            */
            for (iPass = 0; iPass < iMaxPass; iPass++)
            {
               switch (iPass)
               {
               case 0:  /* YELLOW */
                  if (pHandle->Reply.fEmpty.bYellow)
                     continue;

                  pbColorBits = pbYBuffer;
                  pszHack = _ESC_"r4";
                  break;

               case 1:  /* MAGENTA */
                  if (pHandle->Reply.fEmpty.bMagenta)
                     continue;

                  pbColorBits = pbMBuffer;
                  pszHack = _ESC_"r1";
                  break;

               case 2:  /* CYAN */
                  if (pHandle->Reply.fEmpty.bCyan)
                     continue;

                  pbColorBits = pbCBuffer;
                  pszHack = _ESC_"r2";
                  break;

               case 3:  /* BLACK */
                  if (pHandle->Reply.fEmpty.bBlack)
                     continue;

                  pbColorBits = pbKBuffer;
                  pszHack = _ESC_"r0";
                  break;
               }

               // Need to change color modes?
               if (iLastColor != iPass)
               {
                  // Write color select command
                  GplThreadWriteASCIICommand (pddc->pdb->hThread, pszHack);
                  iLastColor = iPass;
               }

               // @176300
               if (pHandle->ulMoveRight        &&
                   pDevice->pCmds->ulCmdSetXPos )
               {
                  GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                   pDevice->pCmds->ulCmdSetXPos,
                                                   pDevice->pCmds->pszCmdSetXPos,
                                                   pHandle->ulMoveRight);
               }

               // Send raster transfer header
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                pResInfo->usCmdLength,
                                                pResInfo->szCmdSelectRes,
                                                1,                          // Compression
                                                3600/pResInfo->ulYRes,      // V density
                                                3600/pResInfo->ulXRes,      // H density
                                                iBandSizes[iBandSizeIdx]);  // V dot count

               // & the count of raster data
               GplThreadWriteCommandWithPrintf (pddc->pdb->hThread,
                                                2,
                                                "%w",
                                                cx);

               // Loop through the scan lines and send the to the printer
               for (sl = 0; sl < iBandSizes[iBandSizeIdx]; sl++)
               {
                  // Get a pointer to the current scan line
                  pbBuffer = pbColorBits
                             + ( iBandSizes[iBandSizeIdx] - 1 - sl )
                               * iOutputBytesInArray;

                  // Compress it and send it to the printer
                  iCompressed = CompressEpsonRLE (pbBuffer,
                                                  iOutputBytesInArray,
                                                  pbCompress,
                                                  pHandle->cBytesInCompress);
                  GplThreadWriteBlock (pddc->pdb->hThread, pbCompress, iCompressed);
               }

               // End the raster line
               GplThreadWriteASCIICommand (pddc->pdb->hThread,
                                           pDevice->pCmds->pszCmdEndRasterGraphicsLine);
            }
         }

         // Move down in the bitmap
         iNumScanLines -= iBandSizes[iBandSizeIdx];
         iWorldY       -= iBandSizes[iBandSizeIdx];
         iScanLineY    -= iBandSizes[iBandSizeIdx];
      }

      // We can no longer do a block this big, so move to the next size
      iBandSizeIdx++;
   }

   return TRUE;

} /* end EpsonColorEsc2PRasterize */

/****************************************************************************/
/* PROCEDURE NAME : EpsonChooseRasterize                                    */
/* AUTHOR         : Mark Hamzy                                              */
/* DATE WRITTEN   : 4/10/94                                                 */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:                                                              */
/*                                                                          */
/* RETURN VALUES:                                                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL _System
EpsonChooseRasterize (PBYTE             pbBits,
                      PBITMAPINFO2      pbmi,
                      PSIZEL            psizelBuffer,
                      PSIZEL            psizelPage,
                      PRECTL            prectlPageLocation,
                      PVOID             pvHandle,
                      PVOID             pvDDC)
{
   PDDC         pddc     = (PDDC)pvDDC;
   PEPSONHANDLE pHandle  = (PEPSONHANDLE)pvHandle;
   PRESINFO     pResInfo = pddc->pdb->pResInfo;

   if (!pHandle->fHaveSetupPrinter)
   {
      EpsonSetupPrinter (pddc, pHandle);

      GplBookletResetOutput (pddc->pdb->hThread); 
   }

   switch (pddc->pdb->pPrintMode->ulColorTech)
   {
   case COLOR_TECH_MONO:
   {
      if (RES_METHOD_EPSON_COLUMN == pResInfo->usPrintMethod)
      {
         if (EPSON_RES_ID_360_360_24 == pResInfo->ulResID)
            return EpsonColumn_360_24 (pbBits,
                                       pbmi,
                                       psizelBuffer,
                                       psizelPage,
                                       prectlPageLocation,
                                       pvHandle,
                                       pvDDC);
         else
            return EpsonColumnRasterize (pbBits,
                                         pbmi,
                                         psizelBuffer,
                                         psizelPage,
                                         prectlPageLocation,
                                         pvHandle,
                                         pvDDC);
      }
      else if (RES_METHOD_EPSON_ESC2P == pResInfo->usPrintMethod)
      {
         return EpsonEsc2PRasterize (pbBits,
                                     pbmi,
                                     psizelBuffer,
                                     psizelPage,
                                     prectlPageLocation,
                                     pvHandle,
                                     pvDDC);
      }
      else
      {
         assertstring ("Unknown Raster Tecnology");
         return FALSE;
      }
   }

   case COLOR_TECH_CMY:
   case COLOR_TECH_CMYK:
   {
      if (RES_METHOD_EPSON_COLUMN == pResInfo->usPrintMethod)
      {
         if (EPSON_RES_ID_360_360_24 == pResInfo->ulResID)
            return EpsonColorColumn_360_24 (pbBits,
                                            pbmi,
                                            psizelBuffer,
                                            psizelPage,
                                            prectlPageLocation,
                                            pvHandle,
                                            pvDDC);
         else
            return EpsonColorColumnRasterize (pbBits,
                                              pbmi,
                                              psizelBuffer,
                                              psizelPage,
                                              prectlPageLocation,
                                              pvHandle,
                                              pvDDC);
      }
      else if (RES_METHOD_EPSON_ESC2P == pResInfo->usPrintMethod)
      {
         return EpsonColorEsc2PRasterize (pbBits,
                                          pbmi,
                                          psizelBuffer,
                                          psizelPage,
                                          prectlPageLocation,
                                          pvHandle,
                                          pvDDC);
      }
      else
      {
         assertstring ("Unknown Raster Tecnology");
         return FALSE;
      }
   }
   }

   return FALSE;

} /* EpsonChooseRasterize */

// *************************************
//           MEMORY TABLE
// *************************************
MEMORYINFO EpsonMemory =
{
  MEMORY_FIXED_SIZE,   // ulMemFlags;
  0,                   // ulBase;
  0,                   // ulMaximum;
  0                    // ulIncrement;
};

// *************************************
//           GAMMA TABLE
// *************************************
ULONG abEpsonModeTable [][2] = {
        MODE_ID_BLK_COLOR,  EPSON_PRINT_MODE_ID_BLK_COLOR,
        MODE_ID_BLK_COLOR,  EPSON_PRINT_MODE_ID_BLK_24BIT,
        MODE_ID_COLOR    ,  EPSON_PRINT_MODE_ID_COLOR    ,
        MODE_ID_COLOR    ,  EPSON_PRINT_MODE_ID_24BIT    ,
        0                ,  0
};

ULONG abEpsonMediaTable [][2] = {
        MEDIA_ID_PLAIN    ,  EPSON_MEDIA_ID_PLAIN      ,
        MEDIA_ID_1_SPECIAL,  EPSON_MEDIA_ID_360_SPECIAL,
        MEDIA_ID_2_SPECIAL,  EPSON_MEDIA_ID_720_SPECIAL,
        0                 ,  0
};

ULONG abEpsonResTable [][2] = {
        RES_ID_120,  EPSON_RES_ID_60_60      ,
        RES_ID_120,  EPSON_RES_ID_80_60      ,
        RES_ID_120,  EPSON_RES_ID_90_60      ,
        RES_ID_120,  EPSON_RES_ID_120_60     ,
        RES_ID_240,  EPSON_RES_ID_240_60     ,
        RES_ID_120,  EPSON_RES_ID_60_72      ,
        RES_ID_120,  EPSON_RES_ID_72_72      ,
        RES_ID_120,  EPSON_RES_ID_80_72      ,
        RES_ID_120,  EPSON_RES_ID_90_72      ,
        RES_ID_120,  EPSON_RES_ID_120_72     ,
        RES_ID_120,  EPSON_RES_ID_144_72     ,
        RES_ID_240,  EPSON_RES_ID_240_72     ,
        RES_ID_120,  EPSON_RES_ID_60_180     ,
        RES_ID_120,  EPSON_RES_ID_120_180    ,
        RES_ID_120,  EPSON_RES_ID_90_180     ,
        RES_ID_180,  EPSON_RES_ID_180_180    ,
        RES_ID_360,  EPSON_RES_ID_360_180    ,
        RES_ID_120,  EPSON_RES_ID_60_360     ,
        RES_ID_120,  EPSON_RES_ID_90_360     ,
        RES_ID_120,  EPSON_RES_ID_120_360    ,
        RES_ID_180,  EPSON_RES_ID_180_360    ,
        RES_ID_360,  EPSON_RES_ID_360_360    ,
        RES_ID_180,  EPSON_RES_ID_180_180_2P ,
        RES_ID_360,  EPSON_RES_ID_360_180_2P ,
        RES_ID_360,  EPSON_RES_ID_360_360_2P ,
        RES_ID_720,  EPSON_RES_ID_720_720_2P ,
        RES_ID_120,  EPSON_RES_ID_60_72_MX80 ,
        RES_ID_120,  EPSON_RES_ID_120_72_MX80,
        RES_ID_360,  EPSON_RES_ID_360_360_24 ,
        RES_ID_720,  EPSON_RES_ID_360_720_2P ,
        0,           0
};

ULONG abEpsonFamily[][2] = {
        1,  PRN_ID_EPSON_9PIN_GENERIC_NARROW ,
        1,  PRN_ID_EPSON_9PIN_GENERIC_WIDE   ,
        1,  PRN_ID_EPSON_24PIN_GENERIC_NARROW,
        1,  PRN_ID_EPSON_24PIN_GENERIC_WIDE  ,
        1,  PRN_ID_EPSON_48PIN_GENERIC       ,
        1,  PRN_ID_EPSON_ESC2P_GENERIC       ,
        1,  PRN_ID_GEN_ESCP24J84             ,
        0,  0
};

GAMMA_STRUCT  abEpsonGammaTable [] = {
 // Element zero is the default
 0, 0,          0,                  0,                 0,                   10,10,10,10,0,

 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 10,10,10,10,0,
 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 10,10,10,10,0,
 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 10,10,10,10,0,
 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 10,10,10,10,0,

 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 15,17,16,17,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 10,10,10,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 16,16,16,16,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 14,14,14,14,0,

 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 14,14,14,14,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 14,14,14,14,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 24,24,24,24,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 24,24,24,24,0,

 1, RES_ID_360, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 32,38,36,38,0,
 1, RES_ID_360, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 22,28,26,28,0,
 1, RES_ID_360, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 44,44,44,44,0,
 1, RES_ID_360, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 39,39,39,39,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 26,26,26,26,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION, 15,19,17,19,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_MATRIX   , 39,39,39,39,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 36,36,36,36,0,

 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   , 120,120,120,120,145,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION,  80, 94, 88, 94,145,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_MATRIX   , 150,170,160,170,145,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_DIFFUSION, 140,140,140,140,145,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_MATRIX   ,  78, 78, 78, 78,145,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_DIFFUSION,  52, 52, 52, 52,145,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_COLOR    , DITHER_ID_MATRIX   ,  90,106, 98,106,145,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_COLOR    , DITHER_ID_DIFFUSION,  96, 96, 96, 96,145,

 // jk1115 start
 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 10,10,10,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 10,10,10,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 10,10,10,10,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 10,10,10,12,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 10,10,10,12,0,
 1, RES_ID_360, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 13,13,13,23,0, //
 1, RES_ID_360, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 13,13,13,22,0, //
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 12,12,12,22,0, //
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 12,12,12,22,0, //
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 28,28,28,16,0, //
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 28,28,28,16,0, //
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_VOID_CLUSTER  , 28,28,28,16,0, //
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_COLOR    , DITHER_ID_VOID_CLUSTER  , 28,28,28,16,0, //

 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_DIFFUSION , 10,10,10,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION , 13,13,13,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_DIFFUSION , 13,13,13,10,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION , 13,13,13,11,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_DIFFUSION , 13,13,13,11,0,
 1, RES_ID_360, MEDIA_ID_PLAIN,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION,  13,13,13,22,0,
 1, RES_ID_360, MEDIA_ID_PLAIN,     MODE_ID_COLOR,     DITHER_ID_NEW_DIFFUSION,  13,13,13,22,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION , 17,17,17,20,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_NEW_DIFFUSION , 17,17,17,20,0,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION , 19,19,19,41,0,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_DIFFUSION , 19,19,19,41,0,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_NEW_DIFFUSION , 19,19,19,41,0,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_COLOR    , DITHER_ID_NEW_DIFFUSION , 19,19,19,41,0,
 // jk1115 end
 
 // sx start
 1, RES_ID_120, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_MATRIX , 34,34,34,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX , 38,38,38,10,0,
 1, RES_ID_180, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_MATRIX , 38,38,38,10,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX , 38,38,38,11,0,
 1, RES_ID_240, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_MATRIX , 38,38,38,11,0,
 1, RES_ID_360, MEDIA_ID_PLAIN,     MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX , 38,38,38,22,0,
 1, RES_ID_360, MEDIA_ID_PLAIN,     MODE_ID_COLOR,     DITHER_ID_NEW_MATRIX , 38,38,38,22,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX , 42,42,42,20,0,
 1, RES_ID_360, MEDIA_ID_1_SPECIAL, MODE_ID_COLOR    , DITHER_ID_NEW_MATRIX , 42,42,42,20,0,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX , 44,44,44,41,0,
 1, RES_ID_720, MEDIA_ID_PLAIN    , MODE_ID_COLOR    , DITHER_ID_NEW_MATRIX , 44,44,44,41,0,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_BLK_COLOR, DITHER_ID_NEW_MATRIX , 44,44,44,41,0,
 1, RES_ID_720, MEDIA_ID_2_SPECIAL, MODE_ID_COLOR    , DITHER_ID_NEW_MATRIX , 44,44,44,41,0,
 // sx end

 // Terminate with zero elements
 0, 0,          0,                  0,                 0,                    0,0,0,0,0
};

GAMMAINFO EPSON_pGammas[] = {
  {
    (PULONG)abEpsonFamily,                 // pFamily
    (PULONG)abEpsonResTable,               // pRes;
    (PULONG)abEpsonMediaTable,             // pMedia;
    (PULONG)abEpsonModeTable,              // pMode;
    (PGAMMA_STRUCT)abEpsonGammaTable,      // pSelect;
  }
};

// *************************************
//           RESOLUTION TABLE
// *************************************

RESINFO EPSON_pRes[] =
{
  {
    EPSON_RES_ID_60_72_MX80,         // ulResID;
    RES_STRING_60_X_72_8_PINS,       // ulNameID;     @181074
    "",                              // szResName[32];
    60,                              // ulXRes;
    72,                              // ulYRes;
    2,                               // usCmdLength;
    _ESC_ ASCII(K),                  // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    72,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_120_72_MX80,        // ulResID;
    RES_STRING_120_X_72_8_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    120,                             // ulXRes;
    72,                              // ulYRes;
    2,                               // usCmdLength;
    _ESC_ ASCII(L),                  // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    72,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_60_60,              // ulResID;
    RES_STRING_60_X_60_24_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    60,                              // ulXRes;
    60,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(00),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    60,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_80_60,              // ulResID;
    RES_STRING_80_X_60_24_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    80,                              // ulXRes;
    60,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(04),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    60,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_90_60,              // ulResID;
    RES_STRING_90_X_60_24_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    90,                              // ulXRes;
    60,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(06),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    60,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_120_60,             // ulResID;
    RES_STRING_120_X_60_24_PINS,     // ulNameID;     @181074
    "",                              // szResName[32];
    120,                             // ulXRes;
    60,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(02),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    FALSE,                           // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    60,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_240_60,             // ulResID;
    RES_STRING_240_X_60_24_PINS,     // ulNameID;     @181074
    "",                              // szResName[32];
    240,                             // ulXRes;
    60,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(03),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    FALSE,                           // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(A) "%c",             // szCmdLineSpacing[8];
    60,                              // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_60_72,              // ulResID;
    RES_STRING_60_X_72_8_PINS,       // ulNameID;     @181074
    "",                              // szResName[32];
    60,                              // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(00),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_72_72,              // ulResID;
    RES_STRING_72_X_72_8_PINS,       // ulNameID;     @181074
    "",                              // szResName[32];
    72,                              // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(05),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_80_72,              // ulResID;
    RES_STRING_80_X_72_8_PINS,       // ulNameID;     @181074
    "",                              // szResName[32];
    80,                              // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(04),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_90_72,              // ulResID;
    RES_STRING_90_X_72_8_PINS,       // ulNameID;     @181074
    "",                              // szResName[32];
    90,                              // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(06),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_120_72,             // ulResID;
    RES_STRING_120_X_72_8_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    120,                             // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(02),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_MEDIUM,                  // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    FALSE,                           // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_144_72,             // ulResID;
    RES_STRING_144_X_72_8_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    144,                             // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(07),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_MEDIUM,                  // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_240_72,             // ulResID;
    RES_STRING_240_X_72_8_PINS,      // ulNameID;     @181074
    "",                              // szResName[32];
    240,                             // ulXRes;
    72,                              // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(03),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    8,                               // usNumPhysicalPins;
    FALSE,                           // bAdjacentDotPrinting;
    8,                               // usDotsPerColumn;
    1,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    216,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_60_180,             // ulResID;
    RES_STRING_60_X_180_24_PINS,     // ulNameID;     @181074
    "",                              // szResName[32];
    60,                              // ulXRes;
    180,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(20),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    180,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_90_180,             // ulResID;
    RES_STRING_90_X_180_24_PINS,     // ulNameID;     @181074
    "",                              // szResName[32];
    90,                              // ulXRes;
    180,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(26),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_LOW,                     // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    180,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_120_180,            // ulResID;
    RES_STRING_120_X_180_24_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    120,                             // ulXRes;
    180,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(21),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_MEDIUM,                  // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    180,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_180_180,            // ulResID;
    RES_STRING_180_X_180_24_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    180,                             // ulXRes;
    180,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(27),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    180,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_360_180,            // ulResID;
    RES_STRING_360_X_180_24_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    360,                             // ulXRes;
    180,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(28),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    24,                              // usNumPhysicalPins;
    FALSE,                           // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(3) "%c",             // szCmdLineSpacing[8];
    180,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_60_360,             // ulResID;
    RES_STRING_60_X_360_48_PINS,     // ulNameID;     @181074
    "",                              // szResName[32];
    60,                              // ulXRes;
    360,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(40),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    48,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    48,                              // usDotsPerColumn;
    6,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_90_360,             // ulResID;
    RES_STRING_90_X_360_48_PINS,     // ulNameID;     @181074
    "",                              // szResName[32];
    90,                              // ulXRes;
    360,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(46),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_DRAFT,                   // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    48,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    48,                              // usDotsPerColumn;
    6,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_120_360,            // ulResID;
    RES_STRING_120_X_360_48_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    120,                             // ulXRes;
    360,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(41),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_MEDIUM,                  // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    48,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    48,                              // usDotsPerColumn;
    6,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_180_360,            // ulResID;
    RES_STRING_180_X_360_48_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    180,                             // ulXRes;
    360,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(47),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    48,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    48,                              // usDotsPerColumn;
    6,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_360_360,            // ulResID;
    RES_STRING_360_X_360_48_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    360,                             // ulXRes;
    360,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(49),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    48,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    48,                              // usDotsPerColumn;
    6,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_180_180_2P,         // ulResID;
    RES_STRING_180_X_180,            // ulNameID;     @181074
    "",                              // szResName[32];
    180,                             // ulXRes;
    180,                             // ulYRes;
    10,                              // usCmdLength;
    _ESC_ ".%c%c%c%c",               // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_LOW,                     // ulDJPInfo
    RES_METHOD_EPSON_ESC2P,          // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    7,                               // usLenVerticalDensity;
    _ESC_ "(U" HEX(01) HEX(00) "%c", // szCmdVerticalDensity[8];
    3600,                            // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_360_180_2P,         // ulResID;
    RES_STRING_360_X_180,            // ulNameID;     @181074
    "",                              // szResName[32];
    360,                             // ulXRes;
    180,                             // ulYRes;
    10,                              // usCmdLength;
    _ESC_ ".%c%c%c%c",               // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_MEDIUM,                  // ulDJPInfo
    RES_METHOD_EPSON_ESC2P,          // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    7,                               // usLenVerticalDensity;
    _ESC_ "(U" HEX(01) HEX(00) "%c", // szCmdVerticalDensity[8];
    3600,                            // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_360_360_2P,         // ulResID;
    RES_STRING_360_X_360,            // ulNameID;     @181074
    "",                              // szResName[32];
    360,                             // ulXRes;
    360,                             // ulYRes;
    10,                              // usCmdLength;
    _ESC_ ".%c%c%c%c",               // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_ESC2P,          // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    7,                               // usLenVerticalDensity;
    _ESC_ "(U" HEX(01) HEX(00) "%c", // szCmdVerticalDensity[8];
    3600,                            // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_360_720_2P,         // ulResID;
    RES_STRING_360_X_720,            // ulNameID;     @181074
    "",                              // szResName[32];
    360,                             // ulXRes;
    720,                             // ulYRes;
    10,                              // usCmdLength;
    _ESC_ ".%c%c%c%c",               // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_ESC2P,          // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    720,                             // ulUnitsLineSpacing;
    7,                               // usLenVerticalDensity;
    _ESC_ "(U" HEX(01) HEX(00) "%c", // szCmdVerticalDensity[8];
    3600,                            // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_720_720_2P,         // ulResID;
    RES_STRING_720_X_720,            // ulNameID;     @181074
    "",                              // szResName[32];
    720,                             // ulXRes;
    720,                             // ulYRes;
    10,                              // usCmdLength;
    _ESC_ ".%c%c%c%c",               // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_ESC2P,          // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    24,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    720,                             // ulUnitsLineSpacing;
    7,                               // usLenVerticalDensity;
    _ESC_ "(U" HEX(01) HEX(00) "%c", // szCmdVerticalDensity[8];
    3600,                            // ulUnitsVerticalDensity;
  }
,
  {
    EPSON_RES_ID_360_360_24,         // ulResID;
    RES_STRING_360_X_360_24_PINS,    // ulNameID;     @181074
    "",                              // szResName[32];
    360,                             // ulXRes;
    360,                             // ulYRes;
    3,                               // usCmdLength;
    _ESC_ ASCII(*) HEX(28),          // szCmdSelectRes[32];
    0,                               // ulCapabilities
    DJP_PQL_HIGH,                    // ulDJPInfo
    RES_METHOD_EPSON_COLUMN,         // usPrintMethod;
    24,                              // usNumPhysicalPins;
    TRUE,                            // bAdjacentDotPrinting;
    48,                              // usDotsPerColumn;
    3,                               // usBytesPerColumn;
    4,                               // usLenLineSpacing;
    _ESC_ ASCII(+) "%c",             // szCmdLineSpacing[8];
    360,                             // ulUnitsLineSpacing;
    0,                               // usLenVerticalDensity;
    HEX(00),                         // szCmdVerticalDensity[8];
    0,                               // ulUnitsVerticalDensity;
  }
};
#define EPSON_DEFINED_RESOLUTIONS      (sizeof (EPSON_pRes)/sizeof (EPSON_pRes[0]))

//****************************************************************************
//* FONT DEFINITION TABLES
//****************************************************************************
//   typedef struct _DEVFONTMETRICS {                      /* dfm */
//           ULONG                  flOptions;
//           PFOCAFONT              pFoca;
//           PVOID                  pUniPanhose;
//           char                   achFileName[CCHMAXPATH];
//           PSZ                    pszFullFamilyName;
//           PSZ                    pszFullFaceName;
//           char                   achGlyphListName[16];
//           PFN                    pfnDevQueryCodePage;
//           LONG                   lDevMatch;
//   } DEVFONTMETRICS;

FONTINFO2 EPSON_pFonts[] =
{
  {
    FONT_ID_MINCHO_WIDE,            // ulFontID
    -1,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    DEVFONT_DEFAULT,                // ulMiscCaps; (GRE FLAGS/PMDDI.H)
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONMWD.FMF",                 // szFileName
    "",                             // szFaceName;
    "MINCHO",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    12,                             // ulCmdLen
    _FS_ ASCII(k) HEX(00) _ESC_ ASCII(P) _ESC_ HEX(20) HEX(0C) _FS_ ASCII(S) HEX(00) HEX(18),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_MINCHO_10CPI,           // ulFontID
    -2,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONM10.FMF",                 // szFileName
    "",                             // szFaceName;
    "MINCHO",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    9,                              // ulCmdLen
    _FS_ ASCII(k) HEX(00) _ESC_ ASCII(P) _FS_ ASCII(S) HEX(00) HEX(0C),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_MINCHO_12CPI,           // ulFontID
    -3,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONM12.FMF",                 // szFileName
    "",                             // szFaceName;
    "MINCHO",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    9,                              // ulCmdLen;
    _FS_  ASCII(k) HEX(00) _ESC_ ASCII(M) _FS_  ASCII(S) HEX(00) HEX(06),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_MINCHO_15CPI,           // ulFontID
    -4,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONM15.FMF",                 // szFileName
    "",                             // szFaceName;
    "MINCHO",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    9,                              // ulCmdLen
    _FS_ ASCII(k) HEX(00) _ESC_ ASCII(g) _FS_ ASCII(S) HEX(00) HEX(00),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_GOTHIC_WIDE,            // ulFontID
    -5,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONGWD.FMF",                 // szFileName
    "",                             // szFaceName;
    "GOTHIC",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    12,                             // ulCmdLen
    _FS_ ASCII(k) HEX(01) _ESC_ ASCII(P) _ESC_ HEX(20) HEX(0C) _FS_ ASCII(S) HEX(00) HEX(18),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_GOTHIC_10CPI,           // ulFontID
    -6,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONG10.FMF",                 // szFileName
    "",                             // szFaceName;
    "GOTHIC",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    9,                              // ulCmdLen
    _FS_ ASCII(k) HEX(01) _ESC_ ASCII(P) _FS_ ASCII(S) HEX(00) HEX(0C),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_GOTHIC_12CPI,           // ulFontID
    -7,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONG12.FMF",                 // szFileName
    "",                             // szFaceName;
    "GOTHIC",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    9,                              // ulCmdLen
    _FS_ ASCII(k) HEX(01) _ESC_ ASCII(M) _FS_ ASCII(S) HEX(00) HEX(06),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  },
  {
    FONT_ID_GOTHIC_15CPI,           // ulFontID
    -8,                             // lDevMatch
    FONT_TYPE_FIXED,                // ulType (Fixed, Prop, Scalable)
    0,                              // ulTypeExtra
    FONT_ORIENT_PORTRAIT,           // ulOrientation;
    0,                              // ulMiscCaps;
    FONT_FILE_TYPE_FOCAMETRICS,     // ulFileType
    "EPSONG15.FMF",                 // szFileName
    "",                             // szFaceName;
    "GOTHIC",                       // szFamilyName
    "PMJPN",                        // szGlyphListName;
    9,                              // ulCmdLen
    _FS_ ASCII(k) HEX(01) _ESC_ ASCII(g) _FS_ ASCII(S) HEX(00) HEX(00),
                                    // szCmdSelectFont;
    CHAR_SET_JIS_X_0208_1983,       // ulFontCharSet;
    CHAR_ENCODING_JIS,              // ulFontCharEncoding;
    0L,                             // ulFontCharSetExtension;
    2L,                             // ulCmdModeInLen;
    _FS_ ASCII(&),                  // szCmdModeIn;
    2L,                             // ulCmdModeOutLen;
    _FS_ ASCII(.),                  // szCmdModeOut;
    NULL,                           // pvUGLToSymbolSet;
    "",                             // szBackupFontName;
    NULL,                           // vpGlobalFocaFont;
    0                               // cbGlobalFocaFont;
  }
} ;

  /**************************************************************************/
  /* end Font                                                               */
  /**************************************************************************/

#define  EPSON_DEFINED_FONTS (sizeof (EPSON_pFonts)/sizeof (EPSON_pFonts[0]))

ULONG EPSON_pulFonts[] = { FONT_ID_MINCHO_WIDE,
                           FONT_ID_MINCHO_10CPI,
                           FONT_ID_MINCHO_12CPI,
                           FONT_ID_MINCHO_15CPI,
                           FONT_ID_GOTHIC_WIDE,
                           FONT_ID_GOTHIC_10CPI,
                           FONT_ID_GOTHIC_12CPI,
                           FONT_ID_GOTHIC_15CPI };

#define  EPSON_SUPPORTED_FONTS (sizeof (EPSON_pulFonts)/sizeof (EPSON_pulFonts[0]))

// *************************************
//           TRAY TABLE
// *************************************

// @TBD - **** us name IDs NOT _STRING_ names ****
TRAYINFO EPSON_pTrays[] =
{
  {
     EPSON_TRAY_ID_REAR_CONTINUOUS,           // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_TRACTOR,                         // ulDJPid;
     TRAY_STRING_R_CONTINUOUS,                // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType;
     3,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(B),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_FRONT_CONTINUOUS,          // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_TRACTOR,                         // ulDJPid;
     TRAY_STRING_F_CONTINUOUS,                // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     3,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(F),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_SINGLE_SHEET,              // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_MANUAL,                          // ulDJPid;
     TRAY_STRING_SINGLE_SHEET,                // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     0,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(?),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_MANUAL_FEED,               // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_MANUAL,                          // ulDJPid;
     TRAY_STRING_MANUAL_FEED,                 // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_MANUAL,                        // ulTrayType
     3,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(?),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_BIN1,                      // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_UPPER,                           // ulDJPid;
     TRAY_STRING_BIN_1,                       // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     3,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(1),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_BIN2,                      // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_LOWER,                           // ulDJPid;
     TRAY_STRING_BIN_2,                       // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     3,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(2),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_CSF,                       // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_AUTO,                            // ulDJPid;
     TRAY_STRING_CSF,                         // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     3,                                       // ulCmdSelectTray;
     _ESC_ _EM_ ASCII(0),                     // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
,
  {
     EPSON_TRAY_ID_CSF_ESCP2,                 // ulTrayID;
     0,                                       // ulTrayCap;
     DJP_TRY_AUTO,                            // ulDJPid;
     TRAY_STRING_CSF,                         // ulNameID;
     "",                                      // pszTrayName;
     TRAY_TYPE_AUTO,                          // ulTrayType
     0,                                       // ulCmdSelectTray;
     "",                                      // szCmdSelectTray;
     0,                                       // ulCompatibleForms;
     (PULONG)NULL                             // pulCompatibleForms;
  }
};
#define EPSON_DEFINED_TRAYS             (sizeof (EPSON_pTrays)/sizeof (EPSON_pTrays[0]))


// *************************************
//           FORM TABLE
// *************************************


// NOTE : We do not fill in xPels or yPels in this structure
//        but determine this value dynamically based on
//        current resolution
//        also HCAPS_SELECTABLE and CURRENT are determined from
//        printer and job props dynamically as well

// @TBD - use NAME_IDs not _STRING directly
FORMINFO2 EPSON_pGenNarrowForms[] =
{
  {
    EPSON_FORM_ID_LETTER,       // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_LETTER,             // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LETTER,         // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32] optional
      21590,                    // cx; (mm)
      27940,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      400,                      // yBottomClip; (mm)
      21590 - 1400,             // xRightClip; (mm)
      27940 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_LEGAL,        // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_LEGAL,              // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LEGAL,          // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      21590,                    // cx; (mm)
      35560,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      400,                      // yBottomClip; (mm)
      21590 - 1400,             // xRightClip; (mm)
      35560 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_A3,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_A3,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A3,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      29693,                    // cx; (mm)
      42012,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      29693 - 1400,             // xRightClip; (mm)
      42012 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_A4,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_A4,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A4,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      21006,                    // cx; (mm)
      29693,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      21006 - 1000,             // xRightClip; (mm)
      29693 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_A5,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_A5,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A5,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      14808,                    // cx; (mm)
      21006,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      14808,                    // xRightClip; (mm)
      21006 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_B5,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_B5,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_B5,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      18288,                    // cx; (mm)
      25654,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      18288,                    // xRightClip; (mm)
      25654 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
}; /* end Forms */
#define EPSON_DEFINED_GENNARROWFORMS         (sizeof (EPSON_pGenNarrowForms)/sizeof (EPSON_pGenNarrowForms[0]))

FORMINFO2 EPSON_pGenWideForms[] =
{
  {
    EPSON_FORM_ID_LETTER,       // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_LETTER,             // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LETTER,         // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32] optional
      21590,                    // cx; (mm)
      27940,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      400,                      // yBottomClip; (mm)
      21590,                    // xRightClip; (mm)
      27940 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_LEGAL,        // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_LEGAL,              // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LEGAL,          // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      21590,                    // cx; (mm)
      35560,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      400,                      // yBottomClip; (mm)
      21590,                    // xRightClip; (mm)
      35560 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_WIDE,         // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_WIDE,               // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_WIDE,           // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32] optional
      34500,                    // cx; (mm)
      27940,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      400,                      // yBottomClip; (mm)
      34500,                    // xRightClip; (mm)
      27940 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_A3,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_A3,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A3,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      29693,                    // cx; (mm)
      42012,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      29693,                    // xRightClip; (mm)
      42012 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_A4,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_A4,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A4,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      21006,                    // cx; (mm)
      29693,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      21006,                    // xRightClip; (mm)
      29693 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_A5,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_A5,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_A5,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      14808,                    // cx; (mm)
      21006,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      14808,                    // xRightClip; (mm)
      21006 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_B5,           // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_B5,                 // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_B5,             // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      18288,                    // cx; (mm)
      25654,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      18288,                    // xRightClip; (mm)
      25654 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
,
  {
    EPSON_FORM_ID_LEDGER,       // ulFormID;
    FORM_CAP_FF_TO_END,         // ulFormCap;
    DJP_PSI_LEDGER,             // ulDJPid;
    FORM_CLASS_SHEET_PAPER,     // ulFormClass;
    FORM_STRING_LEDGER,         // ulNameID;
    0,                          // usLenCmdSelectForm;
    { 0 },                      // szCmdSelectForm;
    {                           // hcInfo
      "",                       // szFormname[32]
      27900,                    // cx; (mm)
      43200,                    // cy; (mm)
      0,                        // xLeftClip; (mm)
      0,                        // yBottomClip; (mm)
      27900,                    // xRightClip; (mm)
      43200 - 2000,             // yTopClip; (mm)
      0,                        // xPels; (pels)
      0,                        // yPels; (pels)
      0                         // flAttributes;
    }, /* end logical block */
    FALSE                       // bHasBeenSetup;
  }
}; /* end Forms */
#define EPSON_DEFINED_GENWIDEFORMS         (sizeof (EPSON_pGenWideForms)/sizeof (EPSON_pGenWideForms[0]))

// *************************************
//           MEDIAS TABLE
// *************************************

MEDIAINFO EPSON_pMedias[] =
{
  {
    EPSON_MEDIA_ID_PLAIN,               // ulMediaID;
    0,                                  // ulMediaCap;
    DJP_MED_PLAIN,                      // ulDJPid;
    MEDIA_STRING_PLAIN,                 // ulNameID;
    "",                                 // szMediaName[LEN_MEDIANAME];
    0,                                  // ulCmdSelectMedia;
    "",                                 // szMediaTypeCmd[LEN_CMD];
    FALSE,                              // fColorAdjustRequired;
    HEAVY_ABSORPTION                    // ulAbsorptionAdjust;
  }  /* end Plain */
,
  {
    EPSON_MEDIA_ID_360_SPECIAL,         // ulMediaID;
    0,                                  // ulMediaCap;
    DJP_MED_SPECIAL,                    // ulDJPid;
    MEDIA_STRING_SPECIAL_360,           // ulNameID;  @181074
    "",                                 // szMediaName[LEN_MEDIANAME];
    0,                                  // ulCmdSelectMedia;
    "",                                 // szMediaTypeCmd[LEN_CMD];
    FALSE,                              // fColorAdjustRequired;
    LIGHT_ABSORPTION                    // ulAbsorptionAdjust;
  }  /* end Plain */
,
  {
    EPSON_MEDIA_ID_720_SPECIAL,         // ulMediaID;
    0,                                  // ulMediaCap;
    DJP_MED_SPECIAL,                    // ulDJPid;
    MEDIA_STRING_SPECIAL_720,           // ulNameID;  @181074
    "",                                 // szMediaName[LEN_MEDIANAME];
    0,                                  // ulCmdSelectMedia;
    "",                                 // szMediaTypeCmd[LEN_CMD];
    FALSE,                              // fColorAdjustRequired;
    LIGHT_ABSORPTION                    // ulAbsorptionAdjust;
  }  /* end Plain */
}; /* end Medias */
#define EPSON_DEFINED_MEDIAS  (sizeof (EPSON_pMedias)/sizeof (EPSON_pMedias[0]))

ULONG EPSON_pulStylusColorMedias[] =
{
            EPSON_MEDIA_ID_PLAIN,
            EPSON_MEDIA_ID_360_SPECIAL,
            EPSON_MEDIA_ID_720_SPECIAL
};
#define EPSON_SUPPORTED_STYLUS_COLOR_MEDIAS (sizeof (EPSON_pulStylusColorMedias)/sizeof (EPSON_pulStylusColorMedias[0]))

ULONG EPSON_pulMedias[] =
{
            EPSON_MEDIA_ID_PLAIN
};
#define EPSON_SUPPORTED_MEDIAS (sizeof (EPSON_pulMedias)/sizeof (EPSON_pulMedias[0]))

// *************************************
//           CONNECTION TABLE
// *************************************

FORMCONNECTION EPSON_pConnects[] =
{
  {EPSON_CONN_ID_1,  EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_LETTER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_6,  EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_A4,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_10, EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_A3,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_2,  EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_A5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_15, EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_B5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_43, EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_LEGAL,  EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_16, EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_LEDGER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_17, EPSON_TRAY_ID_REAR_CONTINUOUS, EPSON_FORM_ID_WIDE,   EPSON_MEDIA_ID_PLAIN      },

  {EPSON_CONN_ID_3,  EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LETTER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_7,  EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A4,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_18, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A3,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_19, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_20, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_B5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_44, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LEGAL,  EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_21, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LEDGER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_22, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_WIDE,   EPSON_MEDIA_ID_PLAIN      },

  {EPSON_CONN_ID_4,  EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LETTER, EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_8,  EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A4,     EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_23, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A3,     EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_24, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A5,     EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_25, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_B5,     EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_45, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LEGAL,  EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_26, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LEDGER, EPSON_MEDIA_ID_360_SPECIAL},
  {EPSON_CONN_ID_27, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_WIDE,   EPSON_MEDIA_ID_360_SPECIAL},

  {EPSON_CONN_ID_5,  EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LETTER, EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_9,  EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A4,     EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_28, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A3,     EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_29, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_A5,     EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_30, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_B5,     EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_46, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LEGAL,  EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_31, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_LEDGER, EPSON_MEDIA_ID_720_SPECIAL},
  {EPSON_CONN_ID_32, EPSON_TRAY_ID_CSF_ESCP2,       EPSON_FORM_ID_WIDE,   EPSON_MEDIA_ID_720_SPECIAL},

  {EPSON_CONN_ID_11, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_LETTER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_12, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_A4,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_33, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_A3,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_34, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_A5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_35, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_B5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_47, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_LEGAL,  EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_36, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_LEDGER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_37, EPSON_TRAY_ID_SINGLE_SHEET,    EPSON_FORM_ID_WIDE,   EPSON_MEDIA_ID_PLAIN      },

  {EPSON_CONN_ID_13, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_LETTER, EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_14, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_A4,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_38, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_A3,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_39, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_A5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_40, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_B5,     EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_48, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_LEGAL,  EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_42, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_WIDE,   EPSON_MEDIA_ID_PLAIN      },
  {EPSON_CONN_ID_49, EPSON_TRAY_ID_MANUAL_FEED,     EPSON_FORM_ID_LEDGER, EPSON_MEDIA_ID_PLAIN      }
}; /* end Connects */
#define EPSON_DEFINED_CONNECTIONS       (sizeof (EPSON_pConnects)/sizeof (EPSON_pConnects[0]))

ULONG EPSON_R_Cont_pulConnects[] = {
                   EPSON_CONN_ID_1,            // RearC    Letter plain
                   EPSON_CONN_ID_43,           //          Legal  plain
                   EPSON_CONN_ID_6,            //          A4     plain
                   EPSON_CONN_ID_10,           //          A3     plain
                   EPSON_CONN_ID_2,            //          A5     plain
                   EPSON_CONN_ID_15,           //          B5     plain

                   EPSON_CONN_ID_13,           // manual   Letter plain
                   EPSON_CONN_ID_48,           //          Legal  plain
                   EPSON_CONN_ID_14,           //          A4     plain
                   EPSON_CONN_ID_38,           //          A3     plain
                   EPSON_CONN_ID_39,           //          A5     plain
                   EPSON_CONN_ID_40            //          B5     plain
};
#define EPSON_R_Cont_SUPPORTED_CONNECTIONS     (sizeof (EPSON_R_Cont_pulConnects)/sizeof (EPSON_R_Cont_pulConnects[0]))

ULONG EPSON_Portable_pulConnects[] = {
                   EPSON_CONN_ID_11,           // Single   Letter plain
                   EPSON_CONN_ID_47,           // Sheet    Legal  plain
                   EPSON_CONN_ID_12,           //          A4     plain
                   EPSON_CONN_ID_33,           //          A3     plain
                   EPSON_CONN_ID_34,           //          A5     plain
                   EPSON_CONN_ID_35,           //          B5     plain

                   EPSON_CONN_ID_13,           // manual   Letter plain
                   EPSON_CONN_ID_48,           //          Legal  plain
                   EPSON_CONN_ID_14,           //          A4     plain
                   EPSON_CONN_ID_38,           //          A3     plain
                   EPSON_CONN_ID_39,           //          A5     plain
                   EPSON_CONN_ID_40            //          B5     plain
};
#define EPSON_PORTABLE_SUPPORTED_CONNECTIONS     (sizeof (EPSON_Portable_pulConnects)/sizeof (EPSON_Portable_pulConnects[0]))

// *************************************
//           PRINT MODE TABLE
// *************************************

PRINTMODE EPSON_pPrintModes[] =
{
  {
    EPSON_PRINT_MODE_ID_MONO,       // ulPrintModeID;
    COLOR_TECH_MONO,                // ulColorTech;
    PRINT_MODE_STRING_MONOCHROME,   // ulNameID;
    "",                             // szPrintMode[32];
    1,                              // usBitsPerPel;   // Physical bit count
    8,                              // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    EPSON_PRINT_MODE_ID_COLOR,      // ulPrintModeID;
    COLOR_TECH_CMY,                 // ulColorTech;
    PRINT_MODE_STRING_COLOR_ONLY,   // ulNameID;
    "",                             // szPrintMode[32];
    8,                              // usBitsPerPel;   // Physical bit count
    8,                              // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    EPSON_PRINT_MODE_ID_BLK_COLOR,  // ulPrintModeID;
    COLOR_TECH_CMYK,                // ulColorTech;
    PRINT_MODE_STRING_BLACK_COLOR,  // ulNameID;
    "",                             // szPrintMode[32];
    8,                              // usBitsPerPel;   // Physical bit count
    8,                              // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    EPSON_PRINT_MODE_ID_24BIT,      // ulPrintModeID;
    COLOR_TECH_CMY,                 // ulColorTech;
    PRINT_MODE_STRING_24_CLR,       // ulNameID;
    "",                             // szPrintMode[32];
    24,                             // usBitsPerPel;   // Physical bit count
    24,                             // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
,
  {
    EPSON_PRINT_MODE_ID_BLK_24BIT,  // ulPrintModeID;
    COLOR_TECH_CMYK,                // ulColorTech;
    PRINT_MODE_STRING_24_BLK_CLR,   // ulNameID;
    "",                             // szPrintMode[32];
    24,                             // usBitsPerPel;   // Physical bit count
    24,                             // usLogBitCount;  // Logical  bit count
    1,                              // usNumPlanes;    // # color planes (always 1 for OS/2)
    0                               // ulDither;       // default Dither Alg. (e.g. pattern/scatter)
  }
}; /* end Print Modes */
#define EPSON_DEFINED_PRINT_MODES       (sizeof (EPSON_pPrintModes)/sizeof (EPSON_pPrintModes[0]))

ULONG EPSON_BW_pulPrintModes[] =
{
       EPSON_PRINT_MODE_ID_MONO
}; /* end EPSON_BW_pulPrintModes */
#define EPSON_BW_SUPPORTED_PRINT_MODES (sizeof (EPSON_BW_pulPrintModes)/sizeof (EPSON_BW_pulPrintModes[0]))

ULONG EPSON_CMYK_pulPrintModes[] =
{
       EPSON_PRINT_MODE_ID_MONO      ,
       EPSON_PRINT_MODE_ID_BLK_COLOR ,
       EPSON_PRINT_MODE_ID_BLK_24BIT
}; /* end EPSON_CMYK_pulPrintModes */
#define EPSON_CMYK_SUPPORTED_PRINT_MODES (sizeof (EPSON_CMYK_pulPrintModes)/sizeof (EPSON_CMYK_pulPrintModes[0]))

ULONG EPSON_CMY_pulPrintModes[] =
{
       EPSON_PRINT_MODE_ID_MONO      ,
       EPSON_PRINT_MODE_ID_COLOR     ,
       EPSON_PRINT_MODE_ID_24BIT
}; /* end EPSON_CMY_pulPrintModes */
#define EPSON_CMY_SUPPORTED_PRINT_MODES (sizeof (EPSON_CMY_pulPrintModes)/sizeof (EPSON_CMY_pulPrintModes[0]))

// @PN60
ULONG EPSON_PORT_pulPrintModes[] =
{
       EPSON_PRINT_MODE_ID_MONO      ,
       EPSON_PRINT_MODE_ID_BLK_COLOR ,
}; /* end EPSON_PORT_pulPrintModes */
#define EPSON_PORT_SUPPORTED_PRINT_MODES (sizeof (EPSON_PORT_pulPrintModes)/sizeof (EPSON_PORT_pulPrintModes[0]))


FEATUREINFO EPSON_Features =
{
  TRUE,                // bSupUserDefTrays;
  TRUE,                // bSupUserDefForms;
  FALSE,               // bSupUserDefFonts;
  TRUE                 // bSupUserDefConnects;
}; /* end EPSON_Features */

// Subclassed functions for Epson Driver
FNSUBCLASS EPSON_Subfunctions =
{
   EpsonBeginJob,          // pfnInitJob;
   EpsonChooseRasterize,   // pfnRasterizeBand;
   EpsonNewFrame,          // pfnNextPage;
   EpsonEndJob,            // pfnTermJob;
   EpsonAbortJob,          // pfnAbortJob;
   EpsonDeviceQuery,       // pfnDeviceQuery;
   NULL,                   // pfnQueryFonts;
   NULL,                   // pfnDeviceOpen;
   NULL,                   // pfnDeviceClose;
   NULL,                   // pfnDeviceWrite;
   NULL,                   // pfnSpoolerOpen;
   NULL,                   // pfnSpoolerStartJob;
   NULL,                   // pfnSpoolerEndJob;
   NULL,                   // pfnSpoolerClose;
   (PFNBANDINGSUPPORT)NULL,// pfnBandingSupport;
   SampEPSQueryPDL         // pfnQueryPDL;
}; /* end EPSON_Subfunctions */

//
// -------------------------------------------------------------------------
// Start of individual device support
// -------------------------------------------------------------------------
//

CMDINFO EPSON_9Generic_Commands =
{
  4                                                  ,  // length
  _ESC_ ASCII(@) HEX(00) HEX(00)                     ,  // pszCmdInit;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTerm;
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdReset;
  3                                                  ,  // length
  HEX(0C) _ESC_ ASCII(@)                             ,  // pszCmdAbort
  1                                                  ,  // length
  _FF_                                               ,  // pszCmdPageEject;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdPortrait
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLandscape
  4                                                  ,  // length
  _ESC_ ASCII($) "%w"                                ,  // pszCmdSetXYPos
  4                                                  ,  // length
  _ESC_ ASCII($) "%w"                                ,  // pszCmdSetXPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetYPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSelfTest;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLineTerm;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextLtoR;
  4                                                  ,  // length
  _ESC_ ASCII(U) "%c"                                ,  // pszCmdBidi;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdBeginRasterGraphics;
  1                                                  ,  // length
  _CR_                                               ,  // pszCmdEndRasterGraphicsLine;
  1                                                  ,  // length
  _LF_                                               ,  // pszCmdMoveToNextRasterGraphicsLine;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEndRasterGraphics;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetResolution;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetCompression
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTransferRasterBlock
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetTopMargin;
  0                                                  ,  // length
  NULL                                                  // pszCmdTransferRasterPlane
}; /* end EPSON_9Generic_Commands */

// Arrays of unique Font, Tray and Form ID's that have been
// previosly defiend by user above
// *NOTE: first value in array is default value for device

ULONG EPSON_9Generic_pulResolutions[] = {
             EPSON_RES_ID_60_72,         //   9 pin resolutions
             EPSON_RES_ID_72_72,         //
             EPSON_RES_ID_80_72,         //
             EPSON_RES_ID_90_72,         //
             EPSON_RES_ID_120_72,        //
             EPSON_RES_ID_240_72         //
};
#define EPSON_9Generic_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_9Generic_pulResolutions)/sizeof (EPSON_9Generic_pulResolutions[0]))

ULONG EPSON_9_w144_72_pulResolutions[] = {
             EPSON_RES_ID_60_72,         //   9 pin resolutions
             EPSON_RES_ID_72_72,         //
             EPSON_RES_ID_80_72,         //
             EPSON_RES_ID_90_72,         //
             EPSON_RES_ID_120_72,        //
             EPSON_RES_ID_144_72,        //  special resolution only some support
             EPSON_RES_ID_240_72         //
};
#define EPSON_9_w144_72_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_9_w144_72_pulResolutions)/sizeof (EPSON_9_w144_72_pulResolutions[0]))

ULONG EPSON_9Generic_pulTrays[] = {
                   EPSON_TRAY_ID_REAR_CONTINUOUS,
                   EPSON_TRAY_ID_FRONT_CONTINUOUS,
                   EPSON_TRAY_ID_SINGLE_SHEET,
                   EPSON_TRAY_ID_MANUAL_FEED,
                   EPSON_TRAY_ID_BIN1,
                   EPSON_TRAY_ID_BIN2,
                   EPSON_TRAY_ID_CSF
};
#define EPSON_9Generic_SUPPORTED_TRAYS           (sizeof (EPSON_9Generic_pulTrays)/sizeof (EPSON_9Generic_pulTrays[0]))

USERDEFDATA EPSON_9Generic_UserData =
{

  FALSE, // BOOL             bTraysInit;
  0,     // USHORT           usNumTrays;
  NULL,  // PUSERTRAY        pUserTRAYS;
  NULL,  // PUSERTRAY        pLastUserTRAYS;
  FALSE, // BOOL             bFormsInit;
  0,     // USHORT           usNumForms;
  NULL,  // PUSERFORM        pUserFORMS;
  NULL,  // PUSERFORM        pLastUserFORMS;
  FALSE, // BOOL             bConnsInit;
  0,     // USHORT           usNumConnects;
  NULL,  // PUSERCONNECT     pUserCONNECTS;
  NULL,  // PUSERCONNECT     pLastUserCONNECTS;
  FALSE, // BOOL             bFontsInit;
  0,     // USHORT           usNumFonts;
  NULL,  // PUSERFONT        pUserFONTS;
  NULL   // PUSERFONT        pLastUserFONTS;
};

/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

CMDINFO EPSON_24Generic_Commands =
{
  4                                                  ,  // length
  _ESC_ ASCII(@) HEX(00) HEX(00)                     ,  // pszCmdInit;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTerm;
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdReset;
  3                                                  ,  // length
  HEX(0C) _ESC_ ASCII(@)                             ,  // pszCmdAbort
  1                                                  ,  // length
  _FF_                                               ,  // pszCmdPageEject;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdPortrait
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLandscape
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetXYPos
  4                                                  ,  // length
  _ESC_ ASCII($) "%w"                                ,  // pszCmdSetXPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetYPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSelfTest;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLineTerm;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextLtoR;
  4                                                  ,  // length
  _ESC_ ASCII(U) "%c"                                ,  // pszCmdBidi;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdBeginRasterGraphics;
  1                                                  ,  // length
  _CR_                                               ,  // pszCmdEndRasterGraphicsLine;
  1                                                  ,  // length
  _LF_                                               ,  // pszCmdMoveToNextRasterGraphicsLine;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEndRasterGraphics;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetResolution;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetCompression
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTransferRasterBlock
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetTopMargin;
  0                                                  ,  // length
  NULL                                                  // pszCmdTransferRasterPlane
}; /* end EPSON_24Generic_Commands */


// Arrays of unique Font, Tray and Form ID's that have been
// previosly defiend by user above
// *NOTE: first value in array is default value for device

ULONG EPSON_24Generic_pulResolutions[] = {
             EPSON_RES_ID_60_180      ,  //
             EPSON_RES_ID_120_180     ,  //
             EPSON_RES_ID_90_180      ,  //  24 pin resolutions
             EPSON_RES_ID_180_180     ,  //
             EPSON_RES_ID_360_180        //
};
#define EPSON_24Generic_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_24Generic_pulResolutions)/sizeof (EPSON_24Generic_pulResolutions[0]))

ULONG EPSON_24_w360_pulResolutions[] = {
             EPSON_RES_ID_60_180      ,  //
             EPSON_RES_ID_120_180     ,  //
             EPSON_RES_ID_90_180      ,  //  24 pin resolutions
             EPSON_RES_ID_180_180     ,  //
             EPSON_RES_ID_360_180     ,  //
             EPSON_RES_ID_360_360_24     //
};
#define EPSON_24_w360_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_24_w360_pulResolutions)/sizeof (EPSON_24_w360_pulResolutions[0]))

ULONG EPSON_24Generic_pulTrays[] = {
                   EPSON_TRAY_ID_REAR_CONTINUOUS,
                   EPSON_TRAY_ID_FRONT_CONTINUOUS,
                   EPSON_TRAY_ID_SINGLE_SHEET,
                   EPSON_TRAY_ID_MANUAL_FEED,
                   EPSON_TRAY_ID_BIN1,
                   EPSON_TRAY_ID_BIN2,
                   EPSON_TRAY_ID_CSF
};
#define EPSON_24Generic_SUPPORTED_TRAYS           (sizeof (EPSON_24Generic_pulTrays)/sizeof (EPSON_24Generic_pulTrays[0]))

USERDEFDATA EPSON_24Generic_UserData =
{

  FALSE, // BOOL             bTraysInit;
  0,     // USHORT           usNumTrays;
  NULL,  // PUSERTRAY        pUserTRAYS;
  NULL,  // PUSERTRAY        pLastUserTRAYS;
  FALSE, // BOOL             bFormsInit;
  0,     // USHORT           usNumForms;
  NULL,  // PUSERFORM        pUserFORMS;
  NULL,  // PUSERFORM        pLastUserFORMS;
  FALSE, // BOOL             bConnsInit;
  0,     // USHORT           usNumConnects;
  NULL,  // PUSERCONNECT     pUserCONNECTS;
  NULL,  // PUSERCONNECT     pLastUserCONNECTS;
  FALSE, // BOOL             bFontsInit;
  0,     // USHORT           usNumFonts;
  NULL,  // PUSERFONT        pUserFONTS;
  NULL   // PUSERFONT        pLastUserFONTS;
};

/*****************************************************************************/
/* EPSON_ESCP24J84_Commands (DDK)                                            */
/*****************************************************************************/

CMDINFO EPSON_ESCP24J84_Commands =
{
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdInit;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTerm;
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdReset;
  3                                                  ,  // length
  HEX(0C) _ESC_ ASCII(@)                             ,  // pszCmdAbort
  1                                                  ,  // length
  _FF_                                               ,  // pszCmdPageEject;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdPortrait
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLandscape
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetXYPos
  4                                                  ,  // length
  _ESC_ HEX(5C) "%w"                                 ,  // pszCmdSetXPos
  3                                                  ,  // length
  _ESC_ ASCII(J) "%c"                                ,  // pszCmdSetYPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSelfTest;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLineTerm;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextLtoR;
  3                                                  ,  // length
  _ESC_ ASCII(U) "%c"                                ,  // pszCmdBidi;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdBeginRasterGraphics;
  1                                                  ,  // length
  _CR_                                               ,  // pszCmdEndRasterGraphicsLine;
  1                                                  ,  // length
  _LF_                                               ,  // pszCmdMoveToNextRasterGraphicsLine;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEndRasterGraphics;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetResolution;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetCompression
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTransferRasterBlock
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetTopMargin;
  0                                                  ,  // length
  NULL                                                  // pszCmdTransferRasterPlane
}; /* end EPSON_ESCP24J84_Commands */

ULONG EPSON_ESCP24J84_pulResolutions[] = {
             EPSON_RES_ID_60_60       ,  //
             EPSON_RES_ID_120_60      ,  //
             EPSON_RES_ID_240_60      ,  //
             EPSON_RES_ID_90_60       ,  //
             EPSON_RES_ID_60_180      ,  //  24 pin resolutions
             EPSON_RES_ID_120_180     ,  //
             EPSON_RES_ID_90_180      ,  //
             EPSON_RES_ID_180_180     ,  //
             EPSON_RES_ID_360_180        //
};
#define EPSON_ESCP24J84_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_ESCP24J84_pulResolutions)/sizeof (EPSON_ESCP24J84_pulResolutions[0]))

/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

CMDINFO EPSON_48Generic_Commands =
{
  4                                                  ,  // length
  _ESC_ ASCII(@) HEX(00) HEX(00)                     ,  // pszCmdInit;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTerm;
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdReset;
  3                                                  ,  // length
  HEX(0C) _ESC_ ASCII(@)                             ,  // pszCmdAbort
  1                                                  ,  // length
  _FF_                                               ,  // pszCmdPageEject;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdPortrait
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLandscape
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetXYPos
  4                                                  ,  // length
  _ESC_ ASCII($) "%w"                                ,  // pszCmdSetXPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetYPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSelfTest;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLineTerm;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextLtoR;
  4                                                  ,  // length
  _ESC_ ASCII(U) "%c"                                ,  // pszCmdBidi;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdBeginRasterGraphics;
  1                                                  ,  // length
  _CR_                                               ,  // pszCmdEndRasterGraphicsLine;
  1                                                  ,  // length
  _LF_                                               ,  // pszCmdMoveToNextRasterGraphicsLine;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEndRasterGraphics;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetResolution;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetCompression
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTransferRasterBlock
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetTopMargin;
  0                                                  ,  // length
  NULL                                                  // pszCmdTransferRasterPlane
}; /* end EPSON_48Generic_Commands */


ULONG EPSON_48Generic_pulResolutions[] = {
             EPSON_RES_ID_60_180,        //
             EPSON_RES_ID_120_180,       //
             EPSON_RES_ID_90_180,        //  24 pin resolutions
             EPSON_RES_ID_180_180,       //
             EPSON_RES_ID_360_180,       //

             EPSON_RES_ID_60_360,        //
             EPSON_RES_ID_90_360,        //
             EPSON_RES_ID_120_360,       //  48 pin resolutions
             EPSON_RES_ID_180_360,       //
             EPSON_RES_ID_360_360        //
};
#define EPSON_48Generic_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_48Generic_pulResolutions)/sizeof (EPSON_48Generic_pulResolutions[0]))

ULONG EPSON_48Generic_pulTrays[] = {
                   EPSON_TRAY_ID_REAR_CONTINUOUS,
                   EPSON_TRAY_ID_FRONT_CONTINUOUS,
                   EPSON_TRAY_ID_SINGLE_SHEET,
                   EPSON_TRAY_ID_MANUAL_FEED,
                   EPSON_TRAY_ID_BIN1,
                   EPSON_TRAY_ID_BIN2,
                   EPSON_TRAY_ID_CSF
};
#define EPSON_48Generic_SUPPORTED_TRAYS           (sizeof (EPSON_48Generic_pulTrays)/sizeof (EPSON_48Generic_pulTrays[0]))

USERDEFDATA EPSON_48Generic_UserData =
{

  FALSE, // BOOL             bTraysInit;
  0,     // USHORT           usNumTrays;
  NULL,  // PUSERTRAY        pUserTRAYS;
  NULL,  // PUSERTRAY        pLastUserTRAYS;
  FALSE, // BOOL             bFormsInit;
  0,     // USHORT           usNumForms;
  NULL,  // PUSERFORM        pUserFORMS;
  NULL,  // PUSERFORM        pLastUserFORMS;
  FALSE, // BOOL             bConnsInit;
  0,     // USHORT           usNumConnects;
  NULL,  // PUSERCONNECT     pUserCONNECTS;
  NULL,  // PUSERCONNECT     pLastUserCONNECTS;
  FALSE, // BOOL             bFontsInit;
  0,     // USHORT           usNumFonts;
  NULL,  // PUSERFONT        pUserFONTS;
  NULL   // PUSERFONT        pLastUserFONTS;
};

/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

CMDINFO EPSON_Esc2P_Commands =
{
  4                                                  ,  // length
  _ESC_ ASCII(@) HEX(00) HEX(00)                     ,  // pszCmdInit;
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdTerm;
  2                                                  ,  // length
  _ESC_ ASCII(@)                                     ,  // pszCmdReset;
  3                                                  ,  // length
  HEX(0C) _ESC_ ASCII(@)                             ,  // pszCmdAbort
  1                                                  ,  // length
  _FF_                                               ,  // pszCmdPageEject;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdPortrait
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLandscape
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetXYPos
  4                                                  ,  // length
  _ESC_ ASCII($) "%w"                                ,  // pszCmdSetXPos
  7                                                  ,  // length
  _ESC_ "(V" HEX(02) HEX(00) "%w"                    ,  // pszCmdSetYPos
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSelfTest;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEOLWrapOff;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdLineTerm;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextLtoR;
  4                                                  ,  // length
  _ESC_ ASCII(U) "%c"                                ,  // pszCmdBidi;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOn;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTextScaleOff;
  6                                                  ,  // length
  _ESC_ "(G" HEX(01) HEX(00) HEX(01)                 ,  // pszCmdBeginRasterGraphics;
  1                                                  ,  // length
  _CR_                                               ,  // pszCmdEndRasterGraphicsLine;
  1                                                  ,  // length
  _LF_                                               ,  // pszCmdMoveToNextRasterGraphicsLine;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdEndRasterGraphics;
  7                                                  ,  // length
  _ESC_ "(U" HEX(01) HEX(00) "%c"                    ,  // pszCmdSetResolution;
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetCompression
  0                                                  ,  // length
  NULL                                               ,  // pszCmdTransferRasterBlock
  0                                                  ,  // length
  NULL                                               ,  // pszCmdSetTopMargin;
  0                                                  ,  // length
  NULL                                                  // pszCmdTransferRasterPlane
}; /* end EPSON_Esc2P_Commands */

ULONG EPSON_Esc2P_pulResolutions[] = {
             EPSON_RES_ID_180_180_2P,   //
             EPSON_RES_ID_360_180_2P,   //   EscP2 resolutions
             EPSON_RES_ID_360_360_2P    //
};
#define EPSON_Esc2P_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_Esc2P_pulResolutions)/sizeof (EPSON_Esc2P_pulResolutions[0]))

ULONG EPSON_Esc2P720_pulResolutions[] = {
             EPSON_RES_ID_180_180_2P,   //
             EPSON_RES_ID_360_180_2P,   //   EscP2 resolutions
             EPSON_RES_ID_360_360_2P,   //
             EPSON_RES_ID_720_720_2P    //
};
#define EPSON_Esc2P720_SUPPORTED_RESOLUTIONS     (sizeof (EPSON_Esc2P720_pulResolutions)/sizeof (EPSON_Esc2P720_pulResolutions[0]))

ULONG EPSON_Esc2PSpecial_pulResolutions[] = {
             EPSON_RES_ID_180_180_2P,   //
             EPSON_RES_ID_360_180_2P,   //   EscP2 resolutions
             EPSON_RES_ID_360_360_2P    //
#if 0
/* @TBD garbage */
         ,
             EPSON_RES_ID_360_720_2P    //
#endif
};
#define EPSON_Esc2PSpecial_SUPPORTED_RESOLUTIONS (sizeof (EPSON_Esc2PSpecial_pulResolutions)/sizeof (EPSON_Esc2PSpecial_pulResolutions[0]))

ULONG EPSON_Esc2P_pulTrays[] = {
                   EPSON_TRAY_ID_REAR_CONTINUOUS,
                   EPSON_TRAY_ID_FRONT_CONTINUOUS,
                   EPSON_TRAY_ID_SINGLE_SHEET,
                   EPSON_TRAY_ID_MANUAL_FEED,
                   EPSON_TRAY_ID_BIN1,
                   EPSON_TRAY_ID_BIN2,
                   EPSON_TRAY_ID_CSF_ESCP2
};
#define EPSON_Esc2P_SUPPORTED_TRAYS           (sizeof (EPSON_Esc2P_pulTrays)/sizeof (EPSON_Esc2P_pulTrays[0]))

USERDEFDATA EPSON_Esc2P_UserData =
{

  FALSE, // BOOL             bTraysInit;
  0,     // USHORT           usNumTrays;
  NULL,  // PUSERTRAY        pUserTRAYS;
  NULL,  // PUSERTRAY        pLastUserTRAYS;
  FALSE, // BOOL             bFormsInit;
  0,     // USHORT           usNumForms;
  NULL,  // PUSERFORM        pUserFORMS;
  NULL,  // PUSERFORM        pLastUserFORMS;
  FALSE, // BOOL             bConnsInit;
  0,     // USHORT           usNumConnects;
  NULL,  // PUSERCONNECT     pUserCONNECTS;
  NULL,  // PUSERCONNECT     pLastUserCONNECTS;
  FALSE, // BOOL             bFontsInit;
  0,     // USHORT           usNumFonts;
  NULL,  // PUSERFONT        pUserFONTS;
  NULL   // PUSERFONT        pLastUserFONTS;
};

/*****************************************************************************/
/*                                                                           */
/*****************************************************************************/

DEVICEINFO EPSON_pDevices[] =
{
  {
    PRN_ID_EPSON_9PIN_GENERIC_NARROW,
    "9-pin 80 Col",                                // pszDeviceName;
    "Epson Generic 9 pin - 80 columns",            // pszDeviceDesc;
    "IBM",                                         // pszDeviceVendor;
    1,                                             // usDeviceMajorVersion;
    0,                                             // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                      // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                               // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                          // Raster info
    0,                                             // ulDeviceBits;
    0,                                             // ulCompressModes;
    &EpsonMemory,                                  // pMemory;
    &EPSON_9Generic_Commands,                      // pCommands;
    &EPSON_Subfunctions,                           // pSubclassedFunctions;
    &EPSON_Features,                               // pFeatures;
    EPSON_9Generic_SUPPORTED_RESOLUTIONS,          // usNumRes;
    EPSON_9Generic_pulResolutions,                 // pulRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_9Generic_SUPPORTED_TRAYS,                // usNumTrays;
    EPSON_9Generic_pulTrays,                       // pTRAYS;
    EPSON_DEFINED_GENNARROWFORMS,                  // ulNumForms;
    EPSON_pGenNarrowForms,                         // pFORMS;
    EPSON_SUPPORTED_MEDIAS,                        // ulNumMedias;
    EPSON_pulMedias,                               // pulMedias;
    NULL,                                          // pInk;
    EPSON_R_Cont_SUPPORTED_CONNECTIONS,            // usNumConnects;
    EPSON_R_Cont_pulConnects,                      // pConnects;
    EPSON_BW_SUPPORTED_PRINT_MODES,                // usNumPrintModes;
    EPSON_BW_pulPrintModes,                        // pulPrintModes;
    0,                                             // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                       // pTripletModifiers;
    &EPSON_9Generic_UserData,
    {
       0,                         // ***IGNORED***
       {0},                       // ***IGNORED*** NOTE the {}! It is an array!
       0,                         // ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       EPSON_CONN_ID_6,           // ulDefConnID
       EPSON_CONN_ID_1,           // ulDefNonMetricConnID
       0,                         // ulDefFontID
       EPSON_RES_ID_240_72,       // ulDefResID
       EPSON_PRINT_MODE_ID_MONO,  // ulDefPrintModeID
       0,                         // ulCompressModes
       COLOR_BLACK,               // usForeground
       COLOR_WHITE,               // usBackground
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_CONDITIONAL,        // bFFControlType
       0,                         // ulJobPhaseControl
       0,                         // ulClrGamma;
       0,                         // ulClrBias;
    }
  } /* end Device */
,
  {
    PRN_ID_EPSON_9PIN_GENERIC_WIDE,
    "9-pin 136 Col",                               // pszDeviceName;
    "Epson Generic 9 pin - 136 columns",           // pszDeviceDesc;
    "IBM",                                         // pszDeviceVendor;
    1,                                             // usDeviceMajorVersion;
    0,                                             // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                      // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                               // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                          // Raster info
    0,                                             // ulDeviceBits;
    0,                                             // ulCompressModes;
    &EpsonMemory,                                  // pMemory;
    &EPSON_9Generic_Commands,                      // pCommands;
    &EPSON_Subfunctions,                           // pSubclassedFunctions;
    &EPSON_Features,                               // pFeatures;
    EPSON_9Generic_SUPPORTED_RESOLUTIONS,          // usNumRes;
    EPSON_9Generic_pulResolutions,                 // pulRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_9Generic_SUPPORTED_TRAYS,                // usNumTrays;
    EPSON_9Generic_pulTrays,                       // pTRAYS;
    EPSON_DEFINED_GENWIDEFORMS,                    // ulNumForms;
    EPSON_pGenWideForms,                           // pFORMS;
    EPSON_SUPPORTED_MEDIAS,                        // ulNumMedias;
    EPSON_pulMedias,                               // pulMedias;
    NULL,                                          // pInk;
    EPSON_R_Cont_SUPPORTED_CONNECTIONS,            // usNumConnects;
    EPSON_R_Cont_pulConnects,                      // pConnects;
    EPSON_BW_SUPPORTED_PRINT_MODES,                // usNumPrintModes;
    EPSON_BW_pulPrintModes,                        // pulPrintModes;
    0,                                             // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                       // pTripletModifiers;
    &EPSON_9Generic_UserData,
    {
       0,                         // ***IGNORED***
       {0},                       // ***IGNORED*** NOTE the {}! It is an array!
       0,                         // ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       EPSON_CONN_ID_6,           // ulDefConnID
       EPSON_CONN_ID_1,           // ulDefNonMetricConnID
       0,                         // ulDefFontID
       EPSON_RES_ID_240_72,       // ulDefResID
       EPSON_PRINT_MODE_ID_MONO,  // ulDefPrintModeID
       0,                         // ulCompressModes
       COLOR_BLACK,               // usForeground
       COLOR_WHITE,               // usBackground
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_CONDITIONAL,        // bFFControlType
       0,                         // ulJobPhaseControl
       0,                         // ulClrGamma;
       0,                         // ulClrBias;
    }
  } /* end Device */
,
  {
    PRN_ID_EPSON_24PIN_GENERIC_NARROW,
    "24-pin 80 Col",                               // pszDeviceName;
    "Epson Generic 24 pin - 80 columns",           // pszDeviceDesc;
    "IBM",                                         // pszDeviceVendor;
    1,                                             // usDeviceMajorVersion;
    0,                                             // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                      // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                               // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                          // Raster info
    0,                                             // ulDeviceBits;
    0,                                             // ulCompressModes;
    &EpsonMemory,                                  // pMemory;
    &EPSON_24Generic_Commands,                     // pCommands;
    &EPSON_Subfunctions,                           // pSubclassedFunctions;
    &EPSON_Features,                               // pFeatures;
    EPSON_24Generic_SUPPORTED_RESOLUTIONS,         // usNumRes;
    EPSON_24Generic_pulResolutions,                // pulRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_24Generic_SUPPORTED_TRAYS,               // usNumTrays;
    EPSON_24Generic_pulTrays,                      // pTRAYS;
    EPSON_DEFINED_GENNARROWFORMS,                  // ulNumForms;
    EPSON_pGenNarrowForms,                         // pFORMS;
    EPSON_SUPPORTED_MEDIAS,                        // ulNumMedias;
    EPSON_pulMedias,                               // pulMedias;
    NULL,                                          // pInk;
    EPSON_R_Cont_SUPPORTED_CONNECTIONS,            // usNumConnects;
    EPSON_R_Cont_pulConnects,                      // pConnects;
    EPSON_BW_SUPPORTED_PRINT_MODES,                // usNumPrintModes;
    EPSON_BW_pulPrintModes,                        // pulPrintModes;
    0,                                             // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                       // pTripletModifiers;
    &EPSON_24Generic_UserData,
    {
       0,                         // ***IGNORED***
       {0},                       // ***IGNORED*** NOTE the {}! It is an array!
       0,                         // ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       EPSON_CONN_ID_6,           // ulDefConnID
       EPSON_CONN_ID_1,           // ulDefNonMetricConnID
       0,                         // ulDefFontID
       EPSON_RES_ID_180_180,      // ulDefResID
       EPSON_PRINT_MODE_ID_MONO,  // ulDefPrintModeID
       0,                         // ulCompressModes
       COLOR_BLACK,               // usForeground
       COLOR_WHITE,               // usBackground
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_CONDITIONAL,        // bFFControlType
       0,                         // ulJobPhaseControl
       0,                         // ulClrGamma;
       0,                         // ulClrBias;
    }
  } /* end Device */
,
  {
    PRN_ID_EPSON_24PIN_GENERIC_WIDE,
    "24-pin 136 Col",                              // pszDeviceName;
    "Epson Generic 24 pin - 136 columns",          // pszDeviceDesc;
    "IBM",                                         // pszDeviceVendor;
    1,                                             // usDeviceMajorVersion;
    0,                                             // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                      // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                               // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                          // Raster info
    0,                                             // ulDeviceBits;
    0,                                             // ulCompressModes;
    &EpsonMemory,                                  // pMemory;
    &EPSON_24Generic_Commands,                     // pCommands;
    &EPSON_Subfunctions,                           // pSubclassedFunctions;
    &EPSON_Features,                               // pFeatures;
    EPSON_24Generic_SUPPORTED_RESOLUTIONS,         // usNumRes;
    EPSON_24Generic_pulResolutions,                // pulRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_24Generic_SUPPORTED_TRAYS,               // usNumTrays;
    EPSON_24Generic_pulTrays,                      // pTRAYS;
    EPSON_DEFINED_GENWIDEFORMS,                    // ulNumForms;
    EPSON_pGenWideForms,                           // pFORMS;
    EPSON_SUPPORTED_MEDIAS,                        // ulNumMedias;
    EPSON_pulMedias,                               // pulMedias;
    NULL,                                          // pInk;
    EPSON_R_Cont_SUPPORTED_CONNECTIONS,            // usNumConnects;
    EPSON_R_Cont_pulConnects,                      // pConnects;
    EPSON_BW_SUPPORTED_PRINT_MODES,                // usNumPrintModes;
    EPSON_BW_pulPrintModes,                        // pulPrintModes;
    0,                                             // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                       // pTripletModifiers;
    &EPSON_24Generic_UserData,
    {
       0,                         // ***IGNORED***
       {0},                       // ***IGNORED*** NOTE the {}! It is an array!
       0,                         // ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       EPSON_CONN_ID_6,           // ulDefConnID
       EPSON_CONN_ID_1,           // ulDefNonMetricConnID
       0,                         // ulDefFontID
       EPSON_RES_ID_180_180,      // ulDefResID
       EPSON_PRINT_MODE_ID_MONO,  // ulDefPrintModeID
       0,                         // ulCompressModes
       COLOR_BLACK,               // usForeground
       COLOR_WHITE,               // usBackground
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_CONDITIONAL,        // bFFControlType
       0,                         // ulJobPhaseControl
       0,                         // ulClrGamma;
       0,                         // ulClrBias;
    }
  } /* end Device */
,
  {
    PRN_ID_EPSON_48PIN_GENERIC,
    "Epson Generic 48 pin",                        // pszDeviceName;
    "Epson Generic 48 pin",                        // pszDeviceDesc;
    "IBM",                                         // pszDeviceVendor;
    1,                                             // usDeviceMajorVersion;
    0,                                             // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                      // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                               // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                          // Raster info
    0,                                             // ulDeviceBits;
    0,                                             // ulCompressModes;
    &EpsonMemory,                                  // pMemory;
    &EPSON_48Generic_Commands,                     // pCommands;
    &EPSON_Subfunctions,                           // pSubclassedFunctions;
    &EPSON_Features,                               // pFeatures;
    EPSON_48Generic_SUPPORTED_RESOLUTIONS,         // usNumRes;
    EPSON_48Generic_pulResolutions,                // pulRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_48Generic_SUPPORTED_TRAYS,               // usNumTrays;
    EPSON_48Generic_pulTrays,                      // pTRAYS;
    EPSON_DEFINED_GENNARROWFORMS,                  // ulNumForms;
    EPSON_pGenNarrowForms,                         // pFORMS;
    EPSON_SUPPORTED_MEDIAS,                        // ulNumMedias;
    EPSON_pulMedias,                               // pulMedias;
    NULL,                                          // pInk;
    EPSON_R_Cont_SUPPORTED_CONNECTIONS,            // usNumConnects;
    EPSON_R_Cont_pulConnects,                      // pConnects;
    EPSON_BW_SUPPORTED_PRINT_MODES,                // usNumPrintModes;
    EPSON_BW_pulPrintModes,                        // pulPrintModes;
    0,                                             // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                       // pTripletModifiers;
    &EPSON_48Generic_UserData,
    {
       0,                         // ***IGNORED***
       {0},                       // ***IGNORED*** NOTE the {}! It is an array!
       0,                         // ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       EPSON_CONN_ID_6,           // ulDefConnID
       EPSON_CONN_ID_1,           // ulDefNonMetricConnID
       0,                         // ulDefFontID
       EPSON_RES_ID_360_360,      // ulDefResID
       EPSON_PRINT_MODE_ID_MONO,  // ulDefPrintModeID
       0,                         // ulCompressModes
       COLOR_BLACK,               // usForeground
       COLOR_WHITE,               // usBackground
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_CONDITIONAL,        // bFFControlType
       0,                         // ulJobPhaseControl
       0,                         // ulClrGamma;
       0,                         // ulClrBias;
    }
  } /* end Device */
,
  {
    PRN_ID_EPSON_ESC2P_GENERIC,
    "Epson Generic ESC2P",                         // pszDeviceName;
    "Epson Generic ESC2P",                         // pszDeviceDesc;
    "IBM",                                         // pszDeviceVendor;
    1,                                             // usDeviceMajorVersion;
    0,                                             // usDeviceMinorVersion;
    CAPS_TECH_RASTER_PRINTER,                      // ulOS2DeviceTechnology;
    OMNI_CAP_MONO | OMNI_CAP_MIRROR,               // ulOmniDeviceTechnology;
    DEV_FUNC_RASTER,                               // ulDeviceFunctions;
    RASTER_TOP_TO_BOTTOM,                          // Raster info
    0,                                             // ulDeviceBits;
    0,                                             // ulCompressModes;
    &EpsonMemory,                                  // pMemory;
    &EPSON_Esc2P_Commands,                         // pCommands;
    &EPSON_Subfunctions,                           // pSubclassedFunctions;
    &EPSON_Features,                               // pFeatures;
    EPSON_Esc2P_SUPPORTED_RESOLUTIONS,             // usNumRes;
    EPSON_Esc2P_pulResolutions,                    // pulRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_Esc2P_SUPPORTED_TRAYS,                   // usNumTrays;
    EPSON_Esc2P_pulTrays,                          // pTRAYS;
    EPSON_DEFINED_GENNARROWFORMS,                  // ulNumForms;
    EPSON_pGenNarrowForms,                         // pFORMS;
    EPSON_SUPPORTED_MEDIAS,                        // ulNumMedias;
    EPSON_pulMedias,                               // pulMedias;
    NULL,                                          // pInk;
    EPSON_R_Cont_SUPPORTED_CONNECTIONS,            // usNumConnects;
    EPSON_R_Cont_pulConnects,                      // pConnects;
    EPSON_BW_SUPPORTED_PRINT_MODES,                // usNumPrintModes;
    EPSON_BW_pulPrintModes,                        // pulPrintModes;
    0,                                             // ulNumModifiers;
    (PTRIPLETMODIFIERS)NULL,                       // pTripletModifiers;
    &EPSON_Esc2P_UserData,
    {
       0,                         // ***IGNORED***
       {0},                       // ***IGNORED*** NOTE the {}! It is an array!
       0,                         // ***IGNORED***
       ORIENTATION_PORTRAIT,      // ulOrientation
       1,                         // cCopies
       EPSON_CONN_ID_6,           // ulDefConnID
       EPSON_CONN_ID_1,           // ulDefNonMetricConnID
       0,                         // ulDefFontID
       EPSON_RES_ID_360_360_2P,   // ulDefResID
       EPSON_PRINT_MODE_ID_MONO,  // ulDefPrintModeID
       0,                         // ulCompressModes
       COLOR_BLACK,               // usForeground
       COLOR_WHITE,               // usBackground
       IMAGE_OPTION_NOMIRROR,     // OR options together
       HT_MAGIC_SQUARES,          // Half Tone Snap as default
       127,                       // Default 127 for HT_LEVEL HaftTone Algorithm
       0,                         // lHue;         // % deviation along circle
       0,                         // lSaturation;  // % deviation along radius
       0,                         // lValue;       // % deviation darkness value
       0,                         // lDarkness;    // % deviation darkness value
       FF_CTL_CONDITIONAL,        // bFFControlType
       0,                         // ulJobPhaseControl
       0,                         // ulClrGamma;
       0,                         // ulClrBias;
    }
  } /* end Device */
}; /* end EPSON_pDevices block */

#define EPSON_SUPPORTED_DEVICES         (sizeof (EPSON_pDevices)/sizeof (EPSON_pDevices[0]))

DRIVERINFO pSAMPEPSONDriver[] =
{
  {
    "Sample EPSON",                                // pszDriverName;
    "IBM",                                         // pszDriverVendor;
    0x0210,                                        // usDriverVersion;
    EPSON_DEFINED_RESOLUTIONS,                     // usNumRes;
    EPSON_pRes,                                    // pRES;
    0,                                             // usNumFonts;
    NULL,                                          // pFONTS;
    EPSON_DEFINED_TRAYS,                           // usNumTrays;
    EPSON_pTrays,                                  // pTRAYS;
    EPSON_DEFINED_GENNARROWFORMS,                  // ulNumForms;
    EPSON_pGenNarrowForms,                         // pFORMS;
    EPSON_DEFINED_MEDIAS,                          // ulNumMedias;
    EPSON_pMedias,                                 // pMEDIAS;
    EPSON_DEFINED_CONNECTIONS,                     // usNumConnects;
    EPSON_pConnects,                               // pCONNECTS;
    EPSON_DEFINED_PRINT_MODES,                     // usNumPrintModes;
    EPSON_pPrintModes,                             // pPrintModes;
    EPSON_SUPPORTED_DEVICES,                       // usNumDevices;
    EPSON_pDevices,                                // pDEVICES;
    EPSON_pGammas                                  // pGammaTables;
  } /* end logical block */
}; /* end Driver block */
