/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/* MODULE NAME    : Form Connection Functions                               */
/* SOURCE NAME    : CONNECT.C                                               */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/26/93                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/

// os2 includes
#define INCL_DOS
#define INCL_PM
#include <os2.h>

// c includes
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

// @DBCS
#define INCL_VMANDDI
#include <ddi.h>
#include <pmddi.h>

#include "def.h"
#include "driver.h"
#include "funcs.h"


/****************************************************************************/
/* PROCEDURE NAME : CreateUniqueConnectID                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10-18-93                                                */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:    */
/*                                                                          */
/* RETURN VALUES: */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ULONG CreateUniqueConnectID( PDEVICEINFO pDevice )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ULONG        ulConnID = 0;
  PUSERCONNECT pUserConnect;
  BOOL         bIDClash = FALSE;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/


  // if we have user defined Connects to check against
  if( pDevice->pUserDefData->usNumConnects > 0 )
  {
    // sanity check pointers
    assert( pDevice->pUserDefData->pUserCONNECTS );

    // as long as we don't exceed some large max ID number
    // we'll keep trying to assign a unique ID
    while( ulConnID < MAX_USER_DEF_ID )
    {

      // assign Connect list pointer to local variable
      pUserConnect = pDevice->pUserDefData->pUserCONNECTS;

      // for each entry in linked list of user defined Connects
      while( (pUserConnect != NULL) && !bIDClash )
      {
        if( pUserConnect->fcUser.ulConnID == ( ulConnID | DEF_TYPE_USER ) )
        {
          bIDClash = TRUE;
          break;
        } /* end if */

        pUserConnect = pUserConnect->pNextConnect;

      } /* end while */

      // we had a clash try next ID
      if( bIDClash )
      {
        ulConnID++;
        bIDClash = FALSE;

      } /* end if */
      else
      {

        // we have found a match let's get out of while loop
        break;

      } /* end else */

    } /* end while */

  } /* end if */

  return( ulConnID | DEF_TYPE_USER );

} /* end CreateUniqueConnectID */



/****************************************************************************/
/* PROCEDURE NAME : AddConnectionEntry                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:    */
/*                                                                          */
/* RETURN VALUES: */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* 06/27/95 - MFR - Separated validation of uniqueness to new function      */
/*                  "IsConnectionUnique"                                    */
/*                                                                          */
/****************************************************************************/
BOOL AddConnectionEntry( PDEVICEINFO pDevice, PFORMCONNECTION pNewConnection )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PUSERCONNECT pUserConn;
  BOOL         bSuccess = FALSE;
  BOOL         rc;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DBPRINTF(( "AddConnectionEntry(): Enter\n" ));

  // allocate connection struct and add to current list
  pUserConn = (PUSERCONNECT) GplMemoryAlloc (globals.pvSharedHeap,
                                             sizeof (USERCONNECT));

  assert( pDevice );
  assert( pUserConn );

  // if memory is allocated all right
  if( pUserConn )
  {
    // Since we are altering global data, request a global semaphore
    // for the changes we need to make...
    rc = DosRequestMutexSem( procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT );
    assertT (rc);

    // Fill in new connection entry
    pUserConn->fcUser.ulConnID  = pNewConnection->ulConnID;
    pUserConn->fcUser.ulTrayID  = pNewConnection->ulTrayID;
    pUserConn->fcUser.ulFormID  = pNewConnection->ulFormID;
    pUserConn->fcUser.ulMediaID = pNewConnection->ulMediaID;
    pUserConn->pNextConnect     = NULL;

    DBPRINTF(( "AddConnectionEntry(): pUserCONNECTS = %x; Tray ID=%lu; Form ID=%lu; Media ID=%lu\n",
    pDevice->pUserDefData->pUserCONNECTS,
    pUserConn->fcUser.ulTrayID,
    pUserConn->fcUser.ulFormID ,
    pUserConn->fcUser.ulMediaID ));

    // Add new Connection structure to linked list
    if( pDevice->pUserDefData->pUserCONNECTS == NULL )
    {
      // first connection being defined add at top of list
      pDevice->pUserDefData->pUserCONNECTS = pUserConn;

    } /* end if */
    else
    {
      // add at end of list
      pDevice->pUserDefData->pLastUserCONNECTS->pNextConnect = pUserConn;

    } /* end else */

    pDevice->pUserDefData->pLastUserCONNECTS = pUserConn;
    pDevice->pUserDefData->usNumConnects++;

    rc = DosReleaseMutexSem( procdata.hmtxGlobalSem );
    assertT (rc);
  } /* end if */

  DBPRINTF(( "AddConnectionEntry(): Exit\n" ));
  bSuccess = TRUE;

#ifndef DEBUG
  rc++;       // Get rid of warning
#endif

  return( bSuccess );

} /* end AddConnectionEntry */


/****************************************************************************/
/* PROCEDURE NAME : IsConnectionUnique                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:    */
/*                                                                          */
/* RETURN VALUES: */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL IsConnectionUnique( PDLGINSTANCE pdi, PFORMCONNECTION pNewConnection )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PDEVICEINFO  pDevice;
  PUSERCONNECT pCurrent;
  ULONG        ul, ulTrayID, ulFormID, ulMediaID;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Setup and validate local variables
  assert( pdi );
  pDevice = pdi->pDevice;
  assert( pDevice );

  /*-----------------------------------------*/
  /* Check for duplicate built-in entry @TBD */
  /*-----------------------------------------*/

  // for each built-in form connection
  for( ul=0; ul < pDevice->usNumConnects; ul++ )
  {
    // get tray, size, and media ids for validation
    GetIDsFromConnID( pdi->pDriver,
                      pDevice,
                      pDevice->pulCONNECTS[ul],
                      &ulTrayID,
                      &ulFormID,
                      &ulMediaID );

    // if duplicate entry found display warning
    if( ulTrayID  == pNewConnection->ulTrayID &&
        ulFormID  == pNewConnection->ulFormID &&
        ulMediaID == pNewConnection->ulMediaID )
    {
      // return a failure
      return( FALSE );

    } /* end if */

  } /* end for */

  /*-----------------------------------------*/
  /* Check for duplicate user-defined entry  */
  /*-----------------------------------------*/

  // get pointer to first user-defined form connection
  pCurrent = pDevice->pUserDefData->pUserCONNECTS;

  // as long as we have user-defined form connections
  while( pCurrent != NULL )
  {
    // if duplicate entry found display warning
    if( pCurrent->fcUser.ulTrayID  == pNewConnection->ulTrayID &&
        pCurrent->fcUser.ulFormID  == pNewConnection->ulFormID &&
        pCurrent->fcUser.ulMediaID == pNewConnection->ulMediaID )
    {

      // return a failure
      return( FALSE );

    } /* end if */

    // else advance to next form entry
    pCurrent = pCurrent->pNextConnect;

  } /* end while */

  return( TRUE );

} /* end IsConnectionUnique */



/****************************************************************************/
/* PROCEDURE NAME : DeleteConnectionEntry                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9-16-93                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/*                                                                          */
/* PARAMETERS     : */
/*                                                                          */
/* RETURN VALUES  : */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL DeleteConnectionEntry( PDEVICEINFO pDevice, ULONG ulConnID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PUSERCONNECT pCurrent;
  PUSERCONNECT pLast = NULL;
  BOOL         bSuccess = FALSE;
  BOOL         rc;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /* Since we are altering global data, request a global semaphore
  ** for the changes we need to make...
  */
  rc = DosRequestMutexSem( procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT );
  assertT (rc);

  // search for Conn ID in linked list and once found delete entry
  // from list and fix chain
  pCurrent = pDevice->pUserDefData->pUserCONNECTS;

  while( pCurrent != NULL )
  {
    // matching ConnID
    if( pCurrent->fcUser.ulConnID == ulConnID )
    {
      // we are in middle of list chain
      if( pLast != NULL )
      {
        pLast->pNextConnect = pCurrent->pNextConnect;

      } /* end if */
      else
      {
        // deleting first entry
        pDevice->pUserDefData->pUserCONNECTS = pCurrent->pNextConnect;

      } /* end else */

      GplMemoryFree( pCurrent );

      pDevice->pUserDefData->usNumConnects--;

      bSuccess = TRUE;

      break;

    } /* end if */

    // advnce list pointers
    pLast = pCurrent;
    pCurrent = pCurrent->pNextConnect;

  } /* end while */

  rc = DosReleaseMutexSem( procdata.hmtxGlobalSem );
  assertT (rc);

#ifndef DEBUG
  rc++;       // Get rid of warning
#endif

  return( bSuccess );

} /* end DeleteConnectionEntry */


/****************************************************************************/
/* PROCEDURE NAME : FreeConnectionList                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:    */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID FreeConnectionList( PDEVICEINFO pDevice )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PUSERCONNECT pDelete, pNext;
  BOOL         rc;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DBPRINTF(( "FreeConnectionList(): Enter\n" ));

  assert( pDevice );

  if (!pDevice || !pDevice->pUserDefData)
     return;

  /* Since we are altering global data, request a global semaphore
  ** for the changes we need to make...
  */
  rc = DosRequestMutexSem( procdata.hmtxGlobalSem, SEM_INDEFINITE_WAIT );
  assertT (rc);

  // search for Conn ID in linked list and once found delete entry
  // from list and fix chain
  pDelete = pDevice->pUserDefData->pUserCONNECTS;

  while( pDelete != NULL )
  {
    pNext = pDelete->pNextConnect;

    GplMemoryFree( pDelete );

    pDelete = pNext;

  } /* end while */

  // reset linked list values
  pDevice->pUserDefData->bConnsInit        = FALSE;
  pDevice->pUserDefData->pUserCONNECTS     = NULL;
  pDevice->pUserDefData->pLastUserCONNECTS = NULL;
  pDevice->pUserDefData->usNumConnects = 0;

  rc = DosReleaseMutexSem( procdata.hmtxGlobalSem );
  assertT (rc);

#ifndef DEBUG
  rc++;       // Get rid of warning
#endif

  DBPRINTF(( "FreeConnectionList(): Exit\n" ));

} /* FreeConnectionList */

/****************************************************************************/
/* PROCEDURE NAME : SaveUserDefinedConnectList                              */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:    */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL SaveUserDefinedConnectList( PSZ pszAppName, PDEVICEINFO pDevice )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  USHORT          usConnectsProcessed = 0;
  USHORT          usTotalConnects;
  PUSERCONNECT    pCurrent;
  CHAR            szTemp[LEN_STRING];
  PFORMCONNECTION pConnect, pBase;
  ULONG           ulDataSize;
  ULONG           ulConnectInfoSize = sizeof( FORMCONNECTION );
  BOOL            bSuccess = TRUE;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // write out Num Connects to INI
  usTotalConnects = pDevice->pUserDefData->usNumConnects;

  _itoa( usTotalConnects, szTemp, 10 );

  PrfWriteProfileString( HINI_SYSTEMPROFILE,
                         pszAppName,
                         (PSZ)KEYNAME_NUM_USER_CONNECTS,
                         (PVOID)szTemp );


  // allocate memory to hold all Connectinfo and cast to FORMCONNECTION struct
  ulDataSize = usTotalConnects * sizeof( FORMCONNECTION );

  // if we have connection strucs to allocate for
  if( ulDataSize > 0 )
  {
    pBase = GplMemoryAlloc (0,  ulDataSize );

    assert( pBase );

    if( pBase )
    {
      pConnect = pBase;

      // set up start of list pointer
      pCurrent = pDevice->pUserDefData->pUserCONNECTS;

      // while we have Connects in linked list to process
      while( pCurrent != NULL && usConnectsProcessed <= usTotalConnects )
      {
        // copy out to memory area
        memcpy( pConnect, &(pCurrent->fcUser), ulConnectInfoSize );

        // increment counter
        usConnectsProcessed++;

        // advance buffer pointer
        pConnect++;

        // advance list pointer
        pCurrent = pCurrent->pNextConnect;

      } /* end while */

      PrfWriteProfileData( HINI_SYSTEMPROFILE,
                           pszAppName,
                           (PSZ)KEYNAME_USER_CONNECTS,
                           (PVOID)pBase,
                           ulDataSize );

      GplMemoryFree( pBase );

    } /* end if */
    else
    {
      bSuccess = FALSE;

    } /* end else */

  } /* end if */
  else
  {
    // overwrite any old entries with NULL
    PrfWriteProfileData( HINI_SYSTEMPROFILE,
                         pszAppName,
                         (PSZ)KEYNAME_USER_CONNECTS,
                         (PVOID)NULL,
                         ulDataSize );

  } /* end else */

  return( bSuccess );

} /* SaveUserDefinedConnectList */


/****************************************************************************/
/* PROCEDURE NAME : ReadUserDefinedConnectList                              */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   :                                                         */
/* DESCRIPTION    :                                                         */
/*                                                                          */
/* PARAMETERS:    */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL ReadUserDefinedConnectList( PSZ pszAppName, PDEVICEINFO pDevice )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PFORMCONNECTION pConnect, pBase;
  ULONG      ulDataSize;
  BOOL       bSuccess = TRUE;
  CHAR       szTemp[LEN_STRING+1];
  USHORT     usConnectsProcessed = 0;
  USHORT     usNumUserConnects = 0;
  BOOL       bRC;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DBPRINTF(( "ReadUserDefinedConnectList(): Enter; Appname = %s\n", pszAppName ));

  assert( pDevice );

  if( !pDevice->pUserDefData->bConnsInit )
  {
    ulDataSize = PrfQueryProfileString( (HINI)HINI_SYSTEM,
                                        pszAppName,
                                        (PSZ)KEYNAME_NUM_USER_CONNECTS,
                                        (PSZ)NULL,
                                        (PSZ)szTemp,
                                        (ULONG)LEN_STRING );

    // DBPRINTF(( "ReadUserDefinedConnectList(): PrfQueryProfileString: ulDataSize = %d\n", ulDataSize ));

    // If we have an INI entry
    if( ulDataSize != 0 )
    {
      // write out Num Connects to INI
      usNumUserConnects = atoi( szTemp );

      // DBPRINTF(( "ReadUserDefinedConnectList(): usNumConnects = %d\n", usNumUserConnects ));

      if( usNumUserConnects > 0)
      {
        // allocate memory to hold all Connectinfo and cast to FORMCONNECTION struct
        ulDataSize = usNumUserConnects * sizeof( FORMCONNECTION );

        pBase = GplMemoryAlloc (0,  ulDataSize );

        assert( pBase );

        if( pBase )
        {
          // read in profile data
          bRC = PrfQueryProfileData( HINI_SYSTEMPROFILE,
                                     pszAppName,
                                     (PSZ)KEYNAME_USER_CONNECTS,
                                     (PVOID)pBase,
                                     &ulDataSize );

          assert( bRC );

          // if data read in correctly
          if( bRC )
          {
            // init current entry pointer to each Connect structure
            pConnect = pBase;

            // while we have Connects in linked buffer to process
            while( usConnectsProcessed < usNumUserConnects )
            {
              AddConnectionEntry( pDevice, pConnect );

              // increment counter
              usConnectsProcessed++;

              // advance buffer pointer
              pConnect++;

            } /* end while */

            // free memory
            GplMemoryFree( pBase );

          } /* end if */
          else
          {
             bSuccess = FALSE;

          } /* end logical block */

        } /* end if */
        else
        {
          bSuccess = FALSE;

        } /* end else */

      } /* end if */

    } /* end if */
    else
    {
      DBPRINTF(( "ReadUserDefinedConnectList(): No INI data\n" ));

    } /* end else */

    // indicate we have read in connection data
    pDevice->pUserDefData->bConnsInit = TRUE;

  } /* end if */

  DBPRINTF(( "ReadUserDefinedConnectList(): Exit\n" ));

  return( bSuccess );

} /* ReadUserDefinedConnectList */



/****************************************************************************/
/* PROCEDURE NAME : GetIDsFromConnID                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/29/93                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL GetIDsFromConnID( PDRIVERINFO pDriver,
                       PDEVICEINFO pDevice,
                       ULONG ulConnID,
                       PULONG pulTrayID,
                       PULONG pulFormID,
                       PULONG pulMediaID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  BOOL            bSuccess = FALSE;
  ULONG           i;
  ULONG           ulNumDefined;
  PFORMCONNECTION pConnections;
  ULONG           ulDefType;

#ifdef DEBUG
  PDEBUGINFO  pDbg = &globals.DebugInfo;
#endif

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): Enter\n", __FUNCTION__ ));

  // Find out how form defined (in driver, in device, by user )
  ulDefType = MAKEULONG( 0, HIUSHORT(ulConnID) );

  switch ( ulDefType )
  {
    case DEF_TYPE_DRIVER:  // No hi word value
    {

      DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): DEF_TYPE_DRIVER\n", __FUNCTION__ ));

      ulNumDefined = pDriver->ulNumConnects;
      pConnections = pDriver->pCONNECTS;

      // For now only driver table definitions allowed
      // loop through table of connects until matching Connect ID
      // found
      for( i=0; i < ulNumDefined; i++ )
      {
         if( ulConnID == pConnections[i].ulConnID )
         {
            if(pulTrayID)
              *pulTrayID  = pConnections[i].ulTrayID;
            if(pulFormID)
              *pulFormID  = pConnections[i].ulFormID;
            if(pulMediaID)
              *pulMediaID = pConnections[i].ulMediaID;

            bSuccess = TRUE;

            break;

         } /* end if */

      } /* end for */

    } /* end case */
    break;

    case DEF_TYPE_DEVICE:
    {
      DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): DEF_TYPE_DEVICE!!! Not yet defined\n", __FUNCTION__ ));

    } /* end case */
    break;

    case DEF_TYPE_USER:
    {
      PUSERCONNECT pUserConnect;

      DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): DEF_TYPE_USER\n", __FUNCTION__ ));

      // assign form list pointer to local variable
      pUserConnect = pDevice->pUserDefData->pUserCONNECTS;

      // for each entry in linked list of user defined forms
      while( pUserConnect != NULL )
      {
        if( pUserConnect->fcUser.ulConnID == ulConnID )
        {
          if(pulTrayID)
            *pulTrayID  = pUserConnect->fcUser.ulTrayID;
          if(pulFormID)
            *pulFormID  = pUserConnect->fcUser.ulFormID;
          if(pulMediaID)
            *pulMediaID = pUserConnect->fcUser.ulMediaID;

          bSuccess = TRUE;

          break;

        } /* end if */

        pUserConnect = pUserConnect->pNextConnect;

      } /* end while */

    } /* end case */

    break;

  } /* end switch */

  if(pulTrayID)
    DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): TrayID = %lu\n", __FUNCTION__, *pulTrayID ));

  if(pulFormID)
    DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): FormID = %lu\n", __FUNCTION__, *pulFormID ));

  if(pulMediaID)
    DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): MediaID = %lu\n", __FUNCTION__, *pulMediaID ));

  DBPRINTIF(( pDbg->bGETIDSFROMCONNID,"%s(): Exit; bSuccess = %d\n", __FUNCTION__, bSuccess ));

  return( bSuccess );

} /* end GetIDsFromConnID */


/****************************************************************************/
/* PROCEDURE NAME : IsFormPartOfAConnection                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/02/94                                                 */
/* DESCRIPTION    : @94455 - Implemented to prevent users from deleting     */
/*                  forms we could potentially use from a connection        */
/*                  definition and trap.                                    */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL IsFormPartOfAConnection( PDEVICEINFO pDevice,
                              ULONG ulFormID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  BOOL            bMatch = FALSE;
  PUSERCONNECT    pUserConnect;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // assign form list pointer to local variable
  pUserConnect = pDevice->pUserDefData->pUserCONNECTS;

  // for each entry in linked list of user defined forms
  while( pUserConnect != NULL )
  {
    if( pUserConnect->fcUser.ulFormID == ulFormID )
    {
      bMatch = TRUE;

      break;

    } /* end if */

    pUserConnect = pUserConnect->pNextConnect;

  } /* end while */

  return( bMatch );

} /* end IsFormPartOfAConnection */

/****************************************************************************/
/* PROCEDURE NAME : IsTrayPartOfAConnection                                 */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 5/01/96                                                 */
/* DESCRIPTION    : @NEW   - Implemented to prevent users from deleting     */
/*                  forms we could potentially use from a connection        */
/*                  definition and trap.                                    */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL IsTrayPartOfAConnection( PDRIVERINFO pDriver,
                              PDEVICEINFO pDevice,
                              ULONG ulTrayID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PUSERCONNECT    pUserConnect;
  ULONG           ulNumDefined;
  PFORMCONNECTION pConnections;
  USHORT          i;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  //-----------------------------------
  // Check Built-in Connections
  //-----------------------------------
  ulNumDefined = pDriver->ulNumConnects;
  pConnections = pDriver->pCONNECTS;

  // For now only driver table definitions allowed
  // loop through table of connects until matching Connect ID
  // found
  for( i=0; i < ulNumDefined; i++ )
  {
    if( pConnections[i].ulTrayID == ulTrayID )
    {
      return( TRUE );

    } /* end if */

  } /* end for */

  //-----------------------------------
  // Check User-defined Connections
  //-----------------------------------

  // assign form list pointer to local variable
  pUserConnect = pDevice->pUserDefData->pUserCONNECTS;

  // for each entry in linked list of user defined forms
  while( pUserConnect != NULL )
  {
    if( pUserConnect->fcUser.ulTrayID == ulTrayID )
    {
      return( TRUE );

    } /* end if */

    pUserConnect = pUserConnect->pNextConnect;

  } /* end while */

  return( FALSE );

} /* end IsTrayPartOfAConnection */



/****************************************************************************/
/* PROCEDURE NAME : FillConnectionList                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/22/93                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID FillConnectionList( HWND hwndDlg, PDLGINSTANCE pdi, PDEVICEINFO pDevice,
                         ULONG ulListboxID, ULONG ulCurID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  USHORT          usNumConnects;
  USHORT          i;
  ULONG           ulConnID;
  ULONG           ulTrayID;
  ULONG           ulFormID;
  ULONG           ulMediaID;
  CHAR            szFormName[LEN_STRING];
  CHAR            szTrayName[LEN_STRING];
  CHAR            szMediaName[LEN_STRING];
  CHAR            szConnect[LEN_STRING*2];
  PUSERCONNECT    pUserConnect;
  USHORT          usMatchIdx = 0;
  BOOL            bMatchFound = FALSE;

#ifdef DEBUG
//PDEBUGINFO  pDbg  = &globals.DebugInfo;
  BOOL        bShow = FALSE;
#endif

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DBPRINTIF(( bShow, "%s(): Enter\n", __FUNCTION__ ));

  // initial pointer validation
  assertF( pdi );
  assertF( pDevice );
  assertF( pDevice->pulCONNECTS );

  // free existing list
  WinSendDlgItemMsg( hwndDlg,
                     ulListboxID,
                     LM_DELETEALL,
                     MPVOID,
                     MPVOID );

  /************************************/
  /* DRIVER CONNECTS                  */
  /************************************/

  usNumConnects = pDevice->usNumConnects;

  // add each form defined in driver table
  // that this device supports
  for( i=0; i < usNumConnects; i++ )
  {
    ulConnID = pDevice->pulCONNECTS[i];

    // if Form ID found and string returned
    if( GetIDsFromConnID( pdi->pDriver, pDevice, ulConnID, &ulTrayID, &ulFormID, &ulMediaID ) )
    {
      // if Tray, Form, and Media ID found and string returned
      if( TrayNameFromID( pdi->pDriver, pDevice, szTrayName, ulTrayID ) &&
          FormNameFromID( pdi->pDriver, pDevice, szFormName, ulFormID ) &&
          MediaNameFromID( pdi->pDriver, pDevice, szMediaName, ulMediaID ))
      {
        SHORT sIdx;

        sprintf( szConnect, "%s : %s : %s\0",
                 szTrayName,
                 szFormName,
                 szMediaName );

        sIdx = (SHORT)WinSendDlgItemMsg( hwndDlg,
                                         ulListboxID,
                                         LM_INSERTITEM,
                                         MPFROM2SHORT( LIT_END, 0 ),
                                         MPFROMP( szConnect ) );

        // Set Item Handle to be the Form ID
        WinSendDlgItemMsg( hwndDlg,
                           ulListboxID,
                           LM_SETITEMHANDLE,
                           MPFROMSHORT( sIdx ),
                           MPFROMLONG( ulConnID ) );

        DBPRINTIF(( bShow, "%s(): Adding Conn: '%s'; ID=%d\n",
                    __FUNCTION__, szConnect, ulConnID ));

        // If we are to highlite correct index
        if( !bMatchFound )
        {
          // If we match default form ID
          if( ulConnID == ulCurID )
          {
            usMatchIdx  = sIdx;
            bMatchFound = TRUE;

          } /* end if */

        } /* end if */

      } /* end if */

    } /* end if */

  } /* end for */

  /************************************/
  /* USER DEFINED CONNECTS            */
  /************************************/

  // get # Connects defined in linked list by user
  usNumConnects = pDevice->pUserDefData->usNumConnects;

  // assign base pointer to linked list
  pUserConnect = pDevice->pUserDefData->pUserCONNECTS;

  // add each Connect defined into listbox
  for( i=0; i < usNumConnects; i++ )
  {
    DBPRINTIF(( bShow, "%s(): Add DEF_TYPE_USER: Tray ID=%lu; Form ID=%lu; Media ID=%lu\n",
               __FUNCTION__,
               pUserConnect->fcUser.ulTrayID,
               pUserConnect->fcUser.ulFormID,
               pUserConnect->fcUser.ulMediaID ));

    // if Connect ID found and string returned
    if( TrayNameFromID( pdi->pDriver, pDevice, szTrayName, pUserConnect->fcUser.ulTrayID ) &&
        FormNameFromID( pdi->pDriver, pDevice, szFormName, pUserConnect->fcUser.ulFormID ) &&
        MediaNameFromID( pdi->pDriver, pDevice, szMediaName, pUserConnect->fcUser.ulMediaID ) )
    {
      SHORT sIdx;

      sprintf( szConnect, "%s : %s : %s\0",
               szTrayName,
               szFormName,
               szMediaName );

      sIdx = (SHORT)WinSendDlgItemMsg( hwndDlg,
                                       ulListboxID,
                                       LM_INSERTITEM,
                                       MPFROM2SHORT( LIT_END, 0 ),
                                       MPFROMP( szConnect ) );

      // Set Item Handle to be the Connect ID
      WinSendDlgItemMsg( hwndDlg,
                         ulListboxID,
                         LM_SETITEMHANDLE,
                         MPFROMSHORT( sIdx ),
                         MPFROMLONG( pUserConnect->fcUser.ulConnID ) );

      DBPRINTIF(( bShow, "%s(): Add DEF_TYPE_USER: '%s'; ID=%x\n", __FUNCTION__,
                  szConnect, pUserConnect->fcUser.ulConnID ));

      // If we are to highlite correct index
      if( !bMatchFound)
      {
        // If we match default form ID
        if( pUserConnect->fcUser.ulConnID == ulCurID )
        {
          usMatchIdx  = sIdx;
          bMatchFound = TRUE;

        } /* end if */

      } /* end if */

    } /* end if */

    // increment list pointer
    pUserConnect = pUserConnect->pNextConnect;

  } /* end for */

  // @TBD - highlite index may need to check if no match found
  // what to do as default then
  WinSendDlgItemMsg( hwndDlg,
                     ulListboxID,
                     LM_SELECTITEM,
                     MPFROMLONG( usMatchIdx ),
                     MPFROMLONG( TRUE ) );

  DBPRINTIF(( bShow, "%s(): Exit\n", __FUNCTION__ ));

} /* end FillConnectionList */


/****************************************************************************/
/* PROCEDURE NAME : MarkSelectableForms                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 11/11/93                                                */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID MarkIfSelectableCurrentForm (PDRIVERINFO pDriver,
                                  PDEVICEINFO pDevice,
                                  ULONG       ulFormID,
                                  PFORMINFO2  pForm)
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  USHORT          usNumConnects;
  USHORT          i;
  PUSERCONNECT    pUserConnect;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DBPRINTF(( "MarkIfSelectableCurrentForm(): Enter; Name = %s\n", pForm->hcInfo.szFormname));

  // initial pointer validation
  assert( pDevice );
  assert( pDevice->pulCONNECTS );

  /************************************/
  /* DRIVER CONNECTS                  */
  /************************************/

  usNumConnects = pDevice->usNumConnects;

  // for each connection find form and fill in fsSelection field
  // with HCCAPS_SELECTABLE
  for( i=0; i < usNumConnects; i++ )
  {
     if (ulFormID == pDevice->pulCONNECTS[i])
     {
        DBPRINTF (("is selectable.\n"));
        pForm->hcInfo.flAttributes |= HCAPS_SELECTABLE;
     }
  } /* end for */

  /************************************/
  /* USER DEFINED CONNECTS            */
  /************************************/

  // get # Connects defined in linked list by user
  usNumConnects = pDevice->pUserDefData->usNumConnects;

  // assign base pointer to linked list
  pUserConnect = pDevice->pUserDefData->pUserCONNECTS;

  // look at each user defined connect
  for( i=0; i < usNumConnects; i++ )
  {
     if (ulFormID == pUserConnect->fcUser.ulFormID)
     {
        DBPRINTF (("is selectable.\n"));
        pForm->hcInfo.flAttributes |= HCAPS_SELECTABLE;
     }

     // increment list pointer
     pUserConnect = pUserConnect->pNextConnect;
  } /* end for */
} /* end MarkIfSelectableCurrentForm */

/****************************************************************************/ 
/* PROCEDURE NAME : Find1stConnFromFromID                                   */ 
/* AUTHOR         : Y.Shih                                                  */ 
/* DATE WRITTEN   : 09/26/96                                                */ 
/* DESCRIPTION    :                                                         */ 
/*                                                                          */ 
/*                  Control Flow:                                           */ 
/*                                                                          */ 
/* PARAMETERS:                                                              */ 
/*                                                                          */ 
/* RETURN VALUES:                                                           */ 
/*                                                                          */ 
/*--------------------------------------------------------------------------*/ 
/* CHANGE/MODIFICATION LOG :                                                */ 
/*--------------------------------------------------------------------------*/ 
/*                                                                          */ 
/****************************************************************************/ 
BOOL Find1stConnFromFormID( PDRIVERINFO pDriver,                               
                            PDEVICEINFO pDevice,                               
                            ULONG       ulFormID,                              
                            PULONG      pulConnID)                             
{                                                                              
  /*------------------------------------------------------------------------*/ 
  /* LOCAL VARIABLES                                                        */ 
  /*------------------------------------------------------------------------*/ 
  USHORT          usNumConnects;                                               
  USHORT          i;                                                           
  ULONG           ulConn;                                                      
  BOOL            bRc = FALSE;                                                 
                                                                               
  /*------------------------------------------------------------------------*/ 
  /* BEGIN CODE                                                             */ 
  /*------------------------------------------------------------------------*/ 
                                                                               
  
  assert( pDevice );                                                           
  assert( pDevice->pulCONNECTS );                                              
                                                                               
  /************************************/                                       
  /* DRIVER CONNECTS                  */                                       
  /************************************/                                       
                                                                               
  usNumConnects = pDevice->usNumConnects;                                      
                                                                               
  for( i=0; i < usNumConnects; i++ )                                           
  {                                                                            
     ulConn = pDevice->pulCONNECTS[i];                                         
     if (ulFormID == pDriver->pCONNECTS[ulConn].ulFormID)                      
     {                                                                         
        *pulConnID = pDriver->pCONNECTS[ulConn].ulConnID;                      
        bRc = TRUE;                                                            
        break;                                                                 
     }                                                                         
  } /* end for */                                                              
                                                                               
  return bRc;                                                                  
} /* end Find1stConnFromFormID */                                              

