/****************************************************************************/
/*                                                                          */
/* By Mark Hamzy 10-7-97                                                    */
/*                                                                          */
/* This is a plugin module for the omni driver that when called will "push" */
/* (or send) a bitmap of the current band to a client using a rather simple */
/* protocol.                                                                */
/*                                                                          */
/****************************************************************************/

extern "C" {

#define INCL_WINSHELLDATA
#define INCL_GPI
#define INCL_OS2
#define INCL_DOS
#define INCL_DOSERRORS
#include <os2.h>

#define INCL_GENPLIB_ASSERT
#define INCL_GENPLIB_HANDLER
#define INCL_GENPLIB_UTIL
#define INCL_GENPLIB_RUNTIME
#include <genplib.h>

}

#include <memory.h>
#include <stdlib.h>

extern "C" {

// TCP/IP includes
#include <types.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <netdb.h>

}

#include "server.hpp"

#define DEFAULT_TIMEOUT   (5 * 60 * 1000)       // 5 minutes in miliseconds
#define DEFAULT_HOST      "127.0.0.1"
#define DEFAULT_PORT      9999

typedef enum _State {
   STATE_NONE,
   STATE_OPEN,
   STATE_CONNECTED,
   STATE_SENT_HEADER,
   STATE_SENDING_SCAN_LINES,
   STATE_DONE
} ESTATE;

/* OK.
**   I am trying something strange out here -- bear with me.
**   In order to provide some protection, there is a difference between the
**   "handle" that an application receives from these APIs and the internal handle
**   that the APIs use.  The internal handle is a pointer to a structure.  The
**   external handle is some 32-bit number (unique for each handle).  If the
**   public header files are distributed, then the application cannot dereference
**   the handle and do things with it.  So this is where the "_" comes into play.
**   internal type defines will have the "_" and external ones will not (where
**   there will be a name collision).  External variable names will have the "_"s
**   and internal names will not [ Note: this is reversed because the internal
**   APIs will use the variable names a lot more than they will use the type
**   defines. ]
**   Another odity is that since the app "handle" is a ULONG and the internal
**   handle is a pointer, the conversion looks a little strange since it is
**   casting a so-called structure to a pointer and a so-called pointer to
**   a structure to a pointer to a pointer to a structure.
*/

#define BHANDLE_SIG    0x68614862         // 'bHan'

typedef ULONG BHANDLE;
typedef struct _BandHandle {
   ULONG              cb;
   ULONG              ulSig;
   struct sockaddr_in tcp_client;
   INT                iSocket;
   ESTATE             eState;
} _BHANDLE_, *_PBHANDLE_;

/* Function prototypes...
*/
PBYTE APIENTRY     CreateInstance           (VOID);
VOID APIENTRY      SendBand                 (BHANDLE      _BHandle_,
                                             PBYTE        pbBits,
                                             PBITMAPINFO2 pbmi,
                                             PSIZEL       psizelBuffer,
                                             PSIZEL       psizelPage,
                                             PRECTL       prectlPageLocation);
VOID APIENTRY      DeleteInstance           (BHANDLE      _BHandle_);
_PBHANDLE_         ValidateHandle           (BHANDLE      _BHandle_);
BOOL               OpenConnection           (_PBHANDLE_   pBHandle);
BOOL               CloseConnection          (_PBHANDLE_   pBHandle);
INT                SocketReadBytes          (INT          iSocket,
                                             PBYTE        pbBytes,
                                             INT          cbBytes);
INT                SocketWriteBytes         (INT          iSocket,
                                             PBYTE        pbBytes,
                                             INT          cbBytes);

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
PBYTE APIENTRY
CreateInstance (VOID)
{
   CHAR               achPlace[255];
   struct hostent    *ent;
   PSZ                pszPort;
   _PBHANDLE_         pBHandle;
   APIRET             rc;
   PBYTE              pbRet         = NULL;

   rc = DosAllocMem ((PPVOID)&pbRet, sizeof (_BHANDLE_), fALLOC);
   assertT (rc);
   if (!pbRet)
      return NULL;

   pBHandle = (_PBHANDLE_)pbRet;
   memset (pbRet, 0, sizeof (_BHANDLE_));

   pBHandle->cb     = sizeof (_BHANDLE_);
   pBHandle->ulSig  = BHANDLE_SIG;
   pBHandle->eState = STATE_NONE;

   bzero (&pBHandle->tcp_client, sizeof (pBHandle->tcp_client));
   pBHandle->tcp_client.sin_family = AF_INET;

   if (1 < PrfQueryProfileString (HINI_USERPROFILE,
                                  (PSZ)"BandView",
                                  (PSZ)"Place",
                                  (PSZ)"",
                                  achPlace,
                                  sizeof (achPlace) - 1))
   {
      DBPRINTF (("OS2.INI / BandView / Place = '%s'\n", achPlace));

      pszPort = (PSZ)strchr (achPlace, ':');
      if (pszPort)
      {
         // There was a port number override
         *pszPort = '\0';
         pszPort++;
         pBHandle->tcp_client.sin_port = htons (atoi ((char *)pszPort));
      }
      else
         pBHandle->tcp_client.sin_port = htons (DEFAULT_PORT);

      // Resolve the host name
      if (INADDR_NONE == inet_addr (achPlace))
      {
         ent = gethostbyname (achPlace);
         if (ent)
            // If resolve was sucessful, copy the address
            bcopy (ent->h_addr,
                   (caddr_t)&pBHandle->tcp_client.sin_addr,
                   ent->h_length);
      }
      else
         // It is in the dotted number format (ex 205.218.112.2)
         pBHandle->tcp_client.sin_addr.s_addr = inet_addr (achPlace);
   }
   else
   {
      // Defaults
      pBHandle->tcp_client.sin_addr.s_addr = inet_addr (DEFAULT_HOST);
      pBHandle->tcp_client.sin_port        = htons (DEFAULT_PORT);
   }

   DBPRINTF (("host = %s\n", inet_ntoa (pBHandle->tcp_client.sin_addr)));
   DBPRINTF (("port = %d\n", ntohs (pBHandle->tcp_client.sin_port)));

   if (0 != sock_init ())
   {
      DBPRINTF (("INET.SYS probably is not running"));
      goto error;
   }

   return pbRet;

error:
   DeleteInstance ((BHANDLE)pBHandle);

   return NULL;
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
VOID APIENTRY
SendBand (BHANDLE           _BHandle_,
          PBYTE             pbBits,
          PBITMAPINFO2      pbmi2,
          PSIZEL            psizelBuffer,
          PSIZEL            psizelPage,
          PRECTL            prectlPageLocation)
{
   _PBHANDLE_         pBHandle       = (_PBHANDLE_)NULL;
   INT                iBytes;
   SCANMESSAGE        ScanMessage;
   ULONG              ulLength;
   register INT       i;

   pBHandle = ValidateHandle (_BHandle_);
   if (!pBHandle)
      return;

   OpenConnection (pBHandle);

   if (STATE_CONNECTED == pBHandle->eState)
   {
      BMAPMESSAGE   BMapMessage;
      INT           iNumColors;

      BMapMessage.cb        = sizeof (BMapMessage);
      BMapMessage.ulSig     = SIG;
      BMapMessage.ulCommand = COMMAND_BMAP_HEADER;

      iNumColors = 1 << pbmi2->cBitCount;
      if (256 >= iNumColors)
         BMapMessage.cb += (iNumColors - 1) * sizeof (RGB2);

      iBytes = SocketWriteBytes (pBHandle->iSocket,
                                 (PBYTE)&BMapMessage,
                                 SIZEOF_BMAPMESSAGE);
      assertT (iBytes != SIZEOF_BMAPMESSAGE);

      iBytes = SocketWriteBytes (pBHandle->iSocket,
                                 (PBYTE)pbmi2,
                                 BMapMessage.cb - SIZEOF_BMAPMESSAGE);
      assertT (iBytes != (BMapMessage.cb - SIZEOF_BMAPMESSAGE));

      pBHandle->eState = STATE_SENT_HEADER;
   }

   if (STATE_SENT_HEADER == pBHandle->eState)
   {
      ulLength = ((((pbmi2->cx * pbmi2->cBitCount) + 31) >> 5 ) << 2);

      pBHandle->eState = STATE_SENDING_SCAN_LINES;
   }

   if (STATE_SENDING_SCAN_LINES == pBHandle->eState)
   {
      for (i = 0; i < psizelBuffer->cy; i++)
      {
         ScanMessage.cb        = SIZEOF_SCANMESSAGE + ulLength;
         ScanMessage.ulSig     = SIG;
         ScanMessage.ulCommand = COMMAND_SCANLINE;

         iBytes = SocketWriteBytes (pBHandle->iSocket,
                                    (PBYTE)&ScanMessage,
                                    SIZEOF_SCANMESSAGE);
         assertT (iBytes != SIZEOF_SCANMESSAGE);

         iBytes = SocketWriteBytes (pBHandle->iSocket,
                                    pbBits + i * ulLength,
                                    ulLength);
         assertT (iBytes != ulLength);
      }

      pBHandle->eState = STATE_DONE;
   }

   CloseConnection (pBHandle);
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
VOID APIENTRY
DeleteInstance (BHANDLE           _BHandle_)
{
   _PBHANDLE_  pBHandle      = (_PBHANDLE_)NULL;
   APIRET      rc;

   pBHandle = ValidateHandle (_BHandle_);
   if (!pBHandle)
      return;

   CloseConnection (pBHandle);

   rc = DosFreeMem (pBHandle);
   assertT (rc);
}

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/
/*>>>>>>>>>>>>>>>>>>>>>>>>>> Internal functions <<<<<<<<<<<<<<<<<<<<<<<<<<<*/
/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>><<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
BOOL
OpenConnection (_PBHANDLE_ pBHandle)
{
   INT                iSocket;    /* socket for accepting connections      */
   INT                iBytes;
///OPENREQUEST        OpenMsg;
   OPENACK            Response;
   INT                iRc;
   BOOL               bRet = FALSE;

   /*
   ** TCP PUSH SERVER
   **
   ** Get a TCP socket for sending data to a client.
   **
   */
   if ((iSocket = socket (AF_INET, SOCK_STREAM, 0)) < 0)
   {
      DBPRINTF (("socket() failed!\n"));
      goto error;
   }

   pBHandle->eState = STATE_OPEN;

   iRc = connect (iSocket,
                  (struct sockaddr *)&pBHandle->tcp_client,
                  sizeof (pBHandle->tcp_client));
   if (0 > iRc)
   {
      DBPRINTF (("connect failed!\n"));
      goto error;
   }

   // Read the response
   iBytes = SocketReadBytes (iSocket,
                             (PBYTE)&Response,
                             sizeof (Response));
   DBPRINTF (("reading the response %d bytes, iBytes = %d\n",
              sizeof (Response), iBytes));

   if (sizeof (Response) != iBytes)
      goto error;

   pBHandle->iSocket = iSocket;
   pBHandle->eState  = STATE_CONNECTED;

   bRet = TRUE;

error:

   return bRet;
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
BOOL
CloseConnection (_PBHANDLE_ pBHandle)
{
   if (STATE_CONNECTED <= pBHandle->eState)
   {
      CLOSEMESSAGE       CloseMsg;
      INT                iBytes;

      // Close the session
      CloseMsg.cb        = sizeof (CloseMsg);
      CloseMsg.ulSig     = SIG;
      CloseMsg.ulCommand = COMMAND_CLOSE;

      iBytes = SocketWriteBytes (pBHandle->iSocket,
                                 (PBYTE)&CloseMsg,
                                 sizeof (CloseMsg));
      assertT (sizeof (CloseMsg) != iBytes);

      pBHandle->eState = STATE_OPEN;
   }

   if (STATE_OPEN <= pBHandle->eState)
   {
      soclose (pBHandle->iSocket);

      pBHandle->eState = STATE_NONE;
   }

   return TRUE;
}

/****************************************************************************/
/*                                                                          */
/****************************************************************************/
_PBHANDLE_
ValidateHandle (BHANDLE _BHandle_)
{
   _PBHANDLE_   pBHandle    = (_PBHANDLE_)_BHandle_;
   _PBHANDLE_   pRetHandle  = NULL;
   REGREC       regrec;
   ULONG        ulException;

   if (!_BHandle_)
      return pRetHandle;

   REGISTERHANDLER (regrec, 0)
   ulException = setjmp (regrec.jmp);
   if (ulException)
   {
      // clean up here as required
      assertT (ulException);

      // check for the killed-thread case
      switch (ulException)
      {
      case XCPT_PROCESS_TERMINATE:
      case XCPT_ASYNC_PROCESS_TERMINATE:
         DosUnsetExceptionHandler ((PEXCEPTIONREGISTRATIONRECORD)&regrec);
         DosExit (EXIT_THREAD, 0);
      }
      // error result
      goto depart;
   }

   // Dereference signature to see if it is valid
   if (BHANDLE_SIG == pBHandle->ulSig)
   {
      // Success!
      pRetHandle = pBHandle;
   }

depart:
   UNREGISTERHANDLER (regrec)

   return pRetHandle;
}

/****************************************************************************/
/* SocketReadBytes                                                          */
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
INT
SocketReadBytes (INT         iSocket,
                 PBYTE       pbBytes,
                 INT         cbBytes)
{
   static CHAR achBitBucket[4096];    // one big bit bucket!
   INT         iReadSocket;
   PSZ         pszBuffer;
   INT         cbBuffer;
   BOOL        fNotTossIntoBitBucket;
   INT         rc;

//DBPRINTF (("SocketReadBytes(%d) cbBytes = %d\n", iSocket, cbBytes));
   if (0 == cbBytes)
      // Special case!
      return 0;

   if (pbBytes)
   {
      fNotTossIntoBitBucket = TRUE;

      pszBuffer = (PSZ)pbBytes;
   }
   else
   {
      // Special case!
      fNotTossIntoBitBucket = FALSE;

      pszBuffer = (PSZ)achBitBucket;
   }

   cbBuffer = cbBytes;

   while (0 < cbBuffer)
   {
      iReadSocket = iSocket;
      rc = select (&iReadSocket,     // array of sockets
                   1,                // # read sockets
                   0,                // # write sockets
                   0,                // # exception sockets
                   DEFAULT_TIMEOUT); // timeout
//////DBPRINTF (("select(%d) = %d\n", iSocket, rc));
      assertT (0 > rc);

      if (1 == rc)
      {
         rc = recv (iSocket,
                    (char *)pszBuffer,
                    (fNotTossIntoBitBucket
                       ? cbBuffer
                       : min (cbBuffer, sizeof (achBitBucket))),
                    0);
/////////DBPRINTF (("recv(%d) = %d\n", iSocket, rc));
         assertT (0 > rc);

         if (0 >= rc)
            /* 0 == socket closed
            ** 0 <  error
            */
            return -1;

         if (fNotTossIntoBitBucket)
            pszBuffer += rc;

         cbBuffer -= rc;
      }
      else
         /* 0 == time out
         ** 0 <  error
         */
         return -1;
   }

///DBPRINTF (("SocketReadBytes(%d) returning %d\n", iSocket, cbBytes));
   return cbBytes;
}

/****************************************************************************/
/* Function SocketWriteBytes                                                */
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
INT
SocketWriteBytes (INT         iSocket,
                  PBYTE       pbBytes,
                  INT         cbBytes)
{
   INT         iWriteSocket;
   PSZ         pszBuffer;
   INT         cbBuffer;
   INT         rc;

//DBPRINTF (("SocketWriteBytes(%d) cbBytes = %d\n", iSocket, cbBytes));
   if (0 == cbBytes)
      // Special case!
      return 0;

   if (NULL == pbBytes)
      // Special case!
      return -1;

   pszBuffer = (PSZ)pbBytes;
   cbBuffer  = cbBytes;

   while (0 < cbBuffer)
   {
      iWriteSocket = iSocket;
      rc = select (&iWriteSocket,    // array of sockets
                   0,                // # read sockets
                   1,                // # write sockets
                   0,                // # exception sockets
                   DEFAULT_TIMEOUT); // timeout
//////DBPRINTF (("select(%d) = %d\n", iSocket, rc));
//////assertT (0 > rc);

      if (1 == rc)
      {
         while (0 < cbBuffer)
         {
            rc = send (iSocket,
                       (char *)pszBuffer,
                       min (cbBuffer, (4 * 1024)),
                       0);
////////////DBPRINTF (("send(%d) = %d\n", iSocket, rc));
            assertT (0 > rc);

            if (0 >= rc)
               /* 0 == socket closed
               ** 0 <  error
               */
               return -1;

            pszBuffer += rc;
            cbBuffer -= rc;
         }
      }
      else
         /* 0 == time out
         ** 0 <  error
         */
         return -1;
   }

///DBPRINTF (("SocketWriteBytes(%d) returning %d\n", iSocket, cbBytes));
   return cbBytes;
}

#ifdef _V3_CRT_
extern "C" {

int  _CRT_init        (void);
void _CRT_term        (void);

}
#endif

// ------------------------------------------------------------------------------------------------------------------
// _DLL_InitTerm is called once at process init, and once at process terminate
// ulFlag == 0, then do DLL init per process
// ulFlag == 1, then DLL freed per process

ULONG APIENTRY
_DLL_InitTerm (ULONG hThisModule, ULONG ulFlag)
{
   REGREC       regrec;
   ULONG        ulException;
   ULONG        ulrc = 0;

   // Is the module loading?  If so, then initialize before anything else
#ifdef _V3_CRT_
   if (0 == ulFlag)
   {
      if (_CRT_init () == -1)       /* Need to initialize runtimes for V3 */
      {
         return 0UL;
      }
   }
#endif

   if (0 == ulFlag)
   {
      GplRuntimeInit ();
   }

   REGISTERHANDLER (regrec, 0)
   ulException = setjmp (regrec.jmp);
   if (ulException)
   {
      // clean up here as required
      assertT (ulException);

      // check for the killed-thread case
      switch (ulException)
      {
      case XCPT_PROCESS_TERMINATE:
      case XCPT_ASYNC_PROCESS_TERMINATE:
         DosUnsetExceptionHandler ((PEXCEPTIONREGISTRATIONRECORD)&regrec);
         DosExit (EXIT_THREAD, 0);
      }
      // error result
      goto depart;
   }

   DBPRINTF (("%s(): Enter\n", __FUNCTION__));

   switch (ulFlag)
   {
   /*------------------------------------------------------------------------*/
   /* Load Module                                                            */
   /*------------------------------------------------------------------------*/
   case 0:
   {
      break;
   }

   /*------------------------------------------------------------------------*/
   /* Free Module                                                            */
   /*------------------------------------------------------------------------*/
   case 1:
   {
      break;
   }

   default:
   {
      // blow up the debug version
      assertstring ("unknown case!\n");
   }
   }

   // success
   ulrc = 1;

depart:
   DBPRINTF (("%s(): Exit; ulrc=%d\n", __FUNCTION__, ulrc));

   // Is the module unloading?  If so, then terminate as the last thing that we do
   if (1 == ulFlag)
   {
      GplRuntimeTerm ();
   }

#ifdef _V3_CRT_
   if (1 == ulFlag)
   {
      _CRT_term();   /* Clean up runtime environment */
   }
#endif

   return ulrc;

} /* end _DLL_InitTerm */
