/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Menu 3D                                                 */
/* AUTHOR         : Matthew F. Rutkowski                                    */
/* FILENAME       : MENU.C                                                  */
/* DATE WRITTEN   : 4-04-94                                                 */
/* DESCRIPTION    : Provides generically cool 3-D menu buttons              */
/*                                                                          */
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*--personal to-do list-----------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/

#define INCL_SPL
#define INCL_DOS
#define INCL_DEV
#define INCL_GPI
#define INCL_GPIERRORS
#define INCL_WIN

// DBCS enable
#define INCL_DOSNLS
// DBCS enable

#include <os2.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <setjmp.h>

#define INCL_GENPLIB_ASSERT
#define INCL_GENPLIB_MEMORY
#include <genplib.h>

#include "menu.h"

/*------------------------------------------------------------------------*/
/* prototypes                                                             */
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/* external prototypes                                                    */
/*------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/
/* GLOBALS                                                                */
/*------------------------------------------------------------------------*/
PFNWP pfnDefMenuProc;

#ifndef MM_SELECTITEM
#define MM_SELECTITEM         0x0189
#endif

BOOL CalcMaxTextWidth( HMENU3D hMenu3d, USHORT usLevel, PULONG pulWidth, ULONG ulType )
{

  BOOL  bError = FALSE;  // return code

  // If we really have a menu to free up
  if ( hMenu3d )
  {
    PBUTTON3D pCurrent, pNext;
    ULONG     ulWidth = 0;

    // Initialize pointer to first node in linked list
    pCurrent = hMenu3d->pFirstEntry;

    while( (hMenu3d->ulNumEntries > 0) && (pCurrent != NULL) )
    {
      // Remember next node in linked list
      pNext = pCurrent->pNext;

      if( pCurrent->usLevel == usLevel )
      {
        switch( ulType )
        {
          case BUTTON_SIZE_TEXT:
          {
            // Add Widths
            ulWidth = max( ulWidth, pCurrent->NameText.ulTextcx );

          } /* end if */
          break;

          case BUTTON_SIZE_CTLTEXT:
          {
            ulWidth = max( ulWidth, pCurrent->CtrlText.ulTextcx );

          } /* end else */
          break;

          case BUTTON_SIZE_BITMAP:
          {
            ulWidth = max( ulWidth, pCurrent->bmi.cx );

          } /* end else */
          break;

        } /* end switch */

        //DBPRINTF (("CalcMaxTextWidth: ulWidth=%lu\n", ulWidth ));

      } /* end if */

      // next node now becomes current node to process
      pCurrent = pNext;

    } /* end while */

    *pulWidth = ulWidth;

  } /* end if */

  return( bError );

} /* end CalcMaxTextWidth */

BOOL CalcMaxTextHeight( HMENU3D hMenu3d, USHORT usLevel, PULONG pulHeight, ULONG ulType )
{
  BOOL  bError = FALSE;  // return code

  // If we really have a menu to free up
  if ( hMenu3d )
  {
    PBUTTON3D pCurrent, pNext;
    ULONG     ulHeight = 0;

    // Initialize pointer to first node in linked list
    pCurrent = hMenu3d->pFirstEntry;

    while( (hMenu3d->ulNumEntries > 0) && (pCurrent != NULL) )
    {
      // Remember next node in linked list
      pNext = pCurrent->pNext;

      if( pCurrent->usLevel == usLevel )
      {
        switch( ulType )
        {
          case BUTTON_SIZE_TEXT:
          {
            // Add heights
            ulHeight = max( ulHeight, pCurrent->NameText.ulTextcy );

          } /* end if */
          break;

          case BUTTON_SIZE_CTLTEXT:
          {
            ulHeight = max( ulHeight, pCurrent->CtrlText.ulTextcy );

          } /* end else */
          break;

          case BUTTON_SIZE_BITMAP:
          {
            ulHeight = max( ulHeight, pCurrent->bmi.cy );

          } /* end else */
          break;

        } /* end switch */

        //DBPRINTF (("CalcMaxTextHeight: ulHeight=%lu\n", ulHeight ));

      } /* end if */

      // next node now becomes current node to process
      pCurrent = pNext;

    } /* end while */

    *pulHeight = ulHeight;

  } /* end if */

  return( bError );

} /* end CalcMaxTextHeight */

BOOL Create3DMenu( HAB hab, PHMENU3D phMenu, HWND hwndMenu, HMODULE hmodBmpExe )
{
  BOOL          bError = FALSE;  // return code
  DEVOPENSTRUC  dop;             // Info for opening Memory DC with display
  HMCB          hmcb = NULL;     // heap handle
  PMENU3D       pMenu3d;         // alloc Menu3d struct off heap
  SIZEL         sizl = { 0, 0 }; // use default page size of device

  // Initialize handle in case of error
  *phMenu = NULL;

  // Create a heap to use for allocations
  hmcb = GplMemoryCreateInstance ( MENU_3D_HEAP_LEN,  // Size
                                   0,                 // Extent
                                   0,                 // Threshold
                                   PROCESS_MEMORY);   // Type

  // Able to create a heap for menu usage
  if( hmcb!=NULL )
  {
    pMenu3d = (PMENU3D) GplMemoryAlloc ( hmcb, sizeof( MENU3D ));

    // Initialize Values in Menu structure
    pMenu3d->hmcb         = hmcb;
    pMenu3d->ulNumEntries = 0;
    pMenu3d->hdcMem       = (HDC)NULL;
    pMenu3d->hpsMem       = (HPS)NULL;
    pMenu3d->pFirstEntry  = NULL;
    pMenu3d->pLastEntry   = NULL;
    pMenu3d->pNextEntry   = NULL;
    pMenu3d->hwnd         = hwndMenu;
    pMenu3d->hmod         = hmodBmpExe;

    // initialize DEVOPENSTRUC
    dop.pszLogAddress = NULL;
    dop.pszDriverName = (PSZ) "DISPLAY";
    dop.pdriv         = NULL;
    dop.pszDataType   = NULL;

    // Open a Memory DC to display for drawing Menu items
    pMenu3d->hdcMem = DevOpenDC( hab, OD_MEMORY, (PSZ)"*", 4L, (PDEVOPENDATA)&dop, (HDC)NULL );

    // Success opening Memory DC
    if ( pMenu3d->hdcMem!=DEV_ERROR )
    {
      // Open a Memory PS to display using Memory DC for drawing menu items
      pMenu3d->hpsMem = GpiCreatePS( hab, pMenu3d->hdcMem, &sizl, PU_PELS | GPIT_MICRO | GPIA_ASSOC );

      // Success opening memory PS to display
      if ( pMenu3d->hpsMem!=GPI_ERROR )
      {
        // return handle to menu to caller SUCCESS!!!!
        *phMenu = pMenu3d;

      } /* end if */

    } /* end if */

  } /* end if */

  return( !bError );

} /* end Create3DMenu */


BOOL Destroy3DMenu( HMENU3D hMenu3d )
{
  BOOL          bError = FALSE;  // return code
  HMCB          hmcb = NULL;     // heap handle

  DBPRINTF(( "DestroyMenu3D(): Enter; hMenu3d = %x\n", hMenu3d ));

  assertF( hMenu3d );

  // If we really have a menu to free up
  if ( hMenu3d )
  {
    PBUTTON3D pCurrent, pNext;

    // remember heap handle (memory instance handle)
    hmcb = hMenu3d->hmcb;

    // Initialize pointer to first node in linked list
    pCurrent = hMenu3d->pFirstEntry;

    DBPRINTF(( "DestroyMenu3D(): pCurrent = %x; ulNumEntries = %d\n", pCurrent, hMenu3d->ulNumEntries ));

    while( (hMenu3d->ulNumEntries > 0) && (pCurrent != NULL) )
    {
      // Remember next node in linked list
      pNext = pCurrent->pNext;

      DBPRINTF(( "DestroyMenu3D(): Freeing Button = '%s'\n", pCurrent->NameText.pszText ));

      // Free any strings allocated in button structure
      if( pCurrent->NameText.pszText )
        GplMemoryFree( pCurrent->NameText.pszText );

      // Free any strings allocated in button structure
      if( pCurrent->CtrlText.pszText )
        GplMemoryFree( pCurrent->CtrlText.pszText );

      // Free current node in linked list
      GplMemoryFree( pCurrent );

      // Decrement # buttons in Menu linked list
      hMenu3d->ulNumEntries--;

      // next node now becomes current node to process
      pCurrent = pNext;

    } /* end while */

    // Free menu structure - @TBD as soon as Lee sets real return code (not 0)
    // add appropriate checks for them
    GplMemoryFree( hMenu3d );

    // If Heap exists free it up
    if( hmcb )
       GplMemoryDeleteInstance( hmcb );

  } /* end if */

  return( !bError );

} /* end Destroy3DMenu */


BOOL Create3DButton( HMENU3D hMenu3d, PHBUTTON3D phButton, ULONG ulControlID, ULONG ulBitmapID, PSZ pszText, ULONG ulStyle, USHORT usLevel )
{

   BOOL        bError = FALSE;  // return code
   PBUTTON3D   pButton3d;

   // Initialize handle in case of error
   *phButton = NULL;

   pButton3d = (PBUTTON3D) GplMemoryAlloc ( hMenu3d->hmcb, sizeof( BUTTON3D ));
   bError = PMAssert( (BOOL)(pButton3d==NULL), hMenu3d->hab, (PSZ)"MemAlloc failed" );

   // Able to allocate 3D button node
   if( !bError )
   {
     // Initialize values
     pButton3d->ulState            = 0;
     pButton3d->pNext              = NULL;
     pButton3d->NameText.ulLength  = 0;
     pButton3d->CtrlText.ulLength  = 0;

     // Copy fields passed in to our structure
     pButton3d->ulStyle            = ulStyle;
     pButton3d->ulControlID        = ulControlID;
     pButton3d->ulBitmapID         = ulBitmapID;
     pButton3d->usLevel            = usLevel;

     // If we have a string (includes control and mneumonics)
     if( pszText )
     {
       PSZ    pszTemp = NULL;
       PSZ    pszCtrl = NULL;
       POINTL aptl[TXTBOX_COUNT];

       // Search for Tab (\t) character and store away
       pszTemp = (PSZ)strchr( pszText, '\t' );

       if( pszTemp!= NULL )
       {
         pszCtrl = pszTemp + 1;

         // Terminate non-control part of string
         *pszTemp = '\0';

       } /* end if */

       // allocate memory to hold strings and initialize text strucs
       if( pszText != NULL )
       {
         pButton3d->NameText.ulLength = strlen( pszText );
         pButton3d->NameText.pszText = (PSZ) GplMemoryAlloc ( hMenu3d->hmcb, pButton3d->NameText.ulLength + 1 );
         strcpy( pButton3d->NameText.pszText, pszText );

         GpiQueryTextBox( hMenu3d->hpsMem, pButton3d->NameText.ulLength, pszText, TXTBOX_COUNT, (PPOINTL)aptl);

         pButton3d->NameText.ulTextcy = max( aptl[TXTBOX_TOPRIGHT].y,
                                             aptl[TXTBOX_TOPLEFT].y) -
                                        min( aptl[TXTBOX_BOTTOMRIGHT].y,
                                             aptl[TXTBOX_BOTTOMLEFT].y );

         pButton3d->NameText.ulTextcx = max( aptl[TXTBOX_TOPRIGHT].x,
                                             aptl[TXTBOX_BOTTOMRIGHT].x) -
                                        min( aptl[TXTBOX_TOPLEFT].x,
                                             aptl[TXTBOX_BOTTOMLEFT].x );

       } /* end if */

       // If we found a control string (shortcut key for menu buttons)
       if( pszCtrl != NULL )
       {
         pButton3d->CtrlText.ulLength = strlen( pszCtrl );
         pButton3d->CtrlText.pszText = (PSZ) GplMemoryAlloc ( hMenu3d->hmcb, pButton3d->CtrlText.ulLength + 1 );
         strcpy( pButton3d->CtrlText.pszText, pszCtrl );

         GpiQueryTextBox( hMenu3d->hpsMem, pButton3d->CtrlText.ulLength, pszCtrl, TXTBOX_COUNT, (PPOINTL)aptl);

         pButton3d->CtrlText.ulTextcy = max( aptl[TXTBOX_TOPRIGHT].y,
                                             aptl[TXTBOX_TOPLEFT].y) -
                                        min( aptl[TXTBOX_BOTTOMRIGHT].y,
                                             aptl[TXTBOX_BOTTOMLEFT].y );

         pButton3d->CtrlText.ulTextcx = max( aptl[TXTBOX_TOPRIGHT].x,
                                             aptl[TXTBOX_BOTTOMRIGHT].x) -
                                        min( aptl[TXTBOX_TOPLEFT].x,
                                             aptl[TXTBOX_BOTTOMLEFT].x );

       } /* end if */

       // Search for Mnemonic character and store away
       pszTemp = (PSZ)strchr( pszText, '~' );

       // We found a Mnemonic so store it away
       if ( pszTemp != NULL )
       {
         // Store character away for later testing
         // @TBD - need to include ctype.h correctly to get tolower
         // pButton3d->chMnemonic = tolower( (*(pszTemp+1)) );
         pButton3d->chMnemonic = (*(pszTemp+1));

       } /* end if */
       else
       {
         // No Mnemonic - set value to NULL character
         pButton3d->chMnemonic = 0;

       } /* end else */

     } /* end if */

     // If we have a bitmap ID to load
     if( ulBitmapID )
     {
       // load Bitmap from this module's resources
       pButton3d->hbm = GpiLoadBitmap( hMenu3d->hpsMem,// presentation space handle
                                       (HMODULE)hMenu3d->hmod,  // loads from application EXE
                                       ulBitmapID,     // bitmap resource #
                                       0L,             // sets width to actual
                                       0L );           // sets height to actual

       // Able to load the bitmap
       if( pButton3d->hbm!=GPI_ERROR )
       {
         // Set the bitmap into the memory PS (make it current)
         GpiSetBitmap( hMenu3d->hpsMem, pButton3d->hbm );

         // Success Selecting bitmap into PS
         // Get information about the bitmap for later use
         GpiQueryBitmapParameters( pButton3d->hbm, &(pButton3d->bmi) );

       } /* end if */

     } /* end if */
     else
     {
       // No bitmap to load initialize height/width to 0
       pButton3d->bmi.cx = 0;
       pButton3d->bmi.cy = 0;
       pButton3d->hbm = NULLHANDLE;

     } /* end else */

     // return handle to menu to caller
     *phButton = pButton3d;

   } /* end if */

   return( !bError );

} /* end Create3DButton */


BOOL Destroy3DButton( HMENU3D hMenu3d, HBUTTON3D hButton )
{
  BOOL bError = FALSE;  // return code

  DBPRINTF(( "Destroy3DButton(): Enter; Name = '%s'\n", hButton->NameText.pszText ));

  assertF( hMenu3d );

  if ( hButton )
  {
     // Free any strings allocated in structure
     if(hButton->NameText.pszText)
       GplMemoryFree( hButton->NameText.pszText );

     // Free any strings allocated in structure
     if(hButton->CtrlText.pszText)
       GplMemoryFree( hButton->CtrlText.pszText );

     // Free button structure
     GplMemoryFree( hButton );

  } /* end if */

  return( !bError );

} /* end Destroy3DButton */


BOOL Add3DButton( HMENU3D hMenu3d, HBUTTON3D hButton )
{
  assertF( hMenu3d );

  // If first entry
  if( hMenu3d->ulNumEntries == 0 )
  {
    // Set up list head pointer
    hMenu3d->pFirstEntry = hButton;

    // Insert button at end of list
    hMenu3d->pLastEntry = hButton;

  }/* end if */
  else
  {
    // Insert button at end of list
    hMenu3d->pLastEntry->pNext = hButton;
    hMenu3d->pLastEntry        = hButton;

  } /* end else */

  // Bump # buttons in Menu linked list
  hMenu3d->ulNumEntries++;

  return TRUE;
} /* end Add3DButton */



BOOL DrawDepressedBorder( HPS hps, POWNERITEM poi, PBUTTON3D pButton3d, POINTL  aptl[BUTTON_3D_TARGET_PTS], RECTL rcl, RECTL rclText, ULONG ulClrBackGround, PSZ pszText  )
{

  ULONG     ulClrForeGround = CLR_DARKGRAY;
  ULONG     ulcyTxtHalf;

  assertF( poi );

  // Place 3D depressed button below all other controls this
  // makes the window behave like a "group" 3D box if desired
  WinSetWindowPos( poi->hwnd, HWND_BOTTOM, 0L, 0L, 0L, 0L, SWP_ZORDER );

  // calculate 1/2 height of text line that will be used
  // to show "group box" title string this allows us to position
  // the string such that the border passes 1/2 way thru the text
  ulcyTxtHalf = pButton3d->NameText.ulTextcy/2;
  aptl[1].y   -= ulcyTxtHalf;

  // draw the outline of the group box
  GpiMove( hps, &(aptl[0]) );
  GpiSetColor( hps, CLR_BLACK );
  GpiBox( hps, DRO_OUTLINE, &(aptl[1]), 0L, 0L );

  // Calculate "group box" rectangle (minus outline box
  // we already drew and 1/2 text height on top)
  rcl.xLeft   += 1;
  rcl.yBottom += 1;
  rcl.xRight  -= 1;
  rcl.yTop    -= ulcyTxtHalf;

  // Calculate text rectangle at top of "group" box
  rclText.xLeft  = rcl.xLeft + BUTTON_GROUP_OFFSET ;
  rclText.xRight = rclText.xLeft + pButton3d->NameText.ulTextcx;
  rclText.yBottom= rcl.yTop - ulcyTxtHalf;
  rclText.yTop   = rcl.yTop + ulcyTxtHalf;

  // Draw the group box
  WinDrawBorder( hps,
                 &rcl,
                 BUTTON_3D_BORDER,
                 BUTTON_3D_BORDER,
                 ulClrForeGround,
                 ulClrBackGround,
                 DB_DEPRESSED );

  {
    RECTL rclTextBdr  = rclText;

    // allow for additional room around text so no
    // letters are clipped @TBD - use fontmetrics
    // to be more precise in future - MFR
    rclTextBdr.xLeft  -= 8;
    rclTextBdr.xRight += 8;
    rclTextBdr.yTop   += 4;
    rclTextBdr.yBottom-= 4;

    // erase area we will draw text into
    WinFillRect( hps, &rclTextBdr, ulClrBackGround );

    // draw a "raised" border around the "group box" text
    WinDrawBorder( hps,
                   &rclTextBdr,
                   BUTTON_3D_BORDER,
                   BUTTON_3D_BORDER,
                   ulClrForeGround,
                   ulClrBackGround,
                   DB_RAISED );

  } /* end logical block */

  WinDrawText( hps, pButton3d->NameText.ulLength, pszText, &rclText, 0L, 0L,
               DT_CENTER | DT_VCENTER | DT_TEXTATTRS | DT_MNEMONIC );

  return( TRUE );

} /* end DrawDepressedBorder */

BOOL Draw3DButton( HMENU3D hMenu3d, POWNERITEM poi, ULONG ulCtlId, ULONG ulFlag )
{

  BOOL      bError = FALSE; // assume error until we blt successfully
  ULONG     ul = 0;
  PBUTTON3D pButton3d;

  if(hMenu3d)
  {
    if( hMenu3d->ulNumEntries > 0 )
    {
      BOOL      bEntryFound = FALSE;
      BOOL      bMatch;

      pButton3d = hMenu3d->pFirstEntry;

      while( (ul < hMenu3d->ulNumEntries) && (pButton3d != NULL) )
      {

        if( ulFlag == BUTTON_ACTION_MNEMONIC )
        {
          bMatch = (pButton3d->chMnemonic == hMenu3d->ch);

        } /* end if */
        else
        {
          assertF( poi );

          bMatch = (pButton3d->ulControlID == poi->idItem);

        } /* end else */

        if( bMatch )
        {
          // let's try to blt it
          bEntryFound = TRUE;
          break;

        } /* end if */

        // Advance to next button in the list
        pButton3d = pButton3d->pNext;
        ul++;

      } /* end while */

      if( bEntryFound )
      {
        switch( ulFlag )
        {
          case BUTTON_ACTION_MNEMONIC:
          {
            MPARAM mp1, mp2;

            mp1 = MPFROM2SHORT( pButton3d->ulControlID, FALSE );
            mp2 = MPFROMLONG( FALSE );
            WinPostMsg( hMenu3d->hwnd, MM_SELECTITEM, mp1, mp2 );

          } /* end case */
          break;

          case BUTTON_ACTION_MEASURE:
          {
            ULONG ulMaxNameWidth, ulMaxCtrlWidth;

            CalcMaxTextWidth( hMenu3d, pButton3d->usLevel, &ulMaxNameWidth, 0 );
            CalcMaxTextWidth( hMenu3d, pButton3d->usLevel, &ulMaxCtrlWidth, 1 );

            poi->rclItem.xLeft   = 0;
            poi->rclItem.yBottom = 0;

            poi->rclItem.xRight  = pButton3d->bmi.cx;
            poi->rclItem.yTop    = pButton3d->bmi.cy;

            if( pButton3d->ulStyle & BS3D_TEXT_RIGHT )
            {
              poi->rclItem.xRight  = pButton3d->bmi.cx + ulMaxNameWidth + ulMaxCtrlWidth + BUTTON_TEXT_LEFT_MARGIN;

            } /* end if */

            if( pButton3d->ulStyle & BS3D_BORDER_3D )
            {

              poi->rclItem.yTop    += BUTTON_3D_BORDER_2X;
              poi->rclItem.xRight  += BUTTON_3D_BORDER_2X;

            } /* end if */

          } /* end case */
          break;

          case BUTTON_ACTION_DRAW:
          {
            POINTL  aptl[BUTTON_3D_TARGET_PTS];
            RECTL   rclText;
            ULONG   ulAlign;
            ULONG   ulRop;
            ULONG   ulClrBottomRight;
            ULONG   ulClrTopLeft;
            ULONG   ulClrText;
            ULONG   ulClrBackGround;
            USHORT  usTextShiftX;
            USHORT  usTextShiftY;
            PSZ     pszText;

            // TBD - try instead WinInvertRect call

            // set colors/values based on hilite
            if( BDS_HILITED & poi->fsState )
            {
               ulClrBottomRight = CLR_WHITE;
               ulClrTopLeft     = CLR_BLACK;
               ulClrText        = CLR_WHITE;
               ulClrBackGround  = CLR_DARKGRAY;
               ulRop            = ROP_NOTSRCCOPY;
               usTextShiftX     = 2;
               usTextShiftY     = 1;

            } /* end if */
            else
            {
               ulClrBottomRight = CLR_DARKGRAY;
               ulClrTopLeft     = CLR_WHITE;
               ulClrText        = CLR_BLACK;
               ulClrBackGround  = CLR_PALEGRAY;
               ulRop            = ROP_SRCCOPY;
               usTextShiftX     = 0;
               usTextShiftY     = 0;

            } /* end else */

            // if we have a bitmap handle
            if( pButton3d->hbm )
            {
              GpiSetBitmap( hMenu3d->hpsMem, pButton3d->hbm );

            } /* end if */

            // able to select 3D button bitmap
            if( !bError )
            {
              HPS    hps = poi->hps;
              RECTL  rcl = poi->rclItem;
              LONG   lLeftOver    = 0;

              // Rectangle area of 3-D control to draw
              aptl[0].x = rcl.xLeft;
              aptl[0].y = rcl.yBottom;

              aptl[1].y = rcl.yTop;
              aptl[1].x = rcl.xRight;

              pszText = pButton3d->NameText.pszText;

              if( pButton3d->ulStyle & BS3D_BORDER_DEPRESSED )
              {
                if( !( BDS_HILITED & poi->fsState ) )
                {
                  DrawDepressedBorder( hps, poi, pButton3d, aptl, rcl, rclText, ulClrBackGround, pszText  );

                } /* end if */

                return( TRUE );

              } /* end if */
              else
              {
                if( pButton3d->ulStyle & BS3D_BORDER_3D )
                {
                  USHORT i;

                  // Bottom-right
                  aptl[2].x = poi->rclItem.xRight;
                  aptl[2].y = poi->rclItem.yBottom;

                  GpiMove( hps, &(aptl[0]) );
                  GpiSetColor( hps, ulClrTopLeft );
                  GpiBox( hps, DRO_FILL, &(aptl[1]) , 0L, 0L );

                  GpiSetColor( hps, CLR_BLACK );
                  GpiBox( hps, DRO_OUTLINE, &(aptl[1]) , 0L, 0L );

                  GpiSetColor( hps, ulClrBottomRight );

                  for( i=0; i < BUTTON_3D_BORDER; i++ )
                  {
                    // Bottom-left
                    aptl[0].x += 1;
                    aptl[0].y += 1;

                    // Top-Right
                    aptl[1].x -= 1;
                    aptl[1].y -= 1;

                    // Bottom-right
                    aptl[2].x -= 1;
                    aptl[2].y += 1;

                    GpiMove( hps, &(aptl[0]) );
                    GpiLine( hps, &(aptl[2]) );
                    GpiLine( hps, &(aptl[1]) );

                  } /* end for */

                  GpiMove( hps, &(aptl[0]) );
                  GpiSetColor( hps, ulClrBackGround );
                  GpiBox( hps, DRO_OUTLINEFILL, &(aptl[1]) , 1L, 1L );
                  GpiSetColor( hps, ulClrText );

                  aptl[1].y += 1;

                } /* end if */

                // Set up source bitmap (picture to be drawn on button)
                aptl[2].x = 0;
                aptl[2].y = 0;

                aptl[3].x = pButton3d->bmi.cx;
                aptl[3].y = pButton3d->bmi.cy;

                // Coordinates for WinDrawText of button text
                rclText.xRight = poi->rclItem.xRight;

                // User wants text drawn on the 3D button
                if( pButton3d->ulStyle & BS3D_TEXT_INSIDE )
                {
                  // Target Bitmap Top-Right coordinate
                  aptl[1].x = poi->rclItem.xRight;

                  // Left edge of text rectangle
                  rclText.xLeft = poi->rclItem.xLeft + usTextShiftX;

                  // Vertical position of text (lower third of bitmap)
                  rclText.yBottom = poi->rclItem.yBottom + BUTTON_3D_BORDER + BUTTON_TEXT_BOTTOM_MARGIN - usTextShiftY;
                  rclText.yTop    = poi->rclItem.yBottom + pButton3d->bmi.cy/2 - BUTTON_3D_BORDER_2X - usTextShiftY;

                  // WinDrawText alignment
                  ulAlign = DT_CENTER;

                } /* end if */
                else
                {
                  // User wants text drawn to right of 3D button
                  if( pButton3d->ulStyle & BS3D_TEXT_RIGHT )
                  {
                    // Target Top-Right coordinate @DELTA - MFR
                    aptl[1].x = poi->rclItem.xLeft + pButton3d->bmi.cx + BUTTON_3D_BORDER;

                    // Left edge of text rectangle (past right edge of bitmap)
                    rclText.xLeft   = aptl[1].x + BUTTON_TEXT_LEFT_MARGIN + usTextShiftX;

                    // Center text vertically
                    rclText.yBottom = poi->rclItem.yBottom - usTextShiftY;
                    rclText.yTop    = poi->rclItem.yTop - usTextShiftY;

                    // WinDrawText alignment
                    ulAlign = DT_LEFT;

                  } /* end if */

                } /* end else */

                // Coordinates for GpiBitBlt of button bitmap
                lLeftOver = ((LONG)(poi->rclItem.yTop - poi->rclItem.yBottom)) - ((LONG)pButton3d->bmi.cy);

                if( lLeftOver > 0L )
                {
                  lLeftOver /= 2;
                  aptl[0].y = poi->rclItem.yBottom + lLeftOver;
                  aptl[1].y = poi->rclItem.yTop    - lLeftOver;

                } /* end if */

                // Not Supported in all display drivers yet...
                //{
                //  ULONG        ulAttrs = IBB_BACK_COLOR | IBB_BACK_MIX_MODE;
                //  IMAGEBUNDLE  ImgBnd;
                //  PIMAGEBUNDLE pImgBnd = &ImgBnd;
                //
                //
                //  ImgBnd.lBackColor    = ulClrBackGround;
                //  ImgBnd.usBackMixMode = BM_SRCTRANSPARENT;
                //
                //  GpiSetAttrs( hps,
                //               PRIM_IMAGE,
                //               ulAttrs,
                //               0L,
                //               &pImgBnd );
                //
                //} /* end logical block */

                // Draw 3D button if we have a bitmap to blt
                if( pButton3d->hbm )
                {
                  GpiBitBlt( hps,                        // target PS
                             hMenu3d->hpsMem,            // source PS
                             (LONG)BUTTON_3D_TARGET_PTS, // four points needed to compress
                             aptl,                       // points to source and target
                             ulRop,                      // copy source replacing target
                             BBO_IGNORE);                // discard extra rows and columns

                } /* end if */

                // make 3-D shading of text an option flag in future
                if( pButton3d->ulStyle & BS3D_TEXT_3D )
                {
                  RECTL rclBack    = rclText;
                  ULONG ulClrShade;

                  rclBack.xRight += 2;
                  rclBack.xLeft  += 2;

                  rclBack.yBottom -= 1;
                  rclBack.yTop    -= 1;

                  ulClrShade = ( ulClrText == (ULONG)CLR_BLACK) ? CLR_WHITE : CLR_BLACK;

                  GpiSetColor( hps, ulClrShade );

                  // Draw 3D button's "Descriptive Name" text (on-top)
                  WinDrawText( hps, pButton3d->NameText.ulLength, pszText, &rclBack, 0L, 0L,
                               ulAlign | DT_VCENTER | DT_TEXTATTRS | DT_MNEMONIC );

                  GpiSetColor( hps, ulClrText );

                } /* end logical block */

                // Draw 3D button's "Descriptive Name" text (on-top)
                WinDrawText( hps, pButton3d->NameText.ulLength, pszText, &rclText, 0L, 0L,
                             ulAlign | DT_VCENTER | DT_TEXTATTRS | DT_MNEMONIC );

                // Only draw Control Text when Button Style is correct (RIGHT)
                if( pButton3d->CtrlText.pszText && (pButton3d->ulStyle & BS3D_TEXT_RIGHT) )
                {
                  ULONG ulMaxNameWidth, ulMaxCtrlWidth;

                  CalcMaxTextWidth( hMenu3d, pButton3d->usLevel, &ulMaxNameWidth, 0 );
                  CalcMaxTextWidth( hMenu3d, pButton3d->usLevel, &ulMaxCtrlWidth, 1 );

                  rclText.xLeft = rclText.xLeft + ulMaxNameWidth + ( 3 * BUTTON_TEXT_LEFT_MARGIN );

                  // Draw 3D button's "Control Key" text to right of desc. name text
                  WinDrawText( hps, pButton3d->CtrlText.ulLength,
                               pButton3d->CtrlText.pszText, &rclText, 0L, 0L,
                               ulAlign | DT_VCENTER | DT_TEXTATTRS | DT_MNEMONIC );

                } /* end if */

              } /* end else */

            } /* end if */ // !berror

            // we have already done all initial calc's relative
            // to parent dialog
            pButton3d->ulState |= BUTTON_REALIZED;

          } /* end case */
          break;

        } /* end switch */

      } /* end if */ // entry found
      else
      {
        // Button not found therefore set error
        bError = TRUE;
      } /* end else */

    } /* end if */ // entries > 0

  } /* end if */


  return( !bError );

} /* end Draw3DButton */

