/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/*                                                                          */
/* PROGRAM NAME   : Font Test Print Queue Access Functions                  */
/* AUTHOR         : Matthew F. Rutkowski                                    */
/* FILENAME       : FTQUEUE.C                                               */
/* DATE WRITTEN   : 02-29-95                                                */
/* DESCRIPTION    : This module contains functions that help access         */
/*                  OS/2 Print Queue information.                           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* Modification/Change log                                                  */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/

#define INCL_DEV
#define INCL_DOSFILEMGR
#define INCL_DOSMEMMGR
#define INCL_DOSPROCESS
#define INCL_ERRORS
#define INCL_GPI
#define INCL_GPIERRORS
#define INCL_SPL
#define INCL_SPLDOSPRINT
#define INCL_WIN

#include <os2.h>
#include <stdlib.h>
#include <stdio.h>

#include "ft32.h"
#include "ftdlg.h"

USHORT FillQueNameListBox( HAB, HWND, PQUEUEDATA );
BOOL   GetDefaultQueueData( HAB, PQUEUEDATA, PPRINTERDATA );
APIRET GetQueueData( HAB, PQUEUEDATA, PPRINTERDATA, BOOL );


/****************************************************************************/
/* PROCEDURE NAME : ParseServerQueueName                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 05/06/95                                                */
/* DESCRIPTION    : */
/*                  */
/* PARAMETERS:      */
/*                  */
/* RETURN VALUES:   */
/*                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL ParseServerQueueName( PSZ pszServerQueue, PSZ pszServer, PSZ pszQueue )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Check to see if it's a network queue by '\\' character
  if( pszServerQueue[0] != '\\' )
  {
    // Not a network queue (only way to check for now)
    strcpy( pszQueue, pszServerQueue );
    pszServer[0] = '\0';

  } /* end if */
  else
  {
    // Parse out Server name from Queue name
    pszServer[0] = '\\';
    pszServer[1] = '\\';
    StrToken( pszServer+2, pszServerQueue, '\\', 3 );
    StrToken( pszQueue, pszServerQueue, '\\', 4 );

  } /* end else */

  return( TRUE );

} /* end ParseServerQueueName */

/****************************************************************************/
/* PROCEDURE NAME : GetDefaultQueueData                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 05/04/93                                                */
/* DESCRIPTION    : If the user never selects a queue using the "Select     */
/*                  Queue..." dialog this routine will be called to         */
/*                  search the OS2.INI file for the system default queue.   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Query OS2.INI for default queue information.        */
/*                                                                          */
/* PARAMETERS:      (HAB)  hab - anchor block handle for error processing   */
/*                                                                          */
/* RETURN VALUES:   (BOOL) ErrorOccurred == FALSE - Successful completion   */
/*                                       == TRUE  - Failure error occurred  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL GetDefaultQueueData( HAB hab, PQUEUEDATA pqd, PPRINTERDATA prd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  CHAR          szBuffer[MAXSTR];      // Buffer for profile string
  BOOL          ErrorOccurred = TRUE;  // Error assertion return code
  ULONG         ulReturn;              // Size to allocate for SplQueryQueue

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-------------------------------------------------*/
  /* QUEUE INFO                                      */
  /*-------------------------------------------------*/

  /* Get default printer name since none selected */
  ulReturn = PrfQueryProfileString( HINI_USER, "PM_SPOOLER", "PRINTER",
                                    ";", szBuffer, (ULONG)sizeof(szBuffer));

  ErrorOccurred = AssertPM( ulReturn == 0, hab, ERR_PRFQRYSTRING_PRINTER );

  // Able to query the printer entry
  if( !ErrorOccurred )
  {
    StrToken( prd->szPrinterName,  szBuffer, ';', 1 );

    /* Get default printer name since none selected */
    ulReturn = PrfQueryProfileString( HINI_USER, "PM_SPOOLER", "QUEUE",
                                      ";", szBuffer, (ULONG)sizeof(szBuffer));

    ErrorOccurred = AssertPM( ulReturn == 0, hab, ERR_PRFQRYSTRING_QUEUE );

    // Able to query the queue entry
    if( !ErrorOccurred )
    {

      StrToken( pqd->QueueName, szBuffer, ';', 1 );

      // Get Queue Description from Queue Name
      ulReturn = PrfQueryProfileString( HINI_SYSTEM, "PM_SPOOLER_QUEUE_DESCR", pqd->QueueName,
                                        ";", szBuffer, (ULONG)sizeof(szBuffer));

      // If we got a string back
      if( ulReturn > 0 )
      {
        // Copy it to our Queue Description
        StrToken( pqd->QueueDesc, szBuffer, ';', 1 );

      } /* end if */

    } /* end if */

  } /* end if */

  if( pqd->QueueName[0] == 0 )
    ErrorOccurred = TRUE;

  return( ErrorOccurred );

} /* GetDefaultQueueData */


/****************************************************************************/
/* PROCEDURE NAME : GetQueueData                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 12/20/92                                                */
/* DESCRIPTION    : Called whenever we are going to access a device on a    */
/*                  a queue (printing, queries etc) to retrieve all details */
/*                  about that queue (drivername, printername etc.).        */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) call GetDefaultQueueData(); if no queue selected    */
/*                  (b) Check for server information                        */
/*                  (c) call SplQueryQueue(); to get driver data from queue */
/*                                                                          */
/* PARAMETERS:      (HAB)  hab - anchor block handle for error processing   */
/*                                                                          */
/* RETURN VALUES:   (BOOL) ErrorOccurred == FALSE - Successful completion   */
/*                                       == TRUE  - Failure error occurred  */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
APIRET GetQueueData( HAB hab, PQUEUEDATA pqd, PPRINTERDATA prd, BOOL bResetDrivData )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  APIRET        apiret = ERROR_QUE_NOT_INITIALIZED;     // Error assertion return code
  PBYTE         pbuf;                  // pointer to printer queue info
  PPRQINFO3     prq;                   // pointer to PRQINFO3 struct
  CHAR          szDDL[MAXSTR];         // Buffer for driver name
  CHAR          szQL[MAXSTR];          // Buffer of Queue names
  CHAR          szServer[MAXSTR];      // Buffer for Server name
  ULONG         ulSize;                // Size to allocate for SplQueryQueue
  BOOL          bError = FALSE;
  SPLERR        SplErr = NO_ERROR;
  BOOL          bDefault = FALSE;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-------------------------------------------------*/
  /* Case 1: No Queue Name passed in                 */
  /*-------------------------------------------------*/
  if( pqd->QueueName[0] == 0 )
  {
    // No Queue name passed so get OS/2 default queue
    if( bError = GetDefaultQueueData( hab, pqd, prd ) )
    {
       // Return failure to find queue to caller
       return( apiret );

    } /* end if */

    bDefault = TRUE;

  } /* end if */

  /*-------------------------------------------------*/
  /* At this point we have a Queue name (or default) */
  /*-------------------------------------------------*/

  ParseServerQueueName( pqd->QueueName, szServer, szQL );

  // Find out how much memory we need to allocate for the passed in queue
  SplErr = SplQueryQueue( szServer, szQL, 3L, (PVOID)NULL, 0L, &ulSize );

  /*-------------------------------------------------*/
  /* Case 2: Queue Not Found (deleted)               */
  /*-------------------------------------------------*/
  if( SplErr == NERR_QNotFound )
  {
    // If we have not yet tried default print queue
    if( !bDefault )
    {
      // Get OS/2 system default queue
      bError = GetDefaultQueueData( hab, pqd, prd );

      // Able to get default queue name and description
      if( !bError )
      {
        ParseServerQueueName( pqd->QueueName, szServer, szQL );

        // Find out how much memory we need to allocate for the passed in queue
        SplErr = SplQueryQueue( szServer, szQL, 3L, (PVOID)NULL, 0L, &ulSize );

      } /* end if */
      else
      {
         // Return failure to find queue to caller
         return( apiret );

      } /* end else */

    } /* end if */
    else
    {
       // Return failure to find queue to caller
       return( apiret );

    } /* end else */

  } /* end if */

  // We expect these return codes (requests for larger buffer)
  if( ( SplErr == ERROR_MORE_DATA ) || ( SplErr == NERR_BufTooSmall ) )
  {
    // Allocate memory requested by spooler
    if( DosAllocMem( (PPVOID)&pbuf, ulSize, PAG_READ |
                     PAG_WRITE | PAG_COMMIT ) == NO_ERROR )
    {
       // Query Queue data for this print queue
       if( SplQueryQueue( szServer, szQL, 3L, (PBYTE)pbuf,
                          ulSize, &ulSize) == NO_ERROR )
       {
         // Format the data into PRQINFO3 structure
         prq = (PPRQINFO3)pbuf;

         // Get Queue Description for the queue name
         strcpy( pqd->QueueDesc, prq->pszComment );

         // Get default driver for queue
         strcpy( szDDL, prq->pszDriverName );
         StrToken( prd->szDriverName, szDDL, '.', 1 );
         StrToken( prd->szDeviceName, szDDL, '.', 2 );

         // Get all printers assigned to queue
         StrToken( prd->szPrinterName, prq->pszPrinters, ',', 1 );

         // if we have previously allocated a copy of Driver Data
         if( pqd->pDrivData )
         {
           // if Driver Data is a different size than what was there before OR
           // we were asked to reset the driver data to a known state
           if( (pqd->pDrivData->cb != prq->pDriverData->cb) || bResetDrivData )
           {
             // free old drivdata memory
             DosFreeMem( pqd->pDrivData );
             pqd->pDrivData = NULL;

           } /* end if */

         } /* end if */

         // If no DRIVDATA allocate and copy it from spooler queue info strucs
         if( pqd->pDrivData == NULL )
         {
           DosAllocMem( (PPVOID)&(pqd->pDrivData),
                         prq->pDriverData->cb,
                         PAG_READ | PAG_WRITE | PAG_COMMIT );

           memcpy( pqd->pDrivData, prq->pDriverData, prq->pDriverData->cb );
         } /* end if */

         // At this point we have valid queue name, description and job props
         apiret = NO_ERROR;

       } /* end if */
       else
       {
         bError = AssertPM( TRUE, hab, ERR_SPLQUERYQUEUE );

       } /* end else */

       // Free up memory from PRQINFO3 struct
       DosFreeMem( pbuf );

    } /* end if */
    else
    {
      bError = AssertPM( TRUE, hab, ERR_DOSALLOCMEM );

    } /* end else */

  } /* end if */
  else
  {
    bError = AssertPM( TRUE, hab, ERR_SPLQUERYQUEUE );

  } /* end else */

  return( apiret );

} /* GetQueueData */


/****************************************************************************/
/* PROCEDURE NAME : FillQueNameListBox                                      */
/* AUTHOR         : Matt R.                                                 */
/* DATE WRITTEN   : 5/6/92                                                  */
/* DESCRIPTION    : Fills a listbox in the Queue dialog with all available  */
/*                  queues found in the system (LOCAL and REMOTE).          */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) call SplEnumQueue(); to enumerate all queues        */
/*                      visible from this workstation.                      */
/*                                                                          */
/*                  (b) Display each queue's comment (desktop name) and     */
/*                      internal queue name in a listbox.                   */
/*                                                                          */
/*                  (c) loop through all queue names and highlite the       */
/*                      one the user selected previously or the system      */
/*                      default queue.                                      */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:      (HAB)    hab  - caller's anchor block                   */
/*                  (HWND)   hwnd - caller's window handle                  */
/*                                                                          */
/* RETURN VALUES:   (USHORT) ulQueuesReturned - Number of queues found      */
/*                                                                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* NOTE: change return value to ULONG                                       */
/*                                                                          */
/****************************************************************************/
USHORT FillQueNameListBox( HAB hab, HWND hwnd, PQUEUEDATA pqd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PBYTE     pbPrqInfoBuf;              // Queue Info buffers (Level 3)
  BOOL      bSavedQueueFound = FALSE;  // Internal queue name match found
  CHAR      pszListboxQueName[MAXSTR]; // Listbox Queue Name buffer
  PPRQINFO3 pPrqInfo3;                 // Pointer to Lvl 3 Queue Info structs
  SPLERR    rc;                        // Spooler API Return Codes
  ULONG     ulIdx;                     // Loop control for current queue
  ULONG     ulBufSize;                 // byte size of pszPrqInfoBuf
  ULONG     ulQueuesReturned = 0;      // # Qs returned by DosPrintQEnum
  ULONG     ulQueuesTotal    = 0;      // # Qs available by DosPrintQEnum
  ULONG     ulNeeded         = 0;      // Memory needed for Queue info
  LONG      lSavedQueueIdx;            // Listbox index of saved queue name
  LONG      lDefaultQueueIdx;          // Listbox index of default queue name
  LONG      lIdxRet;                   // Current Listbox insertion index

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Clear out the listbox from any previous draw
  WinSendDlgItemMsg( hwnd, IDCB_PRINT_QUEUES, LM_DELETEALL, NULL, NULL );

  // Find out how much to allocate for Queue Info (at Level 3)
  rc = SplEnumQueue( (PSZ)NULL, 3L, pbPrqInfoBuf, 0L, &ulQueuesReturned,
                     &ulQueuesTotal, &ulNeeded, NULL );

  // If at least one OS/2 Print queue has been defined
  if( ulQueuesTotal > 0 )
  {
    // Allocate memory to hold Queue Info structures
    if( DosAllocMem( (PPVOID)(&pbPrqInfoBuf), ulNeeded,
                     PAG_READ | PAG_WRITE | PAG_COMMIT ) == NO_ERROR )
    {
      // We have allocated what we needed so set that as our buffer size
      ulBufSize = ulNeeded;

      /*----------------------------------------------------------*/
      /* Enumerate all local or aliases of network queues (Lvl 3) */
      /*----------------------------------------------------------*/
      if( rc = SplEnumQueue( (PSZ)NULL, 3L, pbPrqInfoBuf, ulBufSize,
                              &ulQueuesReturned, &ulQueuesTotal,
                              &ulNeeded, (PVOID)NULL ) == NO_ERROR )
      {
        /*--------------------------------------------------------*/
        /* Display all Queue names found to listbox               */
        /*--------------------------------------------------------*/
        if ( ulQueuesReturned )
        {
          HWND hwndLB = WinWindowFromID( hwnd, IDCB_PRINT_QUEUES );

          /* Get pointer to begining of first PRQINFO3 struct */
          pPrqInfo3 = (PPRQINFO3)pbPrqInfoBuf;

          /* while there are still queues in the PrqInfoBuf */
          for( ulIdx = 0; ulIdx < ulQueuesReturned; ulIdx++ )
          {
            /*----------------------------------------------------*/
            /* Format queue name displayed in listbox             */
            /* First show Queue Description (as seen on desktop)  */
            /* then follow that with actual internal queue name   */
            /*----------------------------------------------------*/
            sprintf( pszListboxQueName, "%s %c%s%c",
                     pPrqInfo3->pszComment,
                     LEAD_QUEUE_TOKEN,
                     pPrqInfo3->pszName,
                     TRAIL_QUEUE_TOKEN );

            // Insert Hybrid Queue name into end of listbox
            lIdxRet = WinInsertLboxItem( hwndLB, LIT_END, pszListboxQueName );

            if(!bSavedQueueFound)
            {
              if( strcmp( pqd->QueueName, pPrqInfo3->pszName ) == 0)
              {
                // Save Listbox Index where queue name matched saved name
                lSavedQueueIdx = lIdxRet;

                // Indicate match found
                bSavedQueueFound = TRUE;

              } /* end if */
            } /* end if */

            // Look to see if OS/2 System default queue found
            if( pPrqInfo3->fsType & PRQ3_TYPE_APPDEFAULT )
               lDefaultQueueIdx = lIdxRet;

            /* increment to next queue info (PRQINFO3) struct */
            pPrqInfo3++;

          } /* end for */

          // Set Highlite index to saved or default queue name
          if(bSavedQueueFound)
            ulIdx = lSavedQueueIdx;
          else
            ulIdx = lDefaultQueueIdx;

          // Highlite saved (if found) or default queue name
          WinSendDlgItemMsg( hwnd,
                             IDCB_PRINT_QUEUES,
                             LM_SELECTITEM,
                             MPFROMLONG(ulIdx),
                             MPFROMLONG(TRUE));
        } /* end if */

      } /* end if */
      else
      {
          AssertPM( TRUE, hab, ERR_SPLENUMQUEUE );

      } /* end else */

      // Free up the queue info buffer
      DosFreeMem( pbPrqInfoBuf );

    } /* end if */
    else
    {
      // Failed to allocate memory display error
      AssertPM( TRUE, hab, ERR_DOSALLOCMEM );

    } /* end else */

  } /* end if */
  else
  {
    // Show user no queue defined "None Available"
    lIdxRet = WinInsertLboxItem( WinWindowFromID( hwnd, IDCB_PRINT_QUEUES ),
                                 LIT_END, NONE_AVAILABLE );

    // Highlite saved (if found) or default queue name
    WinSendDlgItemMsg( hwnd,
                       IDCB_PRINT_QUEUES,
                       LM_SELECTITEM,
                       MPFROMLONG(lIdxRet),
                       MPFROMLONG(TRUE));

    return( ulQueuesTotal );

  } /* end else */

  return( ulQueuesReturned );

} /* end FillQueNameListBox */

