/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/****************************************************************************/
/*                                                                          */
/*      DISCLAIMER OF WARRANTIES.  The following [enclosed] code is         */
/*      sample code created by IBM Corporation. This sample code is not     */
/*      part of any standard or IBM product and is provided to you solely   */
/*      for  the purpose of assisting you in the development of your        */
/*      presentation drivers.  The code is provided "AS IS", without        */
/*      warranty of any kind.  IBM shall not be liable for any damages      */
/*      arising out of your use of the sample code, even if they have been  */
/*      advised of the possibility of such damages.                         */
/*                                                                          */
/****************************************************************************/
/****************************************************************************/
/* PROGRAM NAME   : Font Tester                                             */
/* AUTHOR         : Matthew F. Rutkowski                                    */
/* FILENAME       : FT32.C  (new 32-bit version)                            */
/* DATE WRITTEN   : 4-04-92                                                 */
/* DESCRIPTION    : Font Tester is used to browse and print files on local  */
/*                  or remote printers using any public font.               */
/*                  Also allows the user to view various capabilities of    */
/*                  device attached to the selected queue including font    */
/*                  and hardcopy information.                               */
/*                  Provides WYSIWIG screen to printer for system fonts     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* TAG - DATE - [COMPANY] - AUTHOR - DESCRIPTION                            */
/*--------------------------------------------------------------------------*/
/* @DBCS    -          - DBCS enabling Tag                                  */
/*                                                                          */
/* @MAXLINE - 11/02/93 - Matt Rutkowski [IBM] - Allows browse and print of  */
/*                       huge files where previously files upto 6000 lines  */
/*                       were only allowed.                                 */
/*                                                                          */
/*                       Added 2 new special keys:                          */
/*                       CTRL-HOME     - Browse top of file                 */
/*                       CTRL-END      - Browse bottom of file              */
/*                                                                          */
/* @74652   - 11/02/93 - Matt Rutkowski [IBM] - Allows any number of        */
/*                       characters for document name on DevEscape(         */
/*                       STARTDOC);  previously told driver to use 8 chars  */
/*                       and Epson would truncate document name.            */
/*                                                                          */
/* @74644   - 11/03/93 - Matt Rutkowski [IBM] - New text format option      */
/*                       "Print Line Numbers" added as a suggestion that    */
/*                       was logged as a defect.                            */
/*                                                                          */
/* @75474   - 11/03/93 - Matt Rutkowski [IBM] - fixed problem with search   */
/*                       on huge files (used to incorrectly use SHORTS)     */
/*                                                                          */
/* @TTTTT   - 05/12/94 - Matt Rutkowski [IBM] - Add "Title text" option     */
/*                                                                          */
/* @100698  - 09/28/94 - Matt Rutkowski [IBM] - recalculate char box size   */
/*                       when user selects a new printer resolution.        */
/*                                                                          */
/*--personal to-do list-----------------------------------------------------*/
/*                                                                          */
/* check bitmap fonts for matching resolution.                              */
/*   - perhaps a "suppress" checkbox to disallow display fonts              */
/* check fonts for codepage that matches a system codepage.                 */
/* add margin print option                                                  */
/* show page break/boundaries in client window                              */
/* semaphore global structs                                                 */
/* allow multiple clients/files to be browsed                               */
/* enable  drag/drop                                                        */
/* enable printing of .BMP and .MET                                         */
/* add clipboard support                                                    */
/* Remove float stuff from charbox calcs. use longs to make fixed values    */
/* print preview - in progress for next release                             */
/* add AssertPM header messages to stringtable                              */
/* add context sensitive help (static frame class below client)             */
/* Add "Wrap" option to text search                                         */
/* Add "Size and Position" save option to "Program Settings"                */
/*                                                                          */
/****************************************************************************/

#define INCL_DEV
#define INCL_DOSMISC
#define INCL_DOSPROCESS
#define INCL_ERRORS
#define INCL_GPI
#define INCL_GPIERRORS
#define INCL_SPL
#define INCL_SPLDOSPRINT
#define INCL_WIN
#define INCL_DOSNLS  // @DBCS

#include <os2.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "menu.h"
#include "ft32.h"
#include "ftdlg.h"
#include "fthelp.h"

#define  ENVVARNAME   "PATH"       // Environment variable name
#define  STRNICMP     TRUE

/*------------------------------------------------------------------------*/
/* Function Prototypes (Window Procedures)                                */
/*------------------------------------------------------------------------*/
FNWP AboutDlgProc;
FNWP ClientWndProc;
FNWP ColorsDlgProc;
FNWP FontDlgProc;
FNWP FontWidthDlgProc;
FNWP FontKernDlgProc;
FNWP ProgramSettingsDlgProc;
FNWP ConfirmDlgProc;
FNWP MetricsDlgProc;
FNWP OpenFileDlgProc;
FNWP QueueDlgProc;
FNWP SampleFormatWndProc;
FNWP SearchDialogProc;
FNWP FormatDlgProc;
FNWP UnlistedQueueDialogProc;

/*------------------------------------------------------------------------*/
/* Function Prototypes (Internal)                                         */
/*------------------------------------------------------------------------*/
ERRORID  AccessDevice( PPGMDATA, HAB, LONG, ULONG, ULONG );
VOID     BuildTrailerString( HAB, PSZ, PSZ, PSZ, PSZ, PULONG );
VOID     CalculateCharBox( ULONG, LONG, LONG, PSIZEF );
VOID     CreateThread ( PTID, PFNTHREAD );
VOID     DrawTextBorder( HAB, HPS, POINTL, BOOL, LONG, LONG, LONG );
VOID     DrawTitleString( HAB, HPS, PGMDATA, PPOINTL, PFONTMETRICS );
VOID     DrawTrailerString( HAB, HPS, POINTL, PFONTMETRICS, PULONG, PSZ, PSZ, PSZ, PSZ );
VOID     InitDisplayData( PPGMDATA );
VOID     InitPgmData( PPGMDATA );
VOID     InitializeMenu( ULONG );
BOOL     MapColorToRowColumn( LONG, PUSHORT, PUSHORT );
LONG     MapRowColumnToColor( USHORT usRow, USHORT usColumn );
VOID     PrintCharacterBoxes( HAB, HPS, USHORT, PSZ );
BOOL     PrintTextFile( HAB, HDC, HPS, PGMDATA, LONG, LONG, PFONTMETRICS );
VOID FAR PrintThread( VOID );
BOOL     ReadTextFormatOptions( PPGMDATA, PFORMATDATA, HWND );
BOOL     RetrieveCmdLineParms( USHORT count, CHAR *Parms[] );
VOID     ShowTextFormatOptions( PPGMDATA, PFORMATDATA, HWND );
ULONG    SearchForString ( PSZ, ULONG, PSZ, PSZ );
VOID FAR SearchThread ( VOID );
VOID     SetScrollBars( HWND, HWND, HWND, PLONG, LONG, LONG );

/*------------------------------------------------------------------------*/
/* Function Prototypes (External)                                         */
/*------------------------------------------------------------------------*/
extern VOID    DeleteIniFileData( HWND, ULONG );
extern ERRORID DisplayDeviceFormCaps( HAB, HDC, HWND, PPRINTERDATA, ULONG );
extern ERRORID DisplayDeviceCaps( HAB, HDC, HWND, PPRINTERDATA, ULONG );
extern VOID    FillCapsListBox( HWND, PLONG );
extern BOOL    FillFaceNameListBox( HWND, PPGMDATA, PFONTDATA );
extern VOID    FillFontMetricsListBox( HWND, ULONG, PFONTMETRICS, SHORT *, SHORT * );
extern USHORT  FillFormsListBox( HAB, HWND, LONG );
extern VOID    FillHCCapsListBox( HWND, USHORT );
extern USHORT  FillQueNameListBox( HAB, HWND, PQUEUEDATA );
extern VOID    PaintClientWindow( HPS, LONG, HWND, PRECTL );
extern ERRORID AssertPM( BOOL, HAB, PSZ );
extern BOOL    ReadInTextFile ( HAB, USHORT, ULONG, BOOL );
extern VOID    ReadIniFileData( PPGMDATA, HWND );
extern VOID    ReplaceTabs( PSZ, PSZ, PULONG, CHAR, ULONG );
extern BOOL    StrToken( PSZ, PSZ, CHAR, USHORT );
extern VOID    WriteIniFileData( PPGMDATA, ULONG );

/*------------------------------------------------------------------------*/
/* GLOBALS                                                                */
/*------------------------------------------------------------------------*/
ULONG         cxChar, cyChar, cyDesc;  // Current font's charbox values
FILEDLG       filedlg;                 // Info for std file dlg
HDC           hdc;                     // Device Context handles
HPS           hps;                     // Presentation space handles
HWND          hwndClient;              // Window handle of client window
HWND          hwndTitlebar;            // Window handle of titlebar window
HWND          hwndMenu;                // Window handle of menu window
HWND          hwndHelp;                // Window handle of Help Instance
LONG          lHscrollPos = 0;         // Horizontal scrollbar position
LONG          lVscrollPos = 0;         // Vertical scrollbar position
ULONG         ulTotalLines;            // # of lines in the file
PSZ *         pLinePtrs;               // @MAXLINE: static array ->dynamic ptrs
PSZ           pFileBuff;               // holds pointer to input file
CHAR          szSearchString[MAXSTR];  // Holds Text search string
CHAR          szTitleText[MAXSTR];     // Holds current TitleBar text
PGMDATA       PgmData;                 // Structure of current program data
HMENU3D       hMenu3d;                 // Handle to 3-D Menu
ULONG         CodePage;                // @DBCS - Current code page
CHAR          DbcsEnv[MAXSTR];         // @DBCS - DBCS enviroment table
LONG          cxClient;                // Client window width
LONG          cyClient;                // Client window height
HFILE         hInFile;                 // file handle
HWND          hwndHscroll;             // Horz scrollbar window handle
HWND          hwndVscroll;             // Vert scrollbar window handle

// 3-D Button handles for Menus (Program, File, Query, Search and Help)
HBUTTON3D  hBtnFile, hBtnQuery, hBtnHelp, hBtnSearch;
HBUTTON3D  hBtnOpenFile, hBtnSettings, hBtnDefault, hBtnPrint;
HBUTTON3D  hBtnQCaps, hBtnQHCCaps, hBtnQFont, hBtnQWidth, hBtnQKern;
HBUTTON3D  hBtnSrchText;
HBUTTON3D  hBtnHelpIdx, hBtnHelpKeys, hBtnHelpHelp, hBtnHelpExt, hBtnAbout;
HBUTTON3D  hBtnFntSys, hBtnFntPrn, hBtnFntDsp, hBtnFntATM, hBtnFntBmp;

// 3-D Button handles for "Program Settings" Notebook tabs
HBUTTON3D  hBtnSelectQueue, hBtnSelectFont, hBtnSelectColors, hBtnSelectFormat;

// 3-D buttons used in program dialogs
HBUTTON3D  hBtnOther, hBtnReview, hBtnJobProp;

// Define a color table with the 16 defined OS/2 colors for "Select Colors"
LONG ColorTable[MAX_COLORS] =
{
 CLR_BLACK,
 CLR_WHITE,
 CLR_BLUE,
 CLR_RED,
 CLR_PINK,
 CLR_GREEN,
 CLR_CYAN,
 CLR_YELLOW,
 CLR_DARKGRAY,
 CLR_DARKBLUE,
 CLR_DARKRED,
 CLR_DARKPINK,
 CLR_DARKGREEN,
 CLR_DARKCYAN,
 CLR_BROWN,
 CLR_PALEGRAY,
};


/****************************************************************************/
/* PROCEDURE NAME : MAIN                                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Main 'C' source code module                             */
/*                  Program control starts at this function when Font Test  */
/*                  is executed.                                            */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Validate command line arguments                     */
/*                  (b) Recognize DBCS or SBCS environment                  */
/*                  (c) Initialize Help instance                            */
/*                  (d) Create standard PM window and begin main message    */
/*                      processing loop.                                    */
/*                                                                          */
/* PARAMETERS:      argc - number of command line arguments                 */
/*                  argv - command line arguments                           */
/*                                                                          */
/* RETURN VALUES:   Always returns the numeric constant 0 (zero).           */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
int main ( USHORT argc, CHAR *argv[] )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  static CHAR    szClientClass[] = PROGRAM_TITLE_CNST;

  static ULONG   flFrameFlags = FCF_ICON       | FCF_TITLEBAR      |
                                FCF_SYSMENU    | FCF_SIZEBORDER    |
                                FCF_MINMAX     | FCF_SHELLPOSITION |
                                FCF_TASKLIST   | FCF_MENU          |
                                FCF_VERTSCROLL | FCF_HORZSCROLL    |
                                FCF_ACCELTABLE;

  BOOL           bSuccess = TRUE;
  HELPINIT       helpinit;              // Help instance data structure
  HWND           hwndFrame;             // Window handle of frame window
  BYTE           ResultBuffer[512];     // Path search result
  ULONG          ulStyle = WS_VISIBLE;  // Make ft32 window visible or not

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  InitPgmData( &PgmData );

  PgmData.habPgm = WinInitialize( 0 );
  PgmData.hmqPgm = WinCreateMsgQueue( PgmData.habPgm, 0 );

  // If user passed cmd line arguments to the program
  if ( argc > 1 )
  {
    // retrieve and validate the parameters passed on cmd line
    bSuccess = RetrieveCmdLineParms( argc, argv );

    // user wants to print only (do not bring up PM application window)
    if( PgmData.StateData.bCmdLinePrint )
    {
      ulStyle = 0L;

    } /* end if */

  } /* end if */

  // If parms retrieved and validation passed or no parms
  if( bSuccess )
  {
    /*------------------------------------------------------------------*/
    /* @DBCS - Get process codepage and create a DBCS environment table */
    /*------------------------------------------------------------------*/
    {
      ULONG   len;
      COUNTRYCODE cc;
      UCHAR   buf[12], *pbuf = buf;

      cc.country  = 0;
      DosQueryCp(sizeof(ULONG), &cc.codepage, &len);
      CodePage = cc.codepage;

      memset( DbcsEnv, 1, MAXSTR );

      if(!DosQueryDBCSEnv(sizeof(buf), (PCOUNTRYCODE)&cc, buf)
         && *(PUSHORT)pbuf)
      {
        while(*(PUSHORT)pbuf)
        {
           memset( &(DbcsEnv[*pbuf]), SIZEDBCS, pbuf[1] - pbuf[0] + 1);
           pbuf += 2;

        } /* end while */

      } /* end if */

    } /* end @DBCS */

    // Register our program client class and reserve 4 bytes
    // of data for our window pointer
    WinRegisterClass( PgmData.habPgm, (PSZ)szClientClass, (PFNWP)ClientWndProc,
                      CS_SIZEREDRAW, 4L );

    // Create our standard PM program window
    hwndFrame = WinCreateStdWindow( HWND_DESKTOP, ulStyle,
                                    &flFrameFlags, (PSZ)szClientClass,
                                    (PSZ)NULL, 0L, (HMODULE)NULL,
                                    ID_FONTEST, &hwndClient );

    /*----------------------------------------------*/
    /* If Frame Window created                      */
    /*----------------------------------------------*/
    if( hwndFrame )
    {
      QMSG   qmsg;  // queue message

      /*--------------------------------------------*/
      /* Get Titlebar window handle                 */
      /*--------------------------------------------*/
      hwndTitlebar = WinWindowFromID( hwndFrame, FID_TITLEBAR );

      if( !PgmData.StateData.bCmdLinePrint )
      {
        // Subclass any windows for special processing
        //pfnDefMenuProc = WinSubclassWindow( WinWindowFromID( hwndFrame, FID_MENU ), MenuSubclassProc ); // @NEW

        /*--------------------------------------------*/
        /* Setup IPF help instance                    */
        /*--------------------------------------------*/
        helpinit.cb                 = sizeof( HELPINIT );
        helpinit.ulReturnCode       = 0L;
        helpinit.pszTutorialName    = (PSZ)NULL;
        helpinit.phtHelpTable       = (PHELPTABLE)MAKEULONG( ID_FONTEST, 0xffff );
        helpinit.hmodHelpTableModule      = NULLHANDLE;
        helpinit.hmodAccelActionBarModule = NULLHANDLE;
        helpinit.idAccelTable       = 0L;
        helpinit.idActionBar        = 0L;
        helpinit.pszHelpWindowTitle = (PSZ)PROGRAM_TITLE_CNST;
        helpinit.fShowPanelId       = SHOW_HELP_PANEL_IDS;
        helpinit.pszHelpLibraryName = (PSZ)PROGRAM_HELP_FILE;

        // Search current directory + PATH variable for .HLP
        if(!DosSearchPath( SEARCH_ENVIRONMENT,
                           (PSZ)ENVVARNAME,
                           (PSZ)PROGRAM_HELP_FILE,
                           (PBYTE)ResultBuffer,
                           (ULONG)sizeof(ResultBuffer)))
        {

          helpinit.pszHelpLibraryName = (PSZ)ResultBuffer;

        } /* end if */

        // last try to create help
        hwndHelp = WinCreateHelpInstance( PgmData.habPgm, &helpinit );

      } /* end if */

      /*--------------------------------------------*/
      /* If help instance created                   */
      /*--------------------------------------------*/
      if( hwndHelp )
      {
        if( WinAssociateHelpInstance( hwndHelp, hwndFrame ) )
        {
           // Mark state variable that we found .HLP file
           // and was able to create/associate help instance
           PgmData.StateData.bHelpAvailable = TRUE;

        } /* end if */

      } /* end if */

      /*--------------------------------------------*/
      /* Main message polling loop                  */
      /*--------------------------------------------*/
      while( WinGetMsg( PgmData.habPgm, &qmsg, (HWND)NULL, 0L, 0L ) )
         WinDispatchMsg( PgmData.habPgm, &qmsg );

      /*--------------------------------------------*/
      /* Destroy the Help Instance                  */
      /*--------------------------------------------*/
      WinDestroyHelpInstance( hwndHelp );

      /*--------------------------------------------*/
      /* Destroy the Frame window                   */
      /*--------------------------------------------*/
      WinDestroyWindow(hwndFrame);

    } /* end if */

  } /* end if */

  /*----------------------------------------------*/
  /* Destroy the Message Queue                    */
  /*----------------------------------------------*/
  WinDestroyMsgQueue(PgmData.hmqPgm);

  /*----------------------------------------------*/
  /* Destroy the Anchor Block handle              */
  /*----------------------------------------------*/
  WinTerminate(PgmData.habPgm);

  return(0);

} /* end main */



/****************************************************************************/
/* PROCEDURE NAME : ClientWndProc                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Handles all messages specific to the application's      */
/*                  client window.                                          */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY ClientWndProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ERRORID        rc;
  ULONG          action;                // used by DosOpen
  SIZEF          DisplayCharBox;        // Display's current character box
  FILEFINDBUF    FileInfoBuf;           // Gets info about file being opened
  USHORT         FileInfoBufSize;       // Size of FILEFINDBUF
  double         fInt;                  // holds integer part of division
  double         fFract;                // holds fractional part of division
  double         fNewHeight;            // holds new charbox height
  double         fNewWidth;             // holds new charbox width
  LONG           lFontMatch;            // Return code from GpiCreateLogFont
  BOOL           PostCompleted = FALSE; // Indicates if posted message received
  RECTL          rclInvalid;            // Invalid client window rectangle
  LONG           lHscrollInc;           // # to increment Horz Scrollbar
  LONG           lVscrollInc;           // # to increment Vert Scrollbar
  static LONG    lVscrollMax = 0;       // Maximum Vert scroll position
  USHORT         status;                // Holds error status return codes
  BOOL           Success;               // file read in successfully?
  USHORT         us;                    // USHORT variable used in FOR loops
  USHORT         usResult;              // holds USHORT return codes
  PPGMDATA       ppd;                   // Pointer to Program Data
  PSTATEDATA     pState;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_CLOSE: WinAlarm( HWND_DESKTOP, 4 );
    break;

    case WM_CREATE:
    {
      // Store our program specific data for this window
      WinSetWindowPtr( hwnd, 0L, &PgmData );

      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );
      pState   = &ppd->StateData;

      // Retrieve program's INI data
      ReadIniFileData( ppd, hwnd );

      // If command line print
      if( pState->bCmdLinePrint && !pState->bCmdLineBrowse )
      {
        // open the file without invoking PM interface
        WinSendMsg ( hwnd, WM_COMMAND,
                     MPFROMSHORT( IDM_OPEN_QUIET ),
                     MPVOID );

        // print the file without PM interface
        WinSendMsg ( hwnd, WM_COMMAND,
                     MPFROMSHORT( IDM_PRINT ),
                     MPVOID );
      } /* end if */
      else
      {
        SIZEL sizel = {0,0}; // Init to 0 to get default PS page size
        ULONG ulStyle = BS3D_TEXT_RIGHT | BS3D_BORDER_3D;

        // Cause a susyem "open" noise to occur when client is created
        WinAlarm( HWND_DESKTOP, 3 );

        // Initialize any values dependent on current display
        InitDisplayData( ppd );

        // If we have a high-resolution (SVGA+) display driver
        if( ppd->DisplayData.fHorzDPI > DPI_96 )
        {
          // use 3-D text effect on higher quality moniters
          ulStyle |= BS3D_TEXT_3D;

        } /* end else */

        // Get the window handle of the menu window
        hwndMenu = WinWindowFromID( WinQueryWindow(hwnd,QW_PARENT), FID_MENU );

        // create a 3-D menu to replace boring default one
        Create3DMenu( ppd->habPgm, &hMenu3d, hwndMenu, NULLHANDLE );

        // Create and add all 3D button we will use
        InitializeMenu( ulStyle );

        // Get a display DC for client window
        hdc = WinOpenWindowDC ( hwnd );

        // Create PS to draw to
        hps = GpiCreatePS ( ppd->habPgm, hdc, &sizel, PU_PELS | GPIA_ASSOC );

        /*----------------------------*/
        /* Get the scrollbar handles  */
        /*----------------------------*/
        hwndHscroll = WinWindowFromID( WinQueryWindow(hwnd, QW_PARENT),
                                       FID_HORZSCROLL );

        hwndVscroll = WinWindowFromID( WinQueryWindow(hwnd, QW_PARENT),
                                       FID_VERTSCROLL );

        // Set up client window's WYSIWYG font that was saved
        // This also initializes scrollbars
        WinSendMsg( hwnd, WM_CHANGE_FONT, MPVOID, MPVOID );

        // Fill in creation values for OS/2 common File Dialog
        filedlg.cbSize      = sizeof( FILEDLG );
        filedlg.fl          = FDS_OPEN_DIALOG | FDS_CENTER | FDS_HELPBUTTON;
        filedlg.pszTitle    = (PSZ)PROGRAM_TITLE_CNST;
        filedlg.pszOKButton = (PSZ)OPEN_TITLE_CNST;
        filedlg.pfnDlgProc  = OpenFileDlgProc;

      } /* end else */

      if( pState->bCmdLineBrowse )
      {
        // open the file without invoking PM interface
        WinSendMsg ( hwnd, WM_COMMAND,
                     MPFROMSHORT( IDM_OPEN_QUIET ),
                     MPVOID );

      } /* end if */

    } /* end WM_CREATE */
    break;

    case HM_QUERY_KEYS_HELP:
         return( (MRESULT)PANEL_HELP_KEYS );
    break;

    // @75474
    case WM_SET_CURRENT_LINE:
    {
      lVscrollPos = LONGFROMMP( mp1 );

      // Set both Horz and Vert scrollbars to new line
      SetScrollBars( hwnd, hwndHscroll, hwndVscroll,
                     &lVscrollMax, cxClient, cyClient );

      return(0);

    } /* end case */
    break;

    case WM_CHANGE_FONT:
    {
      PFONTMETRICS pfm;

      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );
      pfm = &(ppd->FontData.fm);

      // Clear out previous font with dummy LCID_LOCAL so we can reuse
      // same LCID_WYSIWIG
      GpiSetCharSet( hps, LCID_LOCAL );
      GpiDeleteSetId( hps, LCID_WYSIWYG );

      // If we have a system font then we can have WYSIWIG b/w
      // text shown in client window and on printer
      if( pfm->lMatch > 0 )
      {
        FATTRS fattrsTmp;

        /*--------------------------------------------------------*/
        /* Fill in a default set of FATTRS for display            */
        /*--------------------------------------------------------*/
        fattrsTmp.usRecordLength  = sizeof(FATTRS);
        for( us=0;fattrsTmp.szFacename[us]=pfm->szFacename[us];++us );
        fattrsTmp.fsSelection     = pfm->fsSelection | ppd->FontData.Options;
        fattrsTmp.lMatch          = pfm->lMatch;
        fattrsTmp.lMaxBaselineExt = pfm->lMaxBaselineExt;
        fattrsTmp.lAveCharWidth   = pfm->lAveCharWidth;

        // @DBCS - set codepage to system codepage so font realized
        // with correct codepage (not font's default codepage
        // (i.e. not fm.usCodepage )
        fattrsTmp.usCodePage      = CodePage;
        fattrsTmp.fsType          = 0;
        fattrsTmp.fsFontUse       = FATTR_FONTUSE_NOMIX &
                                    FATTR_FONTUSE_OUTLINE;

        lFontMatch = GpiCreateLogFont( hps, NULL, LCID_WYSIWYG,
                                       &fattrsTmp );
      } /* end if */
      else
      {
        // The font selected by the user is NOT a system font
        // but a device font.  Therefore simulate using a
        // Courier font (also set correvt fsSelection flags)
        ppd->DisplayData.fattrsDefault.fsSelection =
             pfm->fsSelection | ppd->FontData.Options;

        lFontMatch = GpiCreateLogFont( hps, NULL, LCID_WYSIWYG,
                                       &(ppd->DisplayData.fattrsDefault) );
      } /* end else */

      // Verify GpiCreateLogFont was successful (i.e. we have valid font
      if( AssertPM( lFontMatch==GPI_ERROR, ppd->habPgm, ERR_GPICREATELOGFONT ) == NO_ERROR )
      {
        // Make sure the font created was not the system "default" font
        AssertPM( lFontMatch==FONT_DEFAULT, ppd->habPgm, ERR_TYPE_FONTMATCH );

        // calculate the character box of the newly created font
        CalculateCharBox( ppd->FontData.PointSize,
                          ppd->DisplayData.lHorzFontRes,
                          ppd->DisplayData.lVertFontRes,
                          &DisplayCharBox );

        // Put the new box in effect for the printer pres space
        if( GpiSetCharBox( hps, &DisplayCharBox ) == GPI_ERROR )
           AssertPM( TRUE, ppd->habPgm, ERR_GPISETCHARBOX );

        if( GpiSetCharSet( hps, LCID_WYSIWYG ) == GPI_ERROR )
           AssertPM( TRUE, ppd->habPgm, ERR_GPISETCHARSET );

        if( GpiQueryFontMetrics( hps, sizeof(FONTMETRICS), &(ppd->DisplayData.fm) ) == GPI_ERROR )
           AssertPM( TRUE, ppd->habPgm, ERR_GPIQFONTMETRICS );

        // make sure average character width is not 0
        if( ppd->DisplayData.fm.lAveCharWidth != 0L )
          cxChar = (ULONG) ppd->DisplayData.fm.lAveCharWidth;
        else
          cxChar = (ULONG) ppd->DisplayData.fm.lEmInc;

        cyChar = (ULONG) ppd->DisplayData.fm.lMaxBaselineExt;
        cyDesc = (ULONG) ppd->DisplayData.fm.lMaxDescender;

        // Set both Horz and Vert scrollbars to new size/pos
        SetScrollBars( hwnd, hwndHscroll, hwndVscroll,
                       &lVscrollMax, cxClient, cyClient );

      } /* end if */

      return(0);

    } /* end case */
    break;

    case WM_CHAR:
    {
      USHORT   flKeyFlags  = SHORT1FROMMP( mp1 );
      CHAR     ch          = CHAR1FROMMP( mp2 );
      BOOL     bKeyHandled = FALSE;

      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

      // ignore all messages from the release of a key
      if ( flKeyFlags & KC_KEYUP || ch == '\0' )
          return( (MRESULT) bKeyHandled);

      // check for virtual keys
      if ( flKeyFlags & KC_VIRTUALKEY )
      {
        // If the key matches one we handle tell ourselves to
        // perform the appropriate corresponding action
        switch ( SHORT2FROMMP ( mp2 ) )
        {
          case VK_PAGEUP:
          {
             WinSendMsg ( hwnd, WM_VSCROLL,
                          MPFROMSHORT(FID_VERTSCROLL),
                          MPFROM2SHORT(0,SB_PAGEUP));
             bKeyHandled = TRUE;
          } /* end case */
          break;

          case VK_PAGEDOWN:
          {
             WinSendMsg ( hwnd, WM_VSCROLL,
                          MPFROMSHORT(FID_VERTSCROLL),
                          MPFROM2SHORT(0,SB_PAGEDOWN));
             bKeyHandled = TRUE;
          } /* end case */
          break;

          case VK_LEFT:
          {
             WinSendMsg ( hwnd, WM_HSCROLL,
                          MPFROMSHORT(FID_HORZSCROLL),
                          MPFROM2SHORT(0,SB_LINELEFT));
             bKeyHandled = TRUE;
          } /* end case */
          break;

          case VK_RIGHT:
          {
             WinSendMsg ( hwnd, WM_HSCROLL,
                          MPFROMSHORT(FID_HORZSCROLL),
                          MPFROM2SHORT(0,SB_LINERIGHT));
             bKeyHandled = TRUE;
          } /* end case */
          break;

          case VK_UP:
          {
             WinSendMsg ( hwnd, WM_VSCROLL,
                          MPFROMSHORT(FID_VERTSCROLL),
                          MPFROM2SHORT(0,SB_LINEUP));
             bKeyHandled = TRUE;
          } /* end case */
          break;

          case VK_DOWN:
          {
             WinSendMsg ( hwnd, WM_VSCROLL,
                          MPFROMSHORT(FID_VERTSCROLL),
                          MPFROM2SHORT(0,SB_LINEDOWN));
             bKeyHandled = TRUE;
          } /* end case */
          break;

        } /* end switch */

      } /* end if */

      if( !bKeyHandled )
      {
        // Set nmemonic key to look for
        if( SHORT2FROMMP ( mp2 ) == VK_SHIFT )
          hMenu3d->ch = WinUpperChar( ppd->habPgm, 0L, 0L, (USHORT)ch );
        else
          hMenu3d->ch = ch;

        // Draw3DButton return TRUE if button found else FALSE
        // this matches what we want to return for WM_CHAR
        bKeyHandled = Draw3DButton( hMenu3d, (POWNERITEM)NULL, 0, BUTTON_ACTION_MNEMONIC );

      } /* end else */

      return ((MRESULT) bKeyHandled );

    } /* end case */
    break;

    case WM_COMMAND:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );
      pState   = &ppd->StateData;

      switch( SHORT1FROMMP(mp1) )
      {
         case IDM_SELECT_QUEUE:
         {
           LONG rc;

           ppd->hwndMain = WinLoadDlg( HWND_DESKTOP,
                                       WinQueryActiveWindow(HWND_DESKTOP),
                                       (PFNWP)QueueDlgProc,
                                       (HMODULE)NULL,
                                       IDD_SELECT_QUEUE,
                                       (PVOID)ppd );

           rc = WinProcessDlg( ppd->hwndMain );

           // must destroy window ourselves for WinProcessDlg() loaded
           // dialogs or all close msgs will not be sent
           WinDestroyWindow( ppd->hwndMain );

         } /* end case */
         break;

         case IDM_SETTINGS:
         {
           LONG rc;

           /*------------------------------------------*/
           /* Set the hour-glass pointer               */
           /*------------------------------------------*/
           WinSetPointer (HWND_DESKTOP,
                          WinQuerySysPointer( HWND_DESKTOP,
                                              SPTR_WAIT,
                                              FALSE));

            ppd->hwndMain = WinLoadDlg( HWND_DESKTOP,
                                             WinQueryActiveWindow(HWND_DESKTOP),
                                             (PFNWP)ProgramSettingsDlgProc,
                                             (HMODULE)NULL,
                                             IDD_PROGRAM_SETTINGS,
                                             (PVOID)ppd );

           /*------------------------------------------*/
           /* Set the hour-glass pointer               */
           /*------------------------------------------*/
           WinSetPointer (HWND_DESKTOP,
                          WinQuerySysPointer( HWND_DESKTOP,
                                              SPTR_ARROW,
                                              FALSE));

           rc = WinProcessDlg( ppd->hwndMain );

           // must destroy window ourselves for WinProcessDlg() loaded
           // dialogs or all close msgs will not be sent
           WinDestroyWindow( ppd->hwndMain );

           if( rc !=DID_CANCEL )
           {
             // invalidate entire client window to apply changes made
             WinInvalidateRect ( hwndClient, NULL, TRUE );

             // Tell ourselves to change the currently displayed font
             WinPostQueueMsg( ppd->hmqPgm,
                              (USHORT)WM_CHANGE_FONT,
                              MPVOID, MPVOID );
           } /* end if */

         } /* end case */
         break;

         case IDM_DEFAULT_OPTIONS:
         {
           ULONG ulRet;

           ulRet = WinMessageBox ( (HWND)HWND_DESKTOP,
                                   (HWND)NULL,
                                   (PSZ)MSG_VERIFY_DEFAULT,
                                   (PSZ)ERROR_TYPE_VERIFY,
                                   1L,
                                   MB_OKCANCEL |
                                   MB_DEFBUTTON2 |
                                   MB_ICONEXCLAMATION );

           if( ulRet == MBID_OK )
           {
             DeleteIniFileData( hwnd, FT_KEYID_ALL );
           } /* end if */

         } /* end case */
         break;

         case IDM_EXIT:
         {
           // @75474: NOTE: memory allocated to hold file/line ptrs
           // is implicitely freed on process term.
           // may need to alter this. Perhaps free on some other msg

           // Tell ourselves to quit
           WinPostMsg ( hwnd, WM_QUIT, 0L, 0L );

         } /* end case */
         break;

         case IDM_OPEN_FILE:

              WinFileDlg( HWND_DESKTOP, hwnd, &filedlg );

         case IDM_OPEN_QUIET:
         {
           if( filedlg.lReturn == DID_OK ||
               pState->bCmdLinePrint ||
               pState->bCmdLineBrowse )
           {

             // only do windowing stuff if PM interface is used
             if( !pState->bCmdLinePrint )
             {
               // Enable the printing menu item now that we have a file
               WinEnableMenuItem( hwndMenu, IDM_PRINT,  TRUE );

               // Enable the search menu item now that we have a file
               WinEnableMenuItem( hwndMenu, IDM_SEARCH_TEXT, TRUE );

             } /* end if */

             // open input file
             status = DosOpen2 ( (PSZ)filedlg.szFullFile,
                                 &hInFile, &action,
                                 0L, 0L, 1L,
                                 (ULONG)0x20,
                                 (PEAOP2)0L );

             // If DosOpen failed
             if ( status != 0 )
             {
                WinMessageBox ( (HWND)HWND_DESKTOP, (HWND)NULL,
                                (PSZ)ERR_DOSOPEN, (PSZ)ERROR_TYPE_TRYAGAIN,
                                1L, MB_OK | MB_ICONEXCLAMATION );
             } /* end if */
             else
             {
                FileInfoBufSize = sizeof(FILEFINDBUF);

                // Get file information
                status = DosQFileInfo( hInFile, (USHORT)1,
                                       (PBYTE)&FileInfoBuf,
                                       FileInfoBufSize );

                // Read in file if file info retrieved
                if(status == 0)
                {
                  /*------------------------------------------*/
                  /* Set the hour-glass pointer               */
                  /*------------------------------------------*/
                  WinSetPointer (HWND_DESKTOP,
                                 WinQuerySysPointer( HWND_DESKTOP,
                                                     SPTR_WAIT,
                                                     FALSE));

                  // Read in the text file to memory
                  Success = ReadInTextFile ( ppd->habPgm,
                                             hInFile,
                                             FileInfoBuf.cbFile,
                                             pState->bFileOpen );

                  /*------------------------------------------*/
                  /* Set the arrow pointer                    */
                  /*------------------------------------------*/
                  WinSetPointer (HWND_DESKTOP,
                                 WinQuerySysPointer( HWND_DESKTOP,
                                                     SPTR_ARROW,
                                                     FALSE));

                  // If file read in w/o incident
                  if( Success )
                  {
                    pState->bFileOpen = TRUE;

                    strcpy( szTitleText, "Browsing File: \0" );
                    strcat( szTitleText, filedlg.szFullFile );

                    WinSetWindowText(hwndTitlebar,szTitleText);

                    // set scroll bars if using PM interface
                    if( !pState->bCmdLinePrint )
                      // Set Horz and Vert scrollbars to new size/pos
                      SetScrollBars( hwnd, hwndHscroll, hwndVscroll,
                                     &lVscrollMax, cxClient, cyClient );

                  } /* end if */

               } /* end if */
               else
               {
                  WinMessageBox ( (HWND)HWND_DESKTOP, (HWND)NULL,
                                  (PSZ)filedlg.szFullFile,
                                  (PSZ)ERR_DOSQFILEINFO, 1L,
                                  MB_OK | MB_ICONEXCLAMATION );
               } /* end else */

             } /* end else */

           } /* end if */

           return (MRESULT)NULL;
         } /* end case */
         break;

         case IDM_PAGEUP: // @MAXLINE
         {
           lVscrollPos = 0L;

           // Set both Horz and Vert scrollbars to new size/pos
           SetScrollBars( hwnd, hwndHscroll, hwndVscroll,
                          &lVscrollMax, cxClient, cyClient );

         } /* end case */
         break;

         case IDM_PAGEDOWN:  // @MAXLINE
         {
           lVscrollPos = ulTotalLines;

           // Set both Horz and Vert scrollbars to new size/pos
           SetScrollBars( hwnd, hwndHscroll, hwndVscroll,
                          &lVscrollMax, cxClient, cyClient );

         } /* end case */
         break;

         case IDM_PRINT:
         {
           TID Tid;

           // If a file is open to print
           if(pState->bFileOpen == TRUE)
           {
             // Create a second thread for printing
             CreateThread( &Tid, (PFNTHREAD) PrintThread );

           } /* end if */
           else
           {
             // Inform user to select a file first
             WinMessageBox( (HWND)HWND_DESKTOP, (HWND)NULL,
                            (PSZ)ERR_NO_FILE_SELECTED,
                            (PSZ)ERROR_TYPE_TRYAGAIN,
                            1L, MB_OK );

             if( pState->bCmdLinePrint )
             {
               // Tell ourselves to quit
               WinPostMsg ( hwnd, WM_QUIT, 0L, 0L );

             } /* end if */

           } /* end else */

         } /* end case */
         break;

         case IDM_QUERY_CAPS:
         {
           rc = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_CAPS, DEV_TYPE_PRINTER );
           AssertPM( rc!=NO_ERROR, ppd->habPgm, ERR_NO_QUEUE_TO_QUERY );
         } /* end case */
         break;

         case IDM_QUERY_HCCAPS:
         {
           rc = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_HCCAPS, DEV_TYPE_PRINTER );
           AssertPM( rc!=NO_ERROR, ppd->habPgm, ERR_NO_QUEUE_TO_QUERY );
         } /* end case */
         break;

         case IDM_QUERY_METRICS:
         {
           rc = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_METRICS, DEV_TYPE_PRINTER );
           AssertPM( rc!=NO_ERROR, ppd->habPgm, ERR_NO_QUEUE_TO_QUERY );
         } /* end case */
         break;

         case IDM_QUERY_FNTWDTH:
         {
           rc = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_WIDTHS, DEV_TYPE_PRINTER );
           AssertPM( rc!=NO_ERROR, ppd->habPgm, ERR_NO_QUEUE_TO_QUERY );
         } /* end case */
         break;

         case IDM_QUERY_FNTKERN:
         {
           rc = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_KERN, DEV_TYPE_PRINTER );
           AssertPM( rc!=NO_ERROR, ppd->habPgm, ERR_NO_QUEUE_TO_QUERY );
         } /* end case */
         break;

         case IDM_SEARCH_TEXT:
         {
           if(pState->bFileOpen)
           {
             TID Tid;

             // Get search string
             usResult = (USHORT)WinDlgBox( HWND_DESKTOP,
                                           hwnd,
                                           (PFNWP)SearchDialogProc,
                                           (HMODULE)NULL,
                                           (ULONG)IDD_SEARCH,
                                           szSearchString );

             // start search if user still wants to
             if( usResult == DID_OK )
             {
               if( szSearchString[0] != '\0' )
               {
                 // Start a second thread for search
                 CreateThread( &Tid, (PFNTHREAD)SearchThread );

               } /* end if */
               else
               {
                 AssertPM( TRUE, ppd->habPgm, ERR_NO_SEARCH_STRING );

               } /* end else */

             } /* end if */

           } /* end if */
           else
           {
             // Inform user to select a file first
             WinMessageBox( (HWND)HWND_DESKTOP, (HWND)NULL,
                            (PSZ)ERR_NO_FILE_SELECTED,
                            (PSZ)ERROR_TYPE_TRYAGAIN,
                            1L, MB_OK );
           } /* end else */

         } /* end case */
         break;

         case IDM_SEARCH_NEXT:
         {
           if(pState->bFileOpen)
           {
             if( szSearchString[0] != '\0' )
             {
               TID Tid;

               // Start a second thread for search
               CreateThread( &Tid, (PFNTHREAD)SearchThread );

             } /* end if */
             else
             {
               AssertPM( TRUE, ppd->habPgm, ERR_NO_SEARCH_STRING );

             } /* end else */

           } /* end if */
           else
           {
             // Inform user to select a file first
             WinMessageBox( (HWND)HWND_DESKTOP, (HWND)NULL,
                            (PSZ)ERR_NO_FILE_SELECTED,
                            (PSZ)ERROR_TYPE_TRYAGAIN,
                            1L, MB_OK );
           } /* end else */

         } /* end case */
         break;

         case IDM_HELP_HELP:
              // display panel from .IPF file for "Help Using Font Test"
              if( ppd->StateData.bHelpAvailable )
              {
                WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                            MPFROMSHORT( PANEL_HELP_USING ),
                            MPFROMSHORT( HM_RESOURCEID ));

              } /* end if */
              else
                AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
         break;

         case IDM_EXT_HELP:
              if( ppd->StateData.bHelpAvailable )
                WinSendMsg(hwndHelp, HM_EXT_HELP, NULL, NULL);
              else
                AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
         break;

         case IDM_KEYS_HELP:
              if( ppd->StateData.bHelpAvailable )
                WinSendMsg(hwndHelp, HM_KEYS_HELP, NULL, NULL);
              else
                AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
         break;

         case IDM_HELP_INDEX:
              if( ppd->StateData.bHelpAvailable )
                WinSendMsg(hwndHelp, HM_HELP_INDEX, NULL, NULL);
              else
                AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
         break;

         case IDM_ABOUT:
              // Display about dialog
              WinDlgBox( HWND_DESKTOP,
                         hwnd,
                         (PFNWP)AboutDlgProc,
                         (HMODULE)NULL,
                         (ULONG)IDD_ABOUT,
                         (PVOID)NULL );
         break;
      } /* end switch */

      // We handled the message so return zero
      return(0);

    } /* end case */
    break;

    case WM_DESTROY:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

      // Destroy Display HDC and HPS
      GpiDestroyPS( ppd->DisplayData.hpsMem );
      DevCloseDC( ppd->DisplayData.hdcMem );

      // destroy the micro presentation space
      GpiDestroyPS ( hps );
      DevCloseDC( hdc );

      // Free up Menu Structures (implied free of all buttons)
      Destroy3DMenu( hMenu3d );

      // We handled the message so return zero
      return(0);

    } /* end case */
    break;

    case WM_HSCROLL:
    {
      switch( SHORT2FROMMP(mp2) )
      {
        case SB_LINELEFT:  lHscrollInc = -1;
        break;
        case SB_LINERIGHT: lHscrollInc =  1;
        break;
        case SB_PAGELEFT:  lHscrollInc = min(-1, -(cxClient/cxChar) );
        break;
        case SB_PAGERIGHT: lHscrollInc = max( 1, (cxClient/cxChar) );
        break;
        case SB_SLIDERTRACK:
             lHscrollInc = SHORT1FROMMP(mp2) - lHscrollPos;
        break;
        default: lHscrollInc = 0;
        break;
      } /* end switch */

      // min - check increment against maximum scroll pos
      // if the increment is negative, make sure it is not more negative
      // then we have already moved by testing against cur pos.
      lHscrollInc = max( -lHscrollPos,
                    min(  lHscrollInc, MAX_CHAR_PER_LINE - lHscrollPos) );

      if(lHscrollInc != 0 )
      {
        lHscrollPos += lHscrollInc;

        WinScrollWindow( hwnd, (LONG)(-lHscrollInc*cxChar), 0L,
                         (PRECTL)NULL, (PRECTL)NULL, (HRGN)NULL,
                         (PRECTL)NULL, SW_INVALIDATERGN );

        WinSendMsg( hwndHscroll, SBM_SETPOS, MPFROMSHORT(lHscrollPos),
                    NULL );

        WinUpdateWindow( hwnd );

      } /* end if */

      return(0);

    } /* end case */
    break;

    case WM_MEASUREITEM:
    {
      USHORT     id;
      POWNERITEM poi;

      id  = (USHORT) SHORT1FROMMP(mp1);       // window ID
      poi = (POWNERITEM) PVOIDFROMMP(mp2);    // pointer to OWNERITEM

      Draw3DButton( hMenu3d, poi, id, BUTTON_ACTION_MEASURE );

      return( MRFROMLONG(TRUE) );

    } /* end case */
    break;

    case WM_DRAWITEM:
    {
      USHORT     id;
      POWNERITEM poi;

      id  = (USHORT)     SHORT1FROMMP(mp1);   // window ID
      poi = (POWNERITEM) PVOIDFROMMP (mp2);   // pointer to OWNERITEM

      if( poi->fsAttribute == poi->fsAttributeOld )
      {

        Draw3DButton( hMenu3d, poi, id, BUTTON_ACTION_DRAW );

      } /* end if */

      return( MRFROMLONG(TRUE) );

    } /* end case */
    break;

    case WM_PAINT:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

      // request a cache PS to draw to
      hps = WinBeginPaint ( hwnd, hps, &rclInvalid );

      // fill client with background color
      WinFillRect ( hps, &rclInvalid, ppd->ColorData.WinBackColor );

      // if we have a file open
      if( ppd->StateData.bFileOpen )
      {
        // draw the lines of text on the screen
        PaintClientWindow( hps, ppd->ColorData.WinTextColor,
                           hwnd, &rclInvalid );
      } /* end if */

      // free up cache PS
      WinEndPaint( hps );

      return(0);

    } /* end case */
    break;

    case WM_PRINT_THREAD_READY:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

      /*-------------------------------------------------*/
      /* Show in Titlebar that we have started printing  */
      /*-------------------------------------------------*/
      strcpy( szTitleText, "Printing File: \0" );
      strcat( szTitleText, filedlg.szFullFile );
      WinSetWindowText( hwndTitlebar,szTitleText );

      /*-------------------------------------------------*/
      /* Send message to printing thread to begin        */
      /* printing the selected file.                     */
      /*-------------------------------------------------*/
      WinPostQueueMsg( ppd->PrinterData.hmqPrn,
                       (USHORT)WM_PRINTDOC,
                       MPVOID,
                       MPVOID );

    } /* end case */
    break;

    case WM_SEARCH_FAILED:
    {
       WinMessageBox( (HWND)HWND_DESKTOP,
                      (HWND)NULL,
                      (PSZ)ERR_UNABLE_FIND_STRING,
                      (PSZ)ERROR_TYPE_TRYAGAIN,
                      1L,
                      MB_OK );

       // return zero we handled the call
       return(0);

    } /* end case */
    break;

    case WM_SIZE:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

      cxClient = SHORT1FROMMP( mp2 );
      cyClient = SHORT2FROMMP( mp2 );

      // If we have a file open
      if( ppd->StateData.bFileOpen == TRUE )
      {
        // Set both Horz and Vert scrollbars to new size/pos
        SetScrollBars( hwnd, hwndHscroll, hwndVscroll,
                       &lVscrollMax, cxClient, cyClient );

      } /* end if */

      return(0);

    } /* end case */
    break;

    case WM_VSCROLL:
    {
      ULONG  ulLines;

      ulLines = cyClient / cyChar;

      switch( SHORT2FROMMP(mp2) )
      {
         case SB_LINEUP:   lVscrollInc = -1;
         break;
         case SB_LINEDOWN: lVscrollInc =  1;
         break;
         case SB_PAGEUP:
              lVscrollInc = min( -1, -(ulLines) );
         break;
         case SB_PAGEDOWN:
              lVscrollInc = max( 1, ulLines );
         break;
         case SB_SLIDERTRACK:
              lVscrollInc = SHORT1FROMMP(mp2) - lVscrollPos;
         break;
         default : lVscrollInc = 0;
         break;
      } /* end switch */

      lVscrollInc = max( -lVscrollPos,
                    min( lVscrollInc, lVscrollMax - lVscrollPos));

      if( lVscrollInc != 0 )
      {
         lVscrollPos += lVscrollInc;

         WinScrollWindow( hwnd, 0L, cyChar * lVscrollInc, NULL, NULL,
                          (HRGN)NULL, NULL, SW_INVALIDATERGN );

         WinSendMsg( hwndVscroll,
                     SBM_SETPOS,
                     MPFROMSHORT( min( lVscrollPos, MAXLINES - ulLines ) ),
                     MPVOID );

         WinUpdateWindow( hwnd );

// @MAXLINE - show debug line # output
#if 1
         {
           CHAR szTemp[256];

           sprintf( szTemp, "[%d/%d] %s\0", lVscrollPos, ulTotalLines, szTitleText );

           WinSetWindowText( hwndTitlebar,szTemp );

         } /* end logical block */
#endif

      } /* end if */

      return(0);

    } /* end case */
    break;

  } /* end switch */

  return WinDefWindowProc( hwnd, msg, mp1, mp2 );

} /* end ClientWndProc */


/****************************************************************************/
/* PROCEDURE NAME : AccessDevice                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Main function allowing access to a device               */
/*                  This includes:  Printing text files, querying device    */
/*                  fonts, querying device caps, querying device hardcopy   */
/*                  caps.                                                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) call GetQueueData(); to get driver data             */
/*                  (b) if printing from PM display the                     */
/*                      "Confirm Print Information" dialog to verify        */
/*                      queue, font, color, and text format options.        */
/*                  (c) call DevOpenDC(); to open the output device         */
/*                  (d) Query device or print to it based on input          */
/*                      parameter 'usActionType'.                           */
/*                                                                          */
/*                    Possible Requests                                     */
/*                    ----------------                                      */
/*                                                                          */
/*                    REQUEST_PRINT_FILE :                                  */
/*                                                                          */
/*                       Print Text file from PM interface.                 */
/*                       call DevEscape( DEVESC_STARTDOC );                 */
/*                       call PrintTextFile();                              */
/*                       call DevEscape( DEVESC_ENDDOC );                   */
/*                                                                          */
/*                    REQUEST_PRINT_FILE_NO_PM :                            */
/*                                                                          */
/*                       Print text file from an OS/2 command line.         */
/*                       call DevEscape( DEVESC_STARTDOC );                 */
/*                       call PrintTextFile();                              */
/*                       call DevEscape( DEVESC_ENDDOC );                   */
/*                                                                          */
/*                    REQUEST_QUERY_FONTS :                                 */
/*                                                                          */
/*                       call    GpiQueryFonts();                           */
/*                       display 'Select a Font for Printing' dialog        */
/*                                                                          */
/*                    REQUEST_QUERY_CAPS :                                  */
/*                                                                          */
/*                       call    DevQueryCaps();                            */
/*                       display 'Device Capabilities' dialog               */
/*                                                                          */
/*                    REQUEST_QUERY_HCCAPS :                                */
/*                                                                          */
/*                       call    DevQueryHardCopyCaps();                    */
/*                       display 'Device Form Information' dialog           */
/*                                                                          */
/*                    REQUEST_QUERY_FORM_INFO :                             */
/*                                                                          */
/*                       call    DevQueryHardCopyCaps();                    */
/*                       return  index of HCCAPS_CURRENT form in HCINFO     */
/*                               structure.                                 */
/*                                                                          */
/*                    REQUEST_QUERY_METRICS :                               */
/*                                                                          */
/*                       call    GpiQueryFontMetrics(); for currently       */
/*                               realized font                              */
/*                       display 'Current Font Metrics' dialog              */
/*                                                                          */
/*                                                                          */
/*                  (e) call DevCloseDC(); to close device context          */
/*                                                                          */
/*                                                                          */
/* PARAMETERS:     (HAB)    habLcl       - anchor block handle for error    */
/*                                         processing.                      */
/*                 (LONG)   lDCType      - type of Device Context (DC)      */
/*                                         OD_QUEUED or OD_INFO             */
/*                 (ULONG)  ulRequestType - what action to take on this     */
/*                                         call to AccessDevice();          */
/*                 (ULONG)  ulDevType    - Type of device (printer/display) */
/*                                                                          */
/* RETURN VALUES:  */
/*                 */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
ERRORID AccessDevice( PPGMDATA pPgmData, HAB habLcl, LONG lDCType, ULONG ulRequestType, ULONG ulDevType )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ERRORID       ErrorID = NO_ERROR;    // This functions return code
  ERRORID       bError  = NO_ERROR;    // Error assertion return code
  LONG          lReturn;               // LONG Return codes from APIs
  BOOL          bSuccess;              // BOOL Return codes from APIs

  DEVOPENSTRUC  dop;                   // Device Data for DevOpenDC
  USHORT        us;                    // Loop control variable
  PPRINTERDATA  pPrnData = &pPgmData->PrinterData;  // Local copy of Printer Data
  PQUEUEDATA    pqd = &pPgmData->QueueData;         // Local copy of Queue Data

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-----------------------------------------------------------*/
  /* Fill DEVOPENSTRUC for "printer" or "display"              */
  /*-----------------------------------------------------------*/
  if( ulDevType == DEV_TYPE_PRINTER )
  {
    // Get most recent PRQINFO data (driver name, job properties)
    if( GetQueueData( habLcl, pqd, pPrnData, FALSE ) == NO_ERROR )
    {
      // Fill DevOpenStruc for "printer" device
      dop.pszLogAddress      = pPgmData->QueueData.QueueName;
      dop.pszDriverName      = pPrnData->szDriverName;
      dop.pszDataType        = "PM_Q_STD";
      dop.pszComment         = PROGRAM_TITLE_CNST;
      dop.pszQueueProcName   = NULL;
      dop.pszQueueProcParams = "XFM=0";
      dop.pszSpoolerParams   = NULL;
      dop.pszNetworkParams   = NULL;

      /*-----------------------------------------------------------*/
      /* CONFIRM PRINT                                             */
      /*-----------------------------------------------------------*/
      if( ulRequestType & REQUEST_PRINT_FILE )
      {

        pPgmData->StateData.bPrinting = TRUE;

        // Confirm Job Properties / Job Options before printing
        if( WinDlgBox( HWND_DESKTOP, hwndClient, (PFNWP)ConfirmDlgProc,
                       (HMODULE)NULL, IDD_MAIN_PRINT, (PVOID)pPgmData ) == DID_CANCEL )
        {
          // User cancelled print request so change action type
          ulRequestType = REQUEST_NONE;

        } /* end if */

        pPgmData->StateData.bPrinting = FALSE;

      } /* end if */

      // Get Job Property data after user was allowed to modify
      dop.pdriv=PgmData.QueueData.pDrivData;

    } /* end if */
    else
    {
      // No queue data was found, cancel all requests (actions)
      // NOTE: This means that no valid OS/2 Print Queues were found
      ErrorID = PMERR_SPL_QUEUE_NOT_FOUND;

    } /* end else */

  } /* end if */
  else
  {
    if( ulDevType == DEV_TYPE_DISPLAY )
    {
       // Fill DevOpenStruc for "display" device
       dop.pszLogAddress      = NULL;
       dop.pszDriverName      = (PSZ) "DISPLAY";
       dop.pdriv              = NULL;
       dop.pszDataType        = NULL;
       dop.pszComment         = PROGRAM_TITLE_CNST;
       dop.pszQueueProcName   = NULL;
       dop.pszQueueProcParams = "XFM=0";
       dop.pszSpoolerParams   = NULL;
       dop.pszNetworkParams   = NULL;

    } /* end if */

  } /* end else */

  /*---------------------------------------------------------*/
  /* If 'DevOpenData' and Request Type are valid             */
  /*---------------------------------------------------------*/
  if( ErrorID == NO_ERROR && ulRequestType )
  {
    /*---------------------------------------------------------*/
    /* DevOpenDC() : Open device to get a Device Context       */
    /*---------------------------------------------------------*/
    pPrnData->hdcPrn = DevOpenDC( habLcl,             // anchor block handle
                                  lDCType,            // Type of output (i.e. OD_)
                                  "*",                // Default token (use 'dop')
                                  9L,                 // # of items in 'dop'
                                  (PDEVOPENDATA)&dop, // Device Open Data
                                  (HDC)NULL );        // Compatible HDC (none)

    ErrorID = AssertPM( pPrnData->hdcPrn==DEV_ERROR, habLcl, ERR_DEVOPENDC );

    // If DevOpenDC() was successful (i.e. valid HDC)
    if( ErrorID == NO_ERROR )
    {
      HMF   hmfPrn;      // Metafile handle for job generated

      /*-------------------------------------------------------*/
      /* Create a PS for the printer entire page (default)     */
      /*-------------------------------------------------------*/
      pPrnData->szlPage.cx = pPrnData->szlPage.cy = 0;

      pPrnData->hpsPrn = GpiCreatePS( habLcl, pPrnData->hdcPrn, &pPrnData->szlPage, PU_PELS | GPIA_ASSOC );

      ErrorID = AssertPM( pPrnData->hpsPrn==GPI_ERROR, habLcl, ERR_GPICREATEPS );

      /*-------------------------------------------------------*/
      /* If PS created successfully                            */
      /*-------------------------------------------------------*/
      if( ErrorID == NO_ERROR )
      {
        /***************************************************/
        /* REQUEST_QUERY_CAPS                              */
        /***************************************************/
        if( ulRequestType & REQUEST_QUERY_CAPS )
        {
          DisplayDeviceCaps( habLcl, pPrnData->hdcPrn,
                             hwndClient, pPrnData, ulRequestType );
        } /* end if */

        /***************************************************/
        /* REQUEST_QUERY_HCCAPS                            */
        /***************************************************/
        if( ulRequestType & REQUEST_PRINT_FILE       ||
            ulRequestType & REQUEST_PRINT_FILE_NO_PM ||
            ulRequestType & REQUEST_QUERY_FORM_INFO  ||
            ulRequestType & REQUEST_QUERY_HCCAPS    )
        {
          DisplayDeviceFormCaps( habLcl, pPrnData->hdcPrn,
                                 hwndClient, pPrnData, ulRequestType );

        } /* end if */

        /***************************************************/
        /* REQUEST_QUERY_FONTS                             */
        /***************************************************/
        if( ulRequestType & REQUEST_QUERY_FONTS )
        {
           LONG RemFonts;       // # of public fonts from GpiQueryFonts
           LONG ReqFonts = 0L;  // # fonts requested by GpiQueryFonts

           // @100698 - User may have selected a resolution on the printer
           // different from the one used to originally calculate the char box size.
           DevQueryCaps( pPrnData->hdcPrn, CAPS_HORIZONTAL_FONT_RES, 1L,
                         &(pPrnData->lCaps[CAPS_HORIZONTAL_FONT_RES]) );

           DevQueryCaps( pPrnData->hdcPrn, CAPS_VERTICAL_FONT_RES, 1L,
                         &(pPrnData->lCaps[CAPS_VERTICAL_FONT_RES]) );

           /* Get list of PUBLIC fonts supported by device */
           RemFonts = GpiQueryFonts( pPrnData->hpsPrn, QF_PUBLIC, NULL,
                                     &ReqFonts, 0L, (PFONTMETRICS)NULL );

           ErrorID = AssertPM( RemFonts==GPI_ALTERROR, habLcl, ERR_GPIQUERYFONTS );
           ErrorID = AssertPM( RemFonts==0, habLcl, ERR_NO_FONTS );

           if( ErrorID==NO_ERROR )
           {
              ULONG ulSize;   // Size to allocate for Font Metrics

              if(pPrnData->pFontMetrics)
                 DosFreeMem( pPrnData->pFontMetrics );

              // Figure out how much memory we need
              ulSize = ((ULONG)RemFonts * (ULONG)sizeof(FONTMETRICS) );

              // attempt to allocate it
              ErrorID = DosAllocMem( (PPVOID)&(pPrnData->pFontMetrics), ulSize,
                                     PAG_READ | PAG_WRITE | PAG_COMMIT );

              AssertPM( ErrorID!=NO_ERROR, habLcl, ERR_DOSALLOCMEM );

              // Able to allocate memory for fontmetrics
              if( ErrorID==NO_ERROR )
              {
                // Get all (remaining) public fonts in system
                ReqFonts = RemFonts;

                RemFonts = GpiQueryFonts( pPrnData->hpsPrn,
                                          QF_PUBLIC,
                                          NULL,
                                          &ReqFonts,
                                          (LONG)sizeof(FONTMETRICS),
                                          pPrnData->pFontMetrics );

                ErrorID = AssertPM( RemFonts==GPI_ALTERROR, habLcl, ERR_GPIQUERYFONTS );
                ErrorID = AssertPM( ReqFonts==0, habLcl, ERR_NO_FONTS );

              } /* end if */

           } /* end if */

           // Save # fonts returned in pFontMetrics
           if( ErrorID == NO_ERROR )
             pPrnData->lNumFonts = ReqFonts;
           else
             pPrnData->lNumFonts = 0;

        } /* end REQUEST_QUERY_FONTS */

        /***************************************************/
        /* REQUEST_PRINT_FILE                              */
        /***************************************************/
        if( ( ulRequestType & REQUEST_PRINT_FILE       ) ||
            ( ulRequestType & REQUEST_PRINT_FILE_NO_PM ) ||
            ( ulRequestType & REQUEST_QUERY_KERN       ) ||
            ( ulRequestType & REQUEST_QUERY_WIDTHS     ) ||
            ( ulRequestType & REQUEST_QUERY_METRICS    ))
        {
          /*-----------------------------------------------*/
          /* Start Document  (DEVESC_STARTDOC)             */
          /*-----------------------------------------------*/

          // Start document if we are printing
          if( lDCType != OD_INFO )
          {
            // @74652 - send in full string length for document name
            //          previously had '8' characters hardcoded
            lReturn = DevEscape( pPrnData->hdcPrn,
                                 DEVESC_STARTDOC,
                                 min(strlen(filedlg.szFullFile), MAXCOMMENTSZ),
                                 (PBYTE)filedlg.szFullFile,
                                 (LONG)0,
                                 (PBYTE)NULL );

            ErrorID = AssertPM( lReturn!=DEV_OK, habLcl, ERR_DEVESCSTARTDOC );

          } /* end if */

          /*-----------------------------------------------*/
          /* DEVESC_STARTDOC success                       */
          /*-----------------------------------------------*/
          if( ErrorID == NO_ERROR )
          {
            FATTRS fat;            // Font attributes of current font
            LONG   lFontMatch;     // Return value from GpiCreateLogFont
            LONG   OutCount;       // DevEscape ENDOC code
            USHORT usJobId;        // Job Id number from DEVESC_ENDDOC

            /*---------------------------------------------*/
            /* Font Attributes (FATTRS) for output         */
            /*---------------------------------------------*/
            fat.usRecordLength  = sizeof(FATTRS);
            fat.fsSelection     = pPgmData->FontData.fm.fsSelection |
                                  pPgmData->FontData.Options;
            fat.lMatch          = pPgmData->FontData.fm.lMatch;
            fat.idRegistry      = 0;
            fat.lMaxBaselineExt = 0L;
            fat.lAveCharWidth   = 0L;
            fat.fsType          = 0;
            // @DBCS - set codepage to system codepage so font realized
            // with correct codepage (not font's default codepage (i.e. not 0)
            fat.usCodePage      = CodePage;
            fat.fsFontUse       = FATTR_FONTUSE_NOMIX;
            for (us=0; fat.szFacename[us] =
                       pPgmData->FontData.fm.szFacename[us]; ++us);

            /*---------------------------------------------*/
            /* Create the Font                             */
            /*---------------------------------------------*/
            lFontMatch = GpiCreateLogFont( pPrnData->hpsPrn, NULL, LCID_PRINTER, &fat );

            ErrorID = AssertPM( lFontMatch==GPI_ERROR, habLcl, ERR_GPICREATELOGFONT );

            AssertPM( lFontMatch==FONT_DEFAULT, habLcl, ERR_TYPE_FONTMATCH );

            /*---------------------------------------------*/
            /* Font Match found                            */
            /*---------------------------------------------*/
            if( lFontMatch == FONT_MATCH )
            {
              /*-------------------------------------------*/
              /* SCALABLE FONT                             */
              /*-------------------------------------------*/
              if( pPgmData->FontData.FontScalable == TRUE )
              {
                // Put the new box in effect for the printer pres space
                // otherwise use default (designer's) char size for font
                if( pPgmData->FontData.PointSize !=
                    (pPgmData->FontData.fm.sNominalPointSize/10) )
                {
                  LONG lHorzFontRes;     // @100698 - CAPS_HORIZONTAL_FONT_RES (37L)
                  LONG lVertFontRes;     // @100698 - CAPS_VERTICAL_FONT_RES   (38L)
                  SIZEF sfCharBox;

                  // @100698 - User may have selected a resolution on the printer
                  // different from the one used to originally calculate the char box size.
                  DevQueryCaps( pPrnData->hdcPrn, CAPS_HORIZONTAL_FONT_RES, 1L,
                                &lHorzFontRes );

                  DevQueryCaps( pPrnData->hdcPrn, CAPS_VERTICAL_FONT_RES, 1L,
                                &lVertFontRes );

                  // resolution changed so recalculate char. box
                  CalculateCharBox( pPgmData->FontData.PointSize,
                                    lHorzFontRes,
                                    lVertFontRes,
                                    &sfCharBox );

                  bSuccess = GpiSetCharBox( pPrnData->hpsPrn,
                                            &sfCharBox );

                  bError = AssertPM( bSuccess==GPI_ERROR, habLcl, ERR_GPISETCHARBOX );

                } /* end if */

              } /* end if */

              /*-------------------------------------------*/
              /* Select Font as current                    */
              /*-------------------------------------------*/
              bSuccess = GpiSetCharSet( pPrnData->hpsPrn, LCID_PRINTER );

              ErrorID = AssertPM( bSuccess==GPI_ERROR, habLcl, ERR_GPISETCHARSET );

            } /* end FONT_MATCH */

            /*---------------------------------------------*/
            /* Character Set selected successfully         */
            /*---------------------------------------------*/
            if( ErrorID == NO_ERROR )
            {
              FONTMETRICS  fmPrn;     // FontMetrics of current font

              // Find Font Metrics of current font
              bSuccess = GpiQueryFontMetrics( pPrnData->hpsPrn,
                                              (LONG)sizeof(FONTMETRICS),
                                              &fmPrn);

              // check if font query failed
              ErrorID = AssertPM( !bSuccess, habLcl, ERR_GPIQFONTMETRICS );

              // check if font has zero height
              ErrorID = AssertPM( fmPrn.lMaxBaselineExt== 0, habLcl, ERR_MAX_BASELINE_EXT );

              /**********************************************/
              /* REQUEST_QUERY_METRICS                      */
              /**********************************************/
              if( ( ErrorID == NO_ERROR ) &&
                  ( ulRequestType & REQUEST_QUERY_METRICS ) )
              {
                  // Select FONTEMETRICS so we can add Font Name to information we display
                  pPrnData->pFontMetrics = &fmPrn;

                  WinDlgBox( HWND_DESKTOP,
                             hwndClient,
                             (PFNWP)MetricsDlgProc,
                             (HMODULE)NULL,
                             (ULONG)IDD_METRICS,
                             (PVOID)pPgmData );

              } /* end if */

              /***************************************************/
              /* REQUEST_QUERY_WIDTHS                            */
              /***************************************************/
              if( ( ErrorID == NO_ERROR ) &&
                  ( ulRequestType & REQUEST_QUERY_WIDTHS ) )
              {

                // Select FONTEMETRICS so we can add Font Name to information we display
                pPrnData->pFontMetrics = &fmPrn;

                // Show Font Width Table dialog
                WinDlgBox( HWND_DESKTOP,
                           hwndClient,
                           (PFNWP)FontWidthDlgProc,
                           (HMODULE)NULL,
                           (ULONG)IDD_FONT_WIDTHS,
                           (PVOID)pPgmData );

              } /* end if */

              /***************************************************/
              /* REQUEST_QUERY_KERN                              */
              /***************************************************/
              if( ( ErrorID == NO_ERROR ) &&
                  ( ulRequestType & REQUEST_QUERY_KERN ) )
              {

                // check if Font query width table failed
                ErrorID = AssertPM( !(fmPrn.sKerningPairs > 0), habLcl, ERR_FONT_NO_KERN_PAIRS );

                // We have Kerning pairs to examine
                if( ErrorID == NO_ERROR )
                {
                  // allocate memory to hold kerning pairs
                  ErrorID = DosAllocMem( (PPVOID)&(pPrnData->pFontKernPairs),
                                         fmPrn.sKerningPairs * sizeof(KERNINGPAIRS),
                                         PAG_READ | PAG_WRITE | PAG_COMMIT );

                  AssertPM( ErrorID!=NO_ERROR, habLcl, ERR_DOSALLOCMEM );

                  // Able to allocate memory to hold kerning pairs
                  if( ErrorID == NO_ERROR )
                  {
                    // Select FONTEMETRICS so we can add Font Name to information we display
                    pPrnData->pFontMetrics = &fmPrn;

                    // Query Kerning pairs for currently selected font
                    bSuccess = GpiQueryKerningPairs( pPrnData->hpsPrn,
                                                     (LONG)fmPrn.sKerningPairs,
                                                     pPrnData->pFontKernPairs);

                    // If we got some font widths to show
                    if( bSuccess != GPI_ERROR )
                    {
                      // Show Font Width Table dialog
                      WinDlgBox( HWND_DESKTOP,
                                 hwndClient,
                                 (PFNWP)FontKernDlgProc,
                                 (HMODULE)NULL,
                                 (ULONG)IDD_FONT_KERN_PAIRS,
                                 (PVOID)pPgmData );
                    } /* end if */

                    /* free the memory used to store font */
                    DosFreeMem( pPrnData->pFontKernPairs );

                   } /* end if  */

                } /* end if */

              } /* end if */

              /**********************************************/
              /* REQUEST_PRINT_FILE                         */
              /**********************************************/
              if( ( ErrorID == NO_ERROR ) &&
                  ( ( ulRequestType & REQUEST_PRINT_FILE )    ||
                    ( ulRequestType & REQUEST_PRINT_FILE_NO_PM ) ) )
              {

                LONG lLinesPerPage;  // # of text lines on a printer page
                LONG cyPrintChar;    // Height of output font
                float f;

                // calculate height of a line from char height
                cyPrintChar  = fmPrn.lMaxBaselineExt +
                               fmPrn.lExternalLeading;

                // calculate lines per page based on line height
                // make allowance for first line where font height
                // need not take into account the descender as it is
                // first line on page.
                lLinesPerPage = (LONG)((float)(pPrnData->szlPage.cy+fmPrn.lMaxDescender))/((float)cyPrintChar);

                // If at least 1 line will fit on the page
                if( lLinesPerPage > 0 )
                {
                  PrintTextFile( habLcl,
                                 pPrnData->hdcPrn,
                                 pPrnData->hpsPrn,
                                 PgmData,
                                 cyPrintChar,
                                 lLinesPerPage,
                                 &fmPrn );

                } /* end if */
                else
                {
                   // Font too large to fit on page
                   ErrorID = AssertPM( TRUE, habLcl, ERR_FONT_TOO_LARGE );

                } /* end else */

              } /* end if */

            } /* end if */

            /*--------------------------------------------*/
            /* End Document (DEVESC_ENDDOC)               */
            /*--------------------------------------------*/
            if( lDCType != OD_INFO )
            {
              OutCount = sizeof(usJobId);

              lReturn = DevEscape( pPrnData->hdcPrn,    // HDC   hdc
                                   DEVESC_ENDDOC,       // LONG  lCode
                                   0L,                  // LONG  lInCount
                                   (PBYTE)NULL,         // PBYTE pbInData
                                   &OutCount,           // PLONG plOutCount
                                   (PBYTE)&usJobId );   // PBYTE pbOutData

              ErrorID = AssertPM( lReturn!=DEV_OK, habLcl, ERR_DEVESCENDDOC );

            } /* end if */

          } /* end if */ // DevEscape STARTDOC success

          // If we have encountered an error we need to abort the document
          if( ErrorID != NO_ERROR )
          {
            /*----------------------------------------------*/
            /* Abort Document (DEVESC_ABORTDOC)             */
            /*----------------------------------------------*/
            DevEscape( pPrnData->hdcPrn, DEVESC_ABORTDOC, (LONG)0, (PBYTE)NULL,
                       (LONG)0, (PBYTE)NULL );

          } /* end else */

        } /* end REQUEST_PRINT_FILE */

        /*----------------------------------------------------*/
        /* Disassociate the PS.                               */
        /*----------------------------------------------------*/
        ErrorID = AssertPM( !GpiAssociate( pPrnData->hpsPrn, (HDC)NULL ),
                             habLcl, ERR_GPIASSOCIATE  );

        /*----------------------------------------------------*/
        /* Destroy the PS.                                    */
        /*----------------------------------------------------*/
        ErrorID = AssertPM( !GpiDestroyPS( pPrnData->hpsPrn ),
                            habLcl, (PSZ)ERR_GPIDESTROYPS );

      } /* end if */ // GpiCreatePS successful

      /*------------------------------------------------------*/
      /* Destroy the device context & get metafile handle     */
      /*------------------------------------------------------*/
      hmfPrn = DevCloseDC( pPrnData->hdcPrn );

      ErrorID = AssertPM( hmfPrn==DEV_ERROR, habLcl, ERR_DEVCLOSEDC );

    } /* end if */

  } /* end if */

  return( ErrorID );

} /* end AccessDevice */


/****************************************************************************/
/* PROCEDURE NAME : SearchForString                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : This function searches for the specified string         */
/*                  starting from the specified line.                       */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Start case sensitive search from current line       */
/*                      which is the first one displayed in the client      */
/*                      window.                                             */
/*                                                                          */
/* PARAMETERS:      (PSZ)   pszSearchStr - string to search for             */
/*                  (ULONG) ulFromLine   - line in file where search should */
/*                                         start.                           */
/*                                                                          */
/* RETURN VALUES:   (ULONG) ulFromLine   - Line of file where search string */
/*                                         found or -1 if not found.        */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/* @MAXLINE  - 11/02/93 - Matt Rutkowski [IBM] - changed SHORT sFromLine to */
/*                        ULONG ulFromLine                                  */
/*                                                                          */
/* @75474    - 11/03/93 - Matt Rutkowski [IBM] - fixed problem with search  */
/*                        on huge files (used to incorrectly use SHORTS)    */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
ULONG SearchForString ( PSZ pszSearchStr, ULONG ulFromLine, PSZ pszPre, PSZ pszMatch )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  CHAR  FAR *pszCurLine;           // points to current search line
  CHAR  FAR *p;                    // temporary pointer
  CHAR      szCurLineCopy[MAXSTR]; // temporary line
  CHAR      *pCurLineCopy;         // temporary pointer

  ULONG     ulSearchLen;
  ULONG     ulLinesRemaining;      // @75474 - count of lines remaining
  BOOL      bCaseSensitive;
  BOOL      bMatch = FALSE;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  ulLinesRemaining = (ulTotalLines-ulFromLine)-1;
  pszCurLine       = pLinePtrs[++ulFromLine];
  ulSearchLen      = strlen(pszSearchStr );
  bCaseSensitive   = PgmData.StateData.bCaseSensitive;

  while ( !bMatch && ulLinesRemaining > 0 )
  {
    // TBD - needs to overcome MAXTSTR limits
    // copy current line into temporary string
    for ( p=pszCurLine, pCurLineCopy = szCurLineCopy; *p != '\n';  )
        *pCurLineCopy++ = *p++;

    // null-terminate the temporary string
    *pCurLineCopy = '\0';

    // Reset current line pointer to start of line
    pCurLineCopy=szCurLineCopy;

    // While no match AND search string is shorter than line length remaining
    while( !bMatch &&
           pCurLineCopy < (szCurLineCopy + strlen(szCurLineCopy) - ulSearchLen ))
    {
      // search for string on current line; return line number if found
      if( bCaseSensitive )
      {
        if ( strncmp ( pCurLineCopy, pszSearchStr, strlen (pszSearchStr)) == 0 )
          bMatch = TRUE;

      } /* end if */
      else
      {
#if STRNICMP
        if ( strnicmp( pCurLineCopy, pszSearchStr, strlen (pszSearchStr)) == 0 )
          bMatch = TRUE;
#else
        if ( strncmp ( pCurLineCopy, pszSearchStr, strlen (pszSearchStr)) == 0 )
          bMatch = TRUE;
#endif

      } /* end else */

      // advance to next substring in line (increment by one character)
      if(!bMatch)
      {
        pCurLineCopy++;

      } /* end if */

    } /* end while */

    // else advance to next line
    if(!bMatch)
    {
      pszCurLine = pLinePtrs [++ulFromLine];
      ulLinesRemaining--;

    } /* end if */

  } /* end while */

  // Search has ended with/without a match
  if(bMatch )
  {
    // return matching string
    strncpy( pszMatch, pCurLineCopy, ulSearchLen );
    pszMatch[ulSearchLen] = '\0';

    // return string appearing before matching string
    *pCurLineCopy = '\0';
    strcpy( pszPre, szCurLineCopy );

    // return line match found on
    return( ulFromLine );

  } /* end if */
  else
  {
    // return -1 if no match found
    return ( -1 );
  } /* end else */

} /* end SearchForString */


/****************************************************************************/
/* PROCEDURE NAME : CalculateCharBox                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 09/28/94                                                */
/* DESCRIPTION    : */
/*                  */
/* PARAMETERS:      */
/*                  */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID CalculateCharBox( ULONG   ulPointSize,
                       LONG    lHorzFontRes,
                       LONG    lVertFontRes,
                       PSIZEF  psfCharBox )
{
   float   fWidth, fHeight, fTemp;
   double  fFract, fInt;

   /*--------------------------*/
   /* Char Box Width           */
   /*--------------------------*/
   fTemp = (float)((SHORT)lHorzFontRes);
   fWidth = (((float)ulPointSize)/TWIPS_CNST ) * fTemp;
   fFract = modf( fWidth, &fInt );

   psfCharBox->cx = MAKEFIXED( ((SHORT)fInt), ((USHORT)fFract) );

   /*--------------------------*/
   /* Char Box Height          */
   /*--------------------------*/
   fTemp = (float)((SHORT)lVertFontRes);
   fHeight = (((float)ulPointSize)/TWIPS_CNST ) * fTemp;
   fFract = modf( fHeight, &fInt );

   psfCharBox->cy = MAKEFIXED( ((SHORT)fInt), ((USHORT)fFract) );

} /* end CalculateCharBox */

/****************************************************************************/
/* PROCEDURE NAME : FontDlgProc                                             */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Controls Font/fontmetrics dialog by responding to all   */
/*                  window messages.                                        */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY FontDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  static       SHORT       MaxPointSize;    // Max allowable pt size of font
  static       SHORT       MinPointSize;    // Min allowable pt size of font
  static       FATTRS      fattrMLE;
  static       HWND        hwndMLE;
  static       HWND        hwndItalic, hwndUnderscore, hwndStrikeout, hwndBold;
  static       USERBUTTON  ubStyle,  ubOwner;
  static       PUSERBUTTON pubStyle, pubOwner;

  PPGMDATA     ppd;
  PFONTDATA    pfdTemp;
  PPRINTERDATA prd;
  PNBPAGEINFO  ppi;
  SHORT        sIdx;
  BOOL         bResetMLE;
  SHORT        id;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      ULONG   ulOpts;
      ERRORID ErrorID;

      // Get program data from mp2 which is set by caller
      ppd = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)ppd );

      // Get program variables in east-to-dereference locals
      prd = &ppd->PrinterData;
      ppi = &(ppd->nbi.PageInfoFont);
      pfdTemp = (PFONTDATA)ppi->pvPageData;

      // Make local copy of font data to be used as state data by
      // dialog code (for font page)
      strcpy( pfdTemp->fm.szFacename, ppd->FontData.fm.szFacename );
      *pfdTemp = ppd->FontData;

      // Set state variable (pointer to current font to NULL)
      pfdTemp->pfmCurrent = NULL;

      // Get some window handles we need throughout procedure
      hwndMLE        = WinWindowFromID( hwnd, IDML_FONTSAMP );
      hwndItalic     = WinWindowFromID( hwnd, IDC_FONT_ITALIC );
      hwndUnderscore = WinWindowFromID( hwnd, IDC_FONT_UNDERSCORE );
      hwndStrikeout  = WinWindowFromID( hwnd, IDC_FONT_STRIKEOUT );
      hwndBold       = WinWindowFromID( hwnd, IDC_FONT_BOLD );

      pubStyle = &ubStyle; pubOwner = &ubOwner;
      pubOwner->hwnd = WinWindowFromID( hwnd, IDD_FONTTYPE_OWNER );
      pubStyle->hwnd = WinWindowFromID( hwnd, IDD_FONTTYPE_STYLE );

      pubOwner->fsState = pubOwner->fsStateOld = BDS_DEFAULT;
      pubStyle->fsState = pubStyle->fsStateOld = BDS_DEFAULT;

      // Set check marks next to user-selected font style options in dialog
      WinCheckButton( hwnd, IDC_FONT_ITALIC,     (pfdTemp->Options & FATTR_SEL_ITALIC)!=FALSE );
      WinCheckButton( hwnd, IDC_FONT_UNDERSCORE, (pfdTemp->Options & FATTR_SEL_UNDERSCORE)!=FALSE );
      WinCheckButton( hwnd, IDC_FONT_STRIKEOUT,  (pfdTemp->Options & FATTR_SEL_STRIKEOUT)!=FALSE );
      WinCheckButton( hwnd, IDC_FONT_BOLD,       (pfdTemp->Options & FATTR_SEL_BOLD)!=FALSE );

      // Perform GpiQueryFonts() call against current printer device
      ErrorID = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_FONTS, DEV_TYPE_PRINTER );

      // Unable to get fonts using printer driver (perhaps no print queues) use display
      if( ErrorID != NO_ERROR )
      {
         ErrorID = AccessDevice( ppd, ppd->habPgm, OD_INFO, REQUEST_QUERY_FONTS, DEV_TYPE_DISPLAY );
      } /* end if */

      // We have font facenames and fontmetrics at this point
      if( ErrorID == NO_ERROR )
      {
        // Fill font facename listbox and see if we have a font match
        if( FillFaceNameListBox( hwnd, ppd, pfdTemp ) )
        {
          // Font was matched while filling listbox (name and lmatch)
          // Now set font's point size as user wanted NOT the nominal point
          // size set by our selecting a font facename from combobox.
          pfdTemp->PointSize = ppd->FontData.PointSize;

          // If FONT_MATCH Display stored pointsize overriding nominal
          WinSendDlgItemMsg( hwnd,
                             IDE_POINTSIZE,
                             SPBM_SETCURRENTVALUE,
                             MPFROMLONG(pfdTemp->PointSize),
                             NULL );
        } /* end if */

      } /* end if */

      // fill in the FATTRS used by the "sample text" MLE
      fattrMLE.usRecordLength  = sizeof(FATTRS);
      fattrMLE.fsSelection     = pfdTemp->fm.fsSelection | pfdTemp->Options;
      fattrMLE.lMatch          = pfdTemp->fm.lMatch;
      fattrMLE.idRegistry      = 0;
      fattrMLE.lMaxBaselineExt = 0L;
      fattrMLE.lAveCharWidth   = 0L;
      fattrMLE.fsType          = 0;
      // @DBCS - set codepage to system codepage so font realized
      // with correct codepage (not font's default codepage
      // (i.e. not 0 )
      fattrMLE.usCodePage      = CodePage;
      fattrMLE.fsFontUse       = FATTR_FONTUSE_NOMIX;
      strcpy( fattrMLE.szFacename, pfdTemp->fm.szFacename );
      WinSendMsg( hwndMLE, MLM_SETFONT, (MPARAM)&fattrMLE, MPVOID );

      return(0);

    } /* end case */
    break;

    case WM_CONTROL:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );
      prd = &ppd->PrinterData;
      ppi = &(ppd->nbi.PageInfoFont);
      pfdTemp = (PFONTDATA)ppi->pvPageData;
      bResetMLE = FALSE;

      switch( id = SHORT1FROMMP(mp1) )
      {
        case IDD_FONTTYPE_OWNER:
        case IDD_FONTTYPE_STYLE:
        {
          switch( SHORT2FROMMP(mp1) )
          {
            case BN_PAINT:
            {
               PUSERBUTTON pub;
               OWNERITEM   oi;

               pub = (PUSERBUTTON) PVOIDFROMMP(mp2); /* pointer to OWNERITEM */

               if( id == IDD_FONTTYPE_STYLE )
               {
                 if( pfdTemp->FontScalable )
                   id = ID_FONTATM;
                 else
                   id = ID_FONTBMP;

               } /* end if */
               else
               {
                 if( id == IDD_FONTTYPE_OWNER )
                 {
                   if(pfdTemp->pfmCurrent)
                   {
                     if( pfdTemp->pfmCurrent->lMatch < 0 )
                     {
                       id = ID_FONTPRN;
                     } /* end if */
                     else
                     {
                       if( pfdTemp->pfmCurrent->sXDeviceRes != 1000 ||
                           pfdTemp->pfmCurrent->sYDeviceRes != 1000 )
                       {
                         id = ID_FONTDSP;
                       } /* end if */
                       else
                       {
                         id = ID_FONTSYS;
                       } /* end else */

                     } /* end else */

                   } /* end if */
                   else
                     id= ID_FONTSYS;

                 } /* end if */

               } /* end else */

               oi.hwnd       = pub->hwnd;
               oi.hps        = pub->hps;
               oi.idItem     = id;
               oi.fsState    = pub->fsState;
               oi.fsStateOld = pub->fsStateOld;

               WinQueryWindowRect( oi.hwnd, &(oi.rclItem) );

               Draw3DButton( hMenu3d, &oi, id, BUTTON_ACTION_DRAW );

               return( 0 );

            } /* end case */
            break;

          } /* end switch */

        } /* end case */
        break;

        case IDL_FACENAME:
        {
          // if an item was selected from the listbox
          if( SHORT2FROMMP(mp1) == CBN_LBSELECT )
          {
            // Mark flag to indicate redraw of MLE
            bResetMLE = TRUE;

            // Get index of facename selected by user
            sIdx = SHORT1FROMMR( WinSendDlgItemMsg( hwnd,
                                                    IDL_FACENAME,
                                                    LM_QUERYSELECTION,
                                                    0L,0L) );
            // @LBOX - verify user selected an item
            if( sIdx != LIT_NONE )
            {
              // Get Item Handle (STRING ID)
              pfdTemp->pfmCurrent = (PFONTMETRICS)WinSendDlgItemMsg( hwnd,
                                                     IDL_FACENAME,
                                                     LM_QUERYITEMHANDLE,
                                                     MPFROMSHORT( sIdx ),
                                                     MPVOID );

              FillFontMetricsListBox( hwnd, IDL_FONTMETRICS,
                                      pfdTemp->pfmCurrent,
                                      &MinPointSize, &MaxPointSize );

              pfdTemp->PointSize = pfdTemp->pfmCurrent->sNominalPointSize/10;

              // FATTRS for MLE
              fattrMLE.lMatch      = pfdTemp->pfmCurrent->lMatch;
              fattrMLE.fsSelection = pfdTemp->pfmCurrent->fsSelection | pfdTemp->Options;
              strcpy( fattrMLE.szFacename, pfdTemp->pfmCurrent->szFacename );

              if( pfdTemp->pfmCurrent->fsDefn & FM_DEFN_OUTLINE )
              {
                WinSetDlgItemText( hwnd, IDE_OUTLINE, SCALABLE_MSG );

                /* Set Spin Button to font's point size limits */
                WinSendDlgItemMsg( hwnd,
                                   IDE_POINTSIZE,
                                   SPBM_SETLIMITS,
                                   MPFROMLONG((LONG)MaxPointSize/10),
                                   MPFROMLONG((LONG)MinPointSize/10));

                pfdTemp->FontScalable = TRUE;

              } /* end if */
              else
              {
                WinSetDlgItemText( hwnd, IDE_OUTLINE, FIXED_MSG );

                /* Set Spin Button to font's point size limits */
                WinSendDlgItemMsg( hwnd,
                                   IDE_POINTSIZE,
                                   SPBM_SETLIMITS,
                                   MPFROMLONG((LONG)pfdTemp->PointSize),
                                   MPFROMLONG((LONG)pfdTemp->PointSize));

                pfdTemp->FontScalable = FALSE;

              } /* end else */

              /* show pointsize of currently selected font */
              WinSendDlgItemMsg( hwnd,
                                 IDE_POINTSIZE,
                                 SPBM_SETCURRENTVALUE,
                                 MPFROMLONG((LONG)pfdTemp->PointSize),
                                 NULL );

              WinEnableControl( hwnd, IDE_POINTSIZE,
                                pfdTemp->pfmCurrent->fsDefn & FM_DEFN_OUTLINE );
            } /* end if */

            // Set "Font Type" buttons to appropriate graphic symbols
            pubOwner->hps = WinGetPS( pubOwner->hwnd );
            WinSendMsg( hwnd, WM_CONTROL,
                        MPFROM2SHORT( IDD_FONTTYPE_OWNER, BN_PAINT ),
                        MPFROMP( pubOwner ) );
            WinReleasePS( pubOwner->hps );

            pubStyle->hps = WinGetPS( pubStyle->hwnd );
            WinSendMsg( hwnd, WM_CONTROL,
                        MPFROM2SHORT( IDD_FONTTYPE_STYLE, BN_PAINT ),
                        MPFROMP( pubStyle ) );
            WinReleasePS( pubStyle->hps );

          } /* end if */

        } /* end case */
        break;

        case IDC_FONT_ITALIC :
        {
          bResetMLE = TRUE;

          if( pfdTemp->Options & FATTR_SEL_ITALIC )
          {
            WinSendMsg( hwndItalic, BM_SETCHECK, MPFROMSHORT(FALSE), MPVOID );
            pfdTemp->Options &= ~FATTR_SEL_ITALIC;
          } /* end if */
          else
          {
            WinSendMsg( hwndItalic, BM_SETCHECK, MPFROMSHORT(TRUE), MPVOID );
            pfdTemp->Options |= FATTR_SEL_ITALIC;
          } /* end else */

          fattrMLE.fsSelection = pfdTemp->Options;

        } /* end case */
        break;

        case IDC_FONT_UNDERSCORE :
        {
          bResetMLE = TRUE;

          if( pfdTemp->Options & FATTR_SEL_UNDERSCORE )
          {
            WinSendMsg( hwndUnderscore, BM_SETCHECK, MPFROMSHORT(FALSE), MPVOID );
            pfdTemp->Options &= ~FATTR_SEL_UNDERSCORE;
          } /* end if */
          else
          {
            WinSendMsg( hwndUnderscore, BM_SETCHECK, MPFROMSHORT(TRUE), MPVOID );
            pfdTemp->Options |= FATTR_SEL_UNDERSCORE;
          } /* end else */

          fattrMLE.fsSelection = pfdTemp->Options;

        } /* end case */
        break;

        case IDC_FONT_STRIKEOUT :
        {
          bResetMLE = TRUE;

          if( pfdTemp->Options & FATTR_SEL_STRIKEOUT )
          {
            WinSendMsg( hwndStrikeout, BM_SETCHECK, MPFROMSHORT(FALSE), MPVOID );
            pfdTemp->Options &= ~FATTR_SEL_STRIKEOUT;
          } /* end if */
          else
          {
            WinSendMsg( hwndStrikeout, BM_SETCHECK, MPFROMSHORT(TRUE), MPVOID );
            pfdTemp->Options |= FATTR_SEL_STRIKEOUT;
          } /* end else */

          fattrMLE.fsSelection = pfdTemp->Options;

        } /* end case */
        break;

        case IDC_FONT_BOLD :
        {
          bResetMLE = TRUE;

          if( pfdTemp->Options & FATTR_SEL_BOLD )
          {
            WinSendMsg( hwndBold, BM_SETCHECK, MPFROMSHORT(FALSE), MPVOID );
            pfdTemp->Options &= ~FATTR_SEL_BOLD;
          } /* end if */
          else
          {
            WinSendMsg( hwndBold, BM_SETCHECK, MPFROMSHORT(TRUE), MPVOID );
            pfdTemp->Options |= FATTR_SEL_BOLD;
          } /* end else */

          fattrMLE.fsSelection = pfdTemp->Options;

        } /* end case */
        break;

      } /* end switch */

      // Currently we want to update MLE with new font and/or
      // changed font attributes as all WM_CONTROL msgs for
      // the font dailog affect font appearance
      if( bResetMLE )
      {
        WinPostMsg( hwndMLE, MLM_SETFONT, (MPARAM)&fattrMLE, MPVOID );
      } /* end if */

    } /* end case */
    break;

    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case IDD_FONTTYPE_OWNER:
        case IDD_FONTTYPE_STYLE:
        case DID_CANCEL:
        {
          return(0);

        } /* end case */

      } /* end switch */

    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end FontDlgProc */


/****************************************************************************/
/* PROCEDURE NAME : MetricsDlgProc                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/3/93                                                  */
/* DESCRIPTION    : Displays Fontmetrics of currently selected font being   */
/*                  used by Font Test to display Text.                      */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY MetricsDlgProc ( HWND   hwnd,
                                  ULONG  msg,
                                  MPARAM mp1,
                                  MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PPGMDATA  pPgmData;


  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      SHORT     MinPointSize, MaxPointSize;  // Dummy vars for FillFontMetricsListBox

      // Get program data from mp2 which is set by caller
      pPgmData = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)pPgmData );

      // Fill listbox with font metrics (DISPLAY)
      FillFontMetricsListBox( hwnd, IDL_DSP_FONTMETRICS, &(pPgmData->DisplayData.fm), &MinPointSize, &MaxPointSize );

      // Fill listbox with font metrics (PRINTER)
      FillFontMetricsListBox( hwnd, IDL_PRN_FONTMETRICS, pPgmData->PrinterData.pFontMetrics, &MinPointSize, &MaxPointSize );

      // return zero since we handled this call
      return(0);

    } /* end case */

    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case DID_OK:
             WinDismissDlg( hwnd, DID_OK );
             return(0);
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP:
    {
       pPgmData = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

       if( pPgmData->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_FNTMETRIC ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, pPgmData->habPgm, ERR_NO_HELP );
    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end MetricsDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : FontWidthDlgProc                                        */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10/27/94                                                */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY FontWidthDlgProc ( HWND   hwnd,
                                    ULONG  msg,
                                    MPARAM mp1,
                                    MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PPGMDATA     pPgmData;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      PLONG        pWidths;
      ULONG        ul;
      HWND         hwndLB = WinWindowFromID( hwnd, IDL_FONT_WIDTHS );
      CHAR         szBuffer[MAXSTR];
      PFONTMETRICS pfm;
      BOOL         bSuccess;
      USHORT       usNumWidths ;

      pPgmData = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)pPgmData );

      pWidths = pPgmData->PrinterData.pFontWidthTable;
      pfm     = pPgmData->PrinterData.pFontMetrics;

      usNumWidths = (((USHORT)pfm->sLastChar) - ((USHORT)pfm->sFirstChar)) + 1;

      // NOTE: the call will not be able to return widths for entries
      // greater than 256 in most code pages
      usNumWidths = min( 256, usNumWidths );


      // attempt to allocate it
      if( DosAllocMem( (PPVOID)&pWidths, sizeof(LONG)*usNumWidths,
                   PAG_READ | PAG_WRITE | PAG_COMMIT ) == NO_ERROR )
      {

        bSuccess = GpiQueryWidthTable( pPgmData->PrinterData.hpsPrn, (LONG)pfm->sFirstChar, usNumWidths, pWidths );

        // check if Font query width table failed
        if( bSuccess != GPI_ERROR )
        {
          WinSendDlgItemMsg( hwnd, IDL_FONT_WIDTHS, LM_DELETEALL, NULL, NULL );

          sprintf( szBuffer, "Font Name = %s",  pfm->szFacename );
          WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

          if(pfm->fsType & FM_TYPE_DBCS)
          {
            WinInsertLboxItem( hwndLB, LIT_END, ">> This is a DBCS Font" );

          } /* end if */

          sprintf( szBuffer, ">> # of Font Widths = %d",  usNumWidths );
          WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

          sprintf( szBuffer, ">> First Char = %d", pfm->sFirstChar  );
          WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

          sprintf( szBuffer, ">> Last Char = %d", pfm->sLastChar  );
          WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

          for( ul = 0; ul <= usNumWidths; ul++ )
          {
            switch( pWidths[ul] )
            {
              case EOS:
                sprintf( szBuffer, "Character[%3d] (NULL), Width = %d", ul, pWidths[ul] );
              break;

              case CR:
                sprintf( szBuffer, "Character[%3d] (CR), Width = %d", ul, pWidths[ul] );
              break;

              case LF:
                sprintf( szBuffer, "Character[%3d] (LF), Width = %d", ul, pWidths[ul] );
              break;

              default:
                sprintf( szBuffer, "Character[%3d] (%c), Width = %d", ul, (pfm->sFirstChar+ul), pWidths[ul] );

            } /* end switch */

            WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

          } /* end for */

        } /* end if */
        else
        {
          AssertPM( TRUE, pPgmData->habPgm, ERR_GPIQWIDTHTABLE );

        } /* end else */

        /* free the memory used to store font */
        DosFreeMem( pPgmData->PrinterData.pFontWidthTable );

      } /* end if */

      return(0);

    } /* end case */

    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case DID_OK:
             WinDismissDlg( hwnd, DID_OK );
             return(0);
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP: // TEST !!!!! BAD!!! CHANGE THIS PANEL!!!!
    {
       pPgmData = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

       if( pPgmData->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_FNTWDTH ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, pPgmData->habPgm, ERR_NO_HELP );
    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end FontWidthDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : FontKernDlgProc                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10/27/94                                                */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY FontKernDlgProc ( HWND   hwnd,
                                   ULONG  msg,
                                   MPARAM mp1,
                                   MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PPGMDATA      pPgmData;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      PFONTMETRICS  pfm;
      PKERNINGPAIRS pKernPairs;
      ULONG         ul;
      HWND          hwndLB = WinWindowFromID( hwnd, IDL_FONT_KERN_PAIRS );
      CHAR          szBuffer[MAXSTR];

      pPgmData = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)pPgmData );

      WinSendDlgItemMsg( hwnd, IDL_FONT_KERN_PAIRS, LM_DELETEALL, NULL, NULL );

      pKernPairs = pPgmData->PrinterData.pFontKernPairs;
      pfm        = pPgmData->PrinterData.pFontMetrics;

      sprintf( szBuffer, "Font Name = %s",  pfm->szFacename );
      WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

      for( ul = 0; ul < pfm->sKerningPairs; ul++ )
      {
        sprintf( szBuffer, "Kerning Pair[%3d], (%c%c),  Amt = %d",
                 ul,
                 pKernPairs[ul].sFirstChar,
                 pKernPairs[ul].sSecondChar,
                 pKernPairs[ul].lKerningAmount );

        WinInsertLboxItem( hwndLB, LIT_END, szBuffer );

      } /* end for */

      return(0);

    } /* end case */

    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case DID_OK:
             WinDismissDlg( hwnd, DID_OK );
             return(0);
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP: // TEST !!!!! BAD!!! CHANGE THIS PANEL!!!!
    {
       pPgmData = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

       if( pPgmData->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_FNTKERN ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, pPgmData->habPgm, ERR_NO_HELP );
    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end FontKernDlgProc */


/****************************************************************************/
/* PROCEDURE NAME : ConfirmDlgProc                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Verifies that the user really wishes to print.          */
/*                  This dialog shows all information that will be used     */
/*                  to print the file.  It also allows user to alter job    */
/*                  properties one last time before printing it.            */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY ConfirmDlgProc ( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{

  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PPGMDATA     ppd;
  PPRINTERDATA prd;
  PQUEUEDATA   pqd;
  PCOLORDATA   pcd;
  PFORMATDATA  ptd;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      CHAR szBuffer[MAXSTR];

      // Get program data from mp2 which is set by caller
      ppd = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)ppd );

      // Get program variables in easy-to-dereference locals
      prd = &ppd->PrinterData;

      // Set text limit for entry fields that need it
      WinSendDlgItemMsg( hwnd, IDE_PRNTFILE, EM_SETTEXTLIMIT,
                         MPFROMSHORT( MAXSTR ), MPVOID );

      WinSendDlgItemMsg( hwnd, IDE_QUEUE_DESC, EM_SETTEXTLIMIT,
                         MPFROMSHORT( MAXSTR ), MPVOID );

      // display file name on dlg
      WinSetDlgItemText( hwnd, IDE_PRNTFILE, filedlg.szFullFile );

      // display Queue name on dlg
      WinSetDlgItemText( hwnd, IDE_QUEUE, ppd->QueueData.QueueName );

      // display Queue desc on dlg
      WinSetDlgItemText( hwnd, IDE_QUEUE_DESC, ppd->QueueData.QueueDesc );

      // display driver name on dlg
      WinSetDlgItemText( hwnd, IDE_DRIVER, prd->szDriverName );

      // display device name on dlg
      WinSetDlgItemText( hwnd, IDE_DEVICE, prd->szDeviceName );

      // display font facename (forgot what driver added semi-colon but one did)
      StrToken( szBuffer, ppd->FontData.fm.szFacename, ';', 1 );
      WinSetDlgItemText( hwnd, IDE_FONT, szBuffer );

      // display font point size
      _ltoa( ppd->FontData.PointSize, szBuffer, 10 );
      WinSetDlgItemText( hwnd, IDE_PTSIZE, szBuffer );

      return(0);

    } /* end case */
    break;

    case WM_CONTROL:
    {
      SHORT id;

      switch( id = SHORT1FROMMP(mp1) )
      {
        case IDP_SETTINGS:
        {
          switch( SHORT2FROMMP(mp1) )
          {
            case BN_PAINT:
            {
               PUSERBUTTON pub;
               OWNERITEM   oi;
               CHAR        sz[256];

               pub = (PUSERBUTTON) PVOIDFROMMP(mp2); /* pointer to OWNERITEM */
               oi.hwnd       = pub->hwnd;
               oi.hps        = pub->hps;
               oi.idItem     = id;
               oi.fsState    = pub->fsState;
               oi.fsStateOld = pub->fsStateOld;

               WinQueryWindowRect( oi.hwnd, &(oi.rclItem) );

               Draw3DButton( hMenu3d, &oi, id, BUTTON_ACTION_DRAW );

               return( 0 );

            } /* end case */
            break;

          } /* end switch */

        } /* end case */
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_COMMAND:
    {
      ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

      switch( SHORT1FROMMP(mp1) )
      {
        case IDP_SETTINGS:
        {
          LONG                 rc;

          ppd->hwndMain = WinLoadDlg( HWND_DESKTOP,
                                      WinQueryActiveWindow(HWND_DESKTOP),
                                      (PFNWP)ProgramSettingsDlgProc,
                                      (HMODULE)NULL,
                                      IDD_PROGRAM_SETTINGS,
                                      (PVOID)ppd );

          WinProcessDlg( ppd->hwndMain );

          // must destroy window ourselves for WinProcessDlg() loaded
          // dialogs or all close msgs will not be sent
          WinDestroyWindow( ppd->hwndMain );

          if( rc !=DID_CANCEL )
          {
            // invalidate entire client window to apply changes made
            WinInvalidateRect ( hwndClient, NULL, TRUE );

            // Tell ourselves to change the currently displayed font
            WinPostQueueMsg( ppd->hmqPgm,
                             (USHORT)WM_CHANGE_FONT,
                             MPVOID, MPVOID );
          } /* end if */

          return( 0 );


        } /* end case */
        break;

        case DID_OK:
             WinDismissDlg( hwnd, TRUE );
             return(0);
        break;
        case DID_CANCEL:
             WinDismissDlg( hwnd, FALSE );
             return(0);
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP:
    {
       ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

       if( ppd->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_CONFIRM ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end ConfirmDlgProc */


/****************************************************************************/
/* PROCEDURE NAME : AboutDlgProc                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Displays a text dialog giving a brief description       */
/*                  of app and author/version info.                         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY AboutDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case DID_OK:
             WinDismissDlg( hwnd, DID_OK );
             return(0);
        break;
      } /* end switch */

    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end AboutDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : QueueDlgProc                                            */
/* AUTHOR         : Matt R.                                                 */
/* DATE WRITTEN   : 4/06/93                                                 */
/* DESCRIPTION    : Dialog used to display available (LOCAL and REMOTE)     */
/*                  device queues for user selection.                       */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY QueueDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  SHORT   sSelect;
  CHAR    szBuffer[MAXSTR];
  CHAR    szBuffer2[MAXSTR];
  USHORT  usQueuesFound;
  MRESULT mr;
  USHORT  id;

  PPGMDATA      ppd;
  PQUEUEDATA    pqd;
  PNBPAGEINFO   ppi;
  PNOTEBOOKINFO pnb;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_CLOSE:
    {
      DosFreeMem( pqd->pDrivData );
      pqd->pDrivData = (PDRIVDATA)NULL;
    } /* end case */
    break;

    case WM_INITDLG:
    {
      // Get program data from mp2 which is set by caller
      ppd = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)ppd );

      // allocate temp copy of queue data for scratch work while
      // user in notebook playing around
      DosAllocMem( (PPVOID)&(ppd->nbi.PageInfoPrinter.pvPageData),
                    sizeof(QUEUEDATA),
                    PAG_READ | PAG_WRITE | PAG_COMMIT );

      ppi = &(ppd->nbi.PageInfoPrinter);
      pqd = (PQUEUEDATA)ppi->pvPageData;

      // Fill dialog listbox with all queue names in system
      usQueuesFound = FillQueNameListBox( ppd->habPgm, hwnd, &(ppd->QueueData));

      // Inform user no queues were found in system
      if( usQueuesFound == 0 )
      {
         // Disable buttons and controls on queue notebook page
         WinEnableWindow( WinWindowFromID(hwnd,IDCB_PRINT_QUEUES), FALSE );
         WinEnableWindow( WinWindowFromID(hwnd,IDP_JOB_PROPERTIES), FALSE );

         // Set Printer Driver and Printer Description to "None"
         WinSetWindowText( WinWindowFromID(hwnd,IDT_PRINTER_DRIVER),
                           NONE_AVAILABLE);
         WinSetWindowText( WinWindowFromID(hwnd,IDT_PRINTER_DESCRIPTION),
                           NONE_AVAILABLE);

         // Initialize temporary queue data to NULL
         pqd->QueueName[0] = '\0';
         pqd->QueueDesc[0] = '\0';

      } /* end else */

      return(0);

    } /* end case */
    break;

    case WM_CONTROL:
    {
      switch( id = SHORT1FROMMP(mp1) )
      {
        case IDP_JOB_PROPERTIES:
        case IDP_OTHER_DESTINATION:
        {
          switch( SHORT2FROMMP(mp1) )
          {
            case BN_PAINT:
            {
               PUSERBUTTON pub;
               OWNERITEM   oi;

               /* pointer to OWNERITEM */
               pub = (PUSERBUTTON) PVOIDFROMMP(mp2);
               oi.hwnd       = pub->hwnd;
               oi.hps        = pub->hps;
               oi.idItem     = id;
               oi.fsState    = pub->fsState;
               oi.fsStateOld = pub->fsStateOld;

               WinQueryWindowRect( oi.hwnd, &(oi.rclItem) );
               Draw3DButton( hMenu3d, &oi, id, BUTTON_ACTION_DRAW );

               return( 0 );

            } /* end case */
            break;

          } /* end switch */

        } /* end case */
        break;

        case IDCB_PRINT_QUEUES:
        {
          if( SHORT2FROMMP(mp1) == CBN_LBSELECT )
          {
            // Setup program's Queue Data
            ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );
            ppi = &(ppd->nbi.PageInfoPrinter);
            pqd = (PQUEUEDATA)ppi->pvPageData;

            mr = WinSendDlgItemMsg( hwnd, SHORT1FROMMP(mp1),
                                    LM_QUERYSELECTION, 0L, 0L );

            sSelect = SHORT1FROMMR( mr );

            WinSendDlgItemMsg( hwnd,
                               SHORT1FROMMP(mp1),
                               LM_QUERYITEMTEXT,
                               MPFROM2SHORT(sSelect, sizeof(szBuffer)),
                               MPFROMP(szBuffer) );

            // remember queue description selected
            StrToken( szBuffer2, szBuffer, LEAD_QUEUE_TOKEN, 1 );
            strcpy( pqd->QueueDesc, szBuffer2 );

            // remember queue name selected
            StrToken( szBuffer2, szBuffer, LEAD_QUEUE_TOKEN, 2 );
            StrToken( szBuffer, szBuffer2, TRAIL_QUEUE_TOKEN, 1 );

            strcpy( pqd->QueueName, szBuffer );

            if( pqd->QueueName[0] != '\0' )
            {
               APIRET apiret;

               // get new queue data and new job property data from that queue
               apiret = GetQueueData( ppd->habPgm, pqd, &(ppd->PrinterData), TRUE );

               if( apiret == NO_ERROR )
               {
                 // Show name of driver and device associated with this queue
                 WinSetDlgItemText( hwnd, IDT_PRINTER_DRIVER,
                                    ppd->PrinterData.szDriverName );
                 WinSetDlgItemText( hwnd, IDT_PRINTER_DESCRIPTION,
                                    ppd->PrinterData.szDeviceName  );
               } /* end if */

            } /* end if */

          } /* end if */

        } /* end case */
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_COMMAND:
    {
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );
      ppi = &(ppd->nbi.PageInfoPrinter);
      pqd = (PQUEUEDATA)ppi->pvPageData;

      switch( id = SHORT1FROMMP(mp1) )
      {
        case DID_OK:
        case DID_SAVE:
        {
          /********************************************/
          /* Update QUEUE Data                        */
          /********************************************/

          // Copy Dialog Page's Queue data to program's queue data
          strcpy( ppd->QueueData.QueueName, pqd->QueueName );
          strcpy( ppd->QueueData.QueueDesc, pqd->QueueDesc );

          // Save Job Property data in case user cancel after making a change
          if( ppd->QueueData.pDrivData )
          {
            DosFreeMem( ppd->QueueData.pDrivData );
            ppd->QueueData.pDrivData = NULL;

          } /* end if */

          DosAllocMem( (PPVOID)&(ppd->QueueData.pDrivData),
                        pqd->pDrivData->cb,
                        PAG_READ | PAG_WRITE | PAG_COMMIT );

          memcpy( ppd->QueueData.pDrivData, pqd->pDrivData, pqd->pDrivData->cb );

          // Update INI files
          if( id == DID_SAVE )
          {
            WriteIniFileData( ppd, FT_KEYID_QUEUE  );

          } /* end if */

          WinDismissDlg( hwnd, id );

          return(0);

        } /* end case */
        break;

        case DID_CANCEL:
        {
          WinDismissDlg( hwnd, DID_CANCEL );
          return(0);

        } /* end case */

        case IDP_OTHER_DESTINATION:
        {
          // User wants to print to a queue not listed
          // This could result from network queues that are
          // defined and are attached to but are not enumerable
          // thru OS/2 APIs. (e.g. Novell)
          WinDlgBox( HWND_DESKTOP,
                     hwnd,
                     (PFNWP)UnlistedQueueDialogProc,
                     (HMODULE)NULL,
                     (ULONG)IDD_OTHER_DESTINATION,
                     (PVOID)ppd );

          return(0);

        } /* end case */
        break;

        case IDP_JOB_PROPERTIES:
        {
          APIRET    rc;
          LONG      lJobPropLength = 0;

          // Get size of Job Property data for current queue's printer
          lJobPropLength = DevPostDeviceModes( ppd->habPgm,
                                               NULL,
                                               ppd->PrinterData.szDriverName,
                                               ppd->PrinterData.szDeviceName,
                                               ppd->PrinterData.szPrinterName,
                                               DPDM_QUERYJOBPROP );

          // If we have Job Property data from a previous call
          if( pqd->pDrivData )
          {
            // If size of Job Property data stored is different than current
            if( lJobPropLength != pqd->pDrivData->cb )
            {
              // free the old Job Property data in favor of new
              DosFreeMem( pqd->pDrivData );
              pqd->pDrivData = NULL;

              // allocate for correct Driver data size as reported by driver
              DosAllocMem( (PPVOID)&(pqd->pDrivData),
                            lJobPropLength,
                            PAG_READ | PAG_WRITE | PAG_COMMIT );

              pqd->pDrivData->cb = lJobPropLength;

            } /* end if */
          } /* end if */
          else
          {
            // allocate new data area to hold driver data
            DosAllocMem( (PPVOID)&(pqd->pDrivData),
                          lJobPropLength,
                          PAG_READ | PAG_WRITE | PAG_COMMIT );

            pqd->pDrivData->cb = lJobPropLength;

          } /* end else */

          // Post default job props and allow user to modify
          rc = DevPostDeviceModes( ppd->habPgm,
                                   pqd->pDrivData,
                                   ppd->PrinterData.szDriverName,
                                   ppd->PrinterData.szDeviceName,
                                   ppd->PrinterData.szPrinterName,
                                   DPDM_POSTJOBPROP );

          return(0);

        } /* end case  */

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP:
    {
       ppd = (PPGMDATA) WinQueryWindowPtr( hwnd, QWL_USER );

       if( ppd->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_QUEUE ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end QueueDlgProc */


/****************************************************************************/
/* PROCEDURE NAME : InitNotebook                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 08/15/94                                                */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID InitNotebook( HWND hwnd, HMENU3D hMenu3d, PNOTEBOOKINFO pnbi, ULONG ulSubMenuID )
{

  ULONG ulMaxTextWidth,  ulMaxBmpWidth;
  ULONG ulMaxTextHeight, ulMaxBmpHeight;


  // Initialize notebook structure
  pnbi->pLastPageAdded = NULL;
  pnbi->pPageShowing   = NULL;

  //-----------------------//
  // Set Tab Size (cx,cy)  //
  //-----------------------//

  // calculate what size tabs should be
  CalcMaxTextWidth ( hMenu3d, ulSubMenuID, &ulMaxTextWidth,  BUTTON_SIZE_TEXT );
  CalcMaxTextWidth ( hMenu3d, ulSubMenuID, &ulMaxBmpWidth,   BUTTON_SIZE_BITMAP );
  CalcMaxTextHeight( hMenu3d, ulSubMenuID, &ulMaxTextHeight, BUTTON_SIZE_TEXT );
  CalcMaxTextHeight( hMenu3d, ulSubMenuID, &ulMaxBmpHeight,  BUTTON_SIZE_BITMAP );

  // Add Fudge for incorrect calc of spaces in system prop fonts
  ulMaxTextWidth += 24;

  WinSendDlgItemMsg( hwnd, pnbi->ulNBID, BKM_SETDIMENSIONS,
                     MPFROM2SHORT((SHORT)(ulMaxTextWidth+ulMaxBmpWidth),
                                  (SHORT)(ulMaxTextHeight+ulMaxBmpHeight)),
                     MPFROMSHORT(BKA_MAJORTAB));

  WinSendDlgItemMsg( hwnd, pnbi->ulNBID, BKM_SETDIMENSIONS,
                     MPFROM2SHORT((short)(ulMaxTextHeight*2),
                                  (short)ulMaxTextHeight),
                     MPFROMSHORT(BKA_PAGEBUTTON));

} /* end InitNotebook */

/****************************************************************************/
/* PROCEDURE NAME : InsertNotebookPage                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 05/25/94 !!!!                                           */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID InsertNotebookPage( HWND hwnd,
                         PPGMDATA ppd,
                         PNOTEBOOKINFO pnbi,
                         PNBPAGEINFO   pCurrentPage,
                         ULONG ulDlgID,
                         PFNWP pfnDlg,
                         PSZ   pszText,
                         ULONG ulTabID,
                         ULONG ulPageStyle,
                         ULONG ulPageOrder )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  MRESULT mr;
  ULONG   ulPageID;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  pCurrentPage->ulDlgID = ulDlgID;
  pCurrentPage->pfnDlg  = pfnDlg;
  pCurrentPage->ulTabID = ulTabID;
  pCurrentPage->pszStatusLine = pszText;

  pCurrentPage->hwndPage = WinLoadDlg( hwnd,
                                       hwnd,
                                       (PFNWP)pfnDlg,
                                       (HMODULE)NULL,
                                       ulDlgID,
                                       (PVOID)ppd );

  if(pnbi->pLastPageAdded)
  {
    ulPageID = pnbi->pLastPageAdded->ulPageID;
  } /* end if */
  else
  {
    ulPageID = 0;
  } /* end else */

  pnbi->pLastPageAdded = pCurrentPage;

  // Add notebook page using style/order flags passed in
  mr = WinSendDlgItemMsg( hwnd, pnbi->ulNBID, BKM_INSERTPAGE,
                          MPFROMLONG(ulPageID),
                          MPFROM2SHORT( BKA_AUTOPAGESIZE | BKA_STATUSTEXTON |
                                        ulPageStyle, ulPageOrder));


  pCurrentPage->ulPageID = LONGFROMMR( mr );

  //-----------------------//
  // Set Page Status Text  //
  //-----------------------//
  WinSendDlgItemMsg( hwnd, pnbi->ulNBID, BKM_SETSTATUSLINETEXT,
                     MPFROMLONG(pCurrentPage->ulPageID),
                     MPFROMP( pszText ));

  //-----------------------//
  // Set Page Data         //
  //-----------------------//
  WinSendDlgItemMsg( hwnd, pnbi->ulNBID, BKM_SETPAGEDATA,
                     MPFROMLONG(pCurrentPage->ulPageID),
                     MPFROMP(pCurrentPage));

  WinSendMsg( pnbi->hwndNB, BKM_SETPAGEWINDOWHWND,
              MPFROMLONG( pCurrentPage->ulPageID ),
              MPFROMHWND( pCurrentPage->hwndPage ) );

} /* end InsertNotebookPage */


/****************************************************************************/
/* PROCEDURE NAME : CreateNotebookPages                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 05/25/94 !!!!                                           */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID CreateNotebookPages( HWND hwnd, HMENU3D hMenu3d, PPGMDATA ppd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PNBPAGEINFO   pCurrentPage;
  PNOTEBOOKINFO pNBI;
  ULONG         ulIdx;
  BOOL          bRC;
  TID           TidFont;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // easier pointers to dereference
  pNBI = &ppd->nbi;

  // Set up notebook button sizes/colors/fonts etc
  InitNotebook( hwnd, hMenu3d, pNBI, 6 );

  //--------------------//
  // "Font" Page        //
  //--------------------//

  // allocate temp copy of font data for scratch work while
  // user in notebook playing around
  DosAllocMem( (PPVOID)&(pNBI->PageInfoFont.pvPageData),
                sizeof(FONTDATA),
                PAG_READ | PAG_WRITE | PAG_COMMIT );

  InsertNotebookPage( hwnd,
                      ppd,
                      pNBI,
                      &(pNBI->PageInfoFont),
                      IDD_FONT,
                      FontDlgProc,
                      "Select Program's Font",
                      IDM_SELECT_FONT,
                      BKA_MAJOR,
                      BKA_FIRST );

  //--------------------//
  // "Color" Page       //
  //--------------------//

  // allocate temp copy of color data for scratch work while
  // user in notebook playing around
  DosAllocMem( (PPVOID)&(pNBI->PageInfoColors.pvPageData),
                sizeof(COLORDATA),
                PAG_READ | PAG_WRITE | PAG_COMMIT );

  InsertNotebookPage( hwnd,
                      ppd,
                      pNBI,
                      &(pNBI->PageInfoColors),
                      IDD_SELECT_COLORS,
                      ColorsDlgProc,
                      "Select Program's Colors",
                      IDM_SELECT_COLORS,
                      BKA_MAJOR,
                      BKA_NEXT );

  //--------------------//
  // "Page Format" Page //
  //--------------------//

  // allocate temp copy of format data for scratch work while
  // user in notebook playing around
  DosAllocMem( (PPVOID)&(pNBI->PageInfoFormat.pvPageData),
                sizeof(FORMATDATA),
                PAG_READ | PAG_WRITE | PAG_COMMIT );

  InsertNotebookPage( hwnd,
                      ppd,
                      pNBI,
                      &(pNBI->PageInfoFormat),
                      IDD_FORMAT_OPTIONS,
                      FormatDlgProc,
                      "Select Program's Page Format",
                      IDM_SELECT_FORMAT,
                      BKA_MAJOR,
                      BKA_NEXT );

} /* end CreateNotebookPages */

/****************************************************************************/
/* PROCEDURE NAME : DestroyNotebookPages                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 05/25/94 !!!!                                           */
/* DESCRIPTION    : */
/*                                                                          */
/*                  Control Flow:                                           */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID DestroyNotebookPages( PPGMDATA ppd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PNOTEBOOKINFO pNBI;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // easier pointers to dereference
  pNBI = &ppd->nbi;

  //--------------------//
  // "Printer" Page     //
  //--------------------//
  DosFreeMem( pNBI->PageInfoPrinter.pvPageData );

  //--------------------//
  // "Font" Page        //
  //--------------------//
  DosFreeMem( pNBI->PageInfoFont.pvPageData );

  //--------------------//
  // "Colors" Page      //
  //--------------------//
  DosFreeMem( pNBI->PageInfoColors.pvPageData );

  //--------------------//
  // "Format" Page      //
  //--------------------//
  DosFreeMem( pNBI->PageInfoFormat.pvPageData );

} /* end DestroyNotebookPages */


/****************************************************************************/
/* PROCEDURE NAME : ProgramSettingsDlgProc                                  */
/* AUTHOR         : Matt R.                                                 */
/* DATE WRITTEN   : 2/22/95                                                 */
/* DESCRIPTION    : Dialog used to display                                  */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY ProgramSettingsDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PPGMDATA      ppd;
  PQUEUEDATA    pqd;

  PNOTEBOOKINFO pnb;
  PQUEUEDATA    ppiQue;
  PCOLORDATA    ppiClr;
  PFORMATDATA   ppiFmt;
  PFONTDATA     ppiFnt;

  static BOOL   bTabSizeSet = FALSE;
  ULONG ulID;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      ULONG ul = CLR_WHITE;

      ppd = (PPGMDATA)mp2;

      // store the pointer to instance data
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)ppd );

      ppd->nbi.ulNBID = IDD_SETTINGS_NOTEBOOK;
      ppd->nbi.hwndNB = WinWindowFromID( hwnd, IDD_SETTINGS_NOTEBOOK );

      WinSetPresParam( ppd->nbi.hwndNB, PP_FOREGROUNDCOLORINDEX, 4L, &ul );

      WinSendMsg( ppd->nbi.hwndNB, BKM_SETNOTEBOOKCOLORS, MPFROMLONG( CLR_DARKGRAY ), MPFROMLONG( BKA_BACKGROUNDMAJORCOLORINDEX ) );
      WinSendMsg( ppd->nbi.hwndNB, BKM_SETNOTEBOOKCOLORS, MPFROMLONG( CLR_DARKBLUE ), MPFROMLONG( BKA_BACKGROUNDPAGECOLORINDEX ) );

      CreateNotebookPages( hwnd, hMenu3d, ppd );

      // Select "Printer Info" to be topmost page
      WinSendMsg( ppd->nbi.hwndNB,
                  BKM_TURNTOPAGE,
                  MPFROMLONG( ppd->nbi.PageInfoPrinter.ulPageID ),
                  MPVOID );

      WinEnableControl( hwnd, DID_SAVE, !ppd->StateData.bPrinting );

      bTabSizeSet = TRUE;

      return(0);

    } /* end case */
    break;

    case WM_CONTROL:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case IDD_SETTINGS_NOTEBOOK:
        {
          switch (SHORT2FROMMP(mp1))
          {
            case BKN_PAGESELECTED:
            {
              PAGESELECTNOTIFY *PageData;
              PageData = (PAGESELECTNOTIFY *)mp2;

            } /* end case */
            break;

          } /* end switch */

        } /* end case */
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_DRAWITEM:
    {
      POWNERITEM   poi;

      // get instance data
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );

      if( bTabSizeSet )
      {
        PNBPAGEINFO pPage;

        poi = (POWNERITEM) PVOIDFROMMP(mp2); /* pointer to OWNERITEM */

        pPage = (PNBPAGEINFO)WinSendDlgItemMsg( hwnd, IDD_SETTINGS_NOTEBOOK,
                                                BKM_QUERYPAGEDATA,
                                                MPFROMLONG(poi->hItem),
                                                (MPARAM)NULL);
        if( pPage )
        {
          poi->idItem = pPage->ulTabID;

          Draw3DButton( hMenu3d, poi, SHORT1FROMMP( mp1 ), BUTTON_ACTION_DRAW );

        } /* end if */

      } /* end if */

      return( MRFROMLONG(TRUE) );

    } /* end case */

    case WM_DESTROY:
    {
      // get instance data
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );

      bTabSizeSet = FALSE;

      WinSendDlgItemMsg( hwnd,
                         IDD_SETTINGS_NOTEBOOK,
                         BKM_DELETEPAGE,
                         0L, (MPARAM)BKA_ALL);

      DestroyNotebookPages( ppd );

    } /* end case */
    break;

    case WM_COMMAND:
    {
      switch( ulID = SHORT1FROMMP(mp1) )
      {
        // QQQQQ
        case DID_OK:
        case DID_SAVE:
        {

          // get instance data
          ppd   = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );
          pnb   = &(ppd->nbi);

          ppiClr   = (PCOLORDATA) pnb->PageInfoColors.pvPageData;
          ppiFmt   = (PFORMATDATA)pnb->PageInfoFormat.pvPageData;
          ppiFnt   = (PFONTDATA)  pnb->PageInfoFont.pvPageData;

          /********************************************/
          /* Update COLOR Data                        */
          /********************************************/

          // Copy local color settings to global colors
          ppd->ColorData = *ppiClr;

          /********************************************/
          /* Update FORMAT Data                       */
          /********************************************/
          // Retrieve all selected text format options from
          // the dialogs listbox.
          ReadTextFormatOptions( ppd, ppiFmt, pnb->PageInfoFormat.hwndPage );

          // Copy local format settings to global format options
          ppd->FmtData = *ppiFmt;

          /********************************************/
          /* Update FONT Data                         */
          /********************************************/

          // @100698 -  Get the new pointsize into page data
          WinSendDlgItemMsg( pnb->PageInfoFont.hwndPage,
                             IDE_POINTSIZE, SPBM_QUERYVALUE,
                             MPFROMP(&(ppiFnt->PointSize)),
                             MPVOID );

          // Copy dialog page's font settings to program's
          ppd->FontData = *ppiFnt;

          // Copy dialog page's FONTMETRICS struc to program's FONTMETRICS
          strcpy( ppd->FontData.fm.szFacename, ppiFnt->pfmCurrent->szFacename );
          ppd->FontData.fm = *(ppiFnt->pfmCurrent);

          /********************************************/
          /* Update INI files                         */
          /********************************************/
          if( ulID == DID_SAVE )
          {
            WriteIniFileData( ppd, FT_KEYID_FONT   );
            WriteIniFileData( ppd, FT_KEYID_COLOR  );
            WriteIniFileData( ppd, FT_KEYID_FORMAT );
            WriteIniFileData( ppd, FT_KEYID_TITLE  );

          } /* end if */

          WinDismissDlg( hwnd, ulID );
          return(0);

        } /* end case */
        break;

        case DID_CANCEL:
        {
           WinDismissDlg( hwnd, DID_CANCEL );
           return(0);
        } /* end case */
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP:
    {
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );

      if( ppd->StateData.bHelpAvailable )
        WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                    MPFROMSHORT( PANEL_DLG_SETTINGS ),
                    MPFROMSHORT( HM_RESOURCEID ));
      else
        AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );
    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end ProgramSettingsDlgProc */


/****************************************************************************/
/* PROCEDURE NAME : OpenFileDlgProc                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/3/91                                                  */
/* DESCRIPTION    : Semi-standard dialog allowing user to select a file to  */
/*                  browse and output to a device.                          */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT APIENTRY OpenFileDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
   switch( msg )
   {
      case WM_HELP:
      {
         if( PgmData.StateData.bHelpAvailable )
           WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                       MPFROMSHORT( PANEL_DLG_FILE ),
                       MPFROMSHORT( HM_RESOURCEID ));
         else
           AssertPM( TRUE, PgmData.habPgm, ERR_NO_HELP );
      } /* end case */
      break;

   } /* end switch */

   return WinDefFileDlgProc( hwnd, msg, mp1, mp2 );

} /* end OpenFileDlgProc */



/****************************************************************************/
/* PROCEDURE NAME : UnlistedQueueDlgProc                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 8/4/91                                                  */
/* DESCRIPTION    : Allows user to enter a queue destination that we are    */
/*                  not able to enumerate through SplEnumQueue yet may      */
/*                  be accessible from this system. (i.e. Novell network    */
/*                  queues).                                                */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY UnlistedQueueDialogProc( HWND hwndDlg, ULONG msg, MPARAM mp1,
                                          MPARAM mp2 )
{

  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch( msg )
  {
    case WM_COMMAND:
       switch( SHORT1FROMMP(mp1) )
       {
         case DID_OK:
         {
           CHAR szBuffer[MAXSTR];
           LONG lChars;

           lChars = WinQueryWindowText( WinWindowFromID( hwndDlg,
                                                      IDE_OTHER_DESTINATION ),
                                                      QUENAME_LENGTH,
                                                      szBuffer );

           // If user entered a queue name from this alternative method
           if( lChars )
           {
             // save that away as current queue
             strcpy( PgmData.QueueData.QueueName, szBuffer );
             strcpy( PgmData.QueueData.QueueDesc, szBuffer );

           } /* end if */

           WinDismissDlg( hwndDlg, TRUE );

         } /* end case */
         break;

         case DID_CANCEL:
            WinDismissDlg( hwndDlg, FALSE );
         break;

       } /* end switch */
       return( (MRESULT)NULL );
    break;

    case WM_HELP:
    {
       if( PgmData.StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_OTHER ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, PgmData.habPgm, ERR_NO_HELP );
    } /* end case */
    break;

    default:
       return( WinDefDlgProc( hwndDlg, msg, mp1, mp2 ) );

  } /* end switch */

} /* end UnlistedQueueDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : SearchDlgProc                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Displays dialog allowing user to enter a search string  */
/*                  to search for in the current open file.                 */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY SearchDialogProc ( HWND hwndDlg, ULONG msg, MPARAM mp1,
                                    MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  static PSZ pszEntry;                // pointer to string
  static  hwndCase;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch ( msg )
  {
    case WM_INITDLG:
         // retrieve a pointer to the current string
         pszEntry = PVOIDFROMMP(mp2);

         // Get some window handles we need throughout procedure
         hwndCase = WinWindowFromID( hwndDlg, IDC_SRCH_CASE_SENSE );

         WinSendMsg( hwndCase, BM_SETCHECK,
                     MPFROMLONG(PgmData.StateData.bCaseSensitive),
                     NULL );

         // initialize the entry field text to current string
         WinSetDlgItemText( hwndDlg, IDE_SEARCH, pszEntry );
         return ( (MRESULT) FALSE );

    case WM_CONTROL:
         switch( SHORT1FROMMP(mp1) )
         {
           case IDC_SRCH_CASE_SENSE:

             PgmData.StateData.bCaseSensitive = PgmData.StateData.bCaseSensitive ? FALSE : TRUE;

             WinSendMsg( hwndCase, BM_SETCHECK, MPFROMSHORT(PgmData.StateData.bCaseSensitive), NULL );

           break;
         }
         break;

    case WM_COMMAND:
         switch ( SHORT1FROMMP ( mp1 ) )
         {
           case DID_OK:
                 // read the string from the entry field
                 // into the current string variable
                 WinQueryDlgItemText( hwndDlg, IDE_SEARCH, MAX_SEARCH_LEN,
                                      pszEntry );

                 // dismiss with DID_OK to indicate we changed the string
                 WinDismissDlg( hwndDlg, DID_OK );
                 return ( (MRESULT) NULL );
           break;

           case DID_CANCEL:
                // dismiss with DID_CANCEL to indicate no change
                WinDismissDlg( hwndDlg, DID_CANCEL );
                return ( (MRESULT) NULL );
           break;
         } /* end switch */

    case WM_HELP:
    {
       if( PgmData.StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_SEARCH ),
                     MPFROMSHORT( HM_RESOURCEID ));
       else
         AssertPM( TRUE, PgmData.habPgm, ERR_NO_HELP );
    } /* end case */
    break;

    default:
         return ( WinDefDlgProc( hwndDlg, msg, mp1, mp2 ) );

  } /* end switch */

} /* end SearchDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : CreateThread                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : This function creates a thread for function passed in   */
/*                  and returns it's thread id.                             */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Call DosCreateThread(); for function passed in      */
/*                                                                          */
/* PARAMETERS:      (PTID)      pTid      - pointer to Thread ID that will  */
/*                                          be created by this function     */
/*                  (PFNTHREAD) pfnThread - pointer to function address     */
/*                                          of the function that will act   */
/*                                          as the main message processing  */
/*                                          function of the new thread.     */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID CreateThread ( PTID pTid, PFNTHREAD pfnThread )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  DosCreateThread ( pTid,
                    pfnThread,
                    0L,
                    0L,
                    (ULONG) STACKSIZE );

} /* end CreateThread */

/****************************************************************************/
/* PROCEDURE NAME : SearchThread                                            */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : This function is a separate thread that starts the      */
/*                  search for the specified string at spec line.           */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Search for text string in currently opened file     */
/*                  (b) Send message back to ClientWndProc();               */
/*                      based on search result.                             */
/*                                                                          */
/*                      Search Result                                       */
/*                      -------------                                       */
/*                      Success  - display line where string found as       */
/*                                 first line in the client window.         */
/*                      Failure  - send failure message.                    */
/*                                                                          */
/*                  (c) call DosExit(); to end search thread                */
/*                                                                          */
/* PARAMETERS:      None                                                    */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/* @75474 - 11/03/93 - Matt Rutkowski [IBM] - removed SHORTS and USHORTS    */
/*                     for LONGS and ULONGS and call new message            */
/*                     WM_SET_CURRENT_LINE.                                 */
/*                                                                          */
/*                                                                          */
/****************************************************************************/
VOID FAR SearchThread ( VOID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ULONG      ulFoundLine;            // @75474 - line number of text
  CHAR       szPre[MAXSTR];
  CHAR       szMatch[MAXSTR];

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // if we have a string to search for
  ulFoundLine = SearchForString( szSearchString, lVscrollPos, szPre, szMatch );

  if( ulFoundLine == -1 )
  {
    WinPostMsg ( hwndClient, WM_SEARCH_FAILED, 0L, 0L );

  } /* end if */
  else
  {
    POINTL aptlUnmatched[TXTBOX_COUNT];
    POINTL aptlMatched[TXTBOX_COUNT];
    RECTL  rclClient;
    ULONG  ulHeight;
    POINTL aptl[2];

    //  @75474 - display line of file match found on
    WinPostMsg ( hwndClient, WM_SET_CURRENT_LINE, MPFROMLONG( ulFoundLine ), 0L );

    GpiQueryTextBox( hps, strlen(szPre),
                     szPre,
                     TXTBOX_COUNT,
                     aptlUnmatched );

    GpiQueryTextBox( hps, strlen(szMatch),
                     szMatch,
                     TXTBOX_COUNT,
                     aptlMatched );

    WinQueryWindowRect ( hwndClient, &rclClient );

    ulHeight = aptlMatched[TXTBOX_TOPRIGHT].y -
               aptlMatched[TXTBOX_BOTTOMRIGHT].y;

    aptl[0].x = aptlUnmatched[TXTBOX_BOTTOMRIGHT].x - (lHscrollPos*cxChar) - 1;
    aptl[0].y = cyClient - ulHeight - 1;

    aptl[1].x = aptl[0].x + aptlMatched[TXTBOX_BOTTOMRIGHT].x + 1;
    aptl[1].y = aptl[0].y + ulHeight + 1;

    GpiMove( hps, &(aptl[0]) );

    GpiSetColor( hps, PgmData.ColorData.SearchHighliteColor );

    GpiBox( hps, DRO_OUTLINE, &(aptl[1]), 0L, 0L );

  } /* end else */

  DosExit ( 0, 0 );

} /* SearchThread */


/****************************************************************************/
/* PROCEDURE NAME : PrintThread                                             */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/91                                                  */
/* DESCRIPTION    : Message processing loop for the printer thread.         */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Create a message queue for this thread to receive   */
/*                      messages.                                           */
/*                  (b) Send message to main thread to indicate print       */
/*                      thread is ready to receive messages.                */
/*                  (c) call DosSetPriority(); to lower threads priority    */
/*                  (d) Disable print related menu items                    */
/*                  (e) call AccessDevice(); with the action                */
/*                      REQUEST_PRINT_FILE       - if PM window print.      */
/*                      REQUEST_PRINT_FILE_NO_PM - if command line print    */
/*                  (f) Enable Print related window items                   */
/*                  (g) Destroy message queue                               */
/*                  (h) call DosExit(); to end print thread.                */
/*                                                                          */
/* PARAMETERS:      None                                                    */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID FAR PrintThread( VOID )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  QMSG qmsgPrn;
  ULONG ulAction = REQUEST_PRINT_FILE;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  PgmData.PrinterData.habPrn = WinInitialize( (USHORT)NULL );
  PgmData.PrinterData.hmqPrn = WinCreateMsgQueue( PgmData.PrinterData.habPrn, (SHORT)NULL );

  /* Set thread execution priority lower than main thread priority */
  DosSetPrty( PRTYS_THREAD, PRTYC_NOCHANGE, -1, (TID)NULL );

  WinPostQueueMsg( PgmData.hmqPgm,
                   WM_PRINT_THREAD_READY,
                   (MPARAM)NULL,
                   (MPARAM)NULL );

  if( PgmData.PrinterData.hmqPrn != (HMQ)NULL )
  {
    /* main msg processing loop */
    while( WinGetMsg( PgmData.PrinterData.habPrn, &qmsgPrn, (HWND)NULL,0,0 ))
    {
       // Internal message to print file
       if (qmsgPrn.msg == WM_PRINTDOC )
       {
         /*---------------------------------------------------*/
         /* CMD LINE PRINT                                    */
         /*---------------------------------------------------*/
         if( PgmData.StateData.bCmdLinePrint )
         {
           ulAction = REQUEST_PRINT_FILE_NO_PM;

           AccessDevice( &PgmData, PgmData.PrinterData.habPrn, OD_QUEUED, ulAction, DEV_TYPE_PRINTER );

           WinPostMsg( hwndClient, WM_QUIT, 0L, 0L );

         } /* end if */
         else
         {
           ERRORID ErrorID;

           /*-------------------------------------------------*/
           /* PM PRINT                                        */
           /*-------------------------------------------------*/

           // Disable menuitems during printing
           WinEnableMenuItem( hwndMenu, IDM_OPEN_FILE,     FALSE );
           WinEnableMenuItem( hwndMenu, IDM_QUERY_MENU,    FALSE );
           WinEnableMenuItem( hwndMenu, IDM_PRINT,         FALSE );

           WinEnableMenuItem( hwndMenu, IDM_FILE_MENU,     FALSE );
           WinEnableMenuItem( hwndMenu, IDM_QUERY_MENU,    FALSE );

           ErrorID = AccessDevice( &PgmData,
                                   PgmData.PrinterData.habPrn,
                                   OD_QUEUED,
                                   ulAction,
                                   DEV_TYPE_PRINTER );

           AssertPM( ErrorID!=NO_ERROR, PgmData.PrinterData.habPrn,
                     ERR_UNABLE_TO_PRINT );

           // Re-enable the menuitems
           WinEnableMenuItem( hwndMenu, IDM_OPEN_FILE,     TRUE );
           WinEnableMenuItem( hwndMenu, IDM_QUERY_MENU,    TRUE );
           WinEnableMenuItem( hwndMenu, IDM_PRINT,         TRUE );

           WinEnableMenuItem( hwndMenu, IDM_FILE_MENU,     TRUE );
           WinEnableMenuItem( hwndMenu, IDM_QUERY_MENU,    TRUE );

           /*-------------------------------------------------*/
           /* Show in Titlebar that we have finished printing */
           /*-------------------------------------------------*/
           strcpy( szTitleText, "Browsing File: \0" );
           strcat( szTitleText, filedlg.szFullFile );
           WinSetWindowText(hwndTitlebar,szTitleText);

         } /* end else */

         WinDestroyMsgQueue( PgmData.PrinterData.hmqPrn );
         WinTerminate( PgmData.PrinterData.habPrn );
         DosExit( 0, 0 );

       } /* end if */

    } /* end while */

  } /* end if */

} /* end PrintThread */


/****************************************************************************/
/* PROCEDURE NAME : SetScrollBars                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10/8/92                                                 */
/* DESCRIPTION    : Called to calculate scrollbar positions and limits      */
/*                  if client window is resized or font size changes.       */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Calculate new Horizontal scrollbar position and     */
/*                      thumbsize and send message to ClientWndProc();      */
/*                      to set them to their new values.                    */
/*                  (b) Calculate new Vertical scrollbar position and       */
/*                      thumbsize and send message to ClientWndProc();      */
/*                      to set them to their new values.                    */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd        - window handle of caller          */
/*                  (HWND)   hwndHscroll - window handle of Horizontal      */
/*                                         scroll bar.                      */
/*                  (HWND)   hwndVscroll - window handle of Vertical        */
/*                                         scroll bar.                      */
/*                  (PLONG)  lVscrollMax - Maximum value Vertical scroll    */
/*                                         bar can be set to.               */
/*                  (LONG)   cxClient    - width of PM client window        */
/*                  (LONG)   cyClient    - height of PM client window       */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID SetScrollBars( HWND  hwnd,        HWND hwndHscroll, HWND hwndVscroll,
                    PLONG lVscrollMax, LONG cxClient,    LONG cyClient )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  ULONG ulVertLines;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*-------------------------------------------------*/
  /* HORIZONTAL SCROLLBAR                            */
  /*-------------------------------------------------*/
  lHscrollPos = min( lHscrollPos, MAX_CHAR_PER_LINE );

  // Set initial Horz scrollbar position
  WinSendMsg( hwndHscroll,
              SBM_SETSCROLLBAR,
              MPFROM2SHORT( lHscrollPos, 0 ),
              MPFROM2SHORT( 0, MAX_CHAR_PER_LINE ));

  // Set initial Horz scrollbar thumbsize
  if( cxClient != 0 )
  {
    WinSendMsg( hwndHscroll,
                SBM_SETTHUMBSIZE,
                MPFROM2SHORT( cxClient/cxChar, MAX_CHAR_PER_LINE ),
                MPFROM2SHORT( 0, 0 ));
  } /* end if */

  WinEnableWindow( hwndHscroll, MAX_CHAR_PER_LINE ? TRUE : FALSE );

  /*-------------------------------------------------*/
  /* VERTICAL SCROLLBAR                              */
  /*-------------------------------------------------*/

  // validate character height to prevent % by 0 errors
  if( cyChar > 0 )
  {
    // @MAXLINE - new code
    ulVertLines = cyClient / cyChar;

    if( ulTotalLines > ulVertLines )
    {
      *lVscrollMax = (LONG)(ulTotalLines - ulVertLines);

    } /* end if */
    else
    {
       *lVscrollMax = 0L;
    } /* end else */

    lVscrollPos = min( lVscrollPos, *lVscrollMax );

    // @MAXLINE - Set initial Vert scrollbar position considering MAXLINES
    WinSendMsg( hwndVscroll, SBM_SETSCROLLBAR,
                MPFROM2SHORT( min( lVscrollPos, MAXLINES - ulVertLines), 0 ),
                MPFROM2SHORT( 0, min( *lVscrollMax, MAXLINES - ulVertLines ) ));

    WinSendMsg( hwndVscroll,
                SBM_SETTHUMBSIZE,
                MPFROM2SHORT( ulVertLines, min(ulTotalLines, MAXLINES/2 ) ),
                MPFROM2SHORT( 0, 0 ));

    WinEnableWindow( hwndVscroll, *lVscrollMax ? TRUE : FALSE );

    WinInvalidateRect ( hwnd, NULL, TRUE );

  // @MAXLINE - make debug
#if 1
     {
       CHAR szTemp[256];

       sprintf( szTemp, "[%d/%d] %s\0", lVscrollPos, ulTotalLines, szTitleText );

       WinSetWindowText( hwndTitlebar,szTemp );

     } /* end logical block */

#endif

  } /* end if */
  else
  {
    AssertPM( TRUE, PgmData.habPgm, ERR_CHAR_HEIGHT );
  } /* end else */

} /* end SetScrollBars */


/****************************************************************************/
/* PROCEDURE NAME : PrintCharacterBoxes                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/23/93                                                 */
/* DESCRIPTION    : This procedure is invoked from PrintTextFile();         */
/*                  if the user selected the "Print Character Boxes"        */
/*                  option from the "Text Format Options" dialog            */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Call GpiQueryCharStringPostion(); for every         */
/*                      line in the text file.                              */
/*                  (b) use values returned from (a) to draw each           */
/*                      caharcter box in the string using GpiLine();        */
/*                                                                          */
/* PARAMETERS:      (HAB)    habPrn   - Handle to anchor block of print     */
/*                                      thread.                             */
/*                  (HPS)    hpsPrn   - Handle to Presentaion Space for     */
/*                                      the print thread.                   */
/*                  (USHORT) uslength - length of current text line in      */
/*                                      characters.                         */
/*                  (PSZ)    str1     - current text line                   */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID PrintCharacterBoxes( HAB habPrn, HPS hpsPrn, USHORT uslength, PSZ str1 )
{

  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  POINTL     aptl[TXTBOX_COUNT];
  POINTL     ptlTmp;
  POINTL     ptl2;
  CHAR       cChar[1];
  USHORT     j;
  LONG       alXInc[MAXSTR];
  POINTL     aptlPos[MAXSTR];

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  if( GpiQueryCharStringPos( hpsPrn, 0, uslength, str1,
                             (PLONG)alXInc, (PPOINTL)aptlPos ) == GPI_ERROR )
  {
    AssertPM( TRUE, habPrn, ERR_GPIQCHARSTRINGPOS );

  } /* end if */

  // For each char in string
  for( j = 0; j < uslength; j++ )
  {

     ptlTmp = aptlPos[j];

     cChar[0] = str1[j];

     if( GpiQueryTextBox( hpsPrn, 1L, str1, TXTBOX_COUNT, aptl ) == GPI_ERROR )
     {
       AssertPM( TRUE, habPrn, ERR_GPIQTEXTBOX );

     } /* end if */

     ptl2.x = ptlTmp.x + aptl[TXTBOX_BOTTOMLEFT].x;
     ptl2.y = ptlTmp.y + aptl[TXTBOX_BOTTOMLEFT].y;

     GpiMove( hpsPrn, &ptl2 );

     ptl2.x = ptlTmp.x + aptl[TXTBOX_BOTTOMRIGHT].x;
     ptl2.y = ptlTmp.y + aptl[TXTBOX_BOTTOMRIGHT].y;
     GpiLine( hpsPrn, &ptl2 );

     ptl2.x = ptlTmp.x + aptl[TXTBOX_TOPRIGHT].x;
     ptl2.y = ptlTmp.y + aptl[TXTBOX_TOPRIGHT].y;
     GpiLine( hpsPrn, &ptl2 );

     ptl2.x = ptlTmp.x + aptl[TXTBOX_TOPLEFT].x;
     ptl2.y = ptlTmp.y + aptl[TXTBOX_TOPLEFT].y;
     GpiLine( hpsPrn, &ptl2 );

     ptl2.x = ptlTmp.x + aptl[TXTBOX_BOTTOMLEFT].x;
     ptl2.y = ptlTmp.y + aptl[TXTBOX_BOTTOMLEFT].y;
     GpiLine( hpsPrn, &ptl2 );

     ptlTmp.x += aptl[TXTBOX_CONCAT].x;
     ptlTmp.y += aptl[TXTBOX_CONCAT].y;

     GpiMove( hpsPrn, &ptlTmp );

  } /* end for */

} /* end PrintCharacterBoxes */


/****************************************************************************/
/* PROCEDURE NAME : PrintTextFile                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 3/23/93                                                 */
/* DESCRIPTION    : This procedure is called to outut a text file to the    */
/*                  currently opened printer Device Context passed in       */
/*                  and apply all the Text Format options selected by the   */
/*                  user.                                                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) BuildTrailerString(); if user wants to print date,  */
/*                      time, file name, page numbers etc.                  */
/*                  (b) Use GpiCreateRegion(); GpiSetClipRegion(); and      */
/*                      GpiOffsetClipRegion(); to create region half of the */
/*                      page if "Dual Column Print"                         */
/*                  (c) call GpiSetColor(); to set the text color.          */
/*                  (d) call PrintCharacterBoxes(); if user selected        */
/*                      it as a text format option.                         */
/*                  (e) Use GpiCharStringAt(); to output each line in the   */
/*                      text file                                           */
/*                  (f) call DrawTrailerString(); at the bottom of each     */
/*                      page and if trailer string was built.               */
/*                  (g) call DrawPageBorder(); at the bottom of each page   */
/*                      if user selected this Text Format option.           */
/*                  (h) call DevEscape( NEWFRAME ); to advance to next      */
/*                      page on printer.                                    */
/*                                                                          */
/* PARAMETERS:      (HAB)    habLcl              - handle to anchoe block   */
/*                                                 of caller.               */
/*                  (HDC)    hdcPrn              - handle to Device Context */
/*                                                 for printer              */
/*                  (HPS)    hpsPrn              - handle to Presentation   */
/*                                                 Space for printer        */
/*                  (LONG)   cyPrintChar         - Height of a print char   */
/*                  (LONG)   LinesPerPage        - Number of text lines that*/
/*                                                 fit on a printed page    */
/*                  (LONG)   cyDesc              - height of fonts maximum  */
/*                                                 maximum descender        */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL PrintTextFile( HAB     habLcl,
                    HDC     hdcPrn,
                    HPS     hpsPrn,
                    PGMDATA PgmData,
                    LONG    cyPrintChar,
                    LONG    lLinesPerPage,
                    PFONTMETRICS pfm )
{

  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  LONG     lCurLine = 0;           // Current line index being output
  BOOL     bError = FALSE;         // holds error return codes
  HRGN     hrgn, oldhrgn;          // regions for dual column printing
  LONG     index   = 0L;           // Font index into FontMetrics buffer
  LONG     lReturn = 0;            // holds long return codes
  POINTL   ptl;                    // point in PS
  POINTL   ptlShift;               // point in PS to shift for dual column
  CHAR     str1[MAXSTR+1];         // current text line
  CHAR     str2[MAXSTR+11];        // @74644
  USHORT   us;                     // loop variable
  ULONG    ulLength;               // length of current text line
  BOOL     bFirstColumn = TRUE;    // flag to indicate which column is on
  BOOL     bTrailerExists = FALSE; // flag to indicate a trailer line exists
  BOOL     bTitleExists   = FALSE; // flag to indicate a title   line exists
  CHAR     szFileName[MAXSTR];     // short filename
  CHAR     szPage[12];             // Page in string format
  CHAR     szDate[12];             // Date in string format
  CHAR     szTime[12];             // Time in string format
  CHAR     szTemp[10];             // Temp string variable
  ULONG    ulPageNum = 1;          // Holds current page number
  USHORT   cyTitle = 0;            // Holds amt. of pels to reserve for Title
  PSZ      pszCurLine;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Set initial left margin to left edge of page
  ptl.x = 0;

  if( PgmData.FmtData.bPrintTitleText )
  {
    // Flag that we have a trailer line
    bTitleExists = TRUE;

    // subtract 2 lines to print out trailer info if possible
    if( lLinesPerPage > 2 )
    {
      lLinesPerPage-=2;
      cyTitle = 2 * cyPrintChar;

    } /* end if */
    else
    {
      // Trailer not possible
      AssertPM( TRUE, habLcl, ERR_NO_TITLE );
      bTitleExists = FALSE;

    } /* end else */

  } /* end if */

  // If we have any text options that appear on the trailer line
  if( PgmData.FmtData.bPrintFileName ||
      PgmData.FmtData.bPrintDate     ||
      PgmData.FmtData.bPrintTime     ||
      PgmData.FmtData.bPrintPageNumbers )
  {
    // Flag that we have a trailer line
    bTrailerExists = TRUE;

    // subtract 2 lines to print out trailer info if possible
    if( lLinesPerPage > 2 )
    {
      lLinesPerPage-=2;

      // Build trailer string
      BuildTrailerString( habLcl, szPage, szFileName, szDate, szTime, &ulPageNum );

    } /* end if */
    else
    {

      /* Trailer not possible */
      AssertPM( TRUE, habLcl, ERR_NO_TRAILER );

      bTrailerExists = FALSE;

    } /* end else */

  } /* end if */

  /* Remember if dual column printing was requested */
  if( PgmData.FmtData.bPrintDualColumn )
  {
     RECTL ClipRect;

     ClipRect.xLeft   = 0;
     ClipRect.yBottom = 0;
     ClipRect.yTop    = PgmData.PrinterData.szlPage.cy;
     ClipRect.xRight  = PgmData.PrinterData.szlPage.cx/2;
     ptlShift.x       = ClipRect.xRight;
     ptlShift.y       = 0;

     hrgn = GpiCreateRegion( hpsPrn, 1L, &ClipRect );

     // verify region created successfully
     if( !hrgn )
     {
       AssertPM( TRUE, habLcl, ERR_GPICREATEREGION );

     } /* end if */

     if( GpiSetClipRegion( hpsPrn, hrgn, &oldhrgn ) == RGN_ERROR )
     {
       AssertPM( TRUE, habLcl, ERR_GPISETCLIPREGION );

     } /* end if */

  } /* end if */

  // Set the foreground (text) color
  if( GpiSetColor( hpsPrn, PgmData.ColorData.PrnTextColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, habLcl, ERR_GPISETCOLOR );

  } /* end if*/

  // @MARKV - Set background color if not default of white
  if( PgmData.ColorData.PrnBackColor != CLR_WHITE )
  {
    CHARBUNDLE chbnd;
    BOOL bRC;

    chbnd.lColor = 0L;
    chbnd.lBackColor    = PgmData.ColorData.PrnBackColor;
    chbnd.usBackMixMode = BM_OVERPAINT;

    bRC = GpiSetAttrs( hpsPrn, PRIM_CHAR, CBB_BACK_COLOR | CBB_BACK_MIX_MODE, 0L, &chbnd );
    AssertPM( bRC == GPI_ERROR, habLcl, ERR_GPISETATTRS  );

  } /* end if */

  /*--------------------------------------------*/
  /* Output lines of selected file              */
  /*--------------------------------------------*/
  while( index < ulTotalLines && bError==FALSE )
  {
    // Set current position on page to baseline where first
    // line of text is to be drawn.
    ptl.y = PgmData.PrinterData.szlPage.cy - pfm->lMaxAscender;

    /*------------------------------------------------------*/
    /* Print Title                                          */
    /*------------------------------------------------------*/
    /* user selected text format option for "Title Text"    */
    /*------------------------------------------------------*/
    if( bTitleExists )
    {
      DrawTitleString( habLcl, hpsPrn, PgmData, &ptl, pfm );

    } /* end if */

    /*------------------------------------------*/
    /* Output lines of selected file            */
    /*------------------------------------------*/
    while( lCurLine < lLinesPerPage &&
           index    < ulTotalLines &&
           bError  == FALSE )
    {
      // Tabs do not have any meaning to many printers so for
      // consistency we replace TAB chars with several space chars
      ReplaceTabs( pLinePtrs[index], (PSZ)str1, &ulLength, ' ', 7 );

      /*------------------------------------------------------*/
      /* Print Character Boxes                                */
      /*------------------------------------------------------*/
      /* User selected the text format option                 */
      /* "Print Character Boxes"                              */
      /*------------------------------------------------------*/
      if( PgmData.FmtData.bPrintCharBoxes )
      {
        /* Move to lower left corner of first character box */
        GpiMove ( hpsPrn, &ptl );

        /* Print all character boxes for this string */
        PrintCharacterBoxes( habLcl, hpsPrn, ulLength, str1 );

      } /* end if */

      /*------------------------------------------------------*/
      /* Print Line Numbers                          @74644   */
      /*------------------------------------------------------*/
      /* User selected the text format option                 */
      /* "Print Character Boxes"                              */
      /*------------------------------------------------------*/
      if( PgmData.FmtData.bPrintLineNumbers )
      {
        // Line numbers requested
        sprintf( str2, "%6d: %s\0", index + 1, str1 );
        pszCurLine = (PSZ)str2;
        ulLength   = strlen(str2);

      } /* end if */
      else
      {
        // No line numbers (original way)
        pszCurLine = (PSZ)str1;

      } /* end else */

      lReturn = GpiCharStringAt ( hpsPrn, &ptl, ulLength, pszCurLine );

      bError = AssertPM( lReturn!=GPI_OK, habLcl, ERR_GPICHARSTRAT );

      // increment current line index and decrease cursor on page to
      // baseline for next line of text to be printed by height of
      // current font ( lMaxBaseLineExtent )
      index++;
      lCurLine++;
      ptl.y -=cyPrintChar;

    } /* end while */

    /*------------------------------------------*/
    /* Signal a Page Break.                     */
    /*------------------------------------------*/
    if( index < ulTotalLines )
    {
      /*------------------------------------------------------*/
      /* Print Trailer String                                 */
      /*------------------------------------------------------*/
      /* If user selected any text format options that appear */
      /* in a trailer string (Page #, Date, Time, Filename)   */
      /*------------------------------------------------------*/
      if( bTrailerExists )
      {
         DrawTrailerString( habLcl,
                            hpsPrn,
                            ptl,
                            pfm,
                            &ulPageNum,
                            (PSZ)szPage,
                            (PSZ)szFileName,
                            (PSZ)szDate,
                            (PSZ)szTime );
      } /* end if */

      /*------------------------------------------------------*/
      /* Print Text Border                                    */
      /*------------------------------------------------------*/
      /* user selected text format option for "Text Border"   */
      /*------------------------------------------------------*/
      if( PgmData.FmtData.bPrintTextBorder )
      {
        DrawTextBorder( habLcl,
                        hpsPrn,
                        ptl,
                        bFirstColumn,
                        PgmData.PrinterData.szlPage.cx,
                        PgmData.PrinterData.szlPage.cy,
                        cyDesc );
      } /* end if */

      /* Remember if dual column printing was requested */
      if( (PgmData.FmtData.bPrintDualColumn) && (bFirstColumn) )
      {
        // Shift the clip rect and text over to 2nd column
        if( GpiOffsetClipRegion( hpsPrn, &ptlShift ) == RGN_ERROR )
        {
           AssertPM( TRUE, habLcl, ERR_GPIOFFSETCLIPREGION );

        } /* end if */

        // Move text over too
        ptl.x += ptlShift.x;

        // Set up values for next pass
        ptlShift.x = ptlShift.x * -1;

        bFirstColumn = FALSE;

      } /* end if */
      else
      {

        lReturn = DevEscape( hdcPrn, DEVESC_NEWFRAME, 0L,
                             NULL, 0L, NULL);

        bError = AssertPM( lReturn!=DEV_OK, habLcl, ERR_DEVESCNEWFRAME );

        if( PgmData.FmtData.bPrintDualColumn )
        {
          // Shift the clip rect and text over to 2nd column
          if( GpiOffsetClipRegion( hpsPrn, &ptlShift ) == RGN_ERROR )
          {
             AssertPM( TRUE, habLcl, ERR_GPIOFFSETCLIPREGION );

          } /* end if */

          // Move text back over to 1st column
          ptl.x += ptlShift.x;

          // Set up values for next pass
          ptlShift.x = ptlShift.x * -1;

          bFirstColumn = TRUE;

        } /* end if */

      } /* end else */

    } /* end if */

    // reset current line counter
    lCurLine = 0;

  } /* end while */

  // If we have partially completed page
  if( lCurLine < lLinesPerPage )
  {
    /*------------------------------------------------------*/
    /* Print Title                                          */
    /*------------------------------------------------------*/
    /* user selected text format option for "Title Text"    */
    /*------------------------------------------------------*/
    if( bTitleExists )
    {
        ptl.y = PgmData.PrinterData.szlPage.cy - pfm->lMaxAscender;

        DrawTitleString( habLcl, hpsPrn, PgmData, &ptl, pfm );

    } /* end if */

    /*------------------------------------------------------*/
    /* Print Trailer String                                 */
    /*------------------------------------------------------*/
    /* If user selected any text format options that appear */
    /* in a trailer string (Page #, Date, Time, Filename)   */
    /*------------------------------------------------------*/
    if( bTrailerExists )
    {
       DrawTrailerString( habLcl,
                          hpsPrn,
                          ptl,
                          pfm,
                          &ulPageNum,
                          (PSZ)szPage,
                          (PSZ)szFileName,
                          (PSZ)szDate,
                          (PSZ)szTime );
    } /* end if */

    /*------------------------------------------------------*/
    /* Print Text Border                                    */
    /*------------------------------------------------------*/
    /* user selected text format option for "Text Border"   */
    /*------------------------------------------------------*/
    if( PgmData.FmtData.bPrintTextBorder )
    {
      DrawTextBorder( habLcl,
                      hpsPrn,
                      ptl,
                      bFirstColumn,
                      PgmData.PrinterData.szlPage.cx,
                      PgmData.PrinterData.szlPage.cy,
                      cyDesc );
    } /* end if */

    if( PgmData.FmtData.bPrintDualColumn )
    {
      GpiDestroyRegion( hpsPrn, hrgn );

    } /* end if*/

  } /* end if */

  return( bError );

} /* end PrintTextFile */

/****************************************************************************/
/* PROCEDURE NAME : ColorsDlgProc                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/17/93                                                 */
/* DESCRIPTION    : This is the 'Select Colors' dialog's message processing */
/*                  procedure.  This function manages color selection for   */
/*                  the client window and for printed output.               */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY ColorsDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  static  HWND      hwndColorSet;
  static  PLONG     plTmpItem;

  USHORT  usItem;
  ULONG   ul;
  CHAR    szString[MAXSTR];
  USHORT  usRow, usColumn;
  USHORT  usIndex;
  LONG    lTmpClr;
  ULONG   ulRowColumn;
  SHORT   sIdx;               // @LBOX

  PPGMDATA      ppd;
  PCOLORDATA    pcd;
  PNBPAGEINFO   ppi;
  PNOTEBOOKINFO pnb;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      HWND   hwndLB = WinWindowFromID( hwnd, IDL_COLOR_ITEMS );

      // Get program data from mp2 which is set by caller
      ppd = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)ppd );

      ppi = &(ppd->nbi.PageInfoColors);
      pcd = (PCOLORDATA)ppi->pvPageData;

      // Get window handle of color value set
      hwndColorSet = WinWindowFromID( hwnd, IDV_AVAIL_COLORS );

      // Make local copy of color data
      *pcd = ppd->ColorData;

      // For each row
      for ( usRow=0; usRow < CLR_TBL_ROWS; usRow++ )
      {
        // For each column
        for ( usColumn=0; usColumn < CLR_TBL_COLUMNS; usColumn++ )
        {
            usIndex = (usRow * CLR_TBL_COLUMNS) + usColumn;

            WinSendMsg( hwndColorSet, VM_SETITEM,
                        MPFROM2SHORT( (usRow+1), (usColumn+1) ),
                        MPFROMLONG( ColorTable[usIndex] ) );
        } /* end for */

      } /* end for */

      for ( ul = STRID_CLR_ITEM_BASE; ul < STRID_CLR_ITEM_END ; ul++ )
      {
        // retrieve a pointer to the current string
        WinLoadString( ppd->habPgm, (HMODULE)NULL, (ULONG)ul,
                       (LONG)(MAXSTR-1), (PSZ)szString );

        sIdx = (SHORT)WinInsertLboxItem( hwndLB, LIT_END, szString );

        // @LBOX - Set Item Handle to be the Form ID
        WinSendDlgItemMsg( hwnd,
                           IDL_COLOR_ITEMS,
                           LM_SETITEMHANDLE,
                           MPFROMSHORT( sIdx ),
                           MPFROMLONG( ul ) );

      } /* end for */

      // Tell ourselves to select first item in list as convention
      WinSendDlgItemMsg( hwnd,
                         IDL_COLOR_ITEMS,
                         LM_SELECTITEM,
                         MPFROMSHORT( 0 ),
                         MPFROMSHORT((SHORT)TRUE));

      return(0);

    } /* end case */
    break;

    case WM_CONTROL:
    {
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );
      ppi = &(ppd->nbi.PageInfoColors);
      pcd = (PCOLORDATA)ppi->pvPageData;

      switch( SHORT1FROMMP(mp1) )
      {
        case IDV_AVAIL_COLORS:
        {
          switch( SHORT2FROMMP(mp1) )
          {
             case VN_SELECT:
             {
                ulRowColumn = (ULONG)WinSendMsg( hwndColorSet,
                                                 VM_QUERYSELECTEDITEM,
                                                 MPVOID,
                                                 MPVOID );

                usRow    = SHORT1FROMMP(ulRowColumn);
                usColumn = SHORT2FROMMP(ulRowColumn);

                *plTmpItem = MapRowColumnToColor( usRow, usColumn );

             } /* end case */
             break;

          } /* end switch */

        } /* end case */
        break;

        case IDL_COLOR_ITEMS:
        {
          // Get index of listbox item selected by user
          sIdx = SHORT1FROMMR( WinSendDlgItemMsg( hwnd,
                                                  IDL_COLOR_ITEMS,
                                                  LM_QUERYSELECTION,
                                                  0L,0L) );

          // @LBOX - verify user selected an item
          if( sIdx != LIT_NONE )
          {
            // Get Item Handle (STRING ID)
            usItem = (USHORT)WinSendDlgItemMsg( hwnd,
                                                IDL_COLOR_ITEMS,
                                                LM_QUERYITEMHANDLE,
                                                MPFROMSHORT( sIdx ),
                                                MPVOID );

            switch ( usItem )
            {
              case STRID_CLR_ITEM_TEXT      :
                   lTmpClr = pcd->WinTextColor;
                   plTmpItem  = &(pcd->WinTextColor);
              break;
              case STRID_CLR_ITEM_BACKTEXT  :
                   lTmpClr = pcd->WinBackColor;
                   plTmpItem  = &(pcd->WinBackColor);
              break;
              case STRID_CLR_ITEM_PTEXT     :
                   lTmpClr = pcd->PrnTextColor;
                   plTmpItem  = &(pcd->PrnTextColor);
              break;
              case STRID_CLR_ITEM_PBACKTEXT :
                   lTmpClr = pcd->PrnBackColor;
                   plTmpItem  = &(pcd->PrnBackColor);
              break;
              case STRID_CLR_ITEM_BORDER    :
                   lTmpClr = pcd->TextBorderColor;
                   plTmpItem  = &(pcd->TextBorderColor);
              break;
              case STRID_CLR_ITEM_TRAILER   :
                   lTmpClr = pcd->TextTrailerColor;
                   plTmpItem  = &(pcd->TextTrailerColor);
              break;
              case STRID_CLR_ITEM_TITLETEXT :
                   lTmpClr = pcd->TextTitleColor;
                   plTmpItem  = &(pcd->TextTitleColor);
              break;
              case STRID_CLR_ITEM_SEARCH :
                   lTmpClr = pcd->SearchHighliteColor;
                   plTmpItem  = &(pcd->SearchHighliteColor);
              break;

            } /* end switch */

            MapColorToRowColumn( lTmpClr, &usRow, &usColumn );

            WinSendMsg( hwndColorSet,
                        VM_SELECTITEM,
                        MPFROM2SHORT( usRow, usColumn ),
                        MPVOID );

          } /* end if */

        } /* end case */
        break;

      } /* end switch */

    } /* end case */
    break;

    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case DID_CANCEL:
        {
          return(0);

        } /* end case */

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP:
    {
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );

      if( ppd->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_COLORS ),
                     MPFROMSHORT( HM_RESOURCEID ));
      else
         AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );

    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end ColorsDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : FormatDlgProc                                           */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/9/93                                                  */
/* DESCRIPTION    : This dialog procedure processes messages from the       */
/*                  Text Format dialog such as "Print Dual Column" and      */
/*                  "Print Page Numbers".                                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      (HWND)   hwnd                                           */
/*                  (ULONG)  msg                                            */
/*                  (MPARAM) mp1                                            */
/*                  (MPARAM) mp2                                            */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
MRESULT EXPENTRY FormatDlgProc( HWND hwnd, ULONG msg, MPARAM mp1, MPARAM mp2 )
{
  /*------------------------------------------------------------------------*/
  /* Locals                                                                 */
  /*------------------------------------------------------------------------*/
  PPGMDATA      ppd;
  PFORMATDATA   pfd;
  PNBPAGEINFO   ppi;
  PNOTEBOOKINFO pnb;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/
  switch(msg)
  {
    case WM_INITDLG:
    {
      CHAR   szString[MAXSTR];
      ULONG  ul;
      HWND   hwndLB = WinWindowFromID( hwnd, IDL_FORMAT_OPTIONS );

      // Get program data from mp2 which is set by caller
      ppd = (PPGMDATA)PVOIDFROMMP( mp2 );

      // store the pointer to program data in user's window data area
      WinSetWindowULong( hwnd, QWL_USER, (ULONG)ppd );

      ppi = &(ppd->nbi.PageInfoFormat);
      pfd = (PFORMATDATA)ppi->pvPageData;

      // Make local copy of document format data
      *pfd = ppd->FmtData;

      for ( ul = STRID_FORMAT_BASE; ul < STRID_END ; ul++ )
      {
        // retrieve a pointer to the current string
        WinLoadString( ppd->habPgm, (HMODULE)NULL, (ULONG)ul,
                       (LONG)(MAXSTR-1), (PSZ)szString );

        WinInsertLboxItem( hwndLB, LIT_END, szString );

      } /* end for */

      // Call procedure that displays all text format options
      // to the listbox in the dialog
      ShowTextFormatOptions( ppd, pfd, hwnd );

      return ( (MRESULT) FALSE );

    } /* end case */
    break;

    case WM_COMMAND:
    {
      switch( SHORT1FROMMP(mp1) )
      {
        case DID_CANCEL:
        {
          return(0);

        } /* end case */

      } /* end switch */

    } /* end case */
    break;

    case WM_HELP:
    {
      ppd = (PPGMDATA) WinQueryWindowULong( hwnd, QWL_USER );

      if( ppd->StateData.bHelpAvailable )
         WinSendMsg( hwndHelp, HM_DISPLAY_HELP,
                     MPFROMSHORT( PANEL_DLG_FORMAT ),
                     MPFROMSHORT( HM_RESOURCEID ));
      else
         AssertPM( TRUE, ppd->habPgm, ERR_NO_HELP );

    } /* end case */
    break;

  } /* end switch */

  return WinDefDlgProc( hwnd, msg, mp1, mp2 );

} /* end FormatDlgProc */

/****************************************************************************/
/* PROCEDURE NAME : ReadTextFormatOptions                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/9/93                                                  */
/* DESCRIPTION    : This function retrieves all highlighted entries in the  */
/*                  Available Text Options listbox and saves this           */
/*                  information for later application.                      */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) For each highlighted line in the listbox save       */
/*                      the fact that its corresponding text format option  */
/*                      was selected.                                       */
/*                                                                          */
/* PARAMETERS:      (HWND) hwnd - window handle of caller, in this case     */
/*                                text format option dialog window.         */
/*                                                                          */
/* RETURN VALUES:   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL ReadTextFormatOptions( PPGMDATA ppd, PFORMATDATA pfd, HWND hwnd )

{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  SHORT  sTextOpt;
  USHORT usDlgItem;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Initialize all format options to FALSE
  DefaultFormatOptions( pfd );

  // Query listbox for first highlighted item
  sTextOpt = SHORT1FROMMR( WinSendDlgItemMsg( hwnd,
                                              IDL_FORMAT_OPTIONS,
                                              LM_QUERYSELECTION,
                                              MPFROMSHORT(LIT_FIRST),
                                              MPVOID ) );

  /* while items are still selected in the text format listbox */
  while( sTextOpt != LIT_NONE )
  {
     // indicate the appropriate option was selected
     switch( sTextOpt+STRID_FORMAT_BASE )
     {
       case STRID_FORMAT_CHARBOX:
          pfd->bPrintCharBoxes   = TRUE;
       break;
       case STRID_FORMAT_DUAL:
          pfd->bPrintDualColumn  = TRUE;
       break;
       case STRID_FORMAT_FILE:
          pfd->bPrintFileName    = TRUE;
       break;
       case STRID_FORMAT_DATE:
          pfd->bPrintDate        = TRUE;
       break;
       case STRID_FORMAT_TIME:
          pfd->bPrintTime        = TRUE;
       break;
       case STRID_FORMAT_PAGENUM:
          pfd->bPrintPageNumbers = TRUE;
       break;
       case STRID_FORMAT_BORDER:
          pfd->bPrintTextBorder  = TRUE;
       break;
       // @74644
       case STRID_FORMAT_LINENUM:
          pfd->bPrintLineNumbers = TRUE;
       break;
       // @TTTTT
       case STRID_FORMAT_TITLETEXT:
          pfd->bPrintTitleText   = TRUE;
          WinQueryDlgItemText( hwnd,
                               IDE_TEXT_TITLE,
                               MAX_SEARCH_LEN,
                               ppd->TxtTitle.pszText );

          if( WinQueryButtonCheckstate( hwnd, IDR_TITLE_LEFT   ) ){ ppd->TxtTitle.usTextJust = TEXT_JUSTIFY_LEFT; }
          if( WinQueryButtonCheckstate( hwnd, IDR_TITLE_CENTER ) ){ ppd->TxtTitle.usTextJust = TEXT_JUSTIFY_CENTER; }
          if( WinQueryButtonCheckstate( hwnd, IDR_TITLE_RIGHT  ) ){ ppd->TxtTitle.usTextJust = TEXT_JUSTIFY_RIGHT; }

       break;
     } /* end switch */

     // Query listbox for next highlighted item
     sTextOpt = SHORT1FROMMR( WinSendDlgItemMsg( hwnd,
                                                 IDL_FORMAT_OPTIONS,
                                                 LM_QUERYSELECTION,
                                                 MPFROMSHORT(sTextOpt),
                                                 MPVOID ) );
  } /* end while */

} /* end ReadTextFormatOptions */


/****************************************************************************/
/* PROCEDURE NAME : ShowTextFormatOptions                                   */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/9/93                                                  */
/* DESCRIPTION    : This function is called when the Text Format dialog is  */
/*                  first displayed to highlight text format options that   */
/*                  were previously selected by the user.                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) For each text option in the available text options  */
/*                      listbox check to see if user selected it previously */
/*                      and if so highlite it.                              */
/*                                                                          */
/* PARAMETERS:      (HWND) hwnd - window handle of caller, in this case     */
/*                                text format option dialog window.         */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID ShowTextFormatOptions( PPGMDATA ppd, PFORMATDATA pfd, HWND hwnd )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  SHORT sTextOpt = 0;
  BOOL  bSelect;
  USHORT usDlgItem;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /* For each item to be placed in the text format listbox */
  while( sTextOpt < STRID_END )
  {
     switch( sTextOpt+STRID_FORMAT_BASE )
     {
       case STRID_FORMAT_CHARBOX:
          bSelect = pfd->bPrintCharBoxes;
       break;
       case STRID_FORMAT_DUAL:
          bSelect = pfd->bPrintDualColumn;
       break;
       case STRID_FORMAT_FILE:
          bSelect = pfd->bPrintFileName;
       break;
       case STRID_FORMAT_DATE:
          bSelect = pfd->bPrintDate;
       break;
       case STRID_FORMAT_TIME:
          bSelect = pfd->bPrintTime;
       break;
       case STRID_FORMAT_PAGENUM:
          bSelect = pfd->bPrintPageNumbers;
       break;
       case STRID_FORMAT_BORDER:
          bSelect = pfd->bPrintTextBorder;
       break;
       // @74644
       case STRID_FORMAT_LINENUM:
          bSelect = pfd->bPrintLineNumbers;
       break;
       // @TTTTT
       case STRID_FORMAT_TITLETEXT:
          bSelect = pfd->bPrintTitleText;

          WinSetDlgItemText( hwnd, IDE_TEXT_TITLE, ppd->TxtTitle.pszText );

          switch( ppd->TxtTitle.usTextJust )
          {
            case TEXT_JUSTIFY_LEFT:   usDlgItem = IDR_TITLE_LEFT;
            break;
            case TEXT_JUSTIFY_CENTER: usDlgItem = IDR_TITLE_CENTER;
            break;
            case TEXT_JUSTIFY_RIGHT:  usDlgItem = IDR_TITLE_RIGHT;
            break;
            default: usDlgItem = IDR_TITLE_LEFT;
            break;

          } /* end switch */

          WinCheckButton( hwnd, usDlgItem, TRUE );

       break;
     } /* end switch */

     WinSendDlgItemMsg( hwnd, IDL_FORMAT_OPTIONS, LM_SELECTITEM,
                        MPFROMSHORT(sTextOpt), MPFROMSHORT( bSelect ));
     sTextOpt++;

  } /* end while */

} /* end ShowTextFormatOptions */

/****************************************************************************/
/* PROCEDURE NAME : BuildTrailerString                                      */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/16/93                                                 */
/* DESCRIPTION    : This function is called to retrieve all the strings     */
/*                  that make up the trailer string that will be output at  */
/*                  the bottom of each page of output.                      */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Check each text format option that makes up the     */
/*                      trailer string and fill in the string for that      */
/*                      option                                              */
/*                                                                          */
/* PARAMETERS:      (HAB)    hab         - anchor block handle of caller    */
/*                  (PSZ)    pszPage     - the string for "Page"            */
/*                  (PSZ)    pszFileName - Current file name string         */
/*                  (PSZ)    pszDate     - Current date string              */
/*                  (PSZ)    pszTime     - Current Time String              */
/*                  (PULONG) pulPageNum  - Current Page Number string       */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID BuildTrailerString( HAB hab,
                         PSZ pszPage,
                         PSZ pszFileName,
                         PSZ pszDate,
                         PSZ pszTime,
                         PULONG pulPageNum )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  DATETIME DateTime;               // Holds system date/time info
  PSZ      pszTempFile;
  CHAR     szTemp[10];             // Temp string variable

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  if( PgmData.FmtData.bPrintFileName )
  {
     // Get string after last occurrence of backslash
     pszTempFile = (PSZ)strrchr( filedlg.szFullFile, '\\' );


     // If drive:directory path found
     if( pszTempFile )
     {
       // move past last backslash - filename should follow
       pszTempFile++;

       strcpy( pszFileName, pszTempFile );

     } /* end if */
     else
     {

       // copy in whole thing
       strcpy( pszFileName, filedlg.szFullFile  );

     } /* end else */

     strcat( pszFileName, TRAILER_SPACE_CNST );

  } /* end if */
  else
  {
     pszFileName[0] = '\0';

  } /* end else */

  // Get the date and time anyways
  DosGetDateTime( &DateTime );

  if( PgmData.FmtData.bPrintDate )
  {
     _itoa( DateTime.month, szTemp, 10 );

     strcpy( pszDate, szTemp );
     strcat( pszDate, (PSZ)TRAILER_DATE_SEP_CNST );

     _itoa( DateTime.day, szTemp, 10 );

     strcat( pszDate, szTemp );
     strcat( pszDate, (PSZ)TRAILER_DATE_SEP_CNST );

     _itoa( DateTime.year, szTemp, 10 );

     strcat( pszDate, szTemp );

     strcat( pszDate, TRAILER_SPACE_CNST );

  } /* end if */
  else
  {
    pszDate[0] = '\0';

  } /* end else */

  if( PgmData.FmtData.bPrintTime )
  {
     _itoa( DateTime.hours, szTemp, 10 );

     strcpy( pszTime, szTemp );
     strcat( pszTime, (PSZ)TRAILER_TIME_SEP_CNST );

     _itoa( DateTime.minutes, szTemp, 10 );

     strcat( pszTime, szTemp );
     strcat( pszTime, (PSZ)TRAILER_TIME_SEP_CNST );

     _itoa( DateTime.seconds, szTemp, 10 );

     strcat( pszTime, szTemp );

     strcat( pszTime, TRAILER_SPACE_CNST );

  } /* end if */
  else
  {
    pszTime[0] = '\0';

  } /* end else */

  if( PgmData.FmtData.bPrintPageNumbers )
  {
     strcpy( pszPage, TRAILER_PAGE_CNST );

  } /* end if */
  else
  {
    *pulPageNum = 0;
    pszPage[0] = '\0';

  } /* end else */

} /* end BuildTrailerString */

/****************************************************************************/
/* PROCEDURE NAME : DrawTitleString                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 1/24/95                                                 */
/* DESCRIPTION    : This function is called at the top                      */
/*                                                                          */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID DrawTitleString( HAB          hab,
                      HPS          hps,
                      PGMDATA      PgmData,
                      PPOINTL      pptl,
                      PFONTMETRICS pfm )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  LONG lReturn;                // GpiCharStringAt return code
  POINTL ptlTitle = *pptl;     // Init local position on page
  ULONG ulLen;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Set the foreground (text) color for the title
  if( GpiSetColor( hps, PgmData.ColorData.TextTitleColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, hab, ERR_GPISETCOLOR );

  } /* end if*/

  // If we are justifying title to center it or to right
  // we must calculate the 'x' coordinate
  if( !( PgmData.TxtTitle.usTextJust == TEXT_JUSTIFY_LEFT) )
  {

    POINTL aptl[TXTBOX_COUNT];
    ULONG  ulLen;

    if(GpiQueryTextBox( hps,
                        (LONG)strlen(PgmData.TxtTitle.pszText),
                        (PSZ)PgmData.TxtTitle.pszText,
                        TXTBOX_COUNT,
                        aptl ) == GPI_OK )
    {
      ulLen = abs( aptl[TXTBOX_TOPRIGHT].x - aptl[TXTBOX_TOPLEFT].x ) + 1;

      ptlTitle.x = PgmData.PrinterData.szlPage.cx - ulLen;

      if( PgmData.TxtTitle.usTextJust == TEXT_JUSTIFY_CENTER )
      {
        ptlTitle.x = ptlTitle.x/2;

      } /* end if */

    } /* end if */
    else
    {
      AssertPM( TRUE, hab, ERR_GPIQTEXTBOX );

    } /* end else */

  } /* end if */

  lReturn = GpiCharStringAt ( hps, &ptlTitle,
                              (LONG)strlen(PgmData.TxtTitle.pszText),
                              (PSZ)PgmData.TxtTitle.pszText );

  AssertPM( lReturn!=GPI_OK, hab, ERR_GPICHARSTRAT );

  // Restore the foreground (text) color
  if( GpiSetColor( hps, PgmData.ColorData.PrnTextColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, hab, ERR_GPISETCOLOR );

  } /* end if*/

  // Set position on page for next line of text to be printed
  // We will position by omving down height of title plus
  // height of one blank  line
  pptl->y -= pfm->lMaxAscender + pfm->lMaxBaselineExt + 2 * pfm->lExternalLeading;

} /* end DrawTitleString */

/****************************************************************************/
/* PROCEDURE NAME : DrawTrailerString                                       */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/15/93                                                 */
/* DESCRIPTION    : This function is called at the endof each page of       */
/*                  output to draw the trailer string at the bottom of the  */
/*                  page.                                                   */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Take all strings that make of the trailer and       */
/*                      combine them together.                              */
/*                  (b) call GpiSetColor(); to set dolor for trailer string */
/*                  (c) call GpiCharStringAt(); to print out trailer string */
/*                                                                          */
/* PARAMETERS:      (HAB)    hab         - Anchor block handle              */
/*                  (HPS)    hps         - Presentation Space handle        */
/*                  (POINTL) ptl         - Point where trailer string is    */
/*                                         to be drawn.                     */
/*                  (LONG)   cyDesc      - Current font's maximum descender */
/*                  (PULONG) pulPageNum  - Current Page Number              */
/*                  (PSZ)    pszPage     - Page Name string                 */
/*                  (PSZ)    pszFileName - File Name string                 */
/*                  (PSZ)    pszDate     - Current Date string              */
/*                  (PSZ)    pszTime     - Current Time String              */
/*                                                                          */
/* RETURN VALUES:   None                                                    */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID DrawTrailerString( HAB          hab,
                        HPS          hps,
                        POINTL       ptl,
                        PFONTMETRICS pfm,
                        PULONG       pulPageNum,
                        PSZ          pszPage,
                        PSZ          pszFileName,
                        PSZ          pszDate,
                        PSZ          pszTime )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  CHAR szPageNum[12];          // string to hold page number
  CHAR szTrailer[MAXSTR];      // trailer string (date/time/page number)
  LONG lReturn;                // GpiCharStringAt return code

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Set color for the trailer string
  if( GpiSetColor( hps, PgmData.ColorData.TextTrailerColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, hab, ERR_GPISETCOLOR );

  } /* end if*/

  // Null out trailer string to prevent garbage chars
  szTrailer[0] = '\0';
  szPageNum[0] = '\0';

  // Set position on page for trailer string's baseline
  ptl.y = pfm->lExternalLeading + pfm->lMaxDescender;

  // if we are printing page numbers
  if( (*pulPageNum) > 0 )
  {
    // convert page number to string
    _ltoa( (*pulPageNum), szPageNum, 10 );

    // increment page number
    (*pulPageNum)++;

    // format trailer string w/ page numbers
    sprintf( szTrailer, "%s %-5s %s%s%s\0",
             pszPage,
             szPageNum,
             pszFileName,
             pszDate,
             pszTime );

  } /* end if */
  else
  {
    // format trailer string w/o page numbers
    sprintf( szTrailer, "%s%s%s\0",
             pszFileName,
             pszDate,
             pszTime );

  } /* end else */

  lReturn = GpiCharStringAt ( hps, &ptl,
                              (LONG)strlen(szTrailer),
                              (PSZ)szTrailer );

  AssertPM( lReturn!=GPI_OK, hab, ERR_GPICHARSTRAT );

  // Restore text color now that trailer string is done
  if( GpiSetColor( hps, PgmData.ColorData.PrnTextColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, hab, ERR_GPISETCOLOR );

  } /* end if*/

} /* end DrawTrailerString */


/****************************************************************************/
/* PROCEDURE NAME : DrawTextBorder                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/16/93                                                 */
/* DESCRIPTION    : This function is called after each logical page of      */
/*                  output to draw a border around it if the user selected  */
/*                  the "Print Text Border" text format option.             */
/*                  If the user selected "Print Dual Column" then this      */
/*                  function will be called twice per physical page         */
/*                  (once per column).                                      */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) call GpiSetColor(); to set border color             */
/*                  (b) call GpiMove(); to bottom-left box corner           */
/*                  (c) calculate width and height of box.                  */
/*                  (d) call GpiBox(); to draw the text border              */
/*                  (e) call GpiSetColor(); to restore text color           */
/*                                                                          */
/* PARAMETERS:     (HAB)    hab          - Anchor block handle              */
/*                 (HPS)    hps          - Presentation Space handle        */
/*                 (POINTL) ptl          - Point where trailer string is    */
/*                                         to be drawn.                     */
/*                 (BOOL)   bFirstColumn - Dual Column Text Format On/Off   */
/*                 (LONG)   cxColumn     - Width of column                  */
/*                 (LONG)   cyColumn     - Height of column                 */
/*                 (LONG)   cyDesc       - Current font's maximum descender */
/*                                                                          */
/* RETURN VALUES:  None                                                     */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID DrawTextBorder( HAB    hab,
                     HPS    hps,
                     POINTL ptl,
                     BOOL   bFirstColumn,
                     LONG   cxColumn,
                     LONG   cyColumn,
                     LONG   cyDesc )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  POINTL ptltest;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // Set the text border color selected by the user
  if( GpiSetColor( hps, PgmData.ColorData.TextBorderColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, hab, ERR_GPISETCOLOR );
  } /* end if */

  ptltest.x = ptl.x;
  ptltest.y = 0;

  GpiMove( hps, &ptltest );

  // If printing dual column
  if ( PgmData.FmtData.bPrintDualColumn )
  {
   // If we are printing first column
   if ( bFirstColumn )
   {
      ptltest.x = cxColumn/2 - 1;

   } /* end if */
   else
   {
      ptltest.x = cxColumn - 2;

   } /* end else */

  } /* end if */
  else
  {
    ptltest.x = cxColumn - 1;

  } /* end else */

  ptltest.y = cyColumn - 1;

  // Draw the text box
  GpiBox( hps, DRO_OUTLINE, &ptltest, 0L, 0L );

  // Restore text color now that trailer string is done
  if( GpiSetColor( hps, PgmData.ColorData.PrnTextColor ) == GPI_ERROR )
  {
    AssertPM( TRUE, hab, ERR_GPISETCOLOR );

  } /* end if*/

} /* end DrawTextBorder */




/****************************************************************************/
/* PROCEDURE NAME : MapColorToRowColumn                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/19/93                                                 */
/* DESCRIPTION    : This function helps manage the 'Select Color' dialog's  */
/*                  Available Colors Value set control.  It returns         */
/*                  the correct Row and Column index into the value set     */
/*                  for the color passed                                    */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Loop through the Rows and Columns until color       */
/*                      passed in is found or the end of the color table    */
/*                      is reached.                                         */
/*                                                                          */
/* PARAMETERS:      (LONG)    lColor    -  Color to search for in value set */
/*                  (PUSHORT) pusRow    -  Row where lColor found           */
/*                  (PUSHORT) pusColumn -  Column where lColor found        */
/*                                                                          */
/* RETURN VALUES:   (BOOL)    FALSE     -  No color match                   */
/*                            TRUE      -  Color match found                */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL MapColorToRowColumn( LONG lColor, PUSHORT pusRow, PUSHORT pusColumn )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  USHORT usRow, usColumn, usIndex;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // For each row
  for ( usRow=0; usRow < CLR_TBL_ROWS; usRow++ )
  {
    // For each column
    for ( usColumn=0; usColumn < CLR_TBL_COLUMNS; usColumn++ )
    {
        usIndex = (usRow * CLR_TBL_COLUMNS) + usColumn;

        if( ColorTable[usIndex] == lColor )
        {

          *pusRow    = usRow + 1;
          *pusColumn = usColumn + 1;

          return( TRUE );

        } /* end if */

    } /* end for */

  } /* end for */

  // No color match found
  return( FALSE );

} /* end MapColorToRowColumn */


/****************************************************************************/
/* PROCEDURE NAME : MapRowColumnToColor                                     */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 4/19/93                                                 */
/* DESCRIPTION    : This function helps manage the 'Select Color' dialog's  */
/*                  Available Colors Value set control.  It returns the     */
/*                  color for the Row and Column index passed in.           */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Calculate color index based on the Rows and Column  */
/*                      index passed in.                                    */
/*                                                                          */
/* PARAMETERS:      (USHORT) usRow    -  Row index of desired color         */
/*                  (USHORT) usColumn -  Column index of desired color      */
/*                                                                          */
/* RETURN VALUES:   (LONG)   Color table entry of desired color.            */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
LONG MapRowColumnToColor( USHORT usRow, USHORT usColumn )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  USHORT usIndex;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  usRow--;
  usColumn--;

  usIndex = (usRow * CLR_TBL_COLUMNS) + usColumn;

  // No color match found
  return( ColorTable[usIndex] );

} /* end MapRowColumnToColor */


/****************************************************************************/
/* PROCEDURE NAME : RetrieveCmdLineParms                                    */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 7/8/92                                                  */
/* DESCRIPTION    : Retrieve application parameters passed in, validate     */
/*                  them and set up program state data if valid.            */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) loop through the number of parameters passed in     */
/*                      and process recognized ones in a switch statement   */
/*                                                                          */
/* PARAMETERS:      (USHORT) count   - Number of parameters passed          */
/*                  (CHAR *) Parms[] - Parameter list                       */
/*                                                                          */
/* RETURN VALUES:   (BOOL) FALSE - Incorrect parameter syntax               */
/*                         TRUE  - Correct syntax and parameters            */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
BOOL RetrieveCmdLineParms( USHORT count, CHAR *Parms[] )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  BOOL bRC = TRUE;
  CHAR chParm;
  CHAR chParmPrefix;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  // while parms exist parse them
  while( --count > 0 )
  {
    // advance to next parm since one exists
    ++Parms;

    // get first character of parm which should be a parm prefix
    chParmPrefix = (*Parms)[0];

    // if a proper parm prefix is present
    if(( chParmPrefix == '-') || ( chParmPrefix == '/') )
    {
      // get the 2nd character of parm
      chParm = (*Parms)[1];

      // find out which parm it is and act accordingly
      switch( chParm )
      {
         case 'p':
         case 'P': // User wants to print from cmd line
                   PgmData.StateData.bCmdLinePrint  = TRUE;

                   // Turn off PM browse option
                   PgmData.StateData.bCmdLineBrowse = FALSE;
         break;

         case '?': // Tell user about syntax
                   WinMessageBox ( (HWND)HWND_DESKTOP,
                                   (HWND)NULL,
                                   (PSZ)MSG_SYNTAX_HELP,
                                   (PSZ)ERROR_TYPE_SYNTAX,
                                   1L, MB_OK | MB_ICONEXCLAMATION );
                   bRC = FALSE;

         break;

      } /* end switch */

    } /* end if */
    else
    {

      // copy in filename
      strcpy( filedlg.szFullFile, Parms[0] );

      PgmData.StateData.bCmdLineFileName = TRUE;

      if( PgmData.StateData.bCmdLinePrint != TRUE )
      {
        // Want to at least browse the file
        PgmData.StateData.bCmdLineBrowse = TRUE;

      } /* end if */

    } /* end if */

  } /* end while */

  // If user wants to print from cmd line but did not pass in a filename
  if( PgmData.StateData.bCmdLinePrint && !PgmData.StateData.bCmdLineFileName )
  {

     // Tell user about syntax
     WinMessageBox ( (HWND)HWND_DESKTOP,
                     (HWND)NULL,
                     (PSZ)MSG_SYNTAX_HELP,
                     (PSZ)ERROR_TYPE_SYNTAX,
                     1L, MB_OK | MB_ICONEXCLAMATION );
     bRC = FALSE;
     PgmData.StateData.bCmdLinePrint = FALSE;

  } /* end if */

  return( bRC );

} /* end RetrieveCmdLineParms */

/****************************************************************************/
/* PROCEDURE NAME : InitializeMenu                                          */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10/27/94                                                */
/* DESCRIPTION    : Performs all calls that create and add 3D buttons       */
/*                  to 3D menu control                                      */
/*                                                                          */
/*                  Control Flow:                                           */
/*                  (a) Process message passed in using a switch statement  */
/*                                                                          */
/* PARAMETERS:      */
/*                                                                          */
/* RETURN VALUES:   (MRESULT)                                               */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*                                                                          */
/****************************************************************************/
VOID InitializeMenu( ULONG ulStyle )
{

   // Createe 3-D menu buttons
   Create3DButton( hMenu3d, &hBtnFile,   IDM_FILE_MENU,   ID_FOLDER, "~File",   ulStyle, 0 );
   Create3DButton( hMenu3d, &hBtnQuery,  IDM_QUERY_MENU,  ID_QUERY,  "~Query",  ulStyle, 0 );
   Create3DButton( hMenu3d, &hBtnSearch, IDM_SEARCH_MENU, ID_SEARCH, "~Search", ulStyle, 0 );
   Create3DButton( hMenu3d, &hBtnHelp,   IDM_HELP_MENU,   ID_HELP,   "~Help",   ulStyle, 0 );

   // Create 3-D Menuitem buttons for "File" Submenu
   Create3DButton( hMenu3d, &hBtnSelectFont,  IDM_SELECT_FONT,     ID_FONT,     "Font",         ulStyle, 6 );
   Create3DButton( hMenu3d, &hBtnSelectColors,IDM_SELECT_COLORS,   ID_SELCOLOR, "Colors",       ulStyle, 6 );
   Create3DButton( hMenu3d, &hBtnSelectFormat,IDM_SELECT_FORMAT,   ID_SELFMT,   "Page Format",  ulStyle, 6 );

   Create3DButton( hMenu3d, &hBtnOpenFile,    IDM_OPEN_FILE,       ID_OPENFLDR, "~Open File...\tCtrl-O",      ulStyle, 1 );
   Create3DButton( hMenu3d, &hBtnSelectQueue, IDM_SELECT_QUEUE,    ID_PRINTER,  "Select ~Queue...\tCtrl-Q",   ulStyle, 1 );
   Create3DButton( hMenu3d, &hBtnSettings,    IDM_SETTINGS,        ID_SETTINGS, "Program Settings...",        ulStyle, 1 );
   Create3DButton( hMenu3d, &hBtnDefault,     IDM_DEFAULT_OPTIONS, ID_DEFOPTS,  "Default Settings",           ulStyle, 1 );
   Create3DButton( hMenu3d, &hBtnPrint,       IDM_PRINT,           ID_PRNTFILE, "~Print File...\tCtrl-PrtSc", ulStyle, 1 );

   // Create 3-D Menuitem buttons for "Query" Submenu
   Create3DButton( hMenu3d, &hBtnQCaps,   IDM_QUERY_CAPS,    ID_PRINTER, "DevQueryCaps...",          ulStyle, 2 );
   Create3DButton( hMenu3d, &hBtnQHCCaps, IDM_QUERY_HCCAPS,  ID_PAPER,   "DevQueryHardcopyCaps...",  ulStyle, 2 );
   Create3DButton( hMenu3d, &hBtnQFont,   IDM_QUERY_METRICS, ID_FONT,    "GpiQueryFontmetrics...",   ulStyle, 2 );
   Create3DButton( hMenu3d, &hBtnQWidth,  IDM_QUERY_FNTWDTH, ID_FONT,    "GpiQueryWidthTable...",    ulStyle, 2 );
   Create3DButton( hMenu3d, &hBtnQKern,   IDM_QUERY_FNTKERN, ID_FONT,    "GpiQueryKerningPairs...",  ulStyle, 2 );

   // Create 3-D Menuitem buttons for "Search" Submenu
   Create3DButton( hMenu3d, &hBtnSrchText, IDM_SEARCH_TEXT, ID_SRCHTEXT, "Search For Text...\tCtrl-S", ulStyle, 3 );

   // Create 3-D Menuitem buttons for "Help" Submenu
   Create3DButton( hMenu3d, &hBtnHelpHelp, IDM_HELP_HELP,  ID_REDCROSS, "Using Help...", ulStyle, 4 );
   Create3DButton( hMenu3d, &hBtnHelpIdx,  IDM_HELP_INDEX, ID_BOOK,     "Help Index...",    ulStyle, 4 );
   Create3DButton( hMenu3d, &hBtnHelpKeys, IDM_KEYS_HELP,  ID_KEYBOARD, "Special Keys...",  ulStyle, 4 );
   Create3DButton( hMenu3d, &hBtnHelpExt,  IDM_EXT_HELP,   ID_HELPPLUS, "Extended help...", ulStyle, 4 );
   Create3DButton( hMenu3d, &hBtnAbout,    IDM_ABOUT,      ID_LITEBULB, "About Program...", ulStyle, 4 );

   // Create 3-D pushbutton for "Select Queue..." Dialog
   Create3DButton( hMenu3d, &hBtnOther,   IDP_OTHER_DESTINATION, ID_PRINTER, "Other Destination...", ulStyle, 5 );
   Create3DButton( hMenu3d, &hBtnJobProp, IDP_JOB_PROPERTIES, ID_SELFMT, "Job Properties...",    ulStyle, 5 );
   Create3DButton( hMenu3d, &hBtnReview,  IDP_SETTINGS, ID_SETTINGS, "Review Settings",      ulStyle, 5 );

   // Create 3-D buttons representing Font Styles and Font Owning Device
   Create3DButton( hMenu3d, &hBtnFntSys, ID_FONTSYS, ID_FONTSYS, "", ulStyle, 6 );
   Create3DButton( hMenu3d, &hBtnFntPrn, ID_FONTPRN, ID_FONTPRN, "", ulStyle, 6 );
   Create3DButton( hMenu3d, &hBtnFntDsp, ID_FONTDSP, ID_FONTDSP, "", ulStyle, 6 );
   Create3DButton( hMenu3d, &hBtnFntATM, ID_FONTATM, ID_FONTATM, "", ulStyle, 6 );
   Create3DButton( hMenu3d, &hBtnFntBmp, ID_FONTBMP, ID_FONTBMP, "", ulStyle, 6 );

   // Add all 3-D buttons for menu
   Add3DButton( hMenu3d, hBtnFile );
   Add3DButton( hMenu3d, hBtnQuery );
   Add3DButton( hMenu3d, hBtnSearch );
   Add3DButton( hMenu3d, hBtnHelp );

   // Add all 3-D buttons for the "File" submenu
   Add3DButton( hMenu3d, hBtnOpenFile );
   Add3DButton( hMenu3d, hBtnSelectQueue );
   Add3DButton( hMenu3d, hBtnSelectFont  );
   Add3DButton( hMenu3d, hBtnSelectColors );
   Add3DButton( hMenu3d, hBtnSelectFormat );
   Add3DButton( hMenu3d, hBtnSettings );
   Add3DButton( hMenu3d, hBtnDefault );
   Add3DButton( hMenu3d, hBtnPrint );

   // Add all 3-D buttons for the "Query" submenu
   Add3DButton( hMenu3d, hBtnQCaps );
   Add3DButton( hMenu3d, hBtnQHCCaps );
   Add3DButton( hMenu3d, hBtnQFont );
   Add3DButton( hMenu3d, hBtnQWidth );
   Add3DButton( hMenu3d, hBtnQKern );

   // Add all 3-D buttons for the "Search" submenu
   Add3DButton( hMenu3d, hBtnSrchText );

   // Add all 3-D buttons for the "Help" submenu
   Add3DButton( hMenu3d, hBtnHelpIdx );
   Add3DButton( hMenu3d, hBtnHelpKeys );
   Add3DButton( hMenu3d, hBtnHelpHelp );
   Add3DButton( hMenu3d, hBtnHelpExt );
   Add3DButton( hMenu3d, hBtnAbout );

   // Add all 3-D dialog buttons
   Add3DButton( hMenu3d, hBtnOther );
   Add3DButton( hMenu3d, hBtnJobProp );
   Add3DButton( hMenu3d, hBtnReview );

   // Add all 3-D buttons for Font Styles/Font Owners
   Add3DButton( hMenu3d, hBtnFntSys );
   Add3DButton( hMenu3d, hBtnFntPrn );
   Add3DButton( hMenu3d, hBtnFntDsp );
   Add3DButton( hMenu3d, hBtnFntATM );
   Add3DButton( hMenu3d, hBtnFntBmp );


} /* end InitializeMenu */


/****************************************************************************/
/* PROCEDURE NAME : InitPgmData                                             */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 9/29/94                                                 */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:      */
/*                  */
/*                                                                          */
/* RETURN VALUES:   */
/*                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID InitPgmData( PPGMDATA pPgmData )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PSTATEDATA   pState = &pPgmData->StateData;
  PDISPLAYDATA pDisp  = &pPgmData->DisplayData;
  PPRINTERDATA pPrnt  = &pPgmData->PrinterData;

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  pState->bCmdLineBrowse   = FALSE;
  pState->bCmdLinePrint    = FALSE;
  pState->bCmdLineFileName = FALSE;
  pState->bCaseSensitive   = FALSE;
  pState->bFileOpen        = FALSE;
  pState->bHelpAvailable   = FALSE;

  pDisp->lHorzRes = 0;
  pDisp->lVertRes = 0;
  pDisp->lHorzFontRes = 0;
  pDisp->lVertFontRes = 0;

  pPrnt->pHcInfo = NULL;
  pPrnt->pFontMetrics = NULL;

} /* end InitPgmData */

/****************************************************************************/
/* PROCEDURE NAME : InitDisplayData                                         */
/* AUTHOR         : MFR                                                     */
/* DATE WRITTEN   : 10/13/94                                                */
/* DESCRIPTION    : */
/*                                                                          */
/* PARAMETERS:      */
/*                  */
/*                                                                          */
/* RETURN VALUES:   */
/*                  */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/* CHANGE/MODIFICATION LOG :                                                */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/****************************************************************************/
VOID InitDisplayData( PPGMDATA pPgmData )
{
  /*------------------------------------------------------------------------*/
  /* LOCAL VARIABLES                                                        */
  /*------------------------------------------------------------------------*/
  PDISPLAYDATA pDisplayData = &(pPgmData->DisplayData);
  DEVOPENSTRUC dop;          // Memory DC with display
  SIZEL        sizl = {0,0}; // use default page size of device

  /*------------------------------------------------------------------------*/
  /* BEGIN CODE                                                             */
  /*------------------------------------------------------------------------*/

  /*---------------------------------*/
  /* Open Display Memory DC          */
  /*---------------------------------*/
  dop.pszLogAddress = NULL;
  dop.pszDriverName = (PSZ) "DISPLAY";
  dop.pdriv         = NULL;
  dop.pszDataType   = NULL;

  pDisplayData->hdcMem = DevOpenDC( PgmData.habPgm,
                                    OD_MEMORY,
                                    "*",
                                    4L,
                                    (PDEVOPENDATA)&dop,
                                    NULLHANDLE );

  pDisplayData->hpsMem = GpiCreatePS( PgmData.habPgm,
                                      pDisplayData->hdcMem,
                                      &sizl,
                                      PU_PELS | GPIT_MICRO | GPIA_ASSOC );

  /*---------------------------------*/
  /* Get Display capabilities        */
  /*---------------------------------*/

  // Pels per inch for display
  DevQueryCaps( pDisplayData->hdcMem, CAPS_HORIZONTAL_FONT_RES,
                2L, &(pDisplayData->lHorzFontRes) );

  DevQueryCaps( pDisplayData->hdcMem, CAPS_HORIZONTAL_RESOLUTION,
                2L, &(pDisplayData->lHorzRes) );

  /*---------------------------------*/
  /* Calculate display DPI           */
  /*---------------------------------*/
  pDisplayData->fHorzDPI = (((float)pDisplayData->lHorzRes)/100.0)*2.54;
  pDisplayData->fVertDPI = (((float)pDisplayData->lVertRes)/100.0)*2.54;

  /*--------------------------------------------------------*/
  /* Fill in a default set of FATTRS for display            */
  /*--------------------------------------------------------*/
  pDisplayData->fattrsDefault.usRecordLength  = sizeof(FATTRS);
  pDisplayData->fattrsDefault.fsSelection     = 0;
  pDisplayData->fattrsDefault.lMatch          = 0;
  pDisplayData->fattrsDefault.idRegistry      = 0;
  pDisplayData->fattrsDefault.lMaxBaselineExt = 0L;
  pDisplayData->fattrsDefault.lAveCharWidth   = 0L;
  pDisplayData->fattrsDefault.fsType          = 0;

  // @DBCS - set codepage to system codepage so font realized
  // with correct codepage (not font's default codepage (i.e. not 0)
  pDisplayData->fattrsDefault.usCodePage      = CodePage;
  pDisplayData->fattrsDefault.fsFontUse       = FATTR_FONTUSE_NOMIX &
                                                FATTR_FONTUSE_OUTLINE;

  strcpy( pDisplayData->fattrsDefault.szFacename, FT_DEFAULT_FONT );

  // Get Fontmetrics of font we'll display in client window
  GpiQueryFontMetrics( hps, sizeof(FONTMETRICS), &(pDisplayData->fm) );

  cxChar = (ULONG) pDisplayData->fm.lAveCharWidth;
  cyChar = (ULONG) pDisplayData->fm.lMaxBaselineExt;
  cyDesc = (ULONG) pDisplayData->fm.lMaxDescender;

} /* end InitDisplayData */

