/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = CDSTRAT1.C
 *
 * DESCRIPTIVE NAME = Strategy 1 interface for OS/2 CDROM Device Manager
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION
 *
 *
 * FUNCTIONS   Provides validation and routing of Strategy 1 requests
 *             received from the OS/2 Kernel.
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/


#include "cdh.h"

USHORT NonSCSI_GetLastSessionAddr (NPUNITCB, ULONG FAR *);


CMDFUNC functable[]=                                                    //SD@135221
{                          /*--------------------------------------*/   //SD@135221
   {CD_DriveInit,   TRUE}, /* 0x00  initialize                     */   //SD@135221
   {CD_MediaCheck,  FALSE}, /* 0x01  check the media                */   //SD@135221
   {CD_BuildBPB,    TRUE,}, /* 0x02  build BPB                      */   //SD@135221
   {CmdErr,         TRUE}, /* 0x03  reserved                       */   //SD@135221
   {CD_Read,        FALSE}, /* 0x04  read                           */   //SD@135221
   {CmdErr,         TRUE}, /* 0x05  non-destructive read           */   //SD@135221
   {CmdErr,         TRUE}, /* 0x06  input status                   */   //SD@135221
   {CmdErr,         TRUE}, /* 0x07  input flush                    */   //SD@135221
   {CD_Write,       TRUE}, /* 0x08  write                          */   //SD@135221
   {CD_WriteV,      TRUE}, /* 0x09  write with verify              */   //SD@135221
   {CmdErr,         TRUE}, /* 0x0A  get output status              */   //SD@135221
   {CmdErr,         TRUE}, /* 0x0B  flush output                   */   //SD@135221
   {CmdErr,         TRUE}, /* 0x0C  reserved                       */   //SD@135221
   {StatusComplete, TRUE}, /* 0x0D  open                           */   //SD@135221
   {StatusComplete, TRUE}, /* 0x0E  close                          */   //SD@135221
   {RemovableMedia, TRUE}, /* 0x0F  removable media                */   //SD@135221
   {DriveGenIOCTL,  TRUE},  /* 0x10  generic IOCTL                  */   //SD@135221
   {ResetMedia,     FALSE}, /* 0x11  reset uncertain media          */   //SD@135221
   {GetLogDriveMap, TRUE}, /* 0x12  get Logical Drive Map          */   //SD@135221
   {SetLogDriveMap, TRUE}, /* 0x13  set Logical Drive Map          */   //SD@135221
   {CmdErr,         TRUE}, /* 0x14  de-Install this device         */   //SD@135221
   {CmdErr,         TRUE}, /* 0x15  reserved                       */   //SD@135221
   {PartFixedDisks, TRUE}, /* 0x16  get number of partitions       */   //SD@135221
   {GetUnitMap,     TRUE}, /* 0x17  get unit map                   */   //SD@135221
   {CD_Read,        FALSE}, /* 0x18  no caching read                */   //SD@135221
   {CD_Write,       TRUE}, /* 0x19  no caching write               */   //SD@135221
   {CD_WriteV,      TRUE}, /* 0x1A  no caching write/verify        */   //SD@135221
   {CD_DriveInit,   TRUE}, /* 0x1B  initialize                     */   //SD@135221
   {StatusComplete, TRUE}, /* 0x1C  prepare for shutdown           */   //SD@135221
   {CmdErr,         TRUE}, /* 0x1D  Get Driver Capabilities        */   //SD@135221
};                    /*--------------------------------------*/        //SD@135221


/****************************************************************************
 *
 * FUNCTION NAME = CD_Strat1
 *
 * DESCRIPTION   = OS2CDROM strategy 1 routine
 *
 * INPUT         = ES:BX     - pointer to Request Packet
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 *
 * RETURN-ERROR  =
 *
 ****************************************************************************/

void near CD_Strat1(pRPH)

PRPH pRPH;
{
   NPUNITCB      pUnitCB;
   USHORT        Cmd, Status;


   pRPH->Status = 0;
   pRPH->Flags = 0;

   Cmd = pRPH->Cmd;


   /*
   ** Filter out invalid requests
   */
/*
**if (DDFlags & DDF_NO_MEDIA)
**{
**    Status = STDON + STERR + ERROR_I24_BAD_UNIT;
**    goto ExitDiskDD;
**}
*/
   if (Cmd > MAX_DISKDD_CMD)
   {
      Status = STDON + STERR + ERROR_I24_BAD_COMMAND;
      goto  ExitDiskDD;
   }

   if ( (Get_UnitCB_Addr(pRPH->Unit, (NPUNITCB FAR *) &pUnitCB) != NO_ERROR) &&
        (Cmd != CMDInitBase) &&
        (Cmd != CMDInit) &&
        (Cmd != CMDPartfixeddisks) )
   {
      Status = STDON + STERR + ERROR_I24_BAD_UNIT;
      goto ExitDiskDD;
   }

#ifdef TRACING
   if ( (Cmd != CMDInitBase) && (Cmd != CMDInit) && IsTraceOn() )  /*@Perftrace*/
      Trace(TRACE_STRAT1 | TRACE_ENTRY, (PBYTE) pRPH, pUnitCB); /*@Perftrace*/
#endif

   /*
   ** Call Worker Routine
   */
   if ((CDFlags & CDF_INIT_COMPLETE) &&                                   //SD@135221
       (pUnitCB->DeviceInfo.Audio.capabilities & (DCAPS_CARTRIDGE_CHANGER |DCAPS_INDIVIDUAL_CHANGER)))//SD@135221
   {
      //SD@135221
      // only ATAPI devices support this flag today                       //SD@135221
      if (pUnitCB->pParentUnitCB)                                          //SD@135221
      {
         //SD@135221
         // serialize here                                                 //SD@135221
#define SEM_INDEFINITE_WAIT ((ULONG)(-1L))                        //SD@135221
         DevHelp_SemRequest((ULONG)(PVOID)(&pUnitCB->pParentUnitCB->Semaphore),SEM_INDEFINITE_WAIT);//SD@135221   // 10/23/2000 MB
         // if we have a parent, and the active slot is not ours           //SD@135221
         if (!(pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_SINGLE_MODE)) //SD@135221
         {
            //SD@135221
            if (pUnitCB->pParentUnitCB && (pUnitCB->pParentUnitCB->DeviceInfo.Slots.Current!=pUnitCB->DeviceInfo.Slot))//SD@135221
            {
               //SD@135221
               // load cd for requested Unit, may fail (no disk etc)         //SD@135221
               Status = MakeSlotActive(pUnitCB,(UCHAR)pUnitCB->DeviceInfo.Slot);//SD@135221
               if (Status!=STDON && functable[Cmd].ChangerAllowed==TRUE)      //SD@135221
               {
                  //SD@135221
                  Status=STDON;                                               //SD@135221
               } /* endif */                                               //SD@135221
            }                                                             //SD@135221
            else                                                            //SD@135221
            {
               //SD@135221
               Status=STDON;                                                 //SD@135221
            }                                                             //SD@135221
         } /* endif */                                                   //SD@135221
         else                                                              //SD@135221
         {
            //SD@135221
            Status=STDON;                                                   //SD@135221
         }                                                               //SD@135221
         // if slot is now active one                                      //SD@135221
         if (Status==STDON)                                                 //SD@135221
         {
            //SD@135221
            Status = (*functable[Cmd].Strat1Near)(pRPH, pUnitCB);           //SD@135221
         } /* endif */                                                   //SD@135221
         // deserialize here                                               //SD@135221
         DevHelp_SemClear((ULONG)(PVOID)(&pUnitCB->pParentUnitCB->Semaphore));    //SD@135221   // 10/23/2000 MB
      } /* endif */                                                     //SD@135221
   } /* endif */                                                       //SD@135221
   else                                                                  //SD@135221
      Status = (*functable[Cmd].Strat1Near)(pRPH, pUnitCB);

   /*
   ** Finish up by setting the Status word in the Request Packet Header
   */

   ExitDiskDD:  ;

   DISABLE;

   pRPH->Status = Status;                  /* Save status in Request Packet */

   ENABLE;

#ifdef DEBUG
   dprintf("OS2CDROM CD_Strat1 Cmd=%x, status=%x, pUnitCB=%x\r\n", (PVOID)&Cmd, (PVOID)&Status, (PVOID)&pUnitCB);
#endif

#ifdef TRACING
   if ( (TraceFlags != 0) && (pRPH->Status & STDON) && (Cmd != CMDInitBase) )  /*@Perftrace*/
      Trace(TRACE_STRAT1 | TRACE_EXIT, (PBYTE) pRPH, pUnitCB);  /*@Perftrace*/
#endif
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_MediaCheck
 *
 * DESCRIPTION   = Check the Media    (Command = 0x01)
 *
 *       Checks to see if the media in the drive has changed.
 *
 *       USHORT CD_MediaCheck   (PRP_MEDIACHECK pRP, NPUNITCB pUnitCB)
 *
 *       EFFECTS:  The Return Code in the Request Packet is set to one
 *                 of the following:
 *
 *                     -1 = Media has been changed
 *                      0 = Unsure if media has been changed
 *                      1 = Media unchanged
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT NEAR CD_MediaCheck(pRP, pUnitCB)

PRP_MEDIACHECK    pRP;
NPUNITCB          pUnitCB;
{
   USHORT rc;
   USHORT cTries;                                                    /*V@91985*/
   NPIORB pIORB;
   BOOL   playing;                                                   //SD@135221

   if (pUnitCB->Flags & UCF_UNCERTAIN_MEDIA)
   {
      pRP->rc = -1;
      return (STDON);
   }

   if (pUnitCB->DeviceInfo.playing==FALSE &&                         //SD@135221
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))//SD@135221
   {
      //SD@135221
      rc = GetPlayStatus(pUnitCB, &playing);                         //SD@135221
                                                                     //SD@135221
      if ( (rc == STDON) && playing)                                 //SD@135221
      {
         //SD@135221
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);            //SD@135221
      }                                                              //SD@135221
   }                                                                 //SD@135221

   BuildCDB_TestUnitReady (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

   rc = SubmitIORB_Wait (pUnitCB, pIORB);

   if (pIORB->Status & IORB_ERROR)
   {

      if ( ((NPIORB_CDB)pIORB)->sense_data.additional_sense_code !=     /*@V91985*/
           ASC_MEDIUM_NOT_PRESENT )                /*@V91985*/
      {
         /*@V91985*/
         if (pUnitCB->DeviceInfo.Audio.capabilities & (DCAPS_CARTRIDGE_CHANGER |DCAPS_INDIVIDUAL_CHANGER))//SD@135221
         {
            //SD@135221
            if (pUnitCB->DeviceInfo.product_id_code==TORISAN_C3G)          //SD@135221
            {
               //SD@135221
               UCHAR asc,ascq;                                             //SD@135221
               asc=((NPIORB_CDB)pIORB)->sense_data.additional_sense_code;  //SD@135221
               ascq=((NPIORB_CDB)pIORB)->sense_data.additional_sense_code_qualifier;//SD@135221
               if (asc==ASC_MEDIUM_CHANGED && ascq==0)                      //SD@135221
               {
                  //SD@135221
                  // only ATAPI devices support this flag today             //SD@135221
                  // load cd for requested Unit, may fail (no disk etc)     //SD@135221
                  if (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_SINGLE_MODE)//SD@135221
                  {
                     //SD@135221
                     MakeSlotActive(pUnitCB,(UCHAR)pUnitCB->pParentUnitCB->DeviceInfo.Slots.Current);//SD@135221
                  } /* endif */                                           //SD@135221
                  else                                                      //SD@135221
                  {
                     //SD@135221
                     MakeSlotActive(pUnitCB,(UCHAR)pUnitCB->DeviceInfo.Slot);//SD@135221
                  } /* endelse */                                         //SD@135221
               } /* endif */                                             //SD@135221
            } /* endif */                                               //SD@135221
         }                                                             //SD@135221
         cTries = 0;                                                    /*@V91985*/
         while ( (((NPIORB_CDB)pIORB)->sense_data.sense_key ==          /*@V91985*/
                  SCSI_SK_NOTRDY )         /*@V91985*/
                 && (cTries++ < MAX_NOT_READY_RETRIES ))                /*@V91985*/
         {
            /*@V91985*/
            FreeIORB (pUnitCB, (NPIORB_CDB) pIORB);                     /*@V91985*/
            DevHelp_ProcBlock( (ULONG)(PVOID) pIORB, NOT_READY_WAIT,    /*@V91985*/ // 10/23/2000 MB
                               WAIT_IS_INTERRUPTABLE );         /*@V91985*/
            BuildCDB_TestUnitReady (pUnitCB,                            /*@V91985*/
                                    (NPIORB_CDB FAR *) &pIORB);         /*@V91985*/
            rc = SubmitIORB_Wait (pUnitCB, pIORB);                      /*@V91985*/
         }                                                              /*@V91985*/
      }                                                                 /*@V91985*/
   } /* endif */

   pRP->rc = (UCHAR)(! (pUnitCB->Flags & UCF_UNCERTAIN_MEDIA));   // 10/23/2000 MB

   FreeIORB (pUnitCB, (NPIORB_CDB) pIORB);

   return (rc);

}


/****************************************************************************
 *
 * FUNCTION NAME = CD_BuildBPB
 *
 * DESCRIPTION   = Build the BPB      (Command = 0x02)
 *
 *       Builds the BPB.  This is requested when the media has changed
 *       of when the media type is uncertain.
 *
 *       USHORT BuildBPB   (PRP_BUILDBPB pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT NEAR CD_BuildBPB(pRP, pUnitCB)
PRP_BUILDBPB   pRP;
NPUNITCB       pUnitCB;
{
   USHORT rc;
   ULONG  volume_size;

   if ( (rc = GetVolumeSize (pUnitCB, (ULONG FAR *) &volume_size)) & STERR)
      return (rc);

   DefaultBPB.BigTotalSectors = volume_size;

   pRP->bpb = (PVOID) &(DefaultBPB);

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_Read
 *
 * DESCRIPTION   = Read Command  (Command 0x04)
 *
 *         This is the basic strategy-1 I/O read routine for the driver.
 *         The request is queued and sent to the adapter driver for
 *         processing.
 *
 *         USHORT CD_Read   (PRP_RWV pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_Read (pRP, pUnitCB)

PRP_RWV   pRP;
NPUNITCB  pUnitCB;
{
   USHORT rc;
   BOOL   playing,
   MountDrive = FALSE;                                        /*@V91985*/
   PULONG pBuff;                                                     /*@V91985*/
   PULONG i;                                                         /*@V96674*/
   USHORT ModeFlag;                                                  /*@V91985*/
   NPIORB_CDB pIORB = 0, pModeIORB = 0;

   /*
   ** Check for valid input parms
   */
   if (pRP->NumSectors == 0)
      return (STDON);
/*
** if ((pRP->rba + pRP->NumSectors) > pUnitCB->TotalSectors)
**    return (STDON + STERR + ERROR_I24_SECTOR_NOT_FOUND);
*/
   /*
   ** Check for uncertain media
   */
   if (pUnitCB->Flags & UCF_UNCERTAIN_MEDIA)
   {
      pRP->NumSectors = 0;
      return (STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA);
   }

   /*
   ** Cant read when unit is playing so check play status
   */
   if (pUnitCB->DeviceInfo.playing ||                           //SD@135221
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))//SD@135221
   {
      //SD@135221
      rc = GetPlayStatus(pUnitCB, &playing);

      if ( (rc == STDON) && playing)
      {
         pRP->NumSectors = 0;
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);
      }
   }                                                            //SD@135221

   /*
   ** If multisession photo CD mounted and the target read is one of the
   ** Volume Descriptor Sectors, then remap the RBA to the last session
   */
   if ( (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION) &&
        (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION_MOUNTED) &&
        (pRP->rba >= PRIMARY_VOL_DESCR_RBA) &&
        (pRP->rba <= pUnitCB->DeviceInfo.volume_descr_terminator) )
   {
      pRP->rba = pRP->rba + pUnitCB->DeviceInfo.last_session_addr;
   }

   /*                                                                  @V91985
   ** Determine if the File System is attempting to mount the Drive.   @V91985
   */                                                                /*@V91985*/

   if ( (pRP->rba == 0) &&                                           /*@V91985*/
        (pRP->NumSectors == 1) )                                     /*@V91985*/
   {
      /*@V91985*/
      if ( DevHelp_PhysToVirt( (ULONG) pRP->XferAddr,                /*@V91985*/
                               (USHORT) 2048,                        /*@V96674*/
                               (PVOID)  &pBuff,                      /*@V91985*/
                               (PUSHORT) &ModeFlag   ) )             /*@V91985*/
      {
         /*@V91985*/
         _asm {int 3}                                                /*@V91985*/
      }                                                              /*@V91985*/
      /*@V91985*/
      if (*pBuff == 0x544f4f42l) /* Boot Signature */                /*@V91985*/
      {
         /*@V91985*/
         MountDrive = TRUE;                                          /*@V91985*/
      }                                                              /*@V91985*/
   }                                                                 /*@V91985*/


   /*
   ** Toshiba 3301 & 3401 is vendor unique since we must make sure the
   ** density code is set properly.
   */
   switch (pUnitCB->DeviceInfo.product_id_code)
   {
   case TOSHIBA_3301:
   case TOSHIBA_3401:
      rc = Tosh_Read_2048(pUnitCB, pRP->rba, pRP->NumSectors, pRP->XferAddr);
      break;


   default:

      if (pUnitCB->DeviceInfo.raw_block_size)
      {
         if (!( pUnitCB->DeviceInfo.interface_type == INTERFACE_ATAPI ))
            rc = Submit_ModeSelect (pUnitCB, CD_DENSITY_DEFAULT, 2048);
         pUnitCB->DeviceInfo.raw_block_size = 0;
      }

      if ( (pUnitCB->DeviceInfo.current_block_size != 2048) &&
           (pUnitCB->DeviceInfo.interface_type != INTERFACE_ATAPI) )
      {
         rc = ChainModeSelectRead (pUnitCB, pRP->rba, pRP->NumSectors,
                                   pRP->XferAddr, 2048, 0);
      }
      else
      {
         rc = ReadSector (pUnitCB, pRP->rba, pRP->NumSectors,
                          pRP->XferAddr, 2048);
      }
   }

   /*
   ** Sony 561 can't read 2048 byte Mode 2 Form 2 XA sectors with the
   ** Read 6 command. We must issue the Read CD-XA command.
   */
   if ( (pUnitCB->DeviceInfo.product_id_code == SONY_561) &&
        (rc == STDON + STERR + ERROR_I24_SECTOR_NOT_FOUND) )
   {
      rc = Sony_Read_2048(pUnitCB, pRP->rba, pRP->NumSectors, pRP->XferAddr);
   }

   /*
   ** If reading sector 0 on an NEC SCSI drive and it's an XA disk then change
   ** the density code to XA, and reissue the read.
   */
   if ( (rc == STDON + STERR + ERROR_I24_READ_FAULT) &&
        (pUnitCB->DeviceInfo.vendor_id_code == NEC) &&
        (pUnitCB->DeviceInfo.interface_type == INTERFACE_SCSI) &&
        (pRP->rba == 0) )
   {
      rc = ChainModeSelectRead (pUnitCB, pRP->rba, pRP->NumSectors,
                                pRP->XferAddr, 2048, 0x81);
   }

   if (rc & STERR)
      pRP->NumSectors = 0;

   if (rc == STDON + STERR + ERROR_I24_BAD_COMMAND)
      rc = STDON + STERR + ERROR_I24_SECTOR_NOT_FOUND;

   /*
   ** If we are mounting the volume, then check for multi-session disk
   */
   if ( (MountDrive) &&                                                 /*@142227*/
        (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION) ) /*@142227*/
   {
      Check_MultiSession_Mounted(pUnitCB, pRP->XferAddr);
   }


   /*                                                                  @V91985
   ** If the File System is attempting to mount the drive, and there   @V91985
   ** was an error that was not a drive ready error, give the error    @V91985
   ** (if any) from a read of the volumne descriptor.                  @V91985
   ** Clear the buffer if the read was successful.                     @V91985
   */                                                                /*@V91985*/
   /*@V91985*/
   if (MountDrive &&     // Atempting to mount a disc                /*@V96674*/
       (rc != STDON) &&   // An error was encountered                 /*@V96674*/
       (rc != STDON + STERR + ERROR_I24_NOT_READY) &&                 /*@V91985*/
       (rc != STDON + STERR + ERROR_I24_DISK_CHANGE) &&               /*@V91985*/
       (rc != STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA) )            /*@V91985*/
   {
      /*@V91985*/
      if ((rc = ReadSector(        pUnitCB,                       /*@V96674*/
                                   (ULONG) PRIMARY_VOL_DESCR_RBA,         /*@V96674*/
                                   1,                             /*@V96674*/
                                   pRP->XferAddr,                 /*@V96674*/
                                   2048)) == STDON )              /*@V96674*/
      {

         if ( DevHelp_PhysToVirt( (ULONG) pRP->XferAddr,          /*@V132783*/
                                  (USHORT) 2048,                  /*@V132783*/
                                  (PVOID)  &pBuff,                /*@V132783*/
                                  (PUSHORT) &ModeFlag   ) )       /*@V132783*/
         {
            /*@V132783*/
            _asm {int 3}                                          /*@V132783*/
         }                                                        /*@V132783*/
         /*@V91985*/
         /* zero out buffer (512 ULONGS = 2048 bytes) */
         for (i=pBuff; i<pBuff+512; i++ )                         /*@V96674*/
         {
            /*@V96674*/
            *i = 0l;                                              /*@V96674*/
         }                                                        /*@V96674*/

      } /* if rc=ReadSector */                                    /*@V96674*/
   }                                                              /*@V91985*/

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = RemovableMedia
 *
 * DESCRIPTION   = Check for Removable Media    (Command = 0x0F)
 *
 *        USHORT RemovableMedia (PRPH pRPH, NPUNITCB pUnitCB)
 *
 *        EFFECTS:  The busy bit of the status word is set as follows:
 *
 *                       1 = Media is non-removable
 *                       0 = Media is removable
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT RemovableMedia(pRPH, pUnitCB)

PRPH     pRPH;
NPUNITCB pUnitCB;
{
   if (pUnitCB->UnitInfo.UnitFlags & UF_REMOVABLE)
      return (STDON);
   else
      return (STDON + STBUI);
}



/****************************************************************************
 *
 * FUNCTION NAME = ResetMedia
 *
 * DESCRIPTION   = Reset Uncertain Media  (Command = 0x11)
 *
 *                 USHORT ResetMedia (PRPH pRPH, NPUNITCB pUnitCB)
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT ResetMedia (pRPH, pUnitCB)

PRPH     pRPH;
NPUNITCB pUnitCB;
{
   USHORT rc;
   BOOL playing;                                                //SD@135221

   if (pUnitCB->pParentUnitCB &&                                //SD@135221
       pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying)        //SD@135221
   {
      //SD@135221
      rc = GetPlayStatus(pUnitCB, &playing);                    //SD@135221
                                                                //SD@135221
      if ( (rc == STDON) && playing)                            //SD@135221
      {
         //SD@135221
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);       //SD@135221
      }                                                         //SD@135221
   }                                                            //SD@135221
   pUnitCB->Flags &= ~UCF_UNCERTAIN_MEDIA;
   pUnitCB->Flags &= ~UCF_PACKET_WRITABLE;

   /*
   ** Issue a Mode Select to set default density code to CD-ROM
   ** and default block size to 2048 bytes.
   */
   if (!( pUnitCB->DeviceInfo.interface_type == INTERFACE_ATAPI ))
      rc = Submit_ModeSelect (pUnitCB, CD_DENSITY_DEFAULT, 2048);

   pUnitCB->DeviceInfo.raw_block_size = 0;

   pUnitCB->MediaType = GetMediumType(pUnitCB);
#ifdef   DEBUG
   {
      ULONG uMedia=(ULONG)pUnitCB->MediaType;
      dprintf("OS2CDROM ResetMedia MediaType=%x\r\n", (PVOID)&uMedia);
   }
#endif
   if (isDVDMedia(pUnitCB->MediaType))   // AK: 04/04/02, defect 269726
   {
      pUnitCB->Flags |= UCF_DVD_MEDIA;
   }
   if (pUnitCB->MediaType == DMD_CDRW || pUnitCB->MediaType == DMD_CDR)
   {
      rc = SetPacketInfo(pUnitCB);
      if (rc == STDON)
         pUnitCB->Flags |= UCF_PACKET_WRITABLE;
      FlushBuffers(pUnitCB);    // AK (10/27/00): Free buffers 
   }

   return (STDON);
}


/****************************************************************************
 *
 * FUNCTION NAME = GetLogDriveMap
 *
 * DESCRIPTION   = Get Logical Drive Mapping  (Command = 0x12)
 *
 *        Returns which logical drive is currently mapped onto a particular
 *        physical drive.  A zero is returned if only one logical drive is
 *        mapped to the physical drive.
 *
 *        USHORT GetLogDriveMap (PRPH pRPH, NPUNITCB pUnitCB)
 *
 *        EFFECTS: The logical drive is returned in the unit field
 *                 of the request packet header.  The logical drive
 *                 is actually LogDriveNum + 1, which represents the
 *                 drive letter, i.e. C: = 3.  A zero is returned
 *                 if only one logical drive is mapped to the physical drive.
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT GetLogDriveMap (pRPH, pUnitCB)

PRPH     pRPH;
NPUNITCB pUnitCB;
{

   pRPH->Unit = 0;                             /* ret 0 if one drive mapped */

   return (STDON);

}


/****************************************************************************
 *
 * FUNCTION NAME = SetLogDriveMap
 *
 * DESCRIPTION   = Set Logical Drive Mapping  (Command = 0x13)
 *
 *         Maps the specified logical drive onto the physical drive.
 *
 *         USHORT SetLogDriveMap (PRPH pRPH, NPVOLCB pVolCB)
 *
 *         EFFECTS: The logical drive is returned in the unit field
 *                  of the request packet header.  The logical drive
 *                  is actually LogDriveNum + 1, which represents the
 *                  drive letter, i.e. C: = 3.  A zero is returned
 *                  if only one logical drive is mapped to the physical drive.
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT SetLogDriveMap (pRPH, pUnitCB)

PRPH     pRPH;
NPUNITCB pUnitCB;

{
   pRPH->Unit = 0;                            /* only 1 drive can be mapped */

   return (STDON);
}


/****************************************************************************
 *
 * FUNCTION NAME = PartFixedDisks
 *
 * DESCRIPTION   = Get number of fixed disks  (Command = 0x16)
 *
 *        Returns the number of fixed disks supported by the driver.
 *
 *        USHORT PartFixedDisks (PRP_PARTFIXEDDISKS pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT PartFixedDisks (pRP, pUnitCB)

PRP_PARTFIXEDDISKS  pRP;
NPUNITCB            pUnitCB;
{
   pRP->NumFixedDisks = 0;

   return (STDON);
}


/****************************************************************************
 *
 * FUNCTION NAME = GetUnitMap
 *
 * DESCRIPTION   = Get logical units mapped to a physical unit (Command = 0x17)
 *
 *        This command is only supported for disk media.
 *
 *        USHORT GetUnitMap (PRP_GETUNITMAP pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT GetUnitMap (pRP, pUnitCB)

PRP_GETUNITMAP  pRP;
NPUNITCB        pUnitCB;
{

   pRP->UnitMap = 0;                                     /* No units mapped */

   return (STDON);
}



/*
** Near Entry Point to swappable IOCTL routine.
*/

/****************************************************************************
 *
 * FUNCTION NAME = DriveGenIOCTL
 *
 * DESCRIPTION   =
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT DriveGenIOCTL(pRP, pUnitCB)

PRP_GENIOCTL pRP;
NPUNITCB     pUnitCB;
{
   USHORT rc;

   rc = f_CD_DriveGenIOCTL (pRP, pUnitCB);

   rc |= STDON;

   return (rc);
}


/*
**   Packet Status return functions:
**
**   CmdErr, StatusError, StatusDevReady, StatusComplete
*/

/****************************************************************************
 *
 * FUNCTION NAME = CmdErr
 *
 * DESCRIPTION   =
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT near CmdErr (pRPH, pUnitCB)
PRPH     pRPH;
NPUNITCB pUnitCB;
{
   return (STERR + STDON + ERROR_I24_BAD_COMMAND);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_Write
 *
 * DESCRIPTION   = Write Command  (Commands 0x08,0x19)
 *
 *         This is the basic strategy-1 I/O write routine for the driver.
 *         The request is queued and sent to the adapter driver for
 *         processing.
 *
 *         USHORT CD_Write   (PRP_RWV pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_Write (pRP, pUnitCB)

PRP_RWV   pRP;
NPUNITCB  pUnitCB;
{
   USHORT rc;
   BOOL   playing,
   MountDrive = FALSE;
   PULONG pBuff;
   PULONG i;
   USHORT ModeFlag;
   NPIORB_CDB pIORB = 0, pModeIORB = 0;

   /*
   ** Check for valid input parms
   */
   if (pRP->NumSectors == 0)
      return (STDON);

   /*
   ** Check for uncertain media
   */
   if (pUnitCB->Flags & UCF_UNCERTAIN_MEDIA)
   {
      pRP->NumSectors = 0;
      return (STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA);
   }

   /*
   ** Cant write when unit is playing so check play status
   */
   if (pUnitCB->DeviceInfo.playing ||
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))
   {
      rc = GetPlayStatus(pUnitCB, &playing);

      if ( (rc == STDON) && playing)
      {
         pRP->NumSectors = 0;
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);
      }
   }

   /*
   ** If multisession photo CD mounted and the target write is one of the
   ** Volume Descriptor Sectors, then remap the RBA to the last session
   */
   if ( (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION) &&
        (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION_MOUNTED) &&
        (pRP->rba >= PRIMARY_VOL_DESCR_RBA) &&
        (pRP->rba <= pUnitCB->DeviceInfo.volume_descr_terminator) )
   {
      pRP->rba = pRP->rba + pUnitCB->DeviceInfo.last_session_addr;
   }

   /*
   ** Determine if the File System is attempting to mount the Drive.
   */

   if ( (pRP->rba == 0) &&
        (pRP->NumSectors == 1) )
   {
      if ( DevHelp_PhysToVirt( (ULONG) pRP->XferAddr,
                               (USHORT) 2048,
                               (PVOID)  &pBuff,
                               (PUSHORT) &ModeFlag   ) )
      {
         _asm {int 3}
      }

      if (*pBuff == 0x544f4f42l) /* Boot Signature */
      {
         MountDrive = TRUE;
      }
   }


   if ( (pUnitCB->DeviceInfo.current_block_size != 2048) &&
        (pUnitCB->DeviceInfo.interface_type != INTERFACE_ATAPI) )
   {
      rc = ChainModeSelectWrite (pUnitCB, pRP->rba, pRP->NumSectors,
                                 pRP->XferAddr, 2048, 0, FALSE);
   }
   else
   {
      rc = WriteSector (pUnitCB, pRP->rba, pRP->NumSectors,
                        pRP->XferAddr, 2048, FALSE);
   }


   if (rc & STERR)
      pRP->NumSectors = 0;

   if (rc == STDON + STERR + ERROR_I24_BAD_COMMAND)
      rc = STDON + STERR + ERROR_I24_SECTOR_NOT_FOUND;

   /*
   ** If we are mounting the volume, then check for multi-session disk
   */
   if ( (MountDrive) &&
        (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION) )
   {
      Check_MultiSession_Mounted(pUnitCB, pRP->XferAddr);
   }


   /*
   ** If the File System is attempting to mount the drive, and there
   ** was an error that was not a drive ready error, give the error
   ** (if any) from a write of the volumne descriptor.
   ** Clear the buffer if the write was successful.
   */

   if (MountDrive &&     // Atempting to mount a disc
       (rc != STDON) &&   // An error was encountered
       (rc != STDON + STERR + ERROR_I24_NOT_READY) &&
       (rc != STDON + STERR + ERROR_I24_DISK_CHANGE) &&
       (rc != STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA) )
   {
      if ((rc = WriteSector(        pUnitCB,
                                    (ULONG) PRIMARY_VOL_DESCR_RBA,
                                    1,
                                    pRP->XferAddr,
                                    2048,
                                    FALSE)) == STDON )
      {

         if ( DevHelp_PhysToVirt( (ULONG) pRP->XferAddr,
                                  (USHORT) 2048,
                                  (PVOID)  &pBuff,
                                  (PUSHORT) &ModeFlag   ) )
         {
            _asm {int 3}
         }

         /* zero out buffer (512 ULONGS = 2048 bytes) */
         for (i=pBuff; i<pBuff+512; i++ )
         {
            *i = 0l;
         }

      } /* if rc=WriteSector */
   }

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_WriteV
 *
 * DESCRIPTION   = Write and Verify Command  (Commands 0x09, 0x1A)
 *
 *         This is the basic strategy-1 I/O write and verify routine for the driver.
 *         The request is queued and sent to the adapter driver for
 *         processing.
 *
 *         USHORT CD_WriteV   (PRP_RWV pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_WriteV (pRP, pUnitCB)

PRP_RWV   pRP;
NPUNITCB  pUnitCB;
{
   USHORT rc;
   BOOL   playing,
   MountDrive = FALSE;
   PULONG pBuff;
   PULONG i;
   USHORT ModeFlag;
   NPIORB_CDB pIORB = 0, pModeIORB = 0;

   /*
   ** Check for valid input parms
   */
   if (pRP->NumSectors == 0)
      return (STDON);

   /*
   ** Check for uncertain media
   */
   if (pUnitCB->Flags & UCF_UNCERTAIN_MEDIA)
   {
      pRP->NumSectors = 0;
      return (STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA);
   }

   /*
   ** Cant write when unit is playing so check play status
   */
   if (pUnitCB->DeviceInfo.playing ||
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))
   {
      rc = GetPlayStatus(pUnitCB, &playing);

      if ( (rc == STDON) && playing)
      {
         pRP->NumSectors = 0;
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);
      }
   }

   /*
   ** If multisession photo CD mounted and the target write is one of the
   ** Volume Descriptor Sectors, then remap the RBA to the last session
   */
   if ( (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION) &&
        (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION_MOUNTED) &&
        (pRP->rba >= PRIMARY_VOL_DESCR_RBA) &&
        (pRP->rba <= pUnitCB->DeviceInfo.volume_descr_terminator) )
   {
      pRP->rba = pRP->rba + pUnitCB->DeviceInfo.last_session_addr;
   }

   /*
   ** Determine if the File System is attempting to mount the Drive.
   */

   if ( (pRP->rba == 0) &&
        (pRP->NumSectors == 1) )
   {
      if ( DevHelp_PhysToVirt( (ULONG) pRP->XferAddr,
                               (USHORT) 2048,
                               (PVOID)  &pBuff,
                               (PUSHORT) &ModeFlag   ) )
      {
         _asm {int 3}
      }

      if (*pBuff == 0x544f4f42l) /* Boot Signature */
      {
         MountDrive = TRUE;
      }
   }

   if ( (pUnitCB->DeviceInfo.current_block_size != 2048) &&
        (pUnitCB->DeviceInfo.interface_type != INTERFACE_ATAPI) )
   {
      rc = ChainModeSelectWrite (pUnitCB, pRP->rba, pRP->NumSectors,
                                 pRP->XferAddr, 2048, 0, TRUE);
   }
   else
   {
      rc = WriteSector (pUnitCB, pRP->rba, pRP->NumSectors,
                        pRP->XferAddr, 2048, TRUE);
   }

   if (rc & STERR)
      pRP->NumSectors = 0;

   if (rc == STDON + STERR + ERROR_I24_BAD_COMMAND)
      rc = STDON + STERR + ERROR_I24_SECTOR_NOT_FOUND;

   /*
   ** If we are mounting the volume, then check for multi-session disk
   */
   if ( (MountDrive) &&
        (pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_MULTISESSION) )
   {
      Check_MultiSession_Mounted(pUnitCB, pRP->XferAddr);
   }


   /*
   ** If the File System is attempting to mount the drive, and there
   ** was an error that was not a drive ready error, give the error
   ** (if any) from a write of the volumne descriptor.
   ** Clear the buffer if the write was successful.
   */

   if (MountDrive &&     // Atempting to mount a disc
       (rc != STDON) &&   // An error was encountered
       (rc != STDON + STERR + ERROR_I24_NOT_READY) &&
       (rc != STDON + STERR + ERROR_I24_DISK_CHANGE) &&
       (rc != STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA) )
   {
      if ((rc = WriteSector(        pUnitCB,
                                    (ULONG) PRIMARY_VOL_DESCR_RBA,
                                    1,
                                    pRP->XferAddr,
                                    2048,
                                    TRUE)) == STDON)
      {

         if ( DevHelp_PhysToVirt( (ULONG) pRP->XferAddr,
                                  (USHORT) 2048,
                                  (PVOID)  &pBuff,
                                  (PUSHORT) &ModeFlag   ) )
         {
            _asm {int 3}
         }

         /* zero out buffer (512 ULONGS = 2048 bytes) */
         for (i=pBuff; i<pBuff+512; i++ )
         {
            *i = 0l;
         }

      } /* if rc=WriteSector */
   }

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = StatusDevReady
 *
 * DESCRIPTION   =
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT StatusDevReady(pRPH, pUnitCB)
PRPH     pRPH;
NPUNITCB pUnitCB;
{
   return (STDON + STBUI);
}


/****************************************************************************
 *
 * FUNCTION NAME = StatusComplete
 *
 * DESCRIPTION   =
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT StatusComplete(pRPH, pUnitCB)
PRPH     pRPH;
NPUNITCB pUnitCB;
{
   return (STDON);
}


/****************************************************************************
 *
 * FUNCTION NAME = StatusError
 *
 * DESCRIPTION   =
 *
 * INPUT         = pRPH             - Request Packet Header
 *                 ErrorCode        - Error Code
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT StatusError( pRPH, ErrorCode )

PRPH    pRPH;
USHORT  ErrorCode;
{
   return (STDON + STERR);
}


/****************************************************************************
 *
 * FUNCTION NAME = ChainModeSelectRead
 *
 * DESCRIPTION   = Chain Mode Select and Read Commands
 *
 *         USHORT ChainModeSelectRead (NPUNITCB pUnitCB, ULONG LBA,
 *                                     USHORT transfer_count, ULONG ppDataBuff,
 *                                     USHORT block_length, USHORT density)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - LBA
 *                 transfer_count   - count of sectors to transfer
 *                 ppDataBuff       - phys addr of data buffer
 *                 block_length     - block_length
 *                 density          - density code
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * NOTES:  This routine should not be called for an ATAPI device since
 *         ATAPI drives do not support Mode Selects for density codes
 *         and block lengths.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT ChainModeSelectRead (pUnitCB, LBA, transfer_count, ppDataBuff,
                            block_length, density)

NPUNITCB pUnitCB;
ULONG    LBA;
USHORT   transfer_count;
ULONG    ppDataBuff;
USHORT   block_length;
USHORT   density;

{
   USHORT rc;

   NPIORB_CDB pIORB = 0, pModeIORB = 0;
   NPIORB_DMWORK pDMWork;

   BuildCDB_ModeSelect (pUnitCB, density, block_length,
                        (NPIORB_CDB FAR *) &pModeIORB);

   /*
   ** If transfer count < 255 sectors, issue Read (6) else issue Read (10)
   */
   if (transfer_count <= 255 && LBA < 0x200000L)  // AK:01/17/02, Defect 268091 (LBA check)
   {
      BuildCDB_Read_6  (pUnitCB, LBA, block_length,
                        transfer_count, ppDataBuff,
                        (NPIORB_CDB FAR *) &pIORB);
   }
   else
   {
      BuildCDB_Read_10 (pUnitCB, LBA, block_length,
                        transfer_count, ppDataBuff,
                        (NPIORB_CDB FAR *) &pIORB);
   }

   pDMWork = (NPIORB_DMWORK) &(pModeIORB->apt.iorbh.DMWorkSpace);
   pDMWork->pCoReqIORB = (NPIORB) pIORB;

   QueueIORB (pUnitCB, (NPIORB)pModeIORB);   // 10/23/2000 MB

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   if ( !(pModeIORB->apt.iorbh.Status & IORB_ERROR) )
   {
      pUnitCB->DeviceInfo.current_density = density;
      pUnitCB->DeviceInfo.current_block_size = block_length;

   }

   FreeIORB (pUnitCB, pModeIORB);
   FreeIORB (pUnitCB, pIORB);

   return (rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = ChainModeSelectWrite
 *
 * DESCRIPTION   = Chain Mode Select and Write Commands
 *
 *         USHORT ChainModeSelectWrite (NPUNITCB pUnitCB, ULONG LBA,
 *                                     USHORT transfer_count, ULONG ppDataBuff,
 *                                     USHORT block_length, USHORT density, BOOL verify)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - LBA
 *                 transfer_count   - count of sectors to transfer
 *                 ppDataBuff       - phys addr of data buffer
 *                 block_length     - block_length
 *                 density          - density code
 *                 verify           - verification is necessary
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * NOTES:  This routine should not be called for an ATAPI device since
 *         ATAPI drives do not support Mode Selects for density codes
 *         and block lengths.
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT ChainModeSelectWrite (pUnitCB, LBA, transfer_count, ppDataBuff,
                             block_length, density, verify)

NPUNITCB pUnitCB;
ULONG    LBA;
USHORT   transfer_count;
ULONG    ppDataBuff;
USHORT   block_length;
USHORT   density;
BOOL     verify;

{
   USHORT rc;

   NPIORB_CDB pIORB = 0, pModeIORB = 0;
   NPIORB_DMWORK pDMWork;

   BuildCDB_ModeSelect (pUnitCB, density, block_length,
                        (NPIORB_CDB FAR *) &pModeIORB);

   if (verify)
   {
      BuildCDB_WriteV_10  (pUnitCB, LBA, block_length,
                           transfer_count, ppDataBuff,
                           (NPIORB_CDB FAR *) &pIORB);
   }
   else
   {
      BuildCDB_Write_10 (pUnitCB, LBA, block_length,
                         transfer_count, ppDataBuff,
                         (NPIORB_CDB FAR *) &pIORB);
   }

   pDMWork = (NPIORB_DMWORK) &(pModeIORB->apt.iorbh.DMWorkSpace);
   pDMWork->pCoReqIORB = (NPIORB) pIORB;

   QueueIORB (pUnitCB, (NPIORB)pModeIORB);   // 10/23/2000 MB

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   if ( !(pModeIORB->apt.iorbh.Status & IORB_ERROR) )
   {
      pUnitCB->DeviceInfo.current_density = density;
      pUnitCB->DeviceInfo.current_block_size = block_length;

   }

   FreeIORB (pUnitCB, pModeIORB);
   FreeIORB (pUnitCB, pIORB);

   return (rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = ReadPacket
 *
 * DESCRIPTION   = Issue Read Command
 *
 *        USHORT ReadPacket (NPUNITCB pUnitCB, ULONG LBA,
 *                           USHORT transfer_count, ULONG ppDataBuff,
 *                           USHORT block_length)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - LBA
 *                 transfer_count   - count of sectors to transfer
 *                 ppDataBuff       - phys addr of data buffer
 *                 block_length     - block_length
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT ReadPacket (pUnitCB, LBA, transfer_count, ppDataBuff, block_length)

NPUNITCB pUnitCB;
ULONG    LBA;
USHORT   transfer_count;
ULONG    ppDataBuff;
USHORT   block_length;

{
   USHORT rc;
   NPIORB_CDB pIORB = 0;

   BuildCDB_Read_10 (pUnitCB, LBA, block_length,
                     transfer_count, ppDataBuff,
                     (NPIORB_CDB FAR *) &pIORB);

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   FreeIORB (pUnitCB, pIORB);

   return rc;
}

/****************************************************************************
 *
 * FUNCTION NAME = WritePacket
 *
 * DESCRIPTION   = Issue Write Command
 *
 *        USHORT WritePacket (NPUNITCB pUnitCB, ULONG LBA,
 *                            USHORT transfer_count, ULONG ppDataBuff,
 *                            USHORT block_length, BOOL verify)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - LBA
 *                 transfer_count   - count of sectors to transfer
 *                 ppDataBuff       - phys addr of data buffer
 *                 block_length     - block_length
 *                 verify           - verification is necessary
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT WritePacket (pUnitCB, LBA, transfer_count, ppDataBuff, block_length, verify)

NPUNITCB pUnitCB;
ULONG    LBA;
USHORT   transfer_count;
ULONG    ppDataBuff;
USHORT   block_length;
BOOL     verify;

{
   USHORT rc;
   NPIORB_CDB pIORB = 0;

   if (verify)
   {
      BuildCDB_WriteV_10 (pUnitCB, LBA, block_length,
                          transfer_count, ppDataBuff,
                          (NPIORB_CDB FAR *) &pIORB);
   }
   else
   {
      BuildCDB_Write_10 (pUnitCB, LBA, block_length,
                         transfer_count, ppDataBuff,
                         (NPIORB_CDB FAR *) &pIORB);
   }

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   FreeIORB (pUnitCB, pIORB);

   if (rc != STDON)
      return rc;
		// 08/08/01 AK: defect 260320 (some CD-RW devices require SyncCache)
   if (pUnitCB->Flags & UCF_PACKET_WRITABLE)
   {
      BuildCDB_SyncCache (pUnitCB, LBA, transfer_count, (NPIORB_CDB FAR *) &pIORB);

      rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);
      rc = STDON;

      FreeIORB (pUnitCB, pIORB);
   }

   return rc;
}

/****************************************************************************
 *
 * FUNCTION NAME = ReadSector
 *
 * DESCRIPTION   = Issue Read Command
 *
 *        USHORT ReadSector (NPUNITCB pUnitCB, ULONG LBA,
 *                           USHORT transfer_count, ULONG ppDataBuff,
 *                           USHORT block_length)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - LBA
 *                 transfer_count   - count of sectors to transfer
 *                 ppDataBuff       - phys addr of data buffer
 *                 block_length     - block_length
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT ReadSector (pUnitCB, LBA, transfer_count, ppDataBuff, block_length)

NPUNITCB pUnitCB;
ULONG    LBA;
USHORT   transfer_count;
ULONG    ppDataBuff;
USHORT   block_length;
{
   USHORT rc=STDON + STERR + ERROR_I24_SECTOR_NOT_FOUND; // 10/23/2000 MB
   NPIORB_CDB pIORB = 0;
   ULONG      nBytesRem;                                            /*@151345*/
   USHORT     partialBlockCount;                                    /*@151345*/
   ULONG      start_LBA;
   USHORT     packet_size;
   ULONG      packet_length;
   ULONG      rest_length;
   USHORT     n;
   BOOL       ver1;

   if ((pUnitCB->Flags & UCF_PACKET_WRITABLE) &&
       pUnitCB->PacketInfo.packet_fp &&
       /* AK 10/12/00, D249463: cache is allowed only for 2048-byte blocks */
       (pUnitCB->DeviceInfo.raw_block_size == 0 ||
        pUnitCB->DeviceInfo.raw_block_size == 2048)
      )   /* Incremental (packet) writable */
   {
      nBytesRem = (ULONG)transfer_count * block_length;
      packet_size = (USHORT)pUnitCB->PacketInfo.packet_size;   // 10/23/2000 MB
      packet_length = (ULONG)packet_size * block_length;
      while (nBytesRem > 0)
      {
         rc = STDON;
         start_LBA = LBA - (LBA % packet_size);
         if (LBA == start_LBA && nBytesRem >= packet_length)
         {
            partialBlockCount = packet_size;
            rc = ReadPacket (pUnitCB, LBA, packet_size, ppDataBuff, block_length);
         }
         else
         {
            DevHelp_SemRequest((ULONG)(PVOID)(&pUnitCB->PacketInfo.sem), SEM_INDEFINITE_WAIT); // AK 02/20/02
            rest_length = packet_length - (ULONG)(LBA - start_LBA) * block_length;
            if (nBytesRem < rest_length)
            {
               partialBlockCount = (USHORT)(nBytesRem / block_length);  // 10/23/2000 MB
            }
            else
            {
               partialBlockCount = (USHORT)(rest_length / block_length);   // 10/23/2000 MB
            }
            if (!FindPacket (pUnitCB, start_LBA, (PUSHORT) &n))
            {
               rc = FindLeastUsed (pUnitCB, (PUSHORT) &n);
               if (rc == STDON)
               {
                  if (pUnitCB->PacketInfo.buf_info[n].flags & PAC_UNSAVED)
                  {
                     ver1 = pUnitCB->PacketInfo.buf_info[n].flags & PAC_VERIFY ? TRUE : FALSE;
                     rc = WritePacket (pUnitCB,
                                       pUnitCB->PacketInfo.buf_info[n].LBA,
                                       packet_size,
                                       pUnitCB->PacketInfo.buf_info[n].adr_buf,
                                       block_length,
                                       ver1);
                  }
                  if (rc == STDON)
                  {
                     pUnitCB->PacketInfo.buf_info[n].flags &= ~PAC_UNSAVED;
                     pUnitCB->PacketInfo.buf_info[n].LBA = start_LBA;
                     pUnitCB->PacketInfo.buf_info[n].flags &= ~PAC_ALLOCATED;   //AK(10/26/00)  Clear alloc flag prior reading
                     rc = ReadPacket (pUnitCB, start_LBA, packet_size,
                                      pUnitCB->PacketInfo.buf_info[n].adr_buf,
                                      block_length);
                     if (rc == STDON)                 //AK(10/26/00) Set allocation flag if read
                     {
                        //AK
                        pUnitCB->PacketInfo.buf_info[n].flags |= PAC_ALLOCATED; //AK
                     }                       //AK
                  }
               }
            }
            if (rc == STDON)
            {
               rc = MemCopy( ppDataBuff,
                             pUnitCB->PacketInfo.buf_info[n].adr_buf + (LBA-start_LBA)*block_length,
                             partialBlockCount * block_length);
               SetAccessOrder (pUnitCB, n);
            }
            DevHelp_SemClear((ULONG)(PVOID)(&pUnitCB->PacketInfo.sem));  // AK 02/20/02
         }
         LBA += partialBlockCount;
         ppDataBuff += (ULONG) partialBlockCount * block_length;
         nBytesRem -= (ULONG) partialBlockCount * block_length;
         if (rc != STDON)
            break;
      }
   }
   else
   {
      /*
      ** If transfer count < 255 sectors, issue Read (6) else issue Read (10)
      ** ATAPI drives only support Read 10.
      */
      if ( (transfer_count <= 255) && (LBA < 0x200000L) &&    // AK:01/17/02 defect 268091 (LBA check)
           (pUnitCB->DeviceInfo.interface_type != INTERFACE_ATAPI) )
      {
         BuildCDB_Read_6 (pUnitCB, LBA, block_length,
                          transfer_count, ppDataBuff,
                          (NPIORB_CDB FAR *) &pIORB);
         rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);                 /*@151345*/  // 10/23/2000 MB
         FreeIORB (pUnitCB, pIORB);                                    /*@151345*/
      }
      else
      {
         nBytesRem = (ULONG)transfer_count * block_length;             /*@151345*/
#ifdef DEBUG
{
   dprintf("OS2CDROM ReadSector nBytesRem %lx, LBA %lx, tCount %x, bl %x\r\n",
           (PVOID)&nBytesRem, (PVOID)&LBA, (PVOID)&transfer_count, (PVOID)&block_length); 
}
#endif
         while ( nBytesRem > 0 )                                        /*@151345*/
         {
            /*@151345*/
            if ( nBytesRem > CDATAPI_LIMIT_64K )                        /*@151345*/
            {
               /*@151345*/
               /* Limit to blocks which fit in 64K */                  /*@151345*/
               partialBlockCount = (USHORT)(CDATAPI_LIMIT_64K / block_length);   /*@151345*/ // 10/23/2000 MB
            }                                                          /*@151345*/
            else                                                       /*@151345*/
            {
               /*@151345*/
               /* Do the last partial read */                          /*@151345*/
               partialBlockCount = (USHORT)(nBytesRem / block_length);           /*@151345*/ // 10/23/2000 MB
            }                                                          /*@151345*/
            /*@151345*/
            rc = ReadPacket (pUnitCB, LBA, partialBlockCount, ppDataBuff, block_length);

            /*                                                           @151345
            ** Adjust the buffer pointers for more requests.             @151345
            */                                                         /*@151345*/
            LBA += partialBlockCount;                                  /*@151345*/
            ppDataBuff += (ULONG)partialBlockCount * block_length;     /*@151345*/
            nBytesRem -= (ULONG)partialBlockCount * block_length;      /*@151345*/
            if (rc != STDON)
               break;
         }
      }
   }

   return (rc);
}

/****************************************************************************
 *
 * FUNCTION NAME = WriteSector
 *
 * DESCRIPTION   = Issue Write Command
 *
 *        USHORT WriteSector (NPUNITCB pUnitCB, ULONG LBA,
 *                            USHORT transfer_count, ULONG ppDataBuff,
 *                            USHORT block_length, BOOL verify)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - LBA
 *                 transfer_count   - count of sectors to transfer
 *                 ppDataBuff       - phys addr of data buffer
 *                 block_length     - block_length
 *                 verify           - verification is necessary
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT WriteSector (pUnitCB, LBA, transfer_count, ppDataBuff, block_length, verify)

NPUNITCB pUnitCB;
ULONG    LBA;
USHORT   transfer_count;
ULONG    ppDataBuff;
USHORT   block_length;
BOOL     verify;

{
   USHORT rc;
   NPIORB_CDB pIORB = 0;
   ULONG      nBytesRem;
   USHORT     partialBlockCount;
   ULONG      start_LBA;
   ULONG      packet_length;
   USHORT     packet_size;
   ULONG      rest_length;
   USHORT     n;
   BOOL       ver1;

   if (pUnitCB->DeviceInfo.raw_block_size)
   {     /* Restore block_size, if it was changed by CD_Read_Long */
      if (!( pUnitCB->DeviceInfo.interface_type == INTERFACE_ATAPI ))
         rc = Submit_ModeSelect (pUnitCB, CD_DENSITY_DEFAULT, 2048);

      pUnitCB->DeviceInfo.raw_block_size = 0;
      if (pUnitCB->MediaType == DMD_CDRW || pUnitCB->MediaType == DMD_CDR)
      {
         rc = SetPacketInfo(pUnitCB);
         if (rc == STDON)
            pUnitCB->Flags |= UCF_PACKET_WRITABLE;
      }
   }

   nBytesRem = (ULONG)transfer_count * block_length;
   if (pUnitCB->Flags & UCF_PACKET_WRITABLE) /* Incremental (packet) writable */
   {
      if (pUnitCB->PacketInfo.packet_fp)    /* fixed packets */
      {
         packet_size = (USHORT)pUnitCB->PacketInfo.packet_size;   // 10/23/2000 MB
         packet_length = (ULONG)packet_size * block_length;
         while (nBytesRem > 0)
         {
            rc = STDON;
            start_LBA = LBA - (LBA % packet_size);
            if (LBA == start_LBA && nBytesRem >= packet_length)
            {
               partialBlockCount = packet_size;
               rc = WritePacket (pUnitCB, LBA, packet_size, ppDataBuff,
                                 block_length, verify);
            }
            else
            {
               DevHelp_SemRequest((ULONG)(PVOID)(&pUnitCB->PacketInfo.sem), SEM_INDEFINITE_WAIT);  // AK 02/20/02
               rest_length = packet_length - (ULONG)(LBA - start_LBA) * block_length;
               if (nBytesRem < rest_length)
               {
                  partialBlockCount = (USHORT)(nBytesRem / block_length);  // 10/23/2000 MB
               }
               else
               {
                  partialBlockCount = (USHORT)(rest_length / block_length);   // 10/23/2000 MB
               }
               if (!FindPacket (pUnitCB, start_LBA, (PUSHORT) &n))
               {
                  rc = FindLeastUsed (pUnitCB, (PUSHORT) &n);
                  if (rc == STDON)
                  {
                     if (pUnitCB->PacketInfo.buf_info[n].flags & PAC_UNSAVED)
                     {
                        ver1 = pUnitCB->PacketInfo.buf_info[n].flags & PAC_VERIFY ? TRUE : FALSE;
                        rc = WritePacket (pUnitCB,
                                          pUnitCB->PacketInfo.buf_info[n].LBA,
                                          packet_size,
                                          pUnitCB->PacketInfo.buf_info[n].adr_buf,
                                          block_length,
                                          ver1);
                     }
                     if (rc == STDON)
                     {
                        pUnitCB->PacketInfo.buf_info[n].flags &= ~PAC_UNSAVED;
                        pUnitCB->PacketInfo.buf_info[n].LBA = start_LBA;
                        rc = ReadPacket (pUnitCB, start_LBA, packet_size,
                                         pUnitCB->PacketInfo.buf_info[n].adr_buf,
                                         block_length);
                        if (rc != STDON)
                        {
                           pUnitCB->PacketInfo.buf_info[n].flags &= ~PAC_ALLOCATED;
                        }
                     }
                  }
               }
               if (rc == STDON)
               {
                  rc = MemCopy( pUnitCB->PacketInfo.buf_info[n].adr_buf + (LBA-start_LBA)*block_length,
                                ppDataBuff,
                                partialBlockCount * block_length);
                  SetPacketFlags (pUnitCB, n, TRUE, verify);
                  SetAccessOrder (pUnitCB, n);
               }
               DevHelp_SemClear((ULONG)(PVOID)(&pUnitCB->PacketInfo.sem));  // AK 02/20/02
            }
            LBA += partialBlockCount;
            ppDataBuff += (ULONG) partialBlockCount * block_length;
            nBytesRem -= (ULONG) partialBlockCount * block_length;
            if (rc != STDON)
               break;
         }
      }
      else  /* variable packets */
      {
         rc = WritePacket (pUnitCB, LBA, transfer_count, ppDataBuff,
                           block_length, verify);
      }
   }
   else     /* Random writable */
   {
      while ( nBytesRem > 0 )
      {
         if ( nBytesRem > CDATAPI_LIMIT_64K )
         {
            /* Limit to blocks which fit in 64K */
            partialBlockCount = CDATAPI_LIMIT_64K / block_length;
         }
         else
         {
            /* Do the last partial read */
            partialBlockCount = (USHORT)(nBytesRem / block_length);  // 10/23/2000 MB
         }

         rc = WritePacket (pUnitCB, LBA, partialBlockCount, ppDataBuff, block_length, verify);

         /*
         ** Adjust the buffer pointers for more requests.
         */
         LBA += partialBlockCount;
         ppDataBuff += (ULONG) partialBlockCount * block_length;
         nBytesRem -= (ULONG) partialBlockCount * block_length;
         if (rc != STDON)
            break;
      }
   }

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = GetVolumeSize
 *
 * DESCRIPTION   = Get volume size
 *
 *       Return the volume size of the current CD
 *
 *       USHORT GetVolumeSize  (NPUNITCB pUnitCB, ULONG FAR * volume_size)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 volume_size      - Pointer to return volume size
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT GetVolumeSize (pUnitCB, volume_size)

NPUNITCB  pUnitCB;
ULONG FAR * volume_size;

{
   USHORT rc;
   NPIORB_CDB pIORB;
   BOOL playing;                                         //SD@135221

   struct ReadCapacity_Data NEAR *pCDBData;
   union  ULONGB  ul_volume;
   struct ReadTrackInfo_Data NEAR *pModeTrackInfo;       //07-23-01 AK

   if (pUnitCB->DeviceInfo.playing==FALSE &&             //SD@135221
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))//SD@135221
   {
      //SD@135221
      rc = GetPlayStatus(pUnitCB, &playing);             //SD@135221
                                                         //SD@135221
      if ( (rc == STDON) && playing)                     //SD@135221
      {
         //SD@135221
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);//SD@135221
      }
   }
//------- 07-23-2001 AK
   if (pUnitCB->MediaType == DMD_CDR || pUnitCB->MediaType == DMD_CDRW
       || pUnitCB->MediaType == DMD_DVDR)
   { 
      BuildCDB_ReadTrackInfo (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

      if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
      {
         pModeTrackInfo = (struct ReadTrackInfo_Data NEAR *) &(pIORB->CDB_data);
         ul_volume.ulbytes.byte_0 = pModeTrackInfo->track_size.ulbytes.byte_3;
         ul_volume.ulbytes.byte_1 = pModeTrackInfo->track_size.ulbytes.byte_2;
         ul_volume.ulbytes.byte_2 = pModeTrackInfo->track_size.ulbytes.byte_1;
         ul_volume.ulbytes.byte_3 = pModeTrackInfo->track_size.ulbytes.byte_0;
         *volume_size = ul_volume.dword - 1;
      }

      FreeIORB (pUnitCB, pIORB);

      if (rc == STDON)
         return rc;
   }

   BuildCDB_ReadCapacity(pUnitCB, (NPIORB_CDB FAR *) &pIORB);

   pCDBData = (struct ReadCapacity_Data NEAR *) pIORB->CDB_data;

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   if (rc == STDON)
   {
      ul_volume.ulbytes.byte_3  =  pCDBData->capacity_LBA.ulbytes.byte_0;
      ul_volume.ulbytes.byte_2  =  pCDBData->capacity_LBA.ulbytes.byte_1;
      ul_volume.ulbytes.byte_1  =  pCDBData->capacity_LBA.ulbytes.byte_2;
      ul_volume.ulbytes.byte_0  =  pCDBData->capacity_LBA.ulbytes.byte_3;

      *volume_size = ul_volume.dword;
   }
   else if (rc == STDON + STERR + ERROR_I24_INVALID_PARAMETER)
      ul_volume.dword = 270000L;


   FreeIORB (pUnitCB, pIORB);

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = ClearCheckCondition
 *
 * DESCRIPTION   = Clear check condition and return error code
 *
 *                 USHORT ClearCheckCondition  (NPUNITCB pUnitCB)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT ClearCheckCondition (pUnitCB)

NPUNITCB pUnitCB;
{
   ULONG  volume_size;

   return (GetVolumeSize (pUnitCB, (ULONG FAR *) &volume_size) );
}


/****************************************************************************
 *
 * FUNCTION NAME = GetSectorMode
 *
 * DESCRIPTION   = Get mode for sector zero.
 *
 *       This routine returns the mode of sector zero by reading the header.
 *
 *       USHORT GetSectorMode  (NPUNITCB pUnitCB, USHORT FAR * mode)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 mode             - mode
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT GetSectorMode (pUnitCB, mode)

NPUNITCB pUnitCB;
USHORT   FAR *mode;

{
   USHORT rc;
   NPIORB_CDB pIORB;
   struct ReadHeader_Data NEAR *pCDBData;


   BuildCDB_ReadHeader(pUnitCB, 0, (NPIORB_CDB FAR *) &pIORB);

   pCDBData = (struct ReadHeader_Data NEAR *) pIORB->CDB_data;

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   if (rc == STDON)
   {
      *mode = pCDBData->cdrom_data_mode;
   }

   FreeIORB (pUnitCB, pIORB);
   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = Check_MultiSession_Mounted
 *
 * DESCRIPTION   = Check if multisession mounted
 *
 *      This routine checks to see if a multisession photoCD disk is mounted.
 *      If it is, then the driver reads the volume descriptors until the
 *      volume descriptor terminator is found.  All subsequent reads from
 *      the Primary Volume Descriptor till the Volume Descriptor Terminator
 *      are mapped to the last session prior to the read.  This routine is
 *      only called if the multi_session_support bit is set in the UnitCB.
 *
 *      USHORT Check_MultiSession_Mounted  (NPUNITCB pUnitCB, ULONG physaddr)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 physaddr         - physical address of I/O buffer
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT Check_MultiSession_Mounted (pUnitCB, physaddr)

NPUNITCB pUnitCB;
ULONG    physaddr;

{
   USHORT rc, n, mode_flag;
   ULONG session_addr;
   PBYTE pBuffer;

   BOOL term_found = FALSE;

   /*
   ** Initialize to not multisession disk
   */
   pUnitCB->DeviceInfo.Audio.capabilities &= ~DCAPS_MULTISESSION_MOUNTED;
   pUnitCB->DeviceInfo.last_session_addr = 0;
   pUnitCB->DeviceInfo.volume_descr_terminator = 0;

   switch (pUnitCB->DeviceInfo.product_id_code)
   {
   case TOSHIBA_3401:
      rc = Tosh_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr);
      break;

   case SONY_561:
   case TEXEL_3024K:
   case PIONEER_604X:
   case HITACHI_6750:
   case LMS_215:
   case PLEXTOR_PX20:
      rc = Sony_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr);
      break;

   case CHINON_535:
      rc = Chinon_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr);
      break;

   case NEC_84_1:
      rc = NEC_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr);
      break;

   case NEC_46X:                                                             /*@190032*/
      rc = NEC46X_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr); /*@190032*/
      break;                                                                 /*@190032*/

   case HP_C4324:                                                       /*@V156096*/
      rc = HP_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr);/*@V156096*/
      break;                                                            /*@V156096*/

   default:
      if (pUnitCB->DeviceInfo.interface_type == INTERFACE_PROPRIETARY)
         rc=NonSCSI_GetLastSessionAddr(pUnitCB, (ULONG FAR *) &session_addr);
      else if (pUnitCB->DeviceInfo.interface_type == INTERFACE_ATAPI)
         rc = Sony_GetLastSessionAddr (pUnitCB, (ULONG FAR *) &session_addr);
      else
         rc = STDON + STERR;
      break;
   }
   /*
   ** If no error, then multisession disk.  Read the volume descriptors
   ** till the terminator descriptor is found.
   */
   if (rc == STDON)
   {
      for (n = PRIMARY_VOL_DESCR_RBA; term_found != TRUE; n++)
      {
         if ((rc = ReadSector(pUnitCB, (ULONG) (n + session_addr), 1, physaddr, 2048)) != STDON)  
            break;
         /*
         ** Check for the volume descriptor id of "CD001"
         */
         DevHelp_PhysToVirt (physaddr,
                             (USHORT) 2048,
                             (PBYTE FAR *) &pBuffer,
                             (USHORT FAR *) &mode_flag);


         if ( *(pBuffer+1) != 'C' || *(pBuffer+2) != 'D' ||
              *(pBuffer+3) != '0' || *(pBuffer+4) != '0' ||
              *(pBuffer+5) != '1' )
         {
            rc = STDON + STERR;
            break;
         }
         /*
         ** Check for terminator volume descriptor
         */
         if ( (UCHAR) *pBuffer == VOL_DESCR_TERMINATOR)
         {
            pUnitCB->DeviceInfo.Audio.capabilities |= DCAPS_MULTISESSION_MOUNTED;
            pUnitCB->DeviceInfo.last_session_addr = session_addr;
            pUnitCB->DeviceInfo.volume_descr_terminator = n;
            term_found = TRUE;
         }
      }
      /*
      ** Read back original sector 0
      */
      rc=ReadSector (pUnitCB, (ULONG) 0, 1, physaddr, 2048);
   }

   return (rc); // 10/23/2000 MB
}


/****************************************************************************
 *
 * FUNCTION NAME = NonSCSI_GetLastSessionAddr
 *
 * DESCRIPTION   = Get address of last session on multisession disk
 *
 *  This routine returns the address of the last session of a multisession
 *  photo CD disk.  This routine should only be called is the drive is
 *  a non SCSI CD-ROM.
 *
 *  USHORT NonSCSI_GetLastSessionAddr(NPUNITCB pUnitCB, ULONG FAR *session_addr)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 session_addr     - Pointer to returned session addr
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 *                                    if the STERR bit is on, then the disk
 *                                      is not a multisession disk and the
 *                                      session_addr field is not valid
 *
 *                                    if the STERR bit is NOT set, then the
 *                                      disk is a multi-session disk and the
 *                                      session addr field is valid
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT NonSCSI_GetLastSessionAddr (pUnitCB, session_addr)

NPUNITCB pUnitCB;
ULONG    FAR *session_addr;

{
   USHORT rc;
   NPIORB_CDB pIORB;
   union ADD_ReadDiskInfo_Data NEAR *pCDBData;
   union  AddressType last_session;


   ADD_BuildCDB_ReadDiskInfo (pUnitCB, TYPE_LASTSESSION_INFO,
                              (NPIORB_CDB FAR *) &pIORB);

   pCDBData = (union ADD_ReadDiskInfo_Data NEAR *) pIORB->CDB_data;

   rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

   if (rc == STDON)
   {
      /*
      ** If zero returned in data fields, then not multisession
      */
      if ( (pCDBData->last_session.amin == 0) &&
           (pCDBData->last_session.asec == 0) &&
           (pCDBData->last_session.aframe == 0) )
      {
         rc = STDON + STERR;
      }
      else
      {
         last_session.ul_redbook.min = pCDBData->last_session.amin;
         last_session.ul_redbook.sec = pCDBData->last_session.asec;
         last_session.ul_redbook.frame = pCDBData->last_session.aframe;
         last_session.ul_redbook.zero = 0;

         last_session.dword = RedBookToHSG (last_session.dword);

         *session_addr = last_session.dword;
      }

   }
   FreeIORB (pUnitCB, pIORB);
   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = GetPlayStatus
 *
 * DESCRIPTION   = Get play status
 *
 *                 USHORT GetPlayStatus  (NPUNITCB pUnitCB, BOOL FAR * playing)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 playing          - returned play status
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT  GetPlayStatus (pUnitCB, playing)

NPUNITCB   pUnitCB;
BOOL FAR *playing;
{
   USHORT rc;
   NPIORB_CDB pIORB;
   struct SubChannel_Position NEAR *pCDBData;

   if (pUnitCB->DeviceInfo.playing==FALSE &&             //SD@135221
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))//SD@135221
   {
      //SD@135221
      *playing=TRUE;                                     //SD@135221
      return STDON;                                      //SD@135221
   } /* endif */                                      //SD@135221

   BuildCDB_ReadSubChannel(pUnitCB, RSC_CURRENT_POSITION,
                           (NPIORB_CDB FAR *) &pIORB);

   pCDBData = (struct SubChannel_Position NEAR *) pIORB->CDB_data;

   *playing = FALSE;

   if ( (rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)  // 10/23/2000 MB
   {
      if (pCDBData->sub_channel_hdr.audio_status == AS_PLAY_IN_PROGRESS)
         *playing = TRUE;

      pUnitCB->DeviceInfo.playing = *playing;
      if (pUnitCB->pParentUnitCB)                        //SD@135221
      {
         pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying = pUnitCB->DeviceInfo.playing;//SD@135221
      } /* endif */                                   //SD@135221
   }
   FreeIORB (pUnitCB, pIORB);

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = Submit_ModeSelect
 *
 * DESCRIPTION   = Submit a Mode Select command
 *
 *                 Submit_ModeSelect (NPUNITCB pUnitCB, USHORT density_code,
 *                                                      USHORT block_length)
 *
 * INPUT         = pUnitCB         - pointer to UnitCB
 *                 density_code    - density code
 *                 block_length    - block_length
 *                 pIORBOut        - returned pointer to IORB
 *
 * OUTPUT        = USHORT          - packet status
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT Submit_ModeSelect (pUnitCB, density_code, block_length)

NPUNITCB   pUnitCB;
USHORT     density_code;
USHORT     block_length;
{

   USHORT rc;
   NPIORB_CDB pIORB;

   BuildCDB_ModeSelect (pUnitCB, density_code, block_length,
                        (NPIORB_CDB FAR *) &pIORB);

   rc = SubmitIORB_Wait (pUnitCB, (NPIORB)pIORB);  // 10/23/2000 MB

   FreeIORB (pUnitCB, pIORB);

   if (rc == STDON)
   {
      pUnitCB->DeviceInfo.current_density    = density_code;
      pUnitCB->DeviceInfo.current_block_size = block_length;
   }

   return (rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = Get_UnitCB_Addr
 *
 * DESCRIPTION   = Get UnitCB address
 *
 *       Return the address of the UnitCB for the specified unit.
 *
 *       USHORT Get_UnitCB_Addr  (UCHAR LogDriveNum, NPUNITCB FAR *pUnitCB)
 *
 * INPUT         = unit             - logical drive number
 *                 pUnitCB          - returned pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT Get_UnitCB_Addr (LogDriveNum, pUnitCB)

UCHAR LogDriveNum;
NPUNITCB FAR *pUnitCB;

{
   NPUNITCB pUnitCBx;
   USHORT   found = FALSE;

   pUnitCBx = UnitCB_Head;

   while (pUnitCBx != NULL && found == FALSE)
   {
      if ( (UCHAR) pUnitCBx->LogDriveNum == LogDriveNum )
         found = TRUE;
      else
         pUnitCBx = pUnitCBx->pNextUnitCB;
   }

   if (found == TRUE)
   {
      *pUnitCB = pUnitCBx;
      return (NO_ERROR);
   }
   else
      return (ERROR);
}

/****************************************************************************
 *
 * FUNCTION NAME = SetPacketFlags
 *
 * DESCRIPTION   = Set flags
 *
 *                 BOOL SetPacketFlags  (NPUNITCB pUnitCB, USHORT n, BOOL unsave, BOOL verify)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 n                - Buffer index
 *                 unsave           - UNSAVE flag
 *                 verify           - VERIFY flag
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT SetPacketFlags (pUnitCB, n, unsave, verify)

NPUNITCB   pUnitCB;
USHORT     n;
BOOL       unsave;
BOOL       verify;

{
   NPIORB_CDB pIORB;
   USHORT     rc;

   if (!(pUnitCB->PacketInfo.buf_info[n].flags & PAC_UNSAVED) && unsave)
   {
      if (!(pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_DOOR_LOCKED))
      {
         /*
         ** Issue SCSI PreventAllowRemoval command to lock the media
         */

         BuildCDB_PreventAllowRemoval (pUnitCB, 1, (NPIORB_CDB FAR *) &pIORB);

         rc = SubmitIORB_Wait (pUnitCB, (NPIORB)pIORB);  // 10/23/2000 MB

         FreeIORB (pUnitCB, pIORB);

         if (rc != STDON)
            return rc;

         pUnitCB->DeviceInfo.Audio.capabilities |= DCAPS_DOOR_LOCKED;
      }
   }

   pUnitCB->PacketInfo.buf_info[n].flags |= PAC_ALLOCATED;

   if (unsave)
   {
      pUnitCB->PacketInfo.buf_info[n].flags |= PAC_UNSAVED;
   }
   else
   {
      pUnitCB->PacketInfo.buf_info[n].flags &= ~PAC_UNSAVED;
   }

   if (verify)
   {
      pUnitCB->PacketInfo.buf_info[n].flags |= PAC_VERIFY;
   }
   else
   {
      pUnitCB->PacketInfo.buf_info[n].flags &= ~PAC_VERIFY;
   }

   return STDON;
}


/****************************************************************************
 *
 * FUNCTION NAME = MemCopy
 *
 * DESCRIPTION   = Copy block of physical memory
 *
 *                 USHORT MemCopy (ULONG dest_mem, ULONG src_mem, USHORT count)
 *
 * INPUT         = dest_mem          - Address of destination
 *                 src_mem           - Address of source
 *                 count             - Length
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT MemCopy (dest_mem, src_mem, count)

ULONG   dest_mem;
ULONG   src_mem;
USHORT  count;
{
   PUCHAR   src, dst;
   USHORT   i, rc=STDON;

   if ( DevHelp_PhysToGDTSelector((ULONG) dest_mem,
                                  (USHORT) count,
                                  (SEL) ReservedSelectors[0]) )
   {
      _asm {int 3}
   }
   dst = MAKEP (ReservedSelectors[0], NULL);

   if ( DevHelp_PhysToGDTSelector((ULONG) src_mem,
                                  (USHORT) count,
                                  (SEL) ReservedSelectors[1]) )
   {
      _asm {int 3}
   }
   src = MAKEP (ReservedSelectors[1], NULL);

   for (i = 0; i < count; i++)
      *(dst++) = *(src++);

   return rc;
}

/****************************************************************************
 *
 * FUNCTION NAME = FindPacket
 *
 * DESCRIPTION   = Find packet in the buffer
 *
 *                 BOOL FindPacket  (NPUNITCB pUnitCB, ULONG LBA, PUSHORT i)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 LBA              - Packet starting sector
 *
 * OUTPUT        = i                - Buffer index address
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL FindPacket (pUnitCB, LBA, i)

NPUNITCB   pUnitCB;
ULONG      LBA;
PUSHORT    i;
{
   USHORT m;

   for (m=0; m<NUM_OF_BUFFERS; m++)
   {
      if ((pUnitCB->PacketInfo.buf_info[m].flags & PAC_EXIST) &&
          (pUnitCB->PacketInfo.buf_info[m].flags & PAC_ALLOCATED) &&
          (pUnitCB->PacketInfo.buf_info[m].LBA == LBA))
      {
         *i = m;
         return TRUE;
      }
   }
   return FALSE;
}

/****************************************************************************
 *
 * FUNCTION NAME = FindLeastUsed
 *
 * DESCRIPTION   = Find least used buffer
 *
 *                 USHORT FindLeastUsed  (NPUNITCB pUnitCB, PUSHORT i)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = i                - Buffer index address
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT FindLeastUsed (pUnitCB, i)

NPUNITCB   pUnitCB;
PUSHORT    i;
{
   USHORT rc=STDON;
   UCHAR  maxv;   // 10/23/2000 MB
   USHORT m;

   for (m=0; m<NUM_OF_BUFFERS; m++)
   {
      if (!(pUnitCB->PacketInfo.buf_info[m].flags & PAC_EXIST))
      {
         if (DevHelp_AllocPhys (MAX_PACKET_LENGTH, 0, (PULONG) &(pUnitCB->PacketInfo.buf_info[m].adr_buf)))
            return STDON+STERR;
         *i = m;
         pUnitCB->PacketInfo.buf_info[m].flags |= PAC_EXIST;
         return rc;
      }
   }

   for (m=0; m<NUM_OF_BUFFERS; m++)
   {
      if (!(pUnitCB->PacketInfo.buf_info[m].flags & PAC_ALLOCATED))
      {
         *i = m;
         return rc;
      }
   }

   maxv = 0;
   for (m=0; m<NUM_OF_BUFFERS; m++)
   {
      if (pUnitCB->PacketInfo.buf_info[m].access_order >= maxv)
      {
         maxv = pUnitCB->PacketInfo.buf_info[m].access_order;
         *i = m;
      }
   }

   return rc;
}

/****************************************************************************
 *
 * FUNCTION NAME = FlushBuffers
 *
 * DESCRIPTION   = Free buffers
 *
 *                 USHORT FlushBuffers  (NPUNITCB pUnitCB)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT FlushBuffers (pUnitCB)

NPUNITCB   pUnitCB;
{
   USHORT  i, rc;
   BOOL    ver1;

   if ((pUnitCB->Flags & UCF_PACKET_WRITABLE) &&  /* Incremental (packet) writable */
       pUnitCB->PacketInfo.packet_fp)            /* fixed packets */
   {
      DevHelp_SemRequest((ULONG)(PVOID)(&pUnitCB->PacketInfo.sem), SEM_INDEFINITE_WAIT); // AK 02/20/02
      for (i=0; i<NUM_OF_BUFFERS; i++)
      {
         if (pUnitCB->PacketInfo.buf_info[i].flags & PAC_UNSAVED)
         {
            ver1 = pUnitCB->PacketInfo.buf_info[i].flags & PAC_VERIFY ? TRUE : FALSE;
            rc = WritePacket (pUnitCB,
                              pUnitCB->PacketInfo.buf_info[i].LBA,
                              (USHORT) pUnitCB->PacketInfo.packet_size,
                              pUnitCB->PacketInfo.buf_info[i].adr_buf,
                              CDROM_SECTOR_SIZE,
                              ver1);
            /*
            ** 	After WRITE command some SCSI commands can't work.
            **  MODE SELECT doesn't work for HP 9100/9200
            **  So READ command (read 0 blocks) is used
            */
            ReadPacket (pUnitCB,
                        pUnitCB->PacketInfo.buf_info[i].LBA,
                        (USHORT) 0,
                        pUnitCB->PacketInfo.buf_info[i].adr_buf,
                        CDROM_SECTOR_SIZE);

            pUnitCB->PacketInfo.buf_info[i].flags &= ~PAC_UNSAVED;
         }
         pUnitCB->PacketInfo.buf_info[i].flags &= ~PAC_ALLOCATED;
         pUnitCB->PacketInfo.buf_info[i].access_order = 0;
      }
      DevHelp_SemClear((ULONG)(PVOID)(&pUnitCB->PacketInfo.sem));  // AK 02/20/02
   }

   return (rc); // 10/23/2000 MB
}

/****************************************************************************
 *
 * FUNCTION NAME = SetAccessOrder
 *
 * DESCRIPTION   = Set access order for buffers
 *
 *                 USHORT SetAccessOrder  (NPUNITCB pUnitCB, USHORT i)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *                 i                - Buffer index
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT SetAccessOrder (pUnitCB, i)

NPUNITCB   pUnitCB;
USHORT     i;
{
   USHORT m;

   for (m=0; m<NUM_OF_BUFFERS; m++)
   {
      if ((pUnitCB->PacketInfo.buf_info[m].flags & PAC_ALLOCATED) &&
          (pUnitCB->PacketInfo.buf_info[m].flags & PAC_EXIST))
      {
         if (pUnitCB->PacketInfo.buf_info[m].access_order < 255)
         {
            pUnitCB->PacketInfo.buf_info[m].access_order++;
         }
      }
   }
   pUnitCB->PacketInfo.buf_info[i].access_order = 0;

   return (NO_ERROR); // 10/23/2000 MB
}

/****************************************************************************
 *
 * FUNCTION NAME = SetPacketInfo
 *
 * DESCRIPTION   = Set parameters for Packet Writing
 *
 *                 USHORT SetPacketInfo  (NPUNITCB pUnitCB)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT  SetPacketInfo (pUnitCB)

NPUNITCB   pUnitCB;
{
   USHORT rc;
   NPIORB_CDB pIORB, pIORB1;
   struct ModeSelectParmList_10 NEAR *pModeCap;
   struct Page_Write_Parameters NEAR *pWrite;
   struct ReadDiscInfo_Data NEAR *pModeDiscInfo;
   struct ReadTrackInfo_Data NEAR *pModeTrackInfo;
   union ULONGB ul;


   pUnitCB->PacketInfo.sem = 0;  // AK 02/20/02

   BuildCDB_ReadDiscInfo (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

   if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   //10/23/2000 MB
   {
      pModeDiscInfo = (struct ReadDiscInfo_Data NEAR *) &(pIORB->CDB_data);
      if (pModeDiscInfo->disc_type != 0x20 && pModeDiscInfo->disc_type != 0)
         rc = STDON+STERR;
      if (pModeDiscInfo->last_status == 2)
         rc = STDON+STERR;
   }

   FreeIORB (pUnitCB, pIORB);

   if (rc != STDON)
      return rc;

   BuildCDB_ReadTrackInfo (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

   if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
   {
      pModeTrackInfo = (struct ReadTrackInfo_Data NEAR *) &(pIORB->CDB_data);
      if (pModeTrackInfo->FP)   // 05/22/01 AK - Allow only FP tracks
      {
         if (pModeTrackInfo->RT == 0 && pModeTrackInfo->blank == 0 && pModeTrackInfo->packet_inc == 1)
            ;
         else if (pModeTrackInfo->RT == 0 && pModeTrackInfo->blank == 1 && pModeTrackInfo->packet_inc == 1)
            ;
         else if (pModeTrackInfo->RT == 1 && pModeTrackInfo->blank == 0 && pModeTrackInfo->packet_inc == 1)
            ;
         else
            rc = STDON+STERR;
         pUnitCB->PacketInfo.packet_fp = pModeTrackInfo->FP;
         ul.ulbytes.byte_0 = pModeTrackInfo->FP_size.ulbytes.byte_3;
         ul.ulbytes.byte_1 = pModeTrackInfo->FP_size.ulbytes.byte_2;
         ul.ulbytes.byte_2 = pModeTrackInfo->FP_size.ulbytes.byte_1;
         ul.ulbytes.byte_3 = pModeTrackInfo->FP_size.ulbytes.byte_0;
         pUnitCB->PacketInfo.packet_size = ul.dword;
         pUnitCB->PacketInfo.write_type = 0;
         pUnitCB->PacketInfo.track_mode = pModeTrackInfo->track_mode;
         if (pUnitCB->PacketInfo.packet_size > MAX_PACKET_SIZE)
            rc = STDON+STERR;
         if (pModeTrackInfo->data_mode == 1)
            pUnitCB->PacketInfo.block_type = 8;
         else if (pModeTrackInfo->data_mode == 2)
            pUnitCB->PacketInfo.block_type = 10;
         else
            rc = STDON+STERR;
      } else
         rc = STDON+STERR;	// Non FP track
   }

   FreeIORB (pUnitCB, pIORB);

   if (rc != STDON)
      return rc;

   BuildCDB_ModeSense_10 (pUnitCB, PAGE_WRITE, (NPIORB_CDB FAR *) &pIORB); // 10/23/2000 MB

   if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
   {
      pModeCap = (struct ModeSelectParmList_10 NEAR *) &(pIORB->CDB_data);
      pWrite = (struct Page_Write_Parameters NEAR *) &(pModeCap->Descriptors.WritePar);
      pWrite->FP = pUnitCB->PacketInfo.packet_fp;
      pWrite->write_type = pUnitCB->PacketInfo.write_type;     // 05-21-2001 AK: set Write Type Field as Packet
      pWrite->track_mode = pUnitCB->PacketInfo.track_mode;
      pWrite->data_block_type = pUnitCB->PacketInfo.block_type;
      pWrite->multi_session = 0;
      ul.dword = pUnitCB->PacketInfo.packet_size;
      pWrite->packet_size.ulbytes.byte_0 = ul.ulbytes.byte_3;
      pWrite->packet_size.ulbytes.byte_1 = ul.ulbytes.byte_2;
      pWrite->packet_size.ulbytes.byte_2 = ul.ulbytes.byte_1;
      pWrite->packet_size.ulbytes.byte_3 = ul.ulbytes.byte_0;
      if (pWrite->data_block_type == 8)
      {
         pWrite->session_format = 0;
         pWrite->sub_header_byte_2 = 0x20;
      }
      if (pWrite->data_block_type == 10)
      {
         pWrite->session_format = 0x20;
         pWrite->sub_header_byte_2 = 8;
      }

      BuildCDB_ModeSelect_WP (pUnitCB, pIORB->CDB_data, (NPIORB_CDB FAR *) &pIORB1);

      rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB1);  // 10/23/2000 MB

      FreeIORB (pUnitCB, pIORB1);
   }

   FreeIORB (pUnitCB, pIORB);

   return rc;
}

/****************************************************************************
 *
 * FUNCTION NAME = GetMediumType
 *
 * DESCRIPTION   = Get medium type
 *
 *                 USHORT GetMediumType  (NPUNITCB pUnitCB)
 *
 * INPUT         = pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Medium Type
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

UCHAR  GetMediumType (pUnitCB)

NPUNITCB   pUnitCB;
{
   USHORT rc;
   NPIORB_CDB pIORB;
   struct ModeSenseHdr_10 NEAR *pModeSenseHdr;
   struct ModeSenseParmList_10 NEAR *pModeCap;
   struct Configuration NEAR *pModeConf;
   struct ReadDVDParmList NEAR *pReadDVD;
   UCHAR  media_type = 0;  // 10/23/2000 MB
   UCHAR  dev_write;
   UCHAR  is_page5;
   UCHAR  type;

   if (!(pUnitCB->Flags&UCF_GETCONF_NS))   // 10/23/2000 MB
   {
      BuildCDB_GetConfiguration (pUnitCB, FEATURE_RANDOM_WRITABLE, (NPIORB_CDB FAR *) &pIORB);

      if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
      {
         pModeConf = (struct Configuration NEAR *) &(pIORB->CDB_data);

         if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_DVDRAM)
         {
            if (pModeConf->Descriptors.RandomFeature.Current)
               media_type = DMD_DVDRAM;
            else
               media_type = DMD_DVDRAM | DMD_WP;
         }
         else if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_DVDROM)
            media_type = DMD_DVDROM;
         else if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_CDROM)
            media_type = DMD_CDROM;
      }

      FreeIORB (pUnitCB, pIORB);
   }

   if (rc==STERR + STDON + ERROR_I24_BAD_COMMAND)
      pUnitCB->Flags|=UCF_GETCONF_NS;

   if (media_type)
      return media_type;

   if (!(pUnitCB->Flags&UCF_GETCONF_NS))   // 10/23/2000 MB
   {
      BuildCDB_GetConfiguration (pUnitCB, FEATURE_INCREMENTAL_WRITABLE, (NPIORB_CDB FAR *) &pIORB);

      if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
      {
         pModeConf = (struct Configuration NEAR *) &(pIORB->CDB_data);

         if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_CDR)
         {
            if (pModeConf->Descriptors.IncrFeature.Current)
               media_type = DMD_CDR;
            else
               media_type = DMD_CDR | DMD_WP;
         }
         else if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_CDRW)
         {
            if (pModeConf->Descriptors.IncrFeature.Current)
               media_type = DMD_CDRW;
            else
               media_type = DMD_CDRW | DMD_WP;
         }
         else if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_DVDR)
         {
            if (pModeConf->Descriptors.IncrFeature.Current)
               media_type = DMD_DVDR;
            else
               media_type = DMD_DVDR | DMD_WP;
         }
         else if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_DVDROM)
            media_type = DMD_DVDROM;
         else if (pModeConf->FeatureHdr.CurrentProfile.usbytes.byte_1 == PROFILE_CDROM)
            media_type = DMD_CDROM;
      }

      FreeIORB (pUnitCB, pIORB);
   }

   if (media_type)
      return media_type;

   BuildCDB_ReadDVDStructure (pUnitCB, READDVD_FORMAT_00, (NPIORB_CDB FAR *) &pIORB );  // AK 04/11/02 - check DVD media, defect 269694

   if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)
   {
      pReadDVD = (struct ReadDVDParmList NEAR *) &(pIORB->CDB_data);
      type = pReadDVD->DataFormat.DataFormat_0.book_type;
      if (type == BOOKTYPE_DVDROM)
         media_type = DMD_DVDROM;
      else if (type == BOOKTYPE_DVDRAM)
         media_type = DMD_DVDRAM;
      else if (type == BOOKTYPE_DVDR)
         media_type = DMD_DVDR;
   }

   FreeIORB (pUnitCB, pIORB);

   if (media_type && media_type != DMD_DVDROM)
   {
      BuildCDB_ReadDVDStructure (pUnitCB, READDVD_FORMAT_C0, (NPIORB_CDB FAR *) &pIORB);

      if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
      {
         pReadDVD = (struct ReadDVDParmList NEAR *) &(pIORB->CDB_data);
         if (pReadDVD->DataFormat.DataFormat_C0.WP_flags != 0)
            media_type |= DMD_WP;
      }
      FreeIORB (pUnitCB, pIORB);
   }

   if (media_type)
      return media_type;

   BuildCDB_ModeSense_10 (pUnitCB, PAGE_CAPABILITIES, (NPIORB_CDB FAR *) &pIORB);   // 10/23/2000 MB

   if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
   {
      pModeSenseHdr = (struct ModeSenseHdr_10 NEAR *) &(pIORB->CDB_data);
      type = pModeSenseHdr->medium_type;
      pModeCap = (struct ModeSenseParmList_10 NEAR *) &(pIORB->CDB_data);
      dev_write = pModeCap->Descriptors.Capabilities.write_capabilities;
   }
   else
   {
      media_type = DMD_CDROM;
   }

   FreeIORB (pUnitCB, pIORB);

   if (media_type)
      return media_type;

   BuildCDB_ModeSense_10 (pUnitCB, PAGE_WRITE, (NPIORB_CDB FAR *) &pIORB); //10/23/2000 MB

   if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
   {
      is_page5 = 1;
   }
   else
   {
      is_page5 = 0;
   }

   FreeIORB(pUnitCB, pIORB);

   if (type != 0 && type < 0x80)
   {
      type &= 0xF0;
      if (type == MEDIUM_TYPE_CDROM)
         media_type = DMD_CDROM;
      if (type == MEDIUM_TYPE_CDR)
      {
         if ((dev_write & WRITE_CDR_ENABLED) && is_page5)
            media_type = DMD_CDR;
         else
            media_type = DMD_CDR | DMD_WP;
      }
      if (type == MEDIUM_TYPE_CDRW)
      {
         if ((dev_write & WRITE_CDRW_ENABLED) && is_page5)
            media_type = DMD_CDRW;
         else
            media_type = DMD_CDRW | DMD_WP;
      }
      if (type == MEDIUM_TYPE_DVDROM)
         media_type = DMD_DVDROM;
   }

   if (media_type == 0)
      media_type = DMD_CDROM;

   return (media_type);
}

/****************************************************************************
 *
 * FUNCTION NAME = isDVDMedia
 *
 * DESCRIPTION   = Checks if DVD media
 *
 *                 BOOL isDVDMedia  (UCHAR mtype)
 *
 * INPUT         = mtype          - Media Type
 *
 * RETURN        = BOOL           - TRUE, if DVD media
 *
 ****************************************************************************/

BOOL  isDVDMedia (mtype)

UCHAR   mtype;
{
   UCHAR mt;

   mt = mtype & (UCHAR)(~DMD_WP);
   if (mt == DMD_DVDROM || mt == DMD_DVDRAM || mt == DMD_DVDR || 
       mt == DMD_DVDRW  || mt == DMD_DVDRWPLUS)
     return TRUE;

   return FALSE; 
}

/****************************************************************************
 *
 * FUNCTION NAME = Strncmp
 *
 * DESCRIPTION   = string compare
 *
 *                 Strncmp (NPBYTE string1, NPBYTE string2, USHORT length)
 *
 *                 This is like the standard C function, except it is boolean
 *
 * INPUT         = string1         - string 1
 *                 string2         - string 2
 *                 length          - length of string
 *
 * OUTPUT        = BOOLEAN, 1 if equal, 0 if not equal
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

BOOL Strncmp ( string1, string2, length )

UCHAR   *string1, *string2;
USHORT  length;

{
   BOOL    equal = TRUE;

   while ( length-- > 0 )
   {
      if ( *string1++ != *string2++ )
      {
         equal = FALSE;
         break;
      }
   }
   return ( equal );
}


#define PARM_PACKET_LOCKED   1
#define DATA_PACKET_LOCKED   2

/****************************************************************************
 *
 * FUNCTION NAME = CD_Strat1b
 *
 * DESCRIPTION   = VOID CD_strat1b  ()
 *
 * INPUT         = ES:BX    - Request Packet
 *
 * OUTPUT        =
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

VOID NEAR CD_Strat1b()
{
   PRP_GENIOCTL  pRP;
   USHORT        Cmd, Status, rc;
   PRPINITOUT    pRPO;
   LOCKSTRUC     lock_struc;

   _asm
   {
      push es
      push bx
      mov word ptr pRP[0], bx
      mov word ptr pRP[2], es
   }

   pRPO = (PRPINITOUT) pRP;                  /* Output for Init RP           */
   lock_struc.flags = 0;
   Status = STDON;

   Cmd = pRP->rph.Cmd;

   if (Cmd == CMDInitBase || Cmd == CMDInit)
   {
      pRPO->DataEnd = DataBreakAddress;
      pRPO->CodeEnd = CodeBreakAddress;
      if (CodeBreakAddress == 0)
         Status = STDON + STERR + ERROR_I24_QUIET_INIT_FAIL;
   }
   else if ( (Cmd == CMDGenIOCTL) && (pRP->Category == IOC_CDROM_2) &&
             ((pRP->Function == IOCD_RETURN_DRIVE_LETTER) || (pRP->Function == IOCD_RETURN_FEATURES)) )
   {

/*
 * Fixed error in IOCtl command (Cat:82h, Func:60h)   AK: 03/06/2001
 */
      Status = VerifyParameters(pRP, 0, 4, (LOCKSTRUC FAR*)&lock_struc);

      if (Status == STDON && pRP->DataPacket != NULL)
      {
         if (pRP->Function == IOCD_RETURN_DRIVE_LETTER)
         {
            ((struct DriveLetter_Data FAR *)pRP->DataPacket)->drive_count =
            NumCDROMDrives;

            ((struct DriveLetter_Data FAR *)pRP->DataPacket)->first_drive_number =
            FirstDriveNumber;
         }
         if (pRP->Function == IOCD_RETURN_FEATURES)
         {
            ((struct DriveFeatures_Data FAR *)pRP->DataPacket)->driver_status =
             FEATURE_USB_SUPPORT | FEATURE_CDRW_SUPPORT | FEATURE_EXECMD_SUPPORT;
         }
      }
      else
      {
         rc = STDON + STERR + ERROR_I24_INVALID_PARAMETER;
      }
      if (lock_struc.flags & PARM_PACKET_LOCKED)
      {
         DevHelp_VMUnLock (lock_struc.lin_lock_struc +
            (ULONG)(FIELDOFFSET(LOCKSTRUC, lockhandle_param[0])));
      }

      if (lock_struc.flags & DATA_PACKET_LOCKED)
      {
         DevHelp_VMUnLock (lock_struc.lin_lock_struc +
            (ULONG)(FIELDOFFSET(LOCKSTRUC, lockhandle_data[0])));   /*@V133532*/
      }
   }
   else
      Status = STDON + STERR + ERROR_I24_BAD_COMMAND;

   DISABLE;

   pRP->rph.Status = Status ;         /*Save status in Request Packet */

   ENABLE;

   _asm
   {
      pop bx
      pop es
      LEAVE
      retf
   }
}

