/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/**************************************************************************
 *
 * SOURCE FILE NAME = CDIOCTL.C
 *
 * DESCRIPTIVE NAME = IOCTL handling routines for OS/2 CD-ROM Device Mgr
 *
 *
 * VERSION = V2.0
 *
 * DATE
 *
 * DESCRIPTION
 *
 *
 * FUNCTIONS
 *
 * ENTRY POINTS:
 *
 * DEPENDENCIES:
 *
 * NOTES
 *
 *
 * STRUCTURES
 *
 * EXTERNAL REFERENCES
 *
 * EXTERNAL FUNCTIONS
 *
*/

#include "cdh.h"
#include "ioctl.h"
#include "dskioctl.h"


#define PARM_PACKET_LOCKED   1
#define DATA_PACKET_LOCKED   2
                                  

/****************************************************************************
 *
 * FUNCTION NAME = f_CD_DriveGenIOCTL
 *
 * DESCRIPTION   = Category 8/80/81 IOCTL routines
 *
 *          Category 8/80/81 IOCTL routines and IOCTL function router.
 *
 *          USHORT f_CD_DriveGenIOCTL (PRP_GENIOCTL pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT NEAR f_CD_DriveGenIOCTL (pRP, pUnitCB)

PRP_GENIOCTL  pRP;
NPUNITCB      pUnitCB;

{
   /*
   ** IOCTL Command Router Tables
   */

   /*
   ** Category 8 - Logical Disk Control Command Router Table
   */
    static IOCTL_TABLE_ENTRY Cat8_IOCTL_Table[] =
    {
     {IODC_GP,            CD_GetDeviceParms8,
                          sizeof(DDI_DeviceParameters_param),
                          sizeof(DDI_DeviceParameters_data),
                          TRUE },                                       //SD@135221

     {IODC_RC,            CD_RemovableMediaControl,
                          sizeof(DDI_DsktRemovMediaCtl_param),
                          0,FALSE},                                     //SD@135221

     {IODC_TS,            CD_SetActiveTray,                             //SD@135221
                          0,                                            //SD@135221
                          sizeof(struct SetTray_Parm),                  //SD@135221
                          TRUE},                                        //SD@135221

     {IODC_ST,            CD_RemovableDeviceStatus,
                          0,
                          sizeof(DDI_DsktRemovMediaCtl_data),
                          TRUE },                                       //SD@135221

     {IODC_TQ,            CD_TrayStatus,                                //SD@135221
                          0,                                            //SD@135221
                          sizeof(struct ReturnTrayStatus_Data),         //SD@135221
                          TRUE},                                        //SD@135221

     {IODC_FT,		  CD_FormatVerifyTrack,
			  sizeof(DDI_FormatPacket_param),
			  sizeof(DDI_FormatPacket_data),
			  FALSE},

     {0xff},                // End of Table
    };

    static IOCTL_TABLE_ENTRY Cat9_IOCTL_Table[] =
    {
     {IODC_GP,            CD_GetDeviceParms9,
                          sizeof(DDI_DeviceParameters_param),
                          sizeof(DDI_PhysDeviceParameters_data),
                          TRUE },                                       //SD@135221

     {0xff},                // End of Table
    };

    /*
    ** Category 80 - CD-ROM IOCTL Command Router Table
    */
    static IOCTL_TABLE_ENTRY Cat80_IOCTL_Table[] =
    {
     {IOCD_RESET,         CD_ResetDrive,
                          sizeof(struct ResetDrive),
                          0,TRUE},                                      //SD@135221

     {IOCD_EJECT,         CD_EjectDisk,
                          sizeof(struct EjectDisk),
                          0,TRUE},                                      //SD@135221

     {IOCD_CLOSE_TRAY,    CD_CloseTray,
                          sizeof(struct CloseTray),
                          0,TRUE},                                      //SD@135221

     {IOCD_LOCK_UNLOCK,   CD_LockUnlock,
                          sizeof(struct LockUnlock),
                          0,TRUE},                                      //SD@135221

     {IOCD_SET_ACTIVE_TRAY,                                             //SD@135221
                          CD_SetActiveTray,                             //SD@135221
                          0,                                            //SD@135221
                          sizeof(struct SetTray_Parm),                  //SD@135221
                          TRUE},                                        //SD@135221
                                                                        //SD@135221
/*   {IOCD_WRITE_CTRL,    CD_WriteCtrl},  */

     {IOCD_SEEK,          CD_Seek,
                          sizeof(struct Seek),
                          0,FALSE},                                     //SD@135221


     {IOCD_WRITE_LONG,    CD_WriteLong,
                          sizeof(struct WriteLong),
                          1,FALSE},                                     //SD@135221

     {IOCD_WRITEV_LONG,   CD_WriteVLong,
                          sizeof(struct WriteLong),
                          1,FALSE},                                     //SD@135221

     {IOCD_DEVICE_STATUS, CD_DeviceStatus,
                          sizeof(struct DeviceStatus),
                          sizeof(struct DeviceStatus_Data),TRUE},       //SD@135221

     {IOCD_TRAY_STATUS,   CD_TrayStatus,                                //SD@135221
                          0,                                            //SD@135221
                          sizeof(struct ReturnTrayStatus_Data),TRUE},   //SD@135221

     {IOCD_IDENTIFY,      CD_Identify,
                          sizeof(struct IdentifyCDROMdriver),
                          sizeof(struct IdentifyCDROMdriver_Data),TRUE},//SD@135221

     {IOCD_SECTOR_SIZE,   CD_ReturnSectorSize,
                          sizeof(struct ReturnSectorSize),
                          sizeof(struct ReturnSectorSize_Data),TRUE},   //SD@135221

/*   {IOCD_READ_CTRL,     CD_ReadCtrl},   */

     {IOCD_HEAD_LOCATION, CD_HeadLocation,
                          sizeof(struct LocationOfHead),
                          sizeof(struct LocationOfHead_Data),FALSE},    //SD@135221

     {IOCD_READ_PREFETCH, CD_ReadPrefetch,
                          sizeof(struct ReadPrefetch),
                          0,FALSE},                                     //SD@135221

     {IOCD_READ_LONG,     CD_ReadLong,
                          sizeof(struct ReadLong),
                          1,FALSE},                                     //SD@135221

     {IOCD_READ_LONG_PRE, CD_ReadLongPre,
                          sizeof(struct ReadLongPrefetch),
                          0,FALSE},
                                                                        //SD@135221
     {IOCD_VOLUME_SIZE,   CD_ReturnVolumeSize,
                          sizeof(struct ReturnVolumeSize),
                          sizeof(struct ReturnVolumeSize_Data),FALSE},  //SD@135221

     {IOCD_UPC,           CD_GetUPC,
                          sizeof(struct UPCCode),
                          sizeof(struct UPCCode_Data),FALSE},           //SD@135221

     {IOCD_EXECMD,        CD_ExecCMD,		/* AK: 02-23-2001: func 76 added */
                          sizeof(struct ExecCMD),
                          1,FALSE},

     {0xff},                // End of Table
    };

 /*
 ** Category 81H - CD-ROM AUDIO IOCTL Command Router Table
 */
    static IOCTL_TABLE_ENTRY Cat81_IOCTL_Table[] =
    {
     {IOCD_CHANNEL_CTRL,    CD_AudioChannelCtrl,
                            sizeof(struct AudioChannelControl),
                            sizeof(struct AudioChannelControl_Data),TRUE},//SD@135221

     {IOCD_PLAY,            CD_Play,
                            sizeof(struct PlayAudio),
                            0,FALSE},                                     //SD@135221

     {IOCD_STOP,            CD_Stop,
                            sizeof(struct StopAudio),
                            0,FALSE},                                     //SD@135221


     {IOCD_RESUME,          CD_Resume,
                            sizeof(struct ResumeAudio),
                            0,FALSE},                                     //SD@135221

     {IOCD_CHANNEL_INFO,    CD_AudioChannelInfo,
                            sizeof(struct AudioChannelInfo),
                            sizeof(struct AudioChannelInfo_Data),TRUE},   //SD@135221

     {IOCD_DISK_INFO,       CD_AudioDiskInfo,
                            sizeof(struct AudioDiskInfo),
                            sizeof(struct AudioDiskInfo_Data),FALSE},     //SD@135221

     {IOCD_TRACK_INFO,      CD_AudioTrackInfo,
                            sizeof(struct AudioTrackInfo),
                            sizeof(struct AudioTrackInfo_Data),FALSE},    //SD@135221

     {IOCD_QCHANNEL_INFO,   CD_AudioQChannelInfo,
                            sizeof(struct AudioQChannelInfo),
                            sizeof(struct AudioQChannelInfo_Data),FALSE}, //SD@135221

     {IOCD_SUBCHANNEL_INFO, CD_AudioSubChannelInfo,
                            sizeof(struct AudioSubChannelInfo),
                            0,FALSE},                                    //SD@135221   // 10/23/2000 MB

     {IOCD_STATUS_INFO,     CD_AudioStatusInfo,
                            sizeof(struct AudioStatusInfo),
                            sizeof(struct AudioStatusInfo_Data),FALSE},   //SD@135221

     {0xff},                  // End of Table
    };


   /*
   ** Verify it's an IOCTL the driver supports, verify
   ** the parameter and data packets, and route the
   ** the request to it's command handler.
   */

   USHORT  rc, Unit, i;
   USHORT (NEAR *pIOCTL_Routine)(PRP_GENIOCTL, NPUNITCB);
   IOCTL_TABLE_ENTRY *pTable;

   LOCKSTRUC lock_struc;

#ifdef DEBUG
{
   ULONG lFun=(ULONG)pRP->Function, lCat=(ULONG)pRP->Category, lRc=(ULONG)rc;
   dprintf("OS2CDROM f_CD_DriveGenIOCTL cat=%x, fun=%x, rc=enter\r\n",
           (PVOID)&lCat, (PVOID)&lFun);
}
#endif

   switch (pRP->Category)
   {
      case IOC_DC:
         pTable = Cat8_IOCTL_Table;
         break;

      case IOC_PD:
         pTable = Cat9_IOCTL_Table;
         break;

      case IOC_CDROM:
         pTable = Cat80_IOCTL_Table;
         break;

      case IOC_CDROM_AUDIO:
         pTable = Cat81_IOCTL_Table;
         break;

      default:
         return (STERR + STDON + ERROR_I24_BAD_COMMAND);
   }

   /*
   ** Verify it's an IOCTL Function the  driver supports
   ** and get the entry point of the IOCTL function to call
   */
   for (i = 0; pTable[i].Function != 0xff; i++)
   {
      if (pRP->Function == pTable[i].Function)
      {
         pIOCTL_Routine = pTable[i].pIOCTL_Routine;
         break;
      }
   }

   if (pTable[i].Function == 0xff)
      return (STERR + STDON + ERROR_I24_BAD_COMMAND);

   /*
   ** Valid Category and Command Code.  Now validate access to the
   ** parameter and data packets.  Lock down the data packet if required.
   */

   Unit = pRP->rph.Unit;

   lock_struc.flags = 0;

   if (VerifyParameters(pRP, pTable[i].ParamPktLen,
                             pTable[i].DataPktLen,
                             (LOCKSTRUC FAR *) &lock_struc) == STDON)

      /*
      ** Invoke the command handler for the IOCTL
      */
   {

   if(pUnitCB->DeviceInfo.Audio.capabilities &                            //SD@135221
     (DCAPS_CARTRIDGE_CHANGER |DCAPS_INDIVIDUAL_CHANGER))                 //SD@135221
     {                                                                    //SD@135221
     // only ATAPI devices support this flag today                        //SD@135221
     if(pUnitCB->pParentUnitCB)                                           //SD@135221
       {                                                                  //SD@135221
       // if we have a parent, and the active slot is not ours            //SD@135221
       if(pUnitCB->pParentUnitCB &&                                       //SD@135221
         (pUnitCB->pParentUnitCB->DeviceInfo.Slots.Current!=pUnitCB->DeviceInfo.Slot))//SD@135221
         {                                                                //SD@135221
         if(!(pUnitCB->DeviceInfo.Audio.capabilities & DCAPS_SINGLE_MODE))//SD@135221
           {                                                              //SD@135221
           // if it is allowed for active slot not to be as requested     //SD@135221
           if(pTable[i].ChangerAllowed==TRUE)                             //SD@135221
             {                                                            //SD@135221
             // allow processing to continue                              //SD@135221
             rc=STDON;                                                    //SD@135221
             } /* endif */                                                //SD@135221
           else                                                           //SD@135221
             {                                                            //SD@135221
             // else we have a not ready error, can't see the slot        //SD@135221
             rc=STDON + STERR + ERROR_I24_NOT_READY;                      //SD@135221
             }                                                            //SD@135221
           } /* endif */                                                  //SD@135221
         else                                                             //SD@135221
           {                                                              //SD@135221
           rc=STDON;                                                      //SD@135221
           } /* endelse */                                                //SD@135221
         } /* endif */                                                    //SD@135221
       else                                                               //SD@135221
         rc=STDON;                                                        //SD@135221
       // if slot is now active one                                       //SD@135221
       if(rc==STDON)                                                      //SD@135221
         {                                                                //SD@135221
         rc = (*pIOCTL_Routine)(pRP, pUnitCB);                            //SD@135221
         } /* endif */                                                    //SD@135221
       } /* endif */                                                      //SD@135221
     } /* endif */                                                        //SD@135221
   else                                                                   //SD@135221
       rc = (*pIOCTL_Routine)(pRP, pUnitCB);
   }
   else
   {
      rc = STDON + STERR + ERROR_I24_INVALID_PARAMETER;
   }

   if (lock_struc.flags & PARM_PACKET_LOCKED)
   {
         DevHelp_VMUnLock (lock_struc.lin_lock_struc +
            (ULONG)(FIELDOFFSET(LOCKSTRUC, lockhandle_param[0])));  /*@V133532*/
   }

   if (lock_struc.flags & DATA_PACKET_LOCKED)
   {
         DevHelp_VMUnLock (lock_struc.lin_lock_struc +
            (ULONG)(FIELDOFFSET(LOCKSTRUC, lockhandle_data[0])));   /*@V133532*/
   }

#ifdef DEBUG
{
   ULONG lFun=(ULONG)pRP->Function, lCat=(ULONG)pRP->Category, lRc=(ULONG)rc;
   dprintf("OS2CDROM f_CD_DriveGenIOCTL cat=%x, fun=%x, rc=%x\r\n",
           (PVOID)&lCat, (PVOID)&lFun, (PVOID)&lRc);
}
#endif
   return(rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = VerifyParameters
 *
 * DESCRIPTION   = Verify Parameter and Data Packet Pointers
 *
 *       Verify access to the Parameter and Data Packet pointer is the
 *       IOCTL request packet.  If the request is for an operation where
 *       data will be DMA'ed directly into the data packet, then issue a
 *       a long term lock on the data packet.
 *
 *       USHORT VerifyParameters (PRP_GENIOCTL pRP, USHORT ParmPktLen,
 *                                                  USHORT DataPktLen,
 *                                                  PBYTE plock_struc,
 *
 * INPUT         = pRP              - Request Packet
 *                 ParamPktLen      - Length of Parameter Packet
 *                 DataPktLen       - Length of Data Packet
 *                 plock_struc      - address of lock structure
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT VerifyParameters (pRP, ParamPktLen, DataPktLen, plock_struc)

PRP_GENIOCTL pRP;
USHORT       ParamPktLen;
USHORT       DataPktLen;
PLOCKSTRUC   plock_struc;

{
   USHORT rc;
   ULONG lin_Pkt;                /* Linear address of Parm or Data Packet */
   ULONG lin_lock_struc;
   ULONG lin_lockhandle_param;
   ULONG lin_lockhandle_data;
/* ULONG lin_page_list;                                                   */
   ULONG PageListCount;
   ULONG Length;
   ULONG VMLockFlags;


   rc = 0;

   /*
   ** Verify the string CD01 appears at the beginning of each Cat 80
   ** and Cat 81 parameter packet.
   */
   if (pRP->Category == IOC_CDROM || pRP->Category == IOC_CDROM_AUDIO)
   {
      if ( (pRP->Category == IOC_CDROM) && (pRP->Function == IOCD_IDENTIFY) )
      ;
      else
        if ( ((struct ResetDrive FAR *)pRP->ParmPacket)->ID_code != CD01 )
            return (STDON + STERR + ERROR_I24_INVALID_PARAMETER);
   }

   /*
   ** Get linear address of lock structure on the stack
   */
   DevHelp_VirtToLin(SELECTOROF(plock_struc),
                    (ULONG) OFFSETOF(plock_struc),
                    (PVOID) &lin_lock_struc);

   plock_struc->lin_lock_struc = lin_lock_struc;

/*
** lin_pagelist =  lin_lock_struc +
**                 (ULONG) ((USHORT) &(plock_struc->page_list[0]) -
**                          (USHORT) &(plock_struc));
*/
   /*
   ** Verify access to the parameter packet
   */
   if (ParamPktLen != 0)
   {
      /*
      ** Get linear address of the Parameter Packet
      */
      rc = DevHelp_VirtToLin(SELECTOROF(pRP->ParmPacket),
                            (ULONG) OFFSETOF(pRP->ParmPacket),
                            (PVOID) &lin_Pkt);
      if (rc == 0)
      {
         lin_lockhandle_param = lin_lock_struc +
                       (ULONG) ((USHORT)(ULONG) &(plock_struc->lockhandle_param[0]) -  // 10/23/2000 MB
                                (USHORT)(ULONG) &(plock_struc->flags));                // 10/23/2000 MB

         if ((rc = DevHelp_VMLock(VMDHL_LONG | VMDHL_VERIFY,
                                  lin_Pkt,
                                  ParamPktLen,
                                  -1L,
                                  lin_lockhandle_param,
                                  (PULONG) &PageListCount)) == 0)
         {
            plock_struc->flags = PARM_PACKET_LOCKED;
         }
      }
   }

   /*
   ** Verify access to the data packet.  If it's a ReadLong, then
   ** long term lock and make contig the data packet.
   */
   if (rc == 0 && DataPktLen != 0)
   {
      rc = DevHelp_VirtToLin(SELECTOROF(pRP->DataPacket),
                       (ULONG) OFFSETOF(pRP->DataPacket),
                       (PVOID) &lin_Pkt);
      if (rc == 0)
      {
         lin_lockhandle_data = lin_lock_struc +
                       (ULONG) ((USHORT)(ULONG) &(plock_struc->lockhandle_data[0]) -   // 10/23/2000 MB
                                (USHORT)(ULONG) &(plock_struc->flags));                // 10/23/2000 MB

         Length = DataPktLen;
         VMLockFlags = VMDHL_WRITE | VMDHL_LONG | VMDHL_VERIFY;

         if ((pRP->Category == IOC_CDROM) && (pRP->Function ==IOCD_READ_LONG))
         {
            VMLockFlags = VMDHL_CONTIGUOUS | VMDHL_16M |
                          VMDHL_WRITE | VMDHL_LONG;

            Length = ((struct ReadLong FAR *)pRP->ParmPacket)->transfer_count
                                                      * 2352;
         }

         //
         if ((pRP->Category == IOC_CDROM) && (pRP->Function == IOCD_WRITE_LONG || pRP->Function == IOCD_WRITEV_LONG))
         {
            VMLockFlags = VMDHL_CONTIGUOUS | VMDHL_16M |
                          VMDHL_WRITE | VMDHL_LONG;

            Length = ((struct WriteLong FAR *)pRP->ParmPacket)->transfer_count
                                                      * 2048;
         }
//
         if ((pRP->Category == IOC_CDROM) && (pRP->Function == IOCD_EXECMD)) // AK:02-26-2001 for IOCD_EXECMD function
         {
            VMLockFlags = VMDHL_CONTIGUOUS | VMDHL_16M |
                          VMDHL_WRITE | VMDHL_LONG;

            Length = ((struct ExecCMD FAR *)pRP->ParmPacket)->data_length;
         }

         rc = DevHelp_VMLock(VMLockFlags, lin_Pkt, Length, -1L,
                     lin_lockhandle_data, (PULONG) &PageListCount);
      }

      if (rc == 0)
      {
         plock_struc->flags |= DATA_PACKET_LOCKED;
      }
      else
      {
         if (plock_struc->flags & PARM_PACKET_LOCKED)
         {
            DevHelp_VMUnLock(lin_lockhandle_param);
            plock_struc->flags = 0;
         }
      }
   }

   if (rc)
      rc = STDON + STERR + ERROR_I24_INVALID_PARAMETER;
   else
      rc = STDON;

   return(rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_GetDeviceParms8
 *
 * DESCRIPTION   = Get Device Parms
 *
 *       Transfers the BPB, Number of Cylinders, Device Type and Device
 *       attributes to the IOCTL data packet.
 *
 *       USHORT CD_GetDeviceParms8(PRP_GENIOCTL pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_GetDeviceParms8 (pRP, pUnitCB)

PRP_GENIOCTL pRP;
NPUNITCB     pUnitCB;

{
   PDDI_DeviceParameters_param pParmPkt;
   PDDI_DeviceParameters_data pDataPkt;
   USHORT rc;
   ULONG volume_size;

   pParmPkt = (PDDI_DeviceParameters_param) pRP->ParmPacket;
   pDataPkt = (PDDI_DeviceParameters_data) pRP->DataPacket;

   if (pParmPkt->Command & ~COMMAND_INFO_MASK)
      return(STDON + STERR + ERROR_I24_INVALID_PARAMETER);

   rc = GetVolumeSize (pUnitCB, (ULONG FAR *) &volume_size);

   pDataPkt->bpb = DefaultBPB;

   if (rc == STDON)
      pDataPkt->bpb.BigTotalSectors = volume_size;
   else
      pDataPkt->bpb.BigTotalSectors = 270000L;

   pDataPkt->bpb.MediaType = pUnitCB->MediaType;
   pDataPkt->NumCylinders = DEFAULT_CYLINDER_COUNT;
   pDataPkt->DeviceType = TYPE_CDROM;
/* AK (02/06/2001), Defect 252957: R/W optical disk type setting for CD-RW and DVD-RAM */
   if (pUnitCB->MediaType == DMD_CDRW || pUnitCB->MediaType == DMD_DVDRAM)
      pDataPkt->DeviceType = TYPE_RWOPTICALDISK;

   pDataPkt->DeviceAttr = 0;

   if ( !(pUnitCB->UnitInfo.UnitFlags & UF_REMOVABLE) )
      pDataPkt->DeviceAttr |= DP_DEVICEATTR_NON_REMOVABLE;

/* if (pUnitCB->UnitInfo.UnitFlags & UF_CHANGELINE)                        */
      pDataPkt->DeviceAttr |= DP_DEVICEATTR_CHANGELINE;

   if (pUnitCB->Flags & UCF_16M)
      pDataPkt->DeviceAttr |= DP_DEVICEATTR_GT16MBSUPPORT;


   return(STDON);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_GetDeviceParms9
 *
 * DESCRIPTION   = Cat 9 Get Device Parms
 *
 *           Transfer physical device paramaters
 *
 *           USHORT CD_GetDeviceParms9(PRP_GENIOCTL pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_GetDeviceParms9 (pRP, pUnitCB)

PRP_GENIOCTL pRP;
NPUNITCB     pUnitCB;

{
   PDDI_PhysDeviceParameters_data pDataPkt;

   pDataPkt = (PDDI_PhysDeviceParameters_data) pRP->DataPacket;

   pDataPkt->NumCylinders = DEFAULT_CYLINDER_COUNT;
   pDataPkt->NumHeads = 1;
   pDataPkt->SectorsPerTrack = 0xFFFF;

   return(STDON);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_RemovableMediaControl
 *
 * DESCRIPTION   = Removable Media Control
 *
 *      Cat 8 function to allow lock, unlock or eject of media in the drive.
 *      This function simply calls the equivalent Cat 80 function.
 *
 *      USHORT CD_RemovableMediaControl(PRP_GENIOCTL pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_RemovableMediaControl (pRP, pUnitCB)

PRP_GENIOCTL pRP;
NPUNITCB     pUnitCB;

{
   USHORT rc;
   PDDI_DsktRemovMediaCtl_param pParmPkt;

   pParmPkt = (PDDI_DsktRemovMediaCtl_param) pRP->ParmPacket;

   switch (pParmPkt->Command)
   {
      case UNLOCK_MEDIA:
      case LOCK_MEDIA:
        rc = CD_LockUnlock(pRP, pUnitCB);
        break;

      case EJECT_MEDIA:
        rc = CD_EjectDisk(pRP, pUnitCB);
        break;

      case LOAD_MEDIA:                                              /*@V133532*/
        rc = CD_CloseTray(pRP, pUnitCB);                            /*@V133532*/
        break;                                                      /*@V133532*/

      default:
         rc = STDON + STERR + ERROR_I24_INVALID_PARAMETER;
   }

   return(rc);
}


/****************************************************************************
 *
 * FUNCTION NAME = CD_RemovableDeviceStatus
 *
 * DESCRIPTION   = return device status
 *
 *        Cat 8 function to return removable media status
 *
 *        USHORT CD_RemovableDeviceStatus (PRP_GENIOCTL pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_RemovableDeviceStatus (pRP, pUnitCB)

PRP_GENIOCTL pRP;
NPUNITCB     pUnitCB;

{
   NPIORB_CDB pIORB;
   ULONG status, capabilities;                                       /*V@91985*/
   PDDI_DsktRemovMediaCtl_data pDataPkt;
   struct CapabilitiesParmList_10 NEAR *pCDBData;                    /*V@91985*/
   struct CDB_ModeSense_10  NEAR *pCDB;                              /*V@91985*/

   pDataPkt = (PDDI_DsktRemovMediaCtl_data) pRP->DataPacket;
   
   if (pUnitCB->DeviceInfo.interface_type == INTERFACE_ATAPI)        /*V@91985*/
   {                                                                 /*V@91985*/
       /*                                                              V@91985
       ** Issue Mode Sense to read the CD-ROM capabilites page.        V@91985
       */                                                            /*V@91985*/
                                                                     /*V@91985*/
       BuildCDB_SenseCapabilities (pUnitCB,                          /*V@91985*/
                                   (NPIORB_CDB FAR *) &pIORB);       /*V@91985*/
       pCDB = (struct CDB_ModeSense_10 NEAR *) &(pIORB->CDB);        /*V@91985*/
       pCDB->PC = PC_CURRENT;                                        /*V@91985*/
                                                                     /*V@91985*/
       SubmitIORB_Wait (pUnitCB, (NPIORB)pIORB);                     /*V@91985*/ // 10/23/2000 MB
                                                                     /*V@91985*/
       if ( !(pIORB->apt.iorbh.Status & IORB_ERROR) )                /*V@91985*/
       {                                                             /*V@91985*/
          pCDBData = (struct CapabilitiesParmList_10 NEAR *)         /*V@91985*/
                                            pIORB->CDB_data;         /*V@91985*/
          capabilities = pCDBData->CapPage.capabilities;             /*V@91985*/
                                                                     /*V@91985*/
          if (capabilities & CP_LOCK_STATE )                         /*V@91985*/
             status = 1;     /* Drive Locked, lock supported */      /*V@91985*/
          else                                                       /*V@91985*/
             status = 2;     /* Drive Unlocked, lock supported */    /*V@91985*/
       }                                                             /*V@91985*/
       else                                                          /*V@91985*/
          status = 0;        /* Mode sense failed.            */     /*V@91985*/
                             /* Assume no lock/unlock support */     /*V@91985*/
       FreeIORB (pUnitCB, pIORB);                                    /*V@91985*/
   }                                                                 /*V@91985*/
   else                                                              /*V@91985*/
      status = 3;                    /* Lock supported, but lock status isnt */

   BuildCDB_TestUnitReady (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

   if  (SubmitIORB_Wait (pUnitCB, (NPIORB)pIORB) == STDON)  // 10/23/2000 MB
      status |= MEDIA_IN_DRIVE;

   FreeIORB (pUnitCB, pIORB);

   pDataPkt->Status = status;

   return(STDON);
}

USHORT MakeSlotActive(pUnitCB,Slot)                                               //SD@135221
                                                                                  //SD@135221
NPUNITCB pUnitCB;                                                                 //SD@135221
UCHAR Slot;                                                                       //SD@135221
{                                                                                 //SD@135221
USHORT rc;                                                                        //SD@135221
NPIORB pIORB;                                                                     //SD@135221
UCHAR asc,ascq;                                                                   //SD@135221
                                                                                  //SD@135221
     if(pUnitCB->DeviceInfo.playing==FALSE &&                                     //SD@135221
       pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying==TRUE)                    //SD@135221
       {                                                                          //SD@135221
       rc=STDON + STERR + ERROR_I24_NOT_READY;                                    //SD@135221
       } /* endif */                                                              //SD@135221
     else                                                                         //SD@135221
       {                                                                          //SD@135221
       for(; ; )                                                                  //SD@135221
         {                                                                        //SD@135221
         BuildCDB_ChangerLoad(pUnitCB, (NPIORB_CDB FAR *) &pIORB,Slot);           //SD@135221
                                                                                  //SD@135221
         pUnitCB->DeviceInfo.Slots.LoadInProgress=TRUE;                           //SD@135221
                                                                                  //SD@135221
         rc=SubmitIORB_Wait (pUnitCB, pIORB);                                     //SD@135221
                                                                                  //SD@135221
         pUnitCB->DeviceInfo.Slots.LoadInProgress=FALSE;                          //SD@135221
                                                                                  //SD@135221
         if( ((NPIORB_CDB)pIORB)->apt.iorbh.Status & IORB_ERROR )                 //SD@135221
           {                                                                      //SD@135221
           if(pUnitCB->DeviceInfo.product_id_code==TORISAN_C3G)                   //SD@135221
             {                                                                    //SD@135221
             asc=((NPIORB_CDB)pIORB)->sense_data.additional_sense_code;           //SD@135221
             ascq=((NPIORB_CDB)pIORB)->sense_data.additional_sense_code_qualifier;//SD@135221
                                                                                  //SD@135221
             if( asc==ASC_UNIT_NOT_READY && ascq==1 )                             //SD@135221
               {                                                                  //SD@135221
               // load is not yet finished                                        //SD@135221
               // wait 100ms and try again                                        //SD@135221
               DevHelp_ProcBlock ((ULONG)(PVOID)pIORB, 100L, 1);                  //SD@135221   // 10/23/2000 MB
               // continue loop, and retry                                        //SD@135221
               } /* endif */                                                      //SD@135221
             else if( asc==0x3A && ascq==0 )                                      //SD@135221
               {                                                                  //SD@135221
               // no media in this slot                                           //SD@135221
               rc=STDON + STERR + ERROR_I24_NOT_READY;                            //SD@135221
               break;                                                             //SD@135221
               } /* endif */                                                      //SD@135221
             else if( asc==0 && ascq==0 )                                         //SD@135221
               {                                                                  //SD@135221
               rc=STDON;                                                          //SD@135221
               pUnitCB->pParentUnitCB->DeviceInfo.Slots.Current=Slot;             //SD@135221
               break;                                                             //SD@135221
               } /* endelse */                                                    //SD@135221
             } /* endif */                                                        //SD@135221
           else                                                                   //SD@135221
             {                                                                    //SD@135221
             // say load failed, unit not ready                                   //SD@135221
             rc=STDON + STERR + ERROR_I24_NOT_READY;                              //SD@135221
             break;                                                               //SD@135221
             } /* endelse */                                                      //SD@135221
           }                                                                      //SD@135221
         else                                                                     //SD@135221
           {                                                                      //SD@135221
           rc=STDON;                                                              //SD@135221
           pUnitCB->pParentUnitCB->DeviceInfo.Slots.Current=Slot;                 //SD@135221
           break;                                                                 //SD@135221
           }                                                                      //SD@135221
         FreeIORB (pUnitCB, (NPIORB_CDB)pIORB);                                   //SD@135221
         pIORB=(NPIORB_CDB)0;                                                     //SD@135221
         } /* endfor */                                                           //SD@135221
       if(pIORB)                                                                  //SD@135221
         {                                                                        //SD@135221
         FreeIORB (pUnitCB, (NPIORB_CDB)pIORB);                                   //SD@135221
         } /* endif */                                                            //SD@135221
       } /* endelse */                                                            //SD@135221
     return rc;                                                                   //SD@135221
}                                                                                 //SD@135221

/****************************************************************************
 *
 * FUNCTION NAME = CD_FormatVerifyTrack
 *
 * DESCRIPTION   = return device status
 *
 *        Cat 8 function to format and verify track
 *
 *        USHORT CD_FormatVerifyTrack (PRP_GENIOCTL pRP, NPUNITCB pUnitCB)
 *
 * INPUT         = pRP              - Request Packet
 *                 pUnitCB          - Pointer to UnitCB
 *
 * OUTPUT        = USHORT           - Packet Status word
 *
 * RETURN-NORMAL =
 * RETURN-ERROR  =
 *
 ****************************************************************************/

USHORT CD_FormatVerifyTrack (pRP, pUnitCB)

PRP_GENIOCTL pRP;
NPUNITCB     pUnitCB;

{
   NPIORB_CDB pIORB, pIORB1;
   BOOL  playing;
   PDDI_FormatPacket_param pParmPkt;
   PDDI_FormatPacket_data  pDataPkt;
   ULONG blocks, pregap;
   USHORT packet_size;
   USHORT rc;
   union ULONGB ul;
   struct ModeSelectParmList_10 NEAR *pModeCap;
   struct Page_Write_Parameters NEAR *pWrite;
   struct Sense_Data NEAR *pSense;
   struct ReadDiscInfo_Data NEAR *pDiscInfo;
   struct ReadFormatCapacities_Data NEAR *pFormat;

   pDataPkt = (PDDI_FormatPacket_data) pRP->DataPacket;
   pParmPkt = (PDDI_FormatPacket_param) pRP->ParmPacket;
   pDataPkt->StartSector = 0;

   /*
   ** Check for uncertain media
   */
   if (pUnitCB->Flags & UCF_UNCERTAIN_MEDIA)
   {
      return (STDON + STERR + ERROR_I24_UNCERTAIN_MEDIA);
   }

   /*
   ** Cant format when unit is playing so check play status
   */
   if (pUnitCB->DeviceInfo.playing ||
       (pUnitCB->pParentUnitCB && pUnitCB->pParentUnitCB->DeviceInfo.Parentplaying))
   {
      rc = GetPlayStatus(pUnitCB, &playing);

      if ( (rc == STDON) && playing)
      {
         return (STDON + STERR + ERROR_I24_DEVICE_IN_USE);
      }
   }

   if (pParmPkt->Command & 0x80) /* 0x - Format, 1x - Verify */
   {
      /* Verify state of formatting */
      BuildCDB_TestUnitReady (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

      rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   //10/23/2000 MB

      FreeIORB (pUnitCB, pIORB);

      if (rc == STDON)
      {
         return (STDON + STERR + ERROR_I24_INVALID_PARAMETER);
      }

      BuildCDB_RequestSense (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

      if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
      {
         pSense = (struct Sense_Data NEAR *) &(pIORB->CDB_data);
         if (pSense->sensekey_specific[0] & 0x80)
         {
            pDataPkt->StartSector = (UCHAR)((((ULONG)(pSense->sensekey_specific[1]) * 256L + // 10/23/2000 MB
                                              (ULONG)(pSense->sensekey_specific[2])) * 100L) / 65535L);
            if (pDataPkt->StartSector == 0)
               pDataPkt->StartSector = 1;
         }
         else
            pDataPkt->StartSector = 0;
      }

      FreeIORB (pUnitCB, pIORB);

      return rc;
   }
   else
   {  /* Format media */
      switch (pUnitCB->MediaType & 0x7F)
      {
      case DMD_CDRW:
         /* Format CD-RW media */
         packet_size = pParmPkt->NumSectors;
         if (packet_size > MAX_PACKET_SIZE)
         {
            return (STDON+STERR+ERROR_I24_INVALID_PARAMETER);
         }
         if (packet_size == 0)
            packet_size = MAX_PACKET_SIZE;

         BuildCDB_ReadDiscInfo(pUnitCB, (NPIORB_CDB FAR *) &pIORB);
         
         if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 07/13/01 AK,BR, defect 257548 - count the number of blocks
         {
            pDiscInfo = (struct ReadDiscInfo_Data NEAR *) &(pIORB->CDB_data);

            blocks = (ULONG)pDiscInfo->lead_out.ulbytes.byte_1 * 75L * 60L +
                     (ULONG)pDiscInfo->lead_out.ulbytes.byte_2 * 75L +
                     (ULONG)pDiscInfo->lead_out.ulbytes.byte_3;
            pregap = 150;

            blocks = ((blocks - pregap) / (packet_size + 7)) * packet_size;
         }

         FreeIORB (pUnitCB, pIORB);

         if (rc != STDON)
            return rc;

         BuildCDB_ModeSense_10 (pUnitCB, PAGE_WRITE, (NPIORB_CDB FAR *) &pIORB); // 10/23/2000 MB

         if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
         {
            pModeCap = (struct ModeSelectParmList_10 NEAR *) &(pIORB->CDB_data);
            pWrite = (struct Page_Write_Parameters NEAR *) &(pModeCap->Descriptors.WritePar);

            pWrite->write_type = 0;
            pWrite->FP = 1;
            pWrite->track_mode = 7;
            pWrite->data_block_type = 10;
            pWrite->multi_session = 0;
            ul.dword = packet_size;
            pWrite->packet_size.ulbytes.byte_0 = ul.ulbytes.byte_3;
            pWrite->packet_size.ulbytes.byte_1 = ul.ulbytes.byte_2;
            pWrite->packet_size.ulbytes.byte_2 = ul.ulbytes.byte_1;
            pWrite->packet_size.ulbytes.byte_3 = ul.ulbytes.byte_0;
            pWrite->session_format = 0x20;
            pWrite->sub_header_byte_2 = 8;

            BuildCDB_ModeSelect_WP (pUnitCB, pIORB->CDB_data, (NPIORB_CDB FAR *) &pIORB1);

            rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB1);  // 10/23/2000 MB

            FreeIORB (pUnitCB, pIORB1);
         }

         FreeIORB (pUnitCB, pIORB);

         if (rc != STDON)
            return rc;

         BuildCDB_FormatUnit(pUnitCB, blocks, DMD_CDRW, (NPIORB_CDB FAR *) &pIORB);

         rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

         FreeIORB(pUnitCB, pIORB);

         return rc;

      case DMD_DVDRAM:
         /* Format DVD-RAM media */
         BuildCDB_ReadFormatCapacities (pUnitCB, (NPIORB_CDB FAR *) &pIORB);

         if ((rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB)) == STDON)   // 10/23/2000 MB
         {
            pFormat = (struct ReadFormatCapacities_Data NEAR *) &(pIORB->CDB_data);
            ul.ulbytes.byte_0 = pFormat->num_of_blocks.ulbytes.byte_3;
            ul.ulbytes.byte_1 = pFormat->num_of_blocks.ulbytes.byte_2;
            ul.ulbytes.byte_2 = pFormat->num_of_blocks.ulbytes.byte_1;
            ul.ulbytes.byte_3 = pFormat->num_of_blocks.ulbytes.byte_0;
            blocks = ul.dword;
         }

         FreeIORB (pUnitCB, pIORB);

         if (rc != STDON)
            return rc;

         BuildCDB_FormatUnit(pUnitCB, blocks, DMD_DVDRAM, (NPIORB_CDB FAR *) &pIORB);

         rc = SubmitIORB_Wait(pUnitCB, (NPIORB)pIORB);   // 10/23/2000 MB

         FreeIORB(pUnitCB, pIORB);

         return rc;

      default:
         return (STDON + STERR + ERROR_I24_WRITE_FAULT);
      }
   }
}
