/*DDK*************************************************************************/
/*                                                                           */
/* COPYRIGHT (C) IBM Corporation, 1992                                       */
/* COPYRIGHT    Copyright (C) 1992 IBM Corporation                           */
/*                                                                           */
/*    The following IBM OS/2 source code is provided to you solely for       */
/*    the purpose of assisting you in your development of OS/2 device        */
/*    drivers. You may use this code in accordance with the IBM License      */
/*    Agreement provided in the IBM Developer Connection Device Driver       */
/*    Source Kit for OS/2. This Copyright statement may not be removed.      */
/*                                                                           */
/*****************************************************************************/
/******************************************************************************/
/*   SCCSID = src/dev/dasd/cdrom/ibmatapi/atapiism.c, atapiflt, c.basedd 98/08/11                                                         */
/*                                                                            */
/*                                                                            */
/*   OCO Source Materials                                                     */
/*                                                                            */
/*   Program number (  )                                                      */
/*                                                                            */
/*                                                                            */
/*   The source code for this program is not published or otherwise divested  */
/*   of its tradesecrets, irrespective of what has been deposited with the    */
/*   U. S. Copyright Office.                                                  */
/*                                                                            */
/*   US Government Users Restricted Rights -- Use, duplication or disclosure  */
/*   restricted by GSA ADP Schedule Contract with IBM Corp.                   */
/*                                                                            */
/******************************************************************************/
/*
 *
 * SOURCE FILE NAME =     ATAPIISM.C
 *
 * DESCRIPTIVE NAME =     ATAPI Inner State Machine
 *
 *
 *
 * VERSION = 1.0
 *
 * DATE
 *
 * DESCRIPTION :
 *
 * Purpose:
 *
 *
 *
 *
 *
*/
#define INCL_NOBASEAPI
#define INCL_NOPMAPI
#include "os2.h"
#include "dos.h"
#include "dskinit.h"

#include "iorb.h"
#include "addcalls.h"
#include "dhcalls.h"
#include "apmcalls.h"

#define INCL_INITRP_ONLY
#include "reqpkt.h"

#include "scsi.h"
#include "cdbscsi.h"

#include "atapicon.h"
#include "atapireg.h"
#include "atapityp.h"
#include "atapiext.h"
#include "atapipro.h"

/* ͻ
                                                                             
  StartOSMRequest                                                            
                                                                             
  Outer State Machine Interface:  This routine initializes the inner state   
  machine and sets up the outer state machine to go into a WAITSTATE until   
  the inner state machine finishes the requested operation.                  
                                                                             
ͼ */
VOID FAR StartOSMRequest ( NPACB npACB )
{
  npACB->ISMState      = ACBIS_START_STATE;
  npACB->ISMDoneReturn = StartOSM;

  if(npACB->suspended)
  {
     return;
  }

  StartISM( npACB );
}

/* ͻ
                                    
  StartISM                          
                                    
  Inner State Machine Router        
                                    
*/
VOID NEAR StartISM( NPACB npACB )
{
   VOID (FAR *ISMDoneReturn)( NPACB );

   DISABLE

   npACB->ISMUseCount++;

   if ( npACB->ISMUseCount == 1 )
   {
      do
      {
         ENABLE
         npACB->ISMFlags &= ~ACBIF_WAITSTATE;
         do
         {
            switch ( npACB->ISMState )
            {
               case ACBIS_START_STATE:
                  ISMStartState ( npACB );
                  break;

               case ACBIS_INTERRUPT_STATE:
                  InterruptState( npACB );
                  break;

               case ACBIS_WRITE_ATAPI_PACKET_STATE:
                  WriteATAPIPkt( npACB );
                  break;

               case ACBIS_COMPLETE_STATE:
                  npACB->ISMFlags    |= ACBIF_WAITSTATE;
                  npACB->ISMUseCount  = 1;
                  npACB->OSMState     = ACBOS_ISM_COMPLETE;
                  break;

               default :                                     /* unknown state */
                  npACB->ISMUseCount  = 1;
                  npACB->ISMFlags    |= ACBIF_WAITSTATE;
                  break;
            }
         }
         while ( !(npACB->ISMFlags & ACBIF_WAITSTATE) );

         DISABLE
      }
      while ( --npACB->ISMUseCount );
   }
   ENABLE

   if ( npACB->ISMState == ACBIS_COMPLETE_STATE )
   {
      DISABLE
      if (npACB->ISMDoneReturn)
      {
         ISMDoneReturn        = npACB->ISMDoneReturn;
         npACB->ISMDoneReturn = 0;
         ENABLE
         (*ISMDoneReturn)( npACB );
      } else {
         ENABLE
      }
   }
}

/*
ͻ
                                    
  ISMStartState                     
                                    
  Initializes the Inner State       
  Machine                           
                                    
*/
VOID NEAR ISMStartState ( NPACB npACB )
{
   DISABLE

   if (!( npACB->OSMReqFlags & ACBR_RESET ) &&                      /*V@106915*/
       ( !DRQ_AND_BSY_CLEAR_WAIT( npACB ) ) )                       /*V@93531*/
   {                                           /* Max Poll wait time exceeded */
     if( npACB->npUCB->Flags & UCBF_FORCE )                        /*@V151168*//*@V155162*/
     {                                                             /*@V151168*/
        npACB->npUCB->Flags |= UCBF_NOTPRESENT; //set fake flag  /*@V151168*/
        npACB->IORBError     = IOERR_UNIT_NOT_READY;                   /*@V155162*/
        npACB->ISMState      = ACBIS_COMPLETE_STATE;                    /*@V155162*/

     } else {

        npACB->ISMState     = ACBIS_COMPLETE_STATE;
        npACB->OSMReqFlags |= ACBR_RESET;  /* Set OSM for reset */
        ENABLE

        return;                                                     /*@V151168*/
     }                                                             /*@V151168*/
   } else {

      ENABLE

      InitializeISM( npACB );

      StartATA_ATAPICmd( npACB);
   }
}

/*
ͻ
                                       
  InitializeISM                        
                                       
  Initializes state and flag variables 
                                       
ͼ */
VOID NEAR InitializeISM ( NPACB npACB )
{
   UCHAR  Opcode;
   NPUCB  npUCB = npACB->npUCB;
   SHORT  Port, Data;

   /*Ŀ
    Check ATAPI Command Operations 
   */
   if ( !( npACB->ISMFlags & ACBIF_ATA_OPERATION ) )
   {
      /*Ŀ
       set opcode dependent flags 
      */
      Opcode = npACB->npCmdIO->ATAPIPkt[0];

      npACB->BM_CommandCode = 0;

      switch( Opcode )
      {
         /*------------------------------------------------------*/
         /* Immediate Commands - complete after returning status */
         /*------------------------------------------------------*/
         /*------------------------*/
         /* Completion Status Only */
         /*------------------------*/

         case  FXP_TESTUNIT_READY:
         case  FXP_REZERO_UNIT   :
         case  FXP_SEEK          :
         case  FXP_SEND_DIAG     :

         case  FXP_START_STOPUNIT:
         case  FXP_PRV_ALWMEDRMVL:

         case  FXP_NEW_FORMATUNIT:
         case  FXP_FORMAT_UNIT   :
         case  FXP_WRITE_BUFFER  :
         case  FXP_READ_BUFFER   :
         case  FXP_VERIFY        :

         /*---------------------*/
         /*      Single IO      */
         /*---------------------*/
         case  FXP_READ_CAPACITY :
         case  FXP_INQUIRY       :
         case  FXP_READFORMAT_CAP:
         case  FXP_READ_DEFECT   :
         case  FXP_READ_LONG     :
         case  FXP_WRITE_LONG    :
         case  FXP_REASSIGNBLOCKS:
         case  FXP_REQUEST_SENSE :
         case  FXP_MODE_SENSE    :
         case  FXP_NON_SENSE     :                                  

            npACB->ISMFlags |= ACBIF_WAITSTATE;
            break;

         /*---------------------*/
         /*      Block IO       */
         /*---------------------*/
         case  FXP_READ_10     :
         case  FXP_WRITE_10    :
         case  FXP_READ_12     :
         case  FXP_WRITE_12    :
         case  FXP_WRITE_VERIFY:

            npACB->ISMFlags |= ACBIF_WAITSTATE;

            /* Begin [001] Determine if DMA capable system/CD-ROM and even length transfer */

            if ( (!InitActive) &&
                 !(npACB->ISMFlags & ACBIF_BUSMASTERDMA_FORCEPIO) &&
                 (npUCB->Flags & UCBF_BM_DMA) &&
                 !(npACB->npCmdIO->IOSGPtrs.numTotalBytes & 1) )
            {
              if( !CreateBMSGList( npACB ) )    /* try and create scatter/gather list */
              {
                /* change transfer setup to Bus Master DMA */

                npACB->ISMFlags &= ~ACBIF_MULTIPLEIO;
                npACB->ISMFlags |= ACBIF_BUSMASTERDMAIO;

                /* Shut down Bus Master DMA controller if active */

                Port = npACB->BMISTA;           /* Set port address to BM Status reg */
                inp ( Port, Data );
                if ( Data & ACBX_BMISTA_ACTIVE )
                {
                   StopBMDMA( npACB );
                }
                Port = npACB->BMISTA;           /* set to status reg again */
                Data &= ( ACBX_BMISTA_INTERRUPT |
                          ACBX_BMISTA_D0DMA |
                          ACBX_BMISTA_D1DMA );
                outp ( Port, Data );            /* write to port */

                npACB->BM_CommandCode = (ACBX_BMICOM_RW | ACBX_BMICOM_START);

                #ifdef ENABLE_COUNTERS
      //                  ++npUCB->DeviceCounters.TotalBMReadOperations;
                #endif

                Port = npACB->BMIDTP;           /* address descriptor base register in PIIX */
                Data = (USHORT) npACB->BMDMA_SGList;
                outwp ( Port, Data );
                Port += 2;
                Data = (USHORT) (npACB->BMDMA_SGList >> 16);
                outwp ( Port, Data );

                /*
                 * Bus Master DMA is now ready for the transfer.  It must be started after the
                 * command is issued to the drive, to avoid data corruption in case a spurious
                 * interrupt occurred at the beginning of the xfer and the drive is in an unknown
                 * state.
                 */
              }
              npACB->ISMFlags &= ~ACBIF_BUSMASTERDMA_FORCEPIO;
            }
            break;

         case  FXP_MODE_SELECT    :

            npACB->ISMFlags |= ACBIF_WAITSTATE;

            /*Ŀ
             change default transfer direction 
            */
            npACB->npCmdIO->IOSGPtrs.Mode = SGLIST_TO_PORT;

            break;

         default :
            npACB->ISMFlags |=  ACBIF_WAITSTATE;
            break;
      }
   }

   /*Ŀ
        Odd Byte Transfers                        
   */
   if ( npACB->npCmdIO->IOSGPtrs.numTotalBytes & 1 )
   {
      /*------------------------*/
      /* Convert to even number */
      /*------------------------*/
      if (npACB->npCmdIO->IOSGPtrs.Mode == PORT_TO_SGLIST)
      {
         npACB->npCmdIO->IOSGPtrs.numTotalBytes -= 1;
         npACB->ISMFlags |= ACBIF_ODD_BYTE_XFER;         /* set ODD Byte Flag */

      } else {

         npACB->npCmdIO->IOSGPtrs.numTotalBytes += 1;
      }
   }
}

/* ͻ
                                    
  StartATA_ATAPICmd                 
                                    
  Write an ATA or ATAPI Cmd         
                                    
*/
VOID NEAR StartATA_ATAPICmd ( NPACB npACB )
{
   USHORT  port;
   USHORT  data;
   USHORT  i;
   USHORT  IOMask;

   /*Ŀ
    Do not write if unknown command        
   */
   if (npACB->IORBError & IOERR_CMD_NOT_SUPPORTED)
   {
      npACB->ISMState      = ACBIS_COMPLETE_STATE;
      npACB->ISMDoneReturn = 0;
      return;
   }

   /*Ŀ
    Setup the Drive Select Register        
   */
   npACB->IORegs[FI_PDRVSLCT] = DEFAULT_ATAPI_DRV_SLCT_REG |
                                        ((npACB->UnitId == 0) ? UNIT0 : UNIT1 );

   /*Ŀ
    Processing of the ATA or ATAPI command 
   */
   if ( !(npACB->ISMFlags & ACBIF_ATA_OPERATION) )
   {
      IOMask = FM_PATAPI_CMD;           /* ATAPI Packet Command Register Mask */
      SetATAPICmd( npACB );

   }else{

      IOMask = FM_PATA_CMD;             /* ATA Command Register Mask */
      SetATACmd( npACB );
   }

   /*Ŀ
    Actual write of registers        
   */
   for ( i = FI_PFEATURE; IOMask; i++ )
   {
      IOMask >>= 1;
      if ( IOMask & 0x0001 )
      {
         port = npACB->IOPorts[i];
         data = npACB->IORegs[i];
         outp( port, data);
         IODelay();
      }
   }
}


/*ͻ
                                    
  SetATAPICmd                       
                                    
  Write an ATAPI CMD and Paramters  
                                    
*/
VOID NEAR SetATAPICmd ( NPACB npACB )
{
   ULONG   cBytes = npACB->npCmdIO->cXferBytesRemain;               /*@V190040*/
   USHORT  cntBytes;                                                /*@V190040*/

   /*-----------------------------------*/
   /* Set Feature register  ( DMA OFF ) */
   /*-----------------------------------*/
   npACB->IORegs[FI_PFEATURE] = DEFAULT_ATAPI_FEATURE_REG & ~DMA_ON;

   /*---------------------------------*/
   /* Enable if DMA set for this Xfer */
   /*---------------------------------*/
   if ( npACB->ISMFlags & ACBIF_BUSMASTERDMAIO )
   {
      npACB->IORegs[FI_PFEATURE] |= DMA_ON;
   }

   /*-----------------------------------------------*/
   /* Set Byte Count registers (High and Low order) */
   /*-----------------------------------------------*/
   if ( cBytes > MAX_XFER_BYTES_PER_INTERRUPT )
   {
      cntBytes = MAX_XFER_BYTES_PER_INTERRUPT;                      /*@V190040*/
   } else {
      cntBytes = cBytes;
   }

   npACB->IORegs[FI_PBYTECTH] = cntBytes >> 8;
   npACB->IORegs[FI_PBYTECTL] = cntBytes & LOW_BYTE_MASK;           /*@V190040*/

   /*---------------------------------------------*/
   /* Set Command register with ATAPI PKT Command */
   /*---------------------------------------------*/
   npACB->IORegs[FI_PCMD] = FX_PKTCMD;

   if ( npACB->npUCB->Capabilities & UCBC_INTERRUPT_DRQ )
   {
      DISABLE
      if (npACB->ISMFlags & ACBIF_INTERRUPT)
      {
         return;
      }

      /*-------------------------*/
      /* Set ISM Flags and State */
      /*-------------------------*/
      npACB->ISMFlags |= (ACBIF_WAITSTATE | ACBIF_INTERRUPT);
      npACB->ISMState  = ACBIS_INTERRUPT_STATE;

      /*Ŀ
       Start the interrupt timer 
      */
      if ( ADD_StartTimerMS((PULONG) &npACB->IRQTimeOutHandle,
                            (ULONG)  npACB->IRQTimeOut,
                            (PFN)    IRQTimeOutHandler,
                            (PVOID)  npACB,
                            (ULONG)  0                      ) )
      {
         return;
      }
      ENABLE

   } else {

      npACB->ISMFlags &= ~ACBIF_WAITSTATE;
      npACB->ISMState  = ACBIS_WRITE_ATAPI_PACKET_STATE;
   }
}

/*ͻ
                                    
  SetATACmd                         
                                    
  Write an ATA CMD and Paramters    
                                    
*/
VOID NEAR SetATACmd ( NPACB npACB )
{
   /*Ŀ
      Set Command Register    
   */
   if( npACB->npUCB->ReqFlags & UCBR_RESET ) {
       npACB->IORegs[FI_PCMD] = FX_SOFTRESET;
   }
   else if( npACB->npUCB->ReqFlags & UCBR_IDENTIFY ) {
       npACB->IORegs[FI_PCMD]  = FX_IDENTIFYDRIVE;
       npACB->npUCB->ReqFlags &= ~UCBR_IDENTIFY;
   }
   else if( npACB->npUCB->ReqFlags & UCBR_CK_PWR_MODE ) {
            npACB->IORegs[FI_PCMD] = FX_CKPWRMODE;
   }
   else if( npACB->npUCB->ReqFlags & UCBR_GETMEDIA_STAT ) {
            npACB->IORegs[FI_PCMD] = FX_GETMEDIASTAT;
    }
   else if( npACB->npUCB->ReqFlags & UCBR_EXEC_DRV_DIAG ) {
            /*Ŀ
             Do not issue EXEC_DRV_DIAG   
             All devices will execute it! 
            */
   }
   else if( npACB->npUCB->ReqFlags & UCBR_NOP ) {
            npACB->IORegs[FI_PCMD] = FX_NOP;
   }
   else if( npACB->npUCB->ReqFlags & UCBR_SET_FEATURES ) {
            npACB->IORegs[FI_PCMD] = FX_SETFTURE;
   }
   else if( npACB->npUCB->ReqFlags & UCBR_STANDBY ) {
            npACB->IORegs[FI_PCMD] = FX_STANDBY;
   } 
   else 
   {
      /*Ŀ
          Unknown ATA command    
      */
      
      if (APMResumeEvent)                       /* If we are at Resume, there could be a situation   */
      {                                         /* when an ATA IDENTIFY command has been issued to a */
         APMResumeEvent++;                      /* FDD device. In this case we need to check it and  */ 
         if (APMResumeEvent < MAX_ID_RETRIES)   /* stop ISM from looping in state 1. Defect 259353   */
         {
            return;
         }
      }
      
      else
      {
         return;
      }
   }

   /*Ŀ
     Check for Reset Request  
   */
   if( ( npACB->npUCB->ReqFlags & UCBR_RESET         ) ||
       ( npACB->npUCB->ReqFlags & UCBR_EXEC_DRV_DIAG )    )
   {
      npACB->npUCB->ReqFlags &= ~UCBR_RESET;
      npACB->ISMState   =  ACBIS_COMPLETE_STATE;
      npACB->ISMFlags  &= ~ACBIF_WAITSTATE;

   } else  {

      /*Ŀ
       Continue with NON-Reset ATA Request  
      */
      DISABLE
      if (npACB->ISMFlags & ACBIF_INTERRUPT)
      {
         return;
      }

      /*Ŀ
       Set ISM Flags and State 
      */
      npACB->ISMFlags |= (ACBIF_INTERRUPT | ACBIF_WAITSTATE);
      npACB->ISMState  =  ACBIS_INTERRUPT_STATE;

      /*Ŀ
       Start the interrupt timer 
      */
      if ( ADD_StartTimerMS((PULONG) &npACB->IRQTimeOutHandle,
                            (ULONG)  npACB->IRQTimeOut,
                            (PFN)    IRQTimeOutHandler,
                            (PVOID)  npACB,
                            (ULONG)  0                      ) )
      {
         return;
      }
      ENABLE
   }
}

/*ͻ
                                      
  InterruptState                      
                                      
  Routes interrupt generated threads  
  to the selected interrupt function  
                                      
*/
VOID NEAR InterruptState( NPACB npACB )
{
   USHORT  Status;
   USHORT  INTReason = 0;                                           /*@V151168*/
   ULONG   cXferBytes;                                              /*@V190040*/
   USHORT  cShortBytes;
   USHORT  ByteBucket;
   USHORT  Port;
   USHORT  OddWord;
   USHORT  cExtraBytes = 0;
   USHORT  Reason = 0;
   USHORT  data, port;

   register INT loop;

   if( !(npACB->TimerFlags & ACBT_INTERRUPT) )
   {
      Status = GetRegister( npACB, FI_PSTATUS );

      while( Status & FX_BUSY ) {
         Status = GetRegister( npACB, FI_PSTATUS );
      }

      INTReason = GetRegister( npACB, FI_PINTREASON );

      npACB->IORegs[FI_PSTATUS]    = Status;
      npACB->IORegs[FI_PINTREASON] = INTReason;

                               /* Ŀ*/
                               /* Interrupt Reason  (Defined in ATAPIREG.H)*/
      if (INTReason & IRR_IO)  /*                                          */
         Reason  |= IRM_IO;    /*  IO    DRQ   COD    #define value        */
                               /*   0     1     1     IR_PKTREADY          */
      if (Status & FX_DRQ)     /*   1     1     1     IR_MESSAGE           */
         Reason |= IRM_DRQ;    /*   1     1     0     IR_XFER_FROM_DEVICE  */
                               /*   0     1     0     IR_XFER_TO_DEVICE    */
      if (INTReason & IRR_COD) /*   1     0     1     IR_COMPLETE          */
         Reason |= IRM_COD;    /* */


      /*Ŀ
           Set ATA bits          
      */
      if( npACB->ISMFlags & ACBIF_ATA_OPERATION )
      {
         Reason = IRM_IO | IRM_DRQ;

         if( npACB->IORegs[FI_PCMD] == FX_GETMEDIASTAT )
         {
            Reason = IR_COMPLETE;
         }
      }

      /* Ŀ
       The 1.7 spec indicates that the completion status only relies on the  
       DRQ bit being 0.  If we are in 1.7B compatibility mode and the DRQ    
       bit is 0, set the reason variable to IR_COMPLETE.                     
      */

      else if (/*( npACB->npUCB->Capabilities & UCBC_SPEC_REV_17B) && */
                                                         !(Status & FX_DRQ))
         Reason |= IRM_IO | IRM_COD;

      /*Ŀ
       If there was an error, go to the complete phase 
      */
      if ( Status & FX_ERROR )
         Reason = IR_COMPLETE;

      switch( Reason )
      {
         case IR_PKTREADY:
            npACB->ISMState  = ACBIS_WRITE_ATAPI_PACKET_STATE;
            npACB->ISMFlags &= ~ACBIF_WAITSTATE;
            break;

         case IR_XFER_FROM_DEVICE:
              if (npACB->npCmdIO->IOSGPtrs.Mode != PORT_TO_SGLIST)
              {
#ifdef DEBUG                                                          /*@V155162*/
                 _asm { int 3 }       /* Expecting to go the opposite direction */
#endif                                                                /*@V155162*/
                 //       WAY, expecting to go the opposite direction,/*@V195083*/
                 // fix it instead of trapping.                       /*@V195083*/
                 npACB->npCmdIO->IOSGPtrs.Mode = PORT_TO_SGLIST;      /*@V195083*/
              } /* endif */

            /*Ŀ
             ATA Xfer from Device      
            */

            if ( npACB->ISMFlags & ACBIF_ATA_OPERATION )
            {
               cXferBytes       = npACB->npCmdIO->cXferBytesRemain;
               npACB->ISMFlags &= ~ACBIF_WAITSTATE;
               npACB->ISMState  = ACBIS_COMPLETE_STATE;

            } else {
               /*Ŀ
                ATAPI Xfer from Device    
               */
               npACB->IORegs[FI_PBYTECTH] = GetRegister( npACB, FI_PBYTECTH );
               npACB->IORegs[FI_PBYTECTL] = GetRegister( npACB, FI_PBYTECTL );

               cXferBytes = MAKEUSHORT( npACB->IORegs[FI_PBYTECTL],
                                        npACB->IORegs[FI_PBYTECTH] );
            }

            if (cXferBytes & 1) {
               cXferBytes++;
            }

            /*Ŀ
             If overrun, determine amount of extra bytes to transfer, and set 
             IOSGPtrs count to get only the number of bytes we are expecting  
             */
            if ( cXferBytes > npACB->npCmdIO->cXferBytesRemain )
            {
               cExtraBytes = cXferBytes - npACB->npCmdIO->cXferBytesRemain;
               cXferBytes  = npACB->npCmdIO->cXferBytesRemain;
            }

            npACB->npCmdIO->IOSGPtrs.numTotalBytes = cXferBytes;

            
            
            
            
            if( cXferBytes )                                        
            {                                                       
               /*Ŀ
                Start the interrupt timer if this is an ATAPI transfer 
               */
               if( !(npACB->ISMFlags & ACBIF_ATA_OPERATION) )
               {
                  DISABLE
                  if ( npACB->ISMFlags & ACBIF_INTERRUPT )
                  {
                     return;
                  }

                  npACB->ISMFlags |= (ACBIF_WAITSTATE | ACBIF_INTERRUPT);

                  if ( ADD_StartTimerMS((PULONG) &npACB->IRQTimeOutHandle,
                                        (ULONG)  npACB->IRQTimeOut,
                                        (PFN)    IRQTimeOutHandler,
                                        (PVOID)  npACB,
                                        (ULONG)  0                      ) )
                  {
                     return;
                  }
               }

               /*Ŀ
                Device will prepare for next interrupt  
                after last byte is transfered to host   
               */

               ENABLE
               ADD_XferIOW( &npACB->npCmdIO->IOSGPtrs );
            }                                                       

            /* Ŀ
             if extra bytes remain, put them in the byte bucket
            */
            while( cExtraBytes )
            {
               inwp( Port, ByteBucket );
               cExtraBytes-=2;
            }

            /* Ŀ
             Adjust counts to show last transfer 
            */
            npACB->npCmdIO->cXferBytesComplete += cXferBytes;
            npACB->npCmdIO->cXferBytesRemain   -= cXferBytes;

            /* Ŀ
             If an odd byte count was requested, and we have finished the  
             even portion of the transfer, get the last byte and put it in 
             the S/G list.                                                 
            */
            if ( !(npACB->npCmdIO->cXferBytesRemain) &&
                                      ( npACB->ISMFlags & ACBIF_ODD_BYTE_XFER) )
            {
               inwp( Port, OddWord );
               PutByteInSGList( npACB, OddWord );

               npACB->npCmdIO->cXferBytesComplete++;
            }
            break;

         case IR_XFER_TO_DEVICE:
            if (npACB->npCmdIO->IOSGPtrs.Mode != SGLIST_TO_PORT)    /*@V155162*/
            {
#ifdef DEBUG                                                        /*@V155162*/
               _asm { int 3 }       /* Expecting to go the opposite direction */
#endif                                                              /*@V155162*/
               //       WAY, expecting to go the opposite direction,/*@V195083*/
               // fix it instead of trapping.                       /*@V195083*/
               npACB->npCmdIO->IOSGPtrs.Mode = SGLIST_TO_PORT;      /*@V195083*/
            } /* endif */                                           /*@V155162*/

            npACB->IORegs[FI_PBYTECTH] = GetRegister( npACB, FI_PBYTECTH );
            npACB->IORegs[FI_PBYTECTL] = GetRegister( npACB, FI_PBYTECTL );

            cXferBytes = MAKEUSHORT( npACB->IORegs[FI_PBYTECTL],
                                     npACB->IORegs[FI_PBYTECTH]);

            /*Ŀ
             If UNDERRUN, set IOSGPtrs count to send only the number of bytes 
             the device is expecting.  Set up cExtraBytes to fill remaining   
             bytes with 0's                                                   
            */
            if ( cXferBytes > npACB->npCmdIO->cXferBytesRemain )
            {
               cExtraBytes = cXferBytes - npACB->npCmdIO->cXferBytesRemain;
               cXferBytes = npACB->npCmdIO->cXferBytesRemain;
            }

            npACB->npCmdIO->IOSGPtrs.numTotalBytes = cXferBytes;

            
            
            
            
            if( cXferBytes )                                        
            {                                                       
               /* Ŀ
                Start the interrupt timer 
               */
               DISABLE
               if ( npACB->ISMFlags & ACBIF_INTERRUPT )
               {
                  return;
               }

               npACB->ISMFlags |= (ACBIF_WAITSTATE | ACBIF_INTERRUPT);

               if ( ADD_StartTimerMS((PULONG) &npACB->IRQTimeOutHandle,
                                     (ULONG)  npACB->IRQTimeOut,
                                     (PFN)    IRQTimeOutHandler,
                                     (PVOID)  npACB,
                                     (ULONG)  0                      ) )
               {
                  return;
               }

               /*Ŀ
                Device will prepare for next interrupt   
                after last byte is transfered from host  
               */
               ENABLE
               ADD_XferIOW( &npACB->npCmdIO->IOSGPtrs );
            }                                                       

            /* Ŀ
             if device expected more, give it zeros            
            */
            Port = npACB->npCmdIO->IOSGPtrs.iPortAddress;

            while( cExtraBytes )
            {
               outwp( Port, 0 );
               cExtraBytes-=2;
            }

            npACB->npCmdIO->cXferBytesComplete += cXferBytes;
            npACB->npCmdIO->cXferBytesRemain   -= cXferBytes;

            break;

         case IR_COMPLETE:

            if( npACB->ISMFlags & ACBIF_BUSMASTERDMAIO )
            {
               PerfBMComplete( npACB );
            }

            /*Ŀ
             if an error, get the error register 
            */
            if (Status & FX_ERROR )
            {
               npACB->IORegs[FI_PERROR]  = GetRegister( npACB, FI_PERROR );

               /*Ŀ
                          Request Sense            
               */
               npACB->OSMReqFlags   |= ACBR_SENSE_DATA;
            }
            else if ( !(npACB->ISMFlags & ACBIF_ATA_OPERATION) &&
                       (npACB->npCmdIO->IOSGPtrs.Mode == PORT_TO_SGLIST))
            {

               /* ATAPI from device */
               /* Ŀ
                if the device did not give us as many bytes as we asked   
                for fill the remaining portion of the SG List with 0's.   
               */
               while ( npACB->npCmdIO->cXferBytesRemain )
               {
                  if (npACB->npCmdIO->IOSGPtrs.SGOffset ==
                     npACB->npCmdIO->IOSGPtrs.pSGList
                              [npACB->npCmdIO->IOSGPtrs.iSGList].XferBufLen)
                  {
                     npACB->npCmdIO->IOSGPtrs.SGOffset = 0;
                     npACB->npCmdIO->IOSGPtrs.iSGList++;
                  }

                  PutByteinSGList( npACB, 0 );
                  npACB->npCmdIO->cXferBytesComplete++;
                  npACB->npCmdIO->cXferBytesRemain--;
               }
            }

            npACB->ISMState  = ACBIS_COMPLETE_STATE;
            break;

         default :

            INT3
            //BMK if all is 0 then there was an unknown problem     /*@V151168*/
            npACB->IORegs[FI_PSTATUS] = FX_ERROR;                   /*@V151168*/
            //Do a reset                                            /*@V151168*/
            npACB->ISMFlags    &= ~ACBIF_WAITSTATE;                 /*@V151168*/
            npACB->OSMReqFlags |= ACBR_RESET;                       /*@V151168*/
            npACB->ISMState     = ACBIS_COMPLETE_STATE;
            break;

      } /* endswitch */

   } else {

      /*Ŀ
       An Interrupt Timed Out 
      */

      if( ( npACB->ISMFlags & ACBIF_BUSMASTERDMAIO ) )
      {
         npACB->ISMFlags &= ~ACBIF_BUSMASTERDMAIO;

         StopBMDMA( npACB );                   /* shut down BM DMA controller */

         port = npACB->BMISTA;
         inp( port, data );

         ClearBMISTA_INT( data, npACB );
      }

      npACB->ISMState     = ACBIS_COMPLETE_STATE;
      npACB->ISMFlags    &= ~ACBIF_WAITSTATE;
      npACB->OSMReqFlags |= ACBR_RESET;

      // Fake interrupt timeout to get an error                     /*@V151168*/
      npACB->TimerFlags |= ACBT_INTERRUPT;                          /*@V151168*/
   }
}

/*ͻ
                                       
  PutByteInSGList                      
                                       
  Puts the next byte from the port     
  address in the next SGList location  
                                       
*/
VOID NEAR PutByteInSGList ( NPACB npACB, USHORT Data )
{
   PSCATGATENTRY pSGE;    /* Current S/G List Entry          */
   ULONG         ppDst;   /* Physical Address of Destination */
   PBYTE         pDst;    /* Virtual Address of Destination  */
   USHORT        ModeFlag;

   /* Point to the current entry */

   pSGE  = &npACB->npCmdIO->IOSGPtrs.pSGList[npACB->npCmdIO->IOSGPtrs.iSGList];
   ppDst = pSGE->ppXferBuf + npACB->npCmdIO->IOSGPtrs.SGOffset;

   if ( DevHelp_PhysToVirt( (ULONG)   ppDst,
                             (USHORT)  1,
                             (PVOID)   &pDst,
                             (PUSHORT) &ModeFlag  ) )
   {
      return;
   }

   *pDst = (UCHAR) (Data >> 8);                           /* Store the Byte   */
}

/*ͻ
                                       
  WriteATAPIPkt                        
                                       
  Write ATAPI command packet to data   
  register                             
                                       
*/
VOID NEAR WriteATAPIPkt ( NPACB npACB )
{
   USHORT Port, i;
   USHORT Data;

   if ( !BSY_CLR_DRQ_SET_WAIT( npACB ) )                             /*V@93531*/
             /* Max Poll wait time exceeded */                       /*V@93531*/
   {                                                                 /*V@93531*/
      npACB->ISMState     = ACBIS_COMPLETE_STATE;                    /*V@93531*/
      npACB->OSMReqFlags |= ACBR_RESET;  /* Set OSM for reset */     /*V@93531*/

   } else {

      /* Ŀ
       Start the interrupt timer 
      */
      DISABLE
      if ( npACB->ISMFlags & ACBIF_INTERRUPT )
      {
         return;
      }

      npACB->ISMFlags |= (ACBIF_WAITSTATE | ACBIF_INTERRUPT);
      npACB->ISMState  = ACBIS_INTERRUPT_STATE;

      if ( ADD_StartTimerMS((PULONG) &npACB->IRQTimeOutHandle,
                            (ULONG)  npACB->IRQTimeOut,
                            (PFN)    IRQTimeOutHandler,
                            (PVOID)  npACB,
                            (ULONG)  0                      ) )
      {
         return;
      }

      /*------------------------------------------------------------*/
      /* Start Bus Master DMA transfer after issuing Packet command */
      /*------------------------------------------------------------*/

      if( npACB->BM_CommandCode & ACBX_BMICOM_START )
      {
         Port = npACB->BMISTA;           /* addr status reg */
         inp( Port, Data );

         ClearBMISTA_INT( Data, npACB );

         Port = npACB->BMICOM;           /* Set port address to BM Command reg */
         Data = npACB->BM_CommandCode;   /* BM controller command */
         outp( Port, Data );             /* Start BM controller */
      }

      /* Ŀ
            Write out the ATAPI Packet      
      */
      Port = npACB->IOPorts[FI_PDATA];

      i=0;
      do
      {                                                   /* MAKEUSHORT (h,l) */
         Data = MAKEUSHORT(npACB->npCmdIO->ATAPIPkt[i],
                           npACB->npCmdIO->ATAPIPkt[i+1]);
         i+=2;
         outwp( Port, Data );
         IODelay ();
      } while ( i < npACB->npUCB->CmdPacketLength);

      ENABLE
   }
}

/*
ͻ
                                    
  IRQTimeOutHandler                 
                                    
  Times out the interrupt           
                                    
ͼ */
VOID FAR IRQTimeOutHandler( ULONG TimerHandle, ULONG Parm1, ULONG Parm2 )
{
   NPACB         npACB;

   DISABLE

   ADD_CancelTimer( TimerHandle );

   npACB = (NPACB) Parm1;

   npACB->IRQTimeOutHandle  = 0;
   npACB->ISMFlags         &= ~ACBIF_INTERRUPT;
   npACB->TimerFlags       |=  ACBT_INTERRUPT;   /* Tell intrt state we timedout */

   if ( npACB->ISMState == ACBIS_INTERRUPT_STATE )
   {
      ENABLE
      StartISM( npACB );

   } else {
      ENABLE
   }
}

/*
ͻ
                                    
  AdapterIRQ0                       
                                    
  Routes received IRQs for Adapter  
  0 to generic handler              
                                    
ͼ */
USHORT FAR _loadds AdapterIRQ0()
{
   return( AdptInterrupt( ACBPtrs[0].npACB ) );                   /*V@93531*/
}

/*
ͻ
                                    
  AdapterIRQ1                       
                                    
  Routes received IRQs for Adapter  
  1 to generic handler              
                                    
ͼ */
USHORT FAR _loadds AdapterIRQ1()
{
   return( AdptInterrupt( ACBPtrs[1].npACB ) );                   /*V@93531*/
}

/*
ͻ
                                    
  AdapterIRQ2                       
                                    
  Routes received IRQs for Adapter  
  2 to generic handler              
                                    
ͼ */
USHORT FAR _loadds AdapterIRQ2()
{
   return( AdptInterrupt( ACBPtrs[2].npACB ) );                   /*V@93531*/
}

/*
ͻ
                                    
  AdapterIRQ3                       
                                    
  Routes received IRQs for Adapter  
  3 to generic handler              
                                    
ͼ */
USHORT FAR _loadds AdapterIRQ3()
{
   return( AdptInterrupt( ACBPtrs[3].npACB ) );                   /*V@93531*/
}

/*
ͻ
                                      
  AdptInterrupt                       
                                      
  Determines if IRQ should be claimed 
  and sets state flags accordingly    
                                      
ͼ */
USHORT NEAR AdptInterrupt( NPACB npACB )
{
   USHORT  Claimed = 0;

   if( npACB )
   {
      DISABLE

      Claimed = 1;

      /*
      Ŀ
       Read the controller status register to clear interrupt  
       in case hardware is setup for level triggered operation 
       */
      GetRegister( npACB, FI_PSTATUS );

      if ( npACB->ISMFlags & ACBIF_INTERRUPT )
      {
         npACB->ISMFlags &= ~ACBIF_INTERRUPT;

         if ( npACB->IRQTimeOutHandle )
         {
            ADD_CancelTimer( npACB->IRQTimeOutHandle );
            npACB->IRQTimeOutHandle = 0;

         } else {
            return( ~Claimed );                                     
         }

         DevHelp_EOI( npACB->IRQLevel );

         ENABLE
         StartISM( npACB );

      } else {

         npACB->SpuriousIRQ++;

         DevHelp_EOI( npACB->IRQLevel );
      }
   }

   return( ~Claimed );
}

/*--------------------------------------------*/
/* CreateBMSGList                             */
/* --------------                             */
/*                                            */
/* Arguments:                                 */
/*                                            */
/*                                            */
/* Actions:                                   */
/*      Takes OS/2 scatter/gather list and    */
/*      builds SFF-8038i compatible list for  */
/*      DMA controller.                       */
/*                                            */
/*                                            */
/* Returns:                                   */
/*      0 if successful                       */
/*                                            */
/*                                            */
/*--------------------------------------------*/
int CreateBMSGList( NPACB npACB )
{
  USHORT          i;
  PPRD            pSGL;
  ULONG           BytesLeft;
  ULONG           PhysicalSgl;
  ULONG           PhysicalAddress;
  ULONG           Length;
  ULONG           LengthLeftInBoundary;
  ULONG           DescriptorCount = 0;
  ULONG           TempLength;
  PSCATGATENTRY   pSGList;

  /* get seg:off address of BM DMA scatter/gather table */

  if ( DevHelp_PhysToVirt( npACB->BMDMA_SGList, npACB->BMDMA_SGListSize,
                           &pSGL, &i ) )
  {
    return ( 1 );                       /* fail it */
  }

  BytesLeft = npACB->npCmdIO->cXferBytesRemain;

                            /* process each entry in OS/2 scatter/gather list */

  pSGList =  npACB->npCmdIO->IOSGPtrs.pSGList;

  for (i=0; i < npACB->npCmdIO->IOSGPtrs.cSGList; i++,pSGList++)
  {
    Length          = pSGList->XferBufLen;     /* get length of memory region */
    PhysicalAddress = pSGList->ppXferBuf;

    if ( Length > BytesLeft )
    {
      #ifdef DEBUG_WDCIDEOS
      {
        _asm int 3;
      }
      #endif
      Length = BytesLeft;               /* Don't set up entry bigger than total */
    }                                   /* transfer length reported */

    if( (PhysicalAddress & 1) ||        /* if on odd byte boundary, do PIO */
        (Length & 1) )                  /* if odd transfer length, do PIO */
    {
      #ifdef ENABLE_COUNTERS
      //      ++npACB->npUCB->DeviceCounters.ByteMisalignedBuffers;
      #endif
      return ( 1 );                     /* fail conversion */
    }

    while( Length )
    {
      pSGL->MR_PhysicalBaseAddress = PhysicalAddress;
      TempLength                   = Length;     /* make copy of total length */

                        /* Can't exceed 64KB size per entry in BM DMA SG List */
      if( TempLength > MR_64K_LIMIT )
      {
        TempLength = MR_64K_LIMIT;      /* force to max size */
        #ifdef DEBUG_WDCIDEOS
        {
          _asm mov ax,1
          _asm int 3;
        }
        #endif
      }

      if ( TempLength > BytesLeft )
      {
        TempLength = BytesLeft;         /* Don't exceed remaining transfer length */
        #ifdef DEBUG_WDCIDEOS
        {
          _asm mov ax,2
          _asm int 3;
        }
        #endif
      }

      /* Can't cross 64KB boundary so check for it and adjust */

      LengthLeftInBoundary = MR_64K_LIMIT - (PhysicalAddress & (MR_64K_LIMIT-1));
      if ( TempLength > LengthLeftInBoundary )
      {
        TempLength = LengthLeftInBoundary;
        #ifdef  ENABLE_COUNTERS
        //        ++npACB->npUCB->DeviceCounters.ByteMisalignedBuffers;
        #endif
      }
                                      /* Adjust counts */
      PhysicalAddress += TempLength;
      Length          -= TempLength;
      BytesLeft       -= TempLength;

                                      /* Create SGL descriptor entry */
      pSGL->ByteCountAndEOT = (ULONG) TempLength;
      ++DescriptorCount;
      if(BytesLeft == 0)
      {
        pSGL->ByteCountAndEOT |= PRD_EOT_FLAG;
      }
      else if ( DescriptorCount > npACB->BMDMA_SGListCount )
      {
        #ifdef DEBUG_WDCIDEOS
        {
          _asm mov ax,4
          _asm int 3;
        }
        #endif
        return( 1 );                    /* ran out of descriptors */
      }
      pSGL++;
    }
  }

  return( 0 );
}

/*
ͻ
                                             
  GetRegister                                
                                             
  Get the error data from the error register 
                                             
ͼ */
USHORT NEAR GetRegister ( NPACB npACB, USHORT Register )
{
   USHORT port;
   USHORT data = 0;

   port = npACB->IOPorts[Register];
   inp ( port, data );

   return ( data );
}

/*-------------------------*/
/*                         */
/* StopBMDMA()             */
/*                         */
/*-------------------------*/
VOID StopBMDMA( NPACB npACB )
{
   USHORT data, port;

   port = npACB->BMICOM;                               /* BM Command Register */

   /*****************************************/
   /* Note: Important to write out correct  */
   /* direction bit per Intel               */
   /*****************************************/
   data  = npACB->BM_CommandCode;                       /* get DIRECTION bit  */

   data &= ~ACBX_BMICOM_START;                          /* turn OFF Start bit */

   outp ( port, data );
}

/*-------------------------*/
/*                         */
/* ClearBMISTA_INT()       */
/*                         */
/*-------------------------*/
VOID ClearBMISTA_INT( USHORT Data, NPACB npACB )
{
   USHORT port;

   port = npACB->BMISTA;                        /* BM Status Register */

   Data &= ( ACBX_BMISTA_INTERRUPT |            /* clear INTR flag if set     */
             ACBX_BMISTA_D0DMA     |            /* DMA Capability bit Drive 0 */
             ACBX_BMISTA_D1DMA     );           /*                    Drive 1 */

   outp ( port, Data );
}

/*-------------------------*/
/*                         */
/* ClearBMISTA_error()     */
/*                         */
/*-------------------------*/
VOID ClearBMISTA_error( USHORT Data, NPACB npACB )
{
   USHORT port;

   port = npACB->BMISTA;                    /* BM Status Register */

   Data &= ( ACBX_BMISTA_ERROR |            /* Turn off Error Bit by writing 1*/
             ACBX_BMISTA_D0DMA |            /* DMA Capability bit Drive 0 */
             ACBX_BMISTA_D1DMA   );         /*                    Drive 1 */

   outp ( port, Data );
}


/*-------------------------*/
/*                         */
/*                         */
/*                         */
/*-------------------------*/
VOID PerfBMComplete( NPACB npACB )
{
   USHORT  port,data;
   USHORT  loop;

   npACB->ISMFlags &= ~ACBIF_BUSMASTERDMAIO;
   npACB->npCmdIO->cXferBytesComplete = npACB->npCmdIO->cXferBytesRemain;
   npACB->npCmdIO->cXferBytesRemain = 0;

   /* Wait for BM DMA active to go away for a while. */

   port = npACB->BMISTA;
   for( loop=256 ; loop ; --loop )
   {
     inp(port,data);
     if ( !(data & ACBX_BMISTA_ACTIVE) )
     {
       break;
     }
   }

   if ( data & ACBX_BMISTA_ACTIVE )
   {
      npACB->ISMState     = ACBIS_COMPLETE_STATE;
      npACB->ISMFlags    &= ~ACBIF_WAITSTATE;
      npACB->OSMReqFlags |= ACBR_RESET;
   }

   StopBMDMA( npACB);               /* shut down BM DMA controller */

   ClearBMISTA_INT( data, npACB );

   /* Fix so that Bus Master errors will definitely result in retries */

   inp ( port, data );
   if ( data & ACBX_BMISTA_ERROR )
   {
     ClearBMISTA_error( data, npACB );

     npACB->ISMState     = ACBIS_COMPLETE_STATE;
     npACB->ISMFlags    &= ~ACBIF_WAITSTATE;
     npACB->OSMReqFlags |= ACBR_RESET;
   }
}

