/***********************************************************************

	WSWIN.C - Windows (Win16 and Win32) main program for
	PC Magazine WSEARCH utility

	Copyright 1996, Ziff-Davis Publishing Company.  All Rights Reserved.
	First Published in PC Magazine, US Edition, March 26, 1996.

	Author:  Tom Rawson.

	(This file is formatted for tabs every 3 columns.)

	This is the Windows 16-bit and 32-bit main program.  The bulk of the
	WSEARCH code is in WSEARCH.C.

***********************************************************************/

#include "wswin.h"

#include <commdlg.h>
#include <ctl3d.h>
#include <direct.h>
#include <dos.h>
#include <stdio.h>
#include <string.h>

#define DEFINE_GLOBALS
#include "wsearch.h"


// Global variables
int gfSearching = FALSE;				// search in process
int gfStopping = FALSE; 				// search stop requested
int gfClosing = FALSE;					// window close requested
int gfFlags = 0;							// search flags
#if defined(__NT__)
int gfWin95 = 0;							// Win95 flag
#endif
HWND ghDlg = NULL;						// dialog window handle
HINSTANCE ghInstance;					// our instance handle
char gszProgramName[] = PROGNAME;	// our program name
char gszCmdLine[MAX_COMMAND];			// passed command line
char gszFiles[MAX_STRING];				// current filespecs
char gszExpression[MAX_STRING];		// current expression
char gszHelpName[MAX_PATHNAME + 1];	// help file name
char gszErrString[MAX_PATHNAME];		// error detail
unsigned int guCurDrive;				// starting drive
char gszCurDir[MAX_PATHNAME + 1];	// starting directory
int gnCharWidth;							// average character width (pixels)
int gnLBWidth;								// list box scrollable width (chars)

// Array to match checkbox IDs to flag bits
CHECKBOX CheckBoxList[] = {
	IDD_SUBDIRS,  SUBDIRS,
	IDD_CASE,     CASE_SENSITIVE,
	IDD_NOMATCH,  SHOW_NOMATCH,
	IDD_ALLLINES, MATCH_ALL,
	IDD_LINENUM,  LINE_NUMBERS,
	IDD_FASTRESP, FAST_RESPONSE
};
#define NUMCHECKBOXES (sizeof(CheckBoxList) / sizeof(CHECKBOX))


// External function prototypes
extern int WSParse(PSZ, PSZ, PSZ, int *, PSZ);
extern int WSearch(PSZ, PSZ, int, ULONG *, PSZ);

// Local function prototypes
int WSYield(BOOL);
int FAR PASCAL MasterDlgProc(HWND, UINT, WPARAM, LPARAM);
int FAR PASCAL BrowseHook(HWND, UINT, WPARAM, LPARAM);
int FAR PASCAL AboutDlgProc(HWND, UINT, WPARAM, LPARAM);
void UpdateComboBox(HWND, int, char *);
void ShowError(int, PSZ);


// ---------------------------------------------------------------------
// WSWIN / WSWIN32 main program 
// ---------------------------------------------------------------------
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpszCmdLine, int nCmdShow)
{
	int rval;								// return value from WSParse
	PSZ pszEndHelpPath;					// end of path in help file name
#if defined(__NT__)
	OSVERSIONINFO OSVersionInfo;		// NT structure for version info
	int fCtl3D = FALSE;					// TRUE if we registered with CTL3D
#endif

	// Save instance handle for later use
	ghInstance = hInstance;

	// Set up the help file name -- use the WSEARCH path, if we can find one
	GetModuleFileName(hInstance, gszHelpName, MAX_PATHNAME);
	if ((pszEndHelpPath = strrchr(gszHelpName, '\\')) == NULL)
		gszHelpName[0] = '\0';
	else
		pszEndHelpPath[1] = '\0';
	strcat(gszHelpName, HELPNAME);

	// Get the command line into local storage and parse it
	gszCmdLine[0] = '\0';
	_fstrncpy((LPSTR)gszCmdLine, lpszCmdLine, MAX_COMMAND);
	gszErrString[0] = '\0';
	if ((rval = WSParse(gszCmdLine, gszFiles, gszExpression, &gfFlags, gszErrString)) > 0) {
		ShowError(rval, gszErrString);
		return rval;
	}

	// Enable 3-D support
	// In WSWIN32 we only enable 3-D support if we are not in Win95
	// (because Win95 has its own 3D support)
#if defined(__WINDOWS__)
	Ctl3dRegister(hInstance);
	Ctl3dAutoSubclass(hInstance);
#elif defined(__NT__)
	OSVersionInfo.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	if (GetVersionEx(&OSVersionInfo)) {
		if (OSVersionInfo.dwPlatformId == VER_PLATFORM_WIN32_NT) {
			Ctl3dRegister(hInstance);
			Ctl3dAutoSubclass(hInstance);
			fCtl3D = TRUE;
		}
		else if (OSVersionInfo.dwPlatformId == 1)
			gfWin95 = 1;
	}
#endif

	// Create the dialog
	if ((ghDlg = CreateDialog(hInstance, MAKEINTRESOURCE(WSEARCHDIALOG), 0,
			MakeProcInstance(MasterDlgProc, hInstance))) == (HWND)NULL) {
		// If we can't create the dialog, complain
		MessageBox(0, ERRMSG_DIALOG_LOAD, NULL, MB_ICONEXCLAMATION | MB_OK);
		return 1;
	}

	// Make a minimized window display the proper icon (we have to do this
	// because the window is a dialog, and the default dialog class does not
	// have an associated icon)
#if defined(__WINDOWS__)
	SetClassWord(ghDlg, GCW_HICON, (WORD)LoadIcon(hInstance, MAKEINTRESOURCE(WSEARCHICON)));
#else
	SetClassLong(ghDlg, GCL_HICON, (WORD)LoadIcon(hInstance, MAKEINTRESOURCE(WSEARCHICON)));
#endif
 
	// Display the dialog window as requested
	ShowWindow (ghDlg, nCmdShow);

	// Run the message loop
	rval = WSYield(TRUE);

	// Remove link to CTL3D 
#if defined(__NT__)
	if (fCtl3D)
#endif
		Ctl3dUnregister(hInstance);

	return rval;
}


// ---------------------------------------------------------------------
// Message loop and yield routine
// Call with TRUE to loop until done, FALSE to clean out our messages
// once and return
// ---------------------------------------------------------------------
int WSYield(BOOL fWait)
{
	MSG msg;

	for ( ; ; ) {

		// Clean out the message queue
		while (PeekMessage(&msg, NULL, 0, 0, PM_REMOVE )) {
			if ((ghDlg == NULL) || !IsDialogMessage(ghDlg, &msg)) {
				TranslateMessage(&msg);
				DispatchMessage(&msg);
			}
		}

		// If not waiting, return search flag as result (this lets
		// the search engine know if it is time to stop)
		if (!fWait)
			return gfSearching;

		// If waiting and we got a WM_QUIT, return the result
		if (msg.message == WM_QUIT)
			return msg.wParam;

		// Wait for the next message and do it all again
		WaitMessage();
	}
}


// ---------------------------------------------------------------------
// Dialog procedure to manage the WSEARCH dialog
// ---------------------------------------------------------------------
int FAR PASCAL MasterDlgProc(HWND hDlg, UINT uMessage, WPARAM wParam, LPARAM lParam)
{

	OPENFILENAME OpenFile;					// data structure for common dialog
	TEXTMETRIC TextMetrics;					// text metrics for listbox
	HDC hDC;										// display context so we can get text metrics
	char szFNBuffer[MAX_PATHNAME + 1];	// filename buffer for browse
	char szMsgBuffer[32];					// short message buffer for file count
	ULONG ulFiles;								// file count
	int rval;									// return value from WSearch
	int i;										// array index
 
	switch (uMessage) {
 
		case WM_INITDIALOG:

			// Get text metrics for use in list box
			hDC = GetDC(GetDlgItem(hDlg, IDD_RESULTS));
			GetTextMetrics(hDC, (LPTEXTMETRIC)&TextMetrics);
			gnCharWidth = TextMetrics.tmAveCharWidth;
			ReleaseDC(hDlg, hDC);

			// Get current directory, clear searching data
			_dos_getdrive(&guCurDrive);
			getcwd(gszCurDir, MAX_PATHNAME + 1);
			SetDlgItemText(hDlg, IDD_CURDIR, (LPSTR)gszCurDir);
			SetDlgItemText(hDlg, IDD_SEARCHING, NULLSTR);

			// Set field widths
			SendDlgItemMessage(hDlg, IDD_FILENAME, EM_LIMITTEXT, MAX_STRING, 0L);
			SendDlgItemMessage(hDlg, IDD_EXPRESSION, EM_LIMITTEXT, MAX_STRING, 0L);

			// Set filename and text controls to reflect data from command line
			SetDlgItemText(hDlg, IDD_FILENAME, (LPSTR)gszFiles);
			SetDlgItemText(hDlg, IDD_EXPRESSION, (LPSTR)gszExpression);

			// Set checkboxes based on command line options
			for (i = 0; i < NUMCHECKBOXES; i++)
				CheckDlgButton(hDlg, CheckBoxList[i].nBoxID, (gfFlags & CheckBoxList[i].nFlagBit));

			// Start the search if requested on the command line
			if (gfFlags & START_SEARCH)
				PostMessage(hDlg, WM_COMMAND, IDD_SEARCH, 0L);

			return 1;

		case WM_COMMAND :
			switch (wParam) {

				case IDD_SEARCH:

					// If currently stopping, ignore extra hits on the stop button
					if (gfStopping)
						return 1;

					// See if we need to search, or stop
					if (!gfSearching) {

						// Search requested, retrieve the dialog box info
						GetDlgItemText(hDlg, IDD_FILENAME, (LPSTR)gszFiles, MAX_STRING);
						UpdateComboBox(hDlg, IDD_FILENAME, gszFiles);
						GetDlgItemText(hDlg, IDD_EXPRESSION, (LPSTR)gszExpression, MAX_STRING);
						UpdateComboBox(hDlg, IDD_EXPRESSION, gszExpression);

						// Set the search flags based on dialog checkbox settings
 						for (i = 0; i < NUMCHECKBOXES; i++) {
							if (IsDlgButtonChecked(hDlg, CheckBoxList[i].nBoxID))
								gfFlags |= CheckBoxList[i].nFlagBit;
							else
								gfFlags &= ~CheckBoxList[i].nFlagBit;
						}

						// Change the "Search" button to a "Stop" button, disable
						// other buttons
						SetDlgItemText(hDlg, IDD_SEARCH, (LPSTR)STOP_LABEL);
						EnableWindow(GetDlgItem(hDlg, IDD_EXIT), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDD_HELP), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDD_ABOUT), FALSE);
						EnableWindow(GetDlgItem(hDlg, IDD_BROWSE), FALSE);

						// Clear the results box, set its width to zero, and
						// give it the focus
						SendDlgItemMessage(hDlg, IDD_RESULTS, LB_RESETCONTENT, 0, 0L);
						SendDlgItemMessage(ghDlg, IDD_RESULTS, LB_SETHORIZONTALEXTENT, 0, 0L);
						gnLBWidth = 0;
						SetFocus(GetDlgItem(hDlg, IDD_RESULTS));
						WSYield(FALSE);

						// Start the search engine; holler if any error occurs
						// during the search
						gfSearching = TRUE;
						gfStopping = FALSE;
						gszErrString[0] = '\0';
						if ((rval = WSearch(gszFiles, gszExpression, gfFlags, &ulFiles, gszErrString)) > 0)
							ShowError(rval, gszErrString);

						// Done searching, clean up the display and buttons
						gfSearching = FALSE;
						gfStopping = FALSE;
						EnableWindow(GetDlgItem(hDlg, IDD_EXIT), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDD_HELP), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDD_ABOUT), TRUE);
						EnableWindow(GetDlgItem(hDlg, IDD_BROWSE), TRUE);
						SetDlgItemText(hDlg, IDD_SEARCH, (LPSTR)SEARCH_LABEL);

						// Display Search termination / completion, plus "1 file
						// found" or "n files found"
						sprintf(szMsgBuffer, "%s  (%lu file%s found)", ((rval == TERMINATE) ? SEARCH_TERMINATED : SEARCH_COMPLETED), ulFiles, ((ulFiles == 1) ? NULLSTR : "s"));
						SetDlgItemText(hDlg, IDD_SEARCHING, (LPSTR)szMsgBuffer);

						// If a close was requested, repost the close message now
						// that the search is done
						if (gfClosing) {
							gfClosing = FALSE;
							PostMessage(hDlg, WM_CLOSE, 0, 0L);
						}

					} else {
						// A search is in progress, and the stop button was hit
						// Clearing gfSearching will stop the search engine the
						// next time it yields
						gfSearching = FALSE;
						gfStopping = TRUE;
					}

					return 1;

				case IDD_HELP:
					// Display the help
					WinHelp(hDlg, gszHelpName, HELP_INDEX, 0L);
					break;

				case IDD_ABOUT:
					// DIsplay the About box
					DialogBox(ghInstance, MAKEINTRESOURCE(ABOUTDIALOG), hDlg, MakeProcInstance(AboutDlgProc, ghInstance));
					return 1;

				case IDD_EXIT:
				case IDCANCEL:
					// Ignore these buttons if a search is in process, otherwise
					// close the dialog
					if (!gfSearching)
						PostMessage(hDlg, WM_CLOSE, 0, 0L);
					return 1;

				case IDD_BROWSE:
					// We use the Windows "File Open" common dialog for browsing
					// We have to treat the results a little differently than
					// usual, so we can retrieve the directory name even if no
					// file is selected

					// Initialize the open file structure
					memset(&OpenFile, '\0', sizeof(OPENFILENAME));
					OpenFile.lStructSize = sizeof(OPENFILENAME);
					OpenFile.hwndOwner = hDlg;
					OpenFile.lpstrFilter = OFN_FILTER;
					OpenFile.lpstrFile = szFNBuffer;
					strcpy(szFNBuffer, szWildName);
					OpenFile.nMaxFile = MAX_PATHNAME + 1;
					OpenFile.lpstrTitle = OFN_TITLE;
					OpenFile.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_NOVALIDATE;

					// Display the common dialog to get a filename
					if (GetOpenFileName(&OpenFile)) {

						// Local variables used in this block only
						char szBrowseDir[MAX_PATHNAME + 1];	// browse directory
						int nBrowseLen;			// browse directory path length
						int nCurLen;				// current directory path length
						unsigned int uDummy;		// dummy var for _dos_setdrive

						// Get the current directory in case the dialog changed it
						getcwd(szBrowseDir, MAX_PATHNAME + 1);

#if defined(__NT__)
						// In 32-bit Win95 we don't need to insert the path in
						// front of the name, because Windows does it for us
						// and leaves the full path name in szFNBuffer.  However
						// we still need to make the path relative if possible --
						// whether or not the directory was changed.

						if (gfWin95) {

							// Get string lengths
							nCurLen = strlen(gszCurDir);
							nBrowseLen = strlen(szFNBuffer);
	
							if ((nCurLen < nBrowseLen) && (strnicmp(gszCurDir, szFNBuffer, nCurLen) == 0)) {
								// Eliminate any starting backslash in relative path
								if (szFNBuffer[nCurLen] == '\\')
									nCurLen++;
								memmove(szFNBuffer, &szFNBuffer[nCurLen], (nBrowseLen - nCurLen + 1));
							}
	
							// Switch back to original directory
							_dos_setdrive(guCurDrive, &uDummy);
							chdir(gszCurDir);

						// If not Win95, drop through to code below
						} else {
#endif
						// In Windows and Windows NT we have to figure out the
						// path chosen in the browse dialog by looking at the
						// current directory, and insert it in front of the
						// name.  If the current directory didn't change, then
						// we can skip all this and just use the buffer that came
						// back from the dialog.

						// Check if current directory changed
						if (stricmp(szBrowseDir, gszCurDir) != 0) {

							// Get string lengths
							nBrowseLen = strlen(szBrowseDir);
							nCurLen = strlen(gszCurDir);

							// Convert new directory to a relative path if possible
							if ((nCurLen < nBrowseLen) && (strnicmp(gszCurDir, szBrowseDir, nCurLen) == 0)) {
								// Eliminate any starting backslash in relative path
								if (szBrowseDir[nCurLen] == '\\')
									nCurLen++;
								memmove(szBrowseDir, &szBrowseDir[nCurLen], (nBrowseLen - nCurLen + 1));
								nBrowseLen -= nCurLen;
							}

							// Make sure it ends with a backslash
							if (szBrowseDir[nBrowseLen - 1] != '\\') {
								szBrowseDir[nBrowseLen++] = '\\';
								szBrowseDir[nBrowseLen] = '\0';
							}

							// Insert path in front of returned filename
							memmove(&szFNBuffer[nBrowseLen], szFNBuffer, strlen(szFNBuffer) + 1);
							memcpy(szFNBuffer, szBrowseDir, nBrowseLen);

							// Switch back to original directory
							_dos_setdrive(guCurDrive, &uDummy);
							chdir(gszCurDir);
						}
#if defined(__NT__)
						}
#endif

						// In Windows and Windows NT we have to figure out the
						// path chosen in the browse dialog by looking at the
						// current directory, and insert it in front of the
						// name.  In Windows 95 the dialog does this for us, and
						// all we have to do is make the path relative.


						// Insert the new name in the File(s) box, with a space
						// before it if necessary
						GetDlgItemText(hDlg, IDD_FILENAME, (LPSTR)gszFiles, MAX_STRING);
						if ((strlen(gszFiles) > 0) && (gszFiles[strlen(gszFiles) - 1] != ' '))
							strcat(gszFiles, " ");
						strcat(gszFiles, szFNBuffer);
						SetDlgItemText(hDlg, IDD_FILENAME, (LPSTR)gszFiles);

						// Set the focus to the File(s) box, with the caret at the end
						SetFocus(GetDlgItem(hDlg, IDD_FILENAME));
						SendDlgItemMessage(hDlg, IDD_FILENAME, CB_SETEDITSEL, 0, MAKELPARAM(strlen(gszFiles), strlen(gszFiles)));
					}

					return 1;
			}
			break;

		case WM_CLOSE:
			// If we are waiting for the search to stop, ignore the message
			if (gfClosing)
				return 1;

			// If a search is in process see if the user really wants to exit;
			// if not, ignore the message
			if (gfSearching && (MessageBox(hDlg, (LPSTR)CONFIRM_CLOSE, (LPSTR)gszProgramName, MB_OKCANCEL) != IDOK))
				return 1;

			// If a search is in process and the user really wants to exit, set
			// the flags to stop it, set the close flag, and ignore the message
			// (it will be reposted once the search is finished)
			// This also sets the flag if a close occurs during the stopping
			// process
			if (gfSearching || gfStopping) {
				gfSearching = FALSE;
				gfStopping = TRUE;
				gfClosing = TRUE;
				return 1;
			}

			// If we are not waiting for the search to stop, we are done
			// Close up any open help window, and destroy the dialog
			WinHelp(hDlg, gszHelpName, HELP_QUIT, 0L);
			DestroyWindow(hDlg);
			break;

		case WM_DESTROY :
			PostQuitMessage(0);
			break;
	}

	return 0;
}


// ---------------------------------------------------------------------
// About dialog procedure
// ---------------------------------------------------------------------
int FAR PASCAL AboutDlgProc(HWND hDlg, UINT uMessage, WPARAM wParam, LPARAM lParam)
{

	switch (uMessage) {
 
		case WM_INITDIALOG:
			// Insert the name with version number into the dialog
			SetDlgItemText(hDlg, IDD_VERSION, gszProgramName);
			return 1;

		case WM_COMMAND:
			// Any command closes the dialog
			EndDialog(hDlg, 0);
			return 1;

	}

	return 0;
}


// ---------------------------------------------------------------------
// Add any new unique string to one of the combo box lists
// ---------------------------------------------------------------------
void UpdateComboBox(HWND hDlg, int nControl, char *pszNewItem)
{
	int nItems;								// current item count
	int i;									// array index
	char szCBString[MAX_STRING];		// text of one item

	// Get the current item count
	nItems = (int)SendDlgItemMessage(hDlg, nControl, CB_GETCOUNT, 0, 0L);

	// See if the string is already in the combo box, and return if it is
	for (i = 0; i < nItems; i++) {
		szCBString[0] = '\0';
		SendDlgItemMessage(hDlg, nControl, CB_GETLBTEXT, i, (LPARAM)(LPSTR)szCBString);
		if (strcmp(szCBString, pszNewItem) == 0)
			return;
	}

	// It's not there, so add it as a new string
	SendDlgItemMessage(hDlg, nControl, CB_INSERTSTRING, 0, (LPARAM)(LPSTR)pszNewItem);
}


// ---------------------------------------------------------------------
// Display search results
// Type 0 displays a result in the list box, type 1 displays status in
// the "searching" control
// ---------------------------------------------------------------------
int WSDispResult(int nDispType, char *pszResultString)
{
	int rval;								// return value from listbox messages
	int nNewWidth;							// length of new string

	switch(nDispType) {

		// Display match results
		case 0:

			// Add the string to the list box, complain if it won't fit
			rval = SendDlgItemMessage(ghDlg, IDD_RESULTS, LB_INSERTSTRING, -1, (LPARAM)(LPSTR)pszResultString);
			if ((rval == LB_ERR) || (rval == LB_ERRSPACE))
				return ERR_RESULT_SPACE;

			// If this string is longer than the previous largest string,
			// adjust the listbox width (this enables the horizontal scroll
			// bar for strings that need it)
			if ((nNewWidth = strlen(pszResultString)) > gnLBWidth) {
				SendDlgItemMessage(ghDlg, IDD_RESULTS, LB_SETHORIZONTALEXTENT, nNewWidth * gnCharWidth, 0L);
				gnLBWidth = nNewWidth;
			}
			break;

		// Display search status
		case 1:
			SetDlgItemText(ghDlg, IDD_SEARCHING, (LPSTR)pszResultString);
			break;

	}
	return 0;
}


// ---------------------------------------------------------------------
// Display a message box with an error message
// ---------------------------------------------------------------------
void ShowError(int rval, PSZ pszErrString)
{
	char szWork[MAX_PATHNAME + 64];		// work area for errors

	// COpy the error text
   strcpy(szWork, ErrorList[rval - 1]);

	// If a string was supplied, include it with quotes
	if (pszErrString[0])
		sprintf(&szWork[strlen(szWork)], " \"%s\"", pszErrString);

	// Display the error
	MessageBox(NULL, szWork, gszProgramName, MB_OK);
}

