/***********************************************************************

	WSEARCH.C - Code for PC Magazine WSEARCH utility

	Copyright 1996, Ziff-Davis Publishing Company.  All Rights Reserved.
	First Published in PC Magazine, US Edition, March 26, 1996.

	Author:  Tom Rawson.

	(This file is formatted for tabs every 3 columns.)

	This is the bulk of the WSEARCH code.  The main programs for Windows
	and OS/2 are in WSWIN.C and WSOS2.C respectively.

***********************************************************************/

#if defined(__WINDOWS__) || defined(__NT__)
#include "wswin.h"
#endif

#include <ctype.h>
#include <direct.h>
#include <dos.h>
#include <errno.h>
#include <fcntl.h>
#include <sys\stat.h>
#include <sys\types.h>
#include <io.h>
#include <malloc.h>
#include <process.h>
#include <share.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#if defined(__WINDOWS__)
#include <dos.h>
#endif

#include "wsearch.h"


// State machine and compiler macros
#define State(s) case ST_##s
#define NextState(s) nState = ST_##s
#define ChooseNextState(c, t, f) nState = (c ? ST_##t : ST_##f)
#define NotState(s) (nState != ST_##s)
#define Type(e) ET_##e
#define Element(e) case Type(e)


// ---------------------------------------------------------------------
// Global variables and prototypes
// ---------------------------------------------------------------------

// Arrays to hold pattern elements and file specifications
ELEMENT Pattern[MAX_EXPR_ELEMENTS];
FILESPEC FileSpecList[MAX_FILESPECS];

// Meta-character lists
char StdMetaList[] = MC_STDMETA;
char ClassMetaList[] = MC_CLASSMETA;

// External function prototypes
extern int WSYield(BOOL);						// Windows yield
extern int WSDispResult(int, char *);		// display a result string

// Local function prototypes
int ParseExpr(PCH *, PELEMENT, int *, int, int);
int ParseFileSpec(PCH *, PFILESPEC);
int FindFile(PSZ, UINT, PFINDDATA, PSZ, PSZ);
int SearchFileSpec(PFILESPEC, PELEMENT, int, int, ULONG *, PSZ);
int Match(PCH, PELEMENT, int, int, ULONG *, PSZ);
int ReadNextBuf(PCH, int, PCH, PSEARCHSTATUS, PSZ);
PCH Scan(PCH, PCH, PCH, PCH, PELEMENT, PSTRMATCH, int);
PCH CharScan(PCH, int, UCHAR, int);
PCH TruncateFilename(PCH, PCH, UINT, UINT, PCH, UINT);
void MakeFileName(PCH, PCH);


// ---------------------------------------------------------------------
// WSEARCH command line parser
// ---------------------------------------------------------------------
int WSParse(PSZ pszCmdLine, PSZ pszFiles, PSZ pszExpression, int *pfFlags, PSZ pszErrString)
{
	UCHAR cTerm;						// expression or filespec terminator
	int nChars;							// number of characters in expression / filespec
	int fExpression = FALSE;		// TRUE if we parsed an expression
	int fNextArgExpr = TRUE;		// TRUE if we expect an expression next
	int nFileSpecs = 0;				// filespec counter
	PSZ pszStart;						// start of expression or filespec

	// Parse passed command line
	while (*pszCmdLine != '\0') {

		// Skip whitespace
		while (isspace(*pszCmdLine) || (*pszCmdLine == ';'))
			pszCmdLine++;

		// Stop if we hit end of line after whitespace
		if (*pszCmdLine == '\0')
			break;

		// Handle switches
		if ((*pszCmdLine == SWITCH1) || (*pszCmdLine == SWITCH2)) {

			// Loop through all switches after the switch character
			while ((!isspace(*(++pszCmdLine))) && (*pszCmdLine != '\0')) {

				switch(toupper(*pszCmdLine)) {
	
#if defined(__OS2__)
					// Show command line usage in OS/2
					case SWITCH_USAGE:
						printf(USAGE);
						return -1;
#endif
					// Show all lines
					case SWITCH_ALL:
						*pfFlags |= MATCH_ALL;
						break;
	
					// Case sensitive search
					case SWITCH_CASE:
						*pfFlags |= CASE_SENSITIVE;
						break;
	
					// Filespec follows
					case SWITCH_FILESPEC:
						fNextArgExpr = FALSE;
						break;
	
					// Start the search
					case SWITCH_GO:
						*pfFlags |= START_SEARCH;
						break;
	
					// Matches only
					case SWITCH_NOMATCH:
						*pfFlags |= SHOW_NOMATCH;
						break;
	
					// Line numbers
					case SWITCH_LINENUM:
						*pfFlags |= LINE_NUMBERS;
						break;
	
#if defined(__WINDOWS__) || defined(__NT__)
					// Fast response
					case SWITCH_FASTRESP:
						*pfFlags |= FAST_RESPONSE;
						break;
#endif
	
					// Search subdirectories
					case SWITCH_SUBDIRS:
						*pfFlags |= SUBDIRS;
						break;
	
					// Text pattern follows
					case SWITCH_EXPRESSION:
						fNextArgExpr = TRUE;
						break;
	
					// bad switch
					default:
						pszCmdLine[1] = '\0';
						strcpy(pszErrString, pszCmdLine - 1);
						return ERR_BADSWITCH;
				}
			}
		}

		// Not whitespace or a switch, must be the expression or a filespec
		else {

			if (fNextArgExpr) {

				// Expecting an expression, holler if there are too many
				if (fExpression)
					return ERR_TWOEXPRESSIONS;

				// Set the termination character based on whether the expression
				// is in open text, or double-quoted
				pszStart = pszCmdLine;
				if (*pszCmdLine == '"') {
					cTerm = '"';
					pszCmdLine++;
					pszStart++;
				} else
					cTerm = ' ';

				// Scan the command line for the termination character
				for ( ; *pszCmdLine != '\0'; pszCmdLine++) {
					// If this is an escape, throw away an extra character
					// (this will skip any escaped termination character)
					if (*pszCmdLine == ESCAPE)
						pszCmdLine++;
					else if (*pszCmdLine == cTerm)
						break;
				}

				// Copy the expression to the supplied buffer and terminate it
				if ((nChars = (pszCmdLine - pszStart)) > 0)
					memcpy(pszExpression, pszStart, nChars);
				pszExpression[nChars] = '\0';

				fExpression = TRUE;
				fNextArgExpr = FALSE;

			} else {

				// Expecting a filespec, holler if there are too many
				if (nFileSpecs >= MAX_FILESPECS)
					return ERR_TOOMANYFILES;
				
				// Figure out the terminating character for the filespec
				pszStart = pszCmdLine;
				if (*pszCmdLine == '"') {
					cTerm = '"';
					pszCmdLine++;
				} else
					cTerm = ' ';

				// Scan the command line for the termination character
				for ( ; *pszCmdLine != '\0'; pszCmdLine++) {
					// If this is an escape, throw away an extra character
					// (this will skip any escaped termination character)
					if (*pszCmdLine == ESCAPE)
						pszCmdLine++;
					else if (*pszCmdLine == cTerm)
						break;
				}

				// If we started with a quote be sure to skip the ending quote
				if (cTerm == '"')
					pszCmdLine++;

				// Append the filespec to the supplied buffer and terminate it
				// Ignore empty filespecs, including empty quoted strings
				if (((nChars = (pszCmdLine - pszStart)) > 0) && ((nChars > 2) || (cTerm != '"'))) {
					if (nFileSpecs > 0)
						*pszFiles++ = ' ';
					memcpy(pszFiles, pszStart, nChars);
					pszFiles += nChars;
				}

				nFileSpecs++;
			}
		}
	}

	// Terminate filespec list and return
	*pszFiles = '\0';
	return 0;
}


// ---------------------------------------------------------------------
// Main WSEARCH program -- parse the expression and filenames,
// and do the search
// ---------------------------------------------------------------------
//
// On entry:
//
//		pszFiles					Passed filespec list, or empty string
//		pszExpression			Passed expression, or empty string
//		fFlags					Flags, see WSEARCH.H for bit assignments
//
// Returns:
//		NO_MORE_FILES (-1)	All files processed
//		TERMINATE (-2)			Stopped due to termination request
//		Error code (> 0)		Search failed, code is an index into ErrorList
//

int WSearch(PSZ pszFiles, PSZ pszExpression, int fFlags, ULONG *pulFiles, PSZ pszErrString)
{
	char szFileName[MAX_PATHNAME];	// create file name here if none specified
	PFILESPEC pFileSpec;				// pointer to current search file spec
	int nFileSpecs = 0;				// number of file specifications
	int nElements = 0;				// number of elements in expression
	int nType;							// element type when freeing file specs
	int rval = 0;						// return value;

	// Parse passed expression
	if (*pszExpression != '\0') {
		if ((rval = ParseExpr(&pszExpression, Pattern, &nElements, MAX_EXPR_ELEMENTS, fFlags)) != 0)
			return rval;
	}

	// If no file list passed, create one for all files in the current
	// directory
	if (*pszFiles == '\0')
		pszFiles = strcpy(szFileName, szWildName);

	// Parse filespec list
	while (*pszFiles != '\0') {

		// Skip whitespace
		while (isspace(*pszFiles) || (*pszFiles == ';'))
			pszFiles++;

		// Stop if we hit end of line after whitespace
		if (*pszFiles == '\0')
			break;

		// Check for too many filespecs
		if (nFileSpecs >= MAX_FILESPECS) {
			rval = ERR_TOOMANYFILES;
			goto SearchExit;
		}

		// Parse filespec
		if ((rval = ParseFileSpec(&pszFiles, &FileSpecList[nFileSpecs])) != 0)
			goto SearchExit;
		nFileSpecs++;
	}

	// Search all the filespecs, but stop the search if we hit an error other
	// than "file not found"
	for (pFileSpec = FileSpecList, *pulFiles = 0L; pFileSpec < &FileSpecList[nFileSpecs]; pFileSpec++)
		if (((rval = SearchFileSpec(pFileSpec, Pattern, nElements, fFlags, pulFiles, pszErrString)) != FILE_FOUND) && (rval != FILE_NOT_FOUND))
			break;

SearchExit:

	// Free the pattern string / class memory
	while ((--nElements) > 0)
		if ((Pattern[nElements].nType == ET_CLASS) || (Pattern[nElements].nType == ET_STRING))
			free(Pattern[nElements].text.pString);

	// Free all memory associated with the filespecs
	while ((--nFileSpecs) >= 0) {

		// If this filespec was a regular expression, free the memory for
		// the regular expression strings, then the memory for the regular
		// expression itself
		if ((nElements = FileSpecList[nFileSpecs].nFileElements) != 0) {
			while ((--nElements) > 0) {
				nType = FileSpecList[nFileSpecs].pFileExpr[nElements].nType;
				if ((nType == ET_CLASS) || (nType == ET_STRING))
					free(FileSpecList[nFileSpecs].pFileExpr[nElements].text.pString);
			}
			free(FileSpecList[nFileSpecs].pFileExpr);
		}

		// Free the memory for the file name
		free(FileSpecList[nFileSpecs].pFileName);
	}

	// Search is done!
	return rval;
}


// ---------------------------------------------------------------------
// Parse a regular expression
// ---------------------------------------------------------------------
int ParseExpr(PSZ *ppszLine, PELEMENT pPattern, int *nElements, int nMaxElements, int fFlags)
{
	PSZ pszLine = *ppszLine;		// current scan point
	PSZ pszStart = pszLine;			// original start point
	UCHAR cCurrent;					// current working character from expression
	UCHAR cClassToggleValue;		// value stored in class array when char is toggled
	int nState;							// current state
	int fUseCurrent;					// TRUE to reuse current character
	int fLiteral;						// TRUE if current character is an escaped literal
	int fStdMetaChar;					// TRUE if current character is a standard meta-character
	int fClassMetaChar;				// TRUE if current character is a class meta-character
	int fTermChar;						// TRUE if current character is the termination character
	int fString;						// TRUE if parser is building a string
	int fClass;							// TRUE if parser is within a class definition
	int fRange;							// TRUE if parser is within a range definition
	int fExprScan;						// TRUE if scan (*) character found
	int fCase = (fFlags & CASE_SENSITIVE);	// TRUE for case-sensitive search
	int nClassCount;					// number of characters in class
	UINT uCurrent;						// integer value returned by sscanf
	UINT uLength;						// number of bytes scanned by sscanf
	UINT uRangeChar;					// current character in range
	UINT uRangeStart;					// first character in range
	UINT uRangeEnd;					// last character in range
	UINT uTemp;							// temporary storage for range code
	char StrWork[MAX_STRING];		// string work area

	pszStart = pszLine;

	// State machine to parse the regular expression pattern (see the
	// state diagram in PARSER.BMP for a picture of how this machine
	// works)
	for (NextState(COMPINIT); NotState(COMPEXIT); ) {

		switch(nState) {

			// ------------------------------------------------------------
			// Initialize expression
			State(COMPINIT):

				// Error if expression is empty
				if (*pszLine == '\0')
					return ERR_NOEXPRESSION;

				// Initialize scan variables
				// We decrement pszLine because NextChar will increment it
				// We set fExprScan so that there is automatically a scan for the
				//   first element (unless this is a string pattern)
				*nElements = 0;
				fClass = fString = fUseCurrent = FALSE;
				fExprScan = !(fFlags & MATCH_STRING);
				pszLine--;
				NextState(NEXTCHAR);
				break;


			// ------------------------------------------------------------
			// Move to next character
			//
			// When this state is done we have the following setup:
			//
			//		cCurrent 			current character
			//		pszLine	 			address of last character retrieved from line
			//		fLiteral 			TRUE if cCurrent is to be taken literally,
			//								not as a meta-character
			//		fStdMetaChar	 	TRUE if cCurrent is a standard (non-class)
			//								meta-character or end of line character
			//		fClassMetaChar		TRUE if cCurrent is a class meta-character
			//		fTermChar			TRUE if cCurrent is the terminating character
			//								for the expression
			//
			// Before returning to this state, other code should set up
			// variables as follows:
			//
			//		fUseCurrent 		TRUE to reuse cCurrent, FALSE to get a new
			//								character from pszLine + 1
			//		cCurrent				character to reprocess if fUseCurrent is TRUE
			//		fString				TRUE if currently accumulating a string
			//		fClass				TRUE if currently within a class
			//
			// Note that this allows backup of one character only, multi-character
			// backup during parsing is not supported (or needed)

			State(NEXTCHAR):

				// Disable previous literal character
				fLiteral = FALSE;

				// Get next character, or reuse the old one
				if (!fUseCurrent)
					cCurrent = *(++pszLine);

				// If character is escaped, handle that
				if (cCurrent == ESCAPE) {

					// Get character after escape char
					cCurrent = *(++pszLine);
	
					switch(cCurrent) {
		
						// Tab
						case ESC_TAB:
							cCurrent = TAB;
							break;
			
						// Newline -- always considered a meta-character
						case ESC_EOL:
							cCurrent = EOL;
							break;
			
						// Carriage return
						case ESC_CR:
							cCurrent = CR;
							break;
			
						// Hex number
						case ESC_HEX:
							if (sscanf(pszLine, "%x%n", &uCurrent, &uLength) != 1)
								return ERR_INVALID_HEX;
							cCurrent = (UCHAR)uCurrent;
							// Point to last character of number
							pszLine += (uLength - 1);
							break;
			
						// If there's nothing after the escape it's an error
						case '\0':
							return ERR_EXPR_INCOMPLETE;
		
						// If it's anything else, look for a digit and convert any numeric
						// ASCII value that's present.  All other characters, including
						// escaped meta-characters, quotes, and the escape character
						// itself, use the character as a literal value.
						default:
							if (isdigit(cCurrent)) {
								if (sscanf(pszLine, "%u%n", &uCurrent, &uLength) != 1)
									return ERR_INVALID_DEC;
								cCurrent = (UCHAR)uCurrent;
								// Point to last character of number
								pszLine += (uLength - 1);
							} else
								fLiteral = TRUE;
					}
				}

				// Set character type flags
				if (fTermChar = (cCurrent == '\0'))
					fStdMetaChar = fClassMetaChar = FALSE;
				else {
					fStdMetaChar = ((cCurrent == EOL) || (!fLiteral && (strchr(StdMetaList, (int)cCurrent) != NULL)));
					fClassMetaChar = (!fLiteral && (strchr(ClassMetaList, (int)cCurrent) != NULL));
				}

				// If doing a string, send the character to the string
				if (fString)
					NextState(FILLSTRING);

				// If doing a class, send the character to the class
				else if (fClass)
					NextState(FILLCLASS);

				// If not in a string or class exit if we're done, or start a
				// new element.
				else
					ChooseNextState(fTermChar, COMPEXIT, INITELEMENT);

				// Set defaults for next time
				fUseCurrent = FALSE;

				break;


			// ------------------------------------------------------------
			// Initialize a new pattern element
			State(INITELEMENT):
				// Set the scan flag based on previous occurrence of a
				// "*"; default to no separate upper case, length 1
				pPattern->fScan = fExprScan;
				pPattern->fUpper = FALSE;
				pPattern->nLength = 1;
				ChooseNextState(fStdMetaChar, METACHAR, INITSTRING);
				break;


			// ------------------------------------------------------------
			// Start a new string
			State(INITSTRING):
				// Set string type and flag, initialize length
				pPattern->nType = Type(STRING);
				pPattern->nLength = 0;
				fString = TRUE;
				NextState(FILLSTRING);
				break;


			// ------------------------------------------------------------
			// Add a character to the current string
			State(FILLSTRING):

				// If we are at the end of the expression, or we have any meta-
				// character, terminate the string
				if (fTermChar || fStdMetaChar)
					NextState(STRINGDONE);

				// Otherwise, store the character in the string work area
				else {

					// Error if no more room
					if (pPattern->nLength >= (MAX_STRING - 1))
						return ERR_STR_LENGTH;

					if (fCase)
						// Case-sensitive, just save it
						StrWork[pPattern->nLength++] = cCurrent;
					else {
						// Case-insensitive, save lower case version and set flag
						// if upper case is different
						StrWork[pPattern->nLength++] = tolower(cCurrent);
						if (StrWork[pPattern->nLength] != toupper(cCurrent))
							pPattern->fUpper = TRUE;
					}
					NextState(NEXTCHAR);
				}
				break;


			// ------------------------------------------------------------
			// Finish the current string
			State(STRINGDONE):

				// If only one character, make it a type CHAR
				if (pPattern->nLength == 1) {
					pPattern->nType = Type(CHR);
					pPattern->text.chr = StrWork[0];
				}

				// Otherwise, reserve memory for the string, store the address,
				// and copy the string into place
				else {
					pPattern->text.pString = malloc(pPattern->nLength + 1);
					memcpy(pPattern->text.pString, StrWork, pPattern->nLength);
					pPattern->text.pString[pPattern->nLength] = '\0';
				}

				// String complete; reprocess terminating character
				fString = FALSE;
				fUseCurrent = TRUE;

				NextState(NEXTELEMENT);
				break;


			// ------------------------------------------------------------
			// Meta-character found
			State(METACHAR):

				// Select next state based on which meta-character we have
				// There must be a case below for every character in the
				// MC_STDMETA list in WSEARCH.H, plus the literal EOL
				// character ('\n')
				switch(cCurrent) {

					// Scan (*), just set flag
					case MC_SCAN:
						fExprScan = TRUE;
						NextState(NEXTCHAR);
						break;

					// Any character (?), element is done
					case MC_ANY:
						pPattern->nType = Type(ANY);
						NextState(NEXTELEMENT);
						break;

					// BOL (<), element is done
					case MC_BOL:
						pPattern->nType = Type(BEGINLINE);
						pPattern->nLength = 0;
						NextState(NEXTELEMENT);
						break;

					// EOL (> or '\n'), set EOL flag, element is done
					case MC_EOL:
					case EOL:
						pPattern->nType = Type(ENDLINE);
						NextState(NEXTELEMENT);
						break;

					// Start of class
					case MC_CLASS:
						NextState(INITCLASS);
						break;

					// End of class character outside a class -- error
					case MC_ENDCLASS:
						return ERR_CLASS_NOSTART;
						break;
				}
				break;


			// ------------------------------------------------------------
			// Start a class
			State(INITCLASS):

				// Set type, allocate memory for class array, and clear it
				pPattern->nType = Type(CLASS);
				pPattern->text.pString = malloc(256);
				memset(pPattern->text.pString, '\0', 256);

				// Set class variables and go on to next character
				cClassToggleValue = '\1';
				nClassCount = 0;
				fClass = TRUE;
				fRange = FALSE;
				NextState(NEXTCHAR);
				break;


			// ------------------------------------------------------------
			// Add current character to a class
			State(FILLCLASS):

				// Assume next thing is to get another character
				NextState(NEXTCHAR);

				// If this is a class meta-character, handle it
				if (fClassMetaChar) {

					// Process the class meta-characters
					// There must be a case below for every character in the
					// MC_CLASSMETA list in WSEARCH.H
					switch(cCurrent) {
	
						// Reverse class -- only has this meaning at start
						case MC_REVCLASS:
							if (nClassCount == 0) {
								// To reverse, set all character flags and
								// set the toggle value to 0
								memset(pPattern->text.pString, '\1', 256);
								cClassToggleValue = '\0';
							} else
								goto ClassNotMeta;
							break;

						// Range -- holler if at start or already in a range,
						// otherwise set it up
						case MC_RANGE:
							if (fRange || (nClassCount == 0))
								return ERR_INVALID_CLASS;
							fRange = TRUE;
							break;

						// End of class -- holler if empty or range in process,
						// otherwise just close the class
						case MC_ENDCLASS:
							if (fRange || (nClassCount == 0))
								return ERR_INVALID_CLASS;
							fClass = FALSE;
							NextState(NEXTELEMENT);
							break;
					}

					// Skip remainder of section for metacharacters
					break;
				}

ClassNotMeta:
				// If at end of expression, we have a problem
				if (fTermChar)
					return ERR_INVALID_CLASS;

				// Not a metacharacter or end of expression, so see if we are
				// in a range
				else if (fRange) {

					// Set end of range, handle "backward" ranges
					uRangeEnd = (UINT)cCurrent;
					if (uRangeStart > uRangeEnd) {
						uTemp = uRangeStart;
						uRangeStart = uRangeEnd;
						uRangeEnd = uTemp;
					}

					// Loop through range and toggle all characters that are
					// not already toggled
					for (uRangeChar = uRangeStart + 1; uRangeChar <= uRangeEnd; uRangeChar++) {
						if (pPattern->text.pString[uRangeChar] != cClassToggleValue) 
							ClassToggle(pPattern, (UCHAR)uRangeChar, fCase);
						nClassCount++;
					}

					// Range is done
					fRange = FALSE;
				}

				// Not in a range, must be a regular class character
				// Toggle it if it is not already toggled
				else {
					if (pPattern->text.pString[(UINT)cCurrent] != cClassToggleValue) 
						ClassToggle(pPattern, cCurrent, fCase);
					// Save each character as possible start of range
					uRangeStart = (UINT)cCurrent;
					nClassCount++;
				}

				break;


			// ------------------------------------------------------------
			// Element done, move to next element
			State(NEXTELEMENT):

				if (*nElements >= nMaxElements)
					return ERR_EXPR_LENGTH;
				(*nElements)++;
				pPattern++;
				fExprScan = FALSE;
				NextState(NEXTCHAR);
				break;

		}
	}

	// Convert a pattern consisting of just a single "*" to a separate
	// "match all" element (otherwise no element is generated, since a
	// "*" is not an element itself).
	if ((*nElements == 0) && fExprScan) {
		pPattern->fScan = FALSE;
		pPattern->nType = Type(ALL);
		(*nElements)++;
	}

	// Save new line pointer and return (but don't bump it past
	// the end of the string!)
	*ppszLine = (pszLine + ((*pszLine == '\0') ? 0 : 1));
	return 0;
}


// ---------------------------------------------------------------------
// Toggle the flag for a single character within a class
// ---------------------------------------------------------------------
void ClassToggle(PELEMENT pElement, UCHAR cChar, int fCase)
{
	// If case-sensitive, just toggle the specified character
	if (fCase)
		pElement->text.pString[(UINT)cChar] ^= 1;

	// If case-insensitive, toggle both cases -- e.g. [a] will toggle
	// both 'a' and 'A'
	else {

		cChar = tolower(cChar);
		pElement->text.pString[(int)cChar] ^= 1;
		pElement->text.pString[(int)(toupper(cChar))] ^= 1;

		// Set flag for matcher if upper and lower case are different
		if (cChar != toupper(cChar))
			pElement->fUpper = TRUE;
	}
}


// ---------------------------------------------------------------------
// Parse a single file specification
// ---------------------------------------------------------------------
int ParseFileSpec(PSZ *ppszLine, PFILESPEC pFileSpec)
{
	PSZ pszLine = *ppszLine;		// current scan point
	PSZ pszStart = pszLine;			// original start point
	PCH pEndPath;						// end of path portion of full name
	PCH pName;							// start of name portion
	int nLen;							// length of filename
	int rval;							// return value
	char cTerm = ' ';					// terminating character

	// Handle quoted names by setting terminating character to a quote
	if (*pszLine == '"') {
		cTerm = '"';
		pszLine++;
		pszStart++;
	}

	// Skip until we find the end of the filespec or the end of the line
	while ((*(++pszLine) != '\0') && (*pszLine != cTerm))
		;

	// Calculate the filename length and allocate space for it.
	// We allocate two extra bytes, one for the terminating NUL
	// and one to allow a forced ">" on the end if there is a
	// regular expression in the filespec.
	nLen = pszLine - pszStart;
	pFileSpec->pFileName = malloc(nLen + 2);

	// Copy the filename into place
	memcpy(pFileSpec->pFileName, pszStart, nLen);
	pFileSpec->pFileName[nLen] = '\0';

	// Figure out how long the path portion is -- we need this info
	// so that we can find the break between path and name while
	// parsing and processing the filespec
	if ((pEndPath = strrchr(pFileSpec->pFileName, '\\')) == NULL)
		pFileSpec->nPathLen = 0;
	else
		pFileSpec->nPathLen = (pEndPath - pFileSpec->pFileName + 1);

	// Set name pointer, assume no elements in filename regular
	// expression
	pName = pFileSpec->pFileName + pFileSpec->nPathLen;
	pFileSpec->nFileElements = 0;

	// See if we need to set up a regular expression for this filespec
	// In other words see if it contains any regular expression
	// meta-characters
	if (strpbrk(pName, StdMetaList) != NULL) {

		// Reserve memory for the filespec expression
		if ((pFileSpec->pFileExpr = malloc(MAX_FILENAME_ELEMENTS * sizeof(ELEMENT))) == NULL)
			return ERR_MEMORY;

		// Add a ">" on the end before parsing -- this forces the last
		// character in the pattern to match the last character in the
		// actual file name.
		pFileSpec->pFileName[nLen] = '>';
		pFileSpec->pFileName[nLen+1] = '\0';

		// Parse the filespec as a regular expression
		if ((rval = ParseExpr(&pName, pFileSpec->pFileExpr, &pFileSpec->nFileElements, MAX_FILENAME_ELEMENTS, MATCH_STRING)) != 0)
			return rval;

		// Remove trailing '>'
		pFileSpec->pFileName[nLen] = '\0';
	}

	// Save new line pointer and return (but don't bump it past
	// the end of the string!)
	*ppszLine = (pszLine + ((*pszLine == '\0') ? 0 : 1));
	return 0;
}


// ---------------------------------------------------------------------
// Search all files and directories included in a filespec
// ---------------------------------------------------------------------

int SearchFileSpec(PFILESPEC pFileSpec, PELEMENT pPattern, int nElements, int fFlags, ULONG *pulFiles, PSZ pszErrString)
{
	FINDDATA FindData;						// file search info
	FILESPEC SubDirSpec;						// subdirectory spec
	char szFileName[MAX_PATHNAME + 2];	// current file name
	char szPathName[MAX_PATHNAME + 1];	// current path name
	char szOutBuf[MAX_OUTPUT + 1];		// output text buffer
	PSZ pszSubDirScan;						// temporary buffer for subdir scan
	int rval = 0;								// return value
	int nPathLen;								// path length for name in szFileName
	int fPathOnly = FALSE;					// TRUE if user specified a path with no filename
	int fFound = FALSE;						// TRUE if we found something
	UINT uAttributes;							// filespec attributes

	// Initialize full file name and path length
	strcpy(szFileName, pFileSpec->pFileName);
	nPathLen = pFileSpec->nPathLen;

	// Add wildcard characters if necessary
	if (pFileSpec->nFileElements != 0)
		// If a regular expression was part of the filename, search for all
		// files (* or *.*) in the specified directory; later we use Match
		// to see if each file matches the spec
		strcpy(&szFileName[pFileSpec->nPathLen], szWildName);

	// If the filespec is not a regular expression and it's a directory
	// name, append "*" or "*.*"
	else if ((_dos_getfileattr(szFileName, &uAttributes) == 0) && (uAttributes & _A_SUBDIR)) {
		fPathOnly = TRUE;
		nPathLen = strlen(szFileName);
		// Add a backslash at the end of the name if there isn't one
		if (szFileName[nPathLen - 1] != '\\') {
			szFileName[nPathLen++] = '\\';
			szFileName[nPathLen] = '\0';
		}
		strcat(szFileName, szWildName);
	}

	// Save the path portion (if any)
	if (nPathLen > 0)
		memcpy(szPathName, szFileName, nPathLen);
	szPathName[nPathLen] = '\0';

	// First loop through all files in the filespec
	for (FindData.fInit = FALSE; ((rval = FindFile(szFileName, ATTR_ALL, &FindData,
			szFileName, pszErrString)) != NO_MORE_FILES); ) {

		// Return if there was a real error (not just out of files) in FindFile
		if (rval != 0)
			return rval;

		// If we have a regular expression for this filespec, test to see if
		// it matches the returned name
		if (pFileSpec->nFileElements != 0) {
			rval = Match(szFileName, pFileSpec->pFileExpr, pFileSpec->nFileElements, MATCH_STRING, (ULONG *)NULL, (PSZ)NULL);

			// If no match, try another file
			if (rval == MATCH_NOT_FOUND)
				continue;

			// If an error occurred, get out
			if (rval != MATCH_FOUND)
				return rval;
		}

		// We found the file
		fFound = TRUE;

		// Insert the path in the filename buffer if necessary, so that
		// szFileName contains the path and name (we have to do this
		// because FIndFile returns names only)
		if (nPathLen > 0) {
			memmove(&szFileName[nPathLen], szFileName, strlen(szFileName) + 1);
			memcpy(szFileName, szPathName, nPathLen);
		}

		if (nElements == 0) {

			// We aren't matching text, so just display the filename and count
			// the file
			TruncateFilename(szOutBuf, szFileName, MAX_FN_OUTPUT, FN_PART1_LEN, NULLSTR, 0);
			if ((rval = WSDispResult(0, szOutBuf)) != 0)
				return rval;
	 		(*pulFiles)++;

		} else {

			// If matching text, display this file name as the search status
			// to keep the user updated on our progress
			TruncateFilename(szOutBuf, szFileName, MAX_STATUS_OUTPUT, STATUS_PART1_LEN, NULLSTR, 0);
			if ((rval = WSDispResult(1, szOutBuf)) != 0)
				return rval;

			// Do the text search!
			if (((rval = Match(szFileName, pPattern, nElements, fFlags, pulFiles, pszErrString)) != MATCH_FOUND) && (rval != MATCH_NOT_FOUND))
				return rval;
		}

#if defined(__WINDOWS__) || defined(__NT__)
		// Always yield after processing the file if there is a pattern
		// to match.  If there is no pattern, yield only if fast response
		// was requested.  If we yield and the search was terminated,
		// return with a termination code
		if ((nElements > 0) || ((nElements == 0) && (fFlags & FAST_RESPONSE))) {
			if (!WSYield(FALSE)) {
				WSDispResult(0, SEARCH_TERMINATED);
				return TERMINATE;
			}
		}
#endif
	}

	// If we are doing subdirectories, loop through all of them and call
	// ourselves recursively to search each one
	if (fFlags & SUBDIRS) {

		// Create the path and name to scan for the subdirectory
		if ((pszSubDirScan = malloc(nPathLen + strlen(szWildName) + 1)) == NULL)
			return ERR_MEMORY;
		strcpy(pszSubDirScan, szPathName);
		strcat(pszSubDirScan, szWildName);

		// Create a filespec for the subdirectory
		memcpy(&SubDirSpec, pFileSpec, sizeof(FILESPEC));
		SubDirSpec.pFileName = szFileName;

		for (FindData.fInit = FALSE; ((rval = FindFile(pszSubDirScan, ATTR_DIR, &FindData, szFileName, pszErrString)) != NO_MORE_FILES); ) {

			// Free the subdirectory name buffer, it's no longer needed after
			// the find first, and freeing it here makes it less complex to
			// clean up after an error later
			if (pszSubDirScan != NULL) {
				free(pszSubDirScan);
				pszSubDirScan = NULL;
			}

			// Return if there was a real error (not just out of files) in
			// FindFile
			if (rval != 0)
				return rval;

			// Ignore "." and ".."
			if ((strcmp(szFileName, ".") == 0) || (strcmp(szFileName, "..") == 0))
				continue;
	
			// Insert the path in the filename buffer if necessary
			if (pFileSpec->nPathLen > 0) {
				memmove(&szFileName[nPathLen], szFileName, strlen(szFileName) + 1);
				memcpy(szFileName, szPathName, nPathLen);
			}

			// If we are not matching text, display the new directory name as
			// the search status
			if (nElements == 0) {
				TruncateFilename(szOutBuf, szFileName, MAX_STATUS_OUTPUT, STATUS_PART1_LEN, NULLSTR, 0);
				if ((rval = WSDispResult(1, szOutBuf)) != 0)
					return rval;
			}

			// Add the original filespec (if any) onto the directory name
			if (!fPathOnly) {
				strcat(szFileName, "\\");
				SubDirSpec.nPathLen = strlen(szFileName);
				strcat(szFileName, &pFileSpec->pFileName[pFileSpec->nPathLen]);
			}

			// Recursively search the directory
			if ((rval = SearchFileSpec(&SubDirSpec, pPattern, nElements, (fFlags | IN_SUBDIRS), pulFiles, pszErrString)) == FILE_FOUND)
				fFound = TRUE;
			else if ((rval != FILE_NOT_FOUND) && (rval != NO_MORE_FILES))
				return rval;
		}

		// Free the subdirectory name buffer if it is still reserved (this
		// will oly happen if no subdirectories were found)
		if (pszSubDirScan != NULL)
			free(pszSubDirScan);

	}

	if (fFound) 
		// If something matched, tell the caller
		rval = FILE_FOUND;
	else if (!(fFlags & IN_SUBDIRS)) {
		// If nothing matched, complain
		strcpy(szFileName, pFileSpec->pFileName);
		strcat(szFileName, NO_FILE_MSG);
		TruncateFilename(szOutBuf, szFileName, MAX_FN_OUTPUT, FN_PART1_LEN, NULLSTR, 0);
		if ((rval = WSDispResult(0, szOutBuf)) != 0)
			return rval;
		rval = FILE_NOT_FOUND;
	}

#if defined(__WINDOWS__) || defined(__NT__)
	// Always yield after processing the filespec
	// If we yield and find the search was terminated, return with 
	// the termination code
	if (!WSYield(FALSE)) {
		WSDispResult(0, SEARCH_TERMINATED);
		return TERMINATE;
	}
#endif

	return rval;
}

#if defined(__WINDOWS__)
// ---------------------------------------------------------------------
// Find a file under 16-bit Windows
// ---------------------------------------------------------------------
int FindFile(PSZ pszFileName, UINT uAttrib, PFINDDATA pFindData, PSZ pszResult, PSZ pszErrString)
{
	int rval;										// return value
	int fFindNext = pFindData->fInit;		// TRUE to try a find next

	// Find first
	if (!fFindNext) {

		// Look for a file, if none found return
		rval = _dos_findfirst(pszFileName, (uAttrib & 0x7FFF), &pFindData->FileInfo);
		if ((rval == DOS_FILE_NOT_FOUND) || (rval == DOS_PATH_NOT_FOUND) || (rval == DOS_NO_MORE_FILES))
			return NO_MORE_FILES;

		// Holler if an error occurred
		if (rval != 0) {
			strcpy(pszErrString, pszFileName);
			return ERR_FIND_FILE;
		}

		// Show the scan is initialized
		pFindData->fInit = TRUE;

		// Try again if we need a directory and this isn't one (because
		// _dos_findfirst returns files and directories even when we are
		// looking for just directories)
		if ((uAttrib & 0x8000) && (!(pFindData->FileInfo.attrib & _A_SUBDIR)))
			fFindNext = TRUE;
	}

	// Find next
	while (fFindNext) {

		// Look for next file, clean up if there's an error
		rval = _dos_findnext(&pFindData->FileInfo);
		if ((rval != 0) && (pFindData->fInit)) {
			_dos_findclose(&pFindData->FileInfo);
			pFindData->fInit = FALSE;
		}

		// If no more files, return that
		if (rval == DOS_NO_MORE_FILES)
			return NO_MORE_FILES;

		// Holler if an error occurred
		if (rval != 0)  {
			strcpy(pszErrString, pszFileName);
			return ERR_FIND_FILE;
		}

		// Loop is done if unless we need a directory and this isn't one
		if ((!(uAttrib & 0x8000)) || (pFindData->FileInfo.attrib & _A_SUBDIR))
			fFindNext = FALSE;
	}

	// Return result to caller
	strcpy(pszResult, pFindData->FileInfo.name);
	return 0;
}

#elif defined(__NT__)
// ---------------------------------------------------------------------
// Find a file under Win32
// ---------------------------------------------------------------------
#define DIRATTR FILE_ATTRIBUTE_DIRECTORY
#define HSATTR (FILE_ATTRIBUTE_SYSTEM | FILE_ATTRIBUTE_HIDDEN)
int FindFile(PSZ pszFileName, UINT uAttrib, PFINDDATA pFindData, PSZ pszResult, PSZ pszErrString)
{
	DWORD dwLastError;				// last file system error
	DWORD dwAllowHSAttr = (DWORD)(uAttrib & HSATTR);	// hidden / system portion of attributes
	DWORD dwDirAttr = (DWORD)(uAttrib & DIRATTR);		// directory portion of attributes
	DWORD dwFileHSAttr;				// hidden / system portion of file attributes

	// Loop until we get a file or an error (we have to do this
	// because NT returns all files, it can't select by attributes)
	for ( ; ; ) {

		// Find first
		if (!pFindData->fInit) {
	
			// Look for a file, if none found return
			if ((pFindData->hSearch = FindFirstFile((LPCTSTR)pszFileName, (LPWIN32_FIND_DATA)&pFindData->FileInfo)) == INVALID_HANDLE_VALUE) {
				dwLastError = GetLastError();
				if ((dwLastError == ERROR_FILE_NOT_FOUND) || (dwLastError == ERROR_PATH_NOT_FOUND) || (dwLastError == ERROR_NO_MORE_FILES))
					return NO_MORE_FILES;
				else {
					strcpy(pszErrString, pszFileName);
					return ERR_FIND_FILE;
				}
			}

			// Show the scan is initialized
			pFindData->fInit = TRUE;
		}
	
		// Find next
		else {

			// Look for next file, clean up if there's an error
			if (!FindNextFile(pFindData->hSearch, (LPWIN32_FIND_DATA)&pFindData->FileInfo)) {
				FindClose(pFindData->hSearch);
				pFindData->fInit = FALSE;
				if (GetLastError() == ERROR_NO_MORE_FILES)
					return NO_MORE_FILES;
				else {
					strcpy(pszErrString, pszFileName);
					return ERR_FIND_FILE;
				}
			}
		}
		
		// If the directory attribute is set as requested, and there are no
		// disallowed hidden / system attributes, then we're done
		if ((pFindData->FileInfo.dwFileAttributes & DIRATTR) == dwDirAttr) {
			dwFileHSAttr = (pFindData->FileInfo.dwFileAttributes & HSATTR);
			if (dwFileHSAttr == (dwFileHSAttr & dwAllowHSAttr))
				break;
		}
	}

	// Return result to caller
	strcpy(pszResult, pFindData->FileInfo.cFileName);
	return 0;
}

#elif defined(__OS2__)
// ---------------------------------------------------------------------
// Find a file under OS/2
// ---------------------------------------------------------------------
int FindFile(PSZ pszFileName, UINT uAttrib, PFINDDATA pFindData, PSZ pszResult, PSZ pszErrString)
{
	APIRET rc;
	ULONG ulFiles = 1L;
	FILEFINDBUF3 OS2Result;

	// Find first
	if (!pFindData->fInit) {

		pFindData->hSearch = HDIR_CREATE;

		// Look for a file, if none found return
		rc = DosFindFirst(pszFileName, &(pFindData->hSearch), (ULONG)uAttrib, &OS2Result,
			sizeof(OS2Result), &ulFiles, FIL_STANDARD);
		if ((rc == ERROR_FILE_NOT_FOUND) || (rc == ERROR_PATH_NOT_FOUND) || (rc == ERROR_NO_MORE_FILES))
			return NO_MORE_FILES;

		// Holler if an error occurred
		if (rc != 0) {
			strcpy(pszErrString, pszFileName);
			return ERR_FIND_FILE;
		}

		// Show the scan is initialized
		pFindData->fInit = TRUE;
	}

	// Find next
	else {

		// Look for next file, clean up if there's an error
		rc = DosFindNext(pFindData->hSearch, &OS2Result, sizeof(OS2Result), &ulFiles);
		if ((rc != 0) && (pFindData->fInit)) {
			DosFindClose(pFindData->hSearch);
			pFindData->fInit = FALSE;
		}

		// If no more files, return that
		if (rc == ERROR_NO_MORE_FILES)
			return NO_MORE_FILES;

		// Holler if an error occurred
		if (rc != 0) {
			strcpy(pszErrString, pszFileName);
			return ERR_FIND_FILE;
		}
	}

	// Return result to caller
	strcpy(pszResult, OS2Result.achName);
	return 0;
}
#endif


// ---------------------------------------------------------------------
// Search a file or test a string to find matches for an expression
// ---------------------------------------------------------------------
int Match(PSZ pszFileName, PELEMENT pPattern, int nElements, int fFlags, ULONG *pulFiles, PSZ pszErrString)
{
	// Local variables
	int nState;								// current state
	int nFile = 0;							// current file handle
	int fMatch;								// TRUE if an element matched
	int fMatchFound = FALSE;			// TRUE if at least one full match found
	int fCase = (fFlags & CASE_SENSITIVE);			// TRUE for case-sensitive search
	int fLineNumbers = (fFlags & LINE_NUMBERS);	// TRUE to show line numbers
	int fMatchString = (fFlags & MATCH_STRING);	// TRUE to match a string, not a file
	int fMatchAll = (fFlags & MATCH_ALL);			// TRUE to match all lines
	int fEOLAdded = FALSE;				// TRUE if EOL added for string match
	int fDispLeader;						// TRUE to display leader for this output line
	int fDispTrailer;						// TRUE to display trailer for this output line
	int rval;								// return value
	UINT uMatchLen;						// total characters in MatchBuf or match output
	UINT uPrematchChars;					// prefix characters in MatchBuf
	UINT uMatchLineLen;					// length of matching line for output
	UINT uOutputCount;					// characters added to output buffer
	UINT uDispLen;							// length of actual output, without leader/trailer
	UINT uPadLeft;							// padding at left end of output line
	ULONG ulMatches = 0L;				// number of matches found in this file
	PCH pBuffer = NULL;					// location of file buffer
	PCH pLineSave = NULL;				// value of pLine at character checkpoint
	PCH pMatch;								// location where current match was found
	PCH pFirstMatch;						// location where beginning of match was found
	PCH pEOL;								// location of EOL during line count
	PCH pMatchStart;						// used when manipulating match text for output
	PCH pMatchEnd;							// end of matching text for output
	PCH pMatchLine;						// start of matching line for output
	PCH pMatchLineEnd;					// end of matching line for output
	PCH pDispStart;						// first character for output
	PCH pDispEnd;							// last character for output + 1
	PCH pOut;								// used to set up display output
	PCH pRepl;								// used to replace non-printing chars in output
	PELEMENT pElementSave;				// value of pElement at character checkpoint
	PELEMENT pEndPattern = &(pPattern[nElements]);		// last element in pattern + 1
	PSTRMATCH pfStrMatch;				// string comparison routine
	SEARCHSTATUS Search;					// current search status
	SEARCHSTATUS LineCheckpoint;		// saved status at line checkpoint
	char MatchBuf[MAX_MATCH_SAVE + 1];	// buffer for first matching line
	char szOutBuf[MAX_OUTPUT + 1];	// output text buffer

	// Point to the proper comparison function depending on case sensitivity
	pfStrMatch = (fCase ? memcmp : memicmp);

	// State machine to look for pattern matches in the file
	for (NextState(MATCHINIT); NotState(MATCHEXIT); ) {

		switch(nState) {

			// ------------------------------------------------------------
			// Initialize for scanning this file
			State(MATCHINIT):

				// If not matching a string, reserve the read buffer and
				// open the file
				if (!fMatchString) {

					// Reserve memory for buffer
					if ((pBuffer = malloc(BUF_ALLOC)) == NULL) {
						rval = ERR_MEMORY;
						goto MatchExit;
					}
	
					// Open file
					if ((nFile = sopen(pszFileName, (O_BINARY | O_RDONLY), SH_DENYWR)) == -1) {
						// If we can't open it just show that as the result (file
						// is probably in use)
						strcpy(pBuffer, pszFileName);
						strcat(pBuffer, OPEN_ERROR);
						TruncateFilename(szOutBuf, pBuffer, MAX_FN_OUTPUT, FN_PART1_LEN, NULLSTR, 0);
						if ((rval = WSDispResult(0, szOutBuf)) != 0)
							return rval;
						rval = FILE_NOT_FOUND;
						goto MatchExit;
					}
				}

				// Set start of file, start of pattern
				Search.ulBlockOffset = 0L;
				Search.ulLineNum = 1L;
				Search.uDiscardCount = 0;
				Search.pElement = pPattern;

				// Clear checkpoint
				LineCheckpoint.pLine = NULL;

				NextState(NEXTBUF);
				break;


			// ------------------------------------------------------------
			// Read the next buffer
			State(NEXTBUF):

				// If matching a string, set up the pointers and be sure
				// it ends with a newline
				if (fMatchString) {
					pBuffer = pszFileName;
					Search.pBufEnd = pszFileName + strlen(pszFileName) - 1;
					if (*Search.pBufEnd != '\n') {
						*(++Search.pBufEnd) = '\n';
						Search.pBufEnd[1] = '\0';
						fEOLAdded = TRUE;
					}
				}

				// Otherwise read the buffer
				else {

					if ((rval = ReadNextBuf(pszFileName, nFile, pBuffer, &Search, pszErrString)) == -1) {
						// Quit if we hit the EOF
						NextState(MATCHDONE);
						break;
					} else if (rval != 0)
						// Get out if read failed
						goto MatchExit;

#if defined(__WINDOWS__) || defined(__NT__)
					// Yield after reading each buffer if fast response was
					// requested
					if ((fFlags & FAST_RESPONSE) && (!WSYield(FALSE))) {
						rval = TERMINATE;
						WSDispResult(0, SEARCH_TERMINATED);
						goto MatchExit;
					}
#endif
				}

				// Set up first line
				Search.pLineStart = Search.pLine = pBuffer;
				NextState(NEWLINE);
				break;


			// ------------------------------------------------------------
			// Move to the next line in the buffer
			State(NEXTLINE):
	 
	   		// If we are just matching a string then there are no more lines
				if (fMatchString) {
					rval = MATCH_NOT_FOUND;
					NextState(MATCHEXIT);
					break;
				}

				// Clear old character checkpoint, bump line number
				pLineSave = NULL;
				Search.ulLineNum++;

				// Go to EOL + 1; if that's past end of buffer then we need a new one
				Search.pLineStart = Search.pLineEnd + 1;
				ChooseNextState((Search.pLineStart < Search.pBufEnd), NEWLINE, NEXTBUF);
				break;


			// ------------------------------------------------------------
			// New line available, decide how to process it
			State(NEWLINE):
				
				// Find EOL (NEXTBUF enures there will be one)
				Search.pLine = Search.pLineStart;
				Search.pLineEnd = memchr(Search.pLine, EOL, Search.pBufEnd - Search.pLine + 1);

				// For a file scan at the start of the pattern, scan the whole
				// buffer; otherwise go test for a match at the current position
				ChooseNextState(((Search.pElement == pPattern) &&
					Search.pElement->fScan && !fMatchString), SCANBUF, TESTMATCH);
				break;


			// ------------------------------------------------------------
			// Scan the buffer for a match
			State(SCANBUF):

				// Clear any previous saved text
				uMatchLen = 0;

				// Do the scan
				pFirstMatch = pMatch = Scan(Search.pLine, Search.pBufEnd, Search.pLineStart, Search.pLineEnd, Search.pElement, pfStrMatch, fCase);

				// If we have a match, adjust line pointers and number if
				// necessary
				if (pMatch != NULL) {

					// If match is not on the current line then we need to adjust
					// things
					if (pMatch > Search.pLineEnd) {

						// If we care about line numbers then count EOLs until we
						// find the start of the line where the match occurred
						if (fLineNumbers) {
							for (Search.pLineStart = Search.pLine; Search.pLineStart < pMatch; Search.ulLineNum++) {
								if ((pEOL = memchr(Search.pLineStart, EOL, pMatch - Search.pLineStart + 1)) == NULL)
									break;
								Search.pLineStart = pEOL + 1;
							}
						}

						// If we don't care about line numbers then just scan
						// backward to find the start of the current line (where
						// the match occurred)
						else {
							for (Search.pLineStart = pMatch - 1; ((Search.pLineStart > pBuffer) && (*Search.pLineStart != EOL)); Search.pLineStart--)
								;
							Search.pLineStart++;
						}

						// Either way we have to find the new EOL
						Search.pLineEnd = memchr(pMatch, EOL, Search.pBufEnd - pMatch + 1);
					}

					NextState(CCHECK);
	
				}
				
				// If there's no match count the lines in this buffer if
				// necessary, then get the next buffer
				else {
					if (fLineNumbers) {
						for (Search.pLineStart = Search.pLine; Search.pLineStart < Search.pBufEnd; Search.ulLineNum++) {
							if ((pEOL = memchr(Search.pLineStart, EOL, pMatch - Search.pLineStart + 1)) == NULL)
								break;
							Search.pLineStart = pEOL + 1;
						}
					}
					NextState(NEXTBUF);
				}
				break;


			// ------------------------------------------------------------
			// Set the character checkpoint and skip past it (we return to
			// this checkpoint if a later match fails and we have to scan
			// forward further in the line to keep matching)
			State(CCHECK):

				pLineSave = pMatch + 1;
				pElementSave = Search.pElement;
				Search.pLine = pMatch + Search.pElement->nLength;
				NextState(ADVANCE);
				break;


			// ------------------------------------------------------------
			// Advance to the next pattern element
			State(ADVANCE):

				// Increment the pattern pointer, and choose the next state
				// based on whether the pattern is exhausted, and if not,
				// whether the line is exhausted
				if (++(Search.pElement) < pEndPattern)
					ChooseNextState((Search.pLine <= Search.pLineEnd), TESTMATCH, LCHECK);
				else
					// If the pattern is exhausted, it's a match
					NextState(MATCHED);
				break;


			// ------------------------------------------------------------
			// Test the current pattern element for a match
			State(TESTMATCH):

				// If the scan flag is set start a scan
				if (Search.pElement->fScan) {
					NextState(SCANLINE);
					break;
				}

				// No scan, test the pattern element against the file
				switch(Search.pElement->nType) {

					// Single character
					Element(CHR):
						if (fCase || (!Search.pElement->fUpper))
							fMatch = ((Search.pElement->text.chr) == *(Search.pLine));
						else
							fMatch = (toupper(Search.pElement->text.chr) == toupper(*(Search.pLine)));
						break;

					// String, make sure it fits on the remainder of the current
					// line, and if so then see if it matches
					Element(STRING):
						fMatch = ((Search.pElement->nLength <= (Search.pLineEnd - Search.pLine + 1)) &&
							((*pfStrMatch)(Search.pLine, Search.pElement->text.pString, Search.pElement->nLength) == 0));
						break;

					Element(BEGINLINE):
						// Match if at the beginning of a line
						fMatch = (Search.pLine == Search.pLineStart);
						break;

					Element(ENDLINE):
						// Match if at the end of a line, including at CR of a
						// CRLF pair
						fMatch = (Search.pLine == Search.pLineEnd);
						if (!fMatch && (Search.pLine == (Search.pLineEnd - 1)) && (*Search.pLine == CR)) {
							// CR of a CRLF pair, so skip past it (we skip past
							// the LF below)
							fMatch = TRUE;
							Search.pLine++;
						}
						break;

					Element(ANY):
						// Match any character except EOL
						fMatch = (Search.pLine != Search.pLineEnd);
						break;

					Element(ALL):
						// Match all strings (pattern is a single '*')
						fMatch = TRUE;
						break;

					Element(CLASS):
						// Match if this character flag is set in the class array
						// Any case issues were handled during compilation, no need
						// to worry about case sensitivity here
						fMatch = (Search.pElement->text.pString[(UINT)*(Search.pLine)]);

						// If the class includes the EOL character, check for match
						// on CR of CRLF pair
						if (!fMatch && (Search.pLine == (Search.pLineEnd - 1)) && (*Search.pLine == CR)
								&& (Search.pElement->text.pString[EOL] != 0)) {
							fMatch = TRUE;
							Search.pLine++;
						}
						break;
				}

				if (fMatch) {
					// A match was found, skip past it, and continue with the
					// next pattern element
					Search.pLine += Search.pElement->nLength;
					NextState(ADVANCE);
				} else
					// No match was found, handle the mismatch
					NextState(EMISMATCH);

				break;


			// ------------------------------------------------------------
			// Scan the line for a match
			State(SCANLINE):

				pMatch = Scan(Search.pLine, Search.pLineEnd, Search.pLineStart, Search.pLineEnd, Search.pElement, pfStrMatch, fCase);
				ChooseNextState((pMatch != NULL), CCHECK, LMISMATCH);
				break;


			// ------------------------------------------------------------
			// Element mismatch, see if we can back up within the line
			State(EMISMATCH):
	 
	   		// If EOL or no previous scan, line cannot match
				if ((Search.pLine == Search.pLineEnd) || (pLineSave == NULL))
					NextState(LMISMATCH);

				// Otherwise back up to previous match + 1 (based on info
				// saved in the CCHECK state), and retry the scan
				else {
					Search.pLine = pLineSave;
					Search.pElement = pElementSave;
					pLineSave = NULL;
					NextState(SCANLINE);
				}
				break;


			// ------------------------------------------------------------
			// Line mismatch, back up if necessary
			State(LMISMATCH):

				// See if we need to back up (because the previous match was
				// in a multi-line pattern)
				if (LineCheckpoint.pLine != NULL) {

					// Checkpoint available, back up to it
					if (LineCheckpoint.ulBlockOffset != Search.ulBlockOffset) {
	 
						// If the match started in a different block, seek back
						// to that block
	   				if (lseek(nFile, ((long)LineCheckpoint.ulBlockOffset - (long)LineCheckpoint.uPrefixLen), SEEK_SET) == -1) {
							strcpy(pszErrString, pszFileName);
							rval = ERR_FSEEK;
							goto MatchExit;
						}

						// Re-read the old prefix
						if ((read(nFile, pBuffer, (int)LineCheckpoint.uPrefixLen)) != (int)LineCheckpoint.uPrefixLen) {
							strcpy(pszErrString, pszFileName);
							rval = ERR_FREAD;
							goto MatchExit;
						}

						// Re-read the old block
						if (read(nFile, pBuffer + LineCheckpoint.uPrefixLen, BUF_SIZE) == -1) {
							strcpy(pszErrString, pszFileName);
							rval = ERR_FREAD;
							goto MatchExit;
						}
					}

					// Restore the search status to where it was when the
					// first line matched the first part of the pattern
					memcpy((PCH)&Search, (PCH)&LineCheckpoint, sizeof(SEARCHSTATUS));

					// Clear old character and line checkpoints
					pLineSave = NULL;
					LineCheckpoint.pLine = NULL;

				}

				// Always restart the search at the start of the pattern
				Search.pElement = pPattern;

				NextState(NEXTLINE);
				break;


			// ------------------------------------------------------------
			// Set a line checkpoint if one is not set
			State(LCHECK):

				if (LineCheckpoint.pLine == NULL) {

					// Save the part of the first matching line that we might
					// need for output
					pMatchStart = pFirstMatch - MAX_MATCH_OUTPUT + 1;
					if (pMatchStart < Search.pLineStart)
						pMatchStart = Search.pLineStart;
					uMatchLen = Search.pLineEnd - pMatchStart;
					if (uMatchLen > MAX_MATCH_SAVE)
						uMatchLen = MAX_MATCH_SAVE;
					uPrematchChars = pFirstMatch - pMatchStart;
					memcpy(MatchBuf, pMatchStart, uMatchLen);
					MatchBuf[uMatchLen] = '\0';

					// Create the checkpoint; we'll back up to this point if
					// a later part of the pattern fails to match and we need
					// to restart the process
					memcpy((PCH)&LineCheckpoint, (PCH)&Search, sizeof(SEARCHSTATUS));
				}
				NextState(NEXTLINE);
				break;


			// ------------------------------------------------------------
			// Match found, display it and decide what to do
			State(MATCHED):

				// If this is a string match, we're done
				if (fMatchString) {
					rval = MATCH_FOUND;
					NextState(MATCHEXIT);
					break;
				}

				// If this is the first match, count the file
				if (ulMatches == 0L)
					(*pulFiles)++;

				// Setup to display the match
				if (fMatchAll) {

					// If matching all lines and this is the first match, display
					// the file name
					if (ulMatches == 0L) {
						TruncateFilename(szOutBuf, pszFileName, MAX_FN_OUTPUT, FN_PART1_LEN, ":", 1);
						if ((rval = WSDispResult(0, szOutBuf)) != 0)
							goto MatchExit;
					}

					// Start with two spaces in the output buffer, to provide an
					// indent (matching line is added below)
					pOut = strcpy(szOutBuf, "  ") + 2;

				} else
					// Not matching all lines, so we need the filename in the
					// output buffer (matching line is added below)
					pOut = TruncateFilename(szOutBuf, pszFileName, MAX_FN_OUTPUT, FN_PART1_LEN, ":  ", 3);

				
				// Add a line number if requested, and skip past it
				if (fLineNumbers) {
					sprintf(pOut, "[%lu]  %n", Search.ulLineNum, &uOutputCount);
					pOut += uOutputCount;
				}

				// Find the matching text and its limits ...
				if (uMatchLen == 0) {

					// .. for a match that started on this line
					pMatchStart = pFirstMatch;
					pMatchEnd = Search.pLine;
					pMatchLine = Search.pLineStart;
					pMatchLineEnd = Search.pLineEnd;

				} else {

					// ... for a match that started on a previous line and crossed
					// a line boundary
					pMatchStart = MatchBuf + uPrematchChars;
					pMatchLine = MatchBuf;
					pMatchEnd = pMatchLineEnd = MatchBuf + uMatchLen;
				}

				// Calculate the length of the matching text, and of the line
				// it's in
				uMatchLen = pMatchEnd - pMatchStart;
				uMatchLineLen = pMatchLineEnd - pMatchLine;

				// Set up the output defaults to display the entire line
				// containing the match
				pDispStart = pMatchLine;
				uDispLen = uMatchLineLen;
				fDispLeader = fDispTrailer = FALSE;

				// If the matching line does not fit we have to truncate it
				if (uMatchLineLen > MAX_MATCH_OUTPUT) {

					// In this situation we always display the maximum
					uDispLen = MAX_MATCH_OUTPUT;

					// See if the matching text itself is too long, even without
					// the rest of the line
					if (uMatchLen >= MAX_MATCH_OUTPUT)

						// Match text is too long or just fits, so we can start the
						// display with the match text
						pDispStart = pMatchStart;

					else {

						// There is room for the match text plus some context.
						// Pad on the left with up to half the excess text,
						// limited by what's available on the left.
						uPadLeft = (MAX_MATCH_OUTPUT - uMatchLen) / 2;
						if (uPadLeft > (pMatchStart - pMatchLine))
							uPadLeft = (pMatchStart - pMatchLine);
						pDispStart = pMatchStart - uPadLeft;

						// If we don't have enough padding on the right to fill
						// out the line, use more on the left (we know there's
						// enough, otherwise the line would fit and we wouldn't
						// be here!)
						if ((pMatchLineEnd - pDispStart) < MAX_MATCH_OUTPUT)
							pDispStart = pMatchLineEnd - MAX_MATCH_OUTPUT;

					}

					// Display a leader if the first character displayed is not
					// at the start of the line, and a trailer if the last
					// character is not at the end
					fDispLeader = (pDispStart != pMatchLine);
					fDispTrailer = ((pDispStart + uDispLen) != pMatchLineEnd);
				}

				// Throw away any CR at the end of the line
				if ((pDispStart[uDispLen] == EOL) && (pDispStart[uDispLen - 1] == CR)) 
					uDispLen--;

				pDispEnd = pDispStart + uDispLen;

				// Convert NULs in the displayed line into the replacement
				// character
				pDispStart[uDispLen] = '\0';
				for (pRepl = pDispStart; ((pRepl != NULL) && (pRepl < pDispEnd)); ) {
					if ((pRepl = memchr(pRepl, '\0', pDispEnd - pRepl)) != NULL)
						*pRepl = REPL_CHAR;
				}

#if defined(__OS2__)
				// In OS/2, convert BELs in the line into replacement character
				for (pRepl = pDispStart; ((pRepl != NULL) && (pRepl < pDispEnd)); ) {
					if ((pRepl = strchr(pRepl, BEL)) != NULL)
						*pRepl = REPL_CHAR;
				}
#endif

				// Add the leader, matching line, trailer, and final null
				// We rely on memcpy() returning the value of pOut on each call
				if (fDispLeader)
					pOut = (PCH)memcpy(pOut, LEADER, LEADLEN) + LEADLEN;
				pOut = (PCH)memcpy(pOut, pDispStart, uDispLen) + uDispLen;
				if (fDispTrailer)
					pOut = (PCH)memcpy(pOut, TRAILER, TRAILLEN) + TRAILLEN;
				*pOut = '\0';

				// Display the match output (finally!)
				if ((rval = WSDispResult(0, szOutBuf)) != 0)
					goto MatchExit;

				// Count matches
				ulMatches++;

				// If matching all lines, restart the search for next time
				if (fMatchAll)
					Search.pElement = pPattern;

				ChooseNextState(fMatchAll, NEXTLINE, MATCHDONE);
				break;


			// ------------------------------------------------------------
			// File scan complete
			State(MATCHDONE):

				// Display a message if no matches were found in this file
				if ((ulMatches == 0L) && (fFlags & SHOW_NOMATCH)) {
					strcat(pszFileName, NO_MATCH_MSG);
					TruncateFilename(szOutBuf, pszFileName, MAX_FN_OUTPUT, FN_PART1_LEN, NULLSTR, 0);
					if ((rval = WSDispResult(0, szOutBuf)) != 0)
						goto MatchExit;
				}

				free(pBuffer);
				close(nFile);
				nFile = 0;
				rval = (fMatchFound ? MATCH_FOUND : MATCH_NOT_FOUND);
				NextState(MATCHEXIT);
				break;

		}
	}

MatchExit:

	if (fFlags & MATCH_STRING) {
		// We are matching a string, remove any EOL we added
		if (fEOLAdded)
			*Search.pBufEnd = '\0';
	} else {
		// For file matches, free the buffer and close the file if necessary
		// (these steps are done earlier for a normal exit, but we do it here
		// in case of an error exit)
		if (pBuffer != NULL)
			free(pBuffer);
		if (nFile != 0)
			close(nFile);
	}
	return rval;

}


// ---------------------------------------------------------------------
// Assembler code to find the last EOL in the buffer, used by ReadNextBuf
// (below)
// Returns position of last EOL, or 0xFFFF if not found
// ---------------------------------------------------------------------

// The equivalent C code is:
//		PCH pDiscard;
//		pSearch->uDiscardCount = 0xFFFF;
//		for (pDiscard = pSearch->pBufEnd; ((pDiscard >= pRead) && (pDiscard >= (pSearch->pBufEnd - LINE_MAX))); pDiscard--) {
//			if (*pDiscard == EOL) {
//				pSearch->uDiscardCount = pSearch->pBufEnd - pDiscard;
//				pSearch->pBufEnd = pDiscard;
//				break;
//			}
//		}

// Register equivalences to C variables:
//		EDX	pSearch->UDiscardCount
//		EAX	pRead
//		EBX	&(pSearch->pBufEnd)
//		EDI	pDiscard

// The assembler code scans backward through the buffer looking for an EOL,
// limited by the start of the buffer and the start of the last LINE_MAX
// bytes (whichever is later in the buffer)

// We use assembler because the C runtime library does not handle backward
// searches through memory at hardware string speed.

#if (defined(__OS2__) || defined(__NT__))
	// 32-bit search for last EOL
	extern UINT LastEOL(PCH *ppBufEnd, PCH pRead, UINT uLineMax);
	#pragma aux LastEOL = \
	"				mov		edi,[ebx]" \
	"				mov		edx,edi" \
	"				sub		edx,ecx" \
	"				cmp		eax,edx" \
	"				jae		EOLSetup" \
	"				mov		eax,edx" \
	"EOLSetup:	mov		edx,0xFFFF" \
	"				mov		ecx,edi" \
	"				sub		ecx,eax" \
	"				inc		ecx" \
	"				push		es" \
	"				push		ds" \
	"				pop		es" \
	"				mov		al,0x0A" \
	"				std" \
	"				repne		scasb" \
	"				cld" \
	"				pop		es" \
	"				jne		EOLDone" \
	"				inc		edi" \
	"				mov		edx,[ebx]" \
	"				sub		edx,edi" \
	"				mov		[ebx],edi" \
	"EOLDone:" \
					value		[edx] \
					parm		[ebx] [eax] [ecx]\
					modify 	[eax ebx ecx edx edi]; 
#else
	// 16-bit search for last EOL
	extern UINT LastEOL(PCH *ppBufEnd, PCH pRead, UINT uLineMax);
	#pragma aux LastEOL = \
	"				mov		di,[bx]" \
	"				mov		dx,di" \
	"				sub		dx,cx" \
	"				cmp		ax,dx" \
	"				jae		EOLSetup" \
	"				mov		ax,dx" \
	"EOLSetup:	mov		dx,0xFFFF" \
	"				mov		cx,di" \
	"				sub		cx,ax" \
	"				inc		cx" \
	"				push		es" \
	"				push		ds" \
	"				pop		es" \
	"				mov		al,0x0A" \
	"				std" \
	"				repne		scasb" \
	"				cld" \
	"				pop		es" \
	"				jne		EOLDone" \
	"				inc		di" \
	"				mov		dx,[bx]" \
	"				sub		dx,di" \
	"				mov		[bx],di" \
	"EOLDone:" \
					value		[dx] \
					parm		[bx] [ax] [cx]\
					modify 	[ax bx cx dx di]; 
#endif


// ---------------------------------------------------------------------
// Read the next buffer from the file
// ---------------------------------------------------------------------
int ReadNextBuf(PSZ pszFileName, int nFile, PCH pBuffer, PSEARCHSTATUS pSearch, PSZ pszErrString)
{
	PCH pRead;							// location in buffer for data read
	int nReadBytes;					// bytes read

	// Take any bytes discarded from the end of the last buffer and move
	// them to the start of this one
	pRead = pBuffer;
	if (pSearch->uDiscardCount != 0) {
		memmove(pBuffer, pSearch->pBufEnd + 1, pSearch->uDiscardCount);
		pRead += pSearch->uDiscardCount;
	}
	pSearch->uPrefixLen = pSearch->uDiscardCount;

	// Read another buffer's worth of data
	if ((nReadBytes = read(nFile, pRead, BUF_SIZE)) == -1) {
		// The read failed, complain
		strcpy(pszErrString, pszFileName);
		return ERR_FREAD;
	}

	// If no data available then we're done with this file
	else if (nReadBytes == 0)
		return -1;

	// Adjust current file offset
	pSearch->ulBlockOffset += (ULONG)(UINT)nReadBytes;

	// Find the last EOL in the buffer (see assembler code above)
	pSearch->pBufEnd = (pRead + (unsigned int)nReadBytes - 1);
	pSearch->uDiscardCount = LastEOL(&(pSearch->pBufEnd), pRead, LINE_MAX);

	// If no EOL found, force one at the end of the buffer + 1
	if (pSearch->uDiscardCount == 0xFFFF) {
		pSearch->uDiscardCount = 0;
		*(++(pSearch->pBufEnd)) = EOL;
	}

	return 0;
}


// ---------------------------------------------------------------------
// Scan part of the buffer to see if we can find a match for one element
// Returns match position, or NULL for no match
// ---------------------------------------------------------------------
PCH Scan(PCH pScanStart, PCH pScanEnd, PCH pLineStart, PCH pLineEnd, PELEMENT pElement, PSTRMATCH pfStrMatch, int fCase)
{
	UINT uCharsRemaining;		// data left in buffer
	PCH pMatch = NULL;			// location where match was found -- assume no match
	PCH pScan = pScanStart; 	// current scan location

	uCharsRemaining = pScanEnd - pScan + 1;

	switch(pElement->nType) {

		// Single character
		Element(CHR):
			pMatch = CharScan(pScan, uCharsRemaining, pElement->text.chr, (fCase || !(pElement->fUpper)));
			break;

		Element(STRING):

			// Loop looking for the string
			while (uCharsRemaining > 0) {

				// Find the first character, see if there's room for the rest of
				// the string after it
				if (((pMatch = CharScan(pScan, uCharsRemaining, *(pElement->text.pString), (fCase || !(pElement->fUpper)))) == NULL) ||
						((UINT)pElement->nLength > (UINT)(pScanEnd - pMatch + 1))) {
					// Not found or no room, so no match
					pMatch = NULL;
					break;
				}

				// First character matches, if the rest of the string matches we
				// are done
				if ((*pfStrMatch)(pMatch + 1, pElement->text.pString + 1, pElement->nLength - 1) == 0)
					break;

				// Remainder of string failed the test, try again
				pScan = pMatch + 1;
				uCharsRemaining = pScanEnd - pScan + 1;
				pMatch = NULL;
			}

			break;

		Element(ANY):
			// Scan for any character except EOL
			if (pScan != pLineEnd)
				pMatch = pScan;
			break;

		Element(BEGINLINE):
			// Check if we are at the start of a line
			if (pScan == pLineStart)
				pMatch = pScan;
			break;

		Element(ENDLINE):
			// We can always find the end of a line
			pMatch = pLineEnd;
			break;

		Element(CLASS):
			// Match if we can find a character that matches the class
			// Any case issues were handled during compilation, no need
			// to worry about case sensitivity here
			for (; pScan <= pScanEnd; pScan++)
				// if this character's flag is set in the class array, we have a match
				if (pElement->text.pString[(UINT)*pScan]) {
					pMatch = pScan;
					break;
				}
			break;
	}

	return pMatch;
}


// ---------------------------------------------------------------------
// Assembler code for case-insensitive scan
// Returns match position, or NULL for no match
// ---------------------------------------------------------------------

// The equivalent C code is:
//
//		PCH pScanEnd;
//		for (cScan = toupper(cScan), pScanEnd = pScan + uCharCount; pScan < pScanEnd; pScan++) {
//			if (toupper(*pScan) == cScan)
//				return pScan;
//		}
//		return NULL;

// The assembler code loops through all characters, shifts each alphabetic
// character to lower case, then tests it against the scan character (the
// scan character in a case-insensitive scan is always lower case, this is
// taken care of when the expression is compiled).
//
// This method only works for the US English character set.  If you need
// support for other character sets and your C compiler provides it in the
// toupper() function, switch to the equivalent C code as shown above.  The
// C code shifts characters to upper case (rather than lower case like the
// assembler code) because some case conversion routines only handle non-
// English characters properly when converting to upper case.

#if (defined(__OS2__) || defined(__NT__))
	// 32-bit case-insensitive scan
	extern PCH ScanUL(PCH pScanUL, UINT uCharCountUL, UCHAR cScanUL);
	#pragma aux ScanUL = \
	"ScanLoop:	lodsb" \
	"				cmp		al,'A'" \
	"				jb			ScanTest" \
	"				cmp		al,'Z'" \
	"				ja			ScanTest" \
	"				add		al,20h" \
	"ScanTest:	cmp		al,ah" \
	"				je			ScanFound" \
	"				loop		ScanLoop" \
	"				xor		esi,esi" \
	"				jmp		ScanDone" \
	"ScanFound:	dec		esi" \
	"ScanDone:" \
					value		[esi] \
					parm		[esi] [ecx] [ah] \
					modify 	[eax ecx]; 
#else
	// 16-bit case-insensitive scan
	extern PCH ScanUL(PCH pScanUL, UINT uCharCountUL, UCHAR cScanUL);
	#pragma aux ScanUL = \
	"ScanLoop:	lodsb" \
	"				cmp		al,'A'" \
	"				jb			ScanTest" \
	"				cmp		al,'Z'" \
	"				ja			ScanTest" \
	"				add		al,20h" \
	"ScanTest:	cmp		al,ah" \
	"				je			ScanFound" \
	"				loop		ScanLoop" \
	"				xor		si,si" \
	"				jmp		ScanDone" \
	"ScanFound:	dec		si" \
	"ScanDone:" \
					value		[si] \
					parm		[si] [cx] [ah] \
					modify 	[ax cx]; 
#endif


// ---------------------------------------------------------------------
// Scan for a character, taking case sensitivity into account
// Returns match position, or NULL for no match
// ---------------------------------------------------------------------
PCH CharScan(PCH pScan, UINT uCharCount, UCHAR cScan, int fCaseSensitive)
{

	// If this is a case-sensitive scan just scan memory for the single
	// character
	if (fCaseSensitive)
		return (memchr(pScan, cScan, uCharCount));

	// If case-insensitive use the assembler code
	return ScanUL(pScan, uCharCount, cScan);
}


// ---------------------------------------------------------------------
// Truncate a filename and add a suffix
// Returns address of NUL character at end of name in output buffer
// ---------------------------------------------------------------------
PCH TruncateFilename(PCH pszOutBuf, PCH pszFileName, UINT uMaxLen, UINT uPart1Len, PCH pszSuffix, UINT uSuffixLen)
{
	UINT uFNLen = strlen(pszFileName);	// length of file name
	UINT uPart2Len = uMaxLen - uPart1Len - LEADLEN;	// length of part after "...."

	// If it fits as-is, just copy it; Otherwise, copy in pieces with "...."
	// in the middle
	if (uFNLen < uMaxLen)
		memcpy(pszOutBuf, pszFileName, uFNLen + 1);
	else
		sprintf(pszOutBuf, "%.*s%s%.*s%n", uPart1Len, pszFileName, LEADER, uPart2Len, (pszFileName + uFNLen - uPart2Len), &uFNLen);

	// Skip past the filename in the output buffer
	pszOutBuf += uFNLen;

	// Add the suffix
	if (uSuffixLen > 0) {
		memcpy(pszOutBuf, pszSuffix, uSuffixLen);
		pszOutBuf += uSuffixLen;
		*pszOutBuf = '\0';
	}

	return pszOutBuf;
}


// ---------------------------------------------------------------------
// Glue a filename onto a path
// ---------------------------------------------------------------------
void MakeFileName(PSZ pszBuffer, PSZ pszName)
{
	int nLen = strlen(pszBuffer);		// length of current name

	// Append a backslash if there isn't one there
	if (pszBuffer[nLen - 1] != '\\') {
		pszBuffer[nLen] = '\\';
		pszBuffer[nLen + 1] = '\0';
	}

	// Append the name to the path
	strcat(pszBuffer, pszName);
}


