/* cleandsk.c (emx+gcc) (c) 1996 William D. Ezell */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ftw.h>
#include <string.h>
#include <io.h>
#include <fnmatch.h>

#define DEBUG              /* when defined compilation will produce
                              non-destructive version of program */

#undef DEBUG                /* enable production version compile */

#define WORKBUFSIZE 256    /* size for work buffers */

/* prototypes */
static int walker (const char *name, const struct stat *st, int flag);

/* pattern masks designating files to search for -- TERMINATE LIST WITH \0 */
char *refmasks[] = { "*.old",
                     "*.tmp",
                     "*.syd",
                     "*.fix",
                     "*.chk",
                     "*.??_",
                     "*.~??",
                     "~*.*",
                     "\0"
                   };

/* cleandsk: main function */
int main (int argc, char *argv[])
{
   int rc, i;                 /* ftw() return value, misc index var */
   char ans[WORKBUFSIZE];     /* user response to continue prompt */
   char path[WORKBUFSIZE]; /* buffer for formation of drive/path to scan */

   if (argc > 1) {
      /* use command-line argument if present for drive/path specification */
      strcpy (path, argv[1]);
   } else {
      /* otherwise default to root of drive C: (using UNIX-style path sep) */
      strcpy (path, "c:/");
   } 

   /* perform a crude screen clear and display warning */
   for (i=0; i < 15; i++) {
      printf ("\n");
   }
   
   printf ("CLEANDSK 1.0  (c) 1996 William D. Ezell\n");
   printf ("Address comments or suggestions to  wdezell@ibm.net\n\n");
   printf ("WARNING:    This program will delete ALL files which match the following\n");
   printf ("            pattern masks from the drive and path specified, including files\n");
   printf ("            marked as READ-ONLY and all files in subordinate directories!\n\n");
   printf ("Drive/Path: %s\n", path );
   printf ("Files:      *.chk\t*.fix\t*.syd\t*.tmp\t*.??_\t*.~??\t~*.*\n\n");
   printf ("Do you wish to proceed?  Type 'yes' or 'no' and press ENTER:\a ");

   /* pause until user enters 'yes' or 'no' */
   scanf ("%s", &ans);
   while ((strcmp(strlwr(ans), "yes") != 0) && (strcmp(strlwr(ans), "no") != 0)) {
      scanf ("%s", &ans);
   } /* endwhile */

   if (strcmp(strlwr(ans), "yes") == 0) {
      printf("\n\n");
      /* invoke 'ftw()' to apply a function to every file in the directory tree
         of the specified drive beginning at specified directory */
      rc = ftw (path, walker, 10);
      if (rc < 0) {
         /* ftw encountered some kind of error -- display & return non-zero  */
         perror (path);
         return (1);
      }
   } else {
      printf ("Program aborted\n");
      return(1);
   } /* endif */

   return (0);
}

static int walker (const char *name, const struct stat *st, int flag)
{
   int rc, i;               /* return code from fnmatch(), loop index var */
   char fname[WORKBUFSIZE]; /* 'name' portion of current directory entry */
   char *lastsep;           /* pointers to last path separator & 'dot' char */

   /* only process directory entries of type 'file' */
   if (flag == FTW_F) {

      /* extract file name and extension from fully-qualified filename */
      /* find final path separator */
      lastsep = strrchr (name, '/');

      if (lastsep != NULL) {
         /* found separator -- now copy remainder beyond separator to buffer */
         lastsep++;
         strcpy (fname, lastsep);
      } else {
         /* not found -- entry in root (ftw doesn't alter "\" for root) */
         /* look for "\" separator */
         lastsep = strrchr (name, '\\');
         if (lastsep != NULL) {
            /* found -- now copy remainder beyond separator to buffer */
            lastsep++;
            strcpy (fname, lastsep);
         } else {
            /* still not found -- blank file name */
            strcpy (fname, "\0");
         } /* endif */
      } /* endif */
 
      /* compare current file against each pattern mask in array */
      i = 0;
      while (strcmp(refmasks[i], "\0") != 0) {
         rc = fnmatch (refmasks[i], strlwr(fname), _FNM_OS2);

         if (rc == 0) {
            /* current file matches one or more patterns in mask */

            /* clear R-O attribute if set so file can be deleted */
            if (st->st_mode != S_IWRITE) {
               #ifndef DEBUG     /* compiling production version */
               chmod (name, S_IWRITE);
               #endif
            } /* endif */

            #ifndef DEBUG        /* compiling production version */
            /* delete current file */
            if ((rc = remove (name)) == 0) {
               printf ("Deleted:  %s\n", name);
            } else {
               printf ("Could not delete:\t%s\n", name);
            } /* endif */
            #else                /* compiling debug version */
            printf ("%s\t%s\n", ((st->st_mode & S_IWRITE) ? "r-w" : "r-o"), name);
            #endif

            break;
         }
         i++;

      } /* endwhile */

   }
   return (0);
}


