/*
 * pckmap.c MODULE: ibm pc key mapping
 * C Durland    Public Domain
 * John Burnell Added support for grey keypad 3/92
 * John Burnell Added more support for grey keypad 5/92
 * Patrick McPhee Added OS/2 changes
 */
#include "me2.h"
#include "term.h"

static char what[] = "@(#)PC Keymap";

typedef struct { unsigned int scancode; EKeyCode keycode; } PCKeyMap;

    /* The cursor pad between the number pad and the rest of the keyboard.
     */
static PCKeyMap old_key_map[] =
  {
    0x0047, SOFKEY|'A',		/* Home */
    0x004F, SOFKEY|'B',		/* End */
    0x0048, SOFKEY|'C',		/* Up Arrow */
    0x0050, SOFKEY|'D',		/* Down Arrow */
    0x004D, SOFKEY|'E',		/* Right Arrow */
    0x004B, SOFKEY|'F',		/* Left Arrow */
    0x0053, SOFKEY|'H',		/* Delete */
    0x0052, SOFKEY|'G',		/* Insert */
    0x0051, SOFKEY|'J',		/* Page Down */
    0x0049, SOFKEY|'I',		/* Page Up */

    0x0077, SOFKEY|'A' |CTRL,	/* Control Home */
    0x0075, SOFKEY|'B' |CTRL,	/* Control End */
    0x008d, SOFKEY|'C' |CTRL,	/* Up Arrow */
    0x0091, SOFKEY|'D' |CTRL,	/* Down Arrow */
    0x0074, SOFKEY|'E' |CTRL,	/* Right Arrow */
    0x0073, SOFKEY|'F' |CTRL,	/* Left Arrow */
    0x0093, SOFKEY|'H' |CTRL,	/* Delete */
    0x0092, SOFKEY|'G' |CTRL,	/* Insert */
    0x0076, SOFKEY|'J' |CTRL,   /* Control Page Down */
    0x0084, SOFKEY|'I' |CTRL,   /* Control Page Up */

    0	/* end of table */
  };

static PCKeyMap extended_key_map[] =
  {
    0x0003,  '\0',		/* Control @ */

	/* and now a few ALT mappings */
    0x0081, META|'0', /* '0' not in ascii order. 1..9 mapped in map_key() */
    0x0078, META|'!',
    0x001E, META|'A',
    0x0030, META|'B',
    0x002E, META|'C',
    0x0020, META|'D',
    0x0012, META|'E',
    0x0021, META|'F',
    0x0022, META|'G',
    0x0023, META|'H',
    0x0017, META|'I',
    0x000F, META|'I',		/* shift tab */
    0x0024, META|'J',
    0x0025, META|'K',
    0x0026, META|'L',
    0x0032, META|'M',
    0x0031, META|'N',
    0x0018, META|'O',
    0x0019, META|'P',
    0x0010, META|'Q',
    0x0013, META|'R',
    0x001F, META|'S',
    0x0014, META|'T',
    0x0016, META|'U',
    0x002F, META|'V',
    0x0011, META|'W',
    0x002D, META|'X',
    0x0015, META|'Y',
    0x002C, META|'Z',

    0x0044,	  SOFKEY|'0',	/* F10 */
    0x005D, SHIFT|SOFKEY|'0',	/* Shift F10 */
    0x0067, CTRL |SOFKEY|'0',	/* Control F10 */
    0x0071, META |SOFKEY|'0',	/* Alt F10 */

    0x0001, META|CTRL|'[',	/* Alt ESC */
    0x000E, META|CTRL|'H',	/* Alt backspace */
    0x00A5, META|CTRL|'I',	/* Alt tab */
    0x001C, META|CTRL|'M',	/* Alt enter */
    0x0027, META|';',		/* Alt ; */
    0x0028, META|'\'',		/* Alt ' */
    0x0029, META|'`',		/* Alt ` */
    0x002B, META|'\\',		/* Alt \ */
    0x0033, META|',',		/* Alt , */
    0x0034, META|'.',		/* Alt . */
    0x0035, META|'/',		/* Alt / */
    0x0085,	  SOFKEY|'a',	/* F11 */
    0x0087, SHIFT|SOFKEY|'a',	/* Shift   F11 */
    0x0089, CTRL |SOFKEY|'a',	/* Control F11 */
    0x008B, META |SOFKEY|'a',	/* Alt     F11 */
    0x0086,	  SOFKEY|'b',	/* F12 */
    0x0088, SHIFT|SOFKEY|'b',	/* Shift   F12 */
    0x008A, CTRL |SOFKEY|'b',	/* Control F12 */
    0x008C, META |SOFKEY|'b',	/* Alt     F12 */

    0x004C, SOFKEY|'c',		/* middle of cursor pad (5) */
    0x008F, CTRL|SOFKEY|'c', 	/* ALT middle of cursor pad (5) */

    0xE047,	  SOFKEY|'A',	/* Cursor pad Home */
    0xE04F,	  SOFKEY|'B',	/* Cursor pad End */
    0xE048,	  SOFKEY|'C',	/* Cursor pad Cursor Up */
    0xE050,	  SOFKEY|'D',	/* Cursor pad Cursor Down */
    0xE04D,	  SOFKEY|'E',	/* Cursor pad Cursor Right */
    0xE04B,	  SOFKEY|'F',	/* Cursor pad Cursor Left */
    0xE052,	  SOFKEY|'G',	/* Cursor pad Insert */
    0xE053,	  SOFKEY|'H',	/* Cursor pad Delete */
    0xE049,	  SOFKEY|'I',	/* Cursor pad Page Up */
    0xE051,	  SOFKEY|'J',	/* Cursor pad Page Down */
    0xE077, CTRL |SOFKEY|'A',	/* Cursor pad Control Home */
    0xE075, CTRL |SOFKEY|'B',	/* Cursor pad Control End */
    0xE08D, CTRL |SOFKEY|'C',	/* Cursor pad Control Up Arrow */
    0xE091, CTRL |SOFKEY|'D',	/* Cursor pad Control Down Arrow */
    0xE074, CTRL |SOFKEY|'E',	/* Cursor pad Control Right Arrow */
    0xE073, CTRL |SOFKEY|'F',	/* Cursor pad Control Left Arrow */
    0xE092, CTRL |SOFKEY|'G',	/* Cursor pad Control Insert */
    0xE093, CTRL |SOFKEY|'H',	/* Cursor pad control Delete */
    0xE084, CTRL |SOFKEY|'I',	/* Cursor pad control Page Up */
    0xE076, CTRL |SOFKEY|'J',	/* Cursor pad control Page Down */

    0x0047, SOFKEY|'{',		/* Number pad Home */
    0x004F, SOFKEY|'}',		/* Number pad End */
    0x0049, SOFKEY|'[',		/* Number pad Page Up */
    0x0051, SOFKEY|']',		/* Number pad Page Down */
    0x0048, SOFKEY|'^',		/* Number pad Up Arrow */
    0x0050, SOFKEY|'.',		/* Number pad Down Arrow */
    0x004D, SOFKEY|'>',		/* Number pad Right Arrow */
    0x004B, SOFKEY|'<',		/* Number pad Left Arrow */
    0x0052, SOFKEY|'&',		/* Number pad Insert */
    0x0053, SOFKEY|'#',		/* Number pad Delete */

    0xE00D, SOFKEY|'$',		/* Number pad Enter */
    0xE02A, SOFKEY|'*',		/* Number pad *  */
    0xE02B, SOFKEY|'+',		/* Number pad +  */
    0xE02D, SOFKEY|'-',		/* Number pad -  */
    0xE02F, SOFKEY|'/',		/* Number pad /  */

    0x0077, SOFKEY|'{' |CTRL,	/* Number pad Control Home */
    0x0075, SOFKEY|'}' |CTRL,	/* Number pad Control End */
    0x0084, SOFKEY|'[' |CTRL,	/* Number pad Control Page Up */
    0x0076, SOFKEY|']' |CTRL,	/* Number pad Control Page Down */
    0x008D, SOFKEY|'^' |CTRL,	/* Number pad Control Up Arrow */
    0x0091, SOFKEY|'.' |CTRL,	/* Number pad Control Down Arrow */
    0x0074, SOFKEY|'>' |CTRL,	/* Number pad Control Right Arrow */
    0x0073, SOFKEY|'<' |CTRL,	/* Number pad Control Left Arrow */
    0x0092, SOFKEY|'&' |CTRL,	/* Number pad Control Insert */
    0x0093, SOFKEY|'#' |CTRL,	/* Number pad Control Delete */

    0	/* end of table */
  };

static int look_for_key(key_map, scancode, keycode)
  PCKeyMap *key_map; unsigned int scancode; EKeyCode *keycode;
{
  for (; key_map->scancode; key_map++)
    if (key_map->scancode == scancode)  /* found it */
    { *keycode = key_map->keycode; return TRUE; }

  return FALSE;
}

    /* Convert terminal codes to ME codes.
     * Input:
     *   scancode:  The scan code returned from Kbd fns.  Normally just a
     *     byte but can be 2 bytes (for example with some of the Cursor pad
     *     (grey on the extended keyboard) keypad keys).
     *   keycode:  pointer to a EKeyCode that hold the converted key.
     * Changes: *keycode
     * Returns:
     *   TRUE:  Everything went as expected.
     *   FALSE: Unreconized scancode.
     */
map_key(scancode, keycode)
  unsigned int scancode; EKeyCode *keycode;
{
  /* ************* Function keys ************************************ */
  /* Convert Function key 1..9 to SOFKEY 1...9 */
  if (0x3B <= scancode && scancode <= 0x43)
    { *keycode = SOFKEY | (scancode - 0xA); return TRUE; }

 	/* Convert Shift Function key 1..9 to SHIFT SOFKEY 1...9 */
  if (0x54 <= scancode && scancode <= 0x5C)
    { *keycode = SHIFT |  SOFKEY | (scancode - 0x23); return TRUE; }

 	/* Convert Control Function key 1..9 to CTRL SOFKEY 1...9 */
  if (0x5E <= scancode && scancode <= 0x66)
    { *keycode = CTRL | SOFKEY | (scancode - 0x2D); return TRUE; }

 	/* Convert Alt Function key 1..9 to META SOFKEY 1...9 */
  if (0x68 <= scancode && scancode <= 0x70)
    { *keycode = META | SOFKEY | (scancode - 0x37); return TRUE; }

	/* ************* Alt keys ************************************ */
			/* convert Alt 1...9 to META "1"..."9" */
  if (0x78 <= scancode && scancode <= 0x80)
    { *keycode = META | (scancode - 0x47); return TRUE; }


	/* ************* Everybody else ********************************* */
  if (look_for_key(old_key_map,		 scancode, keycode)) return TRUE;
  if (look_for_key(extended_key_map, scancode, keycode)) return TRUE;

  return FALSE;						  /* not found */
}
