/*
    listPM list files under Presentation Manager. Uses Open Class Libarary.
    Copyright (C) 1996  Paul Elliott

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Paul Elliott
    3987 South Gessner #224
    Houston Tx 77063
    Paul.Elliott@Hrnowl.LoneStar.Org
*/
#include <istring.hpp>
#include "findmle.hpp"
/*
search for given expression in a IMultiLineEdit.
Can search in forward or reverse direction.
Exact case match can be required.
*/
Boolean FindMle(
                   IMultiLineEdit& mle,   // multi line edit field
                   Boolean forward,       // if true search forward.
                   Boolean exact,         // exact case search?
                   const IString str)     // string to search for.
{
         // Freeze multi line edit. until released.
         class FreezeMle
         {
             private:
                  IMultiLineEdit& mle;            // store the multiline edit.
             public:
                  FreezeMle(IMultiLineEdit& mle) : mle(mle)
                  {
                      mle.disableUpdate();        // freeze updates on screens.
                      mle.disable();              // freeze changes.
                  };
                  ~FreezeMle()
                  {
                      mle.enableUpdate();         // unfreeze updates.
                      mle.enable();               // unfreeze changes.
                  };
         } freeze(mle);
         // the above object will freeze the MLE until the object
         // is destroyed.

         IString looked_for;              // looking for this string.
         if ( exact )                     // depends on exact case.
            looked_for = str;
         else
            looked_for = IString::upperCase(str);
         // if not exact case then always work with upcased strings.
         // for both target text and examined text.

         // direction sign is +- 1. direction mult is 0 or 1.
         int direction_sign,direction_mult;

         unsigned long limit;              // last line to look at.

         // search member function forward or reverse?
         // dir_search is a pointer to memeber which will be
         // setup to do the search in the correct direction
         // depending on direction. indexOf for forward.
         // last indexOf for reverse.
         unsigned (I0String::*dir_search)
             (const IString& astring , unsigned startPos ) const;

         // default value 2nd par in above depends on direction!
         unsigned start_pos;

         if(forward)
         {
             direction_sign = direction_mult =1;
             dir_search = I0String::indexOf;          // forward search fn
             start_pos = 0;                           // starting pos for fn.

             limit = mle.numberOfLines() - 1 ;        // last line to look at.
         }
         else
         {
             direction_sign = -1;
             direction_mult = 0;
             dir_search = I0String::lastIndexOf;      // reverse search fn.
             start_pos = UINT_MAX - 1;                // starting pos for fn.
             limit = 0;                               // last line to look at.
         };

         // the current line we are on.
         unsigned long current_line_no = mle.cursorLinePosition();

         // save the current position.
         unsigned long current_pos  = mle.cursorPosition();

         // set the cursor to the begining of this line.
         mle.setCursorLinePosition( current_line_no );

         // mark beginning of this line.
         unsigned long current_bol  = mle.cursorPosition();

         // put cursor position back where it was.
         mle.setCursorPosition(current_pos);

         // offset of our cursor in this line.
         unsigned offset_within_current_line = current_pos - current_bol;

         // get text of current line.
         I0String current_line = mle.text(current_line_no);

         // offset of cursor within 1st line.
         unsigned offset;

         // the fragment of line we are looking at.
         I0String line;
         if ( forward )
         {
             // offset of the fragment within the current line
             offset = offset_within_current_line;

             // get the relevant fragment of current line = the end.
             line = current_line.subString( offset_within_current_line );
         }
         else
         {
             // fragment starts at begin
             offset = 0;

             // get the relevant fragment of current line = the beginning.
             line = current_line.subString( 0 , offset_within_current_line );
         };

         // if !exact case work with upcased line.
         if ( ! exact) line = I0String::upperCase(line);

         // the current line number. for loop.
         unsigned long line_no = current_line_no;

         //  search for text.
         while ( ! line.includes( looked_for ) )
         {

            // after first look at whole line.
            offset = 0;
            if (line_no == limit) return false;

            // increment line number.
            line_no += direction_sign;

            // get new line.
            line = mle.text(line_no);

            // if not exact case, work with upcased string.
            if ( ! exact ) line = I0String::upperCase(line);

         };

         // go to begining of found line
         mle.setCursorLinePosition( line_no );

         // first position of the found string.
         IPair::Coord first = mle.cursorPosition() + offset +
              (line.*dir_search)(looked_for , start_pos );

         // end of the found string.
         IPair::Coord last = first +  str.length() - 1;


         // postion past the found string in the direction of travel.
         mle.setCursorPosition ( first + direction_mult * ( str.length() - 1 ) );

         // select the found string.
         // cursor will be positined at second parameter of the IRange.
         if ( forward )
            mle.selectRange( IRange(first,last) );
         else
            mle.selectRange( IRange(last,first) );
         // we did found it.
         return true;
};
