/*
** Module   :RX.CPP
** Abstract :
**
** Copyright (C) Sergey I. Yevtushenko
**
** Log: Fri  10/04/1998 Created
**
*/

#include <string.h>

#include <boxcoll.h>
#include <stddlg.h>
#include <keynames.h>
#include <fio.h>
#include <varpool.h>

#define INCL_REXXSAA
#define INCL_DOS

#ifdef __EMX__
extern "C" {
#include <os2emx.h>
}
#else
#include <rexxsaa.h>
#endif

/*************************************************************
** Data Types
*/

typedef struct
{
    char *pszName;
    char internalFunc;
    PFN pfnFunc;
} FuncListItem;

/*************************************************************
** Function Prototypes
*/

ULONG APIENTRY fedNoparm  (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedGet     (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedSet     (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedFind    (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedInsChar (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedGetChar (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedOpen    (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedJumpList(PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedMsgBox  (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);

ULONG APIENTRY fedVarSet  (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedVarGet  (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedVarDrop (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);

ULONG APIENTRY fedInput   (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedGetClip (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedSetClip (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);

ULONG APIENTRY fedBind    (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedMenu    (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);
ULONG APIENTRY fedSendKey (PSZ, ULONG, PRXSTRING, PSZ, PRXSTRING);

int makeRC   (PRXSTRING, LONG );
int makeRCstr(PRXSTRING, char*);
int makeRCstr(PRXSTRING, int  );

ULONG ULfromParm(PRXSTRING rxArg);
ULONG ULfromSTR(char *str);

APIRET GetRexxVariable(char* name, char* value, LONG len);
static char* mk_var(char *buff, char *beg, int num, char *end, int mode);

static VarPool RexxVars;

/*************************************************************
** Static Data
*/

extern EditBoxCollection Editor;
extern JumpList JList[];


FuncListItem FunctionList[]=
{
    { "fedAbort",            FUNC_ABORT             , (PFN)fedNoparm },
    { "fedBksp",             FUNC_BKSP              , (PFN)fedNoparm },
    { "fedClose",            FUNC_CLOSE             , (PFN)fedNoparm },
    { "fedCopy",             FUNC_COPY              , (PFN)fedNoparm },
    { "fedCopyright",        FUNC_COPYRIGHT2        , (PFN)fedNoparm },
    { "fedCut",              FUNC_CUT               , (PFN)fedNoparm },
    { "fedDel",              FUNC_DEL               , (PFN)fedNoparm },
    { "fedDelline",          FUNC_DELLINE           , (PFN)fedNoparm },
    { "fedDeltoeol",         FUNC_DELTOEOL          , (PFN)fedNoparm },
    { "fedDelwordleft",      FUNC_DELWORDLEFT       , (PFN)fedNoparm },
    { "fedDelwordright",     FUNC_DELWORDRIGHT      , (PFN)fedNoparm },
    { "fedDown",             FUNC_DOWN              , (PFN)fedNoparm },
    { "fedDownmark",         FUNC_DOWNMARK          , (PFN)fedNoparm },
    { "fedDupLine",          FUNC_DUPLICATE_LINE    , (PFN)fedNoparm },
    { "fedEnd",              FUNC_END               , (PFN)fedNoparm },
    { "fedEndmark",          FUNC_ENDMARK           , (PFN)fedNoparm },
    { "fedExit",             FUNC_EXIT              , (PFN)fedNoparm },
    { "fedFilebegin",        FUNC_FILEBEGIN         , (PFN)fedNoparm },
    { "fedFilebeginmark",    FUNC_FILEBEGINMARK     , (PFN)fedNoparm },
    { "fedFileend",          FUNC_FILEEND           , (PFN)fedNoparm },
    { "fedFileendmark",      FUNC_FILEENDMARK       , (PFN)fedNoparm },
    { "fedFilelist",         FUNC_FILELIST          , (PFN)fedNoparm },
    { "fedFillJumpList",     0                      , (PFN)fedJumpList},
    { "fedFlipautoindent",   FUNC_FLIPAUTOINDENT    , (PFN)fedNoparm },
    { "fedFlipblockmode",    FUNC_FLIPBLOCKMODE     , (PFN)fedNoparm },
    { "fedFliphiliting",     FUNC_FLIPHILITING      , (PFN)fedNoparm },
    { "fedHelpscreen",       FUNC_HELPSCREEN        , (PFN)fedNoparm },
    { "fedHome",             FUNC_HOME              , (PFN)fedNoparm },
    { "fedHomemark",         FUNC_HOMEMARK          , (PFN)fedNoparm },
    { "fedIndent",           FUNC_INDENT            , (PFN)fedNoparm },
    { "fedIns",              FUNC_INS               , (PFN)fedNoparm },
    { "fedInsdate",          FUNC_INSDATE           , (PFN)fedNoparm },
    { "fedInsfilename",      FUNC_INSFILENAME       , (PFN)fedNoparm },
    { "fedInsfilenameshort", FUNC_INSFILENAMESHORT  , (PFN)fedNoparm },
    { "fedJumpcol",          FUNC_JUMPCOL           , (PFN)fedNoparm },
    { "fedJumpline",         FUNC_JUMPLINE          , (PFN)fedNoparm },
    { "fedLeft",             FUNC_LEFT              , (PFN)fedNoparm },
    { "fedLeftmark",         FUNC_LEFTMARK          , (PFN)fedNoparm },
    { "fedLoad",             FUNC_LOAD              , (PFN)fedNoparm },
    { "fedLower",            FUNC_LOWER             , (PFN)fedNoparm },
    { "fedMacrorecend",      FUNC_MACRORECEND       , (PFN)fedNoparm },
    { "fedMacrorecstart",    FUNC_MACRORECSTART     , (PFN)fedNoparm },
    { "fedMatchbracket",     FUNC_MATCHBRACKET      , (PFN)fedNoparm },
    { "fedMatchbracketmark", FUNC_MATCHBRACKETMARK  , (PFN)fedNoparm },
    { "fedMsgBox",           0                      , (PFN)fedMsgBox },
    { "fedMenu",             0                      , (PFN)fedMenu   },
    { "fedNew",              FUNC_NEW               , (PFN)fedNoparm },
    { "fedNextfile",         FUNC_NEXTFILE          , (PFN)fedNoparm },
    { "fedOpenJumpList",     0                      , (PFN)fedJumpList},
    { "fedOpenFile",         0                      , (PFN)fedOpen   },
    { "fedPaste",            FUNC_PASTE             , (PFN)fedNoparm },
    { "fedPgdn",             FUNC_PGDN              , (PFN)fedNoparm },
    { "fedPgdnmark",         FUNC_PGDNMARK          , (PFN)fedNoparm },
    { "fedPgup",             FUNC_PGUP              , (PFN)fedNoparm },
    { "fedPgupmark",         FUNC_PGUPMARK          , (PFN)fedNoparm },
    { "fedPrevfile",         FUNC_PREVFILE          , (PFN)fedNoparm },
    { "fedRight",            FUNC_RIGHT             , (PFN)fedNoparm },
    { "fedRightmark",        FUNC_RIGHTMARK         , (PFN)fedNoparm },
    { "fedSave",             FUNC_SAVE              , (PFN)fedNoparm },
    { "fedSaveall",          FUNC_SAVEALL           , (PFN)fedNoparm },
    { "fedSaveas",           FUNC_SAVEAS            , (PFN)fedNoparm },
    { "fedSearch",           FUNC_SEARCH            , (PFN)fedNoparm },
    { "fedSearchAgain",      FUNC_SEARCHAGAIN       , (PFN)fedNoparm },
    { "fedSort",             FUNC_SORT              , (PFN)fedNoparm },
    { "fedUndo",             FUNC_UNDO              , (PFN)fedNoparm },
    { "fedUnindent",         FUNC_UNINDENT          , (PFN)fedNoparm },
    { "fedUp",               FUNC_UP                , (PFN)fedNoparm },
    { "fedUpmark",           FUNC_UPMARK            , (PFN)fedNoparm },
    { "fedUpper",            FUNC_UPPER             , (PFN)fedNoparm },
    { "fedWordleft",         FUNC_WORDLEFT          , (PFN)fedNoparm },
    { "fedWordleftmark",     FUNC_WORDLEFTMARK      , (PFN)fedNoparm },
    { "fedWordright",        FUNC_WORDRIGHT         , (PFN)fedNoparm },
    { "fedWordrightmark",    FUNC_WORDRIGHTMARK     , (PFN)fedNoparm },
    { "fedInsChar",          0                      , (PFN)fedInsChar},
    { "fedGetChar",          0                      , (PFN)fedGetChar},
    { "fedSet",              0                      , (PFN)fedSet    },
    { "fedGet",              0                      , (PFN)fedGet    },
    { "fedFind",             0                      , (PFN)fedFind   },
    { "fedVarSet",           0                      , (PFN)fedVarSet },
    { "fedVarGet",           0                      , (PFN)fedVarGet },
    { "fedVarDrop",          0                      , (PFN)fedVarDrop},
    { "fedInput",            0                      , (PFN)fedInput  },
    { "fedGetClip",          0                      , (PFN)fedGetClip},
    { "fedSetClip",          0                      , (PFN)fedSetClip},
    { "fedBind"   ,          0                      , (PFN)fedBind   },
    { "fedSendKey",          0                      , (PFN)fedSendKey},
    { 0, 0, 0}

};

Dictionary *pFuncs = 0;

void InitREXX(void)
{
    if(!pFuncs)
    {
        int i;
        pFuncs = new Dictionary(0,0,0);

        for(i = 0; FunctionList[i].pszName; i++)
        {
            pFuncs->Add(&FunctionList[i]);
            RexxRegisterFunctionExe(FunctionList[i].pszName,
#ifdef __EMX__
                                    (RexxFunctionHandler*)FunctionList[i].pfnFunc);
#else
                                    FunctionList[i].pfnFunc);
#endif
        }
    }
}

//-----------------------------------------------------------------------
// REXX API
//-----------------------------------------------------------------------

ULONG APIENTRY fedInput(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    char *hdr = 0;
    char *res = 0;
    int r = 5;
    int c = 5;
    int rc;

    if(lCnt > 3)
        return 40;

    if(lCnt > 0)
        hdr = (char*)rxArgv[0].strptr;

    if(lCnt > 1)
        r = ULfromParm(&rxArgv[1]);

    if(lCnt > 2)
        c = ULfromParm(&rxArgv[2]);

    rc = AskString(r, c, &res, hdr);

    if(!rc)
    {
        int Len = strlen(res);

        makeRCstr(prxRC, Len + 1);

        strcpy((char*)prxRC->strptr, res);
        prxRC->strlength = Len;

        delete res;
        return 0;
    }

    prxRC->strlength = 0;
    return 0;
}

ULONG APIENTRY fedGetClip(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt)
        return 40;

    if(!Clipboard)
    {
        prxRC->strlength = 0;
        return 0;
    }

    Clipboard->from_pm();

    char *res = Clipboard->as_text();

    prxRC->strlength = strlen(res);
    prxRC->strptr    = (PCH)res;

    return 0;
}

ULONG APIENTRY fedSetClip(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt != 1 || !rxArgv[0].strptr)
        return 40;

    if(!Clipboard)
        return 0;

    Clipboard->from_text((char*)rxArgv[0].strptr);
    Clipboard->to_pm();

    return 0;
}

ULONG APIENTRY fedSendKey(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt != 1 || !rxArgv[0].strlength || !rxArgv[0].strptr)
        return 40;

    Editor.SendKey(rxArgv[0].strptr);

    prxRC->strlength = 0;
    return 0;
}

ULONG APIENTRY fedBind(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt)
    {
        if(!rxArgv[0].strptr || !rxArgv[0].strlength)
            return 40;

        char *file = _ld_file((char*)rxArgv[0].strptr);

        if(file)
            Editor.load_profile(file);

        _fr_file(file);
    }
    else
        Editor.doLoadProfile();

    prxRC->strlength = 0;
    return 0;
}

ULONG APIENTRY fedVarSet(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt < 1 || lCnt > 2)
        return 40;

    if(lCnt == 1)
        RexxVars.del((char*)rxArgv[0].strptr);
    else
        RexxVars.set((char*)rxArgv[0].strptr, (char*)rxArgv[1].strptr);

    prxRC->strlength = 0;
    return 0;
}

ULONG APIENTRY fedVarGet(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt < 1 || lCnt > 2)
        return 40;

    char *var = RexxVars.get((char*)rxArgv[0].strptr);

    if(var)
    {
        int mLen = strlen(var);
        if(mLen > prxRC->strlength)
        {
            if (DosAllocMem((PPVOID)&prxRC->strptr, mLen, PAG_COMMIT | PAG_WRITE))
            {
                prxRC->strlength = 0;
                return 0;
            }
        }
        strcpy((char*)prxRC->strptr, var);
    }
    else
        if(lCnt == 1)
            strcpy((char*)prxRC->strptr, strupr((char*)rxArgv[0].strptr));
    	else
        	if(lCnt == 2)
                strcpy((char*)prxRC->strptr, (char*)rxArgv[1].strptr);

    prxRC->strlength = strlen((char*)prxRC->strptr);
    return 0;
}

ULONG APIENTRY fedVarDrop(PSZ pName, ULONG lCnt, PRXSTRING rxArgv, PSZ, PRXSTRING prxRC)
{
    if(lCnt != 1)
        return 40;

    RexxVars.drop((char*)rxArgv[0].strptr);

    prxRC->strlength = 0;
    return 0;
}

//---------------------------
// fedOpen: open file with given name.
ULONG APIENTRY fedOpen(PSZ,
                       ULONG lCnt,
                       PRXSTRING rxArgv,
                       PSZ,
                       PRXSTRING prxRC)
{
    if(lCnt > 1)
        return makeRC(prxRC, -1);

    if(!lCnt || !rxArgv[0].strptr || !rxArgv[0].strlength ||
       strchr((char*)rxArgv[0].strptr, '?') ||
       strchr((char*)rxArgv[0].strptr, '*'))
    {
        Editor.open();
        return makeRC(prxRC, 1);
    }
    else
        Editor.doOpenFile((char*)rxArgv[0].strptr);

    return makeRC(prxRC, 0);
}

ULONG APIENTRY fedMsgBox(PSZ,
                         ULONG lCnt,
                         PRXSTRING rxArgv,
                         PSZ,
                         PRXSTRING prxRC)
{
    if(lCnt < 1 || lCnt > 2 || !rxArgv[0].strptr || !rxArgv[0].strlength)
        return makeRC(prxRC, -1);

    int timeout = 0;

    if(lCnt > 1)
        timeout = ULfromParm(&rxArgv[1]);

    MessageBox((char*)rxArgv[0].strptr, 0, timeout);

    return makeRC(prxRC, 0);
}

//---------------------------
// fedNoparm: rexx API for calls without parameters

ULONG APIENTRY fedNoparm(PSZ pName,
                         ULONG lCnt,
                         PRXSTRING,
                         PSZ,
                         PRXSTRING prxRC)
{
    if(lCnt != 0) //No arguments should be given
        return makeRC(prxRC, -1);

    FuncListItem* pItem = (FuncListItem*)pFuncs->IsIn((char*)pName,0);

    if(pItem && pItem->internalFunc)
    {
        char macro[3];
        macro[0] = FUNC_ESCAPE;
        macro[1] = pItem->internalFunc;
        macro[2] = 0;
        Editor.play_macro(macro);
    }

    return makeRC(prxRC, 0);
}

ULONG APIENTRY fedInsChar(PSZ,
                          ULONG lCnt,
                          PRXSTRING rxArgv,
                          PSZ,
                          PRXSTRING prxRC)
{
    if(lCnt != 1) //One argument should be given
        return makeRC(prxRC, -1);

    Editor.play_macro((char*)rxArgv[0].strptr);
    return makeRC(prxRC, 0);
}

ULONG APIENTRY fedGetChar(PSZ,
                          ULONG lCnt,
                          PRXSTRING rxArgv,
                          PSZ,
                          PRXSTRING prxRC)
{
    //If no arguments given,
    //return current string
    //With one argument, return full string
    //With two arguments, return rest of string from given pos
    //With three arguments, return part of string from given pos;

    int Row = Editor.current()->abs_row();
    int Col = 0;
    int Len = -1;

    switch(lCnt)
    {
        case 3:
            if(rxArgv[2].strlength != 0) //Length
            {
                Len = ULfromParm(&rxArgv[2]);

                if(Len == 0)
                	return makeRC(prxRC, -1);
            }

        case 2:

            if(rxArgv[1].strlength != 0) //Col
            {
                Col = ULfromParm(&rxArgv[1]);

                if(Col == 0)
                    return makeRC(prxRC, -2);

                Col--;

                if(Len < 0)
                    Len = -2;
            }

        case 1:

            if(rxArgv[0].strlength != 0) //Row
            {
                Row = ULfromParm(&rxArgv[0]);

                if(Row == 0)
                    return makeRC(prxRC, -3);
            }

            Row--;

        case 0:

            /*
            prxRC->strlength = (Editor.current()->get_cur_char()) ? 1:0;
            prxRC->strptr[0] = (char)Editor.current()->get_cur_char();
            prxRC->strptr[1] = '\0';
            break;
            */

            if(Editor.current()->Count() < Row)
                return makeRC(prxRC, -1);

            if(Len == -2)
                Len = Editor.current()->line(Row)->len() - Col;

            if(Len == -1)
                Len = Editor.current()->line(Row)->len();

            makeRCstr(prxRC, Len);

            Editor.current()->line(Row)->get_print(Col,
                                                   (char*)prxRC->strptr,
                                                   Len);
            prxRC->strlength = Len;
            break;

        default:
            return makeRC(prxRC, -4);
    }

    return 0;
}


ULONG APIENTRY fedGet(PSZ,
                      ULONG lCnt,
                      PRXSTRING rxArgv,
                      PSZ,
                      PRXSTRING prxRC)
{
    if(lCnt != 1 || rxArgv[0].strlength < 1)
        return makeRC(prxRC, -1);

    //Supported values:
    //  Ins    - Insert mode
    //  Hi     - Hiliting mode
    //  Auto   - Autoindent state
    //  Mark   - Marking state
    //  Row    - Current absolute Row
    //  Col    - Current absolute Col
    //   CH      - Changed status
    //   CR      - Cursor Row
    //   CC      - Cursor Col
    //  CP     - Code Page
    //  Buffer - Current buffer handle
    //  BR     - Marking position start row
    //  BC     - Marking position start col
    //  ? BufferCount - Number of buffers
    //  Name   - Current buffer name
    //  XSize  - Rect size X (Cols)
    //  YSize  - Rect size Y (Rows)
    //  Last   - Last row number
    //  Dir    - Startup dir
    //  WW     - WordWrap Width
    //  WS     - WordWrap Status
    //  WM     - WordWrap Merge
    //  WP     - WordWrap Para

    switch(__to_upper(rxArgv[0].strptr[0]))
    {
        case 'I': return makeRC(prxRC, Editor.current()->get_ins_mode());
        case 'H': return makeRC(prxRC, Editor.current()->get_hiliting()   );
        case 'A': return makeRC(prxRC, Editor.current()->get_auto_indent());
        case 'M': return makeRC(prxRC, Editor.current()->get_mark_state() );
        case 'R': return makeRC(prxRC, Editor.current()->get_edit_row()   );
        case '?': return makeRC(prxRC, (ULONG)Editor.Count());
        case 'X': return makeRC(prxRC, Editor.current()->cols );
        case 'Y': return makeRC(prxRC, Editor.current()->rows );
        //String return
        case 'N': return makeRCstr(prxRC, Editor.current()->get_name());
        case 'D': return makeRCstr(prxRC, StartupDir);

        case 'B':
            if(rxArgv[0].strlength < 2)
                return makeRC(prxRC, (ULONG)Editor.current());

            switch(__to_upper(rxArgv[0].strptr[1]))
            {
                case 'R':
                    if(!Editor.current()->get_mark_state())
                        return makeRC(prxRC, Editor.current()->get_edit_row());

                    return makeRC(prxRC, Editor.current()->get_edit_row_2());

                case 'C':
                    if(!Editor.current()->get_mark_state())
                        return makeRC(prxRC, Editor.current()->get_edit_col());

                    return makeRC(prxRC, Editor.current()->get_edit_col_2());
            }
            break;

        //Possible other cases
        case 'C':
            if(rxArgv[0].strlength < 2)
                return makeRC(prxRC, Editor.current()->get_edit_col());

            switch(__to_upper(rxArgv[0].strptr[1]))
            {
                case 'H': return makeRC(prxRC, Editor.current()->get_changed());
                case 'R': return makeRC(prxRC, Editor.current()->get_cur_row());
                case 'C': return makeRC(prxRC, Editor.current()->get_cur_col());
                case 'P': return makeRCstr(prxRC, Editor.current()->get_cur_cp());
            }
            break;

        case 'W':
            if(rxArgv[0].strlength != 2)
                break;

            switch(__to_upper(rxArgv[0].strptr[1]))
            {
                case 'W': return makeRC(prxRC, Editor.current()->ww_get_width());
                case 'S': return makeRC(prxRC, (Editor.current()->ww_get() & WW_STATE) ? 1:0);
                case 'M': return makeRC(prxRC, (Editor.current()->ww_get() & WW_MERGE) ? 1:0);
                case 'P': return makeRC(prxRC, (Editor.current()->ww_get() & WW_LONG ) ? 1:0);
            }
            break;


        case 'L':
            if(__to_upper(rxArgv[0].strptr[1]) == 'C')  //Last char
                return makeRCstr(prxRC, kiLastKey.KeyName);

            return makeRC(prxRC, Editor.current()->Count());

        default:
            break;
    }
    return makeRC(prxRC, -2);
}

ULONG APIENTRY fedSet(PSZ,
                      ULONG lCnt,
                      PRXSTRING rxArgv,
                      PSZ,
                      PRXSTRING prxRC)

{
    if(lCnt < 1 || rxArgv[0].strlength < 1)
        return makeRC(prxRC, -1);

    //Supported values:
    //  Ins    - Insert mode               (flip)
    //  Hi     - Hiliting mode             (flip)
    //  Auto   - Autoindent state          (flip)
    //  Mark   - Marking state             1 | 0
    //  Row    - Current absolute Row
    //  Col    - Current absolute Col
    //  Page   - Codepage
    //  Buffer - switch to buffer with handle
    //  Name   - Current buffer name
    //  WW     - WordWrap Width
    //  WS     - WordWrap Status
    //  WM     - WordWrap Merge
    //  WP     - WordWrap Para

    switch(__to_upper(rxArgv[0].strptr[0]))
    {
        case 'I':
                if(lCnt == 2 && rxArgv[1].strlength == 1)
                {
                    switch(rxArgv[1].strptr[0])
                    {
                        case '0':
                        case '1':
                            Editor.current()->set_ins_mode(rxArgv[1].strptr[0] - '0');
                            return makeRC(prxRC, 0);

                        default:
                            return makeRC(prxRC, -3);
                    }
                }
                if(lCnt == 2)
                    return makeRC(prxRC, -3);

                Editor.current()->set_ins_mode(1 - Editor.current()->get_ins_mode());

                return makeRC(prxRC, 0);

        case 'A':

                if(lCnt == 2 && rxArgv[1].strlength == 1)
                {
                    switch(rxArgv[1].strptr[0])
                    {
                        case '0':
                        case '1':
                            Editor.current()->set_auto_indent(rxArgv[1].strptr[0] - '0');
                            return makeRC(prxRC, 0);

                        default:
                            return makeRC(prxRC, -4);
                    }
                }
                if(lCnt == 2)
                    return makeRC(prxRC, -4);

                Editor.current()->set_auto_indent(1 - Editor.current()->get_auto_indent());

                return makeRC(prxRC, 0);

        case 'H':

                if(lCnt == 2 && rxArgv[1].strlength == 1)
                {
                    switch(rxArgv[1].strptr[0])
                    {
                        case '0': case '1': case '2': case '3': case '4':
                        case '5': case '6': case '7': case '8': case '9':
                            Editor.current()->set_hiliting(rxArgv[1].strptr[0] - '0');
                            return makeRC(prxRC, 0);

                        default:
                            return makeRC(prxRC, -5);
                    }
                }
                if(lCnt == 2)
                    return makeRC(prxRC, -5);

                Editor.current()->flip_hiliting();

                return makeRC(prxRC, 0);


        //For listed below options, additional parameter is required
        case 'M':

                if(lCnt == 1)
                {
                    Editor.current()->unmark();
                    return makeRC(prxRC, 0);
                }

                if(rxArgv[1].strlength == 1)
                {
                    if(rxArgv[1].strptr[0] == '1')
                    {
                        Editor.current()->mark();
                        return makeRC(prxRC, 0);
                    }
                    else
                    {
                        Editor.current()->unmark();
                        return makeRC(prxRC, 0);
                    }
                }
                return makeRC(prxRC, -6);

        case 'R':
                if(lCnt == 2 && rxArgv[1].strlength > 0)
                {
                    ULONG ulData = ULfromParm(&rxArgv[1]);
                    if(ulData)
                    {
                        Editor.current()->goto_line(*Editor.current(), ulData - 1);
                        return makeRC(prxRC, 0);
                    }
                }

                return makeRC(prxRC, -7);

        case 'C':

                if(lCnt == 2 && rxArgv[1].strlength > 0)
                {
                    ULONG ulData = ULfromParm(&rxArgv[1]);
                    if(ulData)
                    {
                        Editor.current()->goto_col(*Editor.current(), ulData - 1);
                        return makeRC(prxRC, 0);
                    }
                }

                return makeRC(prxRC, -8);

        case 'B':
                if(lCnt == 2 && rxArgv[1].strlength > 0)
                {
                    EditBox* new_box = (EditBox*)ULfromParm(&rxArgv[1]);
                    if(new_box)
                    {
                        Editor.select(new_box);
                        return makeRC(prxRC, 0);
                    }
                }

                return makeRC(prxRC, -9);

        case 'N':
                if(lCnt == 2 && rxArgv[1].strlength > 0)
                    Editor.current()->set_name((char*)rxArgv[1].strptr);
                else
                    return makeRC(prxRC, -10);

                return makeRC(prxRC, 0);

        case 'P':
                if(lCnt == 2 && rxArgv[1].strlength > 0)
                    Editor.current()->set_xlate((char*)rxArgv[1].strptr);
                else
                    return makeRC(prxRC, -11);

                return makeRC(prxRC, 0);

        case 'W':
                {
	                if(rxArgv[0].strlength != 2 || lCnt != 2)
    	               break;

                    Buffer* p = Editor.current();
                    int flags = p->ww_get();
                    int fnew  = 0;
                    int val   = ULfromParm(&rxArgv[1]);

        	        switch(__to_upper(rxArgv[0].strptr[1]))
            	    {
                	    case 'W':
                            p->ww_set_width(val);
                        	return makeRC(prxRC, 0);

                        case 'S': fnew = WW_STATE; break;
                        case 'M': fnew = WW_MERGE; break;
                        case 'P': fnew = WW_LONG ; break;
                    }

                    if(val)
                        flags |= fnew;
                    else
                        flags &= ~fnew;

                    p->ww_set(flags);

                    return makeRC(prxRC, 0);
                }
        	    break;
    }

    return makeRC(prxRC, -2);
}

ULONG APIENTRY fedFind(PSZ,
                       ULONG lCnt,
                       PRXSTRING rxArgv,
                       PSZ,
                       PRXSTRING prxRC)
{
    if(lCnt != 2)
        return makeRC(prxRC, -1);

    int match_len = Editor.current()->search((char*)rxArgv[1].strptr,
                                             (char*)rxArgv[0].strptr);

    return makeRC(prxRC, match_len);
}

ULONG APIENTRY fedMenu(PSZ,
                       ULONG lCnt,
                       PRXSTRING rxArgv,
                       PSZ,
                       PRXSTRING prxRC)
{
    int r = 0;
    int c = 0;

    if(lCnt < 3 ||
       !rxArgv[0].strlength ||
       !rxArgv[1].strlength ||
       !rxArgv[2].strlength)
    {
        return makeRC(prxRC, -1);
    }

    r = ULfromSTR(rxArgv[0].strptr);
    c = ULfromSTR(rxArgv[1].strptr);

    int choice = -1;
    int i;

    Dialog dlg(r, c, 1, 1);
    Menu* menu = new Menu(1, 1, 0);

    for(i = 2; i < lCnt; i++)
    {
        if(rxArgv[i].strlength)
            menu->add_menu(rxArgv[i].strptr);
    }

    dlg.Ins(menu);
    dlg.rows = menu->rows+2;
    dlg.cols = menu->cols+2;

    KeyInfo k;
    do
    {
        dlg.draw();
        vio_read_key(&k);

        if((k.skey & 0x00FF) == kbEnter || (k.skey & 0x00FF) == kbGrEnter)
        {
            choice = menu->first_selected();
            break;
        }

        dlg.do_key(k);

        if(menu->selected_by_key())
        {
            choice = menu->first_selected();
            break;
        }
    }
    while(k.skey != (kbEsc | shIsCtrl));

    return makeRC(prxRC, choice + 1);
}

ULONG APIENTRY fedJumpList(PSZ pName,
                           ULONG lCnt,
                           PRXSTRING rxArgv,
                           PSZ,
                           PRXSTRING prxRC)
{
    char varname[512];
    char varvalue[512];
    char header[512];
    APIRET rc;
    int len;
    int i;
    int list;

    if(lCnt < 1 || rxArgv[0].strlength == 0 || !rxArgv[0].strptr)
        return makeRC(prxRC, -1);

    if(!strcmp((char*)pName, "FEDOPENJUMPLIST"))
    {
        if(lCnt < 1)
            return makeRC(prxRC, -2);

        list = ULfromParm(&rxArgv[0]);

        if(list > 9)
            return makeRC(prxRC, -3);

        int r  = 0;
        int c  = 0;
        int nr = 0;
        int nc = 0;

        if(lCnt >= 2)
            r = ULfromParm(&rxArgv[1]);
        if(lCnt >= 3)
            c = ULfromParm(&rxArgv[2]);
        if(lCnt >= 4)
            nr = ULfromParm(&rxArgv[3]);
        if(lCnt >= 5)
            nc = ULfromParm(&rxArgv[4]);

        return makeRC(prxRC, JumpListBox(&JList[list], r, c, nr, nc));
    }

    if(lCnt != 2)
        return makeRC(prxRC, -2);

    list = ULfromParm(&rxArgv[1]);

    if(list > 9)
        return makeRC(prxRC, -3);

    JList[list].RemoveAll();

    rc = GetRexxVariable(mk_var(varname, (char*)rxArgv[0].strptr,0,"jump_len",0),
                         varvalue, sizeof(varvalue)-1);

    if(rc)
    {
        //mk_var(varname,rxArgv[0].strptr,0,"jump_len not found", 0);
        //MessageBox(varname, 0);
        return makeRC(prxRC, -4);
    }

    len = ULfromSTR(varvalue);

    if(!len)
    {
        return makeRC(prxRC, 0);
    }

    for(i = 0; i < len; i++)
    {
        int row;
        int col;
        APIRET rc2 = 1;

        rc = GetRexxVariable(mk_var(varname,(char*)rxArgv[0].strptr, i, "jump_row", 1),
                             varvalue, sizeof(varvalue)-1);

        if(rc)
        {
            //mk_var(varname,rxArgv[0].strptr, i, "jump_row not found", 1);
            //MessageBox(varname, 0);
            return makeRC(prxRC, -5);
        }

        row = ULfromSTR(varvalue);

        rc = GetRexxVariable(mk_var(varname,(char*)rxArgv[0].strptr, i, "jump_col", 1),
                             varvalue, sizeof(varvalue)-1);

        if(rc)
        {
            //mk_var(varname,rxArgv[0].strptr, i, "jump_col not found", 1);
            //MessageBox(varname, 0);
            return makeRC(prxRC, -6);
        }

        col = ULfromSTR(varvalue);

        rc2 = GetRexxVariable(mk_var(varname,(char*)rxArgv[0].strptr, i, "jump_header", 1),
                             header, sizeof(header)-1);

        rc = GetRexxVariable(mk_var(varname,(char*)rxArgv[0].strptr, i, "jump_file", 1),
                             varvalue, sizeof(varvalue)-1);
        if(rc)
        {
            //mk_var(varname,rxArgv[0].strptr, i, "jump_file not found", 1);
            //MessageBox(varname, 0);
            return makeRC(prxRC, -7);
        }

        if(rc2)
            strcpy(header, varvalue);

        JList[list].add_entry(row, col, header, varvalue);
    }

    return makeRC(prxRC, 0);
}

/*************************************************************
** Utility functions
*/

int makeRC(PRXSTRING prxRC, LONG lData)
{
    char *str ;
    char *buff;
    int  slen = 0;

    buff = (char*)prxRC->strptr;

    if(lData < 0)
    {
        *buff++ = '-';
        slen++;
    }

    str = buff;

    do
    {
        *str++ = (char)((lData % 10) + '0');
        lData /= 10;
        slen++;
    }
    while(lData);

    *str-- = 0;

    for(; str > buff; str--, buff++)
    {
        *buff ^= *str;
        *str  ^= *buff;
        *buff ^= *str;
    }
    prxRC->strlength = slen;
    return 0;
}

int makeRCstr(PRXSTRING prxRC, char *str)
{
    if(!str)
        return 0;

    makeRCstr(prxRC, strlen(str));
    strcpy((char*)prxRC->strptr, str);
    prxRC->strlength = strlen(str);
    return 0;
}

int makeRCstr(PRXSTRING prxRC, int Len)
{
    if(Len > prxRC->strlength)
    {
        DosFreeMem(prxRC->strptr);

        DosAllocMem((PPVOID)&prxRC->strptr,
                     Len,
                     PAG_COMMIT | PAG_WRITE);
    }
    prxRC->strlength = Len;
    return 0;
}

ULONG ULfromSTR(char *str)
{
    ULONG ulRes = 0;

    if(!str)
        return ulRes;
    while(*str)
    {
        if(*str < '0' || *str > '9')
            break;
        ulRes *= 10;
        ulRes += *str - '0';
        str++;
    }
    return ulRes;
}

ULONG ULfromParm(PRXSTRING rxArg)
{
    return ULfromSTR((char*)rxArg->strptr);
}

APIRET GetRexxVariable(char* name, char* value, LONG len)
{
    APIRET rc;
    SHVBLOCK   block;
    block.shvcode = RXSHV_SYFET;
//    block.shvcode = RXSHV_FETCH;
    block.shvret=(UCHAR)0;
    block.shvnext=(PSHVBLOCK)0;

    MAKERXSTRING(block.shvname, name, strlen(name));
    MAKERXSTRING(block.shvvalue, value, len);

    block.shvvaluelen=len;
    rc = RexxVariablePool(&block);
    if(!rc)
        value[block.shvvalue.strlength]=0;
    else
        value[0]=0;
    return rc;
}

char* mk_var(char *buff, char *beg, int num, char *end, int mode)
{
    strcpy(buff, beg);
    if(mode)
    {
        strcat(buff, ".");
        strcat(buff, cvt_num(num,0));
    }
    strcat(buff, ".");
    strcat(buff, end);
    return buff;
}


