/*
** Module   :COMPILE.CPP
** Abstract :Keyboard macro compiler
**
** Copyright (C) Sergey I. Yevtushenko
**
** Log: Fri  27/03/1998     Created
*/

#include <string.h>

#include <boxcoll.h>
#include <version.h>

struct FuncMapping
{
    char *key;
    char func;
};

FuncMapping func_table[]=
{
    {"abort"            ,FUNC_ABORT            },
    {"bind"             ,FUNC_LOAD_KEYS        },
    {"bksp"             ,FUNC_BKSP             },
    {"close"            ,FUNC_CLOSE            },
    {"copy"             ,FUNC_COPY             },
    {"copyright"        ,FUNC_COPYRIGHT2       },
    {"cut"              ,FUNC_CUT              },
    {"del"              ,FUNC_DEL              },
    {"delline"          ,FUNC_DELLINE          },
    {"deltoeol"         ,FUNC_DELTOEOL         },
    {"delwordleft"      ,FUNC_DELWORDLEFT      },
    {"delwordright"     ,FUNC_DELWORDRIGHT     },
    {"down"             ,FUNC_DOWN             },
    {"downmark"         ,FUNC_DOWNMARK         },
    {"dupline"          ,FUNC_DUPLICATE_LINE   },
    {"end"              ,FUNC_END              },
    {"endmark"          ,FUNC_ENDMARK          },
    {"exit"             ,FUNC_EXIT             },
    {"filebegin"        ,FUNC_FILEBEGIN        },
    {"filebeginmark"    ,FUNC_FILEBEGINMARK    },
    {"fileend"          ,FUNC_FILEEND          },
    {"fileendmark"      ,FUNC_FILEENDMARK      },
    {"filelist"         ,FUNC_FILELIST         },
    {"flipautoindent"   ,FUNC_FLIPAUTOINDENT   },
    {"flipblockmode"    ,FUNC_FLIPBLOCKMODE    },
    {"fliphiliting"     ,FUNC_FLIPHILITING     },
    {"fliptype"         ,FUNC_FLIPTYPE         },
    {"flipwordwrap"     ,FUNC_FLIPWORDWRAP     },
    {"flipwordwrapmerge",FUNC_FLIPWWMERGE      },
    {"flipwordwrappara" ,FUNC_FLIPWWLONG       },
    {"helpscreen"       ,FUNC_HELPSCREEN       },
    {"hiliteselect"     ,FUNC_HILITE_ACHOICE   },
    {"home"             ,FUNC_HOME             },
    {"homemark"         ,FUNC_HOMEMARK         },
    {"indent"           ,FUNC_INDENT           },
    {"ins"              ,FUNC_INS              },
    {"insdate"          ,FUNC_INSDATE          },
    {"insfilename"      ,FUNC_INSFILENAME      },
    {"insfilenameshort" ,FUNC_INSFILENAMESHORT },
    {"jumpcol"          ,FUNC_JUMPCOL          },
    {"jumpline"         ,FUNC_JUMPLINE         },
    {"left"             ,FUNC_LEFT             },
    {"leftmark"         ,FUNC_LEFTMARK         },
    {"load"             ,FUNC_LOAD             },
    {"lower"            ,FUNC_LOWER            },
    {"macrorecend"      ,FUNC_MACRORECEND      },
    {"macrorecstart"    ,FUNC_MACRORECSTART    },
    {"markgo0"          ,FUNC_BMK_GET_0        },
    {"markgo1"          ,FUNC_BMK_GET_1        },
    {"markgo2"          ,FUNC_BMK_GET_2        },
    {"markgo3"          ,FUNC_BMK_GET_3        },
    {"markgo4"          ,FUNC_BMK_GET_4        },
    {"markgo5"          ,FUNC_BMK_GET_5        },
    {"markgo6"          ,FUNC_BMK_GET_6        },
    {"markgo7"          ,FUNC_BMK_GET_7        },
    {"markgo8"          ,FUNC_BMK_GET_8        },
    {"markgo9"          ,FUNC_BMK_GET_9        },
    {"markset0"         ,FUNC_BMK_PUT_0        },
    {"markset1"         ,FUNC_BMK_PUT_1        },
    {"markset2"         ,FUNC_BMK_PUT_2        },
    {"markset3"         ,FUNC_BMK_PUT_3        },
    {"markset4"         ,FUNC_BMK_PUT_4        },
    {"markset5"         ,FUNC_BMK_PUT_5        },
    {"markset6"         ,FUNC_BMK_PUT_6        },
    {"markset7"         ,FUNC_BMK_PUT_7        },
    {"markset8"         ,FUNC_BMK_PUT_8        },
    {"markset9"         ,FUNC_BMK_PUT_9        },
    {"matchbracket"     ,FUNC_MATCHBRACKET     },
    {"matchbracketmark" ,FUNC_MATCHBRACKETMARK },
    {"new"              ,FUNC_NEW              },
    {"nextfile"         ,FUNC_NEXTFILE         },
    {"openjumplist0"    ,FUNC_JMP_LST_0        },
    {"openjumplist1"    ,FUNC_JMP_LST_1        },
    {"openjumplist2"    ,FUNC_JMP_LST_2        },
    {"openjumplist3"    ,FUNC_JMP_LST_3        },
    {"openjumplist4"    ,FUNC_JMP_LST_4        },
    {"openjumplist5"    ,FUNC_JMP_LST_5        },
    {"openjumplist6"    ,FUNC_JMP_LST_6        },
    {"openjumplist7"    ,FUNC_JMP_LST_7        },
    {"openjumplist8"    ,FUNC_JMP_LST_8        },
    {"openjumplist9"    ,FUNC_JMP_LST_9        },
    {"paste"            ,FUNC_PASTE            },
    {"pgdn"             ,FUNC_PGDN             },
    {"pgdnmark"         ,FUNC_PGDNMARK         },
    {"pgup"             ,FUNC_PGUP             },
    {"pgupmark"         ,FUNC_PGUPMARK         },
    {"prevfile"         ,FUNC_PREVFILE         },
    {"right"            ,FUNC_RIGHT            },
    {"rightmark"        ,FUNC_RIGHTMARK        },
    {"save"             ,FUNC_SAVE             },
    {"saveall"          ,FUNC_SAVEALL          },
    {"saveas"           ,FUNC_SAVEAS           },
    {"search"           ,FUNC_SEARCH           },
    {"searchagain"      ,FUNC_SEARCHAGAIN      },
    {"setcp"            ,FUNC_SET_XLAT         },
    {"sort"             ,FUNC_SORT             },
    {"undo"             ,FUNC_UNDO             },
    {"unindent"         ,FUNC_UNINDENT         },
    {"up"               ,FUNC_UP               },
    {"upmark"           ,FUNC_UPMARK           },
    {"upper"            ,FUNC_UPPER            },
    {"wordleft"         ,FUNC_WORDLEFT         },
    {"wordleftmark"     ,FUNC_WORDLEFTMARK     },
    {"wordright"        ,FUNC_WORDRIGHT        },
    {"wordrightmark"    ,FUNC_WORDRIGHTMARK    }
};

FunctionDictionary::FunctionDictionary():Dictionary(0,0,0)
{
    for(int i = 0; i < sizeof(func_table)/sizeof(func_table[0]); i++)
        Add(&func_table[i]);
}

static int cx2n(char chr)
{
    int n = 0;

    if(__isdd(chr))
        return chr - '0';

    if(chr >= 'a' || chr <= 'z')
        return chr - 'a' + 0x0A;

    if(chr >= 'A' || chr <= 'Z')
        return chr -'A' + 0x0A;

    return 0;
}

int KeyDefCollection::compile_keydef(char *str, char *out)
{
    int cnt = 0;
    while(*str)
    {
        switch(*str)
        {
            case ' ':
            case ',':
            case '\r':
            case '\n':
            case '\t':
                break;

            case '"':
            case '{':
            case '\'':
                {
                    int iRexx = 0;

                    char delim = *str++;

                    if(delim == '{') //This is rexx?
                    {
                        delim = '}';

                        cnt += 2;

                        if(out)
                        {
                            *out++ = FUNC_ESCAPE;
                            *out++ = FUNC_REXX;
                        }

                        iRexx++;
                    }

                    while(*str)
                    {
                        if(delim != '}' && *str == delim)
                            break;

                        if(delim == '}' && *str == delim && !str[1])
                            break;

                        char chr = *str;

                        if(*str == '\\' && !iRexx) //Escape sequences
                        {
                            str++;

                            switch(*str)
                            {
                                case 'n':
                                    chr = '\n';
                                    break;

                                case 'r':
                                    chr = '\r';
                                    break;

                                case 't':
                                	chr = '\t';
                                    break;

                                case 'x':
                                    {
                                        str++;
                                        chr = 0;
                                        int k;

                                        for(k = 0; k < 2; k++)
                                        {
                                            if(!__ishd(*str))
                                                break;

                                            chr <<= 4;

                                            chr |= cx2n(*str++);
                                        }

                                        if(out)
                                            *out++ = chr++;
                                    }
                                    continue;

                                default:
                                	chr = *str;
                                	break;
                            }
                        }

                        if(out)
                            *out++ = chr;

                        cnt++;
                        str++;
                    }

                    if(*str && delim == '}')
                        return cnt;
                }
                break;

            default:
                if(__isis(*str))
                {
                    char *ptr = str;
                    while(*str && __isic(*str))
                        str++;

                    FuncMapping *pFunc = (FuncMapping *)Func_DIC.IsIn(ptr, str - ptr, 0);

                    if(pFunc)
                    {
                        cnt ++; //bytecode
                        cnt ++; //function code

                        if(out)
                        {
                            *out++ = FUNC_ESCAPE;
                            *out++ = pFunc->func;
                        }
                    }
                    else
                        return -2; //Wrong function name
                }
                else
                    return -1; //Wrong char in keydef
        }
        if(*str)
        	str++;
    }
    return cnt;
}

