/*
** Module   :BUFFER.H
** Abstract :General purpose edit buffer
**
** Copyright (C) Sergey I. Yevtushenko
**
** Log: Sun  06/04/1997   	Created from stripped down EDIT.H
**      Fri  07/11/1997   	Updated for UNDO implementation
*/

#include <line.h>
#include <_ctype.h>

#ifndef  __BUFFER_H
#define  __BUFFER_H

#define WW_STATE        0x01    /* Word wrap state on/off */
#define WW_MERGE        0x02    /* Merge wrapped part with next line */
#define WW_LONG         0x04    /* Reformat until changes are made */

class Buffer;
typedef Buffer* PBuffer;
typedef char (*PerCharFunc)(char);
extern Buffer* Clipboard;

class Buffer: public Collection
{
        int num_cols;
        int col_block;
        int old_abs_col;
        int old_abs_row;
        int start_row;
        int start_col;
        int mark_state;
        int cur_row;
        int cur_col;
        int ins_mode;
        int hiliting;
        int auto_indent;
        int changed;
        int undo_count;
        int found_show;
        int found_row;
        int found_len;
        int found_col;
        int is_unix;
        int custom_syntax;
        int word_wrap;
        int ww_width;

        char cp_in [256];
        char cp_out[256];
        char cur_cp[64];

        struct
        {
	        int start_row;
    	    int start_col;
        } draw_save;

    public:

        int pal_start;
        void ins_line(Line*, int);
        void add_line(Line*);
        Line* del_line(Rect& rect, int line_num);

        void dup_line(Rect& rect, int line_num);

        void fill_hiliting(int start, int initial_state);
        void check_hiliting();

        Line* abs_line()        { return line(abs_row());}
        Parser* gen_parser(int =0);

        struct trackinfo
        {
            int op;
            void *arg1;
            void *arg2;
            trackinfo* next;
        };

        enum track_op
        {
            opAction,
            opCurRow,
            opCurCol,
            opStartRow,
            opStartCol,
            opRestoreLine,
            opDelLine,
            opInsLine,
            opInsBlock,
            opHiliting,
            opMarking,
            opMarkPos,
            opColBlock,
            opInsMode,
            opInsChar,
            opDelChar,
            opRestoreOrder
        };

        struct restore_sort
        {
            int start;
            int len;
            void* ptrs[1];
        };

        int tracking;
        trackinfo *track_head;
        trackinfo *undobuff;

        void track(int op, void *arg1 = 0, void *arg2 = 0);

        void process_block(Rect&, PerCharFunc);
        void init_buffer();

        Buffer(int sz);
        Buffer();

        virtual ~Buffer();

        virtual void Free(Ptr p);

        void draw(Rect&);

        void from_pm();
        void to_pm();

        void from_text(char *);
        char* as_text();

        int abs_row()       { return cur_row + start_row;}
        int abs_col()       { return cur_col + start_col;}

        int get_ins_mode()   { return ins_mode;}
        int get_hiliting()   { return hiliting;}
        int get_mark_state() { return mark_state;}
        int get_auto_indent(){ return auto_indent;}

        int get_cur_row()   { return cur_row;}
        int get_cur_col()   { return cur_col;}
        int get_start_row() { return start_row;}
        int get_start_col() { return start_col;}
        char* get_cur_cp()  { return cur_cp;   }


        int get_changed()   { return changed; }
        int get_unix()      { return is_unix; }
        int get_custom_syntax() { return custom_syntax;}

        int get_edit_row()   { return abs_row() + 1; }
        int get_edit_col()   { return abs_col() + 1; }

        int get_edit_row_2() { return old_abs_row + 1; }
        int get_edit_col_2() { return old_abs_col + 1; }

        int get_cur_char()   { return (unsigned char)abs_line()->char_at(abs_col());}
        int get_rel_pos()    { return abs_row()*100/Count();}

        int get_column_block()  { return col_block;}
        int get_undo_count();

        char chr_in (char c) { return cp_in [c];}
        char chr_out(char c) { return cp_out[c];}

        int do_smart()       { return ((hiliting == HI_CPP ||
                                        hiliting == HI_HTML||
                                        hiliting == HI_PL) &&
                                       auto_indent &&
                                       abs_col()) ? 1:0;}

//-------------------------------------------------------------
        void track_beg();
        void track_end();
        void track_cancel();
        Line* line(int num) { return PLine(Get(num));}

//------------------ Cursor movement
        void text_begin(Rect&);
        void text_end(Rect&);
        void line_begin(Rect&);
        void line_end(Rect&);
        void page_up(Rect&);
        void page_down(Rect&);
        void cursor_up(Rect&);
        void cursor_down(Rect&);
        void cursor_left(Rect&);
        void cursor_right(Rect&);
        void word_left(Rect&);
        void word_right(Rect&);
        void goto_line(Rect&, int);
        void goto_col(Rect&, int);
        void put_mouse(Rect& rect, int new_row, int new_col);
        void match_bracket(Rect&);
        int search(Rect& rect, char* flags, char *pattern);
        int replace(Rect& rect, char *str);

//------------------- Editing
        void ins_char(Rect&, int chr);
        int  del_char(Rect&);
        int  replace_char(Rect&, int chr);
        void del_word_right(Rect&);
        void del_word_left(Rect&);
        void del_to_EOL(Rect&);
        void split_line(Rect&);
        void back_space(Rect&);
        void undo(Rect&);
        void toupper(Rect&);
        void tolower(Rect&);
        void indent();
        void unindent();
        void sort();

//------------------- Clipboard
        Buffer* cut(Rect&);
        Buffer* copy();
        void clear(Rect&);
        void paste(Rect&,Buffer*);
        void paste_over(Rect&,Buffer*);     //Still unimplemented

//------------------- Marking
        void mark();
        void unmark();

//------------------ Other actions
        void set_auto_indent(int mode) { auto_indent = (mode) ? 1:0;}
        void set_column_block(int mode);
        void set_ins_mode(int mode);
        void set_changed(int i = 1);
        void set_found(int start_row, int start_col, int len);
        void set_unix(int mode)        { is_unix = (mode) ? 1:0;}
        void clear_undobuff();
        void set_xlate(char *cp);

        void flip_hiliting();
        void set_hiliting(int mode);
        void set_custom_syntax(int i)  { custom_syntax = i;}

//------------------ Word wrap
        int ww_need(int ln);
        int ww_perform(Rect&, int ln);
        int ww_get()                   { return word_wrap;}
        int ww_get_width()             { return ww_width;}
        void ww_set(int flags)         { word_wrap = flags;}
        void ww_set_width(int width)   { ww_width = (width > 1) ? width:ww_width;}

    protected:
        int is_word_char()       { return __isic(chr_out(abs_line()->char_at(abs_col())));}
};

#endif //__BUFFER_H
