

/*
 *              Author:     Bob Withers     E-Mail: bwit@mo.net
 *              Copyright (c) 1993, All Rights Reserved
 *
 *                              NOTICE
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted
 * provided that the above copyright notice appears in all copies and
 * that both the copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * The author makes no representations about the suitability of this
 * software for any purpose.  This software is provided ``as is''
 * without express or implied warranty.
 *
 *      Define      Operating System        Compiler Tested
 *      -------     ----------------        ---------------
 *      MSDOS_16    DOS                     Microsoft VC++ V1.51
 *      NT_32       Win NT, Win95           Microsoft VC++ V2.1
 *      OS2_16      OS/2 V1+                Microsoft C V6.00
 *      OS2_32      OS/2 V2+                IBM C Set++ V2.1
 *      BORLAND     OS/2                    Borland C++ for OS/2 V1.0
 *      BORLAND     DOS, Win NT             Borland C++ V4.5
 */

#ifndef UNIX

#include <stdlib.h>
#include <string.h>

#if defined(OS2_16) || defined(OS2_32)
#define INCL_DOSFILEMGR
#define INCL_DOSERRORS
#include <os2.h>
#endif

#if defined(OS2_16)
#undef FIL_STANDARD
#define FIL_STANDARD                0
#endif

#if defined(BORLAND)
#define NO_ERROR                    0
#define ERROR_NOT_ENOUGH_MEMORY     -1
#define ERROR_NO_MORE_FILES         ENMFILE
#include <dir.h>
#include <dos.h>
#include <errno.h>
#endif

#if defined(MSDOS_16)
#define NO_ERROR                    0
#define ERROR_NOT_ENOUGH_MEMORY     -1
#define ERROR_NO_MORE_FILES         ENOENT
#include <dos.h>
#include <errno.h>
#endif

#if defined(NT_32)
#define NO_ERROR                    0
#define ERROR_NOT_ENOUGH_MEMORY     -1
#define ERROR_NO_MORE_FILES         ENOENT
#include <io.h>
#include <errno.h>
#endif

#include "dirent.h"

#define DIRENT_INCR                 25


DIR *opendir(char *filename)
{
    auto     DIR               *dirp;
    auto     char              *p;
    auto     int                len;
    auto     int                isdir;

#if  defined(OS2_32)
    auto     APIRET             rc;         /* for 32 bit OS/2          */
    auto     FILEFINDBUF3       ff;
    auto     ULONG              cnt;
    auto     HDIR               hdir;
#endif

#if defined(OS2_16)
    auto     USHORT             rc;         /* for 16 bit OS/2          */
    auto     FILEFINDBUF        ff;
    auto     USHORT             cnt;
    auto     HDIR               hdir;
#endif

#if defined(BORLAND)
    auto     struct ffblk       ff;         /* for Borland C++          */
    auto     int                rc;
#endif

#if defined(MSDOS_16)
    auto     struct _find_t     ff;         /* for Microsoft VC++ 1.52  */
    auto     unsigned int       rc;
#endif

#if defined(NT_32)
    auto     struct _finddata_t ff;
    auto     int                rc;
    auto     long               hdir;
#endif

    if (NULL == filename || '\0' == filename[0])
        filename = ".";

    dirp = malloc(sizeof(*dirp));
    if (NULL == dirp)
        return(NULL);

    len = strlen(filename);
    dirp->dirname = malloc(len + 5);
    if (NULL == dirp->dirname)
    {
        free(dirp);
        return(NULL);
    }

    dirp->max_ent = 0;
    dirp->tot_ent = 0;
    dirp->cur_ent = 0;
    dirp->entp    = NULL;
    strcpy(dirp->dirname, filename);
    for (p = dirp->dirname; *p; ++p)
    {
        if ('/' == *p)
            *p = '\\';
    }

    if ('\\' != dirp->dirname[len - 1])
        strcat(dirp->dirname, "\\");

    strcat(dirp->dirname, "*.*");

#if defined(OS2_16) || defined(OS2_32)
    hdir = HDIR_SYSTEM;
    cnt  = 1;
    rc = DosFindFirst(dirp->dirname, &hdir,
                      FILE_NORMAL | FILE_READONLY | FILE_HIDDEN |
                      FILE_SYSTEM | FILE_DIRECTORY | FILE_ARCHIVED,
                      &ff, sizeof(ff), &cnt, FIL_STANDARD);
#endif

#if defined(BORLAND)
    rc = findfirst(dirp->dirname, &ff, FA_DIREC | FA_ARCH);
    if (0 != rc)
        rc = _doserrno;
#endif

#if defined(MSDOS_16)
    rc = _dos_findfirst(dirp->dirname, _A_ARCH | _A_SUBDIR, &ff);
#endif

#if defined(NT_32)
    hdir = _findfirst(dirp->dirname, &ff);
    rc = (-1 == hdir) ? errno : NO_ERROR;
#endif

    while (NO_ERROR == rc)
    {
        auto     struct dirent     *entp;

        if (dirp->tot_ent >= dirp->max_ent)
        {
            auto     struct dirent    **p;

            dirp->max_ent += DIRENT_INCR;
            p = realloc(dirp->entp, dirp->max_ent * sizeof(entp));
            if (NULL == p)
            {
                rc = ERROR_NOT_ENOUGH_MEMORY;
                break;
            }

            dirp->entp = p;
        }

#if defined(OS2_16) || defined(OS2_32)
        p = (char *) ff.achName;
        len = (int) ff.cchName;
        isdir = (ff.attrFile & FILE_DIRECTORY) ? 1 : 0;
#endif

#if defined(BORLAND)
        p = (char *) ff.ff_name;
        len = strlen(p);
        isdir = (ff.ff_attrib & FA_DIREC) ? 1 : 0;
#endif

#if defined(MSDOS_16) || defined(NT_32)
        p = ff.name;
        len = strlen(p);
        isdir = (ff.attrib & _A_SUBDIR) ? 1 : 0;
#endif

        entp = malloc(sizeof(*entp) + (size_t) len);
        if (NULL == entp)
        {
            rc = ERROR_NOT_ENOUGH_MEMORY;
            break;
        }

        entp->d_ino = 0;
        entp->d_off = dirp->tot_ent;
        entp->d_isdir = isdir;
        entp->d_reclen = (unsigned short) len;
        memcpy(entp->d_name, p, len);
        entp->d_name[len] = '\0';
        dirp->entp[dirp->tot_ent++] = entp;

#if defined(OS2_16) || defined(OS2_32)
        cnt = 1;
        rc = DosFindNext(hdir, &ff, sizeof(ff), &cnt);
#endif

#if defined(BORLAND)
        rc = findnext(&ff);
        if (0 != rc)
            rc = _doserrno;
#endif

#if defined(MSDOS_16)
        rc = _dos_findnext(&ff);
        if (0 != rc)
            rc = ERROR_NO_MORE_FILES;
#endif

#if defined(NT_32)
        rc = _findnext(hdir, &ff);
        if (-1 == rc)
            rc = ERROR_NO_MORE_FILES;
#endif
    }

#if defined(OS2_16) || defined(OS2_32)
    DosFindClose(hdir);
#endif

#if defined(NT_32)
    _findclose(hdir);
#endif

    if (ERROR_NO_MORE_FILES == rc)
        return(dirp);

    closedir(dirp);
    return(NULL);
}


struct dirent *readdir(DIR *dirp)
{
    if (dirp->cur_ent < 0 || dirp->cur_ent >= dirp->tot_ent)
        return(NULL);

    return(dirp->entp[dirp->cur_ent++]);
}


long telldir(DIR *dirp)
{
    return((long) dirp->cur_ent);
}


void seekdir(DIR *dirp, long loc)
{
    dirp->cur_ent = (int) loc;
    return;
}


void rewinddir(DIR *dirp)
{
    dirp->cur_ent = 0;
    return;
}


void closedir(DIR *dirp)
{
    if (dirp)
    {
        if (dirp->dirname)
            free(dirp->dirname);

        if (dirp->entp)
        {
            register int        i;

            for (i = 0; i < dirp->tot_ent; ++i)
                free(dirp->entp[i]);

            free(dirp->entp);
        }
    }

    return;
}


#endif
