;;; CLASS BROWSER FOR C++
;;; $Id: br-compl.el,v 3.1 1995/02/17 18:19:36 mmann Exp $
;;;
;;; **********************************************************************
;;; Copyright (C) 1993, 1994 Gerd Moellmann. All rights reserved.
;;; Altenbergstr. 6, D-40235 Duesseldorf, Germany
;;; 100025.3303@COMPUSERVE.COM
;;; Suggestions, comments and requests for improvements are welcome.
;;; **********************************************************************
;;;
;;; This version works with both Emacs version 18 and 19, and I want
;;; to keep it that way. It requires the CL-19 Common Lisp compatibility
;;; package for Emacs 18 and 19.
;;;
;;; This file contains the code to perform symbol completion.
;;; 

;; This file may be made part of the Emacs distribution at the option
;; of the FSF.

;; This code is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; this code, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

(require 'cl-19 "cl")
(require 'backquote)
(require 'br-macro)
(require 'br-struc)

(defvar browse-last-completion nil
  "A buffer-local variable that contains the text inserted by the
last completion operation.")

(defvar browse-last-completion-start nil
  "A buffer-local variable containing the string which was the basis
for the last completion operation.")

(defvar browse-last-completion-location nil
  "A buffer-local variable containing the buffer position at which
the last completion operation was initiated.")

(defvar browse-last-completion-obarray nil
  "A buffer-local variable containing the member obarray used for the
last completion operation.")

(make-variable-buffer-local 'browse-last-completion-obarray)
(make-variable-buffer-local 'browse-last-completion-location)
(make-variable-buffer-local 'browse-last-completion)
(make-variable-buffer-local 'browse-last-completion-start)


;;;
;;; Get some obarray to base the symbol completion on. If more
;;; than one choice exists, ask the user which to use.
;;; 

(defun browse-some-member-obarray ()
  (let* ((buffers (browse-different-tree-buffers))
	 (buffer (cond ((and (first buffers) (not (second buffers)))
			(first buffers))
		       (t (or (browse-electric-choose-tree)
			      (error "No tree buffer.")))))
	 (header (browse-@value '@header buffer)))
    (browse-member-obarray header)))

;;;
;;; If ITEM is an element of LIST, return the element following
;;; ITEM in the list. If ITEM is the last element, return the
;;; first element as successor.
;;; 

(defun browse-cyclic-successor (string list)
  (or (nth (1+ (position string list :test 'string=)) list)
      (first list)))

;;;
;;; Perform symbol completion. This function is usually bound to
;;; some key in C++ buffers.
;;; 
;;;###autoload

(defun* browse-complete-symbol (prefix)
  "Perform completion on the C++ symbol preceding point. A second call
of this function without changing point inserts the next match. 
A call with prefix reads the symbol to insert from the minibuffer with
completion."
  (interactive "P")
  (let* ((end (point))
	 (begin (save-excursion (skip-chars-backward "a-zA-Z_0-9") (point)))
	 (pattern (buffer-substring begin end))
	 list completion)
    (cond
     ;; With prefix, read name from minibuffer with completion.

     (prefix
      (let* ((members (browse-some-member-obarray))
	     (completion (completing-read "Insert member: "
					  members nil t pattern)))
	(when completion
	  (setf browse-last-completion-location nil)
	  (delete-region begin end)
	  (insert completion))))

     ;; If this function is called at the same point the last expansion
     ;; ended, insert the next expansion.

     ((eq (point) browse-last-completion-location)
      (setf list (all-completions browse-last-completion-start
				  browse-last-completion-obarray)
	    completion (browse-cyclic-successor browse-last-completion list))
      (cond ((null completion)
	     (error "No completion."))
	    ((string= completion pattern)
	     (error "No further completion."))
	    (t
	     (delete-region begin end)
	     (insert completion)
	     (setf browse-last-completion completion
		   browse-last-completion-location (point)))))

     ;; First time the function is called at some position in
     ;; the buffer: Start new completion.

     (t
      (let* ((members (browse-some-member-obarray))
	     (completion (first (all-completions pattern members nil))))
	(cond ((eq completion t))
	      ((null completion)
	       (error "Can't find completion for `%s'." pattern))
	      (t
	       (delete-region begin end)
	       (insert completion)
	      
	       (setf browse-last-completion-location (point)
		     browse-last-completion-start pattern
		     browse-last-completion completion
		     browse-last-completion-obarray members))))))))

(provide 'br-compl)

;; end of `br-compl.el'.
