/*
 *  linux/fs/msdos/inode.c
 *
 *  Written 1992,1993 by Werner Almesberger
 */

#ifdef __IBMC__
#pragma strings(readonly)
#endif

#ifdef MODULE
#include <linux/module.h>
#include <linux/version.h>
#else
#define MOD_INC_USE_COUNT
#define MOD_DEC_USE_COUNT
#endif

#ifdef OS2
#define INCL_DOSERRORS
#define INCL_NOPMAPI
#include <os2.h>                // From the "Developer Connection Device Driver Kit" version 2.0

#include <string.h>                // Standard MS Visual C++ string.h

#include <os2/types.h>
#include <os2/StackToFlat.h>
#include <os2/os2proto.h>

#include <linux/ext2_fs.h>
#include <linux/fs_proto.h>
#include <linux/ext2_proto.h>
#include <os2/fsh32.h>

#endif

#include <linux/xmsdos_fs.h>
#ifndef OS2
#include <linux/kernel.h>
#endif

#include <linux/sched.h>
#include <linux/errno.h>
#ifndef OS2
#include <linux/string.h>
#include <linux/ctype.h>
#else
#include <os2/ctype.h>
#endif

#ifndef OS2
#include <linux/major.h>
#include <linux/blkdev.h>
#endif
#include <linux/fs.h>
#include <linux/stat.h>
#include <linux/locks.h>

#include "msbuffer.h"

#ifndef OS2
#include <asm/segment.h>
#endif

#ifdef DEBUG_LOG
extern char debug_com;                  // output debug info to COM port
extern char debug_vfat_com;             // output VFAT debug info to COM port
#define KERNEL_PRINTF(x) { if (debug_vfat_com) kernel_printf(x);}
#else
#define KERNEL_PRINTF(x)
#endif

extern int *blksize_size[];

#define MAJOR(dev) 0                  // shortcut just for messages
#define MINOR(dev) dev                // shortcut just for messages

#ifdef OS2
extern int vfat_drives[];
#endif

#ifdef OS2
/* copied from vsprintf.c */
unsigned long simple_strtoul(const char *cp,char **endp,unsigned int base)
{
	unsigned long result = 0,value;

	if (!base) {
		base = 10;
		if (*cp == '0') {
			base = 8;
			cp++;
			if ((*cp == 'x') && isxdigit(cp[1])) {
				cp++;
				base = 16;
			}
		}
	}
	while (isxdigit(*cp) && (value = isdigit(*cp) ? *cp-'0' : (islower(*cp)
	    ? toupper(*cp) : *cp)-'A'+10) < base) {
		result = result*base + value;
		cp++;
	}
	if (endp)
		*endp = (char *)cp;
	return result;
}
#endif


void xmsdos_put_inode(struct inode *inode)
{
	struct inode *depend;
	struct super_block *sb;

KERNEL_PRINTF("xmsdos_put_inode");

	if (inode->i_nlink) {
		if (MSDOS_I(inode)->i_busy) cache_inval_inode(inode);
		return;
	}
	inode->i_size = 0;
	xmsdos_truncate(inode);
	depend = MSDOS_I(inode)->i_depend;
	sb = inode->i_sb;
	clear_inode(inode);
	if (depend) {
		if (MSDOS_I(depend)->i_old != inode) {
			printk("Invalid link (0x%p): expected 0x%p, got 0x%p\n",
			    depend, inode, MSDOS_I(depend)->i_old);
			fs_panic(sb,"...");
			return;
		}
		MSDOS_I(depend)->i_old = NULL;
		iput(depend);
	}
}


void xmsdos_put_super(struct super_block *sb)
{
KERNEL_PRINTF("xmsdos_put_super");

	cache_inval_dev(sb->s_dev);
	set_blocksize (sb->s_dev,BLOCK_SIZE);
	lock_super(sb);
	sb->s_dev = 0;
	unlock_super(sb);
	MOD_DEC_USE_COUNT;
	return;
}


static struct super_operations xmsdos_sops = { 
	xmsdos_read_inode,
#ifndef OS2
	xmsdos_notify_change,
#else
        NULL,
#endif
	xmsdos_write_inode,
	xmsdos_put_inode,
	xmsdos_put_super,
	NULL, /* added in 0.96c */
#ifndef OS2
	xmsdos_statfs,
#else
        NULL  /* xmsdos_statfs */,
#endif
	NULL
};

static int parse_options(char *options,char *check,char *conversion,uid_t *uid,
    gid_t *gid,int *umask,int *debug,int *fat,int *quiet,
	int *blksize)
{
	char *this_char,*value;

#if 0
	*check = 'n';
#else
	*check = 'x';		/* Extended MSDOS filesystem */
#endif
	*conversion = 'b';
#ifndef OS2
	*uid = current->uid;
	*gid = current->gid;
	*umask = current->fs->umask;
#endif
	*debug = *fat = *quiet = 0;
	if (!options) return 1;
	for (this_char = strtok(options,","); this_char; this_char = strtok(NULL,",")) {
		if ((value = strchr(this_char,'=')) != NULL)
			*value++ = 0;
		if (!strcmp(this_char,"check") && value) {
			if (value[0] && !value[1] && strchr("rns",*value))
				*check = *value;
			else if (!strcmp(value,"relaxed")) *check = 'r';
			else if (!strcmp(value,"normal")) *check = 'n';
			else if (!strcmp(value,"strict")) *check = 's';
			else return 0;
		}
		else if (!strcmp(this_char,"conv") && value) {
			if (value[0] && !value[1] && strchr("bta",*value))
				*conversion = *value;
			else if (!strcmp(value,"binary")) *conversion = 'b';
			else if (!strcmp(value,"text")) *conversion = 't';
			else if (!strcmp(value,"auto")) *conversion = 'a';
			else return 0;
		}
		else if (!strcmp(this_char,"uid")) {
			if (!value || !*value)
				return 0;
			*uid = simple_strtoul(value,&value,0);
			if (*value)
				return 0;
		}
		else if (!strcmp(this_char,"gid")) {
			if (!value || !*value)
				return 0;
			*gid = simple_strtoul(value,&value,0);
			if (*value)
				return 0;
		}
		else if (!strcmp(this_char,"umask")) {
			if (!value || !*value)
				return 0;
			*umask = simple_strtoul(value,&value,8);
			if (*value)
				return 0;
		}
		else if (!strcmp(this_char,"debug")) {
			if (value) return 0;
			*debug = 1;
		}
		else if (!strcmp(this_char,"fat")) {
			if (!value || !*value)
				return 0;
			*fat = simple_strtoul(value,&value,0);
			if (*value || (*fat != 12 && *fat != 16))
				return 0;
		}
		else if (!strcmp(this_char,"quiet")) {
			if (value) return 0;
			*quiet = 1;
		}
		else if (!strcmp(this_char,"blocksize")) {
			*blksize = simple_strtoul(value,&value,0);
			if (*value)
				return 0;
			if (*blksize != 512 && *blksize != 1024){
				printk ("MSDOS FS: Invalid blocksize (512 or 1024)\n");
			}
		}
		else return 0;
	}
	return 1;
}


/* Read the super block of an MS-DOS FS. */

struct super_block *xmsdos_read_super(struct super_block *sb,void *data,
				     int silent)
{
	struct buffer_head *bh;
	struct msdos_boot_sector *b;
	int data_sectors,logical_sector_size,sector_mult;
	int debug,error,fat,quiet;
	char check,conversion;
	uid_t uid;
	gid_t gid;
	int umask;
	int blksize = 512;

	MOD_INC_USE_COUNT;
#ifndef OS2
	if (hardsect_size[MAJOR(sb->s_dev)] != NULL){
		blksize = hardsect_size[MAJOR(sb->s_dev)][MINOR(sb->s_dev)];
		if (blksize != 512){
			printk ("MSDOS: Hardware sector size is %d\n",blksize);
		}
	}
#endif
	if (!parse_options((char *) data,__StackToFlat(&check),__StackToFlat(&conversion),__StackToFlat(&uid),__StackToFlat(&gid),__StackToFlat(&umask),
	    __StackToFlat(&debug),__StackToFlat(&fat),__StackToFlat(&quiet),__StackToFlat(&blksize))
		|| (blksize != 512 && blksize != 1024)) {
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	cache_init();
debug=1;

	lock_super(sb);
	/* The first read is always 1024 bytes */
	sb->s_blocksize = 1024;
	set_blocksize(sb->s_dev, 1024);
	bh = bread(sb->s_dev, 0, 1024);
	unlock_super(sb);
	if (bh == NULL || !xmsdos_is_uptodate(sb,bh)) {
		brelse (bh);
		sb->s_dev = 0;
		printk("MSDOS bread failed\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	b = (struct msdos_boot_sector *) bh->b_data;
	set_blocksize(sb->s_dev, blksize);
/*
 * The DOS3 partition size limit is *not* 32M as many people think.  
 * Instead, it is 64K sectors (with the usual sector size being
 * 512 bytes, leading to a 32M limit).
 * 
 * DOS 3 partition managers got around this problem by faking a 
 * larger sector size, ie treating multiple physical sectors as 
 * a single logical sector.
 * 
 * We can accommodate this scheme by adjusting our cluster size,
 * fat_start, and data_start by an appropriate value.
 *
 * (by Drew Eckhardt)
 */

#define ROUND_TO_MULTIPLE(n,m) ((n) && (m) ? (n)+(m)-1-((n)-1)%(m) : 0)
    /* don't divide by zero */

	logical_sector_size = CF_LE_W(*(unsigned short *) &b->sector_size);
	sector_mult = logical_sector_size >> SECTOR_BITS;
	MSDOS_SB(sb)->cluster_size = b->cluster_size*sector_mult;
	MSDOS_SB(sb)->fats = b->fats;
	MSDOS_SB(sb)->fat_start = CF_LE_W(b->reserved)*sector_mult;
	MSDOS_SB(sb)->fat_length = CF_LE_W(b->fat_length)*sector_mult;
	MSDOS_SB(sb)->dir_start = (CF_LE_W(b->reserved)+b->fats*CF_LE_W(
	    b->fat_length))*sector_mult;
	MSDOS_SB(sb)->dir_entries = CF_LE_W(*((unsigned short *) &b->dir_entries
	    ));
	MSDOS_SB(sb)->data_start = MSDOS_SB(sb)->dir_start+ROUND_TO_MULTIPLE((
	    MSDOS_SB(sb)->dir_entries << MSDOS_DIR_BITS) >> SECTOR_BITS,
	    sector_mult);
	data_sectors = (CF_LE_W(*((unsigned short *) &b->sectors)) ?
	    CF_LE_W(*((unsigned short *) &b->sectors)) :
	    CF_LE_L(b->total_sect))*sector_mult-MSDOS_SB(sb)->data_start;
	error = !b->cluster_size || !sector_mult;
	if (!error) {
		MSDOS_SB(sb)->clusters = b->cluster_size ? data_sectors/
		    b->cluster_size/sector_mult : 0;
		MSDOS_SB(sb)->fat_bits = fat ? fat : MSDOS_SB(sb)->clusters >
		    MSDOS_FAT12 ? 16 : 12;
		error = !MSDOS_SB(sb)->fats || (MSDOS_SB(sb)->dir_entries &
		    (MSDOS_DPS-1)) || MSDOS_SB(sb)->clusters+2 > MSDOS_SB(sb)->
		    fat_length*SECTOR_SIZE*8/MSDOS_SB(sb)->fat_bits ||
		    (logical_sector_size & (SECTOR_SIZE-1)) || !b->secs_track ||
		    !b->heads;
	}
	brelse(bh);
	/*
		This must be done after the brelse because the bh is a dummy
		allocated by msdos_bread (see buffer.c)
	*/
	sb->s_blocksize = blksize;	/* Using this small block size solve the */
				/* the misfit with buffer cache and cluster */
				/* because cluster (DOS) are often aligned */
				/* on odd sector */
	sb->s_blocksize_bits = blksize == 512 ? 9 : 10;
	if (error || debug) {
		/* The MSDOS_CAN_BMAP is obsolete, but left just to remember */
		printk("[MS-DOS FS Rel. 12,FAT %d,check=%c,conv=%c,"
		    "uid=%d,gid=%d,umask=%03o%s]\n",MSDOS_SB(sb)->fat_bits,check,
		    conversion,uid,gid,umask,MSDOS_CAN_BMAP(MSDOS_SB(sb)) ?
		    ",bmap" : "");
		printk("[me=0x%x,cs=%d,#f=%d,fs=%d,fl=%d,ds=%d,de=%d,data=%d,"
		    "se=%d,ts=%ld,ls=%d]\n",b->media,MSDOS_SB(sb)->cluster_size,
		    MSDOS_SB(sb)->fats,MSDOS_SB(sb)->fat_start,MSDOS_SB(sb)->
		    fat_length,MSDOS_SB(sb)->dir_start,MSDOS_SB(sb)->dir_entries,
		    MSDOS_SB(sb)->data_start,CF_LE_W(*(unsigned short *) &b->
		    sectors),b->total_sect,logical_sector_size);
		printk ("Transaction block size = %d\n",blksize);
	}
	if (error) {
		if (!silent)
			printk("VFS: Can't find a valid MSDOS filesystem on dev 0x%04x.\n",
				   sb->s_dev);
		sb->s_dev = 0;
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	sb->s_magic = MSDOS_SUPER_MAGIC;
	MSDOS_SB(sb)->name_check = check;
	MSDOS_SB(sb)->conversion = conversion;
	/* set up enough so that it can read an inode */
	sb->s_op = &xmsdos_sops;
	MSDOS_SB(sb)->fs_uid = uid;
	MSDOS_SB(sb)->fs_gid = gid;
	MSDOS_SB(sb)->fs_umask = umask;
	MSDOS_SB(sb)->quiet = quiet;
	MSDOS_SB(sb)->free_clusters = -1; /* don't know yet */
	MSDOS_SB(sb)->fat_wait = NULL;
	MSDOS_SB(sb)->fat_lock = 0;
	MSDOS_SB(sb)->prev_free = 0;
	if (!(sb->s_mounted = iget(sb,MSDOS_ROOT_INO))) {
		sb->s_dev = 0;
		printk("get root inode failed\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	return sb;
}

#ifndef OS2
void xmsdos_statfs(struct super_block *sb,struct statfs *buf)
{
	int free,nr;

	put_fs_long(sb->s_magic,&buf->f_type);
	put_fs_long(MSDOS_SB(sb)->cluster_size*SECTOR_SIZE,&buf->f_bsize);
	put_fs_long(MSDOS_SB(sb)->clusters,&buf->f_blocks);
	lock_fat(sb);
	if (MSDOS_SB(sb)->free_clusters != -1)
		free = MSDOS_SB(sb)->free_clusters;
	else {
		free = 0;
		for (nr = 2; nr < MSDOS_SB(sb)->clusters+2; nr++)
			if (!fat_access(sb,nr,-1)) free++;
		MSDOS_SB(sb)->free_clusters = free;
	}
	unlock_fat(sb);
	put_fs_long(free,&buf->f_bfree);
	put_fs_long(free,&buf->f_bavail);
	put_fs_long(0,&buf->f_files);
	put_fs_long(0,&buf->f_ffree);
	put_fs_long(12,&buf->f_namelen);
}
#endif

int xmsdos_bmap(struct inode *inode,int block)
{
	struct msdos_sb_info *sb;
	int cluster,offset;

KERNEL_PRINTF("xmsdos_bmap");

	sb = MSDOS_SB(inode->i_sb);
	if (inode->i_ino == MSDOS_ROOT_INO) {
		return sb->dir_start + block;
	}
	cluster = block/sb->cluster_size;
	offset = block % sb->cluster_size;
	if (!(cluster = get_cluster(inode,cluster))) return 0;
	return (cluster-2)*sb->cluster_size+sb->data_start+offset;
}


void xmsdos_read_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct buffer_head *bh;
	struct msdos_dir_entry *raw_entry;
	int nr;

KERNEL_PRINTF("xmsdos_read_inode");

/* printk("read inode %d\n",inode->i_ino); */
	MSDOS_I(inode)->i_busy = 0;
	MSDOS_I(inode)->i_depend = MSDOS_I(inode)->i_old = NULL;
	MSDOS_I(inode)->i_binary = 1;
	inode->i_uid = MSDOS_SB(inode->i_sb)->fs_uid;
	inode->i_gid = MSDOS_SB(inode->i_sb)->fs_gid;
	if (inode->i_ino == MSDOS_ROOT_INO) {
		inode->i_mode = (S_IRWXUGO & ~MSDOS_SB(inode->i_sb)->fs_umask) |
		    S_IFDIR;
		inode->i_op = &xmsdos_dir_inode_operations;
		inode->i_nlink = xmsdos_subdirs(inode)+2;
		    /* subdirs (neither . nor ..) plus . and "self" */
		inode->i_size = MSDOS_SB(inode->i_sb)->dir_entries*
		    sizeof(struct msdos_dir_entry);
		inode->i_blksize = MSDOS_SB(inode->i_sb)->cluster_size*
		    SECTOR_SIZE;
		inode->i_blocks = (inode->i_size+inode->i_blksize-1)/
		    inode->i_blksize*MSDOS_SB(inode->i_sb)->cluster_size;
		MSDOS_I(inode)->i_start = 0;
		MSDOS_I(inode)->i_attrs = 0;
		inode->i_mtime = inode->i_atime = inode->i_ctime = 0;
		return;
	}
	if (!(bh = bread(inode->i_dev,inode->i_ino >> MSDOS_DPB_BITS,
	    SECTOR_SIZE))) {
		printk("dev = 0x%04X, ino = %ld\n",inode->i_dev,inode->i_ino);
#ifndef OS2
		panic("xmsdos_read_inode: unable to read i-node block");
#else
		ext2_os2_panic(1, "xmsdos_read_inode: unable to read i-node block");
#endif
	}
	raw_entry = &((struct msdos_dir_entry *) (bh->b_data))
	    [inode->i_ino & (MSDOS_DPB-1)];
	if ((raw_entry->attr & ATTR_DIR) && !IS_FREE(raw_entry->name)) {
		inode->i_mode = MSDOS_MKMODE(raw_entry->attr,S_IRWXUGO &
		    ~MSDOS_SB(inode->i_sb)->fs_umask) | S_IFDIR;
		inode->i_op = &xmsdos_dir_inode_operations;
		MSDOS_I(inode)->i_start = CF_LE_W(raw_entry->start);
		inode->i_nlink = xmsdos_subdirs(inode);
		    /* includes .., compensating for "self" */
#ifdef DEBUG
		if (!inode->i_nlink) {
			printk("directory %d: i_nlink == 0\n",inode->i_ino);
			inode->i_nlink = 1;
		}
#endif
		inode->i_size = 0;
		if ((nr = CF_LE_W(raw_entry->start)) != 0)
			while (nr != -1) {
				inode->i_size += SECTOR_SIZE*MSDOS_SB(inode->
				    i_sb)->cluster_size;
				if (!(nr = fat_access(inode->i_sb,nr,-1))) {
					printk("Directory %ld: bad FAT\n",
					    inode->i_ino);
					break;
				}
			}
	}
	else {
		inode->i_mode = MSDOS_MKMODE(raw_entry->attr,(IS_NOEXEC(inode)
		    ? S_IRUGO|S_IWUGO : S_IRWXUGO) & ~MSDOS_SB(inode->i_sb)->fs_umask) |
		    S_IFREG;
		inode->i_op = sb->s_blocksize == 1024
			? &xmsdos_file_inode_operations_1024
			: &xmsdos_file_inode_operations;
		MSDOS_I(inode)->i_start = CF_LE_W(raw_entry->start);
		inode->i_nlink = 1;
		inode->i_size = CF_LE_L(raw_entry->size);
	}
	MSDOS_I(inode)->i_binary = is_binary(MSDOS_SB(inode->i_sb)->conversion,
	    raw_entry->ext);
	MSDOS_I(inode)->i_attrs = raw_entry->attr & ATTR_UNUSED;
	/* this is as close to the truth as we can get ... */
	inode->i_blksize = MSDOS_SB(inode->i_sb)->cluster_size*SECTOR_SIZE;
	inode->i_blocks = (inode->i_size+inode->i_blksize-1)/
	    inode->i_blksize*MSDOS_SB(inode->i_sb)->cluster_size;
	inode->i_mtime = inode->i_atime = inode->i_ctime =
	    date_dos2unix(CF_LE_W(raw_entry->time),CF_LE_W(raw_entry->date));
	brelse(bh);
}


void xmsdos_write_inode(struct inode *inode)
{
	struct super_block *sb = inode->i_sb;
	struct buffer_head *bh;
	struct msdos_dir_entry *raw_entry;
#ifdef OS2
        unsigned short  time;
        unsigned short  date;
#endif

KERNEL_PRINTF("xmsdos_write_inode");

	inode->i_dirt = 0;
	if (inode->i_ino == MSDOS_ROOT_INO || !inode->i_nlink) return;
	if (!(bh = bread(inode->i_dev,inode->i_ino >> MSDOS_DPB_BITS,
	    SECTOR_SIZE))) {
		printk("dev = 0x%04X, ino = %ld\n",inode->i_dev,inode->i_ino);
#ifndef OS2
		panic("xmsdos_write_inode: unable to read i-node block");
#else
		ext2_os2_panic(1, "xmsdos_write_inode: unable to read i-node block");
#endif
	}
	raw_entry = &((struct msdos_dir_entry *) (bh->b_data))
	    [inode->i_ino & (MSDOS_DPB-1)];
	if (S_ISDIR(inode->i_mode)) {
		raw_entry->attr = ATTR_DIR;
		raw_entry->size = 0;
	}
	else {
		raw_entry->attr = ATTR_NONE;
		raw_entry->size = CT_LE_L(inode->i_size);
	}
	raw_entry->attr |= MSDOS_MKATTR(inode->i_mode) |
	    MSDOS_I(inode)->i_attrs;
	raw_entry->start = CT_LE_L(MSDOS_I(inode)->i_start);
#ifndef OS2
	date_unix2dos(inode->i_mtime,&raw_entry->time,&raw_entry->date);
#else
	date_unix2dos(inode->i_mtime, __StackToFlat(&time), __StackToFlat(&date));
        raw_entry->time = time;
        raw_entry->date = date;
#endif
	raw_entry->time = CT_LE_W(raw_entry->time);
	raw_entry->date = CT_LE_W(raw_entry->date);
	mark_buffer_dirty(bh, 1);
	brelse(bh);
}

#ifndef OS2
int xmsdos_notify_change(struct inode * inode,struct iattr * attr)
{
	int error;

	error = inode_change_ok(inode, attr);
	if (error)
		return error;

	if (((attr->ia_valid & ATTR_UID) && 
	     (attr->ia_uid != MSDOS_SB(inode->i_sb)->fs_uid)) ||
	    ((attr->ia_valid & ATTR_GID) && 
	     (attr->ia_gid != MSDOS_SB(inode->i_sb)->fs_gid)) ||
	    ((attr->ia_valid & ATTR_MODE) &&
	     (attr->ia_mode & ~MSDOS_VALID_MODE)))
		error = -EPERM;

	if (error)
		return MSDOS_SB(inode->i_sb)->quiet ? 0 : error;

	inode_setattr(inode, attr);

	if (IS_NOEXEC(inode) && !S_ISDIR(inode->i_mode))
		inode->i_mode &= S_IFMT | S_IRUGO | S_IWUGO;
	else
		inode->i_mode |= S_IXUGO;

	inode->i_mode = ((inode->i_mode & S_IFMT) | ((((inode->i_mode & S_IRWXU
	    & ~MSDOS_SB(inode->i_sb)->fs_umask) | S_IRUSR) >> 6)*S_IXUGO)) &
	    ~MSDOS_SB(inode->i_sb)->fs_umask;
	return 0;
}
#endif
#ifdef OS2
 struct file_system_type xmsdos_fs_type = {
	xmsdos_read_super, "xmsdos", 1, NULL
};

#endif

#ifdef MODULE

char kernel_version[] = UTS_RELEASE;

static struct file_system_type xmsdos_fs_type = {
	xmsdos_read_super, "xmsdos", 1, NULL
};

int init_module(void)
{
	register_filesystem(&xmsdos_fs_type);
	return 0;
}

void cleanup_module(void)
{
	unregister_filesystem(&xmsdos_fs_type);
}

#endif

#ifdef OS2

#define ERROR_VOLUME_NOT_MOUNTED 0xEE00                // IFS specific


/*
 * Structure of the super block
 */
#pragma pack(1)
struct msdos_super_block {
        struct msdos_boot_sector  s_boot_sector;         /* boot sector fields */

        /* offset 0x24 */
        unsigned char s_phys_disk_nr;   /* physical disk number */
        unsigned char s_curr_head;      /* current head - ignored by FAT filesystem but used in Windows NT */
        unsigned char s_signature;      /* signature - must be either 0x28 or 0x29 */
        unsigned long s_serial;         /* Volume Serial Number */
        unsigned char s_vol_lable[11];  /* Volume label */
        unsigned char s_system_id[8];   /* System ID */
};
#pragma pack()



/*
 * This routine reads the superblock and tests if it can be an fat/vfat file system.
 * It does NOT use buffers from the cache.
 */
int check_msdos_magic(struct vpfsi32 *pvpfsi, unsigned short hVPB) {
    int   nb_sec;
    int   rc;
    int   rc2;
    char *buf;
    struct msdos_super_block *es;

char labelname[14];

for (rc=0; rc<sizeof(labelname); rc++) labelname[rc] = pvpfsi->vpi_text[rc];
/*strncpy(labelname, pvpfsi->vpi_text, sizeof(labelname));*/

labelname[13] = '\0';
if (debug_com) kernel_printf("reading drive %d %s", pvpfsi->vpi_drive, __StackToFlat(labelname));

    //
    // Allocates a temporary buffer
    //
    if ((rc = DevHlp32_VMAlloc(BLOCK_SIZE, VMDHA_NOPHYSADDR, VMDHA_FIXED | VMDHA_CONTIG, __StackToFlat(&buf))) == NO_ERROR) {
        //
        // Reads disk block 0 (with blocksize = 1024)
        //
        nb_sec = BLOCK_SIZE / pvpfsi->vpi_bsize;
        if ((rc = fsh32_dovolio(
                                DVIO_OPREAD | DVIO_OPRESMEM | DVIO_OPBYPASS | DVIO_OPNCACHE,
                                DVIO_ALLABORT | DVIO_ALLRETRY | DVIO_ALLFAIL,
                                hVPB,
                                buf,
                                __StackToFlat(&nb_sec),
                                0
                               )) == NO_ERROR) {

            int foundFATsig;

            es = (struct msdos_super_block *)buf;

            if (es->s_system_id[0] == 'F' && es->s_system_id[1] == 'A' && es->s_system_id[2] == 'T')
              foundFATsig = 1;
            else
              foundFATsig = 0;

            if (foundFATsig || vfat_drives[pvpfsi->vpi_drive] == 2) {

                if (foundFATsig)
                  kernel_printf("FAT signature found in superblock (hVPB = 0x%04X)", hVPB);
                else
                  kernel_printf("VFAT forced for superblock (hVPB = 0x%04X)", hVPB);

                /*
                 * The volume serial number is stored in s_serial field
                 */
                pvpfsi->vpi_vid = es->s_serial;

/* volume name should be retrieved later from directory entry */
                /*
                 * The volume name is truncated to the valid OS/2 volume label length (11 chars)
                 */                
         	strncpy(pvpfsi->vpi_text, es->s_vol_lable, sizeof(pvpfsi->vpi_text));
                pvpfsi->vpi_text[sizeof(pvpfsi->vpi_text) - 1] = '\0';

		rc = NO_ERROR;
            } else {
                kernel_printf("FAT signature NOT found in superblock (hVPB = 0x%04X)", hVPB);
		rc = ERROR_VOLUME_NOT_MOUNTED;
            }
        } else {
	    kernel_printf("check_msdos_magic - fsh32_dovolio returned %d", rc);
        }

        if ((rc2 = DevHlp32_VMFree((void *)buf)) == NO_ERROR) {
            /*
             * Nothing else
             */
        } else {
	    kernel_printf("check_msdos_magic - VMFree returned %d", rc);
            rc = rc2;
        }
    } else {
        kernel_printf("check_msdos_magic - VMAlloc returned %d", rc);
    }
    return rc;

}
#endif
