/*******************************
* SAMPLE (C) D. FRIEAUFF, 1993 *
********************************/

#define  INCL_PM
#define  INCL_DOS
#include <os2.h>
#include <b8lib3.h>

#include "rc.h"
#include "sample.h"

/**********************
* External references *
***********************/
extern BOOL         bMove;          /* mouse operation flags */
extern BOOL         bColor;         /* use color (1) or greyscale (0) */
extern BOOL         bTimerActive;   /* timer active */
extern HAB          hab;            /* anchor block handle */
extern HFILE        hfB008;         /* TP DD handle */
extern WIN          w;              /* window */
extern PBITMAPINFO2 pbmi2;          /* pointer to bitmapinfo */

/********************************************
* Global variables with external references *
*********************************************/
PARAMS    params;             /* iteration parameters */
BOOL      bTPbooted = FALSE;  /* flags whether transputer (TP) is booted */
BOOL      bImage = FALSE;     /* flags whether an image has been generated */

/*********************************
* Local function predeclarations *
**********************************/
static MRESULT EXPENTRY StartDlgProc (HWND, USHORT, MPARAM, MPARAM);
static BOOL CmdBoot (HWND hwnd);
static BOOL CmdTest (HWND hwnd, BOOL fUseDMA);
static BOOL WriteTest (HWND hwnd, BOOL fUseDMA, LONG lBlockCount, LONG lBlockSize,
                       BYTE *pBlock, double *dTransferDuration);
static BOOL ReadTest (HWND hwnd, BOOL fUseDMA, LONG lBlockCount, LONG lBlockSize,
                      BYTE *pBlock, double *dTransferDuration, CHAR *szText);

/**************
* Static data *
***************/
static char szText[200];  /* for misc. string operations */

/**********
* Command *
***********/
VOID Command (HWND hwnd, MPARAM mp1)
{
switch (SHORT1FROMMP (mp1)) {
  case IDM_BOOT:  /* boot the transputer system */
    CmdBoot (hwnd);
    break;

  case IDM_ABOUT: /* show the program id */
    About (hwnd);
    break;

  case IDM_EXIT:  /* exit the program */
    WinPostMsg (hwnd, WM_CLOSE, 0L, 0L);
    break;

  case IDM_START: /* start the iteration */
    if (bTimerActive) return; /* timer is active, i.e last iteration not yet finished */
    /* get iteration parameters */
    if (WinDlgBox (HWND_DESKTOP, hwnd, (PFNWP)StartDlgProc, (HMODULE)0, IDD_START, NULL))
       return;
    CmdStart (hwnd); /* start iteration */
    break;

  case IDM_NO_DMA: /* perform link test and measure data transfer performance w/o using DMA */
    if (bTimerActive) return; /* timer is active, i.e last iteration not yet finished */
    CmdTest(hwnd, FALSE);     /* start test */
    break;

  case IDM_DMA: /* perform link test and measure data transfer performance using DMA */    
    if (bTimerActive) return; /* timer is active, i.e last iteration not yet finished */
    CmdTest(hwnd, TRUE);      /* start test */
    break;
  }
}

/***************
* StartDlgProc *
****************/
MRESULT EXPENTRY StartDlgProc (HWND hwnd, USHORT msg, MPARAM mp1, MPARAM mp2)
{
switch (msg)
  {
  case WM_INITDLG:
    /* fill params into edit boxes */
    FillBoxDouble (hwnd, IDD_LEFTBOUND, params.RealStart);
    FillBoxDouble (hwnd, IDD_RIGHTBOUND, params.RealEnd);
    FillBoxDouble (hwnd, IDD_UPPERBOUND, params.ImagStart);
    FillBoxDouble (hwnd, IDD_LOWERBOUND, params.ImagEnd);
    FillBoxDouble (hwnd, IDD_REALCONST, params.RealConst);
    FillBoxDouble (hwnd, IDD_IMAGCONST, params.ImagConst);
    if (bColor)
       WinCheckButton (hwnd, IDD_COLOR, TRUE);
      else
       WinCheckButton (hwnd, IDD_GREY, TRUE);

    /* set focus to first edit box */
    WinSetFocus (HWND_DESKTOP, (HWND)WinWindowFromID (hwnd, IDD_LEFTBOUND));

    return ((MRESULT)0L);
    break;

  case WM_COMMAND:
    switch (SHORT1FROMMP (mp1)) {
      case DID_OK:
        /* get values */
        params.RealStart = GetBoxDouble (hwnd, IDD_LEFTBOUND);
        params.RealEnd   = GetBoxDouble (hwnd, IDD_RIGHTBOUND);
        params.ImagStart = GetBoxDouble (hwnd, IDD_UPPERBOUND);
        params.ImagEnd   = GetBoxDouble (hwnd, IDD_LOWERBOUND);
        params.RealConst = GetBoxDouble (hwnd, IDD_REALCONST);
        params.ImagConst = GetBoxDouble (hwnd, IDD_IMAGCONST);
        bColor = WinQueryButtonCheckstate (hwnd, IDD_COLOR);

        WinDismissDlg (hwnd, FALSE);
        return ((MRESULT)0L);
        break;

      case DID_CANCEL:
        WinDismissDlg (hwnd, TRUE);
        return ((MRESULT)1L);
        break;

      case DID_DEFAULT:
        /* reset parameters to default */
        SetParamDefaults ();
        /* refresh edit boxes */
        WinPostMsg (hwnd, WM_INITDLG, NULL, NULL);
        return ((MRESULT)0L);
        break;
      }
      break;
  }
return WinDefDlgProc (hwnd, msg, mp1, mp2);
}

/***********
* CmdBoot  *
************/
BOOL CmdBoot (HWND hwnd)
{
USHORT  usRet;
ULONG   ulAction, ulBytesTransf;
BYTE    TpData[6];
HFILE   hfBootFile = 0;
FILEDLG fdg;
CHAR    chTitle[MESSAGELEN];
CHAR    chDrive[4];

bTPbooted = FALSE;

memset (&fdg, 0, sizeof(FILEDLG));

WinLoadString (hab, (HMODULE)0L, IDS_BOOTFILE_EXT, MESSAGELEN, fdg.szFullFile);
WinLoadString (hab, (HMODULE)0L, IDS_BOOTTITLE, MESSAGELEN, chTitle);

fdg.cbSize = sizeof(FILEDLG);
fdg.fl = FDS_HELPBUTTON | FDS_CENTER | FDS_OPEN_DIALOG;
fdg.pszTitle = chTitle;

/* open file dialog box to get the file name */
if (!WinFileDlg (HWND_DESKTOP, hwnd, &fdg)) return (TRUE);

/* cancelled */
if (fdg.lReturn != DID_OK) return (TRUE);

/* open file */
if (DosOpen (fdg.szFullFile, &hfBootFile, &ulAction, 0L, FILE_NORMAL, FILE_OPEN,
             OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE, 0L))
   return (ErrorMsg (hwnd, IDS_FILEERROR));

/* reset transputer (for hardware check) */
if (usRet = B8Reset (hfB008)) {
   DosClose (hfBootFile);
   return (ErrorMsg (hwnd, IDS_RESETERROR)); }

/* boot transputer */
if (usRet = B8BootFile (hfB008, hfBootFile, FALSE)) {
   DosClose (hfBootFile);
   return (ErrorMsg (hwnd, IDS_BOOTERROR)); }

/* close boot file */
DosClose (hfBootFile);

/* wait (block this thread) for acknowledge from transputer (see occ\root.occ) */
if (B8Read (hfB008, TpData, SRV_TIMEOUT, 6, &ulBytesTransf))
   return (ErrorMsg (hwnd, IDS_B8READ));
if (*(PUSHORT)&TpData[2] != HCMD_ACK)
   return (ErrorMsg (hwnd, IDS_NOACK));

/* enable locked menu items */
EnableMenu (w.hwndMenu, IDM_START, TRUE);
EnableMenu (w.hwndMenu, IDM_TEST, TRUE);
EnableMenu (w.hwndMenu, IDM_NO_DMA, TRUE);
EnableMenu (w.hwndMenu, IDM_DMA, TRUE);
bTPbooted = TRUE;

/* return (no error) */
return (FALSE);
}

/***********
* CmdStart *
************/
BOOL CmdStart (HWND hwnd)
{
ULONG  ulTpCommand, ulBytesTransf;
PRGB2  pColorTable; /* pointer to bitmap color table */
LONG   lcColors, l;

/* setup color table */
DevQueryCaps (GpiQueryDevice (w.hpsMem), CAPS_COLORS, 1, &lcColors);
pColorTable = (PRGB2)( ((PBYTE)pbmi2) + sizeof (BITMAPINFO2) );
if (bColor)
   GpiQueryRealColors (w.hpsMem, 0, 0, lcColors, (PLONG)pColorTable);
  else {
   for (l = 0; l < lcColors; l++) {
       pColorTable[l].bRed = (BYTE)l;
       pColorTable[l].bGreen = (BYTE)l;
       pColorTable[l].bBlue = (BYTE)l;
       pColorTable[l].fcOptions = PC_EXPLICIT; }
  }

/* setup hidden (not user-controlled) parameters */
params.MaxIt = lcColors - 1;              /* max. number of iterations, determines color */
params.lDataLength = sizeof(params) - 4L; /* length of parameter block */
params.lcx = w.sizel.cx;
params.lcy = w.sizel.cy;

/* select transputer operation: iteration */
ulTpCommand = TP_ITERATE;
if (B8Write (hfB008, (PBYTE)&ulTpCommand, SRV_TIMEOUT, sizeof (ULONG), &ulBytesTransf))
   return (ErrorMsg (hwnd, IDS_B8WRITE));

/* send parameters to transputer */
if (B8Write (hfB008, (PBYTE)&params, SRV_TIMEOUT, sizeof (params), &ulBytesTransf))
   return (ErrorMsg (hwnd, IDS_B8WRITE));

/* Create a timer to periodically poll for available transputer data packets */
if (!WinStartTimer (hab, hwnd, ID_TPTIMER, SRV_POLLINTERVAL))
   return (ErrorMsg (hwnd, IDS_NOTIMER));
bTimerActive = TRUE;

return (FALSE);
}

/***********
* CmdTest  *
************/
BOOL CmdTest (HWND hwnd, BOOL fUseDMA)
{
USHORT   usErr;
BOOL     fErr;
ULONG    ulBytesTransf;
LONG     l;
PBYTE    pBlock;
LONG     lCommand[3];
double   dWrDur1, dWrDur2, dRdDur1, dRdDur2;  /* duration of data transfers, in msec */
double   delta, dRdTransfTime, dRdBlockOvhd, dWrTransfTime, dWrBlockOvhd;

/* display warning */
sprintf (szText, "This may take up to 2 minutes. Continue ?");
if (WinMessageBox (HWND_DESKTOP, hwnd, szText, (PSZ)"Test Start", 0, MB_NOICON | MB_OKCANCEL)
    == MBID_CANCEL) return (FALSE);

/* allocate memory for test */
if (DosAllocMem((PPVOID)&pBlock, MAX_TEST_BLOCK_LENGTH + 6, PAG_COMMIT | OBJ_TILE | PAG_READ | PAG_WRITE))
   return (ErrorMsg (hwnd, IDS_NOMEM));

/* setup test pattern for test 1*/
for (l = 0; l < TEST_BLOCK_LENGTH_1; l++) pBlock[l] = l;

/* perform write test with block length 1 */
fErr = WriteTest (hwnd, fUseDMA, TEST_N_BLOCKS, TEST_BLOCK_LENGTH_1, pBlock, &dWrDur1);
if (fErr) {
   ErrorMsg(hwnd, IDS_B8WRITE);
   DosFreeMem(pBlock);
   return (TRUE); }

/* display warning */
sprintf (szText, "Write test 1 OK. Continue ?");
if (WinMessageBox (HWND_DESKTOP, hwnd, szText, (PSZ)"Link Test", 0, MB_NOICON | MB_OKCANCEL)
    == MBID_CANCEL) {DosFreeMem(pBlock); return (FALSE); }

/* perform read test with block length 1 */
fErr = ReadTest (hwnd, fUseDMA, TEST_N_BLOCKS, TEST_BLOCK_LENGTH_1, pBlock, &dRdDur1, szText);
if (fErr) {
   ErrorMsg(hwnd, IDS_B8READ);
   DosFreeMem(pBlock);
   return (TRUE); }

/* display warning */
sprintf (szText, "Read test 1 OK. Continue ?");
if (WinMessageBox (HWND_DESKTOP, hwnd, szText, (PSZ)"Link Test", 0, MB_NOICON | MB_OKCANCEL)
    == MBID_CANCEL) {DosFreeMem(pBlock); return (FALSE); }

/* setup test pattern for test 2*/
for (l = 0; l < TEST_BLOCK_LENGTH_2; l++) pBlock[l] = l;

/* perform write test with block length 2 */
fErr = WriteTest (hwnd, fUseDMA, TEST_N_BLOCKS, TEST_BLOCK_LENGTH_2, pBlock, &dWrDur2);
if (fErr) {
   ErrorMsg(hwnd, IDS_B8WRITE);
   DosFreeMem(pBlock);
   return (TRUE); }

/* display warning */
sprintf (szText, "Write test 2 OK. Continue ?");
if (WinMessageBox (HWND_DESKTOP, hwnd, szText, (PSZ)"Link Test", 0, MB_NOICON | MB_OKCANCEL)
    == MBID_CANCEL) {DosFreeMem(pBlock); return (FALSE); }

/* perform read test with block length 1 */
fErr = ReadTest (hwnd, fUseDMA, TEST_N_BLOCKS, TEST_BLOCK_LENGTH_2, pBlock, &dRdDur2, szText);
if (fErr) {
   ErrorMsg(hwnd, IDS_B8READ);
   DosFreeMem(pBlock);
   return (TRUE); }

/* evaluate link performance */
delta = (TEST_BLOCK_LENGTH_2 - TEST_BLOCK_LENGTH_1) * TEST_N_BLOCKS;
if (dRdDur1 == dRdDur2) dRdDur2 = dRdDur1 + 31.0;
if (dWrDur1 == dWrDur2) dWrDur2 = dWrDur1 + 31.0;
dRdTransfTime = (dRdDur2 - dRdDur1) / delta;
dWrTransfTime = (dWrDur2 - dWrDur1) / delta;
dRdBlockOvhd = (dRdDur1 * TEST_BLOCK_LENGTH_2 - dRdDur2 * TEST_BLOCK_LENGTH_1) / delta;
dWrBlockOvhd = (dWrDur1 * TEST_BLOCK_LENGTH_2 - dWrDur2 * TEST_BLOCK_LENGTH_1) / delta;

/* setup result text string */
l = sprintf  (szText,     "      Link test successful !\n\n");
l += sprintf (szText + l, "Link write speed is %.2f KBytes/sec\n", 1/dWrTransfTime);
l += sprintf (szText + l, "Link write overhead is %.2f msec / block\n", dWrBlockOvhd);
l += sprintf (szText + l, "Link read  speed is %.2f KBytes/sec\n", 1/dRdTransfTime);
l += sprintf (szText + l, "Link read  overhead is %.2f msec / block", dRdBlockOvhd);

/* display test result */
WinMessageBox (HWND_DESKTOP, hwnd, szText, (PSZ)"Test Result", 0, MB_INFORMATION | MB_OK | MB_MOVEABLE);

/* exit */
DosFreeMem(pBlock);
return (FALSE);
}

/*************
* WriteTest  *
**************/
BOOL WriteTest(HWND hwnd, BOOL fUseDMA, LONG lBlockCount, LONG lBlockSize,
               BYTE *pBlock, double *dTransferDuration)
{
USHORT   usErr;
ULONG    ulBytesTransf, ulStartTime;
LONG     l;
LONG     lCommand[3];

/*** initiate transputer operation: write test ***/
/* setup command and parameters in buffer pBlock */
lCommand[0] = TP_TEST_WR;    /* TP command */
lCommand[1] = lBlockCount;   /* 1. param: number of blocks */
lCommand[2] = lBlockSize;    /* 2. param: block length */

/* write command and parameters to TP */
if (usErr = B8Write (hfB008, (BYTE *)&lCommand, SRV_TIMEOUT, 3 * sizeof (LONG), &ulBytesTransf))
   return (TRUE);

/*** perform test ***/
/* get system time */
ulStartTime = WinGetCurrentTime(hab);

/* send data blocks to transputer */
for (l = 0; l < lBlockCount; l++) {
    if (fUseDMA)
       usErr = B8DMAWrite (hfB008, pBlock, SRV_TIMEOUT, lBlockSize, &ulBytesTransf);
      else
       usErr = B8Write (hfB008, pBlock, SRV_TIMEOUT, lBlockSize, &ulBytesTransf);
    if (usErr) {
       PutError(hwnd, "Error %x writing byte number %ld of block %d to the link", usErr, ulBytesTransf, l);
       return (TRUE); }
    }

/* get transfer time */
*dTransferDuration = (double)(WinGetCurrentTime(hab) - ulStartTime);
if (*dTransferDuration == 0.0) *dTransferDuration = 31.0;

return (FALSE); /* no error */
}

/************
* ReadTest  *
*************/
BOOL ReadTest(HWND hwnd, BOOL fUseDMA, LONG lBlockCount, LONG lBlockSize,
              BYTE *pBlock, double *dTransferDuration, CHAR *szText)
{
USHORT   usErr, usHeaderVal;
ULONG    ulBytesTransf, ulStartTime;
LONG     l;
LONG     lCommand[3];

/*** initiate transputer operation: write test ***/
/* setup command and parameters in buffer pBlock */
lCommand[0] = TP_TEST_RD;    /* TP command */
lCommand[1] = lBlockCount;   /* 1. param: number of blocks */
lCommand[2] = lBlockSize;    /* 2. param: block length */

/* write command and parameters to TP */
if (usErr = B8Write (hfB008, (BYTE *)&lCommand, SRV_TIMEOUT, 3 * sizeof (LONG), &ulBytesTransf))
   return (TRUE);

/*** perform test ***/
/* get system time */
ulStartTime = WinGetCurrentTime(hab);

/* read data blocks from transputer */
for (l = 0; l < lBlockCount; l++) {
    if (fUseDMA)
       usErr = B8DMARead (hfB008, pBlock, SRV_TIMEOUT, lBlockSize + 6, &ulBytesTransf);
      else
       usErr = B8Read (hfB008, pBlock, SRV_TIMEOUT, lBlockSize + 6, &ulBytesTransf);
    if (usErr) {
       PutError(hwnd, "Error %x reading byte number %ld of block %d to the link", usErr, ulBytesTransf, l);
       return (TRUE); }
    }

/* get transfer time */
*dTransferDuration = (double)(WinGetCurrentTime(hab) - ulStartTime);
if (*dTransferDuration == 0.0) *dTransferDuration = 31.0;

/*** check last received data block ***/
/* check total length of return block */
usHeaderVal = *(USHORT *)pBlock;
if (usHeaderVal != lBlockSize + 4) {
   sprintf (szText, "Link test error: total length of return block has unexpected value %d", usHeaderVal);
   return (TRUE); }

/* check return block identifier */
pBlock += 2; /* point to header */
usHeaderVal = *(USHORT *)pBlock;
if (usHeaderVal != HCMD_TESTDATA) {
   sprintf (szText, "Link test error: header of return block has unexpected value %d", usHeaderVal);
   return (TRUE); }

/* check return block data length */
pBlock += 2; /* point to data length */
usHeaderVal = *(USHORT *)pBlock;
if (usHeaderVal != lBlockSize) {
   sprintf (szText, "Link test error: return block data length has unexpected value %d", usHeaderVal);
   return (TRUE); }

/* check contents of return block data; shall match contents of write block (s.a.) */
pBlock += 2;  /* point to data */
for (l = 0; l < lBlockSize; l++) {
    if (pBlock[l] != (BYTE)l) {
       sprintf (szText, "Link test error: bad data (%x instead of %x) at block position %ld",
                pBlock[l], (BYTE)l, l);
       return (TRUE); }
    }

return (FALSE); /* no error */
}
