///////////////////////////////////////////////////////////////////////////////
// SHARED.H
//
// (C) 2000 Dr J A Gow
//
// Extended High-Resolution Timer system
//
// Shared include file providing definitions of IOCtl and IDC interfaces to
// HRTX. It is possible to use the include files shipped with the DDK to
// communicate with this driver in order to maintain full compatibility
// with the original. This file may, however, contains extensions to the
// original protocol
//
///////////////////////////////////////////////////////////////////////////////

#ifndef _SHARED_H_
#define _SHARED_H_

#ifdef __cplusplus
extern "C" {
#endif

///////////////////////////////////////////////////////////////////////////////
// Useful defines..

//
// IOCtl category for HRTX

#define HRTXIOCTL_CATEGORY		0x80

//
// Maximum number of clients allowed

#define HRTX_MAX_CLIENTS		10

///////////////////////////////////////////////////////////////////////////////
// IDC Prototypes..

//
// Prototype for driver callback function

typedef void (_far _loadds *FPFNHRTXCALLBACK)(void);

//
// Prototype of HRTX IDC entry point

typedef int (_far _loadds _cdecl *FPFNHRTXIDC)(int Command, ULONG Parm1, ULONG Parm2);

///////////////////////////////////////////////////////////////////////////////
// Shared structures

//
// Extended version information

typedef struct _VER {

	unsigned char 	ver[7];
	unsigned char	build[5];
	unsigned char	desc[40];
	
} VER;

typedef VER *	PVER;

///////////////////////////////////////////////////////////////////////////////
// HRTX Error code defines

#define	HRTXERR_NOERROR		0x00    // everything sings
#define HRTXERR_MODONINT	0x01  	// attempt to reset timer from IRQ thread
#define HRTXERR_MAX_CLIENTS	0x02	// reached maximum number of clients
#define HRTXERR_NOSUCHCLIENT	0x03	// no such client exists
#define HRTXERR_BADPARAM	0x04	// invalid parameter
#define HRTXERR_HRTCBFAIL	0x05	// problem attaching to the RTC callback
#define HRTXERR_EARLYREGISTER	0x06	// attempt to register before task time
#define HRTXERR_INVALIDHANDLE	0x07	// bad handle supplied

///////////////////////////////////////////////////////////////////////////////
// HRTX Versioning macros
//
// These are identical to those defined for the original TIMER0 and are here for
// compatibility only. New code should use the HRTXxxxx_X_QUERYVERSION call

#define HRTVER_MAJOR(ver)     (((ver) & 0xC000) >> 14)
#define HRTVER_MINOR(ver)     (((ver) & 0x3C00) >> 10)
#define HRTVER_BUGFIX(ver)    (((ver) & 0x03C0) >> 6)
#define HRTVER_PHASE(ver)     (((ver) & 0x0030) >> 4)
#define HRTVER_BUILD(ver)     ((ver) & 0x000F)

#define HRTVER_DEVELOP        0
#define HRTVER_ALPHA          1
#define HRTVER_BETA           2
#define HRTVER_RELEASE        3

/////////////////////////////
// Ordered IOCtl functions //
/////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// IOCtl: 0x80 	fctn 00:	HRTXIOCTL_QUERYVERSION
//
// Returns a composite version number
//
// ParmPacket:
//
// DataPacket: ULONG - version number to be returned
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_QUERYVERSION		0x00

///////////////////////////////////////////////////////////////////////////////
// IOCtl: 0x80	fctn 01:	HRTXIOCTL_GETRESOLUTION
//
// Returns the current timer resolution.
//
// ParmPacket:
//
// DataPacket: ULONG - resolution to be returned
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_GETRESOLUTION		0x01

///////////////////////////////////////////////////////////////////////////////
// IOCtl: 0x80 	fctn 02:	HRTXIOCTL_SETRESOLUTION
//
// Sets the current timer resolution
//
// ParmPacket: ULONG containing timer resolution
//
// DataPacket:
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_SETRESOLUTION		0x02

///////////////////////////////////////////////////////////////////////////////
// IOCtl: 0x80	fctn 03:	HRTXIOCTL_GETPOINTER
//
// Returns a FAR16 (yeuch!) pointer to the timer count.
// Use HRTXxxxx_X_QUERYCOUNTADDR for something a little more friendly..
//
// ParmPacket:
//
// DataPacket: pointer to FAR16 pointer to master millisecond count
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_GETPOINTER		0x03

///////////////////////////////////////////////////////////////////////////////
// IOCtl: 0x80	fctn 04:	HRTXIOCTL_FREEPOINTER
//
// Provided for compatibility reasons only - this function has no effect
//
// ParmPacket: none
//
// DataPacket: none
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_FREEPOINTER		0x04

///////////////////////////////////////////////////////////////////////////////
// IOCtl: 0x80	fctn 05:	HRTXIOCTL_BLOCKUNTIL
//
// Block the thread until the specified time has elapsed
//
// ParmPacket: ULONG containing time in ms to wait
//
// DataPacket:
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_BLOCKUNTIL		0x05



///////////////////////////
// Ordered IDC functions //
///////////////////////////

///////////////////////////////////////////////////////////////////////////////
// IDC:		fctn 00:	HRTXIDC_REGISTER
//
// Register a PDD with the extended high-resolution timer
//
// Parm1: FAR16 - Address of callback function
//
// Parm2: ULONG - Callback repeat time
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIDC_REGISTER		0x00

///////////////////////////////////////////////////////////////////////////////
// IDC:		fctn 01:	HRTXIDC_DEREGISTER
//
// Deregister a PDD with the extended high-resolution timer
//
// Parm1: unused
//
// Parm2: unused
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIDC_DEREGISTER		0x01

///////////////////////////////////////////////////////////////////////////////
// IDC:		fctn 02:	HRTXIDC_GETPOINTER
//
// Returns a FAR16 pointer to the timer count variable
//
// Parm1: FAR16 pointer to location to receive address of master ms count
//
// Parm2:
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIDC_GETPOINTER		0x02



///////////////////////////////////////////////////////////////////////////////
// EXTENDED HRTX API IOCTLS and IDC functions
//
// All extended API calls are accessible from
// both ring 3 and ring 0
//
// This API set is handle-based. When a timer is registered a handle is
// returned, with which that particular timer may be accessed from any time,
// any place, any where including IRQ. ( Martini timer :) )
//
// Each timer is encapsulated, so any operations on one client will not
// affect any of the others.
//


///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 06		HRTXIOCTL_X_QUERYVERSION
// IDC:         fctn 03		HRTXIDC_X_QUERYVERSION
//
// Fills the provided memory with the contents of the device version block as
// defined in the VER structure
//
// Parm1/ParmPacket (none) : unused
//
// Parm2/DataPacket (PVER) : VER structure to fill.
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_QUERYVERSION	0x06
#define HRTXIDC_X_QUERYVERSION		0x03

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 07		HRTXIOCTL_X_QUERYCOMPATHANDLE
// IDC:		fctn 04		HRTXIDC_X_QUERYCOMPATHANDLE
//
// Returns the handle of the timer allocated by the compatibility API, if one
// exists.
//
// Parm1/ParmPacket (none)   : unused
// Parm2/DataPacket (PULONG) : Pointer to ULONG to receive the handle
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_QUERYCOMPATHANDLE	0x07
#define HRTXIDC_X_QUERYCOMPATHANDLE     0x04

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 08		HRTXIOCTL_X_CREATETIMER
// IDC:		fctn 05		HRTXIDC_X_CREATETIMER
//
// Creates a new timer and returns the handle to it. If the timer is not
// created an error is returned
//
// IDC:
//
// 	Parm1 (FAR16)  : Address of callback function
// 	Parm2 (USHORT) : On entry this should contain the callback
//		         interval. On exit it will contain the handle of the
//		         timer
//
// IOCTL:
//
// 	ParmPacket (none)   : unused
// 	DataPacket (PULONG) : Pointer to ULONG to receive the handle.
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_CREATETIMER		0x08
#define HRTXIDC_X_CREATETIMER		0x05

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 09		HRTXIOCTL_X_DESTROYTIMER
// IDC:		fctn 06		HRTXIDC_X_DESTROYTIMER
//
// Destroys the timer referenced by the handle
//
//
// Parm1/ParmPacket (ULONG) : Handle of timer to destroy
// Parm2/DataPacket (none)  : unused
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_DESTROYTIMER	0x09
#define HRTXIDC_X_DESTROYTIMER		0x06

///////////////////////////////////////////////////////////////////////////////
// IDC: 0x80	fctn 07		HRTXIOCTL_X_MODIFYCALLBACKTIMER
//
// Modifies the callback interval of a timer. Only useful from ring-0 where
// callback functions are used
//
// Parm1 (ULONG) : Handle of timer
// Parm2 (USHORT) : New callback interval
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIDC_X_MODIFYCALLBACKTIMER	0x07

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 0a		HRTXIOCTL_X_SETRESOLUTION
// IDC:         fctn 08		HRTXIDC_X_SETRESOLUTION
//
// Sets the resolution of the timer specified by the handle. Currently this
// can be 1 (1ms resampled) or -1 (1/1024 s)
//
// Parm1/ParmPacket (ULONG) : Handle of timer
// Parm2/DataPacket (ULONG) : Resolution to set
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_SETRESOLUTION	0x0a
#define HRTXIDC_X_SETRESOLUTION		0x08

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 0b		HRTXIOCTL_X_GETRESOLUTION
// IDC:		fctn 09		HRTXIDC_X_GETRESOLUTION
//
// Returns the resolution of the timer specified by the handle
//
// Parm1/ParmPacket (ULONG) : Handle of timer
// Parm2/DataPacket (PULONG): ULONG into which to place resolution
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_GETRESOLUTION	0x0b
#define HRTXIDC_X_GETRESOLUTION		0x09

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 0c		HRTXIOCTL_X_BLOCKTHREAD
// IDC:		fctn 0a		HRTXIDC_X_BLOCKTHREAD
//
// Blocks the calling thread for a given length of time. This function will
// return an error if called in an interrupt context
//
// Parm1/ParmPacket (ULONG) : Handle of timer
// Parm2/DataPacket (ULONG) : Time to block for
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_BLOCKTHREAD		0x0c
#define HRTXIDC_X_BLOCKTHREAD		0x0a

///////////////////////////////////////////////////////////////////////////////
// IOCTL: 0x80	fctn 0d		HRTXIOCTL_X_QUERYCOUNTADDR
// IDC:		fctn 0a		HRTXIDC_X_QUERYCOUNTADDR
//
// Returns an address to the timer count variable for the specified timer.
// For the IOCTL - returns a linear address
// For the IDC   - returns a FAR16 address
//
// Parm1/ParmPacket (ULONG)  : Handle of timer
// Parm2/DataPacket (PULONG) : Pointer to a DWORD location to receive the
// 			       address
//
///////////////////////////////////////////////////////////////////////////////

#define HRTXIOCTL_X_QUERYCOUNTADDR	0x0d
#define HRTXIDC_X_QUERYCOUNTADDR	0x0b

#ifdef __cplusplus
}
#endif


//////////////////////
// end of SHARED.H  //
//////////////////////

#endif

